static const char rcsid[] = "$Id: bxh_convert_afni.cpp,v 1.1 2009-01-15 20:56:15 gadde Exp $";

/*
 * afni2bxh.cc --
 * 
 * Creates a BXH file based on given AFNI file(s)
 */

#include "bxh_utils.h"
#include "bxh_datarec.h"

#include <time.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <math.h>

#include <map>
#include <string>

struct afnientry {
    afnientry() : type(), name(), count(0), strdata(NULL), intdata(NULL), floatdata(NULL) { /* null */ }
    std::string type;
    std::string name;
    int count;
    std::string * strdata;
    int * intdata;
    double * floatdata;
};

#ifdef __cplusplus
extern "C" {
#endif

extern char * domutil_errorbuf;

BXHDocPtr
createDocFromAFNI(const char *ifname, const char *ofname, size_t * hintsize, double * hintorigin, double * hintspacing, double * hintgap, char * forceorient)
{
    BXHDocPtr docp = NULL;
    bxhrawdatarec * datarec = NULL;
    BXHElementPtr imagedata = NULL;
    BXHElementPtr acquisitiondata = NULL;
    BXHElementPtr subject = NULL;
    struct stat statbuf;
    int msbfirst;
    std::map<std::string, afnientry> afnimap;
    FILE * iffp = NULL;
    
    msbfirst = 1;
    msbfirst = (((char *)&msbfirst)[0] == 0);

    {
	size_t iflen = 0;
	char type[64];
	char name[256];
	int count = 0;

	iflen = strlen(ifname);

	if ((iffp = fopen(ifname, "rb")) == NULL) {
	    fprintf(stderr, "afni2bxh: Error opening file %s\n", ifname);
	    perror("fopen");
	    goto FAIL;
	}
	
	while (1) {
	    afnientry ae;
	    if (fscanf(iffp, " type = %64s", &type[0]) < 1 ||
		fscanf(iffp, " name = %256s", &name[0]) < 1 ||
		fscanf(iffp, " count = %d", &count) < 1) {
		break;
	    }
	    ae.type = std::string(&type[0]);
	    ae.name = std::string(&name[0]);
	    ae.count = count;
	    if (strcmp(&type[0], "string-attribute") == 0) {
		int vallen = 0;
		int c;
		std::string val(count, '\0');
		while ((c = fgetc(iffp)) != EOF) {
		    if (c == '\'') break;
		}
		while (vallen < count - 1 && (c = fgetc(iffp)) != EOF) {
		    val[vallen++] = c;
		}
		if ((c = fgetc(iffp)) == EOF || c != '~') {
		    fprintf(stderr, "Error reading AFNI string-attribute.\n");
		    goto FAIL;
		}
		ae.strdata = new std::string(val);
	    } else if (strcmp(&type[0], "integer-attribute") == 0) {
		int vallen = 0;
		int val;
		ae.intdata = new int[count];
		while (vallen < count && fscanf(iffp, " %d", &val) == 1) {
		    ((int *)ae.intdata)[vallen++] = val;
		}
		if (vallen != count) {
		    fprintf(stderr, "Error reading AFNI integer-attribute.\n");
		    delete [] (int *)ae.intdata;
		    goto FAIL;
		}
	    } else if (strcmp(&type[0], "float-attribute") == 0) {
		int vallen = 0;
		double val;
		ae.floatdata = new double[count];
		while (vallen < count && fscanf(iffp, " %lf", &val) == 1) {
		    ((double *)ae.floatdata)[vallen++] = val;
		}
		if (vallen != count) {
		    fprintf(stderr, "Error reading AFNI float-attribute.\n");
		    delete [] (double *)ae.floatdata;
		    goto FAIL;
		}
	    } else {
		fprintf(stderr, "AFNI attribute type %s not recognized.\n",
			&type[0]);
		continue;
	    }
	    afnimap[ae.name] = ae;
	}
    }

    if ((docp = bxh_createDocument()) == NULL)
	goto FAIL;

    if ((imagedata = bxh_getDatarec(docp, "image", NULL)) == NULL)
	goto FAIL;

    datarec = (bxhrawdatarec *)malloc(sizeof(bxhrawdatarec));
    memset(datarec, '\0', sizeof(*datarec));
    
    {
	afnientry ae;
	std::map<std::string, afnientry>::iterator iter;
	int dimnum;
	int i;
	int numbriks = 0;
	char * brikname = NULL;
	bxhdimension * dimx = NULL;
	bxhdimension * dimy = NULL;
	bxhdimension * dimz = NULL;
	bxhdimension * dimt = NULL;

	brikname = strdup(ifname);
	if (strcmp(brikname + strlen(brikname) - 5, ".HEAD") != 0) {
	    fprintf(stderr, "Input file doesn't have .HEAD suffix???");
	    goto FAIL;
	}
	strcpy(brikname + strlen(brikname) - 5, ".BRIK");

	if ((iter = afnimap.find(std::string("DATASET_RANK"))) == afnimap.end()) {
	    fprintf(stderr, "Error: Can't find DATASET_RANK field in %s\n", ifname);
	    goto FAIL;
	}
	ae = (*iter).second;
	datarec->numdims = ((int *)ae.intdata)[0] + 1;
	if (datarec->numdims < 3) {
	    fprintf(stderr, "Dataset must have at least 3 dimensions!\n");
	    goto FAIL;
	}
	if (datarec->numdims > 4) {
	    fprintf(stderr, "Dataset must have at most 4 dimensions!\n");
	    goto FAIL;
	}
	numbriks = ((int *)ae.intdata)[1];
	datarec->dimensions = (bxhdimension *)malloc(sizeof(bxhdimension)*datarec->numdims);
	memset(datarec->dimensions, '\0', sizeof(bxhdimension)*datarec->numdims);
	dimx = &datarec->dimensions[0];
	dimy = &datarec->dimensions[1];
	dimz = &datarec->dimensions[2];
	if (datarec->numdims == 4)
	    dimt = &datarec->dimensions[3];
	dimx->type = strdup("x");
	dimy->type = strdup("y");
	dimz->type = strdup("z");
	if (dimt) dimt->type = strdup("t");

	if ((iter = afnimap.find(std::string("DATASET_DIMENSIONS"))) == afnimap.end()) {
	    fprintf(stderr, "Error: Can't find DATASET_DIMENSIONS field in %s\n", ifname);
	    goto FAIL;
	}
	ae = (*iter).second;
	dimx->size = ((int *)ae.intdata)[0];
	dimy->size = ((int *)ae.intdata)[1];
	dimz->size = ((int *)ae.intdata)[2];
	dimx->units = strdup("mm");
	dimy->units = strdup("mm");
	dimz->units = strdup("mm");
	if (dimt) dimt->size = numbriks;
	if (dimt &&
	    (iter = afnimap.find(std::string("TAXIS_FLOATS"))) != afnimap.end()) {
	    ae = (*iter).second;
	    dimt->origin = ((double *)ae.floatdata)[0];
	    dimt->spacing = ((double *)ae.floatdata)[1];
	    dimt->gap = 0;
	}
	if (dimt &&
	    (iter = afnimap.find(std::string("TAXIS_NUMS"))) != afnimap.end()) {
	    ae = (*iter).second;
	    int unitsspec = ((int *)ae.intdata)[2];
	    if (unitsspec == 77001) {
		// ms
		dimt->units = strdup("ms");
	    } else if (unitsspec == 77002) {
		// s
		dimt->units = strdup("ms");
		dimt->spacing *= 1000.0;
	    } else if (unitsspec == 77003) {
		dimt->units = strdup("Hz");
	    }
	}

	if ((iter = afnimap.find(std::string("ORIENT_SPECIFIC"))) != afnimap.end()) {
	    ae = (*iter).second;
	    for (dimnum = 0; dimnum < 3; dimnum++) {
		int orient = ((int *)ae.intdata)[dimnum];
		double * dirp = (double *)malloc(sizeof(double)*3);
		memset(dirp, '\0', sizeof(double)*3);
		switch (orient) {
		case 0: dirp[0] = -1; break;
		case 1: dirp[0] = 1; break;
		case 2: dirp[1] = 1; break;
		case 3: dirp[1] = -1; break;
		case 4: dirp[2] = 1; break;
		case 5: dirp[2] = -1; break;
		}
		datarec->dimensions[dimnum].direction = dirp;
	    }
	}

	if ((iter = afnimap.find(std::string("ORIGIN"))) != afnimap.end()) {
	    ae = (*iter).second;
	    /* AFNI convention is L, P, S are positive (we use R, A, S) */
	    dimx->origin = -1.0 * ((double *)ae.floatdata)[0];
	    dimy->origin = -1.0 * ((double *)ae.floatdata)[1];
	    dimz->origin = ((double *)ae.floatdata)[2];
	}

	if ((iter = afnimap.find(std::string("DELTA"))) != afnimap.end()) {
	    ae = (*iter).second;
	    /* AFNI encodes direction in sign of delta, though this is
	     * redundant, so we just use absolute value */
	    dimx->spacing = fabs(((double *)ae.floatdata)[0]);
	    dimy->spacing = fabs(((double *)ae.floatdata)[1]);
	    dimz->spacing = fabs(((double *)ae.floatdata)[2]);
	}

	if ((iter = afnimap.find(std::string("BRICK_TYPES"))) != afnimap.end()) {
	    ae = (*iter).second;
	    for (i = 0; i < ae.count - 1; i++) {
		if (((int *)ae.intdata)[i] != ((int *)ae.intdata)[i+1]) {
		    fprintf(stderr, "Error: all bricks must be same type!");
		    goto FAIL;
		}
	    }
	    switch (((int *)ae.intdata)[0]) {
	    case 0: datarec->elemtype = strdup("int8"); datarec->elemsize = 1; break;
	    case 1: datarec->elemtype = strdup("int16"); datarec->elemsize = 2; break;
	    case 2: datarec->elemtype = strdup("int32"); datarec->elemsize = 4; break;
	    case 3: datarec->elemtype = strdup("float32"); datarec->elemsize = 4; break;
	    case 4: datarec->elemtype = strdup("float64"); datarec->elemsize = 8; break;
	    default:
		fprintf(stderr, "Brick type %d not supported!\n", ((int *)ae.intdata)[0]);
		goto FAIL;
	    }
	}

	if ((iter = afnimap.find(std::string("BYTEORDER_STRING"))) == afnimap.end()) {
	    fprintf(stderr, "Error: Can't find BYTEORDER_STRING field in %s\n", ifname);
	    goto FAIL;
	}
	ae = (*iter).second;
	if (strcmp(((std::string *)ae.strdata)->c_str(), "LSB_FIRST") == 0) {
	    datarec->msbfirstfrags = 0;
	} else if (strcmp(((std::string *)ae.strdata)->c_str(), "MSB_FIRST") == 0) {
	    datarec->msbfirstfrags = 1;
	} else {
	    fprintf(stderr, "Byte order %s not supported!\n", (char *)ae.strdata);
	    goto FAIL;
	}

	bxh_datarec_addfrag(datarec, brikname, 0, datarec->elemsize * dimx->size * dimy->size * dimz->size * (dimt ? dimt->size : 1), ofname, 1);
	free(brikname);
    }

    {
	const char * hintnames[4] = { "x", "y", "z", "t" };
	int dimnum;
	if (hintsize[0] != (size_t)-1 ||
	    hintsize[1] != (size_t)-1 ||
	    hintsize[2] != (size_t)-1 ||
	    hintsize[3] != (size_t)-1) {
	    fprintf(stderr, "Warning: --hintsize* options are ignored for AFNI files\n");
	}
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    bxhdimension * dimp = &datarec->dimensions[dimnum];
	    int hintnum;
	    for (hintnum = 0; hintnum < 4; hintnum++) {
		if (strcmp(dimp->type, hintnames[hintnum]) == 0)
		    break;
	    }
	    if (hintnum == 4) continue;
	    if (hintorigin[hintnum] != HUGE_VAL)
		dimp->origin = hintorigin[hintnum];
	    if (hintspacing[hintnum] != HUGE_VAL)
		dimp->spacing = hintspacing[hintnum];
	    if (hintgap[hintnum] != HUGE_VAL)
		dimp->gap = hintgap[hintnum];
	    if (forceorient && hintnum < 3) {
		if (dimp->direction == NULL)
		    dimp->direction = (double *)malloc(sizeof(double)*3);
		memset(dimp->direction, '\0', sizeof(double)*3);
		if (tolower(forceorient[hintnum]) == 'r') {
		    dimp->direction[0] = 1;
		} else if (tolower(forceorient[hintnum]) == 'a') {
		    dimp->direction[1] = 1;
		} else if (tolower(forceorient[hintnum]) == 's') {
		    dimp->direction[2] = 1;
		} else if (tolower(forceorient[hintnum]) == 'l') {
		    dimp->direction[0] = -1;
		} else if (tolower(forceorient[hintnum]) == 'p') {
		    dimp->direction[1] = -1;
		} else if (tolower(forceorient[hintnum]) == 'i') {
		    dimp->direction[2] = -1;
		}
	    }
	}
    }

    bxh_datarec_writeToElement(imagedata, datarec);

    if ((subject = bxh_getSubject(docp)) == NULL)
	goto FAIL;

    if ((acquisitiondata = bxh_getAcquisitionData(docp)) == NULL)
	goto FAIL;

    goto EXIT;

FAIL:
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", domutil_errorbuf);
    }
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }

EXIT:
    if (iffp) {
	fclose(iffp); iffp = NULL;
    }
    if (datarec)
	bxh_datarec_free(datarec);
    if (subject)
	bxh_element_unref(subject);
    if (acquisitiondata)
	bxh_element_unref(acquisitiondata);
    if (imagedata)
	bxh_element_unref(imagedata);

    for (std::map<std::string, afnientry>::const_iterator it = afnimap.begin(); it != afnimap.end(); ++it) {
	afnientry ae = (*it).second;
	if (ae.strdata) {
	    delete ae.strdata; ae.strdata = NULL;
	}
	if (ae.intdata) {
	    delete [] ae.intdata; ae.intdata = NULL;
	}
	if (ae.floatdata) {
	    delete [] ae.floatdata; ae.floatdata = NULL;
	}
    }

    return docp;
}

int
createBXHFromAFNI(const char *ifname, const char *ofname, size_t * hintsize, double * hintorigin, double * hintspacing, double * hintgap, char * forceorient)
{
    int result = 0;
    BXHDocPtr docp = NULL;
    struct stat statbuf;

    if (stat(ofname, &statbuf) == 0) {
	fprintf(stderr, "afni2bxh: file exists ('%s')\n", ofname);
	goto FAIL;
    }

    if ((docp = createDocFromAFNI(ifname, ofname, hintsize, hintorigin, hintspacing, hintgap, forceorient)) == NULL)
	goto FAIL;
    if (bxh_writeFile(docp, ofname) != 0)
	goto FAIL;

    goto EXIT;

FAIL:
    fprintf(stderr, "afni2bxh: conversion failed.\n");
    result = -1;

EXIT:
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }

    return result;
}

BXHDocPtr
bxh_afni2doc(const char *ifname)
{
    size_t hintsize[4] = { (size_t)-1, (size_t)-1, (size_t)-1, (size_t)-1 };
    double hintorigin[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    double hintspacing[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    double hintgap[4] = { HUGE_VAL, HUGE_VAL, HUGE_VAL, HUGE_VAL };
    return createDocFromAFNI(ifname, "dummy.bxh", &hintsize[0], &hintorigin[0], &hintspacing[0], &hintgap[0], NULL);
}

#ifdef __cplusplus
}
#endif

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.7  2007/02/28 20:26:37  gadde
 * Deal better with missing unnecessary fields.
 *
 * Revision 1.6  2005/09/19 16:31:54  gadde
 * Documentation and help message updates.
 *
 * Revision 1.5  2005/05/10 19:48:27  gadde
 * Set gap to 0 (previous gap calculation did not work for sample data).
 *
 * Revision 1.4  2005/03/08 18:09:57  gadde
 * Add support for dimension size/spacing/gap/orientation hints
 *
 * Revision 1.3  2004/06/15 16:16:11  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.2  2004/05/20 17:11:22  gadde
 * Fix opts error.
 *
 * Revision 1.1  2004/04/22 12:56:42  gadde
 * Add AFNI support.
 *
 *
 */
