static const char rcsid[] = "$Id: bxh_convert_minc.c,v 1.1 2009-01-15 20:55:18 gadde Exp $";

/*
 * minc2bxh.cc --
 * 
 * Creates a BXH file based on given MINC file.
 * Requires MINC and netCDF libraries.
 */

#include "bxh_utils.h"
#include "bxh_datarec.h"

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <math.h>

#include <rpc/types.h>
#include <rpc/xdr.h>

#include <netcdf.h>
#include <minc.h>

extern char * domutil_errorbuf;

static const char * progname = NULL;

extern int createBXHFromMINC(const char *ifname, const char *ofname);

static int
get_array_type(XDR * xdrp, int * dimtypep)
{
    int retval = 1;

    if (!xdr_int(xdrp, dimtypep)) {
	fprintf(stderr, "minc2bxh: error reading array type\n");
	goto FAIL;
    }
    if (*dimtypep == 0) {
	if (!xdr_int(xdrp, dimtypep)) {
	    fprintf(stderr, "minc2bxh: error reading second absent\n");
	    goto FAIL;
	}
	if (*dimtypep != 0) {
	    fprintf(stderr, "minc2bxh: bad second absent\n");
	    goto FAIL;
	}
    }

    retval = 1;
    goto EXIT;

  FAIL:
    retval = 0;

  EXIT:
    return retval;
}


static int
skip_nc_att_array(XDR * xdrp, int * numattsp)
{
    int retval = 1;

    int attnum;
    if (!xdr_int(xdrp, numattsp)) {
	fprintf(stderr, "minc2bxh: error reading numatts\n");
	goto FAIL;
    }
    for (attnum = 0; attnum < *numattsp; attnum++) {
	int atttype;
	int typesize;
	int attsize;
	int val;
	int numwords = 0;
	int wordnum = 0;
	char attname[32];
	char * attnameptr = &attname[0];
	if (!xdr_string(xdrp, &attnameptr, sizeof(attname))) {
	    fprintf(stderr, "minc2bxh: error reading attname (attnum %d)\n", attnum);
	    goto FAIL;
	}
	if (!xdr_int(xdrp, &atttype)) {
	    fprintf(stderr, "minc2bxh: error reading atttype (attnum %d, attname %s)\n", attnum, &attname[0]);
	    goto FAIL;
	}
	if (!xdr_int(xdrp, &attsize)) {
	    fprintf(stderr, "minc2bxh: error reading attsize (attnum %d, attname %s)\n", attnum, &attname[0]);
	    goto FAIL;
	}
	switch (atttype) {
	case 1: /* NC_BYTE */
	case 2: /* NC_CHAR */
	    typesize = 1;
	    break;
	case 3: /* NC_SHORT */
	    typesize = 2;
	    break;
	case 4: /* NC_LONG */
	case 5: /* NC_FLOAT */
	    typesize = 4;
	    break;
	case 6: /* NC_DOUBLE */
	    typesize = 8;
	    break;
	default:
	    fprintf(stderr, "minc2bxh: bad atttype\n");
	    goto FAIL;
	}
	numwords = ((attsize * typesize) - ((((attsize * typesize) + 3) % 4) - 3)) / 4;
	for (wordnum = 0; wordnum < numwords; wordnum++) {
	    if (!xdr_int(xdrp, &val)) {
		fprintf(stderr, "minc2bxh: error reading attval (att %d, word %d)\n", attnum, wordnum);
		goto FAIL;
	    }
	}
    }

    retval = 1;
    goto EXIT;

  FAIL:
    retval = 0;

  EXIT:
    return retval;
}

static BXHDocPtr
createDocFromMINC(const char *ifname, const char *ofname, size_t * hintsize, double * hintorigin, double * hintspacing, double * hintgap, char * forceorient)
{
    int ncid = -1;
    int dimid = -1;
    int numdims = -1;
    BXHDocPtr docp = NULL;
    bxhdimension * alldimensions = NULL;
    int spatialdimnum = 0;
    bxhrawdatarec * datarec = NULL;
    BXHElementPtr imagedata = NULL;
    BXHElementPtr acquisitiondata = NULL;
    BXHElementPtr subject = NULL;

    if ((ncid = miopen((char *)ifname, NC_NOWRITE)) == -1) {
	fprintf(stderr, "minc2bxh: can't open MINC file ('%s')\n", ifname);
	goto FAIL;
    }

    if (ncinquire(ncid, &numdims, NULL, NULL, NULL) == -1)
	goto FAIL;

    if ((docp = bxh_createDocument()) == NULL)
	goto FAIL;

    datarec = (bxhrawdatarec *)malloc(sizeof(bxhrawdatarec));
    memset(datarec, '\0', sizeof(*datarec));
    datarec->numdims = numdims;
    alldimensions = (bxhdimension *)malloc(sizeof(bxhdimension)*numdims);

    for (dimid = 0; dimid < numdims; dimid++) {
	long dimsize = -1;
	static char dimname[MAX_NC_NAME] = "";
	int dimvarid = -1;
	int widthvarid = -1;
	nc_type dimvartype;
	bxhdimension * dim = &alldimensions[dimid];
	nc_type atttype;
	int attlen = -1;
	double tmpdouble;

	memset(dim, '\0', sizeof(*dim));

	if (ncdiminq(ncid, dimid, &dimname[0], &dimsize) == -1)
	    goto FAIL;
	dim->size = dimsize;
	
	dimvarid = ncvarid(ncid, &dimname[0]);
	
	if (strcmp(&dimname[0], MIxspace) == 0) {
	    dim->type = strdup("x");
	    dim->direction = (double *)malloc(sizeof(double)*3);
	    dim->direction[0] = 1;
	    dim->direction[1] = 0;
	    dim->direction[2] = 0;
	    widthvarid = ncvarid(ncid, MIxspace_width);
	} else if (strcmp(&dimname[0], MIyspace) == 0) {
	    dim->type = strdup("y");
	    dim->direction = (double *)malloc(sizeof(double)*3);
	    dim->direction[0] = 0;
	    dim->direction[1] = 1;
	    dim->direction[2] = 0;
	    widthvarid = ncvarid(ncid, MIyspace_width);
	} else if (strcmp(&dimname[0], MIzspace) == 0) {
	    dim->type = strdup("z");
	    dim->direction = (double *)malloc(sizeof(double)*3);
	    dim->direction[0] = 0;
	    dim->direction[1] = 0;
	    dim->direction[2] = 1;
	    widthvarid = ncvarid(ncid, MIzspace_width);
	} else if (strcmp(&dimname[0], MItime) == 0) {
	    dim->type = strdup("t");
	    widthvarid = ncvarid(ncid, MItime_width);
	} else if (strcmp(&dimname[0], MIxfrequency) == 0) {
	    dim->type = strdup(&dimname[0]);
	    widthvarid = ncvarid(ncid, MIxfrequency_width);
	} else if (strcmp(&dimname[0], MIyfrequency) == 0) {
	    dim->type = strdup(&dimname[0]);
	    widthvarid = ncvarid(ncid, MIyfrequency_width);
	} else if (strcmp(&dimname[0], MIzfrequency) == 0) {
	    dim->type = strdup(&dimname[0]);
	    widthvarid = ncvarid(ncid, MIzfrequency_width);
	} else if (strcmp(&dimname[0], MItfrequency) == 0) {
	    dim->type = strdup(&dimname[0]);
	    widthvarid = ncvarid(ncid, MItfrequency_width);
	} else {
	    dim->type = strdup(&dimname[0]);
	}

	if (dimvarid != -1) {
	    static char dimspacing[MAX_NC_NAME] = "";
	    static char widthspacing[MAX_NC_NAME] = "";
	    int index;
	    double dir[MI_NUM_SPACE_DIMS];

	    index = 0;
	    if (ncvarinq(ncid, dimvarid, NULL, &dimvartype, NULL, &index, NULL) == -1)
		goto FAIL;

	    if (ncattinq(ncid, dimvarid, MIunits, &atttype, &attlen) != -1) {
		dim->units = (char *)malloc(nctypelen(atttype)*(attlen+1));
		ncattget(ncid, dimvarid, MIunits, dim->units);
		dim->units[attlen] = '\0';
	    }

	    if (ncattinq(ncid, dimvarid, MIunits, &atttype, &attlen) != -1) {
		dim->units = (char *)malloc(nctypelen(atttype)*(attlen+1));
		ncattget(ncid, dimvarid, MIunits, dim->units);
		dim->units[attlen] = '\0';
	    }

	    if (miattget(ncid, dimvarid, MIdirection_cosines, NC_DOUBLE, MI_NUM_SPACE_DIMS, &dir[0], NULL) != -1) {
		if (dim->direction)
		    free(dim->direction);
		dim->direction = (double *)malloc(sizeof(double)*MI_NUM_SPACE_DIMS);
		memcpy(dim->direction, &dir[0], sizeof(double)*MI_NUM_SPACE_DIMS);
	    }
	    if (miattget1(ncid, dimvarid, MIstart, NC_DOUBLE, &tmpdouble) != -1) {
		dim->origin = tmpdouble;
	    }

	    
	    strcpy(dimspacing, MI_REGULAR);
	    miattgetstr(ncid, dimvarid, MIspacing, sizeof(dimspacing), &dimspacing[0]);
	    strcpy(widthspacing, MI_REGULAR);
	    if (widthvarid != -1)
		miattgetstr(ncid, widthvarid, MIspacing, sizeof(widthspacing), &widthspacing[0]);
	    /* we ignore irregular widths */
	    miattget1(ncid, dimvarid, MIstep, NC_DOUBLE, &dim->spacing);
	    if (dim->spacing < 0)
		dim->spacing *= -1;
	    if (strcmp(dimspacing, MI_REGULAR) == 0) {
		double width = 0.0;
		if (widthvarid != -1) {
		    long index = 0;
		    if (mivarget1(ncid, widthvarid, &index, NC_DOUBLE, NULL, &width) == -1) {
			width = dim->spacing;
		    }
		} else {
		    width = dim->spacing;
		}
		dim->gap = dim->spacing - width;
	    } else {
		long start = 0;
		long count = dim->size;
		int ind;
		static char numbuf[128];
		double * dimpoints = NULL;
		dimpoints = (double *)malloc(sizeof(double) * dim->size);
		dim->numdpstructs = 1;
		dim->dpstructs = (bxhdatapoints *)malloc(sizeof(bxhdatapoints));
		dim->dpstructs[0].label;
		if (mivarget(ncid, dimvarid, &start, &count, NC_DOUBLE, MI_SIGNED, dimpoints) != -1) {
		    dim->dpstructs[0].values = (char **)malloc(sizeof(char *) * dim->size);
		    for (ind = 0; ind < dim->size; ind++) {
			sprintf(&numbuf[0], "%g", dimpoints[ind]);
			dim->dpstructs[0].values[ind] = strdup(&numbuf[0]);
		    }
		    dim->dpstructs[0].numvalues = dim->size;
		}
	    }
	}
    }
    miclose(ncid);

    /* now go through the actual file to find the data */
    {
	FILE * fp = NULL;
	XDR xdr;
	XDR * xdrp = &xdr;
	char magic[4] = "";
	char * magicptr = &magic[0];
	int numrecs = 0;
	int dimarray = 0;
	int numdims = 0;
	int * dimsizes = NULL;
	int attarray = 0;
	int numatts = 0;
	int vararray = 0;
	int numvars = 0;

	if ((fp = fopen(ifname, "rb")) == NULL) {
	    fprintf(stderr, "minc2bxh: error opening file '%s'\n", ifname);
	    goto FAIL;
	}

	xdrstdio_create(xdrp, fp, XDR_DECODE);
	
	if (!xdr_opaque(xdrp, magicptr, 4)) {
	    fprintf(stderr, "minc2bxh: error reading magic\n");
	    goto FAIL;
	}
	if (strncmp(&magic[0], "CDF", 3) != 0) {
	    fprintf(stderr, "minc2bxh: bad magic\n");
	    goto FAIL;
	}
	if (!xdr_int(xdrp, &numrecs)) {
	    fprintf(stderr, "minc2bxh: error reading numrecs\n");
	    goto FAIL;
	}
	if (numrecs < 0) {
	    fprintf(stderr, "minc2bxh: numrecs negative\n");
	    goto FAIL;
	}

	if (!get_array_type(xdrp, &dimarray)) {
	    fprintf(stderr, "minc2bxh: error reading dim_array\n");
	    goto FAIL;
	}
	if (dimarray == 0) {
	    /* absent */
	} else if (dimarray == 10) { /* NC_DIMENSION */
	    int dimnum;
	    static char dimname[32];
	    char * dimnameptr = &dimname[0];
	    if (!xdr_int(xdrp, &numdims)) {
		fprintf(stderr, "minc2bxh: error reading numdims\n");
		goto FAIL;
	    }
	    dimsizes = (int *)malloc(sizeof(int) * numdims);
	    for (dimnum = 0; dimnum < numdims; dimnum++) {
		if (!xdr_string(xdrp, &dimnameptr, sizeof(dimname))) {
		    fprintf(stderr, "minc2bxh: error reading dimname\n");
		    goto FAIL;
		}
		if (!xdr_int(xdrp, &dimsizes[dimnum])) {
		    fprintf(stderr, "minc2bxh: error reading dimsize\n");
		    goto FAIL;
		}
	    }
	} else {
	    fprintf(stderr, "minc2bxh: bad dim_array\n");
	    goto FAIL;
	}

	if (!get_array_type(xdrp, &attarray)) {
	    fprintf(stderr, "minc2bxh: error reading gatt_array\n");
	    goto FAIL;
	}
	if (attarray == 0) {
	    /* absent */
	} else if (attarray == 12) { /* NC_ATTRIBUTE */
	    if (!skip_nc_att_array(xdrp, &numatts)) {
		fprintf(stderr, "minc2bxh: error reading gatt_array\n");
		goto FAIL;
	    }
	} else {
	    fprintf(stderr, "minc2bxh: bad gatt_array\n");
	    goto FAIL;
	}

	if (!get_array_type(xdrp, &vararray)) {
	    fprintf(stderr, "minc2bxh: error reading var_array\n");
	    goto FAIL;
	}
	if (vararray == 0) {
	    /* absent */
	} else if (vararray == 11) { /* NC_VARIABLE */
	    int varnum;
	    if (!xdr_int(xdrp, &numvars)) {
		fprintf(stderr, "minc2bxh: error reading numvars\n");
		goto FAIL;
	    }
	    for (varnum = 0; varnum < numvars; varnum++) {
		static char varname[64];
		char * varnameptr = &varname[0];
		int numvardims;
		int vardimnum;
		int varsize = 1;
		int vattarray;
		int numvatts;
		int nc_type;
		int typesize;
		int vsize;
		int begin;
		if (!xdr_string(xdrp, &varnameptr, sizeof(varname))) {
		    fprintf(stderr, "minc2bxh: error reading varname (var %d)\n", varnum);
		    goto FAIL;
		}
		if (!xdr_int(xdrp, &numvardims)) {
		    fprintf(stderr, "minc2bxh: error reading varnumdims (var %d, varname %s)\n", varnum, varnameptr);
		    goto FAIL;
		}
		if (strcmp(varnameptr, MIimage) == 0) {
		    datarec->dimensions = (bxhdimension *)malloc(sizeof(bxhdimension) * numvardims);
		}
		for (vardimnum = numvardims-1; vardimnum >= 0; vardimnum--) {
		    int dimid;
		    if (!xdr_int(xdrp, &dimid)) {
			fprintf(stderr, "minc2bxh: error reading vardimid (var %d, varname %s)\n", varnum, varnameptr);
			goto FAIL;
		    }
		    varsize *= dimsizes[dimid];
		    if (strcmp(varnameptr, MIimage) == 0) {
			int dpstructind = 0;
			bxhdimension * dim = &datarec->dimensions[vardimnum];
			bxhdimension * olddim = &alldimensions[dimid];
			memset(dim, '\0', sizeof(*dim));
			if (strcmp(olddim->type, "z") == 0 ||
			    strcmp(olddim->type, "y") == 0 ||
			    strcmp(olddim->type, "x") == 0) {
			    switch (spatialdimnum) {
			    case 0:
				dim->type = strdup("z");
				break;
			    case 1:
				dim->type = strdup("y");
				break;
			    case 2:
				dim->type = strdup("x");
				break;
			    default:
				fprintf(stderr, "more than 3 spatial dimensions?\n");
				return -1;
			    }
			    spatialdimnum++;
			}
			if (olddim->units)
			    dim->units = strdup(olddim->units);
			dim->size = olddim->size;
			dim->origin = olddim->origin;
			dim->gap = olddim->gap;
			dim->spacing = olddim->spacing;
			if (olddim->direction) {
			    dim->direction = (double *)malloc(sizeof(double)*3);
			    memcpy(dim->direction, olddim->direction, sizeof(double)*3);
			}
			for (dpstructind = 0; dpstructind < olddim->numdpstructs; dpstructind++) {
			    int pointnum;
			    bxhdatapoints * dpstructp = dim->dpstructs[dim->numdpstructs];
			    dim->dpstructs = (bxhdatapoints *)realloc(sizeof(bxhdatapoints) * (dim->numdpstructs + 1));
			    dim->numdpstructs++;
			    dpstructp->label = NULL;
			    dpstructp->values = (char **)malloc(sizeof(char *)*dim->size);
			    for (pointnum = 0; pointnum < dim->size; pointnum++) {
				dpstructp->values[pointnum] =
				    strdup(olddim->dpstructs[dpstructind].values[pointnum]);
			    }
			}
		    }
		}

		if (!get_array_type(xdrp, &vattarray)) {
		    fprintf(stderr, "minc2bxh: error reading vatt_array (var %d, varname %s)\n", varnum, varnameptr);
		    goto FAIL;
		}
		if (vattarray == 0) {
		    /* absent */
		} else if (vattarray == 12) { /* NC_ATTRIBUTE */
		    if (!skip_nc_att_array(xdrp, &numvatts)) {
			fprintf(stderr, "minc2bxh: error reading vatt_array (var %d, varname %s)\n", varnum, varnameptr);
			goto FAIL;
		    }
		} else {
		    fprintf(stderr, "minc2bxh: bad vatt_array (var %d, varname %s)\n", varnum, varnameptr);
		    goto FAIL;
		}

		if (!xdr_int(xdrp, &nc_type)) {
		    fprintf(stderr, "minc2bxh: error reading vartype (var %d, varname %s)\n", varnum, varnameptr);
		    goto FAIL;
		}
		switch (nc_type) {
		case 1: /* NC_BYTE */
		case 2: /* NC_CHAR */
		    typesize = 1;
		    break;
		case 3: /* NC_SHORT */
		    typesize = 2;
		    break;
		case 4: /* NC_LONG */
		case 5: /* NC_FLOAT */
		    typesize = 4;
		    break;
		case 6: /* NC_DOUBLE */
		    typesize = 8;
		    break;
		default:
		    fprintf(stderr, "minc2bxh: bad vartype (var %d, varnameptr %s)\n", varnum, varnameptr);
		    goto FAIL;
		}
		varsize *= typesize;
		
		if (!xdr_int(xdrp, &vsize)) {
		    fprintf(stderr, "minc2bxh: error reading vsize (var %d, varnameptr %s)\n", varnum, varnameptr);
		    goto FAIL;
		}

		if (!xdr_int(xdrp, &begin)) {
		    fprintf(stderr, "minc2bxh: error reading varbegin (var %d, varnameptr %s)\n", varnum, varnameptr);
		    goto FAIL;
		}

		if (strcmp(varnameptr, MIimage) == 0) {
		    /* fill in rest of data for image */
		    switch (nc_type) {
		    case 1: /* NC_BYTE */
			datarec->elemtype = strdup("uint8");
			break;
		    case 2: /* NC_CHAR */
			datarec->elemtype = strdup("ascii");
			break;
		    case 3: /* NC_SHORT */
			datarec->elemtype = strdup("int16");
			break;
		    case 4: /* NC_LONG */
			datarec->elemtype = strdup("int32");
			break;
		    case 5: /* NC_FLOAT */
			datarec->elemtype = strdup("float32");
			break;
		    case 6: /* NC_DOUBLE */
			datarec->elemtype = strdup("float64");
			break;
		    default:
			fprintf(stderr, "minc2bxh: bad vartype (var %d, varname %s)\n", varnum, varnameptr);
			goto FAIL;
		    }

		    datarec->msbfirstfrags = 1;
		    datarec->elemsize = typesize;
		    datarec->datasize = varsize;
		    bxh_datarec_addfrag(datarec, ifname, begin, varsize, ofname, 1);
		}
	    }
	} else {
	    fprintf(stderr, "minc2bxh: bad var_array\n");
	    goto FAIL;
	}
    }

    /* Start really creating the BXH file */

    if ((docp = bxh_createDocument()) == NULL)
	goto FAIL;

    if ((imagedata = bxh_getDatarec(docp, "image", NULL)) == NULL)
	goto FAIL;

    {
	char * hintnames[4] = { "x", "y", "z", "t" };
	int dimnum;
	if (hintsize[0] != (size_t)-1 ||
	    hintsize[1] != (size_t)-1 ||
	    hintsize[2] != (size_t)-1 ||
	    hintsize[3] != (size_t)-1) {
	    fprintf(stderr, "Warning: --hintsize* options are ignored for MINC files\n");
	}
	for (dimnum = 0; dimnum < datarec->numdims; dimnum++) {
	    bxhdimension * dimp = &datarec->dimensions[dimnum];
	    int hintnum;
	    for (hintnum = 0; hintnum < 4; hintnum++) {
		if (strcmp(dimp->type, hintnames[hintnum]) == 0)
		    break;
	    }
	    if (hintnum == 4) continue;
	    if (hintorigin[hintnum] != HUGE_VAL)
		dimp->origin = hintorigin[hintnum];
	    if (hintspacing[hintnum] != HUGE_VAL)
		dimp->spacing = hintspacing[hintnum];
	    if (hintgap[hintnum] != HUGE_VAL)
		dimp->gap = hintgap[hintnum];
	    if (forceorient && hintnum < 3) {
		if (dimp->direction == NULL)
		    dimp->direction = (double *)malloc(sizeof(double)*3);
		memset(dimp->direction, '\0', sizeof(double)*3);
		if (tolower(forceorient[hintnum]) == 'r') {
		    dimp->direction[0] = 1;
		} else if (tolower(forceorient[hintnum]) == 'a') {
		    dimp->direction[1] = 1;
		} else if (tolower(forceorient[hintnum]) == 's') {
		    dimp->direction[2] = 1;
		} else if (tolower(forceorient[hintnum]) == 'l') {
		    dimp->direction[0] = -1;
		} else if (tolower(forceorient[hintnum]) == 'p') {
		    dimp->direction[1] = -1;
		} else if (tolower(forceorient[hintnum]) == 'i') {
		    dimp->direction[2] = -1;
		}
	    }
	}
    }

    bxh_datarec_writeToElement(imagedata, datarec);

    if ((subject = bxh_getSubject(docp)) == NULL)
	goto FAIL;

    if ((acquisitiondata = bxh_getAcquisitionData(docp)) == NULL)
	goto FAIL;

    if (bxh_addAutoHistoryEntry(docp, progname, &ifname, 1) != 0)
	goto FAIL;

FAIL:
    fprintf(stderr, "minc2bxh: conversion failed.\n");
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", domutil_errorbuf);
    }
    result = -1;

EXIT:
    if (alldimensions)
	free(alldimensions); /* do something more here! */
    if (datarec)
	bxh_datarec_free(datarec);
    if (subject)
	bxh_element_unref(subject);
    if (acquisitiondata)
	bxh_element_unref(acquisitiondata);
    if (imagedata)
	bxh_element_unref(imagedata);
    if (docp)
	bxh_freeDocument(docp);

    return result;
}

int
createBXHFromMINC(const char *ifname, const char *ofname, size_t * hintsize, double * hintorigin, double * hintspacing, double * hintgap, char * forceorient)
{
    int result = 0;
    struct stat statbuf;
    BXHDocPtr docp = NULL;

    if (stat(ofname, &statbuf) == 0) {
	fprintf(stderr, "minc2bxh: file exists ('%s')\n", ofname);
	goto FAIL;
    }

    if ((docp = createDocFromMINC(ifname, ofname, hintsize, hintorigin, hintspacing, hintgap, forceorient)) == NULL)
	goto FAIL;
    if (bxh_writeFile(docp, ofname) != 0)
	goto FAIL;

    goto EXIT;

FAIL:
    result = -1;

EXIT:
    if (docp) {
	bxh_freeDocument(docp);
	docp = NULL;
    }
    return result;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.14  2006/04/07 14:47:38  gadde
 * Fix memory bug.
 *
 * Revision 1.13  2006/03/23 17:39:15  gadde
 * Solaris fixes.
 *
 * Revision 1.12  2005/09/19 16:31:58  gadde
 * Documentation and help message updates.
 *
 * Revision 1.11  2005/09/14 15:19:17  gadde
 * Some -Wall fixes.
 *
 * Revision 1.10  2005/03/08 18:09:58  gadde
 * Add support for dimension size/spacing/gap/orientation hints
 *
 * Revision 1.9  2004/06/15 19:34:16  gadde
 * Make sure spacing is not negative.
 *
 * Revision 1.8  2004/06/15 16:16:11  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.7  2004/02/24 18:01:07  gadde
 * Move history generation to bxh_utils.cpp, and add option to create BIRN files
 *
 * Revision 1.6  2004/02/20 18:42:48  gadde
 * Add version option and rearrange documentation
 *
 * Revision 1.5  2003/08/19 19:35:41  gadde
 * New options scheme, so that options are available using whichever form of bxhabsorb you use.
 *
 * Revision 1.4  2003/07/21 16:46:49  gadde
 * Code-prettiness updates, for the most part, to further protect BXH
 * library users from particulars of DOM implementation (esp. C vs. C++).
 *
 * Revision 1.3  2003/06/18 16:08:37  gadde
 * fopen in binary mode
 *
 * Revision 1.2  2003/05/10 18:22:12  gadde
 * Add Log.
 *
 */
