static const char* rcsid() { rcsid(); return "$Id: bxh_dicom.cpp,v 1.1 2009/04/24 16:40:57 gadde Exp $"; }

/* bxh_dicom.cpp --
 *
 * Extract DICOM data into BXH files using these functions.
 *
 * Author: Syam Gadde (gadde@biac.duke.edu), Nov. 2002.
 */


#include <osconfig.h>    /* make sure OS specific configuration is included first */

#ifdef LINUX
#include <features.h>
#endif
#ifdef WIN32
#include <windows.h>
#endif
#include <iostream>
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#ifndef WIN32
#include <dirent.h>
#endif
#ifndef WIN32
#include <libgen.h>
#endif
#include <math.h>
#include <float.h>
#include <vector>

#include <gsl/gsl_matrix.h>
#include <gsl/gsl_linalg.h>

#include <map>

/* DICOM includes */
#include <dcdict.h>
#include <dctk.h>
//#include <dcdebug.h>
#include <dcistrmfb.h>
#include <cmdlnarg.h>
#include <ofconapp.h>
#include <ofstd.h>
#include <dcuid.h>    /* for dcmtk version name */
#define SHORTCOL 3
#define LONGCOL 20

#include "bxh_dicom.h"
#include "bxh_utils.h"
#include "bxh_datarec.h"

#include <boost/smart_ptr.hpp>

#ifndef TOPLEVELELEM
#define TOPLEVELELEM "bxh"
#endif

#ifdef __cplusplus
extern "C" {
#endif
extern char * domutil_errorbuf;
#ifdef __cplusplus
}
#endif

//#define DEBUG 1

#define MIN(x, y) (((x) < (y)) ? (x) : (y))

struct dicomfileinfo {
    dicomfileinfo()
	: filename(NULL),
	  fileoffset(0),
	  studyuid(NULL),
	  seriesuid(NULL),
	  imagetype(NULL),
	  inplanephaseencodingdirection(NULL),
	  modality(NULL),
	  manufacturer(NULL),
	  stackid(NULL),
	  rowsr(0), rowsa(0), rowss(0),
	  colsr(0), colsa(0), colss(0),
	  flipangle(0),
	  colspacing(0),
	  rowspacing(0),
	  slicethickness(0),
	  slicespacing(0),
	  originr(0),
	  origina(0),
	  origins(0),
	  tr(0),
	  foundtr(0),
	  te(0),
	  foundte(0),
	  echonumber(0),
	  t(0),
	  sliceacqtime(0),
	  bmatrixorder(0),
	  bvalue(0),
	  scalex(0),
	  scaley(0),
	  scalez(0),
	  frameidentity(0),
	  acqnum(0),
	  gerawdatatype(0),
	  geswapphasefrequency(0),
	  rows(0),
	  columns(0),
	  samplesperpixel(0),
	  bitsallocated(0),
	  instance(0),
	  slicenum(-1),
	  framenum(0),
	  ismosaic(0),
	  foundtimedim(0),
	  oldsiemensmap(),
	  philipsmysteryfield(0),
	  dimsequenceidentifier(),
	  unrecognized_indices()
    { /* null */ }

    ~dicomfileinfo() {
	free(filename); filename = NULL;
	free(studyuid); studyuid = NULL;
	free(seriesuid); seriesuid = NULL;
	free(imagetype); imagetype = NULL;
	free(inplanephaseencodingdirection); inplanephaseencodingdirection = NULL;
	free(manufacturer); manufacturer = NULL;
	free(modality); modality = NULL;
	free(stackid); stackid = NULL;
    }
	  
    char * filename;
    off_t fileoffset;
    char * studyuid;
    char * seriesuid;
    char * imagetype;
    char * inplanephaseencodingdirection;
    char * modality;
    char * manufacturer;
    char * stackid;
    Float64 rowsr, rowsa, rowss;
    Float64 colsr, colsa, colss;
    Float64 flipangle;
    Float64 colspacing;
    Float64 rowspacing;
    Float64 slicethickness;
    Float64 slicespacing;
    Float64 originr;
    Float64 origina;
    Float64 origins;
    Float64 tr;
    Float64 foundtr;
    Float64 te;
    Float64 foundte;
    double echonumber;
    double t;
    double sliceacqtime;
    double bmatrixorder;
    double bvalue;
    double scalex;
    double scaley;
    double scalez;
    double acqnum;
    int frameidentity;
    Sint16 gerawdatatype;
    Sint16 geswapphasefrequency;
    Uint16 rows;
    Uint16 columns;
    Uint16 samplesperpixel;
    Uint16 bitsallocated;
    Sint32 instance;
    Sint32 slicenum;
    int framenum;
    int ismosaic;
    int foundtimedim;
    std::map<OFString,OFString> oldsiemensmap;
    Sint32 philipsmysteryfield;
    boost::shared_ptr<OFString> dimsequenceidentifier;
    boost::shared_ptr<OFString> unrecognized_indices;

    std::string protect(const char * value) const {
	if (value == NULL) {
	    return std::string("<null>");
	}
	bool quote = false;
	std::string newvalue;
	newvalue.reserve(strlen(value) * 2);
	const char * lastrawstart = value;
	const char * iter = NULL;
	for (iter = value; *iter != '\0'; iter++) {
	    char c = *iter;
	    const char *replacement = NULL;
	    if (c == ' ') {
		quote = true;
	    }
	    if (!isprint(c)) {
		replacement = "";
	    } else if (c == '\n') {
		replacement = "\\n";
	    } else if (c == '\'') {
		replacement = "\\'";
	    } else {
		continue;
	    }
	    /* if we got here, we have to flush and print out replacement */
	    size_t buflen = iter - lastrawstart;
	    if (buflen != 0) {
		newvalue.append(lastrawstart, buflen);
	    }
	    newvalue += replacement;
	    lastrawstart = iter + 1;
	}
	size_t buflen = iter - lastrawstart;
	if (buflen != 0) {
	    newvalue.append(lastrawstart, buflen);
	}
	if (quote) {
	    newvalue = std::string("\'") + newvalue + '\'';
	}
	return newvalue;
    }

    friend std::ostream& operator <<(std::ostream& out, const dicomfileinfo& obj);

    void print(std::ostream & out) const {
	out << " filename=" << protect(filename)
	    << " fileoffset=" << fileoffset
	    << " studyuid=" << protect(studyuid)
	    << " seriesuid=" << protect(seriesuid)
	    << " imagetype=" << protect(imagetype)
	    << " inplanephaseencodingdirection=" << protect(inplanephaseencodingdirection)
	    << " modality=" << protect(modality)
	    << " manufacturer=" << protect(manufacturer)
	    << " stackid=" << protect(stackid)
	    << " rowsr=" << rowsr
	    << " rowsa=" << rowsa
	    << " rowss=" << rowss
	    << " colsr=" << colsr
	    << " colsa=" << colsa
	    << " colss=" << colss
	    << " flipangle=" << flipangle
	    << " colspacing=" << colspacing
	    << " rowspacing=" << rowspacing
	    << " slicethickness=" << slicethickness
	    << " slicespacing=" << slicespacing
	    << " originr=" << originr
	    << " origina=" << origina
	    << " origins=" << origins
	    << " tr=" << tr
	    << " te=" << te
	    << " t=" << t
	    << " bmatrixorder=" << bmatrixorder
	    << " bvalue=" << bvalue
	    << " scalex=" << scalex
	    << " scaley=" << scaley
	    << " scalez=" << scalez
	    << " acqnum=" << acqnum
	    << " frameidentity=" << frameidentity
	    << " gerawdatatype=" << gerawdatatype
	    << " geswapphasefrequency=" << geswapphasefrequency
	    << " rows=" << rows
	    << " columns=" << columns
	    << " samplesperpixel=" << samplesperpixel
	    << " bitsallocated=" << bitsallocated
	    << " instance=" << instance
	    << " slicenum=" << slicenum
	    << " framenum=" << framenum
	    << " ismosaic=" << ismosaic
	    << " foundtimedim=" << foundtimedim
	    << " philipsmysteryfield=" << philipsmysteryfield
	    << std::endl;
    }
};

std::ostream &
operator <<(std::ostream& out, const dicomfileinfo& finfo)
{
    finfo.print(out);
}

struct dimmapentry {
    /* dicomfileinfo fields used here are assumed to be doubles */
    double dicomfileinfo::* memberp;   /* pointer to member of dicomfileinfo struct that should be used for distinguishing items along this dimension */
    double dicomfileinfo::* valuep;   /* if not NULL, pointer to member of dicomfileinfo struct that should be used for extracting value */
    const char * dimname; /* name of dimension, should we choose to create it */
    const char * units;   /* units of this dimension */
    bxhdimension ** dim; /* store new dimension pointer into location pointed here */
};

void printfinfos(FILE * fp, int numfinfos, std::vector<struct dicomfileinfo *> finfos, struct dimmapentry * newdimmap) {
	int numnd;
	for (numnd = 0; newdimmap[numnd].dimname; numnd++) {
	    /* null */
	}
	for (int filenum = 0; filenum < numfinfos; filenum++) {
	    dicomfileinfo * finfop = finfos[filenum];
	    fprintf(fp, "%s: instance=%d slicenum=%d", finfop->filename, (int)finfop->instance, (int)finfop->slicenum);
	    int ndnum;
	    for (ndnum = numnd-1; ndnum >= 0; ndnum--) {
		double ival;
		ival = finfop->*(newdimmap[ndnum].memberp);
		fprintf(fp, " %s=%g", newdimmap[ndnum].dimname, ival);
	    }
	    fprintf(fp, " tr=%g te=%g bmatrixorder=%g bvalue=%g acqnum=%g gerawdatatype=%d",
		    finfop->tr, finfop->te, finfop->bmatrixorder, finfop->bvalue, finfop->acqnum, (int)finfop->gerawdatatype);
	    fprintf(fp, " (%g, %g, %g)",
		    finfop->originr,
		    finfop->origina,
		    finfop->origins);
	    fprintf(fp, "\n");
	}
}

/* these are elements that we need to grab from every dataset, so we
 * create a cache structure to store these values, and another table
 * storing the tag names/values so that we can sort and read them in
 * dataset order to avoid backtracking in the dataset.
 * (other values we can just get from the first instance in the group
 *  and don't need to be as careful)
 */
struct dicomelemcache {
    DcmElement * acquisitioncontrast;
    DcmElement * acquisitiondate;
    DcmElement * acquisitionnumber;
    DcmElement * acquisitiontime;
    DcmElement * samplesperpixel;
    DcmElement * bitsallocated;
    DcmElement * columns;
    DcmElement * contentdate;
    DcmElement * contenttime;
    DcmElement * diffusionbvalue;
    DcmElement * diffusiongradientorientation;
    DcmElement * echotime;
    DcmElement * echonumber;
    DcmElement * flipangle;
    DcmElement * imageorientationpatient;
    DcmElement * imagepositionpatient;
    DcmElement * imagetype;
    DcmElement * inplanephaseencodingdirection;
    DcmElement * instancenumber;
    DcmElement * manufacturer;
    DcmElement * numberofframes;
    DcmElement * sopclassuid;
    DcmElement * modality;
    DcmElement * pixelspacing;
    DcmElement * repetitiontime;
    DcmElement * rows;
    DcmElement * seriesinstanceuid;
    DcmElement * slicethickness;
    DcmElement * spacingbetweenslices;
    DcmElement * studyinstanceuid;
    DcmElement * temporalpositionidentifier;
    DcmElement * temporalpositionindex;
    DcmElement * temporalresolution;

    DcmElement * sharedfunctionalgroupssequence;
    DcmElement * perframefunctionalgroupssequence;
    DcmElement * dimensionindexsequence;
    DcmElement * dimensionindexvalues;
    DcmElement * stackid;
    DcmElement * instackpositionnumber;

    DcmElement * pixeldata;

    DcmElement * geautoprescancenterfrequency;
    DcmElement * geautoprescantransmitgain;
    DcmElement * gepulsesequencename;
    DcmElement * gerawdatatype;
    DcmElement * geslopinteger6;
    DcmElement * bvaluebiasfactor;
    DcmElement * geuserdata20;
    DcmElement * geuserdata21;
    DcmElement * geuserdata22;
    DcmElement * geswapphasefrequency;

    DcmElement * siemensbvalue;
    DcmElement * siemensdiffusionvector;
    DcmElement * siemensCSAImageHeaderInfo;

    DcmElement * philipsslicenumbermr;
    DcmElement * philips_2005_1413;
};

struct dicomcacheinit {
    DcmElement * dicomelemcache::*memberp; /* pointer to member of class */
    const char * tagname;
    const char * privCreator;
    Uint16 group;
    Uint16 elem;
    DcmTag tag;
};

/* just for fields that are stored within sequences defined by macros used by
 * the Enhanced MR Image Module */
struct dicomcacheinitenhanced {
    std::vector<DcmTag> sequencetags;
    DcmTag elemtag;
    DcmElement * dicomelemcache::* memberp;
    dicomcacheinitenhanced * subelem;
    dicomcacheinitenhanced(DcmTag sequencetag_in,
			   DcmTag elemtag_in,
			   DcmElement * dicomelemcache::* memberp_in,
			   dicomcacheinitenhanced * subelem_in = (dicomcacheinitenhanced *)NULL)
	: sequencetags(),
	  elemtag(elemtag_in),
	  memberp(memberp_in),
	  subelem(subelem_in)
    {
	this->sequencetags.push_back(sequencetag_in);
    }
    dicomcacheinitenhanced(std::vector<DcmTag> & sequencetags_in,
			   DcmTag elemtag_in,
			   DcmElement * dicomelemcache::* memberp_in,
			   dicomcacheinitenhanced * subelem_in = (dicomcacheinitenhanced *)NULL)
	: sequencetags(sequencetags_in),
	  elemtag(elemtag_in),
	  memberp(memberp_in),
	  subelem(subelem_in)
    {
	/* no-op */
    }
};

struct dicomdatameta {
    DcmDataset * dataset;
    DcmMetaInfo * metainfo;
};

struct enhanceddimindex {
    DcmTagKey funcgroupptr;
    DcmTagKey dimindexptr;

    enhanceddimindex()
	: funcgroupptr(),
	  dimindexptr()
    { /* null */ }
    enhanceddimindex(DcmTagKey funcgroupptr_, DcmTagKey dimindexptr_)
	: funcgroupptr(funcgroupptr_),
	  dimindexptr(dimindexptr_)
    { /* null */ }
};

static
int
bxh_DICOM_read_from_stream(DcmInputFileBufferedStream * myinp, unsigned char * buf, long readsize, long skip)
{
    offile_off_t curpos = myinp->tell();
    if (skip > 0) {
	if (myinp->skip(skip) < skip) {
	    fprintf(stderr, "Error seeking in file to position %u!\n", (unsigned int)(curpos + skip));
	    return -1;
	}
	curpos += skip;
    }
    if (myinp->read(buf, readsize) != readsize) {
	fprintf(stderr, "Error reading %u bytes at byte pos %u!\n", (unsigned int)readsize, (unsigned int)curpos);
	if (!myinp->good()) {
	    fprintf(stderr, "Error: %s\n", myinp->status().text());
	}
	return -1;
    }
    return 0;
}

static
long
bxh_DICOM_pixel_data_offset(DcmInputFileBufferedStream * myinp, Uint32 pixelDataSize, int littleendian, int explicitsyntax)
{
    long retval = -1;
    long bufposend = 0;
    size_t buflen = 0;
    long readsize = 0;
    FILE * fp = NULL;
    static unsigned char buf[1024];
    unsigned char comparestring[13];
    unsigned char comparestring2[13];
    int comparestringlen = 0;
    comparestring[0] = '\0';
    comparestring2[0] = '\0';
#if DEBUG
    fprintf(stderr, "pixelDataSize=%u(0x%x) littleendian=%d explicitsyntax=%d\n", pixelDataSize, pixelDataSize, littleendian, explicitsyntax);
#endif
    if (littleendian) {
	if (explicitsyntax) {
	    memcpy(&comparestring[0], "\xe0\x7f\x10\x00OW\0\0", 8);
	    comparestring[11] = ((pixelDataSize & 0xff000000) >> 24);
	    comparestring[10] = ((pixelDataSize & 0xff0000) >> 16);
	    comparestring[9] = ((pixelDataSize & 0xff00) >> 8);
	    comparestring[8] = (pixelDataSize & 0xff);
	    comparestring[12] = '\0';
	    memcpy(comparestring2, comparestring, 13);
	    comparestring2[5] = 'B';
	    comparestringlen = 12;
	} else {
	    memcpy(&comparestring[0], "\xe0\x7f\x10\x00", 4);
	    comparestring[7] = ((pixelDataSize & 0xff000000) >> 24);
	    comparestring[6] = ((pixelDataSize & 0xff0000) >> 16);
	    comparestring[5] = ((pixelDataSize & 0xff00) >> 8);
	    comparestring[4] = (pixelDataSize & 0xff);
	    comparestring[8] = '\0';
	    comparestringlen = 8;
	}
    } else {
	if (explicitsyntax) {
	    memcpy(&comparestring[0], "\x7f\xe0\x00\x10OW\0\0", 8);
	    comparestring[8] = ((pixelDataSize & 0xff000000) >> 24);
	    comparestring[9] = ((pixelDataSize & 0xff0000) >> 16);
	    comparestring[10] = ((pixelDataSize & 0xff00) >> 8);
	    comparestring[11] = (pixelDataSize & 0xff);
	    comparestring[12] = '\0';
	    memcpy(comparestring2, comparestring, 13);
	    comparestring2[5] = 'B';
	    comparestringlen = 12;
	} else {
	    memcpy(&comparestring[0], "\x7f\xe0\x00\x10", 4);
	    comparestring[4] = ((pixelDataSize & 0xff000000) >> 24);
	    comparestring[5] = ((pixelDataSize & 0xff0000) >> 16);
	    comparestring[6] = ((pixelDataSize & 0xff00) >> 8);
	    comparestring[7] = (pixelDataSize & 0xff);
	    comparestring[8] = '\0';
	    comparestringlen = 8;
	}
    }
#if 0
    {
      int strpos;
      fprintf(stderr, "comparestring (len=%d): ", comparestringlen);
      for (strpos = 0; strpos < comparestringlen; strpos++) {
        fprintf(stderr, " %02x", (unsigned int)(*((unsigned char *)&comparestring[strpos])));
      }
      fprintf(stderr, "\n");
      if (comparestring2[0] == '\0') {
	fprintf(stderr, "comparestring2 (len=%d): ", comparestringlen);
	for (strpos = 0; strpos < comparestringlen; strpos++) {
	  fprintf(stderr, " %02x", (unsigned int)(*((unsigned char *)&comparestring2[strpos])));
	}
	fprintf(stderr, "\n");
      }
    }
#endif
    offile_off_t myinsize = myinp->tell() + myinp->avail();
    //fprintf(stderr, "myinp->tell() = %u\n", (unsigned int)myinp->tell());
    //fprintf(stderr, "myinp->avail() = %u\n", (unsigned int)myinp->avail());
    offile_off_t bufposstart = myinsize - pixelDataSize;
    buflen = 0;
    bufposend = bufposstart + buflen;
    long curpos;
    while (bufposstart > 0) {
	// bufposstart is the file position corresponding to the first
	// byte in the buffer.  We now want to start at a maximum of 512
	// bytes before that.
	curpos = 0;
	if (bufposstart > 512) {
	    readsize = 512;
	} else {
	    readsize = bufposstart;
	}
        //fprintf(stderr, "buf=0x%lx bufposstart=%ld bufposend=%ld buflen=%ld readsize=%ld\n", (long int)buf, bufposstart, bufposend, buflen, readsize);
	if (bufposstart != bufposend) {
	    memcpy(buf + readsize, buf, (buflen + readsize > 1024) ? (1024 - readsize) : (buflen - readsize));
	}
	offile_off_t myinpos = myinp->tell();
	bufposstart -= readsize;
	if (bufposstart > myinpos) {
	    //fprintf(stderr, "Skipping %u bytes to position %u\n", (unsigned int)(bufposstart - myinpos), (unsigned int)bufposstart);
	    if (myinp->skip(bufposstart - myinpos) < bufposstart - myinpos) {
		fprintf(stderr, "Error seeking in file to position %u!\n", (unsigned int)bufposstart);
		return -1;
	    }
	} else if (bufposstart < myinpos) {
	    //fprintf(stderr, "Putting back %u bytes to position %u\n", (unsigned int)(myinpos - bufposstart), (unsigned int)bufposstart);
	    myinp->putback(myinpos - bufposstart);
	    if (!myinp->good()) {
		fprintf(stderr, "Error seeking backwards in file to position %u: %s\n", (unsigned int)bufposstart, myinp->status().text());
		return -1;
	    }
	}
	//fprintf(stderr, "Reading %u bytes\n", (unsigned int)readsize);
	if (myinp->read(buf, readsize) != readsize) {
	    fprintf(stderr, "Error reading %u bytes at byte pos %u!\n", (unsigned int)readsize, (unsigned int)bufposstart);
	    if (!myinp->good()) {
		fprintf(stderr, "Error: %s\n", myinp->status().text());
	    }
	    goto FAIL;
	}
	if (buflen + readsize > 1024) {
	    buflen = 1024;
	} else {
	    buflen = buflen + readsize;
	}
	bufposend = bufposstart + buflen;
	curpos = readsize - 1;
	if (curpos + comparestringlen > buflen) {
	    curpos = buflen - comparestringlen;
	}
	for (/* curpos already set */; curpos >= 0; curpos--) {
	    if (memcmp(&buf[curpos], comparestring, comparestringlen) == 0) {
		retval = bufposstart + curpos + comparestringlen;
		goto FIRSTSUCCESS;
	    }
	    if (comparestring2[0] != '\0') {
		if (memcmp(&buf[curpos], comparestring2, comparestringlen) == 0) {
		    retval = bufposstart + curpos + comparestringlen;
		    goto FIRSTSUCCESS;
		}
	    }
	}
    }
    goto FIRSTSUCCESS;

FIRSTSUCCESS:
    // Now step through past pixel data offset to see if there is more
    // data, and if so, verify that they are valid DICOM elements
    curpos = retval + pixelDataSize;
    if (curpos < myinsize) {
	offile_off_t myinpos = myinp->tell();
	if (bxh_DICOM_read_from_stream(myinp, buf, 4, curpos - myinpos) != 0) {
	    goto FAIL;
	}
	curpos += 4;
	Uint16 group;
	Uint16 elem;
	if (littleendian) {
	    group = (buf[1] << 8) + buf[0];
	    elem = (buf[3] << 8) + buf[2];
	} else {
	    group = (buf[0] << 8) + buf[1];
	    elem = (buf[2] << 8) + buf[3];
	}
	if (group != 0xfffc || elem != 0xfffc) {
	    fprintf(stderr, "Error: any DICOM element after PixelData should be (fffc,fffc), but found (%04x,%04x)!!\n", group, elem);
	    goto FAIL;
	}
	Uint32 vl;
	int reservelen = 0;
	int vllen = 0;
	if (explicitsyntax) {
	    if (bxh_DICOM_read_from_stream(myinp, buf, 2, 0) != 0) {
		goto FAIL;
	    }
	    curpos += 2;
	    if (buf[0] != 'O' || buf[1] != 'B') {
		fprintf(stderr, "Warning: VR of DataSetTrailingPadding element (fffc,fffc) should be 'OB' but found '%c%c'!\n", buf[0], buf[1]);
	    }
	    if ((buf[0] == 'O' && (buf[1] == 'B' || buf[1] == 'W' || buf[1] == 'F')) ||
		(buf[0] == 'S' && buf[1] == 'Q') ||
		(buf[0] == 'U' && (buf[1] == 'T' || buf[1] == 'N'))) {
		reservelen = 2;
		vllen = 4;
	    } else {
		vllen = 2;
	    }
	} else {
	    vllen = 4;
	}
	if (bxh_DICOM_read_from_stream(myinp, buf, vllen, reservelen) != 0) {
	    goto FAIL;
	}
	curpos += reservelen + vllen;
	if (littleendian) {
	    vl = (buf[1] << 8) + buf[0];
	} else {
	    vl = (buf[0] << 8) + buf[1];
	}
	if (vllen == 4) {
	    if (littleendian) {
		vl += (buf[3] << 24) + (buf[2] << 16);
	    } else {
		vl = vl << 16;
		vl += (buf[2] << 8) + buf[1];
	    }
	}
	curpos += vl;
	if (curpos != myinsize) {
	    fprintf(stderr, "Warning: may have found garbage past DICOM dataset padding element!\n");
	}
    }
    goto SUCCESS;

FAIL:
    retval = -1;

SUCCESS:
    if (fp) fclose(fp);
    return retval;
}

struct dicomcont {
    int forceconcat;
    int sortopt;
    int isDataset;
    int numfinfos;
    int cornerorigin;
    std::vector<struct dicomfileinfo *> finfos;

    dicomcont()
	: forceconcat(0),
	  sortopt(0),
	  isDataset(0),
	  numfinfos(0),
	  cornerorigin(0),
	  finfos()
    { /* null */ }

    dicomcont(const dicomcont & cont)
	: forceconcat(cont.forceconcat),
	  sortopt(cont.sortopt),
	  isDataset(cont.isDataset),
	  numfinfos(0),
	  cornerorigin(cont.cornerorigin),
	  finfos()
    { /* null */ }

    ~dicomcont() {
	for (int i = 0; i < finfos.size(); i++) {
	    if (finfos[i] != NULL) {
		delete finfos[i];
		finfos[i] = NULL;
	    }
	}
    }
};

static
int
getPrivateDicomValue(dicomdatameta * ddm,
		     Uint16 group,
		     Uint16 elem,
		     const char * privCreator,
		     DcmStack & stack)
{
    DcmTag searchKey(group, elem);
    searchKey.setPrivateCreator(privCreator);
    searchKey.lookupVRinDictionary();
    stack.clear();
//    cerr << "Looking for tag " << searchKey << endl;
    if (ddm->dataset &&
	ddm->dataset->search(searchKey, stack, ESM_fromHere, OFFalse) == EC_Normal) {
//	cerr << "Found tag " << searchKey << endl;
	return 1;
    } else if (ddm->metainfo &&
	       ddm->metainfo->search(searchKey, stack, ESM_fromHere, OFFalse) == EC_Normal) {
//	cerr << "Found tag " << searchKey << endl;
	return 1;
    }
    return 0;
}

static
int
getDicomValue(dicomdatameta * ddm,
	      DcmTagKey & searchKey,
	      DcmStack & stack)
{
//    cerr << "Looking for tag " << searchKey << endl;
    stack.clear();
    if (ddm->dataset &&
	ddm->dataset->search(searchKey, stack, ESM_fromStackTop, OFFalse) == EC_Normal) {
//	cerr << "Found tag " << searchKey << endl;
	return 1;
    } else if (ddm->metainfo &&
	       ddm->metainfo->search(searchKey, stack, ESM_fromStackTop, OFFalse) == EC_Normal) {
//	cerr << "Found tag " << searchKey << endl;
	return 1;
    }
    return 0;
}

static
int
getDicomValue(dicomdatameta * ddm,
	      const char * tag,
	      DcmStack & stack)
{
    const DcmDataDictionary & globalDataDict = dcmDataDict.rdlock();
    const DcmDictEntry * dicent = globalDataDict.findEntry(tag);
    dcmDataDict.unlock();
    if (dicent == NULL)
	return 0;
    DcmTagKey searchKey(dicent->getGroup(), dicent->getElement());
    return getDicomValue(ddm, searchKey, stack);
}

#undef FUNC
#define FUNC "bxh_DICOM_freeCont"
/**
 * Free the continuation structure returned by bxh_DICOM_readFilesCont.
 * NOTE: do not call this on a continuation structure that you have
 * already sent to bxh_DICOM_readFilesCont.
 *
 * @param cont pointer to continuation structure as returned by bxh_DICOM_readFilesCont
 */
void
bxh_DICOM_freeCont(void * cont)
{
    dicomcont * dc = (dicomcont *)cont;
    delete dc;
    return;
}

#undef FUNC
#define FUNC "bxh_DICOM_readFiles"
/**
 * Extract metadata from DICOM slices and add as many of those slices
 * as we can into BXH document.
 *
 * @param docp BXH document pointer
 * @param numinfiles number of input DICOM files
 * @param filenames array of (char *) DICOM filenames
 * @param isDataset 0=file format, 1=plain dataset
 * @param searchforothers 1=search for other slices in same directory, 0=don't
 * @param forceconcat 1=concat files into volume even if key fields specifying orientation or series UID, etc., differ.  0=only use those slices that match with the first slice.
 * @param sortopt -1=no sort, 0=default (use various fields), 1=sort files by their filename rather than any other field
 * @param cornerorigin 0=no adjusting of origin, 1=assume x/y components of DICOM origin is corner of voxel, so move it to center of voxel
 * @return 0 on success, or non-zero on error.
 */
int
bxh_DICOM_readFiles(BXHDocPtr docp, int numinfiles, const char **filenames, int isDataset, int searchforothers, const char * basepath, int forceconcat, int sortopt, int printfields, int cornerorigin)
{
    int retval;
    void * cont = NULL;
    retval = bxh_DICOM_readFilesStart(docp, numinfiles, filenames, isDataset, searchforothers, basepath, forceconcat, sortopt, printfields, cornerorigin, &cont);
    if (cont) {
	bxh_DICOM_freeCont(cont);
	cont = NULL;
    }
    return retval;
}

#undef FUNC
#define FUNC "getCacheInitsEnhanced"
static
std::vector<dicomcacheinitenhanced> &
getCacheInitsEnhanced()
{
    static std::vector<dicomcacheinitenhanced> cache;
    static bool cacheValid = false;
    if (!cacheValid) {
	cache.clear();
	cache.push_back(dicomcacheinitenhanced(DCM_MRTimingAndRelatedParametersSequence, DCM_RepetitionTime, &dicomelemcache::repetitiontime));
	cache.push_back(dicomcacheinitenhanced(DCM_MRTimingAndRelatedParametersSequence, DCM_FlipAngle, &dicomelemcache::flipangle));
	cache.push_back(dicomcacheinitenhanced(DCM_MREchoSequence, DCM_EffectiveEchoTime, &dicomelemcache::echotime));
	cache.push_back(dicomcacheinitenhanced(DCM_MRFOVGeometrySequence, DCM_InPlanePhaseEncodingDirection, &dicomelemcache::inplanephaseencodingdirection));
	cache.push_back(dicomcacheinitenhanced(DCM_FrameContentSequence, DCM_TemporalPositionIndex, &dicomelemcache::temporalpositionindex));
	cache.push_back(dicomcacheinitenhanced(DCM_PlanePositionSequence, DCM_ImagePositionPatient, &dicomelemcache::imagepositionpatient));
	cache.push_back(dicomcacheinitenhanced(DCM_PlaneOrientationSequence, DCM_ImageOrientationPatient, &dicomelemcache::imageorientationpatient));
	cache.push_back(dicomcacheinitenhanced(DCM_PixelMeasuresSequence, DCM_SliceThickness, &dicomelemcache::slicethickness));
	cache.push_back(dicomcacheinitenhanced(DCM_PixelMeasuresSequence, DCM_PixelSpacing, &dicomelemcache::pixelspacing));
	cache.push_back(dicomcacheinitenhanced(DCM_FrameContentSequence, DCM_DimensionIndexValues, &dicomelemcache::dimensionindexvalues));
	cache.push_back(dicomcacheinitenhanced(DCM_FrameContentSequence, DCM_StackID, &dicomelemcache::stackid));
	cache.push_back(dicomcacheinitenhanced(DCM_FrameContentSequence, DCM_InStackPositionNumber, &dicomelemcache::instackpositionnumber));
	cache.push_back(dicomcacheinitenhanced(DCM_MRDiffusionSequence, DCM_DiffusionBValue, &dicomelemcache::diffusionbvalue));
	std::vector<DcmTag> diffseqs;
	diffseqs.push_back(DCM_MRDiffusionSequence);
	diffseqs.push_back(DCM_DiffusionGradientDirectionSequence);
	cache.push_back(dicomcacheinitenhanced(diffseqs, DCM_DiffusionGradientOrientation, &dicomelemcache::diffusiongradientorientation));

	/* bubble-sort the cacheinitenhanced entries */
	int cachesize = cache.size();
	for (int i = 0; i < cachesize; i++) {
	    for (int j = i + 1; j < cachesize; j++) {
		dicomcacheinitenhanced & ei = cache[i];
		dicomcacheinitenhanced & ej = cache[j];
		int doswap = 0;
		int donoswap = 0;
		int nti = ei.sequencetags.size();
		int ntj = ej.sequencetags.size();
		int tind = 0;
		for (tind = 0; tind < nti && tind < ntj; tind++) {
		    DcmTag & tagi = ei.sequencetags[tind];
		    DcmTag & tagj = ej.sequencetags[tind];
		    if (tagi > tagj) {
			doswap = 1;
			break;
		    } else if (tagi < tagj) {
			donoswap = 1;
			break;
		    }
		}
		if (donoswap) {
		    continue;
		}
		if (!doswap) {
		    DcmTag * tagpi = NULL;
		    DcmTag * tagpj = NULL;
		    if (tind == nti) {
			tagpi = &ei.elemtag;
		    } else {
			tagpi = &ei.sequencetags[tind];
		    }
		    if (tind == ntj) {
			tagpj = &ej.elemtag;
		    } else {
			tagpj = &ej.sequencetags[tind];
		    }
		    if (*tagpi  > *tagpj) {
			doswap = 1;
		    } else if (*tagpi < *tagpj) {
			continue;
		    } else if (nti > ntj) {
			doswap = 1;
		    }
		}
		if (doswap) {
#ifdef DEBUG
		    fprintf(stderr, "swapping entry %d [ ", i);
		    for (int si = 0; si < cache[i].sequencetags.size(); si++) {
			fprintf(stderr, "(%04x,%04x) => ", (unsigned int)cache[i].sequencetags[si].getGTag(), (unsigned int)cache[i].sequencetags[si].getETag());
		    }
		    fprintf(stderr, "(%04x,%04x)", (unsigned int)cache[i].elemtag.getGTag(), (unsigned int)cache[i].elemtag.getETag());
		    fprintf(stderr, " ]\n    with entry %d [ ", j);
		    for (int si = 0; si < cache[j].sequencetags.size(); si++) {
			fprintf(stderr, "(%04x,%04x) => ", (unsigned int)cache[j].sequencetags[si].getGTag(), (unsigned int)cache[j].sequencetags[si].getETag());
		    }
		    fprintf(stderr, "(%04x,%04x) ]\n", (unsigned int)cache[j].elemtag.getGTag(), (unsigned int)cache[j].elemtag.getETag());
#endif
		    dicomcacheinitenhanced swap = cache[i];
		    cache[i] = cache[j];
		    cache[j] = swap;
		}
	    }
	}

	cacheValid = true;
    }
    return cache;
}


#undef FUNC
#define FUNC "bxh_DICOM_readFilesStart"
/**
 * Extract metadata from DICOM slices and add all slices that match with
 * the first slice into BXH document (and return a continuation structure
 * which can be sent to bxh_DICOM_readFilesCont() to do the rest of the
 * slices).
 *
 * @param docp BXH document pointer
 * @param numinfiles number of input DICOM files
 * @param filenames array of (char *) DICOM filenames
 * @param isDataset 0=file format, 1=plain dataset
 * @param searchforothers 1=search for other matching slices in same directory, 0=don't.  numinfiles must be 1.
 * @param basepath location where BXH file will be written (for determining relative pathnames), or directory in which BXH file will be written (must end in path separator [e.g. '/' on UNIX])
 * @param forceconcat 1=concat files into volume even if key fields specifying orientation or series UID, etc., differ.  0=only use those slices that match with the first slice.
 * @param sortopt -1=no sort, 0=default (sort by various fields), 1=sort files by their filename rather than any other field
 * @param cornerorigin 0=no adjusting of origin, 1=assume x/y components of DICOM origin is corner of voxel, so move it to center of voxel
 * @param contp On return, *#contp will have a pointer to a continuation if there are more slices to be processed, otherwise NULL.  Any value *#contp has on entry will be overwritten.
 * @return 0 on success, or non-zero on error.
 */
int
bxh_DICOM_readFilesStart(BXHDocPtr docp, int numinfiles, const char **filenames, int isDataset, int searchforothers, const char * basepath, int forceconcat, int sortopt, int printfields, int cornerorigin, void ** contp)
{
    int retval = 0;
    
    int filenum = 0;

    Uint32 streambufsize = 32768;
    void * streambuf = malloc(streambufsize);
    DcmFileFormat * dfileobj = NULL;
    dicomdatameta ddm;
    dicomdatameta * dfile = &ddm;
    OFString tmpos;
    union {
	Float32 f32;
	Float64 f64;
	Sint16 i16;
	Sint32 i32;
	Uint16 ui16;
	Uint32 ui32;
    } val;

    char ** filenamelist = (char **)filenames;
    char ** newfilenames = NULL;

    dicomcont * cont = NULL;
    int numfinfos = 0;

    struct blistelem {
	double bvalue;
	double scalex;
	double scaley;
	double scalez;
        Sint32 philipsmysteryfield;
    };
    blistelem * blist = NULL;
    int blistsize = 0;

    /* list elements we will want to search for and cache for each
     * input file */
    dicomelemcache elemcache;
    /* here is the table used to populate the basic cache */
    static dicomcacheinit * cacheinits = NULL;
    static dicomcacheinit cacheinitsraw[] = {
	{ &dicomelemcache::acquisitioncontrast, "AcquisitionContrast", NULL, 0, 0 },
	{ &dicomelemcache::acquisitiondate, "AcquisitionDate", NULL, 0, 0 },
	{ &dicomelemcache::acquisitionnumber, "AcquisitionNumber", NULL, 0, 0 },
	{ &dicomelemcache::acquisitiontime, "AcquisitionTime", NULL, 0, 0 },
	{ &dicomelemcache::samplesperpixel, "SamplesPerPixel", NULL, 0, 0 },
	{ &dicomelemcache::bitsallocated, "BitsAllocated", NULL, 0, 0 },
	{ &dicomelemcache::columns, "Columns", NULL, 0, 0 },
	{ &dicomelemcache::contentdate, "ContentDate", NULL, 0, 0 },
	{ &dicomelemcache::contenttime, "ContentTime", NULL, 0, 0 },
	{ &dicomelemcache::diffusionbvalue, "DiffusionBValue", NULL, 0, 0 },
	{ &dicomelemcache::diffusiongradientorientation, "DiffusionGradientOrientation", NULL, 0, 0 },
	{ &dicomelemcache::echotime, "EchoTime", NULL, 0, 0 },
	{ &dicomelemcache::echonumber, "EchoNumbers", NULL, 0, 0 },
	{ &dicomelemcache::flipangle, "FlipAngle", NULL, 0, 0 },
	{ &dicomelemcache::flipangle, "FlipAngle", NULL, 0, 0 },
	{ &dicomelemcache::inplanephaseencodingdirection, "InPlanePhaseEncodingDirection", NULL, 0, 0 },
	{ &dicomelemcache::imageorientationpatient, "ImageOrientationPatient", NULL, 0, 0 },
	{ &dicomelemcache::imagepositionpatient, "ImagePositionPatient", NULL, 0, 0 },
	{ &dicomelemcache::imagetype, "ImageType", NULL, 0, 0 },
	{ &dicomelemcache::instancenumber, "InstanceNumber", NULL, 0, 0 },
	{ &dicomelemcache::manufacturer, "Manufacturer", NULL, 0, 0 },
	{ &dicomelemcache::sopclassuid, "SOPClassUID", NULL, 0, 0 },
	{ &dicomelemcache::modality, "Modality", NULL, 0, 0 },
	{ &dicomelemcache::numberofframes, "NumberOfFrames", NULL, 0, 0 },
	{ &dicomelemcache::pixelspacing, "PixelSpacing", NULL, 0, 0 },
	{ &dicomelemcache::repetitiontime, "RepetitionTime", NULL, 0, 0 },
	{ &dicomelemcache::rows, "Rows", NULL, 0, 0 },
	{ &dicomelemcache::seriesinstanceuid, "SeriesInstanceUID", NULL, 0, 0 },
	{ &dicomelemcache::slicethickness, "SliceThickness", NULL, 0, 0 },
	{ &dicomelemcache::spacingbetweenslices, "SpacingBetweenSlices", NULL, 0, 0 },
	{ &dicomelemcache::studyinstanceuid, "StudyInstanceUID", NULL, 0, 0 },
	{ &dicomelemcache::temporalpositionidentifier, "TemporalPositionIdentifier", NULL, 0, 0 },
	{ &dicomelemcache::temporalpositionindex, "TemporalPositionIndex", NULL, 0, 0 },
	{ &dicomelemcache::temporalresolution, "TemporalResolution", NULL, 0, 0 },
	{ &dicomelemcache::sharedfunctionalgroupssequence, "SharedFunctionalGroupsSequence", NULL, 0,0 },
	{ &dicomelemcache::perframefunctionalgroupssequence, "PerFrameFunctionalGroupsSequence", NULL, 0,0 },
	{ &dicomelemcache::dimensionindexsequence, "DimensionIndexSequence", NULL, 0,0 },
	{ &dicomelemcache::pixeldata, "PixelData", NULL, 0, 0 },
	{ &dicomelemcache::gerawdatatype, NULL, "GEMS_PARM_01", 0x0043, 0x102f },
	{ &dicomelemcache::geautoprescancenterfrequency, NULL, "GEMS_ACQU_01", 0x0019, 0x1093 },
	{ &dicomelemcache::geautoprescantransmitgain, NULL, "GEMS_ACQU_01", 0x0019, 0x1094 },
	{ &dicomelemcache::gepulsesequencename, NULL, "GEMS_ACQU_01", 0x0019, 0x109c },
	{ &dicomelemcache::geslopinteger6, NULL, "GEMS_PARM_01", 0x0043, 0x1039 },
	{ &dicomelemcache::bvaluebiasfactor, NULL, "GEMS_PARM_01", 0x0043, 0x107f },
	{ &dicomelemcache::geuserdata20, NULL, "GEMS_ACQU_01", 0x0019, 0x10bb },
	{ &dicomelemcache::geuserdata21, NULL, "GEMS_ACQU_01", 0x0019, 0x10bc },
	{ &dicomelemcache::geuserdata22, NULL, "GEMS_ACQU_01", 0x0019, 0x10bd },
	{ &dicomelemcache::geswapphasefrequency, NULL, "GEMS_ACQU_01", 0x0019, 0x108f },
	{ &dicomelemcache::siemensbvalue, NULL, "SIEMENS MR HEADER", 0x0019, 0x100c },
	{ &dicomelemcache::siemensdiffusionvector, NULL, "SIEMENS MR HEADER", 0x0019, 0x100e },
	{ &dicomelemcache::siemensCSAImageHeaderInfo, NULL, "SIEMENS CSA HEADER", 0x0029, 0x1010 },
	{ &dicomelemcache::philipsslicenumbermr, NULL, "Philips Imaging DD 001", 0x2001, 0x100a },
	{ &dicomelemcache::philips_2005_1413, NULL, "Philips Imaging DD 001", 0x2005, 0x1413 },
    };
    /* this table includes stuff from Enhanced MR Macros */
    std::vector<dicomcacheinitenhanced> & cacheinitsenhanced = getCacheInitsEnhanced();

    int numcacheinits = sizeof(cacheinitsraw)/sizeof(cacheinits[0]);
    int numcacheinitsenhanced = cacheinitsenhanced.size();
#if 0
    fprintf(stderr, "raw cache init entries:\n");
    for (int i = 0; i < numcacheinits; i++) {
	dicomcacheinit *cip = &cacheinitsraw[i];
	fprintf(stderr, " cip: (");
	if (cip->tag.getPrivateCreator()) {
	    fprintf(stderr, "%s,", cip->tag.getPrivateCreator());
	}
	fprintf(stderr, "%04x,%04x)\n", (unsigned int)cip->tag.getGTag(), (unsigned int)cip->tag.getETag());
    }
    fprintf(stderr, "raw cache init enhanced entries:\n");
    for (int i = 0; i < numcacheinitsenhanced; i++) {
	dicomcacheinitenhanced & cie = cacheinitsenhanced[i];
	fprintf(stderr, " cie: (%04x,%04x) => (%04x,%04x)\n", (unsigned int)cie.sequencetag.getGTag(), (unsigned int)cie.sequencetag.getETag(), (unsigned int)cie.elemtag.getGTag(), (unsigned int)cie.elemtag.getETag());
    }
#endif

    /* add private dictionary entries for DICOM files that come with
     * implicit transfer syntax */
    {
	DcmDataDictionary & globalDataDict = dcmDataDict.wrlock();
	globalDataDict.addEntry(new DcmDictEntry(0x0019, 0x000a, EVR_US, "SiemensNumberSlices", 1, 1, NULL, OFFalse, "SIEMENS MR HEADER"));
	globalDataDict.addEntry(new DcmDictEntry(0x0019, 0x000c, EVR_IS, "SiemensBValue", 1, 1, NULL, OFFalse, "SIEMENS MR HEADER"));
	globalDataDict.addEntry(new DcmDictEntry(0x0019, 0x000e, EVR_FD, "SiemensDiffusionGradientDirection", 3, 3, NULL, OFFalse, "SIEMENS MR HEADER"));
	globalDataDict.addEntry(new DcmDictEntry(0x0043, 0x002f, EVR_SS, "GERawDataType", 1, 1, NULL, OFFalse, "GEMS_PARM_01"));
	globalDataDict.addEntry(new DcmDictEntry(0x0019, 0x0093, EVR_LO, "GEAutoPrescanCenterFrequency", 1, 1, NULL, OFFalse, "GEMS_ACQU_01"));
	globalDataDict.addEntry(new DcmDictEntry(0x0019, 0x0094, EVR_LO, "GEAutoPrescanTransmitGain", 1, 1, NULL, OFFalse, "GEMS_ACQU_01"));
	globalDataDict.addEntry(new DcmDictEntry(0x0019, 0x009c, EVR_LO, "GEPulseSequenceName", 1, 1, NULL, OFFalse, "GEMS_ACQU_01"));
	globalDataDict.addEntry(new DcmDictEntry(0x0043, 0x0039, EVR_IS, "GESlopInteger6", 1, 1, NULL, OFFalse, "GEMS_PARM_01"));
	globalDataDict.addEntry(new DcmDictEntry(0x0043, 0x007f, EVR_IS, "GEBValueBiasFactor", 1, 1, NULL, OFFalse, "GEMS_PARM_01"));
	globalDataDict.addEntry(new DcmDictEntry(0x0019, 0x00bb, EVR_DS, "GEUserData20", 1, 1, NULL, OFFalse, "GEMS_ACQU_01"));
	globalDataDict.addEntry(new DcmDictEntry(0x0019, 0x00bc, EVR_DS, "GEUserData21", 1, 1, NULL, OFFalse, "GEMS_ACQU_01"));
	globalDataDict.addEntry(new DcmDictEntry(0x0019, 0x00bd, EVR_DS, "GEUserData22", 1, 1, NULL, OFFalse, "GEMS_ACQU_01"));
	globalDataDict.addEntry(new DcmDictEntry(0x0019, 0x008f, EVR_SS, "GESwapPhaseFrequency", 1, 1, NULL, OFFalse, "GEMS_ACQU_01"));
	globalDataDict.addEntry(new DcmDictEntry(0x2001, 0x000a, EVR_IS, "SliceNumberMR", 1, 1, NULL, OFFalse, "Philips Imaging DD 001"));
	dcmDataDict.unlock();
    }

    if (cacheinits == NULL) {
	/* fill in the group,elem numbers for those cache init entries that
	 * don't have them
	 */
	for (int i = 0; i < numcacheinits; i++) {
	    dicomcacheinit * cip = &cacheinitsraw[i];
	    if (cip->group == 0 && cip->elem == 0) {
		const DcmDataDictionary & globalDataDict = dcmDataDict.rdlock();
		const DcmDictEntry * dicent = globalDataDict.findEntry(cip->tagname);
		dcmDataDict.unlock();
		if (dicent) {
		    cip->group = dicent->getGroup();
		    cip->elem = dicent->getElement();
		    cip->tag = DcmTag(cip->group, cip->elem);
		    if (cip->privCreator) {
			cip->tag.setPrivateCreator(cip->privCreator);
			cip->tag.lookupVRinDictionary();
		    }
		}
	    } else {
		cip->tag = DcmTag(cip->group, cip->elem);
	    }
	}
#if 0
	fprintf(stderr, "unsorted cache init entries:\n");
	for (int i = 0; i < numcacheinits; i++) {
	    dicomcacheinit *cip = &cacheinitsraw[i];
	    fprintf(stderr, " cip: (");
	    if (cip->tag.getPrivateCreator()) {
		fprintf(stderr, "%s,", cip->tag.getPrivateCreator());
	    }
	    fprintf(stderr, "%04x,%04x)\n", (unsigned int)cip->tag.getGTag(), (unsigned int)cip->tag.getETag());
	}
#endif
	/* bubble-sort the cacheinit entries */
	for (int i = 0; i < numcacheinits; i++) {
	    for (int j = i + 1; j < numcacheinits; j++) {
		int doswap = 0;
		if (cacheinitsraw[i].tag > cacheinitsraw[j].tag) {
		    doswap = 1;
		}
		if (doswap) {
		    dicomcacheinit swap = cacheinitsraw[i];
		    cacheinitsraw[i] = cacheinitsraw[j];
		    cacheinitsraw[j] = swap;
		}
	    }
	}
	cacheinits = cacheinitsraw;
#if DEBUG
	fprintf(stderr, "sorted cache init entries:\n");
	for (int i = 0; i < numcacheinits; i++) {
	    dicomcacheinit *cip = &cacheinits[i];
	    fprintf(stderr, " cip: (");
	    if (cip->tag.getPrivateCreator()) {
		fprintf(stderr, "%s,", cip->tag.getPrivateCreator());
	    }
	    fprintf(stderr, "%04x,%04x)\n", (unsigned int)cip->tag.getGTag(), (unsigned int)cip->tag.getETag());
	}
	fprintf(stderr, "sorted cache init enhanced entries:\n");
	for (int i = 0; i < numcacheinitsenhanced; i++) {
	    dicomcacheinitenhanced & cie = cacheinitsenhanced[i];
	    fprintf(stderr, " cie: ");
	    for (int si = 0; si < cie.sequencetags.size(); si++) {
		fprintf(stderr, "(%04x,%04x) => ", (unsigned int)cie.sequencetags[si].getGTag(), (unsigned int)cie.sequencetags[si].getETag());
	    }
	    fprintf(stderr, "(%04x,%04x)\n", (unsigned int)cie.elemtag.getGTag(), (unsigned int)cie.elemtag.getETag());
	}
#endif
    }


    if (searchforothers && numinfiles != 1) {
        fprintf(stderr, FUNC ": error: searchforothers only allows one input file!\n");
	goto FAIL;
    }

    if (searchforothers) {
	/* check all files in the same directory */
	int newnuminfiles = 0;
#ifdef WIN32
	WIN32_FIND_DATA wfd;
	HANDLE hdl;
	char * pattern = NULL;
	char * curdirname = NULL;
	char * curfilename = NULL;

	pattern = (char *)malloc(sizeof(char)*strlen(filenamelist[0]) + 2);
	strcpy(pattern, filenamelist[0]);
	char * slashpos = strrchr(pattern, '\\');
	if (slashpos == NULL) {
	    free(pattern);
	    pattern = strdup("*");
	    curdirname = strdup(".");
	    curfilename = strdup(filenamelist[0]);
	} else {
	    curdirname = strdup(filenamelist[0]);
	    curdirname[slashpos - pattern] = '\0';
	    curfilename = strdup(slashpos+1);
	    strcpy(slashpos, "\\*");
	}
	//fprintf(stderr, "FindFirstFile(%s)\n", pattern);
	hdl = FindFirstFile(pattern, &wfd);
	int ret = (hdl != INVALID_HANDLE_VALUE);
	//fprintf(stderr, "wfd.cFileName == '%s'\n", wfd.cFileName);
	while (ret) {
	    if (!(wfd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
		char * buf = (char *)malloc(sizeof(char)*(strlen(curdirname) + 1 + strlen(wfd.cFileName) + 1));
		if (strcmp(curdirname, ".") == 0)
		    sprintf(&buf[0], "%s", wfd.cFileName);
		else
		    sprintf(&buf[0], "%s\\%s", curdirname, wfd.cFileName);
		newfilenames = (char **)realloc(newfilenames, sizeof(char *) * (newnuminfiles + 1));
		newfilenames[newnuminfiles] = strdup(&buf[0]);
		newnuminfiles++;
		if (strcmp(wfd.cFileName, curfilename) == 0 &&
		    newnuminfiles != 1) {
		    /* put given input file first in the list */
		    char * swap = newfilenames[0];
		    newfilenames[0] = newfilenames[newnuminfiles-1];
		    newfilenames[newnuminfiles-1] = swap;
		}
		free(buf);
	    }
	    ret = FindNextFile(hdl, &wfd);
	}
	if (hdl != INVALID_HANDLE_VALUE) {
	    FindClose(hdl);
	}
	filenamelist = newfilenames;
	numinfiles = newnuminfiles;
	free(pattern); pattern = NULL;
	free(curdirname); curdirname = NULL;
	free(curfilename); curfilename = NULL;
#else		
	DIR * dir;
	struct dirent * dirent;
	char * curdirname = NULL;
	char * curfilename = NULL;

	curdirname = strdup(filenamelist[0]);
	char * slashpos = strrchr(curdirname, '/');
	if (slashpos == NULL) {
	    free(curdirname);
	    curdirname = strdup(".");
	    curfilename = strdup(filenamelist[0]);
	} else {
	    curfilename = strdup(slashpos+1);
	    *slashpos = '\0';
	}
	dir = opendir(curdirname);
	if (dir == NULL) {
	    fprintf(stderr, FUNC ": error opening directory ('%s')\n", curdirname);
	    goto FAIL;
	}
	while ((dirent = readdir(dir))) {
	    char * buf = NULL;
	    if (strcmp(dirent->d_name, ".") == 0 ||
		strcmp(dirent->d_name, "..") == 0) {
		continue;
	    }
	    buf = (char *)malloc(sizeof(char) * (strlen(curdirname) + 1 + strlen(dirent->d_name) + 1));
	    if (strcmp(curdirname, ".") == 0)
		sprintf(&buf[0], "%s", dirent->d_name);
	    else
		sprintf(&buf[0], "%s/%s", curdirname, dirent->d_name);
	    newfilenames = (char **)realloc(newfilenames, sizeof(char *) * (newnuminfiles + 1));
	    newfilenames[newnuminfiles] = strdup(&buf[0]);
	    newnuminfiles++;
	    if (strcmp(dirent->d_name, curfilename) == 0 &&
		newnuminfiles != 1) {
		/* put given input file first in the list */
		char * swap = newfilenames[0];
		newfilenames[0] = newfilenames[newnuminfiles-1];
		newfilenames[newnuminfiles-1] = swap;
	    }
	    free(buf);
	}
	closedir(dir);
	filenamelist = newfilenames;
	numinfiles = newnuminfiles;
	free(curdirname); curdirname = NULL;
	free(curfilename); curfilename = NULL;
#endif
    }

    numfinfos = numinfiles;
    cont = new dicomcont();
    *contp = cont;
    cont->forceconcat = forceconcat;
    cont->sortopt = sortopt;
    cont->isDataset = isDataset;
    cont->numfinfos = 0;
    cont->finfos.resize(0);
    cont->cornerorigin = cornerorigin;
    for (filenum = 0; filenum < numinfiles; filenum++) {
	dicomfileinfo * finfop = NULL;
	DcmElement * elem = NULL;
	Sint32 numframes = 0;
	dicomelemcache * elemcaches = NULL;
	long pixeldataoffset = -1; /* will be calculated once per file */
	long filesize = -1;
	boost::shared_ptr<OFString> dimsequenceidentifier(new OFString());
	std::vector<enhanceddimindex> dimsequence;
	boost::shared_ptr<OFString> last_unrecognized_indices((OFString *)NULL);
	OFCondition errcode;

#if 0
	fprintf(stderr, "%s\r", filenamelist[filenum]);
#endif
	//DcmInputFileStream myin(filenamelist[filenum]);
	DcmInputFileBufferedStream myin(filenamelist[filenum], 0, streambufsize, streambuf);
	if ( myin.status() != EC_Normal ) {
	    fprintf(stderr, FUNC ": cannot open file: %s\n", filenamelist[filenum]);
	    continue;
	}
	filesize = myin.tell() + myin.avail();

	DcmObject * reader = NULL;
	if (isDataset) {
	    DcmDataset * dataset = new DcmDataset();
	    reader = dataset;
	    dfileobj = new DcmFileFormat(dataset);
	} else {
	    reader = dfileobj = new DcmFileFormat();
	}

	reader->transferInit();
	reader->read(myin, EXS_Unknown, EGL_noChange);
	reader->transferEnd();

	if (dfileobj->error() != EC_Normal)
	{
	    fprintf(stderr, "Error reading file '%s' (may not be DICOM); skipping.\n", filenamelist[filenum]);
	    delete dfileobj;
	    continue;
	}

	//dfileobj->loadAllDataIntoMemory();
	dfile->dataset = ((DcmFileFormat *)dfileobj)->getDataset();
	dfile->metainfo = ((DcmFileFormat *)dfileobj)->getMetaInfo();

	/* cacheinit array is sorted by tag, and so should DICOM dataset, so go
	 * through the list and try to populate element cache in linear time */
	memset(&elemcache, '\0', sizeof(elemcache));
	unsigned int cacheind = 0;
	dicomcacheinit * cip = &cacheinits[cacheind];
	DcmElement * curelem = NULL;
	if (dfile->metainfo) {
	    while (cacheind < numcacheinits &&
		   (curelem = (DcmElement *)dfile->metainfo->nextInContainer(curelem)) != NULL) {
#if 0
		fprintf(stderr, "cip: (");
		if (cip->tag.getPrivateCreator()) {
		    fprintf(stderr, "%s,", cip->tag.getPrivateCreator());
		}
		fprintf(stderr, "%04x,%04x)", (unsigned int)cip->tag.getGTag(), (unsigned int)cip->tag.getETag());
		fprintf(stderr, "  curelem: (");
		if (curelem->getTag().getPrivateCreator()) {
		    fprintf(stderr, "%s,", curelem->getTag().getPrivateCreator());
		}
		fprintf(stderr, "%04x,%04x)", (unsigned int)curelem->getTag().getGTag(), (unsigned int)curelem->getTag().getETag());
		fprintf(stderr, "\n");
#endif
		while (cacheind < numcacheinits &&
		       cip->tag < curelem->getTag()) {
#if 0
		    fprintf(stderr, " skip cip: (");
		    if (cip->tag.getPrivateCreator()) {
			fprintf(stderr, "%s,", cip->tag.getPrivateCreator());
		    }
		    fprintf(stderr, "%04x,%04x)", (unsigned int)cip->tag.getGTag(), (unsigned int)cip->tag.getETag());
		    fprintf(stderr, "  curelem: (");
		    if (curelem->getTag().getPrivateCreator()) {
			fprintf(stderr, "%s,", curelem->getTag().getPrivateCreator());
		    }
		    fprintf(stderr, "%04x,%04x)", (unsigned int)curelem->getTag().getGTag(), (unsigned int)curelem->getTag().getETag());
		    fprintf(stderr, "\n");
#endif
		    cacheind++;
		    cip++;
		}
		if (cip->tag == curelem->getTag()) {
#if 0
		    fprintf(stderr, " metainfo matched! cacheind=%u (", cacheind);
		    if (cip->tag.getPrivateCreator()) {
			fprintf(stderr, "%s,", cip->tag.getPrivateCreator());
		    }
		    fprintf(stderr, "%04x,%04x)\n", (unsigned int)cip->tag.getGTag(), (unsigned int)cip->tag.getETag());
#endif
		    elemcache.*(cip->memberp) = curelem;
		    cacheind++;
		    cip++;
		}
	    }
	}
	while (cacheind < numcacheinits &&
	       (curelem = (DcmElement *)dfile->dataset->nextInContainer(curelem)) != NULL) {
#if 0
	    fprintf(stderr, "cip: (");
	    if (cip->tag.getPrivateCreator()) {
		fprintf(stderr, "%s,", cip->tag.getPrivateCreator());
	    }
	    fprintf(stderr, "%04x,%04x)", (unsigned int)cip->tag.getGTag(), (unsigned int)cip->tag.getETag());
	    fprintf(stderr, "  curelem: (");
	    if (curelem->getTag().getPrivateCreator()) {
		fprintf(stderr, "%s,", curelem->getTag().getPrivateCreator());
	    }
	    fprintf(stderr, "%04x,%04x)", (unsigned int)curelem->getTag().getGTag(), (unsigned int)curelem->getTag().getETag());
	    fprintf(stderr, "\n");
#endif
	    while (cacheind < numcacheinits &&
		   cip->tag < curelem->getTag()) {
#if 0
		fprintf(stderr, " skip cip: (");
		if (cip->tag.getPrivateCreator()) {
		    fprintf(stderr, "%s,", cip->tag.getPrivateCreator());
		}
		fprintf(stderr, "%04x,%04x)", (unsigned int)cip->tag.getGTag(), (unsigned int)cip->tag.getETag());
		fprintf(stderr, "  curelem: (");
		if (curelem->getTag().getPrivateCreator()) {
		    fprintf(stderr, "%s,", curelem->getTag().getPrivateCreator());
		}
		fprintf(stderr, "%04x,%04x)", (unsigned int)curelem->getTag().getGTag(), (unsigned int)curelem->getTag().getETag());
		fprintf(stderr, "\n");
#endif
		cacheind++;
		cip++;
	    }
	    if (cip->tag == curelem->getTag()) {
#if 0
		fprintf(stderr, " dataset matched! cacheind=%u (", cacheind);
		if (cip->tag.getPrivateCreator()) {
		    fprintf(stderr, "%s,", cip->tag.getPrivateCreator());
		}
		fprintf(stderr, "%04x,%04x)\n", (unsigned int)cip->tag.getGTag(), (unsigned int)cip->tag.getETag());
#endif
		elemcache.*(cip->memberp) = curelem;
		cacheind++;
		cip++;
	    }
	}

	if (!elemcache.pixeldata) {
	  fprintf(stderr, "Did not find Pixel Data element in DICOM file '%s'!\n", filenamelist[filenum]);
	  goto PARSEFAIL;
	}

	numframes = 1;
	elemcaches = (dicomelemcache *)malloc(sizeof(dicomelemcache) * numframes);
	elemcaches[0] = elemcache;

	/* do Enhanced MR image processing */
	{
	    if ((elem = elemcache.sopclassuid) == NULL) {
		fprintf(stderr, FUNC ": error: can't get SOPClassUID\n");
		goto PARSEFAIL;
	    }
	    const char * cs;
	    tmpos.clear();
	    elem->getOFString(tmpos, 0);
	    cs = tmpos.c_str();
	    if (strcmp(cs, UID_EnhancedMRImageStorage) == 0) {
		/* parse dimension index sequence for later use */
		if ((elem = elemcache.dimensionindexsequence) != NULL) {
		    int dimnum = -1;
		    DcmObject * curitem = NULL;
		    while ((curitem = elemcache.dimensionindexsequence->nextInContainer(curitem)) != NULL) {
			dimnum++;
			OFString dimorguid;
			DcmTagKey dimindexptr;
			DcmTagKey funcgroupptr;
			DcmObject * curfield = NULL;
			while ((curfield = curitem->nextInContainer(curfield)) != NULL) {
			    if (curfield->getTag() == DCM_DimensionOrganizationUID) {
				char * tmpstr = NULL;
				((DcmElement *)curfield)->getOFString(dimorguid, 0);
			    } else if (curfield->getTag() == DCM_DimensionIndexPointer) {
				((DcmElement *)curfield)->getTagVal(dimindexptr);
			    } else if (curfield->getTag() == DCM_FunctionalGroupPointer) {
				((DcmElement *)curfield)->getTagVal(funcgroupptr);
			    }
			}
			if (dimorguid.length() == 0) {
			    fprintf(stderr, "WARNING: Didn't find DimensionOrganizationUID in DimensionIndexSequence!\n");
			    break;
			}
			(*dimsequenceidentifier) += '|';
			(*dimsequenceidentifier) += dimorguid;
			(*dimsequenceidentifier) += ':';
			(*dimsequenceidentifier) += funcgroupptr.toString();
			(*dimsequenceidentifier) += ':';
			(*dimsequenceidentifier) += dimindexptr.toString();
			dimsequence.push_back(enhanceddimindex(funcgroupptr, dimindexptr));
		    }
		}

		/* create a list of elemcaches, one for each frame */
		if ((elem = elemcache.numberofframes) == NULL) {
		    fprintf(stderr, FUNC ": error: file is of class Enhanced MR Image Storage, but can't get field Number of Frames\n");
		    goto PARSEFAIL;
		}
		elem->getSint32(numframes);
		elemcaches = (dicomelemcache *)realloc(elemcaches, sizeof(dicomelemcache) * numframes);
		DcmSequenceOfItems * sharedfgs = (DcmSequenceOfItems *)elemcache.sharedfunctionalgroupssequence;
		DcmSequenceOfItems * perframefgs = (DcmSequenceOfItems *)elemcache.perframefunctionalgroupssequence;
		/* first elemcache will be the template, and so put all
		 * shared functional group values in there to start.
		 * framenum==-1 means do shared functional group,
		 * otherwise framenum identifies which per-frame functional
		 * group to use. */
		for (int framenum = -1; framenum < numframes; framenum++) {
		    dicomelemcache * curcachep;
		    DcmElement * fgselem = NULL;
		    if (framenum == -1) {
#if DEBUG
			fprintf(stderr, "SHARED FUNCTIONAL GROUPS\n");
#endif
			curcachep = &elemcaches[0];
			fgselem = (DcmElement *)sharedfgs->getItem(0);
		    } else {
#if DEBUG
			fprintf(stderr, "PER-FRAME FUNCTIONAL GROUPS\n");
#endif
			if (framenum > 0) {
			    memcpy(&elemcaches[framenum], &elemcaches[0], sizeof(elemcaches[0]));
			}
			curcachep = &elemcaches[framenum];
			if (perframefgs != NULL) {
			    fgselem = (DcmElement *)perframefgs->getItem(framenum);
			}
		    }
		    if (fgselem == NULL) {
			continue;
		    }
		    /* NOTE: make sure we go in sorted tag order for
		     * performance (we use stack and ESM_afterStackTop to make
		     * sure we don't need to go backwards for anything) */
		    DcmStack stack;
		    std::vector<DcmTag> emptyseq;
		    std::vector<DcmTag> & lastseqtags = emptyseq;
		    std::vector<DcmElement *> mystack;
		    mystack.push_back(fgselem); // note: mystack always has functional groups elems at bottom of stack
		    DcmElement * fgseq = NULL;
		    for (int i = 0; i < numcacheinitsenhanced; i++) {
			dicomcacheinitenhanced & cie = cacheinitsenhanced[i];
			int nst = cie.sequencetags.size();
#if DEBUG
			fprintf(stderr, " Looking for element ");
			for (int si = 0; si < cie.sequencetags.size(); si++) {
			    DcmTag & curtag = cie.sequencetags[si];
			    fprintf(stderr, "(%04x,%04x) [%s] => ", (unsigned int)curtag.getGTag(), (unsigned int)curtag.getETag(), curtag.getTagName());
			}
			fprintf(stderr, "(%04x,%04x) [%s]\n", (unsigned int)cie.elemtag.getGTag(), (unsigned int)cie.elemtag.getETag(), cie.elemtag.getTagName());
			fprintf(stderr, "  last stack: ");
			for (int si = 1; si < mystack.size(); si++) {
			    fprintf(stderr, "(%04x,%04x) => ", (unsigned int)mystack[si]->getTag().getGTag(), (unsigned int)mystack[si]->getTag().getETag());
			}
			fprintf(stderr, "\n");
#endif
			// Find how many initial elements in the last and
			// current sequence hierarchies are the same.
			int lastnst = lastseqtags.size();
			int numcommon = 0;
			for (numcommon = 0; numcommon < nst && numcommon < lastnst; numcommon++) {
			    if (cie.sequencetags[numcommon] != lastseqtags[numcommon]) {
				break;
			    }
			}
#if DEBUG
			fprintf(stderr, "  found %d common stack elements\n", numcommon);
#endif
			// Last sequence list equals current list in the first
			// numcommon tags.
			// If mystack has fewer than numcommon + 1 elements,
			// then we didn't find the common elements last time.
			if (mystack.size() < numcommon + 1) {
			    continue;
			}
			// Pop mystack to the point where we have common
			// elements (if numcommon == 0, then we pop all the
			// way back to the functional groups element).
			while (numcommon < mystack.size() - 1) {
			    mystack.pop_back();
			}
			// mystack now has numcommon elements, and we can
			// start searching from the element at the top of
			// mystack for sub-elements that were different
			// from the last search.
			int si = numcommon;
			for (si = numcommon; si < nst; si++) {
#if DEBUG
			    fprintf(stderr, "   searching from ");
			    for (int tosi = 0; tosi <= si; tosi++) {
				fprintf(stderr, "(%04x,%04x) ", (unsigned int)mystack[tosi]->getTag().getGTag(), (unsigned int)mystack[tosi]->getTag().getETag());
			    }
			    fprintf(stderr, "for (%04x,%04x)\n", (unsigned int)cie.sequencetags[si].getGTag(), (unsigned int)cie.sequencetags[si].getETag());
#endif
			    if (mystack[si]->search(cie.sequencetags[si], stack, ESM_afterStackTop) == EC_Normal) {
				mystack.push_back((DcmElement *)stack.top());
			    } else {
				break;
			    }
			}
			lastseqtags = cie.sequencetags;
			if (si != nst) {
			    // didn't find sequence hierarchy
			    continue;
			}
			if (mystack[mystack.size()-1]->search(cie.elemtag, stack, ESM_afterStackTop) == EC_Normal) {
#if DEBUG
			    fprintf(stderr, "   found element!\n");
#endif
			    curcachep->*(cie.memberp) = (DcmElement *)stack.top();
			}
		    }
		}
	    }
	}

	/* create new finfos */
	cont->finfos.resize(numfinfos + numframes, NULL);
	numfinfos += numframes;
	for (int framenum = 0; framenum < numframes; framenum++) {
	    elemcache = elemcaches[framenum];
	    finfop = new dicomfileinfo;
	    finfop->acqnum = -1;
	    finfop->framenum = framenum;

#if DEBUG
	    fprintf(stderr, "Looking at file %s frame number %d ...\n", filenamelist[filenum], framenum);
#endif

	    if ((elem = elemcache.manufacturer) != NULL) {
		const char * cs;
		tmpos.clear();
		elem->getOFString(tmpos, 0);
		cs = tmpos.c_str();
		finfop->manufacturer = strdup(cs);
	    } else {
		fprintf(stderr, FUNC ": error: can't get Manufacturer\n");
		goto PARSEFAIL;
	    }

	    if (strcmp(finfop->manufacturer, "SIEMENS") == 0) {
		if ((elem = elemcache.imagetype) != NULL) {
		    int valnum;
		    for (valnum = 0; valnum < (int)elem->getVM(); valnum++) {
			tmpos.clear();
			elem->getOFString(tmpos, valnum);
			if (strcmp(tmpos.c_str(), "MOSAIC") == 0)
			    finfop->ismosaic = 1;
		    }
		}
	    }
	    if (strcmp(finfop->manufacturer, "SIEMENS") == 0) {
		if ((elem = elemcache.siemensCSAImageHeaderInfo) != NULL) {
#if DEBUG
		    fprintf(stderr, "Found old Siemens header\n");
#endif
		    Uint8 * bytes = NULL;
		    Uint32 pos;
		    Uint32 len;
		    int byteswap = 0;
		    Uint32 nheaders;

		    len = elem->getLength();
		    elem->getUint8Array(bytes);

		    {
			int dcmbigendian = DcmXfer(dfile->dataset->getOriginalXfer()).isBigEndian();
			int msbfirst = 1;
			msbfirst = (((char *)&msbfirst)[0] == 0);
			byteswap = (dcmbigendian != msbfirst);
		    }

#define _SWAP4(x) {				\
			char s = (x)[0];	\
			(x)[0] = (x)[3];	\
			(x)[3] = s;		\
			s = (x)[1];		\
			(x)[1] = (x)[2];	\
			(x)[2] = s;		\
		    }
#define _READ(tgt, src, srclen, srcoff, copylen, swapifneeded)  {       \
			if ((srcoff) + (copylen) < (srclen)) {		\
			    if ((swapifneeded) && (byteswap)) {		\
				_SWAP##copylen((src) + (srcoff));	\
			    }						\
			    memcpy((char *)(tgt), (char *)(src) + (srcoff), (copylen));	\
			}						\
			srcoff += 4;					\
		    }

	    
		    pos = 0;
		    if (strncmp((char *)bytes, "SV10", 4) != 0) {
			goto FINISH_SIEMENS_0029_1010;
		    }
		    pos += 4;
		    pos += 4; /* skip an unused word */
		    _READ(&nheaders, bytes, len, pos, 4, 1);
		    pos += 4; /* skip an unused word */
		    for (int headern = 0; headern < nheaders && pos < len; headern++) {
			char h_name[65];
			Sint32 h_vm;
			char h_vr[4];
			Sint32 h_syngodt;
			Sint32 h_nitems;
			Sint32 h_xx;
			OFString items;
			if (pos + 64 >= len) {
			    break;
			}
			memcpy(&h_name, &bytes[pos], 64);
			pos += 64;
			h_name[64] = '\0';
			_READ(&h_vm, bytes, len, pos, 4, 1);
			_READ(&h_vr, bytes, len, pos, 4, 0);
			_READ(&h_syngodt, bytes, len, pos, 4, 1);
			_READ(&h_nitems, bytes, len, pos, 4, 1);
			_READ(&h_xx, bytes, len, pos, 4, 1);
			for (int itemn = 0; itemn < h_nitems && pos < len; itemn++) {
			    Uint32 unused;
			    Uint32 itemlen;
			    size_t itemlencropped;
			    _READ(&unused, bytes, len, pos, 4, 1);
			    _READ(&itemlen, bytes, len, pos, 4, 1);
			    _READ(&unused, bytes, len, pos, 4, 1);
			    _READ(&unused, bytes, len, pos, 4, 1);
			    if (itemlen == 0) {
				continue;
			    }
			    if (pos + itemlen >= len) {
				break;
			    }
			    if (itemn > 0) {
				items += "\\";
			    }
			    for (itemlencropped = 0; itemlencropped < itemlen; itemlencropped++) {
				char c = bytes[pos + itemlencropped];
				if (c == 0 || c == ' ') {
				    break;
				}
			    }
			    items += OFString((char *)&bytes[pos], itemlencropped);
			    pos += itemlen + ((4 - (itemlen % 4)) % 4);
			}
#if DEBUG
			fprintf(stderr, "Adding oldsiemensmap['OLD_%s'] = %s\n", h_name, items.c_str());
#endif
			finfop->oldsiemensmap[OFString("OLD_") + h_name] = items;
		    }
		FINISH_SIEMENS_0029_1010:
#if DEBUG
		    fprintf(stderr, "Finished looking through old Siemens header\n");
#endif
		    /* no op */;
		}
	    }

	    if ((elem = elemcache.studyinstanceuid) != NULL) {
		tmpos.clear();
		elem->getOFString(tmpos, 0);
		finfop->studyuid = strdup(tmpos.c_str());
	    } else {
		fprintf(stderr, FUNC ": error: can't get StudyUID\n");
		goto PARSEFAIL;
	    }
	    if ((elem = elemcache.seriesinstanceuid) != NULL) {
		tmpos.clear();
		elem->getOFString(tmpos, 0);
		finfop->seriesuid = strdup(tmpos.c_str());
	    } else {
		fprintf(stderr, FUNC ": error: can't get SeriesUID\n");
		goto PARSEFAIL;
	    }
	    if ((elem = elemcache.imagetype) != NULL) {
		tmpos.clear();
		elem->getOFString(tmpos, 0);
		finfop->imagetype = strdup(tmpos.c_str());
	    } else {
		fprintf(stderr, FUNC ": error: can't get ImageType\n");
		goto PARSEFAIL;
	    }
	    if ((elem = elemcache.inplanephaseencodingdirection) != NULL) {
		tmpos.clear();
		elem->getOFString(tmpos, 0);
		finfop->inplanephaseencodingdirection = strdup(tmpos.c_str());
	    }
	    if ((elem = elemcache.modality) != NULL) {
		tmpos.clear();
		elem->getOFString(tmpos, 0);
		finfop->modality = strdup(tmpos.c_str());
	    } else {
		fprintf(stderr, FUNC ": error: can't get Modality\n");
		goto PARSEFAIL;
	    }
	    if (strcmp(finfop->manufacturer, "SIEMENS") != 0 &&
		strcmp(finfop->modality, "CT") != 0 &&
		(elem = elemcache.acquisitionnumber) != NULL) {
		Sint32 acqnum;
		elem->getSint32(acqnum, 0);
		finfop->acqnum = (double)acqnum;
	    }

	    if ((elem = elemcache.columns) != NULL) {
		elem->getUint16(finfop->columns, 0);
	    }
	    if ((elem = elemcache.rows) != NULL) {
		elem->getUint16(finfop->rows, 0);
	    }
	    if ((elem = elemcache.samplesperpixel) != NULL) {
		elem->getUint16(finfop->samplesperpixel, 0);
	    }
	    if ((elem = elemcache.bitsallocated) != NULL) {
		elem->getUint16(finfop->bitsallocated, 0);
	    }
    
	    if ((elem = elemcache.imageorientationpatient) != NULL) {
		elem->getFloat64(finfop->rowsr, 0);
		elem->getFloat64(finfop->rowsa, 1);
		elem->getFloat64(finfop->rowss, 2);
		elem->getFloat64(finfop->colsr, 3);
		elem->getFloat64(finfop->colsa, 4);
		elem->getFloat64(finfop->colss, 5);
	    }

	    if ((elem = elemcache.repetitiontime) != NULL) {
		elem->getFloat64(finfop->tr, 0);
		finfop->foundtr = 1;
	    }
	    if ((elem = elemcache.echotime) != NULL) {
		elem->getFloat64(finfop->te, 0);
		finfop->foundte = 1;
	    }
	    if ((elem = elemcache.echonumber) != NULL) {
	        Sint32 echonumber;
		elem->getSint32(echonumber, 0);
		finfop->echonumber = (double)echonumber;
	    }
	    if ((elem = elemcache.flipangle) != NULL) {
		elem->getFloat64(finfop->flipangle, 0);
	    }

	    if ((elem = elemcache.pixelspacing) != NULL) {
		elem->getFloat64(finfop->colspacing, 0);
		elem->getFloat64(finfop->rowspacing, 1);
	    }

	    if ((elem = elemcache.slicethickness) != NULL) {
		elem->getFloat64(finfop->slicethickness, 0); /* Z thickness */
	    }
	    if ((elem = elemcache.spacingbetweenslices) != NULL) {
		elem->getFloat64(finfop->slicespacing, 0);
	    } else {
		finfop->slicespacing = finfop->slicethickness;
	    }

	    if (strcmp(finfop->manufacturer, "GE MEDICAL SYSTEMS") == 0) {
		/* Raw Data Type -- when data is provided in multiple types:
		 *  magnitude=0, phase=1, real=2, imaginary=3 */
		if ((elem = elemcache.gerawdatatype) != NULL) {
		    val.i16 = 0;
		    elem->getSint16(val.i16, 0);
		    finfop->gerawdatatype = val.i16;
		}
		/* Swap Phase/Frequency */
		if ((elem = elemcache.geswapphasefrequency) != NULL) {
		    val.i16 = 0;
		    elem->getSint16(val.i16, 0);
		    finfop->geswapphasefrequency = val.i16;
		}
	    }
	    if (strcmp(finfop->manufacturer, "GE MEDICAL SYSTEMS") == 0 &&
		(elem = elemcache.acquisitiondate) != NULL) {
		OFDate ofdate;
		((DcmDate *)elem)->getOFDate(ofdate, 0);
		if ((elem = elemcache.acquisitiontime) != NULL) {
		    struct tm tm;
		    time_t tt;
		    unsigned int hour, min;
		    double sec;
		    struct tm *tmp;
		    time_t lt;
		    
		    memset(&tm, '\0', sizeof(tm));
		    lt = time(NULL);
		    tmp = localtime(&lt);
		    tm = *tmp;
		    tm.tm_isdst = 0;
		    tmpos.clear();
		    ((DcmTime *)elem)->getOFString(tmpos, 0);
		    tm.tm_year = ofdate.getYear() - 1900;
		    tm.tm_mon = ofdate.getMonth() - 1;
		    tm.tm_mday = ofdate.getDay();
		    if (tm.tm_year < 71) { /* make sure time is after epoch */
			tm.tm_year = 71;
		    }
		    if (sscanf(tmpos.c_str(), "%02u%02u", &hour, &min) == 2) {
			char tmpchar;
			sec = atof(tmpos.c_str()+4);
			tm.tm_hour = hour;
			tm.tm_min = min;
			tm.tm_sec = (int)sec;
			tt = mktime(&tm);
			finfop->sliceacqtime = (double)tt;
			finfop->sliceacqtime += sec - (double)tm.tm_sec; /* get fraction */
#ifndef WIN32
			snprintf(&tmpchar, 1, "%g", finfop->sliceacqtime); /* this avoids an optimization anomaly on GCC 3.2.3 on RHEL/CentOS 3*/
#endif
		    }
		}
	    }

#if 0
	    // XXX There is a known major bug in Siemens DICOM where the
	    // XXX AcquisitionTime field is incorrect.  Inaccuracies seem
	    // XXX to be accumulative, so the reported AcquisitionTime of the
	    // XXX last XXX volume may be off by some number of *seconds*, for
	    // XXX long scans.
	    if (finfop->ismosaic &&
		((elem = elemcache.acquisitiondate) != NULL ||
		 (elem = elemcache.contentdate) != NULL)) {
		OFDate ofdate;
		((DcmDate *)elem)->getOFDate(ofdate, 0);
		if ((elem = elemcache.acquisitiontime) != NULL ||
		    (elem = elemcache.contenttime) != NULL) {
		    struct tm tm;
		    time_t tt;
		    unsigned int hour, min;
		    double sec;
		    struct tm *tmp;
		    time_t lt;
		    
		    memset(&tm, '\0', sizeof(tm));
		    lt = time(NULL);
		    tmp = localtime(&lt);
		    tm = *tmp;
		    tm.tm_isdst = 0;
		    tmpos.clear();
		    ((DcmTime *)elem)->getOFString(tmpos, 0);
		    tm.tm_year = ofdate.getYear() - 1900;
		    tm.tm_mon = ofdate.getMonth() - 1;
		    tm.tm_mday = ofdate.getDay();
		    if (tm.tm_year < 71) { /* make sure time is after epoch */
			tm.tm_year = 71;
		    }
		    if (sscanf(tmpos.c_str(), "%02u%02u", &hour, &min) == 2) {
			char tmpchar;
			sec = atof(tmpos.c_str()+4);
			tm.tm_hour = hour;
			tm.tm_min = min;
			tm.tm_sec = (int)sec;
			tt = mktime(&tm);
			finfop->t = (double)tt;
			finfop->t += sec - (double)tm.tm_sec; /* get fraction */
#ifndef WIN32
			snprintf(&tmpchar, 1, "%g", finfop->t); /* this avoids an optimization anomaly on GCC 3.2.3 on RHEL/CentOS 3*/
#endif
			finfop->t *= 1000; /* convert to ms */
			finfop->foundtimedim = 1;
		    }
		}
	    } else
#endif
	    if ((elem = elemcache.temporalpositionidentifier) != NULL) {
	        /* newer DICOM files use this to indicate time points */
		elem->getSint32(val.i32, 0);
		finfop->t = (val.i32 - 1); /* identifier indexed by 1 */
		if ((elem = elemcache.temporalresolution) != NULL) {
		    elem->getFloat64(val.f64, 0);
		    finfop->t *= val.f64;
		} else if ((elem = elemcache.repetitiontime) != NULL) {
		  finfop->t *= finfop->tr;
		}
		finfop->foundtimedim = 1;
	    }
	    if ((elem = elemcache.temporalpositionindex) != NULL) {
		elem->getUint32(val.ui32, 0);
		finfop->t = (val.ui32 - 1); /* identifier indexed by 1 */
		if ((elem = elemcache.repetitiontime) != NULL) {
		    elem->getFloat64(val.f64, 0);
		    finfop->t *= val.f64;
		}
		finfop->foundtimedim = 1;
	    }

	    if ((elem = elemcache.imagepositionpatient) != NULL) {
		elem->getFloat64(finfop->originr, 0);
		finfop->originr *= -1.0;
		elem->getFloat64(finfop->origina, 1);
		finfop->origina *= -1.0;
		elem->getFloat64(finfop->origins, 2);
	    }
	    if ((elem = elemcache.instancenumber) != NULL) {
		elem->getSint32(val.i32, 0);
		finfop->instance = val.i32;
	    }
	    if ((elem = elemcache.philipsslicenumbermr) != NULL) {
		elem->getSint32(val.i32, 0);
		finfop->slicenum = val.i32;
	    }
	    if ((elem = elemcache.dimensionindexsequence) != NULL) {
		finfop->dimsequenceidentifier = dimsequenceidentifier;
	    }
	    if ((elem = elemcache.dimensionindexvalues) != NULL) {
		/* look for unrecognized dimension indices */
		int dimnum = -1;
		DcmObject * curitem = NULL;
		std::vector<enhanceddimindex>::iterator dimiter;
		for (dimiter = dimsequence.begin(), dimnum = 0; dimiter < dimsequence.end(); dimiter++, dimnum++) {
		    DcmTagKey dimindexptr;
		    DcmTagKey funcgroupptr;
		    enhanceddimindex curdimindex = *dimiter;
		    dimindexptr = curdimindex.dimindexptr;
		    funcgroupptr = curdimindex.funcgroupptr;
		    int numcacheinitsenhanced = cacheinitsenhanced.size();
		    int found = 0;
		    for (int i = 0; i < numcacheinitsenhanced; i++) {
			/* XXX we assume the element referenced by the
			 * functional group pointer will always be the
			 * outermost element in the sequence, deducing
			 * from DICOM's in the wild.
			 */
			if (cacheinitsenhanced[i].sequencetags[0] == funcgroupptr &&
			    cacheinitsenhanced[i].elemtag == dimindexptr) {
			    found = 1;
			    break;
			}
		    }
		    if (!found) {
			// this is a dimension index that we don't normally look for, so let's add it as an unrecognized index
			if (elemcache.dimensionindexvalues->getOFString(tmpos, dimnum) != EC_Normal) {
			    fprintf(stderr, "ERROR: can't get %dth value of DimensionIndexValues\n", dimnum);
			    goto PARSEFAIL;
			}
			finfop->unrecognized_indices = boost::shared_ptr<OFString>(new OFString());
			(*finfop->unrecognized_indices) += '\\';
			(*finfop->unrecognized_indices) += tmpos;
		    }
		}
		if (last_unrecognized_indices.get() != NULL &&
		    finfop->unrecognized_indices.get() != NULL &&
		    *last_unrecognized_indices == *finfop->unrecognized_indices) {
		    finfop->unrecognized_indices = last_unrecognized_indices;
		} else {
		    last_unrecognized_indices = finfop->unrecognized_indices;
		}
	    }
	    if ((elem = elemcache.stackid) != NULL) {
		tmpos.clear();
		elem->getOFString(tmpos, 0);
		finfop->stackid = strdup(tmpos.c_str());
	    }
	    /* GEMS_PulseSequenceName */
	    if ((elem = elemcache.gepulsesequencename) != NULL) {
		tmpos.clear();
		elem->getOFString(tmpos, 0);
		//if (tmpos.compare(0,4,"epi2") == 0) {
		    Sint32 bvalue = 0;
		    Sint32 bvaluebiasfactor = 0;
		    Float64 scalex = 0, scaley = 0, scalez = 0;
		    int blistind;
		    Sint16 swapphasefrequency = 0;
		    /* GEMS_SlopInteger6 */
		    if ((elem = elemcache.geslopinteger6) != NULL) {
			elem->getSint32(bvalue, 0);
		    }
		    if ((elem = elemcache.bvaluebiasfactor) != NULL) {
			elem->getSint32(bvaluebiasfactor, 0);
		    }
		    bvalue -= bvaluebiasfactor;
		    /* GEMS_UserData20 */
		    if ((elem = elemcache.geuserdata20) != NULL) {
			elem->getFloat64(scalex, 0);
		    }
		    /* GEMS_UserData21 */
		    if ((elem = elemcache.geuserdata21) != NULL) {
			elem->getFloat64(scaley, 0);
		    }
		    /* GEMS_UserData22 */
		    if ((elem = elemcache.geuserdata22) != NULL) {
			elem->getFloat64(scalez, 0);
		    }
		    if ((elem = elemcache.geswapphasefrequency) != NULL) {
			elem->getSint16(swapphasefrequency);
		    }
		    if (swapphasefrequency != 1) {
			/* We need to swap X and Y */
			Float64 tmpswap = scalex;
			scalex = scaley;
			scaley = tmpswap;
		    }
		    /* the following is to generate a unique value for
		     * each (bvalue, scalex, scaley, scalez) tuple,
		     * which will maintain order when sorted.
		     */
		    for (blistind = 0; blistind < blistsize; blistind++) {
			blistelem * elemp = &blist[blistind];
			if (elemp->bvalue == bvalue &&
			    elemp->scalex == scalex &&
			    elemp->scaley == scaley &&
			    elemp->scalez == scalez) {
			    break;
			}
		    }
		    if (blistind == blistsize) {
			blistelem * elemp = NULL;
			blist = (blistelem *)realloc(blist, sizeof(blistelem)*(blistsize+1));
			elemp = &blist[blistsize];
			elemp->bvalue = bvalue;
			elemp->scalex = scalex;
			elemp->scaley = scaley;
			elemp->scalez = scalez;
			blistsize++;
		    }
		    finfop->bmatrixorder = blistind;
		    finfop->bvalue = (double)bvalue;
		    finfop->scalex = scalex;
		    finfop->scaley = scaley;
		    finfop->scalez = scalez;
		    /* measurement frame is identity for GE data */
		    finfop->frameidentity = 1;
		//}
	    }
	    if (elemcache.gepulsesequencename == NULL &&
		(elemcache.diffusionbvalue != NULL ||
		 (strcmp(finfop->manufacturer, "SIEMENS") == 0 &&
		  (elemcache.siemensbvalue || finfop->oldsiemensmap.count("OLD_B_value"))))) {
		double gradlenthresh = 0.5;
		Float64 bvalue = 0;
		Float64 scalex = 0, scaley = 0, scalez = 0;
		int blistind = -1;
		bool tryoldbmatrix = true;
		if (elemcache.diffusionbvalue != NULL) {
		    // in some Philips data this field is not in a
		    // sequence as specified by supplement 49 -- the
		    // Philips DICOM conformance statements seem to
		    // indicate that this is intentional and an
		    // "extension" of the spec.
		    elem = elemcache.diffusionbvalue;
		    elem->getFloat64(bvalue, 0);
		    if (elemcache.diffusiongradientorientation != NULL) {
		      elem = elemcache.diffusiongradientorientation;
		      elem->getFloat64(scalex, 0);
		      elem->getFloat64(scaley, 1);
		      elem->getFloat64(scalez, 2);
		    } else {
		      scalex = 0;
		      scaley = 0;
		      scalez = 0;
		    }
		    scalex *= -1;
		    scaley *= -1;
		    // measurement frame is identity for instances that specify
		    // diffusion using the standard DICOM tags
		    finfop->frameidentity = 1;
		    tryoldbmatrix = false;
		} else if (elemcache.siemensbvalue) {
		    /* diffusion directions for Siemens */
		    Sint32 intbvalue = 0;
		    elem = elemcache.siemensbvalue;
		    elem->getSint32(intbvalue, 0);
		    if (elemcache.siemensdiffusionvector) {
			elem = elemcache.siemensdiffusionvector;
			elem->getFloat64(scalex, 0);
			elem->getFloat64(scaley, 1);
			elem->getFloat64(scalez, 2);
		    }
		    double scaleall =
			sqrt((scalex * scalex) +
			     (scaley * scaley) +
			     (scalez * scalez)) +
			DBL_MIN;
		    bvalue = (Float64)intbvalue;
		    if (finfop->ismosaic) {
			/* just generate a new blist value so repeated matches (like multiple B0 values) aren't clumped together */
			blistind = blistsize;
		    }
		    if (scaleall >= gradlenthresh) {
			tryoldbmatrix = false;
		    } else {
#if DEBUG
			fprintf(stderr, "Length of diffusion gradient vector is less than %g, so will look for old Siemens B_matrix\n", gradlenthresh);
#endif
		    }
		} else if (finfop->oldsiemensmap.count("OLD_B_value")) {
		    /* diffusion directions for old Siemens */
		    while (1) { /* to allow for an easy way out for errors */
			char *endptr = NULL;
			long int intbvalue = strtol(finfop->oldsiemensmap["OLD_B_value"].c_str(), &endptr, 10);
			if (*endptr != '\0') {
			    /* badly formatted integer */
			    break;
			}
			bvalue = (Float64)intbvalue;
			if (finfop->oldsiemensmap.count("OLD_DiffusionGradientDirection")) {
			    OFString graddir = finfop->oldsiemensmap["OLD_DiffusionGradientDirection"];
			    size_t slashpos1 = graddir.find('\\');
			    if (slashpos1 == std::string::npos) {
				break;
			    }
			    size_t slashpos2 = graddir.find('\\', slashpos1 + 1);
			    if (slashpos2 == std::string::npos) {
				break;
			    }
			    OFString graddirx = graddir.substr(0, slashpos1);
			    OFString graddiry = graddir.substr(slashpos1 + 1, slashpos2 - (slashpos1 + 1));
			    OFString graddirz = graddir.substr(slashpos2 + 1);
			    const char * graddirxstr = graddirx.c_str();
			    const char * graddirystr = graddiry.c_str();
			    const char * graddirzstr = graddirz.c_str();
			    double doublescalex = strtod(graddirxstr, &endptr);
			    if (endptr == graddirxstr || *endptr != '\0') {
				break;
			    }
			    double doublescaley = strtod(graddirystr, &endptr);
			    if (endptr == graddirystr || *endptr != '\0') {
				break;
			    }
			    double doublescalez = strtod(graddirzstr, &endptr);
			    if (endptr == graddirzstr || *endptr != '\0') {
				break;
			    }
			    double scaleall =
				sqrt((doublescalex * doublescalex) +
				     (doublescaley * doublescaley) +
				     (doublescalez * doublescalez)) +
				DBL_MIN;
			    if (scaleall >= gradlenthresh) {
				tryoldbmatrix = false;
			    } else {
#if DEBUG
				fprintf(stderr, "Length of diffusion gradient vector is less than %g, so will look for old Siemens B_matrix\n", gradlenthresh);
#endif
			    }
			    scalex = doublescalex;
			    scaley = doublescaley;
			    scalez = doublescalez;
			}
			if (finfop->ismosaic) {
			    /* just generate a new blist value so repeated matches (like multiple B0 values) aren't clumped together */
			    blistind = blistsize;
			}
			break;
		    }
		}
		if (tryoldbmatrix && finfop->oldsiemensmap.count("OLD_B_matrix")) {
#if DEBUG
		    fprintf(stderr, "Attempting to extract gradient direction from old Siemens B_matrix\n", gradlenthresh);
#endif
		    while (1) {
			// The gradient direction has a small length.  Let's
			// extract it from the B matrix.
			// See:
			//  http://nipy.sourceforge.net/dipy/theory/bmatrix.html
			double bmatrixdoubles[6];
			size_t ind1 = 0;
			size_t ind2 = 0;
			char * endptr = NULL;
			OFString bmatrixstr = finfop->oldsiemensmap["OLD_B_matrix"];
			OFString bvaluestr = finfop->oldsiemensmap["OLD_B_value"];
			long int intbvalue = strtol(finfop->oldsiemensmap["OLD_B_value"].c_str(), &endptr, 10);
			bvalue = intbvalue;
			if (intbvalue == 0) {
			    scalex = 0;
			    scaley = 0;
			    scalez = 0;
			}
			if (*endptr != '\0') {
			    /* badly formatted integer */
			    break;
			}
			int i;
			for (i = 0; i < 6; i++) {
			    if (i == 5) {
				ind2 = bmatrixstr.length();
			    } else {
				ind2 = bmatrixstr.find('\\', ind1);
			    }
			    OFString comp = bmatrixstr.substr(ind1, ind2 - ind1);
			    const char * compstr = comp.c_str();
			    bmatrixdoubles[i] = strtod(compstr, &endptr);
			    if (endptr == compstr || *endptr != '\0') {
				break;
			    }
			    ind1 = ind2 + 1;
			}
			if (i != 6) {
			    break;
			}
			double bmatrix[3][3];
			bmatrix[0][0] = bmatrixdoubles[0];
			bmatrix[0][1] = bmatrixdoubles[1];
			bmatrix[0][2] = bmatrixdoubles[2];
			bmatrix[1][0] = bmatrix[0][1];
			bmatrix[1][1] = bmatrixdoubles[3];
			bmatrix[1][2] = bmatrixdoubles[4];
			bmatrix[2][0] = bmatrix[0][2];
			bmatrix[2][1] = bmatrix[1][2];
			bmatrix[2][2] = bmatrixdoubles[5];
			gsl_matrix * A = NULL;
			gsl_matrix * V = NULL;
			gsl_vector * S = NULL;
			gsl_vector * work = NULL;
			A = gsl_matrix_calloc(3, 3);
			V = gsl_matrix_calloc(3, 3);
			S = gsl_vector_calloc(3);
			work = gsl_vector_calloc(3);
			gsl_matrix_set(A, 0, 0, bmatrixdoubles[0]);
			gsl_matrix_set(A, 0, 1, bmatrixdoubles[1]);
			gsl_matrix_set(A, 0, 2, bmatrixdoubles[2]);
			gsl_matrix_set(A, 1, 0, bmatrixdoubles[1]);
			gsl_matrix_set(A, 1, 1, bmatrixdoubles[3]);
			gsl_matrix_set(A, 1, 2, bmatrixdoubles[4]);
			gsl_matrix_set(A, 2, 0, bmatrixdoubles[2]);
			gsl_matrix_set(A, 2, 1, bmatrixdoubles[4]);
			gsl_matrix_set(A, 2, 2, bmatrixdoubles[5]);
			gsl_linalg_SV_decomp(A, V, S, work);
			double newbvalue = bmatrixdoubles[0] + bmatrixdoubles[3] + bmatrixdoubles[5];
			if (newbvalue == 0) {
			    scalex = 0;
			    scaley = 0;
			    scalez = 0;
			} else {
			    scalex = gsl_matrix_get(A, 0, 0);
			    scaley = gsl_matrix_get(A, 1, 0);
			    scalez = gsl_matrix_get(A, 2, 0);
			}
#if DEBUG
			fprintf(stderr, " Extracted bvalue %g and gradient direction (%g, %g, %g) from old Siemens B_matrix\n", newbvalue, scalex, scaley, scalez);
#endif
			// Do old Siemens CSA headers use DICOM's LPS coordinate space?
			scalex *= -1;
			scaley *= -1;
			break;
		    }
		}
		Sint32 philipsmysteryfield = 0;
		if ((elem = elemcache.philips_2005_1413) != NULL) {
		    /* Philips has a field that seems to provide a new
		     * number for each volume; let's use it */
		    elem->getSint32(philipsmysteryfield);
		}
		if (blistind == -1) {
		    /* the following is to generate a unique value for
		     * each (bvalue, scalex, scaley, scalez) tuple,
		     * which will maintain order when sorted.
		     */
		    for (blistind = 0; blistind < blistsize; blistind++) {
			blistelem * elemp = &blist[blistind];
			if (elemp->philipsmysteryfield == philipsmysteryfield &&
			    elemp->bvalue == bvalue &&
			    elemp->scalex == scalex &&
			    elemp->scaley == scaley &&
			    elemp->scalez == scalez) {
#if DEBUG
			    fprintf(stderr, " Found blist element for diffusion parameters bvalue=%g scalex=%g scaley=%g scalez=%g\n", (double)bvalue, scalex, scaley, scalez);
#endif
			    break;
			}
		    }
		}
		if (blistind >= blistsize) {
		    blist = (blistelem *)realloc(blist, sizeof(blistelem)*(blistind+1));
		    blistsize = blistind + 1;
#if DEBUG
		    fprintf(stderr, " Creating new blist element for diffusion parameters bvalue=%g scalex=%g scaley=%g scalez=%g\n", (double)bvalue, scalex, scaley, scalez);
#endif
		}
		blistelem * elemp = &blist[blistind];
		/* set blist values everytime (because in Philips case we
		 * may get bmatrix indices out of order)
		 */
		elemp->bvalue = bvalue;
		elemp->scalex = scalex;
		elemp->scaley = scaley;
		elemp->scalez = scalez;
		elemp->philipsmysteryfield = philipsmysteryfield;
		finfop->bmatrixorder = blistind;
		finfop->bvalue = (double)bvalue;
		finfop->scalex = scalex;
		finfop->scaley = scaley;
		finfop->scalez = scalez;
		finfop->philipsmysteryfield = philipsmysteryfield;
	    }

	    finfop->filename = strdup(filenamelist[filenum]);
	    if (pixeldataoffset == -1) {
		int littleendian = 1;
		int explicitsyntax = 1;
		E_TransferSyntax ets;
		// hackery at end of this expression because tag values are always padded to even length
		long pixeldatasize = ((((numframes * finfop->rows * finfop->columns * finfop->samplesperpixel * finfop->bitsallocated / 8) + 1) / 2) * 2);

		/* look for pixel data element explicitly because some DICOM
		 * files have extraneous elements after the pixel data */
		ets = dfile->dataset->getOriginalXfer();
		switch (ets) {
		case EXS_LittleEndianImplicit:
		    littleendian = 1;
		    explicitsyntax = 0;
		    break;
		case EXS_BigEndianImplicit:
		    littleendian = 0;
		    explicitsyntax = 0;
		    break;
		case EXS_LittleEndianExplicit:
		    littleendian = 1;
		    explicitsyntax = 1;
		    break;
		case EXS_BigEndianExplicit:
		    littleendian = 0;
		    explicitsyntax = 1;
		    break;
		default:
		    fprintf(stderr, "Warning: unrecognized DICOM transfer type!\n");
		    break;
		}
		if ((pixeldataoffset = bxh_DICOM_pixel_data_offset(&myin, pixeldatasize, littleendian, explicitsyntax)) == -1) {
		    fprintf(stderr, "ERROR: can't find pixel data offset!\n");
		    goto PARSEFAIL;
		}
	    }
	    finfop->fileoffset = pixeldataoffset + (framenum * (finfop->rows * finfop->columns * finfop->samplesperpixel * finfop->bitsallocated / 8));
	    if (printfields) {
		std::cout << "FIELDS:";
		std::cout << *finfop;
	    }
#if DEBUG
	    {
		fprintf(stderr, "%s: instance=%d slicenum=%d", finfop->filename, (int)finfop->instance, (int)finfop->slicenum);
		fprintf(stderr, " tr=%g te=%g bmatrixorder_orig=%g bvalue=%g acqnum=%g gerawdatatype=%d",
			finfop->tr, finfop->te, finfop->bmatrixorder, finfop->bvalue, finfop->acqnum, (int)finfop->gerawdatatype);
		fprintf(stderr, " (%g, %g, %g)",
			finfop->originr,
			finfop->origina,
			finfop->origins);
		fprintf(stderr, "\n");
	    }
#endif
	    cont->finfos[cont->numfinfos] = finfop;
	    cont->numfinfos++;
	}

	delete dfileobj; dfileobj = NULL;
	
	goto PARSESUCCEED;

	PARSEFAIL:
	if (finfop) { delete finfop; finfop = NULL; }
	if (dfileobj) {delete dfileobj; dfileobj = NULL; }
#if DEBUG
	fprintf(stderr, "  ...parsing failed!\n");
#endif
	continue;

	PARSESUCCEED:
	if (elemcaches) { free(elemcaches); elemcaches = NULL; }
#if DEBUG
	fprintf(stderr, "  ...parsing succeeded!\n");
#endif
    }

    if (cont->numfinfos == 0) {
	goto FAIL;
    }

    if (blist != NULL) {
        /* we need to re-order blist based on instance number because the
	 * blist indices are arbitrary */
        /* bubble-sort finfos in cont according to instance number in
	 * preparation for reordering blist */
	int i, j, b;
	int bindmapmax = -1;
	int * bindmap = NULL;
	for (i = 0; i < cont->numfinfos; i++) {
	    for (j = i + 1; j < cont->numfinfos; j++) {
		int doswap = 0;
		if (cont->finfos[j]->instance < cont->finfos[i]->instance) {
		    doswap = 1;
		}
		if (doswap) {
		    dicomfileinfo * swap = cont->finfos[i];
		    cont->finfos[i] = cont->finfos[j];
		    cont->finfos[j] = swap;
		}
	    }
	}
	bindmap = (int *)malloc(sizeof(int)*blistsize);
	for (b = 0; b < blistsize; b++) {
	    bindmap[b] = -1;
	}
	/* reorder blist to match instance order, and rewrite bmatrixorder
	 * fields to match */
	for (i = 0; i < cont->numfinfos; i++) {
	    int bmatrixind = (int)cont->finfos[i]->bmatrixorder;
	    if (bindmap[bmatrixind] == -1) {
		bindmapmax++;
		bindmap[bmatrixind] = bindmapmax;
	    }
	    cont->finfos[i]->bmatrixorder = (double)bindmap[bmatrixind];
	}
	free(bindmap);
	bindmap = NULL;
    }

    if (bxh_DICOM_readFilesCont(docp, basepath, contp) != 0) {
	goto FAIL;
    }

    if (searchforothers && *contp) {
	/* don't return other non-matching files */
	bxh_DICOM_freeCont(*contp);
	*contp = NULL;
    }

    goto EXIT;
    
  FAIL:
    if (*contp) {
	bxh_DICOM_freeCont(*contp);
	*contp = NULL;
    }
    retval = -1;
    
  EXIT:
    if (streambuf) {
	free(streambuf); streambuf = NULL;
    }
    if (newfilenames) {
	for (filenum = 0; filenum < numinfiles; filenum++) {
	    free(newfilenames[filenum]); newfilenames[filenum] = NULL;
	}
	free(newfilenames);
    }
    if (blist)
	free(blist);
    return retval;
}

#undef FUNC
#define FUNC "bxh_DICOM_readFilesCont"
/**
 * Extract metadata from DICOM slices given in continuation structure,
 * and add all slices that match with the first slice into BXH document
 * (and return a continuation structure which can be sent to
 * a subsequent bxh_DICOM_readFilesCont() to do the rest of the slices).
 *
 * @param docp BXH document pointer
 * @param basepath location where BXH file will be written (for determining relative pathnames), or directory in which BXH file will be written (must end in path separator [e.g. '/' on UNIX])
 * @param contp On entry, *#contp must store a pointer to a continuation structure as returned by bxh_DICOM_readFilesStart() or bxh_DICOM_readFilesCont().  On return, *#contp will store a pointer to a new continuation structur, or NULL if there are no more slices to process.  The value returned in *#contp should be sent to a subsequent call to bxh_DICOM_readFilesCont() or freed using bxh_DICOM_freeCont().
 * @return 0 on success, or non-zero on error.
 */
int
bxh_DICOM_readFilesCont(BXHDocPtr docp, const char * basepath, void ** contp)
{
    int retval = 0;
    
    BXHElementPtr bxh = NULL;
    BXHElementPtr imagedata = NULL;
    BXHElementPtr acquisitiondata = NULL;
    BXHElementPtr subject = NULL;
    bxhrawdatarec * rawdatarec = NULL;
    bxhdimension * dimx = NULL;
    bxhdimension * dimztiled = NULL;
    bxhdimension * dimy = NULL;
    bxhdimension * dimz = NULL;
    bxhdimension * dimr = NULL;
    bxhdimension * dima = NULL;
    bxhdimension * dims = NULL;
    int originwrapfiles = 0;
    int extradimstart = 0;

    bxhdimension * dimtr = NULL;
    bxhdimension * dimte = NULL;
    bxhdimension * dimt = NULL;
    bxhdimension * dimdiffdir = NULL;
    bxhdimension * dimacqnum = NULL;
    OFString bvalueos("");

    Float64 rowspacing = 0;
    Float64 colspacing = 0;
    Uint16 acqrows = 0;
    Uint16 acqcols = 0;
    Float64 calcslicespacing = 0;
    Float64 freqfov = 0;
    Float64 phasefov = 0;

    DcmFileFormat * dfileobj = NULL;
    dicomdatameta ddm;
    dicomdatameta * dfile = &ddm;
    DcmStack stack;
    DcmElement * elem;
    OFString tmpos;
    union {
	Float32 f32;
	Float64 f64;
	Sint16 i16;
	Sint32 i32;
	Uint16 ui16;
	Uint32 ui32;
    } val;
    static char tmpbuf[8192];

    /* this is used to discover those values for which we need
     * new dimensions (i.e. those that vary from slice to slice)
     */
    struct dimmapentry newdimmap_orig[] = {
	{ &dicomfileinfo::bmatrixorder, NULL, "diffusiondirection", NULL, &dimdiffdir },
	{ &dicomfileinfo::tr, NULL, "tr", "ms", &dimtr },
	{ &dicomfileinfo::te, NULL, "te", "ms", &dimte },
	{ &dicomfileinfo::echonumber, &dicomfileinfo::echonumber, "te", "ms", &dimte },
	{ &dicomfileinfo::t, NULL, "t", "ms", &dimt },
	{ &dicomfileinfo::acqnum, NULL, "acqnum", NULL, &dimacqnum },
	{ 0, NULL }
    };
    struct dimmapentry * newdimmap = newdimmap_orig;

    /* these are acquisition parameters from the MR Image Module and
     * Enhanced MR Image Module (and related Macros) that will be copied
     * straight into the XML file */
    typedef enum {
	CP_STR,
	CP_STRARRAY,
	CP_ISODATE,
	CP_ISOTIME,
	CP_ISODATETIME,
    } acqparamtype;
    struct copyacqparam {
	DcmTag tag;
	const char * fieldname;
	acqparamtype type;
    };
    static copyacqparam * copyacqparamlist = NULL;
    static struct copyacqparam copyacqparamsraw[] = {
	{ DCM_AngioFlag, "angioflag", CP_STR },
	{ DCM_BeatRejectionFlag, "beatrejectionflag", CP_STR },
	{ DCM_CardiacNumberOfImages, "cardiacnumberofimages", CP_STR },
	{ DCM_Columns, "columns", CP_STR },
	{ DCM_StudyDate, "scandate", CP_ISODATE },
	{ DCM_ContentTime, "scantime", CP_ISOTIME },
	{ DCM_DateOfLastCalibration, "datelastcalibration", CP_STR },
	{ DCM_DeviceSerialNumber, "scannerserialnumber", CP_STR },
	{ DCM_EchoTrainLength, "echotrainlength", CP_STR },
	{ DCM_FlipAngle, "flipangle", CP_STR },
	{ DCM_HeartRate, "heartrate", CP_STR },
	{ DCM_HighRRValue, "highrrvalue", CP_STR },
	{ DCM_ImagedNucleus, "imagednucleus", CP_STR },
	{ DCM_ImagingFrequency, "imagingfrequency", CP_STR },
	{ DCM_InPlanePhaseEncodingDirection, "inplanephaseencodingdirection", CP_STR },
	{ DCM_IntervalsAcquired, "intervalsacquired", CP_STR },
	{ DCM_IntervalsRejected, "intervalsrejected", CP_STR },
	{ DCM_InversionTime, "ti", CP_STR },
	{ DCM_LowRRValue, "lowrrvalue", CP_STR },
	{ DCM_MRAcquisitionType, "mracquisitiontype", CP_STR },
	{ DCM_Manufacturer, "scannermanufacturer", CP_STR },
	{ DCM_ManufacturerModelName, "scannermodelname", CP_STR },
	{ DCM_NominalInterval, "nominalinterval", CP_STR },
	{ DCM_NumberOfAverages, "numaverages", CP_STR },
	{ DCM_NumberOfPhaseEncodingSteps, "numphaseencodingsteps", CP_STR },
	{ DCM_NumberOfTemporalPositions, "numtemporalpositions", CP_STR },
	{ DCM_OperatorsName, "operator", CP_STR },
	{ DCM_PVCRejection, "pvcrejection", CP_STR },
	{ DCM_PercentPhaseFieldOfView, "percentphasefieldofview", CP_STR },
	{ DCM_PercentSampling, "percentsampling", CP_STR },
	{ DCM_PixelPaddingValue, "pixelpaddingvalue", CP_STR },
	{ DCM_ReceiveCoilName, "receivecoilname", CP_STR },
	{ DCM_ReconstructionDiameter, "reconstructiondiameter", CP_STR },
	{ DCM_Rows, "rows", CP_STR },
	{ DCM_SAR, "sar", CP_STR },
	{ DCM_ScanOptions, "scanoptions", CP_STRARRAY },
	{ DCM_ScanningSequence, "scanningsequence", CP_STRARRAY },
	{ DCM_SequenceName, "sequencename", CP_STR },
	{ DCM_SequenceVariant, "sequencevariant", CP_STRARRAY },
	{ DCM_SeriesDescription, "description", CP_STR },
	{ DCM_SkipBeats, "skipbeats", CP_STR },
	{ DCM_SoftwareVersions, "softwareversions", CP_STR },
	{ DCM_SpatialResolution, "spatialresolution", CP_STR },
	{ DCM_StationName, "scanner", CP_STR },
	{ DCM_StudyDescription, "examdescription", CP_STR },
	{ DCM_TemporalPositionIdentifier, "temporalpositionidentifier", CP_STR },
	{ DCM_TemporalResolution, "temporalresolution", CP_STR },
	{ DCM_TimeOfLastCalibration, "timelastcalibration", CP_STR },
	{ DCM_TransmitCoilName, "transmitcoilname", CP_STR },
	{ DCM_TriggerTime, "triggertime", CP_STR },
	{ DCM_TriggerWindow, "triggerwindow", CP_STR },
	{ DCM_VariableFlipAngleFlag, "variableflipangleflag", CP_STR },
	{ DCM_dBdt, "dBdt", CP_STR },
	/* these are from Enhanced MR Image Module */
	{ DCM_InstanceNumber, "instancenumber", CP_STR },
	{ DCM_AcquisitionNumber, "acquisitionnumber", CP_STR },
	{ DCM_AcquisitionDateTime, "acquisitiondatetime", CP_STR },
	{ DCM_AcquisitionDuration, "acquisitionduration", CP_STR },
	{ DCM_NumberOfFrames, "numberofframes", CP_STR },
	{ DCM_ConcatenationFrameOffsetNumber, "concatenationframeoffsetnumber", CP_STR },
	{ DCM_RepresentativeFrameNumber, "representativeframenumber", CP_STR },
	{ DCM_ConcatenationUID, "concatenationuid", CP_STR },
	{ DCM_InConcatenationTotalNumber, "inconcatenationtotalnumber", CP_STR },
	{ DCM_ContentQualification, "contentqualification", CP_STR },
	{ DCM_ResonantNucleus, "resonantnucleus", CP_STR },
	{ DCM_ApplicableSafetyStandardAgency, "applicablesafetystandardagency", CP_STR },
	{ DCM_ApplicableSafetyStandardDescription, "applicablesafetystandarddescription", CP_STR },
	{ DCM_ImageComments, "imagecomments", CP_STR },
	{ DCM_FrameType, "frametype", CP_STR },
	{ DCM_PixelPresentation, "pixelpresentation", CP_STR },
	{ DCM_VolumetricProperties, "volumetricproperties", CP_STR },
	{ DCM_VolumeBasedCalculationTechnique, "volumebasedcalculationtechnique", CP_STR },
	{ DCM_ComplexImageComponent, "compleximagecomponent", CP_STR },
	{ DCM_AcquisitionContrast, "acquisitioncontrast", CP_STR },
	{ DCM_SamplesPerPixel, "samplesperpixel", CP_STR },
	{ DCM_PhotometricInterpretation, "photometricinterpretation", CP_STR },
	{ DCM_BitsAllocated, "bitsallocated", CP_STR },
	{ DCM_BitsStored, "bitsstored", CP_STR },
	{ DCM_HighBit, "highbit", CP_STR },
	{ DCM_SpacingBetweenSlices, "spacingbetweenslices", CP_STR },
	{ DCM_LossyImageCompression, "lossyimagecompression", CP_STR },
	{ DCM_LossyImageCompressionRatio, "lossyimagecompressionratio", CP_STR },
    };
    int numcopyacqparams = sizeof(copyacqparamsraw) / sizeof(copyacqparamsraw[0]);
    struct dicomcont * cont = NULL;
    struct dicomcont * retcont = NULL;

    char numbuf[128];

    if (contp == NULL || *contp == NULL) {
	fprintf(stderr, FUNC ": input argument contp is NULL!");
	return -1;
    }
    cont = (dicomcont *)*contp;

    std::vector<dicomfileinfo *> & finfos = cont->finfos;
    int & numfinfos = cont->numfinfos;

    std::map<OFString,OFString> siemensmap;

#if DEBUG
    {
	fprintf(stderr, "Pre-match series\n");
	printfinfos(stderr, numfinfos, finfos, newdimmap);
    }
#endif

    /* find those files that match with the first file (i.e. same
     * acquisition, te, orientation, etc.).
     * Those that don't match are saved for later. */
    {
	int filenum;
	int newnumfinfos;
	int forceconcat = cont->forceconcat;
	struct dicomfileinfo * fip0 = finfos[0];
	retcont = new dicomcont(*cont);
	retcont->numfinfos = 0;
	retcont->finfos.resize(numfinfos, NULL);
	for (filenum = 1; filenum < numfinfos; filenum++) {
	    struct dicomfileinfo * fipi = finfos[filenum];
	    double epsilon = 0.00002;
	    if (forceconcat) {
		goto MATCHSUCCEED;
	    }

	    if ((((fipi->studyuid == NULL || fip0->studyuid == NULL) &&
		  fipi->studyuid != fip0->studyuid) ||
		 strcmp(fipi->studyuid, fip0->studyuid) != 0)) {
#if 0
		fprintf(stderr, "studyuid: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }
	    if ((((fipi->seriesuid == NULL || fip0->seriesuid == NULL) &&
		  fipi->seriesuid != fip0->seriesuid) ||
		 strcmp(fipi->seriesuid, fip0->seriesuid) != 0)) {
#if 0
		fprintf(stderr, "seriesuid: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }
	    if ((((fipi->imagetype == NULL || fip0->imagetype == NULL) &&
		  fipi->imagetype != fip0->imagetype) ||
		 strcmp(fipi->imagetype, fip0->imagetype) != 0)) {
#if 0
		fprintf(stderr, "imagetype: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }
	    if ((((fipi->modality == NULL || fip0->modality == NULL) &&
		  fipi->modality != fip0->modality) ||
		 strcmp(fipi->modality, fip0->modality) != 0)) {
#if 0
		fprintf(stderr, "modality: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }
	    if (fipi->ismosaic != fip0->ismosaic) {
#if 0
		fprintf(stderr, "ismosaic: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }
	    if (fipi->foundtimedim != fip0->foundtimedim) {
#if 0
		fprintf(stderr, "foundtimedim: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }
	    if (fabs(fipi->colspacing - fip0->colspacing) > epsilon ||
		fabs(fipi->rowspacing - fip0->rowspacing) > epsilon) {
#if 0
		fprintf(stderr, "rowspacing or colspacing: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }

	    if (fabs(fipi->rowsr - fip0->rowsr) > epsilon ||
		fabs(fipi->rowsa - fip0->rowsa) > epsilon ||
		fabs(fipi->rowss - fip0->rowss) > epsilon ||
		fabs(fipi->colsr - fip0->colsr) > epsilon ||
		fabs(fipi->colsa - fip0->colsa) > epsilon ||
		fabs(fipi->colss - fip0->colss) > epsilon) {
		goto MATCHFAIL;
	    }

	    if (fipi->flipangle != fip0->flipangle) {
#if 0
		fprintf(stderr, "flipangle: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }

	    if (fipi->gerawdatatype != fip0->gerawdatatype &&
		strcmp(fip0->manufacturer, "GE MEDICAL SYSTEMS") == 0) {
		/* Raw Data Type */
#if 0
		fprintf(stderr, "gerawdatatype: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }

	    if (fipi->geswapphasefrequency != fip0->geswapphasefrequency &&
		strcmp(fip0->manufacturer, "GE MEDICAL SYSTEMS") == 0) {
		/* Swap Phase/Frequency */
#if 0
		fprintf(stderr, "geswapphasefrequency: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }

	    if (fipi->acqnum != fip0->acqnum &&
		strcmp(fip0->manufacturer, "GE MEDICAL SYSTEMS") == 0) {
#if 0
		fprintf(stderr, "acqnum: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }

	    if (fip0->dimsequenceidentifier.get() != fipi->dimsequenceidentifier.get() &&
		(fip0->dimsequenceidentifier.get() == NULL ||
		 fipi->dimsequenceidentifier.get() == NULL ||
		 *fip0->dimsequenceidentifier != *fipi->dimsequenceidentifier)) {
#if 0
		fprintf(stderr, "dimsequenceidentifier: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }

	    if (fip0->unrecognized_indices.get() != fipi->unrecognized_indices.get() &&
		(fip0->unrecognized_indices.get() == NULL ||
		 fipi->unrecognized_indices.get() == NULL ||
		 *fip0->unrecognized_indices != *fipi->unrecognized_indices)) {
#if 0
		fprintf(stderr, "unrecognized_indices: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }

	    if (fip0->stackid != fipi->stackid &&
		(fip0->stackid == NULL ||
		 fipi->stackid == NULL ||
		 strcmp(fip0->stackid, fipi->stackid) != 0)) {
#if 0
		fprintf(stderr, "stackid: match failed for file 0 and %d\n", filenum);
#endif
		goto MATCHFAIL;
	    }

	  MATCHSUCCEED:
	    continue;

	  MATCHFAIL:
	    retcont->finfos[retcont->numfinfos] = finfos[filenum];
	    retcont->numfinfos++;
	    finfos[filenum] = NULL;
	    continue;
	}
	/* squeeze out empty slots (non-matching files) */
	newnumfinfos = 0;
	for (filenum = 0; filenum < numfinfos; filenum++) {
	    if (finfos[filenum] != NULL) {
		if (filenum != newnumfinfos) {
		    finfos[newnumfinfos] = finfos[filenum];
		    finfos[filenum] = NULL;
		}
		newnumfinfos++;
	    }
	}
	numfinfos = newnumfinfos;
    }

    /* If Bruker, remove acqnum from the list of fields that trigger
     * new dimensions
     */
    if (strcmp(finfos[0]->manufacturer, "Bruker BioSpin MRI GmbH") == 0) {
	int newdimmapsize = 0;
	newdimmap = (struct dimmapentry *)malloc(sizeof(struct dimmapentry) * 1);
	for (int ndnum = 0; newdimmap_orig[ndnum].dimname != NULL; ndnum++) {
	    if (strcmp(newdimmap_orig[ndnum].dimname, "acqnum") == 0) {
		continue;
	    }
	    newdimmap = (struct dimmapentry *)realloc(newdimmap, sizeof(struct dimmapentry) * (newdimmapsize + 2));
	    newdimmap[newdimmapsize] = newdimmap_orig[ndnum];
	    newdimmapsize++;
	}
	/* sentry */
	newdimmap[newdimmapsize].memberp = 0;
	newdimmap[newdimmapsize].dimname = NULL;
    }

#if DEBUG
    {
	fprintf(stderr, "Pre-sort new series\n");
	printfinfos(stderr, numfinfos, finfos, newdimmap);
    }
#endif

    /* bubble-sort finfos according to all relevant dimensions */
    if (cont->sortopt != -1) {
	int i, j;
	int numnd;
	int sortopt = cont->sortopt;
	for (numnd = 0; newdimmap[numnd].dimname; numnd++) {
	    /* null */
	}
	for (i = 0; i < numfinfos; i++) {
	    for (j = i + 1; j < numfinfos; j++) {
		int ndnum;
		int doswap = 0;
		if (sortopt == 1) {
		    if (strcmp(finfos[i]->filename, finfos[j]->filename) <= 0) {
			doswap = 1;
		    }
		    break;
		}
		/* in reverse order of dimensions */
		for (ndnum = numnd-1; ndnum >= 0; ndnum--) {
		    double ival, jval;
		    ival = finfos[i]->*(newdimmap[ndnum].memberp);
		    jval = finfos[j]->*(newdimmap[ndnum].memberp);
		    if (jval < ival) {
			doswap = 1;
			break;
		    }
		    if (jval > ival)
			break;
		}
		if (ndnum < 0) {
		    /* all other dimvals equal, try slicenum, then instancenum */
		    if (finfos[i]->slicenum != finfos[j]->slicenum) {
		        if (finfos[j]->slicenum < finfos[i]->slicenum) {
			    doswap = 1;
			}
		    } else if (finfos[j]->instance < finfos[i]->instance) {
			doswap = 1;
		    } else if (finfos[j]->instance == finfos[i]->instance) {
			/* instance is equal, use framenum */
			if (finfos[j]->framenum < finfos[i]->framenum) {
			    doswap = 1;
			}
		    }
		}
		if (doswap) {
		    struct dicomfileinfo * swap = finfos[i];
		    finfos[i] = finfos[j];
		    finfos[j] = swap;
		}
	    }
	}
    }
    
#if DEBUG
    {
	fprintf(stderr, "Pre-filter New series\n");
	printfinfos(stderr, numfinfos, finfos, newdimmap);
    }
#endif

    /* keep only the first of each run of contiguous repeated origins
     * (unless all files have the same origin or the repetition is
     *  covered by some other dimension) */
    {
	int filenum;
	struct dicomfileinfo * lastfinfop = NULL;
	int numrepeats = 0;
	int lastnumrepeats = 0;
	int consistent = 1;
	for (filenum = 0; filenum < numfinfos; filenum++) {
	    struct dicomfileinfo * finfop = finfos[filenum];
	    if (lastfinfop == NULL || numrepeats == 0) {
		numrepeats++;
	    } else {
		double rdist = finfop->originr - lastfinfop->originr;
		double adist = finfop->origina - lastfinfop->origina;
		double sdist = finfop->origins - lastfinfop->origins;
		double dist = sqrt(rdist*rdist + adist*adist + sdist*sdist);
		const double epsilon = 0.001 * finfop->slicespacing; /* small enough? */
		if (dist < epsilon &&
		    finfop->tr == lastfinfop->tr &&
		    finfop->te == lastfinfop->te &&
		    /* finfop->bmatrixorder == lastfinfop->bmatrixorder &&*/
		    finfop->acqnum == lastfinfop->acqnum) {
		    numrepeats++;
		} else {
		    if (lastnumrepeats != 0 && lastnumrepeats != numrepeats) {
			consistent = 0;
		    }
		    lastnumrepeats = numrepeats;
		    numrepeats = 1;
		}
	    }
	    lastfinfop = finfop;
	}
	if (numrepeats > 1 && numrepeats != numfinfos) {
	    if (consistent != 0) {
		/* we know that there are repeated origins, and they repeat
		 * consistently.  Re-order images. */
		fprintf(stderr, "Warning: found consistently repeated origins, but detected no fields to distinguish between volumes.  Assuming input files are sorted by volume (i.e. no guarantees the output data is correct if this assumption is not valid!)\n");
		int repeatind = 0;
		std::vector<struct dicomfileinfo *> newfinfos;
		for (repeatind = 0; repeatind < numrepeats; repeatind++) {
		    for (filenum = repeatind; filenum < numfinfos; filenum += numrepeats) {
			newfinfos.push_back(finfos[filenum]);
		    }
		}
		finfos = newfinfos;
	    } else {
		/* fallback: repeated origins, but not consistent, so just find
		 * first instance of each origin, and put off the others to the
		 * next continuation */
		int newnumfinfos = 0;
		lastfinfop = finfos[0];
		for (filenum = 1; filenum < numfinfos; filenum++) { /* skip first */
		    struct dicomfileinfo * finfop = finfos[filenum];
		    double rdist = finfop->originr - lastfinfop->originr;
		    double adist = finfop->origina - lastfinfop->origina;
		    double sdist = finfop->origins - lastfinfop->origins;
		    double dist = sqrt(rdist*rdist + adist*adist + sdist*sdist);
		    const double epsilon = 0.001 * finfop->slicespacing; /* small enough? */
		    if (dist < epsilon &&
			finfop->tr == lastfinfop->tr &&
			finfop->te == lastfinfop->te &&
			finfop->bmatrixorder == lastfinfop->bmatrixorder &&
			finfop->acqnum == lastfinfop->acqnum) {
			retcont->finfos[retcont->numfinfos] = finfos[filenum];
			retcont->numfinfos++;
			finfos[filenum] = NULL;
		    }
		    lastfinfop = finfop;
		}
		/* squeeze out empty slots (non-matching files) */
		newnumfinfos = 0;
		for (filenum = 0; filenum < numfinfos; filenum++) {
		    if (finfos[filenum] != NULL) {
			if (filenum != newnumfinfos) {
			    finfos[newnumfinfos] = finfos[filenum];
			    finfos[filenum] = NULL;
			}
			newnumfinfos++;
		    }
		}
		numfinfos = newnumfinfos;
	    }
	}
    }

#if DEBUG
    {
	fprintf(stderr, "Post-filter new series\n");
	printfinfos(stderr, numfinfos, finfos, newdimmap);
    }
#endif

    /* run through origins in each file to see if they wrap around;
     * if so, store the number of files (usually slices) per wrap in
     * originwrapfiles, and if we didn't find a time dimension earlier,
     * assume this is a time series and adjust accordingly. */
    {
	const struct dicomfileinfo * finfop0 = finfos[0];
	int numslices = 0;
	int numslicesperacq = numfinfos;
	int i;
	int checkacqnum = 1;
	if (strcmp(finfop0->manufacturer, "Bruker BioSpin MRI GmbH") == 0) {
	    checkacqnum = 0;
	}
	for (i = 1; i < numfinfos; i++) { /* skip first */
	    /* calculate distance between origins, if less than
	     * epsilon, assume wrap-around */
	    const struct dicomfileinfo * finfopi = finfos[i];
	    double rdist = finfopi->originr - finfop0->originr;
	    double adist = finfopi->origina - finfop0->origina;
	    double sdist = finfopi->origins - finfop0->origins;
	    double dist = sqrt(rdist*rdist + adist*adist + sdist*sdist);
	    const double epsilon = 0.001 * finfopi->slicespacing; /* small enough? */
	    if (numslices == 0 && dist < epsilon) {
		/* double-check that all corresponding slice origins match */
		int validwrap = 1;
	        for (int off = 0; validwrap == 1 && off < i; off++) {
		    const struct dicomfileinfo * finfopa = finfos[off];
		    for (int find = off + i; find < numfinfos; find += i) {
			const struct dicomfileinfo * finfopb = finfos[find];
			double rdist = finfopa->originr - finfopb->originr;
			double adist = finfopa->origina - finfopb->origina;
			double sdist = finfopa->origins - finfopb->origins;
			double dist = sqrt(rdist*rdist + adist*adist + sdist*sdist);
			if (dist >= epsilon) {
			    validwrap = 0;
			    break;
			}
		    }
		}
		if (validwrap == 0) {
		    fprintf(stderr, "Note: detected wrap-around in slice origins, but the slice positions were not in a consistent order!  This may be due to missing files.\n");
		    goto FAIL;
		}
		numslices = i;
		fprintf(stderr, "Note: detected wrap-around in slice origins -- assuming %d images per timepoint.\n", numslices);
	    }
	    if (numslicesperacq == numfinfos &&
		(finfopi->tr != finfop0->tr ||
		 finfopi->te != finfop0->te ||
		 finfopi->bmatrixorder != finfop0->bmatrixorder ||
		 (checkacqnum && finfopi->acqnum != finfop0->acqnum))) {
		/* some non-time dimension values change here.
		 * This is likely the place where the time dimension should
		 * reset.
		 * but make sure that these only change at those spots */
		int j;
		struct dicomfileinfo * lastacqfinfop = finfos[0];
		struct dicomfileinfo * lastfinfop = finfos[0];
		for (j = 1; j < numfinfos; j += 1) {
		    struct dicomfileinfo * finfopj = finfos[j];
		    if (j % i == 0) {
			if (finfopj->tr == lastacqfinfop->tr &&
			    finfopj->te == lastacqfinfop->te &&
			    finfopj->bmatrixorder == lastacqfinfop->bmatrixorder &&
			    finfopj->acqnum == lastacqfinfop->acqnum) {
			    /* values don't change every i files, so break */
			    break;
			}
			lastacqfinfop = finfopj;
		    } else {
			if (finfopj->tr != lastfinfop->tr ||
			    finfopj->te != lastfinfop->te ||
			    finfopj->bmatrixorder != lastfinfop->bmatrixorder ||
			    finfopj->acqnum != lastfinfop->acqnum) {
			    /* values change within a group of i files, so break */
			    break;
			}
		    }
		    lastfinfop = finfopj;
		}
		if (j == numfinfos) {
		    /* we got through everything, so assume this is the right
		     * value */
		    numslicesperacq = i;
		    fprintf(stderr, "Note: assuming %d images per acquisition.\n", numslicesperacq);
		    /* If origins haven't wrapped around yet (i.e. numslices == 0)
		     * we won't bother to create a time dimension.
		     */
		    break;
		}
	    }
	}
	if (numslices > 0 && (numslicesperacq % numslices != 0)) {
	    fprintf(stderr, "Note: assumed slices per acquisition (%d) does not divide evenly by assumed\nslices per timepoint (%d).  Not creating a time dimension.\n", numslicesperacq, numslices);
	} else if (numslices > 0 && !finfos[0]->foundtimedim) {
	    int warnedtr = 0;
	    double timepoint = 0;
	    for (i = 0; i < numfinfos; i++) {
		if (i % numslicesperacq == 0) { /* reset time */
		    timepoint = 0;
		}
		finfos[i]->t = timepoint;
		if ((i + 1) % numslices == 0) { /* wrap around */
		    if (finfos[i]->tr) {
			timepoint += finfos[i]->tr;
		    } else {
			if (!warnedtr) {
			    fprintf(stderr, "Note: couldn't find TR, so assuming 1000ms for spacing between time points\n");
			}
			timepoint += 1000;
		    }
		}
	    }
	}
	originwrapfiles = numslices;
    }

    /* normalize time dimension to origin == 0 */
    {
	int i;
	struct dicomfileinfo * finfop0 = finfos[0];
	for (i = 1; i < numfinfos; i++) { /* skip first */
	    struct dicomfileinfo * finfopi = finfos[i];
	    finfopi->t = (double)finfopi->t - (double)finfop0->t;
	}
	finfop0->t = 0;
    }

    /* now use the first "instance" file as the basis */
    {
	DcmInputFileBufferedStream myin(finfos[0]->filename);
	OFCondition errcode;
	if ( myin.status() != EC_Normal ) {
	    fprintf(stderr, FUNC ": cannot open file: %s\n", finfos[0]->filename);
	    goto FAIL;
	}

	DcmObject * reader = NULL;
	if (cont->isDataset) {
	    DcmDataset * dataset = new DcmDataset();
	    reader = dataset;
	    dfileobj = new DcmFileFormat(dataset);
	} else {
	    reader = dfileobj = new DcmFileFormat();
	}

	dfileobj->transferInit();
	reader->read(myin, EXS_Unknown, EGL_noChange);
	dfileobj->transferEnd();

	if (dfileobj->error() != EC_Normal)
	{
	    fprintf(stderr, FUNC ": error: %s: reading file: %s\n", dfileobj->error().text(), finfos[0]->filename);

	    goto FAIL;
	}

	dfileobj->loadAllDataIntoMemory();
	dfile->dataset = ((DcmFileFormat *)dfileobj)->getDataset();
	dfile->metainfo = ((DcmFileFormat *)dfileobj)->getMetaInfo();
    }

#if DEBUG
    fprintf(stderr, "Creating bxh elements...\n");
#endif
    if ((bxh = bxh_getRootElement(docp)) == NULL)
	goto FAIL;

    if ((imagedata = bxh_getDatarec(docp, "image", NULL)) == NULL)
	goto FAIL;
    if ((acquisitiondata = bxh_getAcquisitionData(docp)) == NULL)
	goto FAIL;
    
    if (finfos[0]->rowspacing != 0 || finfos[0]->colspacing != 0) {
	colspacing = finfos[0]->colspacing;
	rowspacing = finfos[0]->rowspacing;
    }

    if (strcmp(finfos[0]->manufacturer, "SIEMENS") == 0) {
	if (getPrivateDicomValue(dfile, 0x0029, 0x1020, "SIEMENS CSA HEADER", stack)) {
	    const char * begstr = "### ASCCONV BEGIN ";
	    const char * endstr = "### ASCCONV END ###";
	    Uint8 * bytes = NULL;
	    Uint32 pos;
	    Uint32 len;
	    char * lineend = NULL;
	    int begstrlen = strlen(begstr);
		
	    elem = (DcmElement *)stack.top();
	    len = elem->getLength();
	    elem->getUint8Array(bytes);
	    pos = 0;
	    while (pos < len - begstrlen) {
		char * candidate = NULL;
		Uint8 * newline = NULL;
		if ((candidate = (char *)memchr(&bytes[pos], '#', len-pos-begstrlen)) == NULL) {
		    bytes = NULL;
		    break;
		}
		pos = candidate - (char *)bytes;
		if (strncmp(candidate, begstr, strlen(begstr)) == 0 &&
		    (newline = (Uint8 *)memchr(&bytes[pos + begstrlen], '\n', len-pos)) != NULL &&
		    (newline - &bytes[pos + begstrlen] >= 3) &&
		    *(newline-1) == '#' &&
		    *(newline-2) == '#' &&
		    *(newline-3) == '#') {
		    pos += (newline + 1 - &bytes[pos]);
		    break;
		}
		pos++;
	    }
	    while (bytes != NULL && (lineend = (char *)memchr(&bytes[pos], '\n', len - pos)) != NULL) {
		OFString curline((char *)bytes + pos, (char *)lineend + 1 - ((char *)bytes + pos));
		const char * cstr = NULL;
		size_t span = 0;
		OFString name;
		OFString value;
		if (strncmp(curline.c_str(), endstr, strlen(endstr)) == 0)
		    break;
		if (curline.size() > 0 && curline[curline.size()-1] == '\n') {
		    curline.resize(curline.size()-1); /* get rid of newline */
		}
		cstr = curline.data();
		if ((span = strcspn(cstr, " \t")) == 0)
		    goto NEXTLINE;
		name = OFString(cstr, 0, span);
		cstr += span;
		if ((span = strspn(cstr, " \t")) == 0)
		    goto NEXTLINE;
		cstr += span;
		if ((span = strspn(cstr, "=")) == 0)
		    goto NEXTLINE;
		cstr += span;
		if ((span = strspn(cstr, " \t")) == 0)
		    goto NEXTLINE;
		cstr += span;
		value = OFString(cstr);
		if (value.length() >= 4 &&
		    value.substr(0, 2) == "\"\"" &&
		    value.substr(value.length() - 2, 2) == "\"\"") {
		    value = value.substr(2, value.length() - 4);
		}
		siemensmap[name] = value;
#if DEBUG
		fprintf(stderr, "siemensmap[%s] = %s\n", name.data(), siemensmap[name].data());
#endif
	      NEXTLINE:
		pos = (char *)lineend + 1 - (char *)bytes;
	    }
	}
    }

    /* need to grab acqcols and acqrows for mosaic */
    if (finfos[0]->ismosaic &&
        strcmp(finfos[0]->manufacturer, "SIEMENS") == 0 &&
	siemensmap.count("sSliceArray.asSlice[0].dPhaseFOV") &&
	siemensmap.count("sSliceArray.asSlice[0].dReadoutFOV")) {
	OFString ped("");
	if (getDicomValue(dfile, "PhaseEncodingDirection", stack)) {
	    elem = (DcmElement *)stack.top();
	    ped.clear();
	    elem->getOFString(ped, 0);
	} else if (finfos[0]->inplanephaseencodingdirection != NULL) {
	    ped = OFString(finfos[0]->inplanephaseencodingdirection);
	}
	if (ped.length() != 0) {
	    double rowfov = 0;
	    double colfov = 0;
	    if (strcmp(ped.c_str(), "ROW") == 0) {
		rowfov = OFStandard::atof(siemensmap.find("sSliceArray.asSlice[0].dReadoutFOV")->second.c_str());
		colfov = OFStandard::atof(siemensmap.find("sSliceArray.asSlice[0].dPhaseFOV")->second.c_str());
		freqfov = rowfov;
		phasefov = colfov;
	    } else if (strcmp(ped.c_str(), "COL") == 0 || strcmp(ped.c_str(), "COLUMN") == 0) {
		colfov = OFStandard::atof(siemensmap.find("sSliceArray.asSlice[0].dReadoutFOV")->second.c_str());
		rowfov = OFStandard::atof(siemensmap.find("sSliceArray.asSlice[0].dPhaseFOV")->second.c_str());
		freqfov = colfov;
		phasefov = rowfov;
	    } else {
		fprintf(stderr, "Error parsing PhaseEncodingDirection (\"%s\")\n", ped.c_str());
		goto FAIL;
	    }
	    acqcols = (Uint16)(floor((colfov/colspacing)+0.5));
	    acqrows = (Uint16)(floor((rowfov/rowspacing)+0.5));
	} else {
	    fprintf(stderr, "Error finding PhaseEncodingDirection\n");
	    goto FAIL;
	}
    } else if (getDicomValue(dfile, "AcquisitionMatrix", stack)) {
	elem = (DcmElement *)stack.top();
	Uint16 freqrows, freqcols;
	Uint16 phaserows, phasecols;
	elem->getUint16(freqrows, 0);
	elem->getUint16(freqcols, 1);
	elem->getUint16(phaserows, 2);
	elem->getUint16(phasecols, 3);
	if (freqcols == 0 && phaserows == 0) {
	    acqrows = freqrows; acqcols = phasecols;
	} else if (freqrows == 0 && phasecols == 0) {
	    acqcols = freqcols; acqrows = phaserows;
        } else if (finfos[0]->inplanephaseencodingdirection != NULL) {
	    if (strcmp(finfos[0]->inplanephaseencodingdirection, "ROW") == 0) {
		acqrows = phaserows; acqcols = freqcols;
	    } else if (strcmp(finfos[0]->inplanephaseencodingdirection, "COL") == 0 || strcmp(finfos[0]->inplanephaseencodingdirection, "COLUMN") == 0) {
		acqrows = freqrows; acqcols = phasecols;
	    } else {
		fprintf(stderr, "Unrecognized value for InPlanePhaseEncodingDirection: %s\n", finfos[0]->inplanephaseencodingdirection);
		goto FAIL;
	    }
	} else {
	    fprintf(stderr, "Can't interpret DICOM Acquisition Matrix\n");
	    goto FAIL;
	}
    }

    /*** IMAGEDATA ***/
    
#if DEBUG
    fprintf(stderr, " ...IMAGEDATA\n");
#endif

    rawdatarec = (bxhrawdatarec *)malloc(sizeof(bxhrawdatarec));
    memset(rawdatarec, '\0', sizeof(bxhrawdatarec));

    /* prepare for adding new dimensions */
    if (finfos[0]->ismosaic) {
	rawdatarec->numdims = 4;
	rawdatarec->dimensions = (bxhdimension *)malloc(sizeof(bxhdimension) * 4);
	memset(rawdatarec->dimensions, '\0', sizeof(bxhdimension)*4);
	extradimstart = 4;
    } else {
	rawdatarec->numdims = 3;
	rawdatarec->dimensions = (bxhdimension *)malloc(sizeof(bxhdimension) * 3);
	memset(rawdatarec->dimensions, '\0', sizeof(bxhdimension)*3);
	extradimstart = 3;
    }

    /* add new dimensions if necessary */
    {
	int filesingroup = 1;
	if (originwrapfiles) {
	    filesingroup = originwrapfiles;
	}
	int ndnum;
	for (ndnum = 0; numfinfos > filesingroup && newdimmap[ndnum].dimname; ndnum++) {
	    int numdatapoints = 0;
	    char ** datapoints = NULL;
	    char ** bvaluelist = NULL;
	    double firstdiffdp = 0;
	    int diffinited = 0;
	    double firstival = 0;
	    double lastival = 0;
	    double lastivaldp = 0;
	    int evenspacing = 1;
	    int evenenoughspacing = 1;
	    int minrepeats = -1;
	    int numrepeats = 0;
	    int i;

	    datapoints = (char **)malloc(sizeof(char *) * ((numfinfos / filesingroup) + 1));

	    /* make sure value changes on at least one file group */
	    for (i = 0; i < numfinfos; i += filesingroup) {
		double ival;
		double ivaldp;
		ival = finfos[i]->*(newdimmap[ndnum].memberp);
		ivaldp = ival;
		if (newdimmap[ndnum].valuep != NULL) {
		    ivaldp = finfos[i]->*(newdimmap[ndnum].valuep);
		}
		if (i == 0) {
		    firstival = ival;
		    lastival = ival;
		    lastivaldp = ivaldp;
		} else {
		    if (ival == lastival) {
			numrepeats++;
		    } else {
			if (minrepeats == -1 || numrepeats < minrepeats) {
			    minrepeats = numrepeats;
			}
			numrepeats = 0;
		    }
		}
		if (strcmp(newdimmap[ndnum].dimname, "diffusiondirection") == 0) {
		    if (bvalueos.length() != 0) {
			bvalueos.append(" ");
		    }
		    sprintf(&numbuf[0], "%.15g", finfos[i]->bvalue);
		    bvalueos.append(&numbuf[0]);
		    if (bvaluelist == NULL) {
			bvaluelist = (char **)malloc(sizeof(char *) * ((numfinfos / filesingroup) + 1));
		    }
		    bvaluelist[numdatapoints] = strdup(&numbuf[0]);
		    sprintf(&numbuf[0], "%.15g %.15g %.15g", finfos[i]->scalex, finfos[i]->scaley, finfos[i]->scalez);
		    evenspacing = 0; /* force writing of datapoints array */
		    evenenoughspacing = 0;
		} else {
		    sprintf(&numbuf[0], "%.15g", ivaldp);
		}
		datapoints[numdatapoints] = strdup(&numbuf[0]);
		numdatapoints++;
		if (numdatapoints > 1) {
		    if (!diffinited) {
			firstdiffdp = ivaldp - lastivaldp;
			diffinited = 1;
		    }
#if DEBUG
		    fprintf(stderr, " dim %s: firstdiff %g, curdiff %g, abs(curdiff - firstdiff) %g\n", newdimmap[ndnum].dimname, firstdiffdp, ivaldp - lastivaldp, fabs((ivaldp - lastivaldp) - firstdiffdp));
#endif
		    if (ivaldp - lastivaldp != firstdiffdp) {
#if DEBUG
			fprintf(stderr, "   disabling evenspacing\n");
#endif
			evenspacing = 0;
			if (firstdiffdp != 0 && fabs(((ivaldp - lastivaldp) - firstdiffdp) / firstdiffdp) > 1e-6) {
#if DEBUG
			    fprintf(stderr, "   disabling evenenoughspacing\n");
#endif
			    evenenoughspacing = 0;
			}
		    }
		}
		lastival = ival;
		lastivaldp = ivaldp;
	    }
	    if (minrepeats == -1 || numrepeats < minrepeats) {
		minrepeats = numrepeats;
	    }
	    if (minrepeats == 0) {
		/* changed at least once -- we can add it as a dimension */
		bxhdimension * newdim = NULL;

		rawdatarec->dimensions = (bxhdimension *)realloc(rawdatarec->dimensions, sizeof(bxhdimension) * (rawdatarec->numdims + 1));
		newdim = &rawdatarec->dimensions[rawdatarec->numdims];
		rawdatarec->numdims++;
		memset(newdim, '\0', sizeof(bxhdimension));
		newdim->type = strdup(newdimmap[ndnum].dimname);
		if (newdimmap[ndnum].units)
		    newdim->units = strdup(newdimmap[ndnum].units);
		else
		    newdim->units = NULL;
		newdim->size = numdatapoints;
		if (evenspacing) {
		    newdim->origin = firstival;
		    newdim->spacing = firstdiffdp;
		    for (i = 0; i < numdatapoints; i++) {
			free(datapoints[i]); datapoints[i] = NULL;
		    }
		    free(datapoints); datapoints = NULL;
		} else {
		    bxhdatapoints * dps = NULL;
		    newdim->origin = firstival;
		    newdim->spacing = 0;
		    if (evenenoughspacing) {
			newdim->spacing =
			    (lastival - firstival) / (numdatapoints - 1);
		    }
		    newdim->dpstructs = (bxhdatapoints *)realloc(newdim->dpstructs, sizeof(bxhdatapoints) * (newdim->numdpstructs + 1));
		    dps = &newdim->dpstructs[newdim->numdpstructs];
		    dps->label = strdup(newdimmap[ndnum].dimname);
		    dps->values = datapoints;
		    dps->numvalues = numdatapoints;
		    newdim->numdpstructs++;
		    if (bvaluelist != NULL) {
			newdim->dpstructs = (bxhdatapoints *)realloc(newdim->dpstructs, sizeof(bxhdatapoints) * (newdim->numdpstructs + 1));
			dps = &newdim->dpstructs[newdim->numdpstructs];
			dps->label = strdup("bvalues");
			dps->values = bvaluelist;
			dps->numvalues = numdatapoints;
			bvaluelist = NULL; /* so we know we don't have to free it */
			newdim->numdpstructs++;
		    }
		}
		filesingroup *= numdatapoints;
		fprintf(stderr, "Added extra dimension: dimnum=%d dimname=%s dimsize=%u\n", rawdatarec->numdims - 1, newdim->type, (unsigned int)newdim->size);
	    } else {
		for (i = 0; i < numdatapoints; i++) {
		    free(datapoints[i]); datapoints[i] = NULL;
		}
		free(datapoints); datapoints = NULL;
	    }
	    if (bvaluelist != NULL) {
		for (i = 0; i < numdatapoints; i++) {
		    free(bvaluelist[i]); bvaluelist[i] = NULL;
		}
		free(bvaluelist); bvaluelist = NULL;
	    }
	}

	for (ndnum = 0; newdimmap[ndnum].dimname; ndnum++) {
	    int dimnum;
	    const char * dimname = newdimmap[ndnum].dimname;
	    for (dimnum = extradimstart; dimnum < rawdatarec->numdims; dimnum++) {
		const char * dimtype = rawdatarec->dimensions[dimnum].type;
		if (strcmp(dimname, dimtype) == 0)
		    *(newdimmap[ndnum].dim) = &rawdatarec->dimensions[dimnum];
	    }
	}
    }

    /* Fix diffusion directions dimension */
    if (dimdiffdir) {
	free(dimdiffdir->type);
	dimdiffdir->type = strdup("diffusiondirection");
	dimdiffdir->origin = 0;
	dimdiffdir->spacing = 0;
	dimdiffdir->gap = 0;
    }

    /* now we can assign dimx/dimy/etc. (dim. array won't be reallocated) */
    if (finfos[0]->ismosaic) {
	int numslices;
	int sind;
	numslices = (int)OFStandard::atof(siemensmap.find("sSliceArray.lSize")->second.c_str());
	if (getPrivateDicomValue(dfile, 0x0019, 0x100a, "SIEMENS MR HEADER", stack)) {
	    Uint16 numslices2;
	    elem = (DcmElement *)stack.top();
            if (elem->getVR() != EVR_UN) {
		elem->getUint16(numslices2, 0);
		numslices = numslices2;
	    }
	}
	dimx = &rawdatarec->dimensions[0];
	dimztiled = &rawdatarec->dimensions[1];
	dimy = &rawdatarec->dimensions[2];
	dimz = &rawdatarec->dimensions[3];
	memset(dimx, '\0', sizeof(bxhdimension));
	memset(dimztiled, '\0', sizeof(bxhdimension));
	memset(dimy, '\0', sizeof(bxhdimension));
	memset(dimz, '\0', sizeof(bxhdimension));
	dimx->type = strdup("x");
	dimztiled->type = strdup("z-split1");
	dimy->type = strdup("y");
	dimz->type = strdup("z-split2");
	dimx->size = acqcols;
	dimztiled->size = finfos[0]->columns / acqcols;
	dimy->size = acqrows;
	dimz->size = numfinfos * (finfos[0]->rows / acqrows);
	dimz->numoutputselect = numslices;
	dimz->outputselect = (int *)malloc(sizeof(int)*numslices);
	for (sind = 0; sind < numslices; sind++) {
	    dimz->outputselect[sind] = sind;
	}
    } else {
	dimx = &rawdatarec->dimensions[0];
	dimy = &rawdatarec->dimensions[1];
	dimz = &rawdatarec->dimensions[2];
	memset(dimx, '\0', sizeof(bxhdimension));
	memset(dimy, '\0', sizeof(bxhdimension));
	memset(dimz, '\0', sizeof(bxhdimension));
	dimx->type = strdup("x");
	dimy->type = strdup("y");
	dimz->type = strdup("z");
	dimx->size = finfos[0]->columns;
	dimy->size = finfos[0]->rows;
	dimz->size = numfinfos;
    }

    {
	/* fix z size to account for new dimensions (e.g. 't' [time]) */
	int dimnum;
	for (dimnum = extradimstart; dimnum < rawdatarec->numdims; dimnum++) {
	    int dimsize = rawdatarec->dimensions[dimnum].size;
	    if ((double)((int)dimz->size / (int)dimsize) !=
		(double)((double)dimz->size / (double)dimsize)) {
		fprintf(stderr, FUNC ": extra dimensions (dimnum=%d dimname=%s files=%u dimsize=%u) don't divide into equal numbers of files!\n", dimnum, rawdatarec->dimensions[dimnum].type, (unsigned int)dimz->size, (unsigned int)dimsize);
		goto FAIL;
	    }
	    dimz->size /= dimsize;
	}
    }

    if (finfos[0]->foundtr) {
	/* check if slice acquisition time might give us a predictable slice
	 * acquisition order.  Note that no manufacturers actually use this
	 * field for this purpose, but this is to support in-house tools that
	 * encode slice acquisition timing into this field.
	 */
	double epsilon = 0.000001;
	OFString sliceorderstr;
	struct indtime {
	    int ind;
	    double time;
	};
	indtime * indtimes = (indtime *)malloc(sizeof(indtime) * dimz->size);
	bxhdatapoints * dpstructp = NULL;
	double trsec = finfos[0]->tr / 1000.0;
	bool foundequal = false;

	/* first normalize each slice time to the minimum acqtime of the volume */
	for (int volnum = 0; volnum < numfinfos / dimz->size; volnum++) {
	    int volfinfonum = (volnum * dimz->size);
	    double basetime = finfos[volfinfonum]->sliceacqtime;
	    for (int slicenum = 1; slicenum < dimz->size; slicenum++) {
		basetime = MIN(basetime, finfos[volfinfonum + slicenum]->sliceacqtime);
	    }
	    for (int slicenum = 0; slicenum < dimz->size; slicenum++) {
		finfos[volfinfonum + slicenum]->sliceacqtime -= basetime;
#if DEBUG
		fprintf(stderr, "finfo %d: sliceacqtime=%f\n", volfinfonum + slicenum, finfos[volfinfonum + slicenum]->sliceacqtime);
#endif
	    }
	}

	/* now verify that slice timings are consistent */
	bool allsame = true;
	char ** datapoints = (char **)malloc(sizeof(char *) * dimz->size);
	for (int slicenum = 0; slicenum < dimz->size; slicenum++) {
	    datapoints[slicenum] = NULL;
	}

	/* Make sure slice timings are not all the same! */
	for (int finfonum = 0; finfonum < numfinfos; finfonum++) {
	    if (finfos[finfonum]->sliceacqtime != finfos[0]->sliceacqtime) {
		allsame = false;
	    }
	}
	if (allsame) {
	    goto SLICETIMEFAIL;
	}

	// check that each volume has the same slice timing profile
	for (int slicenum = 0; slicenum < dimz->size; slicenum++) {
	    int finfonum = slicenum;
	    double checktime = finfos[finfonum]->sliceacqtime;
	    if (checktime >= trsec) {
		fprintf(stderr, FUNC ": found slice timing (%g) greater than TR (%g),so won't populate sliceorder or acquisitiontimeindex fields using AcquisitionTime\n", checktime, trsec);
		goto SLICETIMEFAIL;
	    }
	    for (finfonum += dimz->size; finfonum < numfinfos; finfonum += dimz->size) {
		if (fabs(finfos[finfonum]->sliceacqtime - checktime) > epsilon) {
		    fprintf(stderr, FUNC ": can't find consistent slice timing, won't populate sliceorder or acquisitiontimeindex fields using AcquisitionTime\n");
		    goto SLICETIMEFAIL;
		}
	    }
	}

	/* slice timings are consistent, put them in datapoints struct */
	for (int slicenum = 0; slicenum < dimz->size; slicenum++) {
	    /* adjust range [0, TR) ->  [1, numslices + 1) */
	    sprintf(&numbuf[0], "%g", ((finfos[slicenum]->sliceacqtime / trsec) * dimz->size) + 1);
	    datapoints[slicenum] = strdup(numbuf);
	}
	dimz->dpstructs = (bxhdatapoints *)realloc(dimz->dpstructs, sizeof(bxhdatapoints) * (dimz->numdpstructs + 1));
	dpstructp = &dimz->dpstructs[dimz->numdpstructs];
	dimz->numdpstructs++;
	dpstructp->label = strdup("acquisitiontimeindex");
	dpstructp->values = datapoints;
	dpstructp->numvalues = dimz->size;

	/* sort slice times and create an appropriate sliceorder field */
	for (int slicenum = 0; slicenum < dimz->size; slicenum++) {
	    indtimes[slicenum].time = finfos[slicenum]->sliceacqtime;
	    indtimes[slicenum].ind = slicenum;
	}
	/* bubble-sort */
	for (int i = 0; i < dimz->size; i++) {
	    for (int j = i + 1; j < dimz->size; j++) {
		int doswap = 0;
		if (indtimes[i].time == indtimes[j].time) {
		    foundequal = true;
		} else if (indtimes[i].time > indtimes[j].time) {
		    doswap = 1;
		}
		if (doswap) {
		    indtime swap = indtimes[i];
		    indtimes[i] = indtimes[j];
		    indtimes[j] = swap;
		}
	    }
	}
	if (!foundequal) {
	    // we only do sliceorder if there are no repeats in the slice timing
	    // in which case there is no sliceorder string that makes sense
	    for (int slicenum = 0; slicenum < dimz->size; slicenum++) {
		if (slicenum != 0) {
		    sliceorderstr += ",";
		}
		sprintf(&numbuf[0], "%d", indtimes[slicenum].ind + 1); /* indexed by one */
		sliceorderstr += numbuf;
	    }
	    if (sliceorderstr.length() > 0) {
		if (bxh_appendChildElement(acquisitiondata, "sliceorder", sliceorderstr.c_str()) != 0)
		    goto FAIL;
	    }
	}
	
	goto SLICETIMESUCCESS;

    SLICETIMEFAIL:
	for (int slicenum = 0; slicenum < dimz->size; slicenum++) {
	    if (datapoints[slicenum] != NULL) {
		free(datapoints[slicenum]);
		datapoints[slicenum] = NULL;
	    }
	}
	free(datapoints); datapoints = NULL;
    SLICETIMESUCCESS:
	if (indtimes != NULL) {
	    free(indtimes);
	    indtimes = NULL;
	}
    }

    /* calculate slice spacing here because we now know true number of slices */
    if (finfos[0]->ismosaic) {
	double epsilon = 0.005;
	int numslices = 0;
	double firstr = 0, firsta = 0, firsts = 0;
	double lastr = 0, lasta = 0, lasts = 0;
	static char tmpname[64];
	std::map<OFString,OFString>::iterator tmpiter;

	numslices = (int)OFStandard::atof(siemensmap.find("sSliceArray.lSize")->second.c_str());
	sprintf(&tmpname[0], "sSliceArray.asSlice[%d].sPosition.dSag", 0);
	if ((tmpiter = siemensmap.find(tmpname)) != siemensmap.end())
	    firstr = -1.0 * OFStandard::atof(tmpiter->second.c_str()); /*lps->ras*/
	sprintf(&tmpname[0], "sSliceArray.asSlice[%d].sPosition.dCor", 0);
	if ((tmpiter = siemensmap.find(tmpname)) != siemensmap.end())
	    firsta = -1.0 * OFStandard::atof(tmpiter->second.c_str()); /*lps->ras*/
	sprintf(&tmpname[0], "sSliceArray.asSlice[%d].sPosition.dTra", 0);
	if ((tmpiter = siemensmap.find(tmpname)) != siemensmap.end())
	    firsts = OFStandard::atof(tmpiter->second.c_str());

	sprintf(&tmpname[0], "sSliceArray.asSlice[%d].sPosition.dSag", numslices-1);
	if ((tmpiter = siemensmap.find(tmpname)) != siemensmap.end())
	    lastr = -1.0 * OFStandard::atof(tmpiter->second.c_str()); /*lps->ras*/
	sprintf(&tmpname[0], "sSliceArray.asSlice[%d].sPosition.dCor", numslices-1);
	if ((tmpiter = siemensmap.find(tmpname)) != siemensmap.end())
	    lasta = -1.0 * OFStandard::atof(tmpiter->second.c_str()); /*lps->ras*/
	sprintf(&tmpname[0], "sSliceArray.asSlice[%d].sPosition.dTra", numslices-1);
	if ((tmpiter = siemensmap.find(tmpname)) != siemensmap.end())
	    lasts = OFStandard::atof(tmpiter->second.c_str());

	if (numslices == 0) {
	    fprintf(stderr, "Warning: cannot find number of slices of mosaic in Siemens private header.  Not double-checking slice spacing.\n");
	    calcslicespacing = finfos[0]->slicespacing;
	} else {
	    calcslicespacing =
		sqrt((lastr - firstr) * (lastr - firstr) +
		     (lasta - firsta) * (lasta - firsta) +
		     (lasts - firsts) * (lasts - firsts))
		/ (double)(numslices - 1);
	    if (fabs(calcslicespacing - finfos[0]->slicespacing) > epsilon) {
		fprintf(stderr, "Warning: reported slice spacing (%g) is different from calculated spacing (%g) by more than %g.  Using calculated number.\n", finfos[0]->slicespacing, calcslicespacing, epsilon);
	    }
	}
    } else if (dimz->size > 1) {
	double epsilon = 0.005;
	int numslices = 0;
	double firstr = 0, firsta = 0, firsts = 0;
	double lastr = 0, lasta = 0, lasts = 0;

	numslices = dimz->size;
	firstr = finfos[0]->originr;
	firsta = finfos[0]->origina;
	firsts = finfos[0]->origins;
	lastr = finfos[numslices-1]->originr;
	lasta = finfos[numslices-1]->origina;
	lasts = finfos[numslices-1]->origins;
	calcslicespacing =
	    sqrt((lastr - firstr) * (lastr - firstr) +
		 (lasta - firsta) * (lasta - firsta) +
		 (lasts - firsts) * (lasts - firsts))
	    / (numslices - 1);
	if (fabs(calcslicespacing - finfos[0]->slicespacing) > epsilon) {
	    fprintf(stderr, "Warning: reported slicespacing (%g) is different from calculated slicespacing (%g) by more than %g.  Using calculated number.\n", finfos[0]->slicespacing, calcslicespacing, epsilon);
	}
    }
    if (finfos[0]->slicespacing == finfos[0]->slicethickness) {
	finfos[0]->slicethickness = calcslicespacing;
    }

    dimx->spacing = colspacing;
    dimy->spacing = rowspacing;
    dimx->units = strdup("mm");
    dimy->units = strdup("mm");

    dimz->spacing = calcslicespacing;
    dimz->gap = (double)calcslicespacing - (double)finfos[0]->slicethickness;
    dimz->units = strdup("mm");

    if (fabs(finfos[0]->rowsr) + fabs(finfos[0]->rowsa) + fabs(finfos[0]->rowss) != 0) {
	Float64 Rr, Ra, Rs; /* row */
	Float64 Cr, Ca, Cs; /* column */
	Float64 Nr, Na, Ns; /* normal */

	Rr = finfos[0]->rowsr;
	Ra = finfos[0]->rowsa;
	Rs = finfos[0]->rowss;
	Cr = finfos[0]->colsr;
	Ca = finfos[0]->colsa;
	Cs = finfos[0]->colss;
	/* DICOM coordinates are in lps, so correct it here */
	Rr *= -1.0;
	Ra *= -1.0;
	Cr *= -1.0;
	Ca *= -1.0;

	Nr = Ra * Cs - Rs * Ca;
	Na = Rs * Cr - Rr * Cs;
	Ns = Rr * Ca - Ra * Cr;

	if (finfos[0]->ismosaic || numfinfos > 1) {
	    /* get normal in the correct data direction */
	    Float64 newNr = 0, newNa = 0, newNs = 0;
	    double Nlen;
	    double epsilon = 0.05;
	    if (finfos[0]->ismosaic) {
		std::map<OFString,OFString>::iterator tmpiter;
		if ((tmpiter = siemensmap.find("sSliceArray.asSlice[0].sNormal.dSag")) != siemensmap.end())
		    newNr = -1.0 * OFStandard::atof(tmpiter->second.c_str()); /*lps->ras*/
		if ((tmpiter = siemensmap.find("sSliceArray.asSlice[0].sNormal.dCor")) != siemensmap.end())
		    newNa = -1.0 * OFStandard::atof(tmpiter->second.c_str()); /*lps->ras*/
		if ((tmpiter = siemensmap.find("sSliceArray.asSlice[0].sNormal.dTra")) != siemensmap.end())
		    newNs = OFStandard::atof(tmpiter->second.c_str());
		if (newNr == 0 && newNa == 0 && newNs == 0) {
		    fprintf(stderr, "Warning: Can't find slice normal in Siemens private header, so we don't know what the slice direction is.  Using the cross-product of row and column directions, ***BUT THIS MAY BE WRONG***!\n");
		    newNr = Nr;
		    newNa = Na;
		    newNs = Ns;
		}
	    } else {
		newNr = finfos[numfinfos-1]->originr - finfos[0]->originr;
		newNa = finfos[numfinfos-1]->origina - finfos[0]->origina;
		newNs = finfos[numfinfos-1]->origins - finfos[0]->origins;
	    }

	    Nlen = sqrt((double)((newNr * newNr) + (newNa * newNa) + (newNs * newNs)));
	    double Nmax = (Nr > Na) ? Nr : Na;
	    Nmax = (Nmax > Ns) ? Nmax : Ns;
	    if (Nlen > (epsilon * Nmax)) {
	        // protect against case where there is only one z slice, and
	        // therefore calculated normal is 0
	        newNr /= Nlen;
		newNa /= Nlen;
		newNs /= Nlen;
		if (!((fabs(newNr - Nr) < epsilon && fabs(newNa - Na) < epsilon && fabs(newNs - Ns) < epsilon) ||
		      (fabs(newNr + Nr) < epsilon && fabs(newNa + Na) < epsilon && fabs(newNs + Ns) < epsilon))) {
		    fprintf(stderr, "Warning: slice origins don't lie along a vector parallel to normal\n");
		    fprintf(stderr, " Cross-product of rows & columns: (%.15g, %.15g, %.15g)\n", Nr, Na, Ns);
		    fprintf(stderr, " Origins vector: (%.15g, %.15g, %.15g)\n", newNr, newNa, newNs);
		    fprintf(stderr, " Continuing anyway...\n");
		}
		Nr = newNr;
		Na = newNa;
		Ns = newNs;
	    }
	}

	if (fabs(Rr) > fabs(Cr) && fabs(Rr) > fabs(Nr)) {
	    dimr = dimx;
	    if (fabs(Ca) > fabs(Ra) && fabs(Ca) > fabs(Na)) {
		dima = dimy; dims = dimz;
	    } else {
		dima = dimz; dims = dimy;
	    }
	} else if (fabs(Cr) > fabs(Rr) && fabs(Cr) > fabs(Nr)) {
	    dimr = dimy;
	    if (fabs(Ra) > fabs(Ca) && fabs(Ra) > fabs(Na)) {
		dima = dimx; dims = dimz;
	    } else {
		dima = dimz; dims = dimx;
	    }
	} else {
	    dimr = dimz;
	    if (fabs(Ra) > fabs(Ca) && fabs(Ra) > fabs(Na)) {
		dima = dimx; dims = dimy;
	    } else {
		dima = dimy; dims = dimx;
	    }
	}

	dimx->direction = (double *)malloc(sizeof(double) * 3);
	dimx->direction[0] = Rr;
	dimx->direction[1] = Ra;
	dimx->direction[2] = Rs;
	dimy->direction = (double *)malloc(sizeof(double) * 3);
	dimy->direction[0] = Cr;
	dimy->direction[1] = Ca;
	dimy->direction[2] = Cs;
	dimz->direction = (double *)malloc(sizeof(double) * 3);
	dimz->direction[0] = Nr;
	dimz->direction[1] = Na;
	dimz->direction[2] = Ns;
    }

    if (finfos[0]->ismosaic) {
	/* mosaic DICOM origin is incorrect; use values in ASCII header */
	std::map<OFString,OFString>::iterator tmpiter;
	double centerR = 0;
	double centerA = 0;
	double centerS = 0;
	if ((tmpiter = siemensmap.find("sSliceArray.asSlice[0].sPosition.dSag")) != siemensmap.end())
	    centerR = -1 * OFStandard::atof(tmpiter->second.c_str()); /* lps->ras */
	if ((tmpiter = siemensmap.find("sSliceArray.asSlice[0].sPosition.dCor")) != siemensmap.end())
	    centerA = -1 * OFStandard::atof(tmpiter->second.c_str()); /* lps->ras */
	if ((tmpiter = siemensmap.find("sSliceArray.asSlice[0].sPosition.dTra")) != siemensmap.end())
	    centerS = OFStandard::atof(tmpiter->second.c_str());
	dimr->origin = centerR + (-0.5 * ((dimx->direction[0] * colspacing * (acqcols - 1)) + (dimy->direction[0] * rowspacing * (acqrows - 1))));
	dima->origin = centerA + (-0.5 * ((dimx->direction[1] * colspacing * (acqcols - 1)) + (dimy->direction[1] * rowspacing * (acqrows - 1))));
	dims->origin = centerS + (-0.5 * ((dimx->direction[2] * colspacing * (acqcols - 1)) + (dimy->direction[2] * rowspacing * (acqrows - 1))));
    } else if (dimr != NULL && dima != NULL && dims != NULL) {
	dimr->origin = finfos[0]->originr;
	dima->origin = finfos[0]->origina;
	dims->origin = finfos[0]->origins;
    }

    if (finfos[0]->ismosaic) {
	std::map<OFString,OFString>::iterator tmpiter;
	if (((tmpiter = siemensmap.find("sSliceArray.ucImageNumbSag")) != siemensmap.end() && dimz == dimr) ||
	    ((tmpiter = siemensmap.find("sSliceArray.ucImageNumbCor")) != siemensmap.end() && dimz == dima) ||
	    ((tmpiter = siemensmap.find("sSliceArray.ucImageNumbTra")) != siemensmap.end() && dimz == dims)) {
	    if (strcmp(tmpiter->second.c_str(), "0x1") == 0) {
		/* this indicates a slice reversal, so z-dimension
		 * direction must be flipped */
		dimz->direction[0] *= -1;
		dimz->direction[1] *= -1;
		dimz->direction[2] *= -1;
	    }
	}
    }


    if (cont->cornerorigin &&
	strcmp(finfos[0]->manufacturer, "GE MEDICAL SYSTEMS") == 0 &&
	dimr != NULL && dima != NULL && dims != NULL) {
	/* fix origin to point to center of voxels
	 * (x and y dimensions use edge of bounding box in GE) */
	dimr->origin +=
	    (dimx->spacing / 2.0) * dimx->direction[0] +
	    (dimy->spacing / 2.0) * dimy->direction[0];
	dima->origin +=
	    (dimx->spacing / 2.0) * dimx->direction[1] +
	    (dimy->spacing / 2.0) * dimy->direction[1];
	dims->origin +=
	    (dimx->spacing / 2.0) * dimx->direction[2] +
	    (dimy->spacing / 2.0) * dimy->direction[2];
    }

    if (getDicomValue(dfile, "PixelRepresentation", stack)) {
	char pixeltype[16] = "";
	elem = (DcmElement *)stack.top();
	elem->getUint16(val.ui16, 0);
	if (val.ui16 == 0) {
	    strcat(pixeltype, "uint");
	} else if (val.ui16 == 1) {
	    strcat(pixeltype, "int");
	} else {
	    fprintf(stderr, "Bad value for DICOM field PixelRepresentation: %u", (unsigned int)val.ui16);
	    goto FAIL;
	}
	if (finfos[0]->bitsallocated == 8) {
	    strcat(pixeltype, "8");
	} else if (finfos[0]->bitsallocated == 16) {
	    strcat(pixeltype, "16");
	} else if (finfos[0]->bitsallocated == 32) {
	    strcat(pixeltype, "32");
	} else {
	    fprintf(stderr, "Only 8, 16, or 32 allowed for DICOM field BitsAllocated: %u", (unsigned int)finfos[0]->bitsallocated);
	    goto FAIL;
	}
	if (getDicomValue(dfile, "PhotometricInterpretation", stack)) {
	    elem = (DcmElement *)stack.top();
	    tmpos.clear();
	    elem->getOFString(tmpos, 0);
	    if (strcmp(tmpos.c_str(), "RGB") == 0) {
		if (strcmp(pixeltype, "uint8") != 0) {
		    fprintf(stderr, FUNC ": RGB data is only supported if each pixel value is uint8! (file: %s)\n", finfos[0]->filename);
		    goto FAIL;
		}
		strcpy(pixeltype, "rgb24");
	    }
	}
	rawdatarec->elemtype = strdup(pixeltype);
    }

    {
	DcmXfer xfer(dfile->dataset->getOriginalXfer());
	if (xfer.isBigEndian()) {
	    rawdatarec->msbfirstfrags = 1;
	} else if (xfer.isLittleEndian()) {
	    rawdatarec->msbfirstfrags = 0;
	} else {
	    fprintf(stderr, "Can't determine byte order from DICOM dataset!\n");
	    goto FAIL;
	}
    }

    /* add frags */
    {
	int i;
	bxh_datarec_frags_free(rawdatarec);
	for (i = 0; i < numfinfos; i++) {
	    bxh_datarec_addfrag(rawdatarec, finfos[i]->filename, finfos[i]->fileoffset, finfos[0]->rows*finfos[0]->columns*finfos[0]->samplesperpixel*finfos[0]->bitsallocated/8, basepath, 1);
	}
    }

    /* measurement frame maps from diffusion gradient coordinate
     * system to patient space.  Assume diffusion directions are
     * relative to the image bounding box, but see exception(s) below.
     */
    if (dimdiffdir && finfos[0]->frameidentity != 0) {
	dimdiffdir->measurementframe = (double *)malloc(sizeof(double)*9);
	dimdiffdir->measurementframe[0] = 1;
	dimdiffdir->measurementframe[1] = 0;
	dimdiffdir->measurementframe[2] = 0;
	dimdiffdir->measurementframe[3] = 0;
	dimdiffdir->measurementframe[4] = 1;
	dimdiffdir->measurementframe[5] = 0;
	dimdiffdir->measurementframe[6] = 0;
	dimdiffdir->measurementframe[7] = 0;
	dimdiffdir->measurementframe[8] = 1;
	dimdiffdir->measurementframeversion = strdup("2");
    } else if (dimdiffdir &&
	       dimx && dimy && dimz &&
	       dimx->direction && dimy->direction && dimz->direction) {
	dimdiffdir->measurementframe = (double *)malloc(sizeof(double)*9);
	dimdiffdir->measurementframe[0] = dimx->direction[0];
	dimdiffdir->measurementframe[1] = dimx->direction[1];
	dimdiffdir->measurementframe[2] = dimx->direction[2];
	dimdiffdir->measurementframe[3] = dimy->direction[0];
	dimdiffdir->measurementframe[4] = dimy->direction[1];
	dimdiffdir->measurementframe[5] = dimy->direction[2];
	dimdiffdir->measurementframe[6] = dimz->direction[0];
	dimdiffdir->measurementframe[7] = dimz->direction[1];
	dimdiffdir->measurementframe[8] = dimz->direction[2];
	dimdiffdir->measurementframeversion = strdup("2");
    }

    /*** ACQUISITIONDATA ***/

#if DEBUG
    fprintf(stderr, " ...ACQUISITIONDATA\n");
#endif

#if DEBUG
    fprintf(stderr, "     StudyID\n");
#endif
    if (getDicomValue(dfile, "StudyID", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "examnumber", tmpos.c_str()) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     PatientID\n");
#endif
    if (getDicomValue(dfile, "PatientID", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "studyid", tmpos.c_str()) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     SeriesNumber\n");
#endif
    if (getDicomValue(dfile, "SeriesNumber", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "seriesnumber", tmpos.c_str()) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     AcquisitionNumber\n");
#endif
    if (getDicomValue(dfile, "AcquisitionNumber", stack)) {
	if (dimacqnum) {
	    static char buf[128];
	    sprintf(&buf[0], "%u-%u", (unsigned int)dimacqnum->origin, (unsigned int)(dimacqnum->origin + (dimacqnum->spacing * (dimacqnum->size - 1))));
	    if (bxh_appendChildElement(acquisitiondata, "runnumber", &buf[0]) != 0)
		goto FAIL;
	} else {
	    elem = (DcmElement *)stack.top();
	    tmpos.clear();
	    elem->getOFString(tmpos, 0);
	    if (bxh_appendChildElement(acquisitiondata, "runnumber", tmpos.c_str()) != 0)
		goto FAIL;
	}
    }

#if DEBUG
    fprintf(stderr, "     InstitutionName\n");
#endif
    if (getDicomValue(dfile, "InstitutionName", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "institution", tmpos.c_str()) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     InstitutionAddress\n");
#endif
    if (getDicomValue(dfile, "InstitutionAddress", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "institutionaddress", tmpos.c_str()) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     InstitutionalDepartmentName\n");
#endif
    if (getDicomValue(dfile, "InstitutionalDepartmentName", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "institutionaldepartmentname", tmpos.c_str()) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     ProtocolName\n");
#endif
    {
	OFString protostr;
	if (getDicomValue(dfile, "ProtocolName", stack)) {
	    elem = (DcmElement *)stack.top();
	    tmpos.clear();
	    elem->getOFString(tmpos, 0);
	    protostr = tmpos;
	}
	if (strcmp(finfos[0]->manufacturer, "GE MEDICAL SYSTEMS") == 0) {
	    OFString paradigmdesc;
	    OFString paradigmname;
	    /* GEMS_ParadigmDescription */
	    if (getPrivateDicomValue(dfile, 0x0043, 0x1071, "GEMS_PARM_01", stack)) {
		elem = (DcmElement *)stack.top();
		tmpos.clear();
		elem->getOFString(tmpos, 0);
		paradigmdesc = tmpos;
	    }
	    /* GEMS_ParadigmName */
	    if (getPrivateDicomValue(dfile, 0x0043, 0x1070, "GEMS_PARM_01", stack)) {
		elem = (DcmElement *)stack.top();
		tmpos.clear();
		elem->getOFString(tmpos, 0);
		paradigmname = tmpos;
	    }
	    if (paradigmname.length() > 0) {
		protostr += " (";
		protostr += tmpos;
	    }
	    if (paradigmdesc.length() > 0) {
		protostr += " [";
		protostr += paradigmdesc;
		protostr += "] ";
	    }
	    if (paradigmname.length() > 0) {
		protostr += ") ";
	    }
	} else if (strcmp(finfos[0]->manufacturer, "SIEMENS") == 0) {
	    std::map<OFString,OFString>::iterator tmpiter;
	    tmpiter = siemensmap.find("tProtocolName");
	    if (tmpiter != siemensmap.end()) {
		protostr += " (";
		protostr += tmpiter->second;
		protostr += ") ";
	    }
	}
	if (protostr.length() > 0) {
	    if (bxh_appendChildElement(acquisitiondata, "protocolname", protostr.c_str()) != 0)
		goto FAIL;
	}
    }

#if DEBUG
    fprintf(stderr, "     pulse sequence name\n");
#endif
    /* GEMS_PulseSequenceName */
    if (getPrivateDicomValue(dfile, 0x0019, 0x109c, "GEMS_ACQU_01", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "psdname", tmpos.c_str()) != 0)
	    goto FAIL;
    }
    /* GEMS_InternalPulseSequenceName */
    if (getPrivateDicomValue(dfile, 0x0019, 0x109e, "GEMS_ACQU_01", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "psdinternalname", tmpos.c_str()) != 0)
	    goto FAIL;
    }
    /* GEMS_PulseSequenceDate */
    if (getPrivateDicomValue(dfile, 0x0019, 0x109d, "GEMS_ACQU_01", stack)) {
	OFDateTime odt;
	elem = (DcmElement *)stack.top();
	((DcmDateTime *)elem)->getOFDateTime(odt, 0);
	odt.getDate().getISOFormattedDate(tmpos);
	if (bxh_appendChildElement(acquisitiondata, "psddate", tmpos.c_str()) != 0)
	((DcmTime *)elem)->getISOFormattedTime(tmpos, 0);
	odt.getTime().getISOFormattedTime(tmpos);
	if (bxh_appendChildElement(acquisitiondata, "psdtime", tmpos.c_str()) != 0)
	    goto FAIL;
    }
    /* GEAutoPrescanCenterFrequency */
    if (getPrivateDicomValue(dfile, 0x0019, 0x1093, "GEMS_ACQU_01", stack)) {
	tmpos.clear();
	elem = (DcmElement *)stack.top();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "autoprescancenterfrequency", tmpos.c_str()) != 0)
	    goto FAIL;
    }
    /* GEAutoPrescanTransmitGain */
    if (getPrivateDicomValue(dfile, 0x0019, 0x1094, "GEMS_ACQU_01", stack)) {
	tmpos.clear();
	elem = (DcmElement *)stack.top();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "autoprescantransmitgain", tmpos.c_str()) != 0)
	    goto FAIL;
    }
    /* Siemens tSequenceFileName */
    if (strcmp(finfos[0]->manufacturer, "SIEMENS") == 0) {
	std::map<OFString,OFString>::iterator tmpiter;
	tmpiter = siemensmap.find("tSequenceFileName");
	if (tmpiter != siemensmap.end()) {
	    if (bxh_appendChildElement(acquisitiondata, "psdname", tmpiter->second.c_str()) != 0)
		goto FAIL;
	}
    }

#if DEBUG
    fprintf(stderr, "     listed acquisition parameters\n");
#endif
    if (copyacqparamlist == NULL) {
	/* bubble-sort the acqparam entries */
	for (int i = 0; i < numcopyacqparams; i++) {
	    for (int j = i + 1; j < numcopyacqparams; j++) {
		int doswap = 0;
		if (copyacqparamsraw[i].tag > copyacqparamsraw[j].tag) {
		    doswap = 1;
		}
		if (doswap) {
		    copyacqparam swap = copyacqparamsraw[i];
		    copyacqparamsraw[i] = copyacqparamsraw[j];
		    copyacqparamsraw[j] = swap;
		}
	    }
	}
	copyacqparamlist = copyacqparamsraw;
    }

    /* go in tag order and use stack to go through dataset in order */
    for (int i = 0; i < numcopyacqparams; i++) {
	copyacqparam * cap = &copyacqparamlist[i];
	DcmStack dstack;
	DcmStack mstack;
#if DEBUG
	const char * fieldtypestr = "(type unknown)";
	switch (cap->type) {
	case CP_STR:
	  fieldtypestr = "CP_STR";
	  break;
	case CP_STRARRAY:
	  fieldtypestr = "CP_STRARRAY";
	  break;
	case CP_ISODATE:
	  fieldtypestr = "CP_ISODATE";
	  break;
	case CP_ISOTIME:
	  fieldtypestr = "CP_ISOTIME";
	  break;
	case CP_ISODATETIME:
	  fieldtypestr = "CP_ISODATETIME";
	  break;
	}
	fprintf(stderr, "     * %s %s\n", cap->fieldname, fieldtypestr);
#endif
	if (dfile->dataset->search(cap->tag, dstack, ESM_afterStackTop, OFTrue) == EC_Normal) {
	    elem = (DcmElement *)dstack.top();
	} else if (dfile->metainfo->search(cap->tag, mstack, ESM_afterStackTop, OFTrue) == EC_Normal) {
	    elem = (DcmElement *)mstack.top();
	} else {
	    continue;
	}
#if DEBUG
	std::cerr << "        ";
	elem->print(std::cerr);
	std::cerr << std::endl;
#endif
	tmpos.clear();
	if (cap->type == CP_STR) {
	    elem->getOFString(tmpos, 0);
	    if (bxh_appendChildElement(acquisitiondata, cap->fieldname, tmpos.c_str()) != 0)
		goto FAIL;
	} else if (cap->type == CP_STRARRAY) {
	    OFString tmpval;
	    unsigned int valnum;
	    for (valnum = 0; valnum < elem->getVM(); valnum++) {
		tmpos.clear();
		elem->getOFString(tmpos, valnum);
		if (valnum > 0)
		    tmpval += "/";
		tmpval += tmpos;
	    }
	    if (bxh_appendChildElement(acquisitiondata, cap->fieldname, tmpval.c_str()) != 0)
		goto FAIL;
	} else if (cap->type == CP_ISODATE) {
	    ((DcmDate *)(elem))->getISOFormattedDate(tmpos, 0);
	    if (bxh_appendChildElement(acquisitiondata, cap->fieldname, tmpos.c_str()) != 0)
		goto FAIL;
	} else if (cap->type == CP_ISOTIME) {
	    ((DcmTime *)(elem))->getISOFormattedTime(tmpos, 0);
	    if (bxh_appendChildElement(acquisitiondata, cap->fieldname, tmpos.c_str()) != 0)
		goto FAIL;
	} else if (cap->type == CP_ISODATETIME) {
	    ((DcmDateTime *)(elem))->getISOFormattedDateTime(tmpos, 0);
	    if (bxh_appendChildElement(acquisitiondata, cap->fieldname, tmpos.c_str()) != 0)
		goto FAIL;
	}
    }

#if DEBUG
    fprintf(stderr, "     MagneticFieldStrength\n");
#endif
    if (getDicomValue(dfile, "MagneticFieldStrength", stack)) {
	elem = (DcmElement *)stack.top();
	elem->getFloat64(val.f64, 0);
	if (strcmp(finfos[0]->manufacturer, "GE MEDICAL SYSTEMS") == 0 &&
	    val.f64 > 1000.0) {
	    val.f64 /= 10000.0; /* GE DICOM is broken -- value is in gauss :( */
	}
	sprintf(&numbuf[0], "%.15g", (double)val.f64);
	if (bxh_appendChildElement(acquisitiondata, "magneticfield", &numbuf[0]) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     ReconstructionDiameter\n");
#endif
    if (getDicomValue(dfile, "ReconstructionDiameter", stack)) {
	Float64 ppfov;
	elem = (DcmElement *)stack.top();
	elem->getFloat64(val.f64, 0);
	if (getDicomValue(dfile, "PercentPhaseFieldOfView", stack)) {
	    elem = (DcmElement *)stack.top();
	    elem->getFloat64(ppfov, 0);
	    freqfov = val.f64;
	    phasefov = val.f64*ppfov/100.0;
	}
    }
    if (freqfov != 0 && phasefov != 0) {
	sprintf(&tmpbuf[0], "%.15g %.15g", (double)freqfov, (double)phasefov);
	if (bxh_appendChildElement(acquisitiondata, "fieldofview", &tmpbuf[0]) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     TR\n");
#endif
    if (dimtr) {
	unsigned int i;
	OFString tros("");
	bxhdatapoints * trdpstructp = NULL;
	size_t size = dimtr->size;
	for (int dpstructind = 0; dpstructind < dimtr->numdpstructs; dpstructind++) {
	    char * label = dimtr->dpstructs[dpstructind].label;
	    if (label == NULL) {
		trdpstructp = &dimtr->dpstructs[dpstructind];
		size = trdpstructp->numvalues;
	    }
	}
	for (i = 0; i < size; i++) {
	    if (i != 0)
		tros.append(" ");
	    if (trdpstructp != NULL) {
		tros.append(trdpstructp->values[i]);
	    } else {
		sprintf(&tmpbuf[0], "%g", dimtr->origin + (dimtr->spacing * i));
		tros.append(&tmpbuf[0]);
	    }
	}
	if (bxh_appendChildElement(acquisitiondata, "tr", tros.c_str()) != 0)
	    goto FAIL;
    } else if (finfos[0]->foundtr) {
	OFString tros("");
	sprintf(&tmpbuf[0], "%g", finfos[0]->tr);
	tros.append(&tmpbuf[0]);
	if (bxh_appendChildElement(acquisitiondata, "tr", tros.c_str()) != 0)
	    goto FAIL;
    } else if (getDicomValue(dfile, "RepetitionTime", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "tr", tmpos.c_str()) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     TE\n");
#endif
    if (dimte) {
	unsigned int i;
	OFString teos("");
	bxhdatapoints * tedpstructp = 0;
	size_t size = dimte->size;
	for (int dpstructind = 0; dpstructind < dimte->numdpstructs; dpstructind++) {
	    char * label = dimte->dpstructs[dpstructind].label;
	    if (label == NULL || strcmp(label, dimte->type) == 0) {
		tedpstructp = &dimte->dpstructs[dpstructind];
		size = tedpstructp->numvalues;
	    }
	}
	for (i = 0; i < size; i++) {
	    if (i != 0)
		teos.append(" ");
	    if (tedpstructp != NULL) {
		teos.append(tedpstructp->values[i]);
	    } else {
		sprintf(&tmpbuf[0], "%g", dimte->origin + (dimte->spacing * i));
		teos.append(&tmpbuf[0]);
	    }
	}
	if (bxh_appendChildElement(acquisitiondata, "te", teos.c_str()) != 0)
	    goto FAIL;
    } else if (strcmp(finfos[0]->manufacturer, "SIEMENS") == 0 &&
	       siemensmap.find("alTE[0]") != siemensmap.end()) {
	char * testr = NULL;
	size_t testrlen = 0;
	std::map<OFString,OFString>::iterator tmpiter;
	int numtes = 0;
	char tmpname[64];

	sprintf(&tmpname[0], "alTE[%d]", numtes);
	while ((tmpiter = siemensmap.find(tmpname)) != siemensmap.end()) {
	    static char tmpcurtestr[128];
	    const char * curte = tmpiter->second.c_str();
	    size_t curtelen = 0;
	    sprintf(tmpcurtestr, "%f", atof(curte)/1000.0);
	    curtelen = strlen(tmpcurtestr);
	    if (testr == NULL) {
		testr = (char *)realloc(testr, curtelen + 1);
		testr[0] = '\0';
		sprintf(testr + testrlen, "%s", tmpcurtestr);
		testrlen += curtelen;
	    } else {
		testr = (char *)realloc(testr, testrlen + curtelen + 1 + 1);
		sprintf(testr + testrlen, " %s", tmpcurtestr);
		testrlen += curtelen + 1;
	    }
	    numtes++;
	    sprintf(&tmpname[0], "alTE[%d]", numtes);
	}
	if (bxh_appendChildElement(acquisitiondata, "te", testr) != 0)
	    goto FAIL;
	free(testr);
    } else if (finfos[0]->foundte) {
	OFString teos("");
	sprintf(&tmpbuf[0], "%g", finfos[0]->te);
	teos.append(&tmpbuf[0]);
	if (bxh_appendChildElement(acquisitiondata, "te", teos.c_str()) != 0)
	    goto FAIL;
    } else if (getDicomValue(dfile, "EchoTime", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "te", tmpos.c_str()) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     EffectiveEchoSpacing\n");
#endif
    /* GEMS_EffectiveEchoSpacing */
    if (getPrivateDicomValue(dfile, 0x0043, 0x102c, "GEMS_PARM_01", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "echospacing", tmpos.c_str()) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     sense factor\n");
#endif
    /* ASSET R Factors */
    if (getPrivateDicomValue(dfile, 0x0043, 0x1083, "GEMS_PARM_01", stack)) {
	double value1, value2;
	elem = (DcmElement *)stack.top();
	elem->getFloat64(value1, 0);
	elem->getFloat64(value2, 1);
	sprintf(&tmpbuf[0], "%.15g", (double)value2/value1);
	if (bxh_appendChildElement(acquisitiondata, "sensefactor", &tmpbuf[0]) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     NumberOfEchoes\n");
#endif
    if (getDicomValue(dfile, "NumberOfEchoes", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "nechos", tmpos.c_str()) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     NumberOfEPIShots\n");
#endif
    if (getDicomValue(dfile, "NumberOfEPIShots", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "nshots", tmpos.c_str()) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     EchoNumber\n");
#endif
    if (getDicomValue(dfile, "EchoNumber", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(acquisitiondata, "echonumber", tmpos.c_str()) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     prescribed slice spacing\n");
#endif
    {
	sprintf(&numbuf[0], "%g", finfos[0]->slicespacing);
	if (bxh_appendChildElement(acquisitiondata, "prescribedslicespacing", &numbuf[0]) != 0)
	    goto FAIL;
    }

#if DEBUG
    fprintf(stderr, "     GE PixelInformation\n");
#endif
    /* Actual prescribed pixel sizes */
    if (getPrivateDicomValue(dfile, 0x0043, 0x10bb, "GEMS_PARM_01", stack)) {
	Float64 pixelinfox;
	Float64 pixelinfoy;
	OFBool success;
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	pixelinfox = OFStandard::atof(tmpos.c_str(), &success);
	if (success) {
	    tmpos.clear();
	    elem->getOFString(tmpos, 1);
	    pixelinfoy = OFStandard::atof(tmpos.c_str(), &success);
	    if (success) {
		sprintf(&tmpbuf[0], "%g %g", (double)pixelinfox, (double)pixelinfoy);
		if (bxh_appendChildElement(acquisitiondata, "prescribedpixelsize", &tmpbuf[0]) != 0)
		    goto FAIL;
		if (fabs(dimx->spacing - pixelinfox) > (0.01 * dimx->spacing) ||
		    fabs(dimy->spacing - pixelinfoy) > (0.01 * dimy->spacing)) {
		    sprintf(tmpbuf, "Image pixel spacing (%gx%g) is different than prescribed pixel spacing (%gx%g), and may be a side-effect of scanner resampling during reconstruction", (double)dimx->spacing, (double)dimy->spacing, (double)pixelinfox, (double)pixelinfoy);
		    bxh_datarec_addcomment(rawdatarec, tmpbuf);
		}
	    }
	}
    }

#if DEBUG
    fprintf(stderr, "     AcquisitionMatrix\n");
#endif
    if (getDicomValue(dfile, "AcquisitionMatrix", stack)) {
	elem = (DcmElement *)stack.top();
	Uint16 freqrows, freqcols;
	Uint16 phaserows, phasecols;
	elem->getUint16(freqrows, 0);
	elem->getUint16(freqcols, 1);
	elem->getUint16(phaserows, 2);
	elem->getUint16(phasecols, 3);
	if (freqcols == 0 && phaserows == 0) {
	    sprintf(&numbuf[0], "%u %u",
		    (unsigned int)freqrows, (unsigned int)phasecols);
	    if (bxh_appendChildElement(acquisitiondata, "acquisitionmatrix", &numbuf[0]) != 0)
		goto FAIL;
	} else if (freqrows == 0 && phasecols == 0) {
	    sprintf(&numbuf[0], "%u %u",
		    (unsigned int)freqcols, (unsigned int)phaserows);
	    if (bxh_appendChildElement(acquisitiondata, "acquisitionmatrix", &numbuf[0]) != 0)
		goto FAIL;
        } else if (getDicomValue(dfile, "InPlanePhaseEncodingDirection", stack)) {
	    elem = (DcmElement *)stack.top();
	    tmpos.clear();
	    elem->getOFString(tmpos, 0);
	    if (strcmp(tmpos.c_str(), "ROW") == 0) {
		sprintf(&numbuf[0], "%u %u",
			(unsigned int)phaserows, (unsigned int)freqcols);
		if (bxh_appendChildElement(acquisitiondata, "acquisitionmatrix", &numbuf[0]) != 0)
		    goto FAIL;
	    } else if (strcmp(tmpos.c_str(), "COL") == 0 || strcmp(tmpos.c_str(), "COLUMN") == 0) {
		sprintf(&numbuf[0], "%u %u",
			(unsigned int)freqrows, (unsigned int)phasecols);
		if (bxh_appendChildElement(acquisitiondata, "acquisitionmatrix", &numbuf[0]) != 0)
		    goto FAIL;
	    } else {
		fprintf(stderr, "Unrecognized value for InPlanePhaseEncodingDirection: %s\n", tmpos.c_str());
		goto FAIL;
	    }
	} else {
	    fprintf(stderr, "Can't interpret DICOM Acquisition Matrix\n");
	    goto FAIL;
	}

#if DEBUG
    fprintf(stderr, "     PixelBandwidth\n");
#endif
	if (getDicomValue(dfile, "PixelBandwidth", stack)) {
	    elem = (DcmElement *)stack.top();
	    elem->getFloat64(val.f64, 0);
	    sprintf(&numbuf[0], "%.15g", (double)val.f64);
	    if (bxh_appendChildElement(acquisitiondata, "pixelbandwidth", &numbuf[0]) != 0)
		goto FAIL;
	    /* BXH wants it per sampling space not per pixel */
	    if (freqrows)
		val.f64 *= freqrows;
	    else
		val.f64 *= freqcols;
	    val.f64 /= 1000.0; /* Hz -> kHz */
	    sprintf(&numbuf[0], "%.15g", (double)val.f64);
	    if (bxh_appendChildElement(acquisitiondata, "bandwidth", &numbuf[0]) != 0)
		goto FAIL;
	}
    }

#if DEBUG
    fprintf(stderr, "     other random info\n");
#endif
    /* add some random info */
    {
	if (strcmp(finfos[0]->manufacturer, "SIEMENS") == 0) {
	    OFString infostr;
	    const char * fieldname = NULL;
	    std::map<OFString,OFString>::iterator tmpiter;

	    fieldname = "sCOIL_SELECT_MEAS.asList[0].sCoilElementID.tCoilID";
	    if (siemensmap.count(fieldname)) {
		infostr.append(fieldname);
		infostr.append(" = ");
		infostr.append(siemensmap.find(fieldname)->second);
		infostr.append("\n");
	    }
	    fieldname = "sRawFilter.lSlope_256";
	    if (siemensmap.count(fieldname)) {
		infostr.append(fieldname);
		infostr.append(" = ");
		infostr.append(siemensmap.find(fieldname)->second);
		infostr.append("\n");
	    }
	    fieldname = "sRawFilter.ucOn";
	    if (siemensmap.count(fieldname)) {
		infostr.append(fieldname);
		infostr.append(" = ");
		infostr.append(siemensmap.find(fieldname)->second);
		infostr.append("\n");
	    }
	    fieldname = "sRawFilter.ucMode";
	    if (siemensmap.count(fieldname)) {
		infostr.append(fieldname);
		infostr.append(" = ");
		infostr.append(siemensmap.find(fieldname)->second);
		infostr.append("\n");
	    }
	    tmpiter = siemensmap.begin();
	    while (tmpiter != siemensmap.end()) {
		if ((*tmpiter).first.find("sWiPMemBlock") == 0) {
		    infostr.append((*tmpiter).first);
		    infostr.append(" = ");
		    infostr.append((*tmpiter).second);
		    infostr.append("\n");
		}
		tmpiter++;
	    }
	    if (bxh_appendChildElement(acquisitiondata, "otherinfo", infostr.c_str()) != 0) {
		goto FAIL;
	    }
	}
    }

    {
#if DEBUG
	fprintf(stderr, "     [InPlane]PhaseEncodingDirection\n");
#endif
	OFString ped("");
	if (getDicomValue(dfile, "PhaseEncodingDirection", stack)) {
	    elem = (DcmElement *)stack.top();
	    ped.clear();
	    elem->getOFString(ped, 0);
	} else if (finfos[0]->inplanephaseencodingdirection != NULL) {
	    ped = OFString(finfos[0]->inplanephaseencodingdirection);
	}
	if (ped.length() != 0) {
	    if (strcmp(ped.c_str(), "ROW") == 0) {
		sprintf(&numbuf[0], "%u", (unsigned int)2);
	    } else if (strcmp(ped.c_str(), "COL") == 0 || strcmp(ped.c_str(), "COLUMN") == 0) {
		sprintf(&numbuf[0], "%u", (unsigned int)1);
	    } else {
		sprintf(&numbuf[0], "%u", (unsigned int)0);
	    }
	    if (bxh_appendChildElement(acquisitiondata, "frequencydirection", &numbuf[0]) != 0)
		goto FAIL;
	}
    }

#if DEBUG
    fprintf(stderr, "     slice order\n");
#endif
    if (strcmp(finfos[0]->manufacturer, "SIEMENS") == 0 &&
	siemensmap.find("sSliceArray.lSize") != siemensmap.end() &&
	siemensmap.find("sSliceArray.ucMode") != siemensmap.end()) {
	OFString sliceorderstr;
	std::map<OFString,OFString>::iterator tmpiter;
	int numslices = (int)OFStandard::atof(siemensmap.find("sSliceArray.lSize")->second.c_str());
	if (getPrivateDicomValue(dfile, 0x0019, 0x100a, "SIEMENS MR HEADER", stack)) {
	    Uint16 numslices2;
	    elem = (DcmElement *)stack.top();
            if (elem->getVR() != EVR_UN) {
		elem->getUint16(numslices2, 0);
		numslices = numslices2;
	    }
	}
	if ((tmpiter = siemensmap.find("sSliceArray.ucMode")) != siemensmap.end()) {
	    char * endptr = NULL;
	    const char * nptr = tmpiter->second.c_str();
	    int ucMode = strtol(nptr, &endptr, 0);
	    if (ucMode == 0 && endptr == nptr) {
		fprintf(stderr, "Error parsing sSliceArray.ucMode (%s)\n", nptr);
		goto FAIL;
	    }
	    if (dimz && dimz == dims && dimz->direction) {
		bxhdatapoints * dpstructp = NULL;
		for (int dpstructind = 0; dpstructind < dimz->numdpstructs; dpstructind++) {
		    if (strcmp(dimz->dpstructs[dpstructind].label, "acquisitiontimeindex") == 0) {
			// overwrite old acquisitiontimeindex field
			dpstructp = &dimz->dpstructs[dpstructind];
			bxh_datarec_datapoints_free(dpstructp);
			break;
		    }
		}
		if (dpstructp == NULL) {
		    dimz->dpstructs = (bxhdatapoints *)realloc(dimz->dpstructs, sizeof(bxhdatapoints) * (dimz->numdpstructs + 1));
		    dpstructp = &dimz->dpstructs[dimz->numdpstructs];
		    dimz->numdpstructs++;
		}
		char ** datapoints = (char **)malloc(sizeof(char *) * numslices);
		dpstructp->label = strdup("acquisitiontimeindex");
		dpstructp->values = datapoints;
		dpstructp->numvalues = numslices;
		/* we only know how to interpret this for axial scans */
		if ((ucMode == 1 && dimz->direction[2] > 0) ||
		    (ucMode == 2 && dimz->direction[2] < 0)) {
		    int slicenum;
		    for (slicenum = 0; slicenum < numslices; slicenum++) {
			sprintf(&numbuf[0], "%d", slicenum+1);
			if (slicenum != 0) {
			    sliceorderstr += ",";
			}
			sliceorderstr += numbuf;
			datapoints[slicenum] = strdup(numbuf);
		    }
		} else if ((ucMode == 2 && dimz->direction[2] > 0) ||
			   (ucMode == 1 && dimz->direction[2] < 0)) {
		    int slicenum;
		    int acqind = 1;
		    for (slicenum = numslices-1; slicenum >= 0; slicenum--, acqind++) {
			sprintf(&numbuf[0], "%d", slicenum+1);
			if (slicenum != numslices-1) {
			    sliceorderstr += ",";
			}
			sliceorderstr += numbuf;
			sprintf(&numbuf[0], "%d", acqind);
			datapoints[slicenum] = strdup(numbuf);
		    }
		} else if (ucMode == 4) {
		    /* NOTE: if the number of slices is even, then, in
		     * interleaved scans, the first slice acquired is the
		     * *second* slice from the bottom. */
		    if (dimz->direction[2] > 0) {
			/* image stored I->S */
			int startslicenum1 = ((numslices % 2) == 0) ? 1 : 0;
			int startslicenum2 = ((numslices % 2) == 0) ? 0 : 1;
			int slicenum;
			int isfirst = 1;
			int acqind = 1;
			for (slicenum = startslicenum1; slicenum < numslices; slicenum+=2, acqind++) {
			    sprintf(&numbuf[0], "%d", slicenum+1);
			    if (!isfirst) {
				sliceorderstr += ",";
			    }
			    isfirst = 0;
			    sliceorderstr += numbuf;
			    sprintf(&numbuf[0], "%d", acqind);
			    datapoints[slicenum] = strdup(numbuf);
			}
			for (slicenum = startslicenum2; slicenum < numslices; slicenum+=2, acqind++) {
			    sprintf(&numbuf[0], "%d", slicenum+1);
			    if (!isfirst) {
				sliceorderstr += ",";
			    }
			    isfirst = 0;
			    sliceorderstr += numbuf;
			    sprintf(&numbuf[0], "%d", acqind);
			    datapoints[slicenum] = strdup(numbuf);
			}
		    } else {
			/* image stored S->I (but Siemens always acquires
			 * axial scans I->S) */
			int startslicenum1 = ((numslices % 2) == 0) ? numslices - 2 : numslices - 1;
			int startslicenum2 = ((numslices % 2) == 0) ? numslices - 1 : numslices - 2;
			int slicenum;
			int acqind = 1;
			int isfirst = 1;
			for (slicenum = startslicenum1; slicenum >= 0; slicenum-=2, acqind++) {
			    sprintf(&numbuf[0], "%d", slicenum+1);
			    if (!isfirst) {
				sliceorderstr += ",";
			    }
			    isfirst = 0;
			    sliceorderstr += numbuf;
			    sprintf(&numbuf[0], "%d", acqind);
			    datapoints[slicenum] = strdup(numbuf);
			}
			for (slicenum = startslicenum2; slicenum >= 0; slicenum-=2, acqind++) {
			    sprintf(&numbuf[0], "%d", slicenum+1);
			    if (!isfirst) {
				sliceorderstr += ",";
			    }
			    isfirst = 0;
			    sliceorderstr += numbuf;
			    sprintf(&numbuf[0], "%d", acqind);
			    datapoints[slicenum] = strdup(numbuf);
			}
		    }
		} else {
		    fprintf(stderr, "Bad sSliceArray.ucMode (%d)\n", ucMode);
		    goto FAIL;
		}
	    }
	    if (sliceorderstr.length() > 0) {
		// use setChildElement (instead of appendChildElement) in case
		// sliceorder was already set
		if (bxh_setChildElement(acquisitiondata, "sliceorder", sliceorderstr.c_str()) != 0)
		    goto FAIL;
	    }
	}
    }

#if DEBUG
    fprintf(stderr, "     bvalues\n");
#endif
    if (dimdiffdir) {
    	if (bvalueos.length() > 0) {
	    if (bxh_appendChildElement(acquisitiondata, "bvalues", bvalueos.c_str()) != 0)
		goto FAIL;
	}
    }

#if DEBUG
    fprintf(stderr, "     AcquisitionNumber\n");
#endif
    /* Number of dummy scans (discarded data acquisitions, "disdaqs") */
    if (getPrivateDicomValue(dfile, 0x0043, 0x1076, "GEMS_PARM_01", stack)) {
	Uint16 disdaqs;
	elem = (DcmElement *)stack.top();
	elem->getUint16(disdaqs, 0);
	sprintf(&tmpbuf[0], "%u", (unsigned int)disdaqs);
	if (bxh_appendChildElement(acquisitiondata, "disdaqs", &tmpbuf[0]) != 0)
	    goto FAIL;
    }

    /*** SUBJECT ***/

#if DEBUG
    fprintf(stderr, " ...SUBJECT\n");
#endif

    if ((subject = bxh_getSubject(docp)) == NULL)
	goto FAIL;

    if (getDicomValue(dfile, "PatientName", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(subject, "name", tmpos.c_str()) != 0)
	    goto FAIL;
    }
    if (getDicomValue(dfile, "PatientAge", stack)) {
	double age;
	char type;
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (tmpos.size() > 0) {
	    age = strtod(tmpos.c_str(), NULL);
	    type = tmpos[tmpos.length()-1];
	    if (type == 'W') {
		age /= 52;
	    } else if (type == 'M') {
		age /= 12;
	    } else if (type == 'D') {
		age /= 365;
	    }
	    sprintf(&numbuf[0], "%.15g", age);
	    if (bxh_appendChildElement(subject, "age", &numbuf[0]) != 0)
		goto FAIL;
	}
    }
    if (getDicomValue(dfile, "PatientSex", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(subject, "sex", tmpos.c_str()) != 0)
	    goto FAIL;
    }
    if (getDicomValue(dfile, "PatientWeight", stack)) {
	elem = (DcmElement *)stack.top();
	tmpos.clear();
	elem->getOFString(tmpos, 0);
	if (bxh_appendChildElement(subject, "weight", tmpos.c_str()) != 0)
	    goto FAIL;
    }

    /* we've created all we need for the datarec */
    if (bxh_datarec_writeToElement(imagedata, rawdatarec) != 0)
	goto FAIL;

    bxh_normalizeNamespaces(bxh);

    goto EXIT;

FAIL:
#if DEBUG
    fprintf(stderr, "bxh_DICOM_readFilesCont() failed.\n");
#endif
    if (domutil_errorbuf[0]) {
	fprintf(stderr, "%s", domutil_errorbuf);
    }
    retval = -1;

EXIT:
#if DEBUG
    fprintf(stderr, "bxh_DICOM_readFilesCont() succeeded.\n");
#endif
    if (rawdatarec) {
	bxh_datarec_free(rawdatarec);
	rawdatarec = NULL;
    }

    if (cont) {
	bxh_DICOM_freeCont(cont); cont = NULL;
    }

    if (dfileobj) {
	delete dfileobj;
	dfileobj = NULL;
    }

    if (retcont && retcont->numfinfos == 0) {
	bxh_DICOM_freeCont(retcont); retcont = NULL;
    }

    if (bxh) bxh_element_unref(bxh);
    if (imagedata) bxh_element_unref(imagedata);
    if (acquisitiondata) bxh_element_unref(acquisitiondata);
    if (subject) bxh_element_unref(subject);

    if (newdimmap != newdimmap_orig) {
	free(newdimmap);
	newdimmap = NULL;
    }

    *contp = retcont;
#if DEBUG
    fprintf(stderr, "bxh_DICOM_readFilesCont() exiting.\n");
#endif
    return retval;
}

/*
 * $Log: bxh_dicom.cpp,v $
 * Revision 1.1  2009/04/24 16:40:57  gadde
 * Initial revision
 *
 * Revision 1.112  2009/04/08 16:20:13  gadde
 * Don't break if SIEMENS CSA HEADER doesn't exist.
 *
 * Revision 1.111  2009/04/06 16:26:54  gadde
 * Fix b-value addition.
 *
 * Revision 1.110  2009/03/11 13:36:35  gadde
 * Add b-value to header.
 *
 * Revision 1.109  2009/02/09 14:46:06  gadde
 * Force datapoint array for DTI data.
 *
 * Revision 1.108  2009/01/21 19:59:14  gadde
 * Allow search of meta-information header for DICOM keys if they are not found
 * in dataset.
 *
 * Revision 1.107  2008/10/17 16:21:13  gadde
 * Fix time wrap-around detection
 *
 * Revision 1.106  2008/07/28 18:33:53  gadde
 * Remove Linux-ism for win32
 *
 * Revision 1.105  2008/07/15 15:50:16  gadde
 * Remove debugging fprintf.
 *
 * Revision 1.104  2008/05/22 18:18:35  gadde
 * Avoid optimization bug with an interspersed snprintf of the computed value.
 *
 * Revision 1.103  2008/03/07 18:19:38  gadde
 * Add raw DICOM pixel bandwidth field.
 *
 * Revision 1.102  2008/02/18 21:03:02  gadde
 * Don't include surrounding ""double-double quotes"" in strings in Siemens map.
 *
 * Revision 1.101  2008/01/21 18:10:21  gadde
 * Add more protocol name and sequence file name fields from private
 * GE and Siemens DICOM.
 *
 * Revision 1.100  2008/01/17 15:35:42  gadde
 * Add Siemens WiP (Work in Progress) block.
 *
 * Revision 1.99  2008/01/08 17:28:04  gadde
 * Add support for extracting diffusion directions directly from DICOM
 * header.
 * Also support repeated dimension values (e.g. three baseline "0 0 0"
 * diffusion images).
 *
 * Revision 1.98  2007/12/05 18:26:30  gadde
 * Add several fields that can be output for XCEDE 2.
 * Also add slice order information for some Siemens data.
 *
 * Revision 1.97  2007/09/14 20:26:24  gadde
 * Add sense factor (for GE).
 *
 * Revision 1.96  2007/07/18 20:38:16  gadde
 * Fix Mosaic interpretation of dReadoutFOV and dPhaseFOV.
 *
 * Revision 1.95  2007/07/09 20:06:36  gadde
 * Add EffectiveEchoSpacing from GE DICOM.
 *
 * Revision 1.94  2007/07/09 19:51:41  gadde
 * If Siemens files specify multiple TEs within one image, put them all in
 * the TE acquisition parameter.
 *
 * Revision 1.93  2007/05/07 15:51:37  gadde
 * Don't expect a newline after ASCCONV END delimiter.
 *
 * Revision 1.92  2007/04/18 18:07:11  gadde
 * Fix search for Siemens text header end (ASCCONV END).
 *
 * Revision 1.91  2007/04/16 20:38:37  gadde
 * Normalize the dimt (time) origin to 0.
 *
 * Revision 1.90  2007/01/12 17:09:27  gadde
 * Add some more Siemens fields.
 *
 * Revision 1.89  2007/01/12 16:06:26  gadde
 * Add some Siemens coil info.
 *
 * Revision 1.88  2007/01/11 21:09:39  gadde
 * Now handle the fact that GE might actually be writing the TR in tesla,
 * as expected. :-)
 *
 * Revision 1.87  2007/01/11 20:15:58  gadde
 * Some initialization fixes
 *
 * Revision 1.86  2006/12/05 16:09:05  gadde
 * Deal better with content dates that are before the epoch
 * (Jan. 1 1970).
 *
 * Revision 1.85  2006/11/01 20:32:12  gadde
 * Check if returned string value is empty before indexing into it!
 *
 * Revision 1.84  2006/10/27 13:56:05  gadde
 * Add --filename-sort option to dicom2bxh
 *
 * Revision 1.83  2006/10/25 20:51:15  gadde
 * Clear tmpos before running getOFString() in case the element has
 * no value (in which case tmpos does not get updated).
 *
 * Revision 1.82  2006/10/20 17:29:44  gadde
 * Fix side-effect bug from last commit.
 *
 * Revision 1.81  2006/10/20 16:30:27  gadde
 * Deal better with DICOM datasets that might have instances
 * of the fields we are looking for in subsequences.  Modify
 * search behavior to exclude subsequences.
 *
 * Revision 1.80  2006/05/25 18:20:41  gadde
 * Fix slice direction for Siemens files (ucImageNumb* fields interpreted
 * incorrectly).
 *
 * Revision 1.79  2006/04/07 19:07:44  gadde
 * Add implicit BIAC tensors and measurementframe
 *
 * Revision 1.78  2006/03/29 16:52:41  gadde
 * Back out accidentally committed change.
 *
 * Revision 1.77  2006/03/29 16:47:44  gadde
 * bump version number to development version
 *
 * Revision 1.76  2006/02/08 19:04:54  gadde
 * Use AcquisitionDate/Time if available for Siemens mosaics
 *
 * Revision 1.75  2005/09/14 14:49:26  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.74  2005/08/03 16:21:29  gadde
 * Update to Siemens orientation calculation.
 * Also, fix siemensmap value extraction.
 *
 * Revision 1.73  2005/08/02 12:53:32  gadde
 * Write frequencydirection based on PhaseEncodingDirection field.
 *
 * Revision 1.72  2005/07/29 19:27:18  gadde
 * Revert flip of rows/cols in frequencydirection mistakenly
 * applied to 1.71.
 *
 * Revision 1.71  2005/07/29 18:53:16  gadde
 * Remove InPlanePhaseEncodingDirection because it is redundant with
 * frequencydirection.
 *
 * Revision 1.70  2005/07/28 14:23:05  gadde
 * Add InPlanePhaseEncodingDirection to acqparams.
 *
 * Revision 1.69  2005/07/25 21:48:31  gadde
 * Fix typo.
 *
 * Revision 1.68  2005/07/25 21:46:08  gadde
 * Don't use datapoints field if it doesn't exist!
 *
 * Revision 1.67  2005/07/07 15:10:51  gadde
 * Memory fixes.
 *
 * Revision 1.66  2005/06/23 16:32:24  gadde
 * Reintroduce check for repeated contiguous origins, but first verify
 * that repetitions aren't accounted for by other dimensions.
 *
 * Revision 1.65  2005/06/17 13:57:49  gadde
 * Update for DTI data from GE EXCITE 12.0 scanner software.
 *
 * Revision 1.64  2005/06/15 21:03:40  gadde
 * Change terminology series->acquisition.
 *
 * Revision 1.63  2005/06/15 16:27:17  gadde
 * Add acquisition number as a potential dimension.
 * Also update code to guess existence/size/order of time dimension.
 *
 * Revision 1.62  2005/05/10 19:47:23  gadde
 * AUTOGEN orientation comment now created in bxh_datarec_writeToElement.
 *
 * Revision 1.61  2005/03/25 22:56:47  gadde
 * Some minor fixes.
 *
 * Revision 1.60  2004/12/10 19:18:02  gadde
 * Add manufacturer and model name to acquisition data.
 *
 * Revision 1.59  2004/12/02 22:33:44  gadde
 * Fix bug in bubble-sort and don't detect a wrap-around if extra
 * dimensions (tr and te) are different.
 *
 * Revision 1.58  2004/08/11 15:20:09  gadde
 * Give a little more info on "new dimension" divide error.
 *
 * Revision 1.57  2004/07/30 19:31:06  gadde
 * Some code cleanup, plus a fix to work with DCMTK-3.5.3, and a fix
 * to convert Siemens origin from lps->ras.
 *
 * Revision 1.56  2004/07/07 17:00:25  gadde
 * Memory fix.
 *
 * Revision 1.55  2004/07/06 14:05:48  gadde
 * Now can perform some checks to more easily determine time series.
 * In the process, made explicit the method of iterating through
 * the slices and picking out only those slices that match, using
 * new functions that use continuations.
 *
 * Revision 1.54  2004/06/16 21:39:00  gadde
 * Guess whether this is a time series by seeing if slice origins
 * wrap around.
 *
 * Revision 1.53  2004/06/16 19:14:16  gadde
 * GE regressor value was a red herring.  Removed check for this.
 *
 * Revision 1.52  2004/06/15 19:02:24  gadde
 * Initial support for some 4D data produced by GE scanners.
 *
 * Revision 1.51  2004/06/15 18:44:41  gadde
 * Add support for distinguishing GE DICOM data by Raw Data Type.
 *
 * Revision 1.50  2004/06/15 16:16:11  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.49  2004/05/17 17:54:21  gadde
 * Add force-concat option.
 *
 * Revision 1.48  2004/04/07 17:20:58  gadde
 * Daylight savings time bug fix.
 *
 * Revision 1.47  2004/02/26 22:13:27  gadde
 * Oops, allow z size to be modified for mosaics too
 *
 * Revision 1.46  2004/02/26 21:51:38  gadde
 * Fix spacing for dual-echo scans
 *
 * Revision 1.45  2004/02/23 21:31:55  gadde
 * Add examdescription field.
 *
 * Revision 1.44  2003/11/14 20:10:48  gadde
 * Add pulse sequence name, internal pulse sequence name, and psd date/time.
 *
 * Revision 1.43  2003/10/22 17:15:42  gadde
 * Fix some bugs uncovered on AIX
 *
 * Revision 1.42  2003/10/21 19:31:04  gadde
 * Change from splitignorestart to more generic outputselect.
 *
 * Revision 1.41  2003/10/14 16:47:49  gadde
 * Hack to get rid of extra slices in Siemens Mosaic (maybe it can be applied generally to other formats with split dimensions)
 *
 * Revision 1.40  2003/09/26 16:23:34  gadde
 * Don't accumulate slice spacings, just use total volume spacing and
 * divide.
 *
 * Revision 1.39  2003/09/26 15:08:37  gadde
 * Fix log message.
 *
 * Revision 1.38  2003/09/26 15:04:11  gadde
 * Fix slice thickness/slice spacing.
 *
 * Revision 1.37  2003/09/04 18:37:55  gadde
 * Fix some casting that results in some subtle optimization bugs.
 * (and use calculated slice spacing -- not documented in real log,
 * added here after the fact).
 *
 * Revision 1.36  2003/09/02 18:23:51  gadde
 * Add Siemens-specific handling for certain fields.
 *
 * Revision 1.35  2003/08/29 16:00:03  gadde
 * Fix mosaic orientation.
 *
 * Revision 1.34  2003/08/22 15:07:29  gadde
 * Add ScanningSequence and SequenceVariant to fields.
 *
 * Revision 1.33  2003/08/18 20:44:48  gadde
 * Get rid of more cruft.
 *
 * Revision 1.32  2003/08/18 20:34:07  gadde
 * Get rid of old many-frag mosaic implementation.
 *
 * Revision 1.31  2003/08/18 19:54:41  gadde
 * Update error message.
 *
 * Revision 1.30  2003/08/18 19:29:31  gadde
 * Typo.
 *
 * Revision 1.29  2003/08/18 19:19:04  gadde
 * Add new getDicomValue function in case we want to start using the
 * macros defined in dcdeftag.h.
 * Check modality and ignore acquisition number if CT.
 * Don't fail if normal isn't really a normal -- just print an error message.
 *
 * Revision 1.28  2003/08/08 18:56:16  gadde
 * P670 updates
 *
 * Revision 1.27  2003/08/01 17:46:45  gadde
 * On the road to merging in xerces branch
 *
 * Revision 1.26  2003/07/28 20:56:54  gadde
 * Slight integration of libxml2/Xerces branches
 *
 * Revision 1.25  2003/07/21 16:46:49  gadde
 * Code-prettiness updates, for the most part, to further protect BXH
 * library users from particulars of DOM implementation (esp. C vs. C++).
 *
 * Revision 1.24  2003/06/26 18:54:16  gadde
 * Don't fail if PercentPhaseFieldOfView is missing...
 *
 * Revision 1.23  2003/06/18 16:14:15  gadde
 * win32 fixes
 *
 * Revision 1.22  2003/06/13 22:17:43  gadde
 * Add Win32 equivalents for opendir().
 *
 * Revision 1.21  2003/06/04 13:52:40  gadde
 * Added support for Siemens "mosaic" format and multiple time points
 * (as used in Picker).
 *
 * Revision 1.20  2003/05/29 20:33:41  gadde
 * Add an epsilon for orientation.
 *
 * Revision 1.19  2003/05/16 16:30:42  gadde
 * Add some more reliable endian checking.
 *
 * Revision 1.18  2003/05/14 21:50:15  gadde
 * Don't fix MagneticFieldStrength in non-GE files.
 *
 * Revision 1.17  2003/05/12 18:42:40  gadde
 * Additional precision in float/double output.
 *
 * Revision 1.16  2003/05/12 00:29:15  gadde
 * Fix origin in GE DICOM files.
 *
 * Revision 1.15  2003/05/10 18:19:12  gadde
 * Don't null-terminate frags.
 *
 * Revision 1.14  2003/04/17 19:31:18  gadde
 * Add support for B-matrix as dimension.
 *
 * Revision 1.13  2003/04/03 15:12:58  gadde
 * Add transmit/receive coil name.
 *
 * Revision 1.12  2003/03/31 17:51:09  gadde
 * Put textual orientation in comment instead of element.
 *
 * Revision 1.11  2003/03/24 16:43:59  gadde
 * Add support for vector datapoints
 *
 * Revision 1.10  2003/02/12 21:07:41  gadde
 * Update for DCMTK 3.5.2.
 *
 * Revision 1.9  2003/01/28 17:21:15  gadde
 * Updated datapoints usage and added support for tr/te dimensions.
 *
 * Revision 1.8  2003/01/17 16:16:17  gadde
 * Add version string to history
 *
 * Revision 1.7  2003/01/16 21:15:11  gadde
 * *Really* fix log weirdness.
 *
 * Revision 1.6  2003/01/16 21:14:45  gadde
 * Fix log weirdness.
 *
 * Revision 1.5  2003/01/16 21:13:28  gadde
 * Fix field of view/percent calculation
 *
 */
