static const char rcsid[] = "$Id: bxh_dom_utils-xerces.cpp,v 1.4 2004-04-02 16:23:53 gadde Exp $";

/* dom_utils.c --
 *
 * Some useful XPath-based utitility functions for use in conjunction
 * with libgdome.
 *
 * Author: Syam Gadde (gadde@biac.duke.edu), July 2002.
 */

#include <limits.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <xercesc/dom/DOM.hpp>
#include <pathan/XPathEvaluator.hpp>

#include "bxh_dom_utils-xerces.h"

static char domutil_errorarray[2048];

/*
 * For functions that return an error, this string may contain
 * a textual description of the error.  All functions initially
 * set this to the null string.  Critical errors are reported
 * straight to stderr.
 */
#ifdef __cplusplus
extern "C" {
#endif
char * domutil_errorbuf = &domutil_errorarray[0];
#ifdef __cplusplus
}
#endif

/**
 * This function initializes the DOM implementation
 *
 * @return 0 on success, -1 on error
 */
int
domutil_initialize()
{
    try {
        XMLPlatformUtils::Initialize();
    } catch(const DOMException & exc) {
        fprintf(stderr, "Error during Xerces-c Initialization.\n  Exception message: %s\n", XMLString::transcode(exc.msg));
	return -1;
    }
    return 0;
}

/**
 * This function creates a comment with the given text and appends it
 * to the given parent element.
 *
 * @param doc the document needed to create new nodes
 * @param parent the parent of the new comment
 * @param valuestr text that will appear inside the comment
 *            If NULL, then the element will be empty.
 * @param prependnewline if non-zero, will prepend a newline before comment
 * @return The new child element, or NULL on error
 */
DOMComment *
domutil_appendComment(DOMDocument * doc,
		      DOMElement * parent,
		      const char * valuestr,
		      int prependnewline)
{
    XMLCh * value = NULL;
    DOMComment * comment = NULL;

    if (prependnewline) {
	DOMNode * tmpnode = NULL;
	value = XMLString::transcode("\n");
	try {
	    tmpnode = doc->createTextNode(value);
	} catch (DOMException & exc) {
	    fprintf(stderr, "Document.createTextNode: failed\n\tException #%s\n", XMLString::transcode(exc.msg));
	    goto FAIL;
	}
	XMLString::release(&value); value = NULL;
	try {
	    parent->appendChild(tmpnode);
	} catch (DOMException & exc) {
	    sprintf(domutil_errorbuf, "Appending text to imagedata failed\n\tException #%s\n", XMLString::transcode(exc.msg));
	    goto FAIL;
	}
	tmpnode = NULL;
    }

    value = XMLString::transcode(valuestr);
    try {
	comment = doc->createComment(value);
    } catch (DOMException & exc) {
	fprintf(stderr, "Document.createComment: failed\n\tException #%s\n", XMLString::transcode(exc.msg));
	goto FAIL;
    }
    XMLString::release(&value); value = NULL;

    try {
	parent->appendChild(comment);
    } catch (DOMException & exc) {
	sprintf(domutil_errorbuf, "Appending comment to imagedata failed\n\tException #%s\n", XMLString::transcode(exc.msg));
	goto FAIL;
    }

    goto EXIT;

FAIL:
    comment = NULL;

EXIT:
    return comment;
}

/**
 * This function is used to create an element which encloses a simple
 * text value, and add this new element as a child to a given parent
 * element.  Caller should dereference the returned element using
 * gdome_el_unref() if it does not need to keep it around.
 *
 * @param doc the document needed to create new nodes
 * @param parent the parent of the new child
 * @param namestr tag name of the new child element
 * @param valuestr text that will appear inside the new child element.
 *            If NULL, then the element will be empty.
 * @param namespaceURI namespace for prefix, if given, or NULL
 * @return The new child element, or NULL on error
 */
DOMElement *
domutil_appendNewChildWithTextValue(DOMDocument * doc,
				    DOMElement * parent,
				    const char * namestr,
				    const char * valuestr,
				    const char * namespaceURI)
{
    char * qnamestr = NULL;
    XMLCh * name = NULL;
    XMLCh * value = NULL;
    XMLCh * ns = NULL;
    DOMElement * retelem = NULL;
    DOMText * tmptxt = NULL;
    const XMLCh * tmpstr = NULL;
    const char * tempnamespaceprefix = "dummyns:";

    domutil_errorbuf[0] = '\0';

    if (namespaceURI && !strchr(namestr, ':')) {
	qnamestr = (char *)malloc(sizeof(char *) * (strlen(namestr) + strlen(tempnamespaceprefix) + 1));
	strcpy(qnamestr, tempnamespaceprefix);
	strcat(qnamestr, namestr);
    } else {
	qnamestr = strdup(namestr);
    }

    if (namespaceURI && (ns = XMLString::transcode(namespaceURI)) == NULL) {
	sprintf(domutil_errorbuf, "domutil_appendNewChildWithValue: create string failed\n");
	goto FAIL;
    }
    if ((name = XMLString::transcode(qnamestr)) == NULL) {
	sprintf(domutil_errorbuf, "domutil_appendNewChildWithValue: create string failed\n");
	goto FAIL;
    }
    if (namespaceURI) {
	try {
	    retelem = doc->createElementNS(ns, name);
	} catch (const DOMException & exc) {
	    sprintf(domutil_errorbuf, "Create '%s' element failed\n\tException %s\n", namestr, XMLString::transcode(exc.msg));
	    goto FAIL;
	}
    } else {
	try {
	    retelem = doc->createElement(name);
	} catch (const DOMException & exc) {
	    sprintf(domutil_errorbuf, "Create '%s' element failed\n\tException %s\n", namestr, XMLString::transcode(exc.msg));
	    goto FAIL;
	}
    }
    XMLString::release(&name); name = NULL;
    try {
	parent->appendChild((DOMNode *)retelem);
    } catch (const DOMException & exc) {
	tmpstr = parent->getNodeName();
	sprintf(domutil_errorbuf, "Appending '%s' to parent '%s' failed\n\tException #%s\n", namestr, XMLString::transcode(tmpstr), XMLString::transcode(exc.msg));
	goto FAIL;
    }
    if (valuestr) {
	value = XMLString::transcode(valuestr);
	try {
	    tmptxt = doc->createTextNode(value);
	} catch (const DOMException & exc) {
	    sprintf(domutil_errorbuf, "Document.createTextNode failed\n\tException #%s\n", XMLString::transcode(exc.msg));
	    goto FAIL;
	}
	XMLString::release(&value); value = NULL;
	try {
	    retelem->appendChild((DOMNode *)tmptxt);
	} catch (const DOMException & exc) {
	    sprintf(domutil_errorbuf, "Appending text to '%s' failed\n\tException #%s\n", namestr, XMLString::transcode(exc.msg));
	    goto FAIL;
	}
    }

    goto EXIT;

FAIL:

EXIT:
    if (qnamestr)
	free(qnamestr);
    if (ns)
	XMLString::release(&ns);
    if (name)
	XMLString::release(&name);
    if (value)
	XMLString::release(&value);
    return retelem;
}

/**
 * This function is used to create an attribute which encloses a simple
 * text value, and add this new attribute as a child to a given parent
 * element.  Caller should dereference the returned attribute using
 * gdome_a_unref() if it does not need to keep it around.
 *
 * @param doc the document needed to create new nodes
 * @param parent the parent of the new attribute
 * @param namestr tag name of the new attribute element
 * @param valuestr text that will appear inside the new attribute.
 *            If NULL, then the attribute will be empty.
 * @param namespaceURI namespace for prefix, if given, or NULL
 * @return 0 on success, -1 on error
 */
int
domutil_setAttributeWithTextValue(DOMDocument * doc,
				  DOMElement * parent,
				  const char * namestr,
				  const char * valuestr,
				  const char * namespaceURI)
{
    char * qnamestr = NULL;
    XMLCh * name = NULL;
    XMLCh * value = NULL;
    XMLCh * ns = NULL;
    const XMLCh * tmpstr = NULL;
    const char * tempnamespaceprefix = "dummyns:";
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if (namespaceURI && !strchr(namestr, ':') && strcmp(namestr, "xmlns") != 0) {
	qnamestr = (char *)malloc(sizeof(char *) * (strlen(namestr) + strlen(tempnamespaceprefix) + 1));
	strcpy(qnamestr, tempnamespaceprefix);
	strcat(qnamestr, namestr);
    } else {
	qnamestr = strdup(namestr);
    }

    if (namespaceURI && (ns = XMLString::transcode(namespaceURI)) == NULL) {
	sprintf(domutil_errorbuf, "domutil_appendNewChildWithValue: create string failed\n");
	goto FAIL;
    }
    if ((name = XMLString::transcode(qnamestr)) == NULL) {
	sprintf(domutil_errorbuf, "domutil_appendNewChildWithValue: create string failed (namestr may be NULL?)\n");
	goto FAIL;
    }
    if ((value = XMLString::transcode(valuestr)) == NULL) {
	sprintf(domutil_errorbuf, "domutil_appendNewChildWithValue: create string failed (valuestr may be NULL?)\n");
	goto FAIL;
    }
    try {
	if (namespaceURI) {
	    if (0 && strcmp(namestr, "xmlns") == 0) {
		/* a bug in gdome2 forces us to do this -- cover your eyes! */
//	    gdome_xmlLinkNsDecl (((DOM_xml_Node *)parent)->n, (xmlChar *)NULL, (xmlChar *)valuestr);
	    } else {
		parent->setAttributeNS(ns, name, value);
	    }
	} else {
	    parent->setAttribute(name, value);
	}
    } catch (const DOMException & exc) {
	tmpstr = parent->getNodeName();
	sprintf(domutil_errorbuf, "Appending '%s' to parent '%s' failed\n\tException #%s\n", namestr, XMLString::transcode(tmpstr), XMLString::transcode(exc.msg));
	goto FAIL;
    }
    XMLString::release(&name); name = NULL;
    XMLString::release(&value); value = NULL;

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (qnamestr)
	free(qnamestr);
    if (ns)
	XMLString::release(&ns);
    if (name)
	XMLString::release(&name);
    if (value)
	XMLString::release(&value);
    return retval;
}

/**
 * This function evaluates an XPath expression in a given context node.
 *
 * @param ctxt the context node in which to evaluate the xpath expression
 * @param exprstr an XPath expression
 * @param resultType a GDOME type to cast the result
 * @return the result of evaluation, or NULL on error
 */
XPathResult *
domutil_getxpresult(DOMNode * ctxt, const char * exprstr, XPathResult::resultType resultType)
{
    XPathEvaluator * eval = NULL;
    XMLCh * expr = NULL;
    XPathResult * result = NULL;
    XPathNSResolver * nsresolv = NULL;
    XPathExpression * parsedexpr = NULL;
    char * newexprstr = NULL;

    domutil_errorbuf[0] = '\0';

    newexprstr = (char *)malloc(sizeof(char)*(strlen(exprstr)+16));
    switch (resultType) {
    case XPathResult::STRING_TYPE:
	sprintf(newexprstr, "string(%s)", exprstr);
	break;
    case XPathResult::NUMBER_TYPE:
	sprintf(newexprstr, "number(%s)", exprstr);
	break;
    default:
	sprintf(newexprstr, "%s", exprstr);
	break;
    }

    eval = XPathEvaluator::createEvaluator();

    try {
	nsresolv = eval->createNSResolver(ctxt);
    } catch (const DOMException & exc) {
	sprintf(domutil_errorbuf, "Creating NSResolver failed.\n\tException #%s\n", XMLString::transcode(exc.msg));
	goto FAIL;
    }
    expr = XMLString::transcode(newexprstr);
    parsedexpr = eval->createExpression(expr, nsresolv);
    try {
	result = parsedexpr->evaluate(ctxt, resultType, NULL);
    } catch (const XPathException & exc) {
	sprintf(domutil_errorbuf, "Attempt to find '%s' failed.\n\tException #%s\n", exprstr, exc.getString().c_str());
	goto FAIL;
    }
    XMLString::release(&expr); expr = NULL;
    
    goto EXIT;

FAIL:
    if (result)
	delete result;
    result = NULL;

EXIT:
    if (newexprstr)
	free(newexprstr);
//     if (parsedexpr)
// 	delete parsedexpr;
//     if (nsresolv)
// 	delete nsresolv;
    if (expr)
	XMLString::release(&expr);
//     if (eval)
// 	delete eval;
    return result;
}

/**
 * Given an XPath expression and a context node, this function returns
 * a newly malloc()ed array of pointers to all matching nodes.  The
 * pointer to the new array will be stored in the location pointed to by
 * retp.  The returned array must eventually be free()d by the caller,
 * and all nodes within the array should also be dereferenced by the
 * caller with gdome_n_unref().  Both can be done using
 * domutil_freeNodeArray(), if desired.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param resultType type of returned node list (should be a node list type)
 * @param retp pointer to result
 * @return number of nodes in array, or -1 on error.  On error, *retp
 *          is unchanged.
 */
int
domutil_getNodeArray(DOMNode * ctxt, const char * path, XPathResult::resultType resultType, DOMNode *** retp)
{
    XPathResult * result = NULL;
    DOMNode * tmpnode = NULL;
    DOMNode ** nodearray = NULL;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if ((result = domutil_getxpresult(ctxt, path, resultType)) == NULL) {
	if (!domutil_errorbuf[0])
	    sprintf(domutil_errorbuf, "getNodeArray: Error getting path '%s'\n", path);
	goto FAIL;
    }
    retval = 0;
    while ((tmpnode = result->iterateNext()) != NULL) {
	if (retval == 0)
	    nodearray = (DOMNode **)malloc(sizeof(DOMNode *) * (retval + 2));
	else
	    nodearray = (DOMNode **)realloc(nodearray, sizeof(DOMNode *) * (retval + 2));
	if (nodearray == NULL)
	    goto FAIL;
	nodearray[retval] = tmpnode;
	retval++;
    }
    if (retval == 0) {
	if (!domutil_errorbuf[0])
	    sprintf(domutil_errorbuf, "getNodeArray: Error getting path '%s'\n", path);
	goto FAIL;
    }
    nodearray[retval] = NULL;
    *retp = nodearray;
    goto EXIT;

FAIL:
    retval = -1;

EXIT:
//     if (result)
// 	delete result;
    return retval;
}

/**
 * A convenient way to free a node array of the kind returned by
 * domutil_getNodeArray().  Send a *pointer to* the pointer
 * to the node pointer array -- i.e. the same argument you sent
 * to domutil_getNodeArray();
 *
 * @param nap pointer to the location holding a pointer to a node pointer array
 */
void
domutil_freeNodeArray(DOMNode *** nap)
{
    domutil_errorbuf[0] = '\0';

    free(*nap);
/*     *nap = NULL; */
}

/**
 * Given an XPath expression and a context node, this function returns
 * a pointer to the single node that matches the expression.  The pointer
 * to the node will be stored in the location pointed to by retp.
 * The returned node eventually should be dereferenced by the caller with
 * gdome_n_unref().  It is an error if the XPath expression matches more
 * than one node.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return 0 on success, or -1 on error.  On error, *retp is unchanged.
 */
int
domutil_getSingleNode(DOMNode * ctxt, const char * path, DOMNode ** retp)
{
    DOMNode ** nodearray = NULL;
    int numnodes;
    int retval = 0;
    
    domutil_errorbuf[0] = '\0';

    if ((numnodes = domutil_getNodeArray(ctxt, path, XPathResult::UNORDERED_NODE_ITERATOR_TYPE, &nodearray)) == -1) {
	if (!domutil_errorbuf[0])
	    sprintf(domutil_errorbuf, "getSingleNode: error getting path '%s'\n", path);
	goto FAIL;
    }
    if (numnodes != 1) {
	if (!domutil_errorbuf[0])
	    sprintf(domutil_errorbuf, "getSingleNode: path '%s' has %d elements (should be 1)\n", path, numnodes);
	goto FAIL;
    }
    *retp = nodearray[0];
    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (nodearray)
	free(nodearray);
    return retval;
}

/**
 * Given an XPath expression and a context node, this function returns
 * a the double-precision floating point number stored in the node that
 * matches the expression.  The double will be stored in the location
 * pointed to by retp.  It is an error if the XPath expression matches
 * more than one node, or if the text in the node is not numeric.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return 0 on success, or -1 on error.  On error, *retp is unchanged.
 */
int
domutil_getDoubleValue(DOMNode * ctxt, const char * path, double * retp)
{
    XPathResult * result = NULL;
    double value;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if ((result = domutil_getxpresult(ctxt, path, XPathResult::NUMBER_TYPE)) == NULL)
	goto FAIL;
    
    try {
	value = result->getNumberValue();
    } catch (const DOMException & exc) {
	goto FAIL;
    }
    *retp = value;
    goto EXIT;

FAIL:
    retval = -1;

EXIT:
//     if (result)
// 	delete result;
    return retval;
}

/**
 * Given an XPath expression and a context node, this function returns
 * an array of double-precision floating point numbers generated from
 * the space-separated list of numbers stored in the node that
 * matches the expression.  A pointer to the new double array is stored
 * in the location pointed to by retp.  It is an error if the XPath
 * expression matches more than one node, or if the text in the node is
 * not a space-separated list of numbers.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return number of items in list, or -1 on error.  On error, *retp
 *          is unchanged.
 */
int
domutil_getDoubleListValue(DOMNode * ctxt, const char * path, double ** retp)
{
    XPathResult * result = NULL;
    const XMLCh * valdomstr = NULL;
    char * tmpcontent = NULL;
    char * endptr = NULL;
    double value;
    int numvalues;
    double * dlist = NULL;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if ((result = domutil_getxpresult(ctxt, path, XPathResult::STRING_TYPE)) == NULL)
	goto FAIL;
    
    try {
	valdomstr = result->getStringValue();
    } catch (const DOMException & exc) {
	goto FAIL;
    }

    tmpcontent = XMLString::transcode(valdomstr);
    endptr = NULL;
    numvalues = 0;
    while (!((value = strtod(tmpcontent, &endptr)) == 0 &&
	     (endptr == tmpcontent))) {
	numvalues++;
	tmpcontent = endptr;
    }
    if (numvalues == 0 ||
	strspn(tmpcontent, " \t\r\n\f\v") != strlen(tmpcontent)) {
	sprintf(domutil_errorbuf, "getDoubleListValue: content of '%s' ('%s') is not a list of doubles\n", path, XMLString::transcode(valdomstr));
	goto FAIL;
    }
    if ((dlist = (double *)malloc(sizeof(double)*numvalues)) == NULL)
	goto FAIL;
    tmpcontent = XMLString::transcode(valdomstr);
    numvalues = 0;
    while (!((value = strtod(tmpcontent, &endptr)) == 0 &&
	     (endptr == tmpcontent))) {
	dlist[numvalues] = value;
	numvalues++;
	tmpcontent = endptr;
    }
    if (strspn(endptr, " \t\r\n\f\v") != strlen(endptr)) {
	sprintf(domutil_errorbuf, "domutil_getDoubleListValue: content of '%s' ('%s') is not a valid list of numbers\n", path, XMLString::transcode(valdomstr));
	goto FAIL;
    }
    retval = numvalues;
    *retp = dlist;
    goto EXIT;

FAIL:
    if (dlist)
	free(dlist);
    retval = -1;

EXIT:
//     if (result)
// 	delete result;
    return retval;
}

/**
 * Given an XPath expression and a context node, this function returns
 * a the (signed) integer stored in the node that matches the expression.
 * The integer will be stored in the location pointed to by retp.  It is
 * an error if the XPath expression matches more than one node, or if the
 * text in the node is not numeric, or if the numeric value cannot be
 * represented as a 32-bit signed integer.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return 0 on success, or -1 on error.  On error, *retp is unchanged.
 */
int
domutil_getIntValue(DOMNode * ctxt, const char * path, int * retp)
{
    double dval;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if (domutil_getDoubleValue(ctxt, path, &dval) == -1)
	goto FAIL;
    if (dval != floor(dval)) {
	sprintf(domutil_errorbuf, "domutil_getIntValue: content of '%s' is not an integer (%g)\n", path, dval);
	goto FAIL;
    }
    if (dval > (double)INT_MAX || dval < (double)INT_MIN) {
	sprintf(domutil_errorbuf, "domutil_getIntValue: content of '%s' results in integer over/underflow (%g)\n", path, dval);
	goto FAIL;
    }
    *retp = (int)dval;
    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    return retval;
}

/**
 * Given an XPath expression and a context node, this function creates
 * an array of (signed) integers generated from the space-separated list
 * of numbers stored in the node that matches the expression.  A pointer
 * to the integer array will be stored in the location pointed to by retp.
 * It is an error if the XPath expression matches more than one node, or
 * if the text in the node is not a space-separated list of numbers, or
 * if the numbers cannot be represented as 32-bit signed integers.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return number of items in list, or -1 on error.  On error, *retp
 *          is unchanged.
 */
int
domutil_getIntListValue(DOMNode * ctxt, const char * path, int ** retp)
{
    double * dlist = NULL;
    int * ilist = NULL;
    int numvals;
    int valnum;

    domutil_errorbuf[0] = '\0';

    if ((numvals = domutil_getDoubleListValue(ctxt, path, &dlist)) == -1)
	goto FAIL;
    if ((ilist = (int *)malloc(sizeof(int) * numvals)) == NULL)
	goto FAIL;
    *retp = ilist;
    valnum = 0;
    while (valnum++ < numvals) {
	double dval = *dlist;
	if (dval != floor(dval)) {
	    sprintf(domutil_errorbuf, "domutil_getIntListValue: content of '%s' is not an integer (%g)\n", path, dval);
	    goto FAIL;
	}
	if (dval > (double)INT_MAX || dval < (double)INT_MIN) {
	    sprintf(domutil_errorbuf, "domutil_getIntListValue: content of '%s' results in integer over/underflow (%g)\n", path, dval);
	    goto FAIL;
	}
	*ilist = (int)dval;
	ilist++;
	dlist++;
    }
    goto EXIT;

FAIL:
    if (dlist)
	free(dlist);
    if (ilist)
	free(ilist);
    numvals = -1;

EXIT:
    return numvals;
}

/**
 * Given an XPath expression and a context node, this function returns
 * a the unsigned integer stored in the node that matches the expression.
 * The integer will be stored in the location pointed to by retp.  It is
 * an error if the XPath expression matches more than one node, or if the
 * text in the node is not numeric, or if the numeric value cannot be
 * represented as a 32-bit unsigned integer.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return 0 on success, or -1 on error.  On error, *retp is unchanged.
 */
int
domutil_getUnsignedIntValue(DOMNode * ctxt, const char * path, unsigned int * retp)
{
    double dval;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if (domutil_getDoubleValue(ctxt, path, &dval) == -1)
	goto FAIL;
    if (dval != floor(dval)) {
	sprintf(domutil_errorbuf, "domutil_getUnsignedIntValue: content of '%s' is not an integer (%g)\n", path, dval);
	goto FAIL;
    }
    if (dval < 0 || dval > (double)UINT_MAX) {
	sprintf(domutil_errorbuf, "domutil_getUnsignedIntValue: content of '%s' results in unsigned integer over/underflow (%g)\n", path, dval);
	goto FAIL;
    }
    *retp = (unsigned int)dval;
    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    return retval;
}

/**
 * Given an XPath expression and a context node, this function creates
 * an array of unsigned integers generated from the space-separated list
 * of numbers stored in the node that matches the expression.  A pointer
 * to the unsigned integer array will be stored in the location pointed
 * to by retp.  It is an error if the XPath expression matches more than
 * one node, or if the text in the node is not a space-separated list of
 * numbers, or if the numbers cannot be represented as 32-bit unsigned
 * integers.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return number of items in list, or -1 on error.  On error, *retp
 *          is unchanged.
 */
int
domutil_getUnsignedIntListValue(DOMNode * ctxt, const char * path, unsigned int ** retp)
{
    double * dlist = NULL;
    unsigned int * ilist = NULL;
    int numvals;
    int valnum;

    domutil_errorbuf[0] = '\0';

    if ((numvals = domutil_getDoubleListValue(ctxt, path, &dlist)) == -1)
	goto FAIL;
    if ((ilist = (unsigned int *)malloc(sizeof(int) * numvals)) == NULL)
	goto FAIL;
    *retp = ilist;
    valnum = 0;
    while (valnum++ < numvals) {
	double dval = *dlist;
	if (dval != floor(dval)) {
	    sprintf(domutil_errorbuf, "domutil_getUnsignedIntListValue: content of '%s' is not an integer (%g)\n", path, dval);
	    goto FAIL;
	}
	if (dval < 0 || dval > (double)UINT_MAX) {
	    sprintf(domutil_errorbuf, "domutil_getUnsignedIntListValue: content of '%s' results in unsigned integer over/underflow (%g)\n", path, dval);
	    goto FAIL;
	}
	*ilist = (unsigned int)dval;
	ilist++;
	dlist++;
    }
    goto EXIT;

FAIL:
    if (dlist)
	free(dlist);
    if (ilist)
	free(ilist);
    numvals = -1;

EXIT:
    return numvals;
}

/**
 * Given an XPath expression and a context node, this function returns
 * the text stored in the node that matches the expression.  The
 * character array will be stored in the location pointed to by retp.
 * It is an error if the XPath expression matches more than one node.
 *
 * @param ctxt XPath context node
 * @param path XPath expression
 * @param retp pointer to result
 * @return 0 on success, or -1 on error.  On error, *retp is unchanged.
 */
int
domutil_getStringValue(DOMNode * ctxt, const char * path, char ** retp)
{
    XPathResult * result = NULL;
    const XMLCh * value = NULL;
    int retval = 0;

    domutil_errorbuf[0] = '\0';

    if ((result = domutil_getxpresult(ctxt, path, XPathResult::STRING_TYPE)) == NULL)
	goto FAIL;
    
    try {
	value = result->getStringValue();
    } catch (const DOMException & exc) {
	goto FAIL;
    }
    *retp = XMLString::transcode(value);
    goto EXIT;

FAIL:
    retval = -1;

EXIT:
//     if (result)
// 	delete result;
    return retval;
}


struct nsdef;
static void domutil_normalizeNamespaces_aux(DOMElement * cur, struct nsdef * oldnsdefs);

/**
 * This function implements, more or less, the namespace normalization
 * algorithm defined in the DOM-Level-3-Core Appendix B.
 * One modification is to replace prefixes with the default namespace
 * if at all possible.
 * There is no return value, this will bomb if something goes wrong.
 *
 * @param start root of the subtree to normalize
 */
void
domutil_normalizeNamespaces(DOMElement * start)
{
    domutil_normalizeNamespaces_aux(start, NULL);
}

typedef struct nsdef {
    int valid;
    char * prefix;
    char * ns;
} nsdef;

static void
domutil_normalizeNamespaces_aux(DOMElement * cur, nsdef * oldnsdefs)
{
    nsdef * nsdefs = NULL;
    int numdefs;

    numdefs = 0;
    nsdefs = (nsdef *)malloc(sizeof(nsdef) * 1);
    nsdefs[numdefs].valid = 0;
    nsdefs[numdefs].prefix = NULL;
    nsdefs[numdefs].ns = NULL;
    
    /* copy old defs */
    {
	int numolddefs = 0;
	int olddefnum;
	for (numolddefs = 0; oldnsdefs && oldnsdefs[numolddefs].valid; numolddefs++) { /* null */ }
	nsdefs = (nsdef *)realloc(nsdefs, sizeof(nsdef) * (numdefs + numolddefs + 1));
	/* put at end of list, for these are old, so just move terminator */
	memmove(nsdefs + numdefs + numolddefs, nsdefs + numdefs, sizeof(nsdef) * 1);
	for (olddefnum = 0; olddefnum < numolddefs; olddefnum++) {
	    nsdefs[numdefs + olddefnum].valid = 1;
	    if (oldnsdefs[olddefnum].prefix)
		nsdefs[numdefs + olddefnum].prefix = strdup(oldnsdefs[olddefnum].prefix);
	    else
		nsdefs[numdefs + olddefnum].prefix = NULL;
	    if (oldnsdefs[olddefnum].ns)
		nsdefs[numdefs + olddefnum].ns = strdup(oldnsdefs[olddefnum].ns);
	    else
		nsdefs[numdefs + olddefnum].ns = NULL;
	}
	numdefs += numolddefs;
    }

    /* find namespace attributes from current element */
    {
	DOMNamedNodeMap * nnm = NULL;
	int numattrs = 0;
	int attrnum;
	if ((nnm = cur->getAttributes()) != NULL)
	    numattrs = nnm->getLength();
	for (attrnum = 0; attrnum < numattrs; attrnum++) {
	    DOMAttr * attr = NULL;
	    char * namespaceURI = NULL;
	    char * prefix = NULL;
	    char * name = NULL;
	    char * value = NULL;
	    attr = (DOMAttr *)nnm->item(attrnum);
	    namespaceURI = XMLString::transcode(attr->getNamespaceURI());
	    prefix = XMLString::transcode(attr->getPrefix());
	    name = XMLString::transcode(attr->getNodeName());
	    value = XMLString::transcode(attr->getNodeValue());
	    if ((prefix && strcmp(prefix, "xmlns") == 0) ||
		(!prefix && name && strcmp(name, "xmlns") == 0)) {
		nsdefs = (nsdef *)realloc(nsdefs, sizeof(nsdef) * (numdefs + 1 + 1));
		memmove(nsdefs + 1, nsdefs, sizeof(nsdef) * (numdefs + 1));
		if (prefix && strcmp(prefix, "xmlns") == 0) {
		    nsdefs[0].valid = 1;
		    nsdefs[0].prefix = strdup(name+6);
		    nsdefs[0].ns = strdup(value);
		} else {
		    nsdefs[0].valid = 1;
		    nsdefs[0].prefix = NULL;
		    if (value)
			nsdefs[0].ns = strdup(value);
		    else
			nsdefs[0].ns = NULL;
		}
		numdefs++;
	    }
	    XMLString::release(&namespaceURI);
	    XMLString::release(&prefix);
	    XMLString::release(&name);
	    XMLString::release(&value);
	}
    }

    /* Now using algorithm in DOM-Level-3-Core Appendix B */

    /* do normalization for this node */
    {
	char * namespaceURI = NULL;
	char * prefix = NULL;
	nsdef * curdefaultns = NULL;
	for (curdefaultns = nsdefs; curdefaultns->valid && curdefaultns->prefix != NULL; curdefaultns++) { /* null */ }
	if (!curdefaultns->valid)
	    curdefaultns = NULL;
	namespaceURI = XMLString::transcode(cur->getNamespaceURI());
	prefix = XMLString::transcode(cur->getPrefix());
	if (namespaceURI != NULL) {
	    if (curdefaultns && curdefaultns->ns &&
		strcmp(curdefaultns->ns, namespaceURI) == 0) {
		if (prefix != NULL) {
		    cur->setPrefix(NULL);
		}
	    } else {
		nsdef * match;
		for (match = nsdefs; match->valid; match++) {
		    if ((prefix && match->prefix && strcmp(match->prefix, prefix) == 0) ||
			(prefix == NULL && match->prefix == NULL)) {
			break;
		    }
		}
		if (match->valid) {
		    if (strcmp(match->ns, namespaceURI) != 0) {
			free(match->ns);
			match->ns = strdup(namespaceURI);
		    }
		} else {
		    nsdefs = (nsdef *)realloc(nsdefs, sizeof(nsdef) * (numdefs + 1 + 1));
		    memmove(nsdefs + 1, nsdefs, sizeof(nsdef) * (numdefs + 1));
		    nsdefs[0].valid = 1;
		    if (prefix == NULL)
			nsdefs[0].prefix = NULL;
		    else
			nsdefs[0].prefix = strdup(prefix);
		    nsdefs[0].ns = strdup(namespaceURI);
		    numdefs++;
		}
	    }
	} else {
	    if (prefix) {
		fprintf(stderr, "domutil_normalizeNamespaces: Element with prefix but no namespace!\n");
	    } else {
		/* element has no prefix and no namespace */
		if (curdefaultns && curdefaultns->ns == NULL) {
		    /* default namespace is empty, so do nothing */
		} else {
		    if (curdefaultns) {
			/* default namespace is not empty, so change it */
			free(curdefaultns->ns);
			curdefaultns->ns = NULL;
		    } else {
			/* create an "empty" default namespace */
			XMLCh * xmlnsstr = NULL;
			XMLCh * emptystr = NULL;
			xmlnsstr = XMLString::transcode("xmlns");
			emptystr = XMLString::transcode("");
			cur->setAttribute(xmlnsstr, emptystr);
			XMLString::release(&emptystr);
			XMLString::release(&xmlnsstr);
			/* add to nsdefs */
			nsdefs = (nsdef *)realloc(nsdefs, sizeof(nsdef)*(numdefs+1+1));
			memmove(nsdefs + 1, nsdefs, sizeof(nsdef) * (numdefs + 1));
			nsdefs[0].valid = 1;
			nsdefs[0].prefix = NULL;
			nsdefs[0].ns = NULL;
			numdefs++;
		    }
		}
	    }
	}
	XMLString::release(&namespaceURI);
	XMLString::release(&prefix);
    }

    /* do the attributes */
    {
	DOMNamedNodeMap * nnm = NULL;
	int numattrs = 0;
	int attrnum;
	nsdef * curdefaultns = NULL;

	for (curdefaultns = nsdefs; curdefaultns->valid && curdefaultns->prefix != NULL; curdefaultns++) { /* null */ }
	if (!curdefaultns->valid)
	    curdefaultns = NULL;

	if ((nnm = cur->getAttributes()) != NULL)
	    numattrs = nnm->getLength();
	for (attrnum = 0; attrnum < numattrs; attrnum++) {
	    /* for all attrs of element */
	    DOMAttr * attr = NULL;
	    char * namespaceURI = NULL;
	    char * prefix = NULL;
	    attr = (DOMAttr *)nnm->item(attrnum);
	    namespaceURI = XMLString::transcode(attr->getNamespaceURI());
	    prefix = XMLString::transcode(attr->getPrefix());
	    if (namespaceURI) {
		if (curdefaultns && curdefaultns->ns &&
		    strcmp(curdefaultns->ns, namespaceURI) == 0) {
		    /* namespace matches default, so unset the prefix */
		    attr->setPrefix(NULL);
		} else {
		    /* if attr has no prefix, or has a prefix that conflicts
		     * with a binding already in scope */
		    nsdef * match;
		    for (match = nsdefs; match->valid; match++) {
			if (prefix && match->prefix && match->ns &&
			    strcmp(match->prefix, prefix) == 0 &&
			    strcmp(match->ns, namespaceURI) != 0)
			    break;
		    }
		    if (prefix == NULL || (match->valid && match->prefix != NULL)) {
			/* if Element is in the scope of a non-default binding
			 * for this namespace */
			nsdef * match = NULL;
			for (match = oldnsdefs; match && match->valid; match++) {
			    if (match->ns &&
				strcmp(match->ns, namespaceURI) == 0)
				break;
			}
			if (oldnsdefs && match->valid && match->prefix != NULL) {
			    /* if one or more prefix bindings are available */
			    nsdef * match;
			    for (match = nsdefs; match->valid; match++) {
				if (match->prefix && match->ns &&
				    strcmp(match->ns, namespaceURI) == 0)
				    break;
			    }
			    if (match->valid) {
				/* pick any one but prefer local bindings */
				XMLCh * newprefix = NULL;
				newprefix = XMLString::transcode(match->prefix);
				attr->setPrefix(newprefix);
				XMLString::release(&newprefix);
			    } else {
				/* create a local namespace declaration attr
				 * for this namespace */
				XMLCh * xmlnsstr = NULL;
				XMLCh * nsstr = NULL;
				XMLCh * prefixstr = NULL;
				static char newprefix[16];
				static char nsattrname[32];
				int nsnum = 1;
				while (1) {
				    nsdef * match;
				    sprintf(&newprefix[0], "xmlns:NS%d", nsnum);
				    for (match = nsdefs; match->valid; match++) {
					if (match->prefix &&
					    strcmp(match->prefix, &newprefix[0]) == 0)
					    break;
				    }
				    if (!match->valid)
					break;
				}
				strcpy(nsattrname, "xmlns:");
				strcat(nsattrname, &newprefix[0]);
				xmlnsstr = XMLString::transcode(&nsattrname[0]);
				nsstr = XMLString::transcode(namespaceURI);
				cur->setAttribute(xmlnsstr, nsstr);
				XMLString::release(&nsstr);
				XMLString::release(&xmlnsstr);
				/* add to nsdefs */
				nsdefs = (nsdef *)realloc(nsdefs, sizeof(nsdef)*(numdefs+1+1));
				memmove(nsdefs + 1, nsdefs, sizeof(nsdef) * (numdefs + 1));
				nsdefs[0].valid = 1;
				nsdefs[0].prefix = strdup(&newprefix[0]);
				nsdefs[0].ns = strdup(namespaceURI);
				numdefs++;
				/* Change the Attr to use this prefix */
				prefixstr = XMLString::transcode(&newprefix[0]);
				attr->setPrefix(prefixstr);
				XMLString::release(&prefixstr);
			    }
			}
		    } else {
			/* prefix does match but... */
			char * attrname = NULL;
			attrname = XMLString::transcode(attr->getNodeName());
			if (strcmp(namespaceURI, "http://www.w3.org/2000/xmlns/") == 0 &&
			    !(prefix && strcmp(prefix, "xmlns") == 0) &&
			    strcmp(attrname, "xmlns") != 0) {
			    /* if there is a non-default binding for this
			     * namespace in scope with a prefix other than
			     * "xmlns" */
			    nsdef * match;
			    for (match = nsdefs; match->valid; match++) {
				if (match->prefix && match->ns &&
				    strcmp(match->ns, namespaceURI) == 0 &&
				    strcmp(match->prefix, "xmlns") != 0)
				    break;
			    }
			    if (match->valid) {
				/* pick any one but prefer local bindings */
				XMLCh * newprefix = NULL;
				newprefix = XMLString::transcode(match->prefix);
				attr->setPrefix(newprefix);
				XMLString::release(&newprefix);
			    } else {
				/* create a local namespace declaration attr
				 * for this namespace */
				XMLCh * xmlnsstr = NULL;
				XMLCh * nsstr = NULL;
				XMLCh * prefixstr = NULL;
				static char newprefix[16];
				static char nsattrname[32];
				int nsnum = 1;
				while (1) {
				    nsdef * match;
				    sprintf(&newprefix[0], "xmlns:NS%d", nsnum);
				    for (match = nsdefs; match->valid; match++) {
					if (match->prefix &&
					    strcmp(match->prefix, &newprefix[0]) == 0)
					    break;
				    }
				    if (!match->valid)
					break;
				}
				strcpy(nsattrname, "xmlns:");
				strcat(nsattrname, &newprefix[0]);
				xmlnsstr = XMLString::transcode(&nsattrname[0]);
				nsstr = XMLString::transcode(namespaceURI);
				cur->setAttribute(xmlnsstr, nsstr);
				XMLString::release(&nsstr);
				XMLString::release(&xmlnsstr);
				/* add to nsdefs */
				nsdefs = (nsdef *)realloc(nsdefs, sizeof(nsdef)*(numdefs+1+1));
				memmove(nsdefs + 1, nsdefs, sizeof(nsdef) * (numdefs + 1));
				nsdefs[0].valid = 1;
				nsdefs[0].prefix = strdup(&newprefix[0]);
				nsdefs[0].ns = strdup(namespaceURI);
				numdefs++;
				/* Change the Attr to use this prefix */
				prefixstr = XMLString::transcode(&newprefix[0]);
				attr->setPrefix(prefixstr);
				XMLString::release(&prefixstr);
			    }
			}
			XMLString::release(&attrname);
		    }
		}
	    } else {
		if (prefix) {
		    fprintf(stderr, "domutil_normalizeNamespaces: Attribute with prefix but no namespace!\n");
		} else {
		    /* attr has no namespaceURI and no prefix */
		}
	    }
	}

	/* recurse */
	{
	    DOMNode * curchild = NULL;
	    DOMNode * lastchild = NULL;
	    for (curchild = cur->getFirstChild(); curchild; curchild = lastchild->getNextSibling()) {
		if (lastchild) {
		    lastchild = NULL;
		}
		if (curchild->getNodeType() == DOMNode::ELEMENT_NODE) {
		    domutil_normalizeNamespaces_aux((DOMElement *)curchild, nsdefs);
		}
		lastchild = curchild;
	    }
	    if (lastchild) {
		lastchild = NULL;
	    }
	}
    }
}

/**
 * Make the document pretty by removing extraneous text elements
 * and adding appropriate spacing for indentation.
 * There is no return value, this will bomb if something goes wrong.
 *
 * @param start root of the subtree to prettify
 * @param startindent number of spaces to indent start element
 * @param indentstep how many spaces to indent each level
 */
void
domutil_prettify(DOMNode * start, int startindent, int indentstep)
{
    DOMNode * parent = NULL;
    DOMDocument * doc = NULL;
    DOMElement * root = NULL;
    int nodetype;

    if (start->getNodeType() == DOMNode::DOCUMENT_NODE) {
	doc = (DOMDocument *)start;
    } else {
	doc = start->getOwnerDocument();
    }
    if (doc == NULL)
	return;

    root = doc->getDocumentElement();

    nodetype = start->getNodeType();

    parent = start->getParentNode();

    /* fix up space before the element */
    if (parent != NULL) {
	DOMNode * cur = NULL;
	DOMNode * prev = NULL;
	char * indentstr = NULL;
	DOMText * indentnode = NULL;
	char * value;
	indentstr = (char *)malloc(sizeof(char *) * (startindent + 2));
	indentstr[0] = '\n';
	memset(indentstr+1, ' ', startindent);
	indentstr[startindent+1] = '\0';
	cur = start;
	while ((prev = cur->getPreviousSibling()) != NULL) {
	    int nodeType;
	    int spacey = 1;
	    nodeType = prev->getNodeType();
	    if (nodeType == DOMNode::ATTRIBUTE_NODE) {
		cur = prev;
		continue;
	    }
	    if (nodeType != DOMNode::TEXT_NODE) {
		break;
	    }
	    value = XMLString::transcode(prev->getNodeValue());
	    if (strspn(value, " \n\t\v\f") != strlen(value))
		spacey = 0;
	    XMLString::release(&value); value = NULL;
	    if (spacey)
		parent->removeChild(prev);
	    if (!spacey)
		break;
	}
	try {
	    indentnode = doc->createTextNode(XMLString::transcode(indentstr));
	} catch (const DOMException & exc) {
	    fprintf(stderr, "domutil_prettify: error creating text node\n\tException: #%s\n", XMLString::transcode(exc.msg));
	}
	free(indentstr); indentstr = NULL;
	try {
	    if (parent != doc) 
		parent->insertBefore((DOMNode *)indentnode, start);
	} catch (const DOMException & exc) {
	    fprintf(stderr, "domutil_prettify: error inserting text node\n\tException: #%s\n", XMLString::transcode(exc.msg));
	}
    }

    if (parent == NULL) {
	startindent -= indentstep;
    }

    /* recurse */
    if (nodetype == DOMNode::DOCUMENT_NODE ||
	nodetype == DOMNode::ELEMENT_NODE) {
	DOMNode * curnode = NULL;
	curnode = start->getFirstChild();
	while (curnode) {
	    DOMNode * nextnode = NULL;
	    int curnodetype;
	    curnodetype = curnode->getNodeType();
	    if (curnodetype == DOMNode::ELEMENT_NODE ||
		curnodetype == DOMNode::COMMENT_NODE) {
		domutil_prettify(curnode, startindent+indentstep, indentstep);
	    }
	    nextnode = curnode->getNextSibling();
	    curnode = nextnode;
	}
    }

    /* fix up space after the element */
    if (parent != NULL) {
	DOMNode * cur = NULL;
	DOMNode * next = NULL;
	char * indentstr = NULL;
	DOMText * indentnode = NULL;
	char * value;
	indentstr = (char *)malloc(sizeof(char *) * (startindent + 2));
	indentstr[0] = '\n';
	if (startindent > indentstep) {
	    memset(indentstr+1, ' ', startindent - indentstep);
	    indentstr[startindent - indentstep + 1] = '\0';
	} else {
	    indentstr[1] = '\0';
	}
	cur = start;
	while ((next = cur->getNextSibling()) != NULL) {
	    int nodeType;
	    int spacey = 1;
	    nodeType = next->getNodeType();
	    if (nodeType == DOMNode::ATTRIBUTE_NODE) {
		cur = next;
		continue;
	    }
	    if (nodeType != DOMNode::TEXT_NODE) {
		break;
	    }
	    value = XMLString::transcode(next->getNodeValue());
	    if (strspn(value, " \n\t\v\f") != strlen(value))
		spacey = 0;
	    XMLString::release(&value); value = NULL;
	    if (spacey) {
		try {
		    parent->removeChild(next);
		} catch (const DOMException & exc) {
		    fprintf(stderr, "domutil_prettify: error removing node\n\tException: #%s\n", XMLString::transcode(exc.msg));
		}
	    }
	    if (!spacey)
		break;
	}
	try {
	    indentnode = doc->createTextNode(XMLString::transcode(indentstr));
	} catch (const DOMException & exc) {
	    fprintf(stderr, "domutil_prettify: error creating text node\n\tException: #%s\n", XMLString::transcode(exc.msg));
	}
	free(indentstr); indentstr = NULL;
	next = cur->getNextSibling(); /* null is OK */
	try {
	    if (parent != doc) 
		parent->insertBefore((DOMNode *)indentnode, next);
	} catch (const DOMException & exc) {
	    fprintf(stderr, "domutil_prettify: error inserting text node\n\tException: #%s\n", XMLString::transcode(exc.msg));
	}
    }
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.3  2003/08/01 20:40:28  gadde
 * Some updates to help ease the migration of code between DOM implementations
 *
 * Revision 1.2  2003/08/01 17:55:55  gadde
 * Merge in XERCES-specific files
 *
 * Revision 1.24.2.2  2003/07/29 16:08:44  gadde
 * Doc fixes.
 *
 * Revision 1.24.2.1  2003/07/25 17:31:38  gadde
 * Committing branch that compiles with Xerces/Pathan
 *
 * Revision 1.22  2003/06/27 15:04:52  gadde
 * Fix early unref()
 *
 * Revision 1.21  2003/06/06 19:43:07  gadde
 * Make child iteration more efficient.
 *
 * Revision 1.20  2003/06/06 02:29:56  gadde
 * Some performance enhancements (to make up for some questionable
 * code in libxml).
 *
 * Revision 1.19  2003/03/31 17:50:48  gadde
 * Prettify indentation of comments too.
 *
 * Revision 1.18  2003/01/06 18:41:58  gadde
 * Many changes/updates to comments.
 * Minor bug fixes.
 *
 * Revision 1.17  2002/12/04 17:21:50  gadde
 * Adding new module files
 *
 * Revision 1.16  2002/12/03 21:08:35  gadde
 * Fixed some docs
 *
 * Revision 1.15  2002/12/03 20:41:04  gadde
 * Big update --
 *  add new datarec module, convert more programs from domutil_ to bxh_,
 *  add new bxh_getElement* functions, and domutil_prettify.
 *
 * Revision 1.14  2002/11/25 16:21:31  gadde
 * Mega-update to merge in new library functions.
 *
 * Revision 1.13  2002/11/15 21:38:19  gadde
 * Added documentation for domutil_normalizeNamespaces
 *
 * Revision 1.12  2002/11/15 21:16:51  gadde
 * More uncommitted namespace changes
 *
 * Revision 1.11  2002/11/15 20:52:33  gadde
 * Now BXH files have the correct namespace declarations (at the expense
 * of some fragile code in dom_utils.c that breaks the GDOME abstraction
 * layer).
 *
 * Revision 1.10  2002/10/25 20:29:51  gadde
 * Moved comment to README
 *
 * Revision 1.9  2002/09/09 19:50:37  gadde
 * Added domutil_appendNewAttributeWithTextValue().
 *
 * Revision 1.8  2002/08/30 14:34:49  gadde
 * Documentation update.
 *
 * Revision 1.7  2002/08/23 20:21:46  gadde
 * *** empty log message ***
 *
 * Revision 1.6  2002/08/23 16:54:18  gadde
 * Whoops.  Fixed some logic errors in number checking.
 * Added more documentation.
 *
 * Revision 1.5  2002/08/22 15:45:57  gadde
 * Fixed some overflow logic.
 *
 * Revision 1.4  2002/08/22 15:39:39  gadde
 * Forgot to #include <limits.h>
 *
 * Revision 1.3  2002/08/22 15:37:58  gadde
 * Added documentation, and more error checking.
 *
 * Revision 1.2  2002/08/20 20:11:29  gadde
 * Added const to rcsid to avoid compiler warning.
 *
 * Revision 1.1  2002/08/20 15:32:33  gadde
 * Everything moved from libxml to libgdome (DOM).  All utility functions
 * prefixed with domutil_.  XPath support still shaky in gdome, but dom_utils.c
 * accounts for this, and extra code can be removed when gdome fully supports
 * the DOM level 3 XPath specification.
 *
 * Revision 1.1  2002/08/16 19:41:15  gadde
 * Initial import
 *
 */
