static const char rcsid[] = "$Id: bxh_utils.cpp,v 1.67 2009-04-03 00:51:57 gadde Exp $";

/* bxh_utils.cpp --
 *
 * Create and modify BXH files using these functions.
 *
 * Author: Syam Gadde (gadde@biac.duke.edu), Nov. 2002.
 */

#include <bxh_config.h>

#include <assert.h>
#include <ctype.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifdef WIN32
#include <io.h>
#define dup _dup
#define dup2 _dup2
#endif
#ifndef WIN32
#include <unistd.h>
#endif

#include <zlib.h>

/* DOM includes */
#include <libxml/parser.h>
#include <gdome.h>
#include "bxh_dom_utils.h"

/* include this after gdome (so it doesn't bork on "interface") */
#ifdef WIN32
#include <winsock.h>
#endif

/* prototypes and type declarations */
#include "bxh_utils.h"
#include "bxh_convert.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no versions specified)"
#endif
#ifndef XMLH_CVSSTR
#define XMLH_CVSSTR "(no CVS versions specified)"
#endif

#ifndef TOPLEVELELEM
#define TOPLEVELELEM "bxh"
#endif
static const char * BXHNS = "http://www.biac.duke.edu/bxh";
static const char * BXHNSATTR = NULL; /* attributes by default shouldn't have namespace */
static const char * XCEDENS = "http://nbirn.net/Resources/Users/Applications/xcede/";
static const char * XCEDE2NS = "http://www.xcede.org/xcede-2";
static const char * XMLNSNS = "http://www.w3.org/2000/xmlns/";

#define DI_DOMIMPLEMENTATION	GdomeDOMImplementation
#define DI_DOMDOCUMENT	GdomeDocument
#define DI_DOMNODE	GdomeNode
#define DI_DOMELEMENT	GdomeElement
#define DI_DOMSTRINGPTR	GdomeDOMString *

#define DI_CREATESTR(x)	gdome_str_mkref_dup(x)
#define DI_DELETESTR(x)	gdome_str_unref(x); x = NULL

#ifdef __cplusplus
extern "C" {
#endif

#ifdef HAVE_LIBXSLT
#include "xsl_xcedetobxh.c"
#include "xsl_bxhtoxcede.c"
#include "xsl_xcede2tobxh.c"
#include "xsl_bxhtoxcede2.c"
#include "libxml/parser.h"
#include "libxslt/xsltInternals.h"
#include "libxslt/transform.h"
int bxh_doXCEDE = 0;
#ifdef HAVE_LIBEXSLT
#include "libexslt/exslt.h"
int bxh_doXCEDE2 = 0;
#endif
/* This is used so we can run libXSLT on the libxml node of a Gdome document */
extern GdomeDocument * gdome_xml_doc_mkref(xmlDoc *n);
struct _Gdome_xml_Document {
    GdomeDocument super;
    const GdomeDocumentVtab *vtab;
    int refcnt;
    xmlDoc *n;
    GdomeAccessType accessType;
    /*Gdome_xml_ListenerList*/ void *ll;
    int livenodes;
};
#endif

#define CONVERTTEMPLATE(inbuf, fromtype, bufsize, retbuf, totype) {	\
    fromtype * buf = NULL;						\
    fromtype * endbuf = (fromtype *)((char *)inbuf + (bufsize));	\
    size_t retsize = sizeof(totype)*((bufsize)/sizeof(*buf));		\
    totype * newbuf = NULL;						\
    newbuf = (totype *)malloc(retsize);					\
    (retbuf) = newbuf;							\
    if ((newbuf) == NULL) {						\
	fprintf(stderr, "Error allocating %lld bytes\n", (long long int)retsize); \
    }									\
    for (buf = (fromtype *)(inbuf); buf < (endbuf); newbuf++, buf++) {	\
	*(newbuf) = (totype)*buf;					\
    }									\
}

char *
bxh_convertBufToChar(const void * inbuf, size_t bufsize, const char * elemtype)
{
    char * retbuf = NULL;
    if (strcmp(elemtype, "int8") == 0) {
	retbuf = (char *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "uint8") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned char, bufsize, retbuf, char);
    } else if (strcmp(elemtype, "int16") == 0) {
	CONVERTTEMPLATE(inbuf, short, bufsize, retbuf, char);
    } else if (strcmp(elemtype, "uint16") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned short, bufsize, retbuf, char);
    } else if (strcmp(elemtype, "int32") == 0) {
	CONVERTTEMPLATE(inbuf, int, bufsize, retbuf, char);
    } else if (strcmp(elemtype, "uint32") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned int, bufsize, retbuf, char); 
    } else if (strcmp(elemtype, "float32") == 0) {
	CONVERTTEMPLATE(inbuf, float, bufsize, retbuf, char);
    } else if (strcmp(elemtype, "float64") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, char);
    } else if (strcmp(elemtype, "double") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, char);
    } else if (strcmp(elemtype, "size_t") == 0) {
	CONVERTTEMPLATE(inbuf, size_t, bufsize, retbuf, char);
    } else if (strcmp(elemtype, "off_t") == 0) {
	CONVERTTEMPLATE(inbuf, off_t, bufsize, retbuf, char);
    }
    return retbuf;
}

unsigned char *
bxh_convertBufToUnsignedChar(const void * inbuf, size_t bufsize, const char * elemtype)
{
    unsigned char * retbuf = NULL;
    if (strcmp(elemtype, "int8") == 0) {
	CONVERTTEMPLATE(inbuf, char, bufsize, retbuf, unsigned char);
    } else if (strcmp(elemtype, "uint8") == 0) {
	retbuf = (unsigned char *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "int16") == 0) {
	CONVERTTEMPLATE(inbuf, short, bufsize, retbuf, unsigned char);
    } else if (strcmp(elemtype, "uint16") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned short, bufsize, retbuf, unsigned char);
    } else if (strcmp(elemtype, "int32") == 0) {
	CONVERTTEMPLATE(inbuf, int, bufsize, retbuf, unsigned char);
    } else if (strcmp(elemtype, "uint32") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned int, bufsize, retbuf, unsigned char); 
    } else if (strcmp(elemtype, "float32") == 0) {
	CONVERTTEMPLATE(inbuf, float, bufsize, retbuf, unsigned char);
    } else if (strcmp(elemtype, "float64") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, unsigned char);
    } else if (strcmp(elemtype, "double") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, unsigned char);
    } else if (strcmp(elemtype, "size_t") == 0) {
	CONVERTTEMPLATE(inbuf, size_t, bufsize, retbuf, unsigned char);
    } else if (strcmp(elemtype, "off_t") == 0) {
	CONVERTTEMPLATE(inbuf, off_t, bufsize, retbuf, unsigned char);
    }
    return retbuf;
}

short *
bxh_convertBufToShort(const void * inbuf, size_t bufsize, const char * elemtype)
{
    short * retbuf = NULL;
    if (strcmp(elemtype, "int8") == 0) {
	CONVERTTEMPLATE(inbuf, char, bufsize, retbuf, short);
    } else if (strcmp(elemtype, "uint8") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned char, bufsize, retbuf, short);
    } else if (strcmp(elemtype, "int16") == 0) {
	retbuf = (short *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "uint16") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned short, bufsize, retbuf, short);
    } else if (strcmp(elemtype, "int32") == 0) {
	CONVERTTEMPLATE(inbuf, int, bufsize, retbuf, short);
    } else if (strcmp(elemtype, "uint32") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned int, bufsize, retbuf, short); 
    } else if (strcmp(elemtype, "float32") == 0) {
	CONVERTTEMPLATE(inbuf, float, bufsize, retbuf, short);
    } else if (strcmp(elemtype, "float64") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, short);
    } else if (strcmp(elemtype, "double") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, short);
    } else if (strcmp(elemtype, "size_t") == 0) {
	CONVERTTEMPLATE(inbuf, size_t, bufsize, retbuf, short);
    } else if (strcmp(elemtype, "off_t") == 0) {
	CONVERTTEMPLATE(inbuf, off_t, bufsize, retbuf, short);
    }
    return retbuf;
}

unsigned short *
bxh_convertBufToUnsignedShort(const void * inbuf, size_t bufsize, const char * elemtype)
{
    unsigned short * retbuf = NULL;
    if (strcmp(elemtype, "int8") == 0) {
	CONVERTTEMPLATE(inbuf, char, bufsize, retbuf, unsigned short);
    } else if (strcmp(elemtype, "uint8") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned char, bufsize, retbuf, unsigned short);
    } else if (strcmp(elemtype, "int16") == 0) {
	CONVERTTEMPLATE(inbuf, short, bufsize, retbuf, unsigned short);
    } else if (strcmp(elemtype, "uint16") == 0) {
	retbuf = (unsigned short *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "int32") == 0) {
	CONVERTTEMPLATE(inbuf, int, bufsize, retbuf, unsigned short);
    } else if (strcmp(elemtype, "uint32") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned int, bufsize, retbuf, unsigned short); 
    } else if (strcmp(elemtype, "float32") == 0) {
	CONVERTTEMPLATE(inbuf, float, bufsize, retbuf, unsigned short);
    } else if (strcmp(elemtype, "float64") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, unsigned short);
    } else if (strcmp(elemtype, "double") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, unsigned short);
    } else if (strcmp(elemtype, "size_t") == 0) {
	CONVERTTEMPLATE(inbuf, size_t, bufsize, retbuf, unsigned short);
    } else if (strcmp(elemtype, "off_t") == 0) {
	CONVERTTEMPLATE(inbuf, off_t, bufsize, retbuf, unsigned short);
    }
    return retbuf;
}

int *
bxh_convertBufToInt(const void * inbuf, size_t bufsize, const char * elemtype)
{
    int * retbuf = NULL;
    if (strcmp(elemtype, "int8") == 0) {
	CONVERTTEMPLATE(inbuf, char, bufsize, retbuf, int);
    } else if (strcmp(elemtype, "uint8") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned char, bufsize, retbuf, int);
    } else if (strcmp(elemtype, "int16") == 0) {
	CONVERTTEMPLATE(inbuf, short, bufsize, retbuf, int);
    } else if (strcmp(elemtype, "uint16") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned short, bufsize, retbuf, int);
    } else if (strcmp(elemtype, "int32") == 0) {
	retbuf = (int *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "uint32") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned int, bufsize, retbuf, int); 
    } else if (strcmp(elemtype, "float32") == 0) {
	CONVERTTEMPLATE(inbuf, float, bufsize, retbuf, int);
    } else if (strcmp(elemtype, "float64") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, int);
    } else if (strcmp(elemtype, "double") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, int);
    } else if (strcmp(elemtype, "size_t") == 0) {
	CONVERTTEMPLATE(inbuf, size_t, bufsize, retbuf, int);
    } else if (strcmp(elemtype, "off_t") == 0) {
	CONVERTTEMPLATE(inbuf, off_t, bufsize, retbuf, int);
    }
    return retbuf;
}

unsigned int *
bxh_convertBufToUnsignedInt(const void * inbuf, size_t bufsize, const char * elemtype)
{
    unsigned int * retbuf = NULL;
    if (strcmp(elemtype, "int8") == 0) {
	CONVERTTEMPLATE(inbuf, char, bufsize, retbuf, unsigned int);
    } else if (strcmp(elemtype, "uint8") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned char, bufsize, retbuf, unsigned int);
    } else if (strcmp(elemtype, "int16") == 0) {
	CONVERTTEMPLATE(inbuf, short, bufsize, retbuf, unsigned int);
    } else if (strcmp(elemtype, "uint16") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned short, bufsize, retbuf, unsigned int);
    } else if (strcmp(elemtype, "int32") == 0) {
	CONVERTTEMPLATE(inbuf, int, bufsize, retbuf, unsigned int);
    } else if (strcmp(elemtype, "uint32") == 0) {
	retbuf = (unsigned int *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "float32") == 0) {
	CONVERTTEMPLATE(inbuf, float, bufsize, retbuf, unsigned int);
    } else if (strcmp(elemtype, "float64") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, unsigned int);
    } else if (strcmp(elemtype, "double") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, unsigned int);
    } else if (strcmp(elemtype, "size_t") == 0) {
	CONVERTTEMPLATE(inbuf, size_t, bufsize, retbuf, unsigned int);
    } else if (strcmp(elemtype, "off_t") == 0) {
	CONVERTTEMPLATE(inbuf, off_t, bufsize, retbuf, unsigned int);
    }
    return retbuf;
}

float *
bxh_convertBufToFloat(const void * inbuf, size_t bufsize, const char * elemtype)
{
    float * retbuf = NULL;
    if (strcmp(elemtype, "int8") == 0) {
	CONVERTTEMPLATE(inbuf, char, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint8") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned char, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "int16") == 0) {
	CONVERTTEMPLATE(inbuf, short, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint16") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned short, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "int32") == 0) {
	CONVERTTEMPLATE(inbuf, int, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "uint32") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned int, bufsize, retbuf, float); 
    } else if (strcmp(elemtype, "float32") == 0) {
	retbuf = (float *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "float64") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "double") == 0) {
	CONVERTTEMPLATE(inbuf, double, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "size_t") == 0) {
	CONVERTTEMPLATE(inbuf, size_t, bufsize, retbuf, float);
    } else if (strcmp(elemtype, "off_t") == 0) {
	CONVERTTEMPLATE(inbuf, off_t, bufsize, retbuf, float);
    }
    return retbuf;
}

double *
bxh_convertBufToDouble(const void * inbuf, size_t bufsize, const char * elemtype)
{
    double * retbuf = NULL;
    if (strcmp(elemtype, "int8") == 0) {
	CONVERTTEMPLATE(inbuf, char, bufsize, retbuf, double);
    } else if (strcmp(elemtype, "uint8") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned char, bufsize, retbuf, double);
    } else if (strcmp(elemtype, "int16") == 0) {
	CONVERTTEMPLATE(inbuf, short, bufsize, retbuf, double);
    } else if (strcmp(elemtype, "uint16") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned short, bufsize, retbuf, double);
    } else if (strcmp(elemtype, "int32") == 0) {
	CONVERTTEMPLATE(inbuf, int, bufsize, retbuf, double);
    } else if (strcmp(elemtype, "uint32") == 0) {
	CONVERTTEMPLATE(inbuf, unsigned int, bufsize, retbuf, double);
    } else if (strcmp(elemtype, "float32") == 0) {
	CONVERTTEMPLATE(inbuf, float, bufsize, retbuf, double);
    } else if (strcmp(elemtype, "float64") == 0) {
	retbuf = (double *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "double") == 0) {
	retbuf = (double *)malloc(bufsize);
	memcpy(retbuf, inbuf, bufsize);
    } else if (strcmp(elemtype, "size_t") == 0) {
	CONVERTTEMPLATE(inbuf, size_t, bufsize, retbuf, double);
    } else if (strcmp(elemtype, "off_t") == 0) {
	CONVERTTEMPLATE(inbuf, off_t, bufsize, retbuf, double);
    }
    return retbuf;
}

#undef FUNC
#define FUNC "bxh_createDocument"
/**
 * Create an empty BXH document.
 *
 * @return The document, or NULL on error
 */
BXHDocPtr
bxh_createDocument()
{
    DI_DOMIMPLEMENTATION * di = NULL;
    DI_DOMSTRINGPTR name = NULL;
    DI_DOMSTRINGPTR value = NULL;
    DI_DOMSTRINGPTR tmpstr = NULL;
    DI_DOMNODE * tmpnode = NULL;
    DI_DOMDOCUMENT * doc = NULL;
    DI_DOMELEMENT * bxh = NULL;

    GdomeNode * inserted = NULL;

    GdomeException exc;

    if (domutil_initialize() != 0) {
	goto FAIL;
    }

    di = gdome_di_mkref();

    name = DI_CREATESTR("bxh:" TOPLEVELELEM);
    tmpstr = DI_CREATESTR(BXHNS);
    if ((doc = gdome_di_createDocument(di, tmpstr, name, NULL, &exc)) == NULL) {
	fprintf(stderr, "DOMImplementation.createDocument: failed\n\tException #%d\n", exc);
	goto FAIL;
    }
    DI_DELETESTR(tmpstr);
    DI_DELETESTR(name);

    if ((bxh = gdome_doc_documentElement(doc, &exc)) == NULL) {
	fprintf(stderr, "Document.documentElement: failed\n\tException #%d\n", exc);
	goto FAIL;
    }

    value = DI_CREATESTR(" This is a BXH (BIAC XML Header) file. ");
    if ((tmpnode = (DI_DOMNODE *)gdome_doc_createComment(doc, value, &exc)) == NULL) {
	fprintf(stderr, "Document.createComment: failed\n\tException #%d\n", exc);
	goto FAIL;
    }
    DI_DELETESTR(value);
    if ((inserted = gdome_n_insertBefore((DI_DOMNODE *)doc, (DI_DOMNODE *)tmpnode, (DI_DOMNODE *)bxh, &exc)) != (DI_DOMNODE *)tmpnode) {
	sprintf(domutil_errorbuf, "Appending comment to doc failed\n\tException #%d\n", exc);
	goto FAIL;
    }
    gdome_n_unref(inserted, &exc); inserted = NULL;
    gdome_n_unref(tmpnode, &exc); tmpnode = NULL;

    if (domutil_setAttributeWithTextValue(doc, bxh, "xmlns:bxh", BXHNS, XMLNSNS) != 0)
	goto FAIL;
    if (domutil_setAttributeWithTextValue(doc, bxh, "xmlns", BXHNS, XMLNSNS) != 0)
	goto FAIL;
    if (domutil_setAttributeWithTextValue(doc, bxh, "version", "1.0", BXHNSATTR) != 0)
	goto FAIL;

    bxh_normalizeNamespaces((BXHElementPtr)bxh);

    goto EXIT;

FAIL:
    if (doc) {
	gdome_doc_unref(doc, &exc);
	doc = NULL;
    }

EXIT:
    if (tmpnode)
	gdome_n_unref(tmpnode, &exc);
    if (inserted)
	gdome_n_unref(inserted, &exc);
    if (name)
	DI_DELETESTR(name);
    if (value)
	DI_DELETESTR(value);
    if (tmpstr)
	DI_DELETESTR(tmpstr);
    if (bxh)
	gdome_el_unref(bxh, &exc);
    if (di)
	gdome_di_unref(di, &exc);

    return (BXHDocPtr)doc;
}

#undef FUNC
#define FUNC "bxh_readFile"
/**
 * Read a document from a BXH file.
 *
 * @param filename Name of BXH file
 * @return The document, or NULL on error
 */
BXHDocPtr
bxh_readFile(const char * filename)
{
    DI_DOMIMPLEMENTATION * di = NULL;
    DI_DOMDOCUMENT * doc = NULL;
    DI_DOMELEMENT * bxh = NULL;
    DI_DOMSTRINGPTR name = NULL;
    DI_DOMSTRINGPTR ns = NULL;

    char * localname = NULL;

    GdomeException exc;

    if (domutil_initialize() != 0) {
	goto FAIL;
    }

    di = gdome_di_mkref();
    
    if ((doc = gdome_di_createDocFromURI(di, filename, GDOME_LOAD_PARSING, &exc)) == NULL) {
	fprintf(stderr, "Error reading XML file!\n");
	goto FAIL;
    }

    if ((bxh = gdome_doc_documentElement(doc, &exc)) == NULL) {
	fprintf(stderr, "Error getting root element!\n");
	goto FAIL;
    }

    if ((name = gdome_el_nodeName(bxh, &exc)) == NULL) {
	fprintf(stderr, "Error getting node name!\n");
	goto FAIL;

    }
    localname = strchr(name->str, ':');
    if (localname == NULL) {
        localname = name->str;
    } else {
        localname++;
    }

    if ((ns = gdome_el_namespaceURI(bxh, &exc)) == NULL) {
	fprintf(stderr, "Error getting namespace!\n");
	goto FAIL;
    }
    
    if (strcmp(localname, "serieslevel") == 0 &&
	strcmp(ns->str, XCEDENS) == 0) {
#ifdef HAVE_LIBXSLT
	xmlDocPtr tmpdoc = NULL;
	xsltStylesheetPtr xsltstylesheet = NULL;
	xmlDocPtr newxmldoc = NULL;
	xmlDocPtr newxmldoc2 = NULL;
	DI_DOMDOCUMENT * newgdomedoc = NULL;
	gdome_str_unref(ns); ns = NULL;
	gdome_str_unref(name); name = NULL;
	gdome_el_unref(bxh, &exc); bxh = NULL;
	tmpdoc = xmlParseMemory(&xcedetobxh[0], sizeof(xcedetobxh));
	xsltstylesheet = xsltParseStylesheetDoc(tmpdoc);
	newxmldoc = xsltApplyStylesheet(xsltstylesheet, ((struct _Gdome_xml_Document *)doc)->n, NULL);
	xsltFreeStylesheet(xsltstylesheet); xsltstylesheet = NULL;
	newxmldoc2 = xmlCopyDoc(newxmldoc, 1); /* this cleans up the dictionary that may be in there, which causes problems later */
	xmlFreeDoc(newxmldoc);
	newgdomedoc = gdome_xml_doc_mkref(newxmldoc2);
	gdome_doc_unref(doc, &exc); doc = NULL;
	doc = newgdomedoc; newgdomedoc = NULL;
	if ((bxh = gdome_doc_documentElement(doc, &exc)) == NULL) {
	    fprintf(stderr, "Error getting root element!\n");
	    goto FAIL;
	}
	if (bxh_setAttribute(bxh, "fromxcede", "yes") != 0) {
	    fprintf(stderr, "Error setting 'fromxcede' attribute!\n");
	    goto FAIL;
	}
#else
	fprintf(stderr, "Cannot parse XCEDE XML file because this program was not compiled with libxslt.\n");
	goto FAIL;
#endif
    } else if (strcmp(localname, "XCEDE") == 0 &&
	       strcmp(ns->str, XCEDE2NS) == 0) {
#ifdef HAVE_LIBEXSLT
	xmlDocPtr tmpdoc = NULL;
	xsltStylesheetPtr xsltstylesheet = NULL;
	xmlDocPtr newxmldoc = NULL;
	xmlDocPtr newxmldoc2 = NULL;
	DI_DOMDOCUMENT * newgdomedoc = NULL;
	exsltCommonRegister(); /* get exslt:node-set() */
	gdome_str_unref(ns); ns = NULL;
	gdome_str_unref(name); name = NULL;
	gdome_el_unref(bxh, &exc); bxh = NULL;
	tmpdoc = xmlParseMemory(&xcede2tobxh[0], sizeof(xcede2tobxh));
	xsltstylesheet = xsltParseStylesheetDoc(tmpdoc);
	newxmldoc = xsltApplyStylesheet(xsltstylesheet, ((struct _Gdome_xml_Document *)doc)->n, NULL);
	xsltFreeStylesheet(xsltstylesheet); xsltstylesheet = NULL;
	newxmldoc2 = xmlCopyDoc(newxmldoc, 1); /* this cleans up the dictionary that may be in there, which causes problems later */
	xmlFreeDoc(newxmldoc);
	newgdomedoc = gdome_xml_doc_mkref(newxmldoc2);
	gdome_doc_unref(doc, &exc); doc = NULL;
	doc = newgdomedoc; newgdomedoc = NULL;
	if ((bxh = gdome_doc_documentElement(doc, &exc)) == NULL) {
	    fprintf(stderr, "Error getting root element!\n");
	    goto FAIL;
	}
	if (bxh_setAttribute(bxh, "fromxcede2", "yes") != 0) {
	    fprintf(stderr, "Error setting 'fromxcede2' attribute!\n");
	    goto FAIL;
	}
#else
	fprintf(stderr, "Cannot parse XCEDE 2 XML file because this program was not compiled with libxslt and libexslt.\n");
	goto FAIL;
#endif
    } else if (strcmp(localname, "bxh") != 0 ||
	       strcmp(ns->str, BXHNS) != 0) {
	fprintf(stderr, "Error: %s doesn't look like a supported XML file!\n",
		filename);
	goto FAIL;
    }


    if (domutil_setAttributeWithTextValue(doc, bxh, "xmlns:bxh", BXHNS, XMLNSNS) != 0)
	goto FAIL;
    if (domutil_setAttributeWithTextValue(doc, bxh, "xmlns", BXHNS, XMLNSNS) != 0)
	goto FAIL;

    bxh_normalizeNamespaces((BXHElementPtr)bxh);
    bxh_removeAutogenComments((BXHElementPtr)bxh);
    goto EXIT;

FAIL:
    if (doc) {
	gdome_doc_unref(doc, &exc);
	doc = NULL;
    }

EXIT:
    if (ns)
	gdome_str_unref(ns);
    if (name)
	gdome_str_unref(name);
    if (bxh)
	gdome_el_unref(bxh, &exc);
    if (di)
	gdome_di_unref(di, &exc);
    return (BXHDocPtr)doc;
}

/**
 * Read a BXH document from a memory buffer
 *
 * @param buf Character buffer
 * @return The document, or NULL on error
 */
BXHDocPtr
bxh_readBuffer(char * buf)
{
    DI_DOMIMPLEMENTATION * di = NULL;
    DI_DOMDOCUMENT * doc = NULL;
    DI_DOMELEMENT * bxh = NULL;

    GdomeException exc;

    if (domutil_initialize() != 0) {
	goto FAIL;
    }

    di = gdome_di_mkref();
    
    if ((doc = gdome_di_createDocFromMemory(di, buf, GDOME_LOAD_PARSING, &exc)) == NULL) {
	fprintf(stderr, "Error reading XML buffer!\n");
	goto FAIL;
    }
    goto EXIT;

    if ((bxh = gdome_doc_documentElement(doc, &exc)) == NULL) {
	fprintf(stderr, "Error getting root element!\n");
	goto FAIL;
    }

    if (domutil_setAttributeWithTextValue(doc, bxh, "xmlns:bxh", BXHNS, XMLNSNS) != 0)
	goto FAIL;
    if (domutil_setAttributeWithTextValue(doc, bxh, "xmlns", BXHNS, XMLNSNS) != 0)
	goto FAIL;

    bxh_normalizeNamespaces((BXHElementPtr)bxh);

FAIL:
    if (doc) {
	gdome_doc_unref(doc, &exc);
	doc = NULL;
    }

EXIT:
    if (di)
	gdome_di_unref(di, &exc);
    return (BXHDocPtr)doc;
}

#undef FUNC
#define FUNC "bxh_writeFile"
/**
 * Write a BXH document to a file.
 *
 * @param docp BXH document pointer
 * @param filename Name of BXH file to write
 * @return 0 on success, non-zero on error.
 */
int
bxh_writeFile(BXHDocPtr docp, const char * filename)
{
    int retval = 0;
    DI_DOMIMPLEMENTATION * di = NULL;
    DI_DOMDOCUMENT * doc = (DI_DOMDOCUMENT *)docp;
    BXHElementPtr root = NULL;
    char * fromxcedestr = NULL;
    char * fromxcede2str = NULL;
    DI_DOMDOCUMENT * newgdomedoc = NULL;

    GdomeException exc;

    root = bxh_getRootElement(docp);

#if defined(HAVE_LIBXSLT)
    fromxcedestr = bxh_getAttributeStringValue(root, "fromxcede");
    fromxcede2str = bxh_getAttributeStringValue(root, "fromxcede2");
    if (fromxcedestr != NULL) {
	bxh_removeAttribute(root, "fromxcede");
    } else if (fromxcede2str != NULL) {
	bxh_removeAttribute(root, "fromxcede2");
    }
    if ((fromxcedestr != NULL && strcmp(fromxcedestr, "yes") == 0) ||
	bxh_doXCEDE) {
	xmlDocPtr tmpdoc = NULL;
	xsltStylesheetPtr xsltstylesheet = NULL;
	xmlDocPtr newxmldoc = NULL;
	tmpdoc = xmlParseMemory(&bxhtoxcede[0], sizeof(bxhtoxcede));
	xsltstylesheet = xsltParseStylesheetDoc(tmpdoc);
	newxmldoc = xsltApplyStylesheet(xsltstylesheet, ((struct _Gdome_xml_Document *)doc)->n, NULL);
	xsltFreeStylesheet(xsltstylesheet); /* frees tmpdoc as a side effect */
	newgdomedoc = gdome_xml_doc_mkref(newxmldoc);
	doc = newgdomedoc;
	docp = (BXHDocPtr)doc;
    } else if ((fromxcede2str != NULL && strcmp(fromxcede2str, "yes") == 0) ||
	bxh_doXCEDE2) {
	xmlDocPtr tmpdoc = NULL;
	xsltStylesheetPtr xsltstylesheet = NULL;
	xmlDocPtr newxmldoc = NULL;
	tmpdoc = xmlParseMemory(&bxhtoxcede2[0], sizeof(bxhtoxcede2));
	xsltstylesheet = xsltParseStylesheetDoc(tmpdoc);
	newxmldoc = xsltApplyStylesheet(xsltstylesheet, ((struct _Gdome_xml_Document *)doc)->n, NULL);
	xsltFreeStylesheet(xsltstylesheet); /* frees tmpdoc as a side effect */
	newgdomedoc = gdome_xml_doc_mkref(newxmldoc);
	doc = newgdomedoc;
	docp = (BXHDocPtr)doc;
    }
#endif

    di = gdome_di_mkref();

    bxh_normalizeNamespaces(root);
    bxh_prettify(docp, 0, 2);

    if (!gdome_di_saveDocToFile(di, doc, filename, GDOME_SAVE_STANDARD, &exc)) {
	fprintf(stderr, "DOMImplementation.saveDocToFile: failed\n\tException #%d\n", exc);
	goto FAIL;
    }

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (fromxcedestr)
	free(fromxcedestr);
    if (fromxcede2str)
	free(fromxcede2str);
    if (di)
	gdome_di_unref(di, &exc);
    if (root)
	bxh_element_unref(root);
    if (newgdomedoc)
	gdome_doc_unref(newgdomedoc, &exc);
    return retval;
}

#undef FUNC
#define FUNC "bxh_freeDocument"
/**
 * Free the memory associated with a BXH document.
 *
 * @param docp BXH document pointer
 */
void
bxh_freeDocument(BXHDocPtr docp)
{
    DI_DOMDOCUMENT * doc = (DI_DOMDOCUMENT *)docp;

    GdomeException exc;

    gdome_doc_unref(doc, &exc);
}

/*************************/
/*** BXH File Sections ***/
/*************************/

#undef FUNC
#define FUNC "bxh_getDatarec"
/**
 * Find and return the \c \<datarec\> element that matches the given type
 * and subtype attributes.  It is an error if more than one element
 * matches.  Either type or subtype may be NULL.
 * If specified datarec doesn't exist, it is created.
 * Returned element should be freed by the caller using bxh_element_unref.
 *
 * @param docp BXH document pointer
 * @param type value for the "type" attribute
 * @param subtype value for the "subtype" attribute
 * @return a BXH element pointer, or NULL on error.
 */
BXHElementPtr
bxh_getDatarec(BXHDocPtr docp, const char * type, const char * subtype)
{
    BXHElementPtr root = NULL;
    DI_DOMDOCUMENT * doc = (DI_DOMDOCUMENT *)docp;
    DI_DOMELEMENT * datarec = NULL;
    DI_DOMNODE * retval = NULL;
    const char * pathstart = "bxh:datarec";
    char * buf = NULL;
    int bufsize = 0;

    GdomeException exc;

    root = bxh_getRootElement(docp);

    bufsize =
	strlen(pathstart) +
	((type || subtype) ? 1 : 0) +
	(type ? (8 + strlen(type)) : 0) +
	((type && subtype) ? 5 : 0) +
	(subtype ? (11 + strlen(subtype)) : 0) +
	((type || subtype) ? 1 : 0) +
	1;
    buf = (char *)malloc(sizeof(char) * bufsize);

    buf[0] = '\0';
    strcat(&buf[0], pathstart);
    if (type || subtype) {
	strcat(&buf[0], "[");
	if (type) {
	    strcat(&buf[0], "@type='");
	    strcat(&buf[0], type);
	    strcat(&buf[0], "'");
	}
	if (type && subtype) {
	    strcat(&buf[0], " and ");
	}
	if (subtype) {
	    strcat(&buf[0], "@subtype='");
	    strcat(&buf[0], subtype);
	    strcat(&buf[0], "'");
	}
	strcat(&buf[0], "]");
    }
    if (domutil_getSingleNode((DI_DOMNODE *)root, &buf[0], &retval) != 0) {
	/* create it */
	if ((datarec = domutil_appendNewChildWithTextValue(doc, (DI_DOMELEMENT *)root, "datarec", NULL, BXHNS)) == NULL)
	    goto FAIL;
	if (type && domutil_setAttributeWithTextValue(doc, datarec, "type", type, BXHNSATTR) != 0)
	    goto FAIL;
	if (subtype && domutil_setAttributeWithTextValue(doc, datarec, "subtype", subtype, BXHNSATTR) != 0)
	    goto FAIL;
	retval = (DI_DOMNODE *)datarec;
    }
    free(buf); buf = NULL;

    if (retval && gdome_n_nodeType(retval, &exc) != GDOME_ELEMENT_NODE) {
	fprintf(stderr, FUNC ": datarec node isn't an element?!?\n");
	gdome_n_unref(retval, &exc); retval = NULL;
	goto FAIL;
    }
    goto EXIT;

FAIL:
    if (domutil_errorbuf[0])
	fprintf(stderr, "%s", &domutil_errorbuf[0]);
    if (datarec) gdome_el_unref(datarec, &exc);

EXIT:
    if (root) bxh_element_unref(root);
    if (buf) free(buf);

    return (BXHElementPtr)retval;
}

#undef FUNC
#define FUNC "bxh_getDatarecArray"
/**
 * Find and return all the \c \<datarec\> elements in the document.
 * Returned elements should be freed by the caller using bxh_element_unref,
 * and the array itself should also be free()d.
 *
 * @param docp BXH document pointer
 * @return a null-terminated array of BXH element pointers, or NULL on error.
 */
BXHElementPtr *
bxh_getDatarecArray(BXHDocPtr docp)
{
    BXHElementPtr root = NULL;
    DI_DOMNODE ** retval = NULL;
    int numnodes;

    GdomeException exc;

    root = bxh_getRootElement(docp);

    if ((numnodes = domutil_getNodeArray((DI_DOMNODE *)root, "bxh:datarec", GDOME_UNORDERED_NODE_ITERATOR_TYPE, &retval)) == -1) {
	if (domutil_errorbuf[0]) {
	    fprintf(stderr, FUNC ": %s\n", domutil_errorbuf);
	}
    }

    if (retval) {
	int nodenum;
	for (nodenum = 0; nodenum < numnodes; nodenum++) {
	    if (gdome_n_nodeType(retval[nodenum], &exc) != GDOME_ELEMENT_NODE) {
		fprintf(stderr, FUNC ": datarec node isn't an element?!?\n");
		domutil_freeNodeArray(&retval);
		retval = NULL;
		break;
	    }
	}
    }

    if (retval) {
	retval = (DI_DOMNODE **)realloc(retval, sizeof(DI_DOMNODE *) * (numnodes + 1));
	retval[numnodes] = NULL;
    }

    if (root) bxh_element_unref(root);
    return (BXHElementPtr *)retval;
}

#undef FUNC
#define FUNC "bxh_getHistoryArray"
/**
 * Find and return all the \c \<history\> elements in the document.
 * Returned elements should be freed by the caller using bxh_element_unref,
 * and the array itself should also be free()d.
 *
 * @param docp BXH document pointer
 * @return a null-terminated array of BXH element pointers, or NULL on error.
 */
BXHElementPtr *
bxh_getHistoryArray(BXHDocPtr docp)
{
    BXHElementPtr root = NULL;
    DI_DOMNODE ** retval = NULL;
    int numnodes;

    GdomeException exc;

    root = bxh_getRootElement(docp);

    if ((numnodes = domutil_getNodeArray((DI_DOMNODE *)root, "bxh:history", GDOME_UNORDERED_NODE_ITERATOR_TYPE, &retval)) == -1) {
	retval = NULL;
    }

    if (retval) {
	int nodenum;
	for (nodenum = 0; nodenum < numnodes; nodenum++) {
	    if (gdome_n_nodeType(retval[nodenum], &exc) != GDOME_ELEMENT_NODE) {
		fprintf(stderr, FUNC ": history node isn't an element?!?\n");
		domutil_freeNodeArray(&retval);
		retval = NULL;
		break;
	    }
	}
    }

    if (retval) {
	retval = (DI_DOMNODE **)realloc(retval, sizeof(DI_DOMNODE *) * (numnodes + 1));
	retval[numnodes] = NULL;
    }

    if (root) bxh_element_unref(root);
    return (BXHElementPtr *)retval;
}

#undef FUNC
#define FUNC "bxh_getAcquisitionData"
/**
 * Find and return the \c \<acquisitiondata\> element.
 * Returned element should be freed by the caller using bxh_element_unref.
 * If the element doesn't exist, it is created.
 *
 * @param docp BXH document pointer
 * @return a BXH element pointer, or NULL on error.
 */
BXHElementPtr
bxh_getAcquisitionData(BXHDocPtr docp)
{
    BXHElementPtr root = NULL;
    DI_DOMNODE * retval = NULL;
    const char * path = "bxh:acquisitiondata";

    GdomeException exc;

    root = bxh_getRootElement(docp);

    if (domutil_getSingleNode((DI_DOMNODE *)root, path, &retval) != 0) {
	DI_DOMDOCUMENT * doc = NULL;
	if ((doc = gdome_n_ownerDocument((DI_DOMNODE *)root, &exc)) != NULL) {
	    retval = (DI_DOMNODE *)domutil_appendNewChildWithTextValue(doc, (DI_DOMELEMENT *)root, "acquisitiondata", NULL, BXHNS);
	    gdome_doc_unref(doc, &exc);
	}
    }

    if (retval && gdome_n_nodeType(retval, &exc) != GDOME_ELEMENT_NODE) {
	fprintf(stderr, FUNC ": acquisitiondata node isn't an element?!?\n");
	gdome_n_unref(retval, &exc);
	retval = NULL;
    }

    if (root) bxh_element_unref(root);

    return (BXHElementPtr)retval;
}

#undef FUNC
#define FUNC "bxh_getSubject"
/**
 * Find and return the \c \<subject\> element.
 * Returned element should be freed by the caller using bxh_element_unref.
 * If the element doesn't exist, it is created.
 *
 * @param docp BXH document pointer
 * @return a BXH element pointer, or NULL on error.
 */
BXHElementPtr
bxh_getSubject(BXHDocPtr docp)
{
    BXHElementPtr root = NULL;
    DI_DOMELEMENT * retval = NULL;
    const char * path = "bxh:subject";

    GdomeException exc;

    root = bxh_getRootElement(docp);

    if (domutil_getSingleNode((DI_DOMNODE *)root, path, (DI_DOMNODE **)&retval) != 0) {
	DI_DOMDOCUMENT * doc = NULL;
	if ((doc = gdome_n_ownerDocument((DI_DOMNODE *)root, &exc)) != NULL) {
	    retval = domutil_appendNewChildWithTextValue(doc, (DI_DOMELEMENT *)root, "subject", NULL, BXHNS);
	    gdome_doc_unref(doc, &exc);
	}
    }

    if (retval && gdome_el_nodeType(retval, &exc) != GDOME_ELEMENT_NODE) {
	fprintf(stderr, FUNC ": subject node isn't an element?!?\n");
	gdome_el_unref(retval, &exc);
	retval = NULL;
    }


    if (root) bxh_element_unref(root);

    return (BXHElementPtr)retval;
}

#undef FUNC
#define FUNC "bxh_getRootElement"
/**
 * Get the element corresponding to the top-level \c \<bxh\> element.
 *
 * @param docp BXH document pointer
 * @return BXH element pointer on success, NULL on error.
 */
BXHElementPtr
bxh_getRootElement(BXHDocPtr docp)
{
    DI_DOMDOCUMENT * doc = (DI_DOMDOCUMENT *)docp;
    DI_DOMNODE * retval = NULL;

    GdomeException exc;

    if ((retval = (DI_DOMNODE *)gdome_doc_documentElement(doc, &exc)) == NULL) {
	fprintf(stderr, FUNC ": Can't get root document element\n");
    }

    if (retval && gdome_n_nodeType(retval, &exc) != GDOME_ELEMENT_NODE) {
	fprintf(stderr, FUNC ": root-level node isn't an element?!?\n");
	gdome_n_unref(retval, &exc);
	retval = NULL;
    }

    return (BXHElementPtr)retval;
}

#undef FUNC
#define FUNC "bxh_addHistoryEntry"
/**
 * Add a history entry to the BXH document.
 *
 * @param docp BXH document pointer
 * @param date timestamp, or 0 if you want none
 * @param desc plain-text description of history entry
 * @return 0 on success, non-zero on error.
 */
int
bxh_addHistoryEntry(BXHDocPtr docp, time_t date, const char * desc)
{
    int retval = 0;
    BXHElementPtr bxh = NULL;
    BXHElementPtr historynode = NULL;
    BXHElementPtr entrynode = NULL;
    BXHElementPtr datenode = NULL;
    BXHElementPtr descnode = NULL;

    bxh = bxh_getRootElement(docp);

    if ((historynode = bxh_getChildElement(bxh, "history")) == NULL) {
	if ((historynode = bxh_appendAndReturnChildElement(bxh, "history", NULL)) == NULL)
	    goto FAIL;
    }
    if ((entrynode = bxh_appendAndReturnChildElement(historynode, "entry", NULL)) == NULL)
	goto FAIL;
    if (date != 0) {
	static char timebuf[64];
	struct tm * tmp;
	tmp = localtime(&date);
	sprintf(timebuf, "%04d-%02d-%02d %02d:%02d:%02d", tmp->tm_year + 1900, tmp->tm_mon+1, tmp->tm_mday, tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
	if ((datenode = bxh_appendAndReturnChildElement(entrynode, "date", timebuf)) == NULL)
	    goto FAIL;
    }
    if ((descnode = bxh_appendAndReturnChildElement(entrynode, "description", desc)) == NULL)
	goto FAIL;

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (descnode) bxh_element_unref(descnode);
    if (datenode) bxh_element_unref(datenode);
    if (entrynode) bxh_element_unref(entrynode);
    if (historynode) bxh_element_unref(historynode);
    if (bxh) bxh_element_unref(bxh);
    return retval;
}

#undef FUNC
#define FUNC "bxh_addXCEDEHistoryEntry"
/**
 * Add a XCEDE-style history entry to the BXH document.
 *
 * @param docp BXH document pointer
 * @param date timestamp, or 0 if you want none
 * @param desc plain-text description of history entry
 * @return 0 on success, non-zero on error.
 */
static int
bxh_addXCEDE2HistoryEntry(BXHDocPtr docp,
			 const char * programName,
			 const char ** programArguments,
			 int numargs)
{
    int retval = 0;
    BXHElementPtr bxh = NULL;
    BXHElementPtr historynode = NULL;
    BXHElementPtr entrynode = NULL;
    BXHElementPtr datenode = NULL;
    BXHElementPtr descnode = NULL;
    BXHElementPtr stepnode = NULL;
    int argnum;
    size_t argslen = 0;
    char * args = NULL;
    const char * envval = NULL;
    time_t date;
    double version;
    static char numbuf[20];
    int truncate = 0;
    static char truncatemsg[] = "[args truncated]";
    const size_t maxargslen = 8192;
    static char timebuf[64];
    BXHElementPtr tmpnode = NULL;

    timebuf[0] = '\0';

    bxh = bxh_getRootElement(docp);

    if ((historynode = bxh_getChildElement(bxh, "history")) == NULL) {
	if ((historynode = bxh_appendAndReturnChildElement(bxh, "history", NULL)) == NULL)
	    goto FAIL;
    }
    if ((entrynode = bxh_appendAndReturnChildElement(historynode, "entry", NULL)) == NULL)
	goto FAIL;
    date = time(NULL);
    if (date != 0) {
	time_t datetzoff = 0;
	struct tm * tmp = NULL;
	int tzoffhr = 0;
	int tzoffmin = 0;
	tmp = gmtime(&date);
	datetzoff = mktime(tmp);
	tmp = localtime(&date);
	tzoffhr = (date - datetzoff) / (60*60);
	tzoffmin = abs((int)(date - datetzoff - (tzoffhr*60*60))) / 60;
	sprintf(timebuf, "%04d-%02d-%02dT%02d:%02d:%02d%c%02d:%02d", tmp->tm_year + 1900, tmp->tm_mon+1, tmp->tm_mday, tmp->tm_hour, tmp->tm_min, tmp->tm_sec, (tzoffhr < 0) ? '-' : '+', abs(tzoffhr), tzoffmin);
	if ((datenode = bxh_appendAndReturnChildElement(entrynode, "date", timebuf)) == NULL)
	    goto FAIL;
    }
    if ((descnode = bxh_appendAndReturnChildElement(entrynode, "description", NULL)) == NULL)
	goto FAIL;
    if ((stepnode = bxh_appendAndReturnChildElement(descnode, "xcede2:processStep", NULL)) == NULL)
	goto FAIL;
    if (bxh_appendChildElement(stepnode, "xcede2:program", programName) != 0)
	goto FAIL;

    argslen = 0;
    for (argnum = 0; argnum < numargs; argnum++) {
	argslen += strlen(programArguments[argnum]) + 1;
    }
    if (argslen > maxargslen) {
	argslen = maxargslen;
	truncate = 1;
    }
    args = (char *)malloc(sizeof(char)*(argslen + 1));
    argslen = 0;
    for (argnum = 0; argnum < numargs; argnum++) {
	size_t arglen = strlen(programArguments[argnum]);
	if (truncate && argslen + arglen + 1 + sizeof(truncatemsg) >= maxargslen) {
	    strcpy(&args[argslen], &truncatemsg[0]);
	    argslen += strlen(&truncatemsg[0]);
	    args[argslen] = ' ';
	    argslen++;
	    break;
	} else {
	    strncpy(&args[argslen], programArguments[argnum], arglen);
	    argslen += strlen(programArguments[argnum]);
	    args[argslen] = ' ';
	    argslen++;
	}
    }
    argslen--;
    args[argslen] = '\0';
    if (bxh_appendChildElement(stepnode, "xcede2:programArguments", args) != 0)
	goto FAIL;
    if (bxh_appendChildElement(stepnode, "xcede2:timeStamp", &timebuf[0]) != 0)
	goto FAIL;
    if ((envval = getenv("USER")) != NULL) {
	if (bxh_appendChildElement(stepnode, "xcede2:user", envval) != 0)
	    goto FAIL;
    }
    if ((envval = getenv("HOSTNAME")) != NULL) {
	if (bxh_appendChildElement(stepnode, "xcede2:hostName", envval) != 0)
	    goto FAIL;
    }
    if ((envval = getenv("MACHTYPE")) != NULL) {
	if (bxh_appendChildElement(stepnode, "xcede2:architecture", envval) != 0)
	    goto FAIL;
    }
    if ((envval = getenv("OSTYPE")) != NULL) {
	if (bxh_appendChildElement(stepnode, "xcede2:platform", envval) != 0)
	    goto FAIL;
    }
    if (bxh_appendChildElement(stepnode, "xcede2:cvs", XMLH_CVSSTR) != 0)
	goto FAIL;
#if defined(__GNUC__)
    envval = "GCC";
#elif defined(_MSC_VER)
    envval = "MSC";
#else
    envval = "UNKNOWN";
#endif
    if ((tmpnode = bxh_appendAndReturnChildElement(stepnode, "xcede2:compiler", envval)) == NULL)
	goto FAIL;
    version = -1;
#if defined(__GNUC__)
#if defined(__GNU_PATCHLEVEL__)
    version = (__GNUC__ * 10000	+ __GNUC_MINOR__ * 100 + __GNUC_PATCHLEVEL__);
# else
    version = (__GNUC__ * 10000 + __GNUC_MINOR__ * 100);
# endif
#elif defined(_MSC_VER)
    version = (_MSC_VER);
#endif
    if (version == -1)
	strcpy(&numbuf[0], "(unknown)");
    else
	sprintf(&numbuf[0], "%g", version);
    if (bxh_setAttribute(tmpnode, "version", &numbuf[0]) != 0)
	goto FAIL;
    bxh_element_unref(tmpnode); tmpnode = NULL;
    date = time(NULL);
    if (date != 0) {
	struct tm * tmp;
	time_t ltt;
	time_t gtt;
	tmp = localtime(&date);
	localtime(&ltt);
	gmtime(&gtt);
	sprintf(timebuf, "%04d-%02d-%02dT%02d:%02d:%02d%c%04d", tmp->tm_year + 1900, tmp->tm_mon+1, tmp->tm_mday, tmp->tm_hour, tmp->tm_min, tmp->tm_sec, (gtt > ltt) ? '-' : '+', (int)(((double)ltt-(double)gtt)/60));
	if ((datenode = bxh_appendAndReturnChildElement(entrynode, "date", timebuf)) == NULL)
	    goto FAIL;
    }
    if (bxh_appendChildElement(stepnode, "xcede2:buildTimeStamp", timebuf) != 0)
	goto FAIL;
    if (bxh_appendChildElement(stepnode, "xcede2:package", XMLH_VERSIONSTR) != 0)
	goto FAIL;

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (args) free(args);
    if (stepnode) bxh_element_unref(stepnode);
    if (descnode) bxh_element_unref(descnode);
    if (datenode) bxh_element_unref(datenode);
    if (entrynode) bxh_element_unref(entrynode);
    if (historynode) bxh_element_unref(historynode);
    if (tmpnode) bxh_element_unref(tmpnode);
    if (bxh) bxh_element_unref(bxh);
    return retval;
}

#undef FUNC
#define FUNC "bxh_addXCEDEHistoryEntry"
/**
 * Add a XCEDE-style history entry to the BXH document.
 *
 * @param docp BXH document pointer
 * @param date timestamp, or 0 if you want none
 * @param desc plain-text description of history entry
 * @return 0 on success, non-zero on error.
 */
static int
bxh_addXCEDEHistoryEntry(BXHDocPtr docp,
			 const char * programName,
			 const char ** programArguments,
			 int numargs)
{
    int retval = 0;
    BXHElementPtr bxh = NULL;
    BXHElementPtr historynode = NULL;
    BXHElementPtr entrynode = NULL;
    BXHElementPtr datenode = NULL;
    BXHElementPtr descnode = NULL;
    BXHElementPtr stepnode = NULL;
    int argnum;
    size_t argslen = 0;
    char * args = NULL;
    const char * envval = NULL;
    time_t date;
    double version;
    static char numbuf[20];
    int truncate = 0;
    static char truncatemsg[] = "[args truncated]";
    const size_t maxargslen = 8192;
    static char timebuf[64];

    bxh = bxh_getRootElement(docp);

    if ((historynode = bxh_getChildElement(bxh, "history")) == NULL) {
	if ((historynode = bxh_appendAndReturnChildElement(bxh, "history", NULL)) == NULL)
	    goto FAIL;
    }
    if ((entrynode = bxh_appendAndReturnChildElement(historynode, "entry", NULL)) == NULL)
	goto FAIL;
    timebuf[0] = '\0';
    date = time(NULL);
    if (date != 0) {
	struct tm * tmp;
	tmp = localtime(&date);
	sprintf(timebuf, "%04d-%02d-%02d %02d:%02d:%02d", tmp->tm_year + 1900, tmp->tm_mon+1, tmp->tm_mday, tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
	if ((datenode = bxh_appendAndReturnChildElement(entrynode, "date", timebuf)) == NULL)
	    goto FAIL;
    }
    if ((descnode = bxh_appendAndReturnChildElement(entrynode, "description", NULL)) == NULL)
	goto FAIL;
    if ((stepnode = bxh_appendAndReturnChildElement(descnode, "processStep", NULL)) == NULL)
	goto FAIL;
    if (bxh_appendChildElement(stepnode, "programName", programName) != 0)
	goto FAIL;

    argslen = 0;
    for (argnum = 0; argnum < numargs; argnum++) {
	argslen += strlen(programArguments[argnum]) + 1;
    }
    if (argslen > maxargslen) {
	argslen = maxargslen;
	truncate = 1;
    }
    args = (char *)malloc(sizeof(char)*(argslen + 1));
    argslen = 0;
    for (argnum = 0; argnum < numargs; argnum++) {
	size_t arglen = strlen(programArguments[argnum]);
	if (truncate && argslen + arglen + 1 + sizeof(truncatemsg) >= maxargslen) {
	    strcpy(&args[argslen], &truncatemsg[0]);
	    argslen += strlen(&truncatemsg[0]);
	    args[argslen] = ' ';
	    argslen++;
	    break;
	} else {
	    strncpy(&args[argslen], programArguments[argnum], arglen);
	    argslen += strlen(programArguments[argnum]);
	    args[argslen] = ' ';
	    argslen++;
	}
    }
    argslen--;
    args[argslen] = '\0';
    if (bxh_appendChildElement(stepnode, "programArgument", args) != 0)
	goto FAIL;
    if (bxh_appendChildElement(stepnode, "version", XMLH_VERSIONSTR) != 0)
	goto FAIL;
    if (bxh_appendChildElement(stepnode, "timeStamp", timebuf) != 0)
	goto FAIL;
    if (bxh_appendChildElement(stepnode, "cvs", XMLH_CVSSTR) != 0)
	goto FAIL;
    if ((envval = getenv("USER")) != NULL) {
	if (bxh_appendChildElement(stepnode, "user", envval) != 0)
	    goto FAIL;
    } else {
	if (bxh_appendChildElement(stepnode, "user", "(unknown)") != 0)
	    goto FAIL;
    }
    if ((envval = getenv("MACHTYPE")) != NULL) {
	if (bxh_appendChildElement(stepnode, "machine", envval) != 0)
	    goto FAIL;
    } else {
	if (bxh_appendChildElement(stepnode, "machine", "(unknown)") != 0)
	    goto FAIL;
    }
    if ((envval = getenv("OSTYPE")) != NULL) {
	if (bxh_appendChildElement(stepnode, "platform", envval) != 0)
	    goto FAIL;
    } else {
	if (bxh_appendChildElement(stepnode, "platform", "(unknown)") != 0)
	    goto FAIL;
    }
    if (bxh_appendChildElement(stepnode, "platformVersion", "(unknown)") != 0)
	goto FAIL;
#if defined(__GNUC__)
    envval = "GCC";
#elif defined(_MSC_VER)
    envval = "MSC";
#else
    envval = "UNKNOWN";
#endif
    if (bxh_appendChildElement(stepnode, "compilerName", "(unknown)") != 0)
	goto FAIL;

    version = -1;
#if defined(__GNUC__)
#if defined(__GNU_PATCHLEVEL__)
    version = (__GNUC__ * 10000	+ __GNUC_MINOR__ * 100 + __GNUC_PATCHLEVEL__);
# else
    version = (__GNUC__ * 10000 + __GNUC_MINOR__ * 100);
# endif
#elif defined(_MSC_VER)
    version = (_MSC_VER);
#endif
    if (version == -1)
	strcpy(&numbuf[0], "(unknown)");
    else
	sprintf(&numbuf[0], "%g", version);
    if (bxh_appendChildElement(stepnode, "compilerVersion", &numbuf[0]) != 0)
	goto FAIL;

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (args) free(args);
    if (stepnode) bxh_element_unref(stepnode);
    if (descnode) bxh_element_unref(descnode);
    if (datenode) bxh_element_unref(datenode);
    if (entrynode) bxh_element_unref(entrynode);
    if (historynode) bxh_element_unref(historynode);
    if (bxh) bxh_element_unref(bxh);
    return retval;
}

#undef FUNC
#define FUNC "bxh_addAutoHistoryEntry"
/**
 * Add a history entry to the BXH document.
 * This is a more automated alternative to the the generic bxh_addHistoryEntry.
 *
 * @param docp BXH document pointer
 * @param programName executable name or other label
 * @param programArguments original argument list
 * @param numargs number of arguments
 * @return 0 on success, non-zero on error.
 */
int
bxh_addAutoHistoryEntry(BXHDocPtr docp,
			const char * programName,
			const char ** programArguments,
			int numargs)
{
    int retval = 0;
    int argnum = 0;
    static char hostname[32];
    size_t descbufsize = 0;
    size_t descbufpos = 0;
    char * descbuf = NULL;
    size_t argslen = 0;
    int truncate = 0;
    static char truncatemsg[] = "[args truncated]";
    const size_t maxargslen = 8192;

    BXHElementPtr root = NULL;
    char * fromxcedestr = NULL;
    char * fromxcede2str = NULL;

    root = bxh_getRootElement(docp);

#ifdef HAVE_LIBXSLT
    if (((fromxcedestr = bxh_getAttributeStringValue(root, "fromxcede")) != NULL &&
	 strcmp(fromxcedestr, "yes") == 0) ||
	bxh_doXCEDE) {
	free(fromxcedestr);
	bxh_element_unref(root);
	return bxh_addXCEDEHistoryEntry(docp, programName, programArguments, numargs);
    } else if (((fromxcede2str = bxh_getAttributeStringValue(root, "fromxcede2")) != NULL &&
		strcmp(fromxcede2str, "yes") == 0) ||
	       bxh_doXCEDE2) {
	free(fromxcede2str);
	bxh_element_unref(root);
	return bxh_addXCEDE2HistoryEntry(docp, programName, programArguments, numargs);
    }
#endif
    
    gethostname(&hostname[0], sizeof(hostname));
    hostname[sizeof(hostname)-1] = '\0';
    descbufsize = 128;
    descbufsize += strlen(hostname) + 1;
    descbufsize += strlen(programName) + 1;
    descbufsize += strlen(XMLH_VERSIONSTR) + 1;
    argslen = 0;
    for (argnum = 0; argnum < numargs; argnum++) {
	argslen += strlen(programArguments[argnum]) + 1;
    }
    if (argslen > maxargslen) {
	argslen = maxargslen;
	truncate = 1;
    }
    descbufsize += argslen;
    descbuf = (char *)malloc(sizeof(char)*(descbufsize + 1));
    if (numargs > 0) {
	descbufpos +=
	    sprintf(&descbuf[descbufpos],
		    "\nGenerated on %s by '%s' (%s) from %d input args:",
		    hostname, programName, XMLH_VERSIONSTR, numargs);
	for (argnum = 0; argnum < numargs; argnum++) {
	    size_t arglen = strlen(programArguments[argnum]);
	    if (truncate && descbufpos + arglen + 1 + sizeof(truncatemsg) + strlen(XMLH_VERSIONSTR) + 1 >= descbufsize) {
		descbuf[descbufpos++] = ' ';
		strcpy(&descbuf[descbufpos], truncatemsg);
		descbufpos += sizeof(truncatemsg);
		break;
	    } else {
		descbuf[descbufpos++] = ' ';
		strncpy(&descbuf[descbufpos], programArguments[argnum], arglen);
		descbufpos += arglen;
	    }
	}
	descbuf[descbufpos++] = '\n';
    } else {
	descbufpos +=
	    sprintf(&descbuf[0],
		    "\nGenerated on %s by '%s'.\n",
		    hostname, programName);
    }
    strcpy(&descbuf[descbufpos], XMLH_VERSIONSTR);
    descbufpos += strlen(XMLH_VERSIONSTR);
    descbuf[descbufpos] = '\0';
    if (bxh_addHistoryEntry(docp, time(NULL), &descbuf[0]) != 0)
	goto FAIL;

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (fromxcedestr)
	free(fromxcedestr);
    if (fromxcede2str)
	free(fromxcede2str);
    if (descbuf)
	free(descbuf);
    if (root)
	bxh_element_unref(root);
    return retval;
}

#undef FUNC
#define FUNC "bxh_dataReadFileStart"
/**
 * High-level datarec reading interface.
 *
 * @param inputfile path to XML file
 * @param type match only datarec with this type
 * @param subtype match only datarec with this subtype
 * @param numorddims number of ordereddims sent in ordereddimnames.  Ignored if #ordereddimnames is NULL.
 * @param dimorder list of dimension names, as sent to bxh_datarec_prepareForReadPermuteSelect().
 * @param dimselect pointer to array of #numorddims selectors, each a character array of comma-separated numbers or ranges (two numbers separated by a colon [:]).  If NULL, no selection is done.
 * @param out_bdr output structure (and continuation for bxh_dataReadFinish)
 * @return 0 on success, non-zero on error.
 */
int
bxh_dataReadFileStart(const char * inputfile, const char * type, const char * subtype, int numorddims, const char ** dimorder, const char ** dimselect, struct bxhdataread * out_bdr)
{
    int retval = 0;
    bxhrawdatarec * tempdatarec = NULL;
    int ** selectors = NULL;
    int dimnum = 0;
    BXHDocPtr docp = NULL;
    int fromtype = -1;

    fromtype = bxh_autodetect_format(inputfile);
    
    if (fromtype == -1) {
	fprintf(stderr, "Can't figure out input data type (use --help for options)\n");
	exit(-1);
    }

    bxh_check_supported_type(fromtype);

    switch (fromtype) {
    case FROM_PFILE:
	fprintf(stderr, "Error: P-file '%s' cannot be used directly in this tool (it requires one or more additional [reconstructed] image data files).  Consider wrapping the P-file and image data with pfile2bxh or bxhabsorb first.\n", inputfile);
	goto FAIL;
	break;
    case FROM_SIGNA5:
	docp = bxh_signafive2doc(inputfile);
	break;
#ifndef WIN32
    case FROM_IOWASIGNA5:
	docp = bxh_iowasignafive2doc(inputfile);
	break;
#endif
    case FROM_XIMG:
	docp = bxh_ximg2doc(inputfile);
	break;
    case FROM_ANALYZE:
	docp = bxh_analyze2doc(inputfile);
	break;
    case FROM_AFNI:
	docp = bxh_afni2doc(inputfile);
	break;
    case FROM_DICOM:
#ifdef HAVE_DCMTK
	docp = bxh_dicom2doc(inputfile);
#endif
	break;
    case FROM_MINC:
#ifdef HAVE_MINC
	docp = bxh_minc2doc(inputfile);
#endif
	break;
    case FROM_MGH:
	docp = bxh_mgh2doc(inputfile);
	break;
    case FROM_BXH:
	docp = bxh_bxh2doc(inputfile);
	break;
    default:
	assert(0);
    }
    if (docp == NULL) {
	goto FAIL;
    }

    memset(out_bdr, '\0', sizeof(*out_bdr));
    out_bdr->docp = docp;
    if ((out_bdr->imagedatap = bxh_getDatarec(out_bdr->docp, "image", NULL)) == NULL) {
	fprintf(stderr, "Getting 'image' datarec failed.\n");
	goto FAIL;
    }
    if ((out_bdr->readdatarec = bxh_datarec_createFromElement(out_bdr->imagedatap, (fromtype == FROM_BXH ? inputfile : "dummy"))) == NULL) {
	fprintf(stderr, "Creating raw datarec failed.\n");
	goto FAIL;
    }
    if (dimselect) {
	bxh_datarec_prepareForReadPermute(out_bdr->readdatarec, dimorder, numorddims);
	tempdatarec = bxh_datarec_postprocess(out_bdr->readdatarec);
	selectors = (int **)malloc(sizeof(int *)*numorddims);
	for (dimnum = 0; dimnum < numorddims; dimnum++) {
	    if (dimnum < tempdatarec->numdims) {
		selectors[dimnum] = bxh_datarec_constructSelector(dimselect[dimnum], 0, tempdatarec->dimensions[dimnum].size-1, NULL);
	    } else {
		selectors[dimnum] = (int *)malloc(sizeof(int)*1);
		selectors[dimnum][0] = -1;
	    }
	    if (selectors[dimnum] == NULL) {
		fprintf(stderr, "Error parsing selector '%s'\n", dimselect[dimnum]);
		goto FAIL;
	    }
	}
	bxh_datarec_free(tempdatarec); tempdatarec = NULL;
    }
    bxh_datarec_prepareForReadPermuteSelect(out_bdr->readdatarec, dimorder, numorddims, selectors);
    out_bdr->datarec = bxh_datarec_postprocess(out_bdr->readdatarec);
    out_bdr->dimsizes = (size_t *)malloc(sizeof(size_t)*out_bdr->datarec->numdims);
    out_bdr->pagesizes = (size_t *)malloc(sizeof(size_t)*out_bdr->datarec->numdims);
    for (dimnum = 0; dimnum < out_bdr->datarec->numdims; dimnum++) {
	out_bdr->dimsizes[dimnum] = out_bdr->datarec->dimensions[dimnum].size;
	if (dimnum == 0)
	    out_bdr->pagesizes[dimnum] = out_bdr->dimsizes[dimnum];
	else
	    out_bdr->pagesizes[dimnum] = out_bdr->pagesizes[dimnum - 1] * out_bdr->dimsizes[dimnum];
    }

    out_bdr->acqdatap = bxh_getAcquisitionData(out_bdr->docp);
    out_bdr->subjectp = bxh_getSubject(out_bdr->docp);
    out_bdr->historyarray = bxh_getHistoryArray(out_bdr->docp);

    goto EXIT;

  FAIL:
    bxh_datareaddata_free(out_bdr);
    retval = -1;

  EXIT:
    if (tempdatarec)
	bxh_datarec_free(tempdatarec);
    if (selectors) {
	for (dimnum = 0; dimnum < numorddims; dimnum++) {
	    free(selectors[dimnum]);
	    selectors[dimnum] = NULL;
	}
	free(selectors);
    }
    return retval;
}

#undef FUNC
#define FUNC "bxh_dataReadFinish"
/**
 * Continues bxh_dataReadStart().
 *
 * @param bdr continuation structure from bxh_dataReadFinish.
 * @param elemtype type data should be converted to on return.  If NULL, don't convert data.
 * @return 0 on success, non-zero on error.
 */
int
bxh_dataReadFinish(struct bxhdataread * bdr, const char * elemtype)
{
    int retval = 0;
    int msbfirst = 1;
    void * buf = NULL;
    const char * bdrtype = NULL;
    size_t elemsize = 0;
    size_t newelemsize = 0;

    msbfirst = (((char *)&msbfirst)[0] == 0);

    if (bdr->datarec) {
	bxh_datarec_free(bdr->datarec);
	bdr->datarec = NULL;
    }
    if ((bdr->datarec = bxh_datarec_readData(bdr->readdatarec, &buf, msbfirst)) == NULL) {
	fprintf(stderr, "Error reading data.\n");
	goto FAIL;
    }

    bdrtype = bdr->datarec->elemtype;
    elemsize = bdr->datarec->elemsize;

    if (elemtype == NULL) {
	bdr->dataptr = buf;
	newelemsize = elemsize;
    } else if ((strcmp(elemtype, "char") == 0 && strcmp(bdrtype, "int8") == 0) ||
	       (strcmp(elemtype, "unsigned char") == 0 && strcmp(bdrtype, "uint8") == 0) ||
	       (strcmp(elemtype, "short") == 0 && strcmp(bdrtype, "int16") == 0) ||
	       (strcmp(elemtype, "unsigned short") == 0 && strcmp(bdrtype, "uint16") == 0) ||
	       (strcmp(elemtype, "int") == 0 && strcmp(bdrtype, "int32") == 0) ||
	       (strcmp(elemtype, "unsigned int") == 0 && strcmp(bdrtype, "uint32") == 0) ||
	       (strcmp(elemtype, "float") == 0 && strcmp(bdrtype, "float32") == 0) ||
	       (strcmp(elemtype, "double") == 0 && strcmp(bdrtype, "float64") == 0)) {
	bdr->dataptr = buf;
	newelemsize = elemsize;
    } else if (strcmp(elemtype, "char") == 0) {
	bdr->dataptr = bxh_convertBufToChar(buf, bdr->datarec->datasize, bdr->datarec->elemtype);
	newelemsize = 1;
    } else if (strcmp(elemtype, "unsigned char") == 0) {
	bdr->dataptr = bxh_convertBufToUnsignedChar(buf, bdr->datarec->datasize, bdr->datarec->elemtype);
	newelemsize = 1;
    } else if (strcmp(elemtype, "short") == 0) {
	bdr->dataptr = bxh_convertBufToShort(buf, bdr->datarec->datasize, bdr->datarec->elemtype);
	newelemsize = 2;
    } else if (strcmp(elemtype, "unsigned short") == 0) {
	bdr->dataptr = bxh_convertBufToUnsignedShort(buf, bdr->datarec->datasize, bdr->datarec->elemtype);
	newelemsize = 2;
    } else if (strcmp(elemtype, "int") == 0) {
	bdr->dataptr = bxh_convertBufToInt(buf, bdr->datarec->datasize, bdr->datarec->elemtype);
	newelemsize = 4;
    } else if (strcmp(elemtype, "unsigned int") == 0) {
	bdr->dataptr = bxh_convertBufToUnsignedInt(buf, bdr->datarec->datasize, bdr->datarec->elemtype);
	newelemsize = 4;
    } else if (strcmp(elemtype, "float") == 0) {
	bdr->dataptr = bxh_convertBufToFloat(buf, bdr->datarec->datasize, bdr->datarec->elemtype);
	newelemsize = 4;
    } else if (strcmp(elemtype, "double") == 0) {
	bdr->dataptr = bxh_convertBufToDouble(buf, bdr->datarec->datasize, bdr->datarec->elemtype);
	newelemsize = 8;
    } else {
	fprintf(stderr, "Don't recognize type '%s'!\n", elemtype);
	goto FAIL;
    }
    if (buf != bdr->dataptr) {
	free(buf);
	buf = NULL;
    }

    bdr->dataptrsize = (bdr->datarec->datasize / elemsize) * newelemsize;
    
    goto EXIT;
    
  FAIL:
    if (bdr->dataptr) {
	free(bdr->dataptr);
	bdr->dataptr = NULL;
    }
    retval = -1;

  EXIT:
    return retval;
}

void
bxh_datareaddata_free(struct bxhdataread * bdr)
{
    if (bdr->readdatarec) {
	bxh_datarec_free(bdr->readdatarec);
	bdr->readdatarec = NULL;
    }
    if (bdr->datarec) {
	bxh_datarec_free(bdr->datarec);
	bdr->datarec = NULL;
    }
    if (bdr->imagedatap) {
	bxh_element_unref(bdr->imagedatap);
	bdr->imagedatap = NULL;
    }
    if (bdr->acqdatap) {
	bxh_element_unref(bdr->acqdatap);
	bdr->acqdatap = NULL;
    }
    if (bdr->subjectp) {
	bxh_element_unref(bdr->subjectp);
	bdr->subjectp = NULL;
    }
    if (bdr->historyarray) {
	BXHElementPtr * historypp = bdr->historyarray;
	while (*historypp) {
	    bxh_element_unref(*historypp);
	    *historypp = NULL;
	}
	free(bdr->historyarray);
	bdr->historyarray = NULL;
    }
    if (bdr->docp) {
	bxh_freeDocument(bdr->docp);
	bdr->docp = NULL;
    }
    if (bdr->dimsizes) {
	free(bdr->dimsizes);
	bdr->dimsizes = NULL;
    }
    if (bdr->pagesizes) {
	free(bdr->pagesizes);
	bdr->pagesizes = NULL;
    }
    if (bdr->dataptr) {
	free(bdr->dataptr);
	bdr->dataptr = NULL;
    }
}

/* generic element functions */

#undef FUNC
#define FUNC "bxh_getElementStringValue"
/**
 * Find and return the string value within the element elemp.
 * It is an error if the child is not a simple element enclosing text.
 * The value returned in retp must be free()d by the caller.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to character array
 * @return 0 on success, -1 on error
 */
int
bxh_getElementStringValue(BXHElementPtr elemp, char ** retp)
{
    return domutil_getStringValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementDoubleValue"
/**
 * Find and return the double value within the element elemp.
 * It is an error if the child is not a simple element enclosing a
 * numeric value.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to double
 * @return 0 on success, -1 on error
 */
int
bxh_getElementDoubleValue(BXHElementPtr elemp, double * retp)
{
    return domutil_getDoubleValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementDoubleListValue"
/**
 * Find and return the double list value within the element elemp.
 * It is an error if the child is not a simple element enclosing a
 * space-separated list of numeric values.
 * The array returned in retp must be free()d by the caller.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to double array
 * @return length of list on success, -1 on error
 */
int
bxh_getElementDoubleListValue(BXHElementPtr elemp, double ** retp)
{
    return domutil_getDoubleListValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementIntValue"
/**
 * Find and return the integer value within the element elemp.
 * It is an error if the child is not a simple element enclosing a
 * integral value.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to int
 * @return 0 on success, -1 on error
 */
int
bxh_getElementIntValue(BXHElementPtr elemp, int * retp)
{
    return domutil_getIntValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementIntListValue"
/**
 * Find and return the integer list value within the element elemp.
 * It is an error if the child is not a simple element enclosing a
 * space-separated list of integral values.
 * The array returned in retp must be free()d by the caller.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to double array
 * @return length of list on success, -1 on error
 */
int
bxh_getElementIntListValue(BXHElementPtr elemp, int ** retp)
{
    return domutil_getIntListValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementUnsignedIntValue"
/**
 * Find and return the unsigned integer value within the element elemp.
 * It is an error if the child is not a simple element enclosing an
 * unsigned integral value.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to unsigned int
 * @return 0 on success, -1 on error
 */
int
bxh_getElementUnsignedIntValue(BXHElementPtr elemp, unsigned int * retp)
{
    return domutil_getUnsignedIntValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementUnsignedIntListValue"
/**
 * Find and return the unsigned integer list value within the element elemp.
 * It is an error if the child is not a simple element enclosing a
 * space-separated list of unsigned integral values.
 * The array returned in retp must be free()d by the caller.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to double array
 * @return length of list on success, -1 on error
 */
int
bxh_getElementUnsignedIntListValue(BXHElementPtr elemp, unsigned int ** retp)
{
    return domutil_getUnsignedIntListValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getElementIntListValue"
/**
 * Find and return the long long integer list value within the element elemp.
 * It is an error if the child is not a simple element enclosing a
 * space-separated list of integral values.
 * The array returned in retp must be free()d by the caller.
 *
 * @param elemp BXH element within which to search
 * @param retp pointer to long long int array
 * @return length of list on success, -1 on error
 */
int
bxh_getElementLongLongIntListValue(BXHElementPtr elemp, long long int ** retp)
{
    return domutil_getLongLongIntListValue((DI_DOMNODE *)elemp, ".", retp);
}

#undef FUNC
#define FUNC "bxh_getChildElement"
/**
 * Find and return the (sole) child element of parentp whose tag
 * matches the name qname.  It is an error if there is more than
 * one child with the same name.
 * Returned element should be freed by the caller using bxh_element_unref.
 *
 * @param parentp BXH element within which to search
 * @param qname name of child element (may include prefix)
 * @return the child element, or NULL on error.
 */
BXHElementPtr
bxh_getChildElement(BXHElementPtr parentp, const char * qname)
{
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    DI_DOMNODE * result = NULL;
    char * buf = NULL;
    
    GdomeException exc;

    buf = (char *)malloc(sizeof(char) * (strlen(qname) + 4 + 1));
    buf[0] = '\0';
    if (!strchr(qname, ':'))
	strcat(&buf[0], "bxh:");
    strcat(&buf[0], qname);

    if (domutil_getSingleNode((DI_DOMNODE *)parent, &buf[0], &result) == -1)
	goto FAIL;
    if (gdome_n_nodeType(result, &exc) != GDOME_ELEMENT_NODE) {
	fprintf(stderr, FUNC ": node %s not an element\n", qname);
	goto FAIL;
    }

    goto EXIT;

FAIL:
    if (result)
	gdome_n_unref(result, &exc);
    result = NULL;

EXIT:
    if (buf)
	free(buf);
    return (BXHElementPtr)result;
}

#undef FUNC
#define FUNC "bxh_getChildElementArray"
/**
 * Find and return the child element(s) of parentp whose tags
 * match the name qname.
 * The returned array will have a NULL pointer in the position past
 * the last valid element.
 * It is an error if any of the matched nodes are not elements.
 * Each element in the returned array should be freed by the caller
 * using bxh_element_unref.
 * The returned array itself should also be free()d by the caller.
 *
 * @param parentp BXH element within which to search
 * @param qname name of child element (may include prefix)
 * @return a null-terminated array of child elements, or NULL on error.
 */
BXHElementPtr *
bxh_getChildElementArray(BXHElementPtr parentp, const char * qname)
{
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    DI_DOMNODE ** result = NULL;
    int numnodes;
    int nodenum;
    char * buf = NULL;
    
    GdomeException exc;

    buf = (char *)malloc(sizeof(char) * (strlen(qname) + 4 + 1));
    buf[0] = '\0';
    if (strpbrk(qname, "*/[]()@+") == NULL) {
	if ((numnodes = domutil_getChildArray(parent, qname, BXHNS, &result)) == -1)
	    goto FAIL;
    } else {
	strcat(&buf[0], qname);

	if ((numnodes = domutil_getNodeArray((DI_DOMNODE *)parent, &buf[0], GDOME_UNORDERED_NODE_ITERATOR_TYPE, &result)) == -1)
	    goto FAIL;
	for (nodenum = 0; nodenum < numnodes; nodenum++) {
	    if (gdome_n_nodeType(result[nodenum], &exc) != GDOME_ELEMENT_NODE) {
		fprintf(stderr, FUNC ": node %s not an element\n", qname);
		goto FAIL;
	    }
	}
	result = (DI_DOMNODE **)realloc(result, sizeof(DI_DOMNODE *) * (numnodes + 1));
	result[numnodes] = NULL;
    }

    goto EXIT;

FAIL:
    if (result) {
	for (nodenum = 0; nodenum < numnodes; nodenum++) {
	    gdome_n_unref(result[nodenum], &exc);
	}
	free(result);
    }
    result = NULL;

EXIT:
    if (buf)
	free(buf);
    return (BXHElementPtr *)result;
}

#undef FUNC
#define FUNC "bxh_getChildElementStringValue"
/**
 * Find and return the value within the (sole) child element of parentp
 * whose tag matches the name qname.  It is an error if there is more
 * than one child with the same name, or if the child is not a simple
 * element enclosing text.  The return value must be free()d by the caller.
 *
 * @param parentp BXH element within which to search
 * @param qname name of child element (may include prefix)
 * @return a newly-allocated string, or NULL on error.
 */
char *
bxh_getChildElementStringValue(BXHElementPtr parentp, const char * qname)
{
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    char * result = NULL;
    char * buf = NULL;
    
    buf = (char *)malloc(sizeof(char) * (strlen(qname) + 4 + 1));
    buf[0] = '\0';
    if (!strchr(qname, ':'))
	strcat(&buf[0], "bxh:");
    strcat(&buf[0], qname);

    if (domutil_getStringValue((DI_DOMNODE *)parent, &buf[0], &result) == -1)
	goto FAIL;

    goto EXIT;

FAIL:
    if (result)
	free(result);
    result = NULL;

EXIT:
    if (buf)
	free(buf);
    return result;
}

#undef FUNC
#define FUNC "bxh_setChildElement"
/**
 * Replace (if any) the sole child element of parentp whose tag matches
 * the name qname with a new child element of the same name, and with
 * the given text value.  It is an error if there exists more than one
 * child with the same name.
 *
 * @param parentp parent BXH element
 * @param qname name of child element (may include prefix)
 * @param value text string value of child element
 * @return 0 on sucess, non-zero on error.
 */
int
bxh_setChildElement(BXHElementPtr parentp,
		    const char * qname, const char * value)
{
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    DI_DOMNODE * child = NULL;
    char * buf = NULL;
    
    GdomeException exc;

    buf = (char *)malloc(sizeof(char) * (strlen(qname) + 4 + 1));
    buf[0] = '\0';
    if (!strchr(qname, ':'))
	strcat(&buf[0], "bxh:");
    strcat(&buf[0], qname);

    if (domutil_getSingleNode((DI_DOMNODE *)parent, &buf[0], &child) == 0) {
	if (bxh_removeChildElement(parent, child) != NULL) {
	    gdome_n_unref(child, &exc);
	}
    }
    return bxh_appendChildElement(parentp, &buf[0], value);
}

#undef FUNC
#define FUNC "bxh_appendChildElement"
/**
 * Add a new child element of parentp with tag qname and value value.
 *
 * @param parentp parent BXH element
 * @param qname name of child element (may include prefix)
 * @param value text string value of child element
 * @return 0 on sucess, non-zero on error.
 */
int
bxh_appendChildElement(BXHElementPtr parentp,
		       const char * qname, const char * value)
{
    BXHElementPtr newchild = bxh_appendAndReturnChildElement(parentp, qname, value);
    if (newchild == NULL)
	return -1;
    bxh_element_unref(newchild); newchild = NULL;
    return 0;
}

#undef FUNC
#define FUNC "bxh_appendAndReturnChildElement"
/**
 * Add a new child element of parentp with tag qname and value value,
 * and return the child.  The returned element should be freed by the
 * caller using bxh_element_unref().
 *
 * @param parentp parent BXH element
 * @param qname name of child element (may include prefix)
 * @param value text string value of child element
 * @return 0 on sucess, non-zero on error.
 */
BXHElementPtr
bxh_appendAndReturnChildElement(BXHElementPtr parentp,
				const char * qname, const char * value)
{
    DI_DOMDOCUMENT * doc = NULL;
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    DI_DOMNODE * child = NULL;
    char * buf = NULL;
    
    GdomeException exc;

    if ((doc = gdome_n_ownerDocument((DI_DOMNODE *)parentp, &exc)) == NULL)
	goto FAIL;

    buf = (char *)malloc(sizeof(char) * (strlen(qname) + 4 + 1));
    buf[0] = '\0';
    if (!strchr(qname, ':'))
	strcat(&buf[0], "bxh:");
    strcat(&buf[0], qname);

    if ((child = (DI_DOMNODE *)domutil_appendNewChildWithTextValue(doc, parent, &buf[0], value, BXHNS)) == NULL)
	goto FAIL;

    goto EXIT;

FAIL:
    if (child)
	gdome_n_unref(child, &exc);
    child = NULL;

EXIT:
    if (buf)
	free(buf);
    if (doc)
	gdome_doc_unref(doc, &exc);
    return (BXHElementPtr)child;
}

#undef FUNC
#define FUNC "bxh_removeChildElement"
/**
 * Remove childp from parentp.
 *
 * @param parentp parent BXH element
 * @param childp child BXH element
 * @return valid reference to child on sucess (must be unref()'d by caller, NULL on error
 */
BXHElementPtr
bxh_removeChildElement(BXHElementPtr parentp, BXHElementPtr childp)
{
    DI_DOMNODE * removed = NULL;
    
    GdomeException exc;

    if ((removed = gdome_n_removeChild((DI_DOMNODE *)parentp, (DI_DOMNODE *)childp, &exc)) == NULL)
	return NULL;
    return removed;
}

#undef FUNC
#define FUNC "bxh_element_unref"
/**
 * Decrement the reference count for the given element.
 *
 * @param elemp BXH element
 * @return 0 on sucess, non-zero on error.
 */
int
bxh_element_unref(BXHElementPtr elemp)
{
    GdomeException exc;
    gdome_el_unref((DI_DOMELEMENT *)elemp, &exc);
    return exc;
}

#undef FUNC
#define FUNC "bxh_getElementName"
/**
 * return the (local) name of the element.
 *
 * @param elemp BXH element
 * @return name string on sucess, NULL on error.  Returned string must be free'd by caller.
 */
char *
bxh_getElementName(BXHElementPtr elemp)
{
    GdomeException exc;
    GdomeDOMString * str;
    char * retval = NULL;
    char * localname = NULL;
    str = gdome_el_nodeName((DI_DOMELEMENT *)elemp, &exc);
    if (str == NULL)
	return NULL;
    localname = strchr(str->str, ':');
    if (localname == NULL) {
        localname = str->str;
    } else {
        localname++;
    }
    retval = strdup(localname);
    gdome_str_unref(str);
    return retval;
}

/* generic attribute functions */

#undef FUNC
#define FUNC "bxh_getAttributeStringValue"
/**
 * Find and return the value of the attribute of parentp whose tag
 * matches the name qname.  The return value must be free()d by the
 * caller.
 *
 * @param parentp BXH element within which to search
 * @param name name of child attribute
 * @return a newly-allocated string, or NULL on error.
 */
char *
bxh_getAttributeStringValue(BXHElementPtr parentp, const char * name)
{
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    char * path = NULL;
    char * result = NULL;

    path = (char *)malloc(sizeof(char)*strlen(name)+2);
    path[0] = '\0';
    strcat(path, "@");
    strcat(path, name);
    
    if (domutil_getStringValue((DI_DOMNODE *)parent, path, &result) == -1)
	goto FAIL;
    
    goto EXIT;

FAIL:
    if (result)
	free(result);
    result = NULL;
    
EXIT:
    if (path)
	free(path);
    return result;
}

#undef FUNC
#define FUNC "bxh_setAttribute"
/**
 * Replace (if any) the attribute of parentp whose tag matches the name
 * name with a new attribute with the given name and text value.
 *
 * @param parentp parent BXH element
 * @param name name of attribute
 * @param value text string value of attribute
 * @return 0 on sucess, non-zero on error.
 */
int
bxh_setAttribute(BXHElementPtr parentp, const char * name, const char * value)
{
    int retval = 0;
    DI_DOMDOCUMENT * doc = NULL;
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;

    GdomeException exc;

    if ((doc = gdome_n_ownerDocument((DI_DOMNODE *)parentp, &exc)) == NULL)
	goto FAIL;

    if (domutil_setAttributeWithTextValue(doc, parent, name, value, BXHNSATTR) == -1)
	goto FAIL;

    goto EXIT;

FAIL:
    retval = -1;

EXIT:
    if (doc)
	gdome_doc_unref(doc, &exc);
    return retval;
}

#undef FUNC
#define FUNC "bxh_removeAttribute"
/**
 * Remove attribute name from parentp.
 *
 * @param parentp parent BXH element
 * @param name attribute name
 */

void
bxh_removeAttribute(BXHElementPtr parentp, const char * name)
{
    DI_DOMSTRINGPTR namestr = NULL;
    
    GdomeException exc;

    namestr = gdome_str_mkref_dup(name);
    gdome_el_removeAttribute((DI_DOMELEMENT *)parentp, namestr, &exc);
    gdome_str_unref(namestr);
}

#undef FUNC
#define FUNC "bxh_attribute_unref"
/**
 * Decrement the reference count for the given attribute.
 *
 * @param attrp BXH attribute
 * @return 0 on sucess, non-zero on error.
 */
int
bxh_attribute_unref(BXHAttributePtr attrp)
{
    GdomeException exc;
    gdome_a_unref((GdomeAttr *)attrp, &exc);
    return exc;
}


/* generic comment functions */

#undef FUNC
#define FUNC "bxh_appendComment"
/**
 * Create a comment with the given text and append it
 * to the given parent element.
 *
 * @param parentp the parent of the new comment
 * @param valuestr text that will appear inside the comment
 *            If NULL, then the element will be empty.
 * @param prependnewline if non-zero, will prepend a newline before comment
 * @return The new child element, or NULL on error
 */
BXHCommentPtr
bxh_appendComment(BXHElementPtr parentp,
		  const char * valuestr,
		  int prependnewline)
{
    DI_DOMDOCUMENT * doc = NULL;
    DI_DOMELEMENT * parent = (DI_DOMELEMENT *)parentp;
    GdomeComment * retval = NULL;

    GdomeException exc;

    if ((doc = gdome_n_ownerDocument((DI_DOMNODE *)parent, &exc)) == NULL)
	goto FAIL;

    if ((retval = domutil_appendComment(doc, parent, valuestr, prependnewline)) == NULL)
	goto FAIL;

    goto EXIT;

FAIL:
    retval = NULL;

EXIT:
    if (doc)
	gdome_doc_unref(doc, &exc);
    return retval;
}

#undef FUNC
#define FUNC "bxh_comment_unref"
/**
 * Decrement the reference count for the given comment.
 *
 * @param commentp BXH comment
 * @return 0 on sucess, non-zero on error.
 */
int
bxh_comment_unref(BXHCommentPtr commentp)
{
    GdomeException exc;
    gdome_c_unref((GdomeComment *)commentp, &exc);
    return exc;
}


/* miscellaneous BXH functions */

/**
 * See domutil_normalizeNamespaces().
 *
 * @param start root of the subtree to normalize
 */
void
bxh_normalizeNamespaces(BXHElementPtr start)
{
    domutil_normalizeNamespaces((DI_DOMELEMENT *)start);
}

/**
 * See domutil_prettify().
 *
 * @param start root of the subtree to prettify
 * @param startindent number of spaces to indent start element
 * @param indentstep how many spaces to indent each level
 */
void
bxh_prettify(BXHDocPtr start, int startindent, int indentstep) {
    domutil_prettify((DI_DOMNODE *)(DI_DOMDOCUMENT *)start, startindent, indentstep);
}

/**
 * See domutil_removeAutogenComments().
 *
 * @param start root of the subtree to normalize
 */
void
bxh_removeAutogenComments(BXHElementPtr start)
{
    domutil_removeAutogenComments((DI_DOMNODE *)start);
}

#define swaps(x) (((x & 0xff) << 8) | ((x & 0xff00) >> 8))
#define swapl(x) (((x & 0xff) << 24) | ((x & 0xff00) << 8) | ((x & 0xff0000) >> 8) | ((x & 0xff000000) >> 24))

static int
ispfile(FILE * fp)
{
    return 0;
}

static int
issignafive(FILE * fp)
{
    char magic[4];
    if (fseek(fp, 0, SEEK_SET) == -1)
	return 0;
    if (fread(&magic[0], 4, 1, fp) != 1)
	return 0;
    if (strncmp(&magic[0], "IMGF", 4) == 0)
	return 1;
    return 0;
}

static int
isximg(FILE * fp)
{
    char magic[4];
    if (fseek(fp, 3228, SEEK_SET) == -1)
	return 0;
    if (fread(&magic[0], 4, 1, fp) != 1)
	return 0;
    if (strncmp(&magic[0], "IMGF", 4) == 0)
	return 1;
    return 0;
}

static int
isdicom(FILE * fp)
{
    int ind;
    char magic[132];
    int msbfirst = 0;
    int domsbfirst = 0;
    if (fseek(fp, 0, SEEK_SET) == -1)
	return 0;
    if (fread(&magic[0], 132, 1, fp) != 1)
	goto ISDICOM_META;
    if (strncmp(&magic[128], "DICM", 4) == 0)
	return 1;

  ISDICOM_META:
    msbfirst = 1;
    msbfirst = (((char *)&msbfirst)[0] == 0);
    for (domsbfirst = 0; domsbfirst < 2; domsbfirst++) {
	int valid = 1;
	int tryelems = 128;
	int numelems = 0;
	unsigned int lastgroup = 0;
	unsigned int lastelem = 0;
	int isexplicit = -1;
	long curpos = 0;
	
	if (fseek(fp, 0, SEEK_SET) == -1)
	    return 0;

	while (curpos != -1 && numelems < tryelems) {
	    unsigned short group = 0;
	    unsigned short elem = 0;
	    int lenlen = 0;
	    unsigned int elemlen = 0;
	    unsigned int tmpuint = 0;
	    unsigned short tmpushort = 0;
	    char vr[2];
	    numelems++;
	    if (fread(&group, 2, 1, fp) != 1)
		break;
	    if (msbfirst != domsbfirst) group = swaps(group);
	    if (fread(&elem, 2, 1, fp) != 1)
		break;
	    if (msbfirst != domsbfirst) elem = swaps(elem);
	    if (group < lastgroup) {
		valid = 0;
		break;
	    } else if (group == lastgroup && elem <= lastelem) {
		valid = 0;
		break;
	    }
	    if (fread(&vr[0], 2, 1, fp) != 1)
		break;
	    if (isexplicit == -1) {
		isexplicit =
		    ((vr[0] == toupper(vr[0])) && (vr[1] == toupper(vr[1]))) &&
		    ((vr[0] != tolower(vr[0])) && (vr[1] != tolower(vr[1])));
	    }
	    lenlen = 2;
	    if (isexplicit) {
		if (strncmp(&vr[0], "OB", 2) == 0 ||
		    strncmp(&vr[0], "UN", 2) == 0 ||
		    strncmp(&vr[0], "OW", 2) == 0 ||
		    strncmp(&vr[0], "SQ", 2) == 0) {
		    lenlen = 4;
		    if (fseek(fp, 2, SEEK_CUR) == -1)
			break;
		}
	    } else {
		lenlen = 4;
		if (fseek(fp, -2, SEEK_CUR) == -1)
		    break;
	    }
	    if (lenlen == 2) {
		if (fread(&tmpushort, lenlen, 1, fp) != 1)
		    break;
		elemlen = tmpushort;
	    } else if (lenlen == 4) {
		if (fread(&tmpuint, lenlen, 1, fp) != 1)
		    break;
		elemlen = tmpuint;
	    }
	    if (fseek(fp, elemlen, SEEK_CUR) == -1)
		break;
	    if (feof(fp))
		break;
	    lastgroup = group;
	    lastelem = elem;
	    curpos = ftell(fp);
	}
	if (valid && ((numelems >= tryelems) || (feof(fp) && numelems > 2)))
	    return 1;
    }
    return 0;
}

static int
isminc(FILE * fp)
{
    char magic[4];
    if (fseek(fp, 0, SEEK_SET) == -1)
	return 0;
    if (fread(&magic[0], 4, 1, fp) != 1)
	return 0;
    if (strncmp(&magic[0], "CDF", 3) == 0)
	return 1;
    return 0;
}

static int
isanalyze(const char * filename)
{
    if (strlen(filename) > 4 &&
	(strcmp(filename + strlen(filename) - 4, ".hdr") == 0 ||
	 strcmp(filename + strlen(filename) - 4, ".img") == 0)) {
	char * hdrname = NULL;
	struct stat statbuf;
	hdrname = strdup(filename);
	strcpy(hdrname + strlen(filename) - 4, ".hdr");
	if (stat(hdrname, &statbuf) == 0 &&
	    statbuf.st_size == 348) {
	    FILE * fp2;
	    if ((fp2 = fopen(hdrname, "rb")) != NULL) {
		char buf[4];
		if (fread(&buf[0], 4, 1, fp2) == 1) {
		    if ((buf[0] == 0x00 && buf[1] == 0x00 &&
			 buf[2] == 0x01 && buf[3] == 0x5C) ||
			(buf[0] == 0x5C && buf[1] == 0x01 &&
			 buf[2] == 0x00 && buf[3] == 0x00)) {
			return 1;
		    }
		}
	    }
	}
    }
    if ((strlen(filename) > 4 &&
	 strcmp(filename + strlen(filename) - 4, ".nii") == 0) ||
	(strlen(filename) > 7 &&
	 strcmp(filename + strlen(filename) - 7, ".nii.gz") == 0)) {
	struct stat statbuf;
	gzFile fp2 = NULL;
	if ((fp2 = gzopen(filename, "rb")) != NULL) {
	    char buf[4];
	    if (gzseek(fp2, 344, SEEK_SET) != -1 &&
		gzread(fp2, &buf[0], 4) == 4) {
		if (strcmp(&buf[0], "ni1") == 0 ||
		    strcmp(&buf[0], "n+1") == 0) {
		    gzclose(fp2); fp2 = NULL;
		    return 1;
		}
	    }
	    gzclose(fp2); fp2 = NULL;
	}
    }
    return 0;
}

static int
isafni(const char * filename)
{
    if (strlen(filename) > 5 &&
	strcmp(filename + strlen(filename) - 5, ".HEAD") == 0) {
	/* what other tests should we do? */
	return 1;
    }
    return 0;
}

static int
ismgh(const char * filename)
{
    if (strlen(filename) > 4 &&
	(strcmp(filename + strlen(filename) - 4, ".mgh") == 0 ||
	 strcmp(filename + strlen(filename) - 4, ".mgz") == 0)) {
	/* what other tests should we do? */
	return 1;
    }
    return 0;
}

static int
isbxh(const char * filename)
{
    BXHDocPtr docp = NULL;
    int newstderr;
    int nullfd;
    int retval = 0;
    if ((newstderr = dup(2)) == -1) {
	fprintf(stderr, "isbxh: Error duplicating stderr!\n");
	return 0;
    }
#ifdef WIN32
    nullfd = open("nul", O_WRONLY);
    if (nullfd == -1) {
	fprintf(stderr, "isbxh: Error opening nul!\n");
	return 0;
    }
#else
    nullfd = open("/dev/null", O_WRONLY);
    if (nullfd == -1) {
	fprintf(stderr, "isbxh: Error opening /dev/null!\n");
	return 0;
    }
#endif
    if (dup2(nullfd, 2) == -1) {
        close(nullfd);
        dup2(newstderr, 2);
	fprintf(stderr, "isbxh: Error duplicating file descriptor!\n");
	return 0;
    }
    if ((docp = bxh_readFile(filename)) != NULL) {
	bxh_freeDocument(docp); docp = NULL;
	retval = 1;
    }
    close(nullfd);
    dup2(newstderr, 2);
    close(newstderr);
    return retval;
}

int
bxh_autodetect_format(const char * filename)
{
    int retval = -1;
    FILE * fp = NULL;
    fp = fopen(filename, "rb");
    if (fp == NULL) {
	fprintf(stderr, "Error opening file %s\n", filename);
	exit(-1);
    }
    if (ispfile(fp))
	retval = FROM_PFILE;
    else if (issignafive(fp))
	retval = FROM_SIGNA5;
    else if (isximg(fp))
	retval = FROM_XIMG;
    else if (isdicom(fp))
	retval = FROM_DICOM;
    else if (isminc(fp))
	retval = FROM_MINC;
    fclose(fp);
    fp = NULL;
    if (retval != -1) 
	1; // no-op
    else if (isanalyze(filename))
	retval = FROM_ANALYZE;
    else if (isafni(filename))
	retval = FROM_AFNI;
    else if (ismgh(filename))
	retval = FROM_MGH;
    else if (isbxh(filename))
	retval = FROM_BXH;
    return retval;
}

void
bxh_check_supported_type(int fromtype)
{
#ifndef HAVE_DCMTK
    if (fromtype == FROM_DICOM) {
	fprintf(stderr, "Sorry, this program was not compiled with DICOM support!\n");
	exit(-1);
    }
#endif
#ifndef HAVE_MINC
    if (fromtype == FROM_MINC) {
	fprintf(stderr, "Sorry, this program was not compiled with MINC support!\n");
	exit(-1);
    }
#endif
    return;
}

#ifdef __cplusplus
}
#endif

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.66  2009/02/24 21:25:23  gadde
 * Actually attempt to read file to see if it is XML/BXH/XCEDE (rather than depend on extension)
 *
 * Revision 1.65  2009/02/17 18:30:30  gadde
 * Accept selected data less than 4-D.
 *
 * Revision 1.64  2009/02/17 14:20:10  gadde
 * Fix relative path for non-XML inputs
 *
 * Revision 1.63  2009/01/15 20:55:18  gadde
 * New organization of data read functions to allow for reading of non-bxh data directly by most tools
 *
 * Revision 1.62  2008/03/07 21:24:19  gadde
 * off_t is not reliably a signed type.
 *
 * Revision 1.61  2008/03/07 18:20:33  gadde
 * Add preliminary XCEDE 2 support.
 *
 * Revision 1.60  2008/01/28 00:59:47  gadde
 * Fix memory leak.
 *
 * Revision 1.59  2007/05/11 15:50:17  gadde
 * Add dataptrsize field to reflect size of data (returned by bxh_dataReadFinish) in memory
 *
 * Revision 1.58  2007/03/14 20:57:16  gadde
 * Add version string to history entry.
 *
 * Revision 1.57  2007/02/23 23:25:56  gadde
 * Fix a nasty memory error due to our onconventional use of gdome2 and libxslt, as well as some other minor memory issues.
 *
 * Revision 1.56  2007/01/26 21:11:25  gadde
 * More memory fixes to last (truncating args) update.
 *
 * Revision 1.55  2007/01/26 19:09:26  gadde
 * Memory fix to last (truncating args) update.
 *
 * Revision 1.54  2007/01/26 18:34:53  gadde
 * Truncate args in history if very large.
 *
 * Revision 1.53  2007/01/16 19:33:51  gadde
 * Memory fixes.
 *
 * Revision 1.52  2007/01/16 19:01:56  gadde
 * Reinstate a previously-problematic unref().
 *
 * Revision 1.51  2007/01/12 16:24:02  gadde
 * Add bxh_getElementName.
 *
 * Revision 1.50  2007/01/09 18:24:05  gadde
 * Fix memory leak.
 *
 * Revision 1.49  2006/08/07 17:49:06  gadde
 * Remove incorrect unref.
 *
 * Revision 1.48  2006/07/07 19:05:01  gadde
 * Remove an unref() that seems to cause problems.
 *
 * Revision 1.47  2006/07/07 18:36:34  gadde
 * A couple memory management fixes
 *
 * Revision 1.46  2006/05/31 21:08:15  gadde
 * Unreference return value of bxh_getRootElement.
 *
 * Revision 1.45  2006/05/31 19:32:50  gadde
 * Fix access to freed memory.
 *
 * Revision 1.44  2006/05/31 14:32:15  gadde
 * Remove unnecessary consts
 *
 * Revision 1.43  2006/05/04 19:22:34  gadde
 * Don't use extra copies when reading data that's already in the data
 * type we need.
 *
 * Revision 1.42  2005/07/07 15:10:51  gadde
 * Memory fixes.
 *
 * Revision 1.41  2005/05/09 20:33:55  gadde
 * Add function to remove AUTOGEN comments, and call it when reading files.
 *
 * Revision 1.40  2005/04/25 13:59:23  gadde
 * Fix some memory leaks and also automatically write XCEDE if input
 * was from XCEDE.
 *
 * Revision 1.39  2005/03/18 17:03:10  gadde
 * Don't attempt to construct selectors for dimensions that don't exist
 * (in bxh_dataReadStart).
 *
 * Revision 1.38  2005/02/25 21:48:04  gadde
 * XCEDE updates
 *
 * Revision 1.37  2005/01/18 16:31:33  gadde
 * Add size_t and off_t to conversion routines.
 *
 * Revision 1.36  2004/12/14 19:38:35  gadde
 * If selector arg to bxh_dataReadStart is NULL, don't construct/send selectors.
 * Fill in rest of fields in bxhdataread.
 *
 * Revision 1.35  2004/12/13 20:26:44  gadde
 * Add documentation to bxh_dataReadStart/Finish and allow for
 * non-conversion of data.
 *
 * Revision 1.34  2004/11/12 21:03:31  gadde
 * Add all arguments to history info.
 *
 * Revision 1.33  2004/11/12 15:03:04  gadde
 * Add some functions for easier data reading.
 *
 * Revision 1.32  2004/09/07 21:16:36  gadde
 * Don't set namespace on attributes.
 *
 * Revision 1.31  2004/07/06 18:33:42  gadde
 * Some memory fixes.
 *
 * Revision 1.30  2004/07/06 13:52:40  gadde
 * Some memory fixes.
 *
 * Revision 1.29  2004/06/09 14:23:28  gadde
 * libgdome/ now  part of -I paths
 *
 * Revision 1.28  2004/05/31 16:38:07  gadde
 * Some memory fixes
 *
 * Revision 1.27  2004/05/27 22:02:30  gadde
 * Some updates to support phantomqa
 *
 * Revision 1.26  2004/04/02 16:23:53  gadde
 * Move dom_utils to bxh_dom_utils, and add some missing dependencies
 *
 * Revision 1.25  2004/03/25 19:00:20  gadde
 * Add xslt include for older versions
 *
 * Revision 1.24  2004/03/12 16:29:04  gadde
 * Minor updates
 *
 * Revision 1.23  2004/02/26 22:21:40  gadde
 * Windows fix.
 *
 * Revision 1.22  2004/02/24 18:01:06  gadde
 * Move history generation to bxh_utils.cpp, and add option to create BIRN files
 *
 * Revision 1.21  2004/02/23 20:10:50  gadde
 * Add preliminary support for BIRN XML (requires libxslt)
 *
 * Revision 1.20  2004/02/23 19:30:33  gadde
 * Add preliminary support for BIRN XML (requires libxslt)
 *
 * Revision 1.19  2003/08/04 19:21:20  gadde
 * Add support for reading from memory buffer.
 *
 * Revision 1.18  2003/08/04 17:15:55  gadde
 * Code rearrangement
 *
 * Revision 1.17  2003/08/01 20:40:28  gadde
 * Some updates to help ease the migration of code between DOM implementations
 *
 * Revision 1.16  2003/07/29 16:06:15  gadde
 * Doc fix.
 *
 * Revision 1.15  2003/07/29 15:19:12  gadde
 * Some -Wall fixes
 *
 * Revision 1.14  2003/07/28 20:56:55  gadde
 * Slight integration of libxml2/Xerces branches
 *
 * Revision 1.13  2003/07/25 17:42:14  gadde
 * Random fixes in the move from C to C++
 *
 * Revision 1.12  2003/07/21 16:46:49  gadde
 * Code-prettiness updates, for the most part, to further protect BXH
 * library users from particulars of DOM implementation (esp. C vs. C++).
 *
 * Revision 1.11  2003/06/18 19:36:59  gadde
 * Fix DOM call.
 *
 * Revision 1.10  2003/06/18 18:52:53  gadde
 * Add function to read BXH file from memory buffer.
 *
 * Revision 1.9  2003/04/17 19:33:45  gadde
 * Use more bxh functions (as opposed to domutils functions).
 * Add bxh_appendAndReturnChildElement for convenience.
 *
 * Revision 1.8  2003/02/12 19:31:36  gadde
 * Updated doxygen documentation.
 *
 * Revision 1.7  2003/02/11 17:49:15  gadde
 * Fix documentation of List functions
 *
 * Revision 1.6  2003/01/06 18:19:07  gadde
 * Fixed off-by-one month error.
 * Many comments changed/updated.
 *
 * Revision 1.5  2002/12/06 17:19:43  gadde
 * Use #defines for xmlns namespace
 *
 * Revision 1.4  2002/12/04 17:21:50  gadde
 * Adding new module files
 *
 * Revision 1.3  2002/12/03 21:08:35  gadde
 * Fixed some docs
 *
 * Revision 1.2  2002/12/03 20:41:03  gadde
 * Big update --
 *  add new datarec module, convert more programs from domutil_ to bxh_,
 *  add new bxh_getElement* functions, and domutil_prettify.
 *
 * Revision 1.1  2002/11/25 16:23:10  gadde
 * Mega-update to merge in new library functions.
 *
 */
