static const char rcsid[] = "$Id: bxr_epochavg.cpp,v 1.104 2009-01-15 20:55:19 gadde Exp $";

/*
 * bxh_epochavg.cpp --
 * 
 *  Create averages of time-series data over all epochs matching particular
 *  events defined in an event file.
 *  Modeled after portions of tstatprofile2.m by Josh Bizzell.
 */

#include <bxh_config.h>

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <libgen.h>

#ifdef WIN32
#define strcasecmp stricmp
#endif

#if !defined(HAVE_LIBGSL) || !defined(HAVE_LIBXML2)
int main(int argc, char * argv[])
{
    fprintf(stderr, "Sorry -- to use this program, this package must be compiled with libxml2 and\nGSL (GNU Scientific Library) support!\n");
    return -1;
}
#else /* #ifndef HAVE_LIBGSL */

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>
#include <time.h>
#include <limits.h>

#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#include <string>
#include <map>
#include <vector>
#include <functional>

#include <gsl/gsl_spline.h>

#include "bxh_datarec.h"
#include "bxh_eventlib.h"
#include "bxh_niftilib.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

#define VERBOSE 0

#define MININTERP_PTS 2

struct eventrange_t {
    int rbegin;
    int rmid;
    int rend;
};

/////////////
// Following from http://www.azillionmonkeys.com/qed/userInput.html

long getInputFrag (
		   /* @out@ */ char * buf, long len,
		   int termchar,
		   int (* cbGetc) (void * ctx), void * cbGetcCtx) {
    long i;
    int c;

    if (!buf || len < 0 || !cbGetc) return -1;

    for (c = (~termchar) | 1, i=0; i < len; i++) {
        if (c == (int) '\0' || c == termchar || EOF == (c = cbGetc (cbGetcCtx))) {
            buf[i] = '\0';
            break;
        }
        buf[i] = (char) c;
    }
    return i;
}

/* Work around an ANSI requirement that casting a function pointer does not
   change its parameters in any way. */

static int xfgetc (void * ctx) {
    return fgetc ((FILE *) ctx);
}

/* Very much like fgets except that it returns the length of the read string 
   or -1 if there was a problem reading the string and it also terminates 
   early if a '\0' is read. A terminating '\0' is always added to the end. */

long fgetstr (char * buff, int n, FILE * fp) {
    long len;

    if (!buff) return -1;

    if (1 == n) {
	buff[0] = '\0';
	return 0;
    }

    if (n < 1 || !fp) return -1;

    len = getInputFrag (buff, n-1, (int) '\n', xfgetc, fp);
    if ((len) >= 0) buff[len] = '\0';
    return len;
}

#define getstr(buff, n) fgetstr ((buff), (n), stdin)

/* Fills *p with a malloced buffer that contains an input line. The line is 
   always '\0' terminated. If p is NULL, it is not filled in. The number of 
   characters read is returned. */

size_t fgetstralloc (char ** p, FILE * fp) {
    size_t mlen, slen, dlen, len;
    char * s, * t;

    if (!fp) return (size_t) -1;

    if (!p) {
	char blk[64];
	for (slen = 0; ; slen += 64) {
	    len = (size_t) getInputFrag (blk, 64, (int) '\n', xfgetc, fp);
	    if (len != 64 || blk[len-1] == '\n' || blk[len-1] == '\0') return slen + len;
	}
    }

    mlen = 8;
    slen = 0;
    if (NULL == (s = (char *) malloc (mlen))) return (size_t) -1;

    for (;;) {
	mlen += mlen;
	if (NULL == (t = (char *)realloc (s, mlen))) break; /* unsafe for passwords */
	len = (size_t) getInputFrag ((s = t) + slen, dlen = mlen - slen - 1, (int) '\n', xfgetc, fp);

	slen += len;
	if (len < dlen || t[slen-1] == '\n' || t[slen-1] == '\0') {
	    s[slen] = '\0';
	    *p = s;
	    return slen;
	}
    }

    free (s);
    return (size_t) -1;
}

#define getstralloc(p) fgetstralloc ((p), stdin)

// End from http://www.azillionmonkeys.com/qed/userInput.html
/////////////

#undef FUNC
#define FUNC "try_reading_timing_file"
static
xmlDocPtr
try_reading_timing_file(const char * eventfile, FILE * logfp)
{
    char * tmpfilename = strdup(eventfile);
    char * tmpbase = basename(tmpfilename);
    char * dotpos = strrchr(tmpbase, '.');
    if (dotpos != NULL) {
	*dotpos = '\0';
    }
    std::string base(tmpbase);
    free(tmpfilename);
    FILE * fp = fopen(eventfile, "r");
    size_t linesize = 0;
    char * lineptr = NULL;
    xmlDocPtr doc = NULL;
    xmlNodePtr events = NULL;
    while ((linesize = fgetstralloc(&lineptr, fp)) != (size_t) -1) {
	float onset=0, duration=0, weight=0;
	char numbuf[64];
	int scanret = sscanf(lineptr, "%f %f %f", &onset, &duration, &weight);
	if (scanret == EOF) {
	    free(lineptr); lineptr = NULL;
	    break;
	}
	if (scanret == 0) {
	    free(lineptr); lineptr = NULL;
	    continue;
	}
	if (scanret > 0 && scanret < 3) {
	    fprintf(stderr, "Bad timing file line, skipping file %s: %s\n", eventfile, lineptr);
	    free(lineptr); lineptr = NULL;
	    if (doc != NULL) {
		xmlFreeDoc(doc);
		doc = NULL;
	    }
	    break;
	}
	if (doc == NULL) {
	    doc = xmlNewDoc((xmlChar *)"1.0");
	    if (doc == NULL) {
		fprintf(stderr, "Error creating XML document!\n");
		free(lineptr); lineptr = NULL;
		break;
	    }
	    events = xmlNewDocNode(doc, NULL, (xmlChar *)"events", NULL);
	    if (events == NULL) {
		fprintf(stderr, "Error creating events node!\n");
		xmlFreeDoc(doc);
		free(lineptr); lineptr = NULL;
		break;
	    }
	    xmlAddChild((xmlNodePtr)doc, events);
	}
	xmlNodePtr eventnode = xmlNewDocNode(doc, NULL, (xmlChar *)"event", NULL);
	xmlAddChild(events, eventnode);
	xmlNodePtr onsetnode = xmlNewDocNode(doc, NULL, (xmlChar *)"onset", NULL);
	xmlAddChild(eventnode, onsetnode);
	sprintf(numbuf, "%f", onset);
	xmlNodePtr onsettext = xmlNewText((xmlChar *)numbuf);
	xmlAddChild(onsetnode, onsettext);
	xmlNodePtr durationnode = xmlNewDocNode(doc, NULL, (xmlChar *)"duration", NULL);
	xmlAddChild(eventnode, durationnode);
	sprintf(numbuf, "%f", duration);
	xmlNodePtr durationtext = xmlNewText((xmlChar *)numbuf);
	xmlAddChild(durationnode, durationtext);
	xmlNodePtr valuenode = xmlNewDocNode(doc, NULL, (xmlChar *)"value", NULL);
	xmlAddChild(eventnode, valuenode);
	xmlAttrPtr valuenameattr = xmlNewProp(valuenode, (xmlChar *)"name", (xmlChar *)"filename");
	xmlNodePtr valuenametext = xmlNewText((xmlChar *)base.c_str());
	xmlAddChild(valuenode, valuenametext);
	free(lineptr); lineptr = NULL;
    }
    if (lineptr != NULL) {
	free(lineptr);
    }
    fclose(fp);
    if (logfp)
      fprintf(logfp,
	      "Converted timing file %s to XML internally.\n"
	      "Use event query   filename('%s')  or XPath query  value[@name='filename']='%s'  to match events in file %s.\n",
	      eventfile, base.c_str(), base.c_str(), eventfile);
    //xmlSaveFormatFileEnc("-", doc, "UTF-8", 1);
    return doc;
}

#undef FUNC
#define FUNC "get_event_list"
static
int
get_event_list(const char * eventfiles, int numqueries, char ** queries, char ** queryfilters, char ** queryepochexcludes, char ** querylabels, double epochdurbefore, double epochdurafter, std::map<double, qvec > & eventlist, FILE * logfp)
{
    int retval = 0;
    const char * queryprefix = "//*[local-name()='events']/*[local-name()='event']";
    std::vector<xmlDocPtr> docs;
    xmlDocPtr mergedoc = NULL;
    xmlDocPtr sortdoc = NULL;
    xmlDocPtr canondoc = NULL;
    size_t indefile = 0;

    std::vector<std::string> eventfilelist;
    const char * curpos = eventfiles;
    while (curpos[0] != '\0') {
	const char * commapos = strchr(curpos, ',');
	if (commapos == NULL) {
	    eventfilelist.push_back(std::string(curpos));
	    curpos += strlen(curpos);
	} else {
	    eventfilelist.push_back(std::string(curpos, (commapos - curpos)));
	    curpos = commapos + 1;
	}
    }

    for (indefile = 0; indefile < eventfilelist.size(); indefile++) {
	xmlDocPtr doc = xmlReadFile(eventfilelist[indefile].c_str(), NULL, 0);
	if (doc == NULL) {
	    doc = try_reading_timing_file(eventfilelist[indefile].c_str(), logfp);
	    if (doc == NULL) {
		fprintf(stderr, "Error: could not parse file %s\n", eventfilelist[indefile].c_str());
		goto FAIL;
	    }
	}
	docs.push_back(doc);
    }

    if ((mergedoc = merge_event_lists(docs, queryprefix, logfp)) == NULL) {
	fprintf(stderr, "Error merging event files!\n");
	goto FAIL;
    }
    if ((sortdoc = xmlCopyDoc(mergedoc, 1)) == NULL) {
	fprintf(stderr, "Error copying XML doc!\n");
	goto FAIL;
    }
    if (sort_event_list(sortdoc, queryprefix, logfp) != 0) {
	fprintf(stderr, "Error sorting events!\n");
	goto FAIL;
    }
    if ((canondoc = xmlCopyDoc(sortdoc, 1)) == NULL) {
	fprintf(stderr, "Error copying XML doc!\n");
	goto FAIL;
    }
    if (canonicalize_event_list(canondoc, queryprefix, logfp) != 0) {
	fprintf(stderr, "Error canonicalizing events!\n");
	goto FAIL;
    }

#if VERBOSE
    xmlDocDump(logfp, canondoc);
#endif

    if (match_events(sortdoc, numqueries, queries, querylabels, eventlist, logfp) != 0) {
	goto FAIL;
    }
    if (filter_events(canondoc, numqueries, queryfilters, querylabels, eventlist, logfp) != 0) {
	goto FAIL;
    }
    if (exclude_epochs(canondoc, numqueries, queryepochexcludes, querylabels, epochdurbefore, epochdurafter, eventlist, logfp) != 0) {
	goto FAIL;
    }

    goto EXIT;
    
  FAIL:
    retval = -1;
    
  EXIT:
    {
	size_t docind;
	for (docind = 0; docind < docs.size(); docind++) {
	    if (docs[docind])
		xmlFreeDoc(docs[docind]);
	}
    }
    if (mergedoc)
	xmlFreeDoc(mergedoc);
    if (sortdoc)
	xmlFreeDoc(sortdoc);
    if (canondoc)
	xmlFreeDoc(canondoc);
    return retval;
}

#undef FUNC
#define FUNC "precompute_event_ranges"
static
void
precompute_event_ranges(std::map<double, qvec > & eventlist,
			double * timepoints,
			size_t dimsize_t,
			unsigned int startpt,
			unsigned int endpt,
			int opt_basestart,
			int opt_baseend,
			double epochdurbefore,
			double epochdurafter,
			double TR,
			int opt_nointerp,
			int opt_mirrorflip,
			FILE * logfp,
			std::pair<double, qvec> * & eventarray,
			eventrange_t * & eventranges)
{
    std::map<double, eventrange_t> eventrangesmap; /* maps event time to time interval [rbegin, rmid, rend] to be used in output */
    std::map<double, qvec >::iterator eventiter;
    size_t eventnum;
    FILE * logerr = NULL;

    if (logfp != NULL && logfp != stderr)
	logerr = stderr;

    {
	size_t indt;
	fprintf(stderr, "precompute_event_ranges:\n");
	fprintf(stderr, " timepoints: ");
	for (indt = 0; indt < dimsize_t; indt++) {
	    fprintf(stderr, " %g", timepoints[indt]);
	}
	fprintf(stderr, "\n");
	fprintf(stderr, " eventlist times: ");
	for (eventiter = eventlist.begin();
	     eventiter != eventlist.end();
	     eventiter++) {
	    double etime = (*eventiter).first;
	    fprintf(stderr, " %g", etime);
	}
	fprintf(stderr, "\n");
	fprintf(stderr, " basestart=%d, baseend=%d, epochdurbefore=%g, epochdurafter=%g, TR=%g\n", opt_basestart, opt_baseend, epochdurbefore, epochdurafter, TR);
    }

    /* pre-compute event data ranges */
    eventiter = eventlist.begin();
    while (eventiter != eventlist.end()) {
	double etime = (*eventiter).first;
	double etimepre  = etime - epochdurbefore;
	double etimepost = etime + epochdurafter;
	off_t lo = 0, med = 0, hi = 0;
	off_t rbegin, rmid, rend;
	double datastart = 0, dataend = 0;
	double blstart = 0, blend = 0;
	double interpstart = 0, interpend = 0;
	std::map<double, qvec >::iterator eventiter2;

	/* find closest timepoints to event range using binary search */

	/* middle of range (event itself) */
	lo = 0;
	hi = dimsize_t;
	while (lo < hi) {
	    med = (lo + hi) / 2;
	    if (etime < timepoints[med]) {
		hi = med;
	    } else if (timepoints[med] < etime) {
		lo = med + 1;
	    } else {
		break;
	    }
	}
	/* if not exact match, get the closest timepoint */
	if (etime < timepoints[med] &&
	    med > 0 &&
	    timepoints[med] - etime > etime - timepoints[med-1]) {
	    med--;
	} else if (etime > timepoints[med] &&
		   med < (off_t)dimsize_t - 1 &&
		   etime - timepoints[med] > timepoints[med+1] - etime) {
	    med++;
	}
	rmid = med;

	/* beginning of range */
	lo = 0;
	hi = dimsize_t;
	while (lo < hi) {
	    med = (lo + hi) / 2;
	    if (etimepre < timepoints[med]) {
		hi = med;
	    } else if (timepoints[med] < etimepre) {
		lo = med + 1;
	    } else {
		break;
	    }
	}
	/* if not exact match, get the timepoint before */
	if (etimepre < timepoints[med]) {
	    med--;
	}
	rbegin = med;

	/* end of range */
	lo = 0;
	hi = dimsize_t;
	while (lo < hi) {
	    med = (lo + hi) / 2;
	    if (etimepost < timepoints[med]) {
		hi = med;
	    } else if (timepoints[med] < etimepost) {
		lo = med + 1;
	    } else {
		break;
	    }
	}
	/* if not exact match, get the timepoint after */
	if (etimepost > timepoints[med]) {
	    med++;
	}
	rend = med;

	datastart = timepoints[startpt];
	dataend = timepoints[endpt] + TR;
	blstart = (rmid + opt_basestart) * TR;
	blend = (rmid + opt_baseend + 1) * TR;
	interpstart = rmid * TR;
	interpend = rmid * TR;
	if (!opt_nointerp && !opt_mirrorflip && etime != timepoints[rmid]) {
	    interpstart = (rbegin - MININTERP_PTS) * TR;
	}
	if (!opt_nointerp && !opt_mirrorflip && etime != timepoints[rmid]) {
	    interpend = (rend + MININTERP_PTS + 1) * TR;
	}

	if (etimepre < datastart || etimepre >= dataend ||
	    etimepost <= datastart || etimepost >= dataend ||
	    blstart < datastart || blstart >= dataend ||
	    blend <= datastart || blstart > dataend ||
	    interpstart < datastart || interpstart >= dataend ||
	    interpend <= datastart || interpend > dataend) {
	    if (logerr)
		fprintf(logerr,
			"Ignoring event at time %g because the following ranges\n"
			"fall outside data range:\t(%g - %g secs)\n",
			etime, datastart, dataend);
	    if (logfp)
		fprintf(logfp,
			"Ignoring event at time %g because the following ranges\n"
			"fall outside data range:\t(%g - %g secs)\n",
			etime, datastart, dataend);
	    if (etimepre < datastart || etimepre >= dataend ||
		etimepost <= datastart || etimepost >= dataend) {
		if (logerr) fprintf(logerr, "    epoch\t\t\t  (%g - %g secs)\n", etimepre, etimepost);
		if (logfp)  fprintf(logfp,  "    epoch\t\t\t  (%g - %g secs)\n", etimepre, etimepost);
	    }
	    if (blstart < datastart || blstart >= dataend ||
		blend <= datastart || blstart > dataend) {
		if (logerr) fprintf(logerr, "    baseline range\t\t  (%g - %g secs)\n", blstart, blend);
		if (logfp)  fprintf(logfp,  "    baseline range\t\t  (%g - %g secs)\n", blstart, blend);
	    }
	    if (interpstart < datastart || interpstart >= dataend ||
		interpend <= datastart || interpend > dataend) {
		if (logerr) fprintf(logerr, "    interpolation range\t\t  (%g - %g secs)\n", interpstart, interpend);
		if (logfp)  fprintf(logfp,  "    interpolation range\t\t  (%g - %g secs)\n", interpstart, interpend);
	    }
	    
	    eventiter2 = eventiter;
	    eventiter++;
	    eventlist.erase(eventiter2);
	    continue;
	}

	eventrangesmap[etime].rbegin = rbegin;
	eventrangesmap[etime].rmid = rmid;
	eventrangesmap[etime].rend = rend;
	eventiter++;
    } /* for each event */

    /* create arrays so we don't have to use iterators
     * through the event list for every voxel.
     */
    {
	eventarray = new std::pair<double, qvec>[eventlist.size()];
	eventranges = new eventrange_t[eventlist.size()];
	for (eventiter = eventlist.begin(), eventnum = 0;
	     eventiter != eventlist.end();
	     eventiter++, eventnum++) {
	    double etime = (*eventiter).first;
	    eventarray[eventnum] = *eventiter;
	    eventranges[eventnum] = eventrangesmap[etime];
	}
    }
}

#undef FUNC
#define FUNC "update_query_counts"
static
void
update_query_counts(int numqueries, std::map<double, qvec > eventlist, char ** opt_querylabels, FILE * logfp, size_t * numepochs)
{
    FILE * logerr = NULL;
    int indquery;

    if (logfp != NULL && logfp != stderr)
	logerr = stderr;

    for (indquery = 0; indquery < numqueries; indquery++) {
	size_t numevents = 0;
	std::map<double, qvec>::iterator eiter;
	size_t qvind;
	for (eiter = eventlist.begin(); eiter != eventlist.end(); eiter++) {
	    qvec & qlist = (*eiter).second;
	    size_t qsize = qlist.size();
	    for (qvind = 0; qvind < qsize; qvind++) {
		if (qlist[qvind].qind == indquery) {
		    numevents++;
		}
	    }
	}
	if (logerr) {
	    fprintf(logerr, "Query '%s' will use %u events in analysis.\n",
		    opt_querylabels[indquery], (unsigned int)numevents);
	}
	if (logfp) {
	    fprintf(logfp,  "Query '%s' will use %u events in analysis.\n",
		    opt_querylabels[indquery], (unsigned int)numevents);
	}
	numepochs[indquery] += numevents;
    }

    if (logerr) fflush(logerr);
    if (logfp)  fflush(logfp);
}

#undef FUNC
#define FUNC "get_timepoints"
static
double *
get_timepoints(bxhdimension * dimt, double opt_forcetr, FILE * logfp)
{
    double * timepoints = NULL;
    bxhdatapoints * dpstructs = dimt->dpstructs;
    FILE * logerr = NULL;

    if (logfp != NULL && logfp != stderr)
	logerr = stderr;

    if (dpstructs) {
	if (logerr) fprintf(logerr, "Warning: \"datapoints\" element is not currently supported and will be ignored.\n");
	if (logfp)  fprintf(logfp,   "Warning: \"datapoints\" element is not currently supported and will be ignored.\n");
	dpstructs = NULL;
    }

    size_t ind;
    double spacing = dimt->spacing;
    if (opt_forcetr != 0) {
	spacing = opt_forcetr * 1000.0;
    }
    if (spacing == 0) {
	fprintf(stderr, "Error: spacing between timepoints is missing or zero.\n");
	goto FAIL;
    }
    timepoints = (double *)malloc(sizeof(double)*dimt->size);
    if (timepoints == NULL) {
	fprintf(stderr, FUNC ": error allocating memory for timepoints\n");
	goto FAIL;
    }
    for (ind = 0; ind < dimt->size; ind++) {
	timepoints[ind] = (ind * spacing);
	timepoints[ind] /= 1000.0; /* convert to seconds */
    }

    if (logfp) {
	size_t ind;
	fprintf(logfp, "timepoints:");
	for (ind = 0; ind < dimt->size; ind++) {
	    fprintf(logfp, " %g", timepoints[ind]);
	}
	fprintf(logfp, "\n");
    }

    goto EXIT;

  FAIL:
    free(timepoints); timepoints = NULL;
    
  EXIT:
    return timepoints;
}

#undef FUNC
#define FUNC "get_mirrored"
static double *
get_mirrored(const double * data, size_t tsize, double * mirroreddata=NULL) {
    double pivot1 = data[0];
    double pivot2 = data[tsize - 1];
    size_t indt;
    if (mirroreddata == NULL) {
	mirroreddata = (double *)malloc(sizeof(double) * (tsize + (2 * MININTERP_PTS)));
    }
    if (mirroreddata + MININTERP_PTS != data) {
	memmove(&mirroreddata[MININTERP_PTS], &data[0], sizeof(double) * tsize);
    }
    for (indt = 1; indt < MININTERP_PTS + 1; indt++) {
	mirroreddata[MININTERP_PTS - indt] = pivot1 + (pivot1 - data[indt]);
	mirroreddata[MININTERP_PTS + (tsize - 1) + indt] = pivot2 + (pivot2 - data[(tsize - 1) - indt]);
    }
    return mirroreddata;
}

#undef FUNC
#define FUNC "write_out_niigz_bxh"
static int
write_out_niigz_bxh(const char * outputbase, void * buf, size_t bufsize, struct bxhdataread * writebdrp)
{
    return writeBXHAndNIIGZ(outputbase, writebdrp, buf, 0);
}

#undef FUNC
#define FUNC "main"
int
main(int argc, char *argv[])
{
    struct stat statbuf;
    int retval = 0;
    int argind;
    int indquery;
    int msbfirst = 1;
    FILE * fp = NULL;
    int numqueries = 0;
    int numtrialsummaries = 0;
    const char * outputprefix = NULL;
    FILE * logfp = NULL;
    
    int oldargc = argc;
    char ** oldargv = argv;

#define MAXQUERIES 128
    char * opt_queries[MAXQUERIES + 1];
    char * opt_queryfilters[MAXQUERIES + 1];
    char * opt_queryepochexcludes[MAXQUERIES + 1];
    char * opt_querylabels[MAXQUERIES + 1];
    int opt_ptsbefore = -1;
    int opt_ptsafter = -1;
    double opt_secsbefore = -1;
    double opt_secsafter = -1;
    unsigned int opt_startpt = 0;
    unsigned int opt_endpt = (unsigned int)-1;
    int opt_basestart = 0;
    int opt_baseend = 0;
    char * opt_maskfile = NULL;
    char * opt_optsfromfile = NULL;
    int opt_overwrite = 0;
    double opt_forcetr = 0;
    int opt_nointerp = 0;
    int opt_mirrorflip = 0;
    char * opt_querylang = NULL;
    int opt_version = 0;
    int opt_scalebl = 0;
    int opt_extracttrials = 0;
    char * opt_trialsummary[MAXQUERIES + 1];
    int opt_tm = 0;
    char * opt_tmroifile = NULL;
    char * opt_tmseed = NULL;
    int opt_extracttimingonly = 0;
    double opt_memorylimit = 0;

    char * xpqueries[MAXQUERIES + 1];
    char * xpqueryfilters[MAXQUERIES + 1];
    char * xpqueryepochexcludes[MAXQUERIES + 1];

    char * outputfilename = NULL;
    char * outputbxhname = NULL;
    char * outputimgname = NULL;
    char * outputbase = NULL;

    struct trialsummarygroup {
	size_t numpoints;
	size_t * points;
    };
    struct trialsummary {
	char * indicesstr;
	size_t numgroups;
	struct trialsummarygroup * groups;
    };
    struct trialsummarylist {
	size_t numsummaries;
	struct trialsummary * summaries;
    };
    struct trialsummarylist trialsummarylists[MAXQUERIES];

    /* these accumulate results of queries */
    float * qres_avg[MAXQUERIES]; /* mean */
    float * qres_std[MAXQUERIES]; /* standard deviation */
    size_t * qres_n[MAXQUERIES]; /* no. of events so far in the query result */
    float * qres_blavg = NULL; /* mean of baseline epochs */
    size_t * qres_nbl = NULL; /* number of points per baseline mean */
    float * qres_temp = NULL; /* staging area for each event */
    float * meanimage = NULL; /* mean image */
    float * qres_tm[MAXQUERIES]; /* for --trialmax */
    float ** qres_summary_avg[MAXQUERIES]; /* for --trialsummary */
    float ** qres_summary_std[MAXQUERIES]; /* for --trialsummary */
    size_t ** qres_summary_n[MAXQUERIES]; /* for --trialsummary */

    /* some random state */
    double TR = -1;
    size_t dimsizexyz[3] = {0, 0, 0};
    size_t volsize = 0;
    
    const char * ordereddimnames[] = { "t", "x", "y", "z" }; /* NOTE: t first, because we traverse time before space */
    const char * maskdimnames[] = { "x", "y", "z", "t" };
    struct bxhdataread bdrs[MAXQUERIES];
    struct bxhdataread writebdr;
    struct bxhdataread maskbdr;
    char * maskdataptr = NULL;
    struct bxhdataread tmroibdr;
    char * tmroidataptr = NULL;
    size_t maxlabellen = 0;
    size_t numptsinresult = 0;
    size_t numptsinbaseline = 0;
    size_t numptsinwork = 0;
    size_t workevoffset = 0;

    /* if memorylimit is specified, we may read subsets of voxels at a time */
    size_t batchdimnum = 0;
    size_t batchdimfullsize = 0;
    size_t batchdimbatchsize = 0;
    size_t batchdimoff = 0;
    const char * selectors[4] = { NULL, NULL, NULL, NULL };
    char selectbuf[64];

    /* for each query, stores number of epochs */
    size_t * numepochs = NULL;

    /* for elapsed time */
    time_t time_start;
    time_t time_end;

    const int numopts = 30;
    opt_data opts[30] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  bxh_epochavg [opts] outputprefix imgfile1 eventfile1[,eventfile1b,...] [imgfile2 eventfile2[,eventfile2b,...] ...]\n\n"
	  "This program \"queries\" a 4-D data set (with corresponding "
	  "event lists) and produces averages of all time courses "
	  "surrounding each event that match the query.  "
	  "Multiple independent queries may be specified, and the width "
	  "and position and duration of each time course relative to the "
	  "event is also user-specified.  "
	  "Multiple event files corresponding to the same image data can be "
	  "specified separated by commas (the filenames/paths themselves are "
	  "therefore prohibited from containing commas)." },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_optsfromfile, 1, "optsfromfile",
	  "Program options (i.e. those starting with '--') will come from "
	  "this file.  "
	  "If this option is specified, then the options in the file "
	  "will be applied after all command-line options.  "
	  "The options (and their arguments) should be specified "
	  "one per line, with the leading '--' omitted." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_overwrite, 1, "overwrite",
	  "Overwrite existing output files (otherwise error and exit). " },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_maskfile, 1, "maskfile",
	  "Use this 3-D mask (should be an XML file) before doing "
	  "calculations." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_querylang, 1, "querylanguage",
	  "The language used for all queries.  Valid values are 'XPath' and "
	  "'event'.  Case is irrelevant.  Default is "
	  "'XPath'." },
	{ OPT_FLAGS_FULL|OPT_FLAGS_NO_OVERFLOW, OPT_VAL_STR, &opt_queries[0], MAXQUERIES, "query",
	  "A query string as an XPath boolean expression.  Will be applied "
	  "as a predicate filter to each event.  Each event node "
	  "may or may not have onset, duration, type, and value elements "
	  "(as well as others).\n"
	  "Examples:\n"
	  "  --query \"value[@name='color']='red'\"\n"
	  "  --query \"value[@name='color']='red' or value[@name='color']='blue'\"\n"
	  "  --query \"(value[@name='color']='red' or value[@name='color']='blue') and not value[@name='field']='upper' and onset>12000\"\n"
	  "Note that some characters in queries may need to be protected "
	  "from the shell with quotes (as in the above examples).  "
	  "Separate instances of the --query option will result in "
	  "independent queries, with separate outputs.  "
	  "Empty queries match all events.  "
	  "NOTE: At least one query must be specified!" },
	{ OPT_FLAGS_FULL|OPT_FLAGS_NO_OVERFLOW, OPT_VAL_STR, &opt_queryfilters[0], MAXQUERIES, "queryfilter",
	  "If this option is specified, it is an XPath query (like --query) "
	  "that is applied to a list of pseudo-events, each pseudo-event "
	  "corresponding to an event matching the original query.  "
	  "Each pseudo-event is a merging of all events that are "
	  "simultaneously in effect at the time of the onset of the real "
	  "event.  If this query matches the pseudo-event, the real event "
	  "passes through the filter.  "
	  "The n-th instance of this option corresponds to the n-th "
	  "specified query.  "
	  "If any --queryfilter options are specified, "
	  "there should be exactly one --queryfilter per --query."
	  "Empty or missing filter queries match everything." },
	{ OPT_FLAGS_FULL|OPT_FLAGS_NO_OVERFLOW, OPT_VAL_STR, &opt_queryepochexcludes[0], MAXQUERIES, "queryepochexclude",
	  "Like --query, --queryepochexclude specifies an XPath-based event "
	  "query.  However, any epoch that includes an event that matches "
	  "this query will be excluded from the analysis.  "
	  "The epoch surrounding an event is specified using --ptsbefore "
	  "and --ptsafter (or --secsbefore and --secsafter).  "
	  "The n-th instance of this option corresponds to the n-th "
	  "specified query.  "
	  "If any --queryepochexclude options are specified, "
	  "there should be exactly one --queryepochexclude per --query."
	  "Empty or missing epoch exclusion queries exclude nothing." },
	{ OPT_FLAGS_FULL|OPT_FLAGS_NO_OVERFLOW, OPT_VAL_STR, &opt_querylabels[0], MAXQUERIES, "querylabel",
	  "A textual label for the corresponding query.  "
	  "The first instance of this option corresponds to the first "
	  "specified query.  "
	  "There should be at most one --querylabel per --query.  "
	  "Default label is the query number." },
	{ OPT_FLAGS_FULL, OPT_VAL_INT, &opt_ptsbefore, 1, "ptsbefore",
	  "How many time points before the event to include in analysis.  "
	  "This option (or --secsbefore) is required." },
	{ OPT_FLAGS_FULL, OPT_VAL_INT, &opt_ptsafter, 1, "ptsafter",
	  "How many time points after the event to include in analysis.  "
	  "This option (or --secsafter) is required." },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_secsbefore, 1, "secsbefore",
	  "How many seconds before the event to include in analysis.  "
	  "This option (or --ptsbefore) is required." },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_secsafter, 1, "secsafter",
	  "How many seconds after the event to include in analysis.  "
	  "This option (or --ptsafter) is required." },
	{ OPT_FLAGS_FULL, OPT_VAL_INT, &opt_basestart, 1, "basestartoffset",
	  "Where to start calculating mean baseline, in number of "
	  "timepoints (TRs) relative to event time.  A negative number "
	  "refers to a timepoint before the event, 0 is at the time of "
	  "the event, and a positive number is after the event.  "
	  "Default is 0." },
	{ OPT_FLAGS_FULL, OPT_VAL_INT, &opt_baseend, 1, "baseendoffset",
	  "Where to end calculating mean baseline, in number of "
	  "timepoints (TRs) relative to event time.  A negative number "
	  "refers to a timepoint before the event, 0 is at the time of "
	  "the event, and a positive number is after the event.  "
	  "Default is 0." },
	{ OPT_FLAGS_FULL, OPT_VAL_UINT, &opt_startpt, 1, "startpt",
	  "This number of time points at the start of the data will "
	  "be ignored.  "
	  "Default is 0." },
	{ OPT_FLAGS_FULL, OPT_VAL_UINT, &opt_endpt, 1, "endpt",
	  "Time points after this point will be ignored.  "
	  "Default is last timepoint." },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_forcetr, 1, "forcetr",
	  "If specified, this value (in seconds) will replace the TR "
	  "specified in the input image file, if any." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_nointerp, 1, "nointerp",
	  "If specified, no interpolation will be done -- events will be "
	  "assumed to occur at the closest TR/image acquisition time." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_mirrorflip, 1, "mirrorflip",
	  "If specified (and if necessary) the input signal will be padded "
	  "on either side with the mirrored/flipped signal to enable "
	  "enough timepoints for interpolation at the boundaries.  If not "
	  "specified, any epochs with an interpolation range outside the "
	  "range of the data will be excluded." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_scalebl, 1, "scalebl",
	  "If specified, values in each epoch are additionally scaled by "
	  "dividing by (after subtracting) the baseline.  This affects the "
	  "'avg' and 'std' output images.  Percent signal-change images are "
	  "not written.  "
	  "WARNING: Know what you are doing before using this option."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_extracttrials, 1, "extracttrials",
	  "If this option is specified, the program will write out epochs for "
	  "*all* extracted trials to a file PREFIX_QUERY_trials.bxh.  "
	  "This file will be a 5-D image file where the 4th dimension "
	  "goes across time points within an epoch, and the 5th dimension "
	  "represents the global trial number."
	},
	{ OPT_FLAGS_FULL|OPT_FLAGS_NO_OVERFLOW, OPT_VAL_STR, &opt_trialsummary[0], MAXQUERIES, "trialsummary",
	  "This option enables the creation of \"summaries\" of trials before "
	  "averaging, where summaries are new trials where each point is "
	  "an average of some number of timepoints in the original trial.  "
	  "The string argument is of the form \"QUERY-PTS\", "
	  "where QUERY is a query label, and PTS is a plus('+')-separated "
	  "list of \"index groups\", an \"index group\" is a "
	  "comma-separated list of indices or ranges (which are two indices "
	  "separated by a colon).  For example \"red-0\" will create a "
	  "summary trials containing only the first point in each trial that "
	  "matches the \"red\" query, and \"red-0:3+4:7+8:11\" or "
	  "\"red-0,1,2,3+4,5,6,7+8,9,10,11\" (both are equivalent) will "
	  "average together the 12 timepoints of each \"red\" trial in groups "
	  "of 4.  The outputs will be similar to other outputs but will look "
	  "like PREFIX_QUERY_summary_PTS_avg.bxh etc., but with the colons "
	  "(':') replaced with dashes ('-') due to problems some file systems "
	  "have with colons.  "
	  "Note that timepoints are indexed from 0.  "
	  "This option may be specified more than once." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_tm, 1, "trialmax",
	  "This is an EXPERIMENTAL option.  If specified, a 'seed' timepoint "
	  "and voxel is found within the optional ROI specified by "
	  "--trialmaxroi.  "
	  "The seed timepoint is defined as the timepoint within the epoch "
	  "average that has the highest mean intensity.  The seed voxel is "
	  "then defined as the voxel with the highest value within the "
	  "seed timepoint.  "
	  "Then, for each voxel, a 'trial sequence' is constructed "
	  "containing the value of that voxel at the seed timepoint within "
	  "each individual epoch (before averaging).  "
	  "The output is a 4-D series of volumes (one for each trial) "
	  "named PREFIX_QUERY_trialmax.bxh that contains the volumes at "
	  "the seed timepoint in each trial.  The seed voxel coordinates "
	  "are written to PREFIX_QUERY_trialmaxseed.txt."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_tmroifile, 1, "trialmaxroi",
	  "The ROI used by --trialmax."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_tmseed, 1, "trialmaxseed",
	  "This specifies an explicit comma-separated coordinate X,Y,Z,T "
	  "for the seed for --trialmax, to be applied to ALL queries.  "
	  "The T coordinate must be in the range [0,s-1] where s is the "
	  "number of time points in the epoch.  "
	  "Note that timepoints are indexed from 0."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_extracttimingonly, 1, "extracttimingonly",
	  "If specified, only the PREFIX_LABEL_timing.txt files will be "
	  "written."
	},
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_memorylimit, 1, "memorylimit",
	  "This specifies the number of megabytes of the input data to read "
	  "at a time.  Default is to read the entire data at once.  If you "
	  "are running out of memory due to high-resolution data, or large "
	  "numbers of timepoints, this is one way to reduce memory usage.  "
	  "This is not an overall memory usage limit -- actual memory usage "
	  "will surely be much higher than this."
	}
    };

    memset(&bdrs, '\0', sizeof(bdrs[0]) * MAXQUERIES);
    memset(&writebdr, '\0', sizeof(writebdr));
    memset(&maskbdr, '\0', sizeof(maskbdr));
    memset(&tmroibdr, '\0', sizeof(tmroibdr));
    memset(&opt_queries[0], '\0', sizeof(char *) * (MAXQUERIES + 1));
    memset(&opt_queryfilters[0], '\0', sizeof(char *) * (MAXQUERIES + 1));
    memset(&opt_queryepochexcludes[0], '\0', sizeof(char *) * (MAXQUERIES + 1));
    memset(&opt_querylabels[0], '\0', sizeof(char *) * (MAXQUERIES + 1));
    memset(&qres_avg[0], '\0', sizeof(qres_avg[0]) * MAXQUERIES);
    memset(&qres_std[0], '\0', sizeof(qres_std[0]) * MAXQUERIES);
    memset(&qres_n[0], '\0', sizeof(qres_n[0]) * MAXQUERIES);
    memset(&qres_summary_avg[0], '\0', sizeof(qres_summary_avg[0]) * MAXQUERIES);
    memset(&qres_summary_std[0], '\0', sizeof(qres_summary_std[0]) * MAXQUERIES);
    memset(&qres_summary_n[0], '\0', sizeof(qres_summary_n[0]) * MAXQUERIES);
    if (opt_tm) {
	memset(&qres_tm[0], '\0', sizeof(qres_tm[0]) * MAXQUERIES);
    }
    memset(&opt_trialsummary[0], '\0', sizeof(char *) * (MAXQUERIES + 1));
    memset(&trialsummarylists[0], '\0', sizeof(trialsummarylists[0]) * MAXQUERIES);

    memset(&xpqueries[0], '\0', sizeof(char *) * (MAXQUERIES + 1));
    memset(&xpqueryfilters[0], '\0', sizeof(char *) * (MAXQUERIES + 1));
    memset(&xpqueryepochexcludes[0], '\0', sizeof(char *) * (MAXQUERIES + 1));

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);
    if (opt_optsfromfile) {
	opt_parsefile(opt_optsfromfile, numopts, &opts[0], 0);
    }
    if (opt_querylang == NULL) {
	opt_querylang = strdup("XPath");
    }

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc < 3 || (argc % 2) != 0) {
	fprintf(stderr, "ERROR: Wrong number of arguments.  Use the --help option for more help.\n");
	fprintf(stderr, "Usage: %s [opts] outputprefix imgfile1 eventfile1 [imgfile2 eventfile2 ...]\n", argv[0]);
	goto FAIL;
    }
    if (opt_ptsbefore != -1 && opt_secsbefore != -1 ||
	opt_ptsbefore == -1 && opt_secsbefore == -1) {
	fprintf(stderr, "ERROR: One (and only one) of --ptsbefore or --secsbefore must be specified.\nUse --help for more help.\n");
	goto FAIL;
    }
    if (opt_ptsafter != -1 && opt_secsafter != -1 ||
	opt_ptsafter == -1 && opt_secsafter == -1) {
	fprintf(stderr, "ERROR: One (and only one) of --ptsafter or --secsafter must be specified.\nUse --help for more help.\n");
	goto FAIL;
    }
    if (opt_queries[0] == NULL) {
	fprintf(stderr, "ERROR: No queries are specified.  Use the --help option for more help.\n");
	goto FAIL;
    }

    if (strcasecmp(opt_querylang, "XPath") != 0 &&
	strcasecmp(opt_querylang, "event") != 0) {
	fprintf(stderr, "ERROR: Query language %s not recognized!\n", opt_querylang);
	goto FAIL;
    }

    outputprefix = argv[1];

    numptsinresult = (opt_ptsbefore + 1 + opt_ptsafter);
    numptsinbaseline = opt_baseend - opt_basestart + 1;
    numptsinwork = 0;
    if (opt_basestart * (-1) > opt_ptsbefore) {
	numptsinwork += opt_basestart * (-1);
	workevoffset = opt_basestart * (-1);
    } else {
	numptsinwork += opt_ptsbefore;
	workevoffset = opt_ptsbefore;
    }
    numptsinwork++;
    if (opt_baseend > opt_ptsafter) {
	numptsinwork += opt_baseend;
    } else {
	numptsinwork += opt_ptsafter;
    }

    msbfirst = (((char *)&msbfirst)[0] == 0);

    /* count queries */
    while (opt_queries[numqueries] != NULL) {
	size_t labellen = 0;
	if (!opt_querylabels[numqueries]) {
	    opt_querylabels[numqueries] = strdup("0000000000");
	    sprintf(opt_querylabels[numqueries], "%03d", (int)numqueries);
	}
	labellen = strlen(opt_querylabels[numqueries]);
	if (labellen > maxlabellen)
	    maxlabellen = labellen;
	numqueries++;
    }

    if (opt_queryepochexcludes[0] != NULL) {
	for (indquery = 1; indquery < numqueries; indquery++) {
	    if (opt_queryepochexcludes[indquery] == NULL) {
		fprintf(stderr, "Number of --queryepochexclude options (%d), if any, must match number of --query options (%d)!\n", indquery, numqueries);
		goto FAIL;
	    }
	}
	if (numqueries + 1 < MAXQUERIES &&
	    opt_queryepochexcludes[numqueries] != NULL) {
	    fprintf(stderr, "Number of --queryepochexclude options (%d), if any, must match number of --query options (%d)!\n", indquery, numqueries);
	    goto FAIL;
	}
    }

    if (opt_queryfilters[0] != NULL) {
	for (indquery = 1; indquery < numqueries; indquery++) {
	    if (opt_queryfilters[indquery] == NULL) {
		fprintf(stderr, "Number of --queryfilter options (%d), if any, must match number of --query options (%d)!\n", indquery, numqueries);
		goto FAIL;
	    }
	}
	if (numqueries + 1 < MAXQUERIES &&
	    opt_queryfilters[numqueries] != NULL) {
	    fprintf(stderr, "Number of --queryfilter options (%d), if any, must match number of --query options (%d)!\n", indquery, numqueries);
	    goto FAIL;
	}
    }

    for (indquery = 0; indquery < numqueries; indquery++) {
	char * newquery = NULL;
	if (opt_queries[indquery]) {
	    if (opt_queries[indquery][0] == '\0') {
		/* empty queries match everything */
		newquery = strdup("true()");
	    } else if (strcasecmp(opt_querylang, "event") == 0 &&
		       (newquery = query2xpath(opt_queries[indquery])) == NULL) {
		fprintf(stderr, "Bad query:\n%s\n", opt_queries[indquery]);
		goto FAIL;
	    } else if (strcasecmp(opt_querylang, "xpathevent") == 0 &&
		       (newquery = expand_xpath_event(opt_queries[indquery])) == NULL) {
		fprintf(stderr, "Bad query:\n%s\n", opt_queries[indquery]);
		goto FAIL;
	    }
	    if (newquery == NULL)
		newquery = strdup(opt_queries[indquery]);
	    xpqueries[indquery] = newquery;
	}

	newquery = NULL;
	if (opt_queryfilters[indquery] == NULL ||
	    opt_queryfilters[indquery][0] == '\0') {
	    /* empty queryfilters match everything */
	    if (opt_queryfilters[indquery])
		free(opt_queryfilters[indquery]);
	    opt_queryfilters[indquery] = strdup("true()");
	} else if (strcasecmp(opt_querylang, "event") == 0 &&
		   (newquery = query2xpath(opt_queryfilters[indquery])) == NULL) {
	    fprintf(stderr, "Bad query '%s'!\n", opt_queryfilters[indquery]);
	    goto FAIL;
	} else if (strcasecmp(opt_querylang, "xpathevent") == 0 &&
		   (newquery = expand_xpath_event(opt_queryfilters[indquery])) == NULL) {
	    fprintf(stderr, "Bad query '%s'!\n", opt_queryfilters[indquery]);
	    goto FAIL;
	}
	if (newquery == NULL && opt_queryfilters[indquery] != NULL)
	    newquery = strdup(opt_queryfilters[indquery]);
	xpqueryfilters[indquery] = newquery;

	newquery = NULL;
	if (opt_queryepochexcludes[indquery] == NULL ||
	    opt_queryepochexcludes[indquery][0] == '\0') {
	    /* empty queryepochexcludes exclude nothing */
	    if (opt_queryepochexcludes[indquery])
		free(opt_queryepochexcludes[indquery]);
	    opt_queryepochexcludes[indquery] = strdup("false()");
	} else if (strcasecmp(opt_querylang, "event") == 0 &&
		   (newquery = query2xpath(opt_queryepochexcludes[indquery])) == NULL) {
	    fprintf(stderr, "Bad query '%s'!\n", opt_queryepochexcludes[indquery]);
	    goto FAIL;
	} else if (strcasecmp(opt_querylang, "xpathevent") == 0 &&
		   (newquery = expand_xpath_event(opt_queryepochexcludes[indquery])) == NULL) {
	    fprintf(stderr, "Bad query '%s'!\n", opt_queryepochexcludes[indquery]);
	    goto FAIL;
	}
	if (newquery == NULL && opt_queryepochexcludes[indquery] != NULL)
	    newquery = strdup(opt_queryepochexcludes[indquery]);
	xpqueryepochexcludes[indquery] = newquery;
    }

    /* count and parse trial summary options, if specified */
    while (opt_trialsummary[numtrialsummaries] != NULL) {
	char * querylabel = NULL;
	char * curpos = opt_trialsummary[numtrialsummaries];
	int queryind;
	struct trialsummarylist * tslistp = NULL;
	struct trialsummary * tsp;
	struct trialsummarygroup * tsg;
	querylabel = curpos;
	if ((curpos = strchr(curpos, '-')) == NULL || *(curpos+1) == '\0') {
	    fprintf(stderr, "--trialsummary argument '%s' must have the form \"QUERY-PTS\"!\n", opt_trialsummary[numtrialsummaries]);
	    goto FAIL;
	}
	*curpos = '\0';
	curpos++;
	for (queryind = 0; queryind < numqueries; queryind++) {
	    if (strcmp(querylabel, opt_querylabels[queryind]) == 0) {
		break;
	    }
	}
	if (queryind == numqueries) {
	    fprintf(stderr, "Error parsing --trialsummary query label '%s': not a valid query label!\n", querylabel);
	    goto FAIL;
	}
	tslistp = &trialsummarylists[queryind];
	tslistp->summaries = (struct trialsummary *)realloc(tslistp->summaries, sizeof(struct trialsummary)*(tslistp->numsummaries+1));
	memset(&tslistp->summaries[tslistp->numsummaries], '\0', sizeof(tslistp->summaries[tslistp->numsummaries]));
	tsp = &tslistp->summaries[tslistp->numsummaries];
	tslistp->numsummaries++;
	tsp->groups = (struct trialsummarygroup *)realloc(tsp->groups, sizeof(struct trialsummarygroup)*(tsp->numgroups+1));
	tsg = &tsp->groups[tsp->numgroups];
	memset(tsg, '\0', sizeof(*tsg));
	tsp->numgroups++;
	tsp->indicesstr = strdup(curpos);
	while (*curpos != '\0') {
	    char * endptr = NULL;
	    long index;
	    index = strtol(curpos, &endptr, 10);
	    if (index == 0 && endptr == curpos) {
		fprintf(stderr, "Can't parse \"trialsummary\" index at '%s'!\n", curpos);
		goto FAIL;
	    }
	    if ((index == LONG_MAX && errno == ERANGE) ||
		(index < 0 || index >= (long)numptsinresult)) {
		fprintf(stderr, "--trialsummary index '%*s' is out of range (must be between 0 and %u)!\n", (int)(endptr-curpos), curpos, (unsigned int)numptsinresult-1);
		goto FAIL;
	    }
	    if (*endptr != '\0' && *endptr != ',' && *endptr != ':' && *endptr != '+') {
		fprintf(stderr, "--trialsummary indices '%s' are not in the correct form!\n(parse error at '%s')\n", tsp->indicesstr, curpos);
		goto FAIL;
	    }
	    if (*endptr == ':') {
		long index2;
		long curindex;
		curpos = endptr;
		if (*endptr != '\0')
		    curpos = endptr + 1;
		index2 = strtol(curpos, &endptr, 10);
		if (index2 == 0 && endptr == curpos) {
		    fprintf(stderr, "Can't parse \"trialsummary\" indexat '%s'!\n", curpos);
		    goto FAIL;
		}
		if ((index2 == LONG_MAX && errno == ERANGE) ||
		    (index2 < 0 || index2 >= (long)numptsinresult)) {
		    fprintf(stderr, "--trialsummary index '%*s' is out of range (must be between 0 and %u)!\n", (int)(endptr-curpos), curpos, (unsigned int)numptsinresult-1);
		    goto FAIL;
		}
		if (index2 < index) {
		    fprintf(stderr, "--trialsummary index '%s' contains a backwards range (%u:%u)!\n", tsp->indicesstr, (unsigned int)index, (unsigned int)index2);
		    goto FAIL;
		}
		if (*endptr != '\0' && *endptr != ',' && *endptr != '+') {
		    fprintf(stderr, "--trialsummary indices '%s' are not in the correct form!\n(parse error at '%s')\n", tsp->indicesstr, curpos);
		    goto FAIL;
		}
		tsg->points = (size_t *)realloc(tsg->points, sizeof(size_t)*(tsg->numpoints+(index2+1-index)));
		for (curindex = index; curindex <= index2; curindex++) {
		    tsg->points[tsg->numpoints] = curindex;
		    tsg->numpoints++;
		}
	    } else {
		tsg->points = (size_t *)realloc(tsg->points, sizeof(size_t)*(tsg->numpoints+1));
		tsg->points[tsg->numpoints] = index;
		tsg->numpoints++;
	    }
	    if (*endptr == '+') {
		tsp->groups = (struct trialsummarygroup *)realloc(tsp->groups, sizeof(struct trialsummarygroup)*(tsp->numgroups+1));
		tsg = &tsp->groups[tsp->numgroups];
		memset(tsg, '\0', sizeof(*tsg));
		tsp->numgroups++;
	    }
	    curpos = endptr;
	    if (*curpos != '\0') {
		curpos++;
	    }
	}
	numtrialsummaries++;
    }

    for (indquery = 0; indquery < numqueries; indquery++) {
	size_t indsummary;
	struct trialsummarylist * tslistp = &trialsummarylists[indquery];
	for (indsummary = 0; indsummary < tslistp->numsummaries; indsummary++) {
	    struct trialsummary * tsp = &tslistp->summaries[indsummary];
	    char * colon = tsp->indicesstr;
	    while ((colon = strchr(colon, ':'))) {
		*colon = '-';
		colon++;
	    }
	}
    }

    outputfilename = (char *)malloc(sizeof(char)*(strlen(outputprefix)+maxlabellen+100));
    if (outputfilename == NULL) {
	fprintf(stderr, FUNC ": error allocating memory for outputfilename\n");
	goto FAIL;
    }
    outputbxhname = (char *)malloc(sizeof(char)*(strlen(outputprefix)+maxlabellen+100));
    if (outputbxhname == NULL) {
	fprintf(stderr, FUNC ": error allocating memory for outputbxhname\n");
	goto FAIL;
    }
    outputimgname = (char *)malloc(sizeof(char)*(strlen(outputprefix)+maxlabellen+100));
    if (outputimgname == NULL) {
	fprintf(stderr, FUNC ": error allocating memory for outputimgname\n");
	goto FAIL;
    }
    outputbase = (char *)malloc(sizeof(char)*(strlen(outputprefix)+maxlabellen+100));
    if (outputbase == NULL) {
	fprintf(stderr, FUNC ": error allocating memory for outputbase\n");
	goto FAIL;
    }

    if (!opt_overwrite) {
	sprintf(outputfilename, "%s_QUERIES.txt", outputprefix);
	if (stat(outputfilename, &statbuf) == 0) {
	    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
	    return -1;
	}
	sprintf(outputfilename, "%s_LOG_EPOCHAVG.txt", outputprefix);
	if (stat(outputfilename, &statbuf) == 0) {
	    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
	    return -1;
	}
	sprintf(outputfilename, "%s_eventtiming.txt", outputprefix);
	if (stat(outputfilename, &statbuf) == 0) {
	    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
	    return -1;
	}
    }
    if (!opt_overwrite && !opt_extracttimingonly) {
	sprintf(outputfilename, "%s_baselineAvg.bxh", outputprefix);
	if (stat(outputfilename, &statbuf) == 0) {
	    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
	    return -1;
	}
	sprintf(outputfilename, "%s_baselineAvg.nii.gz", outputprefix);
	if (stat(outputfilename, &statbuf) == 0) {
	    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
	    return -1;
	}
	for (indquery = 0; indquery < numqueries; indquery++) {
	    sprintf(outputfilename, "%s_%s_avg.bxh", outputprefix, opt_querylabels[indquery]);
	    if (stat(outputfilename, &statbuf) == 0) {
		fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		return -1;
	    }
	    sprintf(outputfilename, "%s_%s_avg.nii.gz", outputprefix, opt_querylabels[indquery]);
	    if (stat(outputfilename, &statbuf) == 0) {
		fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		return -1;
	    }
	    if (!opt_scalebl) {
		sprintf(outputfilename, "%s_%s_avg_percent.bxh", outputprefix, opt_querylabels[indquery]);
		if (stat(outputfilename, &statbuf) == 0) {
		    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		    return -1;
		}
		sprintf(outputfilename, "%s_%s_avg_percent.nii.gz", outputprefix, opt_querylabels[indquery]);
		if (stat(outputfilename, &statbuf) == 0) {
		    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		    return -1;
		}
	    }
	    sprintf(outputfilename, "%s_%s_std.bxh", outputprefix, opt_querylabels[indquery]);
	    if (stat(outputfilename, &statbuf) == 0) {
		fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		return -1;
	    }
	    sprintf(outputfilename, "%s_%s_std.nii.gz", outputprefix, opt_querylabels[indquery]);
	    if (stat(outputfilename, &statbuf) == 0) {
		fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		return -1;
	    }
	    if (!opt_scalebl) {
		sprintf(outputfilename, "%s_%s_std_percent.bxh", outputprefix, opt_querylabels[indquery]);
		if (stat(outputfilename, &statbuf) == 0) {
		    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		    return -1;
		}
		sprintf(outputfilename, "%s_%s_std_percent.nii.gz", outputprefix, opt_querylabels[indquery]);
		if (stat(outputfilename, &statbuf) == 0) {
		    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		    return -1;
		}
	    }
	    sprintf(outputfilename, "%s_%s_n.bxh", outputprefix, opt_querylabels[indquery]);
	    if (stat(outputfilename, &statbuf) == 0) {
		fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		return -1;
	    }
	    sprintf(outputfilename, "%s_%s_n.nii.gz", outputprefix, opt_querylabels[indquery]);
	    if (stat(outputfilename, &statbuf) == 0) {
		fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		return -1;
	    }
	    if (opt_extracttrials) {
		sprintf(outputfilename, "%s_%s_trials.bxh", outputprefix, opt_querylabels[indquery]);
		if (stat(outputfilename, &statbuf) == 0) {
		    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		    return -1;
		}
		sprintf(outputfilename, "%s_%s_trials.img", outputprefix, opt_querylabels[indquery]);
		if (stat(outputfilename, &statbuf) == 0) {
		    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		    return -1;
		}
	    }
	    if (opt_tm) {
		sprintf(outputfilename, "%s_%s_trialmax.bxh", outputprefix, opt_querylabels[indquery]);
		if (stat(outputfilename, &statbuf) == 0) {
		    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		    return -1;
		}
		sprintf(outputfilename, "%s_%s_trialmax.nii.gz", outputprefix, opt_querylabels[indquery]);
		if (stat(outputfilename, &statbuf) == 0) {
		    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		    return -1;
		}
		sprintf(outputfilename, "%s_%s_trialmaxseed.txt", outputprefix, opt_querylabels[indquery]);
		if (stat(outputfilename, &statbuf) == 0) {
		    fprintf(stderr, "%s: output file '%s' exists.\nRemove file or use --overwrite.\n", argv[0], outputfilename);
		    return -1;
		}
	    }
	}
    }

    {
	char * logfile = NULL;
	logfile = (char *)malloc(sizeof(char)*(strlen(outputprefix)+20));
	if (logfile == NULL) {
	    fprintf(stderr, FUNC ": error allocating memory for logfile\n");
	    goto FAIL;
	}
	sprintf(logfile, "%s_LOG_EPOCHAVG.txt", outputprefix);
	if ((logfp = fopen(logfile, "w")) == NULL) {
	    fprintf(stderr, "Error opening %s for writing.\n", logfile);
	    perror("fopen");
	    goto FAIL;
	}
	free(logfile);
    }

    fprintf(logfp, "%s log file\n", oldargv[0]);
    fprintf(logfp, "rcsid: %s\n", rcsid);

    time_start = time(NULL);
    fprintf(stderr, "Start time: %s\n", ctime(&time_start));
    fprintf(logfp,  "Start time: %s\n", ctime(&time_start));

    {
	int argnum = 0;
	fprintf(logfp, "Command line:\n");
	for (argnum = 0; argnum < oldargc; argnum++) {
	    const char * oldarg = oldargv[argnum];
	    char quote = '\0';
	    char altquote = '\0';
	    if (strchr(oldarg, '"') != NULL) {
		quote = '\'';
		altquote = '"';
	    } else if (strchr(oldarg, '\'') != NULL) {
		quote = '"';
		altquote = '\'';
	    } else if (strcspn(oldarg, "|&;$()<> \t\n\v\\") != strlen(oldarg)) {
		quote = '\'';
		altquote = '"';
	    }
	    if (oldarg[0] == '\0') {
		fprintf(logfp, " ''");
	    } else if (quote == '\0') {
		fprintf(logfp, " %s", oldargv[argnum]);
	    } else {
		char * newarg = (char *)malloc(sizeof(char)*(5*strlen(oldargv[argnum]) + 3));
		if (newarg == NULL) {
		    fprintf(stderr, FUNC ": error allocating memory for newarg\n");
		    goto FAIL;
		}
		const char * oldpos = oldarg;
		char * newpos = newarg;
		*(newpos++) = quote;
		while (*oldpos != '\0') {
		    if (*oldpos == quote) {
			*(newpos++) = quote;
			*(newpos++) = altquote;
			*(newpos++) = *(oldpos++);
			*(newpos++) = altquote;
			*(newpos++) = quote;
		    } else {
			*(newpos++) = *(oldpos++);
		    }
		}
		*(newpos++) = quote;
		*(newpos++) = '\0';
		fprintf(logfp, " %s", newarg);
		free(newarg);
	    }
	}
	fprintf(logfp, "\n");
	if (opt_optsfromfile) {
	    char buf[8192];
	    size_t bytesread;
	    FILE * optfp = fopen(opt_optsfromfile, "r");
	    if (optfp == NULL) {
		fprintf(stderr, "Error opening opts file %s for reading:\n", opt_optsfromfile);
		perror("fopen");
		goto FAIL;
	    }
	    fprintf(logfp, "--- Begin contents of '%s' ---\n", opt_optsfromfile);
	    while (1) {
		bytesread = fread(&buf[0], 1, sizeof(buf), optfp);
		fwrite(&buf[0], bytesread, 1, logfp);
		if (bytesread != sizeof(buf)) {
		    break;
		}
	    }
	    fprintf(logfp, "\n--- End contents of '%s' ---\n", opt_optsfromfile);
	    fclose(optfp);
	}
	fprintf(logfp, "\n");
    }

    for (indquery = 0; indquery < numqueries; indquery++) {
	fprintf(stderr, "Query '%s':\n", opt_querylabels[indquery]);
	fprintf(logfp,  "Query '%s':\n", opt_querylabels[indquery]);
	fprintf(stderr, " primary:       %s\n", opt_queries[indquery]);
	fprintf(logfp,  " primary:       %s\n", opt_queries[indquery]);
	fprintf(stderr, "  [XPath]       %s\n", xpqueries[indquery]);
	fprintf(logfp,  "  [XPath]       %s\n", xpqueries[indquery]);
	if (opt_queryfilters[indquery]) {
	    fprintf(stderr, " filter:        %s\n", opt_queryfilters[indquery]);
	    fprintf(logfp,  " filter:        %s\n", opt_queryfilters[indquery]);
	    fprintf(stderr, "  [XPath]       %s\n", xpqueryfilters[indquery]);
	    fprintf(logfp,  "  [XPath]       %s\n", xpqueryfilters[indquery]);
	}
	if (opt_queryepochexcludes[indquery]) {
	    fprintf(stderr, " epoch exclude: %s\n", opt_queryepochexcludes[indquery]);
	    fprintf(logfp,  " epoch exclude: %s\n", opt_queryepochexcludes[indquery]);
	    fprintf(stderr, "  [XPath]       %s\n", xpqueryepochexcludes[indquery]);
	    fprintf(logfp,  "  [XPath]       %s\n", xpqueryepochexcludes[indquery]);
	}
    }
    
    if (opt_extracttimingonly) {
	opt_maskfile = NULL;
	opt_tmroifile = NULL;
    }

    /* read mask if specified */
    if (opt_maskfile) {
	if (bxh_dataReadFileStart(opt_maskfile, "image", NULL, 3, maskdimnames, NULL, &maskbdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
	if (maskbdr.datarec->numdims != 3) {
	    fprintf(stderr, "Mask must be 3-dimensional.\n");
	    goto FAIL;
	}
	if (bxh_dataReadFinish(&maskbdr, "char") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", opt_maskfile);
	    goto FAIL;
	}
	maskdataptr = (char *)maskbdr.dataptr;
    }

    /* read trialmaxroi if specified */
    if (opt_tmroifile) {
	if (bxh_dataReadFileStart(opt_tmroifile, "image", NULL, 3, maskdimnames, NULL, &tmroibdr) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", opt_tmroifile);
	    goto FAIL;
	}
	if (tmroibdr.datarec->numdims != 3) {
	    fprintf(stderr, "ROI '%s' must be 3-dimensional.\n", opt_tmroifile);
	    goto FAIL;
	}
	if (bxh_dataReadFinish(&tmroibdr, "char") != 0) {
	    fprintf(stderr, "Error finishing data read for '%s'.\n", opt_tmroifile);
	    goto FAIL;
	}
	tmroidataptr = (char *)tmroibdr.dataptr;
    }

    /* pre-read data headers to get dimensions, etc. and to verify that
     * all are valid */
    for (argind = 2; argind < argc; argind += 2) {
	int imnum = (argind - 2) / 2;
	bxhdimension * dimx = NULL;
	bxhdimension * dimy = NULL;
	bxhdimension * dimz = NULL;
	bxhdimension * dimt = NULL;

	BXHElementPtr trelemp = NULL;

	const char * imagefile = argv[argind];

	unsigned int endpt = opt_endpt;

	if (bxh_dataReadFileStart(imagefile, "image", NULL, 4, ordereddimnames, NULL, &bdrs[imnum]) != 0) {
	    fprintf(stderr, "Error preparing data read for '%s'.\n", imagefile);
	    goto FAIL;
	}
	if (bdrs[imnum].datarec->numdims != 4) {
	    fprintf(stderr, "Error: data must be 4-dimensional (%s).\n", imagefile);
	    goto FAIL;
	}
	if (bdrs[imnum].datarec->dimensions[3].size == 0) {
	    fprintf(stderr, "Number of time points must be greater than 0 in %s!\n", imagefile);
	    goto FAIL;
	}
	if (opt_forcetr != 0 && bdrs[imnum].acqdatap == NULL) {
	    fprintf(stderr, "Can't find acquisition data in %s!\n", imagefile);
	    goto FAIL;
	}
	dimt = &bdrs[imnum].datarec->dimensions[0];
	dimx = &bdrs[imnum].datarec->dimensions[1];
	dimy = &bdrs[imnum].datarec->dimensions[2];
	dimz = &bdrs[imnum].datarec->dimensions[3];
	if (imnum == 0) {
	    dimsizexyz[0] = dimx->size;
	    dimsizexyz[1] = dimy->size;
	    dimsizexyz[2] = dimz->size;
	    volsize = dimsizexyz[0] * dimsizexyz[1] * dimsizexyz[2];
	} else {
	    if (dimsizexyz[0] != dimx->size ||
		dimsizexyz[1] != dimy->size ||
		dimsizexyz[2] != dimz->size) {
		fprintf(stderr, "Dimensions of data in %s do not match previous data.\n", imagefile);
		goto FAIL;
	    }
	}

	if (opt_forcetr == 0) {
	    double readTR;
	    if ((trelemp = bxh_getChildElement(bdrs[0].acqdatap, "tr")) == NULL ||
		bxh_getElementDoubleValue(trelemp, &readTR) != 0) {
		readTR = dimt->spacing;
// 		fprintf(stderr, "Unable to read TR from image header!\n");
// 		goto FAIL;
	    }
	    if (trelemp) {
	      bxh_element_unref(trelemp); trelemp = NULL;
	    }
	    readTR /= 1000.0; /* convert to seconds */
	    if (imnum == 0)
		TR = readTR;
	    if (TR != readTR) {
		fprintf(stderr, "TR in %s (%.11g) does not match TR in previous data (%.11g).\n", imagefile, readTR, TR);
		goto FAIL;
	    }
	} else {
	    TR = opt_forcetr;
	}

	if (endpt == (unsigned int)-1) {
	    endpt = dimt->size - 1;
	}
	if (endpt >= dimt->size) {
	    fprintf(stderr, "Specified end point %u too high (>= %u).\n", endpt, (unsigned int)dimt->size);
	    goto FAIL;
	}
    }

    /* find out whether we need to read voxel subsets */
    opt_memorylimit *= 1024 * 1024; /* convert from MB to bytes */
    if (opt_memorylimit == 0) {
	batchdimnum = 3;
	batchdimfullsize = dimsizexyz[2];
	batchdimbatchsize = dimsizexyz[2];
    } else {
	int dimnum;
	size_t chunksize = bdrs[0].datarec->datasize;
	for (dimnum = 3; dimnum >= 1; dimnum--) {
	    size_t dimsize = dimsizexyz[dimnum-1];
	    batchdimnum = dimnum;
	    batchdimfullsize = dimsize;
	    batchdimbatchsize = dimsize;
	    if (chunksize < opt_memorylimit) {
		/* we're fine now */
		break;
	    }
	    if ((size_t)(chunksize / opt_memorylimit) <= dimsize) {
		/* use this dimension as the selector, as we can get under
		 * our memory limit by selecting portions of this dimension */
		batchdimnum = dimnum;
		batchdimbatchsize = (size_t)(dimsize * opt_memorylimit / chunksize);
		break;
	    }
	    /* go to next dimension */
	    chunksize /= dimsize;
	}
	if (dimnum < 1) {
	    fprintf(stderr, "ERROR: Specified memorylimit (%g MB) is too low for this data.\n", opt_memorylimit);
	    goto FAIL;
	}
    }

    /* extract timing to file */
    {
	sprintf(outputfilename, "%s_eventtiming.txt", outputprefix);
	if ((fp = fopen(outputfilename, "w")) == NULL) {
	    fprintf(stderr, "Error opening file %s\n", outputfilename);
	    fclose(fp);
	    goto FAIL;
	}

	for (argind = 2; argind < argc; argind += 2) {
	    double epochdurbefore = -1;
	    double epochdurafter = -1;

	    std::map<double, qvec > eventlist; /* maps event time to list of matching queries */
	    size_t eventnum;

	    const char * imagefile = argv[argind];
	    const char * eventfiles = argv[argind + 1];

	    fprintf(stderr, "event file(s) %s\n", eventfiles);

	    epochdurbefore = opt_secsbefore;
	    epochdurafter = opt_secsafter;
	    if (epochdurbefore == -1) {
		epochdurbefore = opt_ptsbefore * TR;
	    }
	    if (epochdurafter == -1) {
		epochdurafter = opt_ptsafter * TR;
	    }

	    if (get_event_list(eventfiles, numqueries, &xpqueries[0], &xpqueryfilters[0], &xpqueryepochexcludes[0], &opt_querylabels[0], epochdurbefore, epochdurafter, eventlist, NULL) != 0)
		goto FAIL;

	    for (indquery = 0; indquery < numqueries; indquery++) {
		std::map<double, qvec >::iterator eventiter;
		for (eventiter = eventlist.begin(), eventnum = 0;
		     eventiter != eventlist.end();
		     eventiter++, eventnum++) {
		    double etime = (*eventiter).first;
		    qvec & qlist = (*eventiter).second;
		    size_t qind;
		    for (qind = 0; qind < qlist.size(); qind++) {
			if (indquery != qlist[qind].qind)
			    continue;
			if (fprintf(fp, "%s,%s,%g\n", imagefile, opt_querylabels[indquery], etime) <= 0) {
			    fprintf(stderr, "Error writing to file %s\n", outputfilename);
			    fclose(fp);
			    goto FAIL;
			}
		    }
		}
	    }
	}

	fclose(fp);
	fp = NULL;
    }

    if (opt_extracttimingonly)
	goto EXIT;

    if (opt_maskfile &&
	dimsizexyz[0] != maskbdr.datarec->dimensions[0].size &&
	dimsizexyz[1] != maskbdr.datarec->dimensions[1].size &&
	dimsizexyz[2] != maskbdr.datarec->dimensions[2].size) {
	fprintf(stderr, "Mask (%ux%ux%u) and image (%ux%ux%u) data dimensions don't match!\n",
		(unsigned int)maskbdr.datarec->dimensions[0].size,
		(unsigned int)maskbdr.datarec->dimensions[1].size,
		(unsigned int)maskbdr.datarec->dimensions[2].size,
		(unsigned int)dimsizexyz[0],
		(unsigned int)dimsizexyz[1],
		(unsigned int)dimsizexyz[2]);
	goto FAIL;
    }

    if (opt_tmroifile &&
	dimsizexyz[0] != tmroibdr.datarec->dimensions[0].size &&
	dimsizexyz[1] != tmroibdr.datarec->dimensions[1].size &&
	dimsizexyz[2] != tmroibdr.datarec->dimensions[2].size) {
	fprintf(stderr, "ROI (%ux%ux%u) and image (%ux%ux%u) data dimensions don't match!\n",
		(unsigned int)tmroibdr.datarec->dimensions[0].size,
		(unsigned int)tmroibdr.datarec->dimensions[1].size,
		(unsigned int)tmroibdr.datarec->dimensions[2].size,
		(unsigned int)dimsizexyz[0],
		(unsigned int)dimsizexyz[1],
		(unsigned int)dimsizexyz[2]);
	goto FAIL;
    }

    /* allocate memory for results */
    for (indquery = 0; indquery < numqueries; indquery++) {
	qres_avg[indquery] = (float *)malloc(sizeof(float)*volsize*numptsinresult);
	qres_std[indquery] = (float *)malloc(sizeof(float)*volsize*numptsinresult);
	qres_n[indquery] = (size_t *)malloc(sizeof(size_t)*volsize*numptsinresult);
	if (qres_avg[indquery] == NULL ||
	    qres_std[indquery] == NULL ||
	    qres_n[indquery] == NULL) {
	    fprintf(stderr, FUNC ": error allocating memory for results for query %d\n", indquery);
	    goto FAIL;
	}
	memset(qres_avg[indquery], '\0', sizeof(float)*volsize*numptsinresult);
	memset(qres_std[indquery], '\0', sizeof(float)*volsize*numptsinresult);
	memset(qres_n[indquery], '\0', sizeof(size_t)*volsize*numptsinresult);
    }
    for (indquery = 0; indquery < numqueries; indquery++) {
	size_t indsummary;
	struct trialsummarylist * tslistp = &trialsummarylists[indquery];
	qres_summary_avg[indquery] = (float **)malloc(sizeof(float *)*tslistp->numsummaries);
	qres_summary_std[indquery] = (float **)malloc(sizeof(float *)*tslistp->numsummaries);
	qres_summary_n[indquery] = (size_t **)malloc(sizeof(size_t *)*tslistp->numsummaries);
	for (indsummary = 0; indsummary < tslistp->numsummaries; indsummary++) {
	    struct trialsummary * tsp = &tslistp->summaries[indsummary];
	    qres_summary_avg[indquery][indsummary] = (float *)malloc(sizeof(float)*volsize*tsp->numgroups);
	    qres_summary_std[indquery][indsummary] = (float *)malloc(sizeof(float)*volsize*tsp->numgroups);
	    qres_summary_n[indquery][indsummary] = (size_t *)malloc(sizeof(size_t)*volsize*tsp->numgroups);
	    memset(qres_summary_avg[indquery][indsummary], '\0', sizeof(float)*volsize*tsp->numgroups);
	    memset(qres_summary_std[indquery][indsummary], '\0', sizeof(float)*volsize*tsp->numgroups);
	    memset(qres_summary_n[indquery][indsummary], '\0', sizeof(size_t)*volsize*tsp->numgroups);
	}
    }
    qres_blavg = (float *)malloc(sizeof(float)*volsize);
    qres_nbl = (size_t *)malloc(sizeof(size_t)*volsize);
    if (qres_blavg == NULL || qres_nbl == NULL) {
	fprintf(stderr, FUNC ": error allocating memory for baseline\n");
	goto FAIL;
    }
    meanimage = (float *)malloc(sizeof(float)*volsize);
    if (meanimage == NULL) {
	fprintf(stderr, FUNC ": error allocating memory for mean image\n");
	goto FAIL;
    }
    memset(qres_blavg, '\0', sizeof(float)*volsize);
    memset(qres_nbl, '\0', sizeof(size_t)*volsize);

    numepochs = (size_t *)malloc(sizeof(numepochs[0])*numqueries);
    if (numepochs == NULL) {
	fprintf(stderr, FUNC ": error allocating memory for numepochs\n");
	goto FAIL;
    }
    memset(numepochs, '\0', sizeof(numepochs[0])*numqueries);

    /* do epoch averaging */
    for (argind = 2; argind < argc; argind += 2) {
	int imnum = (argind - 2) / 2;

	/* temporary workspaces */
	gsl_interp_accel * acc = NULL;
	gsl_spline * spline = NULL;
	float ** avgptrs = NULL;
	float ** stdptrs = NULL;
	size_t ** nptrs = NULL;
	double * tempinterp = NULL;

	float * dataptr = NULL;
	double * timepoints = NULL;
	double * mirroredtimepoints = NULL;
	double * curtimepoints = NULL;
	bxhdimension * dimt = NULL;
	size_t tsize = 0;

	unsigned int startpt = opt_startpt;
	unsigned int endpt = opt_endpt;

	double epochdurbefore = -1;
	double epochdurafter = -1;

	std::map<double, qvec > eventlist; /* maps event time to list of matching queries */
	std::pair<double, qvec> * eventarray = NULL; /* for faster access to eventlist */
	eventrange_t * eventranges = NULL; /* maps event index to time interval [rbegin, rmid, rend] to be used in output */
	size_t eventnum;

	const char * imagefile = argv[argind];
	const char * eventfiles = argv[argind + 1];
	
	double * dptr = NULL;

	fprintf(stderr, "image file %s\n", imagefile);
	fprintf(stderr, "event file(s) %s\n", eventfiles);

	dimt = &bdrs[imnum].datarec->dimensions[0];
	tsize = dimt->size;

	if (endpt == (unsigned int)-1) {
	    endpt = tsize - 1;
	}
	
	epochdurbefore = opt_secsbefore;
	epochdurafter = opt_secsafter;
	if (epochdurbefore == -1) {
	    epochdurbefore = opt_ptsbefore * TR;
	}
	if (epochdurafter == -1) {
	    epochdurafter = opt_ptsafter * TR;
	}

	if (tsize < MININTERP_PTS) {
	    fprintf(stderr, "Not enough timepoints (%u) in data to perform interpolation (need at least %u)!", (unsigned int)tsize, (unsigned int)MININTERP_PTS);
	    goto FAIL;
	}

	if ((timepoints = get_timepoints(dimt, opt_forcetr, logfp)) == NULL) {
	    fprintf(stderr, FUNC ": error getting timepoints from %s\n", imagefile);
	    goto FAIL;
	}
	curtimepoints = timepoints;

	/* dptr: single voxel time-series data holding area */
	if (opt_mirrorflip) { 
	    dptr = (double *)malloc(sizeof(double) * (tsize + (2 * MININTERP_PTS)));
	    mirroredtimepoints = get_mirrored(timepoints, tsize);
	    curtimepoints = mirroredtimepoints;
	} else {
	    dptr = (double *)malloc(sizeof(double) * tsize);
	}

	if (get_event_list(eventfiles, numqueries, &xpqueries[0], &xpqueryfilters[0], &xpqueryepochexcludes[0], &opt_querylabels[0], epochdurbefore, epochdurafter, eventlist, logfp) != 0)
	    goto FAIL;

	/* create arrays for more efficient event list access */
	precompute_event_ranges(eventlist, timepoints, tsize, startpt, endpt, opt_basestart, opt_baseend, epochdurbefore, epochdurafter, TR, opt_nointerp, opt_mirrorflip, logfp, eventarray, eventranges);

	update_query_counts(numqueries, eventlist, opt_querylabels, logfp, numepochs);

	/* allocate temporary workspaces */
	acc = gsl_interp_accel_alloc();
	spline = gsl_spline_alloc(gsl_interp_cspline, tsize);
	avgptrs = (float **)malloc(sizeof(float *)*numqueries);
	stdptrs = (float **)malloc(sizeof(float *)*numqueries);
	nptrs = (size_t **)malloc(sizeof(size_t *)*numqueries);
	tempinterp = (double *)malloc(sizeof(double)*numptsinwork);
	if (avgptrs == NULL || stdptrs == NULL || nptrs == NULL ||
	    tempinterp == NULL) {
	    fprintf(stderr, FUNC ": error allocating temporary workspaces\n");
	    goto FAIL;
	}

	/* initialize mean image */
	memset(meanimage, '\0', sizeof(float)*volsize);

	for (batchdimoff = 0; batchdimoff < batchdimfullsize; batchdimoff += batchdimbatchsize) {
	    size_t batchindxyz;
	    size_t numbatchvoxels = 0;
	    size_t xyzoff = batchdimoff;
	    size_t batchdimbatchend = batchdimbatchsize;
	    struct bxhdataread bdr;
	    if (batchdimoff + batchdimbatchsize > batchdimfullsize) {
		batchdimbatchend = batchdimfullsize - batchdimoff;
	    }
	    numbatchvoxels = batchdimbatchend;
	    if (batchdimnum != 1) {
		size_t dimnum;
		for (dimnum = 1; dimnum < batchdimnum; dimnum++) {
		    numbatchvoxels *= dimsizexyz[dimnum-1];
		    xyzoff *= dimsizexyz[dimnum-1];
		}
	    }
	    memset(&selectors[0], '\0', sizeof(selectors));
	    sprintf(&selectbuf[0], "%u:%u", (unsigned int)batchdimoff, (unsigned int)(batchdimoff + batchdimbatchend - 1));
	    selectors[batchdimnum] = &selectbuf[0];
	    fprintf(stderr, "Reading voxels %u - %u (last is %u)...\n", (unsigned int)xyzoff, (unsigned int)(xyzoff + numbatchvoxels - 1), (unsigned int)(dimsizexyz[0] * dimsizexyz[1] * dimsizexyz[2]) - 1);
	    fprintf(logfp,  "Reading voxels %u - %u (last is %u)...\n", (unsigned int)xyzoff, (unsigned int)(xyzoff + numbatchvoxels - 1), (unsigned int)(dimsizexyz[0] * dimsizexyz[1] * dimsizexyz[2]) - 1);
	    if (bxh_dataReadFileStart(imagefile, "image", NULL, 4, ordereddimnames, &selectors[0], &bdr) != 0) {
		fprintf(stderr, "Error preparing data read for '%s'.\n", imagefile);
		goto FAIL;
	    }
	    if (bxh_dataReadFinish(&bdr, "float") != 0) {
		fprintf(stderr, "Error finishing data read for '%s'.\n", imagefile);
		goto FAIL;
	    }

	    dataptr = (float *)bdr.dataptr;

	    /* calculate mean image */
	    for (batchindxyz = 0; batchindxyz < numbatchvoxels; batchindxyz++) {
		size_t indxyz = batchindxyz + xyzoff;
		size_t indt;
		for (indt = 0; indt < tsize; indt++) {
		    meanimage[indxyz] += (dataptr[batchindxyz*tsize + indt] - meanimage[indxyz]) / (indt + 1);
		}
	    }

	    /* do averaging, one voxel at a time */
	    for (batchindxyz = 0; batchindxyz < numbatchvoxels; batchindxyz++) {
		size_t indxyz = batchindxyz + xyzoff;
		size_t indt;
		size_t numevents = eventlist.size();
	    
		if (indxyz % 1000 == 0) {
		    fprintf(stderr, "voxel %u/%u\r", (unsigned int)indxyz, (unsigned int)volsize);
		}

		for (indt = 0; indt < tsize; indt++) {
		    size_t tmpt = indt;
		    if (opt_mirrorflip) {
			tmpt += MININTERP_PTS;
		    }
		    dptr[tmpt] = (double)dataptr[batchindxyz*tsize + indt];
		}
		if (opt_mirrorflip) {
		    dptr = get_mirrored(dptr + MININTERP_PTS, tsize, dptr);
		}

		if (maskdataptr && maskdataptr[indxyz] == 0)
		    continue;

		for (indquery = 0; indquery < numqueries; indquery++) {
		    avgptrs[indquery] = &qres_avg[indquery][indxyz*numptsinresult];
		    stdptrs[indquery] = &qres_std[indquery][indxyz*numptsinresult];
		    nptrs[indquery] = &qres_n[indquery][indxyz*numptsinresult];
		}

		gsl_spline_init(spline, curtimepoints, dptr, tsize);

		/* go through list of events */
		for (eventnum = 0; eventnum < numevents; eventnum++) {
		    std::pair<double, qvec> & epair = eventarray[eventnum];
		    double etime = epair.first;
		    double etimepre  = etime - (opt_ptsbefore * TR);
		    double etimepost = etime + (opt_ptsafter  * TR);
		    float baseval;
		    qvec & qlist = epair.second;
		    size_t qind;
		    eventrange_t & eventrange = eventranges[eventnum];
		    double * eptr = NULL;
		    size_t rbegin, rmid, rend;
		    rbegin = eventrange.rbegin;
		    rmid = eventrange.rmid;
		    rend = eventrange.rend;
	    
		    if (!opt_nointerp &&
			(etimepre != timepoints[rbegin] ||
			 etimepost != timepoints[rend])) {
			double t = etime - (TR * (int)workevoffset);
			for (indt = 0; indt < numptsinwork; indt++) {
			    tempinterp[indt] = gsl_spline_eval(spline, t, acc);
			    t += TR;
			}
			eptr = tempinterp;
		    } else {
			/* just point to existing data */
			eptr = &dptr[rmid - workevoffset + (opt_mirrorflip ? MININTERP_PTS : 0)];
		    }

		    {
			size_t bloffset = workevoffset + opt_basestart; /* index of start of baseline in eptr */
			baseval = 0;
			for (indt = 0; indt < numptsinbaseline; indt++) {
			    baseval += (eptr[bloffset+indt] - baseval) / (indt + 1);
			}
			qres_blavg[indxyz] +=
			    (baseval - qres_blavg[indxyz]) *
			    ((1.0 * numptsinbaseline * qlist.size()) /
			     (qres_nbl[indxyz] + (numptsinbaseline * qlist.size())));
			qres_nbl[indxyz] += (numptsinbaseline * qlist.size());
		    }
		    for (qind = 0; qind < qlist.size(); qind++) {
			int indquery = qlist[qind].qind;
			float oldmean;
			float oldssd;
			float newmean;
			float newssd;
			size_t newn;
			float newpoint;
			float * avgptr = avgptrs[indquery];
			float * stdptr = stdptrs[indquery];
			size_t * nptr = nptrs[indquery];
			size_t epochoffset = workevoffset - opt_ptsbefore; /* index of start of baseline in eptr */

			for (indt = 0; indt < numptsinresult; indt++) {
			    /* update mean and std. dev.
			     * qres_std actually only has sum of squared
			     * deviations for now to reduce number of
			     * operations. */
			    newpoint = eptr[epochoffset + indt] - baseval;
			    if (opt_scalebl) {
				newpoint /= baseval;
			    }
			    oldmean = avgptr[indt];
			    oldssd = stdptr[indt];
			    newn = nptr[indt] + 1;
			    newmean = oldmean + (newpoint - oldmean) / newn;
			    newssd = oldssd + (newpoint - oldmean) * (newpoint - newmean);
			    avgptr[indt] = newmean;
			    stdptr[indt] = newssd;
			    nptr[indt] = newn;
			}
			if (trialsummarylists[indquery].numsummaries > 0) {
			    struct trialsummarylist * tslistp = &trialsummarylists[indquery];
			    float ** avgsummpp = qres_summary_avg[indquery];
			    float ** stdsummpp = qres_summary_std[indquery];
			    size_t ** nsummpp = qres_summary_n[indquery];
			    size_t indsummary;
			    for (indsummary = 0; indsummary < tslistp->numsummaries; indsummary++) {
				struct trialsummary * tsp = &tslistp->summaries[indsummary];
				size_t numgroups = tsp->numgroups;
				float * avgsummptr = &avgsummpp[indsummary][indxyz*numgroups];
				float * stdsummptr = &stdsummpp[indsummary][indxyz*numgroups];
				size_t * nsummptr = &nsummpp[indsummary][indxyz*numgroups];
				size_t indgroup;
				for (indgroup = 0; indgroup < numgroups; indgroup++) {
				    struct trialsummarygroup * tsg = &tsp->groups[indgroup];
				    size_t indpoint;
				    newpoint = 0;
				    for (indpoint = 0; indpoint < tsg->numpoints; indpoint++) {
					float curpoint = eptr[epochoffset + tsg->points[indpoint]] - baseval;
					newpoint += (curpoint - newpoint) / (indpoint + 1);
				    }
				    oldmean = avgsummptr[indgroup];
				    oldssd = stdsummptr[indgroup];
				    newn = nsummptr[indgroup] + 1;
				    newmean = oldmean + (newpoint - oldmean) / newn;
				    newssd = oldssd + (newpoint - oldmean) * (newpoint - newmean);
				    avgsummptr[indgroup] = newmean;
				    stdsummptr[indgroup] = newssd;
				    nsummptr[indgroup] = newn;
				}
			    }
			}
		    }

		} /* for each event */

	    } /* for each voxel in batch */

	    bxh_datareaddata_free(&bdr);
	} /* for each batch of voxels */

	delete [] eventarray;
	delete [] eventranges;

	if (timepoints)
	    free(timepoints);
	if (mirroredtimepoints)
	    free(mirroredtimepoints);

	/* free temporary workspaces */
	free(avgptrs); avgptrs = NULL;
	free(stdptrs); stdptrs = NULL;
	free(nptrs); nptrs = NULL;
	free(tempinterp); tempinterp = NULL;
	gsl_spline_free(spline); spline = NULL;
	gsl_interp_accel_free(acc); acc = NULL;

	free(dptr); dptr = NULL;
    } /* for each image file */

    /* fix std. dev. */
    for (indquery = 0; indquery < numqueries; indquery++) {
	size_t ressize = volsize * numptsinresult;
	float * stdptr = qres_std[indquery];
	size_t * nptr = qres_n[indquery];
	float temp;
	size_t indxyzt;
	for (indxyzt = 0; indxyzt < ressize; indxyzt++) {
	    temp = *stdptr;
	    if (*nptr == 0 || *nptr == 1)
		temp = 0;
	    else
		temp /= (*nptr - 1); /* unbiased std. dev. */
	    *stdptr = sqrt(temp);
	    stdptr++;
	    nptr++;
	}
    }
    for (indquery = 0; indquery < numqueries; indquery++) {
	struct trialsummarylist * tslistp = &trialsummarylists[indquery];
	float ** stdsummpp = qres_summary_std[indquery];
	size_t ** nsummpp = qres_summary_n[indquery];
	size_t indsummary;
	for (indsummary = 0; indsummary < tslistp->numsummaries; indsummary++) {
	    struct trialsummary * tsp = &tslistp->summaries[indsummary];
	    size_t numgroups = tsp->numgroups;
	    size_t ressize = volsize * numgroups;
	    float * stdptr = stdsummpp[indsummary];
	    size_t * nptr = nsummpp[indsummary];
	    float temp;
	    size_t indxyzt;
	    for (indxyzt = 0; indxyzt < ressize; indxyzt++) {
		temp = *stdptr;
		if (*nptr == 0)
		    temp = 0;
		else
		    temp /= (*nptr - 1); /* unbiased std. dev. */
		*stdptr = sqrt(temp);
		stdptr++;
		nptr++;
	    }
	}
    }

    if (opt_extracttrials) {
	int indquery;
	double * tempinterp = NULL;
	/* all of the following vectors indexed by query */

	fprintf(stderr, "Extracting trials...\n");
	fprintf(logfp,  "Extracting trials...\n");

	tempinterp = (double *)malloc(sizeof(double)*numptsinwork);
	if (tempinterp == NULL) {
	    fprintf(stderr, FUNC ": error allocating temporary workspace\n");
	    fprintf(logfp,  FUNC ": error allocating temporary workspace\n");
	    goto FAIL;
	}

	/* extract trials */
	for (indquery = 0; indquery < numqueries; indquery++) {
	    size_t numtrialsinquery = 0;
	    FILE * trialfp = NULL;
	    if (opt_extracttrials) {
		sprintf(outputimgname, "%s_%s_trials.img", outputprefix, opt_querylabels[indquery]);
		if ((trialfp = fopen(outputimgname, "w")) == NULL) {
		    fprintf(stderr, FUNC ": error opening %s for writing\n", outputimgname);
		    fprintf(logfp, FUNC ": error opening %s for writing\n", outputimgname);
		    goto FAIL;
		}
	    }
	    fprintf(stderr, "Extracting trials for query '%s'\n",
		    opt_querylabels[indquery]);
	    fprintf(logfp,  "Extracting trials for query '%s'\n",
		    opt_querylabels[indquery]);
	    for (argind = 2; argind < argc; argind += 2) {
		int imnum = (argind - 2) / 2;

		/* temporary workspaces */
		gsl_interp_accel * acc = NULL;
		gsl_spline * spline = NULL;

		float * dataptr = NULL;
		double * timepoints = NULL;
		double * mirroredtimepoints = NULL;
		double * curtimepoints = NULL;
		bxhdimension * dimt = NULL;
		size_t tsize = 0;

		unsigned int startpt = opt_startpt;
		unsigned int endpt = opt_endpt;

		double epochdurbefore = -1;
		double epochdurafter = -1;

		size_t numevents;

		std::map<double, qvec > eventlist; /* maps event time to list of matching queries */
		std::pair<double, qvec> * eventarray = NULL; /* for faster access to eventlist */
		eventrange_t * eventranges = NULL; /* maps event index to time interval [rbegin, rmid, rend] to be used in output */
		size_t eventnum;

		const char * imagefile = argv[argind];
		const char * eventfiles = argv[argind + 1];

		double * dptr = NULL;

		size_t numtrialsinfile = 0;
	    
		float * trials = NULL; /* stores trials for one query */

		fprintf(stderr, " image file %s\n", imagefile);

		dimt = &bdrs[imnum].datarec->dimensions[0];
		tsize = dimt->size;

		acc = gsl_interp_accel_alloc();
		spline = gsl_spline_alloc(gsl_interp_cspline, dimt->size);

		if (endpt == (unsigned int)-1) {
		    endpt = dimt->size - 1;
		}
	
		epochdurbefore = opt_secsbefore;
		epochdurafter = opt_secsafter;
		if (epochdurbefore == -1) {
		    epochdurbefore = opt_ptsbefore * TR;
		}
		if (epochdurafter == -1) {
		    epochdurafter = opt_ptsafter * TR;
		}

		if ((timepoints = get_timepoints(dimt, opt_forcetr, logfp)) == NULL) {
		    fprintf(stderr, FUNC ": error getting timepoints from %s\n", imagefile);
		    goto FAIL;
		}
		curtimepoints = timepoints;

		/* dptr: single voxel time-series data holding area */
		if (opt_mirrorflip) { 
		    dptr = (double *)malloc(sizeof(double) * (tsize + (2 * MININTERP_PTS)));
		    mirroredtimepoints = get_mirrored(timepoints, tsize);
		    curtimepoints = mirroredtimepoints;
		} else {
		    dptr = (double *)malloc(sizeof(double) * tsize);
		}

		if (get_event_list(eventfiles, numqueries, &xpqueries[0], &xpqueryfilters[0], &xpqueryepochexcludes[0], &opt_querylabels[0], epochdurbefore, epochdurafter, eventlist, NULL) != 0)
		    goto FAIL;

		/* create arrays for more efficient event list access */
		precompute_event_ranges(eventlist, timepoints, dimt->size, startpt, endpt, opt_basestart, opt_baseend, epochdurbefore, epochdurafter, TR, opt_nointerp, opt_mirrorflip, NULL, eventarray, eventranges);

		numevents = eventlist.size();

		numtrialsinfile = 0;
		for (eventnum = 0; eventnum < numevents; eventnum++) {
		    qvec & qlist = eventarray[eventnum].second;
		    size_t qind;
		    for (qind = 0; qind < qlist.size(); qind++) {
			if (indquery == qlist[qind].qind)
			    numtrialsinfile++;
		    }
		}
		numtrialsinquery += numtrialsinfile;

		if (numtrialsinfile > 0) {
		    if (opt_extracttrials) {
			trials = (float *)malloc(sizeof(float)*volsize*numptsinresult*numtrialsinfile);
			if (trials == NULL) {
			    fprintf(stderr, FUNC ": error allocating memory for trials\n");
			    fprintf(logfp,  FUNC ": error allocating memory for trials\n");
			    goto FAIL;
			}
		    }

		    for (batchdimoff = 0; batchdimoff < batchdimfullsize; batchdimoff += batchdimbatchsize) {
			size_t batchindxyz;
			size_t numbatchvoxels = 0;
			size_t xyzoff = batchdimoff;
			size_t batchdimbatchend = batchdimbatchsize;
			struct bxhdataread bdr;
			if (batchdimoff + batchdimbatchsize > batchdimfullsize) {
			    batchdimbatchend = batchdimfullsize - batchdimoff;
			}
			numbatchvoxels = batchdimbatchend;
			if (batchdimnum != 1) {
			    size_t dimnum;
			    for (dimnum = 1; dimnum < batchdimnum; dimnum++) {
				numbatchvoxels *= dimsizexyz[dimnum-1];
				xyzoff *= dimsizexyz[dimnum-1];
			    }
			}
			memset(&selectors[0], '\0', sizeof(selectors));
			sprintf(&selectbuf[0], "%u:%u", (unsigned int)batchdimoff, (unsigned int)(batchdimoff + batchdimbatchend - 1));
			selectors[batchdimnum] = &selectbuf[0];
			fprintf(stderr, " Reading voxels %u - %u (last is %u)...\n", (unsigned int)xyzoff, (unsigned int)(xyzoff + numbatchvoxels - 1), (unsigned int)(dimsizexyz[0] * dimsizexyz[1] * dimsizexyz[2]) - 1);
			fprintf(logfp,  " Reading voxels %u - %u (last is %u)...\n", (unsigned int)xyzoff, (unsigned int)(xyzoff + numbatchvoxels - 1), (unsigned int)(dimsizexyz[0] * dimsizexyz[1] * dimsizexyz[2]) - 1);
			if (bxh_dataReadFileStart(imagefile, "image", NULL, 4, ordereddimnames, &selectors[0], &bdr) != 0) {
			    fprintf(stderr, "Error preparing data read for '%s'.\n", imagefile);
			    goto FAIL;
			}
			if (bxh_dataReadFinish(&bdr, "float") != 0) {
			    fprintf(stderr, "Error finishing data read for '%s'.\n", imagefile);
			    goto FAIL;
			}

			dataptr = (float *)bdr.dataptr;

			for (batchindxyz = 0; batchindxyz < numbatchvoxels; batchindxyz++) {
			    size_t indxyz = batchindxyz + xyzoff;
			    size_t indt;
			    size_t trialind = 0;

			    if (indxyz % 1000 == 0) {
				fprintf(stderr, " voxel %u/%u\r", (unsigned int)batchindxyz, (unsigned int)numbatchvoxels);
			    }

			    for (indt = 0; indt < dimt->size; indt++) {
				size_t tmpt = indt;
				if (opt_mirrorflip) {
				    tmpt += MININTERP_PTS;
				}
				dptr[tmpt] = (double)dataptr[batchindxyz*tsize + indt];
			    }
			    if (opt_mirrorflip) {
				dptr = get_mirrored(dptr + MININTERP_PTS, tsize, dptr);
			    }

			    gsl_spline_init(spline, curtimepoints, dptr, dimt->size);

			    for (eventnum = 0; eventnum < numevents; eventnum++) {
				std::pair<double, qvec> & epair = eventarray[eventnum];
				double etime = epair.first;
				double etimepre  = etime - (opt_ptsbefore * TR);
				double etimepost = etime + (opt_ptsafter  * TR);
				float baseval;
				qvec & qlist = epair.second;
				size_t qind;
				eventrange_t & eventrange = eventranges[eventnum];
				double * eptr = NULL;
				size_t rbegin, rmid, rend;
				/* only go through with this if the current query
				   collects this event */
				for (qind = 0; qind < qlist.size(); qind++) {
				    if (indquery == qlist[qind].qind)
					break;
				}
				if (qind == qlist.size())
				    goto NEXTEVENT;

				rbegin = eventrange.rbegin;
				rmid = eventrange.rmid;
				rend = eventrange.rend;

				if (!opt_nointerp &&
				    (etimepre != timepoints[rbegin] ||
				     etimepost != timepoints[rend])) {
				    /* Need to interpolate into qres_temp */
				    double t = etime - (TR * (int)workevoffset);
				    for (indt = 0; indt < numptsinwork; indt++) {
					tempinterp[indt] = gsl_spline_eval(spline, t, acc);
					t += TR;
				    }
				    eptr = tempinterp;
				} else {
				    /* just point to existing data */
				    eptr = &dptr[rmid - workevoffset + (opt_mirrorflip ? MININTERP_PTS : 0)];
				}

				{
				    size_t bloffset = workevoffset + opt_basestart; /* index of start of baseline in eptr */
				    baseval = 0;
				    for (indt = 0; indt < numptsinbaseline; indt++) {
					baseval += (eptr[bloffset+indt] - baseval) / (indt + 1);
				    }
				}

				{
				    double newpoint;
				    size_t epochoffset = workevoffset - opt_ptsbefore; /* index of start of baseline in eptr */
				    if (trials) {
					for (indt = 0; indt < numptsinresult; indt++) {
					    newpoint = eptr[epochoffset + indt] - baseval;
					    trials[(((trialind * numptsinresult) + indt) * volsize) + indxyz] = newpoint;
					}
				    }
				}
				trialind++;
			      NEXTEVENT: /* null */;
			    } /* for each event */
			} /* for each voxel in batch */

			bxh_datareaddata_free(&bdr);
		    } /* for each batch of voxels */

		    /* write out trials */
		    if (trials) {
			if (fwrite(trials, sizeof(float)*volsize*numptsinresult*numtrialsinfile, 1, trialfp) != 1) {
			    fprintf(stderr, "Error writing to file %s\n", outputimgname);
			    fclose(trialfp);
			    goto FAIL;
			}
		    }
		} /* if (numtrialsinfile > 0) */

		delete [] eventarray;
		delete [] eventranges;

		if (timepoints)
		    free(timepoints);
		if (mirroredtimepoints)
		    free(mirroredtimepoints);

		gsl_spline_free(spline); spline = NULL;
		gsl_interp_accel_free(acc); acc = NULL;

		free(dptr); dptr = NULL;
		free(trials);
	    } /* for each image */

	    /* close output trials file, and make .bxh file */
	    {
		static const char * trialdimnames[] = { "x", "y", "z", "t" };
		struct bxhdataread trialbdr;
		fclose(trialfp);
		/* get a new dataread struct template for output .bxh */
		sprintf(outputbxhname, "%s_%s_trials.bxh", outputprefix, opt_querylabels[indquery]);
		if (bxh_dataReadFileStart(argv[2], "image", NULL, 4, trialdimnames, NULL, &trialbdr) != 0) {
		    fprintf(stderr, "Error reading '%s'.\n", argv[2]);
		    goto FAIL;
		}
		trialbdr.datarec->msbfirstfrags = msbfirst; /* because we don't call bxh_dataReadFinish, we don't get a converted datarec */
		for (int dpstructind = 0;
		     dpstructind < trialbdr.datarec->dimensions[3].numdpstructs;
		     dpstructind++) {
		    bxhdimension * dimp = &trialbdr.datarec->dimensions[3];
		    bxhdatapoints * dpstructp = &dimp->dpstructs[dpstructind];
		    unsigned int dpnum;
		    for (dpnum = 0; dpnum < dpstructp->numvalues; dpnum++) {
			free(dpstructp->values[dpnum]);
			dpstructp->values[dpnum] = NULL;
		    }
		    free(dpstructp->values);
		    dpstructp->values = NULL;
		    if (dpstructp->label != NULL) {
			free(dpstructp->label);
			dpstructp->label = NULL;
		    }
		}
		if (trialbdr.datarec->dimensions[3].dpstructs != NULL) {
		    free(trialbdr.datarec->dimensions[3].dpstructs);
		    trialbdr.datarec->dimensions[3].dpstructs = NULL;
		}
		trialbdr.datarec->dimensions =
		    (bxhdimension *)realloc(trialbdr.datarec->dimensions,
					    sizeof(bxhdimension)*5);
		trialbdr.datarec->numdims = 5;
		trialbdr.datarec->dimensions[3].size = numptsinresult;
		{
		    bxhdimension * trialdim = &trialbdr.datarec->dimensions[4];
		    memset(trialdim, '\0', sizeof(bxhdimension));
		    trialdim->type = strdup("trial");
		    trialdim->size = numtrialsinquery;
		}
		if (bxh_addAutoHistoryEntry(trialbdr.docp, oldargv[0], (const char **)&oldargv[1], oldargc-1) != 0) {
		    fprintf(stderr, "Error adding history entry\n");
		    goto FAIL;
		}
		free(trialbdr.datarec->elemtype);
		trialbdr.datarec->elemtype = strdup("float32");

		bxh_datarec_frags_free(trialbdr.datarec);
		bxh_datarec_addfrag(trialbdr.datarec, outputimgname, 0, sizeof(float) * volsize * numptsinresult * numtrialsinquery, outputbxhname, 1);
		if (bxh_datarec_writeToElement(trialbdr.imagedatap, trialbdr.datarec) != 0) {
		    fprintf(stderr, "Failed writing datarec\n");
		    goto FAIL;
		}
		if (bxh_writeFile(trialbdr.docp, outputbxhname) != 0) {
		    fprintf(stderr, "Error writing output file %s\n", outputbxhname);
		    goto FAIL;
		}
		bxh_datareaddata_free(&trialbdr);
	    } /* end writing trials */
	} /* for each query */

	free(tempinterp); tempinterp = NULL;
    } /* end extract trials */

    if (opt_tm) {
	int indquery;
	double * tempinterp = NULL;
	/* all of the following vectors indexed by query */
	std::vector<size_t> seedxyz;
	std::vector<size_t> seedt;
	size_t * curepochs = NULL; /* finished epochs per query */
	size_t * tmpepochs = NULL; /* finished epochs per query */

	tempinterp = (double *)malloc(sizeof(double)*numptsinwork);
	if (tempinterp == NULL) {
	    fprintf(stderr, FUNC ": error allocating temporary workspace\n");
	    goto FAIL;
	}

	if (opt_tmseed) {
	    int dimnum = 0;
	    size_t seed[4];
	    size_t tmpseedxyz;
	    char * curpos = opt_tmseed;
	    while (*curpos != '\0' && dimnum < 4) {
		size_t valuelen = 0;
		char * comma = NULL;
		char * endptr = NULL;
		int newdimnum = 0;
		comma = strchr(curpos, ',');
		if (comma) {
		    valuelen = comma - curpos;
		    *comma = '\0';
		} else {
		    valuelen = strlen(curpos);
		}
		/* rearrange from X,Y,Z,T to T,X,Y,Z */
		if (dimnum >= 0 || dimnum <= 2) {
		    newdimnum = dimnum + 1;
		} else if (dimnum == 3) {
		    newdimnum = 0;
		}
		seed[newdimnum] = (size_t)strtod((char *)curpos, &endptr);
		dimnum++;
		if (*endptr != '\0') {
		    fprintf(stderr, FUNC ": error: bad seed coordinate value %s\n", (char *)curpos);
		    free(curpos);
		    goto FAIL;
		}
		if (comma) {
		    *comma = ',';
		    curpos = comma + 1;
		} else {
		    curpos += valuelen;
		}
	    }
	    if (dimnum != 4) {
		fprintf(stderr, "--trialmaxseed '%s' must be in the form X,Y,Z,T\n", opt_tmseed);
		goto FAIL;
	    }
	    for (dimnum = 0; dimnum < 4; dimnum++) {
		if (seed[dimnum] >= bdrs[0].datarec->dimensions[dimnum].size) {
		    fprintf(stderr, FUNC ": error: seed coordinate %d is greater than size of dimension (%d)\n", seed[dimnum], bdrs[0].datarec->dimensions[dimnum].size);
		    goto FAIL;
		}
	    }
	    tmpseedxyz = seed[1] +
		(seed[2] * dimsizexyz[0]) +
		(seed[3] * dimsizexyz[0] * dimsizexyz[1]);
	    for (indquery = 0; indquery < numqueries; indquery++) {
		seedxyz.push_back(tmpseedxyz);
		seedt.push_back(seed[0]);
	    }
	} else {
	    fprintf(stderr, "Finding seed voxel for --trialmax...\n");
	    fprintf(logfp,  "Finding seed voxel for --trialmax...\n");

	    for (indquery = 0; indquery < numqueries; indquery++) {
		seedxyz.push_back(0);
		seedt.push_back(0);
		/* First calculate mean intensities per timepoint in epoch
		 * and find the timepoint with the maximum mean intensity.
		 */
		size_t indxyz, indt;
		float seedvoxelval = 0;
		float seedtimeval = 0;
		float * qavgptr = qres_avg[indquery];
		for (indt = 0; indt < numptsinresult; indt++) {
		    size_t numroivoxels = 0;
		    float meanval = 0;
		    for (indxyz = 0; indxyz < volsize; indxyz++) {
			if (tmroidataptr && tmroidataptr[indxyz] == 0)
			    continue;
			meanval +=
			    qavgptr[(indxyz * numptsinresult) + indt] /
			    (numroivoxels + 1);
			numroivoxels++;
		    }
		    if (indt == 0 || meanval > seedtimeval) {
			seedt[indquery] = indt;
			seedtimeval = meanval;
		    }
		}
		/* seedt now has the timepoint with the maximum intensity.
		 * Now find the voxel within that timepoint with the maximum
		 * intensity.
		 */
		for (indxyz = 0; indxyz < volsize; indxyz++) {
		    if (tmroidataptr && tmroidataptr[indxyz] == 0)
			continue;
		    float tmpval = qavgptr[(indxyz * numptsinresult) + seedt[indquery]];
		    if (indxyz == 0 || tmpval > seedvoxelval) {
			seedxyz[indquery] = indxyz;
			seedvoxelval = tmpval;
		    }
		}

		{
		    static char coords[128];

		    size_t prefixlen = 0;
		    size_t labellen = 0;

		    prefixlen = strlen(outputprefix);
		    labellen = strlen(opt_querylabels[indquery]);

		    sprintf(&coords[0], "%d %d %d",
			     (int)(seedxyz[indquery] % dimsizexyz[0]),
			     (int)((seedxyz[indquery] / dimsizexyz[0]) % dimsizexyz[1]),
			     (int)(((seedxyz[indquery] / dimsizexyz[0]) / dimsizexyz[1]) % dimsizexyz[2]));
		    fprintf(stderr, "Seed voxel is %s...\n", &coords[0]);
		    fprintf(logfp,  "Seed voxel is %s...\n", &coords[0]);

		    sprintf(outputfilename, "%s_%s_trialmaxseed.txt", outputprefix, opt_querylabels[indquery]);
		    if ((fp = fopen(outputfilename, "w")) == NULL) {
			fprintf(stderr, "Error opening file %s\n", outputfilename);
			fclose(fp);
			goto FAIL;
		    }
		    if (fprintf(fp, "%s\n", &coords[0]) <= 0) {
			fprintf(stderr, "Error writing to file %s\n", outputfilename);
			fclose(fp);
			goto FAIL;
		    }
		    fclose(fp);
		}
	    }
	}

	fprintf(stderr, "Extracting trial data...\n");
	fprintf(logfp,  "Extracting trial data...\n");

	for (indquery = 0; indquery < numqueries; indquery++) {
	    qres_tm[indquery] =
		(float *)
		malloc(sizeof(float) * volsize * numepochs[indquery]);
	    if (qres_tm[indquery] == NULL) {
		fprintf(stderr, FUNC ": error allocating memory for trial data results for query %d\n", indquery);
		goto FAIL;
	    }
	    memset(qres_tm[indquery], '\0', sizeof(float)*volsize*numepochs[indquery]);
	}
	curepochs = (size_t *)malloc(sizeof(curepochs[0])*numqueries);
	memset(curepochs, '\0', sizeof(curepochs[0])*numqueries);
	tmpepochs = (size_t *)malloc(sizeof(tmpepochs[0])*numqueries);
	memset(tmpepochs, '\0', sizeof(tmpepochs[0])*numqueries);

	/* extract trials */
	for (argind = 2; argind < argc; argind += 2) {
	    int imnum = (argind - 2) / 2;

	    /* temporary workspaces */
	    gsl_interp_accel * acc = NULL;
	    gsl_spline * spline = NULL;

	    float * dataptr = NULL;
	    double * timepoints = NULL;
	    double * mirroredtimepoints = NULL;
	    double * curtimepoints = NULL;
	    bxhdimension * dimt = NULL;
	    size_t tsize = 0;

	    unsigned int startpt = opt_startpt;
	    unsigned int endpt = opt_endpt;

	    double epochdurbefore = -1;
	    double epochdurafter = -1;

	    size_t numevents;

	    std::map<double, qvec > eventlist; /* maps event time to list of matching queries */
	    std::pair<double, qvec> * eventarray = NULL; /* for faster access to eventlist */
	    eventrange_t * eventranges = NULL; /* maps event index to time interval [rbegin, rmid, rend] to be used in output */
	    size_t eventnum;

	    const char * imagefile = argv[argind];
	    const char * eventfiles = argv[argind + 1];

	    double * dptr = NULL;

	    fprintf(stderr, "image file %s\n", imagefile);
	    fprintf(stderr, "event file(s) %s\n", eventfiles);

	    dimt = &bdrs[imnum].datarec->dimensions[0];
	    tsize = dimt->size;

	    acc = gsl_interp_accel_alloc();
	    spline = gsl_spline_alloc(gsl_interp_cspline, dimt->size);

	    if (endpt == (unsigned int)-1) {
		endpt = dimt->size - 1;
	    }
	
	    epochdurbefore = opt_secsbefore;
	    epochdurafter = opt_secsafter;
	    if (epochdurbefore == -1) {
		epochdurbefore = opt_ptsbefore * TR;
	    }
	    if (epochdurafter == -1) {
		epochdurafter = opt_ptsafter * TR;
	    }

	    if ((timepoints = get_timepoints(dimt, opt_forcetr, logfp)) == NULL) {
		fprintf(stderr, FUNC ": error getting timepoints from %s\n", imagefile);
		goto FAIL;
	    }
	    curtimepoints = timepoints;

	    /* dptr: single voxel time-series data holding area */
	    if (opt_mirrorflip) { 
		dptr = (double *)malloc(sizeof(double) * (tsize + (2 * MININTERP_PTS)));
		mirroredtimepoints = get_mirrored(timepoints, tsize);
		curtimepoints = mirroredtimepoints;
	    } else {
		dptr = (double *)malloc(sizeof(double) * tsize);
	    }

	    if (get_event_list(eventfiles, numqueries, &xpqueries[0], &xpqueryfilters[0], &xpqueryepochexcludes[0], &opt_querylabels[0], epochdurbefore, epochdurafter, eventlist, logfp) != 0)
		goto FAIL;

	    /* create arrays for more efficient event list access */
	    precompute_event_ranges(eventlist, timepoints, dimt->size, startpt, endpt, opt_basestart, opt_baseend, epochdurbefore, epochdurafter, TR, opt_nointerp, opt_mirrorflip, logfp, eventarray, eventranges);

	    numevents = eventlist.size();

	    for (batchdimoff = 0; batchdimoff < batchdimfullsize; batchdimoff += batchdimbatchsize) {
		size_t batchindxyz;
		size_t numbatchvoxels = 0;
		size_t xyzoff = batchdimoff;
		size_t batchdimbatchend = batchdimbatchsize;
		struct bxhdataread bdr;
		if (batchdimoff + batchdimbatchsize > batchdimfullsize) {
		    batchdimbatchend = batchdimfullsize - batchdimoff;
		}
		numbatchvoxels = batchdimbatchend;
		if (batchdimnum != 1) {
		    size_t dimnum;
		    for (dimnum = 1; dimnum < batchdimnum; dimnum++) {
			numbatchvoxels *= dimsizexyz[dimnum-1];
			xyzoff *= dimsizexyz[dimnum-1];
		    }
		}
		memset(&selectors[0], '\0', sizeof(selectors));
		sprintf(&selectbuf[0], "%u:%u", (unsigned int)batchdimoff, (unsigned int)(batchdimoff + batchdimbatchend - 1));
		selectors[batchdimnum] = &selectbuf[0];
		fprintf(stderr, "Reading voxels %u - %u (last is %u)...\n", (unsigned int)xyzoff, (unsigned int)(xyzoff + numbatchvoxels - 1), (unsigned int)(dimsizexyz[0] * dimsizexyz[1] * dimsizexyz[2]) - 1);
		fprintf(logfp,  "Reading voxels %u - %u (last is %u)...\n", (unsigned int)xyzoff, (unsigned int)(xyzoff + numbatchvoxels - 1), (unsigned int)(dimsizexyz[0] * dimsizexyz[1] * dimsizexyz[2]) - 1);
		if (bxh_dataReadFileStart(imagefile, "image", NULL, 4, ordereddimnames, &selectors[0], &bdr) != 0) {
		    fprintf(stderr, "Error preparing data read for '%s'.\n", imagefile);
		    goto FAIL;
		}
		if (bxh_dataReadFinish(&bdr, "float") != 0) {
		    fprintf(stderr, "Error finishing data read for '%s'.\n", imagefile);
		    goto FAIL;
		}

		dataptr = (float *)bdr.dataptr;

		for (batchindxyz = 0; batchindxyz < numbatchvoxels; batchindxyz++) {
		    size_t indxyz = batchindxyz + xyzoff;
		    size_t indt;

		    if (indxyz % 1000 == 0) {
			fprintf(stderr, "voxel %u/%u\r", (unsigned int)indxyz, (unsigned int)volsize);
		    }

		    for (indt = 0; indt < tsize; indt++) {
			size_t tmpt = indt;
			if (opt_mirrorflip) {
			    tmpt += MININTERP_PTS;
			}
			dptr[tmpt] = (double)dataptr[batchindxyz*tsize + indt];
		    }
		    if (opt_mirrorflip) {
			dptr = get_mirrored(dptr + MININTERP_PTS, tsize, dptr);
		    }

		    memcpy(tmpepochs, curepochs, sizeof(size_t)*numqueries);
		
		    gsl_spline_init(spline, curtimepoints, dptr, dimt->size);

		    for (eventnum = 0; eventnum < numevents; eventnum++) {
			std::pair<double, qvec> & epair = eventarray[eventnum];
			double etime = epair.first;
			double etimepre  = etime - (opt_ptsbefore * TR);
			double etimepost = etime + (opt_ptsafter  * TR);
			float baseval;
			qvec & qlist = epair.second;
			size_t qind;
			eventrange_t & eventrange = eventranges[eventnum];
			double * eptr = NULL;
			size_t rbegin, rmid, rend;
			rbegin = eventrange.rbegin;
			rmid = eventrange.rmid;
			rend = eventrange.rend;
	    
			if (!opt_nointerp &&
			    (etimepre != timepoints[rbegin] ||
			     etimepost != timepoints[rend])) {
			    /* Need to interpolate into qres_temp */
			    double t = etime - (TR * (int)workevoffset);
			    for (indt = 0; indt < numptsinwork; indt++) {
				tempinterp[indt] = gsl_spline_eval(spline, t, acc);
				t += TR;
			    }
			    eptr = tempinterp;
			} else {
			    /* just point to existing data */
			    eptr = &dptr[rmid - workevoffset + (opt_mirrorflip ? MININTERP_PTS : 0)];
			}

			{
			    size_t bloffset = workevoffset + opt_basestart; /* index of start of baseline in eptr */
			    baseval = 0;
			    for (indt = 0; indt < numptsinbaseline; indt++) {
				baseval += (eptr[bloffset+indt] - baseval) / (indt + 1);
			    }
			}
			for (qind = 0; qind < qlist.size(); qind++) {
			    int indquery = qlist[qind].qind;
			    double newpoint;
			    size_t epochoffset = workevoffset - opt_ptsbefore; /* index of start of baseline in eptr */
			    newpoint = eptr[epochoffset + seedt[indquery]] - baseval;
			    qres_tm[indquery][(tmpepochs[indquery] * volsize) + indxyz] = newpoint;
			    tmpepochs[indquery]++;
			}
		    } /* for each event */
		
		} /* for each voxel in batch */

		bxh_datareaddata_free(&bdr);
	    } /* for each batch of voxels */

	    /* update query counts */
	    for (eventnum = 0; eventnum < numevents; eventnum++) {
		std::pair<double, qvec> & epair = eventarray[eventnum];
		qvec & qlist = epair.second;
		size_t qind;
		for (qind = 0; qind < qlist.size(); qind++) {
		    int indquery = qlist[qind].qind;
		    curepochs[indquery]++;
		}
	    }

	    delete [] eventarray;
	    delete [] eventranges;

	    if (timepoints)
		free(timepoints);
	    if (mirroredtimepoints)
		free(mirroredtimepoints);

	    gsl_spline_free(spline); spline = NULL;
	    gsl_interp_accel_free(acc); acc = NULL;

	    free(dptr); dptr = NULL;
	} /* for each image */
	free(tempinterp); tempinterp = NULL;
    } /* end trialmax */

    /* permute results from txyz to xyzt */
    {
	float * dtemp =
	    (float *)malloc(sizeof(float)*volsize*numptsinresult);
	size_t * stemp =
	    (size_t *)malloc(sizeof(size_t)*volsize*numptsinresult);
	if (dtemp == NULL || stemp == NULL) {
	    fprintf(stderr, FUNC ": error allocating memory for permutation\n");
	    goto FAIL;
	}
	for (indquery = 0; indquery < numqueries; indquery++) {
	    float * qavg = qres_avg[indquery];
	    float * qstd = qres_std[indquery];
	    size_t * qn = qres_n[indquery];
	    size_t indxyz, indt;
	    for (indt = 0; indt < numptsinresult; indt++) {
		size_t preoff = indt;
		size_t postoff = indt * volsize;
		for (indxyz = 0; indxyz < volsize; indxyz++) {
		    dtemp[postoff] = qavg[preoff];
		    preoff += numptsinresult;
		    postoff++;
		}
	    }
	    qres_avg[indquery] = dtemp;
	    dtemp = qavg;

	    for (indt = 0; indt < numptsinresult; indt++) {
		size_t preoff = indt;
		size_t postoff = indt * volsize;
		for (indxyz = 0; indxyz < volsize; indxyz++) {
		    dtemp[postoff] = qstd[preoff];
		    preoff += numptsinresult;
		    postoff++;
		}
	    }
	    qres_std[indquery] = dtemp;
	    dtemp = qstd;

	    for (indt = 0; indt < numptsinresult; indt++) {
		size_t preoff = indt;
		size_t postoff = indt * volsize;
		for (indxyz = 0; indxyz < volsize; indxyz++) {
		    stemp[postoff] = qn[preoff];
		    preoff += numptsinresult;
		    postoff++;
		}
	    }
	    qres_n[indquery] = stemp;
	    stemp = qn;

	    /* do the same for summaries */
	    {
		struct trialsummarylist * tslistp = &trialsummarylists[indquery];
		size_t indsummary;
		for (indsummary = 0; indsummary < tslistp->numsummaries; indsummary++) {
		    struct trialsummary * tsp = &tslistp->summaries[indsummary];
		    size_t numgroups = tsp->numgroups;
		    float * dsummtemp =
			(float *)malloc(sizeof(float)*volsize*numgroups);
		    size_t * ssummtemp =
			(size_t *)malloc(sizeof(size_t)*volsize*numgroups);
		    if (dsummtemp == NULL || ssummtemp == NULL) {
			fprintf(stderr, FUNC ": error allocating memory for permutation\n");
			goto FAIL;
		    }

		    qavg = qres_summary_avg[indquery][indsummary];
		    qstd = qres_summary_std[indquery][indsummary];
		    qn = qres_summary_n[indquery][indsummary];
		    for (indt = 0; indt < numgroups; indt++) {
			size_t preoff = indt;
			size_t postoff = indt * volsize;
			for (indxyz = 0; indxyz < volsize; indxyz++) {
			    dsummtemp[postoff] = qavg[preoff];
			    preoff += numgroups;
			    postoff++;
			}
		    }
		    qres_summary_avg[indquery][indsummary] = dsummtemp;
		    dsummtemp = qavg;

		    for (indt = 0; indt < numgroups; indt++) {
			size_t preoff = indt;
			size_t postoff = indt * volsize;
			for (indxyz = 0; indxyz < volsize; indxyz++) {
			    dsummtemp[postoff] = qstd[preoff];
			    preoff += numgroups;
			    postoff++;
			}
		    }
		    qres_summary_std[indquery][indsummary] = dsummtemp;
		    dsummtemp = qstd;

		    for (indt = 0; indt < numgroups; indt++) {
			size_t preoff = indt;
			size_t postoff = indt * volsize;
			for (indxyz = 0; indxyz < volsize; indxyz++) {
			    ssummtemp[postoff] = qn[preoff];
			    preoff += numgroups;
			    postoff++;
			}
		    }
		    qres_summary_n[indquery][indsummary] = ssummtemp;
		    ssummtemp = qn;

		    free(dsummtemp);
		    free(ssummtemp);
		}
	    }
	}

	free(dtemp);
	free(stemp);
    }

    writebdr = bdrs[0];
    writebdr.datarec->msbfirstfrags = msbfirst; /* because we don't call bxh_dataReadFinish, we don't get a converted datarec */
    /* move dimensions around because we read data in txyz order */
    {
	bxhdimension tmpdim;
	memcpy(&tmpdim, &writebdr.datarec->dimensions[0], sizeof(tmpdim));
	memmove(&writebdr.datarec->dimensions[0], &writebdr.datarec->dimensions[1], sizeof(writebdr.datarec->dimensions[1])*3);
	memcpy(&writebdr.datarec->dimensions[3], &tmpdim, sizeof(tmpdim));
    }

    fprintf(stderr, "Writing results...\n");
    /* create BXH file for outputs */
    writebdr.datarec->dimensions[3].size = numptsinresult;
    for (int dpstructind = 0;
	 dpstructind < writebdr.datarec->dimensions[3].numdpstructs;
	 dpstructind++) {
	bxhdimension * dimp = &writebdr.datarec->dimensions[3];
	bxhdatapoints * dpstructp = &dimp->dpstructs[dpstructind];
	unsigned int dpnum;
	for (dpnum = 0; dpnum < dpstructp->numvalues; dpnum++) {
	    free(dpstructp->values[dpnum]);
	    dpstructp->values[dpnum] = NULL;
	}
	free(dpstructp->values);
	dpstructp->values = NULL;
	if (dpstructp->label != NULL) {
	    free(dpstructp->label);
	    dpstructp->label = NULL;
	}
    }
    if (writebdr.datarec->dimensions[3].dpstructs != NULL) {
	free(writebdr.datarec->dimensions[3].dpstructs);
	writebdr.datarec->dimensions[3].dpstructs = NULL;
    }
    if (bxh_addAutoHistoryEntry(writebdr.docp, oldargv[0], (const char **)&oldargv[1], oldargc-1) != 0) {
	fprintf(stderr, "Error adding history entry\n");
	goto FAIL;
    }
    free(writebdr.datarec->elemtype);
    writebdr.datarec->elemtype = strdup("float32");
    for (indquery = 0; indquery < numqueries; indquery++) {
	size_t prefixlen = 0;
	size_t labellen = 0;

	prefixlen = strlen(outputprefix);
	labellen = strlen(opt_querylabels[indquery]);

	sprintf(outputbase, "%s_%s_avg", outputprefix, opt_querylabels[indquery]);
	if (write_out_niigz_bxh(outputbase, qres_avg[indquery], sizeof(float) * volsize * numptsinresult, &writebdr) != 0) goto FAIL;

	sprintf(outputbase, "%s_%s_std", outputprefix, opt_querylabels[indquery]);
	if (write_out_niigz_bxh(outputbase, qres_std[indquery], sizeof(float) * volsize * numptsinresult, &writebdr) != 0) goto FAIL;

	{
	    struct trialsummarylist * tslistp = &trialsummarylists[indquery];
	    float ** avgsummpp = qres_summary_avg[indquery];
	    float ** stdsummpp = qres_summary_std[indquery];
	    size_t indsummary;
	    for (indsummary = 0; indsummary < tslistp->numsummaries; indsummary++) {
		struct trialsummary * tsp = &tslistp->summaries[indsummary];
		size_t oldtsize = writebdr.datarec->dimensions[3].size;
		writebdr.datarec->dimensions[3].size = tsp->numgroups;

		sprintf(outputbase, "%s_%s_summary_%s_avg", outputprefix, opt_querylabels[indquery], tsp->indicesstr);
		if (write_out_niigz_bxh(outputbase, avgsummpp[indsummary], sizeof(float) * volsize * tsp->numgroups, &writebdr) != 0) goto FAIL;

		sprintf(outputbase, "%s_%s_summary_%s_std", outputprefix, opt_querylabels[indquery], tsp->indicesstr);
		if (write_out_niigz_bxh(outputbase, stdsummpp[indsummary], sizeof(float) * volsize * tsp->numgroups, &writebdr) != 0) goto FAIL;

		writebdr.datarec->dimensions[3].size = oldtsize;
	    }
	}
    }
    free(writebdr.datarec->elemtype);
    writebdr.datarec->elemtype = strdup("uint16");
    for (indquery = 0; indquery < numqueries; indquery++) {
	/* Write out n */
	unsigned short * newbuf = NULL;
	size_t prefixlen = 0;
	size_t labellen = 0;

	prefixlen = strlen(outputprefix);
	labellen = strlen(opt_querylabels[indquery]);

	sprintf(outputbase, "%s_%s_n", outputprefix, opt_querylabels[indquery]);
	newbuf = bxh_convertBufToUnsignedShort(qres_n[indquery], volsize * numptsinresult * sizeof(size_t), "size_t");
	if (write_out_niigz_bxh(outputbase, newbuf, sizeof(unsigned short) * volsize * numptsinresult, &writebdr) != 0) goto FAIL;
	free(newbuf);

	{
	    struct trialsummarylist * tslistp = &trialsummarylists[indquery];
	    size_t ** nsummpp = qres_summary_n[indquery];
	    size_t indsummary;
	    for (indsummary = 0; indsummary < tslistp->numsummaries; indsummary++) {
		struct trialsummary * tsp = &tslistp->summaries[indsummary];
		size_t oldtsize = writebdr.datarec->dimensions[3].size;
		writebdr.datarec->dimensions[3].size = tsp->numgroups;

		sprintf(outputbase, "%s_%s_summary_%s_n", outputprefix, opt_querylabels[indquery], tsp->indicesstr);
		newbuf = bxh_convertBufToUnsignedShort(nsummpp[indsummary], volsize * tsp->numgroups * sizeof(size_t), "size_t");
		if (write_out_niigz_bxh(outputbase, newbuf, sizeof(unsigned short) * volsize * tsp->numgroups, &writebdr) != 0) goto FAIL;
		free(newbuf);

		writebdr.datarec->dimensions[3].size = oldtsize;
	    }
	}
    }
    if (!opt_scalebl) {
	/* convert avg to percent of mean image, and fix std too */
	for (indquery = 0; indquery < numqueries; indquery++) {
	    size_t indxyz;
	    size_t indt;
	    float * avgptr = qres_avg[indquery];
	    float * stdptr = qres_std[indquery];
	    for(indt = 0; indt < numptsinresult; indt++) {
		for (indxyz = 0; indxyz < volsize; indxyz++) {
		    size_t fullind = indt*volsize + indxyz;
		    float bl = meanimage[indxyz];
		    float oldavg = avgptr[fullind];
		    float oldstd = stdptr[fullind];
		    if (bl == 0) {
			avgptr[fullind] = 0;
			stdptr[fullind] = 0;
		    } else {
			avgptr[fullind] = 100.0 * (oldavg / bl);
			stdptr[fullind] = 100.0 * (oldstd / bl);
		    }
		}
	    }
	    {
		struct trialsummarylist * tslistp = &trialsummarylists[indquery];
		float ** avgsummpp = qres_summary_avg[indquery];
		float ** stdsummpp = qres_summary_std[indquery];
		size_t indsummary;
		for (indsummary = 0; indsummary < tslistp->numsummaries; indsummary++) {
		    struct trialsummary * tsp = &tslistp->summaries[indsummary];
		    for(indt = 0; indt < tsp->numgroups; indt++) {
			for (indxyz = 0; indxyz < volsize; indxyz++) {
			    size_t fullind = indt*volsize + indxyz;
			    float bl = meanimage[indxyz];
			    float oldavg = avgsummpp[indsummary][fullind];
			    float oldstd = stdsummpp[indsummary][fullind];
			    if (bl == 0) {
				avgsummpp[indsummary][fullind] = 0;
				stdsummpp[indsummary][fullind] = 0;
			    } else {
				avgsummpp[indsummary][fullind] = 100.0 * (oldavg / bl);
				stdsummpp[indsummary][fullind] = 100.0 * (oldstd / bl);
			    }
			}
		    }
		}
	    }
	}
	free(writebdr.datarec->elemtype);
	writebdr.datarec->elemtype = strdup("float32");
	for (indquery = 0; indquery < numqueries; indquery++) {
	    size_t prefixlen = 0;
	    size_t labellen = 0;

	    prefixlen = strlen(outputprefix);
	    labellen = strlen(opt_querylabels[indquery]);

	    sprintf(outputbase, "%s_%s_avg_percent", outputprefix, opt_querylabels[indquery]);
	    if (write_out_niigz_bxh(outputbase, qres_avg[indquery], sizeof(float) * volsize * numptsinresult, &writebdr) != 0) goto FAIL;

	    sprintf(outputbase, "%s_%s_std_percent", outputprefix, opt_querylabels[indquery]);
	    if (write_out_niigz_bxh(outputbase, qres_std[indquery], sizeof(float) * volsize * numptsinresult, &writebdr) != 0) goto FAIL;

	    {
		struct trialsummarylist * tslistp = &trialsummarylists[indquery];
		float ** avgsummpp = qres_summary_avg[indquery];
		float ** stdsummpp = qres_summary_std[indquery];
		size_t indsummary;
		for (indsummary = 0; indsummary < tslistp->numsummaries; indsummary++) {
		    struct trialsummary * tsp = &tslistp->summaries[indsummary];
		    size_t oldtsize = writebdr.datarec->dimensions[3].size;
		    writebdr.datarec->dimensions[3].size = tsp->numgroups;

		    sprintf(outputbase, "%s_%s_summary_%s_avg_percent", outputprefix, opt_querylabels[indquery], tsp->indicesstr);
		    if (write_out_niigz_bxh(outputbase, avgsummpp[indsummary], sizeof(float) * volsize * tsp->numgroups, &writebdr) != 0) goto FAIL;

		    sprintf(outputbase, "%s_%s_summary_%s_std_percent", outputprefix, opt_querylabels[indquery], tsp->indicesstr);
		    if (write_out_niigz_bxh(outputbase, stdsummpp[indsummary], sizeof(float) * volsize * tsp->numgroups, &writebdr) != 0) goto FAIL;

		    writebdr.datarec->dimensions[3].size = oldtsize;
		}
	    }
	}
    }
    free(writebdr.datarec->elemtype);
    writebdr.datarec->elemtype = strdup("float32");
    {
	/* Write out baseline mean */
	sprintf(outputbase, "%s_baselineAvg", outputprefix);
	writebdr.datarec->numdims = 3;
	if (write_out_niigz_bxh(outputbase, qres_blavg, sizeof(float) * volsize, &writebdr) != 0) goto FAIL;
	writebdr.datarec->numdims = 4;
    }

    if (opt_tm) {
	free(writebdr.datarec->elemtype);
	writebdr.datarec->elemtype = strdup("float32");
	for (indquery = 0; indquery < numqueries; indquery++) {
	    /* Write out trialmax data */
	    size_t prefixlen = 0;
	    size_t labellen = 0;
	    size_t oldtsize = 0;

	    prefixlen = strlen(outputprefix);
	    labellen = strlen(opt_querylabels[indquery]);

	    sprintf(outputbase, "%s_%s_trialmax", outputprefix, opt_querylabels[indquery]);
 	    oldtsize = writebdr.datarec->dimensions[3].size;
	    writebdr.datarec->dimensions[3].size = numepochs[indquery];
	    if (write_out_niigz_bxh(outputbase, qres_tm[indquery], sizeof(float) * volsize * numepochs[indquery], &writebdr) != 0) goto FAIL;
	    writebdr.datarec->dimensions[3].size = oldtsize;
	}
    }
    {
	FILE * fp = NULL;

	strcpy(outputfilename, outputprefix);
	strcat(outputfilename, "_QUERIES.txt");
	
	if ((fp = fopen(outputfilename, "w")) == NULL) {
	    fprintf(stderr, "Error opening file %s\n", outputfilename);
	    fclose(fp);
	    goto FAIL;
	}
	for (indquery = 0; indquery < numqueries; indquery++) {
	    if (fprintf(fp, "%s: %s\n", opt_querylabels[indquery], opt_queries[indquery]) < 0) {
		fprintf(stderr, "Error writing to file %s\n", outputfilename);
		fclose(fp);
		goto FAIL;
	    }
	}
	fclose(fp);
    }

    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:

    time_end = time(NULL);
    fprintf(stderr, "End time: %s\n", ctime(&time_end));
    if (logfp)
	fprintf(logfp, "End time: %s\n", ctime(&time_end));

    if (logfp && logfp != stderr)
	fclose(logfp);

    for (argind = 2; argind < argc; argind += 2) {
	int imnum = (argind - 2) / 2;
	if (imnum == 0) {
	    /* we used bdrs[0] for writebdr, and subsequently modified writebdr, so free that instead */
	    bxh_datareaddata_free(&writebdr);
	} else {
	    bxh_datareaddata_free(&bdrs[imnum]);
	}
    }
    bxh_datareaddata_free(&maskbdr);

    for (indquery = 0; indquery < numqueries; indquery++) {
	if (opt_queries[indquery]) {
	    free(opt_queries[indquery]); opt_queries[indquery] = NULL;
	}
	if (opt_queryfilters[indquery]) {
	    free(opt_queryfilters[indquery]); opt_queryfilters[indquery] = NULL;
	}
	if (opt_queryepochexcludes[indquery]) {
	    free(opt_queryepochexcludes[indquery]); opt_queryepochexcludes[indquery] = NULL;
	}
	if (opt_querylabels[indquery]) {
	    free(opt_querylabels[indquery]); opt_querylabels[indquery] = NULL;
	}
	if (opt_trialsummary[indquery]) {
	    free(opt_trialsummary[indquery]); opt_trialsummary[indquery] = NULL;
	}
    }
    if (opt_maskfile) {
	free(opt_maskfile); opt_maskfile = NULL;
    }
    if (opt_optsfromfile) {
	free(opt_optsfromfile); opt_optsfromfile = NULL;
    }
    if (opt_querylang) {
	free(opt_querylang); opt_querylang = NULL;
    }
    if (opt_tmroifile) {
	free(opt_tmroifile); opt_tmroifile = NULL;
    }
    if (opt_tmseed) {
	free(opt_tmseed); opt_tmseed = NULL;
    }

    for (indquery = 0; indquery < numqueries; indquery++) {
	if (xpqueries[indquery])
	    free(xpqueries[indquery]);
	if (xpqueryfilters[indquery])
	    free(xpqueryfilters[indquery]);
	if (xpqueryepochexcludes[indquery])
	    free(xpqueryepochexcludes[indquery]);
	if (qres_avg[indquery])
	    free(qres_avg[indquery]);
	if (qres_std[indquery])
	    free(qres_std[indquery]);
	if (qres_n[indquery])
	    free(qres_n[indquery]);
    }
    for (indquery = 0; indquery < numqueries; indquery++) {
	size_t indsummary;
	struct trialsummarylist * tslistp = &trialsummarylists[indquery];
	for (indsummary = 0; indsummary < tslistp->numsummaries; indsummary++) {
	    size_t indgroup;
	    struct trialsummary * tsp = &tslistp->summaries[indsummary];
	    free(qres_summary_avg[indquery][indsummary]);
	    qres_summary_avg[indquery][indsummary] = NULL;
	    free(qres_summary_std[indquery][indsummary]);
	    qres_summary_std[indquery][indsummary] = NULL;
	    free(qres_summary_n[indquery][indsummary]);
	    qres_summary_n[indquery][indsummary] = NULL;

	    for (indgroup = 0; indgroup < tsp->numgroups; indgroup++) {
		struct trialsummarygroup * tsg = &tsp->groups[indgroup];
		free(tsg->points); tsg->points = NULL;
	    }

	    free(tsp->groups); tsp->groups = NULL;
	    free(tsp->indicesstr); tsp->indicesstr = NULL;
	}
	free(qres_summary_avg[indquery]);
	qres_summary_avg[indquery] = NULL;
	free(qres_summary_std[indquery]);
	qres_summary_std[indquery] = NULL;
	free(qres_summary_n[indquery]);
	qres_summary_n[indquery] = NULL;

	free(tslistp->summaries); tslistp->summaries = NULL;
    }
    if (qres_temp)
	free(qres_temp);
    if (qres_blavg)
	free(qres_blavg);
    if (qres_nbl)
	free(qres_nbl);

    if (meanimage)
	free(meanimage);

    free(outputfilename);
    free(outputbxhname);
    free(outputimgname);
    free(outputbase);

    free(numepochs);

    return retval;
}

#endif /* #ifndef HAVE_LIBGSL #else */

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.103  2008/04/21 16:58:44  gadde
 * Remove unused code.
 *
 * Revision 1.102  2007/12/10 16:39:24  gadde
 * Move Analyze/NIFTI generation to bxh_niftilib.c
 *
 * Revision 1.101  2007/12/05 18:29:17  gadde
 * Write out .nii.gz files (instead of just .img files) for most outputs.
 *
 * Revision 1.100  2007/05/11 15:47:22  gadde
 * For stddev, set to zero if n==1 (already did this for n==0).
 *
 * Revision 1.99  2007/04/02 19:12:33  gadde
 * Write percent signal change for summaries too.
 *
 * Revision 1.98  2007/03/20 17:44:18  gadde
 * Allow use of arbitrary reference volumes
 *
 * Revision 1.97  2007/02/12 21:19:04  gadde
 * use higher precision when printing out TR in error message
 *
 * Revision 1.96  2007/01/17 17:10:50  gadde
 * Change colons to dashes when writing files (Windows [and Mac?] filenames
 * will have problems with colons).
 *
 * Revision 1.95  2007/01/16 19:39:42  gadde
 * Make sure to write correct byte order in XML file.
 *
 * Revision 1.94  2007/01/16 19:02:32  gadde
 * Add --trialsummary option.
 *
 * Revision 1.93  2006/08/18 15:56:30  gadde
 * Be more careful incrementing iterators when erasing elements.
 *
 * Revision 1.92  2006/07/03 16:35:41  gadde
 * Don't fail in trial extraction if there are no query matches.
 *
 * Revision 1.91  2006/06/23 21:45:28  gadde
 * Fix byte order in output "trials" .bxh file.
 *
 * Revision 1.90  2006/06/23 21:19:01  gadde
 * Add --extracttrials option.
 *
 * Revision 1.89  2006/06/01 16:14:15  gadde
 * Don't send timing extraction output to stderr twice.
 *
 * Revision 1.88  2006/05/31 21:10:48  gadde
 * Several memory leak fixes.
 *
 * Revision 1.87  2006/05/31 19:11:24  gadde
 * Add timing info.
 *
 * Revision 1.86  2006/05/31 17:56:23  gadde
 * Fix --memorylimit bugs.
 *
 * Revision 1.85  2006/05/31 14:30:04  gadde
 * Add --memorylimit option.
 *
 * Revision 1.84  2006/05/05 14:54:38  gadde
 * use float instead of double for input data
 *
 * Revision 1.83  2006/05/04 18:18:54  gadde
 * Use float instead of double for reading input images.
 *
 * Revision 1.82  2006/04/14 18:22:09  gadde
 * Don't overwrite eventtiming file for each input argument!
 *
 * Revision 1.81  2006/04/14 18:14:03  gadde
 * Add --extracttimingonly option.
 *
 * Revision 1.80  2006/03/24 16:14:14  gadde
 * Don't use snprintf.
 *
 * Revision 1.79  2006/03/16 19:28:08  gadde
 * dimsize_t needs to change if files have different sizes for t dimension!
 *
 * Revision 1.78  2006/03/02 21:47:30  gadde
 * Add specification of explicit trialmax seed.
 *
 * Revision 1.77  2006/02/23 17:48:39  gadde
 * Add --trialmax and --trialmaxroi options.
 * Various code reorganization and variable renaming.
 *
 * Revision 1.76  2006/02/21 22:05:43  gadde
 * Move some things out of main() into functions.
 *
 * Revision 1.75  2006/02/09 17:01:34  gadde
 * Update error messages.
 *
 * Revision 1.74  2006/02/08 22:08:16  gadde
 * Add --scalebl option.
 *
 * Revision 1.73  2006/02/01 21:30:53  gadde
 * Add epochsecsafter/epochsecsbefore options.
 *
 * Revision 1.72  2005/09/20 18:37:55  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.71  2005/09/19 16:31:56  gadde
 * Documentation and help message updates.
 *
 * Revision 1.70  2005/09/13 14:59:30  gadde
 * Fix fprintf arguments.
 *
 * Revision 1.69  2005/08/16 19:47:36  gadde
 * Reset meanimage for each run.
 *
 * Revision 1.68  2005/08/09 13:49:24  gadde
 * Also write out avg/std as percent of mean image.
 *
 * Revision 1.67  2005/07/27 15:59:45  gadde
 * Make endpt specific to each input image (allows for inputs with different
 * numbers of timepoints).
 *
 * Revision 1.66  2005/07/27 14:35:45  gadde
 * Don't overwrite _std.img with std_percbase!
 * Also, #ifdef out the percbase calculation until it's ready.
 *
 * Revision 1.65  2005/07/15 20:26:13  gadde
 * avg is already shifted to baseline, so just divide for percbase calculation.
 *
 * Revision 1.64  2005/07/15 19:06:04  gadde
 * Write out avg and std as percent of baseline too.
 *
 * Revision 1.63  2005/07/08 16:50:52  gadde
 * Allocate enough memory for output filenames.
 *
 * Revision 1.62  2005/05/24 15:25:38  gadde
 * Don't complain if t dimension changes between runs.
 *
 * Revision 1.61  2005/04/29 20:12:54  gadde
 * Fix use of queryepochexcludes.
 *
 * Revision 1.60  2005/04/27 14:31:59  gadde
 * Don't send NULL queries to filter_events or exclude_epochs.
 *
 * Revision 1.59  2005/04/19 14:39:51  gadde
 * Don't add more than one history entry.
 *
 * Revision 1.58  2005/04/01 22:27:22  gadde
 * Updates to allow more than one event per timepoint.
 *
 * Revision 1.57  2005/03/28 20:48:36  gadde
 * Win32 updates
 *
 * Revision 1.56  2005/03/14 22:13:18  gadde
 * Remove origin from baseline/interp ranges.
 *
 * Revision 1.55  2005/03/03 19:09:49  gadde
 * Replace query language name 'new' with 'event'.
 *
 * Revision 1.54  2005/02/18 17:11:19  gadde
 * Fix variable re-initialization bug.
 *
 * Revision 1.53  2005/02/18 16:35:23  gadde
 * Move event stuff into separate source file (and add to library).
 *
 * Revision 1.52  2005/02/04 19:49:13  gadde
 * Print out how many events to be used for each query in analysis.
 *
 * Revision 1.51  2005/01/25 16:01:54  gadde
 * Fix indexing of txyz data dimensions.
 *
 * Revision 1.50  2005/01/24 22:38:56  gadde
 * Clarify an error message.
 *
 * Revision 1.49  2005/01/24 22:14:53  gadde
 * Brain mask fix.
 *
 * Revision 1.48  2005/01/21 21:52:15  gadde
 * Some very minor performance improvements.
 *
 * Revision 1.47  2005/01/20 22:29:08  gadde
 * Fix numptsinwork calculation.
 *
 * Revision 1.46  2005/01/20 21:54:12  gadde
 * Add --nointerp option
 *
 * Revision 1.45  2005/01/20 21:54:26  gadde
 * A number of fixes to allow for the possibility that baseline
 * epoch may be larger than event epoch.  Also allows us to use
 * interpolated data for calculating baseline data.
 *
 * Revision 1.44  2005/01/20 20:52:06  gadde
 * Add origin to baseline/interp ranges.
 *
 * Revision 1.43  2005/01/18 22:09:52  gadde
 * Write out _n.bxh files with uint16 elemtype.
 *
 * Revision 1.42  2005/01/18 21:32:51  gadde
 * Fix signed/unsigned warning.
 *
 * Revision 1.41  2005/01/18 20:51:03  gadde
 * Be smarter about printing ranges if epoch is out of range.
 *
 * Revision 1.40  2005/01/18 20:40:45  gadde
 * Write out qres_n as four dimensions.
 *
 * Revision 1.39  2005/01/18 16:31:00  gadde
 * Write out n as well.  Also, write unbiased stddev, rather than biased.
 *
 * Revision 1.38  2005/01/13 20:32:02  gadde
 * Add CVS id to log file.
 *
 * Revision 1.37  2005/01/13 20:26:51  gadde
 * Add --forcetr option.
 * Write out mean baseline image.
 * Re-order operations (don't read data until absolutely necessary).
 * Streamline timepoint exclusion code.
 * MAJOR: Don't zero out results before each image/event file
 * pair.  This bug was present from version 1.16.
 *
 * Revision 1.36  2005/01/07 20:44:16  gadde
 * Flush output before reading data.
 *
 * Revision 1.35  2005/01/07 19:38:57  gadde
 * Annotate an excessive verbosity as such.
 *
 * Revision 1.34  2005/01/07 16:59:17  gadde
 * Fix node merging bug.  Also make merged nodes prettier.
 * Don't allow multiple instances of the same query in a qvec.
 * Add ability to filter events using --queryfilter.
 * Fix option validation.
 * Fix command line printing for empty string arguments.
 *
 * Revision 1.33  2005/01/06 21:27:31  gadde
 * Label some functions "static".
 *
 * Revision 1.32  2005/01/06 21:15:00  gadde
 * Apply queries only to merged and sorted event list.
 * Apply queryepochexcludes to canonicalized event list.
 *
 * Revision 1.31  2005/01/06 20:08:40  gadde
 * Allow queries to be empty.
 * Verify that number of queryepochexcludes, if any exit, is same as
 * number of queries.
 *
 * Revision 1.30  2005/01/04 18:51:24  gadde
 * Add ability to read multiple event files per image file, and
 * to exclude epochs that include flagged events.
 *
 * Revision 1.29  2004/12/14 19:36:46  gadde
 * Move to bxh_dataReadStart/Finish style of reading.
 * Add option to use maskfile.
 * Also fix minor errors in writing command line to log file.
 *
 * Revision 1.28  2004/12/13 19:24:37  gadde
 * Fix help doc.
 *
 * Revision 1.27  2004/12/10 16:06:40  gadde
 * Accept any namespace for event elements.
 * Also, don't crash if no event elements match.
 *
 * Revision 1.26  2004/12/09 21:30:18  gadde
 * Update docs and replace "condition"s with "event"s.
 *
 * Revision 1.25  2004/12/09 21:24:41  gadde
 * Quote shell metacharacters in writing arguments to log.
 * Also add contents of file specified in --optsfromfile to log.
 *
 * Revision 1.24  2004/12/09 17:12:21  gadde
 * Update trigger for quoting args in log.
 *
 * Revision 1.23  2004/12/09 16:59:53  gadde
 * Quote appropriately when logging program arguments.
 *
 * Revision 1.22  2004/12/09 15:03:33  gadde
 * Go back to full-time-course splines for speed.
 * This (1.22) is a new reference implementation.
 *
 * Revision 1.21  2004/12/09 14:52:16  gadde
 * Make event list an array instead of iterating through the map.
 *
 * Revision 1.20  2004/12/08 21:50:19  gadde
 * Memory allocation fix.
 *
 * Revision 1.19  2004/12/08 21:44:40  gadde
 * Fixes for AIX.
 *
 * Revision 1.18  2004/12/08 21:09:54  gadde
 * Revert to piece-wise splines (per-interval) rather than full
 * time-course, to match data from version 1.11.
 *
 * Revision 1.17  2004/12/08 20:29:40  gadde
 * Change everything to txyz order for speed.
 *
 * Revision 1.16  2004/12/08 19:47:19  gadde
 * Read data in txyz order and represent query nums as simple struct
 * rather than STL vector.
 *
 * Revision 1.15  2004/12/08 18:11:48  gadde
 * Create splines over full time course, rather than piece-wise.
 *
 * Revision 1.14  2004/12/08 17:27:32  gadde
 * Interpolate using same number of points as version 1.11 (oops!).
 *
 * Revision 1.13  2004/12/08 14:17:43  gadde
 * Be a little smarter about memory allocation.
 *
 * Revision 1.12  2004/12/07 19:27:12  gadde
 * Rearrange computation to minimize redundant operations.
 *
 * Revision 1.11  2004/12/07 18:27:57  gadde
 * Make se_less::operator() const.
 *
 * Revision 1.10  2004/12/07 16:17:32  gadde
 * Save old arguments to log file.
 *
 * Revision 1.9  2004/12/07 15:55:23  gadde
 * Save old argc/argv for logging.
 *
 * Revision 1.8  2004/12/06 22:41:38  gadde
 * Several additions, including:
 *  query labels
 *  log file
 *  querying overlapping events
 *  bug fixes involving units
 *
 * Revision 1.7  2004/11/19 22:09:55  gadde
 * Don't overwrite output files by default.
 *
 * Revision 1.6  2004/11/18 15:38:39  gadde
 * Update help message.
 *
 * Revision 1.5  2004/11/17 21:53:31  gadde
 * Double-check spacing in 't' dimension, and fix diagnostic message.
 *
 * Revision 1.4  2004/11/17 19:57:24  gadde
 * Queries are now XPath predicates filtering the condition nodes.
 *
 * Revision 1.3  2004/11/15 14:41:40  gadde
 * Update usage info.
 *
 * Revision 1.2  2004/11/12 21:04:13  gadde
 * Fix help messages.
 *
 * Revision 1.1  2004/11/12 15:03:35  gadde
 * Initial commit.
 *
 *
 */
