static const char rcsid[] = "$Id: bxh_event2table.cpp,v 1.9 2005-09-20 18:37:55 gadde Exp $";

/*
 * bxh_event2table.cpp --
 * 
 *  Given event files and some queries, write a simple text event table.
 */

#include <bxh_config.h>

#include <stdio.h>

#ifdef WIN32
#define strcasecmp stricmp
#endif

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <math.h>

#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#include <string>
#include <map>
#include <vector>

#include "bxh_datarec.h"
#include "bxh_eventlib.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

#define VERBOSE 0

#undef FUNC
#define FUNC "get_event_list"
static
xmlDocPtr
get_event_list(std::vector<std::string> eventfilelist,
	       char * query, char * filterquery,
	       std::multimap<double, xmlNodePtr> & eventlist,
	       FILE * logfp)
{
    xmlDocPtr retval = NULL;
    const char * queryprefix = "//*[local-name()='events']/*[local-name()='event']";
    std::vector<xmlDocPtr> docs;
    xmlDocPtr newdoc = NULL;
    size_t indefile = 0;
    char * eventlabel = (char *)"EVENT";
    xmlNodePtr eventsnode = NULL;

    for (indefile = 0; indefile < eventfilelist.size(); indefile++) {
	xmlXPathContextPtr xpctxt = NULL;
	xmlXPathObjectPtr xpobj = NULL;
	xmlDocPtr doc = xmlParseFile(eventfilelist[indefile].c_str());
	if (doc == NULL) {
	    fprintf(stderr, FUNC ": could not parse file %s\n", eventfilelist[indefile].c_str());
	    goto FAIL;
	}
	docs.push_back(doc);
	if ((xpctxt = xmlXPathNewContext(doc)) == NULL) {
	    fprintf(stderr, FUNC ": unable to create new XPath context\n");
	    goto FAIL;
	}
	if ((xpobj = xmlXPathEvalExpression((xmlChar *)"//*[local-name()='events']/*[local-name()='event']", xpctxt)) != NULL &&
	    xpobj->nodesetval->nodeNr > 0) {
	    int nodenum;
	    int numnodes = xpobj->nodesetval->nodeNr;
	    for (nodenum = 0; nodenum < numnodes; nodenum++) {
		xmlNodePtr matched = xpobj->nodesetval->nodeTab[nodenum];
		xmlNodePtr newvalue = NULL;
		if ((newvalue = xmlNewChild(matched, NULL, (xmlChar *)"value", (xmlChar *)eventfilelist[indefile].c_str())) == NULL) {
		    fprintf(stderr, FUNC ": error creating new child\n");
		    xmlXPathFreeObject(xpobj); xpobj = NULL;
		    xmlXPathFreeContext(xpctxt); xpctxt = NULL;
		    goto FAIL;
		}
		if (xmlSetProp(newvalue, (xmlChar *)"name", (xmlChar *)"$eventfile") == NULL) {
		    fprintf(stderr, FUNC ": error replacing attribute\n");
		    xmlXPathFreeObject(xpobj); xpobj = NULL;
		    xmlXPathFreeContext(xpctxt); xpctxt = NULL;
		    goto FAIL;
		}
	    }
	}
	if (xpobj) {
	    xmlXPathFreeObject(xpobj); xpobj = NULL;
	}
	xmlXPathFreeContext(xpctxt); xpctxt = NULL;
    }

    if ((newdoc = xmlNewDoc((xmlChar *)"1.0")) == NULL) {
	fprintf(stderr,"Error: unable to create new XML document\n");
	goto FAIL;
    }
    if ((eventsnode = xmlNewChild((xmlNodePtr)newdoc, NULL, (xmlChar *)"events", NULL)) == NULL) {
	fprintf(stderr,"Error: unable to create XML element 'events'\n");
	goto FAIL;
    }
    

    eventlist.clear();

    for (indefile = 0; indefile < eventfilelist.size(); indefile++) {
	xmlDocPtr curdoc = docs[indefile];
	xmlDocPtr sortdoc = NULL;
	xmlDocPtr canondoc = NULL;
	std::map<double, qvec> eventmap;
	std::map<double, qvec>::iterator eventiter;

	if ((sortdoc = xmlCopyDoc(curdoc, 1)) == NULL) {
	    fprintf(stderr, "Error copying XML doc!\n");
	    goto FAIL;
	}
	if (sort_event_list(sortdoc, queryprefix, logfp) != 0) {
	    fprintf(stderr, "Error sorting events!\n");
	    goto FAIL;
	}
	if ((canondoc = xmlCopyDoc(sortdoc, 1)) == NULL) {
	    fprintf(stderr, "Error copying XML doc!\n");
	    goto FAIL;
	}
	if (canonicalize_event_list(canondoc, queryprefix, logfp) != 0) {
	    fprintf(stderr, "Error canonicalizing events!\n");
	    goto FAIL;
	}

#if VERBOSE
	xmlDocDump(logfp, canondoc);
#endif

	if (match_events(sortdoc, 1, &query, &eventlabel, eventmap, logfp) != 0) {
	    goto FAIL;
	}
	if (filter_events(canondoc, 1, &filterquery, &eventlabel, eventmap, logfp) != 0) {
	    goto FAIL;
	}
	
	eventiter = eventmap.begin();
	while (eventiter != eventmap.end()) {
#if VERBOSE
	    fprintf(stderr, "Adding event %g to list\n", (*eventiter).first);
#endif
	    std::vector<xmlNodePtr>::iterator nodeiter;
	    for (nodeiter = (*eventiter).second[0].qnodes.begin();
		 nodeiter != (*eventiter).second[0].qnodes.end();
		 nodeiter++) {
		xmlNodePtr nodeptr = *nodeiter;
		xmlUnlinkNode(nodeptr);
		if ((nodeptr = xmlAddChild(eventsnode, nodeptr)) == NULL) {
		    fprintf(stderr, "Error adding child.\n");
		    xmlFreeDoc(canondoc);
		    xmlFreeDoc(sortdoc);
		    goto FAIL;
		}
		eventlist.insert(std::pair<double,xmlNodePtr>((*eventiter).first, nodeptr));
	    }
	    eventiter++;
	}

	xmlFreeDoc(canondoc);
	xmlFreeDoc(sortdoc);
    }

    {
	size_t docind;
	for (docind = 0; docind < docs.size(); docind++) {
	    if (docs[docind]) {
		xmlFreeDoc(docs[docind]);
		docs[docind] = NULL;
	    }
	}
    }

    goto EXIT;
    
  FAIL:
    eventlist.clear();
    retval = NULL;
    
  EXIT:
    {
	size_t docind;
	for (docind = 0; docind < docs.size(); docind++) {
	    if (docs[docind])
		xmlFreeDoc(docs[docind]);
	}
    }
    retval = newdoc;
    return retval;
}

#undef FUNC
#define FUNC "main"
int
main(int argc, char *argv[])
{
    int retval = 0;
    int argind;
    FILE * logfp = NULL;
    xmlDocPtr sortdoc = NULL;
    
    char * opt_optsfromfile = NULL;
    char * opt_query = NULL;
    char * opt_filterquery = NULL;
    const char * opt_querylang = "XPath";
    const char * opt_colsep = "\t";
    int opt_version = 0;

    char * xpquery = NULL;
    char * xpfilterquery = NULL;

    std::multimap<double, xmlNodePtr> eventlist;

    std::vector<std::string> eventfilelist;

    const int numopts = 8;
    opt_data opts[8] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  bxh_event2table [opts] eventfiles...\n\n"
	  "This program takes XML event files as input, selects events "
	  "(given user-specified queries), and writes a table of these "
	  "events and associated metadata to standard output.  "
	  "Each row is one event, and each column represents a different "
	  "metadata element (like onset, duration, and other values "
	  "specified in the events file)." },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_optsfromfile, 1, "optsfromfile",
	  "Program options (i.e. those starting with '--') will come from "
	  "this file.  "
	  "If this option is specified, then the options in the file "
	  "will be applied after all command-line options.  "
	  "The options (and their arguments) should be specified "
	  "one per line, with the leading '--' omitted." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_querylang, 1, "querylanguage",
	  "The language used for all queries.  Valid values are 'XPath' and "
	  "'event'.  Case is irrelevant.  Default is "
	  "'XPath'." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_query, 1, "query",
	  "A query string to match events.  "
	  "This option is required." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_filterquery, 1, "filterquery",
	  "A query string to filter matched events." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_colsep, 1, "colsep",
	  "String to separate columns (default is tab)." }
    };

    argc -= opt_parse(argc, argv, numopts, &opts[0], 0);
    if (opt_optsfromfile) {
	opt_parsefile(opt_optsfromfile, numopts, &opts[0], 0);
    }

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (opt_query == NULL) {
	fprintf(stderr, "Query is missing.\nUse the --help option for more help.\n");
	goto FAIL;
    }
    
    if (argc < 2) {
	fprintf(stderr, "Usage: %s [opts] inputeventfiles...\n", argv[0]);
	fprintf(stderr, "Not enough arguments.  Use the --help option for more help.\n");
	goto FAIL;
    }

    if (strcasecmp(opt_querylang, "XPath") != 0 &&
	strcasecmp(opt_querylang, "event") != 0) {
	fprintf(stderr, "Query language %s not recognized!\n", opt_querylang);
	goto FAIL;
    }

    {
	char ** queryplist[2] = { &opt_query, &opt_filterquery };
	char ** xpqueryplist[2] = { &xpquery, &xpfilterquery };
	int tmpind = 0;
	for (tmpind = 0; tmpind < 2; tmpind++) {
	    char * newquery = NULL;
	    char ** curqueryp = queryplist[tmpind];
	    if (*curqueryp == NULL || **curqueryp == '\0') {
		/* empty queryfilters match everything */
		newquery = strdup("true()");
	    } else if (strcasecmp(opt_querylang, "event") == 0 &&
		       (newquery = query2xpath(*curqueryp)) == NULL) {
		fprintf(stderr, "Bad query '%s'!\n", *curqueryp);
		goto FAIL;
	    }
	    if (newquery == NULL) {
		newquery = strdup(*curqueryp);
	    } else {
		fprintf(stderr, "query '%s' converted to XPath '%s'\n", *curqueryp, newquery);
	    }
	    *(xpqueryplist[tmpind]) = newquery;
	}
    }

    for (argind = 1; argind < argc; argind++) {
	eventfilelist.push_back(std::string(argv[argind]));
    }
    if ((sortdoc = get_event_list(eventfilelist,
				  xpquery, xpfilterquery,
				  eventlist,
				  stderr)) == NULL) {
	fprintf(stderr, "%s: Error getting event list!\n", argv[0]);
	goto FAIL;
    }

    {
	std::multimap<double, std::vector<std::string> > eventparams;
	std::vector<std::string> paramnames;
	std::multimap<double, xmlNodePtr>::iterator eventiter;
	xmlXPathContextPtr xpctxt = NULL;
	xmlXPathObjectPtr xpobj = NULL;
	for (eventiter = eventlist.begin();
	     eventiter != eventlist.end();
	     eventiter++) {
	    std::vector<std::string> newparamlist;
	    double onset = (*eventiter).first;
	    xmlNodePtr nodep = (*eventiter).second;
	    xmlNodePtr matchednodep = NULL;
	    char * addparamname = NULL;
	    char * addparamvalue = NULL;
	    unsigned int paramind;
	    if ((xpctxt = xmlXPathNewContext(nodep->doc)) == NULL) {
		fprintf(stderr,"Error: unable to create new XPath context\n");
		goto FAIL;
	    }
	    xpctxt->node = nodep;
	    if ((xpobj = xmlXPathEvalExpression((xmlChar *)"*[local-name()='onset']", xpctxt)) != NULL &&
		xpobj->nodesetval->nodeNr > 0) {
		matchednodep = xpobj->nodesetval->nodeTab[0];
		addparamname = (char *)"$onset";
		addparamvalue = (char *)xmlNodeGetContent(matchednodep);
		for (paramind = 0; paramind < paramnames.size(); paramind ++) {
		    if (paramnames[paramind] == addparamname)
			break;
		}
		if (paramind == paramnames.size()) {
		    paramnames.push_back(std::string(addparamname));
		    paramind = paramnames.size() - 1;
		}
		if (paramind >= newparamlist.size()) {
		    newparamlist.resize(paramind+1);
		}
		newparamlist[paramind] = std::string(addparamvalue);
		free(addparamvalue); addparamvalue = NULL;
		xmlXPathFreeObject(xpobj); xpobj = NULL;
	    }
	    if ((xpobj = xmlXPathEvalExpression((xmlChar *)"*[local-name()='duration']", xpctxt)) != NULL &&
		xpobj->nodesetval->nodeNr > 0) {
		matchednodep = xpobj->nodesetval->nodeTab[0];
		addparamname = (char *)"$duration";
		addparamvalue = (char *)xmlNodeGetContent(matchednodep);
		for (paramind = 0; paramind < paramnames.size(); paramind ++) {
		    if (paramnames[paramind] == addparamname)
			break;
		}
		if (paramind == paramnames.size()) {
		    paramnames.push_back(std::string(addparamname));
		    paramind = paramnames.size() - 1;
		}
		if (paramind >= newparamlist.size()) {
		    newparamlist.resize(paramind+1);
		}
		newparamlist[paramind] = std::string(addparamvalue);
		free(addparamvalue); addparamvalue = NULL;
		xmlXPathFreeObject(xpobj); xpobj = NULL;
	    }
	    if ((xpobj = xmlXPathEvalExpression((xmlChar *)"*[local-name()='value']", xpctxt)) != NULL &&
		xpobj->nodesetval->nodeNr > 0) {
		xmlXPathContextPtr xpctxt2 = NULL;
		xmlXPathObjectPtr xpobj2 = NULL;
		int nodenum = 0;
		for (nodenum = 0; nodenum < xpobj->nodesetval->nodeNr; nodenum++) {
		    matchednodep = xpobj->nodesetval->nodeTab[nodenum];
		    if ((xpctxt2 = xmlXPathNewContext(matchednodep->doc)) == NULL) {
			fprintf(stderr,"Error: unable to create new XPath context\n");
			goto FAIL;
		    }
		    xpctxt2->node = matchednodep;
		    if ((xpobj2 = xmlXPathEvalExpression((xmlChar *)"@name", xpctxt2)) != NULL) {
			addparamname = (char *)xmlNodeGetContent(xpobj2->nodesetval->nodeTab[0]);
			addparamvalue = (char *)xmlNodeGetContent(matchednodep);
			for (paramind = 0; paramind < paramnames.size(); paramind ++) {
			    if (paramnames[paramind] == addparamname)
				break;
			}
			if (paramind == paramnames.size()) {
			    paramnames.push_back(std::string(addparamname));
			    paramind = paramnames.size() - 1;
			}
			if (paramind >= newparamlist.size()) {
			    newparamlist.resize(paramind+1);
			}
			newparamlist[paramind] = std::string(addparamvalue);
			free(addparamname); addparamname = NULL;
			free(addparamvalue); addparamvalue = NULL;
			xmlXPathFreeObject(xpobj2); xpobj2 = NULL;
		    }
		    xmlXPathFreeContext(xpctxt2); xpctxt2 = NULL;
		}
		xmlXPathFreeObject(xpobj); xpobj = NULL;
	    }
	    xmlXPathFreeContext(xpctxt); xpctxt = NULL;
	    eventparams.insert(std::pair<double,std::vector<std::string> >(onset, newparamlist));
	}
	{
	    std::multimap<double, std::vector<std::string> >::iterator epiter;
	    unsigned int paramnum;
	    for (paramnum = 0; paramnum < paramnames.size(); paramnum++) {
		if (paramnum > 0) {
		    fprintf(stdout, "%s", opt_colsep);
		}
		fprintf(stdout, "%s", paramnames[paramnum].c_str());
	    }
	    fprintf(stdout, "\n");
	    for (epiter = eventparams.begin(); epiter != eventparams.end(); epiter++) {
		std::vector<std::string> & paramlist = (*epiter).second;
		for (paramnum = 0; paramnum < paramlist.size(); paramnum++) {
		    if (paramnum > 0) {
			fprintf(stdout, "%s", opt_colsep);
		    }
		    fprintf(stdout, "%s", paramlist[paramnum].c_str());
		}
		for (/*null*/; paramnum < paramnames.size(); paramnum++) {
		    if (paramnum > 0) {
			fprintf(stdout, "%s", opt_colsep);
		    }
		}
		fprintf(stdout, "\n");
	    }
	}
    }

    goto EXIT;

  FAIL:
    retval = -1;

  EXIT:

    if (logfp)
	fclose(logfp);

    if (xpquery)
	free(xpquery);
    if (xpfilterquery)
	free(xpfilterquery);

    if (sortdoc)
	xmlFreeDoc(sortdoc);

    return retval;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.8  2005/09/19 16:31:56  gadde
 * Documentation and help message updates.
 *
 * Revision 1.7  2005/09/09 20:06:11  gadde
 * Don't merge unrelated events files!  Events that had the same onsets
 * in separate files will all remain now, and the $eventfile column should
 * now be correct.
 *
 * Revision 1.6  2005/07/12 17:26:55  gadde
 * Don't send NULL for empty filterquery to filter_events.
 *
 * Revision 1.5  2005/04/18 16:28:09  gadde
 * Memory fixes.
 *
 * Revision 1.4  2005/04/05 16:47:45  gadde
 * Simplify the addition of filenames to events.
 *
 * Revision 1.3  2005/04/04 21:39:13  gadde
 * Add filename to the values printed out.
 *
 * Revision 1.2  2005/04/04 13:58:38  gadde
 * Fix missing index.
 *
 * Revision 1.1  2005/04/01 22:28:19  gadde
 * Add bxh_event2table
 *
 *
 */
