static const char rcsid[] = "$Id: bxhabsorb.c,v 1.39 2009-01-15 20:55:19 gadde Exp $";

/*
 * bxhabsorb.c --
 * 
 * Creates a BXH file to encapsulate/describe given data.
 */

#include <bxh_config.h>
#include <bxh_utils.h>

#include <assert.h>
#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#ifndef WIN32
#include <strings.h>
#endif
#ifndef WIN32
#include <unistd.h>
#endif

#include <zlib.h>

#include "opts.h"

#ifdef WIN32
#define strcasecmp stricmp
#endif

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

static char * opt_fromtypestr = NULL;
static char * opt_inputsfromfile = NULL;
static int opt_version = 0;
#ifdef HAVE_LIBXSLT
static int opt_xcede = 0;
extern int bxh_doXCEDE;
#ifdef HAVE_LIBEXSLT
static int opt_xcede2 = 0;
extern int bxh_doXCEDE2;
#endif
#endif
size_t opt_hintsize[4] = { (size_t)-1, (size_t)-1, (size_t)-1, (size_t)-1 };
double opt_hintorigin[4];
double opt_hintspacing[4];
double opt_hintgap[4];
char * opt_forceorient = NULL;
static int gnumopts = 0;
static opt_data * gopts = NULL;

typedef int mainfn(int argc, char *argv[]);
extern mainfn main_bxhabsorb;
extern mainfn main_pfile2bxh;
extern mainfn main_signafive2bxh;
#ifndef WIN32
extern mainfn main_iowasignafive2bxh;
#endif
extern mainfn main_ximg2bxh;
extern mainfn main_analyze2bxh;
extern mainfn main_afni2bxh;
#ifdef HAVE_DCMTK
extern mainfn main_dicom2bxh;
#endif
#ifdef HAVE_MINC
extern mainfn main_minc2bxh;
#endif
extern mainfn main_mgh2bxh;
extern mainfn main_bxh2bxh;

typedef int addoptsfn(opt_data * optsin[], int numoptsin);
extern addoptsfn addopts_pfile2bxh;
#ifndef WIN32
extern addoptsfn addopts_iowasignafive2bxh;
#endif
extern addoptsfn addopts_signafive2bxh;
extern addoptsfn addopts_ximg2bxh;
extern addoptsfn addopts_analyze2bxh;
extern addoptsfn addopts_afni2bxh;
#ifdef HAVE_DCMTK
extern addoptsfn addopts_dicom2bxh;
#endif
#ifdef HAVE_MINC
extern addoptsfn addopts_minc2bxh;
#endif
extern addoptsfn addopts_mgh2bxh;
extern addoptsfn addopts_nrrd2bxh;

extern addoptsfn addopts_bxhabsorb_pfile2bxh;
#ifndef WIN32
extern addoptsfn addopts_bxhabsorb_iowasignafive2bxh;
#endif
extern addoptsfn addopts_bxhabsorb_signafive2bxh;
extern addoptsfn addopts_bxhabsorb_ximg2bxh;
extern addoptsfn addopts_bxhabsorb_analyze2bxh;
extern addoptsfn addopts_bxhabsorb_afni2bxh;
#ifdef HAVE_DCMTK
extern addoptsfn addopts_bxhabsorb_dicom2bxh;
#endif
#ifdef HAVE_MINC
extern addoptsfn addopts_bxhabsorb_minc2bxh;
#endif
extern addoptsfn addopts_bxhabsorb_mgh2bxh;
extern addoptsfn addopts_bxhabsorb_bxh2bxh;
extern addoptsfn addopts_bxhabsorb_nrrd2bxh;

int
addopts_bxhabsorb_xcede(opt_data * optsin[], int numoptsin)
{
    const int numopts = 1;
    opt_data opts[1] = {
#ifdef HAVE_LIBXSLT
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_xcede, 1, "xcede",
	  "Write XCEDE-style XML files." }
#else
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", NULL }
#endif
    };
    *optsin = (opt_data *)realloc(*optsin, sizeof(opt_data)*(numoptsin+numopts));
    memcpy(&(*optsin)[numoptsin], &opts[0], sizeof(opt_data)*numopts);
    return numopts;
}

int
addopts_bxhabsorb_xcede2(opt_data * optsin[], int numoptsin)
{
    const int numopts = 1;
    opt_data opts[1] = {
#ifdef HAVE_LIBEXSLT
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_xcede2, 1, "xcede2",
	  "Write XCEDE 2-style XML files." }
#else
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", NULL }
#endif
    };
    *optsin = (opt_data *)realloc(*optsin, sizeof(opt_data)*(numoptsin+numopts));
    memcpy(&(*optsin)[numoptsin], &opts[0], sizeof(opt_data)*numopts);
    return numopts;
}

int
addopts_bxhabsorb_common(opt_data * optsin[], int numoptsin)
{
    int numopts = 19;
    opt_data opts[19] = {
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_inputsfromfile, 1, "inputsfromfile",
	  "Read list of input files from this file." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_SIZE, &opt_hintsize[0], 1, "hintsizex", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_SIZE, &opt_hintsize[1], 1, "hintsizey", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_SIZE, &opt_hintsize[2], 1, "hintsizez", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_SIZE, &opt_hintsize[3], 1, "hintsizet", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_hintorigin[0], 1, "hintoriginx", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_hintorigin[1], 1, "hintoriginy", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_hintorigin[2], 1, "hintoriginz", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_hintorigin[3], 1, "hintorigint", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_hintspacing[0], 1, "hintspacingx", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_hintspacing[1], 1, "hintspacingy", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_hintspacing[2], 1, "hintspacingz", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_hintspacing[3], 1, "hintspacingt", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_hintgap[0], 1, "hintgapx", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_hintgap[1], 1, "hintgapy", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_hintgap[2], 1, "hintgapz", NULL },
	{ OPT_FLAGS_FULL, OPT_VAL_DOUBLE, &opt_hintgap[3], 1, "hintgapt",
	  "These options will provide a hint to the program of the 'size', "
	  "'origin', 'spacing', or 'gap' of the specified dimension.  Some "
	  "image types will not use all these values.  In particular, sizex "
	  "and sizey are assumed correct in most image headers, but they, as "
	  "well as sizez and sizet options may be useful with image type "
	  "'pfile'.  "
	  "Origin and spacing hints will be used by most image types.  " },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_forceorient, 1, "forceorientation",
	  "This option will force the labeled orientation of the image to "
	  "match the given three letter orientation code.  Each letter must "
	  "come from the following groups in any order: R(ight) or L(eft); "
	  "A(nterior) or P(osterior); S(uperior) or I(nferior).  Only one "
	  "letter from each group is allowed." }
    };
    *optsin = (opt_data *)realloc(*optsin, sizeof(opt_data)*(numoptsin+numopts));
    memcpy(&(*optsin)[numoptsin], &opts[0], sizeof(opt_data)*numopts);
    numopts += addopts_bxhabsorb_xcede(optsin, numoptsin+numopts);
    numopts += addopts_bxhabsorb_xcede2(optsin, numoptsin+numopts);
    return numopts;
}

int
addopts_bxhabsorb_specific(opt_data * optsin[], int numoptsin)
{
    const int numopts = 2;
    opt_data opts[2] = {
	{ 0x0, OPT_VAL_NONE, NULL, 0, "",
	  "Usage:\n"
	  "  bxhabsorb [ --fromtype type ] [ type-specific-opts... ] inputfiles... bxhoutputfile\n"
	  "  bxhabsorb [ --fromtype type ] [ --inputsfromfile inputlistfile ] [ type-specific-opts...] bxhoutputfile\n"
	  "\n"
	  "This program creates a BXH or XCEDE wrapper for the specified "
	  "input images.  If the --fromtype option is not specified, it "
	  "attempts to auto-detect the format of the input image files.  "
	  "If it cannot auto-detect the format, it will exit with an error." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_fromtypestr, 1, "fromtype",
	  "Type of the input data (pfile, signa5, signafive, "
#ifndef WIN32
	  "iowa-signa5, iowa-signafive, "
#endif
	  "ximg, analyze, afni, nrrd"
#ifdef HAVE_DCMTK
	  ", dicom"
#endif
#ifdef HAVE_MINC
	  ", minc"
#endif
	  ").  If this option is not specified, attempt to autodetect "
	  "format of inputfiles." }
    };
    *optsin = (opt_data *)realloc(*optsin, sizeof(opt_data)*(numoptsin+numopts));
    memcpy(&(*optsin)[numoptsin], &opts[0], sizeof(opt_data)*numopts);
    return numopts;
}

int
main_bxhabsorb(int argc, char *argv[])
{
    int retval = 0;
    int fromtype = -1;
    int numoptargs;
    char * progfile = NULL;
    int addargc = 0;
    char **addargv = NULL;
    char **newargv = NULL;

    opt_hintsize[0] = opt_hintsize[1] =
	opt_hintsize[2] = opt_hintsize[3] = (size_t)-1;
    opt_hintorigin[0] = opt_hintorigin[1] =
	opt_hintorigin[2] = opt_hintorigin[3] = HUGE_VAL;
    opt_hintspacing[0] = opt_hintspacing[1] =
	opt_hintspacing[2] = opt_hintspacing[3] = HUGE_VAL;
    opt_hintgap[0] = opt_hintgap[1] =
	opt_hintgap[2] = opt_hintgap[3] = HUGE_VAL;

#ifdef WIN32
    if ((progfile = strrchr(argv[0], '\\')) != NULL) {
	progfile++;
    } else if ((progfile = strrchr(argv[0], ':')) != NULL) {
	progfile++;
    } else {
	progfile = argv[0];
    }
#define EXE ".exe"
#else
    if ((progfile = strrchr(argv[0], '/')) != NULL) {
	progfile++;
    } else {
	progfile = argv[0];
    }
#define EXE ""
#endif

    if (strcmp(progfile, "pfile2bxh" EXE) == 0) {
	fromtype = FROM_PFILE;
    } else if (strcmp(progfile, "signafive2bxh" EXE) == 0) {
	fromtype = FROM_SIGNA5;
#ifndef WIN32
    } else if (strcmp(progfile, "iowa-signafive2bxh" EXE) == 0) {
	fromtype = FROM_IOWASIGNA5;
#endif
    } else if (strcmp(progfile, "ximg2bxh" EXE) == 0) {
	fromtype = FROM_XIMG;
    } else if (strcmp(progfile, "analyze2bxh" EXE) == 0) {
	fromtype = FROM_ANALYZE;
    } else if (strcmp(progfile, "afni2bxh" EXE) == 0) {
	fromtype = FROM_AFNI;
    } else if (strcmp(progfile, "nrrd2bxh" EXE) == 0) {
	fromtype = FROM_NRRD;
    } else if (strcmp(progfile, "dicom2bxh" EXE) == 0) {
	fromtype = FROM_DICOM;
    } else if (strcmp(progfile, "minc2bxh" EXE) == 0) {
	fromtype = FROM_MINC;
    } else if (strcmp(progfile, "mgh2bxh" EXE) == 0) {
	fromtype = FROM_MGH;
    } else if (strcmp(progfile, "xcede2bxh" EXE) == 0) {
	fromtype = FROM_BXH;
#ifdef HAVE_LIBXSLT
    } else if (strcmp(progfile, "pfile2xcede" EXE) == 0) {
	fromtype = FROM_PFILE;
	bxh_doXCEDE = 1;
    } else if (strcmp(progfile, "signafive2xcede" EXE) == 0) {
	fromtype = FROM_SIGNA5;
	bxh_doXCEDE = 1;
#ifndef WIN32
    } else if (strcmp(progfile, "iowa-signafive2xcede" EXE) == 0) {
	fromtype = FROM_IOWASIGNA5;
	bxh_doXCEDE = 1;
#endif
    } else if (strcmp(progfile, "ximg2xcede" EXE) == 0) {
	fromtype = FROM_XIMG;
	bxh_doXCEDE = 1;
    } else if (strcmp(progfile, "analyze2xcede" EXE) == 0) {
	fromtype = FROM_ANALYZE;
	bxh_doXCEDE = 1;
    } else if (strcmp(progfile, "afni2xcede" EXE) == 0) {
	fromtype = FROM_AFNI;
	bxh_doXCEDE = 1;
    } else if (strcmp(progfile, "nrrd2xcede" EXE) == 0) {
	fromtype = FROM_NRRD;
	bxh_doXCEDE = 1;
    } else if (strcmp(progfile, "dicom2xcede" EXE) == 0) {
	fromtype = FROM_DICOM;
	bxh_doXCEDE = 1;
    } else if (strcmp(progfile, "minc2xcede" EXE) == 0) {
	fromtype = FROM_MINC;
	bxh_doXCEDE = 1;
    } else if (strcmp(progfile, "mgh2xcede" EXE) == 0) {
	fromtype = FROM_MGH;
	bxh_doXCEDE = 1;
    } else if (strcmp(progfile, "bxh2xcede" EXE) == 0) {
	fromtype = FROM_BXH;
	bxh_doXCEDE = 1;
    } else if (strcmp(progfile, "pfile2xcede2" EXE) == 0) {
	fromtype = FROM_PFILE;
	bxh_doXCEDE2 = 1;
    } else if (strcmp(progfile, "signafive2xcede2" EXE) == 0) {
	fromtype = FROM_SIGNA5;
	bxh_doXCEDE2 = 1;
#ifndef WIN32
    } else if (strcmp(progfile, "iowa-signafive2xcede2" EXE) == 0) {
	fromtype = FROM_IOWASIGNA5;
	bxh_doXCEDE2 = 1;
#endif
    } else if (strcmp(progfile, "ximg2xcede2" EXE) == 0) {
	fromtype = FROM_XIMG;
	bxh_doXCEDE2 = 1;
    } else if (strcmp(progfile, "analyze2xcede2" EXE) == 0) {
	fromtype = FROM_ANALYZE;
	bxh_doXCEDE2 = 1;
    } else if (strcmp(progfile, "afni2xcede2" EXE) == 0) {
	fromtype = FROM_AFNI;
	bxh_doXCEDE2 = 1;
    } else if (strcmp(progfile, "nrrd2xcede2" EXE) == 0) {
	fromtype = FROM_NRRD;
	bxh_doXCEDE2 = 1;
    } else if (strcmp(progfile, "dicom2xcede2" EXE) == 0) {
	fromtype = FROM_DICOM;
	bxh_doXCEDE2 = 1;
    } else if (strcmp(progfile, "minc2xcede2" EXE) == 0) {
	fromtype = FROM_MINC;
	bxh_doXCEDE2 = 1;
    } else if (strcmp(progfile, "mgh2xcede2" EXE) == 0) {
	fromtype = FROM_MGH;
	bxh_doXCEDE2 = 1;
    } else if (strcmp(progfile, "bxh2xcede2" EXE) == 0) {
	fromtype = FROM_BXH;
	bxh_doXCEDE2 = 1;
#endif /* #ifdef HAVE_LIBXSLT */
    }

    bxh_check_supported_type(fromtype);

    /* add options if program was run as <TYPE>2bxh */
    if (fromtype == FROM_PFILE)
	gnumopts += addopts_pfile2bxh(&gopts, gnumopts);
    if (fromtype == FROM_SIGNA5)
	gnumopts += addopts_signafive2bxh(&gopts, gnumopts);
#ifndef WIN32
    if (fromtype == FROM_IOWASIGNA5)
	gnumopts += addopts_iowasignafive2bxh(&gopts, gnumopts);
#endif
    if (fromtype == FROM_XIMG)
	gnumopts += addopts_ximg2bxh(&gopts, gnumopts);
    if (fromtype == FROM_ANALYZE)
	gnumopts += addopts_analyze2bxh(&gopts, gnumopts);
    if (fromtype == FROM_AFNI)
	gnumopts += addopts_afni2bxh(&gopts, gnumopts);
    if (fromtype == FROM_NRRD)
	gnumopts += addopts_nrrd2bxh(&gopts, gnumopts);
#ifdef HAVE_MINC
    if (fromtype == FROM_MINC)
	gnumopts += addopts_minc2bxh(&gopts, gnumopts);
#endif
#ifdef HAVE_DCMTK
    if (fromtype == FROM_DICOM)
	gnumopts += addopts_dicom2bxh(&gopts, gnumopts);
#endif
    if (fromtype == FROM_MGH)
	gnumopts += addopts_mgh2bxh(&gopts, gnumopts);
    if (fromtype == FROM_BXH)
	gnumopts += addopts_bxh2bxh(&gopts, gnumopts);

    /* add options if program was not run as <TYPE>2bxh */
    if (fromtype == -1) {
	gnumopts += addopts_bxhabsorb_specific(&gopts, gnumopts);
	gnumopts += addopts_bxhabsorb_common(&gopts, gnumopts);
	gnumopts += addopts_bxhabsorb_pfile2bxh(&gopts, gnumopts);
	gnumopts += addopts_bxhabsorb_signafive2bxh(&gopts, gnumopts);
#ifndef WIN32
	gnumopts += addopts_bxhabsorb_iowasignafive2bxh(&gopts, gnumopts);
#endif
	gnumopts += addopts_bxhabsorb_ximg2bxh(&gopts, gnumopts);
	gnumopts += addopts_bxhabsorb_analyze2bxh(&gopts, gnumopts);
	gnumopts += addopts_bxhabsorb_afni2bxh(&gopts, gnumopts);
	gnumopts += addopts_bxhabsorb_nrrd2bxh(&gopts, gnumopts);
#ifdef HAVE_MINC
	gnumopts += addopts_bxhabsorb_minc2bxh(&gopts, gnumopts);
#endif
#ifdef HAVE_DCMTK
	gnumopts += addopts_bxhabsorb_dicom2bxh(&gopts, gnumopts);
#endif
	gnumopts += addopts_bxhabsorb_mgh2bxh(&gopts, gnumopts);
	gnumopts += addopts_bxhabsorb_bxh2bxh(&gopts, gnumopts);
#ifndef WIN32
    } else if (fromtype == FROM_IOWASIGNA5) {
	/* signafive2bxh */
	gnumopts += addopts_bxhabsorb_xcede(&gopts, gnumopts);
	gnumopts += addopts_bxhabsorb_xcede2(&gopts, gnumopts);
#endif
    } else {
	gnumopts += addopts_bxhabsorb_common(&gopts, gnumopts);
    }

    numoptargs = opt_parse(argc, argv, gnumopts, &gopts[0], 0);
    argc -= numoptargs;

    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
#ifdef HAVE_LIBXSLT
    if (opt_xcede) {
	bxh_doXCEDE = 1;
    }
#ifdef HAVE_LIBEXSLT
    if (opt_xcede2) {
	bxh_doXCEDE2 = 1;
    }
#endif
#endif

    if (opt_inputsfromfile != NULL) {
	char * buf = NULL;
	unsigned int bufsize = 0, bufend = 0;
	char * newarg;
	int i;
	FILE * fp = NULL;
	
	addargc = 0;
	addargv = (char **)malloc(sizeof(char *)*1);
	addargv[0] = NULL;

	if ((fp = fopen(opt_inputsfromfile, "r")) == NULL) {
	    fprintf(stderr, "Error opening file '%s'\n", opt_inputsfromfile);
	    perror("fopen");
	    exit(-1);
	}
	
	bufsize = 8192;
	buf = (char *)malloc(sizeof(char)*bufsize);
	buf[bufsize-1] = '\0';

	bufend = 0;
	while ((bufend += fread(&buf[bufend], sizeof(char), bufsize-1-bufend, fp)) != 0) {
	    char *eol;

	    buf[bufend] = '\0';
	    eol = strchr(buf, '\n');
	    if (eol == NULL && !feof(fp)) {
		bufsize *= 2;
		buf = (char *)realloc(buf, sizeof(char)*bufsize);
		buf[bufsize-1] = '\0';
		continue;
	    }
	    if (eol)
		*eol = '\0';

	    /* add option (and argument if any) to argv array */
	    newarg = (char *)malloc(sizeof(char)*(strlen(buf)+1));
	    strcpy(newarg, buf);
	    addargv = (char **)realloc(addargv, sizeof(char *)*(addargc+2));
	    addargv[addargc] = newarg;
	    addargv[addargc+1] = NULL;
	    addargc++;

	    if (eol) {
		bufend -= eol + 1 - &buf[0];
		memmove(&buf[0], eol+1, bufend);
	    } else {
		bufend = 0;
	    }
	}
	fclose(fp);
	newargv = (char **)malloc(sizeof(char *)*(argc+addargc+1));
	newargv[0] = argv[0];
	for (i = 0; i < addargc; i++) {
	    newargv[i+1] = addargv[i];
	}
	for (i = 1; i < argc; i++) {
	    newargv[1+addargc+(i-1)] = argv[i];
	}
	newargv[argc+addargc] = NULL;
	argv = newargv;
	argc += addargc;
	free(buf);
    }
    
    if (argc < 3) {
	fprintf(stderr, "Not enough arguments!\n");
	opt_usage(gnumopts, &gopts[0]);
	return 1;
    }

    if (fromtype == -1 && opt_fromtypestr != NULL) {
	if (strcasecmp(opt_fromtypestr, "pfile") == 0) {
	    fromtype = FROM_PFILE;
	    argv[0] = "pfile2bxh";
	} else if (strcasecmp(opt_fromtypestr, "signa5") == 0) {
	    fromtype = FROM_SIGNA5;
	    argv[0] = "signafive2bxh";
	} else if (strcasecmp(opt_fromtypestr, "signafive") == 0) {
	    fromtype = FROM_SIGNA5;
	    argv[0] = "signafive2bxh";
#ifndef WIN32
	} else if (strcasecmp(opt_fromtypestr, "iowa-signa5") == 0) {
	    fromtype = FROM_IOWASIGNA5;
	    argv[0] = "iowa-signafive2bxh";
	} else if (strcasecmp(opt_fromtypestr, "iowa-signafive") == 0) {
	    fromtype = FROM_IOWASIGNA5;
	    argv[0] = "iowa-signafive2bxh";
#endif
	} else if (strcasecmp(opt_fromtypestr, "ximg") == 0) {
	    fromtype = FROM_XIMG;
	    argv[0] = "ximg2bxh";
	} else if (strcasecmp(opt_fromtypestr, "analyze") == 0) {
	    fromtype = FROM_ANALYZE;
	    argv[0] = "analyze2bxh";
	} else if (strcasecmp(opt_fromtypestr, "afni") == 0) {
	    fromtype = FROM_AFNI;
	    argv[0] = "afni2bxh";
	} else if (strcasecmp(opt_fromtypestr, "nrrd") == 0) {
	    fromtype = FROM_NRRD;
	    argv[0] = "nrrd2bxh";
	} else if (strcasecmp(opt_fromtypestr, "dicom") == 0) {
	    fromtype = FROM_DICOM;
	    argv[0] = "dicom2bxh";
	} else if (strcasecmp(opt_fromtypestr, "minc") == 0) {
	    fromtype = FROM_MINC;
	    argv[0] = "minc2bxh";
	} else if (strcasecmp(opt_fromtypestr, "mgh") == 0) {
	    fromtype = FROM_MGH;
	    argv[0] = "mgh2bxh";
	} else if (strcasecmp(opt_fromtypestr, "bxh") == 0) {
	    fromtype = FROM_BXH;
	    argv[0] = "bxh2xcede";
	} else if (strcasecmp(opt_fromtypestr, "xcede") == 0) {
	    fromtype = FROM_BXH;
	    argv[0] = "xcede2bxh";
	} else {
	    fprintf(stderr, "Bad --fromtype '%s'\n", opt_fromtypestr);
	    opt_usage(gnumopts, &gopts[0]);
	    exit(-1);
	}
    }

    if (fromtype == -1 && argc > 1) {
	fromtype = bxh_autodetect_format(argv[1]);
    }
    
    if (fromtype == -1) {
	fprintf(stderr, "Can't figure out input data type (use --help for options)\n");
	exit(-1);
    }

    bxh_check_supported_type(fromtype);

    switch (fromtype) {
    case FROM_PFILE:
	retval = main_pfile2bxh(argc, argv);
	break;
    case FROM_SIGNA5:
	retval = main_signafive2bxh(argc, argv);
	break;
#ifndef WIN32
    case FROM_IOWASIGNA5:
	retval = main_iowasignafive2bxh(argc, argv);
	break;
#endif
    case FROM_XIMG:
	retval = main_ximg2bxh(argc, argv);
	break;
    case FROM_ANALYZE:
	retval = main_analyze2bxh(argc, argv);
	break;
    case FROM_AFNI:
	retval = main_afni2bxh(argc, argv);
	break;
    case FROM_NRRD:
	retval = main_nrrd2bxh(argc, argv);
	break;
    case FROM_DICOM:
#ifdef HAVE_DCMTK
	retval = main_dicom2bxh(argc, argv);
#endif
	break;
    case FROM_MINC:
#ifdef HAVE_MINC
	retval = main_minc2bxh(argc, argv);
#endif
	break;
    case FROM_MGH:
	retval = main_mgh2bxh(argc, argv);
	break;
    case FROM_BXH:
	retval = main_bxh2bxh(argc, argv);
	break;
    default:
	assert(0);
    }

    free(gopts); gopts = NULL;
    if (addargv) {
	int i;
	for (i = 0; i < addargc; i++) {
	    free(addargv[i]); addargv[i] = NULL;
	}
	free(addargv);
    }
    if (newargv) {
	free(newargv);
    }
    return retval;
}

int
main(int argc, char *argv[])
{
    return main_bxhabsorb(argc, argv);
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.38  2008/03/07 18:20:54  gadde
 * Add preliminary XCEDE 2 support.
 *
 * Revision 1.37  2007/02/23 23:24:40  gadde
 * Add {bxh|xcede}2{bxh|xcede}
 *
 * Revision 1.36  2007/02/03 20:01:26  gadde
 * Revert to uninitialized opt_hintorigin (and spacing and gap), and
 * initialize them in main_bxhabsorb.  HUGE_VAL does not count as a
 * constant on AIX.  Not sure why the initialization in main_bxhabsorb
 * doesn't work?
 *
 * Revision 1.35  2007/01/26 15:41:32  gadde
 * initialize some global variables
 *
 * Revision 1.34  2007/01/11 20:17:19  gadde
 * Add some xcede programs
 *
 * Revision 1.33  2006/10/20 17:03:12  gadde
 * Add .mgh/.mgz support
 *
 * Revision 1.32  2006/05/19 14:38:08  gadde
 * Support automatic detection of nifti (and gzipped nifti).
 *
 * Revision 1.31  2005/11/02 15:12:39  gadde
 * Allow XCEDE option for iowa-signafive2bxh.
 *
 * Revision 1.30  2005/09/20 18:37:56  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.29  2005/09/19 16:31:57  gadde
 * Documentation and help message updates.
 *
 * Revision 1.28  2005/03/08 18:45:15  gadde
 * Some AIX fixes
 *
 * Revision 1.27  2005/03/08 18:09:57  gadde
 * Add support for dimension size/spacing/gap/orientation hints
 *
 * Revision 1.26  2005/02/25 21:48:03  gadde
 * XCEDE updates
 *
 * Revision 1.25  2004/07/06 13:55:01  gadde
 * Some memory fixes.
 *
 * Revision 1.24  2004/05/27 22:02:30  gadde
 * Some updates to support phantomqa
 *
 * Revision 1.23  2004/04/27 13:47:07  gadde
 * replace comma in quotes
 *
 * Revision 1.22  2004/04/22 12:56:12  gadde
 * Add AFNI support.
 *
 * Revision 1.21  2004/02/26 22:38:02  gadde
 * Fxi typo.
 *
 * Revision 1.20  2004/02/26 22:36:17  gadde
 * Don't compile iowa for windows.
 *
 * Revision 1.19  2004/02/24 18:01:07  gadde
 * Move history generation to bxh_utils.cpp, and add option to create BIRN files
 *
 * Revision 1.18  2004/02/20 18:42:48  gadde
 * Add version option and rearrange documentation
 *
 * Revision 1.17  2004/01/08 17:47:18  gadde
 * Update docs
 *
 * Revision 1.16  2004/01/08 16:33:27  gadde
 * Add iowa GE support and other minor fixes
 *
 * Revision 1.15  2003/08/19 19:35:41  gadde
 * New options scheme, so that options are available using whichever form of bxhabsorb you use.
 *
 * Revision 1.14  2003/08/08 19:27:42  gadde
 * Don't use bxhabsorb usage info for subsidiary apps.
 *
 * Revision 1.13  2003/08/01 18:34:06  gadde
 * Let Windows version switch on executable name too
 *
 * Revision 1.12  2003/07/29 19:01:09  gadde
 * Move analyze detection to function
 *
 * Revision 1.11  2003/06/26 14:17:28  gadde
 * Fix byte order bug
 *
 * Revision 1.10  2003/06/18 16:14:14  gadde
 * win32 fixes
 *
 * Revision 1.9  2003/06/18 16:07:50  gadde
 * fopen in binary mode
 *
 * Revision 1.8  2003/06/06 19:41:56  gadde
 * Add Analyze support.
 *
 * Revision 1.7  2003/06/05 13:47:54  gadde
 * Allow list of input files to be read from a file.
 *
 * Revision 1.6  2003/05/16 18:27:35  gadde
 * Update usage information.
 *
 * Revision 1.5  2003/05/16 16:33:58  gadde
 * Add some autodetecting logic (borrowed from Matlab autodetect code)
 *
 * Revision 1.4  2003/05/12 18:44:12  gadde
 * Conditionally compile data formats in if libraries are found.
 *
 * Revision 1.3  2003/05/12 13:37:19  gadde
 * #ifdef MINC support
 *
 * Revision 1.2  2003/05/10 18:21:09  gadde
 * Add MINC support.
 *
 * Revision 1.1  2003/04/28 19:44:59  gadde
 * First import of catch-all BXH encapsulation program
 *
 */
