static const char rcsid[] = "$Id: bxhreorient.c,v 1.17 2009-04-03 00:54:14 gadde Exp $";

/*
 * bxhreorient.c --
 * 
 * Reorients a BXH file to given orientation
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <sys/stat.h>
#include <math.h>
#include "bxh_types.h"
#include "bxh_datarec.h"
#include "bxh_niftilib.h"
#include "opts.h"

#ifndef XMLH_VERSIONSTR
#define XMLH_VERSIONSTR "(no version specified)"
#endif

static char * opt_orientation = "ras";
static int opt_inplace = 0;
static int opt_version = 0;
static int opt_overwrite = 0;

#define PROGNAME "bxhreorient"

int
main(int argc, char *argv[])
{
    struct stat statbuf;
    char * ifname = NULL;
    char * ofbase = NULL;
    char * ofname = NULL;
    char * odatafname = NULL;
    BXHElementPtr * sliceorderp = NULL;
    bxhdimension *rasdims[3] = { NULL, NULL, NULL };
    char * databuf = NULL;
    int i;
    int needreorient = 0;
    int fromdim, todim;
    int dim2ras[3] = { 0, 0, 0 };
    int reqras[3] = { 0, 0, 0 };
    int mmatrix[3][4];
    int result = 0;
    const int numopts = 5;
    const char * ordereddimnames[] = { "x", "y", "z", "t" };
    struct bxhdataread bdr;

    int oldargc = argc;
    char ** oldargv = argv;

    opt_data myOpts[5] = {
	{ 0x0, 0x0, NULL, 0, "",
	  "Usage:\n"
	  "  " PROGNAME " [options] inputfile [ outputfile [datafileout] ]\n\n"
	  "This program reorients the image data given by the input BXH or "
	  "XCEDE file to an orientation specified by the user using the "
	  "--orientation option.  It is assumed that the orientation vectors "
	  "in the BXH/XCEDE file are correct with respect to the image data.  "
	  "outputfile is required if not using --inplace option.  Output "
	  "is also a BXH or XCEDE file, pointing to an image data file "
	  "(named by datafileout if specified)." },
	{ 0x0, OPT_VAL_NONE, NULL, 0, "", "" },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_version, 1, "version",
	  "Print version string and exit." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_overwrite, 1, "overwrite",
	  "Overwrite output files if they exist." },
	{ OPT_FLAGS_FULL, OPT_VAL_STR, &opt_orientation, 1, "orientation",
	  "This option specifies the new orientation by R/L A/P S/I letters, "
	  "upper- or lower-case, in X,Y,Z order, "
	  "where R means that dimension starts on the left "
	  "and goes TO THE RIGHT, "
	  "A means the dimension goes from posterior TO ANTERIOR, etc.  "
	  "For example, IPR means X goes S->I, Y goes A->P, and Z goes L->R.  "
	  "Default is RAS (neurological axial, as used by SPM)." },
	{ OPT_FLAGS_FULL, OPT_VAL_BOOL, &opt_inplace, 1, "inplace",
	  "Do the reorientation in-place, overwriting the original files "
	  "(both BXH and data) with new data.  WARNING: THIS OPTION IS "
	  "DANGEROUS AS IT WILL ALTER IMAGE DATA BUT MAY NOT UPDATE OTHER "
	  "IMAGE HEADER METADATA IN THE WRAPPED DATA FILES! (e.g. if the raw "
	  "data is DICOM or NIFTI, the DICOM or NIFTI headers will not be "
	  "updated, only the wrapper will have the correct metadata).  "
	  "This option may be removed in the future." }
    }; 

    memset(&bdr, '\0', sizeof(bdr));

    argc -= opt_parse(argc, argv, numopts, &myOpts[0], 0);
    
    if (opt_version) {
	fprintf(stdout, "%s\n", XMLH_VERSIONSTR);
	exit(0);
    }
    if (argc < 2 || (!opt_inplace && argc < 3) || argc > 4) {
	opt_usage(numopts, &myOpts[0]);
	return -1;
    }

    if (strlen(opt_orientation) != 3) {
	fprintf(stderr, PROGNAME ": orientation '%s' must have three letters\n", opt_orientation); 
	goto FAIL;
    }
    if (isupper(opt_orientation[0])) {
	opt_orientation[0] = tolower(opt_orientation[0]);
    }
    if (isupper(opt_orientation[1])) {
	opt_orientation[1] = tolower(opt_orientation[1]);
    }
    if (isupper(opt_orientation[2])) {
	opt_orientation[2] = tolower(opt_orientation[2]);
    }

    ifname = strdup(argv[1]);
    if (opt_inplace) {
	ofname = strdup(ifname);
    } else {
	ofname = strdup(argv[2]);
    }
    if (argc == 4) {
	odatafname = strdup(argv[3]);
    }

    if (!opt_inplace && odatafname == NULL) {
	char * dotpos = NULL;
	ofbase = strdup(ofname);
	odatafname = (char *)malloc(sizeof(char) * (strlen(ofname) + 8));
	strcpy(odatafname, ofname);
	dotpos = strrchr(ofbase, '.');
	if (dotpos != NULL &&
#ifdef _WIN32
	    strchr(dotpos, '\\') == NULL &&
#endif
	    strchr(dotpos, '/') == NULL) {
	    *dotpos = '\0';
	}
	strcpy(odatafname, ofbase);
	strcat(odatafname, ".nii.gz");
    }

    if (stat(ifname, &statbuf) != 0) {
	fprintf(stderr, PROGNAME ": Input BXH file('%s') does not exist \n", ifname); 
	goto FAIL;
    }
    if (!opt_overwrite && !opt_inplace && stat(ofname, &statbuf) == 0) {
	fprintf(stderr, PROGNAME ": Output BXH file('%s') exists!\n", ofname); 
	goto FAIL;
    }
    if (!opt_overwrite && !opt_inplace && odatafname && stat(odatafname, &statbuf) == 0) {
	fprintf(stderr, PROGNAME ": Output data file('%s') exists!\n", odatafname); 
	goto FAIL;
    }

    /* read the input file and data */
    if (bxh_dataReadFileStart(ifname, "image", NULL, 4, ordereddimnames, NULL, &bdr) != 0) {
	fprintf(stderr, "Error preparing data read for '%s'.\n", ifname);
	goto FAIL;
    }
    if (bxh_dataReadFinish(&bdr, NULL) != 0) {
	fprintf(stderr, "Error finishing data read for '%s'.\n", ifname);
	goto FAIL;
    }

    for (i = 0; i < 3; i++) {
	bxhdimension * dimp = &bdr.datarec->dimensions[i];
	if (dimp->direction == NULL) {
	    fprintf(stderr, PROGNAME ": Dimension '%s' is missing direction field!\n", dimp->type);
	    goto FAIL;
	}
    }

    /* calculate new origin.
     * This basically involves finding out what new corner to use --
     * easiest way is to find out what axes we are flipping and adjust
     * the origin to account for those. */
    {
	double rasorigin[3] = { 0, 0, 0 };
	int rasoriginflip[3] = { 0, 0, 0 };
	for (i = 0; i < 3; i++) {
	    bxhdimension * dimp = &bdr.datarec->dimensions[i];
	    double curmax = 0;
	    char rasletter = ' ';
	    char opprasletter = ' ';
	    bxhdimension ** setthisdim = NULL;
	    int * setthisflip = NULL;
	    curmax = dimp->direction[0];
	    if (dimp->direction[0] > 0) {
		rasletter = 'r';
		opprasletter = 'l';
	    } else {
		rasletter = 'l';
		opprasletter = 'r';
	    }
	    setthisdim = &rasdims[0];
	    setthisflip = &rasoriginflip[0];
	    if (fabs(dimp->direction[1]) > fabs(curmax)) {
		if (dimp->direction[1] > 0) {
		    rasletter = 'a';
		    opprasletter = 'p';
		} else {
		    rasletter = 'p';
		    opprasletter = 'a';
		}
		curmax = dimp->direction[1];
		setthisdim = &rasdims[1];
		setthisflip = &rasoriginflip[1];
	    }
	    if (fabs(dimp->direction[2]) > fabs(curmax)) {
		if (dimp->direction[2] > 0) {
		    rasletter = 's';
		    opprasletter = 'i';
		} else {
		    rasletter = 'i';
		    opprasletter = 's';
		}
		curmax = dimp->direction[2];
		setthisdim = &rasdims[2];
		setthisflip = &rasoriginflip[2];
	    }
	    *setthisdim = dimp;
	    if (strchr(opt_orientation, opprasletter) != NULL) {
		*setthisflip = 1;
	    }
	}
	for (i = 0; i < 3; i++) {
	    if (rasdims[i] == NULL) {
		fprintf(stderr, PROGNAME ": Could not find appropriate RAS assignment for dimensions!\n");
		goto FAIL;
	    }
	}
	rasorigin[0] = rasdims[0]->origin;
	rasorigin[1] = rasdims[1]->origin;
	rasorigin[2] = rasdims[2]->origin;
	for (i = 0; i < 3; i++) {
	    bxhdimension * dimp = rasdims[i];
	    if (rasoriginflip[i]) {
		rasorigin[0] += dimp->direction[0] * dimp->spacing * (dimp->size - 1);
		rasorigin[1] += dimp->direction[1] * dimp->spacing * (dimp->size - 1);
		rasorigin[2] += dimp->direction[2] * dimp->spacing * (dimp->size - 1);
	    }
	}
	rasdims[0]->origin = rasorigin[0];
	rasdims[1]->origin = rasorigin[1];
	rasdims[2]->origin = rasorigin[2];
    }

    /* create matrix that transforms dimensions in original
     * orientation to that of new orientation */
    for (i = 0; i < 3; i++) {
	bxhdimension * dimp = &bdr.datarec->dimensions[i];
	double curmax = 0;
	dim2ras[i] = 1;
	curmax = dimp->direction[0];
	if (fabs(dimp->direction[1]) > fabs(curmax)) {
	    dim2ras[i] = 2;
	    curmax = dimp->direction[1];
	}
	if (fabs(dimp->direction[2]) > fabs(curmax)) {
	    dim2ras[i] = 3;
	    curmax = dimp->direction[2];
	}
	if (curmax < 0) {
	    dim2ras[i] *= -1;
	}

	if (opt_orientation[i] == 'r') {
	    reqras[i] = 1;
	} else if (opt_orientation[i] == 'l') {
	    reqras[i] = -1;
	} else if (opt_orientation[i] == 'a') {
	    reqras[i] = 2;
	} else if (opt_orientation[i] == 'p') {
	    reqras[i] = -2;
	} else if (opt_orientation[i] == 's') {
	    reqras[i] = 3;
	} else if (opt_orientation[i] == 'i') {
	    reqras[i] = -3;
	} else {
	    fprintf(stderr, "Unrecognized orientation letter '%c'!\n",
		    opt_orientation[i]);
	}
    }
    memset(mmatrix, '\0', sizeof(mmatrix));
    for (todim = 0; todim < 3; todim++) {
	for (fromdim = 0; fromdim < 3; fromdim++) {
	    if (dim2ras[fromdim] == reqras[todim]) {
		mmatrix[todim][fromdim] = 1;
		if (fromdim != todim) {
		    needreorient = 1;
		}
	    } else if (dim2ras[fromdim] == -1 * reqras[todim]) {
		mmatrix[todim][fromdim] = -1;
		mmatrix[todim][3] = bdr.datarec->dimensions[fromdim].size - 1;
		needreorient = 1;
	    }
	}
    }

    if (opt_inplace && !needreorient) {
	fprintf(stderr, PROGNAME ": Data unchanged.\n");
	goto EXIT;
    }

    if ((sliceorderp = bxh_getChildElement(bdr.acqdatap, "sliceorder")) != NULL) {
	int writeoriginal = 0;
	char * sliceorderstr = NULL;
	if (bxh_getElementStringValue(sliceorderp, &sliceorderstr) != 0) {
	    fprintf(stderr, PROGNAME ": Error getting sliceorder value.\n");
	    bxh_element_unref(sliceorderp);
	    free(sliceorderstr);
	    goto FAIL;
	}
	if (needreorient && mmatrix[2][2] != 0) {
	    /* slice axis does not change but may be flipped */
	    if (mmatrix[2][2] == -1) {
		/* need to update sliceorder element */
		int numslices = bdr.datarec->dimensions[2].size;
		char * curpos = NULL;
		char * newcurpos = NULL;
		char * newsliceorderstr = NULL;
		newsliceorderstr = (char *)malloc(sizeof(char)*(strlen(sliceorderstr) + 1));
		newsliceorderstr[0] = '\0';
		newcurpos = newsliceorderstr;
		for (curpos = sliceorderstr; curpos != NULL; ((curpos = strchr(curpos, ',')) != NULL) && curpos++) {
		    int slicenum = -1;
		    if (sscanf(curpos, "%d", &slicenum) != 1) {
			fprintf(stderr, PROGNAME ": Error reading sliceorder component.\n");
			bxh_element_unref(sliceorderp);
			free(sliceorderstr);
			goto FAIL;
		    }
		    if (newsliceorderstr[0] != '\0') {
			newcurpos += sprintf(newcurpos, ",");
		    }
		    newcurpos += sprintf(newcurpos, "%d", (numslices - slicenum) + 1);
		}
		if (bxh_setChildElement(bdr.acqdatap, "sliceorder", newsliceorderstr) != 0) {
		    fprintf(stderr, PROGNAME ": Error setting sliceorder element.\n");
		    bxh_element_unref(sliceorderp);
		    free(sliceorderstr);
		    goto FAIL;
		}
		writeoriginal = 1;
	    }
	} else if (needreorient && mmatrix[2][2] == 0) {
	    /* slice axis changes to another dimension -- change sliceorder element to originalsliceorder */
	    BXHCommentPtr * commentp = NULL;
	    writeoriginal = 1;
	    if (bxh_removeChildElement(bdr.acqdatap, sliceorderp) == NULL) {
		fprintf(stderr, PROGNAME ": Error removing sliceorder element.\n");
		bxh_element_unref(sliceorderp);
		free(sliceorderstr);
		goto FAIL;
	    }
	    bxh_element_unref(commentp); commentp = NULL;
	}
	if (writeoriginal) {
	    BXHCommentPtr * commentp = NULL;
	    if ((commentp = bxh_appendComment(bdr.acqdatap, " The following <originalsliceorder> element is the original slice order saved for reference, due to re-orientation of the data; actual slice order for the current orientation of the data, if applicable, would be in the <sliceorder> element. ", 1)) == NULL) {
		fprintf(stderr, PROGNAME ": Error appending comment.\n");
		bxh_element_unref(sliceorderp);
		free(sliceorderstr);
		goto FAIL;
	    }
	    if (bxh_appendChildElement(bdr.acqdatap, "originalsliceorder", sliceorderstr) != 0) {
		fprintf(stderr, PROGNAME ": Error creating originalsliceorder element.\n");
		bxh_element_unref(sliceorderp);
		free(sliceorderstr);
		goto FAIL;
	    }
	    bxh_element_unref(commentp); commentp = NULL;
	}
	bxh_element_unref(sliceorderp);
	free(sliceorderstr);
    }

    databuf = (char *)bdr.dataptr;
    if (needreorient) {
	bxhrawdatarec * newdatarec = NULL;
	char * tmpbuf = NULL;
	int * sizes = NULL;
	int * newsizes = NULL;
	int * coords = NULL;
	int * newcoords = NULL;
	size_t elemnum = 0;
	size_t newelemnum = 0;
	int dimnum = 0;
	int numdims = 0;
	size_t elemsize = 0;
	size_t numelems = 0;

	fprintf(stderr, "Reorienting data...\n");

	/* cache some values */
	numdims = bdr.datarec->numdims;
	elemsize = bdr.datarec->elemsize;
	numelems = bdr.datarec->datasize / bdr.datarec->elemsize;

	newdatarec = bxh_datarec_copy(bdr.datarec);
	{
	    bxhdimension tmpdims[3];
	    for (dimnum = 0; dimnum < 3; dimnum++) {
		int copydimnum = -1;
		if (mmatrix[dimnum][0])
		    copydimnum = 0;
		else if (mmatrix[dimnum][1])
		    copydimnum = 1;
		else if (mmatrix[dimnum][2])
		    copydimnum = 2;
		else {
		    fprintf(stderr, PROGNAME ": Can't properly determine input orientation -- perhaps orientation vectors are extremely non-orthogonal?\n");
		    goto FAIL;
		}
		memcpy(&tmpdims[dimnum], &newdatarec->dimensions[copydimnum],
		       sizeof(tmpdims[dimnum]));
		if (mmatrix[dimnum][copydimnum] == -1) {
		    int dpstructind;
		    /* swap/mirror the data points */
		    for (dpstructind = 0;
			 dpstructind < tmpdims[dimnum].numdpstructs;
			 dpstructind++) {
			bxhdatapoints * dpstructp = &tmpdims[dimnum].dpstructs[dpstructind];
			int numvalues = dpstructp->numvalues;
			unsigned int dpnum;
			for (dpnum = 0; dpnum < numvalues / 2; dpnum++) {
			    char * swap = dpstructp->values[dpnum];
			    dpstructp->values[dpnum] = dpstructp->values[numvalues - dpnum - 1];
			    dpstructp->values[numvalues - dpnum - 1] = swap;
			}
		    }
		    /* flip the direction */
		    tmpdims[dimnum].direction[0] *= -1;
		    tmpdims[dimnum].direction[1] *= -1;
		    tmpdims[dimnum].direction[2] *= -1;
		}
	    }
	    memcpy(newdatarec->dimensions, &tmpdims[0], sizeof(tmpdims));
	    free(newdatarec->dimensions[0].type);
	    free(newdatarec->dimensions[1].type);
	    free(newdatarec->dimensions[2].type);
	    newdatarec->dimensions[0].type = strdup("x");
	    newdatarec->dimensions[1].type = strdup("y");
	    newdatarec->dimensions[2].type = strdup("z");
	}
	if ((tmpbuf = (char *)malloc(sizeof(char)*bdr.datarec->datasize)) == NULL) {
	    fprintf(stderr, "  Couldn't allocate enough memory!\n");
	    goto FAIL;
	}
	sizes = (int *)malloc(sizeof(int) * bdr.datarec->numdims);
	newsizes = (int *)malloc(sizeof(int) * newdatarec->numdims);
	for (dimnum = 0; dimnum < bdr.datarec->numdims; dimnum++) {
	    sizes[dimnum] = bdr.datarec->dimensions[dimnum].size;
	    newsizes[dimnum] = newdatarec->dimensions[dimnum].size;
	}
	coords = (int *)malloc(sizeof(int) * bdr.datarec->numdims);
	memset(coords, '\0', sizeof(int) * bdr.datarec->numdims);
	newcoords = (int *)malloc(sizeof(int) * newdatarec->numdims);
	memset(newcoords, '\0', sizeof(int) * newdatarec->numdims);
	elemnum = 0;
	while (elemnum < numelems) {
	    for (dimnum = 0; dimnum < 3; dimnum++) {
		newcoords[dimnum] =
		    mmatrix[dimnum][0] * coords[0] +
		    mmatrix[dimnum][1] * coords[1] +
		    mmatrix[dimnum][2] * coords[2] +
		    mmatrix[dimnum][3] * 1;
	    }
	    for (dimnum = 3; dimnum < bdr.datarec->numdims; dimnum++) {
		newcoords[dimnum] = coords[dimnum];
	    }
	    newelemnum = 0;
	    for (dimnum = numdims - 1; dimnum >= 0; dimnum--) {
		newelemnum += newcoords[dimnum];
		if (dimnum >= 1) {
		    newelemnum *= newsizes[dimnum-1];
		}
	    }
	    /* copy */
	    for (i = 0; i < (int)elemsize; i++) {
		tmpbuf[newelemnum * elemsize + i] =
		    databuf[elemnum * elemsize + i];
	    }
	    /* increment coords */
	    elemnum++;
	    for (dimnum = 0; dimnum < numdims; dimnum++) {
		coords[dimnum]++;
		if (coords[dimnum] < sizes[dimnum])
		    break; /* our job is done */
		coords[dimnum] = 0;
		/* loop around to increment next dimension */
	    }
	}
	bxh_datarec_free(bdr.datarec);
	bdr.datarec = newdatarec;
	databuf = tmpbuf;
	free(bdr.dataptr);
	bdr.dataptr = (void *)tmpbuf;

	if (sizes)
	    free(sizes);
	if (newsizes)
	    free(newsizes);
	if (coords)
	    free(coords);
	if (newcoords)
	    free(newcoords);
    }

    /* databuf/datarec now contains reoriented data/dimensions */
    /* write the data/BXH file out */
    if (opt_inplace) {
	int fragnum = 0;
	off_t curpos = 0;
	FILE * outfp = NULL;

	if (bxh_datarec_writeToElement(bdr.imagedatap, bdr.datarec) == -1) {
	    fprintf(stderr, "Error writing new frags to datarec element\n");
	    goto FAIL;
	} 
	
	if (bxh_addAutoHistoryEntry(bdr.docp, oldargv[0], (const char **)&oldargv[1], oldargc-1) != 0) {
	    fprintf(stderr, "Error adding history entry\n");
	    goto FAIL;
	}

	if (bxh_writeFile(bdr.docp, ofname) != 0) {
	    fprintf(stderr, "Error writing output BXH file\n");
	    goto FAIL;
	} 
	
	curpos = 0;
	for (fragnum = 0; fragnum < bdr.datarec->numfrags; fragnum++) {
	    bxhfragp fragp = NULL;
	    char * fullfilename = NULL;
	    struct stat statbuf;
	    int outputisgz = 0;
	    fragp = bdr.datarec->frags[fragnum];
	    fullfilename = bxh_datarec_getfullfilename(bdr.datarec->bxhpath, fragp->filename);
	    if (stat(fullfilename, &statbuf) == -1) {
		char buf[8192];
		char * fullfilenamegz = NULL;
		char * tempgzout = NULL;
		z_off_t gzoff = 0;
		gzFile gzinfp = NULL;
		gzFile gzoutfp = NULL;
		fullfilenamegz = (char *)malloc(sizeof(char)*strlen(fullfilename) + 4);
		strcpy(fullfilenamegz, fullfilename);
		strcat(fullfilenamegz, ".gz");
		if (stat(fullfilenamegz, &statbuf) == -1) {
		    fprintf(stderr, "ERROR: You specified --inplace, but neither '%s' nor '%s' exist!", fullfilename, fullfilenamegz);
		    free(fullfilenamegz);
		    goto FAIL;
		}
		free(fullfilename);
		outputisgz = 1;
		tempgzout = (char *)malloc(sizeof(char)*strlen(fullfilenamegz) + 5);
		strcpy(tempgzout, fullfilenamegz);
		strcat(tempgzout, ".tmp");
		if ((gzinfp = gzopen(fullfilenamegz, "rb")) == NULL) {
		    fprintf(stderr, PROGNAME ": Error opening data file '%s' for reading\n", fullfilenamegz);
		    goto FAIL;
		}
		if ((gzoutfp = gzopen(tempgzout, "wb")) == NULL) {
		    fprintf(stderr, PROGNAME ": Error opening data file '%s' for writing\n", tempgzout);
		    goto FAIL;
		}
		while (gzoff < fragp->fileoffset) {
		    unsigned int readlen = sizeof(buf);
		    if (readlen > fragp->fileoffset - gzoff) {
			readlen = fragp->fileoffset - gzoff;
		    }
		    if (gzread(gzinfp, buf, readlen) != readlen) {
			fprintf(stderr, PROGNAME ": Error reading from '%s' at byte offset %u (len %u)\n", fullfilenamegz, (unsigned int)gzoff, (unsigned int)readlen);
			gzclose(gzinfp);
			gzclose(gzoutfp);
			goto FAIL;
		    }
		    if (gzwrite(gzoutfp, buf, readlen) != readlen) {
			fprintf(stderr, PROGNAME ": Error writing %u byes to '%s'\n", (unsigned int)readlen, tempgzout);
			gzclose(gzinfp);
			gzclose(gzoutfp);
			goto FAIL;
		    }
		    gzoff += readlen;
		}
		if (gzwrite(gzoutfp, databuf + curpos, fragp->fragsize) != fragp->fragsize) {
		    fprintf(stderr, PROGNAME ": Error writing to file '%s'\n", tempgzout);
		    gzclose(gzinfp);
		    gzclose(gzoutfp);
		    goto FAIL;
		}
		if (gzseek(gzinfp, gzoff + fragp->fragsize, SEEK_SET) == -1) {
		    fprintf(stderr, PROGNAME ": Error seeking in file '%s'\n", tempgzout);
		    gzclose(gzinfp);
		    gzclose(gzoutfp);
		    goto FAIL;
		}
		while (1) {
		    int bytesread;
		    bytesread = gzread(gzinfp, buf, sizeof(buf));
		    if (bytesread == -1) {
			fprintf(stderr, PROGNAME ": Error reading from file '%s'\n", fullfilenamegz);
			gzclose(gzinfp);
			gzclose(gzoutfp);
			goto FAIL;
		    } else if (bytesread == 0) {
			/* end of file */
			break;
		    }
		    if (gzwrite(gzoutfp, buf, bytesread) != bytesread) {
			fprintf(stderr, PROGNAME ": Error writing to file '%s'\n", tempgzout);
			gzclose(gzinfp);
			gzclose(gzoutfp);
			goto FAIL;
		    }
		}
		gzclose(gzoutfp);
		gzclose(gzinfp);
		unlink(fullfilenamegz);
		rename(tempgzout, fullfilenamegz);
	    } else {
		if ((outfp = fopen(fullfilename, "rb+")) == NULL) {
		    fprintf(stderr, PROGNAME ": Error opening output data file '%s'\n", fullfilename);
		    perror("fopen");
		    goto FAIL;
		}
		if (fseek(outfp, fragp->fileoffset, SEEK_SET) != 0) {
		    fprintf(stderr, PROGNAME ": Error seeking in data file '%s'\n", fullfilename);
		    perror("fseek");
		    goto FAIL;
		}
		if (fwrite(databuf + curpos, fragp->fragsize, 1, outfp) != 1) {
		    fprintf(stderr, PROGNAME ": Error writing to data file '%s'\n", fullfilename);
		    goto FAIL;
		}
		fclose(outfp);
	    }
	    curpos += fragp->fragsize;
	}
    } else {
	/* write out data */
	if (bxh_datarec_writeToElement(bdr.imagedatap, bdr.datarec) != 0) {
	    fprintf(stderr, "Failed writing datarec\n");
	    goto FAIL;
	}
	if (bxh_addAutoHistoryEntry(bdr.docp, argv[0], (const char **)&ifname, 1) != 0) {
	    fprintf(stderr, "Error adding history entry\n");
	    goto FAIL;
	}
	writeBXHAndNIIGZ(ofbase, &bdr, (void *)databuf, 0);
    }
    goto EXIT;

FAIL:
    fprintf(stderr, PROGNAME ": failed.\n");
    result = -1;

EXIT:
    if (ifname)
	free(ifname);
    if (ofname)
	free(ofname);
    if (ofbase)
	free(ofbase);
    if (odatafname)
	free(odatafname);
    
    bxh_datareaddata_free(&bdr);

    return result;
}

/*
 * $Log: In-line log eliminated on transition to SVN; use svn log instead. $
 * Revision 1.16  2005/09/20 18:37:56  gadde
 * Updates to versioning, help and documentation, and dependency checking
 *
 * Revision 1.15  2005/09/19 16:31:57  gadde
 * Documentation and help message updates.
 *
 * Revision 1.14  2005/09/14 14:49:31  gadde
 * Type conversion updates to fix win32 warnings
 *
 * Revision 1.13  2005/09/13 14:59:04  gadde
 * Allow output file to have non-.bxh extension
 *
 * Revision 1.12  2005/05/09 20:08:20  gadde
 * Add history entries to output.
 *
 * Revision 1.11  2005/04/21 12:24:51  gadde
 * Make sure output x/y/z dimensions are labeled in the proper order.
 *
 * Revision 1.10  2005/04/05 18:31:55  gadde
 * Add missing spaces to help.
 *
 * Revision 1.9  2005/04/05 18:24:27  gadde
 * Make usage statement a little clearer.
 *
 * Revision 1.8  2004/06/15 16:16:11  gadde
 * Several -Wall fixes and addition of bxh_datarec_addfrag()
 *
 * Revision 1.7  2004/01/02 18:41:43  gadde
 * -Wall fixes
 *
 * Revision 1.6  2003/11/17 21:58:41  gadde
 * Add math.h #include.
 *
 * Revision 1.5  2003/11/14 20:16:45  gadde
 * Updated help.
 *
 * Revision 1.4  2003/11/14 15:03:47  gadde
 * Add diagnostic.
 *
 * Revision 1.3  2003/11/14 15:01:49  gadde
 * Fix calculation of newcoords.
 *
 * Revision 1.2  2003/11/13 21:09:30  gadde
 * Add log.
 *
 */
