/*
 * Decompiled with CFR 0.152.
 */
package tractography;

import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Arrays;
import misc.LoggedException;
import numerics.Point3D;
import numerics.RealMatrix;
import numerics.Vector3D;
import tractography.Voxel;
import tractography.VoxelList;

public final class Tract {
    private int capacity;
    private double growBy;
    private int numberOfPoints;
    private Point3D[] points;
    private double[] displacements;
    private double[] voxelPathLengths;
    private int seedPointIndex;
    private int seedPointVoxelIndex;
    private VoxelList voxels = null;
    private boolean displacementsCalculated = false;
    private boolean voxelPathLengthsCalculated = false;

    public Tract(int n, double d) {
        this.numberOfPoints = 0;
        this.seedPointIndex = 0;
        this.capacity = n;
        this.growBy = d;
        this.initialise();
    }

    public Tract(Tract tract) {
        this.numberOfPoints = tract.numberOfPoints;
        this.capacity = tract.capacity + 1;
        this.growBy = tract.growBy;
        this.seedPointIndex = tract.seedPointIndex;
        this.initialise();
        this.displacementsCalculated = tract.displacementsCalculated;
        System.arraycopy(tract.points, 0, this.points, 0, this.numberOfPoints);
        System.arraycopy(tract.displacements, 0, this.displacements, 0, this.numberOfPoints);
    }

    public Tract() {
        this.numberOfPoints = 0;
        this.seedPointIndex = 0;
        this.capacity = 100;
        this.growBy = 100.0;
        this.initialise();
    }

    private void initialise() {
        this.points = new Point3D[this.capacity];
        this.displacements = new double[this.capacity];
    }

    private void growArray() {
        int n = (int)((double)this.capacity + (double)this.capacity * this.growBy / 100.0);
        if (n == this.capacity) {
            n = this.capacity + 1;
        }
        Point3D[] point3DArray = new Point3D[n];
        double[] dArray = new double[n];
        System.arraycopy(this.points, 0, point3DArray, 0, this.numberOfPoints);
        System.arraycopy(this.displacements, 0, dArray, 0, this.numberOfPoints);
        this.capacity = n;
        this.displacements = dArray;
        this.points = point3DArray;
    }

    protected void joinTract(Tract tract) {
        int n;
        if (tract.numberOfPoints == 0) {
            return;
        }
        this.voxels = null;
        int n2 = tract.numberOfPoints;
        int n3 = this.numberOfPoints + n2;
        Point3D[] point3DArray = new Point3D[n3];
        double[] dArray = new double[n3];
        for (n = 0; n < n2 - 1; ++n) {
            point3DArray[n] = tract.points[n2 - 1 - n];
            dArray[n] = tract.displacements[n2 - 1 - n];
        }
        this.seedPointIndex = n2 - 1;
        for (n = 0; n < this.numberOfPoints; ++n) {
            point3DArray[n + (n2 - 1)] = this.points[n];
            dArray[n + (n2 - 1)] = this.displacements[n];
        }
        this.numberOfPoints += n2 - 1;
        this.capacity = n3;
        this.points = point3DArray;
        this.displacements = dArray;
        this.displacementsCalculated = this.displacementsCalculated && tract.displacementsCalculated;
    }

    public void addPoint(Point3D point3D) {
        this.voxels = null;
        this.voxelPathLengthsCalculated = false;
        this.displacementsCalculated = false;
        this.points[this.numberOfPoints] = point3D;
        ++this.numberOfPoints;
        if (this.numberOfPoints == this.capacity) {
            this.growArray();
        }
    }

    public void addPoint(Point3D point3D, double d) {
        this.voxels = null;
        this.voxelPathLengthsCalculated = false;
        this.points[this.numberOfPoints] = point3D;
        this.displacements[this.numberOfPoints] = d;
        ++this.numberOfPoints;
        if (this.numberOfPoints == this.capacity) {
            this.growArray();
        }
    }

    private void calculateDisplacements() {
        Point3D point3D;
        Point3D point3D2;
        int n;
        if (this.numberOfPoints < 2) {
            return;
        }
        int n2 = this.seedPointIndex;
        for (n = n2 - 1; n >= 0; --n) {
            point3D2 = this.points[n];
            point3D = this.points[n + 1];
            this.displacements[n] = Math.sqrt((point3D2.x - point3D.x) * (point3D2.x - point3D.x) + (point3D2.y - point3D.y) * (point3D2.y - point3D.y) + (point3D2.z - point3D.z) * (point3D2.z - point3D.z));
        }
        for (n = n2 + 1; n < this.numberOfPoints; ++n) {
            point3D2 = this.points[n];
            point3D = this.points[n - 1];
            this.displacements[n] = Math.sqrt((point3D2.x - point3D.x) * (point3D2.x - point3D.x) + (point3D2.y - point3D.y) * (point3D2.y - point3D.y) + (point3D2.z - point3D.z) * (point3D2.z - point3D.z));
        }
        this.displacementsCalculated = true;
    }

    public double pathLengthFromSeed(int n) {
        if (!this.displacementsCalculated) {
            this.calculateDisplacements();
        }
        if (n == this.seedPointIndex) {
            return 0.0;
        }
        if (n < 0 || n >= this.numberOfPoints) {
            throw new IllegalArgumentException("Illegal index " + n + " is < 0 or exceeds index of last point");
        }
        if (n < this.seedPointIndex) {
            double d = 0.0;
            for (int i = this.seedPointIndex - 1; i >= n; --i) {
                d += this.displacements[i];
            }
            return d;
        }
        double d = 0.0;
        for (int i = this.seedPointIndex + 1; i <= n; ++i) {
            d += this.displacements[i];
        }
        return d;
    }

    public double length() {
        if (!this.displacementsCalculated) {
            this.calculateDisplacements();
        }
        double d = 0.0;
        for (int i = 0; i < this.numberOfPoints; ++i) {
            d += this.displacements[i];
        }
        return d;
    }

    public int numberOfPoints() {
        return this.numberOfPoints;
    }

    public VoxelList toVoxelList(double[] dArray) {
        return this.toVoxelList(dArray[0], dArray[1], dArray[2]);
    }

    public VoxelList toVoxelList(double d, double d2, double d3) {
        VoxelList voxelList;
        Object[] objectArray;
        if (this.voxels != null) {
            objectArray = this.voxels.getVoxelDims();
            if (d == objectArray[0] && d2 == objectArray[1] && d3 == objectArray[2]) {
                return this.voxels;
            }
            this.voxels = null;
            this.voxelPathLengthsCalculated = false;
        }
        objectArray = new Voxel[this.numberOfPoints];
        int n = (int)(this.points[0].x / d);
        int n2 = (int)(this.points[0].y / d2);
        int n3 = (int)(this.points[0].z / d3);
        objectArray[0] = (double)new Voxel(n, n2, n3);
        int n4 = 1;
        int n5 = 0;
        for (int i = 1; i < this.numberOfPoints; ++i) {
            n = (int)(this.points[i].x / d);
            n2 = (int)(this.points[i].y / d2);
            n3 = (int)(this.points[i].z / d3);
            if (n != objectArray[n4 - 1].x || n2 != objectArray[n4 - 1].y || n3 != objectArray[n4 - 1].z) {
                objectArray[n4] = (double)new Voxel(n, n2, n3);
                ++n4;
            }
            if (i != this.seedPointIndex) continue;
            n5 = n4 - 1;
        }
        Voxel[] voxelArray = new Voxel[n4];
        System.arraycopy(objectArray, 0, voxelArray, 0, n4);
        Vector3D vector3D = null;
        Vector3D vector3D2 = null;
        vector3D = this.seedPointIndex < this.numberOfPoints - 1 ? new Vector3D(this.points[this.seedPointIndex + 1], this.points[this.seedPointIndex]) : (this.numberOfPoints > 1 ? new Vector3D(this.points[this.seedPointIndex], this.points[this.seedPointIndex - 1]) : new Vector3D(1.0, 0.0, 0.0));
        vector3D2 = this.seedPointIndex > 0 ? new Vector3D(this.points[this.seedPointIndex - 1], this.points[this.seedPointIndex]) : vector3D.negated();
        this.voxels = voxelList = new VoxelList(voxelArray, n5, d, d2, d3, vector3D, vector3D2);
        return this.voxels;
    }

    public String toString() {
        StringBuffer stringBuffer = new StringBuffer();
        stringBuffer.append("tractography.Tract\n" + this.numberOfPoints + " points:\nseed point index:" + this.seedPointIndex + "\n");
        for (int i = 0; i < this.numberOfPoints; ++i) {
            stringBuffer.append(this.points[i].x);
            stringBuffer.append("\t");
            stringBuffer.append(this.points[i].y);
            stringBuffer.append("\t");
            stringBuffer.append(this.points[i].z);
            stringBuffer.append("\n");
        }
        return stringBuffer.toString();
    }

    public Point3D getPoint(int n) {
        if (n >= 0 && n < this.numberOfPoints) {
            return this.points[n];
        }
        throw new IndexOutOfBoundsException("Point " + n + " does not exist");
    }

    public Point3D getSeedPoint() {
        return this.points[this.seedPointIndex];
    }

    public Point3D[] getPoints() {
        Point3D[] point3DArray = new Point3D[this.numberOfPoints];
        for (int i = 0; i < this.numberOfPoints; ++i) {
            point3DArray[i] = this.points[i];
        }
        return point3DArray;
    }

    public int seedPointIndex() {
        return this.seedPointIndex;
    }

    public Tract resample(double d) {
        if (!(d > 0.0)) {
            throw new LoggedException("Can't use step size of " + d);
        }
        Tract tract = new Tract(2 * this.numberOfPoints, 100.0);
        tract.addPoint(this.points[0]);
        for (int i = 1; i < this.numberOfPoints; ++i) {
            Vector3D vector3D = new Vector3D(this.points[i], this.points[i - 1]);
            double d2 = vector3D.mod();
            int n = (int)(d2 / d);
            for (int j = 0; j < n; ++j) {
                tract.addPoint(this.points[i - 1].displace(vector3D.scaled((double)(j + 1) * d / d2)));
            }
            if (d2 - (double)n * d > 0.001 || n == 0) {
                tract.addPoint(this.points[i]);
            }
            if (i != this.seedPointIndex) continue;
            tract.seedPointIndex = tract.numberOfPoints - 1;
        }
        return tract;
    }

    public void truncateToMaxPoints(int n) {
        if (this.numberOfPoints <= n) {
            return;
        }
        if (n == 0) {
            this.numberOfPoints = 0;
            return;
        }
        if (n < 0) {
            throw new IllegalArgumentException("can't truncate tract to " + n + " points");
        }
        int n2 = n / 2;
        int n3 = this.seedPointIndex - n2;
        int n4 = this.seedPointIndex + n2;
        if (n % 2 == 0) {
            ++n3;
        }
        if (n3 < 0) {
            n4 += -n3;
            n3 = 0;
        }
        if (n4 > this.numberOfPoints - 1) {
            n3 -= n4 - this.numberOfPoints + 1;
            n4 = this.numberOfPoints - 1;
        }
        if (n4 - n3 + 1 != n) {
            throw new LoggedException("Could not truncate tract to " + n + " points. Indices are " + n3 + " " + n4 + ". Tract is\n" + this.toString());
        }
        this.chop(n3, n4);
    }

    public void truncateToMaxLength(double d) {
        if (!(d > 0.0)) {
            throw new IllegalArgumentException("can't truncate tract to " + d + " mm");
        }
        if (!(this.length() > d)) {
            return;
        }
        int n = this.seedPointIndex;
        int n2 = this.seedPointIndex;
        double d2 = 0.0;
        double d3 = 0.0;
        double d4 = 0.0;
        if (n > 0 && d2 + this.displacements[n - 1] < d) {
            d2 += this.displacements[--n];
            d3 += this.displacements[n];
        }
        if (n2 < this.numberOfPoints - 1 && d2 + this.displacements[n2 + 1] < d) {
            d2 += this.displacements[++n2];
            d4 += this.displacements[n2];
        }
        boolean bl = true;
        while (d2 < d && bl) {
            double d5 = d2;
            if (d3 < d4) {
                if (n2 < this.numberOfPoints - 1 && d2 + this.displacements[n2 + 1] < d) {
                    d2 += this.displacements[++n2];
                    d4 += this.displacements[n2];
                }
                if (n > 0 && d2 + this.displacements[n - 1] < d) {
                    d2 += this.displacements[--n];
                    d3 += this.displacements[n];
                }
            } else {
                if (n > 0 && d2 + this.displacements[n - 1] < d) {
                    d2 += this.displacements[--n];
                    d3 += this.displacements[n];
                }
                if (n2 < this.numberOfPoints - 1 && d2 + this.displacements[n2 + 1] < d) {
                    d2 += this.displacements[++n2];
                    d4 += this.displacements[n2];
                }
            }
            bl = d2 > d5;
        }
        this.chop(n, n2);
    }

    public void chop(int n, int n2) {
        int n3;
        if (n > this.seedPointIndex || n2 >= this.numberOfPoints || n2 < this.seedPointIndex) {
            throw new LoggedException("invalid indices " + n + " " + n2);
        }
        this.voxels = null;
        this.voxelPathLengthsCalculated = false;
        int n4 = n2 - n + 1;
        for (n3 = 0; n3 < n4; ++n3) {
            this.points[n3] = this.points[n3 + n];
            this.displacements[n3] = this.displacements[n3 + n];
        }
        for (n3 = n4; n3 < this.numberOfPoints; ++n3) {
            this.points[n3] = null;
            this.displacements[n3] = 0.0;
        }
        this.numberOfPoints = n4;
        this.seedPointIndex -= n;
    }

    public void transform(RealMatrix realMatrix) {
        this.voxels = null;
        this.voxelPathLengthsCalculated = false;
        this.displacementsCalculated = false;
        Point3D[] point3DArray = new Point3D[this.numberOfPoints + 1];
        for (int i = 0; i < this.numberOfPoints; ++i) {
            point3DArray[i] = this.points[i].transform(realMatrix);
        }
        this.points = point3DArray;
    }

    public void writeRaw(DataOutputStream dataOutputStream) throws IOException {
        dataOutputStream.writeFloat(this.numberOfPoints);
        dataOutputStream.writeFloat(this.seedPointIndex);
        for (int i = 0; i < this.numberOfPoints; ++i) {
            dataOutputStream.writeFloat((float)this.points[i].x);
            dataOutputStream.writeFloat((float)this.points[i].y);
            dataOutputStream.writeFloat((float)this.points[i].z);
        }
    }

    public double[] toArray() {
        double[] dArray = new double[2 + 3 * this.numberOfPoints];
        dArray[0] = this.numberOfPoints;
        dArray[1] = this.seedPointIndex;
        for (int i = 0; i < this.numberOfPoints; ++i) {
            dArray[2 + i * 3] = this.points[i].x;
            dArray[3 + i * 3] = this.points[i].y;
            dArray[4 + i * 3] = this.points[i].z;
        }
        return dArray;
    }

    public boolean equals(Object object) {
        if (!(object instanceof Tract)) {
            return false;
        }
        if (object == null) {
            return false;
        }
        Tract tract = (Tract)object;
        if (tract == this) {
            return true;
        }
        if (tract.numberOfPoints != this.numberOfPoints) {
            return false;
        }
        if (tract.seedPointIndex != this.seedPointIndex) {
            return false;
        }
        for (int i = 0; i < this.numberOfPoints; ++i) {
            if (this.points[i].equals(tract.points[i])) continue;
            return false;
        }
        return true;
    }

    public int hashCode() {
        return 13 * (int)this.points[this.seedPointIndex].x + 29 * this.numberOfPoints + 37 * this.seedPointIndex;
    }

    public double[] getVoxelPathLengths(double[] dArray) {
        return this.getVoxelPathLengths(dArray[0], dArray[1], dArray[2]);
    }

    public double[] getVoxelPathLengths(double d, double d2, double d3) {
        double[] dArray;
        if (this.voxelPathLengthsCalculated && this.voxels != null) {
            dArray = this.voxels.getVoxelDims();
            if (d == dArray[0] && d2 == dArray[1] && d3 == dArray[2]) {
                double[] dArray2 = new double[this.voxelPathLengths.length];
                System.arraycopy(this.voxelPathLengths, 0, dArray2, 0, this.voxelPathLengths.length);
                return dArray2;
            }
            this.toVoxelList(d, d2, d3);
        } else if (this.voxels != null) {
            dArray = this.voxels.getVoxelDims();
            if (d != dArray[0] || d2 != dArray[1] || d3 != dArray[2]) {
                this.toVoxelList(d, d2, d3);
            }
        } else {
            this.toVoxelList(d, d2, d3);
        }
        dArray = new double[this.voxels.size()];
        int n = 0;
        Voxel voxel = this.voxels.getVoxel(0);
        int n2 = voxel.x;
        int n3 = voxel.y;
        int n4 = voxel.z;
        for (int i = 0; i < this.numberOfPoints - 1; ++i) {
            int n5 = (int)(this.points[i + 1].x / d);
            int n6 = (int)(this.points[i + 1].y / d2);
            int n7 = (int)(this.points[i + 1].z / d3);
            Vector3D vector3D = new Vector3D(this.points[i + 1], this.points[i]);
            double[] dArray3 = new double[6];
            double d4 = vector3D.mod();
            if (n2 != n5 || n3 != n6 || n4 != n7) {
                vector3D = vector3D.scaled(1.0 / d4);
                double d5 = d * (double)(1 + n2);
                double d6 = d * (double)n2;
                double d7 = d2 * (double)(1 + n3);
                double d8 = d2 * (double)n3;
                double d9 = d3 * (double)(1 + n4);
                double d10 = d3 * (double)n4;
                if (vector3D.x != 0.0) {
                    dArray3[0] = (d6 - this.points[i].x) / vector3D.x;
                    dArray3[1] = (d5 - this.points[i].x) / vector3D.x;
                } else {
                    dArray3[0] = Double.MAX_VALUE;
                    dArray3[1] = Double.MAX_VALUE;
                }
                if (vector3D.y != 0.0) {
                    dArray3[2] = (d7 - this.points[i].y) / vector3D.y;
                    dArray3[3] = (d8 - this.points[i].y) / vector3D.y;
                } else {
                    dArray3[2] = Double.MAX_VALUE;
                    dArray3[3] = Double.MAX_VALUE;
                }
                if (vector3D.z != 0.0) {
                    dArray3[4] = (d9 - this.points[i].z) / vector3D.z;
                    dArray3[5] = (d10 - this.points[i].z) / vector3D.z;
                } else {
                    dArray3[4] = Double.MAX_VALUE;
                    dArray3[5] = Double.MAX_VALUE;
                }
                Arrays.sort(dArray3);
                int n8 = 0;
                while (dArray3[n8] < 0.0) {
                    ++n8;
                }
                int n9 = n;
                dArray[n9] = dArray[n9] + dArray3[n8];
                double d11 = d4 - dArray3[n8];
                int n10 = n + 1;
                dArray[n10] = dArray[n10] + (d11 > 0.0 ? d11 : 0.0);
                ++n;
                n2 = n5;
                n3 = n6;
                n4 = n7;
                continue;
            }
            int n11 = n;
            dArray[n11] = dArray[n11] + d4;
        }
        this.voxelPathLengths = dArray;
        this.voxelPathLengthsCalculated = true;
        double[] dArray4 = new double[this.voxelPathLengths.length];
        System.arraycopy(this.voxelPathLengths, 0, dArray4, 0, this.voxelPathLengths.length);
        return dArray4;
    }

    public final void transformToPhysicalSpace(RealMatrix realMatrix, double d, double d2, double d3) {
        this.voxels = null;
        this.voxelPathLengthsCalculated = false;
        this.displacementsCalculated = false;
        for (int i = 0; i < this.numberOfPoints; ++i) {
            Point3D point3D = new Point3D(this.points[i].x / d - 0.5, this.points[i].y / d2 - 0.5, this.points[i].z / d3 - 0.5);
            this.points[i] = point3D.transform(realMatrix);
        }
    }

    public final void transformToCaminoSpace(RealMatrix realMatrix, double d, double d2, double d3) {
        this.voxels = null;
        this.voxelPathLengthsCalculated = false;
        this.displacementsCalculated = false;
        for (int i = 0; i < this.numberOfPoints; ++i) {
            Point3D point3D = this.points[i].transform(realMatrix);
            this.points[i] = new Point3D((point3D.x + 0.5) * d, (point3D.y + 0.5) * d2, (point3D.z + 0.5) * d3);
        }
    }
}

