package edu.masi.hyperadvisor;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;

import util.OutputType;

import cart.CART;

import edu.masi.hyperadvisor.algorithms.KNN;
import edu.masi.hyperadvisor.structures.Algo;
import edu.masi.hyperadvisor.structures.AlgoCollection;
import edu.masi.hyperadvisor.util.Log;

/**
 * This is the Service Class that Axis2 accesses.
 * When testing outside of Axis2, the main class should be here.
 * It makes sure the database is accessible, and handles calls to aggregate and estimate
 * 
 * @author covingkj
 *
 */
public class Hyperadvisor {	
	
	private int REQUIRED_SAMPLES 	 = 10;
	private String DATABASE_ADDR 	 = "jdbc:mysql://127.0.0.1:3306/hypervisor";
	private String DATABASE_USERNAME = "hypervisor_user";
	private String DATABASE_PASSWORD = "hypervisor_pass";
	private String DATABASE_NAME	 = "hypervisor";
    private int[] dummy = {60000,42};

	/**
	 * Gets the MD5 hash value of the algorithm name,
	 * @param algoName	the algorithm's name
	 * @return			the MD5 hash of algoName
	 */
	private String getHash(String algoName){
		MessageDigest md;
	    byte[] hash = null;
		try {
			md 		= MessageDigest.getInstance("MD5");
			hash 	= md.digest(algoName.getBytes("UTF-8"));
		} catch (NoSuchAlgorithmException e) {
			Log.writeToLog("Error in getting MD5 hash algorithm.");
			Log.writeToLog(e.getMessage());
			
			return null;
		} catch (UnsupportedEncodingException e) {
			Log.writeToLog("Error in getting UTF-8 byte encoding for hash.");
			Log.writeToLog(e.getMessage());
			
			return null;
		}
		
		return String.format("%1$032X", new BigInteger(1,hash));
	}
	
	/**
	 * Make sure that the database exists. 
	 * This may not be useful due to database permissions, but is useful to know and log immediately.
	 * 
	 * @return				true if no exceptions are thrown when connecting the database and making sure it has been created
	 */
	private boolean setupDatabase()
	{		
		FileReader ris;
		try {
			ris = new FileReader(System.getProperty("user.home") + "/hypervisor_preferences.txt");
		    BufferedReader reader = new BufferedReader(ris);
	    	String text = "";
	    	
	        //read in preferences and set the appropriate variables
	        while((text = reader.readLine()) != null){
	        	String [] fieldAndValuePair = text.split("[ ]*=[ ]*");
	        	if(fieldAndValuePair.length == 2){
	        		if(fieldAndValuePair[0].equalsIgnoreCase("DATABASE_ADDR")){
	        			DATABASE_ADDR = fieldAndValuePair[1];
	        		}
	        		if(fieldAndValuePair[0].equalsIgnoreCase("DATABASE_USERNAME")){
	        			DATABASE_USERNAME = fieldAndValuePair[1];
	        		}
	        		if(fieldAndValuePair[0].equalsIgnoreCase("DATABASE_PASSWORD")){
	        			DATABASE_PASSWORD = fieldAndValuePair[1];
	        		}
	        		if(fieldAndValuePair[0].equalsIgnoreCase("DATABASE_NAME")){
	        			DATABASE_NAME = fieldAndValuePair[1];
	        		}
	        	}
	        }
	    
	        //Do nothing, since their are default options that may still work
		} catch (FileNotFoundException e1) {
	
		} catch (IOException e) { }
		
		Connection connection = null;
	    Statement statement = null;
	    try {
	      Class.forName("com.mysql.jdbc.Driver").newInstance();
	      connection = DriverManager.getConnection(DATABASE_ADDR, DATABASE_USERNAME, DATABASE_PASSWORD);

	      statement = connection.createStatement();
	      statement.execute("CREATE DATABASE IF NOT EXISTS " + DATABASE_NAME);
	      statement.execute("USE " + DATABASE_NAME);
	      statement.execute("CREATE TABLE IF NOT EXISTS module_name_lookup(hash VARCHAR(255) NOT NULL, name VARCHAR(255) NOT NULL, UNIQUE (hash, name));");
	    } catch (Exception e) {
	    	Log.writeToLog("Either database or lookup table was not configured successfully.");
	    	Log.writeToLog(e.getMessage());
	    	
	    	return false;
	    } finally {
		      if (statement != null) {
		        try {
		        	statement.close();
		        } catch (SQLException e) {
		        	return false;
		        }
		      }
	      
		      if (connection != null) {
		        try {
		        	connection.close();
		        } catch (SQLException e) {
		        	return false;
		        } 
		      }
	    }
		return true;
	}
	
	/** 
	 * Updates the database with a new instance of a particular algorithm execution
	 * 
	 * @param alg			algorithm information, specifically name, input and output parameters
	 * @return true if database update seems successful, false if some exception is encountered
	 */
	private boolean updateRoutine(Algo alg,String machineID,String isUseGrid){		
		Log.writeToLog("Update Called");
		Connection connection = null;
	    Statement statement = null;
	    String insertionCommand = "";
	    
	    String safeAlgName = getHash(alg.algName);
	    if(safeAlgName == null){
	    	return false;
	    }
	    
	    try {
	      Class.forName("com.mysql.jdbc.Driver").newInstance();
	      connection = DriverManager.getConnection(DATABASE_ADDR, DATABASE_USERNAME, DATABASE_PASSWORD);

	      statement = connection.createStatement();
	      
	      statement.executeUpdate("INSERT IGNORE INTO module_name_lookup (hash,name) VALUES('" + safeAlgName + "','" + alg.algName + "');");
	      
	      //Add input parameters to the table listing
	      String databaseVals = safeAlgName + " (";
	      for (int i = 0; i < alg.inputs.size(); i++){
	    	  databaseVals += "input" + i + " VARCHAR(30), ";
	      }
	      databaseVals += "machineID varchar(60),";
	      databaseVals += "usingGrid varchar(10),";
	      //Add output parameters to the table listing
	      databaseVals += "usedTime INT, ";
	      databaseVals += "usedMem INT, ";
	      databaseVals += "allocTime INT, ";
	      databaseVals += "allocMem INT";
	      
	      databaseVals += ");";
	      
	      statement.executeUpdate("CREATE TABLE IF NOT EXISTS " + databaseVals);
	      
	      String columnVals 	= "(";
	      databaseVals 	= "VALUES(";
	      for (int i = 0; i < alg.inputs.size(); i++){
	    	  columnVals 	+= "input" + i + ", ";
	    	  String data = alg.inputs.get(i).toString();
	    	  databaseVals 	+= "\"" + data.subSequence(1, data.length()-1) + "\", ";
	      }
	      columnVals += "machineID,";
	      columnVals += "usingGrid,";
	      columnVals += "usedTime, ";
	      columnVals += "usedMem, ";
	      columnVals += "allocTime, ";
	      columnVals += "allocMem";
	      
	      databaseVals += "\""+machineID+"\",";
	      if(isUseGrid.equalsIgnoreCase("true"))
	    	  databaseVals+= "\"true\",";
	      else
	    	  databaseVals+="\"false\",";
	      databaseVals += alg.usedTime + ", ";
	      databaseVals += alg.usedMem + ", ";
	      databaseVals += alg.allocTime + ", ";
	      databaseVals += alg.allocMem;
	      
	      columnVals 	+= ")";
	      databaseVals 	+= ");";
	      
	      insertionCommand = "INSERT INTO " + safeAlgName + " "  + columnVals + " " + databaseVals;
	      statement.executeUpdate(insertionCommand);
	    } catch (Exception e) {
	    	Log.writeToLog("Error in either table creation or in data insertion.");
	    	Log.writeToLog(e.getMessage());
	    	return false;
	    } finally {
		      if (statement != null) {
		        try {
		        	statement.close();
		        } catch (SQLException e) {
		        	return false;
		        }
		      }
	      
		      if (connection != null) {
		        try {
		        	connection.close();
		        } catch (SQLException e) {
		        	return false;
		        } 
		      }
	    }
		return true;
	}
	
	/** Gets the recommended wall time and memory estimates based on previous executions
	 * 
	 * @param alg			algorithm information, specifically name, input and class parameters
	 * @param threshold		the desired probability of success for the proposed algorithm
	 * @return int[2] where the first value is wall time and the second is memory 
	 */
	private int [] estimationRoutine(Algo alg, float threshold,String machineID,String isUseGrid)
	{
		Connection connection = null;
	    Statement statement = null;
	    
	    String safeAlgName = getHash(alg.algName);
	    if(safeAlgName == null){
	    	return dummy;
	    }
	    
	    try {
	      Class.forName("com.mysql.jdbc.Driver").newInstance();
	      connection = DriverManager.getConnection(DATABASE_ADDR, DATABASE_USERNAME, DATABASE_PASSWORD);

	      statement = connection.createStatement();
	      
	      //Add input parameters to the table listing
	      String databaseVals = safeAlgName + " (";
	      for (int i = 0; i < alg.inputs.size(); i++){
	    	  databaseVals += "input" + i + " VARCHAR(30), ";
	      }
	      databaseVals += "machineID VARCHAR(60),";
	      databaseVals += "usingGrid varchar(10),";
	      
	      //Add output parameters to the table listing
	      databaseVals += "usedTime INT, ";
	      databaseVals += "usedMem INT, ";
	      databaseVals += "allocTime INT, ";
	      databaseVals += "allocMem INT";	      
	      
	      databaseVals += ");";
	      
	      statement.executeUpdate("CREATE TABLE IF NOT EXISTS " + databaseVals);
	      
	      ResultSet res = statement.executeQuery("SELECT * FROM " + safeAlgName + ";");
	      
	      ArrayList<ArrayList<ArrayList<String>>> prevIns = new  ArrayList<ArrayList<ArrayList<String>>>();
	      ArrayList<ArrayList<Integer>> prevOuts = new ArrayList<ArrayList<Integer>>();
	      
	      int inputEnd = res.findColumn("machineID")-1;//res.findColumn("usedTime")-1;
	      int count = 0;
	      int max2d = 0;
	      int max3d = 0;
	      while(res.next()){
	    	  prevIns.add(new ArrayList<ArrayList<String>>());
	    	  prevOuts.add(new ArrayList<Integer>());
		      for(int i = 0; i < inputEnd; i++){
		    	  prevIns.get(count).add(new ArrayList<String>());
		    	  
		    	  String vals = res.getString("input" + i);
		    	  prevIns.get(count).get(i).addAll(Arrays.asList(vals.split(",")));
		    	  
		    	  if(prevIns.get(count).size() > max2d){
		    		  max2d = prevIns.get(count).size();
		    	  }
		    	  if(prevIns.get(count).get(i).size() > max3d){
		    		  max3d = prevIns.get(count).get(i).size();
		    	  }
		      }
		      for(int i = inputEnd+2; i < inputEnd+6; i++){
		    	  prevOuts.get(count).add(res.getInt(i+1));
		      }
		      count++;
	      }
	      
	      int [][] outsAsArray = new int[prevOuts.size()][4];
	      for(int i = 0; i < prevOuts.size(); i++){
	    	  for(int j = 0; j < 4; j++){
	    		  outsAsArray[i][j] = prevOuts.get(i).get(j);
	    	  }
	      }
	      
	      String [][][] prevInsAsArray = new String[prevIns.size()][max2d][max3d];
	      for(int i = 0; i < prevIns.size(); i++){
	    	  for(int j = 0; j < max2d; j++){
	    		  for(int k = 0; k < max3d; k++){
	    			  if(prevIns.get(i).get(j).size() > k){
	    				  prevInsAsArray[i][j][k] = prevIns.get(i).get(j).get(k);
	    			  } else {
	    				  prevInsAsArray[i][j][k] = null;
	    			  }
	    		  }
	    	  }
	      }
	      
	      AlgoCollection algC = new AlgoCollection(alg.algName, prevInsAsArray, alg.classes.toArray(new String [0]), outsAsArray);
	      
	      //Don't try to estimate when there isn't enough data available

	      if(prevIns.size() >= REQUIRED_SAMPLES){
	    	  Log.writeToLog("Got Here");
	    	  return KNN.getEstimate(alg, algC, (int)Math.sqrt(prevIns.size()), threshold);
	    	  //return KernelDensity.getEstimate(alg, algC);
	    	  //return dummy;
	      } else {
	    	  return dummy;
	      }
	      
	    } catch (Exception e) {
			try {
				//FileWriter fstream  = new FileWriter(new File(System.getProperty("user.home") + "/hypervisor_log2.txt"), true);
				//BufferedWriter out = new BufferedWriter(fstream);
				FileOutputStream fos = new FileOutputStream(new File(System.getProperty("user.home") + "/hypervisor_errorLog.txt"));  
				PrintStream ps = new PrintStream(fos);  
				e.printStackTrace(ps);
				ps.flush();
				ps.close();
				//e.printStackTrace(out);
				//out.flush();
				//out.close();
			} catch (IOException a) { }
	    	Log.writeToLog("e.writeMessage Was supposed to be here");
	    	return dummy;
	    } finally {
		      if (statement != null) {
		        try {
		        	statement.close();
		        } catch (SQLException e) {
		        	return dummy;
		        }
		      }
	      
		      if (connection != null) {
		        try {
		        	connection.close();
		        } catch (SQLException e) {
		        	return dummy;
		        } 
		      }
	    }
	}
			
	/**
	 * Adds the inputs and results of a particular algorithm excecution to the database specified
	 * 
	 * @param algoName		the name of the algorithm to be updated
	 * @param inputValues	the input data given to this execution of the algorithm
	 * @param outputValues	the memory and time resources used by this execution of the algorithm
	 * @return				true if the inputs and outputs were successfully logged to the database
	 */
	public boolean aggregateNewData(String algoName, String [][] inputValues, int [] outputValues,String machineID,String isUseGrid) {	
		
		//If there's nothing wrong with our access to the database...
		if(setupDatabase()){			
			try {

				//The algo class maintains this algorithm instance's input and output values
					//Catch any exceptions here so that the correct status is returned to the user
				Algo alg = new Algo(algoName, inputValues, outputValues);
				System.gc();
			    updateRoutine(alg,machineID,isUseGrid);

			//If there's an exception, the algorithm simply returns and does not aggregate
				//so exceptions do not need to be explicitly handled
			} catch (Exception e) {
				Log.writeToLog("Error in initialization of Algo object. Returning false to caller.");
				Log.writeToLog(e.getMessage());
				System.gc();
				return false;
			}
			
			String shortName = algoName.substring(algoName.lastIndexOf(".")+1);
			Log.writeToLog(shortName);
			CART CPUTree = null;
			CART memTree = null;
			try
			{
				CPUTree = CART.getCARTFromFile(shortName, OutputType.CPU_TYPE);

				memTree = CART.getCARTFromFile(shortName, OutputType.MEM_TYPE);
				Algo alg = new Algo(algoName, inputValues, outputValues);
				String[] cartInputs = new String[alg.inputs.size()];
				for (int i = 0; i < alg.inputs.size(); i++){
					String data = alg.inputs.get(i).toString();
					cartInputs[i] = (String) data.subSequence(1, data.length()-1);
				}
				CPUTree.updateValues(cartInputs, machineID, isUseGrid, outputValues[0]);
				memTree.updateValues(cartInputs,machineID,isUseGrid,outputValues[1]);
				writeTreeToFile(CPUTree, shortName+"_CPUTIME");
				writeTreeToFile(memTree, shortName+"_MEMUSED");
				Log.writeToLog("update success");
				return true;

			}
			catch(Exception e){Log.writeToLog(e.getMessage());}
			Log.writeToLog("Other Fail");
			
			return false;
			
			
		}
		System.gc();
		//Database is incorrectly configured
		return false;
	}
    private void writeTreeToFile(CART tree, String title)
    {
        //String gsonString = gson.toJson(tree);
        try{
           // FileWriter fstream = 
          //          new FileWriter(OUTPUT_TRAINING_DIR
          //          +title+".txt");
         //   BufferedWriter out = new BufferedWriter(fstream);
         //   out.write(gsonString);
         //   out.close();    
            //System.out.println(title+".txt written");
            
            
            FileOutputStream f_out = new 
                    FileOutputStream(System.getProperty("user.home")+"/trainingFiles/"
                    +title+".obj");

            // Write object with ObjectOutputStream
            ObjectOutputStream obj_out = new
                    ObjectOutputStream (f_out);
            obj_out.writeObject (tree);
            obj_out.close();
        }
        catch(IOException e)
        {
            e.printStackTrace();
        }
    }  
	/**
	 * Based on the inputs to a specific algorithm, retrieves time and memory consumption estimates with a KNN implementation
	 * 
	 * @param algoName		the name of the algorithm to be updated
	 * @param inputs	the input data given to this execution of the algorithm
	 * @param inputClasses	the JIST parameter type of each input, which determines how difference is measured
	 * @return				If successful, int[2] where index 0 is memory and index 1 is run time. Otherwise returns null.
	 */
	public int [] getEstimates(String algoName, String [][] inputs, String [] inputClasses, String machineID, String isUseGrid,float cpuThreshold, float memThreshold)
	{
		Log.writeToLog("Estimation Called");
		/*Log.writeToLog("InputClasses:");
		for(String s: inputClasses)
		{
			Log.writeToLog("	"+s);
		}
		Log.writeToLog("Inputs:");
		for(String[]ins:inputs)
		{
			for(String s:ins)
			{
				Log.writeToLog("	"+s);
			}
			Log.writeToLog("New Line");
		}*/
		String shortName = algoName.substring(algoName.lastIndexOf(".")+1);
		Log.writeToLog(shortName);
		CART CPUTree = null;
		CART memTree = null;
		try
		{
			CPUTree = CART.getCARTFromFile(shortName, OutputType.CPU_TYPE);

			memTree = CART.getCARTFromFile(shortName, OutputType.MEM_TYPE);
			Algo alg = new Algo(algoName, inputs, inputClasses);
			String[] cartInputs = new String[alg.inputs.size()];
			for (int i = 0; i < alg.inputs.size(); i++){
				String data = alg.inputs.get(i).toString();
				cartInputs[i] = (String) data.subSequence(1, data.length()-1);
			}
			double cpuEstimate = CPUTree.getEstimate(cartInputs,machineID,isUseGrid,(double)cpuThreshold);
			double memEstimate = memTree.getEstimate(cartInputs,machineID,isUseGrid,(double)memThreshold);
			int[] returnValue = new int[2];
			returnValue[0] = (int)cpuEstimate;
			returnValue[1] = (int)memEstimate;
			Log.writeToLog(returnValue[0]+"");
			return returnValue;

		}
		catch(Exception e){Log.writeToLog(e.getMessage());}
		Log.writeToLog("Other Fail");
		return dummy;
		/*//If there's nothing wrong with our access to the database...
		if(setupDatabase()){
			
			//AlgoCollection retrieves all previous data and constructs estimates using KNN
			Algo alg = new Algo(algoName, inputs, inputClasses);
			System.gc();
			return estimationRoutine(alg, threshold,machineID,isUseGrid);
		}
		
		//Database in incorrectly configured
		System.gc();
		return dummy;*/
	}
	 
	//For testing purposes only...
	/*public static void main(String [] args){
		Hyperadvisor test = new Hyperadvisor();
		
		String [][] inputs = new String[2][1];
		inputs[0][0] = "17039712";
		inputs[1][0] = "100.0";
		int [] outputs = new int[4];
		outputs[0] = 1;
		outputs[1] = 2;
		outputs[2] = 3;
		outputs[3] = 4;
		String [] classes = new String[2];
		classes[0] = "ParamVolume";
		classes[1] = "ParamNumber";
		
		@SuppressWarnings("unused")
		int [] res = test.getEstimates("edu.jhu.ece.iacl.plugins.segmentation.CreateMask", inputs, classes, .5f);
		//test.aggregateNewData("edu.jhu.ece.iacl.plugins.segmentation.CreateMask", inputs, outputs);
	}*/
		
}
