package edu.jhu.ece.iacl.jist.cli;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.helpers.XMLReaderFactory;

import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.JistPreferences;
import edu.jhu.ece.iacl.jist.pipeline.PipeAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.PipePort.type;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.JISTInternalParam;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamFile.DialogType;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamPerformance;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataIntent;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import gov.nih.mipav.model.scripting.ParserException;

@SuppressWarnings("rawtypes")
public class JistCLI {

	Options parserOptions;
	ProcessingAlgorithm module;
	PipeAlgorithm pipe;
	public CommandLine cliCommand;
	static final String cvsversion = "$Revision: 1.28 $".replace("Revision: ", "").replace("$", "").replace(" ", "");
	static String VERSION = cvsversion;
	LinkedHashMap<String, ParamModel> paramList;

	public void addOption(String opt, boolean hasArg, String description) {
		parserOptions.addOption(opt, hasArg, description);
	}

	public String getOptionValue(String key) {

		String value = cliCommand.getOptionValue(key);
		if(value==null && cliCommand.hasOption(key))
			return "set";
		return value;

	}

	public static String uniqueString(String suggestedTag,HashMap<String, ParamModel> tags, String base) {
		String []split = suggestedTag.trim().split("\\s");
		String tag = base;
		int i=0;
		if(split.length>0 && split[0].length()>0) {
			tag = split[0];
			i++;
		}

		String curtag = tag;
		while(null!=tags.get(curtag)) {
			i=i+1;
			curtag = tag+i;
		}
		return curtag;
	}

	public JistCLI(ProcessingAlgorithm module) {
		this.module = module;
		parserOptions = new Options();
		paramList = new LinkedHashMap<String,ParamModel>();

		// parse and add the input arguments
		ParamCollection inputs = module.getInput();
		parseParamCollection((ParamCollection)inputs,paramList,parserOptions,"in",true);

		// parse and add the output arguments
		ParamCollection outA = module.getOutput();
		ParamCollection outputs = new ParamCollection();
		// Script parser sets the last input param to the output params. Need to clear
		for(int i=0;i<outA.size();i++) {
			ParamModel p = outA.getValue(i);
			if (p.getName().equals("Execution Time") && (p instanceof ParamPerformance))
				continue;
			
			// fix any empty descriptions
			if (p.getDescription().equals("null")) {
				p.setDescription(p.getLabel());
			}
			outputs.add(p);
		}
		parseParamCollection((ParamCollection)outputs,paramList,parserOptions,"out",false);

		// add the additional preference options
		addPreferenceOptions();
	}

	public JistCLI(PipeAlgorithm pipe) {
		this.pipe = pipe;
		parserOptions = new Options();
		paramList = new LinkedHashMap<String,ParamModel>();

		// parse and add the input arguments
		ParamCollection inA= pipe.getInputParams();
		ParamCollection inputs = new ParamCollection();
		// Script parser sets the last input param to the output params. Need to clear
		for(int i=0;i<inA.size()-1;i++) {
			ParamModel p = inA.getValue(i);
			if(p.getName().equals("Use Layout Output Directory") && (p instanceof ParamBoolean))
				continue;
			if(p.getName().equals("Show Process Manager") && (p instanceof ParamBoolean))
				continue;
			
			// fix any empty descriptions
			if (p.getDescription().equals("null")) {
				p.setDescription(p.getLabel());
			}
			inputs.add(p);
		
		}
		parseParamCollection((ParamCollection)inputs,paramList,parserOptions,"in",true);
		
		// parse and add the output arguments
		ParamCollection outA = (ParamCollection) inA.getValue(inA.size()-1);
		ParamCollection outputs = new ParamCollection();
		// Script parser sets the last input param to the output params. Need to clear
		for(int i=0;i<outA.size();i++) {
			ParamModel p = outA.getValue(i);
			if (p.getName().equals("Execution Time") && (p instanceof ParamPerformance))
				continue;
			
			// fix any empty descriptions
			if (p.getDescription().equals("null")) {
				p.setDescription(p.getLabel());
			}
			outputs.add(p);
		}
		parseParamCollection((ParamCollection)outputs,paramList,parserOptions,"out",false);

		// add the additional preference options
		addPreferenceOptions();
	}

	private void addPreferenceOptions(){
		
		// add option for the processing directory
		ParamFile procdir = new ParamFile("Processing Directory (default current)",DialogType.DIRECTORY);
		procdir.setShortLabel("Proc");
		procdir.setMandatory(false);
		parserOptions.addOption("xDir", true, procdir.getDescription()+" ["+procdir.getHumanReadableDataType()+"] (optional)");
		
		// add option for the XML summary file
		ParamFile outfile = new ParamFile("XML Summary File (default: None)",DialogType.FILE);
		outfile.setShortLabel("XMLSummary");
		outfile.setMandatory(false);
		parserOptions.addOption("xFile", true, outfile.getDescription()+" ["+outfile.getHumanReadableDataType()+"] (optional)");
		
		parserOptions.addOption("h","help",false,"Print this message.");
		parserOptions.addOption("xml",false,"Prints an xml that is compatible as a 3D Slicer Module");
		parserOptions.addOption("xPrefExt",true,"Set Prefered Output Image Extension");
		parserOptions.addOption("xDefaultMem",true,"Set Set default maximum heap size");
		parserOptions.addOption("xDebugLvl",true,"Set Debug output level[0-9]: 0 = Error messages only, 1 = Major events and warnings, ect. Lvl 5+ maybe cause performance decrease.");
		parserOptions.addOption("xJreLoc",true,"Set location of JRE to use");
		parserOptions.addOption("xMaxProcess",true,"Set default maximum number of processes.");
		parserOptions.addOption("xUseGridEng",true,"Sets the use of the Grid Engine[true/false]");
		//Shunxing edit
		parserOptions.addOption("xUseAwsEC2",true,"Sets the use of the Aws EC2 cloud[true/false]");
		//Shunxing edit
		//parserOptions.addOption("xSavePrefChanges",false,"Saves any changes made to the JIST preference");
	}

	private static void parseParamCollection(ParamCollection inputs,
											 LinkedHashMap<String, ParamModel> paramList,
											 Options parserOptions,
											 String preTag,
											 boolean isInput) {
		
		
		
		for(int i=0;i<inputs.size();i++) {
			ParamModel input = inputs.getValue(i);
			if(!input.isHidden() && !(input instanceof JISTInternalParam)) {

				String tag = uniqueString(input.getShortLabel(),paramList,preTag).replace("-", "");
				paramList.put(tag, input);

				String req ="";
				if(input.isMandatory()) {
					String val = input.probeDefaultValue();
					if(isInput) { 
						if(val==null)
							req = "(required)";
						else
							req = "(default="+val+")";
					}
				} else {
					req = "(optional)";
				}
				parserOptions.addOption(preTag+tag, true, input.getDescription()+" ["+input.getHumanReadableDataType()+"] "+req);
				input.setCliTag(preTag+tag);
				input.setPortType((!isInput?type.OUTPUT:type.INPUT));
			}
			if(input instanceof ParamCollection) {
				parseParamCollection((ParamCollection)input,paramList,parserOptions,preTag,isInput);
			}
		}

	}

	public String getHumanReadableHelpMessage() {

		StringWriter sw = new StringWriter();
		PrintWriter io = new PrintWriter(sw);
		// automatically generate the help statement
		HelpFormatter formatter = new HelpFormatter();
		formatter.setOptionComparator(new OptionComparator());
		
		AlgorithmInformation info=null;
		if(module!=null)
			info = module.getAlgorithmInformation();
		else 
			info = pipe.getAlgorithm().getAlgorithmInformation();
		formatter.printHelp(io,
				HelpFormatter.DEFAULT_WIDTH,
				(module==null?"layoutfile":module.getClass().getCanonicalName()),
				"\n" +
				info.getLabel()+" "+info.getVersion()+" "+info.getStatusString()+"\n" +
				info.getDescription()+"\n" +
				"\n",
				parserOptions,
				HelpFormatter.DEFAULT_LEFT_PAD,HelpFormatter.DEFAULT_DESC_PAD,
				"\n" +
				"Provided by: JIST (Java Image Science Toolkit) Command Line Interface version " + JistCLI.VERSION +
				"\n" +
				"http://www.nitrc.org/projects/jist/"
				,true); 		
		return sw.toString();
	}

	//Stolen from: http://www.captaindebug.com/2011/04/checking-for-well-formed-xml.html#.VHPLE9Yu4d8
	
	  private void validate(String xml) throws SAXException, IOException {
	    XMLReader parser = XMLReaderFactory.createXMLReader();
	    parser.setContentHandler(new DefaultHandler());
	    InputSource source = new InputSource(new ByteArrayInputStream(xml.getBytes()));
	    parser.parse(source);
	  }

	
	public String getSlicerXML(){
		String XML=tagWithXML("Developer Tools","category",true);

		AlgorithmInformation info = module.getAlgorithmInformation();
		XML = XML + tagWithXML(info.getLabel(),"title",false)+"\n";
		String Citations = "";

		for(int i = 0; i< info.getCitations().size();i++){
			Citations = Citations + info.getCitations().get(i).getText()+"\n";
		}
		XML = XML + tagWithXML(info.getDescription()+"\n"+Citations,"description",true);
		XML = XML + tagWithXML(info.getVersion()+"."+info.getStatusString(),"version",false)+"\n";

		String Authors = "";
		for(int i = 0; i< info.getAuthors().size();i++){
			Authors = Authors + info.getAuthors().get(i).toString()+"\n";
		}
		XML = XML + tagWithXML(Authors, "contributor", true);
		XML = XML + tagWithXML(info.getWebsite(),"documentation-url",true);

		ParamCollection inputs = module.getInput();
		ParamCollection outputs = module.getOutput();

		//write inputs and outputs
		XML = XML + slicerXMLaddParamCollection(inputs, true);
		XML = XML + slicerXMLaddParamCollection(outputs, false);

		//add final tags
		XML = "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n" + tagWithXML(XML, "executable", true);
		
		//Try to validate the XML to see if it is well-formed
		try {
			validate(XML);
		} catch (SAXException e) {
			JistLogger.logOutput(JistLogger.SEVERE,"ERROR: XML is not well-formed");
			JistLogger.logError(JistLogger.SEVERE, e.toString());
		} catch (IOException e) {
			JistLogger.logOutput(JistLogger.SEVERE, "Caught IOException");
			JistLogger.logError(JistLogger.SEVERE, e.toString());
			e.printStackTrace();
		}
		
		return XML;
	}

	public String slicerXMLaddParamCollection(ParamCollection inputs, boolean isInput){
		boolean paramFlag = false;
		boolean firstParam = true;
		String XML="";
		if(isInput){//Add memory parameter to inputs
			XML = XML + "<parameters advanced=\"true\">\n";
			XML = XML + "<label>Memory Allocation</label>\n";
			XML = XML + "<description>Amount of Memory Allowed for Algorithm Execution</description>\n";
			XML = XML + "<integer>\n";
			XML = XML + "<longflag>--maxMemoryUsage</longflag>\n";
			XML = XML + "<label>Max Memory (in MB)</label>\n";
			XML = XML + "<description>Maximum Memory Allowed (in MegaBytes). Increase or decrease this depending on java virtual machine heap size requirements.</description>\n";
			XML = XML + "<default>1400</default>\n";
			XML = XML + "</integer>\n";
			XML = XML + "</parameters>\n";
		}
		for(int i = 0; i < inputs.size();i++){
			ParamModel inParam = inputs.getValue(i);

			if(inParam instanceof ParamCollection){

				//Close a parameter tag if one is open
				if(paramFlag){
					XML = XML + "</parameters>\n";
					paramFlag = false;
				}

				if(firstParam){
					XML = XML + "<parameters>\n";	
					firstParam=false;
				}
				else XML = XML + "<parameters advanced=\"true\">\n";

				XML = XML + tagWithXML(inParam.getName(), "label",false)+"\n";
				XML = XML + tagWithXML(inParam.getDescription(), "description",false)+"\n";
				XML = XML + slicerXMLparseCollection((ParamCollection)inParam, isInput);
				XML = XML + "</parameters>\n";
			}
			else if(!inParam.isHidden() && !(inParam instanceof JISTInternalParam)){
				//Start a parameter tag if one isn't open
				if(!paramFlag){

					if(firstParam){
						XML = XML + "<parameters>\n";	
						firstParam=false;
					}
					else XML = XML + "<parameters advanced=\"true\">\n";

					if(isInput){ 
						XML = XML + "<label>Main</label>\n";
						XML = XML + "<description>Main Parameters</description>\n";
					}
					else {
						XML = XML + "<label>Output</label>\n";
						XML = XML + "<description>Output Parameters</description>\n";
						XML = XML + "<string-enumeration>\n";
						XML = XML + "<label>Output File Type</label>\n";
						XML = XML + "<description>Output File Type</description>\n";
						XML = XML + "<longflag>--xPrefExt</longflag>\n";
						XML = XML + "<element>nrrd</element>\n";
						XML = XML + "<default>nrrd</default>\n";
						XML = XML + "</string-enumeration>\n";


					}
					paramFlag = true;
				}
				XML = XML + slicerXMLaddParam(inParam, isInput);
			}

		}
		//Close a parameter tag if one is open
		if(paramFlag){
			XML = XML + "</parameters>\n";
			paramFlag = false;
		}

		return XML;
	}

	public String slicerXMLparseCollection(ParamCollection inCollect, boolean isInput){

		String XML="";

		for(int i = 0; i < inCollect.size(); i++){
			ParamModel inParam = inCollect.getValue(i);
			if( inParam instanceof ParamCollection ){
				XML = XML + slicerXMLparseCollection((ParamCollection)inParam, isInput);
			}
			else if(!inParam.isHidden() && !(inParam instanceof JISTInternalParam)){
				XML = XML + slicerXMLaddParam(inParam, isInput);
			}
		}

		return XML;
	}

	public String slicerXMLaddParam(ParamModel inParam, boolean isInput){
		String XML = "";

		XML = XML + tagWithXML("--"+inParam.getCliTag(), "longflag", false) +"\n";
		XML = XML + tagWithXML(inParam.getName(), "label", false)+"\n";
		//XML = XML + tagWith(inParam.getDescription(), "description",false);

		String req ="";
		if(isInput) { 
			/*
	    if(inParam.isMandatory()) {

			req = "(required)";
		} else {
			req = "(optional)";
		}*/
		}		
		XML = XML + tagWithXML(inParam.getDescription() + req, "description",false);

		String paramType = inParam.getHumanReadableDataType();

		if(paramType.equals("file")){
			if(isInput)	XML = XML + "\n" + tagWithXML("input","channel",false);
			else XML = XML + "\n" + tagWithXML("output","channel",false);

			if(inParam instanceof ParamVolume){
				paramType = "image";
				if(((ParamVolume)inParam).getDataIntent() == ImageDataIntent.NIFTI_INTENT_LABEL)
					paramType = paramType + " type=\"label\"";
			}
			
		}else if(paramType.contains("option")){
			String optionChoices[] = paramType.split(":|\\|");// look for ":" or "|" symbol as dividers
			for(int k = 1; k < optionChoices.length; k++){ //starts at 1 because first is always "option"
				XML = XML + "\n" + tagWithXML("\""+optionChoices[k]+"\"","element",false);
			}
			paramType = "string-enumeration";
		}else if(paramType.equals("boolean")){
			XML = XML + "\n" + tagWithXML("\"true\"","element",false);
			XML = XML + "\n" + tagWithXML("\"false\"","element",false);
			paramType = "string-enumeration";
			
		//fix for https://www.nitrc.org/tracker/index.php?func=detail&aid=7235&group_id=228&atid=942
		} else if(paramType.equals("directory")){
			if(isInput)	XML = XML + "\n" + tagWithXML("input","channel",false);
			else XML = XML + "\n" + tagWithXML("output","channel",false);
		}

		String val = inParam.probeDefaultValue();
		if(val != null){
		if(paramType.equals("string-enumeration") || paramType.equals("file") ) XML = XML + "\n"+tagWithXML("\""+val+"\"", "default", false);
		else XML = XML + "\n"+tagWithXML(val, "default", false);

		}

		XML =tagWithXML(XML, paramType, true);
		return XML;

	}

	public String tagWithXML(String in, String tag, boolean breakAtEnd){
		String[] tmp= tag.split(" ");
		if (tmp.length>1){
			tag = tmp[0];
		}
		
		String XML = "<" + tag +">";
		if(breakAtEnd) XML = XML + "\n";
		XML = XML + in; 
		if(breakAtEnd) XML = XML + "\n";
		XML = XML + "</"  +tag.split(" ")[0] + ">";//leave out arguments in closing tag
		if(breakAtEnd) XML = XML + "\n";

		return XML;
	}

	String parseStatusString = null; 
	boolean parseError;
	private File myOutFile;
	private File myOutDir;
	public boolean encounteredParseError() {
		return parseError;
	}

	public void unmarshal() {
		parseError = false;

		ParamModel[] optionList = paramList.values().toArray(new ParamModel[0]);
		StringWriter parseStatus = new StringWriter();
		if(module!=null)
			parseStatus.append(module.getClass().getCanonicalName()+"\n");
		else
			parseStatus.append(pipe.getLabel()+"\n");	
		for(int i=0;i<optionList.length;i++) 
		{
			ParamModel option = optionList[i];
			if(!option.isHidden() && !(option instanceof JISTInternalParam)) {
				String tag = option.getCliTag();

				String arg = cliCommand.getOptionValue(tag);
				
				//Old way
				//if((arg == null) && (option.isMandatory() && !option.isOutputPort()) && (null==option.probeDefaultValue())) {
				if((arg == null) && (option.isMandatory() && !option.isOutputPort()) && (null==option.probeDefaultValue())) {
					parseStatus.append("PARSE ERROR: Missing required tag: "+tag+"\n");
					parseError=true;
				}
				if(arg!=null) {
					option.setXMLValue(arg);
					parseStatus.append("\t"+tag+" = "+arg+"\n");
				}	else {
					parseStatus.append("\t"+tag+" = <default> = "+option.toString()+"\n");
				}
			}
		}	

		String procdir = cliCommand.getOptionValue("xDir");
		if(procdir==null) {
			try {
				procdir = new File(".").getCanonicalPath();
			} catch (IOException e) {
				procdir =null;
			}
		}  
		if(procdir == null) {
			parseStatus.append("PARSE ERROR: Unable to find canonical output directory: "+"outProc"+"\n");
			parseError=true;
		} else {
			parseStatus.append("\t"+"outProc"+" = "+procdir+"\n");
		}
		
		if(module!=null)
			module.setOutputDirectory(new File(procdir));
		myOutDir = new File(procdir);
		
		// handle the output directory
		if (!myOutDir.exists()) {
			// the directory does not exist
			System.err.format("Warning: Directory %s does not exist, trying to make it now.\n", procdir);
			if (!myOutDir.mkdir()) {
				System.err.format("Error: Unable to make directory %s.\n", procdir);
				System.exit(1);
			}
		} else if (!myOutDir.isDirectory()) {
			// the file exists but it is not a directory
			System.err.format("Error: Can not set output directory to %s -- File exists.\n", procdir);
			System.exit(1);
		}

		String outfile = cliCommand.getOptionValue("xFile");
		if(outfile==null)
			outfile = "output.txt";
		if (cliCommand.getOptionValue("xFile") != null)
			parseStatus.append("\t"+"outResult"+" = "+outfile+"\n");
		if(module!=null)
			module.setOutputMetaFile(new File(outfile));
		myOutFile = new File(outfile);
		parseStatusString  = parseStatus.toString();	
		
		//Preference Options
		String preferedExt = cliCommand.getOptionValue("xPrefExt");
		if(preferedExt != null)JistPreferences.getPreferences().setPreferredExtension(preferedExt);

		if(cliCommand.getOptionValue("xDefaultMem") != null){
			int defaultMem = Integer.parseInt(cliCommand.getOptionValue("xDefaultMem"));
			if(defaultMem > 0 )JistPreferences.getPreferences().setDefaultMemory(defaultMem);
		}

		if(cliCommand.getOptionValue("xDebugLvl") != null){
			int debugLvl = Integer.parseInt(cliCommand.getOptionValue("xDebugLvl"));
			if(debugLvl >= 0 && debugLvl <= 9)JistPreferences.getPreferences().setDebugLevel(debugLvl);
		}

		String jreLoc = cliCommand.getOptionValue("xJreLoc");
		if(jreLoc != null)JistPreferences.getPreferences().setJre(jreLoc);

		if(cliCommand.getOptionValue("xMaxProcess") != null){
			int maxProcess = Integer.parseInt(cliCommand.getOptionValue("xMaxProcess"));
			if(maxProcess > 0 )JistPreferences.getPreferences().setDefaultMaxProcesses(maxProcess);
		}

		if(cliCommand.getOptionValue("xUseGridEng") != null){
			boolean useGridEng = Boolean.parseBoolean(cliCommand.getOptionValue("xUseGridEng"));
			JistPreferences.getPreferences().setUseGridEngine(useGridEng);
		}
		
		//Shunxing edit
		if(cliCommand.getOptionValue("xUseAwsEC2") != null){
			boolean useAwsEC2 = Boolean.parseBoolean(cliCommand.getOptionValue("xUseAwsEC2"));
			JistPreferences.getPreferences().setUseAwsEC2(useAwsEC2);
		}
		//Shunxing edit

	}

	public ProcessingAlgorithm getModule() {
		return module;
	}

	public PipeAlgorithm getPipe() {
		return pipe;
	}

	public File getOutFile() {
		return myOutFile;
	}

	public File getOutDir() {
		return myOutDir;
	}

	public void parse(String[] args) throws ParseException {
		
		
		CommandLineParser parser = new PosixParser();
		cliCommand = parser.parse(parserOptions, args);
		
		//Simple check for duplicates
		Option[] targs = cliCommand.getOptions();
		final Set<String> duplicates = new HashSet<String>();
		final Set<String> holder = new HashSet<String>();
		for (int i=0; i<targs.length;i++){
			if (!holder.add(targs[i].getOpt())){
				duplicates.add(targs[i].getOpt());
			}
		}
		if (duplicates.size()!=0){
			throw new ParseException(String.format("FATAL - Duplicate arg(s) found:\n\t%s\n\tPlease check your CLI call.",duplicates.toString()));
		}
	}

	public boolean showHelp() {
		if(cliCommand!=null) {
			return cliCommand.hasOption("help");
		}
		return true;
	}

	public boolean checkSlicerXMLoption(){
		if(cliCommand!=null) {
			return cliCommand.hasOption("xml");
		}
		return false;		
	}

	public String getParseStatus() {
		// TODO Auto-generated method stub
		return parseStatusString;
	}
	
	public static void prepareCLITags(ParamCollection inputs,ParamCollection outputs) {
		Options opts = new Options();
		LinkedHashMap<String,ParamModel> map = new LinkedHashMap<String,ParamModel>();
		parseParamCollection((ParamCollection)inputs,map,opts,"in",true);
		parseParamCollection((ParamCollection)outputs,map,opts,"out",false);

	}
	
	public class OptionComparator implements Comparator<Option> {

		@Override
		public int compare(Option o1, Option o2) {
			
			// convert the options to the expected string form
			String opt1 = convertOption(o1);
			String opt2 = convertOption(o2);
			
			// get which options are actually in the paramList
			boolean in1 = paramList.containsKey(opt1);
			boolean in2 = paramList.containsKey(opt2);
			
			// both are valid options
			if (in1 && in2) {
				Integer ind1 = -1;
				Integer ind2 = -1;
				int cc = 0;
				for (Map.Entry<String, ParamModel> entry : paramList.entrySet()) {
					if (opt1.equals(entry.getKey()))
						ind1 = cc;
					if (opt2.equals(entry.getKey()))
						ind2 = cc;
					cc++;
				}
				return(ind1.compareTo(ind2));
			// only first is valid option
			} else if (in1 && !in2) {
				return(-1);
			// only second is valid option
			} else if (!in1 && in2) {
				return(1);
			// both are invalid options
			} else {
				return(-1 * opt1.compareTo(opt2));
			}
		}
		
		private String convertOption(Option o) {
			String opt = o.getOpt();
			if (opt.startsWith("in"))
				opt = opt.substring(2);
			else if (opt.startsWith("out"))
				opt = opt.substring(3);
			return(opt);
		}
		
	}
}
