/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.factory;

import java.io.File;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import javax.swing.ProgressMonitor;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.MutableTreeNode;

import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel;
import edu.jhu.ece.iacl.jist.pipeline.view.input.ParamCollectionInputView;
import edu.jhu.ece.iacl.jist.pipeline.view.input.ParamInputView;
import edu.jhu.ece.iacl.jist.pipeline.view.output.ParamCollectionOutputView;
import edu.jhu.ece.iacl.jist.pipeline.view.output.ParamOutputView;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import gov.nih.mipav.model.scripting.ParserException;
import gov.nih.mipav.model.scripting.parameters.ParameterException;
import gov.nih.mipav.model.scripting.parameters.ParameterTable;
import gov.nih.mipav.view.dialogs.AlgorithmParameters;

/**
 * Parameter Collection Factory.
 * 
 * @author Blake Lucas
 */
public class ParamCollectionFactory extends ParamFactory {
	
	/** The params. */
	private ParamCollection params;

	/**
	 * Instantiates a new param collection factory.
	 * 
	 * @param params
	 *            the params
	 */
	public ParamCollectionFactory(ParamCollection params) {
		this.params = params;
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.factory.ParamFactory#createMipavParameter(gov.nih.mipav.view.dialogs.AlgorithmParameters)
	 */
	public void createMipavParameter(AlgorithmParameters scriptParams) throws ParserException {
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.factory.ParamFactory#createTreeNode()
	 */
	public MutableTreeNode createTreeNode() {
		if (getParameter().isHidden()) {
			return null;
		}
		DefaultMutableTreeNode root = new DefaultMutableTreeNode("<HTML><B><I><font color='#009900'>"
				+ getParameter().getLabel() + "</font></I></B></HTML>");
		for (ParamModel model : getParameter().getChildren()) {
			if (!model.isHidden()) {
				root.add(model.createTreeNode());
			}
		}
		return root;
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.factory.ParamFactory#equals(java.lang.Object)
	 */
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		} else if (obj instanceof ParamFactory) {
			return this.equals(((ParamFactory) obj).getParameter());
		} else if (obj instanceof ParamCollection) {
			Vector<ParamModel> children = ((ParamCollection) obj).getValue();
			Vector<ParamModel> mychildren = getParameter().getValue();
			boolean equals = false;
			if (children.size() == mychildren.size()) {
				equals = true;
				for (int i = 0; i < children.size(); i++) {
					if (!children.get(i).getFactory().equals(mychildren.get(i))) {
						equals = false;
					}
				}
			}
			if (!equals) {
			}
			return equals;
		} else {
			return false;
		}
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.factory.ParamFactory#getInputView()
	 */
	public ParamInputView getInputView() {
		if (inputView == null) {
			inputView = new ParamCollectionInputView(params);
		}
		return inputView;
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.factory.ParamFactory#getOutputView()
	 */
	public ParamOutputView getOutputView() {
		if (outputView == null) {
			outputView = new ParamCollectionOutputView(params);
		}
		return outputView;
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.factory.ParamFactory#getParameter()
	 */
	public ParamCollection getParameter() {
		return params;
	}

	/**
	 * Gets the parameter.
	 * 
	 * @param name
	 *            the name
	 * @return the parameter
	 */
	public ParamModel getParameter(String name) {
		for (ParamModel param : params.getChildren()) {
			if (param.getName().equals(name)) {
				return param;
			}
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.factory.ParamFactory#importMipavParameter(gov.nih.mipav.model.scripting.parameters.ParameterTable)
	 */
	public void importMipavParameter(ParameterTable paramTable) throws ParameterException {
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.factory.ParamFactory#importParameter(edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel)
	 */
	public boolean importParameter(ParamModel model) {
		boolean ret = true;
		if (model instanceof ParamCollection) {
			Hashtable<String, ParamModel> hash = ((ParamCollection) model).getChildrenHash();
			Hashtable<String, ParamModel> targetHash=params.getChildrenHash();
			Enumeration<String> keys=targetHash.keys();
			while(keys.hasMoreElements()){
				String key=keys.nextElement();

				ParamModel mod=targetHash.get(key);
				ParamModel otherMod = hash.get(key);
				if (otherMod != null) {
					if (!mod.importParameter(otherMod)) {
						ret = false;
					}
				} else {
					System.err.println(getClass().getCanonicalName()+"Could not find parameter matching " + key+" "+mod.getName());
				}
			}
		} else {
			return false;
		}
		return ret;
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.factory.ParamFactory#loadResources(edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel, javax.swing.ProgressMonitor)
	 */
	public boolean loadResources(ParamModel foreign, ProgressMonitor monitor) {
		boolean ret = false;
		if (foreign instanceof ParamCollection) {
			ret = true;
			Hashtable<String, ParamModel> hash = ((ParamCollection) foreign).getChildrenHash();
			Hashtable<String, ParamModel> targetHash=params.getChildrenHash();
			Enumeration<String> keys=targetHash.keys();
			if (monitor != null) {
				monitor.setMaximum(targetHash.keySet().size());
			}
			int progress = 0;
			while(keys.hasMoreElements()){
				String key=keys.nextElement();
				ParamModel mod=targetHash.get(key);
				if (!mod.isHidden()) {
					ParamModel fmod = hash.get(key);
					if (monitor != null) {
						monitor.setNote(params.getLabel() + ":" + fmod.getLabel());
						if (monitor.isCanceled()) {
							break;
						}
					}
					if (fmod != null) {
						if (!mod.loadResources(fmod, monitor)) {
							ret = false;
						}
					} else {
						System.err.println(getClass().getCanonicalName()+"Could not find parameter matching " + mod.getName());
					}
					if (monitor != null) {
						monitor.setProgress(++progress);
					}
				}
			}
		}
		return ret;
	}

	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.factory.ParamFactory#saveResources(java.io.File)
	 */
	public boolean saveResources(File dir, boolean saveSubDirectoryOverride) {
		boolean ret = true;
		String name = params.getName();
		if (name != null && !saveSubDirectoryOverride) {
			File cur = new File(dir, File.separatorChar + edu.jhu.ece.iacl.jist.utility.FileUtil.forceSafeFilename(name));
			if (!cur.exists()) {
				cur.mkdir();
			}
			dir = cur.getAbsoluteFile();
		}
		for (ParamModel param : params.getChildren()) {
			if (!param.saveResources(dir, saveSubDirectoryOverride)) {
				JistLogger.logError(JistLogger.WARNING, getClass().getCanonicalName()+"\t"+"ParamCollectionFactory: Resource Save Failed.");
				return false;
			}
		}
		return ret;
	}
}
