/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.gui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.SpinnerNumberModel;

import edu.jhu.ece.iacl.jist.io.MipavController;
import edu.jhu.ece.iacl.jist.pipeline.PipeLayout;
import edu.jhu.ece.iacl.jist.pipeline.PipeLayout.RunParameters;
import edu.jhu.ece.iacl.jist.processcontrol.DispatchController;

/**
 * Preference panel.
 * 
 * @author Blake Lucas (bclucas@jhu.edu)
 */
public class LayoutPreferencePanel extends JPanel implements ActionListener {	
	/**
	 * Select dir.
	 * 
	 * @param oldDir
	 *            the old dir
	 * @return the file
	 */
	private static File selectDir(File oldDir) {
		JFileChooser loadDialog = new JFileChooser("Specify Directory");
		if (oldDir!=null&&oldDir.exists()) {
			loadDialog.setCurrentDirectory(oldDir);
		} else {
			loadDialog.setCurrentDirectory(MipavController.getDefaultWorkingDirectory());
		}
		loadDialog.setDialogType(JFileChooser.OPEN_DIALOG);
		loadDialog.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		int returnVal = loadDialog.showOpenDialog(null);
		if ((returnVal == JFileChooser.APPROVE_OPTION) && loadDialog.getSelectedFile().isDirectory()) {
			return loadDialog.getSelectedFile();
		} else {
			return null;
		}
	}

	/**
	 * Show dialog.
	 * 
	 * @param comp
	 *            the comp
	 * @param layout
	 *            the layout
	 * @return true, if successful
	 */
	public static boolean showDialog(Component comp, PipeLayout layout) {
		LayoutPreferencePanel panel = new LayoutPreferencePanel(layout);
		while (true) {
			int n = JOptionPane.showConfirmDialog(comp, panel, "Preferences [" + layout.getTitle() + "]",
					JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE);
			if (n == 0) {
				if (panel.update()) {
					return true;
				} else {
					JOptionPane.showMessageDialog(comp, "Invalid parameter.", "Run Parameter Error",
							JOptionPane.ERROR_MESSAGE);
				}
			} else {
				return false;
			}
		}
	}

	/** The browse input. */
	private JButton browseInput;
	
	
	/** The browse output. */
	private JButton browseOutput;
	
	/** The dir input. */
	private JTextField dirInput;
	
	
	/** The dir output. */
	private JTextField dirOutput;
	
	//Shunxing edit
	private JComboBox<String> useApproach;
	//Shunxing edit
	
//	/** The use grid Engine. */
//	private JCheckBox useGridEngine;

	
	/** The email address the PBS reports back to */
	private JTextField gridEngineEmail;
	
	/** The use Hypervisor. */
	private JCheckBox useHypervisor;
	
	private JCheckBox sendDataToHypervisor;
	
	/** The Hypervisor IP. */
	private JTextField hypervisorIP;
	
	/** Probability of not underestimating when using Hypervisor */
	private JTextField hypervisorProbability;

	private JTextField cpuTimeField;
	private JTextField memField;
	
	
	/** The use relative. */
	private JCheckBox useRelative;
	
	/** The wall time. */
	private JSpinner wallTimeH, wallTimeM, wallTimeS;
	
	/** The heap size. */
	private JSpinner heapSize;
	
	/** The sim procs. */
	private JSpinner simProcs;
	
	/** The params. */
	private RunParameters params;
	
	/** The layout. */
	private PipeLayout layout;
	
	
	/** naming convention */
	//private JComboBox namingConvention;

	/**
	 * Constructor.
	 * 
	 * @param layout
	 *            layout
	 */
	public LayoutPreferencePanel(PipeLayout layout) {
		super();
		createPane();
		this.layout = layout;
		init(layout);
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent evt) {
		if (evt.getSource() == browseInput) {
			params.setInputDirectory(selectDir(params.getInputDirectory()));
			if(params.getInputURI()!=null){
				dirInput.setText(params.getInputURI().toString());
			}
		} else if (evt.getSource() == browseOutput) {
			File f=selectDir(params.getOutputDirectory());
			layout.setOutputDir(f);
			if(params.getOutputURI()!=null){
				dirOutput.setText(params.getOutputURI().toString());
			}
		} else if (evt.getSource() == useRelative) {
			enableRelative(useRelative.isSelected());
		}
	}

	/**
	 * Create panel to display preferences.
	 */
	protected void createPane() {
		this.setLayout(new BorderLayout());
		JPanel small = new JPanel();
		BoxLayout layout = new BoxLayout(small, BoxLayout.PAGE_AXIS);
		small.setLayout(layout);
		this.add(small, BorderLayout.NORTH);
		
		JPanel itemPane = new JPanel(new BorderLayout());
		//itemPane.add(new JLabel("Relative Path "), BorderLayout.WEST);
		JPanel relativePane = new JPanel(new BorderLayout());
		relativePane.add(dirInput = new JTextField(20), BorderLayout.CENTER);
		relativePane.add(browseInput = new JButton("Browse"), BorderLayout.EAST);
		relativePane.add(useRelative = new JCheckBox("Use Relative Path"), BorderLayout.SOUTH);
		useRelative.setEnabled(false);
		//itemPane.add(relativePane, BorderLayout.CENTER);
		small.add(itemPane);
		
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Output Path "), BorderLayout.WEST);
		itemPane.add(dirOutput = new JTextField(20), BorderLayout.CENTER);
		itemPane.add(browseOutput = new JButton("Browse"), BorderLayout.EAST);
		small.add(itemPane);
		
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Max Heap Size"), BorderLayout.WEST);
		itemPane.add(heapSize = new JSpinner(new SpinnerNumberModel(1024, 16, 1000000, 20)), BorderLayout.EAST);
		small.add(itemPane);
		
		itemPane = new JPanel(new GridBagLayout());
		GridBagConstraints c = new GridBagConstraints();
        c.gridx = 1;
        c.gridy = 1;
        c.insets = new Insets(5,5,5,5);
        c.weightx = c.weighty = 1.0;
        c.fill = GridBagConstraints.NONE;
        c.gridwidth = 1;
		itemPane.add(new JLabel("Max Wall Time"), c);
		c.gridx++;
		itemPane.add(wallTimeH = new JSpinner(new SpinnerNumberModel(0, 0, 1000000, 1)), c);
		c.gridx++;
		itemPane.add(new JLabel("H "), c);
		c.gridx++;
		itemPane.add(wallTimeM = new JSpinner(new SpinnerNumberModel(0, 0, 59, 1)), c);
		c.gridx++;
		itemPane.add(new JLabel("M "), c);
		c.gridx++;
		itemPane.add(wallTimeS = new JSpinner(new SpinnerNumberModel(0, 0, 59, 1)), c);
		c.gridx++;
		itemPane.add(new JLabel("S "), c);
		small.add(itemPane);
		
		itemPane = new JPanel(new BorderLayout());
		//int maxProcs = Runtime.getRuntime().availableProcessors();
		//itemPane.add(new JLabel("Simultaneous Processes ("+maxProcs+" Available CPU"+((maxProcs>1)?"s":"")+")"), BorderLayout.WEST);
		itemPane.add(new JLabel("Simultaneous Processes "), BorderLayout.WEST);
		itemPane.add(simProcs = new JSpinner(new SpinnerNumberModel(4, 1, 100, 1)), BorderLayout.EAST);
		small.add(itemPane);
		
		/*
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Output Directory Naming Convention"), BorderLayout.WEST);
		itemPane.add(namingConvention =new JComboBox(new String[]{"Tree Hierarchy","Universal ID"}), BorderLayout.EAST);
		small.add(itemPane);
		*/
		
		//Shunxig edit
//		itemPane = new JPanel(new BorderLayout());
//		String GridString = ((new DispatchController()).testClusterAvailability())?"(Online)":"(Offline)";
//		itemPane.add(new JLabel("Use Grid Engine If Available "+GridString), BorderLayout.WEST);
//		useGridEngine = new JCheckBox("",false);
//		itemPane.add(useGridEngine, BorderLayout.EAST);
//		small.add(itemPane);
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(useApproach = new JComboBox<String>(), BorderLayout.WEST);
		//Approach 1: native way
		useApproach.addItem("Use Native Method");
		//Approach 2: Use grid engine
		String GridString = ((new DispatchController()).testClusterAvailability())?"(Online)":"(Offline)";
		useApproach.addItem("Use Grid Engine If Available "+GridString);
		//Approach 3: Use Aws EC2
		String AwsEC2String = "Offline";
		File awsPreference = new File(System.getProperties().getProperty("user.home")
				+ "/JistAwsEc2"+"/JISTAWSEC2.preferences");
		if(awsPreference.exists() && !awsPreference.isDirectory()){
			AwsEC2String = ((new DispatchController()).testAwsEC2Availability())?"(Online)":"(Offline)";
		}
		useApproach.addItem("Use AWS EC2 If Available "+AwsEC2String);
		small.add(itemPane);
		
		//Shunxing edit
		
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Email address for Grid Engine reporting: "), BorderLayout.WEST);
		itemPane.add(gridEngineEmail = new JTextField(20), BorderLayout.EAST);
		small.add(itemPane);
		
		/*Hypervisor changes*/
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Use Hypervisor Estimate If Available"), BorderLayout.WEST);
		itemPane.add(useHypervisor = new JCheckBox("",true), BorderLayout.EAST);
		small.add(itemPane);

		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Send Data To Hypervisor"), BorderLayout.WEST);
		itemPane.add(sendDataToHypervisor = new JCheckBox("",true), BorderLayout.EAST);
		small.add(itemPane);		
		
		itemPane = new JPanel(new BorderLayout());
		itemPane.add(new JLabel("Hypervisor IP:"), BorderLayout.WEST);
		itemPane.add(hypervisorIP = new JTextField(20), BorderLayout.EAST);
		small.add(itemPane);
		
		itemPane = new JPanel(new BorderLayout());
		/*JLabel hyperThreshold = new JLabel("Hypervisor Threshold:");
		hyperThreshold.setToolTipText("Values less than 1 give estimations that will not underestimate for threshold*100 percent of jobs." +
				"Values greater than or equal to one are multiplied by the biggest known resource requirement for jobs similar to the one to be run." +
				"The tradeoff is reduced queue time verus safety.");
		itemPane.add(hyperThreshold, BorderLayout.WEST);
		itemPane.add(hypervisorProbability = new JTextField(20), BorderLayout.EAST);*/
		itemPane.add(new JLabel("CPU Time Alpha Value"), BorderLayout.WEST);
		itemPane.add(cpuTimeField = new JTextField(20), BorderLayout.EAST);
		small.add(itemPane);
		
		itemPane = new JPanel(new BorderLayout());
		/*JLabel hyperThreshold = new JLabel("Hypervisor Threshold:");
		hyperThreshold.setToolTipText("Values less than 1 give estimations that will not underestimate for threshold*100 percent of jobs." +
				"Values greater than or equal to one are multiplied by the biggest known resource requirement for jobs similar to the one to be run." +
				"The tradeoff is reduced queue time verus safety.");
		itemPane.add(hyperThreshold, BorderLayout.WEST);
		itemPane.add(hypervisorProbability = new JTextField(20), BorderLayout.EAST);*/
		itemPane.add(new JLabel("Memory Alpha Value"), BorderLayout.WEST);
		itemPane.add(memField = new JTextField(20), BorderLayout.EAST);
		small.add(itemPane);
		/*End Hypervisor changes*/	
		
		browseInput.addActionListener(this);
		browseOutput.addActionListener(this);
		useRelative.addActionListener(this);
	}

	/**
	 * Enable relative.
	 * 
	 * @param enabled
	 *            the enabled
	 */
	protected void enableRelative(boolean enabled) {
		useRelative.setSelected(enabled);
		dirInput.setEnabled(enabled);
		browseInput.setEnabled(enabled);
		if (!enabled) {
			layout.setInputDir((File)null);
		}
	}
	
	/**
	 * Initialize preference panel for layout.
	 * 
	 * @param layout
	 *            the layout
	 */
	protected void init(PipeLayout layout) {
		params = layout.getRunParameters();
		dirInput.setText((params.getInputURI() != null) ? params.getInputURI().toString() : "");
		dirOutput.setText((params.getOutputURI() != null) ? params.getOutputURI().toString() : "");
		heapSize.setValue(params.getMaxHeap());
		
		String wallTime = params.getWallTime();
		int first = wallTime.indexOf(":");
		String h = wallTime.substring(0,first);
		String m = wallTime.substring(first+1,wallTime.lastIndexOf(":"));
		String s = wallTime.substring(wallTime.lastIndexOf(":")+1);
		wallTimeH.setValue(Integer.parseInt(h));
		wallTimeM.setValue(Integer.parseInt(m));
		wallTimeS.setValue(Integer.parseInt(s));
		
		simProcs.setValue(params.getMaxProcs());
		//Shunxing edit
		if (params.isUseGridEngine() == true){
			useApproach.setSelectedIndex(1);
		} else if (params.isUseAwsEC2() == true){
			useApproach.setSelectedIndex(2);
		} else 
			useApproach.setSelectedIndex(0);
		
		//useGridEngine.setSelected(params.isUseGridEngine());
		//Shunxing edit
		
//		useGridEngine.setSelected(params.isUseGridEngine());
		gridEngineEmail.setText(params.getGridEngineEmail());
        useHypervisor.setSelected(params.isUseHypervisor());
		sendDataToHypervisor.setSelected(params.isSendDataToHypervisor());
		hypervisorIP.setText(params.getHypervisorIP());
		//hypervisorProbability.setText(String.valueOf(params.getHypervisorProbability()));
		cpuTimeField.setText(String.valueOf(params.getCpuFudge()));
		memField.setText(String.valueOf(params.getMemFudge()));

		/*
		NamingConvention convention=params.getNamingConvention();
		if(convention!=null){
			namingConvention.setSelectedIndex(params.getNamingConvention().ordinal());
		}
		*/
		enableRelative((params.getInputDirectory() != null));
	}
	
	/**
	 * @return Path to current project directory
	 */
	public String getProjectPath(PipeLayout layout) {
		return layout.getRunParameters().getOutputDirectory().getParentFile().getAbsolutePath();
	}
	
	/**
	 * Update.
	 * 
	 * @return true, if successful
	 */
	protected boolean update() {
		try {
			layout.setMaxHeap(((Integer) heapSize.getValue()).intValue());
			layout.setMaxProcs(((Integer) simProcs.getValue()).intValue());
			
		} catch (NumberFormatException e) {
			return false;
		}
		//layout.setNamingConvention(NamingConvention.values()[namingConvention.getSelectedIndex()]);
		//Shunxing edit
		String tmpApproach = useApproach.getSelectedItem().toString();
		if(tmpApproach.contains("Native")){
			layout.setUseGridEngine(false);
			layout.setUseAwsEC2(false);
		} else if(tmpApproach.contains("Grid")){
			layout.setUseGridEngine(true);
			layout.setUseAwsEC2(false);
		} else if(tmpApproach.contains("AWS")){
			layout.setUseAwsEC2(true);
			layout.setUseGridEngine(false);
		}		
		//Shunxing edit
//		layout.setUseGridEngine(useGridEngine.isSelected());
		layout.setGridEngineEmail(gridEngineEmail.getText());
		layout.setUseHypervisor(useHypervisor.isSelected());
		layout.setHypervisorIP(hypervisorIP.getText());
		layout.setSendDataToHypervisor(sendDataToHypervisor.isSelected());
		try{
			//float newProbability = Float.parseFloat(hypervisorProbability.getText());
			//layout.setHypervisorProbability(newProbability);
			float cpuFudge = Float.parseFloat(cpuTimeField.getText());
			float memFudge = Float.parseFloat(memField.getText());
			layout.setCPUFudge(cpuFudge);
			layout.setMemFudge(memFudge);
		} catch (Exception e) {}
		
		
		try {
			String wallTime = String.valueOf(wallTimeH.getValue()) + ":" + String.valueOf(wallTimeM.getValue()) + ":" + String.valueOf(wallTimeS.getValue());
			layout.setWallTime(wallTime);
		} catch (Exception e) {
			return false;
		}
		
		File f;
		URI uri;
		if (useRelative.isSelected()) {
			
			try {
				f = new File(uri=new URI(dirInput.getText()));
			} catch (URISyntaxException e) {
				f = new File(dirInput.getText());
				uri=f.toURI();
			}
			if (f.exists() && f.isDirectory()) {
				layout.setInputDir(uri);
			} else {
				return false;
			}
		} else {
			layout.setInputDir((File)null);
		}
		
		// try and set the output directory, if it is empty, then it will be set when the layout is saved
		String newOutputDir = dirOutput.getText();
		if (!newOutputDir.isEmpty()) {
			
			// try and set the URI for the output directory
			try {
				f = new File(uri=new URI(newOutputDir));
			} catch (URISyntaxException e) {
				f = new File(newOutputDir);
				uri=f.toURI();
			} catch (IllegalArgumentException e) {
				f = new File(newOutputDir);
				uri = f.toURI();
			}
			
			// set the output directory
			if (!f.exists()) {
				String msg = "Do you want to make the following directory:\n" + uri.getPath();
				int n = JOptionPane.showOptionDialog(PipelineLayoutTool.getInstance(),
													 msg, "Output directory does not exist",
													 JOptionPane.OK_CANCEL_OPTION, 
													 JOptionPane.WARNING_MESSAGE,
													 null, null, null);
				if (n == 0) {
					
					// try and make the directory
					boolean success = f.mkdir();
					if (success)
						layout.setOutputDir(uri);
					else
						JOptionPane.showMessageDialog(PipelineLayoutTool.getInstance(),
													  "Unable to make output directory.",
													  "Error",
													  JOptionPane.ERROR_MESSAGE);
				}
			} else if (!f.isDirectory()) {
				return false;
			} else {
				layout.setOutputDir(uri);
			}
		}
		return true;
	}
}
