/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.src;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import edu.jhu.ece.iacl.jist.pipeline.PipeSource;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamDouble;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamModel;

/**
 * Advanced sweep through double values.
 * 
 * @author Aaron Carass
 *
 * value_{i+1} = a * (value_{i}^{p}) + c
 *
 * a = mulInc
 * p = powerTerm
 * c = addInc
 */
public class PipeAdvancedDoubleSweepSource extends PipeSource {

	/** Start */
	transient protected double start;

	/** Value never to be exceeded */
	transient protected double end;

	/** Additive Increment */
	transient protected double addInc;

	/** Multiplicative/Ratio Term */
	transient protected double mulInc;

	/** Power Term */
	transient protected double powerTerm;

	/** The value, a temp value, and the previous value */
	transient protected double val, tmpVal, prevVal;


	/** Params */
	protected ParamDouble startParam, endParam;
	protected ParamDouble addIncParam, mulIncParam, powerTermParam;
	protected ParamDouble valParam;


	protected boolean xmlEncodeModule(Document document, Element parent) {
		boolean val = super.xmlEncodeModule(document, parent);		
		return val;
	}


	public void xmlDecodeModule(Document document, Element el) {
		super.xmlDecodeModule(document, el);
		startParam = (ParamDouble) inputParams.getFirstChildByName("Start Value");
		endParam = (ParamDouble) inputParams.getFirstChildByName("End Value");
		mulIncParam = (ParamDouble) inputParams.getFirstChildByName("Multiplicative Increment");
		powerTermParam = (ParamDouble) inputParams.getFirstChildByName("Power Term");
		addIncParam = (ParamDouble) inputParams.getFirstChildByName("Additive Increment");
		valParam = (ParamDouble) outputParams.getFirstChildByName("Float");
		getParentPort().setParameter(valParam);
	}


	/**
	 * Default constructor.
	 */
	public PipeAdvancedDoubleSweepSource() {
		super();
		getParentPort().setParameter(valParam);
	}


	/**
	 * Create input parameters.
	 * 
	 * @return the param collection
	 */
	public ParamCollection createInputParams() {
		ParamCollection group = new ParamCollection();
		group.setLabel("Advanced Float Sweep");
		group.setName("advancedfloatsweep");

		group.add(startParam = new ParamDouble("Start Value"));
		group.add(endParam = new ParamDouble("End Value"));
		group.add(mulIncParam = new ParamDouble("Multiplicative Increment"));
		group.add(powerTermParam = new ParamDouble("Power Term"));
		group.add(addIncParam = new ParamDouble("Additive Increment"));

		group.setCategory("Number.Float");
		return group;
	}


	/**
	 * Create ouptut parameters.
	 * 
	 * @return the param collection
	 */
	public ParamCollection createOutputParams() {
		ParamCollection group = new ParamCollection();
		group.setLabel("Float");
		group.add(valParam = new ParamDouble("Float"));
		return group;
	}


	/**
	 * Get output parameter.
	 * 
	 * @return the output param
	 */
	public ParamModel getOutputParam() {
		return valParam;
	}


	/**
	 * Return true if iterator has more elements.
	 * 
	 * @return true, if checks for next
	 *
	 * As long as the value is changing every iteration
	 * (ie. prevVal < val) then we proceed.
	 */
	public boolean hasNext() {
		return (super.hasNext() || (prevVal < val));
	}


	/**
	 * Iterate through double values.
	 * 
	 * @return true, if iterate
	 */
	public boolean iterate() {
		if (hasNext()) {
			if (!super.iterate()) {
				if ((prevVal < val) && (val <= end)) {
					valParam.setValue(val);
					push();
					prevVal = val;
					tmpVal = mulInc * Math.pow(val, powerTerm);
					val = tmpVal + addInc;
				} else {
					reset();
					isReset = true;
					return false;
				}
			}
			return true;
		} else {
			reset();
			isReset = true;
			return false;
		}
	}


	/**
	 * Reset iterator.
	 */
	public void reset() {
		super.reset();
		start     = startParam.getDouble();
		end       = endParam.getDouble();
		mulInc    = mulIncParam.getDouble();
		powerTerm = powerTermParam.getDouble();
		addInc    = addIncParam.getDouble();
		val       = start;
		prevVal   = val - 1.0;

		valParam.setValue(val);
		tmpVal = mulInc * Math.pow(val, powerTerm);
		val = tmpVal + addInc;
		push();
	}
}
