/**
 * Java Image Science Toolkit (JIST)
 *
 * Image Analysis and Communications Laboratory &
 * Laboratory for Medical Image Computing &
 * The Johns Hopkins University
 * 
 * http://www.nitrc.org/projects/jist/
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.  The license is available for reading at:
 * http://www.gnu.org/copyleft/lgpl.html
 *
 */
package edu.jhu.ece.iacl.jist.pipeline.src;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import edu.jhu.ece.iacl.jist.pipeline.PipeSource;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;

/**
 * Advanced sweep through range of integers.
 * 
 * @author Aaron Carass
 *
 * value_{i+1} = a * (value_{i}^{p}) + c
 *
 * a = mulInc
 * p = powerTerm
 * c = addInc
 */
public class PipeAdvancedIntegerSweepSource extends PipeSource {

	/** The start. */
	transient protected int start;

	/** The end. */
	transient protected int end;

	/** Additive Increment */
	transient protected int addInc;

	/** Multiplicative/Ratio Increment */
	transient protected int mulInc;

	/** Power Term */
	transient protected int powerTerm;

	/** The value, a temp value, and the previous value */
	transient protected int val, tmpVal, prevVal;

	/** Params */
	protected ParamInteger startParam, endParam;
	protected ParamInteger addIncParam, mulIncParam, powerTermParam;
	protected ParamInteger valParam;


	protected boolean xmlEncodeModule(Document document, Element parent) {
		boolean val = super.xmlEncodeModule(document, parent);		
		return val;
	}


	public void xmlDecodeModule(Document document, Element el) {
		super.xmlDecodeModule(document, el);
		startParam = (ParamInteger) inputParams.getFirstChildByName("Start Value");
		endParam = (ParamInteger) inputParams.getFirstChildByName("End Value");

		mulIncParam = (ParamInteger) inputParams.getFirstChildByName("Multiplicative Increment");
		powerTermParam = (ParamInteger) inputParams.getFirstChildByName("Power Term");
		addIncParam = (ParamInteger) inputParams.getFirstChildByName("Increment");
		valParam = (ParamInteger) outputParams.getFirstChildByName("Integer");

		getParentPort().setParameter(valParam);
	}


	/**
	 * Instantiates a new pipe integer sweep source.
	 */
	public PipeAdvancedIntegerSweepSource() {
		super();
		getParentPort().setParameter(valParam);
	}


	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.PipeSource#createInputParams()
	 */
	public ParamCollection createInputParams() {
		ParamCollection group = new ParamCollection();
		group.setLabel("Advanced Integer Sweep");
		group.setName("AdvancedIntegerSweep");

		group.add(startParam = new ParamInteger("Start Value"));
		group.add(endParam = new ParamInteger("End Value"));
		group.add(mulIncParam = new ParamInteger("Multiplicative Increment"));
		group.add(powerTermParam = new ParamInteger("Power Term"));
		group.add(addIncParam = new ParamInteger("Increment"));

		group.setCategory("Number.Integer");
		return group;
	}


	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.PipeSource#createOutputParams()
	 */
	public ParamCollection createOutputParams() {
		ParamCollection group = new ParamCollection();
		group.setLabel("Integer");
		group.add(valParam = new ParamInteger("Integer"));
		return group;
	}


	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.PipeSource#getOutputParam()
	 */
	public ParamInteger getOutputParam() {
		return valParam;
	}


	/**
	 * Return true if iterator has more elements.
	 *
	 * @return true, if checks for next
	 *
	 * As long as the value is changing every iteration
	 * (ie. prevVal < val) then we proceed.
	 */
	public boolean hasNext() {
		return (super.hasNext() || (prevVal < val));
	}


	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.PipeSource#iterate()
	 */
	public boolean iterate() {
		if (hasNext()) {
			if (!super.iterate()) {
				if ((prevVal < val) && (val <= end)) {
					valParam.setValue(val);
					push();
					prevVal = val;
					tmpVal = mulInc * (int) (Math.pow((double) val, (double) powerTerm));
					val = tmpVal + addInc;
				} else {
					reset();
					isReset = true;
					return false;
				}
			}
			return true;
		} else {
			reset();
			isReset = true;
			return false;
		}
	}


	/* (non-Javadoc)
	 * @see edu.jhu.ece.iacl.jist.pipeline.PipeSource#reset()
	 */
	public void reset() {
		super.reset();
		start     = startParam.getInt();
		end       = endParam.getInt();
		mulInc    = mulIncParam.getInt();
		powerTerm = powerTermParam.getInt();
		addInc    = addIncParam.getInt();
		val       = start;
		prevVal   = val - 1;

		valParam.setValue(val);
		tmpVal = mulInc * (int) (Math.pow((double) val, (double) powerTerm));
		val = tmpVal + addInc;
		push();
	}
}
