package edu.jhu.ece.iacl.jist.processcontrol;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.List;

import edu.jhu.ece.iacl.jist.pipeline.ExecutionContext;
import edu.jhu.ece.iacl.jist.pipeline.JistPreferences;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import edu.vanderbilt.masi.jistcloud.MyUtils;


public class ProcessControllerQsub implements ProcessController {

	private long JobSubmitTime;		
	private Integer exitCode = -2;		
	private InputStream stdout, stderr;
	private ProcessStatus JobStatus;
	
	private String hypervisorIP;
	private boolean useHypervisor;
	
	String maxMem = "4999mb";
	String maxWallTime = "3:59:59";
	String defaultOutputLoc = "";
	String email;
	
	private ExecutionContext context = null;
	
	private String JobID;	
	private String OUTPUT_LOC;	
	private String PBS_NAME;
	private String PBS_LOC;
	private String COMPLETE;
	
	public ProcessControllerQsub(){super();}
	
	/**
	 * sets the output location
	 * for use by the dispatch controller
	 * @param loc
	 */
	public ProcessControllerQsub(String loc, ExecutionContext con){
		super();
		OUTPUT_LOC = loc;
		if(!OUTPUT_LOC.endsWith("/")){
			OUTPUT_LOC = OUTPUT_LOC.concat("/");
		}
		
		context = con;
		
	}
	
	public int getAllocMem(){
		return Integer.parseInt(maxMem);
	}
	
	@Override
	public boolean Setup(List<String> command) {
        try {       	
        	File temp = JistPreferences.getPreferences().getGridEngineTemplate();
        	
        	hypervisorIP = JistPreferences.getPreferences().getHypervisorIP();
        	useHypervisor = JistPreferences.getPreferences().isUseHypervisor();
        	email = JistPreferences.getPreferences().getGridEngineEmail();
        	      	
        	PBS_NAME = "experiment_pbs_file.pbs";
        	PBS_LOC = OUTPUT_LOC + PBS_NAME;

        	COMPLETE = OUTPUT_LOC + "complete.job";
        	
        	File checkFolders = new File(OUTPUT_LOC);
        	if(!checkFolders.exists()){
        		if(!checkFolders.mkdir()){
        			JistLogger.logError(JistLogger.SEVERE, "OUTPUT folder does not exist and could not be created.");
        		} 
        	}
        
        	if(useHypervisor && !hypervisorIP.equalsIgnoreCase("none") &&
        			context.getEstimatedMemory() != null && context.getEstimatedTime() != null &&
        			!context.getEstimatedMemory().equalsIgnoreCase("NA") && !context.getEstimatedTime().equalsIgnoreCase("NA")){
        		
        		maxMem 		= context.getEstimatedMemory();
		        maxWallTime = context.getEstimatedTime();
        	}
        	else{
        		maxMem = Integer.toString(context.getLayout().getRunParameters().getMaxHeap());
				maxWallTime = context.getLayout().getRunParameters().getWallTime();
				JistLogger.logError(JistLogger.WARNING, "Hyperadvisor could not provide a satisfactory estimate...using default memory and runtime settings.");
        	}
        	
        	String text;
        	String cmd = "";
        	
        	for(int i = 0; i < command.size(); i++){
        		text = command.get(i);
        		
        		if(text.contains("-xDir"))
        		{
        			OUTPUT_LOC = text.split("[-xDir][ ]+")[1];
        		}
        		cmd = cmd.concat(text);
        		cmd = cmd.concat(" ");
        	}
        	
        	FileReader ris = new FileReader(temp);
        	BufferedReader reader = new BufferedReader(ris);

        	FileWriter fstream = new FileWriter(new File(PBS_LOC));
        	BufferedWriter out = new BufferedWriter(fstream);
        	
        	//Used when memory values will be interpreted as 1024 bytes per kilobyte, etc.
        	Double doubleAdjusted = Math.ceil(Integer.parseInt(maxMem)*3);
        	Integer adjustedMem = Math.round(doubleAdjusted.intValue());
        	
            /*read in template and create pbs file*/
            while((text = reader.readLine()) != null){
            	if(text.contains("%%EMAIL%%")){
            		text = text.replace("%%EMAIL%%", email);
            	}
            	if(text.contains("%%MEMORY%%")){
            		text = text.replace("%%MEMORY%%", maxMem);
            	}
            	
            	// config max job memory for PBS
            	if(text.contains("%%ADJUSTED_MEMORY%%")){
            		int maxJobMemory = MyUtils.getIntegerFromProperties(context.getLayout(), "max_job_memory");
            		
            		if (maxJobMemory < Integer.parseInt(maxMem)) {
            			maxJobMemory = adjustedMem;
            		}
            		
            		text = text.replace("%%ADJUSTED_MEMORY%%", String.valueOf(maxJobMemory));
            	}
            	
            	if(text.contains("%%WALLTIME%%")){
            		text = text.replace("%%WALLTIME%%", maxWallTime);
            	}
            	if(text.contains("%%OUTPUTLOC%%")){
            		text = text.replace("%%OUTPUTLOC%%", OUTPUT_LOC);
            	}
            	if(text.contains("%%COMMAND%%")){
            		text = text.replace("%%COMMAND%%", cmd);
            	}
            	
            	out.write(text);
            	out.newLine();
            }
            
            out.write("touch " + COMPLETE);
            out.newLine();
	        out.close();
	        
		} catch (IOException e) {
			e.printStackTrace();
			return false;
		}
        
		return true;
	}

	@Override
	public void controlJob(ProcessControlAction Action) {
		// TODO Auto-generated method stub
		if(Action == ProcessControlAction.SUSPEND ){
			JobStatus = ProcessStatus.USER_SYSTEM_SUSPENDED;
		}
		else if(Action == ProcessControlAction.RELEASE ){
			try {
				Runtime.getRuntime().exec("qrls " + JobID);
			} catch (IOException e) {
				e.printStackTrace();
			}
			JobStatus = ProcessStatus.READY;
		}
		else if(Action == ProcessControlAction.HOLD ){
			try {
				Runtime.getRuntime().exec("qhold " + JobID);
			} catch (IOException e) {
				e.printStackTrace();
			}
			JobStatus = ProcessStatus.USER_SYSTEM_ON_HOLD;
		}
		else if (Action == ProcessControlAction.RESUME ){
			// TODO Auto-generated method stub
//			try {
//				Runtime.getRuntime().exec("qdel " + JobID);
//			} catch (IOException e) {
//				e.printStackTrace();
//			}
			JobStatus = ProcessStatus.RUNNING;
		}
		else if (Action == ProcessControlAction.TERMINATE){
			if(exitCode != 0){
				JobStatus = ProcessStatus.FAILED;
			}
			
			JistLogger.logOutput(JistLogger.INFO, "qdel " + JobID + " executed.");
			//Just to make sure the process is really stopped
			//Delete whether or not exit code is 0
			try {
				Runtime.getRuntime().exec("qdel " + JobID);
				
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		JistLogger.logOutput(JistLogger.INFO, getClass().getCanonicalName()+"\t"+JobStatus);
	}

	@Override
	public boolean destroy() {
		controlJob(ProcessControlAction.TERMINATE);
		return true;
	}

	@Override
	public Integer getExitCode() {
		return exitCode;
	}

	@Override
	public Process getProcess() {
		return null;
	}

	@Override
	public ProcessStatus getStatus() {
		return JobStatus;
	}

	@Override
	public InputStream getStderrFile() {
		return stderr;
	}

	@Override
	public InputStream getStdoutFile() {
		return stdout;
	}

	@Override
	public long getSubmissionTime() {
		return JobSubmitTime;
	}

	@Override
	public boolean submit() {
		JobSubmitTime = System.currentTimeMillis();
       
        try
        {
        	JobStatus = ProcessStatus.RUNNING;
        	Process p = Runtime.getRuntime().exec("qsub " + PBS_LOC);

            stdout = p.getInputStream();
            stderr = p.getErrorStream();

            InputStreamReader insrout=new InputStreamReader(stdout);
            InputStreamReader insrerr=new InputStreamReader(stderr);

            BufferedReader stdoutReader = new BufferedReader(insrout);

            BufferedReader stderrReader = new BufferedReader(insrerr);

            JobID = stdoutReader.readLine();
            if (JobID != null)
            {
            	//Test line to make sure that we're getting the job id properly
                JistLogger.logOutput(JistLogger.INFO, "Job ID is " + JobID);
                
                //Write that Job ID to file.
                File job = new File(OUTPUT_LOC + "jobid.txt");

            	try{
                	BufferedWriter out = new BufferedWriter(new FileWriter(job,false));
                	out.write(JobID);
                	out.flush();
                	out.close();
            	}
            	catch(IOException e){
            		JistLogger.logError(JistLogger.SEVERE, "Job ID file could not be written.");
            	}
            }
            
            
            while (true)
            {
                String line = stdoutReader.readLine();
                if (line == null)
                {
                    break;
                }
                JistLogger.logOutput(JistLogger.INFO, line);
            }

            while (true)
            {
                String line = stderrReader.readLine();
                if (line == null)
                {    break;}
                JistLogger.logOutput(JistLogger.INFO, line);
            }
        }
        catch (IOException e)
        {
            e.printStackTrace(System.err);
            e.getMessage();
            System.exit(2);
        }	        		
		return false;
	}
	
	@Override
	public int waitFor() throws InterruptedException {
		
		File isDone = new File(COMPLETE);
	    while(!isDone.exists()){	    	
	    	try {
	    		Thread.sleep(1000);
			} catch (InterruptedException e) {
				JobStatus = ProcessStatus.FAILED;
				exitCode = -1;
				return exitCode;
			}	    	
	    }
	    exitCode = 0;
	    JobStatus = ProcessStatus.DONE;	
	    Thread.sleep(3000);
		return exitCode;
	}

}
