package edu.jhu.ece.iacl.jist.structures.geom;

import edu.jhu.ece.iacl.jist.algorithms.graphics.GeomUtil;
import edu.jhu.ece.iacl.jist.utility.JistLogger;
import gov.nih.mipav.view.renderer.J3D.model.structures.ModelTriangleMesh;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;

import javax.vecmath.Color3f;
import javax.vecmath.Color4f;
import javax.vecmath.Matrix3f;
import javax.vecmath.Point2d;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Tuple3f;
import javax.vecmath.Vector3f;

import Jama.Matrix;

// TODO: Auto-generated Javadoc
/**
 * The Class EmbeddedSurface.
 */
public class EmbeddedSurface extends ModelTriangleMesh implements Cloneable,
Comparable<EmbeddedSurface> {

	/**
	 * The Enum Direction.
	 */
	public enum Direction {

		/** The CLOCKWISE. */
		CLOCKWISE, 
		/** The COUNTE r_ clockwise. */
		COUNTER_CLOCKWISE
	}

	/** The texture coords. */
	protected double[][] textureCoords=null;

	/**
	 * The Class Edge.
	 */
	public static class Edge implements Comparable<Edge> {

		/**
		 * Hash code long.
		 * 
		 * @param v1 the v1
		 * @param v2 the v2
		 * 
		 * @return the long
		 */
		public static long hashCodeLong(int v1, int v2) {
			return v1 + v2 * 10000000l;
		}

		/** The v2. */
		public int v1, v2;

		/** The f2. */
		public int f1, f2;

		/** The id. */
		public int id;

		/**
		 * Instantiates a new edge.
		 * 
		 * @param v1 the v1
		 * @param v2 the v2
		 * @param id the id
		 */
		public Edge(int v1, int v2, int id) {
			this.v1 = v1;
			this.v2 = v2;
			this.f1 = -1;
			this.f2 = -1;
			this.id = id;
		}

		/* (non-Javadoc)
		 * @see java.lang.Comparable#compareTo(java.lang.Object)
		 */
		public int compareTo(Edge e) {
			return (int) Math.signum(this.hashCodeLong() - e.hashCodeLong());
		}

		/**
		 * Contains.
		 * 
		 * @param v the v
		 * 
		 * @return true, if successful
		 */
		public boolean contains(int v) {
			return (v == v1 || v == v2);
		}

		/**
		 * Equals.
		 * 
		 * @param v1 the v1
		 * @param v2 the v2
		 * 
		 * @return true, if successful
		 */
		public boolean equals(int v1, int v2) {
			return ((this.v1 == v1 && this.v2 == v2) || (this.v1 == v2 && this.v2 == v1));
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#equals(java.lang.Object)
		 */
		@Override
		public boolean equals(Object obj) {
			if (obj instanceof Edge) {
				return (v1 == ((Edge) obj).v1 && v2 == ((Edge) obj).v2);
			}
			return false;
		}

		/**
		 * Gets the vector.
		 * 
		 * @param surf the surf
		 * 
		 * @return the vector
		 */
		public Vector3f getVector(EmbeddedSurface surf) {
			Vector3f v = new Vector3f();
			v.sub(surf.getVertex(v2), surf.getVertex(v1));
			return v;
		}

		/**
		 * Hash code long.
		 * 
		 * @return the long
		 */
		public long hashCodeLong() {
			return v1 + v2 * 10000000l;
		}

		/**
		 * Opposite.
		 * 
		 * @param v the v
		 * 
		 * @return the int
		 */
		public int opposite(int v) {
			return (v == v1) ? v2 : v1;
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#toString()
		 */
		@Override
		public String toString() {
			return "<" + v1 + "," + v2 + ">";
		}
	}

	/**
	 * The Class EdgeSplit.
	 */
	public static class EdgeSplit extends Edge {

		/** The mid. */
		public int mid;

		/**
		 * Instantiates a new edge split.
		 * 
		 * @param e the e
		 * @param mid the mid
		 */
		public EdgeSplit(Edge e, int mid) {
			super(e.v1, e.v2, e.id);
			this.mid = mid;
		}

		/**
		 * Instantiates a new edge split.
		 * 
		 * @param v1 the v1
		 * @param v2 the v2
		 * @param mid the mid
		 * @param id the id
		 */
		public EdgeSplit(int v1, int v2, int mid, int id) {
			super(v1, v2, id);
			this.mid = mid;
		}

		/* (non-Javadoc)
		 * @see edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface.Edge#toString()
		 */
		@Override
		public String toString() {
			return "<" + v1 + "," + mid + "," + v2 + ">";
		}
	}

	/**
	 * The Class Face.
	 */
	public static class Face {

		/** The edges. */
		public Edge[] edges;

		/** The id. */
		public int id;

		/**
		 * Instantiates a new face.
		 * 
		 * @param e1 the e1
		 * @param e2 the e2
		 * @param e3 the e3
		 * @param id the id
		 */
		public Face(Edge e1, Edge e2, Edge e3, int id) {
			edges = new Edge[] { e1, e2, e3 };

			this.id = id;
			if (e1.f1 == -1)
				e1.f2 = id;
			else
				e1.f1 = id;
			if (e2.f1 == -1)
				e2.f2 = id;
			else
				e2.f1 = id;
			if (e3.f1 == -1)
				e3.f2 = id;
			else
				e3.f1 = id;
		}

		/**
		 * Instantiates a new face.
		 * 
		 * @param id the id
		 */
		public Face(int id) {
			edges = new Edge[3];
			this.id = id;
		}

		/**
		 * Instantiates a new face.
		 * 
		 * @param v1 the v1
		 * @param v2 the v2
		 * @param v3 the v3
		 * @param id the id
		 */
		public Face(int v1, int v2, int v3, int id) {
			this(new Edge(v1, v2, -1), new Edge(v2, v3, -1), new Edge(v3, v1,
					-1), id);
		}

		/**
		 * Gets the area.
		 * 
		 * @param mesh the mesh
		 * 
		 * @return the area
		 */
		public double getArea(EmbeddedSurface mesh) {
			Vector3f v1 = edges[0].getVector(mesh);
			Vector3f v2 = edges[1].getVector(mesh);
			Vector3f mag = new Vector3f();
			mag.cross(v1, v2);
			return 0.5 * mag.length();
		}

		/**
		 * Gets the vertexes.
		 * 
		 * @return the vertexes
		 */
		public int[] getVertexes() {
			int v1 = edges[0].v1;
			int v2;
			int v3;
			if (edges[1].contains(v1)) {
				v2 = v1;
				v1 = edges[0].v2;
			} else {
				v2 = edges[0].v2;
			}
			v3 = edges[1].opposite(v2);
			return new int[] { v1, v2, v3 };
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#toString()
		 */
		@Override
		public String toString() {
			return "FACE " + id + ") " + edges[0] + " " + edges[1] + " "
			+ edges[2];
		}
	}

	/**
	 * Builds the edge hash.
	 * 
	 * @param edges the edges
	 * 
	 * @return the hashtable< long, edge>
	 */
	public static Hashtable<Long, Edge> buildEdgeHash(Edge[] edges) {
		Hashtable<Long, Edge> edgeHash = new Hashtable<Long, Edge>(edges.length);
		for (Edge e : edges) {
			edgeHash.put(e.hashCodeLong(), e);
		}
		return edgeHash;
	}

	/**
	 * Builds the edge table.
	 * 
	 * @param mesh the mesh
	 * 
	 * @return the edge[]
	 */
	public static Edge[] buildEdgeTable(ModelTriangleMesh mesh) {
		int indexCount = mesh.getIndexCount();
		Edge[] edgeTable = new Edge[indexCount / 2];
		int v1, v2, v3;
		int index = 0;
		for (int i = 0; i < indexCount; i += 3) {
			v1 = mesh.getCoordinateIndex(i);
			v2 = mesh.getCoordinateIndex(i + 1);
			v3 = mesh.getCoordinateIndex(i + 2);
			if (v1 < v2) {
				edgeTable[index] = new Edge(v1, v2, index);
				index++;
			}
			if (v2 < v3) {
				edgeTable[index] = new Edge(v2, v3, index);
				index++;
			}
			if (v3 < v1) {
				edgeTable[index] = new Edge(v3, v1, index);
				index++;
			}
		}
		return edgeTable;
	}

	/**
	 * Builds the face table.
	 * 
	 * @param mesh the mesh
	 * @param edges the edges
	 * @param edgeHash the edge hash
	 * 
	 * @return the face[]
	 */
	public static Face[] buildFaceTable(ModelTriangleMesh mesh, Edge[] edges,
			Hashtable<Long, Edge> edgeHash) {
		int indexCount = mesh.getIndexCount();
		Face[] faceTable = new Face[indexCount / 3];
		int v1, v2, v3;
		int index = 0;
		Face f;

		for (int i = 0; i < indexCount; i += 3) {
			v1 = mesh.getCoordinateIndex(i);
			v2 = mesh.getCoordinateIndex(i + 1);
			v3 = mesh.getCoordinateIndex(i + 2);
			faceTable[i / 3] = f = new Face(i / 3);
			if (v1 < v2) {
				f.edges[0] = edges[index];
				index++;
			} else {
				f.edges[0] = edgeHash.get(Edge.hashCodeLong(v2, v1));
			}
			if (v2 < v3) {
				f.edges[1] = edges[index];
				index++;
			} else {
				f.edges[1] = edgeHash.get(Edge.hashCodeLong(v3, v2));
			}
			if (v3 < v1) {
				f.edges[2] = edges[index];
				index++;
			} else {
				f.edges[2] = edgeHash.get(Edge.hashCodeLong(v1, v3));
			}
		}
		return faceTable;
	}

	/**
	 * Builds the neighbor edge face table.
	 * 
	 * @param mesh the mesh
	 * @param faces the faces
	 * @param edgeHash the edge hash
	 * @param edges the edges
	 * 
	 * @return the face[][]
	 */
	public static Face[][] buildNeighborEdgeFaceTable(ModelTriangleMesh mesh,
			Face[] faces, Hashtable<Long, Edge> edgeHash,Edge[] edges) {
		int indexCount = mesh.getIndexCount();
		Face[][] neighborFaceTable = new Face[indexCount / 2][2];
		int v1, v2, v3;
		int index = 0;
		int tmp = 0;
		Edge e;
		for (int i = 0; i < indexCount; i += 3) {
			v1 = mesh.getCoordinateIndex(i);
			v2 = mesh.getCoordinateIndex(i + 1);
			v3 = mesh.getCoordinateIndex(i + 2);

			if (v1 < v2) {
				neighborFaceTable[index][0] = faces[i / 3];
				edges[index].f1=i / 3;
				index++;
			} else {
				tmp = (e=edgeHash.get(Edge.hashCodeLong(v2, v1))).id;
				neighborFaceTable[tmp][1] = faces[i / 3];
				e.f2=i/3;
			}
			if (v2 < v3) {

				neighborFaceTable[index][0] = faces[i / 3];
				edges[index].f1=i / 3;
				index++;
			} else {
				tmp = (e=edgeHash.get(Edge.hashCodeLong(v3, v2))).id;
				e.f2=i/3;
				neighborFaceTable[tmp][1] = faces[i / 3];
			}
			if (v3 < v1) {
				neighborFaceTable[index][0] = faces[i / 3];
				edges[index].f1=i / 3;
				index++;
			} else {
				;
				tmp = (e=edgeHash.get(Edge.hashCodeLong(v1, v3))).id;
				neighborFaceTable[tmp][1] = faces[i / 3];
				e.f2=i / 3;
			}
		}
		return neighborFaceTable;
	}

	/**
	 * Builds the neighbor face face table.
	 * 
	 * @param mesh the mesh
	 * @param faces the faces
	 * @param neighborEdgeFaceTable the neighbor edge face table
	 * 
	 * @return the face[][]
	 */
	public static Face[][] buildNeighborFaceFaceTable(ModelTriangleMesh mesh,
			Face[] faces, Face[][] neighborEdgeFaceTable) {
		int faceCount = faces.length;
		Face[][] neighborFaceFaceTable = new Face[faceCount][3];
		int j;
		for (int i = 0; i < faceCount; i++) {
			Face f = faces[i];
			j = 0;
			for (Edge edge : f.edges) {
				neighborFaceFaceTable[i][j++] = (neighborEdgeFaceTable[edge.id][0] == f) ? neighborEdgeFaceTable[edge.id][1]
				                                                                                                          : neighborEdgeFaceTable[edge.id][0];
			}
		}
		return neighborFaceFaceTable;
	}

	/**
	 * Builds the neighbor vertex edge table.
	 * 
	 * @param neighborVertexTable the neighbor vertex table
	 * @param edges the edges
	 * @param edgeHash the edge hash
	 * 
	 * @return the edge[][]
	 */
	public static Edge[][] buildNeighborVertexEdgeTable(
			int[][] neighborVertexTable, Edge[] edges,
			Hashtable<Long, Edge> edgeHash) {
		Edge[][] neighborEdgeTable = new Edge[neighborVertexTable.length][0];
		int[] neighbors;
		for (int i = 0; i < neighborVertexTable.length; i++) {
			neighbors = neighborVertexTable[i];
			neighborEdgeTable[i] = new Edge[neighbors.length];
			for (int j = 0; j < neighbors.length; j++) {
				if (i < neighbors[j]) {
					neighborEdgeTable[i][j] = edgeHash.get(Edge.hashCodeLong(i,
							neighbors[j]));
				} else {
					neighborEdgeTable[i][j] = edgeHash.get(Edge.hashCodeLong(
							neighbors[j], i));
				}
			}
		}
		return neighborEdgeTable;
	}

	/**
	 * Builds the all tables.
	 */
	public void buildAllTables(){
		edges=EmbeddedSurface.buildEdgeTable(this);
		Hashtable<Long,EmbeddedSurface.Edge> hash=EmbeddedSurface.buildEdgeHash(edges);
		faces=EmbeddedSurface.buildFaceTable(this, edges,hash);
		neighborVertexVertexTable=EmbeddedSurface.buildNeighborVertexVertexTable(this, EmbeddedSurface.Direction.CLOCKWISE);	
		neighborEdgeFaceTable=EmbeddedSurface.buildNeighborEdgeFaceTable(this,faces,hash,edges);
		neighborVertexEdgeTable=EmbeddedSurface.buildNeighborVertexEdgeTable(neighborVertexVertexTable, edges,hash);
		neighborVertexFaceTable=EmbeddedSurface.buildNeighborVertexFaceTable(neighborVertexEdgeTable,neighborEdgeFaceTable);
		neighborFaceFaceTable=EmbeddedSurface.buildNeighborFaceFaceTable(this, faces, neighborEdgeFaceTable);
	}

	/**
	 * Dispose all tables.
	 */
	public void disposeAllTables(){
		edges=null;
		faces=null;
		neighborEdgeFaceTable=null;
		neighborVertexEdgeTable=null;
		neighborVertexFaceTable=null;
		neighborVertexVertexTable=null;
		System.gc();
	}

	/**
	 * Builds the neighbor vertex face table.
	 * 
	 * @param neighborEdgeTable the neighbor edge table
	 * @param neighborFaceTable the neighbor face table
	 * 
	 * @return the face[][]
	 */
	public static Face[][] buildNeighborVertexFaceTable(
			Edge[][] neighborEdgeTable, Face[][] neighborFaceTable) {
		Face[][] neighborFaceVertexTable = new Face[neighborEdgeTable.length][0];
		Edge[] neighbors;
		Edge e1, e2;
		for (int i = 0; i < neighborEdgeTable.length; i++) {
			neighbors = neighborEdgeTable[i];
			neighborFaceVertexTable[i] = new Face[neighbors.length];
			for (int j = 0; j < neighbors.length; j++) {
				e1 = neighbors[j];
				e2 = neighbors[(j + 1) % neighbors.length];
				if (neighborFaceTable[e1.id][0] == neighborFaceTable[e2.id][0]
				                                                            || neighborFaceTable[e1.id][0] == neighborFaceTable[e2.id][1]) {
					neighborFaceVertexTable[i][j] = neighborFaceTable[e1.id][0];
				} else if (neighborFaceTable[e1.id][1] == neighborFaceTable[e2.id][0]
				                                                                   || neighborFaceTable[e1.id][1] == neighborFaceTable[e2.id][1]) {
					neighborFaceVertexTable[i][j] = neighborFaceTable[e1.id][1];
				} else {
					JistLogger.logError(JistLogger.SEVERE, "jist.base"+"NEIGHBORING FACE NOT FOUND");
				}
			}
		}
		return neighborFaceVertexTable;
	}

	/**
	 * Construct a neighbor table ordered in a particular direction. This
	 * algorithm assumes a closed manifold with no holes
	 * 
	 * @param mesh triangle mesh
	 * @param dir rotation direction
	 * 
	 * @return list of nodes by their neighbors (variable size array list)
	 */
	public static int[][] buildNeighborVertexVertexTable(
			ModelTriangleMesh mesh, Direction dir) {

		int vertexCount = mesh.getVertexCount();
		int indexCount = mesh.getIndexCount();
		int v1, v2, v3;
		int[][] neighborTable = new int[vertexCount][0];
		ArrayList<Integer>[] tmpTable = new ArrayList[vertexCount];
		for (int i = 0; i < vertexCount; i++) {
			tmpTable[i] = new ArrayList<Integer>();
		}
		if (dir == Direction.CLOCKWISE) {
			for (int i = 0; i < indexCount; i += 3) {
				v1 = mesh.getCoordinateIndex(i);
				v2 = mesh.getCoordinateIndex(i + 1);
				v3 = mesh.getCoordinateIndex(i + 2);
				tmpTable[v1].add(v2);
				tmpTable[v1].add(v3);
				tmpTable[v2].add(v3);
				tmpTable[v2].add(v1);
				tmpTable[v3].add(v1);
				tmpTable[v3].add(v2);
			}
		} else if (dir == Direction.COUNTER_CLOCKWISE) {
			for (int i = 0; i < indexCount; i += 3) {
				v1 = mesh.getCoordinateIndex(i);
				v2 = mesh.getCoordinateIndex(i + 1);
				v3 = mesh.getCoordinateIndex(i + 2);
				tmpTable[v1].add(v3);
				tmpTable[v1].add(v2);
				tmpTable[v2].add(v1);
				tmpTable[v2].add(v3);
				tmpTable[v3].add(v2);
				tmpTable[v3].add(v1);
			}
		}
		int pivot;
		int count = 0;
		ArrayList<Integer> neighbors;
		boolean found;
		for (int i = 0; i < vertexCount; i++) {
			neighborTable[i] = new int[tmpTable[i].size() / 2];
			neighbors = tmpTable[i];
			count = 0;
			if(neighbors.size()==0)continue;
			neighbors.remove(0);
			neighborTable[i][count++] = pivot = neighbors.remove(0);
			while (neighbors.size() > 0) {
				found = false;
				for (int k = 0; k < neighbors.size(); k += 2) {
					if (neighbors.get(k) == pivot) {
						neighbors.remove(k);
						neighborTable[i][count++] = pivot = neighbors.remove(k);
						found = true;
						break;
					}
				}
				if (!found) {
					neighbors.remove(0);
					neighborTable[i][count++] = pivot = neighbors.remove(0);
				}
			}
		}
		return neighborTable;
	};

	/**
	 * Convert colors.
	 * 
	 * @param colors the colors
	 * 
	 * @return the color4f[]
	 */
	static private Color4f[] convertColors(Color3f[] colors) {
		Color4f[] colors4 = new Color4f[colors.length];
		for (int i = 0; i < colors.length; i++) {
			colors4[i] = new Color4f(colors[i].get());
		}
		return colors4;
	}

	/**
	 * Gets the genus.
	 * 
	 * @param mesh the mesh
	 * 
	 * @return the genus
	 */
	public static int getGenus(ModelTriangleMesh mesh) {
		int vertCount = mesh.getVertexCount();
		int indexCount = mesh.getIndexCount();
		int faceCount = indexCount / 3;
		// Edge[] edges=new Edge[indexCount];
		long edgeHash[] = new long[indexCount];
		for (int i = 0; i < indexCount; i += 3) {
			int i1 = mesh.getCoordinateIndex(i);
			int i2 = mesh.getCoordinateIndex(i + 1);
			int i3 = mesh.getCoordinateIndex(i + 2);
			if (i1 < i2) {
				edgeHash[i] = i1 + i2 * 10000000l;
			} else {
				edgeHash[i] = i2 + i1 * 10000000l;
			}
			if (i2 < i3) {
				edgeHash[i + 1] = i2 + i3 * 10000000l;
			} else {
				edgeHash[i + 1] = i3 + i2 * 10000000l;
			}
			if (i3 < i1) {
				edgeHash[i + 2] = i3 + i1 * 10000000l;
			} else {
				edgeHash[i + 2] = i1 + i3 * 10000000l;
			}
		}
		Arrays.sort(edgeHash);
		long lastIndex = -1;
		int edgeCount = 1;
		for (long index : edgeHash) {
			if (index != lastIndex) {
				edgeCount++;
			}
			lastIndex = index;
		}
		return -(vertCount - edgeCount + faceCount - 2) / 2;
	}

	/** The vertex data. */
	protected double[][] vertexData;

	/** The cell data. */
	protected double[][] cellData;

	/**
	 * Gets the edge table.
	 * 
	 * @return the edge table
	 */
	public int[][] getEdgeTable() {
		return edgeTable;
	}

	/**
	 * Gets the neighbor face face table.
	 * 
	 * @return the neighbor face face table
	 */
	public EmbeddedSurface.Face[][] getNeighborFaceFaceTable() {
		return neighborFaceFaceTable;
	}

	/**
	 * Gets the neighbor vertex vertex table.
	 * 
	 * @return the neighbor vertex vertex table
	 */
	public int[][] getNeighborVertexVertexTable() {
		return neighborVertexVertexTable;
	}

	/**
	 * Gets the neighbor edge face table.
	 * 
	 * @return the neighbor edge face table
	 */
	public EmbeddedSurface.Face[][] getNeighborEdgeFaceTable() {
		return neighborEdgeFaceTable;
	}

	/**
	 * Gets the neighbor vertex face table.
	 * 
	 * @return the neighbor vertex face table
	 */
	public EmbeddedSurface.Face[][] getNeighborVertexFaceTable() {
		return neighborVertexFaceTable;
	}

	/**
	 * Gets the edges.
	 * 
	 * @return the edges
	 */
	public EmbeddedSurface.Edge[] getEdges() {
		return edges;
	}

	/**
	 * Gets the neighbor vertex edge table.
	 * 
	 * @return the neighbor vertex edge table
	 */
	public EmbeddedSurface.Edge[][] getNeighborVertexEdgeTable() {
		return neighborVertexEdgeTable;
	}

	/**
	 * Gets the faces.
	 * 
	 * @return the faces
	 */
	public EmbeddedSurface.Face[] getFaces() {
		return faces;
	}

	/** The edge table. */
	protected int[][] edgeTable;

	/** The neighbor vertex vertex table. */
	protected int[][] neighborVertexVertexTable;

	/** The neighbor edge face table. */
	protected EmbeddedSurface.Face[][] neighborEdgeFaceTable;

	/** The neighbor vertex face table. */
	protected EmbeddedSurface.Face[][] neighborVertexFaceTable;

	/** The edges. */
	protected EmbeddedSurface.Edge[] edges;

	/** The neighbor vertex edge table. */
	protected EmbeddedSurface.Edge[][] neighborVertexEdgeTable;

	/** The faces. */
	protected EmbeddedSurface.Face[] faces;

	/** The neighbor face face table. */
	protected EmbeddedSurface.Face[][] neighborFaceFaceTable;

	/** The origin. */
	protected Point3f origin = new Point3f(0, 0, 0);

	/** The scale. */
	protected Vector3f scale = new Vector3f(1, 1, 1);

	/**
	 * Instantiates a new embedded surface.
	 * 
	 * @param mesh the mesh
	 */
	public EmbeddedSurface(EmbeddedSurface mesh) {
		this((ModelTriangleMesh) mesh);
		double[][] dat = mesh.getVertexData();
		if (dat != null && dat.length > 0) {
			vertexData = new double[dat.length][dat[0].length];
		} else {
			vertexData = new double[dat.length][0];
		}
		if (dat != null) {
			for (int i = 0; i < dat.length; i++) {
				for (int j = 0; j < dat[0].length; j++) {
					vertexData[i][j] = dat[i][j];
				}
			}
		}
		this.setName(mesh.getName());
	}

	/**
	 * Instantiates a new embedded surface.
	 * 
	 * @param mesh the mesh
	 * @param data the data
	 */
	public EmbeddedSurface(EmbeddedSurface mesh, double[][] data) {
		super(mesh.getVertexCopy(), mesh.getNormalCopy(), mesh.getIndexCopy());
		if (data.length != mesh.getVertexCount()) {
			System.err
			.println("Not the same number of data labels and mesh vertexes");
			//System.exit(1);
			return;
		}
		this.setName(mesh.getName());
		double[][] oldData = mesh.getVertexData();

		if (oldData != null) {
			int off = oldData[0].length;
			double[][] newData = new double[oldData.length][off
			                                                + data[0].length];

			for (int i = 0; i < oldData.length; i++) {
				for (int j = 0; j < oldData[i].length; j++) {
					newData[i][j] = oldData[i][j];
				}
				for (int j = 0; j < data[i].length; j++) {
					newData[i][j + off] = data[i][j];
				}
			}
			this.vertexData = newData;
		} else {
			this.vertexData = data;
		}
	}

	/**
	 * Instantiates a new embedded surface.
	 * 
	 * @param mesh the mesh
	 */
	public EmbeddedSurface(ModelTriangleMesh mesh) {
		super(mesh.getVertexCopy(), mesh.getNormalCopy(), mesh.getIndexCopy());
		setName("surface");
		vertexData = new double[this.getVertexCount()][0];
		this.setName(mesh.getName());
	}

	/**
	 * Instantiates a new embedded surface.
	 * 
	 * @param mesh the mesh
	 * @param data the data
	 */
	public EmbeddedSurface(ModelTriangleMesh mesh, double[][] data) {
		super(mesh.getVertexCopy(), mesh.getNormalCopy(), mesh.getIndexCopy());
		this.setName(mesh.getName());
		this.vertexData = data;
	}

	/**
	 * Instantiates a new embedded surface.
	 * 
	 * @param points the points
	 * @param indexes the indexes
	 */
	public EmbeddedSurface(Point3f[] points, int[] indexes) {
		super(points, NormalGenerator.generate(points, indexes), indexes);
		setName("surface");
		vertexData = new double[points.length][0];
	}

	/**
	 * Instantiates a new embedded surface.
	 * 
	 * @param points the points
	 * @param indexes the indexes
	 * @param data the data
	 */
	public EmbeddedSurface(Point3f[] points, int[] indexes, double[][] data) {
		super(points, NormalGenerator.generate(points, indexes), indexes);
		setName("surface");
		this.vertexData = data;
	}

	/**
	 * Instantiates a new embedded surface.
	 * 
	 * @param points the points
	 * @param arg1 the arg1
	 * @param arg2 the arg2
	 * @param arg3 the arg3
	 */
	public EmbeddedSurface(Point3f[] points, Vector3f[] arg1, Color3f[] arg2,
			int[] arg3) {
		this(points, arg1, convertColors(arg2), arg3);
	}

	/**
	 * Instantiates a new embedded surface.
	 * 
	 * @param points the points
	 * @param arg1 the arg1
	 * @param arg2 the arg2
	 * @param arg3 the arg3
	 * @param data the data
	 */
	public EmbeddedSurface(Point3f[] points, Vector3f[] arg1, Color3f[] arg2,
			int[] arg3, double[][] data) {
		this(points, arg1, convertColors(arg2), arg3, data);
	}

	/**
	 * Instantiates a new embedded surface.
	 * 
	 * @param points the points
	 * @param arg1 the arg1
	 * @param arg2 the arg2
	 * @param arg3 the arg3
	 */
	public EmbeddedSurface(Point3f[] points, Vector3f[] arg1, Color4f[] arg2,
			int[] arg3) {
		super(points, arg1, arg2, arg3);
		setName("surface");
		vertexData = new double[points.length][0];
	}

	/**
	 * Instantiates a new embedded surface.
	 * 
	 * @param points the points
	 * @param arg1 the arg1
	 * @param arg2 the arg2
	 * @param arg3 the arg3
	 * @param data the data
	 */
	public EmbeddedSurface(Point3f[] points, Vector3f[] arg1, Color4f[] arg2,
			int[] arg3, double[][] data) {
		super(points, arg1, arg2, arg3);
		setName("surface");
		this.vertexData = data;
	}

	/**
	 * Instantiates a new embedded surface.
	 * 
	 * @param points the points
	 * @param normals the normals
	 * @param indexes the indexes
	 */
	public EmbeddedSurface(Point3f[] points, Vector3f[] normals, int[] indexes) {
		super(points, normals, indexes);
		setName("surface");
		vertexData = new double[points.length][0];
	}

	/**
	 * Instantiates a new embedded surface.
	 * 
	 * @param points the points
	 * @param normals the normals
	 * @param indexes the indexes
	 * @param data the data
	 */
	public EmbeddedSurface(Point3f[] points, Vector3f[] normals, int[] indexes,
			double[][] data) {
		super(points, normals, indexes);
		setName("surface");
		this.vertexData = data;
	}
	/*
	public void buildAllLookUpTables() {
		edges = EmbeddedSurface.buildEdgeTable(this);
		Hashtable<Long, EmbeddedSurface.Edge> hash = EmbeddedSurface
				.buildEdgeHash(edges);
		faces = EmbeddedSurface.buildFaceTable(this, edges, hash);
		neighborVertexVertexTable = EmbeddedSurface
				.buildNeighborVertexVertexTable(this,
						EmbeddedSurface.Direction.CLOCKWISE);
		neighborEdgeFaceTable = EmbeddedSurface.buildNeighborEdgeFaceTable(
				this, faces, hash);
		neighborVertexEdgeTable = EmbeddedSurface.buildNeighborVertexEdgeTable(
				neighborVertexVertexTable, edges, hash);
		neighborVertexFaceTable = EmbeddedSurface.buildNeighborVertexFaceTable(
				neighborVertexEdgeTable, neighborEdgeFaceTable);
	}
	 */
	/* (non-Javadoc)
	 * @see java.lang.Object#clone()
	 */
	@Override
	public EmbeddedSurface clone() {
		return new EmbeddedSurface(this);
	}

	/* (non-Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(EmbeddedSurface surf) {
		return (int) Math.signum(this.getVertexCount() - surf.getVertexCount());
	}

	/**
	 * Convert to cartesian.
	 */
	public void convertToCartesian() {
		if (vertexData[0].length == 5) {
			for (int i = 0; i < vertexData.length; i++) {
				Point3f r = GeomUtil.toCartesian(new Point2d(vertexData[i][1],
						vertexData[i][2]));
				Point3f r2 = GeomUtil.toCartesian(new Point2d(vertexData[i][3],
						vertexData[i][4]));
				double thick = vertexData[i][0];
				setVertexData(i, new double[] { thick, r.x, r.y, r.z, r2.x,
						r2.y, r2.z });
			}
		} else {
			JistLogger.logError(JistLogger.WARNING, getClass().getCanonicalName()+"\t"+"NOT CORRECT DIMENSION " + vertexData[0].length);
		}
	}

	/**
	 * Convert to spherical.
	 */
	public void convertToSpherical() {
		if (vertexData[0].length == 7) {
			for (int i = 0; i < vertexData.length; i++) {
				Point2d r = GeomUtil.toUnitSpherical(getInnerPoint(i));
				Point2d r2 = GeomUtil.toUnitSpherical(getOuterPoint(i));
				double thick = getThickness(i);
				setVertexData(i, new double[] { thick, r.x, r.y, r2.x, r2.y });
			}
		} else {
			JistLogger.logError(JistLogger.WARNING, getClass().getCanonicalName()+"\t"+"NOT CORRECT DIMENSION " + vertexData[0].length);
		}
	}


	/* (non-Javadoc)
	 * @see gov.nih.mipav.view.renderer.J3D.model.structures.ModelTriangleMesh#getCellData()
	 */
	public double[][] getCellData() {
		return cellData;
	}

	/**
	 * Gets the center of mass.
	 * 
	 * @return the center of mass
	 */
	public Point3f getCenterOfMass() {
		int indexCount = getIndexCount();
		// Compute center of mass
		Point3f p1, p2, p3;
		Vector3f edge1 = new Vector3f();
		Vector3f edge2 = new Vector3f();
		Vector3f norm = new Vector3f();
		Point3f centroid = new Point3f();
		Point3f massCenter = new Point3f();
		float areaSum = 0;
		float area;
		for (int in = 0; in < indexCount; in += 3) {
			p1 = getVertex(getCoordinateIndex(in));
			p2 = getVertex(getCoordinateIndex(in + 1));
			p3 = getVertex(getCoordinateIndex(in + 2));
			edge1.sub(p2, p1);
			edge2.sub(p3, p1);
			norm.cross(edge2, edge1);
			area = 0.5f * norm.length();
			centroid = new Point3f(p1);
			centroid.add(p2);
			centroid.add(p3);
			centroid.scale(0.333333f * area);
			massCenter.add(centroid);
			areaSum += area;
		}
		massCenter.scale(1.0f / areaSum);
		return massCenter;
	}

	/**
	 * Gets the centroid.
	 * 
	 * @param fid the fid
	 * 
	 * @return the centroid
	 */
	public Point3f getCentroid(int fid) {
		fid *= 3;
		Point3f p1 = getVertex(getCoordinateIndex(fid));
		Point3f p2 = getVertex(getCoordinateIndex(fid + 1));
		Point3f p3 = getVertex(getCoordinateIndex(fid + 2));
		p1.add(p2);
		p1.add(p3);
		p1.scale(0.333333333333f);
		return p1;
	}

	/**
	 * Gets the centroid at offset.
	 * 
	 * @param fid the fid
	 * @param offset the offset
	 * 
	 * @return the centroid at offset
	 */
	public Point3f getCentroidAtOffset(int fid, int offset) {
		fid *= 3;
		Point3f p1 = getPointAtOffset(getCoordinateIndex(fid), offset);
		Point3f p2 = getPointAtOffset(getCoordinateIndex(fid + 1), offset);
		Point3f p3 = getPointAtOffset(getCoordinateIndex(fid + 2), offset);
		p1.add(p2);
		p1.add(p3);
		p1.scale(0.333333333333f);
		return p1;
	}

	/**
	 * Gets the embedded sphere.
	 * 
	 * @param offset the offset
	 * @param polar the polar
	 * 
	 * @return the embedded sphere
	 */
	public EmbeddedSurface getEmbeddedSphere(int offset, boolean polar) {
		int vertCount = this.getVertexCount();
		int indexCount = this.getIndexCount();
		Point3f[] verts = new Point3f[vertCount];
		int[] indexes = new int[indexCount];
		Color3f[] color = new Color3f[vertCount];
		Point2d r;
		for (int i = 0; i < vertCount; i++) {
			if (polar) {
				r = new Point2d(vertexData[i][offset],
						vertexData[i][offset + 1]);
				verts[i] = GeomUtil.toCartesian(r);
			} else {
				verts[i] = new Point3f((float) vertexData[i][offset],
						(float) vertexData[i][offset + 1],
						(float) vertexData[i][offset + 2]);
				r = GeomUtil.toUnitSpherical(verts[i]);
			}
			color[i] = new Color3f(new Color(Color.HSBtoRGB((float) (r.x
					/ Math.PI + 0.5f), 1, (float) (0.2 + 0.8 * r.y / Math.PI))));
		}
		for (int i = 0; i < indexCount; i++) {
			indexes[i] = this.getCoordinateIndex(i);
		}
		EmbeddedSurface sphere = new EmbeddedSurface(verts, NormalGenerator
				.generate(verts, indexes), color, indexes);
		sphere.setName(this.getName() + "_sphere");
		return sphere;
	}

	/**
	 * Gets the face area.
	 * 
	 * @param fid the fid
	 * 
	 * @return the face area
	 */
	public float getFaceArea(int fid) {
		fid *= 3;
		Point3f p1 = getVertex(getCoordinateIndex(fid));
		Point3f p2 = getVertex(getCoordinateIndex(fid + 1));
		Point3f p3 = getVertex(getCoordinateIndex(fid + 2));
		Vector3f edge1 = new Vector3f();
		Vector3f edge2 = new Vector3f();
		Vector3f cross = new Vector3f();
		edge1.sub(p2, p1);
		edge2.sub(p3, p1);
		cross.cross(edge1, edge2);
		return cross.length();
	}

	/**
	 * Gets the face count.
	 * 
	 * @return the face count
	 */
	public int getFaceCount() {
		return getIndexCount() / 3;
	}

	/**
	 * Gets the face points.
	 * 
	 * @param fid the fid
	 * 
	 * @return the face points
	 */
	public Point3f[] getFacePoints(int fid) {
		fid *= 3;
		Point3f[] pts = new Point3f[3];
		pts[0] = getVertex(getCoordinateIndex(fid));
		pts[1] = getVertex(getCoordinateIndex(fid + 1));
		pts[2] = getVertex(getCoordinateIndex(fid + 2));
		return pts;
	}

	/**
	 * Gets the face vertex ids.
	 * 
	 * @param fid the fid
	 * 
	 * @return the face vertex ids
	 */
	public int[] getFaceVertexIds(int fid) {
		fid *= 3;
		int[] pts = new int[3];
		pts[0] = (getCoordinateIndex(fid));
		pts[1] = (getCoordinateIndex(fid + 1));
		pts[2] = (getCoordinateIndex(fid + 2));
		return pts;
	}

	/**
	 * Gets the inner point.
	 * 
	 * @param i the i
	 * 
	 * @return the inner point
	 */
	public Point3f getInnerPoint(int i) {
		return new Point3f((float) vertexData[i][1], (float) vertexData[i][2],
				(float) vertexData[i][3]);
	}

	/**
	 * Gets the max.
	 * 
	 * @return the max
	 */
	public Point3f getMax() {
		Point3f max = new Point3f(Float.MIN_VALUE, Float.MIN_VALUE,
				Float.MIN_VALUE);
		for (int i = 0; i < getVertexCount(); i++) {
			Point3f p = getVertex(i);
			max.x = Math.max(p.x, max.x);
			max.y = Math.max(p.y, max.y);
			max.z = Math.max(p.z, max.z);
		}
		return max;
	}

	/**
	 * Gets the max angle.
	 * 
	 * @param fid the fid
	 * 
	 * @return the max angle
	 */
	public float getMaxAngle(int fid) {
		fid *= 3;
		Point3f p1 = getVertex(getCoordinateIndex(fid));
		Point3f p2 = getVertex(getCoordinateIndex(fid + 1));
		Point3f p3 = getVertex(getCoordinateIndex(fid + 2));
		Vector3f edge1 = new Vector3f();
		Vector3f edge2 = new Vector3f();
		Vector3f edge3 = new Vector3f();
		edge1.sub(p2, p1);
		edge2.sub(p3, p1);
		edge3.sub(p3, p2);
		float angle = Math.max(
				Math.max(edge1.angle(edge2), edge2.angle(edge3)), edge1
				.angle(edge3));
		return angle;
	}

	/**
	 * Gets the min.
	 * 
	 * @return the min
	 */
	public Point3f getMin() {
		Point3f min = new Point3f(Float.MAX_VALUE, Float.MAX_VALUE,
				Float.MAX_VALUE);
		for (int i = 0; i < getVertexCount(); i++) {
			Point3f p = getVertex(i);
			min.x = Math.min(p.x, min.x);
			min.y = Math.min(p.y, min.y);
			min.z = Math.min(p.z, min.z);
		}
		return min;
	}

	/**
	 * Gets the min angle.
	 * 
	 * @param fid the fid
	 * 
	 * @return the min angle
	 */
	public float getMinAngle(int fid) {
		fid *= 3;
		Point3f p1 = getVertex(getCoordinateIndex(fid));
		Point3f p2 = getVertex(getCoordinateIndex(fid + 1));
		Point3f p3 = getVertex(getCoordinateIndex(fid + 2));
		Vector3f edge1 = new Vector3f();
		Vector3f edge2 = new Vector3f();
		Vector3f edge3 = new Vector3f();
		edge1.sub(p2, p1);
		edge2.sub(p3, p1);
		edge3.sub(p3, p2);
		float angle = Math.min(
				Math.min(edge1.angle(edge2), edge2.angle(edge3)), edge1
				.angle(edge3));
		return angle;
	}

	/**
	 * Gets the normal.
	 * 
	 * @param i the i
	 * 
	 * @return the normal
	 */
	public Vector3f getNormal(int i) {
		Vector3f p = new Vector3f();
		this.getNormal(i, p);
		return p;
	}

	/**
	 * Gets the number of holes.
	 * 
	 * @return the number of holes
	 */
	public long getNumberOfHoles() {
		return getGenus(this);
	}

	/**
	 * Gets the origin.
	 * 
	 * @return the origin
	 */
	public Point3f getOrigin() {
		return origin;
	}

	/**
	 * Gets the outer point.
	 * 
	 * @param i the i
	 * 
	 * @return the outer point
	 */
	public Point3f getOuterPoint(int i) {
		return new Point3f((float) vertexData[i][4], (float) vertexData[i][5],
				(float) vertexData[i][6]);
	}

	/**
	 * Gets the point at offset.
	 * 
	 * @param i the i
	 * @param offset the offset
	 * 
	 * @return the point at offset
	 */
	public Point3f getPointAtOffset(int i, int offset) {
		return new Point3f((float) vertexData[i][offset],
				(float) vertexData[i][offset + 1],
				(float) vertexData[i][offset + 2]);
	}

	/**
	 * Gets the scale.
	 * 
	 * @return the scale
	 */
	public Vector3f getScale() {
		return scale;
	}

	/**
	 * Gets the statistics.
	 * 
	 * @param i the i
	 * 
	 * @return the statistics
	 */
	protected double[] getStatistics(int i) {
		int count = getVertexCount();
		double sum = 0;
		double sqrs = 0;
		double val;
		double min = Double.MAX_VALUE;
		double max = Double.MIN_VALUE;
		for (int n = 0; n < count; n++) {
			val = getVertexData(n)[i];
			sum += val;
			sqrs += val * val;
			min = Math.min(min, val);
			max = Math.max(max, val);
		}
		double mean = (sum / count);
		double stdev = Math.sqrt((sqrs - sum * sum / count) / (count - 1));
		return new double[] { mean, stdev, min, max };
	}

	/**
	 * Gets the stats.
	 * 
	 * @return the stats
	 */
	public double[] getStats() {
		double sqrs = 0;
		double sum = 0;
		double count = 0;
		double maxThickness = 0;
		double minThickness = 1E10;
		double thick = 0;
		for (int i = 0; i < vertexData.length; i++) {
			thick = (vertexData[i].length > 0) ? vertexData[i][0] : 0;
			maxThickness = Math.max(thick, maxThickness);
			minThickness = Math.min(thick, minThickness);
			sqrs += (thick) * (thick);
			sum += thick;
			count++;
		}
		return new double[] { (sum / count),
				Math.sqrt((sqrs - sum * sum / count) / (count - 1)),
				minThickness, maxThickness };
	}

	/**
	 * Gets the thickness.
	 * 
	 * @param i the i
	 * 
	 * @return the thickness
	 */
	public double getThickness(int i) {
		return vertexData[i][0];
	}
	/*
	public IntersectorTriangle getTriangle(int i) {
		IntersectorTriangle tri = new IntersectorTriangle(this.getCoordinateIndex(3 * i),
				this.getCoordinateIndex(3 * i + 1), this
						.getCoordinateIndex(3 * i + 2), this);
		return tri;
	}
	 */
	/**
	 * Gets the vector at offset.
	 * 
	 * @param i the i
	 * @param offset the offset
	 * 
	 * @return the vector at offset
	 */
	public Vector3f getVectorAtOffset(int i, int offset) {
		return new Vector3f((float) vertexData[i][offset],
				(float) vertexData[i][offset + 1],
				(float) vertexData[i][offset + 2]);
	}

	/**
	 * Gets the vertex.
	 * 
	 * @param i the i
	 * 
	 * @return the vertex
	 */
	public Point3f getVertex(int i) {
		Point3f p = new Point3f();
		this.getCoordinate(i, p);
		return p;
	}

	/**
	 * Gets the vertex double copy.
	 * 
	 * @return the vertex double copy
	 */
	public double[] getVertexDoubleCopy() {
		int vertCount=this.getVertexCount();
		double[] pts = new double[vertCount*3];
		for(int i=0;i<vertCount;i++){
			Point3d p=new Point3d();
			this.getCoordinate(i, p);
			pts[i*3]=p.x;
			pts[i*3+1]=p.y;
			pts[i*3+2]=p.z;
		}
		return pts;
	}

	/* (non-Javadoc)
	 * @see gov.nih.mipav.view.renderer.J3D.model.structures.ModelTriangleMesh#getVertexData()
	 */
	public double[][] getVertexData() {
		return vertexData;
	}


	/* (non-Javadoc)
	 * @see gov.nih.mipav.view.renderer.J3D.model.structures.ModelTriangleMesh#getVertexData(int)
	 */
	public double[] getVertexData(int i) {
		if (vertexData == null) {
			return null;
		} else {
			return vertexData[i];
		}
	}

	/**
	 * Gets the vertex data at offset.
	 * 
	 * @param i the i
	 * @param offset the offset
	 * 
	 * @return the vertex data at offset
	 */
	public double getVertexDataAtOffset(int i, int offset) {
		return vertexData[i][offset];
	}

	/**
	 * Checks for embedded data.
	 * 
	 * @return true, if successful
	 */
	public boolean hasEmbeddedData() {
		return (vertexData != null && vertexData.length > 0 && vertexData[0].length > 0);
	}

	/**
	 * Mid point.
	 * 
	 * @param e the e
	 * 
	 * @return the point3f
	 */
	public Point3f midPoint(Edge e) {
		Point3f p1 = getVertex(e.v1);
		Point3f p2 = getVertex(e.v2);
		return new Point3f((p1.x + p2.x) * 0.5f, (p1.y + p2.y) * 0.5f,
				(p1.z + p2.z) * 0.5f);
	}

	/**
	 * Mid values.
	 * 
	 * @param e the e
	 * 
	 * @return the double[]
	 */
	public double[] midValues(Edge e) {
		double[] ret = new double[vertexData[0].length];
		for (int i = 0; i < ret.length; i++) {
			ret[i] = 0.5 * (vertexData[e.v1][i] + vertexData[e.v2][i]);
		}
		return ret;
	}

	/**
	 * Repair degenerate mappings.
	 * 
	 * @param minDist the min dist
	 * @param perturbation the perturbation
	 */
	public void repairDegenerateMappings(float minDist, float perturbation) {
		edges = EmbeddedSurface.buildEdgeTable(this);
		Hashtable<Long, EmbeddedSurface.Edge> hash = EmbeddedSurface
		.buildEdgeHash(edges);
		faces = EmbeddedSurface.buildFaceTable(this, edges, hash);
		neighborVertexVertexTable = EmbeddedSurface
		.buildNeighborVertexVertexTable(this,
				EmbeddedSurface.Direction.CLOCKWISE);
		int vertexCount = this.getVertexCount();
		int nb;
		Point3f np;
		float len;
		for (int i = 0; i < vertexCount; i++) {
			Point3f p = getInnerPoint(i);
			for (int j = 0; j < neighborVertexVertexTable[i].length; j++) {
				nb = neighborVertexVertexTable[i][j];
				np = this.getInnerPoint(nb);
				if (np.distance(p) < minDist) {
					for (int k = 0; k < neighborVertexVertexTable[i].length; k++) {
						nb = neighborVertexVertexTable[i][k];
						np = this.getInnerPoint(nb);
						if ((len = np.distance(p)) > perturbation) {
							np = new Point3f(p.x + perturbation * (np.x - p.x)
									/ len, p.y + perturbation * (np.y - p.y)
									/ len, p.z + perturbation * (np.z - p.z)
									/ len);
							// "+nb);
							this.setInnerPoint(i, np);
							break;
						}
					}
				}
			}

		}
		edges = null;
		faces = null;
		neighborVertexVertexTable = null;
	}

	/**
	 * Repair degenerate triangles.
	 * 
	 * @param minDist the min dist
	 * @param perturbation the perturbation
	 */
	public void repairDegenerateTriangles(float minDist, float perturbation) {
		edges = EmbeddedSurface.buildEdgeTable(this);
		Hashtable<Long, EmbeddedSurface.Edge> hash = EmbeddedSurface
		.buildEdgeHash(edges);
		faces = EmbeddedSurface.buildFaceTable(this, edges, hash);
		neighborVertexVertexTable = EmbeddedSurface
		.buildNeighborVertexVertexTable(this,
				EmbeddedSurface.Direction.CLOCKWISE);
		int vertexCount = this.getVertexCount();
		int nb;
		Point3f np;
		float len;
		for (int i = 0; i < vertexCount; i++) {
			Point3f p = getVertex(i);
			for (int j = 0; j < neighborVertexVertexTable[i].length; j++) {
				nb = neighborVertexVertexTable[i][j];
				np = this.getVertex(nb);
				if (np.distance(p) < minDist) {
					for (int k = 0; k < neighborVertexVertexTable[i].length; k++) {
						nb = neighborVertexVertexTable[i][k];
						np = this.getVertex(nb);
						if ((len = np.distance(p)) > 2 * perturbation) {
							np = new Point3f(p.x + perturbation * (np.x - p.x)
									/ len, p.y + perturbation * (np.y - p.y)
									/ len, p.z + perturbation * (np.z - p.z)
									/ len);
							this.setVertex(i, np);
							break;
						}
					}
				}
			}

		}
		edges = null;
		faces = null;
		neighborVertexVertexTable = null;
	}

	/**
	 * Resize data.
	 * 
	 * @param dim the dim
	 */
	public void resizeData(int dim) {
		double[] tmp;
		for (int i = 0; i < vertexData.length; i++) {
			tmp = new double[dim];
			for (int j = 0; j < tmp.length && j < vertexData[i].length; j++) {
				tmp[j] = vertexData[i][j];
			}
			setVertexData(i, tmp);
		}
	}

	/**
	 * Scale data.
	 * 
	 * @param scalar the scalar
	 */
	public void scaleData(float scalar) {
		for (int i = 0; i < vertexData.length; i++) {
			for (int j = 0; j < vertexData[0].length; j++) {
				vertexData[i][j] *= scalar;
			}
		}
	}

	/**
	 * Scale vertices.
	 * 
	 * @param scalar the scalar
	 */
	public void scaleVertices(float scalar) {
		int vertexCount = this.getVertexCount();
		for (int i = 0; i < vertexCount; i++) {
			Point3f p = getVertex(i);
			p.scale(scalar);
			this.setVertex(i, p);
		}
	}

	/**
	 * Rotate vertices.
	 * 
	 * @param theta the theta
	 * @param phi the phi
	 */
	public void rotateVertices(float theta,float phi) {
		Matrix3f m=GeomUtil.rotateMatrix3f(theta, phi);
		int vertexCount = this.getVertexCount();
		for (int i = 0; i < vertexCount; i++) {
			Point3f p = getVertex(i);
			m.transform(p);
			this.setVertex(i, p);
		}
	}

	/**
	 * Scale vertices.
	 * 
	 * @param scalar the scalar
	 */
	public void scaleVertices(float[] scalar) {
		int vertexCount = this.getVertexCount();
		for (int i = 0; i < vertexCount; i++) {
			Point3f p = getVertex(i);
			p.x *= scalar[0];
			p.y *= scalar[1];
			p.z *= scalar[2];
			this.setVertex(i, p);
		}
	}

	/**
	 * Affine transform vertices.
	 * 
	 * @param m the m
	 * @param t the t
	 */
	public void affineTransformVertices(Matrix3f m, Point3f t) {
		int vertexCount = this.getVertexCount();
		for (int i = 0; i < vertexCount; i++) {
			Point3f p = getVertex(i);
			m.transform(p);
			p.add(t);
			this.setVertex(i, p);
		}
	}


	/* (non-Javadoc)
	 * @see gov.nih.mipav.view.renderer.J3D.model.structures.ModelTriangleMesh#setCellData(double[][])
	 */
	public void setCellData(double[][] cellData) {
		this.cellData = cellData;
	}

	/**
	 * Sets the color.
	 * 
	 * @param i the i
	 * @param c the c
	 */
	public void setColor(int i, Color c) {
		this.setColor(i, new Color4f(c));
	}

	/**
	 * Sets the inner point.
	 * 
	 * @param i the i
	 * @param p the p
	 */
	public void setInnerPoint(int i, Point3f p) {
		vertexData[i][1] = p.x;
		vertexData[i][2] = p.y;
		vertexData[i][3] = p.z;
	}

	/**
	 * Sets the origin.
	 * 
	 * @param scale the new origin
	 */
	public void setOrigin(Point3f scale) {
		this.origin = scale;
	}

	/**
	 * Sets the outer point.
	 * 
	 * @param i the i
	 * @param p the p
	 */
	public void setOuterPoint(int i, Point3f p) {
		vertexData[i][4] = p.x;
		vertexData[i][5] = p.y;
		vertexData[i][6] = p.z;
	}

	/**
	 * Sets the scale.
	 * 
	 * @param scale the new scale
	 */
	public void setScale(Vector3f scale) {
		this.scale = scale;
	}

	/**
	 * Sets the thickness.
	 * 
	 * @param i the i
	 * @param thick the thick
	 */
	public void setThickness(int i, double thick) {
		vertexData[i][0] = thick;
	}

	/**
	 * Sets the vertex.
	 * 
	 * @param i the i
	 * @param p the p
	 */
	public void setVertex(int i, Tuple3f p) {
		this.setCoordinate(i, new float[] { p.x, p.y, p.z });
	}


	/* (non-Javadoc)
	 * @see gov.nih.mipav.view.renderer.J3D.model.structures.ModelTriangleMesh#setVertexData(double[][])
	 */
	public void setVertexData(double[][] data) {
		if(data!=null&&getVertexCount()!=data.length){
			JistLogger.logError(JistLogger.SEVERE, "jist.base"+"VERTEX DATA IS NOT CORRECT LENGTH! Expected: "+getVertexCount()+" Actual: "+data.length);
			//System.exit(1);
		} else {
			this.vertexData = data;
		}
	}


	/* (non-Javadoc)
	 * @see gov.nih.mipav.view.renderer.J3D.model.structures.ModelTriangleMesh#setVertexData(int, double)
	 */
	public void setVertexData(int i, double val) {
		vertexData[i] = new double[] { val };
	}


	/* (non-Javadoc)
	 * @see gov.nih.mipav.view.renderer.J3D.model.structures.ModelTriangleMesh#setVertexData(int, double[])
	 */
	public void setVertexData(int i, double[] array) {
		vertexData[i] = array;
	}

	/* (non-Javadoc)
	 * @see gov.nih.mipav.view.renderer.J3D.model.structures.ModelTriangleMesh#setVertexData(int, int, double)
	 */
	public void setVertexData(int i, int j, double val) {
		vertexData[i][j] = val;
	}

	/* (non-Javadoc)
	 * @see javax.media.j3d.SceneGraphObject#toString()
	 */
	@Override
	public String toString() {
		return getName();
	}

	/**
	 * Translate.
	 * 
	 * @param offset the offset
	 */
	public void translate(Point3f offset) {
		int vertexCount = this.getVertexCount();
		for (int i = 0; i < vertexCount; i++) {
			Point3f p = getVertex(i);
			p.add(offset);
			this.setVertex(i, p);
		}
	}

	/**
	 * Transform.
	 * 
	 * @param m the m
	 */
	public void transform(Matrix m) {
		int vertexCount = this.getVertexCount();
		Matrix vec=new Matrix(4,1);
		vec.set(3, 0, 1);
		for (int i = 0; i < vertexCount; i++) {
			Point3f p = getVertex(i);
			vec.set(0, 0, p.x);
			vec.set(1, 0, p.y);
			vec.set(2, 0, p.z);
			Matrix r=m.times(vec);
			double w=r.get(3, 0);
			this.setVertex(i, new Point3f((float)(r.get(0, 0)/w),(float)(r.get(1, 0)/w),(float)(r.get(2, 0)/w)));
		}
	}

	/**
	 * Uncrop.
	 * 
	 * @param params the params
	 */
	/*public void uncrop(CropParameters params) {
		Point3f offset = new Point3f(params.xmin, params.ymin, params.zmin);
		translate(offset);
	}*/

	/**
	 * Gets the texture coordinates.
	 * 
	 * @return the texture coordinates
	 */
	public double[][] getTextureCoordinates() {
		return textureCoords;
	}

	/**
	 * Sets the texture coordinates.
	 * 
	 * @param textureCoords the new texture coordinates
	 */
	public void setTextureCoordinates(double[][] textureCoords) {
		this.textureCoords = textureCoords;
	}
}