/*
 * Author: Elliot Hall <elliot.t.hall@vanderbilt.edu>
 * Description: the central part to the unit testing for JIST.  Collects
 * 	testfiles, shunts data off into appropriate singletons, and prepares and
 * 	returns a TestSuite that ant will run.
 */

package JIST_test;

import java.io.*;
import java.util.*;

import java.lang.Math;
import java.lang.Double;
import java.lang.Exception;
import java.lang.Runtime;

import org.json.*;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestResult;
import junit.framework.TestSuite;
import junit.framework.TestFailure;

public class JTest extends TestSuite {

	public static Test suite () {

		System.out.println("\n\tJIST JUnit Testing Version 2.0\n");

		Properties 		build_prop 	= new Properties();
		TestSuite 		suite		= new TestSuite();

		// Initialize data containers

		VolumeCompareSingleton  vcs	= VolumeCompareSingleton.getInstance();
		FileExistenceSingleton  fes	= FileExistenceSingleton.getInstance();
		CommandRunnerSingleton  crs = CommandRunnerSingleton.getInstance();
		SurfaceCompareSingleton scs = SurfaceCompareSingleton.getInstance();

		// We need this properties file for the mipav directory (unless it's 
		// 	passed from the command line, in which case, it's in the System
		// 	Properties) and the number of threads to use on this system.
		try {
			File f 					= new File("build.properties");
			InputStream prop_is 	= new FileInputStream(f);
			build_prop.load(prop_is);
		} catch (IOException ioe) {
			System.err.println("You should have a build.properties file.");
		}
		
		String classpath = null, extdirs, mpvdir, userdir, sharedir, testdir, 
			homedir, jredir, datadir;

		Properties props 	 	= System.getProperties();

		// get all of our directories, etc

		classpath 		= props.getProperty("java.class.path", null) + ";" + 
							  props.getProperty("myclasspath", null);
		extdirs 		= props.getProperty("java.ext.dirs", null);
		homedir			= props.getProperty("user.home", null);
		userdir 		= props.getProperty("user.dir", null) + "/JIST_TEST/src/JIST_Test"; 
		mpvdir 			= props.getProperty("mipav", null);

		// get number of threads
		crs.set_tc(Integer.parseInt(build_prop.getProperty("threads")));
		if (build_prop.getProperty("datadir").equals("default")) {
			datadir	= userdir + "/datadir";
		} else {
			datadir = "/share/jtest/datadir";
		}

		System.out.println("We are working with " + crs.get_tc() + " threads");
		System.out.println("We have " + Runtime.getRuntime().availableProcessors() + " processors");

		System.out.println("user directory: "  + userdir);
		System.out.println("home directory: "  + homedir);
		System.out.println("mipav directory: " + mpvdir);

		if (mpvdir == null) {
			System.err.println("null mipav directory.");
			return suite;
		}

		mpvdir 		= mpvdir.replaceAll(";", "");

		jredir 		= mpvdir + "jre/bin/java";
		testdir 	= userdir + "/testfiles";

		System.out.println("looking for test directory: " + testdir);

		// Ugly, I know, but as far as I can tell it's the best way to iterate
		// 	through a directory looking for files.
		Vector<File> testfiles = 
			new Vector<File>(Arrays.asList((new File(testdir)).listFiles()));

		// By adding the CommandRunner class and telling it to run the test
		// 	<runCommands()>, we guarantee that MIPAV will run all the necessary
		// 	commands before actually comparing the output of said commands.

		suite.addTest(new CommandRunner("runCommands"));

		for (File testfile : testfiles) {

			// if it's not an appropriate testfile, skip it
			if (!testfile.isFile() || 
				!testfile.getName().endsWith(".jtest")) {
				continue;
			}

			System.out.println("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");

			// <directory> is where all of our test-respective files go
			String directory = 
				mk_dirs(testfile.getName().replace(".jtest","_dir/"), userdir);

			// parse the testfile
			TestcaseStrings tcs = new TestcaseStrings();

			try {
				tcs = parse_input(testfile);
			} catch (Exception e) {
				e.printStackTrace();
				continue;
			}

			// will be null if an error is encountered - skip the file if so
			if (tcs == null) continue;

			// annoying loop to fill in appropriate data directory
			for (int i = 0; i < tcs.input_files.size(); ++i) {
				tcs.input_files.set
					(i, tcs.input_files.get(i).replace("{datadir}", datadir));
			}

			// Build the command
			String cli = (jredir + " -Xmx" + tcs.memx + "m -classpath "
							+ classpath + " -Djava.ext.dirs=" + extdirs
							+ " edu.jhu.ece.iacl.jist.cli.run " 
							+ tcs.module).replace(";",":");

			// add flags from the testfile
			for (int i = 0; i < tcs.input_files.size(); ++i) {
				cli += " " + tcs.input_flags.get(i);
				cli += " " + tcs.input_files.get(i);
			}

			// specifies output directory and output result file
			cli += " -xDir "  + directory;
			cli += " -xFile " + directory + "output.txt";

			System.out.println("\tRunning test " + testfile.getName());

			// add it to the CommandRunnerSingleton to be accessed when the
			// 	CommandRunner Testcase gets run
			MipavCommand mc = new MipavCommand(cli, 
											   testfile.getName(), 
											   tcs.grid_run,
											   userdir);
			crs.add_command(mc);

			if (tcs.debug_print) {
				System.out.println("Added command:");
				System.out.println(mc);
			}


			/*
			 * This is super, super annoying.  There's a better way to do it
			 * 	and it involves the much less annoying technique of 
			 * 	inheritance.  But maybe the if/else-if construct isn't so bad
			 * 	when we're only talking 4 unique test tpyes.  If we start
			 * 	drastically increasing the test types though we need to be 
			 * 	prepared to change this technique.
			 */

			if (tcs.test_type.equals("VolumeTest")) {

				vcs.set_data(tcs, directory, datadir, userdir);
				suite.addTest(new VolumeCompareTest("testVolume"));

			} else if (tcs.test_type.equals("SurfaceTest")) {

				scs.set_data(tcs, directory, datadir, userdir, jredir, classpath, extdirs);
				suite.addTest(new SurfaceCompareTest("testSurface"));

			} else if (tcs.test_type.equals("StapleTest")) {

				vcs.set_data(tcs, directory, datadir, userdir);
				suite.addTest(new VolumeCompareTest("testStaple"));

			} else if (tcs.test_type.equals("FileExistence")) {

				fes.set_data(directory + tcs.test_criteria_subdir, tcs.test_value);
				suite.addTest(new FileExistenceTest("testFiles"));

			} else {

				System.err.println("Test " + tcs.test_type + " is unsupported "
									+ "at this time");
			}

			System.out.println("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~");
		} // FOR (TESTFILES)

		return suite;

	} // End of main function
	
	/* 
	 * make a directory for each test we find 
	 */

	private static String mk_dirs(String dir_name, String userdir) {

		dir_name = userdir + "/" + dir_name;
	
		(new File(dir_name)).mkdir();

		return dir_name;
	
	}

	/*
	 * returns a <TestcaseStrings> object populated with all the relevant
	 * 	values from the JSON testfile.
	 *
	 */

	private static TestcaseStrings parse_input(File f) throws Exception {
		
		JSONObject tc_obj = null, test_criteria_obj = null, input_obj = null,
					surf_obj = null;
		TestcaseStrings tcs = new TestcaseStrings();

		try { // to parse the test file using the class defined below.

			tc_obj 	 = 	new JSONObject(
						new JSONTokener(
						new BufferedReader(
						new FileReader(f))));
			// getting nested objects
			if (tc_obj.has("test_criteria"))
				test_criteria_obj 	= (JSONObject) tc_obj.get("test_criteria");
			if (tc_obj.has("input"))
				input_obj 	= (JSONObject) tc_obj.get("input");
			if (tc_obj.has("surface_test"))
				surf_obj 	= (JSONObject) tc_obj.get("surface_test");
			
			// getting values
			if (tc_obj.has("memx"))
				tcs.memx 		= (String) tc_obj.get("memx");
			if (tc_obj.has("module"))
				tcs.module 		= (String) tc_obj.get("module");
			if (tc_obj.has("test"))
				tcs.test_type 	= (String) tc_obj.get("test");
			if (tc_obj.has("expected_data"))
				tcs.ed_file 	= (String) tc_obj.get("expected_data");

			if (tc_obj.has("debug_print"))
				tcs.debug_print = (Boolean) tc_obj.get("debug_print");
			if (tc_obj.has("grid_run"))
				tcs.grid_run = (Boolean) tc_obj.get("grid_run");

			if (surf_obj != null) {
				tcs.template_volume 	= (String) surf_obj.get("template_volume");
				tcs.surface_extension 	= (String) surf_obj.get("surface_extension");
			}

			if (test_criteria_obj != null) {
				tcs.test_criteria_type 	= (String) test_criteria_obj.get("type");
				tcs.test_value  		= (Double) test_criteria_obj.get("val");

				if (test_criteria_obj.has("subdir")) {
					tcs.test_criteria_subdir = 
						(String) test_criteria_obj.get("subdir");
				}
			}

			if (input_obj != null) {
				JSONArray flags = (JSONArray) input_obj.get("flags");
				JSONArray files = (JSONArray) input_obj.get("values");
					
				for (int i = 0; i < flags.length(); ++i)
					tcs.input_flags.add(flags.getString(i));

				for (int i = 0; i < files.length(); ++i)
					tcs.input_files.add(files.getString(i));

				if (tcs.input_flags.size() != tcs.input_files.size()) {
					throw new Exception("Input flags do not match parameters.");
				}

			}

		} catch (JSONException je) {
			System.err.println("JSON error in file " + f.getName());
			System.err.println(je.getMessage());
			tcs = null;
		} catch (Exception e) {
			e.printStackTrace();
			tcs = null;
		} finally {
			return tcs;
		}

	}

}
