package edu.jhmi.rad.medic.algorithms;

import edu.jhmi.rad.medic.methods.*;
import edu.jhmi.rad.medic.utilities.*;


import gov.nih.mipav.model.algorithms.AlgorithmBase;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.model.file.FileInfoBase;
import gov.nih.mipav.view.*;
import gov.nih.mipav.view.dialogs.*;

import java.io.*;


/**
 *	LesionTOADS: TOpology-preserving Anatomy-Driven Segmentation adapted for lesions.
 *	<p>
 *	This is the main version of the segmentation. It includes inhomogeneity correction,
 *	template registration and adaptive coefficients.
 *
 *	@version    Jan 2008
 *	@author     Pierre-Louis Bazin Navid Shiee
 *  @see 		JDialogLesionToads
 *	@see 		LesionToads
 *	@see		RegisterTemplate
 *
*/
public class AlgorithmLesionToads extends AlgorithmBase {
	public AlgorithmLesionToads() {
	}

	private static final String cvsversion = "$Revision: 1.3 $";
	public static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");

	public String get_version() {
		return revnum;
	}

    // Fuzzy images require 1 image for each class
    // Hard images 1 image with assigned classes
    private ModelImage  		destImage[];
    private ModelImage  		srcImage[];
    private int         		destNum=0;
    private String      		resultOutput;
    private ViewUserInterface   userInterface;
		
    // image size
	private int 		classes; 		// number of classes
	private int 		nx,ny,nz,nc;    // original image dimensions
	private int 		ntx,nty,ntz; 	// original template dimensions
	private	float		rx,ry,rz;		// image resolutions
	private	float		rtx,rty,rtz;	// template resolutions
	private int			orient;			// image orientation
	private int			orx,ory,orz;	// image axis orientation
	private	String[]	modality;		// image modality
	private	String[]	psfType;		// image psf type
	private	String		algorithmMode;		// variants of the algorithm
	private	String		connectivityType;		// choice of connectivity
	
    // segmentation parameters
    private	String				atlasName;
    private	LesionToadAtlas		atlas=null;
    private byte[]				objType;

	//private	boolean		segment = true;
	private float   	smoothing;
	private float		outlier;
	private int 		iterations;
	private float   	maxDistance;
	private float		bgthreshold;
	private	boolean		doGrowing;
	
	//private	boolean		useTopologyPrior;
	private float		firstLimit;
	private float		lastLimit;
	private float		spread;
	
	private short       maxGMDist;
	private short      maxBstemDist;
	private short       maxVentDist;
	
	//private	boolean		useShapePrior	= false;
	private	float		atlasFirstCoefficient, atlasScale;
	private	float		atlasRegistrationCoefficient = 10.0f;
	
	//private boolean		useIntensityPrior;
	
	private	String		centroidMode;
	private	float		centroidSmoothness = 0.5f;
	private boolean		lesionWeight = false;
	private String      normType;
	private	String		registrationMode = "rigid";
	
	private	boolean		register = false;
	private	int			levels;
	private	int			firstIter;
	private	int			topIter;
	private	int			lastIter;
	private	String		transformType;
	
	private	boolean		alignShapes = true;
	private	int			alignIter = 1;
	//private	String		alignType = "fully_affine";
	//private	String		alignType = "rigid";
	//private	String		alignType = "single_scale";
	private	int			initAlignIter = 10;
	
    private	boolean		correctInhomogeneity = false;
	private	int			polynomialDegree;
	private	float		splineKernel;
	private	int			correctionMethod;
	private float   	lowEratio = 0.1f;
	
	private boolean     includeLesionnClassification = false;
	private boolean     outputClassificationFromMembership = true;
	private boolean     outputField = false;
	private boolean		verbose = true;
	private boolean     debug   = true;
	
    /**
    *	Constructor for 3D images in which changes are placed in a predetermined destination image.
    *   @param destImg_      Image model where result image is to stored.
    *   @param srcImg_       Source image model.
    */
	public AlgorithmLesionToads(ModelImage[] srcImg_, int nInput_,
								String[] imgModal_, String[] psf_,
								String aName_, LesionToadAtlas atlas_, String segOutput_, 
								float smooth_, float out_, int nIterMax_, float distMax_, float bgth_,
								boolean outputMaxMem_, 
								float fLim_, float lLim_, float spread_,
								short maxGMDist_, short maxBstemDist_, short maxVentDist_,
								boolean inludeLesions_,
								float aFirstCoeff_, float aScale_,
								String relMode_, 
								String centMode_, float smoothCentr_,
								boolean lesionWeight_,
								String regMode_,
								String nrmType_,
								boolean register_, 
								int lev_, int first_, int top_, int last_,
								boolean correct_, boolean outputField_,
								String correctType_,
								int poly_, float kernel_,
								String algo_,
								String connect_) {
        super(null, srcImg_[0]);
		userInterface = ViewUserInterface.getReference();
		srcImage = srcImg_;
        nc = nInput_;
		modality = imgModal_;
		psfType = psf_;
		
		atlasName = aName_;
		atlas = atlas_;

		resultOutput = segOutput_;
        
		includeLesionnClassification = inludeLesions_;
		smoothing = smooth_;
		outlier = out_;
		iterations = Numerics.abs(nIterMax_);
		doGrowing = (nIterMax_>0);
        maxDistance = distMax_;
		lowEratio = (float)Math.sqrt(maxDistance);
		bgthreshold = bgth_;
		
		outputClassificationFromMembership = outputMaxMem_;
		firstLimit = fLim_;
		lastLimit = lLim_;
		spread = spread_;
		
		maxGMDist = maxGMDist_;
		maxBstemDist = maxBstemDist_;
		maxVentDist = maxVentDist_;
		
		//useShapePrior = useShape_;
		atlasFirstCoefficient = aFirstCoeff_;
		atlasScale = aScale_;

		//useIntensityPrior = useIntensity_;
		
		centroidMode = centMode_;
		centroidSmoothness = smoothCentr_;
		lesionWeight = lesionWeight_;
		normType = nrmType_;
		
		
		algorithmMode = algo_;
		registrationMode = regMode_;
		
		connectivityType = connect_;
		
		register = register_;
		levels = lev_;
		firstIter = first_;
		topIter = top_;
		lastIter = last_;
		
		alignIter = topIter;
		initAlignIter = firstIter;
		
		correctInhomogeneity = correct_;
		outputField = outputField_;
		if (correctInhomogeneity) {
			polynomialDegree = poly_;
			splineKernel = kernel_;
			if (correctType_.equals("Splines")) correctionMethod = InhomogeneityCorrection2.SPLINES;
			else correctionMethod = InhomogeneityCorrection2.CHEBYSHEV;
		} else {
			polynomialDegree = 0;
			splineKernel = 0.0f;
			correctionMethod = InhomogeneityCorrection2.CHEBYSHEV;
		}

	}

    /**
    *	Prepares this class for destruction.
    */
	public void finalize(){
	    destImage   = null;
	    srcImage    = null;
        super.finalize();
        System.gc();
	}

	
    /**			
    *   Starts the segmentation.
    */
	public void runAlgorithm() {

				
		if (srcImage  == null) {
            displayError("Source Image is null");
            //notifyListeners(this);
            return;
        }
        
		// start the timer to compute the elapsed time
        setStartTime();

		//constructLog(true);
		
		float[][] buffer = null;
		if (srcImage[0].getNDims() == 2){
			try {
				// image length is length in 2 dims
				int length = srcImage[0].getExtents()[0]  * srcImage[0].getExtents()[1];
				 buffer = new float[nc][];
				for (int t=0;t<nc;t++) {
					buffer[t] = new float[length];
					srcImage[t].exportData(0,length, buffer[t]); // locks and releases lock
				}
			}
			catch (IOException error) {
				buffer = null;
				errorCleanUp("Segmentation reports: source image locked", true);
				return;
			}
			catch (OutOfMemoryError e){
				buffer = null;
				errorCleanUp("Segmentation reports: out of memory", true);
				return;
			}
			// init dimensions
			nx = srcImage[0].getExtents()[0];
			ny = srcImage[0].getExtents()[1];
			nz = 1;
			
			ntx = atlas.getImageDim()[0];
			nty = atlas.getImageDim()[1];
			ntz = 1;
			
			rx = srcImage[0].getFileInfo()[0].getResolutions()[0];
			ry = srcImage[0].getFileInfo()[0].getResolutions()[1];
			rz = 1.0f;
			
			// check for non-mm units
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[0]==FileInfoBase.CENTIMETERS) rx = 10.0f*rx;
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[1]==FileInfoBase.CENTIMETERS) ry = 10.0f*ry;
			
			rtx = atlas.getImageRes()[0];
			rty = atlas.getImageRes()[1];
			rtz = 1.0f;
			
			orient = srcImage[0].getImageOrientation();
			orx = srcImage[0].getFileInfo()[0].getAxisOrientation(0);
			ory = srcImage[0].getFileInfo()[0].getAxisOrientation(1);
			orz = FileInfoBase.UNKNOWN_ORIENT;
			
			// main segmentation
			calcSegmentation(buffer);
			
		}
		else if (srcImage[0].getNDims() > 2) {
		   try {
				// image length is length in 3 dims
				int length = srcImage[0].getExtents()[0] *
							 srcImage[0].getExtents()[1] *
							 srcImage[0].getExtents()[2];
									 buffer = new float[nc][];
				for (int t=0;t<nc;t++) {
					buffer[t] = new float[length];
					srcImage[t].exportData(0,length, buffer[t]); // locks and releases lock
				}
				fireProgressStateChanged(srcImage[0].getImageName(), "Processing image ...");
			}
			catch (IOException error) {
				buffer = null;
				errorCleanUp("Segmentation: source image locked", true);
				return;
			}
			catch (OutOfMemoryError e){
				buffer = null;
				errorCleanUp("Segmentation: Out of memory creating process buffer", true);
				return;
			}
			// init dimensions
			nx = srcImage[0].getExtents()[0];
			ny = srcImage[0].getExtents()[1];
			nz = srcImage[0].getExtents()[2];
			
			ntx = atlas.getImageDim()[0];
			nty = atlas.getImageDim()[1];
			ntz = atlas.getImageDim()[2];
			
			rx = srcImage[0].getFileInfo()[0].getResolutions()[0];
			ry = srcImage[0].getFileInfo()[0].getResolutions()[1];
			rz = srcImage[0].getFileInfo()[0].getResolutions()[2];
			
			// check for non-mm units
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[0]==FileInfoBase.CENTIMETERS) rx = 10.0f*rx;
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[1]==FileInfoBase.CENTIMETERS) ry = 10.0f*ry;
			if (srcImage[0].getFileInfo()[0].getUnitsOfMeasure()[2]==FileInfoBase.CENTIMETERS) rz = 10.0f*rz;
			
			rtx = atlas.getImageRes()[0];
			rty = atlas.getImageRes()[1];
			rtz = atlas.getImageRes()[2];
			
			orient = srcImage[0].getImageOrientation();
			orx = srcImage[0].getFileInfo()[0].getAxisOrientation(0);
			ory = srcImage[0].getFileInfo()[0].getAxisOrientation(1);
			orz = srcImage[0].getFileInfo()[0].getAxisOrientation(2);
			
			classes = atlas.getNumber();
	
			// main segmentation
			calcSegmentation(buffer);
			
			
			
        }

        // compute the elapsed time
        computeElapsedTime();
    } // end run()

    /**
    *	produces the segmentation of the input image
    */
    private void calcSegmentation(float img[][]) {
		byte[][][]   	classification = null;
		byte[][][]   	classification_mem = null;
		float[][]		centroids = null;
		float[][][][] 	memberships = null;
        float[][][][] 	field = null;
        float[][][]   	lesions = null; 
        //float[][][]   	blackHole = null;
		short[][][]   lesionClassification = null;
		short[][][]   lesionClassification_mem = null;
		float[][][]   lesionRelation = null;
		boolean[][][]	obj;
		float[] 		intensityPrior;
		float   	num,den;
		float   	ngb, neighbors;
		float[]     buffer;
		byte[]		bytebuffer;
		short[]     shortbuffer;
		LesionToadSegmentation 	segmentation=null;
		LesionToadPreprocess		preprocess=null;
		InhomogeneityCorrection2[]	correction=null;
		String      info = "";
		float		energy, Eprev, Eratio;
		float		correctRate;
		float[]		misclass = new float[classes];
		float[]		transform = null;
		
		int mod;
		int progress;

		boolean stop;
		
		if (verbose) {
			MedicUtilPublic.displayMessage("Lesion-TOADS\n");
			MedicUtilPublic.displayMessage(getParameterText());
		}
		
		fireProgressStateChanged(srcImage[0].getImageName(), "Initialization");
		fireProgressStateChanged(10);
		
		//---------------------------------------/
		//-          MAIN SEGMENTATION          -/
		//---------------------------------------/
		
		// record time
		long start_time = System.currentTimeMillis();
		long inner_loop_time;

        memberships = null;
        classification = null;
		
		// pre-processing
        if (verbose) MedicUtilPublic.displayMessage("Initialization\n");
		preprocess = new LesionToadPreprocess(img, nc, nx, ny, nz, rx, ry, rz,
												orient, orx, ory, orz,
												atlas.getTemplate(), ntx, nty, ntz, rtx, rty, rtz,
												classes, (byte)1, 0.0f, 10, userInterface);
		
		if (debug) System.out.println(preprocess.displayTransform());
		// put all intensities in [0,1]										
		preprocess.normalizeImages();
		// find unused regions
		preprocess.findCroppingBoundaries();
		// crop images and atlas
		preprocess.cropImages();
		if (debug) System.out.println("cropped: "+preprocess.displayTransform());
		int[] dim = preprocess.getCurrentImageDimensions();
		nx = dim[0]; ny = dim[1]; nz = dim[2];
		float[] res = preprocess.getSignedImageResolutions();
		rx = res[0]; ry = res[1]; rz = res[2];
		
		// first centroids, memberships : tissue types
		centroids = preprocess.initialCentroids(atlas.getIntensityPriors(modality,nc),classes);
		
		fireProgressStateChanged(20);
		
		//Longitudinal
		/*atlas.setImageInfo(nx,ny,nz,rx,ry,rz,orient,orx,ory,orz);	
		atlas.computeTransformedTemplate();
		
		// re-compute the centroids to get better estimates
		preprocess.updateTransformedTemplate(atlas.getTemplate());*/
		//Longitudinal
		// align the atlas
		if (alignShapes) {
			if (verbose) MedicUtilPublic.displayMessage("Atlas alignment\n");
			memberships = preprocess.initialMemberships(centroids, classes);
			
			// align both images centers ? not very efficient
			//preprocess.alignImagesAndTopolgyCenters(atlas.getLabels(), centroids, classes);
			//System.out.println("IMAGE DIMENSIONS "+nx+" "+ny+" "+nz);
			atlas.setImageInfo(nx,ny,nz,rx,ry,rz,orient,orx,ory,orz);
			
			//atlas.initShapeRegistration(memberships, registrationMode, initAlignIter, levels, 1.0f);				
			atlas.initShapeRegistration(memberships, "rigid", initAlignIter, levels, 1.0f);				
			atlas.setTransform(preprocess.getTransform());
			
			if (debug) System.out.println("pre-process: "+preprocess.displayTransform());
			
			//atlas.updateShapeRegistration(memberships, "fully_affine", initAlignIter, levels);
			//atlas.updateShapeRegistration(memberships, "single_scale", initAlignIter, levels);
			//atlas.updateShapeRegistration(memberships, registrationMode, initAlignIter, levels, 1.0f);
			atlas.updateShapeRegistration(memberships, "rigid", initAlignIter, levels, 1.0f);
			if (debug) System.out.println("atlas: "+atlas.displayTransform(atlas.getTransform()));
			
			fireProgressStateChanged(30);
			
			atlas.registerShapesPyramid();
			
			// add a last step ? not needed, can be nice as an option
			//atlas.updateShapeRegistration(memberships, registrationMode, lastIter, 1, 1.0f);
			if (debug) System.out.println("atlas: "+atlas.displayTransform(atlas.getTransform()));
			
			fireProgressStateChanged(50);
			
			//atlas.registerShapes();
			
			memberships = null;
			
			if (debug) System.out.println("aligned atlas: "+atlas.displayTransform(atlas.getTransform()));
			if (debug) System.out.println("aligned pre-process: "+preprocess.displayTransform());
			
			preprocess.uncropImages();
			preprocess.findCroppingBoundaries();
			preprocess.cropImages();
			
			if (debug) System.out.println("final pre-process: "+preprocess.displayTransform());
			if (debug) System.out.println("final aligned atlas: "+atlas.displayTransform(atlas.getTransform()));
			
			dim = preprocess.getCurrentImageDimensions();
			nx = dim[0]; ny = dim[1]; nz = dim[2];
			
			atlas.setImageInfo(nx,ny,nz,rx,ry,rz,orient,orx,ory,orz);
			
			atlas.computeTransformedTemplate();
			
			// re-compute the centroids to get better estimates
			preprocess.updateTransformedTemplate(atlas.getTemplate());
			//centroids = preprocess.initialCentroids(atlas.getLabels(),atlas.getIntensityPriors(modality,nc),classes);		
		}
		
		fireProgressStateChanged(60);
		
		//classification = atlas.getTemplate();
				
		
		// create the segmentation algorithm
		segmentation = new LesionToadSegmentation(preprocess.getImages(), atlas, 
								nx, ny, nz, nc, 
								smoothing, outlier,
								firstLimit, lastLimit, 
								0, spread, 
								maxGMDist, maxBstemDist, maxVentDist,
								atlasFirstCoefficient, atlasScale,
								modality,
								algorithmMode, normType, connectivityType,
								centroidMode, centroidSmoothness,
								userInterface, getProgressChangeListener());

		// get the centroids from initialization
		//segmentation.setCentroids(centroids);
		segmentation.initCentroids(centroids);
		segmentation.setLesionCentroid(preprocess.initialLesionCentroid(atlas.getLesionPriors(modality,nc)));
		segmentation.setBlackHoleCentroid(preprocess.initialBlackHoleCentroid(atlas.getBlackHolePriors(modality,nc)));
		if (lesionWeight) 
			segmentation.setLesionWeight();
		else
			segmentation.reSetLesionWeight();
		segmentation.setIntensityMax(preprocess.getIntensityMax());
		segmentation.setIntensityScale(preprocess.getIntensityScale());
		segmentation.setGMflag();
		if (verbose) MedicUtilPublic.displayMessage("initial "+segmentation.displayCentroids()); 
		if (verbose) MedicUtilPublic.displayMessage("initial "+segmentation.displayLesionCentroid());
		if (verbose) MedicUtilPublic.displayMessage("initial "+segmentation.displayBlackHoleCentroid());
		
		// re-initialize the atlas alignment
		atlas.updateShapeRegistration(segmentation.getMemberships(), registrationMode, alignIter, 1, 1.0f);
		if (verbose) MedicUtilPublic.displayMessage("final aligned atlas: "+atlas.displayTransform(atlas.getTransform()));
			
		fireProgressStateChanged(70);
		
		if (correctInhomogeneity) {
			//fireProgressStateChanged("initialization (inhomogeneity)");
				
			correction = new InhomogeneityCorrection2[nc];
			for (int c=0;c<nc;c++) {
				// create the inhomogeneity correction segmentation
				/*correction[c] = new InhomogeneityCorrection(polynomialDegree,
															preprocess.getImages()[c], 
															segmentation.getClassification(),
															segmentation.getMemberships(),
															segmentation.getCentroids(c),
															nx,ny,nz,rx,ry,rz,
															classes, atlas.getLabels(),
															atlas.getTopology(), 3,
															userInterface, getProgressChangeListener());*/
				correction[c] = new InhomogeneityCorrection2(preprocess.getImages()[c],
						segmentation.getMemberships(),
						segmentation.getCentroids(c),
						classes,
						segmentation.getSegmentation(),atlas.getTopology(),
						polynomialDegree, splineKernel,
						InhomogeneityCorrection2.IMAGE,
						InhomogeneityCorrection2.GLOBAL,
						correctionMethod,
						InhomogeneityCorrection2.FIXED,
						InhomogeneityCorrection2.EQUAL_VARIANCE,
						2.0f, null, null,
						nx,ny,nz,rx,ry,rz,
						getProgressChangeListener());

				// integrate it into the segmentation
				segmentation.addInhomogeneityCorrection(correction[c].getField(),c);
			}
		} else correction = null;
		
		if (debug) System.out.print("pre-processing time: " + (System.currentTimeMillis()-start_time)+"\n"); 

		fireProgressStateChanged(80);
		// relations
		segmentation.computeRelations();
		//segmentation.displayRelationList();
		
		if (debug) System.out.print("relation initialization time: " + (System.currentTimeMillis()-start_time)+"\n"); 
		
		segmentation.computeVariances();
		if (debug) MedicUtilPublic.displayMessage(segmentation.displayVariances());
		segmentation.computeChannelWeights();
		
		fireProgressStateChanged(90);
		// memberships & energy
		//segmentation.setInit();
		//segmentation.reSetGMflag();
		energy = segmentation.computeMemberships();
		//segmentation.reSetInit();
		segmentation.setGMflag();
		userInterface.setGlobalDataText("initial energy : "+energy+"\n");
			
		if (debug) System.out.print("membership initialization time: " + (System.currentTimeMillis()-start_time)+"\n"); 
			
		Eprev = energy;
		Eratio = 1.0f;
			
		fireProgressStateChanged(99);
			
		if (debug) System.out.print("final initialization time: " + (System.currentTimeMillis()-start_time)+"\n"); 
			
		if (debug) System.out.print("algorithm mode: " +algorithmMode+"\n"); 
		
		// 3. main loop
		//--------------------------------------------------------------------
		stop = false;
		int iter = 0;
		int deg = 0;
		if (iterations==0) stop=true;
		while ((!stop) && (!threadStopped) && segmentation.isWorking()) {
			long iterationTime = 0;
			if (debug) iterationTime = System.currentTimeMillis();
				
			iter++;
			if (verbose) MedicUtilPublic.displayMessage("iteration "+iter+"\n");
			
			// a. thinning
			//----------------------------------------------------
			fireProgressStateChanged("iteration " + iter + " (thinning)");
			fireProgressStateChanged(10);
			
			if (algorithmMode.equals("narrow_band_competition")) {
				segmentation.propagateCompetition();
			} else {
				segmentation.propagateThinning();
			}
			
			if (debug) System.out.print("thinning time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
				
			// b. update parameter information
			//------------------------------------------------------------
			fireProgressStateChanged("iteration " + iter + " (parameters)");
			fireProgressStateChanged(20);
					
			// inhomogeneity field
			if (correctInhomogeneity) {
				if ( (deg<polynomialDegree) && (Eratio < lowEratio) ) deg++;
				for (int c=0;c<nc;c++) correction[c].computeCorrectionField(deg);
							
				/*correctRate = 0.0f;
				for (int c=0;c<nc;c++) {
					correctRate = Numerics.max(correctRate,correction[c].computeCorrectionField(deg));
				}*/
				
				fireProgressStateChanged(30);

				if (debug) System.out.print("inhomogeneity time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
				if (debug) iterationTime = System.currentTimeMillis();
			}
				
			// centroids
			//if (Eratio < lowEratio) segmentation.setLesionCentroidFactor(0.0f);
			segmentation.computeCentroids();
			if (verbose) MedicUtilPublic.displayMessage(segmentation.displayCentroids());
			if (verbose) MedicUtilPublic.displayMessage(segmentation.displayLesionCentroid());
			if (verbose) MedicUtilPublic.displayMessage(segmentation.displayBlackHoleCentroid());
			
			// cluster-specific variations
			segmentation.computeVariances();
			if (verbose) MedicUtilPublic.displayMessage(segmentation.displayVariances());
			if (verbose) MedicUtilPublic.displayMessage(segmentation.displayLesionVariance());
			//if (verbose) MedicUtilPublic.displayMessage(segmentation.displayBlackHoleVariance());
			
			/*segmentation.computCovariance();
			System.out.println(segmentation.displayCovariance());
			System.out.println(segmentation.displayLesionCovariance());*/
			//Channel Weights 
			segmentation.computeChannelWeights();
			if (verbose) MedicUtilPublic.displayMessage(segmentation.displayChannelWeights());
			
			fireProgressStateChanged(40);
			
			if (debug) System.out.print("centroid time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
				
			// memberships & energy
			fireProgressStateChanged("iteration " + iter + " (memberships)");
			Eprev = energy;
			energy = segmentation.computeMemberships();
			Eratio = Numerics.abs( 2.0f*(Eprev-energy)/(Eprev+energy) );
			fireProgressStateChanged(60);			
			
			if (debug) System.out.print("membership time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
			
			
			
			// shape alignment
			fireProgressStateChanged("iteration " + iter + " (atlas alignment)");
			if (alignShapes) atlas.registerShapes();
				
			if (debug) System.out.print("alignment time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
			
			fireProgressStateChanged(70);
			// check for convergence
			if ( Eratio < maxDistance) stop = true;
			if (verbose) MedicUtilPublic.displayMessage("energy: "+energy+", " +"ratio: "+Eratio+"\n");
			
			if (iter >= iterations) stop = true;
				
			// c. growing
			//---------------------------------------------------------------
			fireProgressStateChanged("iteration " + iter + " (growing)");
			if (algorithmMode.equals("narrow_band_competition")) {
				// do nothing
			} else {
				if ( (!stop) || doGrowing){
					segmentation.propagateGrowing();
				}
				
				if (debug) System.out.print("growing time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
				if (debug) iterationTime = System.currentTimeMillis();
			}
			fireProgressStateChanged(90);
					
				
			// d. update classification only at the end
			//--------------------------------------------------------------
			if (stop) {
				segmentation.updateClassificationFromLabels();
				segmentation.updateLesionClassifcation();
			}
			
			// update the relation list
			if (!stop){
				fireProgressStateChanged("iteration " + iter + " (relations)");
				segmentation.computeRelations();
			}
			//else segmentation.displayRelationList();
				
			// decrease the smoothing if the skeleton is good enough
			// (must be done after the growth: changes the scaling)
			fireProgressStateChanged(99);
				
			if (debug) System.out.print("update time: " + (System.currentTimeMillis()-iterationTime)+"\n"); 
			if (debug) iterationTime = System.currentTimeMillis();
		}
		
		if (verbose){
			float secs = (float)((System.currentTimeMillis()-start_time)/1000.0f);
			int mins = Numerics.floor(secs/60.0f);
			secs -= 60*mins;  
			MedicUtilPublic.displayMessage("total processing time : " + mins + " minute(s) " + Numerics.round(secs) + " second(s) \n" ) ;
			MedicUtilPublic.displayMessage("Preparing output images\n");
		}
		segmentation.cleanUp();
		
		fireProgressStateChanged("creating result images...");
		fireProgressStateChanged(30);
		
		classification = segmentation.exportClassification();
		lesionClassification = segmentation.exportLesionClassification();
		if (outputClassificationFromMembership){
			segmentation.updateClassificationFromMemberships();
			classification_mem = segmentation.getClassification();
		}
		if (resultOutput.equals("class+member")) {
			memberships = segmentation.exportMemberships();
			lesions = segmentation.getLesionMemberships();
			//blackHole = segmentation.getBlackHoleMem()
		} /*else if (resultOutput.equals("all_images")) {
			memberships = segmentation.exportMemberships();
			lesions = segmentation.exportLesionMembership();
			//blackHole = segmentation.getBlackHoleMem();
			if (correctInhomogeneity) {
				field = new float[nc][][][];
				for (int c=0;c<nc;c++) field[c] = correction[c].exportField();
			}
		}*/ else if (resultOutput.equals("cruise inputs")){
			memberships = segmentation.generateCruiseInputs();
		} else if (resultOutput.equals("dura removal inputs")){
			memberships = segmentation.generateDuraRemovalOutputs();
		}
		if (correctInhomogeneity && outputField) {
			field = new float[nc][][][];
			for (int c=0;c<nc;c++) field[c] = correction[c].exportField();
		}
		/* animation stuff 
		 byte[][][][] thin = segmentation.getThinFrames();
		byte[][][][] thin_Seg = segmentation.getThinSegFrames();
		byte[][][][] grow = segmentation.getGrowFrames();
		byte[][][][] grow_Seg = segmentation.getGrowSegFrames();
		byte[][][][] distances = segmentation.exportDistances();*/
		/*byte[][][] firstSkleton = segmentation.getFistSkleton();
		byte[][][] lastSkleton = segmentation.getLastSkleton();*/
		
		fireProgressStateChanged(99);
		// clean-up the algorithms
		String[] names = atlas.getNames();
		//if (atlas!=null) atlas.finalize();
		//atlas = null;
		if (segmentation!=null) segmentation.finalize();
		segmentation = null;
		if (correction!=null) {
			for (int c=0;c<nc;c++) if (correction[c]!=null) correction[c].finalize();
			correction = null;
		}
		
		if (includeLesionnClassification)
			for (int x =0; x<nx; x++) for (int y =0; y<ny; y++) for (int z =0; z<nz; z++){
				classification[x][y][z] -= 15*lesionClassification[x][y][z];
				if (classification[x][y][z] == 26) classification[x][y][z] =25;
				
			}
		fireProgressStateChanged("export results...");
		fireProgressStateChanged(20);
		
		try {
			// Calculate the number of result images.
			if (resultOutput.equals("class+member")) destNum = 3; //classes+1(lesion) + hardClass + LesionHard  
			/*else if (resultOutput.equals("all_images")) {
				destNum = 3; //like class+member
				if (correctInhomogeneity) destNum ++;
			}*/else if (resultOutput.equals("cruise inputs")||resultOutput.equals("dura removal inputs")) destNum = 6;
			else destNum = 2; // classification
			
			if (outputClassificationFromMembership) destNum++;//for classification from mem
			if (outputField) destNum++; //for inhomogeneity field
			// create the output array
			// for animation
			//destNum += 5;
			destImage = new ModelImage[destNum];
			
			// output dimensions
			int Nout=0;
            
			/*  Generating Thinnig Growing animation
			int growFrames = grow.length;
            destImage[Nout] = new ModelImage(ModelStorageBase.BYTE, preprocess.getOriginalImageArrayDimensions(growFrames),
					JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_grow_progress"));
        	for (int k=0;k<growFrames;k++) {
        		bytebuffer = preprocess.uncropAndBuffer(grow[k],(byte)0);
        		destImage[Nout].importData(k*preprocess.getOriginalImageSize(), bytebuffer, true);
        	}
        	Nout++;
        	growFrames = grow_Seg.length;
            destImage[Nout] = new ModelImage(ModelStorageBase.BYTE, preprocess.getOriginalImageArrayDimensions(growFrames),
					JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_grow_progress"));
        	for (int k=0;k<growFrames;k++) {
        		bytebuffer = preprocess.uncropAndBuffer(grow_Seg[k],(byte)0);
        		destImage[Nout].importData(k*preprocess.getOriginalImageSize(), bytebuffer, true);
        	}
        	Nout++;
        	int thinFrames = thin.length;
            destImage[Nout] = new ModelImage(ModelStorageBase.BYTE, preprocess.getOriginalImageArrayDimensions(thinFrames),
					JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_thin_progress"));
        	for (int k=0;k<thinFrames;k++) {
        		bytebuffer = preprocess.uncropAndBuffer(thin[k],(byte)0);
        		destImage[Nout].importData(k*preprocess.getOriginalImageSize(), bytebuffer, true);
        	}
        	Nout++;
        	thinFrames = thin_Seg.length;
            destImage[Nout] = new ModelImage(ModelStorageBase.BYTE, preprocess.getOriginalImageArrayDimensions(thinFrames),
					JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_thin_progress"));
        	for (int k=0;k<thinFrames;k++) {
        		bytebuffer = preprocess.uncropAndBuffer(thin_Seg[k],(byte)0);
        		destImage[Nout].importData(k*preprocess.getOriginalImageSize(), bytebuffer, true);
        	}
        	Nout++;
        	bytebuffer =null;Generating Thinning growing animation
        	destImage[Nout] = new ModelImage(ModelStorageBase.BYTE, preprocess.getOriginalImageArrayDimensions(3),
					JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_distances"));
        	for (int k=0;k<3;k++) {
        		bytebuffer = preprocess.uncropAndBuffer(distances[k],(byte)0);
        		destImage[Nout].importData(k*preprocess.getOriginalImageSize(), bytebuffer, true);
        	}
        	Nout++;*/
        	
        	
        	
			if (resultOutput.equals("all_images")
				|| resultOutput.equals("class+member")
				) {
				/*for (int k=0;k<classes;k++) {
					destImage[k] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
													JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_"+names[k]));
 
					buffer = preprocess.uncropAndBuffer(memberships[k],0.0f);
					//destImage[k].reallocate(preprocess.getImageDimensions());
					destImage[k].importData(0, buffer, true);
				}
				Nout = classes;
				destImage[Nout] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
                        JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_lesions"));
						
				buffer = preprocess.uncropAndBuffer(lesions,0.0f);

				destImage[Nout].importData(0, buffer, true);
				buffer = null;
				Nout++;*/
            	destImage[Nout] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(classes+1),
						JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_mems"));
            	for (int k=0;k<classes;k++) {
            		buffer = preprocess.uncropAndBuffer(memberships[k],0.0f);
            		destImage[Nout].importData(k*preprocess.getOriginalImageSize(), buffer, true);
            	}
            	buffer = preprocess.uncropAndBuffer(lesions,0.0f);
            	destImage[Nout].importData(classes*preprocess.getOriginalImageSize(), buffer, true);
            	memberships = null;
            	buffer = null;
            	Nout++;
			}
            
            fireProgressStateChanged(60);
			
            /*if (resultOutput.equals("classification")
				|| resultOutput.equals("class+member")
				|| resultOutput.equals("all_images")
				|| resultOutput.equals("cruise inputs")
				|| resultOutput.equals("dura removal inputs")) {*/
				destImage[Nout] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                                                        JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_class"));				

				bytebuffer = preprocess.uncropAndBuffer(classification,(byte)0);
				destImage[Nout].importData(0, bytebuffer, true);
				Nout++;
				
				if (outputClassificationFromMembership){
					destImage[Nout] = new ModelImage(ModelStorageBase.UBYTE, preprocess.getOriginalImageDimensions(),
                        JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_class_max_mem"));				
					bytebuffer = preprocess.uncropAndBuffer(classification_mem,(byte)0);
					destImage[Nout].importData(0, bytebuffer, true);
					Nout++;
				}

//				lesion classification
				destImage[Nout] = new ModelImage(ModelStorageBase.SHORT, preprocess.getOriginalImageDimensions(),
                        JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_lesions_hard"));
				shortbuffer = preprocess.uncropAndBuffer(lesionClassification,(short)0);
				destImage[Nout].importData(0, shortbuffer, true);
				Nout ++;
				
				buffer = null;
				bytebuffer = null;
				shortbuffer = null;
				classification = null;
				lesionClassification = null;
				lesionRelation = null;

			//}
			

			fireProgressStateChanged(90);
			if ( (resultOutput.equals("cruise inputs")) || (resultOutput.equals("dura removal inputs"))) {
				
				destImage[Nout] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
												  JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_csf"));
				destImage[Nout+1] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
						  JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_gm"));
				destImage[Nout+2] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
						  JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_wmFill"));
				destImage[Nout+3] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
						  JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_wmMask"));
				
				for (int i=0;i<4;i++){
					buffer = preprocess.uncropAndBuffer(memberships[i],0.0f);
					destImage[Nout+i].importData(0, buffer, true);
				}
				Nout += 4;
				buffer = null;
			}
			
            //if (resultOutput.equals("all_images") && correctInhomogeneity) {
			if (outputField && correctInhomogeneity){
            	if (nc>1) {
					destImage[Nout] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageArrayDimensions(nc),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_fields"));
					for (int c=0;c<nc;c++) {
						buffer = preprocess.uncropAndBuffer(field[c],1.0f);
						destImage[Nout].importData(c*preprocess.getOriginalImageSize(), buffer, true);
					}
				} else {
					destImage[Nout] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
														JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_field"));
					buffer = preprocess.uncropAndBuffer(field[0],1.0f);
					destImage[Nout].importData(0, buffer, true);
				}
            	/*for (int c=0;c<nc;c++) {
					destImage[Nout] = new ModelImage(ModelStorageBase.FLOAT, preprocess.getOriginalImageDimensions(),
															JDialogBase.makeImageName(srcImage[0].getImageName(), "_lToads_field"+(c+1)));

					buffer = preprocess.uncropAndBuffer(field[c],1.0f);
					destImage[Nout].importData(0, buffer, true);
					buffer = null;
					field[c] = null;
					Nout++;
				}*/
			}
            
            
            fireProgressStateChanged(99);
			
			buffer = null;
			
		} catch (OutOfMemoryError e) {
			buffer = null;
			errorCleanUp("Segmentation LesionTOADS: Out of memory creating hard classification", true);
			System.err.println(e.getMessage());
			finalize();
			
			setCompleted(false);
			return;
		} catch (IOException error) {
			errorCleanUp("Segmentation TOADS: export problem to destImage[]", true);
			System.err.println(error.getMessage());
			finalize();
			
			setCompleted(false);
			return;
		}
        if (debug) MedicUtilPublic.displayMessage("output...\n");

		// clean-up the algorithms
		if (preprocess!=null) preprocess.finalize();
		preprocess = null;
		
		memberships = null;
		classification = null;
		lesions = null;
		
		
		setCompleted(true);
    } // calcSegmentation
	
    public ModelImage[] getResultImages(){
    	return destImage;
    }
    
   
    /**
     * Construct a readable list of the parameters to this segmentation.
     * @return       the parameter string
     */
	public String getParameterText() {
		String delim = "\n";
        String str = new String();
		for (int c=0;c<nc;c++) {
			str += "image "+(c+1)+": "+  srcImage[c].getImageName() + " (modality: "+modality[c]+")"+ delim;
		}
        str += "atlas name: " + atlasName + delim;
        str += "results: "+ resultOutput + " ("+destNum+") "+delim;
		
		str += "include lesionn classification: "+ includeLesionnClassification + delim;
		str += "smoothing: " + smoothing + delim;
        str += "outliers: "+ outlier + delim;
        str += "max difference: "+ maxDistance + delim;
        str += "iterations: "+ iterations + delim;
        str += "background threshold: "+ bgthreshold + delim;
        
        str += "max GM distance: " + maxGMDist + delim;
        str += "max Ventricle distance: "+ maxVentDist + delim;
        str += "max Brainstem distance: "+ maxBstemDist + delim;
		
		//str += "use topology: "+ useTopologyPrior + delim;
		str += "first limit: "+ firstLimit + delim;
        str += "last limit: "+ lastLimit + delim;
        str += "spread: "+ spread + delim;
		
		str += "output classification from membership: "+ outputClassificationFromMembership + delim;
		str += "atlas first coefficient: "+ atlasFirstCoefficient + delim;
		str += "atlas scale: "+ atlasScale + delim;
		
		//str += "use intensity prior: "+ useIntensityPrior + delim;
		
		str += "centroid mode: "+ centroidMode + delim;
		str += "centroid smoothing: "+ centroidSmoothness + delim;
		str += "use Lesion Weights: " + lesionWeight + delim;
		
		str += "Norm mode: " + normType + delim;
		
		str += "register: "+ register + delim;
		str += "registration steps (first,top,last): " + firstIter +", "+ topIter +", "+ lastIter + delim;
        str += "registration levels: "+ levels + delim;
        
		str += "inhomogeneity correction: "+ correctInhomogeneity + delim;
		str += "output inhomogeneity field: " + outputField + delim;
		str += "correction method: "+ correctionMethod + delim;
        str += "degree: "+ polynomialDegree + delim;
        str += "kernel: "+ splineKernel + delim;
        
        return str;
    }

	/** access the atlas information */
	public final LesionToadAtlas getAtlas() { return atlas; }
}
