package edu.jhmi.rad.medic.algorithms;

import edu.jhmi.rad.medic.dialogs.*;
import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.methods.*;
import edu.jhmi.rad.medic.structures.BinaryTree;
import edu.jhmi.rad.medic.structures.Graph;
import edu.jhmi.rad.medic.structures.Octree;
import edu.jhmi.rad.medic.utilities.*;

import gov.nih.mipav.model.algorithms.AlgorithmBase;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.view.*;

import java.io.*;
import java.util.*;
import java.lang.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import WildMagic.LibFoundation.Mathematics.Vector3f;
 
/**
 *	computes topological quantities or operations on sets of binary objects.
 *  <p>
 *	This algorithm groups several useful tools, including Euler number computation,
 *	topology-preserving morphological operations, connectivity check, etc. 
 *	Some of the tools may not be fully functional, or have limitations.
 *	
 *	@version    July 2004
 *	@author     Pierre-Louis Bazin
 *  @see 		JDialogObjectTopology
 *  @see 		TopologyPropagation
 *		
 *
*/
public class AlgorithmObjectTopology extends AlgorithmBase {
	private static final String cvsversion = "$Revision: 1.1 $";
	public static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");

	public static String get_version() {
		return revnum;
	}
    // Fuzzy images require 1 image for each class
    // Hard images 1 image with assigned clusters
    private ModelImage  destImage;
    private ViewUserInterface       userInterface;

    
     // information on the parts of the image to process
    private int         nxmin,nymin,nzmin,nxmax,nymax,nzmax;

    // image size
	private int 		nx,ny,nz; // original image dimensions
	private int			mx,my,mz; // reduced image dimensions
    private int         x0,xN,y0,yN,z0,zN;
	private float 		rx,ry,rz; // image resolution
	
    // algorithm parameters
	private int   	objId;
    private int   	bgId;
    private int   	iterations;
    private int         dim;
    private int         cObj;
    private int         cBack;
    private String      type;
    
	private boolean		verbose = true;
	private boolean		debug = false;
	private	int			border=2;
	
    /**
    *	Constructor for 3D images in which changes are placed in a predetermined destination image.
    *   @param destImage_      Image model where result image is to stored.
    *   @param srcImage_       Source image model.
    */
	public AlgorithmObjectTopology(ModelImage destImage_, ModelImage srcImage_,
                                    String type_,
                                    int objId_, int bgId_, 
									int iter_, 
									int dim_, int cO_, int cB_) {
        super(null, srcImage_);
		userInterface = ViewUserInterface.getReference();
		destImage = destImage_;  // Put results in destination image.
        
        type = type_;
		objId = objId_;
        bgId = bgId_;
		iterations = iter_;
        dim = dim_;
        cObj = cO_;
        cBack = cB_;
	
		if (type.equals("Distance_function") || type.equals("Voronoi_FT")) border += iterations;
	}

    /**
    *	Prepares this class for destruction.
    */
	public void finalize(){
	    destImage   = null;
	    srcImage    = null;
        System.gc();
        super.finalize();
	}

    /**
    *	Constructs a string of the contruction parameters and outputs the string to the messsage frame if the logging
    *   procedure is turned on.
    *	@param destinationFlag	If true the log includes the name of the destination flag.
    */
	/*
    private void constructLog(boolean destinationFlag) {
        if ( destinationFlag == false) {
            historyString = new String( "ObSeg(" + ")");
        }
        else  {
            historyString = new String( "ObSeg(" + ")");
        }
        historyString += "\n";  // append a newline onto the string
        writeLog();
    }
	*/
	
    /**
    *   Starts the algorithm.
    */
	public void runAlgorithm() {

        if (srcImage  == null) {
            displayError("Source Image is null");
            //notifyListeners(this);
            return;
        }
        if (destImage  == null) {
            displayError("Source Image is null");
            //notifyListeners(this);
            return;
        }

        // start the timer to compute the elapsed time
        setStartTime();

        if (destImage != null){     // if there exists a destination image
            //constructLog(true);
			
			float [] buffer;
            if (srcImage.getNDims() == 2){
				try {
					// image length is length in 2 dims
					int length = srcImage.getExtents()[0]  * srcImage.getExtents()[1];
					buffer       = new float[length];
					srcImage.exportData(0,length, buffer); // locks and releases lock
				}
				catch (IOException error) {
					buffer = null;
					errorCleanUp("Algorithm TopologyMap reports: source image locked", true);
					return;
				}
				catch (OutOfMemoryError e){
					buffer = null;
					errorCleanUp("Algorithm TopoFactory reports: out of memory", true);
					return;
				}

				// init dimensions
				nx = srcImage.getExtents()[0];
				ny = srcImage.getExtents()[1];
				nz = 1;
				
				rx = srcImage.getFileInfo()[0].getResolutions()[0];
				ry = srcImage.getFileInfo()[0].getResolutions()[1];
				rz = 1.0f;
				
				// main algorithm
				calcTopology(buffer);
				
			}
	        else if (srcImage.getNDims() > 2) {
			   try {
					// image length is length in 3 dims
					int length = srcImage.getExtents()[0] *
							 srcImage.getExtents()[1] *
							 srcImage.getExtents()[2];
					buffer = new float[length];
					srcImage.exportData(0,length, buffer); // locks and releases lock
					fireProgressStateChanged(srcImage.getImageName(), "Processing image ...");
				}
				catch (IOException error) {
					buffer = null;
					errorCleanUp("Algorithm TopoFactory: source image locked", true);
					return;
				}
				catch (OutOfMemoryError e){
					buffer = null;
					errorCleanUp("Algorithm TopoFactory: Out of memory creating process buffer", true);
					return;
				}
		
				// init dimensions
				nx = srcImage.getExtents()[0];
				ny = srcImage.getExtents()[1];
				nz = srcImage.getExtents()[2];
				
				rx = srcImage.getFileInfo()[0].getResolutions()[0];
				ry = srcImage.getFileInfo()[0].getResolutions()[1];
				rz = srcImage.getFileInfo()[0].getResolutions()[2];
				
				// main algorithm
				calcTopology(buffer);
				
			}
        }

        // compute the elapsed time
        computeElapsedTime();
    } // end run()

    /**
    *	produces a topology propagation
    */
    private void calcTopology(float img[]){
		float[][][]   	result=null; 
        float[][][]     image;
		boolean[][][]     objectMask;
		boolean[][][]     obj = null;
		boolean[][]     obj2D = null;
		int[][][]     label = null;
		int[][]     label2D = null;
		//int 		x,y,z,k,m,i,j,l;
		float   	num,den;
		float   	ngb, neighbors;
		float   	dist;
		float[][][]   	ext = null; 
        float[]     buffer;
		boolean[][][]     paint;
		boolean[][][]     zer;
		TopologyPropagation algorithm = null;
		String info;
		float[][][] lb = null;
        float[][][] tmp = null;
            
		int progress;

		if (debug) MedicUtilPublic.displayMessage("start propagation\n");
				
		fireProgressStateChanged(srcImage.getImageName(), "Processing image ...");
		
		if (debug) MedicUtilPublic.displayMessage("extract data..\n");
		
		/* pre-processing : expand boundaries, so that we don't have to worry for them */
        try {
            expandSize();
            image = expandBoundaries(img);
            paint = null;
            // dispose of img
            img = null;
        } catch (OutOfMemoryError e) {
            img = null;
            errorCleanUp("Algorithm RFCM: Out of memory creating process buffer", true);
            setCompleted(false);
            return;
        }
		if (debug) MedicUtilPublic.displayMessage("create mask..\n");
		
		// if use mask, process only parts with objId or bgId)
		//objectMask = createObjectMask(image, objId, bgId);
        objectMask = createFullMask(image);
        
		// create smaller images for speed and memory
		if (debug) MedicUtilPublic.displayMessage("compute boundaries..\n");
		computeProcessingBoundaries(objectMask);
		
		if (debug) MedicUtilPublic.displayMessage("shrink images..\n");
		image = reduceImageSize(image); 
		objectMask = reduceImageSize(objectMask);
        if (debug) MedicUtilPublic.displayMessage("new dimensions: "+mx+"x"+my+"x"+mz+"\n");
		
		/***************************************/
		/*          MAIN ALGORITHM             */
		/***************************************/
		if (debug) MedicUtilPublic.displayMessage("start algorithm\n");
 
		/* record time */
		long start_time = System.currentTimeMillis();
		long inner_loop_time;

		fireProgressStateChanged(type);
		
        
		if (type.equals("ID_Objects")) {
			// find all separate objects
			result = new float[mx][my][mz];
            for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				result[x][y][z] = 0.0f;
			}
			// select the region for the label k
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
			
			// label all pieces
			if (cObj==6) label = ObjectProcessing.connected6Object3D(obj, mx, my, mz);
			else if (cObj==18) label = ObjectProcessing.connected18Object3D(obj, mx, my, mz);
			else label = ObjectProcessing.connected26Object3D(obj, mx, my, mz);  

			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				result[x][y][z] = label[x][y][z];
			}
			if (verbose) MedicUtilPublic.displayMessage("number of objects: "+ObjectProcessing.countLabels(label, mx, my, mz)+"\n");
		} 
		else if (type.equals("Main_Object")) {
			// find all separate objects
			result = new float[mx][my][mz];
            for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				result[x][y][z] = 0.0f;
			}
			// select the region for the label k
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
			
			// label all pieces
			if (cObj==6) label = ObjectProcessing.connected6Object3D(obj, mx, my, mz);
			else if (cObj==18) label = ObjectProcessing.connected18Object3D(obj, mx, my, mz);
			else label = ObjectProcessing.connected26Object3D(obj, mx, my, mz);  

			// find largest
			obj = ObjectProcessing.largestObjectFromLabel(label, mx, my, mz);
			
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				if (obj[x][y][z]) result[x][y][z] = 1.0f;
			}
		} 
		else if (type.equals("T-objects")) {
			// create a spherical appromixation inside the object 
			result = new float[mx][my][mz];
            ext = new float[mx][my][mz];
			zer = new boolean[mx][my][mz];
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				result[x][y][z] = 0.0f;
			}
			// select the region for the label k
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
			
			// label all pieces
			if (cObj==6) label = ObjectProcessing.connected6Object3D(obj, mx, my, mz);
			else if (cObj==18) label = ObjectProcessing.connected18Object3D(obj, mx, my, mz);
			else label = ObjectProcessing.connected26Object3D(obj, mx, my, mz);  
			int Nlb = ObjectProcessing.countLabels(label,mx,my,mz);
			if (verbose) MedicUtilPublic.displayMessage("separate objects: "+(Nlb-1)+"\n");
			
			// process the largest recursively
			for (int l=0;l<iterations;l++) {
				obj = ObjectProcessing.largestObjectFromLabel(label, Nlb, mx, my, mz);
				if (verbose) MedicUtilPublic.displayMessage("part "+(l+1)+"\n");
				// convert obj
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if (obj[x][y][z]) {
						ext[x][y][z] = 1.0f;
					} else {
						ext[x][y][z] = 0.0f;
					}
					zer[x][y][z] = true;
				}
				
				// compute distance function
				algorithm = new TopologyPropagation(ext, zer, mx, my, mz, 
										0.5f, 1.0f, 0.0f, 0.01f,
										dim, cObj, cBack, false, 
										"intensity", null, null, userInterface, getProgressChangeListener());

				algorithm.propagateUpwardUnconstrainedGeometricDistance();
				lb = algorithm.exportDistance();
				algorithm.finalize();
				
				// use the (maxDistance)-levelset of distances for skeleton init
				float Dmax = 0.0f;
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if (lb[x][y][z] > Dmax) Dmax = lb[x][y][z];
				}
				if (verbose) MedicUtilPublic.displayMessage("max dist: "+Dmax+"\n");
				// set the template
				boolean stop = false;
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) if (!stop) {
					if (lb[x][y][z] == Dmax) {
						stop = true;
						lb[x][y][z] = Dmax+1.0f;
					}	
				}
				
				algorithm = new TopologyPropagation(lb, obj, mx, my, mz, 
										0.0f, Dmax+1.0f, 0.0f, 0.01f,
										dim, cObj, cBack, false, 
										"intensity", null, null, userInterface, getProgressChangeListener());
				// compute topological distance function
				algorithm.propagateDownwardGeometricDistance();
				lb = algorithm.exportLabel();
				algorithm.finalize();
				
				// add to new label
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if (lb[x][y][z]==TopologyPropagation.OBJECT) {
						result[x][y][z] = objId;
					}
				}
				
				// erase object
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if (obj[x][y][z]) label[x][y][z] = 0;
				}
			}
		} 
		else if (type.equals("T-enveloppe")) {
			// clean up the template skeleton: 
			// for each object, find the sub-parts
			// then compute the distance function
			// (outside only) and then the topological distance
			// from the lowest point
			float level,Dmax;
			//find number of classes
            int Nc=0;
            float[] labels = new float[100];
            boolean isNew;
            result = new float[mx][my][mz];
            ext = new float[mx][my][mz];
			zer = new boolean[mx][my][mz];
            for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				ext[x][y][z] = 1.0f;
			}
            for (int x=2;x<mx-2;x++) for (int y=2;y<my-2;y++) for (int z=2;z<mz-2;z++) {
				ext[x][y][z] = 0.0f;
			}
			// select the region for the label k
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
			// process all pieces as one object
			if (verbose) MedicUtilPublic.displayMessage("template object \n");
			// start from the image boundary, use object as mask
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				obj[x][y][z] = !obj[x][y][z];
				//if (obj[x][y][z]) result[x][y][z] = 1.0f;
				if (ext[x][y][z]==1.0f) zer[x][y][z] = true;
				else zer[x][y][z] = false;
			}	
					
			algorithm = new TopologyPropagation(ext, obj, mx, my, mz, 
									0.5f, 1.0f, 1.0f, 0.01f,
									dim, cBack, cObj, false, 
									"intensity", null, null, userInterface, getProgressChangeListener());
			// compute topological distance function
			//algorithm.initDownwardGeometricLabels();
			algorithm.propagateDownwardUnconstrainedGeometricDistance();
			lb = algorithm.exportDistance();
			algorithm.finalize();
			
			algorithm = new TopologyPropagation(lb, obj, mx, my, mz, 
									0.0f, 1.0f, 1.0f, 0.01f,
									dim, cBack, cObj, false, 
									"paint mask", null, zer, userInterface, getProgressChangeListener());
			
			
			algorithm.propagateDownwardExactSmoothing();
			lb = algorithm.exportLabel();
			algorithm.finalize();

				// add to new label
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				if (lb[x][y][z] < TopologyPropagation.OBJECT) {
					if (result[x][y][z]==0) {
						result[x][y][z] = objId;
					} else {
						//already set!
						result[x][y][z] = -1.0f;
					}
				}
			}
		}  
		else if (type.equals("C-wrapping")) {
			// make sure a class wraps around the other ones
			result = new float[mx][my][mz];
            ext = new float[mx][my][mz];
			int xi,yj,zl;
			boolean change;
			int nConnect;
			if (cObj==6) nConnect = 3;
			else if (cObj==18) nConnect = 2;
			else nConnect = 1;
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				result[x][y][z] = image[x][y][z];
                ext[x][y][z] = 1.0f;
				if (image[x][y][z] == bgId) {
					change = false;
					for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) 
						if (i*i+j*j+l*l<=nConnect) {
							xi = Math.max(0, Math.min(x+i, mx-1));
							yj = Math.max(0, Math.min(y+j, my-1));
							zl = Math.max(0, Math.min(z+l, mz-1));
							if ( (image[xi][yj][zl]!=bgId) && (image[xi][yj][zl]!=objId) ) {
								change = true;
							}
						}
					if (change) result[x][y][z] = objId;
				}
			}
		} 
		else if (type.equals("T-dilation")) {
			float level,Dmax;
			//find number of classes
            result = new float[mx][my][mz];
            ext = new float[mx][my][mz];
			boolean[][][] dil = new boolean[mx][my][mz];
			zer = new boolean[mx][my][mz];
			for (int n=0;n<iterations;n++) {
				// select the region for the object
				obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
				// dilate the object
				dil = Morphology.dilateObject(obj,mx,my,mz,1,1,1);
				// add up object and image, set mask to union
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if (obj[x][y][z]) ext[x][y][z] = 2.0f;
					else if (dil[x][y][z]) ext[x][y][z] = 1.0f;
					else ext[x][y][z] = 0.0f;
					if ( (image[x][y][z]==objId) || (image[x][y][z]==bgId) ) {
						zer[x][y][z] = true;
					} else {
						zer[x][y][z] = false;
					}
				}
			
				// compute distance function
				algorithm = new TopologyPropagation(ext, zer, mx, my, mz, 
											0.75f, 1.0f, 0.5f, 0.01f,
											dim, cObj, cBack, true, 
											"paint mask", null, obj, userInterface, getProgressChangeListener());
	
				// compute topological distance function
				algorithm.propagateDownwardExactSmoothing();
				lb = algorithm.exportLabel();
				algorithm.finalize();
				
				// add to new label
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if (lb[x][y][z]==TopologyPropagation.OBJECT) {
						result[x][y][z] = objId;
						image[x][y][z] = objId;
					}
				}
			}
		} 
		else if (type.equals("T-erosion")) {
			float level,Dmax;
			//find number of classes
            result = new float[mx][my][mz];
            ext = new float[mx][my][mz];
			boolean[][][] ero = new boolean[mx][my][mz];
			zer = new boolean[mx][my][mz];
			for (int n=0;n<iterations;n++) {
				// select the region for the object
				obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
				// erode the object
				ero = Morphology.erodeObject(obj,mx,my,mz,1,1,1);
				// add up object and image, set mask to union
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if (ero[x][y][z]) ext[x][y][z] = 0.0f;
					else if (obj[x][y][z]) ext[x][y][z] = 1.0f;
					else ext[x][y][z] = 2.0f;
					if (ext[x][y][z]>0.0f) {
						zer[x][y][z] = true;
					} else {
						zer[x][y][z] = false;
					}
					obj[x][y][z] = !obj[x][y][z];
				}
				result = ext;
			
				// compute distance function
				algorithm = new TopologyPropagation(ext, zer, mx, my, mz, 
											0.75f, 1.0f, 0.5f, 0.01f,
											dim, cBack, cObj, true, 
											"paint mask", null, obj, userInterface, getProgressChangeListener());
	
				// compute topological distance function
				algorithm.propagateDownwardExactSmoothing();
				lb = algorithm.exportLabel();
				algorithm.finalize();
				
				// add to new label
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if (lb[x][y][z]==TopologyPropagation.OBJECT) {
						result[x][y][z] = bgId;
						image[x][y][z] = bgId;
					} else if (!obj[x][y][z]) {
						result[x][y][z] = objId;
						image[x][y][z] = objId;
					}
				}
			}
		} 
		else if (type.equals("TL-dilation")) {
			float level,Dmax;
			//find number of classes
            result = new float[mx][my][mz];
            ext = new float[mx][my][mz];
			boolean[][][] dil = new boolean[mx][my][mz];
			boolean[][][] bg;
			zer = new boolean[mx][my][mz];
			for (int n=0;n<iterations;n++) {
				// select the region for the object
				obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
				bg = ObjectProcessing.objectFromImage(image,mx,my,mz,bgId,bgId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
				//for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) nbg[x][y][z] = !nbg[x][y][z]; 
				// erode the object
				dil = Morphology.dilateObject(obj,mx,my,mz,1,1,1);
				// add up object and image, set mask to union
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if (obj[x][y][z]) ext[x][y][z] = 2.0f;
					else if ( (!bg[x][y][z]) && (!obj[x][y][z]) ) ext[x][y][z] = 2.0f;
					else if ( (dil[x][y][z]) && (bg[x][y][z]) ) ext[x][y][z] = 1.0f;
					else if ( (bg[x][y][z]) && (!dil[x][y][z]) ) ext[x][y][z] = 0.0f;
					else ext[x][y][z] = -1.0f;
					
					if (ext[x][y][z]>=0.0f) {
						zer[x][y][z] = true;
					} else {
						zer[x][y][z] = false;
					}
					//obj[x][y][z] = !obj[x][y][z];
				}
				result = ext;
			
				
				// compute distance function
				algorithm = new TopologyPropagation(ext, zer, mx, my, mz, 
											0.75f, 1.0f, 0.5f, 0.01f,
											dim, cBack, cObj, true, 
											"paint mask", null, obj, userInterface, getProgressChangeListener());
											
				// compute topological distance function
				algorithm.propagateDownwardExactSmoothing();
				lb = algorithm.exportLabel();
				algorithm.finalize();
				
				
				// add to new label
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if ( (!obj[x][y][z]) && (!bg[x][y][z]) ) {
						result[x][y][z] = image[x][y][z];
					} else if (lb[x][y][z]==TopologyPropagation.OBJECT) {
						result[x][y][z] = objId;
						image[x][y][z] = objId;
					} else if (bg[x][y][z]) {
						result[x][y][z] = bgId;
						image[x][y][z] = bgId;
					}
				}
				
				//result = lb;
				
			}
		} 
		else if (type.equals("TL-erosion")) {
			float level,Dmax;
			//find number of classes
            result = new float[mx][my][mz];
            ext = new float[mx][my][mz];
			boolean[][][] ero = new boolean[mx][my][mz];
			boolean[][][] bg;
			zer = new boolean[mx][my][mz];
			for (int n=0;n<iterations;n++) {
				// select the region for the object
				obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
				bg = ObjectProcessing.objectFromImage(image,mx,my,mz,bgId,bgId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
				//for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) nbg[x][y][z] = !nbg[x][y][z]; 
				// erode the object
				ero = Morphology.dilateObject(bg,mx,my,mz,1,1,1);
				// add up object and image, set mask to union
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if ( (!obj[x][y][z]) && (!bg[x][y][z]) ) ext[x][y][z] = 2.0f;
					else if ( (bg[x][y][z]) && (ero[x][y][z]) ) ext[x][y][z] = 2.0f;
					else if ( (ero[x][y][z]) && (obj[x][y][z]) ) ext[x][y][z] = 1.0f;
					else if ( (obj[x][y][z]) && (!ero[x][y][z]) ) ext[x][y][z] = 0.0f;
					else ext[x][y][z] = -1.0f;
					
					if (ext[x][y][z]>=0.0f) {
						zer[x][y][z] = true;
					} else {
						zer[x][y][z] = false;
					}
					obj[x][y][z] = !obj[x][y][z];
				}
				
				// compute distance function
				algorithm = new TopologyPropagation(ext, zer, mx, my, mz, 
											0.75f, 1.0f, 0.5f, 0.01f,
											dim, cBack, cObj, true, 
											"paint mask", null, obj, userInterface, getProgressChangeListener());
											
				// compute topological distance function
				algorithm.propagateDownwardExactSmoothing();
				lb = algorithm.exportLabel();
				algorithm.finalize();
								
				// add to new label
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if ( (obj[x][y][z]) && (!bg[x][y][z]) ) {
						result[x][y][z] = image[x][y][z];
					} else if (lb[x][y][z]==TopologyPropagation.OBJECT) {
						result[x][y][z] = bgId;
						image[x][y][z] = bgId;
					} else if (!obj[x][y][z]) {
						result[x][y][z] = objId;
						image[x][y][z] = objId;
					}					
				}				
			}
		} 
		else if (type.equals("Add_Points")) {
			float level,Dmax;
			//find number of classes
            result = new float[mx][my][mz];
            ext = new float[mx][my][mz];
			boolean[][][] dil = new boolean[mx][my][mz];
			boolean[][][] bg;
			zer = new boolean[mx][my][mz];

			// select the region for the object
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
			bg = ObjectProcessing.objectFromImage(image,mx,my,mz,bgId,bgId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
			// add up object and image, set mask to union
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				if (obj[x][y][z]) ext[x][y][z] = 2.0f;
				else if ( (bg[x][y][z]) ) ext[x][y][z] = 1.0f;
				else if ( (!bg[x][y][z]) && (!obj[x][y][z]) ) ext[x][y][z] = 0.0f;
				else ext[x][y][z] = -1.0f;
					
				if (ext[x][y][z]>=0.0f) {
					zer[x][y][z] = true;
				} else {
					zer[x][y][z] = false;
				}
			}
			result = ext;
		
			
			// compute distance function
			algorithm = new TopologyPropagation(ext, zer, mx, my, mz, 
										0.75f, 1.0f, 0.5f, 0.01f,
										dim, cBack, cObj, true, 
										"paint mask", null, obj, userInterface, getProgressChangeListener());
										
			// compute topological distance function
			algorithm.propagateDownwardExactSmoothing();
			lb = algorithm.exportLabel();
			algorithm.finalize();
			
			
			// add to new label
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				if ( (!obj[x][y][z]) && (!bg[x][y][z]) ) {
					result[x][y][z] = image[x][y][z];
				} else if (lb[x][y][z]==TopologyPropagation.OBJECT) {
					result[x][y][z] = objId;
					image[x][y][z] = objId;
				} else if (bg[x][y][z]) {
					result[x][y][z] = bgId;
					image[x][y][z] = bgId;
				}
			}
			
			//result = lb;
		} 
		else if (type.equals("Remove_Points")) {
			float level,Dmax;
			//find number of classes
            result = new float[mx][my][mz];
            ext = new float[mx][my][mz];
			boolean[][][] ero = new boolean[mx][my][mz];
			boolean[][][] bg;
			zer = new boolean[mx][my][mz];
			// select the region for the object
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
			bg = ObjectProcessing.objectFromImage(image,mx,my,mz,bgId,bgId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
			// add up object and image, set mask to union
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				if ( (!obj[x][y][z]) && (!bg[x][y][z]) ) ext[x][y][z] = 2.0f;
				else if ( (bg[x][y][z])  ) ext[x][y][z] = 1.0f;
				else if ( (obj[x][y][z]) ) ext[x][y][z] = 0.0f;
				else ext[x][y][z] = -1.0f;
				
				if (ext[x][y][z]>=0.0f) {
					zer[x][y][z] = true;
				} else {
					zer[x][y][z] = false;
				}
				obj[x][y][z] = !obj[x][y][z];
			}
			
			// compute distance function
			algorithm = new TopologyPropagation(ext, zer, mx, my, mz, 
										0.75f, 1.0f, 0.5f, 0.01f,
										dim, cBack, cObj, true, 
										"paint mask", null, obj, userInterface, getProgressChangeListener());
										
			// compute topological distance function
			algorithm.propagateDownwardExactSmoothing();
			lb = algorithm.exportLabel();
			algorithm.finalize();
							
			// add to new label
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				if ( (obj[x][y][z]) && (!bg[x][y][z]) ) {
					result[x][y][z] = image[x][y][z];
				} else if (lb[x][y][z]==TopologyPropagation.OBJECT) {
					result[x][y][z] = bgId;
					image[x][y][z] = bgId;
				} else if (!obj[x][y][z]) {
					result[x][y][z] = objId;
					image[x][y][z] = objId;
				}					
			}				
		} 
		else if (type.equals("Critical_Points")) {
			result = new float[mx][my][mz];
            byte[][][] map = new byte[mx][my][mz];;
			byte[][][] bgmap = new byte[mx][my][mz];;
			// select the region for the object
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				if (obj[x][y][z]) { map[x][y][z] = 1; bgmap[x][y][z] = 0; }
				else { map[x][y][z] = 0; bgmap[x][y][z] = 1; }
			}
			
			// compute distance function
			algorithm = new TopologyPropagation(result, obj, mx, my, mz, 
										0.75f, 1.0f, 0.5f, 0.01f,
										dim, cBack, cObj, true, 
										"paint mask", null, obj, userInterface, getProgressChangeListener());
										
			// find critical points
			for (int x=1;x<mx-1;x++) for (int y=1;y<my-1;y++) for (int z=1;z<mz-1;z++) {
				result[x][y][z] = 1.0f;
				if (obj[x][y][z]) { if (algorithm.isRegularObjectPoint(map,x,y,z)) result[x][y][z] = 0.0f; }
				else { if (algorithm.isRegularBackgroundPoint(bgmap,x,y,z)) result[x][y][z] = 0.0f; }				
			}				
		} 
		else if (type.equals("Euler")) {
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
			num = ObjectProcessing.eulerCharacteristic(obj,mx,my,mz,cObj,cBack);
			MedicUtilPublic.displayMessage("Euler characteristic: "+num+"\n");
			result = null;
		} 
		else if (type.equals("Find_Slices")) {
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
			result = null;
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				if (obj[x][y][z]) MedicUtilPublic.displayMessage("Slice: "+z+"\n");
			}
		} 
		else if (type.equals("Connectivity")) {
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
			result = ObjectProcessing.connectivity(obj,mx,my,mz);
		} 
		else if (type.equals("Well_Composed")) {
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
			result = ObjectProcessing.wellComposed(obj,mx,my,mz);
		} 
		else if (type.equals("Curvature")) {
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
			result = ObjectProcessing.signedDistanceFunction(obj,mx,my,mz);
			result = ImageFunctions.meanCurvature(result,mx,my,mz);
		} 
		else if (type.equals("Distance_function")) {
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
			result = ObjectProcessing.signedDistanceFunction(obj,mx,my,mz);
		} 
		else if (type.equals("Voronoi_FT")) {
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
			result = ObjectProcessing.voronoiFeatureSquaredDistance(obj,mx,my,mz);
		} 
		else if (type.equals("Convexity_score")) {
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
			result = ObjectProcessing.convexityMap(obj,mx,my,mz);
		} 
		else if (type.equals("Structure_measurements")) {
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
			result = null;
			int vol = ObjectProcessing.volume(obj,mx,my,mz);
			MedicUtilPublic.displayMessage("Size (voxels): "+vol+"\n");
			MedicUtilPublic.displayMessage("Size (mm): "+((float)vol*rx*ry*rz)+"\n");
			float mean = ObjectProcessing.meanConvexity(obj,mx,my,mz);
			MedicUtilPublic.displayMessage("Mean convexity: "+mean+"\n");
			MedicUtilPublic.displayMessage("Abs  convexity: "+ObjectProcessing.absConvexity(obj,mx,my,mz)+"\n");
			MedicUtilPublic.displayMessage("Std  convexity: "+ObjectProcessing.stdConvexity(obj,mean,mx,my,mz)+"\n");
			// all neighbors : distances + direction ?
			float[] center = ObjectProcessing.center(obj,mx,my,mz);
			int Nb = ObjectProcessing.countLabels(image,mx,my,mz);
			float[] list = ObjectProcessing.listLabels(image,mx,my,mz);
			MedicUtilPublic.displayMessage("Distance to objects\n");
			for (int n=0;n<Nb;n++) {
				obj = ObjectProcessing.objectFromImage(image,mx,my,mz,list[n],list[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
				float[] next = ObjectProcessing.center(obj,mx,my,mz);
				MedicUtilPublic.displayMessage(list[n]+" : "
					+Math.sqrt((next[0]-center[0])*rx*(next[0]-center[0])*rx
							  +(next[1]-center[1])*ry*(next[1]-center[1])*ry
							  +(next[2]-center[2])*rz*(next[2]-center[2])*rz));
				if (  ( (next[0]-center[0])*rx*(next[0]-center[0])*rx > (next[1]-center[1])*ry*(next[1]-center[1])*ry )
					&&( (next[0]-center[0])*rx*(next[0]-center[0])*rx > (next[2]-center[2])*rz*(next[2]-center[2])*rz ) ) {
					if (next[0]>center[0]) MedicUtilPublic.displayMessage(" +x\n");
					else MedicUtilPublic.displayMessage(" -x\n");
				} else if (  ( (next[1]-center[1])*ry*(next[1]-center[1])*ry > (next[0]-center[0])*rx*(next[0]-center[0])*rx )
						   &&( (next[1]-center[1])*ry*(next[1]-center[1])*ry > (next[2]-center[2])*rz*(next[2]-center[2])*rz ) ) {
					if (next[1]>center[1]) MedicUtilPublic.displayMessage(" +y\n");
					else MedicUtilPublic.displayMessage(" -y\n");
				} else if (  ( (next[2]-center[2])*rz*(next[2]-center[2])*rz > (next[0]-center[0])*rx*(next[0]-center[0])*rx )
						   &&( (next[2]-center[2])*rz*(next[2]-center[2])*rz > (next[1]-center[1])*ry*(next[1]-center[1])*ry ) ) {
					if (next[2]>center[2]) MedicUtilPublic.displayMessage(" +z\n");
					else MedicUtilPublic.displayMessage(" -z\n");
				} else MedicUtilPublic.displayMessage("  0\n");
			}
			// bounding neighbors : id
			
		} 
		else if (type.equals("Full_structure")) {
			result = null;
			int Nb = ObjectProcessing.countLabels(image,mx,my,mz);
			float[] list = ObjectProcessing.listOrderedLabels(image,mx,my,mz);
			float[] meandist = new float[Nb];
			float[][] meandir = new float[Nb][3];
			float[][] meanloc = new float[Nb][3];
			float[] varloc;
			float[] vardir;
			float[] origin = new float[3];
			float[][] center = new float[Nb][3];
			
			for (int i=0;i<3;i++) origin[i] = 0.0f;
			float Norig = 0.0f;
			for (int n=0;n<Nb;n++) if (list[n]>0) {
				obj = ObjectProcessing.objectFromImage(image,mx,my,mz,list[n],list[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
				center[n] = ObjectProcessing.center(obj,mx,my,mz);
				for (int i=0;i<3;i++) origin[i] += center[n][i];
				Norig++;
			}
			for (int i=0;i<3;i++) origin[i] = origin[i]/Norig;
			MedicUtilPublic.displayMessage("\n");
			
			MedicUtilPublic.displayMessage("Labels:\n");
			for (int n=0;n<Nb;n++) if (list[n]>0) MedicUtilPublic.displayMessage(list[n]+"	");
			MedicUtilPublic.displayMessage(" \n");
			
			MedicUtilPublic.displayMessage("Volume:\n");	
			for (int n=0;n<Nb;n++) if (list[n]>0) {
				obj = ObjectProcessing.objectFromImage(image,mx,my,mz,list[n],list[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
				int vol = ObjectProcessing.volume(obj,mx,my,mz);
				MedicUtilPublic.displayMessage(((float)vol*rx*ry*rz)+"	");
			}
			MedicUtilPublic.displayMessage(" \n");
				
			MedicUtilPublic.displayMessage("Area:\n");
			for (int n=0;n<Nb;n++) if (list[n]>0) {
				obj = ObjectProcessing.objectFromImage(image,mx,my,mz,list[n],list[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
				MedicUtilPublic.displayMessage(ObjectProcessing.boundaryArea(obj,mx,my,mz,rx,ry,rz)+"	");
			}
			MedicUtilPublic.displayMessage(" \n");
			
			MedicUtilPublic.displayMessage("Location:\n");
			for (int n=0;n<Nb;n++) if (list[n]>0) {
				MedicUtilPublic.displayMessage(((center[n][0]-origin[0])*rx)+"	");
				MedicUtilPublic.displayMessage(((center[n][1]-origin[1])*ry)+"	");
				MedicUtilPublic.displayMessage(((center[n][2]-origin[2])*rz)+"	");
			}
			MedicUtilPublic.displayMessage(" \n");
			for (int n=0;n<Nb;n++) if (list[n]>0) {
				obj = ObjectProcessing.objectFromImage(image,mx,my,mz,list[n],list[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
				float[] dev = ObjectProcessing.deviation(obj,center[n],mx,my,mz);
				MedicUtilPublic.displayMessage((dev[0]*rx)+"	");
				MedicUtilPublic.displayMessage((dev[1]*ry)+"	");
				MedicUtilPublic.displayMessage((dev[2]*rz)+"	");
			}
			MedicUtilPublic.displayMessage(" \n");
			
			MedicUtilPublic.displayMessage("Mean Convexity:\n");
			for (int n=0;n<Nb;n++) if (list[n]>0) {
				obj = ObjectProcessing.objectFromImage(image,mx,my,mz,list[n],list[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
				float mean = ObjectProcessing.meanConvexity(obj,mx,my,mz);
				MedicUtilPublic.displayMessage(mean+"	");
			}
			MedicUtilPublic.displayMessage(" \n");
			
			MedicUtilPublic.displayMessage("Std Convexity:\n");
			for (int n=0;n<Nb;n++) if (list[n]>0) {
				obj = ObjectProcessing.objectFromImage(image,mx,my,mz,list[n],list[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
				float mean = ObjectProcessing.meanConvexity(obj,mx,my,mz);
				MedicUtilPublic.displayMessage(ObjectProcessing.stdConvexity(obj,mean,mx,my,mz)+"	");
			}
			MedicUtilPublic.displayMessage(" \n");
			
			MedicUtilPublic.displayMessage("Distance:\n");
			for (int n=0;n<Nb;n++) if (list[n]>0) {	
				for (int p=0;p<Nb;p++) if (list[p]>0) {
					obj = ObjectProcessing.objectFromImage(image,mx,my,mz,list[p],list[p],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
					meandist[p] = ObjectProcessing.meanDistance(obj,center[n][0],center[n][1],center[n][2],mx,my,mz,rx,ry,rz);
					MedicUtilPublic.displayMessage(meandist[p]+"	");
				}
				MedicUtilPublic.displayMessage(" \n");
			}
			for (int n=0;n<Nb;n++) if (list[n]>0) {	
				for (int p=0;p<Nb;p++) if (list[p]>0) {
					obj = ObjectProcessing.objectFromImage(image,mx,my,mz,list[p],list[p],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
					meandist[p] = ObjectProcessing.meanDistance(obj,center[n][0],center[n][1],center[n][2],mx,my,mz,rx,ry,rz);
					MedicUtilPublic.displayMessage(ObjectProcessing.stdDistance(obj,center[n][0],center[n][1],center[n][2],meandist[p],mx,my,mz,rx,ry,rz)+"	");
				}
				MedicUtilPublic.displayMessage(" \n");
			}
			MedicUtilPublic.displayMessage(" \n");
			
			MedicUtilPublic.displayMessage("Contact Area:\n");
			for (int n=0;n<Nb;n++) if (list[n]>0) {	
				for (int p=0;p<Nb;p++) if (list[p]>0) {
					if (p!=n) MedicUtilPublic.displayMessage(ObjectProcessing.sharedBoundaryArea(image,list[n],list[p],mx,my,mz,rx,ry,rz)+"	");
					else MedicUtilPublic.displayMessage( 0.0f+"	");
				}
				MedicUtilPublic.displayMessage(" \n");
			}
			MedicUtilPublic.displayMessage(" \n");
				
			MedicUtilPublic.displayMessage("Direction:\n");
			for (int n=0;n<Nb;n++) if (list[n]>0) {	
				for (int p=0;p<Nb;p++) if (list[p]>0) {
					obj = ObjectProcessing.objectFromImage(image,mx,my,mz,list[p],list[p],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
					meandir[p] = ObjectProcessing.meanDirection(obj,center[n][0],center[n][1],center[n][2],mx,my,mz,rx,ry,rz);
					MedicUtilPublic.displayMessage(meandir[p][0]+"	");
					MedicUtilPublic.displayMessage(meandir[p][1]+"	");
					MedicUtilPublic.displayMessage(meandir[p][2]+"	");
				}
				MedicUtilPublic.displayMessage(" \n");
			}
			for (int n=0;n<Nb;n++) if (list[n]>0) {	
				for (int p=0;p<Nb;p++) if (list[p]>0) {
					obj = ObjectProcessing.objectFromImage(image,mx,my,mz,list[p],list[p],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
					meandir[p] = ObjectProcessing.meanDirection(obj,center[n][0],center[n][1],center[n][2],mx,my,mz,rx,ry,rz);
					vardir = ObjectProcessing.stdDirection(obj,center[n][0],center[n][1],center[n][2],meandir[p],mx,my,mz,rx,ry,rz);
					MedicUtilPublic.displayMessage(vardir[0]+"	");
					MedicUtilPublic.displayMessage(vardir[1]+"	");
					MedicUtilPublic.displayMessage(vardir[2]+"	");
				}
				MedicUtilPublic.displayMessage(" \n");
			}
			MedicUtilPublic.displayMessage(" \n");
		} 
		else if (type.equals("Full_Euler")) {
			int Nb = ObjectProcessing.countLabels(image,mx,my,mz);
			float[] list = ObjectProcessing.listOrderedLabels(image,mx,my,mz);
			float[] nums = new float[Nb];
			// each separate object
			MedicUtilPublic.displayMessage("Euler characteristic:\n");
			MedicUtilPublic.displayMessage("-- single objects --\n");
			for (int n=0;n<Nb;n++) if (list[n]>0) {
				MedicUtilPublic.displayMessage((int)list[n]+"	");
			}
			MedicUtilPublic.displayMessage("\n");
			for (int n=0;n<Nb;n++) if (list[n]>0) {
				obj = ObjectProcessing.objectFromImage(image,mx,my,mz,list[n],list[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
				nums[n] = ObjectProcessing.eulerCharacteristic(obj,mx,my,mz,cObj,cBack);
				MedicUtilPublic.displayMessage(nums[n]+"	");
			}
			MedicUtilPublic.displayMessage("\n");
		} 
		else if (type.equals("Full_Euler_4")) {
			int N = ObjectProcessing.countLabels(image,mx,my,mz);
			float[] list = ObjectProcessing.listOrderedLabels(image,mx,my,mz);
			float[] nums = new float[N];
			float[] pairs = new float[N*(N-1)];
			float[] triplets = new float[N*(N-1)*(N-2)];
			float[] quadets = new float[N*(N-1)*(N-2)*(N-3)];
			String signature = "Euler signature: ";
			int ind,ter,qad;
			
			for (int n=0;n<N;n++) if (list[n]>0) {
				boolean[][][] obj1 = ObjectProcessing.objectFromImage(image,mx,my,mz,list[n],list[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
				nums[n] = ObjectProcessing.eulerCharacteristic(obj1,mx,my,mz,cObj,cBack);
				
				for (int p=n+1;p<N;p++) if (list[p]>0) {
					boolean[][][] obj2 = ObjectProcessing.objectFromImage(image,mx,my,mz,list[p],list[p],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
					for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
						if (obj1[x][y][z]) obj2[x][y][z] = true;
					}
					pairs[n+N*(p-1)] = ObjectProcessing.eulerCharacteristic(obj2,mx,my,mz,cObj,cBack);
					
					for (int q=p+1;q<N;q++) if (list[q]>0) {
						boolean[][][] obj3 = ObjectProcessing.objectFromImage(image,mx,my,mz,list[q],list[q],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
						for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
							if (obj2[x][y][z]) obj3[x][y][z] = true;
						}
						triplets[n+N*(p-1)+N*(N-1)*(q-2)] = ObjectProcessing.eulerCharacteristic(obj3,mx,my,mz,cObj,cBack);
						
						for (int r=q+1;r<N;r++) if (list[r]>0) {
							boolean[][][] obj4 = ObjectProcessing.objectFromImage(image,mx,my,mz,list[r],list[r],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
							for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
								if (obj3[x][y][z]) obj4[x][y][z] = true;
							}
							quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)] = ObjectProcessing.eulerCharacteristic(obj4,mx,my,mz,cObj,cBack);
						}
					}
				}
			}
			// singles
			for (int n=0;n<N;n++) if (list[n]>0) {
				if (n!=N-1) signature += (int)nums[n]+",";
				else signature += (int)nums[n];
			}
			System.out.print(signature+"\n");
			//pairs
			for (int n=0;n<N;n++) if (list[n]>0) {
				for (int p=n+1;p<N;p++) if (list[p]>0) {
					if (pairs[n+N*(p-1)]!=nums[n]+nums[p])
						signature += ";"+(int)pairs[n+N*(p-1)];
				}
			}
			System.out.print(signature+"\n");
			//triplets
			for (int n=0;n<N;n++) if (list[n]>0) {
				for (int p=n+1;p<N;p++) if (list[p]>0) {
					for (int q=p+1;q<N;q++) if (list[q]>0) {
						if (triplets[n+N*(p-1)+N*(N-1)*(q-2)]!=
							nums[n]+nums[p]+nums[q]
							-pairs[n+N*(p-1)]-pairs[n+N*(q-1)]-pairs[p+N*(q-1)])
							signature += "."+(int)triplets[n+N*(p-1)+N*(N-1)*(q-2)];
					}
				}
			}
			System.out.print(signature+"\n");
			// quadets
			for (int n=0;n<N;n++) if (list[n]>0) {
				for (int p=n+1;p<N;p++) if (list[p]>0) {
					for (int q=p+1;q<N;q++) if (list[q]>0) {
						for (int r=q+1;r<N;r++) if (list[r]>0) {
							if (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=
								nums[n]+nums[p]+nums[q]+nums[r]
								-pairs[n+N*(p-1)]-pairs[n+N*(q-1)]-pairs[n+N*(r-1)]-pairs[p+N*(q-1)]-pairs[p+N*(r-1)]-pairs[q+N*(r-1)]
								+triplets[n+N*(p-1)+N*(N-1)*(q-2)]+triplets[n+N*(p-1)+N*(N-1)*(r-2)]+triplets[n+N*(q-1)+N*(N-1)*(r-2)]+triplets[p+N*(q-1)+N*(N-1)*(r-2)])
							signature += ":"+(int)quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)];
						}
					}
				}
			}
			System.out.print(signature+"\n");
			MedicUtilPublic.displayMessage("\n");
			MedicUtilPublic.displayMessage(signature+"\n");
			
			// each separate object
			MedicUtilPublic.displayMessage("Euler characteristic:\n");
			MedicUtilPublic.displayMessage("-- single objects --\n");
			for (int n=0;n<N;n++) if (list[n]>0) 
				MedicUtilPublic.displayMessage((int)list[n]+"	");
			MedicUtilPublic.displayMessage("\n");
			for (int n=0;n<N;n++) if (list[n]>0) 
				MedicUtilPublic.displayMessage((int)nums[n]+"	");
			MedicUtilPublic.displayMessage("\n");
			
			MedicUtilPublic.displayMessage("-- pairs --\n");
			for (int n=0;n<N;n++) if (list[n]>0) 
				for (int p=n+1;p<N;p++) if (list[p]>0) 
					if (pairs[n+N*(p-1)]!=nums[n]+nums[p]) MedicUtilPublic.displayMessage((int)list[n]+"-"+(int)list[p]+"	");
			MedicUtilPublic.displayMessage("\n");
			for (int n=0;n<N;n++) if (list[n]>0) 
				for (int p=n+1;p<N;p++) if (list[p]>0) 
					if (pairs[n+N*(p-1)]!=nums[n]+nums[p]) MedicUtilPublic.displayMessage((int)pairs[n+N*(p-1)]+"	");
			MedicUtilPublic.displayMessage("\n");
			
			MedicUtilPublic.displayMessage("-- triplets --\n");
			for (int n=0;n<N;n++) if (list[n]>0) 
				for (int p=n+1;p<N;p++) if (list[p]>0) 
					for (int q=p+1;q<N;q++) if (list[q]>0) 
						if ( (triplets[n+N*(p-1)+N*(N-1)*(q-2)]!=nums[n]+nums[p]+nums[q])
							&& (triplets[n+N*(p-1)+N*(N-1)*(q-2)]!=pairs[n+N*(p-1)]+nums[q])
							&& (triplets[n+N*(p-1)+N*(N-1)*(q-2)]!=pairs[n+N*(q-1)]+nums[p])
							&& (triplets[n+N*(p-1)+N*(N-1)*(q-2)]!=pairs[p+N*(q-1)]+nums[n]) ) 
							MedicUtilPublic.displayMessage((int)list[n]+"-"+(int)list[p]+"-"+(int)list[q]+"	");
			MedicUtilPublic.displayMessage("\n");
			for (int n=0;n<N;n++) if (list[n]>0) 
				for (int p=n+1;p<N;p++) if (list[p]>0) 
					for (int q=p+1;q<N;q++) if (list[q]>0) 
						if ( (triplets[n+N*(p-1)+N*(N-1)*(q-2)]!=nums[n]+nums[p]+nums[q])
							&& (triplets[n+N*(p-1)+N*(N-1)*(q-2)]!=pairs[n+N*(p-1)]+nums[q])
							&& (triplets[n+N*(p-1)+N*(N-1)*(q-2)]!=pairs[n+N*(q-1)]+nums[p])
							&& (triplets[n+N*(p-1)+N*(N-1)*(q-2)]!=pairs[p+N*(q-1)]+nums[n]) ) 
							MedicUtilPublic.displayMessage((int)triplets[n+N*(p-1)+N*(N-1)*(q-2)]+"	");
			MedicUtilPublic.displayMessage("\n");
			
			MedicUtilPublic.displayMessage("-- quadets --\n");
			for (int n=0;n<N;n++) if (list[n]>0) 
				for (int p=n+1;p<N;p++) if (list[p]>0) 
					for (int q=p+1;q<N;q++) if (list[q]>0) 
						for (int r=q+1;r<N;r++) if (list[r]>0) 
							if ( (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=nums[n]+nums[p]+nums[q]+nums[r])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=pairs[n+N*(p-1)]+pairs[q+N*(r-1)])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=pairs[n+N*(q-1)]+pairs[p+N*(r-1)])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=pairs[n+N*(r-1)]+pairs[p+N*(q-1)])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=triplets[n+N*(p-1)+N*(N-1)*(q-2)]+nums[r])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=triplets[p+N*(q-1)+N*(N-1)*(r-2)]+nums[n])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=triplets[n+N*(q-1)+N*(N-1)*(r-2)]+nums[p])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=triplets[n+N*(p-1)+N*(N-1)*(r-2)]+nums[q]) )			 
							MedicUtilPublic.displayMessage((int)list[n]+"-"+(int)list[p]+"-"+(int)list[q]+"-"+(int)list[r]+"	");
			MedicUtilPublic.displayMessage("\n");
			for (int n=0;n<N;n++) if (list[n]>0) 
				for (int p=n+1;p<N;p++) if (list[p]>0) 
					for (int q=p+1;q<N;q++) if (list[q]>0) 
						for (int r=q+1;r<N;r++) if (list[r]>0) 
							if ( (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=nums[n]+nums[p]+nums[q]+nums[r])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=pairs[n+N*(p-1)]+pairs[q+N*(r-1)])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=pairs[n+N*(q-1)]+pairs[p+N*(r-1)])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=pairs[n+N*(r-1)]+pairs[p+N*(q-1)])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=triplets[n+N*(p-1)+N*(N-1)*(q-2)]+nums[r])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=triplets[p+N*(q-1)+N*(N-1)*(r-2)]+nums[n])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=triplets[n+N*(q-1)+N*(N-1)*(r-2)]+nums[p])
								&& (quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]!=triplets[n+N*(p-1)+N*(N-1)*(r-2)]+nums[q]) )			 
							MedicUtilPublic.displayMessage(quadets[n+N*(p-1)+N*(N-1)*(q-2)+N*(N-1)*(N-2)*(r-3)]+"	");
			MedicUtilPublic.displayMessage("\n");
		} 
		else if (type.equals("Full_Composed_4")) {
			int N = ObjectProcessing.countLabels(image,mx,my,mz);
			float[] list = ObjectProcessing.listOrderedLabels(image,mx,my,mz);
			result = new float[mx][my][mz];
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				result[x][y][z] = 0.0f;
			}
			
			for (int n=0;n<N;n++) if (list[n]>0) {
				boolean[][][] obj1 = ObjectProcessing.objectFromImage(image,mx,my,mz,list[n],list[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
				float[][][] pts = ObjectProcessing.wellComposed(obj1,mx,my,mz);
				for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
					if (pts[x][y][z]==2.0f) result[x][y][z] += (n+1);
				}
				
				for (int p=n+1;p<N;p++) if (list[p]>0) {
					boolean[][][] obj2 = ObjectProcessing.objectFromImage(image,mx,my,mz,list[p],list[p],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
					for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
						if (obj1[x][y][z]) obj2[x][y][z] = true;
					}
					pts = ObjectProcessing.wellComposed(obj2,mx,my,mz);
					for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
						if (pts[x][y][z]==2.0f) result[x][y][z] += (n+1)*(p+1);
					}
				
					for (int q=p+1;q<N;q++) if (list[q]>0) {
						boolean[][][] obj3 = ObjectProcessing.objectFromImage(image,mx,my,mz,list[q],list[q],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
						for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
							if (obj2[x][y][z]) obj3[x][y][z] = true;
						}
						pts = ObjectProcessing.wellComposed(obj2,mx,my,mz);
						for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
							if (pts[x][y][z]==2.0f) result[x][y][z] += (n+1)*(p+1)*(q+1);
						}
						
						for (int r=q+1;r<N;r++) if (list[r]>0) {
							boolean[][][] obj4 = ObjectProcessing.objectFromImage(image,mx,my,mz,list[r],list[r],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);			
							for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
								if (obj3[x][y][z]) obj4[x][y][z] = true;
							}
							pts = ObjectProcessing.wellComposed(obj2,mx,my,mz);
							for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
								if (pts[x][y][z]==2.0f) result[x][y][z] += (n+1)*(p+1)*(q+1)*(r+1);
							}
						}
					}
				}
			}
		} 
		else if (type.equals("Boundaries")) {
			// find the boundaries between labels
			result = new float[mx][my][mz];
			float[][] lbs = new float[100][];
			int Nlb=1;
			float[] newlb = new float[100];
			int Nb;
			int nConnect;
			if (cObj==6) nConnect = 3;
			else if (cObj==18) nConnect = 2;
			else nConnect = 1;
			boolean isFound,isFoundLine,newValue;
			for (int x=1;x<mx-1;x++) for (int y=1;y<my-1;y++) for (int z=1;z<mz-1;z++) {
				// find boundaries
				for (int n=0;n<100;n++) newlb[n] = 0.0f;
				Nb = 0;
				for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
					if (i*i+j*j+l*l<=nConnect) {
						if (image[x][y][z]!=image[x+i][y+j][z+l]) {
							newValue = true;
							for (int b=0;b<Nb;b++) 
								if (image[x+i][y+j][z+l]==newlb[b])
									{ newValue = false; break; }
							if (newValue) {
								newlb[Nb] = image[x+i][y+j][z+l];
								Nb++;
							}
						}
					}
				}
				if (newlb[0]!=0.0f) { newlb[Nb] = image[x][y][z]; Nb++; }
				// check if boundary unique
				if (newlb[0]!=0.0f) {
					isFoundLine = false;
					for (int n=1;n<Nlb;n++) {
						isFoundLine = true;
						for (int b=0;b<Nb;b++) {
							isFound = false;
							for (int c=0;c<lbs[n].length;c++)
								if (newlb[b]==lbs[n][c]) { isFound = true; break; }
							if (!isFound) isFoundLine = false;
						}
						if (isFoundLine) {
							result[x][y][z] = (float)n;
							break;
						}
					}
					if (!isFoundLine) {
						// create a new entry
						lbs[Nlb] = new float[Nb];
						for (int b=0;b<Nb;b++) lbs[Nlb][b] = newlb[b];
						result[x][y][z] = (float)Nlb;
						Nlb++;
					}
				} else {
					result[x][y][z] = (float)0.0f;
				}
			}
			// output the list of boundaries
			MedicUtilPublic.displayMessage("boundary labels: \n");
			for (int n=1;n<Nlb;n++) {
				MedicUtilPublic.displayMessage(""+n+": (");
				for (int b=0;b<lbs[n].length;b++) MedicUtilPublic.displayMessage(" "+lbs[n][b]+" ");
				MedicUtilPublic.displayMessage(")\n");
			}
		}
		else if (type.equals("Distance_smoothing")) {
			float dmax = iterations*iterations;

			// select the region for the object
			obj = ObjectProcessing.objectFromImage(image,mx,my,mz,objId,objId,ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
		
			boolean exists = false;
			boolean[][][] bnd = new boolean[mx][my][mz];
			for (int x=1;x<mx-1;x++) for (int y=1;y<my-1;y++) for (int z=1;z<mz-1;z++) {
				if ( (obj[x][y][z]) 
					&& (  !obj[x+1][y][z] || !obj[x-1][y][z] || !obj[x][y+1][z] || !obj[x][y-1][z] || !obj[x][y][z+1] || !obj[x][y][z-1] ) ) {
					bnd[x][y][z] = true;
					exists = true;
				} else {
					bnd[x][y][z] = false;
				}
			}
			// compute distance function
			result = ObjectProcessing.voronoiFeatureSquaredDistance(bnd,mx,my,mz);
				
			// compute linear membership
			for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
				if (result[x][y][z] < dmax) {
					result[x][y][z] = 0.5f*(float)(1.0f-Math.sqrt(result[x][y][z]/dmax));
				} else {
					result[x][y][z] = 0.0f;
				}
				if (obj[x][y][z]) result[x][y][z] = 1.0f-result[x][y][z];
			}
		} 
		// free unused arrays
		//algorithm.cleanUp();

        if (threadStopped) {
			algorithm.finalize();
			finalize();
			;
			setCompleted(false);
        	return;
        } 
		
		fireProgressStateChanged("data extraction...");
		
		if (result!=null) {
			result = extendImageSize(result);

			// debug
			if (debug) MedicUtilPublic.displayMessage("total time: (milliseconds): " + (System.currentTimeMillis()-start_time)); 
			//MipavUtil.displayInfo("total time: (milliseconds): " + (System.currentTimeMillis()-start_time)); 
	
			// extract results (segmentation and/or classes) and store them in destImage[]
			fireProgressStateChanged("creating result images...");
			
			nx = nx-2*border;
			ny = ny-2*border;
			nz = nz-2*border;
			try {
				buffer = new float[nx*ny*nz];
				for (int x=0;x<nx;x++) {
					for (int y=0;y<ny;y++) {
						for (int z=0;z<nz;z++) {
							buffer[x + nx*y + nx*ny*z] = result[x+border][y+border][z+border];
						}
					}
				}
				destImage.importData(0, buffer, true);
			} catch (OutOfMemoryError e) {
				buffer = null;
				errorCleanUp("Algorithm Topology: Out of memory creating hard classification", true);
				finalize();
				;
				setCompleted(false);
				return;
			} catch (IOException error) {
				errorCleanUp("Algorithm Topology: export problem to destImage[]", true);
				finalize();
				;
				setCompleted(false);
				return;
			}
		} else {
			destImage = null;
		}
        image = null;
		objectMask = null;
		result = null;

		;
		setCompleted(true);
    } // calcSegmentation
	
	/** expand boundaries for spatial comutations */
	private void expandSize() {
		nx = nx+2*border;
		ny = ny+2*border;
		nz = nz+2*border;
	}			
	private float[][][] expandBoundaries(float[] image) {
		//int 		x,y,z;
		float[][][] 	tmp;
		
		tmp = new float[nx][ny][nz];
		for (int x=0;x<nx;x++)
			for (int y=0;y<ny;y++)
				for (int z=0;z<nz;z++)
					tmp[x][y][z] = -1.0f;
		for (int x=border;x<nx-border;x++)
			for (int y=border;y<ny-border;y++)
				for (int z=border;z<nz-border;z++)
					tmp[x][y][z] = image[ (x-border) + (nx-2*border)*(y-border) + (nx-2*border)*(ny-2*border)*(z-border) ];
		
		return tmp;
	}
	private boolean[][][] expandBoundaries(BitSet image) {
		//int 		x,y,z;
		boolean[][][] 	tmp;
		
		tmp = new boolean[nx][ny][nz];
		for (int x=border;x<nx-border;x++)
			for (int y=border;y<ny-border;y++)
				for (int z=border;z<nz-border;z++)
					tmp[x][y][z] = image.get( (x-border) + (nx-2*border)*(y-border) + (nx-2*border)*(ny-2*border)*(z-border) );
		
		return tmp;
	}
	
	/** creates a mask for unused data */
	private boolean[][][] createObjectMask(float[][][] image, int id1, int id2) {
		//int 		x,y,z;
		boolean[][][]  	objMask;

		// uses only values over the threshold, if mask used
		objMask = new boolean[nx][ny][nz];
		for (int x=0;x<nx;x++)
			for (int y=0;y<ny;y++)
				for (int z=0;z<nz;z++) {
                    if ( (image[x][y][z] == id1) || (image[x][y][z] == id2) ) 
						objMask[x][y][z] = true;
                    else
                        objMask[x][y][z] = false;
				}
		// remove the boundary from the computations
		for (int x=0;x<nx;x++)
			for (int y=0;y<ny;y++) {
				objMask[x][y][0] = false;
				objMask[x][y][nz-1] = false;
			}
		for (int y=0;y<ny;y++)
			for (int z=0;z<nz;z++) {
				objMask[0][y][z] = false;
				objMask[nx-1][y][z] = false;
			}
		for (int z=0;z<nz;z++)
			for (int x=0;x<nx;x++) {
				objMask[x][0][z] = false;
				objMask[x][ny-1][z] = false;
			}

		return objMask;
	} // createObjectMask
        
	/** creates a mask for unused data (keep everything here */
	private boolean[][][] createFullMask(float[][][] image) {
		//int 		x,y,z;
		boolean[][][]  	objMask;

		// uses only values over the threshold, if mask used
		objMask = new boolean[nx][ny][nz];
		for (int x=0;x<nx;x++)
			for (int y=0;y<ny;y++)
				for (int z=0;z<nz;z++) {
                    if (image[x][y][z]>0 ) 
						objMask[x][y][z] = true;
                    else
                        objMask[x][y][z] = false;
              	}
		// remove the boundary from the computations
		for (int x=0;x<nx;x++)
			for (int y=0;y<ny;y++) {
				objMask[x][y][0] = false;
				objMask[x][y][nz-1] = false;
			}
		for (int y=0;y<ny;y++)
			for (int z=0;z<nz;z++) {
				objMask[0][y][z] = false;
				objMask[nx-1][y][z] = false;
			}
		for (int z=0;z<nz;z++)
			for (int x=0;x<nx;x++) {
				objMask[x][0][z] = false;
				objMask[x][ny-1][z] = false;
			}

		return objMask;
	} // createFullMask
        
    /** sets the processing lower and upper boundaries */
    private void computeProcessingBoundaries(boolean[][][] objMask) {
		//int 		x,y,z;
        
        x0 = nx;
        xN = 0;
        y0 = ny;
        yN = 0;
        z0 = nz;
        zN = 0;
        for (int x=1;x<nx-1;x++)
			for (int y=1;y<ny-1;y++)
				for (int z=1;z<nz-1;z++) {
                    if (objMask[x][y][z]) {
                        if (x < x0) x0 = x;
                        if (x > xN) xN = x;
                        if (y < y0) y0 = y;
                        if (y > yN) yN = y;
                        if (z < z0) z0 = z;
                        if (z > zN) zN = z;
                    }
                }
		// update the smaller size parameters (include an extended boundary)
		mx = xN - x0 + 1 + 2*border;
		my = yN - y0 + 1 + 2*border;
		mz = zN - z0 + 1 + 2*border;
		
        // debug
        System.out.print("boundaries: ["+x0+","+xN+"] ["+y0+","+yN+"] ["+z0+","+zN+"]\n");
        
        return;
    }
	/** create smaller image (for saving memory) */
	private float[][][] reduceImageSize(float[][][] image) {
		float[][][] smaller = new float[mx][my][mz];

		for (int x=x0-border;x<=xN+border;x++) {
            for (int y=y0-border;y<=yN+border;y++) {
                for (int z=z0-border;z<=zN+border;z++) {
					smaller[x-x0+border][y-y0+border][z-z0+border] = image[x][y][z];
				}
			}
		}
		return smaller;
	}
	private boolean[][][] reduceImageSize(boolean[][][] image) {
		boolean[][][] smaller = new boolean[mx][my][mz];

		for (int x=x0-border;x<=xN+border;x++) {
            for (int y=y0-border;y<=yN+border;y++) {
                for (int z=z0-border;z<=zN+border;z++) {
					smaller[x-x0+border][y-y0+border][z-z0+border] = image[x][y][z];
				}
			}
		}
		return smaller;
	}
	private void reducePointPosition(Vector3f[] pt) {
		for (int n=0;n<pt.length;n++) {
            // from image to image+boundary to reduced image
            pt[n].X = pt[n].X - x0 + border + 2*border;
            pt[n].Y = pt[n].Y - y0 + border + 2*border;
            pt[n].Z = pt[n].Z - z0 + border + 2*border;
		}
		return;
	}
	/** retrieve original size from smaller image */
	private float[][][] extendImageSize(float[][][] image) {
		float[][][] larger = new float[nx][ny][nz];

		float min = ImageFunctions.minimum(image,mx,my,mz);
		for (int x=0;x<nx;x++) {
			for (int y=0;y<ny;y++) {
				for (int z=0;z<nz;z++) {
					larger[x][y][z] = min;
				}
			}
		}		
 		for (int x=x0-border;x<=xN+border;x++) {
            for (int y=y0-border;y<=yN+border;y++) {
                for (int z=z0-border;z<=zN+border;z++) {
					larger[x][y][z] = image[x-x0+border][y-y0+border][z-z0+border];
				}
			}
		}
		return larger;
	}
	/** retrieve original size from smaller image */
	private float[][][] extendImageSize(int[][][] image) {
		float[][][] larger = new float[nx][ny][nz];

		int min = ImageFunctions.minimum(image,mx,my,mz);
		for (int x=0;x<nx;x++) {
			for (int y=0;y<ny;y++) {
				for (int z=0;z<nz;z++) {
					larger[x][y][z] = min;
				}
			}
		}		
 		for (int x=x0-border;x<=xN+border;x++) {
            for (int y=y0-border;y<=yN+border;y++) {
                for (int z=z0-border;z<=zN+border;z++) {
					larger[x][y][z] = (float)image[x-x0+border][y-y0+border][z-z0+border];
				}
			}
		}
		return larger;
	}
	/** retrieve original size from smaller image */
	private float[][][] extendImageSize(byte[][][] image) {
		float[][][] larger = new float[nx][ny][nz];

		byte min = ImageFunctions.minimum(image,mx,my,mz);
		for (int x=0;x<nx;x++) {
			for (int y=0;y<ny;y++) {
				for (int z=0;z<nz;z++) {
					larger[x][y][z] = min;
				}
			}
		}		
 		for (int x=x0-border;x<=xN+border;x++) {
            for (int y=y0-border;y<=yN+border;y++) {
                for (int z=z0-border;z<=zN+border;z++) {
					larger[x][y][z] = (float)image[x-x0+border][y-y0+border][z-z0+border];
				}
			}
		}
		return larger;
	}
    /** retrieve original size from smaller image */
	private float[][][] extendImageSize(boolean[][][] image) {
		float[][][] larger = new float[nx][ny][nz];

		for (int x=0;x<nx;x++) {
			for (int y=0;y<ny;y++) {
				for (int z=0;z<nz;z++) {
					larger[x][y][z] = 0.0f;
				}
			}
		}		
 		for (int x=x0-border;x<=xN+border;x++) {
            for (int y=y0-border;y<=yN+border;y++) {
                for (int z=z0-border;z<=zN+border;z++) {
                    if (image[x-x0+border][y-y0+border][z-z0+border])
                        larger[x][y][z] = 1.0f;
                    else
                        larger[x][y][z] = 0.0f;
				}
			}
		}
		return larger;
	}
}
