package edu.jhmi.rad.medic.algorithms;

import edu.jhmi.rad.medic.dialogs.*;
import edu.jhmi.rad.medic.methods.*;
import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.utilities.*;

import gov.nih.mipav.model.algorithms.AlgorithmBase;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.view.*;

import java.io.*;
import java.util.*;
import java.lang.*;
import java.text.*;
 
/**
 *
 *   computes several measurements for quantifying volumetric images.
 *   <p>
 *	 It can compare hard segmentations and fuzzy memberships,
 *	 computing volume, overlap, topology, intensity measures.
 *
 *	@version    May 2007
 *	@author     Pierre-Louis Bazin
 *  @see 		JDialogVolumeStatistics
 *		
 *
*/
public class AlgorithmVolumeStatistics2 extends AlgorithmBase {

    private ModelImage  intImage;
    private ModelImage  refImage;
	
    private ViewUserInterface       userInterface;

    // image information
	private int 		nx,ny,nz,nt; 			// original image dimensions
	private float 		rx,ry,rz; 			// original image resolutions
	
	// algorithm parameters
	private ArrayList<String>	statistics;		// the type of algorithm to use
	private	boolean			useVOI;				// whether to use VOIs or not
	
	private	String			connectivity;			// connectivity	
	private	int				cObj, cBg;
	
	private ArrayList<String>	output;		// the type of algorithm to use
	private boolean		writeToFile;
	private	String		filename;
	private	final		String 		delim;
		    
	private boolean		verbose = true;
	private     boolean		debug = true;
	
	private	static final	float	INF = 1e30f;
	private	static final	float	ZERO = 1e-30f;
	
	/**
    *	Constructor for 3D images in which changes are placed in a predetermined destination image.
    *   @param targetImg      Image model where result image is to stored.
    *   @param srcImg       Source image model.
    */
	public AlgorithmVolumeStatistics2(ModelImage srcImg, ModelImage intImg, ModelImage refImg, 
										ArrayList<String> stats_, 
										boolean voi_,
										String connect_,
										boolean tf_, String file_, String del_) {
			
		super(null, srcImg);
		userInterface = ViewUserInterface.getReference();
        
		intImage = intImg;
		refImage = refImg;
		
		statistics = stats_;
		useVOI = voi_;
		connectivity = connect_;
			 if (connectivity.equals("6/18")) { cObj = 6; cBg = 18; }
		else if (connectivity.equals("6/26")) { cObj = 6; cBg = 26; }
		else if (connectivity.equals("18/6")) { cObj = 18; cBg = 6; }
		else if (connectivity.equals("26/6")) { cObj = 26; cBg = 6; }

		
		writeToFile = tf_;
		filename = file_;
		delim = del_;
	}

    /**
    *	Prepares this class for destruction.
    */
	public void finalize(){
        super.finalize();
	    srcImage = null;
        intImage = null;
		refImage = null;
        System.gc();
	}

    /**
    *	Constructs a string of the contruction parameters and outputs the string to the messsage frame if the logging
    *   procedure is turned on.
    *	@param destinationFlag	If true the log includes the name of the destination flag.
    */
	/*
    private void constructLog(boolean destinationFlag) {
        if ( destinationFlag == false) {
            historyString = new String( "VolumeStatistics2(" + ")");
        }
        else  {
            historyString = new String( "VolumeStatistics2(" + ")");
        }
        historyString += "\n";  // append a newline onto the string
        writeLog();
    }
	*/

    /**
    *   Starts the algorithm.
    */
	public void runAlgorithm() {

        if (srcImage  == null) {
            displayError("Source image is null");
            return;
        }
        
		// start the timer to compute the elapsed time
        setStartTime();

		// computations here
		//constructLog(true);
		
		// init dimensions and resolutions
		if (srcImage.getNDims() == 2) {
			nx = srcImage.getExtents()[0];
			ny = srcImage.getExtents()[1];
			nz = 1;
			nt = 1;
			
			rx = srcImage.getFileInfo()[0].getResolutions()[0];
			ry = srcImage.getFileInfo()[0].getResolutions()[1];
			rz = 1.0f;			
		} else if (srcImage.getNDims() == 3) {
			nx = srcImage.getExtents()[0];
			ny = srcImage.getExtents()[1];
			nz = srcImage.getExtents()[2];
			nt = 1;
			
			rx = srcImage.getFileInfo()[0].getResolutions()[0];
			ry = srcImage.getFileInfo()[0].getResolutions()[1];
			rz = srcImage.getFileInfo()[0].getResolutions()[2];
		} else if (srcImage.getNDims() == 4) {
			nx = srcImage.getExtents()[0];
			ny = srcImage.getExtents()[1];
			nz = srcImage.getExtents()[2];
			nt = srcImage.getExtents()[3];
			
			rx = srcImage.getFileInfo()[0].getResolutions()[0];
			ry = srcImage.getFileInfo()[0].getResolutions()[1];
			rz = srcImage.getFileInfo()[0].getResolutions()[2];
		}
		
		for (int t=0;t<nt;t++) {
		
			// 1. Main image
			int[] 	imgbuffer = null;
			float[]	intbuffer = null;
			int[]	refbuffer = null;
			try {
				if (srcImage.isColorImage()) {
					imgbuffer = new int[4*nx*ny*nz];	
					srcImage.exportData(t*4*nx*ny*nz,4*nx*ny*nz, imgbuffer); // locks and releases lock				
				} else {
					imgbuffer = new int[nx*ny*nz];	
					srcImage.exportData(t*nx*ny*nz,nx*ny*nz, imgbuffer); // locks and releases lock
				}
				if (intImage!=null) {
					intbuffer = new float[nx*ny*nz];
					intImage.exportData(t*nx*ny*nz,nx*ny*nz, intbuffer); // locks and releases lock
				}
				if (refImage!=null) {
					if (refImage.isColorImage()) {
						refbuffer = new int[4*nx*ny*nz];
						refImage.exportData(t*4*nx*ny*nz,4*nx*ny*nz, refbuffer); // locks and releases lock
					} else {
						refbuffer = new int[nx*ny*nz];
						refImage.exportData(t*nx*ny*nz,nx*ny*nz, refbuffer); // locks and releases lock
					}
				}
			} catch (IOException error) {
				imgbuffer = null;
				intbuffer = null;
				refbuffer = null;
				errorCleanUp("Algorithm: source images locked", true);
				return;
			} catch (OutOfMemoryError e){
				imgbuffer = null;
				intbuffer = null;
				refbuffer = null;
				errorCleanUp("Algorithm: Out of memory creating process buffer", true);
				return;
			}
			
			computeStatistics(imgbuffer, intbuffer, refbuffer);
		}
		// compute the elapsed time
        computeElapsedTime();

        setCompleted(true);
    } // end runAlgorithm()
	
	/**
    *	compute the size and overlap of the new image with regard to reference
	*	it is assumed that the labels may differ but follow the same ordering in intensity
    */
    private void computeStatistics(int[] img, float[] intens, int[] ref){
		int[][][]				image;
		float[][][]				intensity = null;
		int[][][]				reference = null;
		ImageCropper			crop = new ImageCropper(nx,ny,nz);
		
		int[][] imgMapping = null;
		int[][] refMapping = null;
		
		int mod;
		int progress;

		fireProgressStateChanged("initialization...");
        
		try {
            if (srcImage.isColorImage()) {
				imgMapping = crop.mapColorsToLabels(img);
				image = crop.convertColorToLabelImage(img,imgMapping);
			} else {
				image = crop.convertArray(img);
			}
            img = null;
            if (intens!=null) intensity = crop.convertArray(intens);
            intens = null;
            if (ref!=null) {
				if (refImage.isColorImage()) {
					refMapping = crop.mapColorsToLabels(ref);
					reference = crop.convertColorToLabelImage(ref,refMapping);
				} else {
					reference = crop.convertArray(ref);
				}
			}
            ref = null;
        } catch (OutOfMemoryError e) {
            img = null; image = null;
            intens = null; intensity = null;
            ref = null; reference = null;
            errorCleanUp("Algorithm: Out of memory creating process buffer", true);
            setCompleted(false);
            return;
        }

		//***************************************//
		//*          MAIN ALGORITHM             *//
		//***************************************//
		
		// record time
		long start_time = System.currentTimeMillis();
		
		String line;
		
		String imgtag,reftag,noreftag,inttag,nointtag;
		imgtag = delim+srcImage.getImageName();
		reftag = delim+refImage.getImageName();
		noreftag = delim+"no_reference";
		inttag = delim+intImage.getImageName();
		nointtag = delim+"no_intensity";
		
		// standard computations for the object
		output = new ArrayList();
		
		// get the labels
		int Nlabels = ObjectProcessing.countLabels(image,nx,ny,nz);
		int[] imgLabels = ObjectProcessing.listOrderedLabels(image,nx,ny,nz);
			
		// remove the zero label
		if (imgLabels[0]==0) {
			int[] tmp = new int[Nlabels-1];
			for (int n=1;n<Nlabels;n++)
				tmp[n-1] = imgLabels[n];
			Nlabels--;
			imgLabels = tmp;
		}
		line = "Image_labels"+imgtag+noreftag+nointtag;
		for (int n=0;n<Nlabels;n++) line += (delim+imgLabels[n]);
		line +=("\n");
		output.add(line);
		
		// display the color mapping
		if (srcImage.isColorImage()) {
			line = "Image_color_1"+imgtag+noreftag+nointtag;
			for (int n=0;n<Nlabels;n++) line += (delim+imgMapping[n][0]);
			line +=("\n");
			output.add(line);
			line = "Image_color_2"+imgtag+noreftag+nointtag;
			for (int n=0;n<Nlabels;n++) line += (delim+imgMapping[n][1]);
			line +=("\n");
			output.add(line);
			line = "Image_color_3"+imgtag+noreftag+nointtag;
			for (int n=0;n<Nlabels;n++) line += (delim+imgMapping[n][2]);
			line +=("\n");
			output.add(line);
			line = "Image_color_4"+imgtag+noreftag+nointtag;
			for (int n=0;n<Nlabels;n++) line += (delim+imgMapping[n][3]);
			line +=("\n");
			output.add(line);
		}
		
		// build object list from VOIs
		if (useVOI) {
			// get list of VOI names
			int Nvoi;
			String[] voiName;
		
			// transfer each VOI into a mask
			boolean[][][][] voiMask;
			// TO DO
		}
		
		int Nreflb = 0;
		int[] refLabels = null;
		if (reference!=null) {
			Nreflb= ObjectProcessing.countLabels(reference,nx,ny,nz);
			refLabels = ObjectProcessing.listOrderedLabels(reference,nx,ny,nz);
			
			// remove the zero label
			if (refLabels[0]==0) {
				int[] trf = new int[Nreflb-1];
				for (int n=1;n<Nreflb;n++)
					trf[n-1] = refLabels[n];
				Nreflb--;
				refLabels = trf;
			}
			/* not needed ?
			line = "Reference_labels"+imgtag+reftag+nointtag;
			for (int n=0;n<Nreflb;n++) line = line+(delim+refLabels[n]);
			line +=("\n");
			output.add(line);
			*/
		}
		
		// compute the statistics
		for (int s=0; s<statistics.size(); s++) {
			if (statistics.get(s).equals("Voxels")) {
				// compute the volumes
				float[] volume = new float[Nlabels];
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[n],imgLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					volume[n] = ObjectProcessing.volume(obj,nx,ny,nz);
				}
				line = "Voxels"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels;n++) line += (delim+(int)volume[n]);
				line +="\n";
				output.add(line);
			}
			if (statistics.get(s).equals("Volume")) {
				// compute the volumes
				float[] volume = new float[Nlabels];
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[n],imgLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					volume[n] = ObjectProcessing.volume(obj,nx,ny,nz);
				}
				line = "Volume"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels;n++) line += (delim+volume[n]*rx*ry*rz);
				line +="\n";
				output.add(line);
			}
			if (statistics.get(s).equals("Surface")) {
				// compute the volumes
				float[] surface = new float[Nlabels];
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[n],imgLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					surface[n] = ObjectProcessing.boundaryArea(obj,nx,ny,nz,rx,ry,rz);
				}
				line = "Surface"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels;n++) line += (delim+surface[n]*rx*ry*rz);
				line +="\n";
				output.add(line);
			}
			
			if (statistics.get(s).equals("Euler_characteristic")) {
				// compute the Euler characteristics, number of parts and holes
				float[] euler = new float[Nlabels];
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[n],imgLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					euler[n] = ObjectProcessing.eulerCharacteristic(obj,nx,ny,nz,cObj,cBg);
				}
				
				line = "Euler_characteristic"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+(int)euler[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Object_parts")) {
				int[] parts = new int[Nlabels];
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[n],imgLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					parts[n] = ObjectProcessing.countParts(obj,nx,ny,nz,cObj,cBg);
				}
				line = "Object_parts"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+parts[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Object_cavities")) {
				int[] holes = new int[Nlabels];
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[n],imgLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					holes[n] = ObjectProcessing.countHoles(obj,nx,ny,nz,cObj,cBg);
				}
				line = "Object_cavities"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+holes[n]);
				line+=("\n");
				output.add(line);
			}
			
			if (statistics.get(s).equals("Surface_ripples")) {
				// compute the volumes
				float[] surface = new float[Nlabels];
				float[] opened = new float[Nlabels];
				float[] closed = new float[Nlabels];
				boolean[][][] mask = new boolean[3][3][3];
				for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
					if (i*i+j*j+l*l<=1) mask[1+i][1+j][1+l] = true;
					else mask[1+i][1+j][1+l] = false;
				}
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[n],imgLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					surface[n] = ObjectProcessing.boundaryArea(obj,nx,ny,nz,rx,ry,rz);
					boolean[][][] obj2 = Morphology.erodeObject(obj,nx,ny,nz,mask,1,1,1);
					obj2 = Morphology.dilateObject(obj2,nx,ny,nz,mask,1,1,1);
					opened[n] = ObjectProcessing.boundaryArea(obj2,nx,ny,nz,rx,ry,rz);
					obj2 = Morphology.dilateObject(obj,nx,ny,nz,mask,1,1,1);
					obj2 = Morphology.erodeObject(obj2,nx,ny,nz,mask,1,1,1);
					closed[n] = ObjectProcessing.boundaryArea(obj2,nx,ny,nz,rx,ry,rz);
				}
				line = "Surface_ripples"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels;n++) line += (delim+((2*surface[n]-opened[n]-closed[n])/(ZERO+surface[n])));
				line +="\n";
				output.add(line);
			}
			
			if (statistics.get(s).equals("Dice_overlap")) {
				// compare: requires same labels (use the reference as basis)
				float[] dice = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] xor = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.XOR,nx,ny,nz);
						float diff = ObjectProcessing.volume(xor,nx,ny,nz);
										
						dice[n] = ((vol1+vol2-diff)/(vol1+vol2));
					}
				}
				line = "Dice_overlap"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+dice[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Jaccard_overlap")) {
				// compare: requires same labels (use the reference as basis)
				float[] jaccard = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] xor = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.XOR,nx,ny,nz);
						float diff = ObjectProcessing.volume(xor,nx,ny,nz);
										
						jaccard[n] = ((vol1+vol2-diff)/(vol1+vol2+diff));
					}
				}
				line = "Jaccard_overlap"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+jaccard[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Mountford_overlap")) {
				// compare: requires same labels (use the reference as basis)
				float[] mountford = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] xor = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.XOR,nx,ny,nz);
						float diff = ObjectProcessing.volume(xor,nx,ny,nz);
										
						mountford[n] = (vol1+vol2-diff)/(2.0f*vol1*vol2 - (vol1+vol2)*(vol1+vol2-diff)/2.0f);
					}
				}
				line = "Mountford_overlap"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+mountford[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("True_positive_ratio")) {
				// compare: requires same labels (use the reference as basis)
				float[] truepos = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float inter = ObjectProcessing.volume(and,nx,ny,nz);
										
						truepos[n] = (inter/vol2);
					}
				}
				line = "True_positive_ratio"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+truepos[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("False_positive_ratio")) {
				// compare: requires same labels (use the reference as basis)
				float[] falsepos = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float inter = ObjectProcessing.volume(and,nx,ny,nz);
										
						falsepos[n] = ((vol1-inter)/vol2);
					}
				}
				line = "False_positive_ratio"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+falsepos[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("False_negative_ratio")) {
				// compare: requires same labels (use the reference as basis)
				float[] falseneg = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float inter = ObjectProcessing.volume(and,nx,ny,nz);
										
						falseneg[n] = ((vol2-inter)/vol2);
					}
				}
				line = "False_negative_ratio"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+falseneg[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("True_segmented_ratio")) {
				// compare: requires same labels (use the reference as basis)
				float[] trueseg = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float inter = ObjectProcessing.volume(and,nx,ny,nz);
										
						trueseg[n] = (inter/vol1);
					}
				}
				line = "True_segmented_ratio"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+trueseg[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("False_segmented_ratio")) {
				// compare: requires same labels (use the reference as basis)
				float[] falseseg = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float inter = ObjectProcessing.volume(and,nx,ny,nz);
										
						falseseg[n] = ((vol1-inter)/vol1);
					}
				}
				line = "False_segmented_ratio"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+falseseg[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Volume_difference")) {
				// compare: requires same labels (use the reference as basis)
				float[] voldiff = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
										
						voldiff[n] = Numerics.abs(vol1/vol2-1.0f);
					}
				}
				line = "Volume_difference"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+voldiff[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Containment_index(seg_in_ref)")) {
				// compare: requires same labels (use the reference as basis)
				float[] seginref = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float over = ObjectProcessing.volume(and,nx,ny,nz);
										
						seginref[n] = (over/vol1);
					}
				}
				line = "Containment_index(seg_in_ref)"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+seginref[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Containment_index(ref_in_seg)")) {
				// compare: requires same labels (use the reference as basis)
				float[] refinseg = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float over = ObjectProcessing.volume(and,nx,ny,nz);
										
						refinseg[n] = (over/vol2);
					}
				}
				line = "Containment_index(ref_in_seg)"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+refinseg[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Average_surface_distance")) {
				// compare: requires same labels (use the reference as basis)
				float[] dist = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						
						dist[n] = ObjectProcessing.averageSurfaceDistance(obj1,obj2,rx,ry,rz,nx,ny,nz);
						
						System.out.println("asd "+n+": "+dist[n]);
					}
				}
				line = "Average_surface_distance"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+dist[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Average_surface_difference")) {
				// compare: requires same labels (use the reference as basis)
				float[] dist = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						
						dist[n] = ObjectProcessing.averageSurfaceDifference(obj1,obj2,rx,ry,rz,nx,ny,nz);
						
						System.out.println("asd "+n+": "+dist[n]);
					}
				}
				line = "Average_surface_difference"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+dist[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Average_squared_surface_distance")) {
				// compare: requires same labels (use the reference as basis)
				float[] dist = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						
						dist[n] = ObjectProcessing.averageSquaredSurfaceDistance(obj1,obj2,rx,ry,rz,nx,ny,nz);
						
						System.out.println("assd "+n+": "+dist[n]);
					}
				}
				line = "Average_squared_surface_distance"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+dist[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Hausdorff_distance")) {
				// compare: requires same labels (use the reference as basis)
				float[] hausdorff = new float[Nreflb];
				for (int n=0;n<Nreflb;n++) {
					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[m],imgLabels[m],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,refLabels[n],refLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						
						hausdorff[n] = ObjectProcessing.hausdorffDistance(obj1,obj2,rx,ry,rz,nx,ny,nz);
						
						System.out.println("hd "+n+": "+hausdorff[n]);
					}
				}
				line = "Hausdorff_distance"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels;n++) line+=(delim+hausdorff[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Mean_intensity")) {
				float[] mean = new float[Nlabels];
				float[] den = new float[Nlabels];
				for (int n=0;n<Nlabels;n++) {
					mean[n] = 0.0f;
					den[n] = 0.0f;
				}
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					for (int n=0;n<Nlabels;n++) if (image[x][y][z]==imgLabels[n]) {
						mean[n] += intensity[x][y][z];
						den[n] += 1.0f;
					}
				}
				for (int n=0;n<Nlabels;n++) {
					if (den[n]>0) mean[n] = mean[n]/den[n];
				}
				line = "Mean_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels;n++) line+=(delim+mean[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Min_intensity")) {
				float[] min = new float[Nlabels];
				for (int n=0;n<Nlabels;n++) {
					min[n] = INF;
				}
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					for (int n=0;n<Nlabels;n++) if (image[x][y][z]==imgLabels[n]) {
						if (intensity[x][y][z]<min[n]) min[n] = intensity[x][y][z];
					}
				}
				line = "Min_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels;n++) line+=(delim+min[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Max_intensity")) {
				float[] max = new float[Nlabels];
				for (int n=0;n<Nlabels;n++) {
					max[n] = -INF;
				}
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					for (int n=0;n<Nlabels;n++) if (image[x][y][z]==imgLabels[n]) {
						if (intensity[x][y][z]>max[n]) max[n] = intensity[x][y][z];
					}
				}
				line = "Max_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels;n++) line+=(delim+max[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Stdev_intensity")) {
				float[] mean = new float[Nlabels];
				float[] std = new float[Nlabels];
				float[] den = new float[Nlabels];
				for (int n=0;n<Nlabels;n++) {
					mean[n] = 0.0f;
					std[n] = 0.0f;
					den[n] = 0.0f;
				}
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					for (int n=0;n<Nlabels;n++) if (image[x][y][z]==imgLabels[n]) {
						mean[n] += intensity[x][y][z];
						den[n] += 1.0f;
					}
				}
				for (int n=0;n<Nlabels;n++) {
					if (den[n]>0) mean[n] = mean[n]/den[n];
				}
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					for (int n=0;n<Nlabels;n++) if (image[x][y][z]==imgLabels[n]) {
						std[n] += (intensity[x][y][z]-mean[n])*(intensity[x][y][z]-mean[n]);
					}
				}
				for (int n=0;n<Nlabels;n++) {
					if (den[n]>0) std[n] = (float)Math.sqrt(std[n]/den[n]);
				}
				line = "Stdev_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels;n++) line+=(delim+std[n]);
				line+=("\n");
				output.add(line);
			}			
		}
		
		// write the output to standard window or file
		if (!writeToFile) {
			for (int n=0; n<output.size(); n++) {
				MedicUtilPublic.displayMessage(output.get(n));
			}
		} else {
			addStatisticsToFile(filename);
		}
		setCompleted(true);
    } // calcSegmentation
	
	public final ArrayList<String> getOutput() { return output; }
	
	private final void addStatisticsToFile(String name) {
		// check if the statistics file exists
		
		// open the file
		ArrayList<String> 	previous = loadStatisticsFile(filename);
		
		// merge the output
		appendStatistics(previous, output);
		
		// save the result
		writeStatisticsFile(previous, name);
	}
	
	private	final void appendStatistics(ArrayList<String> main, ArrayList<String> added) {
		for (int n=0;n<added.size();n++) {
			// extract statistics type
			String type = added.get(n).substring(0,added.get(n).indexOf(delim));
			System.out.println(added.get(n));
			System.out.println(type);
			
			// find the last line with this type
			int last=-1;
			for (int m=0;m<main.size();m++) {
				if (main.get(m).indexOf(delim)>-1)
					if (main.get(m).substring(0,main.get(m).indexOf(delim)).equals(type)) last = m;
			}
			if (last>-1) {
				main.add(last+1, added.get(n));
			} else {
				// add a space after each different statistic
				main.add(added.get(n));
				main.add(" \n");
			}
		}
	}
	
	private final ArrayList<String> loadStatisticsFile(String name) {
		ArrayList<String> list = new ArrayList();
		try {
            System.out.println("reading previous statistic file: "+name);
            File f = new File(filename);
            FileReader fr = new FileReader(f);
            BufferedReader br = new BufferedReader(fr);
            String line = br.readLine();
			
			// Exact corresponding template for first line ?
            if (!line.startsWith("MIPAV Volumetric Statistics File")) {
                System.out.println("not a proper MIPAV statistics file");
                br.close();
                fr.close();
                return null;
            }
			line = br.readLine();
			while (line!=null) {
				list.add(line+"\n");
				line = br.readLine();
				System.out.println(line);
			}
			br.close();
            fr.close();
        }
        catch (FileNotFoundException e) {
            System.out.println(e.getMessage());
        }
        catch (IOException e) {
            System.out.println(e.getMessage());
        } 
		catch (OutOfMemoryError e){
			System.out.println(e.getMessage());
		}
		catch (Exception e) {
			System.out.println(e.getMessage());
        }
		return list;	
	}

	private final void writeStatisticsFile(ArrayList<String> list, String name) {
		try {
            File f = new File(filename);
            FileWriter fw = new FileWriter(f);
            PrintWriter pw = new PrintWriter( fw );
			pw.write("MIPAV Volumetric Statistics File\n");
            for (int n=0;n<list.size();n++) {
				pw.write(list.get(n));
				System.out.print(list.get(n));
			}
			pw.close();
            fw.close();
       }
        catch (FileNotFoundException e) {
            System.out.println(e.getMessage());
        }
        catch (IOException e) {
            System.out.println(e.getMessage());
        } 
		catch (OutOfMemoryError e){
			System.out.println(e.getMessage());
		}
		catch (Exception e) {
			System.out.println(e.getMessage());
        }
	}
}		

