package edu.jhmi.rad.medic.methods;
 import java.io.*;
import java.util.*;
import gov.nih.mipav.view.*;

import gov.nih.mipav.model.structures.jama.*;

import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.structures.*;
import edu.jhmi.rad.medic.utilities.*;

/**
 *
 *  This algorithm segments objects while preserving
 *  their topology using a fast marching technique.
 *	<p>
 *	This is be the release version for the algorithm as
 *  of April 2006: works on regular Fantasm tissue classification,
 *	with optional registration and inhomogeneity correction.
 *	Usual parameters are fixed, and experimental options have been
 *	removed.
 *	
 *
 *	@version    Apr 2006
 *	@author     Pierre-Louis Bazin
 *		
 *
 */
 
public class MultipleToadSegmentation {
	
	// object types
	public static final byte	OBJECT		=	1;
	public static final byte	MASK		=	2;
	public static final byte	OUTLIER		=	3;
	public static final byte	LESION		=	4;
	public static final byte	OUTMASK		=	5;
	public static final byte	OPTIMIZED_CSF		=	6;
	public static final byte	OPTIMIZED_GM		=	7;
	public static final byte	OPTIMIZED_WM		=	8;
	public static final byte	RAYLEIGH		=	9;
			
	// numerical quantities
	private static final	float   INF			=	1e10f;
	private static final	float   ZERO		=	1e-10f;
	private	static final 	float	INVSQ2 		= 	1.0f/(float)Math.sqrt(2.0f);
	private	static final 	float	INVSQ3 		= 	1.0f/(float)Math.sqrt(3.0f);
	private	static final 	float	SQR2 		= 	(float)Math.sqrt(2.0f);
	private	static final 	float	SQR3 		= 	(float)Math.sqrt(3);
	private static final	float	SUM26 		= 	6.0f+12.0f*INVSQ2+8.0f*INVSQ3;
	private static final	float 	PI4 		= 	(float)Math.PI/4.0f;
	private static final	float 	LOGPI2MPI2 	= 	(float)Math.log(Math.PI/2.0f) - (float)Math.PI/2.0f;
		
	// invalid label flag, must be <0
	private	static	final	byte	EMPTY = -1;
			
	// data and membership buffers
	private 	float[][][][] 	images;  			// original images
	private 	float[][][][]   mems;   			// membership functions for each class
	private 	byte[][][]		classification;  	// hard classification
	private 	float[][][]   	ordering;   		// distance functions computed during the propagation
	private 	float[][] 		centroid;   		// mean intensity for each class
	private 	float[][] 		stddev;   			// stddev of intensity for each class
	private 	float[] 		varImg;   			// variance of intensity for each image
	private 	BinaryHeap4D  	tree;		   		// the binary tree used for the fast marching
	private static	int 		nx,ny,nz,nc;   		// images dimensions
	private static	int 	  	classes;			// total number of classes (i.e. except outliers)
	private		byte[]			templateLabel;		// intensity labels for the template images
	private		byte[]			objType;			// The type of object for each class
	private		float[]			volume;				// Count the volume of each object
	private		float[]			Ihigh;				// The min max of the centroid range
	private		float[]			Iscale;				// The min max of the centroid range
	private		float[]			lesionCentroid;		// the centroids for the lesion class
	private		float[]			lesionStddev;		// the centroids std dev for the lesion class
	private		float[][][]		lesionMembership;	// the membership for the lesion class
	private		float[][][]		outlierMembership;	// the membership for the outlier class
	private		byte[][][][]	boundaryDist;		// the outside distance to a boundary, for better relation weighting
	private 	float[][] 		priorCentroid;   	// prior intensity for each class
	
	private		CriticalPointLUT	lut;				// the LUT for critical points
	private		boolean				checkComposed;		// check if the objects are well-composed too (different LUTs)
	
	// parameters
	private 	float   		smoothing;   	// RFCM smoothing parameter
	private 	float			limit;			// The boundary for the skeleton
	private		float			volumeRatio;	// The limit for the volume loss in thinning
	private		float			maxDistance;	// The limit between regular and forced values, in the approx case
	private		float			outlier;		// The distance at which a value is considered an outlier
	
	// relations
	private 	short[][][]		relationMap;   	// labels for the relations between neighboring objects (intensity only, simple)
	//private 	boolean[][]		relationPair;   // list of paired relations between neighboring objects (topolgy)
	private		boolean[][]		relationList;		// the relations ordered by sets
	private 	byte[][][]		segmentation;   	// current classification for relations
	private		int				Nrelations;
	private		byte[][][] 	skeleton;
	private		float			lesion = 0.5f;
	private		byte[][][][]	closestLabel;
		
	// new stuff
	boolean[][][][]				available;		// to check wether the point can be added to the tree
	
	public static final int		RC = 0;
	public static final int		RX = 1;
	public static final int		RY = 2;
	public static final int		RZ = 3;
	
	// atlas prior
	private		float			priorCoefficient = 0.2f;	// the amount of the energy spent on priors 
	private		float			priorScale = 0.05f;			// the scale for the centroid difference in priors
	private		float			priorFactor = 1.0f;			// the factor for prior relationships
	private		MultipleToadDeformableAtlas	atlas;
	private		String[]		modality;
	private		float[][] 		similarity;
	private		float[][]		modweight;

	private		float[][]		varweight;
	private		float[][]		varprior;
	
	// variable metrics
	private		float[]	pcsf;	
	private		float[]	pgm;	
	private		float[]	pwm;	
	private		PowerTable[]	powercsf,powergm,powerwm;
	
	// centroid constraints
	private		String			centroidMode;
	private		float			centroidSmoothness;
	private		int[][]		centroidLink;
	
	// option flags
	private 	boolean 		useField;
	private 	float[][][][]	field;  			// inhomogeneity field
	
	private		boolean				useApprox = false;
	private		byte[][][][]		best;
	private		int					approx;
	
	private		float			psfRidge;
	//private		float			psfSpread;
	//private		float[]			psfWeight;
	//private		int				npsf;
	
	// computation flags
	private 	boolean 		isWorking;
	private 	boolean 		isCompleted;
	
	// for debug and display
	ViewJProgressBar            progressBar;
	private static final boolean		debug=true;
	private static final boolean		verbose=true;
	
    
	/**
	 *  constructors for different cases: with/out outliers, with/out selective constraints
	 */
	public MultipleToadSegmentation(float[][][][] images_, MultipleToadDeformableAtlas atlas_,
						int nx_, int ny_, int nz_, int nc_,
						float smooth_, float out_,
						float ridge_, float lastIncr_, float marchingLimit_, float volumeLimit_, 
						int steps_, float lesion_, 
						float priorCoeff_, float priorScale_, float priorFactor_,
						String[] imgModal_,
						String algoMode, String connectivityType,
						String centMode, float centSmo_,
						int approx_,
						ViewJProgressBar bar_) {
		images = images_;
		atlas = atlas_;
		classification = atlas.getTemplate();
		nx = nx_;
		ny = ny_;
		nz = nz_;
		nc = nc_;
		
		// object information
		classes = atlas.getNumber();
		
		objType = new byte[classes];
		for (int k=0;k<classes;k++) {
				 if (atlas.getTopology()[k].equals("obj"))   objType[k] = OBJECT;
			else if (atlas.getTopology()[k].equals("mask"))  objType[k] = MASK;
			else if (atlas.getTopology()[k].equals("out"))   objType[k] = OUTLIER;
			else if (atlas.getTopology()[k].equals("lesion"))  objType[k] = LESION;
			else if (atlas.getTopology()[k].equals("outmask"))   objType[k] = OUTMASK;
			else if (atlas.getTopology()[k].equals("optimized_csf"))   objType[k] = OPTIMIZED_CSF;
			else if (atlas.getTopology()[k].equals("optimized_gm"))   objType[k] = OPTIMIZED_GM;
			else if (atlas.getTopology()[k].equals("optimized_wm"))   objType[k] = OPTIMIZED_WM;
			else if (atlas.getTopology()[k].equals("rayleigh"))   objType[k] = RAYLEIGH;
			else objType[k] = OBJECT;
		}
		templateLabel = atlas.getLabels();
		
		modality = imgModal_;
		
		// coefficient (assumes normalized imagess)
		smoothing = smooth_/6.0f/(float)classes;
		outlier = out_;
		
		limit = marchingLimit_;
		volumeRatio = volumeLimit_;
		
		//psfNext = ridge_/(1.0f + ridge_)/27.0f;
		//psfMain = 1.0f/(1.0f + ridge_);
		psfRidge = ridge_;
		
		//psfSpread = lastIncr_;
		//npsf = Numerics.max(1,Numerics.ceil(3.0f*psfSpread));
		
		priorCoefficient = priorCoeff_/(float)classes;
		priorFactor = priorFactor_;
		priorScale = priorScale_;
		lesion = lesion_;
		
		centroidMode = centMode;
		centroidSmoothness = centSmo_;
		
		approx = approx_;
		if (approx<classes && approx>1) useApprox = true;
		else useApprox = false;
		
		progressBar = bar_;
		if (debug) for (int i=0;i<objType.length;i++) MedicUtilPublic.displayMessage(":"+objType[i]+":");
			
		useField = false;
		field = new float[nc][][][];
		for (int c=0;c<nc;c++) field[c] = null;
		
		// init all the arrays
		try {
			if (useApprox) {
				mems = new float[nx][ny][nz][approx];
				best = new byte[nx][ny][nz][approx];
			} else {
				mems = new float[nx][ny][nz][classes];
				best = null;
			}
			centroid = new float[nc][classes];
			stddev = new float[nc][classes+1];
			varImg = new float[nc];
			Ihigh = new float[nc];
			Iscale = new float[nc];
			lesionCentroid = new float[nc];
			lesionStddev = new float[nc];
			lesionMembership = new float[nx][ny][nz];
			outlierMembership = new float[nx][ny][nz];
			ordering = new float[nx][ny][nz];
			tree = new BinaryHeap4D(nx*ny+ny*nz+nz*nx, BinaryHeap4D.MAXTREE);
			checkComposed = false;
			if (connectivityType.equals("26/6")) lut = new CriticalPointLUT("critical266LUT.raw.gz",200);
			else if (connectivityType.equals("6/26")) lut = new CriticalPointLUT("critical626LUT.raw.gz",200);
			else if (connectivityType.equals("18/6")) lut = new CriticalPointLUT("critical186LUT.raw.gz",200);
			else if (connectivityType.equals("6/18")) lut = new CriticalPointLUT("critical618LUT.raw.gz",200);
			else if (connectivityType.equals("6/6")) lut = new CriticalPointLUT("critical66LUT.raw.gz",200);
			else if (connectivityType.equals("wcs")) {
				lut = new CriticalPointLUT("critical66LUT.raw.gz",200);
				checkComposed=true;
			}
			else if (connectivityType.equals("wco")) {
				lut = new CriticalPointLUT("criticalNoLUT.raw.gz",200);
				checkComposed=true;
			}
			else if (connectivityType.equals("no")) lut = new CriticalPointLUT("criticalNoLUT.raw.gz",200);
			else lut = new CriticalPointLUT("critical266LUT.raw.gz",200);
			if (!lut.loadCompressedPattern()) {
				isWorking = false;
				finalize();
				System.out.println("Problem loading the algorithm's LUT from: "+lut.getFilename());
				MedicUtilPublic.displayMessage("Problem loading the algorithm's LUT from: "+lut.getFilename()+"\n");
			} else {
				System.out.println("LUT loaded from: "+lut.getFilename());
			}
			relationMap = new short[nx][ny][nz];
			segmentation = new byte[nx][ny][nz]; 
			skeleton = new byte[nx][ny][nz];
			available = new boolean[nx][ny][nz][classes];
			volume = new float[classes];
			similarity = new float[classes][classes];
			modweight = atlas.getModalityWeights(modality,nc);
			boundaryDist = new byte[nx][ny][nz][classes];
			varprior = atlas.getIntensityVariancePriors(modality,nc);
			varweight = new float[nc][classes];
			closestLabel = new byte[nx][ny][nz][4]; 
			
			
			pcsf = new float[nc];
			pgm = new float[nc];
			pwm = new float[nc];
			for (int k=0;k<classes;k++) if (atlas.getTopology()[k].startsWith("optimized_")) {
				powercsf = new PowerTable[nc];
				powergm = new PowerTable[nc];
				powerwm = new PowerTable[nc];
				setOptimizedFactors(modality,nc);
			}
			
			centroidLink = atlas.getIntensityGroups(modality,nc);
			priorCentroid = new float[nc][classes];
			System.out.print(displayCentroidGroups());
			
			//psfWeight = new float[npsf*npsf];
			
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
		isWorking = true;
		
		// init values
		for (int k=0;k<classes;k++) volume[k] = 0;
		boolean isObj;
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			ordering[x][y][z] = 0.0f;
			segmentation[x][y][z] = EMPTY;
			skeleton[x][y][z] = EMPTY;
			relationMap[x][y][z] = EMPTY;	
			if (useApprox) for (int a=0;a<approx;a++) {
				mems[x][y][z][a] = 0.0f;
			}
			for (byte k=0;k<classes;k++) {
				if (!useApprox) mems[x][y][z][k] = 0.0f;
				if (classification[x][y][z] == templateLabel[k]) {
					volume[k]++;	
					segmentation[x][y][z] = k;
					skeleton[x][y][z] = k;
					relationMap[x][y][z] = k;
					for (int c=0;c<nc;c++) {
						centroid[c][k] += images[c][x][y][z];
					}
				}
				available[x][y][z][k] = false;
				boundaryDist[x][y][z][k] = 0;
			}
			if (segmentation[x][y][z]==EMPTY) {
				if (debug) System.out.println("error: empty regions in original template - aborting ("+x+", "+y+", "+z+"): "+classification[x][y][z]);
				else System.out.println("error: empty regions in original template - aborting computations\n");
				isWorking = false;
				return;
			}
			lesionMembership[x][y][z] = lesion;
			outlierMembership[x][y][z] = 0.5f;
		}
		for (int c=0;c<nc;c++) {
			for (int k=0;k<classes;k++) {
				centroid[c][k] = centroid[c][k]/volume[k];
				stddev[c][k] = 1.0f;
				varweight[c][k] = varprior[c][k];
			}
			varImg[c] = 1.0f;
			Ihigh[c] = 1.0f;
			Iscale[c] = 1.0f;
			lesionCentroid[c] = 0.0f;
		}
		for (int k=0;k<classes;k++) for (int m=0;m<classes;m++) {
			if (k==m) similarity[k][m] = 0.0f;
			else similarity[k][m] = 1.0f;
		}
		/*
		for (int d=0;d<npsf*npsf;d++) {
			psfWeight[d] = (float)Math.exp(-d/(2.0f*psfSpread*psfSpread));
		}
		float norm = 0.0f;
		for (int i=-npsf+1;i<npsf;i++) for (int j=-npsf+1;j<npsf;j++) for (int l=-npsf+1;l<npsf;l++) {
			int d = i*i+j*j+l*l;
			if (d<npsf*npsf) {
				norm += psfWeight[d];
			}
		}
		for (int d=0;d<npsf*npsf;d++) psfWeight[d] /= norm;
		*/
		if (debug) MedicUtilPublic.displayMessage("initialization\n");
	}
		
	public void finalize() {
		images = null;
		mems = null;
		centroid = null;
		tree = null;
		lut = null;
	}
	
	/**
	 *	clean up the computation arrays
	 */
	public final void cleanUp() {
        images = null;
		tree.finalize();
		tree = null;
		lut.finalize();
		lut = null;
		System.gc();
	}
    

	public final float[] getCentroids(int c) { return centroid[c]; }
	public final float[][] getCentroids() { return centroid; }
	public final void initCentroids(float[][] cent) { 
		for (int  c=0;c<nc;c++) {
			for (int k=0;k<classes;k++) {
				centroid[c][k] = cent[c][k];
				priorCentroid[c][k] = cent[c][k];
			}
		}
	}
		
	public final float[][][][] getMemberships() { return mems; }
	public final void setMemberships(float[][][][] m_) { mems = m_; }

	public final byte[][][][] getBest() { return best; }

	public final void setOrdering(float[][][] o_) { ordering = o_; }
	public final byte[][][] getClassification() { return classification; }
	public final byte[][][] getSegmentation() { return segmentation; }

	public final void setIntensityMax(float[] I) { Ihigh = I; }
	public final void setIntensityScale(float[] I) { Iscale = I; }
	
	public final void setLesionCentroid(float[] cent) { lesionCentroid = cent; }
	
	/*
	public final void setPriorCoefficients(float pC, float pF, float pS) { 
		priorCoefficient = pC;
		priorFactor = pF;
		priorScale = pS;
		if (debug) System.out.println("priors: "+priorCoefficient+", "+priorFactor+", "+priorScale);
	}
	*/
	
	final public void setOptimizedFactors(String[] modality, int nc) {
		System.out.println("SET OPTIMIZED PARAMETERS");
		for (int n=0;n<nc;n++) {
			pcsf[n] = atlas.getOptimizedFactor(modality[n])[0];
			pgm[n]  = atlas.getOptimizedFactor(modality[n])[1];
			pwm[n]  = atlas.getOptimizedFactor(modality[n])[2];
			System.out.println(n+" CSF "+pcsf[n]+" GM "+pgm[n]+" WM "+pwm[n]);
			powercsf[n] = new PowerTable(0.0f , 4.0f , 0.000001f , pcsf[n] );
			powergm[n] = new PowerTable(0.0f , 4.0f , 0.000001f , pgm[n] );
			powerwm[n] = new PowerTable(0.0f , 4.0f , 0.000001f , pwm[n] );

		}
		return;
	}
	
	public final void resetClassification() { 
		classification = atlas.getTemplate();
	}

	/** add inhomogeneity correction */
    public final void addInhomogeneityCorrection(float[][][] field_, int c) {
        field[c] = field_;
        useField = true;
    }
    
	public final boolean isWorking() { return isWorking; }
	public final boolean isCompleted() { return isCompleted; }
	
	private final boolean isNotEmpty() {
		return tree.isNotEmpty();
	}
	
	/**
	 *  initialize the labels for a new iteration
	 */
	public final void initGrowingLabels() {
		float   order;
		float	mem;
		
		// fix for homeomorphic version
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			if (skeleton[x][y][z]==EMPTY) {
				for (int n=0;n<classes;n++) {
					available[x][y][z][n] = true;
				}
			} else {
				for (int n=0;n<classes;n++) {
					available[x][y][z][n] = false;
				}
			}
		}

		// init the labels from the skeleton
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			for (byte k=0;k<classes;k++) {
				if (available[x][y][z][k]) {
					if ( (skeleton[x-1][y][z]==k) || (skeleton[x+1][y][z]==k)
					  || (skeleton[x][y-1][z]==k) || (skeleton[x][y+1][z]==k)
					  || (skeleton[x][y][z-1]==k) || (skeleton[x][y][z+1]==k) ) {
					  
					  	// new boundary: add to the sorting tree
						available[x][y][z][k]=false;

						if (useApprox) order = ordering[x][y][z] + maxApproxMembershipRatio(x,y,z,k);
						else order = ordering[x][y][z] + maxMembershipRatio(x,y,z,k,bestMembership(x,y,z));
						tree.addValue(order,x,y,z,k);
					}
				}
			}
		}
	}//initGlobalGrowingLabels
		
	/**
	 *  initialize the labels for a new iteration
	 */
	public final void initThinningLabels() {
		boolean isBoundary = false;
		float   order=0.0f;
		
		//reset the volume here before first (k==0)
		for (int k=0;k<classes;k++) 
				volume[k] = 0;
		
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			skeleton[x][y][z] = segmentation[x][y][z];
				
			if (useApprox) {
				for (int a=0;a<approx;a++) {
					if (segmentation[x][y][z]==best[x][y][z][a]) {
						volume[best[x][y][z][a]]+=mems[x][y][z][a];
					}
				}
				for (int k=0;k<classes;k++) {
					// available only inside k
					if (segmentation[x][y][z]==k) {
						available[x][y][z][k] = true;
					} else {
						available[x][y][z][k] = false;
					}
				}
			} else {
				for (int k=0;k<classes;k++) {
					// available only inside k
					if (segmentation[x][y][z]==k) {
						volume[k]+=mems[x][y][z][k];
						available[x][y][z][k] = true;
					} else {
						available[x][y][z][k] = false;
					}
				}
			}
		}
		//if (debug) System.out.println("volume("+k+")="+volume[k]);
				
		int[] lb = new int[6];
		float[] score = new float[6];
		int Nlb,best=0;
		boolean found;
		int[] Nbound = new int[classes];
		for (int k=0;k<classes;k++) Nbound[k] = 0;
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			for (int k=0;k<classes;k++) {
				if (segmentation[x][y][z]==k) {
					// neighbors outside the object ?
					isBoundary = false;
					Nlb=0;
					int Nngb=0;
					int ni=0,nj=0,nl=0;
					for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
						if (i*i+j*j+l*l==1 && segmentation[x+i][y+j][z+l]!=k) {
							isBoundary = true;
						} else if (i*i+j*j+l*l>0) {
							Nngb++;
							ni=x+i;nj=y+j;nl=z+l;
						}
					}
					
					if (isBoundary) {
						// no search for best change yet
						available[x][y][z][k] = false;
						
						// check if it is a line endpoint
						boolean endpoint=false;
						if (Nngb==1) {
							Nngb=0;
							for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) if (i*i+j*j+l*l>0) {
								if (segmentation[ni+i][nj+j][nl+l]==k) Nngb++;
							}
							if (Nngb<=2) endpoint=true;
						}
						
						// best to use the membership of the object to be removed
						//order = - highMembershipScore(x,y,z,k);
						if (endpoint) order = - 0.0f;
						else {
							if (useApprox) order = - 1.0f/maxApproxMembershipRatio(x,y,z,k);
							else order = - 1.0f/maxMembershipRatio(x,y,z,k,bestMembership(x,y,z));
						}
						tree.addValue(order,x,y,z,k);
						Nbound[k]++;
					}
				}
			}
		}
		/*
		for (int k=0;k<classes;k++) {
			if (debug) System.out.print("new thinning "+k+": "+Nbound[k]+" boundary points\n");
			if (debug) System.out.print("volume "+k+": "+volume[k]+"\n");
		}
		*/
	}//initGlobalThinningLabels
		
	/** 
	 *  compute the FCM membership functions given the centroids
	 *	with the different options (outliers, field, edges, MRF)
	 *	with q=2
	 */
    final public float computeMemberships() {
		if (useApprox) return computeApproxMemberships();
		else return computeExactMemberships();
	}
    final public float computeExactMemberships() {
        float dist, dist0;
        float den,num;
        float numL,numO;
        float neighbors, ngb;
		float smooth;
		float mean,count;
		boolean	used;
		float[] shape = new float[classes];
		
		byte relationType;
		byte DIRECT = 2;
		byte NEIGHBOR = 1;
		byte FARAWAY = 0;
		byte PRIOR = 3;
		byte DISTANCE = 4;
		
		float[] energy = new float[classes];
        
		float[] psfBias = new float[nc];
        for (int c=0;c<nc;c++) psfBias[c] = 0.0f;
				
		// update the variance 
		computeImageVariance();
		if (debug) System.out.print(displayImageVariance());
				
		// compute the centroid similarity coefficient and scaling factor
		float centroidDist, lesionDist;
		for (int k=0;k<classes;k++) {
			for (int m=0;m<classes;m++) if (m!=k) {
				if ( (objType[k]==LESION) && (objType[m]==LESION) ) centroidDist = 0.0f;						 
				else {
					centroidDist = 0;
					for (int c=0;c<nc;c++) centroidDist += varImg[c]*(centroid[c][k]-centroid[c][m])*(centroid[c][k]-centroid[c][m])
															 			/(stddev[c][k]+stddev[c][m])/(stddev[c][k]+stddev[c][m]);
					
					if (objType[k]==LESION) {
						lesionDist = 0;
						for (int c=0;c<nc;c++) lesionDist += varImg[c]*(lesionCentroid[c]-centroid[c][m])*(lesionCentroid[c]-centroid[c][m])
																 		/(lesionStddev[c]+stddev[c][m])/(lesionStddev[c]+stddev[c][m]);
					} else if (objType[m]==LESION) {
						lesionDist = 0;
						for (int c=0;c<nc;c++) lesionDist += varImg[c]*(lesionCentroid[c]-centroid[c][k])*(lesionCentroid[c]-centroid[c][k])
																 		/(lesionStddev[c]+stddev[c][k])/(lesionStddev[c]+stddev[c][k]);
					} 
																 
				}
													 
				similarity[k][m] = 1.0f/(1.0f + (1-priorScale)/priorScale*centroidDist);

				/* debug 
				if (atlas.getShapePriors()[k]*similarity[k][m]>0.1f)
					if (debug) System.out.println("similarity ["+k+"->"+m+"]: "+similarity[k][m]+", distance: "+centroidDist);
				*/
			} else {
				similarity[k][m] = 0.0f;
			}
		}
		
		// main loop
		dist = 0.0f;
		atlas.precomputeTransformMatrix(1.0f);
		for (short x=1;x<nx-1;x++) for (short y=1;y<ny-1;y++) for (short z=1;z<nz-1;z++) {
			den = 0;
			// precompute the priors
			if (priorCoefficient > 0) {
				shape = atlas.getTransformedShape(x,y,z);
			}
			// precompute the psf bias
			if (psfRidge>0) {
				if (isBoundary(x,y,z)) {
					for (int c=0;c<nc;c++) {
						if (useField) psfBias[c] = -(27.0f/26.0f)*field[c][x][y][z]*images[c][x][y][z];
						else psfBias[c] = -(27.0f/26.0f)*images[c][x][y][z];
						for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
							if (useField) psfBias[c] += (1.0f/26.0f)*field[c][x+i][y+j][z+l]*images[c][x+i][y+j][z+l];
							else psfBias[c] += (1.0f/26.0f)*images[c][x+i][y+j][z+l];
						}
						psfBias[c] *= psfRidge;
					}
				} else {
					for (int c=0;c<nc;c++) psfBias[c] = 0.0f;
				}
			}
			for (int k=0;k<classes;k++) {
				/* replace
				if (relationList[relationMap[x][y][z]][k]) relationType = DIRECT;
				else relationType = PRIOR;
				//else relationType = DISTANCE;
				with: */
				if (closestLabel[x][y][z][0]==k || closestLabel[x][y][z][1]==k) {
					relationType = DIRECT;
				} else if (closestLabel[x][y][z][2]==k || closestLabel[x][y][z][3]==k) {
					relationType = NEIGHBOR;
				} else {
					relationType = PRIOR;
				}
				
				if (relationType==PRIOR && shape[k]==0) { 
					mems[x][y][z][k] = 0.0f;
				} else {
					// data term
					num = 0.0f;
					numL = 0.0f;
					numO = 0.0f;
					if (objType[k]==OBJECT) {
						if (useField) {
							for (int c=0;c<nc;c++)
								num += modweight[c][0]*varweight[c][k]*varImg[c]*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k]);
						} else {
							for (int c=0;c<nc;c++)
								num += modweight[c][0]*varweight[c][k]*varImg[c]*(images[c][x][y][z]-psfBias[c]-centroid[c][k])*(images[c][x][y][z]-psfBias[c]-centroid[c][k]);
						}
					} else if (objType[k]==MASK) {
						for (int c=0;c<nc;c++)
							num += modweight[c][0]*varweight[c][k]*varImg[c]*maskDistance(images[c][x][y][z],0.0f);
					} else if (objType[k]==OUTLIER) {
						if (useField) {
							for (int c=0;c<nc;c++) {
								num  += modweight[c][0]*varweight[c][k]*varImg[c]*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k]);
								numO += modweight[c][0]*varweight[c][k]*varImg[c]*outlier;
							}
						} else {
							for (int c=0;c<nc;c++) {
								num  += modweight[c][0]*varweight[c][k]*varImg[c]*(images[c][x][y][z]-psfBias[c]-centroid[c][k])*(images[c][x][y][z]-psfBias[c]-centroid[c][k]);
								numO += modweight[c][0]*varweight[c][k]*varImg[c]*outlier;
							}
						}
					} else if (objType[k]==OUTMASK) {
						for (int c=0;c<nc;c++) {
							num  += modweight[c][0]*varweight[c][k]*varImg[c]*maskDistance(images[c][x][y][z],0.0f);
							numO += modweight[c][0]*varweight[c][k]*varImg[c]*outlier;
						}
					} else if (objType[k]==LESION) {
						if (useField) {
							for (int c=0;c<nc;c++) {
								num += modweight[c][0]*varweight[c][k]*varImg[c]*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k]);
								if (segmentation[x][y][z]==k) numL += modweight[c][1]*varweight[c][k]*varImg[c]*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-lesionCentroid[c])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-lesionCentroid[c]);
							}
						} else {
							for (int c=0;c<nc;c++) {
								num += modweight[c][0]*varweight[c][k]*varImg[c]*(images[c][x][y][z]-psfBias[c]-centroid[c][k])*(images[c][x][y][z]-psfBias[c]-centroid[c][k]);
								if (segmentation[x][y][z]==k) numL += modweight[c][1]*varweight[c][k]*varImg[c]*(images[c][x][y][z]-psfBias[c]-lesionCentroid[c])*(images[c][x][y][z]-psfBias[c]-lesionCentroid[c]);
							}
						}
					} else if (objType[k]==RAYLEIGH) {
						if (useField) {
							for (int c=0;c<nc;c++)
								num += modweight[c][0]*varweight[c][k]*varImg[c]*rayleighDistance(field[c][x][y][z]*images[c][x][y][z]-psfBias[c],centroid[c][k]);
						} else {
							for (int c=0;c<nc;c++)
								num += modweight[c][0]*varweight[c][k]*varImg[c]*rayleighDistance(images[c][x][y][z]-psfBias[c],centroid[c][k]);
						}
					} else if (objType[k]==OPTIMIZED_CSF) {
						if (useField) {
							for (int c=0;c<nc;c++)
								num += modweight[c][0]*varweight[c][k]*varImg[c]*powercsf[c].lookup( (field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k]), pcsf[c]);
						} else {
							for (int c=0;c<nc;c++)
								num += modweight[c][0]*varweight[c][k]*varImg[c]*powercsf[c].lookup( (images[c][x][y][z]-psfBias[c]-centroid[c][k])*(images[c][x][y][z]-psfBias[c]-centroid[c][k]), pcsf[c]);
						}
					} else if (objType[k]==OPTIMIZED_GM) {
						if (useField) {
							for (int c=0;c<nc;c++)
								num += modweight[c][0]*varweight[c][k]*varImg[c]*powergm[c].lookup( (field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k]), pgm[c]);
						} else {
							for (int c=0;c<nc;c++)
								num += modweight[c][0]*varweight[c][k]*varImg[c]*powergm[c].lookup( (images[c][x][y][z]-psfBias[c]-centroid[c][k])*(images[c][x][y][z]-psfBias[c]-centroid[c][k]), pgm[c]);
						}
					} else if (objType[k]==OPTIMIZED_WM) {
						if (useField) {
							for (int c=0;c<nc;c++)
								num += modweight[c][0]*varweight[c][k]*varImg[c]*powerwm[c].lookup( (field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k]), pwm[c]);
						} else {
							for (int c=0;c<nc;c++)
								num += modweight[c][0]*varweight[c][k]*varImg[c]*powerwm[c].lookup( (images[c][x][y][z]-psfBias[c]-centroid[c][k])*(images[c][x][y][z]-psfBias[c]-centroid[c][k]), pwm[c]);
						}
					} else {
						num = outlier;
					}
					
					// energy term
					if (objType[k]==LESION) {
						if (segmentation[x][y][z]==k) {
							energy[k] = (1.0f-lesionMembership[x][y][z])*num + lesionMembership[x][y][z]*numL;
						} else {
							energy[k] = num;
						}
					} else if (objType[k]==OUTLIER) {
						energy[k] = (1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO;
					} else if (objType[k]==OUTMASK) {
						energy[k] = (1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO;
					} else {
						energy[k] = num;
					}
				
				
					// spatial smoothing
					if (smoothing > 0.0f) { 
						ngb = 0.0f;  
						// case by case	: X+
						for (int m=0;m<classes;m++) if (m!=k) {
							ngb += mems[x+1][y][z][m]*mems[x+1][y][z][m];
						}
						// case by case	: X-
						for (int m=0;m<classes;m++) if (m!=k) {
							ngb += mems[x-1][y][z][m]*mems[x-1][y][z][m];
						}
						// case by case	: Y+
						for (int m=0;m<classes;m++) if (m!=k) {
							ngb += mems[x][y+1][z][m]*mems[x][y+1][z][m];
						}
						// case by case	: Y-
						for (int m=0;m<classes;m++) if (m!=k) {
							ngb += mems[x][y-1][z][m]*mems[x][y-1][z][m];
						}
						// case by case	: Z+
						for (int m=0;m<classes;m++) if (m!=k) {
							ngb += mems[x][y][z+1][m]*mems[x][y][z+1][m];
						}
						// case by case	: Z-
						for (int m=0;m<classes;m++) if (m!=k) {
							ngb += mems[x][y][z-1][m]*mems[x][y][z-1][m];
						}
						num += atlas.getSmoothingPriors()[k]*smoothing*ngb;
						if (objType[k]==LESION) if (segmentation[x][y][z]==k) numL += atlas.getSmoothingPriors()[k]*smoothing*ngb;
						else if (objType[k]==OUTLIER) numO += atlas.getSmoothingPriors()[k]*smoothing*ngb;
						else if (objType[k]==OUTMASK) numO += atlas.getSmoothingPriors()[k]*smoothing*ngb;
						energy[k] += 0.5f*atlas.getSmoothingPriors()[k]*smoothing*ngb;
					}
					
					if (priorCoefficient > 0) {
						float invprior=0;
						for (int m=0;m<classes;m++) if (m!=k) {
							invprior += atlas.getShapePriors()[m]*similarity[k][m]*shape[m]*shape[m];
						}
						num += priorCoefficient*invprior;
						if (objType[k]==LESION) { if (segmentation[x][y][z]==k) numL += priorCoefficient*invprior; }
						else if (objType[k]==OUTLIER) numO += priorCoefficient*invprior;
						else if (objType[k]==OUTMASK) numO += priorCoefficient*invprior;
						energy[k] += 0.5f*priorCoefficient*invprior;
					}

					/* not needed
					// spatial smoothing : encourage neighbors
					ngb = 0.0f;
					for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
						ngb += mems[x+i][y+j][z+l][k]*mems[x+i][y+j][z+l][k];
					}
					float ngbfactor = 1.0f - ngb/26.0f;
					
					//num = ngbfactor*num;
					*/
					
					// invert the result
					if (num>ZERO) num = 1.0f/num;
					else num = INF;
					if ( (objType[k]==LESION) && (segmentation[x][y][z]==k) ) {
						if (numL>ZERO) numL = 1.0f/numL;
						else numL = INF;
					} else if (objType[k]==OUTLIER) {
						if (numO>ZERO) numO = 1.0f/numO;
						else numO = INF;
					} else if (objType[k]==OUTMASK) {
						if (numO>ZERO) numO = 1.0f/numO;
						else numO = INF;
					}
					
					if (objType[k]==LESION) {
						if (relationType==DIRECT) {
							if (segmentation[x][y][z]==k) {
								mems[x][y][z][k] = (1.0f-lesionMembership[x][y][z])*num + lesionMembership[x][y][z]*numL;
								lesionMembership[x][y][z] = lesionMembership[x][y][z]*numL;
							} else {
								mems[x][y][z][k] = num;
								lesionMembership[x][y][z] = 0.0f;
							}
						} else if (relationType==NEIGHBOR) {
							if (segmentation[x][y][z]==k) {
								mems[x][y][z][k] = neighborFactor(shape[k])*((1.0f-lesionMembership[x][y][z])*num + lesionMembership[x][y][z]*numL);
								lesionMembership[x][y][z] = neighborFactor(shape[k])*lesionMembership[x][y][z]*numL;
							} else {
								mems[x][y][z][k] = num;
								lesionMembership[x][y][z] = 0.0f;
							}
						} else if (relationType==PRIOR) {
							if (segmentation[x][y][z]==k) {
								mems[x][y][z][k] = shapeFactor(shape[k])*((1.0f-lesionMembership[x][y][z])*num + lesionMembership[x][y][z]*numL);
								lesionMembership[x][y][z] = shapeFactor(shape[k])*lesionMembership[x][y][z]*numL;
							} else {
								mems[x][y][z][k] = shapeFactor(shape[k])*num;
								lesionMembership[x][y][z] = 0.0f;
							}
						} else if (relationType==DISTANCE) {
							if (segmentation[x][y][z]==k) {
								mems[x][y][z][k] = distanceFactor(x,y,z,k)*((1.0f-lesionMembership[x][y][z])*num + lesionMembership[x][y][z]*numL);
								lesionMembership[x][y][z] = distanceFactor(x,y,z,k)*lesionMembership[x][y][z]*numL;
							} else {
								mems[x][y][z][k] = distanceFactor(x,y,z,k)*num;
								lesionMembership[x][y][z] = 0.0f;
							}
						}
					} else if (objType[k]==OUTLIER) {
						if (relationType==DIRECT) {
							mems[x][y][z][k] = (1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO;
							outlierMembership[x][y][z] = outlierMembership[x][y][z]*numO;
						} else if (relationType==NEIGHBOR) {
							mems[x][y][z][k] = neighborFactor(shape[k])*((1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO);
							outlierMembership[x][y][z] = neighborFactor(shape[k])*outlierMembership[x][y][z]*numO;
						} else if (relationType==PRIOR) {
							mems[x][y][z][k] = shapeFactor(shape[k])*((1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO);
							outlierMembership[x][y][z] = shapeFactor(shape[k])*outlierMembership[x][y][z]*numO;
						} else if (relationType==DISTANCE) {
							mems[x][y][z][k] = distanceFactor(x,y,z,k)*((1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO);
							outlierMembership[x][y][z] = distanceFactor(x,y,z,k)*outlierMembership[x][y][z]*numO;
						}
					} else if (objType[k]==OUTMASK) {
						if (relationType==DIRECT) {
							mems[x][y][z][k] = (1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO;
							outlierMembership[x][y][z] = outlierMembership[x][y][z]*numO;
						} else if (relationType==NEIGHBOR) {
							mems[x][y][z][k] = neighborFactor(shape[k])*((1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO);
							outlierMembership[x][y][z] = neighborFactor(shape[k])*outlierMembership[x][y][z]*numO;
						} else if (relationType==PRIOR) {
							mems[x][y][z][k] = shapeFactor(shape[k])*((1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO);
							outlierMembership[x][y][z] = shapeFactor(shape[k])*outlierMembership[x][y][z]*numO;
						} else if (relationType==DISTANCE) {
							mems[x][y][z][k] = distanceFactor(x,y,z,k)*((1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO);
							outlierMembership[x][y][z] = distanceFactor(x,y,z,k)*outlierMembership[x][y][z]*numO;
						}
					} else {
						if (relationType==DIRECT) {
							mems[x][y][z][k] = num;
						} else if (relationType==NEIGHBOR) {
							mems[x][y][z][k] = neighborFactor(shape[k])*num;
						} else if (relationType==PRIOR) {
							mems[x][y][z][k] = shapeFactor(shape[k])*num;
						} else if (relationType==DISTANCE) {
							mems[x][y][z][k] = distanceFactor(x,y,z,k)*num;
						}
					} 
					den += mems[x][y][z][k];
				}
			}
			
			// normalization
			if (den>0.0f) {
				for (int k=0;k<classes;k++) {
					mems[x][y][z][k] = mems[x][y][z][k]/den;
					if ((objType[k]==LESION) && (segmentation[x][y][z]==k)) {
						lesionMembership[x][y][z] = lesionMembership[x][y][z]/den;
					} else if (objType[k]==OUTLIER) {
						outlierMembership[x][y][z] = outlierMembership[x][y][z]/den;
					} else if (objType[k]==OUTMASK) {
						outlierMembership[x][y][z] = outlierMembership[x][y][z]/den;
					}
					// compute energy
					dist += energy[k]*mems[x][y][z][k]*mems[x][y][z][k];
				}
			}
		}
       return dist;
    }// computeMembershipsWithPrior
	
    final public float computeApproxMemberships() {
        float dist, dist0;
        float den,num;
        float numL,numO;
        float neighbors, ngb;
		float smooth;
		float mean,count;
		boolean	used;
		float[] shape = new float[classes];
		
		byte relationType;
		byte DIRECT = 2;
		byte NEIGHBOR = 1;
		byte FARAWAY = 0;
		byte PRIOR = 3;
		byte DISTANCE = 4;
		
		float[] energy = new float[classes];
        
		float[] psfBias = new float[nc];
        for (int c=0;c<nc;c++) psfBias[c] = 0.0f;
				
		// update the variance 
		computeImageVariance();
		if (debug) System.out.print(displayImageVariance());
				
		// compute the centroid similarity coefficient and scaling factor
		float centroidDist, lesionDist;
		for (int k=0;k<classes;k++) {
			for (int m=0;m<classes;m++) if (m!=k) {
				if ( (objType[k]==LESION) && (objType[m]==LESION) ) centroidDist = 0.0f;						 
				else {
					centroidDist = 0;
					for (int c=0;c<nc;c++) centroidDist += varImg[c]*(centroid[c][k]-centroid[c][m])*(centroid[c][k]-centroid[c][m])
															 			/(stddev[c][k]+stddev[c][m])/(stddev[c][k]+stddev[c][m]);
					
					if (objType[k]==LESION) {
						lesionDist = 0;
						for (int c=0;c<nc;c++) lesionDist += varImg[c]*(lesionCentroid[c]-centroid[c][m])*(lesionCentroid[c]-centroid[c][m])
																 		/(lesionStddev[c]+stddev[c][m])/(lesionStddev[c]+stddev[c][m]);
					} else if (objType[m]==LESION) {
						lesionDist = 0;
						for (int c=0;c<nc;c++) lesionDist += varImg[c]*(lesionCentroid[c]-centroid[c][k])*(lesionCentroid[c]-centroid[c][k])
																 		/(lesionStddev[c]+stddev[c][k])/(lesionStddev[c]+stddev[c][k]);
					} 
																 
				}
													 
				similarity[k][m] = 1.0f/(1.0f + (1-priorScale)/priorScale*centroidDist);

				/* debug 
				if (atlas.getShapePriors()[k]*similarity[k][m]>0.1f)
					if (debug) System.out.println("similarity ["+k+"->"+m+"]: "+similarity[k][m]+", distance: "+centroidDist);
				*/
			} else {
				similarity[k][m] = 0.0f;
			}
		}
		
		// centroid range for distances ?
		//computeCentroidRange();
		
		
		// main loop
		dist = 0.0f;
		atlas.precomputeTransformMatrix(1.0f);
		for (short x=1;x<nx-1;x++) for (short y=1;y<ny-1;y++) for (short z=1;z<nz-1;z++) {
			den = 0;
			// precompute the priors
			if (priorCoefficient > 0) {
				shape = atlas.getTransformedShape(x,y,z);
			}
			// precompute the psf bias
			if (psfRidge>0) {
				if (isBoundary(x,y,z)) {
					for (int c=0;c<nc;c++) {
						if (useField) psfBias[c] = -(27.0f/26.0f)*field[c][x][y][z]*images[c][x][y][z];
						else psfBias[c] = -(27.0f/26.0f)*images[c][x][y][z];
						for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
							if (useField) psfBias[c] += (1.0f/26.0f)*field[c][x+i][y+j][z+l]*images[c][x+i][y+j][z+l];
							else psfBias[c] += (1.0f/26.0f)*images[c][x+i][y+j][z+l];
						}
						psfBias[c] *= psfRidge;
					}
				} else {
					for (int c=0;c<nc;c++) psfBias[c] = 0.0f;
				}
			}
			
			float[] mem = new float[classes];
			
			for (int k=0;k<classes;k++) {
				
				if (relationList[relationMap[x][y][z]][k]) relationType = DIRECT;
				//else  relationType = DISTANCE;
				else  relationType = PRIOR;
				
				if (relationType!=DIRECT && shape[k]==0) {
					mem[k] = 0.0f;
				} else {
					// data term
					num = 0.0f;
					numL = 0.0f;
					numO = 0.0f;
					if (objType[k]==OBJECT) {
						if (useField) {
							for (int c=0;c<nc;c++) 
								num += modweight[c][0]*varweight[c][k]*varImg[c]*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k]);
						} else {
							for (int c=0;c<nc;c++) 
								num += modweight[c][0]*varweight[c][k]*varImg[c]*(images[c][x][y][z]-psfBias[c]-centroid[c][k])*(images[c][x][y][z]-psfBias[c]-centroid[c][k]);
						}
					} else if (objType[k]==MASK) {
						for (int c=0;c<nc;c++)
							num += modweight[c][0]*varweight[c][k]*varImg[c]*maskDistance(images[c][x][y][z],0.0f);
					} else if (objType[k]==OUTLIER) {
						if (useField) {
							for (int c=0;c<nc;c++) {
								num  += modweight[c][0]*varweight[c][k]*varImg[c]*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k]);
								numO += modweight[c][0]*varweight[c][k]*varImg[c]*outlier;
							}
						} else {
							for (int c=0;c<nc;c++) {
								num  += modweight[c][0]*varweight[c][k]*varImg[c]*(images[c][x][y][z]-psfBias[c]-centroid[c][k])*(images[c][x][y][z]-psfBias[c]-centroid[c][k]);
								numO += modweight[c][0]*varweight[c][k]*varImg[c]*outlier;
							}
						}
					} else if (objType[k]==OUTMASK) {
						for (int c=0;c<nc;c++) {
							num  += modweight[c][0]*varweight[c][k]*varImg[c]*maskDistance(images[c][x][y][z],0.0f);
							numO += modweight[c][0]*varweight[c][k]*varImg[c]*outlier;
						}
					} else if (objType[k]==LESION) {
						if (useField) {
							for (int c=0;c<nc;c++) {
								num += modweight[c][0]*varweight[c][k]*varImg[c]*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k]);
								if (segmentation[x][y][z]==k) numL += modweight[c][1]*varweight[c][k]*varImg[c]*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-lesionCentroid[c])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-lesionCentroid[c]);
							}
						} else {
							for (int c=0;c<nc;c++) {
								num += modweight[c][0]*varweight[c][k]*varImg[c]*(images[c][x][y][z]-psfBias[c]-centroid[c][k])*(images[c][x][y][z]-psfBias[c]-centroid[c][k]);
								if (segmentation[x][y][z]==k) numL += modweight[c][1]*varweight[c][k]*varImg[c]*(images[c][x][y][z]-psfBias[c]-lesionCentroid[c])*(images[c][x][y][z]-psfBias[c]-lesionCentroid[c]);
							}
						}
					} else if (objType[k]==RAYLEIGH) {
							if (useField) {
								for (int c=0;c<nc;c++)
									num += modweight[c][0]*varweight[c][k]*varImg[c]*rayleighDistance(field[c][x][y][z]*images[c][x][y][z]-psfBias[c],centroid[c][k]);
							} else {
								for (int c=0;c<nc;c++)
									num += modweight[c][0]*varweight[c][k]*varImg[c]*rayleighDistance(images[c][x][y][z]-psfBias[c],centroid[c][k]);
							}
					} else if (objType[k]==OPTIMIZED_CSF) {
						if (useField) {
							for (int c=0;c<nc;c++) {
								num += modweight[c][0]*varweight[c][k]*varImg[c]*powercsf[c].lookup( (field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k]), pcsf[c]);
							}
						} else {
							for (int c=0;c<nc;c++) {
								num += modweight[c][0]*varweight[c][k]*varImg[c]*powercsf[c].lookup( (images[c][x][y][z]-psfBias[c]-centroid[c][k])*(images[c][x][y][z]-psfBias[c]-centroid[c][k]), pcsf[c]);
							}
						}
					} else if (objType[k]==OPTIMIZED_GM) {
						if (useField) {
							for (int c=0;c<nc;c++) {
								num += modweight[c][0]*varweight[c][k]*varImg[c]*powergm[c].lookup( (field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k]), pgm[c]);
							}
						} else {
							for (int c=0;c<nc;c++) {
								num += modweight[c][0]*varweight[c][k]*varImg[c]*powergm[c].lookup( (images[c][x][y][z]-psfBias[c]-centroid[c][k])*(images[c][x][y][z]-psfBias[c]-centroid[c][k]), pgm[c]);
							}
						}
					} else if (objType[k]==OPTIMIZED_WM) {
						if (useField) {
							for (int c=0;c<nc;c++) {
								num += modweight[c][0]*varweight[c][k]*varImg[c]*powerwm[c].lookup( (field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-psfBias[c]-centroid[c][k]), pwm[c]);
							}
						} else {
							for (int c=0;c<nc;c++) {
								num += modweight[c][0]*varweight[c][k]*varImg[c]*powerwm[c].lookup( (images[c][x][y][z]-psfBias[c]-centroid[c][k])*(images[c][x][y][z]-psfBias[c]-centroid[c][k]), pwm[c]);
							}
						}
					} else {
						num = outlier;
					}
					
					if (objType[k]==LESION) {
						if (segmentation[x][y][z]==k) {
							energy[k] = (1.0f-lesionMembership[x][y][z])*num + lesionMembership[x][y][z]*numL;
						} else {
							energy[k] = num;
						}
					} else if (objType[k]==OUTLIER) {
						energy[k] = (1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO;
					} else if (objType[k]==OUTMASK) {
						energy[k] = (1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO;
					} else {
						energy[k] = num;
					}
				
					// spatial smoothing
					if (smoothing > 0.0f) { 
						ngb = 0.0f;  
						// case by case	: X+
						for (int m=0;m<approx;m++) if (best[x+1][y][z][m]!=k) {
							ngb += mems[x+1][y][z][m]*mems[x+1][y][z][m];
						}
						// case by case	: X-
						for (int m=0;m<approx;m++) if (best[x-1][y][z][m]!=k) {
							ngb += mems[x-1][y][z][m]*mems[x-1][y][z][m];
						}
						// case by case	: Y+
						for (int m=0;m<approx;m++) if (best[x][y+1][z][m]!=k) {
							ngb += mems[x][y+1][z][m]*mems[x][y+1][z][m];
						}
						// case by case	: Y-
						for (int m=0;m<approx;m++) if (best[x][y-1][z][m]!=k) {
							ngb += mems[x][y-1][z][m]*mems[x][y-1][z][m];
						}
						// case by case	: Z+
						for (int m=0;m<approx;m++) if (best[x][y][z+1][m]!=k) {
							ngb += mems[x][y][z+1][m]*mems[x][y][z+1][m];
						}
						// case by case	: Z-
						for (int m=0;m<approx;m++) if (best[x][y][z-1][m]!=k) {
							ngb += mems[x][y][z-1][m]*mems[x][y][z-1][m];
						}
						num += atlas.getSmoothingPriors()[k]*smoothing*ngb;
						if (objType[k]==LESION) { if (segmentation[x][y][z]==k) numL += atlas.getSmoothingPriors()[k]*smoothing*ngb; }
						else if (objType[k]==OUTLIER) numO += atlas.getSmoothingPriors()[k]*smoothing*ngb;
						else if (objType[k]==OUTMASK) numO += atlas.getSmoothingPriors()[k]*smoothing*ngb;
						energy[k] += 0.5f*atlas.getSmoothingPriors()[k]*smoothing*ngb;
					}
					
					if (priorCoefficient > 0) {
						float invprior=0;
						for (int m=0;m<classes;m++) if (m!=k) {
							invprior += atlas.getShapePriors()[m]*similarity[k][m]*shape[m]*shape[m];
						}
						num += priorCoefficient*invprior;
						if (objType[k]==LESION) { if (segmentation[x][y][z]==k) numL += priorCoefficient*invprior; }
						else if (objType[k]==OUTLIER) numO += priorCoefficient*invprior;
						else if (objType[k]==OUTMASK) numO += priorCoefficient*invprior;
						energy[k] += 0.5f*priorCoefficient*invprior;
					}
					
					// invert the result
					if (num>ZERO) num = 1.0f/num;
					else num = INF;
					if ( (objType[k]==LESION) && (segmentation[x][y][z]==k) ) {
						if (numL>ZERO) numL = 1.0f/numL;
						else numL = INF;
					} else if (objType[k]==OUTLIER) {
						if (numO>ZERO) numO = 1.0f/numO;
						else numO = INF;
					} else if (objType[k]==OUTMASK) {
						if (numO>ZERO) numO = 1.0f/numO;
						else numO = INF;
					}
					
					if (objType[k]==LESION) {
						if (relationType==DIRECT) {
							if (segmentation[x][y][z]==k) {
								mem[k] = (1.0f-lesionMembership[x][y][z])*num + lesionMembership[x][y][z]*numL;
								lesionMembership[x][y][z] = lesionMembership[x][y][z]*numL;
							} else {
								mem[k] = num;
								lesionMembership[x][y][z] = 0.0f;
							}
						} else if (relationType==PRIOR) {
							if (segmentation[x][y][z]==k) {
								mem[k] = shapeFactor(shape[k])*((1.0f-lesionMembership[x][y][z])*num + lesionMembership[x][y][z]*numL);
								lesionMembership[x][y][z] = shapeFactor(shape[k])*lesionMembership[x][y][z]*numL;
							} else {
								mem[k] = shapeFactor(shape[k])*num;
								lesionMembership[x][y][z] = 0.0f;
							}
						} else if (relationType==DISTANCE) {
							if (segmentation[x][y][z]==k) {
								mem[k] = distanceFactor(x,y,z,k)*((1.0f-lesionMembership[x][y][z])*num + lesionMembership[x][y][z]*numL);
								lesionMembership[x][y][z] = distanceFactor(x,y,z,k)*lesionMembership[x][y][z]*numL;
							} else {
								mem[k] = shapeFactor(shape[k])*num;
								lesionMembership[x][y][z] = 0.0f;
							}
						}
					} else if (objType[k]==OUTLIER) {
						if (relationType==DIRECT) {
							mem[k] = (1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO;
							outlierMembership[x][y][z] = outlierMembership[x][y][z]*numO;
						} else if (relationType==PRIOR) {
							mem[k] = shapeFactor(shape[k])*((1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO);
							outlierMembership[x][y][z] = shapeFactor(shape[k])*outlierMembership[x][y][z]*numO;
						} else if (relationType==DISTANCE) {
							mem[k] = distanceFactor(x,y,z,k)*((1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO);
							outlierMembership[x][y][z] = distanceFactor(x,y,z,k)*outlierMembership[x][y][z]*numO;
						}
					} else if (objType[k]==OUTMASK) {
						if (relationType==DIRECT) {
							mem[k] = (1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO;
							outlierMembership[x][y][z] = outlierMembership[x][y][z]*numO;
						} else if (relationType==PRIOR) {
							mem[k] = shapeFactor(shape[k])*((1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO);
							outlierMembership[x][y][z] = shapeFactor(shape[k])*outlierMembership[x][y][z]*numO;
						} else if (relationType==DISTANCE) {
							mem[k] = distanceFactor(x,y,z,k)*((1.0f-outlierMembership[x][y][z])*num + outlierMembership[x][y][z]*numO);
							outlierMembership[x][y][z] = distanceFactor(x,y,z,k)*outlierMembership[x][y][z]*numO;
						}
					} else {
						if (relationType==DIRECT) {
							mem[k] = num;
						} else if (relationType==PRIOR) {
							mem[k] = shapeFactor(shape[k])*num;
						} else if (relationType==DISTANCE) {
							mem[k] = distanceFactor(x,y,z,k)*num;
						}
					}
					den += mem[k];
				}
			}
			
			// select the best
			Numerics.bestIndex(best[x][y][z],mems[x][y][z],mem,approx);

			// normalization
			if (den>0.0f) {
				for (int a=0;a<approx;a++) {
					int k = best[x][y][z][a];
					mems[x][y][z][a] = mems[x][y][z][a]/den;
					if ((objType[k]==LESION) && (segmentation[x][y][z]==k)) {
						lesionMembership[x][y][z] = lesionMembership[x][y][z]/den;
					} else if (objType[k]==OUTLIER) {
						outlierMembership[x][y][z] = outlierMembership[x][y][z]/den;
					} else if (objType[k]==OUTMASK) {
						outlierMembership[x][y][z] = outlierMembership[x][y][z]/den;
					}
					// compute energy
					dist += energy[k]*mems[x][y][z][a]*mems[x][y][z][a];
				}
			} else {
				//System.err.print("/0:"+den);
			}
		}
        return dist;
    }// computeMembershipsWithPrior

	final public void propagateExactMemberships() {
		
		float[][][] tmp = new float[nx][ny][nz];
		
		for (int t=0;t<10;t++) {
			for (int k=0;k<classes;k++) {
				for (short x=1;x<nx-1;x++) for (short y=1;y<ny-1;y++) for (short z=1;z<nz-1;z++) {
					tmp[x][y][z] = mems[x][y][z][k];
					for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
						float sim = (float)Math.exp( 0.5f*(images[0][x][y][z]-images[0][x+i][y+j][z+l])*(images[0][x][y][z]-images[0][x+i][y+j][z+l])/stddev[0][k] )/26.0f;
						tmp[x][y][z] += 0.5f*sim*mems[x+i][y+j][z+l][k];
					}
					tmp[x][y][z] = Numerics.min(tmp[x][y][z],1.0f);
				}
				for (short x=1;x<nx-1;x++) for (short y=1;y<ny-1;y++) for (short z=1;z<nz-1;z++) {
					mems[x][y][z][k] = tmp[x][y][z];
				}
			}
		}
		return;
    }// computeMembershipsWithPrior
	

	private final float distanceFactor(int x, int y, int z, int k) {
		return 1.0f/(1.0f+boundaryDist[x][y][z][k]/limit);
	}
	
	private final float shapeFactor(float sh) {
		return Numerics.min(1.0f, 2.0f*priorFactor*sh);
	}
	
	private final float neighborFactor(float sh) {
		return Numerics.bounded(2.0f*priorFactor*sh, 0.25f, 1.0f);
	}
	
	/** pre-compute the smoothing and the relations */ 
	public final void computeBoundaryDistances() {
		byte[][][] 		tmp = new byte[nx][ny][nz];

		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			// find boundaries
			for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
				if (i*i+j*j+l*l==1) {
					if (segmentation[x][y][z]!=segmentation[x+i][y+j][z+l]) {
						boundaryDist[x+i][y+j][z+l][segmentation[x][y][z]] = 1;
						boundaryDist[x][y][z][segmentation[x+i][y+j][z+l]] = 1;
					}
				}
			}		
		}
		
		// diffuse the coefficients
		boolean change=true;
		for (int t=1;t<5*limit && change;t++) {
			for (int k=0;k<classes;k++) {
				change = false;
				for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
					tmp[x][y][z] = boundaryDist[x][y][z][k];
					if (segmentation[x][y][z]!=k && boundaryDist[x][y][z][k]==0) {
						for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
							if (i*i+j*j+l*l==1) {
								if (boundaryDist[x+i][y+j][z+l][k]>0) {
									tmp[x][y][z] = (byte)Numerics.min(tmp[x][y][z],boundaryDist[x+i][y+j][z+l][k]+1);
								}
							}
						}
					}
				}
				for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
					boundaryDist[x][y][z][k] = tmp[x][y][z];
				}
			}
		}
		return;
	}
	
	/** pre-compute the smoothing and the relations */ 
	public final void computeRelations() {
		short[][][] 			tmp = new short[nx][ny][nz];
		float				num,den;
		String info;
		float factor;
	
		// start with the proper boundaries
		buildRelationList();
		
		// diffuse the coefficients
		boolean change=true;
		for (int t=1;t<10*limit && change;t++) { 
			change = false;
			for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
				if (relationMap[x][y][z]<classes) {
					change = true;
					float bestMem = 0.0f;
					tmp[x][y][z] = relationMap[x][y][z];	
					for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) if (i*i+j*j+l*l==1) {
						if (relationMap[x+i][y+j][z+l]>=classes) {
							// choose relation with higest joint memberships ?
							float memSum = 0.0f;
							if (useApprox) {
								for (int a=0;a<approx;a++) {
									int k = best[x+i][y+j][z+l][a];
									if (relationList[relationMap[x+i][y+j][z+l]][k])
										memSum += mems[x+i][y+j][z+l][a];
								}
							} else {
								for (int k=0;k<classes;k++) 
									if (relationList[relationMap[x+i][y+j][z+l]][k])
										memSum += mems[x+i][y+j][z+l][k];	
							}
							if (memSum>bestMem) {
								bestMem = memSum;
								tmp[x][y][z] = relationMap[x+i][y+j][z+l];
							}
							/* this would require some pretty complicated relation handling, but it might be worth it
							/* TO DO: make the relationList more flexible (easy addition? map all conbinations?)
							/* +fast recognition of mappings
							/* MGDM would be really useful here!!! -> MGDM init for up to N labels => easier handling of relations too
							// it is better to add new relations ? unfortunately yes
							boolean[] rel = new boolean[classes];
							for (int k=0;k<classes;k++) rel[k] = false;
							if (useApprox) {
								for (int a=0;a<approx;a++) {
									int k = best[x+i][y+j][z+l][a];
									if (relationList[relationMap[x+i][y+j][z+l]][k])
										rel[k] = true;
								}
							} else {
								for (int k=0;k<classes;k++) 
									if (relationList[relationMap[x+i][y+j][z+l]][k])
										rel[k] = true;	
							}
							// search among existing relations : should be an exact find (because of topology)
							for (int n=classes;n<Nrelations;n++) {
								boolean found = true;
								for (int k=0;k<classes;k++) if (rel[k]!=relationList[n][k]) found = false;
								if (found) {
									tmp[x][y][z] = (short)n;
									n = Nrelations;
								}
							}
							*/
						}
					}
				} else tmp[x][y][z] = relationMap[x][y][z];
			}
			for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
				relationMap[x][y][z] = tmp[x][y][z];
			}
		}
		return;
	}				
	
	/** pre-compute the smoothing and the relations */ 
	public final void computeClosestLabels() {
		// to remove
		byte[][][][] tmpLabel = new byte[nx][ny][nz][4];
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			closestLabel[x][y][z][0] = segmentation[x][y][z];
			closestLabel[x][y][z][1] = EMPTY;
			closestLabel[x][y][z][2] = EMPTY;
			closestLabel[x][y][z][3] = EMPTY;
			for (int n=0;n<4;n++) tmpLabel[x][y][z][n] = closestLabel[x][y][z][n];
		}
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) if (i*i+j*j+l*l==1) {
				if (segmentation[x+i][y+j][z+l]!=segmentation[x][y][z]) {
					if (closestLabel[x][y][z][1]==EMPTY) closestLabel[x][y][z][1] = segmentation[x+i][y+j][z+l];
					else if (closestLabel[x][y][z][2]==EMPTY && closestLabel[x][y][z][1]!=segmentation[x+i][y+j][z+l])
						closestLabel[x][y][z][2] = segmentation[x+i][y+j][z+l];
					else if (closestLabel[x][y][z][3]==EMPTY && closestLabel[x][y][z][1]!=segmentation[x+i][y+j][z+l]
						&& closestLabel[x][y][z][2]!=segmentation[x+i][y+j][z+l]) 
						closestLabel[x][y][z][3] = segmentation[x+i][y+j][z+l];
				}
			}
			for (int n=0;n<4;n++) tmpLabel[x][y][z][n] = closestLabel[x][y][z][n];
		}
		// diffuse the labels
		for (int t=1;t<5*limit;t++) { 
			for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
				if (tmpLabel[x][y][z][1]==EMPTY) {
					for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) if (i*i+j*j+l*l==1) {
						if (closestLabel[x+i][y+j][z+l][1]!=EMPTY && tmpLabel[x][y][z][0]!=closestLabel[x+i][y+j][z+l][1])
							tmpLabel[x][y][z][1] = closestLabel[x+i][y+j][z+l][1];
					}
				} else if (tmpLabel[x][y][z][2]==EMPTY) {
					for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) if (i*i+j*j+l*l==1) {
						if (closestLabel[x+i][y+j][z+l][1]!=EMPTY && tmpLabel[x][y][z][0]!=closestLabel[x+i][y+j][z+l][1]
							&& tmpLabel[x][y][z][1]!=closestLabel[x+i][y+j][z+l][1])
							tmpLabel[x][y][z][2] = closestLabel[x+i][y+j][z+l][1];
						else if (closestLabel[x+i][y+j][z+l][2]!=EMPTY && tmpLabel[x][y][z][0]!=closestLabel[x+i][y+j][z+l][2]
							&& tmpLabel[x][y][z][1]!=closestLabel[x+i][y+j][z+l][2])
							tmpLabel[x][y][z][2] = closestLabel[x+i][y+j][z+l][2];
					}
				} else if (tmpLabel[x][y][z][3]==EMPTY) {
					for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) if (i*i+j*j+l*l==1) {
						if (closestLabel[x+i][y+j][z+l][1]!=EMPTY && tmpLabel[x][y][z][0]!=closestLabel[x+i][y+j][z+l][1]
							&& tmpLabel[x][y][z][1]!=closestLabel[x+i][y+j][z+l][1] && tmpLabel[x][y][z][2]!=closestLabel[x+i][y+j][z+l][1])
							tmpLabel[x][y][z][3] = closestLabel[x+i][y+j][z+l][1];
						else if (closestLabel[x+i][y+j][z+l][2]!=EMPTY && tmpLabel[x][y][z][0]!=closestLabel[x+i][y+j][z+l][2]
							&& tmpLabel[x][y][z][1]!=closestLabel[x+i][y+j][z+l][2] && tmpLabel[x][y][z][2]!=closestLabel[x+i][y+j][z+l][2])
							tmpLabel[x][y][z][3] = closestLabel[x+i][y+j][z+l][2];
						else if (closestLabel[x+i][y+j][z+l][3]!=EMPTY && tmpLabel[x][y][z][0]!=closestLabel[x+i][y+j][z+l][3]
							&& tmpLabel[x][y][z][1]!=closestLabel[x+i][y+j][z+l][3] && tmpLabel[x][y][z][2]!=closestLabel[x+i][y+j][z+l][3])
							tmpLabel[x][y][z][3] = closestLabel[x+i][y+j][z+l][3];
					}
				}
			}
			for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) for (int n=0;n<4;n++) {
				closestLabel[x][y][z][n] = tmpLabel[x][y][z][n];
			}
		}
		return;
	}				
	
	/**
	 *  distances and other functions for computing memberships
	 */
	private final float maskDistance(float img, float mean) {
		if (img==mean) return 0.0f;
		else return 1.0f;
	}
	
	
	private final float rayleighDistance(float val, float sig) {
		if (val<=0) return 1.0f;
		else return val*val/sig*sig - 2.0f*(float)Math.log(val/sig) + LOGPI2MPI2;
	}
			
	/**
	 *  critical relation detection: groups objects with relations
	 */
    private final boolean isHomeomorphicSegmentation(short x, short y, short z, short k) {
		// is the new object regular ? not the growing object, the original one!!
		
		// inside the original object ?
		if (segmentation[x][y][z]==k) return true;
		
		boolean [][][] obj = new boolean[3][3][3];
		
		// does it change the topology of the new object ?
		for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
			if (segmentation[x+i][y+j][z+l]==k) {
				obj[1+i][1+j][1+l] = true;
			} else {
				obj[1+i][1+j][1+l] = false;
			}
		}
		obj[1][1][1] = true;
		
		// check well-composedness for new object(s)
		if (checkComposed) if (!ObjectProcessing.isWellComposed(obj,1,1,1)) return false;
		
		if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
			
		// does it change the topology of the object it modifies ?
		for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
			if (segmentation[x+i][y+j][z+l]==segmentation[x][y][z]) {
				obj[1+i][1+j][1+l] = true;
			} else {
				obj[1+i][1+j][1+l] = false;
			}
		}
		obj[1][1][1] = false;
		
		if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;

		// does it change the topology of a relation between the modified object and its neighbors ?
		int  Nconfiguration = 0;
		short[] lb = new short[26];
		for (short i=-1;i<=1;i++) for (short j=-1;j<=1;j++) for (short l=-1;l<=1;l++) {
			if ( (i*i+j*j+l*l>0) 
				&& (segmentation[x+i][y+j][z+l]!=k) 
				&& (segmentation[x+i][y+j][z+l]!=segmentation[x][y][z]) ) {
				boolean found = false;
				for (int n=0;n<Nconfiguration;n++) 
					if (segmentation[x+i][y+j][z+l]==lb[n]) { found = true; break; }
				
				if (!found) {
					lb[Nconfiguration] = segmentation[x+i][y+j][z+l];
					Nconfiguration++;
				}
			}
		}
		// pairs

		for (int n=0;n<Nconfiguration;n++) {
			// in relation with previous object
			for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
				if ( (segmentation[x+i][y+j][z+l]==segmentation[x][y][z])
					|| (segmentation[x+i][y+j][z+l]==lb[n]) ) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}
			obj[1][1][1] = false;
			if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
		}
		for (int n=0;n<Nconfiguration;n++) {
			// in relation with new object
			for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
				if ( (segmentation[x+i][y+j][z+l]==k)
					|| (segmentation[x+i][y+j][z+l]==lb[n]) ) {
					obj[1+i][1+j][1+l] = true;
				} else {
					obj[1+i][1+j][1+l] = false;
				}
			}
			obj[1][1][1] = true;
			if (checkComposed) if (!ObjectProcessing.isWellComposed(obj,1,1,1)) return false;
			if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
		}

		// triplets
		for (int n=0;n<Nconfiguration;n++) {
			for (int m=n+1;m<Nconfiguration;m++) {
				// in relation with previous object
				for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
					if ( (segmentation[x+i][y+j][z+l]==segmentation[x][y][z])
						|| (segmentation[x+i][y+j][z+l]==lb[n])
						|| (segmentation[x+i][y+j][z+l]==lb[m]) ) {
						obj[1+i][1+j][1+l] = true;
					} else {
						obj[1+i][1+j][1+l] = false;
					}
				}
				obj[1][1][1] = false;
				if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
			}
		}
		for (int n=0;n<Nconfiguration;n++) {
			for (int m=n+1;m<Nconfiguration;m++) {
				// in relation with new object
				for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
					if ( (segmentation[x+i][y+j][z+l]==k)
						|| (segmentation[x+i][y+j][z+l]==lb[n]) 
						|| (segmentation[x+i][y+j][z+l]==lb[m]) ) {
						obj[1+i][1+j][1+l] = true;
					} else {
						obj[1+i][1+j][1+l] = false;
					}
				}
				obj[1][1][1] = true;
				if (checkComposed) if (!ObjectProcessing.isWellComposed(obj,1,1,1)) return false;
				if (!lut.get(lut.keyFromPattern(obj,1,1,1))) return false;
			}
		}

		// else, it works
		return true;
    }
	
	/**
	 *  critical relation detection: groups objects with relations
	 */
    private final boolean isWellComposed(boolean[][][] obj) {
		// 18-C
		if (obj[0][0][1] && !(obj[0][1][1] || obj[1][0][1]) ) return false;
		if (obj[1][0][0] && !(obj[1][0][1] || obj[1][1][0]) ) return false;
		if (obj[0][1][0] && !(obj[1][1][0] || obj[0][1][1]) ) return false;
		
		if (obj[0][2][1] && !(obj[0][1][1] || obj[1][2][1]) ) return false;
		if (obj[1][0][2] && !(obj[1][0][1] || obj[1][1][2]) ) return false;
		if (obj[2][1][0] && !(obj[1][1][0] || obj[2][1][1]) ) return false;
		
		if (obj[2][0][1] && !(obj[2][1][1] || obj[1][0][1]) ) return false;
		if (obj[1][2][0] && !(obj[1][2][1] || obj[1][1][0]) ) return false;
		if (obj[0][1][2] && !(obj[1][1][2] || obj[0][1][1]) ) return false;
		
		if (obj[2][2][1] && !(obj[2][1][1] || obj[1][2][1]) ) return false;
		if (obj[1][2][2] && !(obj[1][2][1] || obj[1][1][2]) ) return false;
		if (obj[2][1][2] && !(obj[1][1][2] || obj[2][1][1]) ) return false;
		
		// 26-C
		if (obj[0][0][0] && !( (obj[0][1][1] && obj[0][0][1]) || (obj[0][1][1] && obj[0][1][0]) 
							|| (obj[1][0][1] && obj[1][0][0]) || (obj[1][0][1] && obj[0][0][1]) 
							|| (obj[1][1][0] && obj[0][1][0]) || (obj[1][1][0] && obj[1][0][0]) ) ) return false;
		
		if (obj[0][0][2] && !( (obj[0][1][1] && obj[0][0][1]) || (obj[0][1][1] && obj[0][1][2]) 
							|| (obj[1][0][1] && obj[1][0][2]) || (obj[1][0][1] && obj[0][0][1]) 
							|| (obj[1][1][2] && obj[0][1][2]) || (obj[1][1][2] && obj[1][0][2]) ) ) return false;
		
		if (obj[0][2][0] && !( (obj[0][1][1] && obj[0][2][1]) || (obj[0][1][1] && obj[0][1][0]) 
							|| (obj[1][2][1] && obj[1][2][0]) || (obj[1][2][1] && obj[0][2][1]) 
							|| (obj[1][1][0] && obj[0][1][0]) || (obj[1][1][0] && obj[1][2][0]) ) ) return false;
		
		if (obj[2][0][0] && !( (obj[2][1][1] && obj[2][0][1]) || (obj[2][1][1] && obj[2][1][0]) 
							|| (obj[1][0][1] && obj[1][0][0]) || (obj[1][0][1] && obj[2][0][1]) 
							|| (obj[1][1][0] && obj[2][1][0]) || (obj[1][1][0] && obj[1][0][0]) ) ) return false;
		
		if (obj[0][2][2] && !( (obj[0][1][1] && obj[0][2][1]) || (obj[0][1][1] && obj[0][1][2]) 
							|| (obj[1][2][1] && obj[1][2][2]) || (obj[1][2][1] && obj[0][2][1]) 
							|| (obj[1][1][2] && obj[0][1][2]) || (obj[1][1][2] && obj[1][2][2]) ) ) return false;
		
		if (obj[2][0][2] && !( (obj[2][1][1] && obj[2][0][1]) || (obj[2][1][1] && obj[2][1][2]) 
							|| (obj[1][0][1] && obj[1][0][2]) || (obj[1][0][1] && obj[2][0][1]) 
							|| (obj[1][1][2] && obj[2][1][2]) || (obj[1][1][2] && obj[1][0][2]) ) ) return false;
		
		if (obj[2][2][0] && !( (obj[2][1][1] && obj[2][2][1]) || (obj[2][1][1] && obj[2][1][0]) 
							|| (obj[1][2][1] && obj[1][2][0]) || (obj[1][2][1] && obj[2][2][1]) 
							|| (obj[1][1][0] && obj[2][1][0]) || (obj[1][1][0] && obj[1][2][0]) ) ) return false;
		
		if (obj[2][2][2] && !( (obj[2][1][1] && obj[2][2][1]) || (obj[2][1][1] && obj[2][1][2]) 
							|| (obj[1][2][1] && obj[1][2][2]) || (obj[1][2][1] && obj[2][2][1]) 
							|| (obj[1][1][2] && obj[2][1][2]) || (obj[1][1][2] && obj[1][2][2]) ) ) return false;
		
		return true;
	}
	
	/**
	 *  critical relation detection: groups objects with relations
	 */
    private final boolean isBoundary(short x, short y, short z) {
		for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
			if (segmentation[x+i][y+j][z+l]!=segmentation[x][y][z]) {
				return true;
			}
		}
		return false;
    }
	
	/** global score ratio for object k */
	private final float maxMembershipRatio(int x, int y, int z, int k, int b) {
		if (k==b) return 1.0f;
		else {	
			return (mems[x][y][z][b]+ZERO)/(mems[x][y][z][k]+ZERO);
		}
	}
	
	private final byte bestMembership(int x, int y, int z) {
		if (useApprox) return best[x][y][z][0];
		else {
			byte id = 0;
			float max = 0.0f;
			
			for (byte m=0;m<classes;m++) if (mems[x][y][z][m] > max) {
				id = m;
				max = mems[x][y][z][m];
			}
			return id;
		}
	}
		
	/** global score ratio for object k */
	private final float maxApproxMembershipRatio(int x, int y, int z, int k) {
		if (k==best[x][y][z][0]) return 1.0f;
		else {
			float cur = 0.0f;
			for (int a=1;a<approx;a++) 
				if (best[x][y][z][a]==k)
					cur = mems[x][y][z][a];
				
			return (mems[x][y][z][0]+ZERO)/(cur+ZERO);
		}
	}
	
	/**
	 *  propagate the skeleton out toward the boundaries
	 *  (geometric progression)
	 */
	public final void propagateGrowing() {	
		float   	val,order=0,factor;
		int 		xi,yj,zl;
		short 		x,y,z,k;
		boolean		isRegular;
		int			d2;
		
		// init: reset the boundary tree, the labels
		tree.reset();
		tree.setMinTree();
		
		initGrowingLabels();
		
		// loop on the boundary
		int nprint=0;
		while ( isNotEmpty()) {
			
			// get the next value
			x = tree.getFirstX();
			y = tree.getFirstY();
			z = tree.getFirstZ();
			k = tree.getFirstK();
			val = tree.getFirst();
			tree.removeFirst();
			
			if (skeleton[x][y][z]==EMPTY) {
				// test for update
				isRegular = isHomeomorphicSegmentation(x,y,z,k);
				
				if (isRegular) {
					// regular point : set the distance
					skeleton[x][y][z] = (byte)k;
					for (int m=0;m<classes;m++) {
						available[x][y][z][m] = false;
					}
					
					// record ordering
					ordering[x][y][z] = val;
					
					// update the volume
					if (useApprox) {
						for (int a=0;a<approx;a++)
							if (best[x][y][z][a]==k) 
								volume[k]+=mems[x][y][z][a];
					} else {
						volume[k]+=mems[x][y][z][k];
					}
					
					// update current classif
					segmentation[x][y][z] = (byte)k;
					
					// search for new neighbors
					for (short i=-1;i<=1;i++) for (short j=-1;j<=1;j++) for (short l=-1;l<=1;l++) if (i*i+j*j+l*l==1) {
						xi=x+i; yj=y+j; zl=z+l;
						if (available[xi][yj][zl][k]) {
							
							if (useApprox) {
								order = val + maxApproxMembershipRatio(xi,yj,zl,k);
							} else {
								order = val + maxMembershipRatio(xi,yj,zl,k,bestMembership(xi,yj,zl));
							}
							available[xi][yj][zl][k] = false;
							tree.addValue(order,xi,yj,zl,k);            
						}
					}
				} else {
					// critical point: back to background
					available[x][y][z][k] = true;
				}
			}
		}

		return;
	}//propagateGrowing

	/**
	 *  secondary function: propagate memberships with a fast marching technique
	 *  and maintain the topology; the distances are fixed but the memberships
	 *  can get lower if needed, introducing structural outliers
	 */
	public final void propagateThinning() {
		float   	val,order=0,factor;
		int 		xi,yj,zl;
		short 		x,y,z,k;
		boolean		isRegular;
		int			d2;
		float 		mem;
		int			rel = 0;
		boolean		changeBoundary;
		float[]		minVolume = new float[classes];
		
		// init: reset the boundary tree, the labels
		tree.reset();
		tree.setMaxTree();
		
		initThinningLabels();
		
		for (k=0;k<classes;k++) minVolume[k] = Numerics.max(100,volumeRatio*volume[k]);
		
		/*
		for (k=0;k<classes;k++) {
			if (debug) System.out.print("volume "+k+": "+volume[k]+"\n");
			if (debug) System.out.print("min volume "+k+": "+minVolume[k]+"\n");
		}
		*/
		
		// thinning
		val = 0.0f;
		changeBoundary = true;
		while (isNotEmpty() && (val > -limit)) {


			// get the next value
			x = tree.getFirstX();
			y = tree.getFirstY();
			z = tree.getFirstZ();
			k = tree.getFirstK();
			val = tree.getFirst();
			tree.removeFirst();
				
			if (volume[k] > minVolume[k]) {
					
				if (skeleton[x][y][z]==k) {
						
					byte id;
					if (useApprox) id = best[x][y][z][0];
					else id = bestMembership(x,y,z);
					boolean connected = false;
					boolean direct = false;
					float bestprior = -0.1f;
					byte next = EMPTY;
					boolean nextconnected = false;
					if (id!=k) {
						for (short i=-1;i<=1;i++) for (short j=-1;j<=1;j++) for (short l=-1;l<=1;l++) if (i*i+j*j+l*l==1) {
							xi=x+i; yj=y+j; zl=z+l;
							if (segmentation[xi][yj][zl]==id) {
								direct = true;
								connected = (skeleton[xi][yj][zl]==id);
							} else if (segmentation[xi][yj][zl]!=k && relationList[segmentation[xi][yj][zl]][id]) {
								// find the neighbor *connected* to the desired tissue with highest desired prior
								//if (atlas.getTransformedShape(xi,yj,zl)[id]>bestprior) {
								// or find the neighbor *connected* to the desired tissue with highest desired membership
								if (atlas.getTransformedShape(xi,yj,zl)[id]>bestprior) {
									next = segmentation[xi][yj][zl];
									nextconnected = (skeleton[xi][yj][zl]==next);
									bestprior = atlas.getTransformedShape(xi,yj,zl)[id];
								}
							}
						}
					}
					
					if (id==k) {
						// no change if it is already the best
						changeBoundary = false;
					} else if (direct) {
						// the best candidate is a local neighbor
						changeBoundary = true;
					} else if (next!=EMPTY) {
						// the best candidate is closest in this direction
						changeBoundary = true;
						id = next;
						connected = nextconnected;
					} else {
						// nothing is connected: no change possible
						changeBoundary = false;
					}
					
				
					// test for update
					if (changeBoundary) 
						changeBoundary = isHomeomorphicSegmentation(x,y,z,id);
				
					// much better (faster, and more flexible)
					isRegular = (volume[k]>minVolume[k]);
					
					if (isRegular) {
						// reset labels
						available[x][y][z][k] = false;	
					
						// record the ordering
						ordering[x][y][z] = val;
					
						// update the volume
						if (useApprox) {
							for (int a=0;a<approx;a++)
								if (best[x][y][z][a]==k)
									volume[k]-=mems[x][y][z][a];
						} else {
							volume[k]-=mems[x][y][z][k];
						}
						// update skeleton
						skeleton[x][y][z] = EMPTY;
						
						// update the classification in good cases
						if (changeBoundary) segmentation[x][y][z] = id;
					
						// search for new or critical neighbors
						for (short i=-1;i<=1;i++) for (short j=-1;j<=1;j++) for (short l=-1;l<=1;l++) if (i*i+j*j+l*l==1) {
							xi=x+i; yj=y+j; zl=z+l; 
							if (available[xi][yj][zl][k]) {
								// check for endpoints ?
								int Nngb=0;
								for (short ni=-1;ni<=1;ni++) for (short nj=-1;nj<=1;nj++) for (short nl=-1;nl<=1;nl++) {
									if (i*i+j*j+l*l>0 && segmentation[xi+ni][yj+nj][zl+nl]==k) {
										Nngb++;
									}
								}
								if (Nngb==1) {
									order = val;
								} else {
									if (useApprox) {
										order = val - 1.0f/maxApproxMembershipRatio(xi,yj,zl,k);
									} else{
										order = val - 1.0f/maxMembershipRatio(xi,yj,zl,k,bestMembership(xi,yj,zl));
									}
								}
								/*
								if (useApprox) {
									order = val - 1.0f/maxApproxMembershipRatio(xi,yj,zl,k);
								} else{
									order = val - 1.0f/maxMembershipRatio(xi,yj,zl,k,bestMembership(xi,yj,zl));
								}
								*/
								if ( (volume[k] > minVolume[k]) && (order > -limit) ) {
									available[xi][yj][zl][k] = false;
									tree.addValue(order,xi,yj,zl,k);
								}
							}			
						}
					} else {
						// critical point: make available again
						available[x][y][z][k] = true;
					}
				}
			}
		}
		//if (debug) System.out.print("final volume "+k+": "+volume[k]+"\n");		
	}

	/**
	 *	compute the cluster centroids given the distances 
	 */
    public final void computeCentroids() {
		if (useApprox) computeApproxCentroids();
		else computeExactCentroids();
	}
    public final void computeExactCentroids() {
        float mem;
		float[] num,den;
		float numL,denL,d0,dL;
        num = new float[classes];
		den = new float[classes];
		
        for (int c=0;c<nc;c++) {
			numL = 0;
			denL = 0;
			for (int k=0;k<classes;k++) {
				num[k] = 0;
				den[k] = 0;
				for (short x=1;x<nx-1;x++) for (short y=1;y<ny-1;y++) for (short z=1;z<nz-1;z++) {
					 if (objType[k]==OBJECT) {
						mem = mems[x][y][z][k];
						float psfEqual = 1.0f; 
						float img;
						if (useField) img = field[c][x][y][z]*images[c][x][y][z];
						else img = images[c][x][y][z];
						/*
						if (psfSpread>0) {
							psfEqual = psfWeight[0];
							for (int i=-npsf+1;i<npsf;i++) for (int j=-npsf+1;j<npsf;j++) for (int l=-npsf+1;l<npsf;l++) {
								if (x+i>=0 && x+i<nx && y+j>=0 && y+j<ny && z+l>=0 && z+l<nz) {
									int d = i*i+j*j+l*l;
									if (d>0 && d<npsf*npsf) {
										short m = bestMembership(x+i,y+j,z+l);
										if (m==k) psfEqual += psfWeight[d];
										else img -= psfWeight[d]*centroid[c][m];
									}
								}
							}
						}  else */
						if (psfRidge>0) {
							// use Laplacian on the boundaries
							if (isBoundary(x,y,z)) {
								img += psfRidge*(28.0f/27.0f)*images[c][x][y][z];
								for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
									img -= psfRidge*(1.0f/27.0f)*images[c][x+i][y+j][z+l];
								}
							}
						}
						num[k] += mem*mem*psfEqual*img;
						den[k] += mem*mem*psfEqual*psfEqual;	
					} else if (objType[k]==LESION) {
						mem = mems[x][y][z][k];
						
						if (segmentation[x][y][z]==k) {
							float memObj = mem*(1.0f-lesionMembership[x][y][z]);
							float memLes = mem*lesionMembership[x][y][z];
							
							if (useField) {
								num[k] += memObj*memObj*(field[c][x][y][z]*images[c][x][y][z]);
								den[k] += memObj*memObj;
							
								numL += memLes*memLes*(field[c][x][y][z]*images[c][x][y][z]);
								denL += memLes*memLes;
							} else {
								num[k] += memObj*memObj*images[c][x][y][z];
								den[k] += memObj*memObj;
							
								numL += memLes*memLes*images[c][x][y][z];
								denL += memLes*memLes;
							}
						} else {
							if (useField) {
								num[k] += mem*mem*(field[c][x][y][z]*images[c][x][y][z]);
								den[k] += mem*mem;
							} else {
								num[k] += mem*mem*images[c][x][y][z];
								den[k] += mem*mem;
							}
						}
					} else if (objType[k]==OUTLIER) {
						mem = mems[x][y][z][k]*(1.0f-outlierMembership[x][y][z]);
						
						if (useField) {
							num[k] += mem*mem*(field[c][x][y][z]*images[c][x][y][z]);
							den[k] += mem*mem;
						} else {
							num[k] += mem*mem*(images[c][x][y][z]);
							den[k] += mem*mem;
						}
					} else if (objType[k]==OPTIMIZED_CSF) {
						mem = mems[x][y][z][k];
						
						float dist;
						float factor = 1.0f;
							
						if (useField) dist = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
						else dist = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
						
						if (dist>ZERO) factor = (float)(powercsf[c].lookup(dist, pcsf[c])/dist);
							
						if (useField) num[k] += factor*mem*mem*(field[c][x][y][z]*images[c][x][y][z]);
						else num[k] += factor*mem*mem*(images[c][x][y][z]);
						den[k] += factor*mem*mem;	
					} else if (objType[k]==OPTIMIZED_GM) {
						mem = mems[x][y][z][k];
						
						float dist;
						float factor = 1.0f;
							
						if (useField) dist = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
						else dist = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
						
						if (dist>ZERO) factor = (float)(powergm[c].lookup(dist, pgm[c])/dist);
							
						if (useField) num[k] += factor*mem*mem*(field[c][x][y][z]*images[c][x][y][z]);
						else num[k] += factor*mem*mem*(images[c][x][y][z]);
						den[k] += factor*mem*mem;	
					} else if (objType[k]==OPTIMIZED_WM) {
						mem = mems[x][y][z][k];
						
						float dist;
						float factor = 1.0f;
							
						if (useField) dist = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
						else dist = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
						
						if (dist>ZERO) factor = (float)(powerwm[c].lookup(dist, pwm[c])/dist);
							
						if (useField) num[k] += factor*mem*mem*(field[c][x][y][z]*images[c][x][y][z]);
						else num[k] += factor*mem*mem*(images[c][x][y][z]);
						den[k] += factor*mem*mem;	
					} else if (objType[k]==RAYLEIGH) {
						mem = mems[x][y][z][k];
						
						if (useField) num[k] += mem*mem*(field[c][x][y][z]*images[c][x][y][z])*(field[c][x][y][z]*images[c][x][y][z]);
						else num[k] += mem*mem*(images[c][x][y][z])*(images[c][x][y][z]);
						den[k] += mem*mem;	
					} else {
						mem = mems[x][y][z][k];
						
						if (useField) num[k] += mem*mem*(field[c][x][y][z]*images[c][x][y][z]);
						else num[k] += mem*mem*(images[c][x][y][z]);
						den[k] += mem*mem;	
					}
				}
			}
			for (int k=0;k<classes;k++) {
				if (centroidMode.equals("none")) {
					if (den[k]>0.0) {
						if (objType[k]==RAYLEIGH) centroid[c][k] = (float)Math.sqrt(num[k]/(2.0f*den[k]));
						else centroid[c][k] = num[k]/den[k];
					} else {
						centroid[c][k] = 0.0f;
					}
				} else if (centroidMode.equals("linked")) {
					float nsum,dsum;
					nsum = num[k];
					dsum = den[k];
					for (int l=0;l<classes;l++) {
						if (l!=k && centroidLink[c][k]==centroidLink[c][l]) {
							nsum += centroidSmoothness*num[l];
							dsum += centroidSmoothness*den[l];
						}
					}
					if (dsum>0.0) {
						if (objType[k]==RAYLEIGH) centroid[c][k] = (float)Math.sqrt(nsum/(2.0f*dsum));
						else centroid[c][k] = nsum/dsum;	
					} else {
						centroid[c][k] = 0.0f;
					}
				} else if (centroidMode.equals("damped")) {
					if (den[k]>0.0) {
						if (objType[k]==RAYLEIGH) centroid[c][k] = (1.0f-centroidSmoothness)*(float)Math.sqrt(num[k]/(2.0f*den[k])) + centroidSmoothness*centroid[c][k];
						else centroid[c][k] = (1.0f-centroidSmoothness)*num[k]/den[k] + centroidSmoothness*centroid[c][k];
					} else {
						// no change: previous value
					}
				} else if (centroidMode.equals("prior")) {
					float nsum,dsum;
					nsum = num[k];
					dsum = den[k];
					
					nsum += centroidSmoothness*1e4f*priorCentroid[c][k];
					dsum += centroidSmoothness*1e4f;
					
					if (dsum>0.0) {
						centroid[c][k] = nsum/dsum;	
					} else {
						centroid[c][k] = 0.0f;
					}
				} else {
					if (den[k]>0.0) {
						if (objType[k]==RAYLEIGH) centroid[c][k] = (float)Math.sqrt(num[k]/(2.0f*den[k]));
						else centroid[c][k] = num[k]/den[k];
					} else {
						centroid[c][k] = 0.0f;
					}
				}
			}
			if (denL>0.0) {
				lesionCentroid[c] = numL/denL;
			} else {
				lesionCentroid[c] = 0.0f;
			}
			
		}
        return;
    } // computeCentroids
    
    public final void computeApproxCentroids() {
        float mem;
		float[] num,den;
		float numL,denL,d0,dL;
        num = new float[classes];
		den = new float[classes];
		
        for (int c=0;c<nc;c++) {
			numL = 0;
			denL = 0;
			for (int k=0;k<classes;k++) {
				num[k] = 0;
				den[k] = 0;
			}
			for (short x=0;x<nx;x++) for (short y=0;y<ny;y++) for (short z=0;z<nz;z++) {
				for (int a=0;a<approx;a++) {
					int k = best[x][y][z][a];
					
					if (objType[k]==OBJECT) {
						mem = mems[x][y][z][a];
						float psfEqual = 1.0f; 
						float img;
						if (useField) img = field[c][x][y][z]*images[c][x][y][z];
						else img = images[c][x][y][z];
						/*
						if (psfSpread>0) {
							psfEqual = psfWeight[0];
							for (int i=-npsf+1;i<npsf;i++) for (int j=-npsf+1;j<npsf;j++) for (int l=-npsf+1;l<npsf;l++) {
								if (x+i>=0 && x+i<nx && y+j>=0 && y+j<ny && z+l>=0 && z+l<nz) {
									int d = i*i+j*j+l*l;
									if (d>0 && d<npsf*npsf) {
										short m = bestMembership(x+i,y+j,z+l);
										if (m==k) psfEqual += psfWeight[d];
										else img -= psfWeight[d]*centroid[c][m];
									}
								}
							}
						}  else */
						if (psfRidge>0) {
							// use Laplacian on the boundaries
							if (isBoundary(x,y,z)) {
								img += psfRidge*(28.0f/27.0f)*images[c][x][y][z];
								for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
									img -= psfRidge*(1.0f/27.0f)*images[c][x+i][y+j][z+l];
								}
							}
						}
						num[k] += mem*mem*psfEqual*img;
						den[k] += mem*mem*psfEqual*psfEqual;
					} else if (objType[k]==LESION) {
						mem = mems[x][y][z][a];
						
						if (segmentation[x][y][z]==k) {
							float memObj = mem*(1.0f-lesionMembership[x][y][z]);
							float memLes = mem*lesionMembership[x][y][z];
								
							if (useField) {
								num[k] += memObj*memObj*(field[c][x][y][z]*images[c][x][y][z]);
								den[k] += memObj*memObj;
							
								numL += memLes*memLes*(field[c][x][y][z]*images[c][x][y][z]);
								denL += memLes*memLes;
							} else {
								num[k] += memObj*memObj*images[c][x][y][z];
								den[k] += memObj*memObj;
							
								numL += memLes*memLes*images[c][x][y][z];
								denL += memLes*memLes;
							}
						} else {
							if (useField) {
								num[k] += mem*mem*(field[c][x][y][z]*images[c][x][y][z]);
								den[k] += mem*mem;
							} else {
								num[k] += mem*mem*images[c][x][y][z];
								den[k] += mem*mem;
							}
						}
					} else if (objType[k]==OUTLIER) {
						mem = mems[x][y][z][a]*(1.0f-outlierMembership[x][y][z]);
						
						if (useField) {
							num[k] += mem*mem*(field[c][x][y][z]*images[c][x][y][z]);
							den[k] += mem*mem;
						} else {
							num[k] += mem*mem*(images[c][x][y][z]);
							den[k] += mem*mem;
						}
					} else if (objType[k]==OPTIMIZED_CSF) {
						mem = mems[x][y][z][a];
						
						float dist;
						float factor = 1.0f;
								
						if (useField) dist = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
						else dist = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
							
						if (dist>ZERO) factor = (float)(powercsf[c].lookup(dist, pcsf[c])/dist);
								
						if (useField) num[k] += factor*mem*mem*(field[c][x][y][z]*images[c][x][y][z]);
						else num[k] += factor*mem*mem*(images[c][x][y][z]);
						den[k] += factor*mem*mem;	
					} else if (objType[k]==OPTIMIZED_GM) {
						mem = mems[x][y][z][a];
						
						float dist;
						float factor = 1.0f;
							
						if (useField) dist = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
						else dist = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
						
						if (dist>ZERO) factor = (float)(powergm[c].lookup(dist, pgm[c])/dist);
							
						if (useField) num[k] += factor*mem*mem*(field[c][x][y][z]*images[c][x][y][z]);
						else num[k] += factor*mem*mem*(images[c][x][y][z]);
						den[k] += factor*mem*mem;	
					} else if (objType[k]==OPTIMIZED_WM) {
						mem = mems[x][y][z][a];
						
						float dist;
						float factor = 1.0f;
								
						if (useField) dist = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
						else dist = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
						
						if (dist>ZERO) factor = (float)(powerwm[c].lookup(dist, pwm[c])/dist);
							
						if (useField) num[k] += factor*mem*mem*(field[c][x][y][z]*images[c][x][y][z]);
						else num[k] += factor*mem*mem*(images[c][x][y][z]);
						den[k] += factor*mem*mem;	
					} else if (objType[k]==RAYLEIGH) {
						mem = mems[x][y][z][a];
							
						if (useField) num[k] += mem*mem*(field[c][x][y][z]*images[c][x][y][z])*(field[c][x][y][z]*images[c][x][y][z]);
						else num[k] += mem*mem*(images[c][x][y][z])*(images[c][x][y][z]);
						den[k] += mem*mem;	
					} else {
						mem = mems[x][y][z][a];

						if (useField) num[k] += mem*mem*(field[c][x][y][z]*images[c][x][y][z]);
						else num[k] += mem*mem*(images[c][x][y][z]);
						den[k] += mem*mem;	
					}
				}
			}
			for (int k=0;k<classes;k++) {
				if (centroidMode.equals("none")) {
					if (den[k]>0.0) {
						if (objType[k]==RAYLEIGH) centroid[c][k] = (float)Math.sqrt(num[k]/(2.0f*den[k]));
						else centroid[c][k] = num[k]/den[k];
					} else {
						centroid[c][k] = 0.0f;
					}
				} else if (centroidMode.equals("linked")) {
					float nsum,dsum;
					nsum = num[k];
					dsum = den[k];
					for (int l=0;l<classes;l++) {
						if (l!=k && centroidLink[c][k]==centroidLink[c][l]) {
							nsum += centroidSmoothness*num[l];
							dsum += centroidSmoothness*den[l];
						}
					}
					if (dsum>0.0) {
						if (objType[k]==RAYLEIGH) centroid[c][k] = (float)Math.sqrt(nsum/(2.0f*dsum));
						else centroid[c][k] = nsum/dsum;	
					} else {
						centroid[c][k] = 0.0f;
					}
				} else if (centroidMode.equals("damped")) {
					if (den[k]>0.0) {
						if (objType[k]==RAYLEIGH) centroid[c][k] = (1.0f-centroidSmoothness)*(float)Math.sqrt(num[k]/(2.0f*den[k])) + centroidSmoothness*centroid[c][k];
						else centroid[c][k] = (1.0f-centroidSmoothness)*num[k]/den[k] + centroidSmoothness*centroid[c][k];
					} else {
						// no change: previous value
					}
				} else {
					if (den[k]>0.0) {
						if (objType[k]==RAYLEIGH) centroid[c][k] = (float)Math.sqrt(num[k]/(2.0f*den[k]));
						else centroid[c][k] = num[k]/den[k];
					} else {
						centroid[c][k] = 0.0f;
					}
				}
			}
			if (denL>0.0) {
				lesionCentroid[c] = numL/denL;
			} else {
				lesionCentroid[c] = 0.0f;
			}
			
		}
        return;
    } // computeCentroids
    
	/**
	 *	compute the stddev of the cluster centroids given the distances 
	 */
    public final void computeVariances() {
		if (useApprox) computeApproxVariances();
		else computeExactVariances();
	}
    public final void computeExactVariances() {
        float num,den,mem,diff;
        float numL,denL,memL,diffL;
        
        for (int c=0;c<nc;c++) {
			for (int k=0;k<classes;k++) {
				num = 0; numL = 0;
				den = 0; denL = 0;
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					if (segmentation[x][y][z]==k) {
						if (objType[k]==LESION) {
							mem = mems[x][y][z][k];
							
							float memObj = mem*(1.0f-lesionMembership[x][y][z]);
						
							if (useField) diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
							else diff = (images[c][x][y][z]-centroid[c][k]);
							
							num += (memObj*memObj+1e-30f)*diff*diff;
							den += (memObj*memObj+1e-30f);
						
							memL = mem*lesionMembership[x][y][z];
						
							if (useField) diffL = (field[c][x][y][z]*images[c][x][y][z]-lesionCentroid[c]);
							else diffL = (images[c][x][y][z]-lesionCentroid[c]);
							
							numL += (memL*memL+1e-30f)*diffL*diffL;
							denL += (memL*memL+1e-30f);
						
						} else {
							mem = mems[x][y][z][k];
							
							if (useField) diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
							else diff = (images[c][x][y][z]-centroid[c][k]);
							
							num += (mem*mem+1e-30f)*diff*diff;
							den += (mem*mem+1e-30f);
						}
					}
				}
				if (den>0.0) {
					stddev[c][k] = (float)Math.sqrt(num/den);
				} else {
					stddev[c][k] = 0.0f;
				}
				if (objType[k]==LESION) {
					if (denL>0.0) {
					lesionStddev[c] = (float)Math.sqrt(numL/denL);
					} else {
						lesionStddev[c] = 0.0f;
					}
				}
			}
		}
        return;
    } // computeVariances
    public final void computeApproxVariances() {
        float mem,diff;
        float numL,denL,memL,diffL;
        float[] num,den;
		
		num = new float[classes];
		den = new float[classes];
        for (int c=0;c<nc;c++) {
			for (int k=0;k<classes;k++) {
				num[k] = 0; 
				den[k] = 0; 
			}
			numL = 0;
			denL = 0;
			for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
				for (int a=0;a<approx;a++) {
					int k = best[x][y][z][a];
					if (segmentation[x][y][z]==k) {
						if (objType[k]==LESION) {
							mem = mems[x][y][z][a];
							
							float memObj = mem*(1.0f-lesionMembership[x][y][z]);
						
							if (useField) diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
							else diff = (images[c][x][y][z]-centroid[c][k]);
							
							num[k] += (memObj*memObj+1e-30f)*diff*diff;
							den[k] += (memObj*memObj+1e-30f);
						
							memL = mem*lesionMembership[x][y][z];
						
							if (useField) diffL = (field[c][x][y][z]*images[c][x][y][z]-lesionCentroid[c]);
							else diffL = (images[c][x][y][z]-lesionCentroid[c]);
							
							numL += (memL*memL+1e-30f)*diffL*diffL;
							denL += (memL*memL+1e-30f);
						
						} else {
							mem = mems[x][y][z][a];
							
							if (useField) diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
							else diff = (images[c][x][y][z]-centroid[c][k]);
							
							num[k] += (mem*mem+1e-30f)*diff*diff;
							den[k] += (mem*mem+1e-30f);
						}
					}
				}
			}
			for (int k=0;k<classes;k++) {
				if (den[k]>0.0) {
					stddev[c][k] = (float)Math.sqrt(num[k]/den[k]);
				} else {
					stddev[c][k] = 0.0f;
				}
				if (objType[k]==LESION) {
					if (denL>0.0) {
					lesionStddev[c] = (float)Math.sqrt(numL/denL);
					} else {
						lesionStddev[c] = 0.0f;
					}
				}
			}
		}
        return;
    } // computeVariances
    
	/**
	 *	compute the relative weight of inverse variance of the cluster centroids for each image
	 */
    public final void computeImageVariance() {
		if (useApprox) computeApproxImageVariance();
		else computeExactImageVariance();
	}
    public final void computeExactImageVariance() {
        float num,den,mem,diff,diffO,diffL;
		float sum;
        
		sum = 0;
        for (int c=0;c<nc;c++) {
			num = 0;
			den = 0;
			for (int k=0;k<classes;k++) {
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					mem = mems[x][y][z][k];
						
					// data term
					diff = 0; diffO = 0; diffL = 0; 
					if (objType[k]==OBJECT) {
						if (useField) diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
						else diff = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
						
						num += (mem*mem+1e-30f)*diff*diff;
						den += (mem*mem+1e-30f);
						
					} else if (objType[k]==OUTLIER) {
						if (useField) {
							diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
							diffO = outlier;
						} else {
							diff = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
							diffO = outlier;
						}
						float omem = outlierMembership[x][y][z];
						num += (mem*mem*(1.0f-omem)*(1.0f-omem)+1e-30f)*diff*diff
								+(mem*mem*omem*omem)*diffO*diffO;
						den += (mem*mem+1e-30f);
						
					} else if (objType[k]==LESION) {
						if (useField) {
							diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
							if (segmentation[x][y][z]==k) diffL = (field[c][x][y][z]*images[c][x][y][z]-lesionCentroid[c])*(field[c][x][y][z]*images[c][x][y][z]-lesionCentroid[c]);
						} else {
							diff = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
							if (segmentation[x][y][z]==k) diffL = (images[c][x][y][z]-lesionCentroid[c])*(images[c][x][y][z]-lesionCentroid[c]);
						}
						float lmem = lesionMembership[x][y][z];
						num += (mem*mem*(1.0f-lmem)*(1.0f-lmem)+1e-30f)*diff*diff 
								+(mem*mem*lmem*lmem)*diffL*diffL;
						den += (mem*mem+1e-30f);
						
					} else if (objType[k]==OPTIMIZED_CSF || objType[k]==OPTIMIZED_GM || objType[k]==OPTIMIZED_WM) {
						if (useField) diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
						else diff = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
						
						num += (mem*mem+1e-30f)*diff*diff;
						den += (mem*mem+1e-30f);
					}			
				}
			}
			if (num>0.0) {
				varImg[c] = den/num;
			} else {
				varImg[c] = INF;
			}
			sum += varImg[c];
		}
		for (int c=0;c<nc;c++) {
			stddev[c][classes] = (float)Math.sqrt(varImg[c]);
			varImg[c] = varImg[c]/sum;
		}
		return;
    } // computeVariances
     public final void computeApproxImageVariance() {
        float num,den,mem,diff,diffO,diffL;
		float sum;
        
		sum = 0;
        for (int c=0;c<nc;c++) {
			num = 0;
			den = 0;
			for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
				for (int a=0;a<approx;a++) {
					int k = best[x][y][z][a];
					mem = mems[x][y][z][a];
						
					// data term
					diff = 0; diffO = 0; diffL = 0; 
					if (objType[k]==OBJECT) {
						if (useField) diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
						else diff = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
						
						num += (mem*mem+1e-30f)*diff*diff;
						den += (mem*mem+1e-30f);
						
					} else if (objType[k]==OUTLIER) {
						if (useField) {
							diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
							diffO = outlier;
						} else {
							diff = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
							diffO = outlier;
						}
						float omem = outlierMembership[x][y][z];
						num += (mem*mem*(1.0f-omem)*(1.0f-omem)+1e-30f)*diff*diff
								+(mem*mem*omem*omem)*diffO*diffO;
						den += (mem*mem+1e-30f);
						
					} else if (objType[k]==LESION) {
						if (useField) {
							diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
							if (segmentation[x][y][z]==k) diffL = (field[c][x][y][z]*images[c][x][y][z]-lesionCentroid[c])*(field[c][x][y][z]*images[c][x][y][z]-lesionCentroid[c]);
						} else {
							diff = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
							if (segmentation[x][y][z]==k) diffL = (images[c][x][y][z]-lesionCentroid[c])*(images[c][x][y][z]-lesionCentroid[c]);
						}
						float lmem = lesionMembership[x][y][z];
						num += (mem*mem*(1.0f-lmem)*(1.0f-lmem)+1e-30f)*diff*diff 
								+(mem*mem*lmem*lmem)*diffL*diffL;
						den += (mem*mem+1e-30f);
						
					} else if (objType[k]==OPTIMIZED_CSF || objType[k]==OPTIMIZED_GM || objType[k]==OPTIMIZED_WM) {
						if (useField) diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k])*(field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
						else diff = (images[c][x][y][z]-centroid[c][k])*(images[c][x][y][z]-centroid[c][k]);
						
						num += (mem*mem+1e-30f)*diff*diff;
						den += (mem*mem+1e-30f);
					}			
				}
			}
			if (num>0.0) {
				varImg[c] = den/num;
			} else {
				varImg[c] = INF;
			}
			sum += varImg[c];
		}
		for (int c=0;c<nc;c++) {
			stddev[c][classes] = (float)Math.sqrt(varImg[c]);
			varImg[c] = varImg[c]/sum;
		}
		return;
    } // computeVariances
   
	/**
	 *	compute the stddev of the cluster centroids given the distances 
	 */
	/* 
    public final void computeVarianceWeights() {
		if (useApprox) computeApproxVarianceWeights();
		else computeExactVarianceWeights();
	}
    public final void computeExactVarianceWeights() {
        float num,den,mem,diff;
        float numT, denT;
		float pcount = 100.0f;
        
        for (int c=0;c<nc;c++) {
			numT = 0.0f;
			denT = 0.0f;
			for (int k=0;k<classes;k++) {
				// prior value
				num = pcount*varprior[c][k]; 
				den = pcount; 
				if (objType[k]!=MASK && objType[k]!=OUTMASK) {
					for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
						if (segmentation[x][y][z]==k) {
							mem = mems[x][y][z][k];
							
							if (useField) diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
							else diff = (images[c][x][y][z]-centroid[c][k]);
							
							num += (mem*mem+1e-30f)*diff*diff;
							den += (mem*mem+1e-30f);
						}
					}
				}
				// recombine 
				varweight[c][k] = num/den;
				
				numT += num;
				denT += den;
			}
			// compute ratio
			for (int k=0;k<classes;k++) {
				varweight[c][k] = classes*(numT/denT)/varweight[c][k];
			}
		}
        return;
    } // computeVarianceWeights
    public final void computeApproxVarianceWeights() {
        float mem,diff;
        float numT,denT;
        float[] num,den;
		float pcount = 100.0f;
        
		num = new float[classes];
		den = new float[classes];
        for (int c=0;c<nc;c++) {
			numT = 0.0f;
			denT = 0.0f;
			for (int k=0;k<classes;k++) {
				num[k] = pcount*varprior[c][k]; 
				den[k] = pcount; 
			}
			for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
				for (int a=0;a<approx;a++) {
					int k = best[x][y][z][a];
					if (objType[k]!=MASK && objType[k]!=OUTMASK) {
						if (segmentation[x][y][z]==k) {
							mem = mems[x][y][z][a];
							
							if (useField) diff = (field[c][x][y][z]*images[c][x][y][z]-centroid[c][k]);
							else diff = (images[c][x][y][z]-centroid[c][k]);
							
							num[k] += (mem*mem+1e-30f)*diff*diff;
							den[k] += (mem*mem+1e-30f);
						}
					}
				}
			}
			for (int k=0;k<classes;k++) {
				// recombine 
				varweight[c][k] = num[k]/den[k];
				
				numT += num[k];
				denT += den[k];
			}
			// compute ratio
			for (int k=0;k<classes;k++) {
				varweight[c][k] = classes*(numT/denT)/varweight[c][k];
			}
		}
        return;
    } // computeVarianceWeights
    */
	
	/**
	 *	output the centroids
	 */
    public final String displayCentroids() {
        String output="";
		for (int c=0;c<nc;c++) {
			output += "centroids("+(c+1)+") ";
			for (int k=0;k<classes;k++) output +=" | "+centroid[c][k]*Iscale[c];
			output += "\n";
		}
		return output;
	}
	
    public final String displayCentroidGroups() {
        String output="";
		for (int c=0;c<nc;c++) {
			output += "centroids groups("+(c+1)+") ";
			for (int k=0;k<classes;k++) output +=" | "+centroidLink[c][k]*Iscale[c];
			output += "\n";
		}
		return output;
	}
	
	/**
	 *	output the centroids
	 */
    public final String displayVariances() {
        String output="";
		for (int c=0;c<nc;c++) {
			output += "stddevs("+(c+1)+") ";
			for (int k=0;k<classes;k++) output +=" | "+stddev[c][k]*Iscale[c];
			output += "\n";
		}
		return output;
	}
	
    public final String displayImageVariance() {
        String output = "varImg ";
		for (int c=0;c<nc;c++) {
			output +=" | "+Math.sqrt(varImg[c])*Iscale[c];
		}
		output += "\n";
		return output;
	}
	
    public final String displayVarianceWeights() {
        String output="";
		for (int c=0;c<nc;c++) {
			output += "varweight("+(c+1)+") ";
			for (int k=0;k<classes;k++) output +=" | "+varweight[c][k];
			output += "\n";
		}
		return output;
	}
	
	/**
	 *	output the lesion centroids
	 */
    public final String displayLesionCentroid() {
        String output = "lesion centroid ";
		for (int c=0;c<nc;c++) {
			output +=" | "+lesionCentroid[c]*Iscale[c];
		}
		output += "\n";
		
		return output;
	}
	
	/**
	 *	output the lesion centroid variance
	 */
    public final String displayLesionVariance() {
        String output = "lesion stddev ";
		for (int c=0;c<nc;c++) {
			output +=" | "+lesionStddev[c]*Iscale[c];
		}
		output += "\n";
		
		return output;
	}
	
	/**
	 *  analyze the template classes to make relation map 
	 */
	public final void buildRelationList() {
		// find the boundaries between labels
		int MAXLB=500;
		short[][] lbs = new short[MAXLB][];
		short Nlb=0;
		short[] newlb = new short[MAXLB];
		int Nb;
		boolean isFound,isFoundLine,newValue;
		// start with the single class relations
		for (short m=0;m<classes;m++) {
			lbs[Nlb] = new short[1];
			lbs[Nlb][0] = m;
			Nlb++;
		}
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			// find boundaries
			for (int n=0;n<MAXLB;n++) newlb[n] = EMPTY;
			Nb = 0;
			for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
				if (i*i+j*j+l*l==1) {
					if (segmentation[x][y][z]!=segmentation[x+i][y+j][z+l]) {
						newValue = true;
						for (int b=0;b<Nb;b++) 
							if (segmentation[x+i][y+j][z+l] == newlb[b])
								{ newValue = false; break; }
						if (newValue) {
							newlb[Nb] = segmentation[x+i][y+j][z+l];
							Nb++;
						}
					}
				}
			}
			if (newlb[0]!=EMPTY) {
				newlb[Nb] = segmentation[x][y][z]; 
				Nb++; 
				// check if boundary unique
				isFoundLine = false;
				for (short n=0;n<Nlb;n++) {
					if (lbs[n].length!=Nb) {
						isFoundLine = false;
					} else {
						isFoundLine = true;
						for (int b=0;b<Nb;b++) {
							isFound = false;
							for (int c=0;c<lbs[n].length;c++)
								if (newlb[b]==lbs[n][c]) { isFound = true; break; }
							if (!isFound) isFoundLine = false;
						}
					}
					if (isFoundLine) {
						// same labels
						relationMap[x][y][z] = n;
						break;
					}
				}
				if (!isFoundLine) {
					// create a new entry
					lbs[Nlb] = new short[Nb];
					for (int b=0;b<Nb;b++) lbs[Nlb][b] = newlb[b];
					relationMap[x][y][z] = Nlb;
					Nlb++;
				}
			} else {
				// not a boundary: find the object label
				relationMap[x][y][z] = segmentation[x][y][z];
			}
		}
		// convert into boolean relations
		Nrelations = Nlb;
		relationList = new boolean[Nlb][classes];
		for (int n=0;n<Nlb;n++) {
			for (int m=0;m<classes;m++) relationList[n][m] = false;
			
			for (int b=0;b<lbs[n].length;b++) {
				if (lbs[n][b]!=EMPTY) relationList[n][lbs[n][b]] = true;
			}
		}
		// for the first relations: list all related objects
		for (int k=0;k<classes;k++) {
			for (int n=classes;n<Nlb;n++) {
				if (relationList[n][k]) {
					for (int m=0;m<classes;m++) {
						if (relationList[n][m]) relationList[k][m] = true;
					}
				}
			}
		}
		lbs = null;
		newlb = null;

		return;
	}//updateRelationList

	/**
	 *  show the relation list
	 */
	public final void displayRelationList() {
		// output the list of boundaries
		MedicUtilPublic.displayMessage("relation list: \n");
		MedicUtilPublic.displayMessage(" : (");
		for (int m=0;m<classes;m++) MedicUtilPublic.displayMessage(""+templateLabel[m]+" ");
		MedicUtilPublic.displayMessage(")\n");
		for (int n=0;n<Nrelations;n++) {
			MedicUtilPublic.displayMessage(""+n+": (");
			//for (int m=0;m<classes+pairs;m++) 
			for (int m=0;m<classes;m++) 
				if (relationList[n][m]) MedicUtilPublic.displayMessage(" 1 ");
				else MedicUtilPublic.displayMessage(" 0 ");
			MedicUtilPublic.displayMessage(")\n");
		}

		MedicUtilPublic.displayMessage("intensity relations: \n");
		for (int k=0;k<classes;k++) {
			MedicUtilPublic.displayMessage(templateLabel[k]+" : ");
			for (int m=0;m<classes;m++) if (relationList[k][m]) MedicUtilPublic.displayMessage(templateLabel[m]+" ");
			MedicUtilPublic.displayMessage("\n");
		}
		MedicUtilPublic.displayMessage("relation labels: \n");
		for (int n=classes;n<Nrelations;n++) {
			MedicUtilPublic.displayMessage(n+" > ");
			for (int k=0;k<classes;k++) if (relationList[n][k]) MedicUtilPublic.displayMessage(templateLabel[k]+" ");
			MedicUtilPublic.displayMessage("\n");	
		}
		return;
	} //displayRelationList

	/**
	 *	compute the classification from the labels 
	 */
    public final int updateClassificationFromLabels() {
        int classDiff = 0;
		byte prev;
        
        for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			prev = classification[x][y][z];
			classification[x][y][z] = templateLabel[segmentation[x][y][z]];
			
			if (prev != classification[x][y][z]) classDiff++;		
       }
	   return classDiff;
	}

	/**
	 *	compute the classification from the labels 
	 */
    public final int updateClassificationFromSkeleton() {
        int classDiff = 0;
		byte prev;
        
        for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			prev = classification[x][y][z];
			classification[x][y][z] = 0;
			if (skeleton[x][y][z]!=EMPTY)
				classification[x][y][z] = templateLabel[skeleton[x][y][z]];
			if (prev != classification[x][y][z]) classDiff++;		
       }
	   return classDiff;
	}


	/**
	 *	compute the classification from the memberships
	 */
    public final int updateClassificationFromMemberships() {
        int classDiff = 0;
		byte prev;
		int b;
        
        for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			prev = classification[x][y][z];
			if (useApprox) {
				classification[x][y][z] = templateLabel[best[x][y][z][0]];
			} else {
				classification[x][y][z] = 0;
				b = 0;
				for (int k=1;k<classes;k++) if (mems[x][y][z][k]>mems[x][y][z][b]) {
					b = k;
				}
				if (mems[x][y][z][b]>0) classification[x][y][z] = templateLabel[b];
			}
			if (prev != classification[x][y][z]) classDiff++;		
       }
	   return classDiff;
	}

	/** 
	 *	export membership functions 
	 */
	public final float[][][][] exportMemberships() {
		float[][][][]	Mems = new float[classes][nx][ny][nz];
		
		if (useApprox) {
			for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
				for (int k=0;k<classes;k++) Mems[k][x][y][z]=0.0f;
				for (int a=0;a<approx;a++) Mems[best[x][y][z][a]][x][y][z] = mems[x][y][z][a];
			}
		} else {
			for (int k=0;k<classes;k++) {
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					Mems[k][x][y][z] = mems[x][y][z][k];
				}
			}
		}
		return Mems;
	} // exportMemberships
	
	public final byte[][][] exportClassification() {
		byte[][][]	Classif = new byte[nx][ny][nz];
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			Classif[x][y][z] = templateLabel[segmentation[x][y][z]];
		}
		return Classif;
	} // exportClassification
	
	public final byte[][][] exportSkeleton() {
		byte[][][]	Classif = new byte[nx][ny][nz];
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			if (skeleton[x][y][z]!=EMPTY) Classif[x][y][z] = templateLabel[skeleton[x][y][z]];
			else Classif[x][y][z] = 0;
		}
		return Classif;
	} // exportClassification
	
	public final float[][][] exportOrdering() {
		float[][][]	Order = new float[nx][ny][nz];
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			Order[x][y][z] = ordering[x][y][z];
		}
		return Order;
	} // exportOrdering
	
	public final float[][][] exportLesionMembership() {
		float[][][]	Lesion = new float[nx][ny][nz];
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			Lesion[x][y][z] = lesionMembership[x][y][z];
		}
		return Lesion;
	} // exportLesionMembership
	
	public final float[][][] exportOutlierMembership() {
		float[][][]	Outlier = new float[nx][ny][nz];
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			Outlier[x][y][z] = outlierMembership[x][y][z];
		}
		return Outlier;
	} // exportOutlierMembership
	
	public final byte[][][] exportRelationMap() {
		byte[][][]	Relations = new byte[nx][ny][nz];
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			Relations[x][y][z] = (byte)relationMap[x][y][z];
		}
		return Relations;
	} // exportRelationMap
   
	public final byte[][][] exportClosestLabelMap(int n) {
		byte[][][]	Map = new byte[nx][ny][nz];
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			Map[x][y][z] = (byte)closestLabel[x][y][z][n];
		}
		return Map;
	} // exportRelationMap
   
	/**  generate atlas image from information
	 */
    final public byte[][][] generateClassification() {
		float dist,max,count;
		int b=0;
		byte[][][] img = new byte[nx][ny][nz];
		
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			if (useApprox) {
				img[x][y][z] = templateLabel[best[x][y][z][0]];
			} else {
				// compute each class probability : attribute the highest
				max = 0; b = -1;
				for (int k=0;k<classes;k++) {
					if (mems[x][y][z][k]>max) {
						b = k;
						max = mems[x][y][z][k];
					}
				}
				if (b>-1) img[x][y][z] = templateLabel[b];
				else img[x][y][z] = 0;
			}
		}
		return img;
	}
	
	/** 
	 *	generate membership functions and WM mask for CRUISE 
	 */
	public final float[][][][] generateCruiseOutputs() {
		float[][][][]	Mems = new float[4][nx][ny][nz];
		
		// CSF
		for (int k=0;k<classes;k++) {
			if (atlas.getNames()[k].equals("SulcalCSF") 
				|| atlas.getNames()[k].equals("Sulcal-CSF")) {
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					if (useApprox) {
						for (int a=0;a<approx;a++)
							if (best[x][y][z][a]==k)
								Mems[0][x][y][z] = mems[x][y][z][a];
					} else {
						Mems[0][x][y][z] = mems[x][y][z][k];
					}
				}
			}
		}
		// GM
		for (int k=0;k<classes;k++) {
			if (atlas.getNames()[k].equals("CorticalGM") 
				|| atlas.getNames()[k].equals("Cerebrum-GM") 
				|| atlas.getNames()[k].equals("CerebralGM")) {
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					if (useApprox) {
						for (int a=0;a<approx;a++)
							if (best[x][y][z][a]==k)
								Mems[1][x][y][z] = mems[x][y][z][a];
					} else {
						Mems[1][x][y][z] = mems[x][y][z][k];
					}
				}
			}
		}
		// WM (everything else except background and cerebellum / brainstem
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			Mems[2][x][y][z] = 0.0f;
			Mems[3][x][y][z] = 0.0f;
		}
		for (int k=0;k<classes;k++) {
			if (!atlas.getNames()[k].equals("SulcalCSF") 
				&& !atlas.getNames()[k].equals("Sulcal-CSF")
				&& !atlas.getNames()[k].equals("CorticalGM") 
				&& !atlas.getNames()[k].equals("Cerebrum-GM") 
				&& !atlas.getNames()[k].equals("CerebralGM")
				&& !atlas.getNames()[k].startsWith("Cerebell")
				&& !atlas.getNames()[k].equals("Brainstem")
				&& !atlas.getNames()[k].equals("Background")) {
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					if (useApprox) {
						for (int a=0;a<approx;a++)
							if (best[x][y][z][a]==k)
								Mems[2][x][y][z] += mems[x][y][z][a];
					} else {
						Mems[2][x][y][z] += mems[x][y][z][k];
					}
					if (segmentation[x][y][z]==k) Mems[3][x][y][z] = 1.0f;
				}
			}
		}
			
		return Mems;
	} // generateCruiseOutputs
	
public final float[][][][] generateDuraRemovalOutputs(){
    	
    	float[][][][]	Mems = new float[4][nx][ny][nz];
    			
    			// CSF
    			for (int k=0;k<classes;k++) {
    				if (atlas.getNames()[k].equals("SulcalCSF") 
    					|| atlas.getNames()[k].equals("Sulcal-CSF")) {
    					for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) 
    						Mems[0][x][y][z] = mems[x][y][z][k] ;
    				}
    			}
    			
    			
    			// WM (everything else except background and cerebellum-GM 
    			for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
    				Mems[1][x][y][z] = 0.0f;
    				Mems[2][x][y][z] = 0.0f;
    				Mems[3][x][y][z] = 0.0f;
    			}
    			for (int k=0;k<classes;k++) {
    				if (!atlas.getNames()[k].equals("SulcalCSF") 
    					&& !atlas.getNames()[k].equals("Sulcal-CSF")
    					&& !atlas.getNames()[k].equals("CorticalGM") 
    					&& !atlas.getNames()[k].equals("Cerebrum-GM") 
    					&& !atlas.getNames()[k].equals("CerebralGM")
    					&& !atlas.getNames()[k].equals("Cerebellum-GM")
    					&& !atlas.getNames()[k].equals("CerebellarGM")
    					&& !atlas.getNames()[k].equals("Background")) {
    					for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
    						Mems[2][x][y][z] += (mems[x][y][z][k]);
    						if (segmentation[x][y][z]==k) Mems[3][x][y][z] = 1.0f;
    					}
    				}
    			}
    			
    			// GM (Cerebellum and Cortical)
    			for (int k=0;k<classes;k++) {
    				if (atlas.getNames()[k].equals("CorticalGM") 
    					|| atlas.getNames()[k].equals("Cerebrum-GM") 
    					|| atlas.getNames()[k].equals("CerebralGM")
    					|| atlas.getNames()[k].equals("Cerebellum-GM")
    					|| atlas.getNames()[k].equals("CerebellarGM")) {
    					for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
    							Mems[1][x][y][z] += mems[x][y][z][k];
    							//include all the cerebellum in WM mask
    							if (atlas.getNames()[segmentation[x][y][z]].equals("Cerebellum-GM")
    								|| atlas.getNames()[segmentation[x][y][z]].equals("CerebellarGM)"))
    								Mems[3][x][y][z] = 1.0f;
    					}
    				}
    			}

    			return Mems;
    		} // generateDuraRemovalOutputs
	
	/** 
	 *	generate thalamus membership function for MGDM
	 */
	public final float[][][][] generateThalamusOutput() {
		float[][][][]	Mems = new float[1][nx][ny][nz];
		
		// Thalamus
		for (int k=0;k<classes;k++) {
			if (atlas.getNames()[k].equals("Thalamus")) {
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					if (useApprox) {
						for (int a=0;a<approx;a++)
							if (best[x][y][z][a]==k)
								Mems[0][x][y][z] = mems[x][y][z][a];
					} else {
						Mems[0][x][y][z] = mems[x][y][z][k];
					}
				}
			}
		}
		return Mems;
	} // generateThalamusOutput
	

}