package edu.jhu.ece.iacl.algorithms.graphics.intersector;

import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.BBox;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdPoint3;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdSegment;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * A mesh triangle object designed for self-intersection detection.
 * 
 * @author Blake Lucas
 * 
 */
public class SelfIntersectionTriangle {
	// constant used for avoiding numerical accuracy
	public KdPoint3 minPoint, maxPoint;
	public KdPoint3 pts[];
	public int[] ids;
	public int fid;
	public static final double EPS = 5E-5;

	/**
	 * Self-intersection triangle
	 * 
	 * @param id1
	 *            vertex id 1
	 * @param id2
	 *            vertex id 2
	 * @param id3
	 *            vertex id 3
	 * @param fid
	 *            face id
	 * @param mesh
	 *            surface mesh
	 */
	public SelfIntersectionTriangle(int id1, int id2, int id3, int fid,
			EmbeddedSurface mesh) {
		ids = new int[3];
		this.ids[0] = id1;
		this.ids[1] = id2;
		this.ids[2] = id3;
		this.fid = fid;
		pts = new KdPoint3[3];
		minPoint = new KdPoint3();
		maxPoint = new KdPoint3();
		pts[0] = new KdPoint3(mesh.getVertex(id1));
		pts[1] = new KdPoint3(mesh.getVertex(id2));
		pts[2] = new KdPoint3(mesh.getVertex(id3));
		update();
		// Find plane normal
	}

	/**
	 * Constructor
	 */
	public SelfIntersectionTriangle() {
		super();
	}

	/**
	 * test if bounding boxes for two triangles intersect
	 * 
	 * @param test
	 * @return
	 */
	public boolean intersectsBoundingBox(SelfIntersectionTriangle test) {
		double dx = Math.min(test.maxPoint.x, maxPoint.x)
				- Math.max(test.minPoint.x, minPoint.x);
		double dy = Math.min(test.maxPoint.y, maxPoint.y)
				- Math.max(test.minPoint.y, minPoint.y);
		double dz = Math.min(test.maxPoint.z, maxPoint.z)
				- Math.max(test.minPoint.z, minPoint.z);
		return (dx >= 0) && (dy >= 0) && (dz >= 0);
	}

	/**
	 * Test if bounding boxes for two triangles intersect
	 * 
	 * @param minPoint
	 *            min point for bounding box
	 * @param maxPoint
	 *            max point for bounding box
	 * @return
	 */
	public boolean intersectsBoundingBox(Point3f minPoint, Point3f maxPoint) {
		double dx = Math.min(maxPoint.x, maxPoint.x)
				- Math.max(minPoint.x, minPoint.x);
		double dy = Math.min(maxPoint.y, maxPoint.y)
				- Math.max(minPoint.y, minPoint.y);
		double dz = Math.min(maxPoint.z, maxPoint.z)
				- Math.max(minPoint.z, minPoint.z);
		return (dx >= 0) && (dy >= 0) && (dz >= 0);
	}

	/**
	 * Test for intersection of two triangles. This is a port of the triangle
	 * intersection test from WildMagic.
	 * 
	 * @param tri
	 *            triangle
	 * @return true if two triangles intersect
	 */
	public boolean intersects(SelfIntersectionTriangle tri) {
		for (int i = 0; i < 3; i++) {
			for (int j = 0; j < 3; j++) {
				if (ids[i] == tri.ids[j]) {
					if ((ids[(i + 2) % 3] == tri.ids[(j + 1) % 3] || ids[(i + 1) % 3] == tri.ids[(j + 2) % 3])) {
						return false;
					}
				}
			}
		}
		if (!intersectsBoundingBox(tri))
			return false;
		// get edge vectors for triangle0
		Vector3f akE0[] = new Vector3f[3];
		akE0[0] = new Vector3f();
		akE0[0].sub(pts[1], pts[0]);
		akE0[1] = new Vector3f();
		akE0[1].sub(pts[2], pts[1]);
		akE0[2] = new Vector3f();
		akE0[2].sub(pts[0], pts[2]);
		// get normal vector of triangle0
		Vector3f kN0 = new Vector3f();
		kN0.cross(akE0[0], akE0[1]);

		// project triangle1 onto normal line of triangle0, test for separation
		double fN0dT0V0 = GeometricUtilities.dot(kN0, pts[0]);
		double[] fMin1 = new double[1];
		double[] fMax1 = new double[1];
		ProjectOntoAxis(tri, kN0, fMin1, fMax1);
		if (fN0dT0V0 < fMin1[0] || fN0dT0V0 > fMax1[0]) {
			return false;
		}

		// get edge vectors for triangle1
		Vector3f akE1[] = new Vector3f[3];
		akE1[0] = new Vector3f();
		akE1[0].sub(tri.pts[1], tri.pts[0]);
		akE1[1] = new Vector3f();
		akE1[1].sub(tri.pts[2], tri.pts[1]);
		akE1[2] = new Vector3f();
		akE1[2].sub(tri.pts[0], tri.pts[2]);

		// get normal vector of triangle1
		Vector3f kN1 = new Vector3f();
		kN1.cross(akE1[0], akE1[1]);

		Vector3f kDir = new Vector3f();
		double[] fMin0 = new double[1];
		double[] fMax0 = new double[1];
		int i0, i1;

		Vector3f kN0xN1 = new Vector3f();
		kN0xN1.cross(kN0, kN1);
		if (kN0xN1.dot(kN0xN1) >= EPS) {

			// triangles are not parallel

			// Project triangle0 onto normal line of triangle1, test for
			// separation.
			double fN1dT1V0 = GeometricUtilities.dot(kN1, tri.pts[0]);
			ProjectOntoAxis(this, kN1, fMin0, fMax0);
			if (fN1dT1V0 < fMin0[0] + EPS || fN1dT1V0 + EPS > fMax0[0]) {
				return false;
			}

			// directions E0[i0]xE1[i1]
			for (i1 = 0; i1 < 3; i1++) {
				for (i0 = 0; i0 < 3; i0++) {
					kDir.cross(akE0[i0], akE1[i1]);
					ProjectOntoAxis(this, kDir, fMin0, fMax0);
					ProjectOntoAxis(tri, kDir, fMin1, fMax1);
					if (fMax0[0] < fMin1[0] + EPS || fMax1[0] < fMin0[0] + EPS) {
						return false;
					}
				}
			}
		} else // triangles are parallel (and, in fact, coplanar)
		{
			return false;
			// Do not consider case where two triangles are parallel and lie
			// within an epsilon of each other.
			/*
			 * // directions N0xE0[i0] for (i0 = 0; i0 < 3; i0++) {
			 * kDir.cross(kN0,akE0[i0]); ProjectOntoAxis(this,kDir,fMin0,fMax0);
			 * ProjectOntoAxis(tri,kDir,fMin1,fMax1); if (fMax0[0] <fMin1[0]+EPS
			 * || fMax1[0] < fMin0[0]+EPS ) { return false; } }
			 * 
			 * // directions N1xE1[i1] for (i1 = 0; i1 < 3; i1++) {
			 * kDir.cross(kN1,akE1[i1]); ProjectOntoAxis(this,kDir,fMin0,fMax0);
			 * ProjectOntoAxis(tri,kDir,fMin1,fMax1); if (fMax0[0] <
			 * fMin1[0]+EPS || fMax1[0] < fMin0[0]+EPS ) { return false; } }
			 */
		}

		return true;
	}

	/**
	 * Project triangle coordinates onto axis.
	 * 
	 * @param rkTri
	 *            triangle
	 * @param rkAxis
	 *            axis
	 * @param rfMin
	 *            resulting min position along axis
	 * @param rfMax
	 *            resulting max position along axis
	 */
	protected void ProjectOntoAxis(SelfIntersectionTriangle rkTri,
			Vector3f rkAxis, double[] rfMin, double[] rfMax) {
		double fDot0 = GeometricUtilities.dot(rkAxis, rkTri.pts[0]);
		double fDot1 = GeometricUtilities.dot(rkAxis, rkTri.pts[1]);
		double fDot2 = GeometricUtilities.dot(rkAxis, rkTri.pts[2]);
		rfMin[0] = fDot0;
		rfMax[0] = rfMin[0];

		if (fDot1 < rfMin[0]) {
			rfMin[0] = fDot1;
		} else if (fDot1 > rfMax[0]) {
			rfMax[0] = fDot1;
		}

		if (fDot2 < rfMin[0]) {
			rfMin[0] = fDot2;
		} else if (fDot2 > rfMax[0]) {
			rfMax[0] = fDot2;
		}
	}

	/**
	 * Detect if a point is inside a test point
	 */
	public boolean inside(Point3f test) {
		return (test.x >= minPoint.x) && (test.x <= maxPoint.x)
				&& (test.y >= minPoint.y) && (test.y <= maxPoint.y)
				&& (test.z >= minPoint.z) && (test.z <= maxPoint.z);
	}

	public String toString() {
		return ("BBox " + minPoint + " " + maxPoint);
	}

	/**
	 * Update bounding box for triangle
	 */
	public void update() {
		minPoint.x = Math.min(Math.min(pts[0].x, pts[1].x), pts[2].x)
				- (float) EPS;
		minPoint.y = Math.min(Math.min(pts[0].y, pts[1].y), pts[2].y)
				- (float) EPS;
		minPoint.z = Math.min(Math.min(pts[0].z, pts[1].z), pts[2].z)
				- (float) EPS;

		maxPoint.x = Math.max(Math.max(pts[0].x, pts[1].x), pts[2].x)
				+ (float) EPS;
		maxPoint.y = Math.max(Math.max(pts[0].y, pts[1].y), pts[2].y)
				+ (float) EPS;
		maxPoint.z = Math.max(Math.max(pts[0].z, pts[1].z), pts[2].z)
				+ (float) EPS;
	}

	/**
	 * Get normal for triangle
	 * 
	 * @return triangle normal
	 */
	public Vector3f getNormal() {
		Vector3f kEdge1 = new Vector3f();
		Vector3f kEdge2 = new Vector3f();
		Vector3f kNormal = new Vector3f();
		kEdge1.sub(pts[1], pts[0]);
		kEdge2.sub(pts[2], pts[0]);
		kNormal.cross(kEdge1, kEdge2);
		kNormal.normalize();
		return kNormal;
	}

	/**
	 * Get triangle area
	 * 
	 * @return area
	 */
	public float getArea() {
		Vector3f kEdge1 = new Vector3f();
		Vector3f kEdge2 = new Vector3f();
		Vector3f kNormal = new Vector3f();
		kEdge1.sub(pts[1], pts[0]);
		kEdge2.sub(pts[2], pts[0]);
		kNormal.cross(kEdge1, kEdge2);
		return kNormal.length() * 0.5f;
	}

	/**
	 * Get triangle centroid
	 * 
	 * @return centroid
	 */
	public Point3f getCentroid() {
		Point3f p = new Point3f();
		p.add(pts[0]);
		p.add(pts[1]);
		p.add(pts[2]);
		p.scale(0.333333f);
		return p;
	}

	/**
	 * Interpolate position using barycentric coordinates
	 * 
	 * @param b
	 *            barycentric coordinate
	 * @return
	 */
	public Point3f getPointCoords(Point3d b) {
		return new Point3f((float) (pts[0].x * b.x + pts[1].x * b.y + pts[2].x
				* b.z), (float) (pts[0].y * b.x + pts[1].y * b.y + pts[2].y
				* b.z), (float) (pts[0].z * b.x + pts[1].z * b.y + pts[2].z
				* b.z));
	}

	/**
	 * Compute barycentric coordinates
	 * 
	 * @param p
	 *            position on or near triangle
	 * @return barycentric coordinate
	 */
	public Point3d getBaryCoords(Point3f p) {
		double a = pts[0].x - pts[2].x;
		double b = pts[1].x - pts[2].x;
		double c = pts[2].x - p.x;

		double d = pts[0].y - pts[2].y;
		double e = pts[1].y - pts[2].y;
		double f = pts[2].y - p.y;

		double g = pts[0].z - pts[2].z;
		double h = pts[1].z - pts[2].z;
		double i = pts[2].z - p.z;

		double l1 = (b * (f + i) - c * (e + h)) / (a * (e + h) - b * (d + g));
		double l2 = (a * (f + i) - c * (d + g)) / (b * (d + g) - a * (e + h));
		if (Double.isNaN(l1) || Double.isInfinite(l1)) {
			l1 = 0;
		}
		if (Double.isNaN(l2) || Double.isInfinite(l2)) {
			l2 = 0;
		}
		if (l1 > 1 || l2 > 1 || l1 + l2 > 1 || l1 < 0 || l2 < 0) {
			// System.err.println("Barycentric Coordinate Invalid R1:"+r1+" R2:"+r2+" R3:"+r3+" P:"+p);
			l1 = Math.max(Math.min(l1, 1), 0);
			l2 = Math.max(Math.min(l2, 1), 0);
			if (l1 + l2 > 1) {
				double diff = 0.5 * (1 - l1 - l2);
				l1 += diff;
				l2 += diff;
			}
		}
		return new Point3d(l1, l2, 1 - l1 - l2);
	}

}