package edu.jhu.ece.iacl.algorithms.graphics.locator.balltree;

import java.util.LinkedList;
import java.util.List;

import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3d;

import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.utilities.quickhull.QPoint3;

/**
 * A ball data structure used as part of a tree structure.
 * 
 * @author Blake Lucas
 * 
 */
public abstract class Ball extends QPoint3 implements Comparable<Ball> {
	protected Ball parent = null;
	protected static final double EPS = 1E-6;

	public Ball(double x, double y, double z) {
		super(x, y, z);
	}

	public Ball(Point3d p) {
		super(p.x, p.y, p.z);
	}

	public Ball(Point3f p) {
		super(p.x, p.y, p.z);
	}

	public Ball() {
		super();
	}

	/**
	 * Get number of balls in tree
	 * 
	 * @return number of balls
	 */
	public int getTreeSize() {
		int sz = 1;
		for (Ball b : getChildren()) {
			sz += getTreeSize();
		}
		return sz;
	}

	/**
	 * Get all descendant balls and append them to list
	 * 
	 * @param collection
	 */
	public void getAllDescendants(List<Ball> list) {
		list.add(this);
		if (getChildren() == null)
			return;
		for (Ball b : getChildren()) {
			b.getAllDescendants(list);
		}
	}

	/**
	 * Get all descendants rooted at this ball.
	 * 
	 * @return list of all descendants
	 */
	public LinkedList<Ball> getAllDescendants() {
		LinkedList<Ball> list = new LinkedList<Ball>();
		getAllDescendants(list);
		return list;
	}

	/**
	 * Return true if leaf ball
	 * 
	 * @return true if leaf
	 */
	public abstract boolean isLeaf();

	/**
	 * Get tree depth for ball
	 * 
	 * @return depth
	 */
	public abstract int getDepth();

	/**
	 * Set tree depth for ball
	 * 
	 * @param d
	 *            depth
	 */
	public abstract void setDepth(int d);

	/**
	 * Set ball radius
	 * 
	 * @return radius
	 */
	public abstract double getRadius();

	/**
	 * Get all children for ball
	 * 
	 * @return list of all children
	 */
	public abstract List<Ball> getChildren();

	/**
	 * Update descendants after leaf positions have changed
	 */
	public abstract void updateDescendants();

	/**
	 * Update ancestors after lead positions have changed
	 */
	public abstract void updateAncestors();

	/**
	 * Factor to compute volume of ball
	 */
	protected static final double VOLUME_SCALE = (4 / 3.0) * Math.PI;

	/**
	 * Return ball volume.
	 * 
	 * @return
	 */
	public double volume() {
		double radius = getRadius();
		return VOLUME_SCALE * radius * radius * radius;
	}

	/**
	 * Determine if ball intersects this ball.
	 * 
	 * @param test
	 *            ball to test for intersection
	 * @return Return true if ball intersects this ball.
	 */
	public boolean intersects(Ball test) {
		return (test.distance(this) <= getRadius() + test.getRadius());
	}

	/**
	 * Determine if ball is completely contained in this ball
	 * 
	 * @param test
	 *            ball to test for containment
	 * @return true if this ball contains ball
	 */
	public boolean contains(Ball test) {
		return (test.distance(this) <= getRadius() - test.getRadius());
	}

	/**
	 * Check if all children balls are contained in this ball
	 * 
	 * @return true if this ball contains all its children
	 */
	public boolean containmentCheck() {
		if (isLeaf())
			return true;
		for (Ball b : getChildren()) {
			if (!this.contains(b)) {
				System.out.println("CHECK FAILED\n" + this + "\n" + b);
				return false;
			}
		}
		for (Ball b : getChildren()) {
			if (!b.containmentCheck()) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Detect if a point is inside this ball
	 * 
	 * @param test
	 *            point location
	 * @return true if this ball contains a point
	 */
	public boolean contains(Point3d test) {
		return (test.distance(this) <= getRadius());
	}

	/**
	 * Calculate distance from point to this ball
	 * 
	 * @param p
	 * @return
	 */
	public double distanceToSphere(Point3d p) {
		return Math.max(p.distance(this) - getRadius(), 0);
	}

	/**
	 * Compare the size of two balls by volume
	 */
	public int compareTo(Ball ball) {
		return (int) Math.signum(this.volume() - ball.volume());
	}

	public String toString() {
		return ("[Ball " + this.getClass().getSimpleName() + " (" + this.x
				+ "," + this.y + "," + this.z + ") Radius=" + getRadius()
				+ " Depth=" + getDepth() + " Children="
				+ ((isLeaf()) ? 0 : getChildren().size()) + "]");
	}

	/**
	 * Project ball onto vector and return the minimum position of its children
	 * along that vector
	 * 
	 * @param v
	 *            vector
	 * @return minimum position of children along axis
	 */
	public double getMin(Vector3d v) {
		double minR = Double.MAX_VALUE;
		for (Ball b : getChildren()) {
			minR = Math.min(minR, GeometricUtilities.dot(b, v) - b.getRadius());
		}
		return minR;
	}

	/**
	 * Project ball onto vector and return the maximum position of its children
	 * along that vector
	 * 
	 * @param v
	 *            vector
	 * @return maximum position of children along axis
	 */

	public double getMax(Vector3d v) {
		double maxR = Double.MIN_VALUE;
		for (Ball b : getChildren()) {
			maxR = Math.max(maxR, GeometricUtilities.dot(b, v) + b.getRadius());
		}
		return maxR;
	}

	/**
	 * Get parent ball
	 * 
	 * @return parent
	 */
	public Ball getParent() {
		return parent;
	}

	/**
	 * Set parent
	 * 
	 * @param parent
	 *            parent ball
	 */
	public void setParent(Ball parent) {
		this.parent = parent;
	}

}
