package edu.jhu.ece.iacl.algorithms.graphics.locator.sphere;

import java.util.LinkedList;

import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.EmbeddedSurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.EmbeddedTriangle;
import edu.jhu.ece.iacl.algorithms.graphics.utilities.quickhull.QuickHull3D;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Interpolate data by intersecting ray against convex hull of a surface. It's
 * assumed that all vertices lie on a sphere.
 * 
 * @author Blake Lucas
 * 
 */
public class ConvexHullLocator extends AbstractCalculation {
	protected EmbeddedSurfaceIntersector intersector;
	protected EmbeddedSurface hull;
	protected int[] ptMap;
	public static String getVersion(){
		return VersionUtil.parseRevisionNumber("$Revision: 1.1 $");
	}
	/**
	 * Constructor
	 */
	public ConvexHullLocator() {
		super();
	}

	/**
	 * Constructor
	 * 
	 * @param parent
	 *            parent calculation
	 */
	public ConvexHullLocator(AbstractCalculation parent) {
		super(parent);
	}

	/**
	 * Constructor
	 * 
	 * @param surf
	 *            surface
	 */
	public ConvexHullLocator(EmbeddedSurface surf) {
		super();
		init(surf);
	}

	public ConvexHullLocator(AbstractCalculation parent, EmbeddedSurface surf) {
		super(parent);
		init(surf);
	}

	/**
	 * Initialize convex hull with vertices from a surface. The surface is
	 * assumed to be a spherical parameterization.
	 * 
	 * @param surf
	 *            surface
	 */
	public void init(EmbeddedSurface surf) {
		init(surf.getVertexDoubleCopy());
		if (surf.getVertexData() != null)
			setEmbeddedData(surf.getVertexData());
	}

	/**
	 * Initialize convex hull with points that are stored as triplets in a 1D
	 * array.
	 * 
	 * @param ptsArray
	 *            points array
	 */
	public void init(double[] ptsArray) {
		QuickHull3D qhull = new QuickHull3D();
		qhull.build(ptsArray, ptsArray.length / 3);
		hull = qhull.getSurface(0);
		ptMap = qhull.getVertexPointIndices();
		intersector = new EmbeddedSurfaceIntersector(this, 12, hull);
	}

	/**
	 * Set embedded data for convex data
	 * 
	 * @param origData
	 *            vertex data
	 */
	public void setEmbeddedData(double[][] origData) {
		double[][] data = new double[ptMap.length][origData[0].length];
		for (int i = 0; i < ptMap.length; i++) {
			data[i] = origData[ptMap[i]];
		}
		hull.setVertexData(data);
	}

	/**
	 * Interpolate embedded vector by casting ray in vector direction
	 * 
	 * @param q
	 *            direction
	 * @param offset
	 *            offset into embedded vertex data
	 * @return interpolated vector
	 */
	public Vector3f interpolateEmbeddedVector(Point3f q, int offset) {
		double d = intersector.intersectRayDistance(new Point3f(),
				new Vector3f(q));
		Point3f pt = intersector.getLastIntersectionPoint();
		EmbeddedTriangle tri = intersector.getLastIntersectionTriangle();
		Point3d b = tri.getBaryCoords(pt);
		Vector3f vec = new Vector3f();
		vec.x = (float) tri.getScalar(hull, b, offset);
		vec.y = (float) tri.getScalar(hull, b, offset + 1);
		vec.z = (float) tri.getScalar(hull, b, offset + 2);
		return vec;
	}

	/**
	 * Interpolate embedded scalar by casting ray in vector direction
	 * 
	 * @param q
	 *            direction
	 * @param offset
	 *            offset into embedded vertex data
	 * @return interpolated scalar
	 */
	public double interpolateEmbeddedScalar(Point3f q, int offset) {
		double d = intersector.intersectRayDistance(new Point3f(),
				new Vector3f(q));
		Point3f pt = intersector.getLastIntersectionPoint();
		EmbeddedTriangle tri = intersector.getLastIntersectionTriangle();
		Point3d b = tri.getBaryCoords(pt);
		return tri.getScalar(hull, b, offset);
	}
}
