package edu.jhu.ece.iacl.algorithms.graphics.map;

import java.util.ArrayList;
import java.util.List;
import javax.vecmath.Point2d;
import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.edit.RemeshSurfacesUsingSphericalMap;
import edu.jhu.ece.iacl.algorithms.graphics.locator.sphere.SphericalMapLocator;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.CurveCollection;
import edu.jhu.ece.iacl.jist.structures.geom.CurvePath;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Remap surfaces that are implicitly mapped to each other via a spherical map.
 * The remapping is performed both source->target and target->source. Connectors
 * are generated to indicate corresponding vertices.
 * 
 * @author Blake Lucas
 * 
 */
public class CoupleSphericallyMappedSurfaces extends AbstractCalculation {
	protected CurveCollection paths;

	protected ArrayList<EmbeddedSurface> remapedSurfs;

	public CoupleSphericallyMappedSurfaces() {
		super();
		paths = new CurveCollection();
	}

	public CoupleSphericallyMappedSurfaces(AbstractCalculation parent) {
		super(parent);
		paths = new CurveCollection();
	}

	public CurveCollection getConnectors() {
		return paths;
	}

	public ArrayList<EmbeddedSurface> getRemeshedSurfaces() {
		return remapedSurfs;
	}

	/**
	 * Generate connectors between spherically mapped surfaces.
	 * 
	 * @param surfs
	 *            list of surface
	 * @param offset
	 * @param minEdgeLength
	 * @param useConnectors
	 */
	public void solve(List<EmbeddedSurface> surfs, int offset,
			double minEdgeLength, boolean useConnectors) {
		remapedSurfs = new ArrayList<EmbeddedSurface>(surfs.size() - 1);
		EmbeddedSurface source = surfs.get(0);
		EmbeddedSurface target = surfs.get(surfs.size() - 1);
		setLabel("Couple Surfaces");
		RemeshSurfacesUsingSphericalMap rs = new RemeshSurfacesUsingSphericalMap(
				this, source, target, offset);
		rs.remap();
		// rs.decimate(minEdgeLength);
		EmbeddedSurface remapedSource = rs.getRemeshedSource();
		EmbeddedSurface remapedTarget = rs.getRemeshedTarget();
		remapedSurfs.add(remapedSource);
		double val, minT = 1, maxT = 0, maxP = 0, minP = 1;
		CurvePath path;
		setTotalUnits(surfs.size());
		int vertCount = remapedSource.getVertexCount();
		double sphereCoords[][] = new double[vertCount][2];
		double[][] lengths = new double[vertCount][1];
		if (useConnectors) {
			for (int i = 0; i < vertCount; i++) {
				path = new CurvePath();
				path.add(remapedSource.getVertex(i));
				paths.add(path);
			}
		}
		for (int id = 0; id < vertCount; id++) {
			lengths[id][0] = 0;
			Point3f p = remapedSource.getPointAtOffset(id, offset);
			Point2d r = GeometricUtilities.toUnitSpherical(p);
			sphereCoords[id][0] = val = Math.abs(r.x / Math.PI);
			minT = Math.min(minT, val);
			maxT = Math.max(maxT, val);
			sphereCoords[id][1] = val = Math.abs(r.y / Math.PI);
			minP = Math.min(minP, val);
			maxP = Math.max(maxP, val);
		}
		/*
		 * for(int id=0;id<vertCount;id++){
		 * sphereCoords[id][0]=(sphereCoords[id][0]-minT)/(maxT-minT);
		 * sphereCoords[id][1]=(sphereCoords[id][1]-minP)/(maxP-minP); }
		 */
		incrementCompletedUnits();
		for (int i = 1; i < surfs.size() - 1; i++) {

			remapedSurfs.add(coupleSurfaces(remapedSource, surfs.get(i),
					offset, false));
			source = remapedSurfs.get(i - 1);
			target = remapedSurfs.get(i);
			System.out.println("ITERMEDIATE " + source.getName() + " "
					+ target.getName());
			for (int id = 0; id < vertCount; id++) {
				((CurvePath) paths.get(id)).add(target.getVertex(id));
				lengths[id][0] += source.getVertex(id).distance(
						target.getVertex(id));
			}
			incrementCompletedUnits();
		}

		remapedSurfs.add(remapedTarget);

		source = remapedSurfs.get(surfs.size() - 2);
		target = remapedTarget;

		System.out.println("LAST ITERATION " + source.getName() + " "
				+ target.getName());
		for (int id = 0; id < vertCount; id++) {
			lengths[id][0] += source.getVertex(id).distance(
					target.getVertex(id));
		}
		if (useConnectors) {
			for (int i = 0; i < vertCount; i++) {
				path = (CurvePath) paths.get(i);
				(path).add(remapedTarget.getVertex(i));
			}
		}

		incrementCompletedUnits();

		if (useConnectors) {
			paths.setName(surfs.get(surfs.size() - 1).getName() + "_lines");
		}
		EmbeddedSurface targetSphere = remapedTarget.getEmbeddedSphere(offset,
				false);
		for (int i = 0; i < surfs.size(); i++) {
			source = remapedSurfs.get(i);
			System.out.println("SURFACE " + i + " " + source.getName());
			source.setVertexData(lengths);
			source.setTextureCoordinates(sphereCoords);
		}
		targetSphere.setVertexData(lengths);
		targetSphere.setTextureCoordinates(sphereCoords);
		remapedSurfs.add(targetSphere);
		markCompleted();
	}

	public enum SurfParam {
		INNER, CENTRAL, OUTER
	};

	/**
	 * Solve for mapping between inner and outer surfaces using inner spherical
	 * map and outer spherical map
	 * 
	 * @param innerInner
	 *            Inner surface with inner spherical map
	 * @param innerOuter
	 *            Outer surface with inner spherical map
	 * @param outerInner
	 *            Inner surface with outer spherical map
	 * @param outerOuter
	 *            Outer surface with outer spherical map
	 * @param param
	 *            Use parameterization from either inner or outer surface
	 * @param offset
	 *            offset into vertex data to retrieve spherical coordinates
	 */
	public void solve(EmbeddedSurface innerInner, EmbeddedSurface innerOuter,
			EmbeddedSurface outerInner, EmbeddedSurface outerOuter,
			SurfParam param, int offset) {
		EmbeddedSurface remapedInner = null, remapedOuter = null;
		EmbeddedSurface remapedInner2 = null, remapedOuter2 = null;
		remapedSurfs = new ArrayList<EmbeddedSurface>(2);
		Point3f sp1, sp2, tp;
		CurvePath path;
		double[][] lens;

		int vertCount;
		if (param == SurfParam.OUTER) {
			remapedOuter = outerOuter.clone();
			vertCount = outerOuter.getVertexCount();
			lens = new double[vertCount][1];

			paths.setName(outerOuter.getName() + "_lines");
			remapedOuter.setName(outerOuter.getName() + "_mapped");
			remapedInner = coupleSurfaces(innerOuter, innerInner, offset, false);
			remapedInner2 = coupleSurfaces(outerOuter, outerInner, offset,
					false);
			for (int id = 0; id < vertCount; id++) {
				sp1 = remapedInner.getVertex(id);
				sp2 = remapedInner2.getVertex(id);
				tp = outerOuter.getVertex(id);
				path = new CurvePath();
				if (sp2.distance(tp) < sp1.distance(tp)) {
					remapedInner.setVertex(id, sp2);
					path.add(sp2);
				} else {
					path.add(sp1);
				}
				path.add(tp);
				paths.add(path);
				lens[id][0] = path.getValue();
			}
			int faceCount = remapedOuter.getFaceCount();
			double[][] areas = new double[faceCount][1];
			for (int fid = 0; fid < faceCount; fid++) {
				areas[fid][0] = remapedInner.getFaceArea(fid);
			}
			remapedInner.setCellData(areas);
			remapedOuter.setCellData(areas);

			remapedInner.setVertexData(lens);
			remapedOuter.setVertexData(lens);
		}
		if (param == SurfParam.INNER) {
			remapedInner = outerInner.clone();
			vertCount = outerInner.getVertexCount();
			lens = new double[vertCount][1];
			paths.setName(outerInner.getName() + "_lines");
			remapedInner.setName(outerInner.getName() + "_mapped");
			remapedOuter = coupleSurfaces(innerInner, innerOuter, offset, false);
			remapedOuter2 = coupleSurfaces(outerInner, outerOuter, offset,
					false);
			for (int id = 0; id < vertCount; id++) {
				sp1 = remapedOuter.getVertex(id);
				sp2 = remapedOuter2.getVertex(id);
				tp = outerInner.getVertex(id);
				path = new CurvePath();
				path.add(tp);
				if (sp2.distance(tp) < sp1.distance(tp)) {
					remapedOuter.setVertex(id, sp2);
					path.add(sp2);
				} else {
					path.add(sp1);
				}
				paths.add(path);
				lens[id][0] = path.getValue();
			}
			int faceCount = remapedOuter.getFaceCount();
			double[][] areas = new double[faceCount][1];
			for (int fid = 0; fid < faceCount; fid++) {
				areas[fid][0] = remapedOuter.getFaceArea(fid);
			}
			remapedInner.setCellData(areas);
			remapedOuter.setCellData(areas);

			remapedInner.setVertexData(lens);
			remapedOuter.setVertexData(lens);

		}
		remapedSurfs.add(remapedInner);
		remapedSurfs.add(remapedOuter);
	}

	/*
	 * public void solveTripleMapping(EmbeddedSurface inner, EmbeddedSurface
	 * central, EmbeddedSurface outer, SurfParam param, int offset) {
	 * EmbeddedSurface remapedCentral = null, remapedInner = null, remapedOuter
	 * = null; remapedSurfs = new ArrayList<EmbeddedSurface>(3);
	 * 
	 * switch (param) { case INNER: remapedInner = inner.clone(); remapedCentral
	 * = coupleSurfaces(inner, central, offset, false); remapedOuter =
	 * coupleSurfaces(inner, outer, offset, false);
	 * paths.setName(remapedInner.getName() + "_lines");
	 * remapedInner.setName(inner.getName() + "_mapped"); break; case CENTRAL:
	 * remapedInner = coupleSurfaces(central, inner, offset, false);
	 * remapedCentral = central.clone(); remapedOuter = coupleSurfaces(central,
	 * outer, offset, false); paths.setName(remapedCentral.getName() +
	 * "_lines"); remapedCentral.setName(central.getName() + "_mapped"); break;
	 * case OUTER: remapedInner = coupleSurfaces(outer, inner, offset, false);
	 * remapedCentral = coupleSurfaces(outer, central, offset, false);
	 * remapedOuter = outer.clone(); paths.setName(remapedOuter.getName() +
	 * "_lines"); remapedOuter.setName(outer.getName() + "_mapped"); break; }
	 * int vertCount = remapedInner.getVertexCount(); CurvePath path;
	 * remapedSurfs.add(remapedInner); remapedSurfs.add(remapedCentral);
	 * remapedSurfs.add(remapedOuter); EmbeddedSurface targetSphere =
	 * remapedOuter.getEmbeddedSphere(offset, false);
	 * remapedSurfs.add(targetSphere); targetSphere.setVertexData(new
	 * double[vertCount][1]); double len = 0; for (int id = 0; id < vertCount;
	 * id++) { path = new CurvePath(); path.add(remapedInner.getVertex(id)); //
	 * path.add(remapedCentral.getVertex(id));
	 * path.add(remapedOuter.getVertex(id)); paths.add(path); len =
	 * path.getValue(); remapedInner.setVertexData(id, 0, len);
	 * remapedCentral.setVertexData(id, 0, len); remapedOuter.setVertexData(id,
	 * 0, len); targetSphere.setVertexData(id, 0, len); } }
	 */
	/**
	 * Solve for mapping between inner and outer surfaces and use vertex data
	 * associated with lookup surface
	 */
	public void solveDoubleMapping(EmbeddedSurface inner,
			EmbeddedSurface outer, EmbeddedSurface lookup, SurfParam param,
			int offset, boolean createPaths) {
		EmbeddedSurface remapedCentral = null, remapedInner = null, remapedOuter = null;
		remapedSurfs = new ArrayList<EmbeddedSurface>(2);
		switch (param) {
		case INNER:
			remapedInner = inner.clone();
			remapedOuter = coupleSurfaces(inner, outer, lookup, offset, false);

			remapedInner.setName(inner.getName() + "_mapped");
			remapedInner.setVertexData(remapedOuter.getVertexData());
			paths = new CurveCollection();
			paths.setName(remapedInner.getName() + "_lines");
			break;
		case OUTER:
			remapedInner = coupleSurfaces(outer, inner, lookup, offset, false);
			remapedOuter = outer.clone();
			remapedOuter.setName(outer.getName() + "_mapped");
			remapedOuter.setVertexData(remapedInner.getVertexData());
			paths = new CurveCollection();
			paths.setName(remapedInner.getName() + "_lines");
			break;
		}
		int vertCount = remapedInner.getVertexCount();
		CurvePath path;
		remapedSurfs.add(remapedInner);
		remapedSurfs.add(remapedOuter);
		double len = 0;
		System.out.println("CREATING PATHS " + vertCount + " " + paths.size());
		if (createPaths) {
			setTotalUnits(vertCount);
			for (int id = 0; id < vertCount; id++) {
				path = new CurvePath();
				path.add(remapedInner.getVertex(id));
				// path.add(remapedCentral.getVertex(id));
				path.add(remapedOuter.getVertex(id));
				paths.add(path);
				incrementCompletedUnits();
			}
		}
		markCompleted();
	}
	/**
	 * Couple surfaces using spherical maps embedded in both surfaces
	 * @param source source surface
	 * @param target target surface
	 * @param lookup surface to copy vertex data into remaped surfaces
	 * @param offset offset into vertex data to retrieve spherical coordinates
	 * @param appendPath append connectors to collection of paths
	 * @return remaped target with parameterization of source
	 */
	private EmbeddedSurface coupleSurfaces(EmbeddedSurface source,
			EmbeddedSurface target, EmbeddedSurface lookup, int offset,
			boolean appendPath) {
		SphericalMapLocator targetLocator = new SphericalMapLocator(this,
				target, offset);
		int vertSourceCount = source.getVertexCount();
		Point3f tp;
		Point3f[] pts = new Point3f[vertSourceCount];
		double[][] data = new double[vertSourceCount][2];
		setTotalUnits(vertSourceCount);
		boolean lookupIsSource = (lookup.getVertexCount() == vertSourceCount);
		for (int i = 0; i < vertSourceCount; i++) {
			// Get embedded point on target
			Point3f p = source.getPointAtOffset(i, 0);
			// locate point on source
			tp = targetLocator.locatePoint(p);
			pts[i] = tp;
			data[i][0] = source.getVertex(i).distance(tp);
			data[i][1] = (lookupIsSource) ? lookup.getVertexDataAtOffset(i,
					offset) : targetLocator.getLastIntersectionTriangle()
					.mapToScalar(lookup, tp, offset);
			if (appendPath && paths.size() > 0) {
				((CurvePath) paths.get(i)).add(tp);
			}
			incrementCompletedUnits();
		}
		EmbeddedSurface remapedSurf = new EmbeddedSurface(pts, source
				.getIndexCopy(), data);
		remapedSurf.setName(target.getName() + "_mapped");
		return remapedSurf;
	}
	/**
	 * Remap surfaces using their embedded spherical maps
	 * @param source source surface
	 * @param target target surface
	 * @param offset offset into vertex data to retrieve spherical coordinates
	 * @param appendPath append connectors to curve paths
	 * @return remaped target
	 */
	private EmbeddedSurface coupleSurfaces(EmbeddedSurface source,
			EmbeddedSurface target, int offset, boolean appendPath) {
		SphericalMapLocator targetLocator = new SphericalMapLocator(this,
				target, offset);
		int vertSourceCount = source.getVertexCount();
		Point3f tp;
		Point3f[] pts = new Point3f[vertSourceCount];
		setTotalUnits(vertSourceCount);
		for (int i = 0; i < vertSourceCount; i++) {
			// Get embedded point on target
			Point3f p = source.getPointAtOffset(i, offset);
			// locate point on source
			tp = targetLocator.locatePoint(p);
			pts[i] = tp;
			if (appendPath && paths.size() > 0) {
				((CurvePath) paths.get(i)).add(tp);
			}
			incrementCompletedUnits();
		}
		EmbeddedSurface remapedSurf = new EmbeddedSurface(pts, source
				.getIndexCopy(), source.getVertexData());
		remapedSurf.setName(target.getName() + "_mapped");
		return remapedSurf;
	}
}
