package edu.jhu.ece.iacl.algorithms.graphics.map;

import javax.vecmath.*;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.XuGraphicsWrapper;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.IntersectorTriangle;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Generate spherical map of surface by solving for planar conformal mapping and
 * then stereographically projecting the surface to a sphere. 
 * The algorithm crashes if the surface contains degenerate triangles.
 * The algorithm is bijective in vertice space, but is not a
 * continuous bijection.
 * 
 * @author Blake Lucas
 * 
 */
public class SurfaceToComplex extends XuGraphicsWrapper {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.1 $");
	}

	private static final int avgIters = 5000;
	private static final int imax = 20000;// max iteration number for
	// conjugate gradient",

	private static final double error = 1e-6; // error tolerance for conjugate
	// gradient",

	private static final int update = 50;

	private static final double scaleIncrement = 250;

	private static final double minScale = 100;

	private static final double maxScale = 20000;
	private static boolean regularize = false;


	public SurfaceToComplex(EmbeddedSurface mesh) {
		super(mesh, true);

		mesh.repairDegenerateTriangles(0.0005f, 0.001f);
		this.setLabel("Map Surface to Sphere");
	}


	public SurfaceToComplex(AbstractCalculation parent, EmbeddedSurface mesh) {
		super(parent, mesh, true);
		mesh.repairDegenerateTriangles(0.0005f, 0.001f);
		this.setLabel("Map Surface to Sphere");
	}


	private double mysquare(double val) {
		return val * val;
	}


	/**
	 * Regularize surface by snapping vertices to grid points
	 * @param reg true if regularize
	 */
	public void setRegularize(boolean reg) {
		regularize = reg;
	}


	/**
	 * Locate north pole triangle by casting a ray in Z direction from reference point
	 * @param p reference point
	 * @return face id
	 */
	public int locateNorthPole(Point3f p) {
		int inCount = surf.getIndexCount();
		IntersectorTriangle tri = null;
		Vector3f vpos = new Vector3f(0, 0, 1);
		Vector3f vneg = new Vector3f(0, 0, -1);
		int index = -1;
		for (int i = 0; i < inCount / 3; i++) {
			tri = getTriangle(surf, i);
			if (tri.intersectionPoint(p, vneg) != null
					|| tri.intersectionPoint(p, vpos) != null) {
				index = i;
				break;
			}
		}
		if (index == -1) {
			System.err.println("Could not locate north pole from point " + p);
			p = surf.getCenterOfMass();
			for (int i = 0; i < inCount / 3; i++) {
				tri = getTriangle(surf, i);
				if (tri.intersectionPoint(p, vneg) != null
						|| tri.intersectionPoint(p, vpos) != null) {
					index = i;
					break;
				}
			}
			if (index == -1) {
				System.err
						.println("Still could not locate north pole from center of mass "
								+ p);
			} else {
				System.err.println("Located north pole from center of mass "
						+ p);
			}
		}
		return index;
	}


	/**
	 * Get intersection triangle for specified face id
	 * @param surf surface
	 * @param i face id
	 * @return intersection triangle
	 */
	static public IntersectorTriangle getTriangle(EmbeddedSurface surf, int i) {
		IntersectorTriangle tri = new IntersectorTriangle(surf
				.getCoordinateIndex(3 * i), surf.getCoordinateIndex(3 * i + 1),
				surf.getCoordinateIndex(3 * i + 2), i, surf);
		return tri;
	}


	/**
	 * Map surface to spherical coordinates
	 * @param p reference point
	 * @return spherical map in spherical coordinates
	 */
	public double[][] mapToSphericalCoordinates(Point3f p) {
		return mapToSphericalCoordinates(locateNorthPole(p));
	}


	/**
	 * Map surface to cartesian coordinates
	 * @param p reference point
	 * @return spherical map in cartesian coordinates
	 */
	public double[][] mapToCartesianCoordinates(Point3f p) {
		return mapToCartesianCoordinates(locateNorthPole(p));
	}


	/**
	 * Map surface to spherical coordinates
	 * @param n_poly north pole face id
	 * @return spherical map in spherical coordinates
	 */
	public double[][] mapToSphericalCoordinates(int n_poly) {
		double[][] cart = mapToCartesianCoordinates(n_poly);
		if (cart == null)
			return null;
		double x, y, z, theta, phi;
		for (int i = 0; i < cart.length; i++) {
			x = cart[i][0];
			y = cart[i][1];
			z = cart[i][2];
			theta = Math.atan2(y, x);
			phi = Math.acos(z);
			cart[i] = new double[] { theta, phi };
		}
		return cart;
	}


	/**
	 * Map surface to cartesian coordinates
	 * @param n_poly north pole face id
	 * @return spherical map in cartesian coordinates
	 */
	public double[][] mapToCartesianCoordinates(int n_poly) {
		double[][] cmplx = mapToComplexCoordinates(n_poly);
		if (cmplx == null)
			return null;
		double[][] cartesian = new double[cmplx.length][3];
		System.out.println("START POLYGON " + n_poly);
		double u, v, wsqr, x, y, z;
		;
		double lastSpread = 0;
		double bestScale = minScale;
		double spread;
		double phi = 0;
		double sum = 0;
		double sqrs = 0;
		for (double scale = minScale; scale <= maxScale; scale += scaleIncrement) {
			sum = 0;
			sqrs = 0;
			for (int i = 0; i < cmplx.length; i++) {
				u = scale * cmplx[i][0];
				v = scale * cmplx[i][1];
				wsqr = u * u + v * v;
				phi = Math.acos((wsqr - 4) / (wsqr + 4));
				sum += phi;
				sqrs += phi * phi;
			}
			spread = Math.sqrt((sqrs - sum * sum / cmplx.length)
					/ (cmplx.length - 1));
			// System.out.println("SCALE " + scale + " SPREAD " + spread);
			if (spread < lastSpread) {
				break;
			} else {
				lastSpread = spread;
				bestScale = scale;
			}
		}
		System.out.println("BEST SCALE " + bestScale + " SPREAD " + lastSpread);
		for (int i = 0; i < cmplx.length; i++) {
			u = bestScale * cmplx[i][0];
			v = bestScale * cmplx[i][1];
			wsqr = u * u + v * v;
			x = 4 * u / (wsqr + 4);
			y = 4 * v / (wsqr + 4);
			z = (wsqr - 4) / (wsqr + 4);
			cartesian[i][0] = x;
			cartesian[i][1] = y;
			cartesian[i][2] = -z;
		}
		return cartesian;
	}


	/**
	 * Map surface to complex coordinates
	 * @param n_poly north pole face id
	 * @return spherical map in complex coordinates
	 */
	public double[][] mapToComplexCoordinates(int n_poly) {
		int j;
		// check the three neighboring triangles of the north pole triangle
		// want to pick the one that is closer to be an equilateral triangle

		// North Pole default

		double[] angles = new double[3];
		double stdevN, stdevS;
		Vector3f a, b, c, CA, BA;
		int vida, vidb, vidc;
		double magb, magc, I;
		vida = XUGetPolyVertexId(n_poly, 0);
		a = XUGetVertex(vida);
		vidb = XUGetPolyVertexId(n_poly, 1);
		b = XUGetVertex(vidb);
		vidc = XUGetPolyVertexId(n_poly, 2);
		c = XUGetVertex(vidc);
		CA = XUVecSub(c, a);
		magb = XUVecMag(CA);
		BA = XUVecSub(b, a);
		magc = XUVecMag(BA);
		I = XUVecDot(CA, BA);
		angles[0] = Math.acos(I / (magb * magc));
		CA = XUVecSub(a, b);
		magb = XUVecMag(CA);
		BA = XUVecSub(c, b);
		magc = XUVecMag(BA);
		I = XUVecDot(CA, BA);
		angles[1] = Math.acos(I / (magb * magc));
		CA = XUVecSub(b, c);
		magb = XUVecMag(CA);
		BA = XUVecSub(a, c);
		magc = XUVecMag(BA);
		I = XUVecDot(CA, BA);
		angles[2] = Math.acos(I / (magb * magc));
		stdevN = Math.sqrt((mysquare(angles[0] - Math.PI / 3)
				+ mysquare(angles[1] - Math.PI / 3) + mysquare(angles[2]
				- Math.PI / 3)) / 2);
		System.out.printf("n-poly's angle stdev : %g ---> ", stdevN);
		int i;
		int k = n_poly;
		int s_poly;
		for (i = 0; i < 3; ++i) {
			s_poly = XUGetPolyNeighborPoly(n_poly, i);
			vida = XUGetPolyVertexId(s_poly, 0);
			a = XUGetVertex(vida);
			vidb = XUGetPolyVertexId(s_poly, 1);
			b = XUGetVertex(vidb);
			vidc = XUGetPolyVertexId(s_poly, 2);
			c = XUGetVertex(vidc);
			CA = XUVecSub(c, a);
			magb = XUVecMag(CA);
			BA = XUVecSub(b, a);
			magc = XUVecMag(BA);
			I = XUVecDot(CA, BA);
			angles[0] = Math.acos(I / (magb * magc));
			CA = XUVecSub(a, b);
			magb = XUVecMag(CA);
			BA = XUVecSub(c, b);
			magc = XUVecMag(BA);
			I = XUVecDot(CA, BA);
			angles[1] = Math.acos(I / (magb * magc));
			CA = XUVecSub(b, c);
			magb = XUVecMag(CA);
			BA = XUVecSub(a, c);
			magc = XUVecMag(BA);
			I = XUVecDot(CA, BA);
			angles[2] = Math.acos(I / (magb * magc));
			stdevS = Math.sqrt((mysquare(angles[0] - Math.PI / 3)
					+ mysquare(angles[1] - Math.PI / 3) + mysquare(angles[2]
					- Math.PI / 3)) / 2);
			if (stdevS < stdevN)
				k = s_poly;
		}

		n_poly = k;

		int VN = surf.getVertexCount();

		if (regularize) {
			for (i = 0; i < VN; ++i) {
				Vector3f V = XUGetVertex(i);
				V.x = (float) (Math.floor(V.x) + Math.ceil(V.x - (int) (V.x)) * 0.5f);
				V.y = (float) (Math.floor(V.y) + Math.ceil(V.y - (int) (V.y)) * 0.5f);
				V.z = (float) (Math.floor(V.z) + Math.ceil(V.z - (int) (V.z)) * 0.5f);
				surf.setVertex(i, V);
			}
		}

		// calculate the number of non-zero elements
		int maxnum;
		maxnum = 0;
		for (i = 0; i < VN; ++i)
			maxnum += XUGetNeighborEN(i);
		maxnum += VN;
		double diag, offdiag; // dumy for computation of matrix entries
		int vid, vid1, vid2, eid, pid, temp, EN;
		Vector3f P, Q, R, PQ, PR, QR;
		EmbeddedSurface.Edge e;
		double magPR, magQR, cosr, cotr;
		EmbeddedSurface.Edge[] edgelist;
		int[] row_start = new int[VN + 1];
		int[] col_idx = new int[maxnum];
		double[] value = new double[maxnum];
		// printf("initialize matrix ... ");
		row_start[0] = 0;
		// computation of offdiagonal elements
		for (i = 0; i < VN; ++i) {
			vid = i;
			temp = 0;
			diag = 0;
			P = XUGetVertex(vid);// get the coordinate of vertex
			EN = XUGetNeighborEN(vid);// get the number of neighbor edges
			row_start[i + 1] = row_start[i] + EN + 1;// assign the row start
			// for the next row
			edgelist = XUGetNeighborElist(vid);// point to the edge list of v
			for (k = 0; k < EN; ++k) {
				vid1 = XUGetVertexNeighborVId(vid, k);// vertex id of kth
				// neighbor
				Q = XUGetVertex(vid1);// vertex coordinate of kth neighbor
				e = edgelist[k];

				// initialize the matrix entry at P=i,Q=kth neighbor
				offdiag = 0;
				for (j = 0; j < 2; ++j) {
					pid = XUGetEdgePolyId(e, j);// get the id of jth triangle
					// that shares the edge e
					vid2 = XUGetPolyVertexId(pid, 0);// get the vertex id of
					// R or S
					if ((vid2 == vid) || (vid2 == vid1))
						vid2 = XUGetPolyVertexId(pid, 1);
					if ((vid2 == vid) || (vid2 == vid1))
						vid2 = XUGetPolyVertexId(pid, 2);
					R = XUGetVertex(vid2);
					PR = XUVecSub(P, R);
					QR = XUVecSub(Q, R);

					magPR = XUVecMag(PR);
					magQR = XUVecMag(QR);
					if (magPR == 0 || magQR == 0) {
						System.err
								.printf("%d %d %d magnitudes are zero!!!!\n\n",
										i, j, k);
						System.err.println(P + "," + Q + "," + R);
						return null;
					}
					cosr = XUVecDot(PR, QR) / (magPR * magQR);
					if ((cosr <= -1) || (cosr >= 1)) {
						System.err.println("Cosine invalid " + cosr + " "
								+ (magPR * magQR) + " " + XUVecDot(PR, QR));
						cosr = 1 - 10e-4;
					}
					cotr = cosr / Math.sqrt(1 - mysquare(cosr));

					offdiag -= (cotr / 2);
				}
				// System.out.println("WEIGHT "+vid+" "+vid1+" "+(-offdiag));
				diag -= offdiag;
				temp = row_start[i] + k;
				col_idx[temp] = vid1;
				value[temp] = offdiag;
			}// end of the sub for loop for each column of a particular row
			temp = row_start[i] + EN;
			col_idx[temp] = vid;
			value[temp] = diag;
		}// end of main for loop for each vertex (for each row)
		// printf("create sparse matrix\n");
		// create the a and -b vectors depending on the choosen point
		double theta, cosalpha, h;
		double[] avector = new double[VN];
		double[] bvector = new double[VN];
		// printf("initialize vectors ... ");
		vida = XUGetPolyVertexId(n_poly, 0);
		a = XUGetVertex(vida);
		vidb = XUGetPolyVertexId(n_poly, 1);
		b = XUGetVertex(vidb);
		vidc = XUGetPolyVertexId(n_poly, 2);
		c = XUGetVertex(vidc);

		CA = XUVecSub(c, a);
		magb = XUVecMag(CA);
		BA = XUVecSub(b, a);
		magc = XUVecMag(BA);
		I = XUVecDot(CA, BA);
		// theta = I/mysquare(magc);
		theta = I / magc;
		// cosalpha = I/(magc*magb);
		// h = magb*sqrt(1-mysquare(cosalpha));
		h = Math.sqrt(magb * magb - theta * theta);

		avector[vida] = -1 / magc;
		avector[vidb] = 1 / magc;
		bvector[vida] = (1 - theta) / h;
		bvector[vidb] = theta / h;
		bvector[vidc] = -1 / h;
		// printf("create vectors\n\n");
		// conjugate gradient algorithm for
		double dnewa, dnewb, d0a, d0b, alfaa, alfab, dolda, doldb, betaa, betab;
		double[] x = new double[VN]; // soln vectors
		double[] y = new double[VN];
		double[] ra = new double[VN];// residue
		double[] rb = new double[VN];
		double[] da = new double[VN];
		double[] db = new double[VN];
		double[] qa = new double[VN];
		double[] qb = new double[VN];
		// printf("begin conjugate gradient ... ");
		// for Dx = a
		for (i = 0; i < VN; ++i) {
			ra[i] = da[i] = avector[i];
			rb[i] = db[i] = bvector[i];
		}

		dnewa = vvm(VN, ra, ra);
		dnewb = vvm(VN, rb, rb);
		d0a = dnewa;
		d0b = dnewb;

		k = 1;
		setTotalUnits(100);
		double oldError = -1;
		while ((k < imax)
				&& ((dnewa > (error * error * d0a)) || (dnewb > (error * error * d0b)))) {
			if (oldError == -1) {
				oldError = Math.min(dnewa, dnewb);
			}
			// System.out.println("ERRORS "+dnewa+" "+dnewb);
			if (k % 100 == 0) {
				double err = (error * error * Math.max(d0a, d0b));
				setCompletedUnits(1 - (Math.min(dnewa, dnewb) - err)
						/ (oldError - err));
			}
			smvm(VN, maxnum, value, col_idx, row_start, da, qa);
			smvm(VN, maxnum, value, col_idx, row_start, db, qb);
			alfaa = dnewa / vvm(VN, da, qa);
			alfab = dnewb / vvm(VN, db, qb);

			for (i = 0; i < VN; ++i) {
				x[i] = x[i] + alfaa * da[i];
				ra[i] = ra[i] - alfaa * qa[i];
				y[i] = y[i] + alfab * db[i];
				rb[i] = rb[i] - alfab * qb[i];
			}
			if (k % update == 0) {
				// System.out.println("Surface to Complex Iteration " +
				// k+" ERROR A "+dnewa+" "+(error * error *
				// d0a)+" ERROR B "+dnewb +" "+(error * error * d0b));
				smvm(VN, maxnum, value, col_idx, row_start, x, qa);
				smvm(VN, maxnum, value, col_idx, row_start, y, qb);
				for (i = 0; i < VN; ++i) {
					ra[i] = avector[i] - qa[i];
					rb[i] = bvector[i] - qb[i];
				}
			}

			dolda = dnewa;
			dnewa = vvm(VN, ra, ra);
			betaa = dnewa / dolda;
			doldb = dnewb;
			dnewb = vvm(VN, rb, rb);
			betab = dnewb / doldb;

			for (i = 0; i < VN; ++i) {
				da[i] = ra[i] + betaa * da[i];
				db[i] = rb[i] + betab * db[i];
			}
			k += 1;
		}

		double[][] complex = new double[VN][2];
		for (i = 0; i < VN; ++i) {
			complex[i][0] = x[i];
			complex[i][1] = y[i];
			// System.out.printf("%f+%fi,\n", x[i],y[i]);
		}
		/*
		 * double[] xr = new double[VN]; smvm(VN, maxnum, value, col_idx,
		 * row_start, x, xr); for (i = 0; i < VN; ++i) { xr[i] -= avector[i]; }
		 * double norm = vvm(VN, xr, xr);
		 * System.out.printf("%.12f residue of x vector\n", Math.sqrt(norm));
		 * 
		 * smvm(VN, maxnum, value, col_idx, row_start, y, xr); for (i = 0; i <
		 * VN; ++i) { xr[i] -= bvector[i]; } norm = vvm(VN, xr, xr);
		 * System.out.printf("%.12f residue of y vector\n", Math.sqrt(norm));
		 */
		markCompleted();
		return complex;
	}


	/** ***sparse matrix vector multiplication**** */
	void smvm(int m, int max, double[] val, int[] col, int[] rowstart,
			double[] x, double[] y) {
		int i, j;

		for (i = 0; i < m; ++i) {
			y[i] = 0;
			for (j = rowstart[i]; j < rowstart[i + 1]; j++) {
				y[i] += (val[j]) * x[col[j]];
			}
		}
	}


	/** ***vector vector multiplication**** */
	double vvm(int m, double[] v1, double[] v2) {
		int i;
		double norm;

		norm = 0;
		for (i = 0; i < m; i++) {
			norm += (v1[i]) * (v2[i]);
		}
		return (norm);
	}
	/*
	 * find the kth polygon adjacent to VId, The polygon is labeled from p0, p1,
	 * ..., pn, the edge is labeled from e0, e1, ... en, and the order is e0 p0
	 * e1 p1, ..., en pn e0 i.e., e0^e1 = p0
	 */
}
