package edu.jhu.ece.iacl.algorithms.graphics.utilities.rbf;

import Jama.*;
import javax.vecmath.Point2f;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

/**
 * Use radial basis functions to interpolate z position for a function z=f(x,y)
 * where f(.) and its derivative are known for several positions. For the
 * derivative vector specified as <dx,dy,dz>, The derivative is assumed to be
 * <dz/dx,dz/dy>
 * 
 * @author Blake Lucas
 * 
 */
public class RadialBasisFunc3DWithDerivConst extends RadialBasisFunctionFloat {

	protected Vector3f[] norms;

	public RadialBasisFunc3DWithDerivConst(Point3f[] points, Vector3f[] norms,
			RadialFunctionType func) {
		this.points = points;
		this.norms = norms;
		radialFunc = func;
		computeCoefficients();
	}

	public float interpolate(float x, float y) {
		float ret = 0;
		ret += chi[0] + chi[1] * x + chi[2] * y;
		Point3f p = new Point3f();
		p.x = x;
		p.y = y;
		for (int i = 0; i < points.length; i++) {
			ret += (lambda[i] + lambda[i + points.length] * x + lambda[i + 2
					* points.length]
					* y)
					* evalFunc(p, points[i]);
		}
		return ret;
	}

	protected void computeCoefficients() {
		Matrix A = new Matrix(3 * points.length + 3, 3 * points.length + 3);
		Matrix b = new Matrix(3 * points.length + 3, 1);
		double d;
		Vector3f deriv;
		// Populate s(x) entries
		for (int i = 0; i < points.length; i++) {
			Point3f p = points[i];
			A.set(i, 3 * points.length, 1);
			A.set(i, 3 * points.length + 1, p.x);
			A.set(i, 3 * points.length + 2, p.y);
			b.set(i, 0, p.z);
			A.set(3 * points.length, i, 1);
			A.set(3 * points.length + 1, i, p.x);
			A.set(3 * points.length + 2, i, p.y);
			for (int j = 0; j < points.length; j++) {
				d = evalFunc(p, points[j]);
				A.set(i, j, d);
				A.set(i, j + points.length, points[i].x * d);
				A.set(i, j + 2 * points.length, points[i].y * d);
			}
		}

		for (int i = 0; i < points.length; i++) {
			Point3f p = points[i];
			Vector3f n = norms[i];
			// Populate ds(x)/dx entries
			b.set(i + points.length, 0, (n.z != 0) ? -n.x / n.z : 0);

			A.set(i + points.length, 3 * points.length, 0);
			A.set(i + points.length, 3 * points.length + 1, 1);
			A.set(i + points.length, 3 * points.length + 2, 0);

			A.set(3 * points.length, i + points.length, 0);
			A.set(3 * points.length + 1, i + points.length, 1);
			A.set(3 * points.length + 2, i + points.length, 0);

			// Populate ds(x)/dy entries
			b.set(i + 2 * points.length, 0, (n.z != 0) ? -n.y / n.z : 0);

			A.set(i + 2 * points.length, 3 * points.length, 0);
			A.set(i + 2 * points.length, 3 * points.length + 1, 0);
			A.set(i + 2 * points.length, 3 * points.length + 2, 1);

			A.set(3 * points.length, i + 2 * points.length, 0);
			A.set(3 * points.length + 1, i + 2 * points.length, 0);
			A.set(3 * points.length + 2, i + 2 * points.length, 1);

			for (int j = 0; j < points.length; j++) {
				deriv = evalFuncGradient(p, points[j]);
				d = evalFunc(p, points[j]);
				A.set(i + points.length, j, deriv.x);
				A.set(i + points.length, j + points.length, points[i].x
						* deriv.x + d);
				A.set(i + points.length, j + 2 * points.length, points[i].y
						* deriv.x);

				A.set(i + 2 * points.length, j, deriv.y);
				A.set(i + 2 * points.length, j + points.length, points[i].x
						* deriv.y);
				A.set(i + 2 * points.length, j + 2 * points.length, points[i].y
						* deriv.y + d);
			}
		}
		for (int i = 0; i < 3; i++) {
			for (int j = 0; j < 3; j++) {
				A.set(3 * points.length + i, 3 * points.length + j, 0);
			}
			b.set(3 * points.length + i, 0, 0);
		}
		Matrix coeff = A.inverse().times(b);
		chi = new double[3];
		lambda = new double[3 * points.length];
		chi[0] = coeff.get(3 * points.length, 0);
		chi[1] = coeff.get(3 * points.length + 1, 0);
		chi[2] = coeff.get(3 * points.length + 2, 0);
		for (int i = 0; i < 3 * points.length; i++) {
			lambda[i] = coeff.get(i, 0);
		}
	}

	public float evalFunc(Point3f p1, Point3f p2) {
		float r = (float) Math.sqrt((p1.x - p2.x) * (p1.x - p2.x)
				+ (p1.y - p2.y) * (p1.y - p2.y));
		switch (radialFunc) {
		case LINEAR:
			return r;
		case THIN_PLATE:
			return (float) ((r > 0) ? r * r * Math.log(r) : 0);
		case GAUSSIAN:
			return (float) Math.exp(-alpha * r * r);
		case MULTIQUADRIC:
			return (float) Math.sqrt((r * r + C * C));
		default:
			return 0;
		}
	}

	public Vector3f evalFuncGradient(Point3f p1, Point3f p2) {
		float r = (float) Math.sqrt((p1.x - p2.x) * (p1.x - p2.x)
				+ (p1.y - p2.y) * (p1.y - p2.y));
		float rx = (r != 0) ? (p1.x - p2.x) / r : 0;
		float ry = (r != 0) ? (p1.y - p2.y) / r : 0;
		float d = 0;
		switch (radialFunc) {
		case LINEAR:
			d = 1;
			break;
		case THIN_PLATE:
			d = (float) ((r > 0) ? 2 * r * Math.log(Math.abs(r)) + r : 0);
			break;
		case GAUSSIAN:
			d = (float) (-2 * alpha * r * Math.exp(-alpha * r * r));
			break;
		case MULTIQUADRIC:
			d = (float) (r / Math.sqrt((r * r + C * C)));
			break;
		default:
		}
		return new Vector3f(d * rx, d * ry, 0);
	}
}
