package edu.jhu.ece.iacl.algorithms.thickness;

import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.isosurf.IsoSurfaceOnGrid;
import edu.jhu.ece.iacl.algorithms.thickness.grid.SimpleThicknessNode;
import edu.jhu.ece.iacl.algorithms.topology.ConnectivityRule;
import edu.jhu.ece.iacl.jist.structures.geom.CurveCollection;
import edu.jhu.ece.iacl.jist.structures.geom.CurvePath;

/**
 * Implementation of original Jones method for measuring the length of
 * streamlines in an inter-surface conformal map.
 * 
 * @author Blake Lucas (bclucas@jhu.edu)
 * 
 */
public class LagrangeThickness extends ThicknessSolver {
	public LagrangeThickness(double lambda, double isoVal, double lagrangeDeltaT) {
		super(lambda, isoVal, lagrangeDeltaT);
		setLabel("Lagrange Thickness Solver");
		streamlines = new CurveCollection();
	}

	protected CurveCollection streamlines;
	protected double streamThresh = 5;
	SurfaceIntersector intersector;

	protected void computeThickness() {
		int n;

		SimpleThicknessNode cd;
		setTotalUnits(corticalData.length * 2);

		IsoSurfaceOnGrid surfGen = new IsoSurfaceOnGrid();
		outerMesh = (ADJUST_BOUNDARY) ? surfGen.solveNoMove(outerVol,
				ConnectivityRule.CONNECT_18_6, (float) isoVal, false) : surfGen
				.solveOriginal(outerVol, ConnectivityRule.CONNECT_18_6,
						(float) isoVal, false);
		innerMesh = (ADJUST_BOUNDARY) ? surfGen.solveNoMove(innerVol,
				ConnectivityRule.CONNECT_18_6, (float) isoVal, false) : surfGen
				.solveOriginal(innerVol, ConnectivityRule.CONNECT_18_6,
						(float) isoVal, false);

		System.out.println("HOLES IN OUTER SURFACE "
				+ outerMesh.getNumberOfHoles());

		intersector = new SurfaceIntersector(this, 12, outerMesh);

		System.gc();
		this.setLabel("Computing Outer Length");
		SimpleThicknessNode.setCompareInnerLength(false);
		for (n = 0; n < corticalData.length; n++) {
			cd = corticalData[n];
			// System.out.println("TRACE "+cd.getLocation());
			cd.setLength(trace(cd.getLocation(), true));
			if (cd.getLength() > streamThresh) {
				createStreamline(cd.getLocation(), true);
			}
			incrementCompletedUnits();
		}
		markCompleted();

		System.out.println("HOLES IN INNER SURFACE "
				+ innerMesh.getNumberOfHoles());
		intersector = new SurfaceIntersector(this, 12, innerMesh);
		System.gc();
		this.setLabel("Computing Inner Length");
		SimpleThicknessNode.setCompareInnerLength(true);
		for (n = 0; n < corticalData.length; n++) {
			cd = corticalData[n];
			cd.setLength(trace(cd.getLocation(), false));
			if (cd.getLength() > streamThresh) {
				createStreamline(cd.getLocation(), false);
			}
			incrementCompletedUnits();
		}
		intersector = null;

		markCompleted();

	}

	/**
	 * Get collection of streamlines
	 * 
	 * @return streamlines
	 */
	public CurveCollection getStreamlines() {
		streamlines.setName(innerVol.getName() + "_streams");
		return streamlines;
	}

	/**
	 * Create streamlines for particular starting location
	 * 
	 * @param p1
	 *            starting location
	 * @param boundary
	 *            false indicates trace distance to inner boundary, true
	 *            indicates trace distance to outer boundary
	 */
	protected void createStreamline(Point3f p1, boolean boundary) {
		CurvePath path = new CurvePath();
		double len = 0;
		Point3f p2 = null;
		int count = 0;
		double step = (boundary) ? lagrangeDeltaT : -lagrangeDeltaT;
		double mag;
		final int maxCount = 2000;
		final float maxLength = 100;
		Point3f start = p1;
		double d;
		path.add(p1);
		Point3f tan_1 = null;
		Point3f tan_2 = null;
		Point3f tan_3 = null;
		Point3f tan;

		while (len < maxLength && count < maxCount) {
			tan_1 = tan_2;
			tan_2 = tan_3;
			tan_3 = interpolateTanget(p1);
			if (tan_2 != null) {
				if (tan_1 != null) {
					tan = new Point3f(0.166666f * tan_3.x + 0.666668f * tan_2.x
							+ 0.166666f * tan_1.x, 0.166666f * tan_3.y
							+ 0.666668f * tan_2.y + 0.166666f * tan_1.y,
							0.166666f * tan_3.z + 0.666668f * tan_2.z
									+ 0.166666f * tan_1.z);
				} else {
					tan = new Point3f(0.5f * (tan_3.x + tan_2.x),
							0.5f * (tan_3.y + tan_2.y),
							0.5f * (tan_3.z + tan_2.z));
				}
			} else {
				tan = tan_3;
			}
			mag = Math.sqrt(tan.x * tan.x + tan.y * tan.y + tan.z * tan.z);
			p2 = new Point3f((float) (p1.x + step * tan.x),
					(float) (p1.y + step * tan.y),
					(float) (p1.z + step * tan.z));
			double dist = intersector.intersectSegmentDistance(p1, p2);

			if (dist < 0) {
				len += lagrangeDeltaT * mag;
				path.add(p2);
				p1 = p2;
			} else {
				path.add(intersector.getLastIntersectionPoint());
				streamlines.add(path);
				return;
			}
			count++;
		}
		streamlines.add(path);
	}
	/**
	 * Trace distance from point
	 * @param p1 point
	 * @param boundary
	 *            false indicates trace distance to inner boundary, true
	 *            indicates trace distance to outer boundary
	 * @return
	 */
	protected double trace(Point3f p1, boolean boundary) {
		double len = 0;
		Point3f p2 = null;
		int count = 0;
		double step = (boundary) ? lagrangeDeltaT : -lagrangeDeltaT;
		double mag;
		final int maxCount = 1000;
		final float maxLength = 40;
		Point3f start = p1;
		double d;
		Point3f tan_1 = null;
		Point3f tan_2 = null;
		Point3f tan_3 = null;
		Point3f tan;
		// Use Runge-Kutta 4th order scheme
		while (len < maxLength && count < maxCount) {
			tan_1 = tan_2;
			tan_2 = tan_3;
			tan_3 = interpolateTanget(p1);
			if (tan_2 != null) {
				if (tan_1 != null) {
					tan = new Point3f(0.166666f * tan_3.x + 0.666668f * tan_2.x
							+ 0.166666f * tan_1.x, 0.166666f * tan_3.y
							+ 0.666668f * tan_2.y + 0.166666f * tan_1.y,
							0.166666f * tan_3.z + 0.666668f * tan_2.z
									+ 0.166666f * tan_1.z);
				} else {
					tan = new Point3f(0.5f * (tan_3.x + tan_2.x),
							0.5f * (tan_3.y + tan_2.y),
							0.5f * (tan_3.z + tan_2.z));
				}
			} else {
				tan = tan_3;
			}
			mag = Math.sqrt(tan.x * tan.x + tan.y * tan.y + tan.z * tan.z);
			p2 = new Point3f((float) (p1.x + step * tan.x),
					(float) (p1.y + step * tan.y),
					(float) (p1.z + step * tan.z));
			double dist = intersector.intersectSegmentDistance(p1, p2);
			if (dist < 0) {
				len += lagrangeDeltaT * mag;
				p1 = p2;
			} else {
				return len + dist;
			}
			count++;
		}
		// System.out.println(start+" "+tags[(int)Math.round(start.x)][(int)Math.round(start.y)][(int)Math.round(start.z)]+" "+p2+" "+tags[(int)Math.round(p2.x)][(int)Math.round(p2.y)][(int)Math.round(p2.z)]);
		d = intersector.distance(start);
		// System.err.println("MAXED OUT "+count+" "+start+" "+p1+" "+p2+" "+len);
		return d;
	}
}
