package edu.jhu.ece.iacl.algorithms.volume;

import edu.jhmi.rad.medic.structures.RotationMatrix;
import edu.jhmi.rad.medic.utilities.Numerics;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmRuntimeException;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataMipav;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataMipavWrapper;
import edu.jhu.ece.iacl.jist.structures.image.ImageHeader;
import gov.nih.mipav.model.algorithms.AlgorithmTransform;
import gov.nih.mipav.model.file.FileInfoBase;
import gov.nih.mipav.model.structures.ModelImage;
import gov.nih.mipav.model.structures.TransMatrix;
/**
 * Reorient volume to particular anatomical orientation.
 * 
 * @author Blake Lucas
 *
 */
public class ReorientVolume {
	private static final String[] orientTypes = { "Axial", "Coronal", "Sagittal" };
	private static final String[] resolutionTypes = { "Unchanged", "Finest-Cubic","Coarsest-Cubic", "Same_as_template" };
	private static final String[] interpTypes = { "Nearest_Neighbor", "Linear","Windowed_Sinc" };
	private static final float ISQRT2 = (float)(1.0/Math.sqrt(2.0f));
	public enum Orientation{AXIAL,CORONAL,SAGITTAL};
	public enum Resolution{UNCHANGED,FINEST_CUBIC,COARSEST_CUBIC,SAME_AS_TEMPLATE};
	public enum Mask{BINARY,PROBABILITY};
	public enum Interpolation{NEAREST_NEIGHBOR,LINEAR,WINDOWED_SINC};
	public static ImageDataMipav solve(ImageDataMipav source,ImageDataMipav target,Orientation or,Resolution res,Interpolation interpMethod){
		String orientType = orientTypes[or.ordinal()];
		String resolutionType = resolutionTypes[res.ordinal()];
		String interpType = interpTypes[interpMethod.ordinal()];
		ModelImage image = source.getModelImageCopy();
		ModelImage template = null;
		if(resolutionType.equals("Same_as_template")){
			try{
				template = target.getModelImageCopy();
			}catch(NullPointerException e){
				System.err.println("Resolution option was set as \"Same as template\" but no template specified!");
				e.printStackTrace();
			}
		}
		FileInfoBase fileInfo = (FileInfoBase) (image.getFileInfo()[0].clone());
		// set resampled resolutions, dimensions
		float r0x = image.getFileInfo()[0].getResolutions()[0];
		float r0y = image.getFileInfo()[0].getResolutions()[1];
		float r0z = image.getFileInfo()[0].getResolutions()[2];

		int n0x = image.getExtents()[0];
		int n0y = image.getExtents()[1];
		int n0z = image.getExtents()[2];

		float rx = r0x, ry = r0y, rz = r0z;
		int nx = n0x, ny = n0y, nz = n0z;

		if (resolutionType.equals("Finest-Cubic")) {
			float rn = Numerics.min(rx, ry, rz);
			nx = Numerics.round(nx * rx / rn);
			rx = rn;
			ny = Numerics.round(ny * ry / rn);
			ry = rn;
			nz = Numerics.round(nz * rz / rn);
			rz = rn;
		} else if (resolutionType.equals("Coarsest-Cubic")) {
			float rn = Numerics.max(rx, ry, rz);
			nx = Numerics.round(nx * rx / rn);
			rx = rn;
			ny = Numerics.round(ny * ry / rn);
			ry = rn;
			nz = Numerics.round(nz * rz / rn);
			rz = rn;
		} else if (resolutionType.equals("Same_as_template")) {
			rx = template.getFileInfo()[0].getResolutions()[0];
			ry = template.getFileInfo()[0].getResolutions()[1];
			rz = template.getFileInfo()[0].getResolutions()[2];
			nx = template.getExtents()[0];
			ny = template.getExtents()[1];
			nz = template.getExtents()[2];
		}

		// compute the matrix from re-orientation information
		RotationMatrix rot = new RotationMatrix();

		// retrieve image info
		int orient = image.getImageOrientation();
		int orx = image.getFileInfo()[0].getAxisOrientation(0);
		int ory = image.getFileInfo()[0].getAxisOrientation(1);
		int orz = image.getFileInfo()[0].getAxisOrientation(2);

		float[] ao = new float[3];
		for (int i = 0; i < 3; i++)
			ao[i] = 1.0f;
		float[] at = new float[3];
		for (int i = 0; i < 3; i++)
			at[i] = 0.0f;

		if (orientType.equals("Axial")) {
			// note : assumes a rotation around the image center
			if (orient == FileInfoBase.AXIAL) {
				rot.setParameters(0.0f, 0.0f, 0.0f);

				if (orx == FileInfoBase.ORI_L2R_TYPE) {
					ao[0] = -1.0f;
					at[0] = r0x * (n0x - 1);
				}
				if (ory == FileInfoBase.ORI_P2A_TYPE) {
					ao[1] = -1.0f;
					at[1] = r0y * (n0y - 1);
				}
				if (orz == FileInfoBase.ORI_S2I_TYPE) {
					ao[2] = -1.0f;
					at[2] = r0z * (n0z - 1);
				}

			} else if (orient == FileInfoBase.CORONAL) {
				rot.setParameters(-ISQRT2, 0.0f, 0.0f);
				at[2] = r0y * (n0y - 1);

				if (orx == FileInfoBase.ORI_L2R_TYPE) {
					ao[0] = -1.0f;
					at[0] = r0x * (n0x - 1);
				}
				if (orz == FileInfoBase.ORI_P2A_TYPE) {
					ao[1] = -1.0f;
					at[1] = r0z * (n0z - 1);
				}
				if (ory == FileInfoBase.ORI_I2S_TYPE) {
					ao[2] = -1.0f;
					at[2] = 0.0f;
				}

				float rt = ry;
				ry = rz;
				rz = rt;
				int nt = ny;
				ny = nz;
				nz = nt;
			} else if (orient == FileInfoBase.SAGITTAL) {
				rot.setParameters(-0.5f, -0.5f, 0.5f);
				at[0] = r0z * (n0z - 1);
				at[2] = r0y * (n0y - 1);

				if (orz == FileInfoBase.ORI_R2L_TYPE) {
					ao[0] = -1.0f;
					at[0] = 0.0f;
				}
				if (orx == FileInfoBase.ORI_P2A_TYPE) {
					ao[1] = -1.0f;
					at[1] = r0x * (n0x - 1);
				}
				if (ory == FileInfoBase.ORI_I2S_TYPE) {
					ao[2] = -1.0f;
					at[2] = 0.0f;
				}

				float rt = rz;
				rz = ry;
				ry = rx;
				rx = rt;
				int nt = nz;
				nz = ny;
				ny = nx;
				nx = nt;
			}
			fileInfo.setImageOrientation(FileInfoBase.AXIAL);
			fileInfo.setAxisOrientation(FileInfoBase.ORI_R2L_TYPE, 0);
			fileInfo.setAxisOrientation(FileInfoBase.ORI_A2P_TYPE, 1);
			fileInfo.setAxisOrientation(FileInfoBase.ORI_I2S_TYPE, 2);
		} else if (orientType.equals("Coronal")) {
			// note : assumes a rotation around the image center
			if (orient == FileInfoBase.CORONAL) {
				rot.setParameters(0.0f, 0.0f, 0.0f);

				if (orx == FileInfoBase.ORI_L2R_TYPE) {
					ao[0] = -1.0f;
					at[0] = r0x * (n0x - 1);
				}
				if (ory == FileInfoBase.ORI_I2S_TYPE) {
					ao[1] = -1.0f;
					at[1] = r0y * (n0y - 1);
				}
				if (orz == FileInfoBase.ORI_P2A_TYPE) {
					ao[2] = -1.0f;
					at[2] = r0z * (n0z - 1);
				}

			} else if (orient == FileInfoBase.AXIAL) {
				rot.setParameters(ISQRT2, 0.0f, 0.0f);
				at[1] = r0z * (n0z - 1);

				if (orx == FileInfoBase.ORI_L2R_TYPE) {
					ao[0] = -1.0f;
					at[0] = r0x * (n0x - 1);
				}
				if (orz == FileInfoBase.ORI_S2I_TYPE) {
					ao[1] = -1.0f;
					at[1] = 0.0f;
				}
				if (ory == FileInfoBase.ORI_P2A_TYPE) {
					ao[2] = -1.0f;
					at[2] = r0y * (n0y - 1);
				}

				float rt = ry;
				ry = rz;
				rz = rt;
				int nt = ny;
				ny = nz;
				nz = nt;
			} else if (orient == FileInfoBase.SAGITTAL) {
				rot.setParameters(0.0f, -ISQRT2, 0.0f);
				at[0] = r0z * (n0z - 1);

				if (orz == FileInfoBase.ORI_R2L_TYPE) {
					ao[0] = -1.0f;
					at[0] = 0.0f;
				}
				if (ory == FileInfoBase.ORI_I2S_TYPE) {
					ao[1] = -1.0f;
					at[1] = r0y * (n0y - 1);
				}
				if (orx == FileInfoBase.ORI_P2A_TYPE) {
					ao[2] = -1.0f;
					at[2] = r0x * (n0x - 1);
				}

				float rt = rz;
				rz = rx;
				rx = rt;
				int nt = nz;
				nz = nx;
				nx = nt;
			}
			fileInfo.setImageOrientation(FileInfoBase.CORONAL);
			fileInfo.setAxisOrientation(FileInfoBase.ORI_R2L_TYPE, 0);
			fileInfo.setAxisOrientation(FileInfoBase.ORI_S2I_TYPE, 1);
			fileInfo.setAxisOrientation(FileInfoBase.ORI_A2P_TYPE, 2);
		} else if (orientType.equals("Sagittal")) {
			// note : assumes a rotation around the image center
			if (orient == FileInfoBase.SAGITTAL) {
				rot.setParameters(0.0f, 0.0f, 0.0f);

				if (orx == FileInfoBase.ORI_P2A_TYPE) {
					ao[0] = -1.0f;
					at[0] = r0x * (n0x - 1);
				}
				if (ory == FileInfoBase.ORI_I2S_TYPE) {
					ao[1] = -1.0f;
					at[1] = r0y * (n0y - 1);
				}
				if (orz == FileInfoBase.ORI_R2L_TYPE) {
					ao[2] = -1.0f;
					at[2] = r0z * (n0z - 1);
				}

			} else if (orient == FileInfoBase.CORONAL) {
				rot.setParameters(0.0f, ISQRT2, 0.0f);
				at[2] = r0x * (n0x - 1);

				if (orz == FileInfoBase.ORI_P2A_TYPE) {
					ao[0] = -1.0f;
					at[0] = r0z * (n0z - 1);
				}
				if (ory == FileInfoBase.ORI_I2S_TYPE) {
					ao[1] = -1.0f;
					at[1] = r0y * (n0y - 1);
				}
				if (orx == FileInfoBase.ORI_L2R_TYPE) {
					ao[2] = -1.0f;
					at[2] = 0.0f;
				}

				float rt = rz;
				rz = rx;
				rx = rt;
				int nt = nz;
				nz = nx;
				nx = nt;
			} else if (orient == FileInfoBase.AXIAL) {
				rot.setParameters(0.5f, 0.5f, -0.5f);
				at[1] = r0z * (n0z - 1);
				at[2] = r0x * (n0x - 1);

				if (ory == FileInfoBase.ORI_P2A_TYPE) {
					ao[0] = -1.0f;
					at[0] = r0y * (n0y - 1);
				}
				if (orz == FileInfoBase.ORI_S2I_TYPE) {
					ao[1] = -1.0f;
					at[1] = 0.0f;
				}
				if (orx == FileInfoBase.ORI_L2R_TYPE) {
					ao[2] = -1.0f;
					at[2] = 0.0f;
				}

				float rt = rx;
				rx = ry;
				ry = rz;
				rz = rt;
				int nt = nx;
				nx = ny;
				ny = nz;
				nz = nt;
			}
			fileInfo.setImageOrientation(FileInfoBase.SAGITTAL);
			fileInfo.setAxisOrientation(FileInfoBase.ORI_A2P_TYPE, 0);
			fileInfo.setAxisOrientation(FileInfoBase.ORI_S2I_TYPE, 1);
			fileInfo.setAxisOrientation(FileInfoBase.ORI_L2R_TYPE, 2);
		}

		fileInfo.setResolutions(rx, 0);
		fileInfo.setResolutions(ry, 1);
		fileInfo.setResolutions(rz, 2);
		fileInfo.setExtents(nx, 0);
		fileInfo.setExtents(ny, 1);
		fileInfo.setExtents(nz, 2);

		TransMatrix transform = new TransMatrix(4);
		transform.MakeIdentity();
		for (int i = 0; i < 3; i++) {
			for (int j = 0; j < 3; j++) {
				transform.Set(i, j, rot.getMatrix(i, j) * ao[i]);
			}
			transform.Set(i, 3, at[i]);
		}
		// transform.invert();
		System.out.println(transform.toString());

		int interp = AlgorithmTransform.TRILINEAR;
		if (interpType.equals("Nearest_Neighbor")) {
			interp = AlgorithmTransform.NEAREST_NEIGHBOR;
		} else if (interpType.equals("Linear")) {
			interp = AlgorithmTransform.TRILINEAR;
		} else if (interpType.equals("Windowed_Sinc")) {
			interp = AlgorithmTransform.WSINC;
		}

		AlgorithmTransform algoTrans = new AlgorithmTransform(image, transform, interp, rx, ry,
				rz, nx, ny, nz, true, false, false);
		algoTrans.setUpdateOriginFlag(true);
		algoTrans.run();
		ImageDataMipav img=new ImageDataMipavWrapper(algoTrans.getTransformedImage());
		img.setHeader(new ImageHeader(fileInfo));
		return img;
	}
}
