package edu.jhu.ece.iacl.plugins.classification;

import java.awt.CardLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.SpringLayout;

import edu.jhmi.rad.medic.algorithms.AlgorithmFantasm;
import edu.jhmi.rad.medic.algorithms.AlgorithmFantasmLongitudinal;
import edu.jhmi.rad.medic.algorithms.AlgorithmFantasmMultichannel;
import edu.jhmi.rad.medic.algorithms.AlgorithmMultipleToads;
import edu.jhmi.rad.medic.methods.MultipleToadAtlas;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.algorithms.PrinceGroupAuthors;
import edu.jhu.ece.iacl.algorithms.ReferencedPapers;
import edu.jhu.ece.iacl.jist.io.FileExtensionFilter;
import edu.jhu.ece.iacl.jist.io.MipavController;

import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.AlgorithmAuthor;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation.Citation;
import edu.jhu.ece.iacl.jist.pipeline.parameter.*;
import edu.jhu.ece.iacl.jist.pipeline.view.input.ParamCollectionInputView;
import edu.jhu.ece.iacl.jist.pipeline.view.input.ParamCollectionPaneInputView;
import edu.jhu.ece.iacl.jist.pipeline.view.input.ParamCollectionWindowInputView;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataMipav;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataMipavWrapper;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataUByte;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;

import gov.nih.mipav.model.algorithms.AlgorithmBase;
import gov.nih.mipav.model.structures.ModelImage;
import gov.nih.mipav.model.structures.ModelStorageBase;
import gov.nih.mipav.view.MipavUtil;
import gov.nih.mipav.view.ViewJProgressBar;
import gov.nih.mipav.view.ViewUserInterface;
import gov.nih.mipav.view.dialogs.JDialogBase;

/**
 * Medic wrapper for Fantasm
 * 
 * @version July 2008
 * @author Blake Lucas
 */
public class MedicAlgorithmFantasm extends ProcessingAlgorithm {
	private ParamVolumeCollection inputImage;
	private ParamOption outputTypeParam;
	private ParamBoolean scaleData;
	private ParamInteger numOfClasses;
	private ParamBoolean correctInhomogeneity;
	private ParamBoolean backgroundCrop;
	private ParamDouble smoothParam;
	private ParamDouble maxDifference;
	private ParamInteger maxIters;
	private ParamOption initOptions;
	private ParamOption maskMode;
	private ParamInteger fieldDegree;
	private ParamDouble bgThreshold;
	private ParamDouble fuzzyCoeff;
	private ParamVolume classification;
	private ParamVolumeCollection field;
	//private ParamVolumeCollection memberships;

	private ParamVolume csf;
	private ParamVolume gm;
	private ParamVolume wmfill;
	private static String[] algoTypes = { "single_image", "multichannel" };
	private static String[] initModes = { "range", "modes", "manual" };
	private static String[] maskingModes = { "one", "all" };
	private static String[] outputTypes = { "all_result_images", "hard_segmentation", "fuzzy_segmentation",
			"both_fuzzy_&_hard" };
	// private ParamVolumeCollection volClasses;
	private HashMap<String, String> connMap;

	private static final String cvsversion = "$Revision: 1.2 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "");
	/**
	 * Create Input parameters for TOADS as specified in AlgorithmMultipleToads
	 * The boundaries for these variables do not necessarily match those
	 * specified in the original dialog
	 */
	protected void createInputParameters(ParamCollection inputParams) {
		algorithmInformation.add(PrinceGroupAuthors.dzungPham);
		inputParams.add(inputImage = new ParamVolumeCollection("Image to segment"));
		inputParams.add(numOfClasses = new ParamInteger("Number of classes", 1, 10, 3));
		inputParams.add(correctInhomogeneity = new ParamBoolean("Correct Inhomogeneity", true));
		inputParams.add(backgroundCrop = new ParamBoolean("Background Cropping", true));
		inputParams.add(outputTypeParam = new ParamOption("Output Type", outputTypes));
		inputParams.add(smoothParam = new ParamDouble("Smooting parameter", 0, 1E10, 0.01));
		inputParams.add(maxDifference = new ParamDouble("Maximum difference", 0, 1E10, 0.01));
		inputParams.add(maxIters = new ParamInteger("Maximum iterations", 0, 100000, 50));
		inputParams.add(initOptions = new ParamOption("Initialization", initModes));
		inputParams.add(maskMode = new ParamOption("Masking Mode", maskingModes));
		inputParams.add(fieldDegree = new ParamInteger("Inhomogeneity field degree", 1, 4, 3));
		inputParams.add(bgThreshold = new ParamDouble("Background threshold", 0, 1E10, 0));
		inputParams.add(fuzzyCoeff = new ParamDouble("Fuzziness Coefficient", 0, 1E10, 2.0));
		outputTypeParam = new ParamOption("Output images", outputTypes);
		inputParams.setPackage("IACL");
		inputParams.setCategory("Classification");
		inputParams.setName("fantasm");
		inputParams.setLabel("Fantasm");
			
		AlgorithmInformation info=getAlgorithmInformation();
		info.setWebsite("http://medic.rad.jhmi.edu/");
		info.setDescription("Fuzzy and Noise Tolerant Adaptive Segmentation Method. "
				+ "Outputs WM/GM/CSF classes.");
		info.setAffiliation("Johns Hopkins University, Department of Electrical and Computer Engineering");
		info.add(new Citation("D.L. Pham," 
				+ "'Robust Fuzzy Segmentation of Magnetic Resonance Images', "
				+ "Proc. 14th IEEE Symp. on Computer-based Medical Systems (CBMS 2001), "
				+ "Bethesda, MD, pp. 127-131, 2001."));
		info.setVersion(revnum);
		info.setEditable(false);
		info.setStatus(DevelopmentStatus.Release);
	}

	/**
	 * Create output Parameters for TOADS. Note: Not all output fields are
	 * populated and the non-populated fields will create an error that is
	 * caught by the dialog.
	 */
	protected void createOutputParameters(ParamCollection outputParams) {
		// not working; is it set before the input parameters are received ??
		outputParams.add(classification = new ParamVolume("Hard segmentation"));
		outputParams.add(field = new ParamVolumeCollection("Inhomogeneity Field"));
		//outputParams.add(memberships = new ParamVolumeCollection("Membership Functions"));
		outputParams.add(csf = new ParamVolume("Sulcal CSF embership"));
		outputParams.add(gm = new ParamVolume("Cortical GM Membership"));
		outputParams.add(wmfill = new ParamVolume("Filled WM Membership"));
		classification.setMandatory(false);
		field.setMandatory(false);
		//memberships.setMandatory(false);
		csf.setMandatory(false);
		gm.setMandatory(false);
		wmfill.setMandatory(false);
		// outputParams.add(volClasses = new ParamVolumeCollection("Membership
		// Classes",VoxelType.UBYTE));
	}

	/**
	 * Execute the TOADS/CRUISE algorithm given the input parameters
	 */
	protected class FantasmWrapper extends AbstractCalculation {
		private AlgorithmFantasmWrapper algo = null;
		private AlgorithmFantasmMultiChannelWrapper algo3 = null;
		private ModelImage image; // reference image
		private ModelImage images[]; // source images
		private ModelImage resultImage[] = null; // result images
		private int resultNumber;
		private int imageNumber;
		private int presentNumber;
		private int destExtents[];
		private ViewUserInterface userInterface;
		private String titles[];
		private int[] order;
		private int nOrder;
		private int borders = 0;
		// parameters
		// classification
		private int nClasses = 3;
		private int nIterations = 50;
		private float maxDiff = 0.01f;
		private float smoothing = 0.01f;
		private float fuzziness = 2.0f;
		private boolean addOutliers = false;
		private float outlierRatio = 0.25f;
		private String algoType = "single_image";
		private String[] algoTypes = { "single_image", "multichannel" };
		private String initMode = "range";
		private String[] initModes = { "range", "modes", "manual" };
		private String maskingMode = "one";
		private String[] maskingModes = { "one", "all" };
		// temporal / multi channel
		private float temporal = 0.05f;
		// registration
		private boolean useRegistration = false;
		private int topIter = 10;
		private int firstIter = 2;
		private int lastIter = 3;
		private int levels = 3;
		// inhomogeneity
		private boolean correctField = true;
		private int polynomialDegree = 3;
		private String[] polynomialDegrees = { "1", "2", "3", "4" };
		// edges
		private boolean useEdges = false;
		private float edgeSmoothness = 0.05f;
		private float edgeContrast = 0.05f;
		private float edgePrior = 0.001f;
		// general
		private float backThreshold = 0.0f;
		private boolean cropBackground = true;
		private boolean useRelative = true;
		private String outputType = "hard_segmentation";
		private String[] outputTypes = { "all_result_images", "hard_segmentation", "fuzzy_segmentation",
				"both_fuzzy_&_hard" };

		public FantasmWrapper() {
			setLabel("Fantasm");
		}
		

		public void execute() {
			int i, Nimg;
			nClasses=numOfClasses.getInt();
			
			correctField=correctInhomogeneity.getValue();
			cropBackground=backgroundCrop.getValue();
			outputType=outputTypeParam.getValue();
			smoothing=smoothParam.getFloat();
			maxDiff=maxDifference.getFloat();
			nIterations=maxIters.getInt();
			initMode=initOptions.getValue();
			maskingMode=maskMode.getValue();
			polynomialDegree=fieldDegree.getInt();
			backThreshold=bgThreshold.getFloat();
			fuzziness=fuzzyCoeff.getFloat();
			imageNumber=inputImage.size();
			algoType=(imageNumber>1)?algoTypes[1]:algoTypes[0];
			
			images=new ModelImage[imageNumber];
			List<ImageData> temp = inputImage.getImageDataList();
			for(int j=0; j<imageNumber;j++){
				images[j]=temp.get(j).getModelImageCopy();
			}
			
			image = images[0];
			String name = (image.getImageName()+"");
			AlgorithmBase algorithm = null;
			String suffix = "";
			// Calculate the number of result images.
			resultNumber = 0;
			if (outputType.equals("hard_segmentation"))
				resultNumber = 1;
			else if (outputType.equals("fuzzy_segmentation")) {
				resultNumber = nClasses;
				if (addOutliers)
					resultNumber++;
			} else {
				resultNumber = nClasses + 1;
				if (addOutliers)
					resultNumber++;
				if (outputType.equals("all_result_images")) {
					if (correctField) {
						if (algoType.equals("multichannel"))
							resultNumber += imageNumber;
						else
							resultNumber++;
					}
					if (useEdges)
						resultNumber += 3;
				}
			}
			if (algoType.equals("longitudinal"))
				Nimg = imageNumber;
			else
				Nimg = 1;
			// add suffix for algorithm type
			if (algoType.equals("longitudinal"))
				suffix = "_long";
			else if (algoType.equals("multichannel"))
				suffix = "_multi";
			if (image.getNDims() == 2) { // source image is 2D
				destExtents = new int[2];
				destExtents[0] = image.getExtents()[0]; // X dim
				destExtents[1] = image.getExtents()[1]; // Y dim
			} else { // source image is 3D or more (?)
				destExtents = new int[3];
				destExtents[0] = image.getExtents()[0];
				destExtents[1] = image.getExtents()[1];
				destExtents[2] = image.getExtents()[2];
			}
			resultImage = new ModelImage[resultNumber * Nimg];
			presentNumber = 0;
			for (int n = 0; n < Nimg; n++) {
				if (!outputType.equals("hard_segmentation")) {
					for (i = 0; i < nClasses; i++) {
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents, (images[n].getImageName()+ suffix + "_class" + (i + 1)));
					}
					if (addOutliers) {
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents, (images[n].getImageName()+suffix + "_out"));
					}
				}
				if (!outputType.equals("fuzzy_segmentation")) {
					resultImage[presentNumber++] = new ModelImage(ModelStorageBase.UBYTE, destExtents, (images[n].getImageName()+ suffix + "_seg"));
				}
				if (outputType.equals("all_result_images")) {
					if (correctField) {
						if (algoType.equals("multichannel")) {
							for (int t = 0; t < imageNumber; t++) {
								resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
										(images[n].getImageName()+ suffix + "_field"+(t+1)));
							}
						} else {
							resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents,
									(images[n].getImageName()+ suffix + "_field"));
						}
					}

					if (useEdges) {
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents, (images[n].getImageName()+ suffix + "_edgeX"));
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents, (images[n].getImageName()+ suffix + "_edgeY"));
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents, (images[n].getImageName()+ suffix + "_edgeZ"));
					}
				}
			}
			for(int n=0;n<resultImage.length;n++){
				System.out.println(n+") "+resultImage[n].getImageName());
			}
			// Create algorithm
			if (algoType.equals("single_image")) {
				String correct;
				if (correctField)
					correct = "image";
				else
					correct = "none";
				algo = new AlgorithmFantasmWrapper(resultImage, images[0], resultNumber, initMode, outputType, nClasses,
						nIterations, maxDiff, smoothing, fuzziness, addOutliers, outlierRatio, correct,
						polynomialDegree, useEdges, edgeSmoothness, edgeContrast, edgePrior, cropBackground,
						backThreshold, useRelative, false);
				algo.setObserver(this);
				algorithm = algo;
			} else if (algoType.equals("multichannel")) {
				algo3 = new AlgorithmFantasmMultiChannelWrapper(resultImage, images, imageNumber, resultNumber, initMode,
						outputType, nClasses, nIterations, maxDiff, smoothing, fuzziness, addOutliers, outlierRatio,
						correctField, polynomialDegree, useEdges, edgeSmoothness, edgeContrast, edgePrior,
						useRegistration, levels, firstIter, topIter, lastIter, cropBackground, backThreshold,
						maskingMode);
				algo3.setObserver(this);
				algorithm = algo3;
			}
			// This is very important. Adding this object as a listener allows
			// the algorithm to
			// notify this object when it has completed or failed. See algorithm
			// performed event.
			// This is made possible by implementing AlgorithmedPerformed
			// interface
			algorithm.run();
			presentNumber = 0;
			for (int n = 0; n < Nimg; n++) {
				if (!outputType.equals("hard_segmentation")) {
					csf.setValue(new ImageDataMipavWrapper(resultImage[presentNumber++]));
					gm.setValue(new ImageDataMipavWrapper(resultImage[presentNumber++]));
					wmfill.setValue(new ImageDataMipavWrapper(resultImage[presentNumber++]));
					if (addOutliers) {
						
						presentNumber++;
					}
					/*
					if (addOutliers) {
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents, JDialogBase
								.(images[n].getImageName(), suffix + "_out"));
					}
					*/
				}
				if (!outputType.equals("fuzzy_segmentation")) {
					classification.setValue(new ImageDataMipavWrapper(resultImage[presentNumber++]));
				}
				if (outputType.equals("all_result_images")) {
					if (correctField) {
						
						if (algoType.equals("multichannel")) {
							for (int t = 0; t < imageNumber; t++) {
								field.add(resultImage[presentNumber++]);
							}
						} else {
							field.add(resultImage[presentNumber++]);
						}
					}
					/*
					if (useEdges) {
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents, JDialogBase
								.(images[n].getImageName(), suffix + "_edgeX"));
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents, JDialogBase
								.(images[n].getImageName(), suffix + "_edgeY"));
						resultImage[presentNumber++] = new ModelImage(ModelStorageBase.FLOAT, destExtents, JDialogBase
								.(images[n].getImageName(), suffix + "_edgeZ"));
					}
					*/
				}
			}
			
			for(int j=0; j<imageNumber;j++){
				images[j].disposeLocal();
			}
		}
		
		
	}
	protected class AlgorithmFantasmMultiChannelWrapper extends AlgorithmFantasmMultichannel{

		/**
		 * @param destImg_
		 * @param srcImg_
		 * @param Nimg_
		 * @param destNum_
		 * @param init_
		 * @param output_
		 * @param classes_
		 * @param iterMax_
		 * @param distMax_
		 * @param smooth_
		 * @param fuzzy_
		 * @param addOut_
		 * @param outVal_
		 * @param correct_
		 * @param poly_
		 * @param edges_
		 * @param smooth_2
		 * @param contr_
		 * @param pr_
		 * @param reg_
		 * @param lev_
		 * @param first_
		 * @param top_
		 * @param last_
		 * @param cropBack_
		 * @param cropVal_
		 * @param scaling_
		 */
		public AlgorithmFantasmMultiChannelWrapper(ModelImage[] destImg_, ModelImage[] srcImg_, int Nimg_,
				int destNum_, String init_, String output_, int classes_, int iterMax_, float distMax_, float smooth_,
				float fuzzy_, boolean addOut_, float outVal_, boolean correct_, int poly_, boolean edges_,
				float smooth_2, float contr_, float pr_, boolean reg_, int lev_, int first_, int top_, int last_,
				boolean cropBack_, float cropVal_, String scaling_) {
			super(destImg_, srcImg_, Nimg_, destNum_, init_, output_, classes_, iterMax_, distMax_, smooth_, fuzzy_, addOut_,
					outVal_, correct_, poly_, edges_, smooth_2, contr_, pr_, reg_, lev_, first_, top_, last_, cropBack_, cropVal_,
					scaling_);
			// TODO Auto-generated constructor stub
		}
		protected AbstractCalculation observer;


		public void setObserver(AbstractCalculation observer) {
			this.observer = observer;
		}

		public void runAlgorithm() {
			observer.setTotalUnits(100);
			super.runAlgorithm();
			observer.markCompleted();
		}

		/**
		 * Notifies all listeners that have registered interest for notification
		 * on this event type.
		 * 
		 * @param value
		 *            the value of the progress bar.
		 */
		protected void fireProgressStateChanged(int value) {
			super.fireProgressStateChanged(value);
			observer.setCompletedUnits(value);
		}

		/**
		 * Updates listeners of progress status. Without actually changing the
		 * numerical value
		 * 
		 * @param imageName
		 *            the name of the image
		 * @param message
		 *            the new message to display
		 */
		protected void fireProgressStateChanged(String imageName, String message) {
			super.fireProgressStateChanged(imageName, message);
			observer.setLabel(message);
		}
		
	}
	protected class AlgorithmFantasmWrapper extends AlgorithmFantasm {
		/**
		 * @param destImg_
		 * @param srcImg_
		 * @param destNum_
		 * @param init_
		 * @param output_
		 * @param classes_
		 * @param iterMax_
		 * @param distMax_
		 * @param smooth_
		 * @param fuzzy_
		 * @param addOut_
		 * @param outVal_
		 * @param correct_
		 * @param poly_
		 * @param edges_
		 * @param smooth_2
		 * @param contr_
		 * @param pr_
		 * @param cropBack_
		 * @param cropVal_
		 * @param useRel_
		 * @param adaptSmooth_
		 */
		public AlgorithmFantasmWrapper(ModelImage[] destImg_, ModelImage srcImg_, int destNum_, String init_,
				String output_, int classes_, int iterMax_, float distMax_, float smooth_, float fuzzy_,
				boolean addOut_, float outVal_, String correct_, int poly_, boolean edges_, float smooth_2,
				float contr_, float pr_, boolean cropBack_, float cropVal_, boolean useRel_, boolean adaptSmooth_) {
			super(destImg_, srcImg_, destNum_, init_, output_, classes_, iterMax_, distMax_, smooth_, fuzzy_, addOut_, outVal_,
					correct_, poly_, edges_, smooth_2, contr_, pr_, cropBack_, cropVal_, useRel_, adaptSmooth_);
			// TODO Auto-generated constructor stub
		}

		protected AbstractCalculation observer;


		public void setObserver(AbstractCalculation observer) {
			this.observer = observer;
		}

		public void runAlgorithm() {
			observer.setTotalUnits(100);
			super.runAlgorithm();
			observer.markCompleted();
		}

		/**
		 * Notifies all listeners that have registered interest for notification
		 * on this event type.
		 * 
		 * @param value
		 *            the value of the progress bar.
		 */
		protected void fireProgressStateChanged(int value) {
			super.fireProgressStateChanged(value);
			observer.setCompletedUnits(value);
		}

		/**
		 * Updates listeners of progress status. Without actually changing the
		 * numerical value
		 * 
		 * @param imageName
		 *            the name of the image
		 * @param message
		 *            the new message to display
		 */
		protected void fireProgressStateChanged(String imageName, String message) {
			super.fireProgressStateChanged(imageName, message);
			observer.setLabel(message);
		}
	}

	protected void execute(CalculationMonitor monitor) {
		FantasmWrapper toads = new FantasmWrapper();
		monitor.observe(toads);
		toads.execute();
	}

	public Dimension getPreferredSize() {
		return new Dimension(300, 450);
	}
}
