package edu.jhu.ece.iacl.plugins.segmentation.gdm;

import java.awt.Dimension;

import javax.vecmath.Point3f;
import edu.jhu.ece.iacl.algorithms.PrinceGroupAuthors;
import edu.jhu.ece.iacl.algorithms.ReferencedPapers;
import edu.jhu.ece.iacl.algorithms.graphics.isosurf.IsoSurfaceOnGrid;
import edu.jhu.ece.iacl.algorithms.tgdm.*;
import edu.jhu.ece.iacl.algorithms.tgdm.ProfileTGDM.Tracking;
import edu.jhu.ece.iacl.algorithms.topology.ConnectivityRule;
import edu.jhu.ece.iacl.algorithms.volume.DistanceField;
import edu.jhu.ece.iacl.jist.pipeline.AlgorithmInformation;
import edu.jhu.ece.iacl.jist.pipeline.CalculationMonitor;
import edu.jhu.ece.iacl.jist.pipeline.DevelopmentStatus;
import edu.jhu.ece.iacl.jist.pipeline.ProcessingAlgorithm;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamBoolean;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamDouble;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamInteger;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamOption;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamSurface;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamSurfaceCollection;
import edu.jhu.ece.iacl.jist.pipeline.parameter.ParamVolume;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataUByte;
import edu.jhu.ece.iacl.jist.structures.image.VoxelType;

import gov.nih.mipav.model.structures.ModelImage;
import gov.nih.mipav.model.structures.ModelStorageBase;
import gov.nih.mipav.view.ViewUserInterface;

public class MedicAlgorithmNestedTGDM extends ProcessingAlgorithm {
	private ParamVolume gvfVol;
	private ParamVolume gmAceVol;
	private ParamVolume gmVol;
	private ParamVolume wmVol;
	private ParamVolume initVol;
	private ParamVolume skelVol;
	private ParamVolume innerVol;
	private ParamVolume centralVol;
	private ParamVolume outerVol;
	private ParamVolume sphereInnerVol;
	private ParamVolume sphereCentralVol;
	private ParamVolume sphereOuterVol;
	private ParamVolume sgnVol;
	private ParamVolume brainMaskVol;
	private ParamBoolean runInner, runCentral, runOuter;

	private ParamCollection dataCollection;
	private ParamCollection paramCollection;
	private ParamDouble initCurvatureForce;
	private ParamDouble innerCurvatureForce, innerExternalForce,
			innerPressureForce;
	private ParamDouble centralCurvatureForce, centralExternalForce,
			centralPressureForce;
	private ParamDouble outerCurvatureForce, outerExternalForce,
			outerPressureForce;
	private ParamInteger initIterations;
	private ParamInteger innerIterations;
	private ParamInteger centralIterations;
	private ParamInteger outerIterations;
	private ParamBoolean gensurfs;
	private ParamSurface embeddedInnerSurface;
	private ParamSurface innerSurf;
	private ParamSurface centralSurf;
	private ParamSurface outerSurf;
	private ParamOption connectivity;
	private ParamOption tracking;
	private ParamDouble initIsoLevel;

	private ParamBoolean isMembership;
	private ParamDouble isoWm, isoGm;

	private static final String cvsversion = "$Revision: 1.1 $";
	private static final String revnum = cvsversion.replace("Revision: ", "").replace("$", "").replace(" ", "");

	protected void createInputParameters(ParamCollection inputParams) {

		dataCollection = new ParamCollection("Data");
		dataCollection.add(initVol = new ParamVolume("Initial Level Set",
				VoxelType.FLOAT));
		dataCollection.add(isMembership = new ParamBoolean(
				"Initial Volume is Membership [0,255]", true));
		dataCollection.add(wmVol = new ParamVolume("White Matter Membership",
				VoxelType.FLOAT));
		dataCollection.add(gmVol = new ParamVolume("Gray Matter Membership",
				VoxelType.FLOAT));
		dataCollection.add(gmAceVol = new ParamVolume(
				"ACE Gray Matter Membership", VoxelType.FLOAT));
		dataCollection.add(skelVol = new ParamVolume("Thinned ACE Skeleton",
				VoxelType.UBYTE));
		dataCollection.add(gvfVol = new ParamVolume("Gradient Vector Field",
				VoxelType.FLOAT, -1, -1, -1, 3));
		dataCollection.add(brainMaskVol = new ParamVolume(
				"Brain Mask Level Set", VoxelType.FLOAT));
		brainMaskVol.setMandatory(false);
		dataCollection.add(embeddedInnerSurface = new ParamSurface(
				"Embedded Inner Surface"));
		wmVol.setMandatory(false);
		gmVol.setMandatory(false);
		gmAceVol.setMandatory(false);
		skelVol.setMandatory(false);
		gvfVol.setMandatory(false);
		embeddedInnerSurface.setMandatory(false);
		dataCollection.add(tracking = new ParamOption("Tracking", new String[] {
				"Off", "Sphere", "Point", "Shell" }));
		dataCollection.add(runInner = new ParamBoolean(
				"Generate Inner Surface", true));
		dataCollection.add(runCentral = new ParamBoolean(
				"Generate Central Surface", true));
		dataCollection.add(runOuter = new ParamBoolean(
				"Generate Outer Surface", true));
		inputParams.add(dataCollection);
		paramCollection = new ParamCollection("Parameters");
		paramCollection.add(initIsoLevel = new ParamDouble("Initial Iso Level",
				126.9));
		paramCollection.add(isoWm = new ParamDouble("White Matter Iso Level",
				0, 1, 0.5));
		paramCollection.add(isoGm = new ParamDouble("Gray Matter Iso Level", 0,
				1, 0.6));
		paramCollection.add(initCurvatureForce = new ParamDouble(
				"Initial Smoothing Curvature Force", 0, 10,3));

		paramCollection.add(initIterations = new ParamInteger(
				"Iterations for Initial Smoothing", 0, 1000, 6));
		paramCollection.add(innerCurvatureForce = new ParamDouble(
				"Inner Surface Curvature Force", 0, 10, 1.0));
		paramCollection.add(innerPressureForce = new ParamDouble(
				"Inner Surface Pressure Force", 0, 10, 1));
		paramCollection.add(innerExternalForce = new ParamDouble(
				"Inner Surface External Force", 0, 10, 0));
		paramCollection.add(innerIterations = new ParamInteger(
				"Iterations for Inner Surface", 0, 1000, 4));
		paramCollection.add(centralCurvatureForce = new ParamDouble(
				"Central Surface Curvature Force", 0, 10, 0.15));
		paramCollection.add(centralPressureForce = new ParamDouble(
				"Central Surface Pressure Force", 0, 10, 1.5));
		paramCollection.add(centralExternalForce = new ParamDouble(
				"Central Surface External Force", 0, 10, 1));
		paramCollection.add(centralIterations = new ParamInteger(
				"Iterations for Central Surface", 0, 1000, 7));
		paramCollection.add(outerCurvatureForce = new ParamDouble(
				"Outer Surface Curvature Force", 0, 10, 0.25));
		paramCollection.add(outerPressureForce = new ParamDouble(
				"Outer Surface Pressure Force", 0, 10, 1));
		paramCollection.add(outerExternalForce = new ParamDouble(
				"Outer Surface External Force", 0, 10, 0));
		paramCollection.add(outerIterations = new ParamInteger(
				"Iterations for Outer Surface", 0, 1000, 7));
		paramCollection.add(connectivity = new ParamOption(
				"Connectivity (Foreground,Background)", new String[] {
						"(18,6)", "(6,18)", "(26,6)", "(6,26)" }));
		connectivity.setValue(0);
		paramCollection.add(gensurfs = new ParamBoolean(
				"Generate Iso-Surfaces", true));
		inputParams.add(paramCollection);

		this.setPreferredSize(new Dimension(350, 500));

		inputParams.setName("nested_tgdm");
		inputParams.setLabel("Nested TGDM");

		inputParams.setPackage("IACL");
		inputParams.setCategory("Segmentation.Geometric_Deformable_Model");

		AlgorithmInformation info = getAlgorithmInformation();
		info.setWebsite("http://www.iacl.ece.jhu.edu/");
		info.setVersion(revnum);
		info.setEditable(false);
		info.setDescription("Geometric deformable model to find inner, central, and outer coritcal surfaces.");
		info.setDescription("There are slight numerical differences between this implementation and the original implementation, but those differences are far less than the precision of the model (0.1 voxel).");
		info.setStatus(DevelopmentStatus.RC);
		info.add(ReferencedPapers.tgdm);
		info.add(PrinceGroupAuthors.xiaoHan);
		info.add(PrinceGroupAuthors.blakeLucas);
	}

	protected void createOutputParameters(ParamCollection outputParams) {
		outputParams.add(innerVol = new ParamVolume("Inner Level Set",
				VoxelType.FLOAT));
		outputParams.add(centralVol = new ParamVolume("Central Level Set",
				VoxelType.FLOAT));
		outputParams.add(outerVol = new ParamVolume("Outer Level Set",
				VoxelType.FLOAT));
		outputParams.add(sgnVol = new ParamVolume("Signed Volume",
				VoxelType.UBYTE));
		outputParams.add(sphereInnerVol = new ParamVolume("Sphere Inner Map",
				VoxelType.FLOAT, -1, -1, -1, 3));
		outputParams.add(sphereCentralVol = new ParamVolume(
				"Sphere Central Map", VoxelType.FLOAT, -1, -1, -1, 3));
		outputParams.add(sphereOuterVol = new ParamVolume("Sphere Outer Map",
				VoxelType.FLOAT, -1, -1, -1, 3));
		outputParams.add(innerSurf = new ParamSurface("Inner Surface"));
		outputParams.add(centralSurf = new ParamSurface("Central Surface"));
		outputParams.add(outerSurf = new ParamSurface("Outer Surface"));
		innerVol.setMandatory(false);
		centralVol.setMandatory(false);
		outerVol.setMandatory(false);
		sgnVol.setMandatory(false);
		innerSurf.setMandatory(false);
		centralSurf.setMandatory(false);
		outerSurf.setMandatory(false);
		sphereInnerVol.setMandatory(false);
		sphereCentralVol.setMandatory(false);
		sphereOuterVol.setMandatory(false);
		outputParams.setName("nested_tgdm");
		outputParams.setLabel("Nested TGDM");
	}

	protected void execute(CalculationMonitor monitor) {
		int conn = 0;
		switch (connectivity.getIndex()) {
		case 0:
			conn = ConnectivityRule.CONNECT_18_6;
			break;
		case 1:
			conn = ConnectivityRule.CONNECT_6_18;
			break;
		case 2:
			conn = ConnectivityRule.CONNECT_26_6;
			break;
		case 3:
			conn = ConnectivityRule.CONNECT_6_26;
			break;
		default:
			break;
		}
		ImageDataFloat brainMask = (brainMaskVol.getImageData() != null) ? new ImageDataFloat(
				brainMaskVol.getImageData())
				: null;
		if(brainMask!=null)System.out.println("Use brain mask "+brainMask.getName());
		Tracking track = Tracking.values()[tracking.getIndex()];
		// String name = initVol.getImageData().getName();
		String name = gmVol.getImageData().getName();
		// MedicUtil.displayMessage("Before anything in NestedTGDM: name="+name+"\n");
		GenericTGDM tgdm = (track == Tracking.OFF) ? new GenericTGDM(conn)
				: new TrackingTGDM(conn);
		if (track != Tracking.OFF) {
			((TrackingTGDM) tgdm).setTracking(track);
			((TrackingTGDM) tgdm).setMapInterpolationEnabled(false);
			((TrackingTGDM) tgdm).setPdeEnabled(false);
		}
		monitor.observe(tgdm);
		ImageDataFloat inner, central, outer;
		float isomem = isoWm.getFloat();
		if (runInner.getValue()) {
			ImageDataFloat phi;
			if (isMembership.getValue()) {
				ImageData VpMImg = initVol.getImageData();
				int rows = VpMImg.getRows();
				int cols = VpMImg.getCols();
				int slices = VpMImg.getSlices();
				DistanceField df = new DistanceField();
				phi = new ImageDataFloat(VpMImg);
				float[][][] Phi = phi.toArray3d();
				for (int x = 0; x < rows; x++)
					for (int y = 0; y < cols; y++)
						for (int z = 0; z < slices; z++) {
							Phi[x][y][z] = initIsoLevel.getFloat()
									- Phi[x][y][z];
						}
				System.out.println("Fast Marching...");
				phi = df.solve(phi, 8);
				phi = tgdm.solveSmoothSurface(phi, initCurvatureForce
						.getFloat(), initIterations.getInt());
			} else {
				phi = new ImageDataFloat(initVol.getImageData());
			}
			inner = tgdm.solveInnerSurfaceFromOuter(phi, brainMask, wmVol
					.getImageData(), null, innerCurvatureForce.getFloat(),
					innerExternalForce.getFloat(), innerPressureForce
							.getFloat(), initIsoLevel.getFloat(), isomem,
					innerIterations.getInt());

			inner.setName(name + "_innerlevset");
			innerVol.setValue(inner);
			//innerVol.getImageData().getModelImageCopy().copyFileTypeInfo(gmVol.getImageData().getModelImageCopy());
			innerVol.getImageData().setHeader(gmVol.getImageData().getHeader());
			if (gensurfs.getValue()) {
				EmbeddedSurface surf = tgdm.getFinalSurface();
				// MedicUtil.displayMessage("Center of mass = "+surf.getCenterOfMass()+", name="+name+"_InnerSurf\n");
				surf.setName(name + "_InnerSurf");
				innerSurf.setValue(surf);
			}
			if (track != Tracking.OFF) {
				ImageDataFloat vol = ((TrackingTGDM) tgdm).getCorrespondence();
				vol.setName(name + "_map_inner");
				sphereInnerVol.setValue(vol);
			}
		} else {
			inner = new ImageDataFloat(initVol.getImageData());
		}
		ImageData GM = gmAceVol.getImageData();
		ImageData WM = wmVol.getImageData();
		ImageDataFloat signVol = new ImageDataFloat(GM.getRows(), GM.getCols(),
				GM.getSlices());
		float sign, gm, wm, csf;
		float isowm = isoWm.getFloat(), isogm = isoGm.getFloat();
		for (int i = 0; i < signVol.getRows(); i++) {
			for (int j = 0; j < signVol.getCols(); j++) {
				for (int k = 0; k < signVol.getSlices(); k++) {
					gm = (float) GM.getFloat(i, j, k) / 255.0f;
					wm = (float) WM.getFloat(i, j, k) / 255.0f;
					csf = 1.0f - wm - gm;
					if (wm > csf) {
						sign = (wm - isowm) / (1.0f - isowm);
						if (sign < 0)
							sign = 0;
					} else {
						sign = (1 - csf - isogm) / isogm;
						if (sign > 0)
							sign = 0;
					}
					sign = (int) (sign * 127 + 127);
					signVol.set(i, j, k, (int) ((sign < 254) ? sign : 254));
				}
			}
		}
		signVol.setName(name + "_sign");
		sgnVol.setValue(signVol);
		//sgnVol.getImageData().getModelImageCopy().copyFileTypeInfo(gmVol.getImageData().getModelImageCopy());
		sgnVol.getImageData().setHeader(gmVol.getImageData().getHeader());
		if (embeddedInnerSurface.getSurface() != null) {
			EmbeddedSurface sphericalMap = embeddedInnerSurface.getSurface();
			if (track != Tracking.OFF) {
				((TrackingTGDM) tgdm).setPdeEnabled(false);
				((TrackingTGDM) tgdm).setMapInterpolationEnabled(false);
				((TrackingTGDM) tgdm).setSphericalMap(sphericalMap);
			}
		}
		if (runCentral.getValue()) {
			central = tgdm.solveCentralSurface(inner, brainMask, signVol,
					gvfVol.getImageData(), centralCurvatureForce.getFloat(),
					centralExternalForce.getFloat(), centralPressureForce
							.getFloat(), centralIterations.getInt());
			central.setName(name + "_centrallevset");
			centralVol.setValue(central);
			//centralVol.getImageData().getModelImageCopy().copyFileTypeInfo(gmVol.getImageData().getModelImageCopy());
			centralVol.getImageData().setHeader(gmVol.getImageData().getHeader());
			if (gensurfs.getValue()) {
				EmbeddedSurface surf = tgdm.getFinalSurface();
				surf.setName(name + "_CentralSurf");
				centralSurf.setValue(surf);
				surf = tgdm.getInitialSurface();
				surf.setName(name + "_InnerSurf");
				innerSurf.setValue(surf);
			}
			if (track != Tracking.OFF) {
				ImageDataFloat vol = ((TrackingTGDM) tgdm).getCorrespondence();
				vol.setName(name + "_map_central");
				sphereCentralVol.setValue(vol);
				// Initialize spherical map for outer surface if that branch is
				// executed
				((TrackingTGDM) tgdm).setSphericalMap(centralSurf.getSurface());
			}
		} else {
			central = new ImageDataFloat(initVol.getImageData());
		}
		if (runOuter.getValue()) {
			outer = tgdm.solveOuterSurface(central, brainMask, gmVol
					.getImageData(), null, skelVol.getImageData(),
					outerCurvatureForce.getFloat(), outerExternalForce
							.getFloat(), outerPressureForce.getFloat(), isogm,
					outerIterations.getInt());
			outer.setName(name + "_outerlevset");
			outerVol.setValue(outer);
			//outerVol.getImageData().getModelImageCopy().copyFileTypeInfo(gmVol.getImageData().getModelImageCopy());
			outerVol.getImageData().setHeader(gmVol.getImageData().getHeader());
			
			if (gensurfs.getValue()) {
				EmbeddedSurface surf = tgdm.getFinalSurface();
				surf.setName(name + "_OuterSurf");
				outerSurf.setValue(surf);
				surf = tgdm.getInitialSurface();

				centralSurf.setValue(surf);
				centralSurf.getSurface().setName(name + "_CentralSurf");
			}
			if (track != Tracking.OFF) {
				ImageDataFloat vol = ((TrackingTGDM) tgdm).getCorrespondence();
				vol.setName(name + "_map_outer");
				sphereOuterVol.setValue(vol);
			}
		} else {
			outer = new ImageDataFloat(initVol.getImageData());
		}
	}
}
