package edu.jhu.ece.iacl.algorithms.graphics;

import java.util.Arrays;

import javax.vecmath.Point3d;
import Jama.*;

import javax.vecmath.Matrix3d;
import javax.vecmath.Matrix3f;
import javax.vecmath.Point2d;
import javax.vecmath.Point2f;
import javax.vecmath.Point3f;
import javax.vecmath.Tuple2d;
import javax.vecmath.Tuple3d;
import javax.vecmath.Tuple3f;
import javax.vecmath.Vector2d;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;

import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Geometric utilities. There is a similar class in JIST called in GeomUtil that
 * is largely redundant and should eventually be removed or simplified.
 * 
 * @author Blake
 * 
 */
public class GeometricUtilities {
	public static Matrix rotateInverseMatrix(Point3f p) {
		return rotateInverseMatrix(Math.atan2(p.y, p.x), Math.acos(p.z));
	}

	public static double length(Tuple3f p) {
		return Math.sqrt(p.x * p.x + p.y * p.y + p.z * p.z);
	}

	public static double length(Tuple3d p) {
		return Math.sqrt(p.x * p.x + p.y * p.y + p.z * p.z);
	}

	public static double length(Tuple2d p) {
		return Math.sqrt(p.x * p.x + p.y * p.y);
	}

	public static double distance(Tuple3f p1, Tuple3f p2) {
		return Math.sqrt((p1.x - p2.x) * (p1.x - p2.x) + (p1.y - p2.y)
				* (p1.y - p2.y) + (p1.z - p2.z) * (p1.z - p2.z));
	}

	public static double distance(Tuple3f p1, Tuple3d p2) {
		return Math.sqrt((p1.x - p2.x) * (p1.x - p2.x) + (p1.y - p2.y)
				* (p1.y - p2.y) + (p1.z - p2.z) * (p1.z - p2.z));
	}

	public static double distance(Tuple3d p1, Tuple3d p2) {
		return Math.sqrt((p1.x - p2.x) * (p1.x - p2.x) + (p1.y - p2.y)
				* (p1.y - p2.y) + (p1.z - p2.z) * (p1.z - p2.z));
	}

	public static Matrix rotateInverseMatrix(double theta, double phi) {
		double cost = Math.cos(-theta);
		double sint = Math.sin(-theta);
		double cosp = Math.cos(0.5 * Math.PI - phi);
		double sinp = Math.sin(0.5 * Math.PI - phi);
		Matrix rotPhi = new Matrix(new double[][] { { cosp, 0, sinp },
				{ 0, 1, 0 }, { -sinp, 0, cosp } });
		Matrix rotTheta = new Matrix(new double[][] { { cost, -sint, 0 },
				{ sint, cost, 0 }, { 0, 0, 1 } });
		return rotPhi.times(rotTheta);
	}

	public static Matrix3d rotateInverseMatrix3d(Point3f p) {
		return rotateInverseMatrix3d(Math.atan2(p.y, p.x), Math.acos(p.z));
	}

	public static Matrix3d rotateInverseMatrix3d(double theta, double phi) {
		double cost = Math.cos(-theta);
		double sint = Math.sin(-theta);
		double cosp = Math.cos(0.5 * Math.PI - phi);
		double sinp = Math.sin(0.5 * Math.PI - phi);
		Matrix3d rotPhi = new Matrix3d(new double[] { cosp, 0, sinp, 0, 1, 0,
				-sinp, 0, cosp });
		Matrix3d rotTheta = new Matrix3d(new double[] { cost, -sint, 0, sint,
				cost, 0, 0, 0, 1 });
		rotPhi.mul(rotTheta);
		return rotPhi;
	}

	/*
	 * public void rewrap(Point2d p){ if(p.y<0){ p.y=-p.y;
	 * if(p.x<0)p.x+=Math.PI; else p.x-=Math.PI; } if(p.y>Math.PI){
	 * p.y=2*Math.PI-p.y; if(p.x<0)p.x+=Math.PI; else p.x-=Math.PI; }
	 * if(p.x>Math.PI)p.x-=2*Math.PI; if(p.x<-Math.PI)p.x+=2*Math.PI; }
	 */
	public static Point3f multMatrix(Matrix A, Point3f p) {
		if (p == null) {
			return new Point3f();
		} else {
			Matrix v = new Matrix(3, 1);
			v.set(0, 0, p.x);
			v.set(1, 0, p.y);
			v.set(2, 0, p.z);
			Matrix result = A.times(v);
			return new Point3f((float) result.get(0, 0), (float) result.get(1,
					0), (float) result.get(2, 0));
		}
	}

	public static Point3f multMatrix(Matrix A, double[] p) {
		Matrix v = new Matrix(3, 1);
		v.set(0, 0, p[0]);
		v.set(1, 0, p[1]);
		v.set(2, 0, p[2]);
		Matrix result = A.times(v);
		return new Point3f((float) result.get(0, 0), (float) result.get(1, 0),
				(float) result.get(2, 0));
	}

	public static Point3f multMatrix(Matrix3d A, double[] p) {
		double px = p[0];
		double py = p[1];
		double pz = p[2];
		return new Point3f((float) (px * A.m00 + py * A.m01 + pz * A.m02),
				(float) (px * A.m10 + py * A.m11 + pz * A.m12), (float) (px
						* A.m20 + py * A.m21 + pz * A.m22));
	}

	public static Point3d multMatrix3d(Matrix3d A, Point3f p) {
		double px = p.x;
		double py = p.y;
		double pz = p.z;
		return new Point3d((px * A.m00 + py * A.m01 + pz * A.m02), (px * A.m10
				+ py * A.m11 + pz * A.m12), (px * A.m20 + py * A.m21 + pz
				* A.m22));
	}

	public static Point3d multMatrix3d(Matrix3d A, Point3d p) {
		double px = p.x;
		double py = p.y;
		double pz = p.z;
		return new Point3d((px * A.m00 + py * A.m01 + pz * A.m02), (px * A.m10
				+ py * A.m11 + pz * A.m12), (px * A.m20 + py * A.m21 + pz
				* A.m22));
	}

	public static Point3f multMatrix(Matrix3d A, Point3f p) {
		double px = p.x;
		double py = p.y;
		double pz = p.z;
		return new Point3f((float) (px * A.m00 + py * A.m01 + pz * A.m02),
				(float) (px * A.m10 + py * A.m11 + pz * A.m12), (float) (px
						* A.m20 + py * A.m21 + pz * A.m22));
	}

	public static Vector3f multMatrix(Matrix3d A, Vector3f p) {
		double px = p.x;
		double py = p.y;
		double pz = p.z;
		return new Vector3f((float) (px * A.m00 + py * A.m01 + pz * A.m02),
				(float) (px * A.m10 + py * A.m11 + pz * A.m12), (float) (px
						* A.m20 + py * A.m21 + pz * A.m22));
	}

	public static Point2d sphereToStereo(Tuple3f p) {
		return new Point2d((p.z != 1) ? p.x / (1 - p.z) : p.x * 1E20,
				(p.z != 1) ? p.y / (1 - p.z) : p.y * 1E20);
	}

	public static Point3f stereoToSphere(Tuple2d p) {
		double r = (1 + p.x * p.x + p.y * p.y);
		if (Math.abs(r) > 1E-10) {
			return new Point3f((float) (2 * p.x / r), (float) (2 * p.y / r),
					(float) ((p.x * p.x + p.y * p.y - 1) / r));
		} else {
			return new Point3f(0, 0, 1);
		}
	}

	public static Point3d multMatrix(Matrix3d A, Point3d p) {
		double px = p.x;
		double py = p.y;
		double pz = p.z;
		return new Point3d((px * A.m00 + py * A.m01 + pz * A.m02), (px * A.m10
				+ py * A.m11 + pz * A.m12), (px * A.m20 + py * A.m21 + pz
				* A.m22));
	}

	public static Point2d toUnitSpherical(Tuple3f p) {
		return new Point2d(Math.atan2(p.y, p.x), Math.acos(p.z));
	}

	public static Point3d toSpherical(Tuple3f p) {
		double r = length(p);
		if (r < 1E-10) {
			return new Point3d(0, 0, 0);
		} else {
			return new Point3d(r, Math.atan2(p.y / r, p.x / r), Math.acos(p.z
					/ r));
		}
	}

	public static Point3f toCartesian(Point2d r) {
		return new Point3f((float) (Math.cos(r.x) * Math.sin(r.y)),
				(float) (Math.sin(r.x) * Math.sin(r.y)), (float) Math.cos(r.y));
	}

	public static Point3f toCartesian(Point3d r) {
		return new Point3f((float) (r.x * Math.cos(r.y) * Math.sin(r.z)),
				(float) (r.x * Math.sin(r.y) * Math.sin(r.z)),
				(float) (r.x * Math.cos(r.z)));
	}

	public static Point3f toCartesian(Point2f r) {
		return new Point3f((float) (Math.cos(r.x) * Math.sin(r.y)),
				(float) (Math.sin(r.x) * Math.sin(r.y)), (float) Math.cos(r.y));
	}

	public static double normalizedAngle(Tuple3f p1, Tuple3f p2) {
		return Math.acos(Math.max(-1, Math.min(1, p1.x * p2.x + p1.y * p2.y
				+ p1.z * p2.z)));
	}

	public static double normalize(Tuple3f p1) {
		double l1 = Math.sqrt(p1.x * p1.x + p1.y * p1.y + p1.z * p1.z);
		p1.scale((float) ((l1 > 1E-6) ? 1 / l1 : 0));
		return l1;
	}

	public static void normalize(Tuple3d p1) {
		double l1 = Math.sqrt(p1.x * p1.x + p1.y * p1.y + p1.z * p1.z);
		p1.scale(((l1 > 1E-6) ? 1 / l1 : 0));
	}

	public static void normalizeSqr(Tuple3f p1) {
		double l1 = p1.x * p1.x + p1.y * p1.y + p1.z * p1.z;
		p1.scale((float) ((l1 > 1E-6) ? 1 / l1 : 0));
	}

	public static void normalizeSqr(Tuple3d p1) {
		double l1 = p1.x * p1.x + p1.y * p1.y + p1.z * p1.z;
		p1.scale(((l1 > 1E-6) ? 1 / l1 : 0));
	}

	public static double angle(Tuple3f p1, Tuple3f p2) {
		double l1 = Math.sqrt(p1.x * p1.x + p1.y * p1.y + p1.z * p1.z);
		double l2 = Math.sqrt(p2.x * p2.x + p2.y * p2.y + p2.z * p2.z);
		return Math.acos(Math.min(1, Math.max(-1,
				(p1.x * p2.x + p1.y * p2.y + p1.z * p2.z) / (l1 * l2))));
	}

	public static double angle(Tuple2d p1, Tuple2d p2) {
		double l1 = Math.sqrt(p1.x * p1.x + p1.y * p1.y);
		double l2 = Math.sqrt(p2.x * p2.x + p2.y * p2.y);
		return Math.acos(Math.min(1, Math.max(-1, (p1.x * p2.x + p1.y * p2.y)
				/ (l1 * l2))));
	}

	public static double sphericalAngle(Vector3d v2, Vector3d v1, Vector3d v3) {
		Vector3d v12 = new Vector3d();
		Vector3d v23 = new Vector3d();
		v12.cross(v1, v2);
		v23.cross(v3, v2);
		double ang = -Math.atan2(v2.length() * v1.dot(v23), v12.dot(v23));
		if (ang < -1E-5) {
			return 2 * Math.PI + ang;
		} else
			return ang;
	}

	public static double sphericalAngle(Vector3f v2, Vector3f v1, Vector3f v3) {
		Vector3f v12 = new Vector3f();
		Vector3f v23 = new Vector3f();
		v12.cross(v1, v2);
		v23.cross(v3, v2);
		double ang = -Math.atan2(v2.length() * v1.dot(v23), v12.dot(v23));
		if (ang < -1E-5) {
			return 2 * Math.PI + ang;
		} else
			return ang;
	}

	public static Vector3f toCartesianVector(Point2d r) {
		return new Vector3f((float) (Math.cos(r.x) * Math.sin(r.y)),
				(float) (Math.sin(r.x) * Math.sin(r.y)), (float) Math.cos(r.y));
	}

	public static Matrix rotateMatrix(double theta, double phi) {
		double cost = Math.cos(theta);
		double sint = Math.sin(theta);
		double cosp = Math.cos(phi - 0.5 * Math.PI);
		double sinp = Math.sin(phi - 0.5 * Math.PI);
		Matrix rotPhi = new Matrix(new double[][] { { cosp, 0, sinp },
				{ 0, 1, 0 }, { -sinp, 0, cosp } });
		Matrix rotTheta = new Matrix(new double[][] { { cost, -sint, 0 },
				{ sint, cost, 0 }, { 0, 0, 1 } });
		return rotTheta.times(rotPhi);
	}

	public static Matrix3d rotateMatrix3d(double theta, double phi) {
		double cost = Math.cos(theta);
		double sint = Math.sin(theta);
		double cosp = Math.cos(phi - 0.5 * Math.PI);
		double sinp = Math.sin(phi - 0.5 * Math.PI);
		Matrix3d rotPhi = new Matrix3d(new double[] { cosp, 0, sinp, 0, 1, 0,
				-sinp, 0, cosp });
		Matrix3d rotTheta = new Matrix3d(new double[] { cost, -sint, 0, sint,
				cost, 0, 0, 0, 1 });
		Matrix3d m = new Matrix3d();
		m.mul(rotTheta, rotPhi);
		return m;
	}

	public static Matrix3f rotateMatrix3f(double theta, double phi) {
		float cost = (float) Math.cos(theta);
		float sint = (float) Math.sin(theta);
		float cosp = (float) Math.cos(phi - 0.5 * Math.PI);
		float sinp = (float) Math.sin(phi - 0.5 * Math.PI);
		Matrix3f rotPhi = new Matrix3f(new float[] { cosp, 0, sinp, 0, 1, 0,
				-sinp, 0, cosp });
		Matrix3f rotTheta = new Matrix3f(new float[] { cost, -sint, 0, sint,
				cost, 0, 0, 0, 1 });
		Matrix3f m = new Matrix3f();
		m.mul(rotTheta, rotPhi);
		return m;
	}

	public static Vector3f slerp(Tuple3f v1, Tuple3f v2, double t) {
		double l1 = Math.abs(GeometricUtilities.length(v1) - 1);
		double l2 = Math.abs(GeometricUtilities.length(v2) - 1);
		if (l1 > 0.01) {
			return new Vector3f(v2);
		} else if (l2 > 0.01) {
			return new Vector3f(v1);
		}
		double ang = angle(v1, v2);
		double sina = Math.sin(ang);
		double w1 = (Math.abs(sina) < 1E-10) ? (1 - t) : Math
				.sin(ang * (1 - t))
				/ sina;
		double w2 = (Math.abs(sina) < 1E-10) ? t : Math.sin(ang * (t)) / sina;
		Vector3f v3 = new Vector3f((float) (w1 * v1.x + w2 * v2.x), (float) (w1
				* v1.y + w2 * v2.y), (float) (w1 * v1.z + w2 * v2.z));
		normalize(v3);
		return v3;
	}

	public static Vector3f interp(Tuple3f v1, Tuple3f v2, double t) {
		double w1 = (1 - t);
		double w2 = t;
		Vector3f v3 = new Vector3f((float) (w1 * v1.x + w2 * v2.x), (float) (w1
				* v1.y + w2 * v2.y), (float) (w1 * v1.z + w2 * v2.z));
		return v3;
	}

	public static Vector2d slerp(Tuple2d v1, Tuple2d v2, double t) {
		double ang = angle(v1, v2);
		double sina = Math.sin(ang);
		double w1 = (Math.abs(sina) < 1E-10) ? (1 - t) : Math
				.sin(ang * (1 - t))
				/ sina;
		double w2 = (Math.abs(sina) < 1E-10) ? t : Math.sin(ang * (t)) / sina;
		Vector2d v3 = new Vector2d((float) (w1 * v1.x + w2 * v2.x), (float) (w1
				* v1.y + w2 * v2.y));
		v3.normalize();
		return v3;
	}

	public static Point2d[] unwrap(Point3f p1, Point3f p2, Point3f p3) {
		Point2d s1 = toUnitSpherical(p1);
		Matrix A = rotateInverseMatrix(s1.x, s1.y);
		s1 = toUnitSpherical(multMatrix(A, p1));
		Point2d s2 = toUnitSpherical(multMatrix(A, p2));
		Point2d s3 = toUnitSpherical(multMatrix(A, p3));
		return new Point2d[] { s1, s2, s3 };
	}

	public static synchronized double angle(Point3f v0, Point3f v1, Point3f v2) {
		Vector3f v = new Vector3f();
		Vector3f w = new Vector3f();
		v.sub(v0, v2);
		w.sub(v1, v2);
		double len = v.length();
		return (len > 1E-8) ? Math.acos(v.dot(w) / (len * len)) : 0;
	}

	// Bisect angle at point 3
	public static Point3f angleBisect(Point3f p1, Point3f p2, Point3f p3) {
		Vector3f v1 = new Vector3f();
		Vector3f v2 = new Vector3f();
		Vector3f v3 = new Vector3f();
		Vector3f v4 = new Vector3f();
		v1.sub(p1, p3);
		v3.sub(p1, p3);

		v2.sub(p2, p3);
		v4.sub(p1, p2);

		v1.normalize();
		v2.normalize();

		v2.sub(v2, v1);
		double t = v3.dot(v2) / v4.dot(v2);

		v4.scale(-(float) t);

		Point3f p = new Point3f(p1);
		p.add(v4);
		return p;
	}

	public static double intersectionTime(Tuple2d x1, Tuple2d v1, Tuple2d x2,
			Tuple2d v2) {
		Vector2d v3 = new Vector2d();
		v3.sub(x2, x1);
		double denom = (v1.x * v2.y - v2.x * v1.y);
		return (denom != 0) ? (v3.x * v2.y - v2.x * v3.y) / denom : 1E30;
	}

	public static double intersectionAngle(Vector3f x1, Vector3f x2,
			Vector3f x3, Vector3f x4) {
		Vector3f n1 = new Vector3f();
		n1.cross(x1, x2);
		Vector3f n2 = new Vector3f();
		n2.cross(x3, x4);
		Vector3f x0 = new Vector3f();
		x0.cross(n1, n2);
		x0.normalize();
		if (x0.dot(x1) < 0) {
			x0.negate();
		}
		n2.cross(x1, x0);
		return Math.signum(n1.dot(n2)) * GeometricUtilities.angle(x1, x0);
	}

	public static double triangleArea(Tuple3f v0, Tuple3f v1, Tuple3f v2) {
		Vector3f v = new Vector3f();
		Vector3f w = new Vector3f();
		Vector3f z = new Vector3f();
		v.sub(v0, v2);
		w.sub(v1, v2);
		z.cross(v, w);
		return 0.5 * z.length();
	}

	public static double triangleArea(Tuple2d v1, Tuple2d v2, Tuple2d v3) {
		return 0.5 * Math.abs(-v2.x * v1.y + v3.x * v1.y + v1.x * v2.y - v3.x
				* v2.y - v1.x * v3.y + v2.x * v3.y);
	}

	public static Point3f triangleCenter(Tuple3f v0, Tuple3f v1, Tuple3f v2) {
		Point3f avg = new Point3f();
		avg.add(v0);
		avg.add(v1);
		avg.add(v2);
		avg.scale(0.333333333f);
		return avg;
	}

	public static double tanAngle(Tuple3f v0, Tuple3f v1, Tuple3f v2) {
		Vector3f v = new Vector3f();
		Vector3f w = new Vector3f();
		v.sub(v0, v2);
		w.sub(v1, v2);
		double len = v.length() * w.length();
		if (len > 0) {
			double dot = Math.max(-0.9999, Math.min(v.dot(w) / len, 0.9999));
			return Math.sqrt(1 - dot * dot) / dot;
		} else {
			return 0;
		}
	}

	public static double tanHalfAngle(Tuple3f v0, Tuple3f v1, Tuple3f v2) {
		Vector3f v = new Vector3f();
		Vector3f w = new Vector3f();
		v.sub(v0, v2);
		w.sub(v1, v2);
		double len = v.length() * w.length();
		if (len > 0) {
			double dot = Math.max(-0.9999, Math.min(v.dot(w) / len, 0.9999));
			return (1 - dot) / Math.sqrt(1 - dot * dot);
		} else {
			return 0;
		}
	}

	public static double cotAngle(Tuple3f v0, Tuple3f v1, Tuple3f v2) {
		Vector3f v = new Vector3f();
		Vector3f w = new Vector3f();
		v.sub(v0, v2);
		w.sub(v1, v2);
		double len = v.length() * w.length();
		if (len > 0) {
			double dot = Math.max(-0.9999, Math.min(v.dot(w) / len, 0.9999));
			return dot / Math.sqrt(1 - dot * dot);
		} else {
			return 0;
		}
	}

	public static double crossProduct(Tuple2d u, Tuple2d v) {
		return u.x * v.y - v.x * u.y;
	}

	public static double dot(Tuple3d u, Tuple3d v) {
		return u.x * v.x + u.y * v.y + u.z * v.z;
	}

	public static double dot(Tuple3f u, Tuple3f v) {
		return u.x * v.x + u.y * v.y + u.z * v.z;
	}

	public static double cosAngle(Vector3f v1, Vector3f v2) {
		return v1.dot(v2) / (v1.length() * v2.length());
	}

	public static double cosAngle(Vector2d v1, Vector2d v2) {
		return v1.dot(v2) / (v1.length() * v2.length());
	}

	public static Matrix jacobianStereoToCartesian(Point3f q1, Point3f q2,
			Point3d q3, Point2d p1, Point2d p2, Point2d p3) {
		Point3f jacx = new Point3f();
		float A = (float) (1.0 / ((p2.x - p1.x) * (p3.y - p1.y) - (p3.x - p1.x)
				* (p2.y - p1.y)));
		jacx.x = A
				* (float) (q1.x * (p2.y - p3.y) + q2.x * (p3.y - p1.y) + q3.x
						* (p1.y - p2.y));
		jacx.y = A
				* (float) (q1.y * (p2.y - p3.y) + q2.y * (p3.y - p1.y) + q3.y
						* (p1.y - p2.y));
		jacx.z = A
				* (float) (q1.z * (p2.y - p3.y) + q2.z * (p3.y - p1.y) + q3.z
						* (p1.y - p2.y));

		Point3f jacy = new Point3f();
		jacy.x = A
				* (float) (q1.x * (p3.x - p2.x) + q2.x * (p1.x - p3.x) + q3.x
						* (p2.x - p1.x));
		jacy.y = A
				* (float) (q1.y * (p3.x - p2.x) + q2.y * (p1.x - p3.x) + q3.y
						* (p2.x - p1.x));
		jacy.z = A
				* (float) (q1.z * (p3.x - p2.x) + q2.z * (p1.x - p3.x) + q3.z
						* (p2.x - p1.x));
		Matrix m = new Matrix(3, 2);
		m.set(0, 0, jacx.x);
		m.set(1, 0, jacx.y);
		m.set(2, 0, jacx.z);

		m.set(0, 1, jacy.x);
		m.set(1, 1, jacy.y);
		m.set(2, 1, jacy.z);
		return m;
	}

	public static Matrix jacobianStereoToSphere(Point2d p) {
		Matrix J = new Matrix(3, 2);
		double r = (1 + p.x * p.x + p.y * p.y);
		r = 1 / (r * r);
		J.set(0, 0, -2 * (p.x * p.x - p.y * p.y - 1) * r);
		J.set(1, 0, -4 * p.x * p.y * r);
		J.set(2, 0, 4 * p.x * r);

		J.set(0, 1, -4 * p.x * p.y * r);
		J.set(1, 1, -2 * (p.y * p.y - p.x * p.x - 1) * r);
		J.set(2, 1, 4 * p.y * r);
		return J;
	}

	public static Matrix jacobianSphereToStereo(Point2d p) {
		Matrix J = new Matrix(2, 3);
		double r = (1 + p.x * p.x + p.y * p.y);
		r = (r * r);
		J.set(0, 0, r / (-2 * (p.x * p.x - p.y * p.y - 1)));
		J.set(0, 1, r / (-4 * p.x * p.y));
		J.set(0, 2, r / (4 * p.x));

		J.set(1, 0, r / (-4 * p.x * p.y));
		J.set(1, 1, r / (-2 * (p.y * p.y - p.x * p.x - 1)));
		J.set(1, 2, r / (4 * p.y));
		return J;
	}

	public static Matrix jacobianCartesianToSphere(Point3f q1, Point3f q2,
			Point3d q3, Point2d p1, Point2d p2, Point2d p3, Point2d p) {
		Matrix J1 = jacobianCartesianToStereo(q1, q2, q3, p1, p2, p3);
		Matrix J2 = jacobianStereoToSphere(p);
		Matrix J3 = J2.times(J1);
		return J3;
	}

	public static double fractionalAnisotropy(Matrix m) {
		EigenvalueDecomposition ed = new EigenvalueDecomposition(m);
		Matrix D = ed.getD();
		double l1 = Math.abs(D.get(0, 0));
		double l2 = Math.abs(D.get(1, 1));
		double l3 = Math.abs(D.get(2, 2));
		double lm = (l1 + l2 + l3) * 0.3333333333;
		return Math.sqrt((1.5)
				* ((l1 - lm) * (l1 - lm) + (l2 - lm) * (l2 - lm) + (l3 - lm)
						* (l3 - lm)) / (l1 * l1 + l2 * l2 + l3 * l3));
	}

	public static double meanDiffusivity(Matrix m) {
		EigenvalueDecomposition ed = new EigenvalueDecomposition(m);
		Matrix D = ed.getD();
		double l1 = Math.abs(D.get(0, 0));
		double l2 = Math.abs(D.get(1, 1));
		double l3 = Math.abs(D.get(2, 2));
		double lm = (l1 + l2 + l3) * 0.3333333333;
		return lm;
	}

	public static double meanStretch(Matrix m) {
		EigenvalueDecomposition ed = new EigenvalueDecomposition(m);
		Matrix D = ed.getD();
		double l1 = D.get(0, 0);
		double l2 = D.get(1, 1);
		double l3 = D.get(2, 2);
		return Math.sqrt((l1 * l1 + l2 * l2 + l3 * l3) * 0.3333333333);
	}

	public static double maxStretch(Matrix m) {
		EigenvalueDecomposition ed = new EigenvalueDecomposition(m);
		Matrix D = ed.getD();
		double l1 = Math.abs(D.get(0, 0));
		double l2 = Math.abs(D.get(1, 1));
		double l3 = Math.abs(D.get(2, 2));
		return Math.max(Math.max(l1, l2), l3);
	}

	public static Matrix triangle3Dto2D(Point3f pt1, Point3f pt2, Point3f pt3) {
		Matrix M = new Matrix(3, 2);
		Vector3f v1 = new Vector3f();
		Vector3f v2 = new Vector3f();
		Vector3f v3 = new Vector3f();
		Vector3f v4 = new Vector3f();
		v1.sub(pt2, pt1);
		v2.sub(pt3, pt1);
		v3.cross(v1, v2);
		v4.cross(v3, v1);
		double l = GeometricUtilities.normalize(v1);
		GeometricUtilities.normalize(v4);
		M.set(0, 0, 0);
		M.set(0, 1, 0);
		M.set(1, 0, l);
		M.set(1, 1, 0);
		M.set(2, 0, v1.dot(v2));
		M.set(2, 1, v4.dot(v2));

		return M;
	}

	private static Matrix PtoP3d = new Matrix(new double[][] { { 0, 1, -1 },
			{ -1, 0, 1 }, { 1, -1, 0 } });
	private static Matrix PtoP2d = new Matrix(new double[][] { { 0, -1 },
			{ 1, 0 } });

	public static Matrix jacobian3dto3d(Point3f p1, Point3f p2, Point3f p3,
			Point3f q1, Point3f q2, Point3f q3) {

		Matrix M1 = triangle3Dto2D(p1, p2, p3);
		float A = (float) ((M1.get(1, 0) - M1.get(0, 0))
				* (M1.get(2, 1) - M1.get(0, 1)) - (M1.get(2, 0) - M1.get(0, 0))
				* (M1.get(1, 1) - M1.get(0, 1)));
		Matrix M2 = triangle3Dto2D(q1, q2, q3);
		Matrix J = PtoP2d.times(M1.transpose()).times(PtoP3d).times(M2).times(
				(Math.abs(A) > 1E-10) ? 1 / A : 0);
		return J;
	}

	public static double[] jacobianEigenValues2D(Matrix J) {
		double E = J.get(0, 0) * J.get(0, 0) + J.get(1, 0) * J.get(1, 0);
		double F = J.get(0, 0) * J.get(0, 1) + J.get(1, 0) * J.get(1, 1);
		double G = J.get(0, 1) * J.get(0, 1) + J.get(1, 1) * J.get(1, 1);
		double tmp = Math.sqrt((E - G) * (E - G) + 4 * F * F);
		double sig1 = Math.sqrt(Math.abs(0.5 * (E + G) + tmp));
		double sig2 = Math.sqrt(Math.abs(0.5 * (E + G) - tmp));
		if (sig1 < sig2) {
			double tmp2 = sig1;
			sig1 = sig2;
			sig2 = tmp2;
		}
		double[] sigs = new double[] { sig1, sig2 };
		return sigs;
	}

	public static Matrix jacobianCartesianToStereo(Point3f q1, Point3f q2,
			Point3d q3, Point2d p1, Point2d p2, Point2d p3) {
		Point3f jacx = new Point3f();
		float A = (float) (1.0 / ((p2.x - p1.x) * (p3.y - p1.y) - (p3.x - p1.x)
				* (p2.y - p1.y)));
		jacx.x = A
				* (float) (q1.x * (p2.y - p3.y) + q2.x * (p3.y - p1.y) + q3.x
						* (p1.y - p2.y));
		jacx.y = A
				* (float) (q1.y * (p2.y - p3.y) + q2.y * (p3.y - p1.y) + q3.y
						* (p1.y - p2.y));
		jacx.z = A
				* (float) (q1.z * (p2.y - p3.y) + q2.z * (p3.y - p1.y) + q3.z
						* (p1.y - p2.y));

		Point3f jacy = new Point3f();
		jacy.x = A
				* (float) (q1.x * (p3.x - p2.x) + q2.x * (p1.x - p3.x) + q3.x
						* (p2.x - p1.x));
		jacy.y = A
				* (float) (q1.y * (p3.x - p2.x) + q2.y * (p1.x - p3.x) + q3.y
						* (p2.x - p1.x));
		jacy.z = A
				* (float) (q1.z * (p3.x - p2.x) + q2.z * (p1.x - p3.x) + q3.z
						* (p2.x - p1.x));
		Matrix m = new Matrix(2, 3);
		m.set(0, 0, 1 / jacx.x);
		m.set(0, 1, 1 / jacx.y);
		m.set(0, 2, 1 / jacx.z);

		m.set(1, 0, 1 / jacy.x);
		m.set(1, 1, 1 / jacy.y);
		m.set(1, 2, 1 / jacy.z);
		return m;
	}

	public static double determinant(double fX0, double fY0, double fZ0,
			double fW0, double fX1, double fY1, double fZ1, double fW1,
			double fX2, double fY2, double fZ2, double fW2, double fX3,
			double fY3, double fZ3, double fW3) {
		double fA0 = fX0 * fY1 - fX1 * fY0;
		double fA1 = fX0 * fY2 - fX2 * fY0;
		double fA2 = fX0 * fY3 - fX3 * fY0;
		double fA3 = fX1 * fY2 - fX2 * fY1;
		double fA4 = fX1 * fY3 - fX3 * fY1;
		double fA5 = fX2 * fY3 - fX3 * fY2;
		double fB0 = fZ0 * fW1 - fZ1 * fW0;
		double fB1 = fZ0 * fW2 - fZ2 * fW0;
		double fB2 = fZ0 * fW3 - fZ3 * fW0;
		double fB3 = fZ1 * fW2 - fZ2 * fW1;
		double fB4 = fZ1 * fW3 - fZ3 * fW1;
		double fB5 = fZ2 * fW3 - fZ3 * fW2;
		return fA0 * fB5 - fA1 * fB4 + fA2 * fB3 + fA3 * fB2 - fA4 * fB1 + fA5
				* fB0;
	}

	public static double determinant(double fX0, double fY0, double fZ0,
			double fX1, double fY1, double fZ1, double fX2, double fY2,
			double fZ2) {
		double fC00 = fY1 * fZ2 - fY2 * fZ1;
		double fC01 = fY2 * fZ0 - fY0 * fZ2;
		double fC02 = fY0 * fZ1 - fY1 * fZ0;
		return fX0 * fC00 + fX1 * fC01 + fX2 * fC02;
	}

	public static int relationToCircumsphere(Point3f P, Point3f V0, Point3f V1,
			Point3f V2, Point3f V3) {
		double fS0x = V0.x + P.x;
		double fD0x = V0.x - P.x;
		double fS0y = V0.y + P.y;
		double fD0y = V0.y - P.y;
		double fS0z = V0.z + P.z;
		double fD0z = V0.z - P.z;
		double fS1x = V1.x + P.x;
		double fD1x = V1.x - P.x;
		double fS1y = V1.y + P.y;
		double fD1y = V1.y - P.y;
		double fS1z = V1.z + P.z;
		double fD1z = V1.z - P.z;
		double fS2x = V2.x + P.x;
		double fD2x = V2.x - P.x;
		double fS2y = V2.y + P.y;
		double fD2y = V2.y - P.y;
		double fS2z = V2.z + P.z;
		double fD2z = V2.z - P.z;
		double fS3x = V3.x + P.x;
		double fD3x = V3.x - P.x;
		double fS3y = V3.y + P.y;
		double fD3y = V3.y - P.y;
		double fS3z = V3.z + P.z;
		double fD3z = V3.z - P.z;
		double fW0 = fS0x * fD0x + fS0y * fD0y + fS0z * fD0z;
		double fW1 = fS1x * fD1x + fS1y * fD1y + fS1z * fD1z;
		double fW2 = fS2x * fD2x + fS2y * fD2y + fS2z * fD2z;
		double fW3 = fS3x * fD3x + fS3y * fD3y + fS3z * fD3z;
		double fDet4 = GeometricUtilities.determinant(fD0x, fD0y, fD0z, fW0,
				fD1x, fD1y, fD1z, fW1, fD2x, fD2y, fD2z, fW2, fD3x, fD3y, fD3z,
				fW3);
		return ((fDet4 > (double) 0.0) ? 1 : ((fDet4 < (double) 0.0) ? -1 : 0));
	}

	public static int relationToPlane(Point3f rkP, Point3f rkV0, Point3f rkV1,
			Point3f rkV2) {
		double fX0 = rkP.x - rkV0.x;
		double fY0 = rkP.y - rkV0.y;
		double fZ0 = rkP.z - rkV0.z;
		double fX1 = rkV1.x - rkV0.x;
		double fY1 = rkV1.y - rkV0.y;
		double fZ1 = rkV1.z - rkV0.z;
		double fX2 = rkV2.x - rkV0.x;
		double fY2 = rkV2.y - rkV0.y;
		double fZ2 = rkV2.z - rkV0.z;
		double fDet3 = GeometricUtilities.determinant(fX0, fY0, fZ0, fX1, fY1,
				fZ1, fX2, fY2, fZ2);
		return ((fDet3 > (double) 0.0) ? +1 : ((fDet3 < (double) 0.0) ? -1 : 0));
	}

	public static Point3d getBaryCoords(Tuple2d p, Tuple2d pt1, Tuple2d pt2,
			Tuple2d pt3) {
		double a = triangleArea(pt1, pt2, pt3);
		Point3d bary = new Point3d();
		if (a > 0) {
			bary.x = Math.max(0, Math.min(1, triangleArea(p, pt2, pt3) / a));
			bary.y = Math.max(0, Math.min(1, triangleArea(p, pt3, pt1) / a));
		}
		bary.z = 1 - bary.x - bary.y;
		return bary;
	}

	public static Point3d getBaryCoords(Tuple3f p, Tuple3f pt1, Tuple3f pt2,
			Tuple3f pt3) {
		double a = pt1.x - pt3.x;
		double b = pt2.x - pt3.x;
		double c = pt3.x - p.x;

		double d = pt1.y - pt3.y;
		double e = pt2.y - pt3.y;
		double f = pt3.y - p.y;

		double g = pt1.z - pt3.z;
		double h = pt2.z - pt3.z;
		double i = pt3.z - p.z;

		double l1 = (b * (f + i) - c * (e + h)) / (a * (e + h) - b * (d + g));
		double l2 = (a * (f + i) - c * (d + g)) / (b * (d + g) - a * (e + h));
		if (Double.isNaN(l1) || Double.isInfinite(l1)) {
			l1 = 0;
		}
		if (Double.isNaN(l2) || Double.isInfinite(l2)) {
			l2 = 0;
		}
		if (l1 > 1 || l2 > 1 || l1 + l2 > 1 || l1 < 0 || l2 < 0) {
			l1 = Math.max(Math.min(l1, 1), 0);
			l2 = Math.max(Math.min(l2, 1), 0);
			if (l1 + l2 > 1) {
				double diff = 0.5 * (1 - l1 - l2);
				l1 += diff;
				l2 += diff;
			}
		}
		return new Point3d(l1, l2, 1 - l1 - l2);
	}

	public static Vector3f interpolateVectorFromBary(Point3d b, Tuple3f p1,
			Tuple3f p2, Tuple3f p3) {
		return new Vector3f((float) (p1.x * b.x + p2.x * b.y + p3.x * b.z),
				(float) (p1.y * b.x + p2.y * b.y + p3.y * b.z), (float) (p1.z
						* b.x + p2.z * b.y + p3.z * b.z));
	}

	public static Point3f interpolatePointFromBary(Point3d b, Tuple3f p1,
			Tuple3f p2, Tuple3f p3) {
		return new Point3f((float) (p1.x * b.x + p2.x * b.y + p3.x * b.z),
				(float) (p1.y * b.x + p2.y * b.y + p3.y * b.z), (float) (p1.z
						* b.x + p2.z * b.y + p3.z * b.z));
	}

	public static double interpolateQuantityFromBary(Point3d b, double v1,
			double v2, double v3) {
		return v1 * b.x + v2 * b.y + v3 * b.z;
	}

	public static double[] interpolateVectorFromBary(Point3d b, double[] v1,
			double[] v2, double[] v3) {
		double[] v = new double[v1.length];
		for (int i = 0; i < v.length; i++) {
			v[i] = v1[i] * b.x + v2[i] * b.y + v3[i] * b.z;
		}
		return v;
	}
	public static double[] interpolateVectorFromBary(Point3d b, double[] v1,
			double[] v2, double[] v3,boolean linearInterpolation) {
		double[] v = new double[v1.length];
		if(linearInterpolation){
			for (int i = 0; i < v.length; i++) {
				v[i] = v1[i] * b.x + v2[i] * b.y + v3[i] * b.z;
			}
		} else {
			for (int i = 0; i < v.length; i++) {
				if(b.x>b.y){
					if(b.x>b.z){
						v[i]=v1[i];
					} else {
						v[i]=v3[i];
					}
				} else {
					if(b.y>b.z){
						v[i]=v2[i];
					} else {
						v[i]=v3[i];
					}
				}
			}
		}
		return v;
	}
	public static Point2d getPointFromBary(Point3d b, Tuple2d p1, Tuple2d p2,
			Tuple2d p3) {
		return new Point2d((p1.x * b.x + p2.x * b.y + p3.x * b.z), (p1.y * b.x
				+ p2.y * b.y + p3.y * b.z));
	}

	public static void mapSphereToSurf(EmbeddedSurface sphere,
			EmbeddedSurface surf) {
		int vertCount = surf.getVertexCount();
		double[][] vertData = new double[vertCount][3];
		for (int i = 0; i < vertCount; i++) {
			Point3f pt = sphere.getVertex(i);
			vertData[i][0] = pt.x;
			vertData[i][1] = pt.y;
			vertData[i][2] = pt.z;
		}
		surf.setVertexData(vertData);
	}
}
