package edu.jhu.ece.iacl.algorithms.graphics.edit;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.Stack;

import javax.vecmath.Point3d;
import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SelfIntersectionTriangle;
import edu.jhu.ece.iacl.algorithms.graphics.surf.HierarchicalSurface;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface;
import edu.jhu.ece.iacl.algorithms.graphics.surf.SelfIntersectingPreventionProgressiveSurface;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface.EdgeCollapse;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface.MeshTopologyOperation;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface.SurfaceDistanceVertexMetric;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.data.BinaryMinHeap;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.VertexIndexed;

/**
 * Re-map two surfaces that are in 1-to-1 correspondence so that a mesh has the
 * same parameterization as the other mesh.
 * 
 * @author Blake Lucas
 */
public class RemapCoupledSurfaces extends AbstractCalculation{
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.1 $");
	}
	protected EmbeddedSurface source;
	protected EmbeddedSurface target;
	protected EmbeddedSurface refSource;
	protected EmbeddedSurface refTarget;
	protected EmbeddedSurface remappedSource = null;
	protected EmbeddedSurface remappedTarget = null;
	protected static int THICKNESS = 0;
	protected static int MATCHED_POINT = 1;
	protected static int FID = 4;
	protected boolean linearInterpolation=false;
	/**
	 * Constructor
	 * @param parent parent calculation
	 * @param source source surface
	 * @param target target surface
	 * @param refSource reference source surface
	 * @param refTarget reference target surface
	 */
	public RemapCoupledSurfaces(AbstractCalculation parent,
			EmbeddedSurface source, EmbeddedSurface target,
			EmbeddedSurface refSource, EmbeddedSurface refTarget) {
		super(parent);
		this.source = source;
		this.target = target;
		this.refSource = refSource;
		this.refTarget = refTarget;
		setLabel("Remap Surfaces");
	}
	/**
	 * Constructor
	 * @param source source surface
	 * @param target target surface
	 * @param refSource reference source surface
	 * @param refTarget reference target surface
	 */
	public RemapCoupledSurfaces(EmbeddedSurface source, EmbeddedSurface target,
			EmbeddedSurface refSource, EmbeddedSurface refTarget,boolean linearInterpolation) {
		this.source = source;
		this.target = target;
		this.refSource = refSource;
		this.refTarget = refTarget;
		this.linearInterpolation=linearInterpolation;
		setLabel("Remap Surfaces");
	}
	/**
	 * Remap surfaces
	 */
	public void remap() {
		int vertSourceCount = (source != null) ? source.getVertexCount() : 0;
		int vertTargetCount = (target != null) ? target.getVertexCount() : 0;
		setTotalUnits((vertSourceCount + vertTargetCount));
		Point3d sp, tp;
		int v1, v2, v3, fid;
		double[] stats;
		double[][] vertDataSource = null, vertDataTarget = null;
		// create vertex data lists
		if (refSource.getVertexData() != null) {
			vertDataSource = new double[vertTargetCount][0];
		}
		if (refTarget.getVertexData() != null) {
			vertDataTarget = new double[vertSourceCount][0];
		}
		//If source is specified
		if (source != null) {
			remappedTarget = source.clone();
			remappedTarget.setName(refSource.getName() + "_src");
			for (int i = 0; i < vertSourceCount; i++) {
				// Get the barycentric coordinates for the source
				stats = source.getVertexData(i);
				tp = new Point3d(stats[MATCHED_POINT],
						stats[MATCHED_POINT + 1], stats[MATCHED_POINT + 2]);
				fid = (int) stats[MATCHED_POINT + 3];
				v1 = refTarget.getCoordinateIndex(fid * 3);
				v2 = refTarget.getCoordinateIndex(fid * 3 + 1);
				v3 = refTarget.getCoordinateIndex(fid * 3 + 2);
				// Interpolate the vertex location using barycentric coordinates
				// embedded at each vertex location
				if (vertDataTarget != null)
					vertDataTarget[i] = GeometricUtilities
							.interpolateVectorFromBary(tp, refTarget
									.getVertexData(v1), refTarget
									.getVertexData(v2), refTarget
									.getVertexData(v3),linearInterpolation);
				remappedTarget.setVertex(i, GeometricUtilities
						.interpolatePointFromBary(tp, refTarget.getVertex(v1),
								refTarget.getVertex(v2), refTarget
										.getVertex(v3)));
				incrementCompletedUnits();
			}
			markCompleted();
			remappedTarget.setVertexData(vertDataTarget);
		}
		//If target is specified
		if (target != null) {
			remappedSource = target.clone();
			remappedSource.setName(refSource.getName() + "_dest");
			for (int i = 0; i < vertTargetCount; i++) {
				stats = target.getVertexData(i);
				sp = new Point3d(stats[MATCHED_POINT],
						stats[MATCHED_POINT + 1], stats[MATCHED_POINT + 2]);
				fid = (int) stats[MATCHED_POINT + 3];
				v1 = refSource.getCoordinateIndex(fid * 3);
				v2 = refSource.getCoordinateIndex(fid * 3 + 1);
				v3 = refSource.getCoordinateIndex(fid * 3 + 2);
				// Interpolate the vertex location using barycentric coordinates
				// embedded at each vertex location
				if (vertDataSource != null)
					vertDataSource[i] = GeometricUtilities
							.interpolateVectorFromBary(sp, refSource
									.getVertexData(v1), refSource
									.getVertexData(v2), refSource
									.getVertexData(v3),linearInterpolation);

				remappedSource.setVertex(i, GeometricUtilities
						.interpolatePointFromBary(sp, refSource.getVertex(v1),
								refSource.getVertex(v2), refSource
										.getVertex(v3)));
				incrementCompletedUnits();
			}
			remappedSource.setVertexData(vertDataSource);
		}
		markCompleted();
	}
	/**
	 * Get remapped source
	 * @return remapped source
	 */
	public EmbeddedSurface getRemappedSource() {
		return remappedSource;
	}
	/**
	 * Get remapped target
	 * @return remapped target
	 */
	public EmbeddedSurface getRemappedTarget() {
		return remappedTarget;
	}
}
