package edu.jhu.ece.iacl.algorithms.graphics.intersector;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.Vector;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import edu.jhmi.rad.medic.libraries.NeighborConnectivity;
import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree.KdTriangle;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface.Direction;

/**
 *A self-intersection detection algorithm using a hash table data table. There
 * are numerical precision limits on the ability of this algorithm to detect
 * self-intersections. Triangles that are nearly parallel or extremely close
 * together maybe labeled as a self-intersection.
 * 
 * @author Blake Lucas
 * 
 */
public class SurfaceSelfIntersector extends AbstractCalculation {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.3 $");
	}

	protected int hashResolution;
	protected EmbeddedSurface surf;
	protected int neighborVertexVertex[][];
	protected float hashPadding = 10.01f;
	protected static float EPS = 1E-6f;

	/**
	 * Get padding around 3D hash structure
	 * 
	 * @return padding in voxels
	 */
	public float getHashPadding() {
		return hashPadding;
	}

	/**
	 * Set hash padding
	 * 
	 * @param hashPadding
	 *            set hash padding
	 */
	public void setHashPadding(float hashPadding) {
		this.hashPadding = hashPadding;
	}

	protected LinkedList<SelfIntersectionTriangle> triangles;
	protected LinkedList<SelfIntersectionTriangle>[][][] triangleLUT;

	/**
	 * Constructor
	 * 
	 * @param surf
	 *            surface
	 * @param hashRes
	 *            3D hash resolution
	 */
	public SurfaceSelfIntersector(EmbeddedSurface surf, int hashRes) {
		this.hashResolution = hashRes;
		this.surf = surf;
		setLabel("Surface Self-Intersection Test");
	}

	/**
	 * Constructor
	 * 
	 * @param surf
	 *            surface
	 */
	public SurfaceSelfIntersector(EmbeddedSurface surf) {
		this(surf, 256);
	}

	/**
	 * Get 3D hash
	 * 
	 * @return hash structure
	 */
	public LinkedList<SelfIntersectionTriangle>[][][] getTriangleHash() {
		return triangleLUT;
	}

	/**
	 * Get volume indicating number of triangles in each hash bounding box.
	 * 
	 * @return hash triangle count.
	 */
	public int[][][] getTriangleCountVolume() {
		int[][][] vol = new int[hashResolution][hashResolution][hashResolution];
		for (int i = 0; i < hashResolution; i++) {
			for (int j = 0; j < hashResolution; j++) {
				for (int k = 0; k < hashResolution; k++) {
					vol[i][j][k] = (triangleLUT[i][j][k] == null) ? 0
							: triangleLUT[i][j][k].size();
				}
			}
		}
		return vol;
	}

	/**
	 * Get surface
	 * 
	 * @return surface
	 */
	public EmbeddedSurface getLabeledSurf() {
		return surf;
	}

	double minX, minY, minZ;
	double maxX, maxY, maxZ;
	double edgeLengthX;
	double edgeLengthY;
	double edgeLengthZ;

	/**
	 * Initialize hash table
	 */
	public void init() {
		System.out.println("INITIALIZE SELF INTERSECTOR");
		int vertCount = surf.getVertexCount();
		int indexCount = surf.getIndexCount();
		minX = 1E10f;
		minY = 1E10f;
		minZ = 1E10f;
		maxX = -1E10f;
		maxY = -1E10f;
		maxZ = -1E10f;
		for (int i = 0; i < vertCount; i++) {
			Point3f p = surf.getVertex(i);
			minX = Math.min(minX, p.x);
			maxX = Math.max(maxX, p.x);
			minY = Math.min(minY, p.y);
			maxY = Math.max(maxY, p.y);
			minZ = Math.min(minZ, p.z);
			maxZ = Math.max(maxZ, p.z);
		}
		int id1, id2, id3;
		SelfIntersectionTriangle t;
		triangles = new LinkedList<SelfIntersectionTriangle>();
		triangleLUT = new LinkedList[hashResolution][hashResolution][hashResolution];
		neighborVertexVertex = EmbeddedSurface.buildNeighborVertexVertexTable(
				surf, Direction.COUNTER_CLOCKWISE);
		for (int i = 0; i < indexCount; i += 3) {
			id1 = surf.getCoordinateIndex(i);
			id2 = surf.getCoordinateIndex(i + 1);
			id3 = surf.getCoordinateIndex(i + 2);
			t = new SelfIntersectionTriangle(id1, id2, id3, i / 3, surf);
			triangles.add(t);
		}
		minX -= hashPadding;
		minY -= hashPadding;
		minZ -= hashPadding;
		maxX += hashPadding;
		maxY += hashPadding;
		maxZ += hashPadding;
		edgeLengthX = (maxX - minX) / hashResolution;
		edgeLengthY = (maxY - minY) / hashResolution;
		edgeLengthZ = (maxZ - minZ) / hashResolution;

		for (SelfIntersectionTriangle tri : triangles) {
			add(tri);
		}
	}

	/**
	 * Swap edges if edge swap will not create self-intersection.
	 * 
	 * @param v1
	 *            edge1->v1
	 * @param v2
	 *            edge1->v2
	 * @param v3
	 *            edge2->v1
	 * @param v4
	 *            edge2->v2
	 * @return true if edge swap was successful
	 */
	public boolean swapEdge(int v1, int v2, int v3, int v4) {
		SelfIntersectionTriangle t1 = findFace(v3, v1, v4);
		SelfIntersectionTriangle t2 = findFace(v4, v2, v3);
		if (t1 == null || t2 == null)
			return false;
		removeFace(t1);
		removeFace(t2);
		for (int i = 0; i < 3; i++) {
			if (t1.ids[i] == v3) {
				t1.ids[i] = v2;
				t1.pts[i].set(surf.getVertex(v2));
			}
			if (t2.ids[i] == v4) {
				t2.ids[i] = v1;
				t2.pts[i].set(surf.getVertex(v1));
			}
		}
		t1.update();
		t2.update();
		// Could not swap
		if (intersects(t1) || intersects(t2)) {
			// swap back
			for (int i = 0; i < 3; i++) {
				if (t1.ids[i] == v2) {
					t1.ids[i] = v3;
					t1.pts[i].set(surf.getVertex(v3));
				}
				if (t2.ids[i] == v1) {
					t2.ids[i] = v4;
					t2.pts[i].set(surf.getVertex(v4));
				}
			}
			t1.update();
			t2.update();

			add(t1);
			add(t2);
			return false;
		} else {
			add(t1);
			add(t2);
			return true;
		}
	}

	/**
	 * For triangle identified by vertices {v1,v2,v3}, find self-intersection
	 * triangle in hash with those vertices.
	 * 
	 * @param vid1
	 *            v1
	 * @param vid2
	 *            v2
	 * @param vid3
	 *            v3
	 * @return self-intersection triangle
	 */
	public SelfIntersectionTriangle findFace(int vid1, int vid2, int vid3) {
		Point3f p = surf.getVertex(vid1);
		int stX = (int) Math.max(0, Math
				.floor((p.x - minX - EPS) / edgeLengthX));
		int stY = (int) Math.max(0, Math
				.floor((p.y - minY - EPS) / edgeLengthY));
		int stZ = (int) Math.max(0, Math
				.floor((p.z - minZ - EPS) / edgeLengthZ));

		int edX = (int) Math.min(hashResolution - 1, Math
				.ceil((p.x - minX + EPS) / edgeLengthX));
		int edY = (int) Math.min(hashResolution - 1, Math
				.ceil((p.y - minY + EPS) / edgeLengthY));
		int edZ = (int) Math.min(hashResolution - 1, Math
				.ceil((p.z - minZ + EPS) / edgeLengthZ));
		LinkedList<SelfIntersectionTriangle> hash = null;
		for (int i = stX; i < edX; i++) {
			for (int j = stY; j < edY; j++) {
				for (int k = stZ; k < edZ; k++) {
					hash = triangleLUT[i][j][k];
					if (hash == null)
						continue;
					for (SelfIntersectionTriangle t : hash) {
						if (t.ids[0] == vid1 && t.ids[1] == vid2
								&& t.ids[2] == vid3) {
							return t;
						} else if (t.ids[0] == vid2 && t.ids[1] == vid3
								&& t.ids[2] == vid1) {
							return t;
						} else if (t.ids[0] == vid3 && t.ids[1] == vid1
								&& t.ids[2] == vid2) {
							return t;
						}
					}
				}
			}
		}
		System.out.println("BRUTE FORCE SEARCH " + vid1 + "," + vid2 + ","
				+ vid3);
		for (int i = 0; i < hashResolution; i++) {
			for (int j = 0; j < hashResolution; j++) {
				for (int k = 0; k < hashResolution; k++) {
					hash = triangleLUT[i][j][k];
					if (hash == null)
						continue;
					for (SelfIntersectionTriangle t : hash) {
						if (t.ids[0] == vid1 && t.ids[1] == vid2
								&& t.ids[2] == vid3) {
							return t;
						} else if (t.ids[0] == vid2 && t.ids[1] == vid3
								&& t.ids[2] == vid1) {
							return t;
						} else if (t.ids[0] == vid3 && t.ids[1] == vid1
								&& t.ids[2] == vid2) {
							return t;
						}
					}
				}
			}
		}
		System.out.println("COULD NOT FIND " + vid1 + "," + vid2 + "," + vid3);
		return null;
	}

	/**
	 * Find all faces that contain the specified vertex
	 * 
	 * @param vid
	 *            vertex id
	 * @return list of faces
	 */
	public LinkedList<SelfIntersectionTriangle> findFaces(int vid) {
		Point3f p = surf.getVertex(vid);
		int stX = (int) Math.max(0, Math.floor((p.x - minX) / edgeLengthX));
		int stY = (int) Math.max(0, Math.floor((p.y - minY) / edgeLengthY));
		int stZ = (int) Math.max(0, Math.floor((p.z - minZ) / edgeLengthZ));

		int edX = (int) Math.min(hashResolution - 1, Math.ceil((p.x - minX)
				/ edgeLengthX));
		int edY = (int) Math.min(hashResolution - 1, Math.ceil((p.y - minY)
				/ edgeLengthY));
		int edZ = (int) Math.min(hashResolution - 1, Math.ceil((p.z - minZ)
				/ edgeLengthZ));
		LinkedList<SelfIntersectionTriangle> faceList = new LinkedList<SelfIntersectionTriangle>();
		for (int i = stX; i < edX; i++) {
			for (int j = stY; j < edY; j++) {
				for (int k = stZ; k < edZ; k++) {
					LinkedList<SelfIntersectionTriangle> hash = triangleLUT[i][j][k];
					if (hash == null)
						continue;
					for (SelfIntersectionTriangle t : hash) {
						if (t.ids[0] == vid || t.ids[1] == vid
								|| t.ids[2] == vid) {
							if (!faceList.contains(t))
								faceList.add(t);
						}
					}
				}
			}
		}
		return faceList;
	}

	/**
	 * Update position of specified vertex as long as moving the vertex will not
	 * cause self-intersection of the surface.
	 * 
	 * @param vid
	 *            vertex id
	 * @param p
	 *            updated vertex location
	 * @return true if vertex position could be changed without causing
	 *         self-intersection of the surface.
	 */
	public boolean updateVertex(int vid, Point3f p) {
		Point3f np = surf.getVertex(vid);

		int stX = (int) Math.max(0, Math.floor((np.x - minX - EPS)
				/ edgeLengthX));
		int stY = (int) Math.max(0, Math.floor((np.y - minY - EPS)
				/ edgeLengthY));
		int stZ = (int) Math.max(0, Math.floor((np.z - minZ - EPS)
				/ edgeLengthZ));

		int edX = (int) Math.min(hashResolution - 1, Math
				.ceil((np.x - minX + EPS) / edgeLengthX));
		int edY = (int) Math.min(hashResolution - 1, Math
				.ceil((np.y - minY + EPS) / edgeLengthY));
		int edZ = (int) Math.min(hashResolution - 1, Math
				.ceil((np.z - minZ + EPS) / edgeLengthZ));
		Vector<SelfIntersectionTriangle> updateTriangleList = new Vector<SelfIntersectionTriangle>();
		for (int i = stX; i < edX; i++) {
			for (int j = stY; j < edY; j++) {
				for (int k = stZ; k < edZ; k++) {
					LinkedList<SelfIntersectionTriangle> hash = triangleLUT[i][j][k];

					if (hash == null)
						continue;
					for (SelfIntersectionTriangle t : hash) {
						if (t.ids[0] == vid) {
							t.pts[0].set(p);
							if (!updateTriangleList.contains(t))
								updateTriangleList.add(t);
						} else if (t.ids[1] == vid) {
							t.pts[1].set(p);
							if (!updateTriangleList.contains(t))
								updateTriangleList.add(t);
						} else if (t.ids[2] == vid) {
							t.pts[2].set(p);
							if (!updateTriangleList.contains(t))
								updateTriangleList.add(t);
						}
					}
				}
			}
		}
		for (SelfIntersectionTriangle t : updateTriangleList) {
			removeFace(t);
			t.update();
		}
		boolean hit = false;
		for (SelfIntersectionTriangle t : updateTriangleList) {
			hit = intersects(t);
			if (hit)
				break;
		}
		if (!hit) {
			for (SelfIntersectionTriangle t : updateTriangleList) {
				add(t);
			}
			surf.setVertex(vid, p);
			return true;
		} else {
			for (SelfIntersectionTriangle t : updateTriangleList) {
				if (t.ids[0] == vid) {
					t.pts[0].set(np);
				} else if (t.ids[1] == vid) {
					t.pts[1].set(np);
				} else if (t.ids[2] == vid) {
					t.pts[2].set(np);
				}
				t.update();
			}
			for (SelfIntersectionTriangle t : updateTriangleList) {
				add(t);
			}
			return false;
		}

	}

	/**
	 * Remove vertex from hash and all its connected faces
	 * 
	 * @param vid
	 *            vertex id
	 */
	public void removeVertex(int vid) {
		Point3f p = surf.getVertex(vid);
		int stX = (int) Math.max(0, Math.floor((p.x - minX) / edgeLengthX));
		int stY = (int) Math.max(0, Math.floor((p.y - minY) / edgeLengthY));
		int stZ = (int) Math.max(0, Math.floor((p.z - minZ) / edgeLengthZ));

		int edX = (int) Math.min(hashResolution - 1, Math.ceil((p.x - minX)
				/ edgeLengthX));
		int edY = (int) Math.min(hashResolution - 1, Math.ceil((p.y - minY)
				/ edgeLengthY));
		int edZ = (int) Math.min(hashResolution - 1, Math.ceil((p.z - minZ)
				/ edgeLengthZ));
		for (int i = stX; i < edX; i++) {
			for (int j = stY; j < edY; j++) {
				for (int k = stZ; k < edZ; k++) {
					LinkedList<SelfIntersectionTriangle> hash = triangleLUT[i][j][k];
					if (hash == null)
						continue;
					for (int n = 0; n < hash.size(); n++) {
						SelfIntersectionTriangle t = hash.get(n);
						if (t.ids[0] == vid || t.ids[1] == vid
								|| t.ids[2] == vid) {
							hash.remove(t);
							n--;
						}
					}
				}
			}
		}
	}

	/**
	 * Remove triangle face from hash.
	 * 
	 * @param tri
	 *            triangle
	 */
	public void removeFace(SelfIntersectionTriangle tri) {
		if (tri == null)
			return;
		Point3f minP = tri.minPoint;
		Point3f maxP = tri.maxPoint;
		int stX = (int) Math.max(0, Math.floor((minP.x - minX - EPS)
				/ edgeLengthX));
		int stY = (int) Math.max(0, Math.floor((minP.y - minY - EPS)
				/ edgeLengthY));
		int stZ = (int) Math.max(0, Math.floor((minP.z - minZ - EPS)
				/ edgeLengthZ));

		int edX = (int) Math.min(hashResolution - 1, Math
				.ceil((maxP.x - minX + EPS) / edgeLengthX));
		int edY = (int) Math.min(hashResolution - 1, Math
				.ceil((maxP.y - minY + EPS) / edgeLengthY));
		int edZ = (int) Math.min(hashResolution - 1, Math
				.ceil((maxP.z - minZ + EPS) / edgeLengthZ));
		for (int i = stX; i < edX; i++) {
			for (int j = stY; j < edY; j++) {
				for (int k = stZ; k < edZ; k++) {
					if (triangleLUT[i][j][k] == null)
						continue;
					triangleLUT[i][j][k].remove(tri);
					if (triangleLUT[i][j][k].size() == 0)
						triangleLUT[i][j][k] = null;
				}
			}
		}
	}

	/**
	 * Add self-intersection triangle to hash
	 * 
	 * @param tri
	 */
	public void add(SelfIntersectionTriangle tri) {
		Point3f minP = tri.minPoint;
		Point3f maxP = tri.maxPoint;
		int stX = (int) Math.max(0, Math.floor((minP.x - minX - EPS)
				/ edgeLengthX));
		int stY = (int) Math.max(0, Math.floor((minP.y - minY - EPS)
				/ edgeLengthY));
		int stZ = (int) Math.max(0, Math.floor((minP.z - minZ - EPS)
				/ edgeLengthZ));

		int edX = (int) Math.min(hashResolution - 1, Math
				.ceil((maxP.x - minX + EPS) / edgeLengthX));
		int edY = (int) Math.min(hashResolution - 1, Math
				.ceil((maxP.y - minY + EPS) / edgeLengthY));
		int edZ = (int) Math.min(hashResolution - 1, Math
				.ceil((maxP.z - minZ + EPS) / edgeLengthZ));
		boolean added = false;
		for (int i = stX; i < edX; i++) {
			for (int j = stY; j < edY; j++) {
				for (int k = stZ; k < edZ; k++) {
					if (triangleLUT[i][j][k] == null) {
						triangleLUT[i][j][k] = new LinkedList<SelfIntersectionTriangle>();
					}
					triangleLUT[i][j][k].add(tri);
					added = true;
				}
			}
		}
		if (!added) {
			System.out.println("COULD NOT ADD " + tri.ids[0] + "," + tri.ids[1]
					+ "," + tri.ids[2] + ": " + tri.minPoint + " "
					+ tri.maxPoint);
		}
	}

	/**
	 * Determine if triangle intersects any other triangles in the mesh
	 * 
	 * @param tri
	 *            triangle
	 * @return true if there is a self-intersection
	 */
	public boolean intersects(SelfIntersectionTriangle tri) {
		Point3f minP = tri.minPoint;
		Point3f maxP = tri.maxPoint;
		int stX = (int) Math.max(0, Math.floor((minP.x - minX - EPS)
				/ edgeLengthX));
		int stY = (int) Math.max(0, Math.floor((minP.y - minY - EPS)
				/ edgeLengthY));
		int stZ = (int) Math.max(0, Math.floor((minP.z - minZ - EPS)
				/ edgeLengthZ));

		int edX = (int) Math.min(hashResolution - 1, Math
				.ceil((maxP.x - minX + EPS) / edgeLengthX));
		int edY = (int) Math.min(hashResolution - 1, Math
				.ceil((maxP.y - minY + EPS) / edgeLengthY));
		int edZ = (int) Math.min(hashResolution - 1, Math
				.ceil((maxP.z - minZ + EPS) / edgeLengthZ));
		for (int i = stX; i < edX; i++) {
			for (int j = stY; j < edY; j++) {
				for (int k = stZ; k < edZ; k++) {
					if (triangleLUT[i][j][k] == null)
						continue;
					for (SelfIntersectionTriangle t : triangleLUT[i][j][k]) {
						if (t != tri && tri.intersects(t))
							return true;
					}
				}
			}
		}
		return false;
	}

	/**
	 * Check if surface contains self-intersections
	 * 
	 * @return true if surface contains self-intersections
	 */
	public boolean containsSelfIntersections() {
		int m, n, sz;
		SelfIntersectionTriangle t2;
		for (int i = 0; i < hashResolution; i++) {
			for (int j = 0; j < hashResolution; j++) {
				for (int k = 0; k < hashResolution; k++) {
					if (triangleLUT[i][j][k] == null)
						continue;
					LinkedList<SelfIntersectionTriangle> hash = triangleLUT[i][j][k];
					m = 1;
					sz = hash.size();
					for (SelfIntersectionTriangle t1 : hash) {
						for (n = m; n < sz; n++) {
							t2 = hash.get(n);
							if (t1.intersects(t2)) {
								return true;
							}
						}
						m++;
					}
				}
			}
		}
		return false;
	}

	/**
	 * Apply rotational vector field to mesh to test self-interesction
	 * prevention
	 */
	public void applyTestDeformation() {
		int maxIters = 50;
		double l, d;
		Vector3f v = new Vector3f();
		int vertCount = surf.getVertexCount();
		Point3f center = surf.getCenterOfMass();
		double maxRadius = 0;
		for (int i = 0; i < vertCount; i++) {
			maxRadius = Math.max(GeometricUtilities.distance(center, surf
					.getVertex(i)), maxRadius);
		}
		int rejectedCount = 0;
		for (int iter = 0; iter < maxIters; iter++) {
			System.out.println("Iteration " + (iter + 1));
			rejectedCount = 0;
			for (int i = 0; i < vertCount; i++) {
				Point3f p = surf.getVertex(i);
				d = Math.min(1, GeometricUtilities.distance(center, p)
						/ maxRadius);
				l = Math.sqrt((p.x - center.x) * (p.x - center.x)
						+ (p.y - center.y) * (p.y - center.y));
				v.x = (float) ((p.y - center.y) / l);
				v.y = (float) (-(p.x - center.x) / l);
				v.z = 0;
				v.scale((float) (0.5 * d));
				p.add(v);

				if (!updateVertex(i, p)) {
					rejectedCount++;
				}
			}
			System.out.println("Rejected " + rejectedCount);
		}
	}

	/**
	 * Get calculated normal
	 * 
	 * @param id
	 *            vertex id
	 * @param neighborVertexVertexTable
	 *            vertex-vertex lookup table
	 * @return normal vector
	 */
	protected Vector3f getCalculatedNormal(int id,
			int[][] neighborVertexVertexTable) {
		Vector3f norm = new Vector3f();
		Vector3f edge1 = new Vector3f();
		Vector3f edge2 = new Vector3f();
		Vector3f cross = new Vector3f();
		Point3f pivot = surf.getVertex(id);
		int len = neighborVertexVertexTable[id].length;
		for (int i = 0; i < len; i++) {
			edge1.sub(surf.getVertex(neighborVertexVertexTable[id][i]), pivot);
			edge2.sub(surf.getVertex(neighborVertexVertexTable[id][(i + 1)
					% len]), pivot);
			cross.cross(edge1, edge2);
			norm.add(cross);
		}
		GeometricUtilities.normalize(norm);
		return norm;
	}

	/**
	 * Apply force to drive surface to a sphere. This usually creates
	 * self-intersections for any surface that is not star-shaped.
	 */
	public void applySphericalDeformation() {

		int maxIters = 500;
		double l, d;
		Vector3f v = new Vector3f();
		int vertCount = surf.getVertexCount();
		Point3f center = surf.getCenterOfMass();
		double radius = Math.sqrt(surf.area() / 4 * Math.PI);
		int rejectedCount = 0;
		float speed = 0;
		double fuzz = 0.2;
		setLabel("Deform to Sphere");
		setTotalUnits(maxIters);
		System.out.println("RADIUS " + radius);
		double error = 0;
		int[][] neighborVertexVertexTable = EmbeddedSurface
				.buildNeighborVertexVertexTable(surf,
						Direction.COUNTER_CLOCKWISE);
		for (int iter = 0; iter < maxIters; iter++) {
			System.out.println("Iteration " + (iter + 1));
			rejectedCount = 0;
			error = 0;
			for (int i = 0; i < vertCount; i++) {
				Point3f p = surf.getVertex(i);
				double off = p.distance(center) / radius;
				speed = (float) (2 * 1.0 / (1 + Math.exp(fuzz * (1 - off)))) - 1;
				v = getCalculatedNormal(i, neighborVertexVertexTable);
				v.scale(0.5f * speed);
				p.add(v);
				error += Math.abs(1 - off);
				surf.setVertex(i, p);
				/*
				 * if(!updateVertex(i,p)){ rejectedCount++; }
				 */
			}
			error /= vertCount;
			incrementCompletedUnits();
			System.out.println("Rejected " + rejectedCount + " Error " + error);
		}
		markCompleted();
	}

	/**
	 * Label the number of self-intersections in the surface.
	 * 
	 * @return number of self-intersections
	 */
	public int labelIntersections() {
		double[][] intersectionFaceCount = new double[triangles.size()][1];
		double[][] intersectionVertexCount = new double[surf.getVertexCount()][1];
		int m, n, sz;
		boolean hit = false;
		SelfIntersectionTriangle t2;
		int count = 0;
		LinkedList<SelfIntersectionTriangle> triangles = new LinkedList<SelfIntersectionTriangle>();
		for (int i = 0; i < hashResolution; i++) {
			for (int j = 0; j < hashResolution; j++) {
				for (int k = 0; k < hashResolution; k++) {
					if (triangleLUT[i][j][k] == null)
						continue;
					LinkedList<SelfIntersectionTriangle> hash = triangleLUT[i][j][k];
					m = 1;
					sz = hash.size();
					hit = false;
					for (SelfIntersectionTriangle t1 : hash) {
						for (n = m; n < sz; n++) {
							t2 = hash.get(n);
							if (t1.intersects(t2)) {
								intersectionFaceCount[t1.fid][0]++;
								System.out.println("T1 " + t1.fid + " "
										+ t1.ids[0] + "," + t1.ids[1] + ","
										+ t1.ids[2] + " " + t1.pts[0] + " "
										+ t1.pts[1] + " " + t1.pts[2]);
								System.out.println("T2 " + t2.fid + " "
										+ t2.ids[0] + "," + t2.ids[1] + ","
										+ t2.ids[2] + " " + t2.pts[0] + " "
										+ t2.pts[1] + " " + t2.pts[2]);
								triangles.add(t1);
								triangles.add(t2);
								hit = true;
								if (t1.fid < t2.fid)
									count++;
								break;
							}
						}
						if (hit)
							break;
						m++;
					}
				}
			}
		}
		System.out.println("Intersection Count " + count);
		for (SelfIntersectionTriangle t : triangles) {
			intersectionVertexCount[t.ids[0]][0] = intersectionFaceCount[t.fid][0];
			intersectionVertexCount[t.ids[1]][0] = intersectionFaceCount[t.fid][0];
			intersectionVertexCount[t.ids[2]][0] = intersectionFaceCount[t.fid][0];
		}
		surf.setVertexData(intersectionVertexCount);
		surf.setCellData(intersectionFaceCount);
		return count;
	}

	/**
	 * Sanity check for self-intersections
	 * 
	 * @param surf
	 * @return
	 */
	public boolean sanityCheck(EmbeddedSurface surf) {
		this.surf = surf;
		init();
		return (labelIntersections() == 0);
	}

}
