package edu.jhmi.rad.medic.algorithms;

import edu.jhmi.rad.medic.dialogs.*;
import edu.jhmi.rad.medic.methods.*;
import edu.jhmi.rad.medic.utilities.*;

import gov.nih.mipav.model.algorithms.AlgorithmBase;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.view.*;

import java.io.*;
import java.util.*;
import java.lang.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
 
/**
 *
 *  FANTASM for longitudinal series of 3D images.
 *  <p>
 *	This version assumes a correlation between images by extending the MRF
 *	field of FANTASM to the 4th dimension. The images may be registered in the process.
 *
 *	
 *	@version    March 2004
 *	@author     Pilou Bazin
 *  @see        AlgorithmFantasm
 *  @see 		JDialogFantasms
 *	@see		LongitudinalFCM
 *	@see		RegisterSegmentation
 *
*/
public class AlgorithmFantasmLongitudinal extends AlgorithmBase {

    // Fuzzy images require 1 image for each class
    // Hard images 1 image with assigned clusters
    private ModelImage          destImage[];
    private ModelImage          srcImage[];
    private int                 destNum=0;
    private ViewUserInterface   userInterface;
    private float[]             Imin;
    private float[]             Imax;
    private float[]             scaling;
    private float             	rx,ry,rz;
    
     // information on the parts of the image to process
    private     float[]     backThreshold;
    private     boolean     cropBackground;
	private		boolean		useRelative;
    private     int         nxmin,nymin,nzmin,nxmax,nymax,nzmax;
    private     int         x0,xN,y0,yN,z0,zN;
    private     int         X0,XN,Y0,YN,Z0,ZN;
    private     int         mx,my,mz;
	private		int			borderSize;
	private		String		maskingMode;

    // image size
	private int         clusters, classes; // classes include the outliers, clusters does not
	private int 		nx,ny,nz,nt;
	private int			dim;

    // algorithm parameters
	private		String		initMode;
	private		String		outputType;
    
	private 	float	 	smoothing;
	private 	float	 	temporal;
	private     int 		iterations;
	private     float   	maxDistance;
	private 	float	 	fuzziness;
	
	private		boolean		addOutliers;
	private		float		outlier;        
	
	private		boolean		correctInhomogeneity;
	private		int			fieldDegree;
	
	private		boolean		useEdges;
	private		float		edgeSmoothness, edgeContrast, edgePrior;
	
	private		boolean		useRegistration;
	private		int			levels;
	private		int			firstIter;
	private		int			topIter;
	private		int			lastIter;
	
	private     boolean		verbose = true;
	private     boolean		debug = false;
	
    /**
    *	Constructor for 3D images in which changes are placed in a predetermined destination image.
    *   @param destImg_      Image model where result image is to stored.
    *   @param srcImg_       Source image model.
    */
	public AlgorithmFantasmLongitudinal(ModelImage[] destImg_, ModelImage[] srcImg_, int Nimg_, int destNum_,
							String init_, String output_,
							int nClasses_, int nIterMax_, float distMax_, float smooth_, float temp_, 
							float fuzzy_,
                            boolean addOut_, float outVal_, 
							boolean correct_, int poly_,
							boolean edges_, float eSmooth_, float eContr_, float ePr_,
                            boolean reg_, int lev_, int first_, int top_, int last_,
                            boolean cropBack_, float cropVal_, String masking_, boolean useRel_) {
								
        super(null, srcImg_[0]);
        srcImage = srcImg_;
        destImage = destImg_;
        userInterface = ViewUserInterface.getReference();
		destNum = destNum_;
		nt = Nimg_;
		
		initMode = init_;
        outputType = output_;
        
        cropBackground = cropBack_;
		backThreshold = new float[nt];
		for (int t=0;t<nt;t++) backThreshold[t] = cropVal_;
		maskingMode = masking_;
        useRelative = useRel_;
         
		addOutliers = addOut_;
		outlier = outVal_;
		        
		clusters = nClasses_;
		classes = nClasses_;
        if (addOutliers) classes++;
		
		iterations = nIterMax_;
        maxDistance = distMax_;
		smoothing = smooth_;
		temporal = temp_;
		fuzziness = fuzzy_;
		
        correctInhomogeneity = correct_;
        fieldDegree = poly_;
		
		useEdges = edges_;
		edgeSmoothness = eSmooth_;
		edgeContrast = eContr_;
		edgePrior = ePr_;
        
		useRegistration = reg_;
		levels = lev_;
		firstIter = first_;
		topIter = top_;
		lastIter = last_;
        
		if (debug) {
			MedicUtilPublic.displayMessage("image list:\n");
			for (int t=0;t<nt;t++)
				MedicUtilPublic.displayMessage(""+t+". "+srcImage[t].getImageName()+"\n");
		}
	}


    /**
    *	Prepares this class for destruction.
    */
	public void finalize(){
	    destImage   = null;
	    srcImage    = null;
        System.gc();
        super.finalize();
	}

    /**
    *	Constructs a string of the contruction parameters and outputs the string to the messsage frame if the logging
    *   procedure is turned on.
    *	@param destinationFlag	If true the log includes the name of the destination flag.
    */
	/*
    private void constructLog(boolean destinationFlag) {
        if ( destinationFlag == false) {
            historyString = new String( "Fantasm(" + ")");
        }
        else  {
            historyString = new String( "Fantasm(" + ")");
        }
        historyString += "\n";  // append a newline onto the string
        writeLog();
    }
	*/

     /**
     * Writes the logString to the appropriate log area. Overrides the AlgorithmBase <code>writeLog()</code> to append
     * the history to all of the destination images we've created.
     */
	 /*
    protected void writeLog() {
        // write to the history area
        if (Preferences.is(Preferences.PREF_LOGGING_ENABLED) && isCompleted()) {
			if (destImage != null) {
				for (int i = 0; i < destImage.length; i++) {
					if ((destImage[i] != null) && (destImage[i].getHistoryArea() != null)) {
						if (srcImage != null) {
                            destImage[i].getHistoryArea().setText(srcImage[0].getHistoryArea().getText());
                        }
						if (historyString != null) {
                            destImage[i].getHistoryArea().append(historyString);
                        }
                    }
                }
            } else if (srcImage != null) {
				if (historyString != null) {
					for (int i = 0; i < srcImage.length; i++) {
						if ((srcImage[i] != null) && (srcImage[i].getHistoryArea() != null)) {
							srcImage[i].getHistoryArea().append(historyString);
						}
					}
                }
            }
        }
    }
	*/

    /**
    *   Starts the algorithm.
    */
	public void runAlgorithm() {

        if (srcImage  == null) {
            displayError("Source Image is null");
            //notifyListeners(this);
            return;
        }
        if (destImage  == null) {
            displayError("Destination Image is null");
            //notifyListeners(this);
            return;
        }

        // start the timer to compute the elapsed time
        setStartTime();

        if (destImage != null){     // if there exists a destination image
            //constructLog(true);
			
			float[][] buffer;
            if (srcImage[0].getNDims() == 2) {
			   try {
					// image length is length in 2 dims
					int length = srcImage[0].getExtents()[0] 
                                * srcImage[0].getExtents()[1];
                    
                    // retrieve all data: for each volume
                    buffer = new float[nt][];
                    for (int t=0;t<nt;t++) {
                        buffer[t] = new float[length];
                        srcImage[t].exportData(0,length, buffer[t]); // locks and releases lock
                    }
                    fireProgressStateChanged(srcImage[0].getImageName(), "Processing image ...");
				}
				catch (IOException error) {
					buffer = null;
					errorCleanUp("Algorithm: source image locked", true);
					return;
				}
				catch (OutOfMemoryError e){
					buffer = null;
					errorCleanUp("Algorithm: Out of memory creating process buffer", true);
					return;
				}
		
				// init dimensions
				nx = srcImage[0].getExtents()[0];
				ny = srcImage[0].getExtents()[1];
				nz = 1;
				dim = 2;
				
                // main algorithm
				calcSegmentation(buffer);
				
			} else if (srcImage[0].getNDims() == 3) {
			   try {
					// image length is length in 3 dims
					int length = srcImage[0].getExtents()[0] 
                                * srcImage[0].getExtents()[1] * srcImage[0].getExtents()[2];
                    
                    // retrieve all data: for each volume
                    buffer = new float[nt][];
                    for (int t=0;t<nt;t++) {
                        buffer[t] = new float[length];
                        srcImage[t].exportData(0,length, buffer[t]); // locks and releases lock
                    }
                    fireProgressStateChanged(srcImage[0].getImageName(), "Processing image ...");
				}
				catch (IOException error) {
					buffer = null;
					errorCleanUp("Algorithm: source image locked", true);
					return;
				}
				catch (OutOfMemoryError e){
					buffer = null;
					errorCleanUp("Algorithm: Out of memory creating process buffer", true);
					return;
				}
		
				// init dimensions
				nx = srcImage[0].getExtents()[0];
				ny = srcImage[0].getExtents()[1];
				nz = srcImage[0].getExtents()[2];
				dim = 3;
				
                // main algorithm
				calcSegmentation(buffer);
				
			}
        }

        // compute the elapsed time
        computeElapsedTime();

        //if (!threadStopped) {  closingLog();  }
        //notifyListeners(this);

    } // end run()

    /**
    *	produces a RFCM fuzzy segmentation of the input images
    */
    private void calcSegmentation(float img[][]){
		boolean[][][][]         objectMask;
		boolean[][][]         	commonMask;
		float[][][][]           image;
		int                     x,y,z,t,k,m,l;
		float                   dist;
		int                     indx;
        float[]                 buffer;
		byte[]                 	bytebuffer;
		LongitudinalFCM			segmentation;
		MultichannelFCM			initialization;
		InhomogeneityCorrection[]	correction;
		EdgeAdaptation[]			edge;
		ClusterSearch			search;
		RegisterSegmentation	register;	
		TransMatrix[]			transmat;
		int[]					id;

		int mod;
		int progress;

		boolean stop,stopReg;
		int n;
		float distance;
		
		float maxDist0 = (float)Math.sqrt(maxDistance);
		int iter0 = (int)Math.sqrt(iterations);

		 
		// increase the dimension to make boundaries
		nx = nx+2; ny = ny+2; nz = nz+2;
		// pre-processing : expand boundaries, so that we don't have to worry for them
        try {
            image = new float[nt][][][];
            for (t=0;t<nt;t++)
                image[t] = expandImage(img[t]);
            // dispose of img
            img = null;
        } catch (OutOfMemoryError e) {
            img = null; image = null;
            errorCleanUp("Algorithm: Out of memory creating process buffer", true);
            setCompleted(false);
            return;
        }

		// compute intensity boundaries and scale parameters
        Imin = new float[nt];
        Imax = new float[nt];
		scaling = new float[nt];
        
        for (t=0;t<nt;t++) {
            srcImage[t].calcMinMax();
		
			Imin[t] = (float)srcImage[t].getMin();
			Imax[t] = (float)srcImage[t].getMax();
			scaling[t] = Imax[t]-Imin[t];
			backThreshold[t] = Imin[t] + backThreshold[t]*scaling[t];
		}
        for (t=0;t<nt;t++) {
            // squared factors
            //smoothing = smoothing*smoothing;
            //temporal = temporal*temporal;
            //edgeSmoothness = edgeSmoothness*edgeSmoothness;
            //edgeContrast = edgeContrast*edgeContrast;
            //edgePrior = edgePrior*edgePrior;
        }
		// resolution: same for all images
		rx = srcImage[0].getFileInfo()[0].getResolutions()[0];
		ry = srcImage[0].getFileInfo()[0].getResolutions()[1];
		rz = srcImage[0].getFileInfo()[0].getResolutions()[2];
        if (debug) MedicUtilPublic.displayMessage("resolutions: "+rx+"x"+ry+"x"+rz+"\n");
		
		//if use mask, process only parts where intensity > min (mask stripped areas)
        objectMask = new boolean[nt][][][];
        X0 = nx; XN = 0;
		Y0 = ny; YN = 0;
		Z0 = nz; ZN = 0;
		for (t=0;t<nt;t++) {
            objectMask[t] = createObjectMask(image[t], backThreshold[t] );
        
            computeProcessingBoundaries(objectMask[t]);
            // global boundaries
            if (x0 < X0) X0 = x0;
            if (xN > XN) XN = xN;
            if (y0 < Y0) Y0 = y0;
            if (yN > YN) YN = yN;
            if (z0 < Z0) Z0 = z0;
            if (zN > ZN) ZN = zN;
        }
		// use small-sized boundaries
        x0 = Math.max(1,X0); xN = Math.min(nx-2,XN);
        y0 = Math.max(1,Y0); yN = Math.min(ny-2,YN);
        z0 = Math.max(1,Z0); zN = Math.min(nz-2,ZN);
        mx = xN-x0+1+2;
        my = yN-y0+1+2;
        mz = zN-z0+1+2;
        // rescale images
		if (debug) MedicUtilPublic.displayMessage("shrink images..\n");
		for (t=0;t<nt;t++) {
            image[t] = reduceImageSize(image[t]); 
            objectMask[t] = reduceImageSize(objectMask[t]); 
        }
        if (debug) MedicUtilPublic.displayMessage("new dimensions: "+mx+"x"+my+"x"+mz+"\n");

        //***************************************//
		//*          MAIN ALGORITHM             *//
		//***************************************//
 
		// record time
		long start_time = System.currentTimeMillis();
		long inner_loop_time;

		// create the initial segmentation and registration algorithm
		initialization = new MultichannelFCM(image, objectMask, mx, my, mz, nt,
											rx, ry, rz, 
											classes, clusters, 
											smoothing, fuzziness, scaling, 
											addOutliers, outlier,
											useRegistration, backThreshold,
											maskingMode,
											Imin, Imax,
											userInterface, getProgressChangeListener());

		// create auxiliary algorithms
		if (correctInhomogeneity) {
			correction = new InhomogeneityCorrection[nt];
			for (t=0;t<nt;t++) {
				// a correction algorithm for each image...
				correction[t] = new InhomogeneityCorrection(fieldDegree,  InhomogeneityCorrection.IMAGE,
														image[t], objectMask[t], 
														initialization.getMemberships(),
														initialization.getCentroids(t),
														mx,my,mz,rx,ry,rz,
														classes,clusters,dim,
														getProgressChangeListener());
				// integrate it into the segmentation
				initialization.addInhomogeneityCorrection(t, correction[t].getField());
			}
		} else correction = null;
		
		if (useEdges) {
			// create the first edge map algorithm
			edge = new EdgeAdaptation[nt];
			edge[0] = new EdgeAdaptation(initialization.getMemberships(),
											mx,my,mz,classes,clusters,
											edgeSmoothness,edgeContrast,edgePrior,
											userInterface, getProgressChangeListener());
			// integrate it into the segmentation
			initialization.addEdgeMap(edge[0].getEdges());
		} else edge = null;
		
        // initial guess for centroids: first image only
		if (initMode.equals("modes")) {
			search = new ClusterSearch(clusters);
			search.computeHistogram(image[0],objectMask[0],mx,my,mz);
			search.findCentroids();	
			initialization.importCentroids(0,search.exportCentroids());
			search.finalize();
			search = null;	
		} else if (initMode.equals("range")) {
			float[] cent = new float[clusters];
			cent[0] = Imin[0] + 0.5f*(Imax[0]-Imin[0])/(float)clusters;
			for (k=1;k<clusters;k++)
				cent[k] = cent[k-1] + (Imax[0]-Imin[0])/(float)clusters;
			initialization.importCentroids(0,cent);
		} else if (initMode.equals("manual")) {
		}
        
        // initialize the first segmentation: iterates until quick convergence
		initialization.setMRF(0.0f);
		n=0;
		stop = false;
		distance = 0.0f;
		while (!stop) {
			distance = initialization.computeMemberships(1);
			initialization.computeCentroids(1);
			// no inhomogeneity correction at first 
			if (useEdges) edge[0].computeEdgeMap();
			if (n==0) initialization.setMRF(smoothing);
			n++;
			if (n >= iter0) stop = true;
			if (distance < maxDist0) stop = true;            
		}
		
		// add images one by one, until they are all there
		for (t=1;t<nt;t++) {
			// compute centroids for images 0 to t :
			// use membership function to init new centroids
			// and to register new image (if needed)
			fireProgressStateChanged("adding image " + t + " (max: " + distance + ")"); 
			fireProgressStateChanged(Math.round( (float)t/(float)nt)*100);
			
			if (useRegistration) {
				// register all ? only the new one ?
				register = new RegisterSegmentation(image[t], initialization.getMemberships(),
													mx, my, mz, rx, ry, rz, 
													classes, clusters,
													outlier,
													levels, firstIter, lastIter, topIter,
													0.0f, backThreshold[t],
													userInterface, getProgressChangeListener());
				register.runFullMultigrid();

				initialization.setTransform(t, register.exportTransformMatrix());
				initialization.computeTransformedCentroids(t+1);
				if (useEdges) edge[0].computeEdgeMap();
				initialization.computeTransformedMemberships(t+1);
			} else {	
				initialization.computeCentroids(t+1);
				if (useEdges) edge[0].computeEdgeMap();
				initialization.computeMemberships(t+1);
			}
		}
		
		// create a global mask and align the images if needed
		commonMask = initialization.createCommonTransformedMask();
		for (t=0;t<nt;t++) {
			if (useRegistration) {
				image[t] = initialization.exportTransformedImage(t);
			}
			objectMask[t] = commonMask;
		}
		
		// save the transforms for output
		if (useRegistration) {
			transmat = new TransMatrix[nt];
			for (t=0;t<nt;t++) {
				transmat[t] = new TransMatrix(4);
				double[][] tmp = extendTransform(initialization.convertTransform(t));
				for (int i=0;i<4;i++) for (int j=0;j<4;j++)
					transmat[t].set(i,j,tmp[i][j]);
			}
		} else {
			transmat = null;
		}

		
		// create the appropriate segmentation algorithm
		segmentation = new LongitudinalFCM(image, objectMask, mx, my, mz, nt,
											rx, ry, rz, 
											classes, clusters, 
											smoothing, temporal, fuzziness, outlier,
											scaling, backThreshold,
											Imin, Imax,
											userInterface, getProgressChangeListener());

		// import initial results
		for (t=0;t<nt;t++) {
			segmentation.importMemberships(t, initialization.getMemberships());
			segmentation.importCentroids(t, initialization.getCentroids(t));
		}
		// clear the init
		initialization.finalize();
		initialization = null;
		
		// modify auxiliary algorithms
		if (correctInhomogeneity) {
			for (t=0;t<nt;t++) {
				// a correction algorithm for each image...
				correction[t].setMemberships(segmentation.getMemberships(t));
				correction[t].setMask(commonMask);
				// integrate it into the segmentation
				segmentation.addInhomogeneityCorrection(t, correction[t].getField());
			}
		} else correction = null;
		
		if (useEdges) {
			edge[0].setMemberships(segmentation.getMemberships(0));
			segmentation.addEdgeMap(0, edge[0].getEdges());
			for (t=1;t<nt;t++) {
				// create the edge map algorithm
				edge[t] = new EdgeAdaptation(segmentation.getMemberships(t),
											mx,my,mz,classes,clusters,
											edgeSmoothness,edgeContrast,edgePrior,
											userInterface, getProgressChangeListener());
				// copy the first one
				edge[t].importEdges(edge[0].getEdges());
				// integrate it into the segmentation
				segmentation.addEdgeMap(t, edge[t].getEdges());
			}
		} else edge = null;
				
					
		// main iterations: compute the classes on the image
		// two steps: iterate quickly on each inhomogeneity correction degree, then refine
		distance = 0.0f;
		n = 0;
		int Niterations = 1;
		for (int d=0;d<=fieldDegree;d++) {
			stop = false;
			n = 0;
			while ((!stop) && (!threadStopped)) {
				fireProgressStateChanged("iteration " + Niterations + " (max: " + distance + ")"); 
				fireProgressStateChanged(Math.round( (float)n/(float)iterations)*100);
				if (debug) MedicUtilPublic.displayMessage("iteration " + Niterations + " (max: " + distance + ")\n");
				
				// update external fields
				if (correctInhomogeneity) {
					for (t=0;t<nt;t++) {
						correction[t].computeCorrectionField(d);
					}
				}
				if (useEdges) {
					for (t=0;t<nt;t++) {
						edge[t].computeEdgeMap();
					}
				}
				
				// update centroids
				segmentation.computeCentroids();
				
				// update membership
				distance = segmentation.computeMemberships();
				
				// check for segmentation convergence 
				n++;
				Niterations++;
				if (n >= iter0) stop = true;
				if (distance < maxDist0) stop = true;            
			}
		}
		
		// second loop at maximum field degree
		stop = false;
		n = 0;
		while ((!stop) && (!threadStopped)) {
			fireProgressStateChanged("iteration " + Niterations + " (max: " + distance + ")"); 
			fireProgressStateChanged(Math.round( (float)n/(float)iterations)*100);
			if (debug) MedicUtilPublic.displayMessage("iteration " + Niterations + " (max: " + distance + ")\n");
            
			// update external fields
			if (correctInhomogeneity) {
				for (t=0;t<nt;t++) {
					correction[t].computeCorrectionField(fieldDegree);
				}
			}
			if (useEdges) {
				for (t=0;t<nt;t++) {
					edge[t].computeEdgeMap();
				}
			}
			
			// update centroids
			segmentation.computeCentroids();
			
            // update membership
			distance = segmentation.computeMemberships();
			
			// check for segmentation convergence 
			n++;
			Niterations++;
			if (n >= iterations) stop = true;
			if (distance < maxDistance) stop = true;            
        }

        /* if cancelled, return the current result
		if (threadStopped) {
        	finalize();
        	return;
        }
		*/
        
        if (debug) MedicUtilPublic.displayMessage("total time: (milliseconds): " + (System.currentTimeMillis()-start_time)+"\n"); 
		
		// order the classes in increasing order
		id = segmentation.computeCentroidOrder();
		
        // extract results (segmentation and/or classes) and store them in destImage[]
		fireProgressStateChanged("creating result images...");
		
        int Ndest = 0;
        try {        
			for (t=0;t<nt;t++) {
				if (useRegistration) {
					srcImage[t].setMatrix(transmat[t]);
					// interpolate the source image to the transformed coordinates
					buffer = bufferFromImage(image[t]);
					srcImage[t].importData(0, buffer, true);
				}
				if (!outputType.equals("hard_segmentation")) {
					fireProgressStateChanged("memberships...");
					for (k=0;k<classes;k++) {
						buffer = bufferFromImage(segmentation.exportMembership(t)[k]);
						destImage[Ndest+id[k+1]-1].importData(0, buffer, true);
					}
					Ndest+=classes;
				}
				if (!outputType.equals("fuzzy_segmentation")) {
					fireProgressStateChanged("final classification...");
					bytebuffer = orderedBufferFromImage(segmentation.exportHardClassification(t),id);
					destImage[Ndest].importData(0, bytebuffer, true);
					Ndest++;
				}
				if (outputType.equals("all_result_images")) {
					if (correctInhomogeneity) {
						fireProgressStateChanged("gain field...");
						buffer = bufferFromImage(correction[t].exportField());
						destImage[Ndest].importData(0, buffer, true);
						Ndest++;
					}
					if (useEdges) {
						fireProgressStateChanged("edge maps...");
						buffer = bufferFromImage(edge[t].exportEdges()[0]);
						destImage[Ndest].importData(0, buffer, true);
						Ndest++;
						buffer = bufferFromImage(edge[t].exportEdges()[1]);
						destImage[Ndest].importData(0, buffer, true);
						Ndest++;
						buffer = bufferFromImage(edge[t].exportEdges()[2]);
						destImage[Ndest].importData(0, buffer, true);
						Ndest++;
					}
				}
			}
			segmentation.finalize();
			segmentation = null;
			if (correctInhomogeneity) {
				for (t=0;t<nt;t++) {
					correction[t].finalize();
					correction[t] = null;
				}
			}
			if (useEdges) {
				for (t=0;t<nt;t++) {
					edge[t].finalize();
					edge[t] = null;
				}
			}
		} catch (OutOfMemoryError e) {
			segmentation.finalize();
			segmentation = null;				
            buffer = null;
            errorCleanUp("Algorithm: Out of memory creating hard classification", true);
            finalize();
            setCompleted(false);
            return;
        } catch (IOException error) {
            errorCleanUp("Algorithm: export problem to destImage[]", true);
            finalize();
			setCompleted(false);
            return;
        }
        
        threadStopped = false;
		setCompleted(true);
    } // calcSegmentation
	
	/** brings image in correct array */
	private float[][][] expandImage(float[] image) {
		int 		x,y,z;
		float[][][] 	tmp;
		
		tmp = new float[nx][ny][nz];
		for (x=1;x<nx-1;x++)
			for (y=1;y<ny-1;y++)
				for (z=1;z<nz-1;z++)
					tmp[x][y][z] = image[ (x-1) + (nx-2)*(y-1) + (nx-2)*(ny-2)*(z-1) ];
		
		return tmp;
	}
		
	/** creates a mask for unused data */
	private boolean[][][] createObjectMask(float[][][] image, float val) {
		int 		x,y,z;
		boolean[][][]  	objMask;
        boolean useWholeImage = true;

		// uses only values over the threshold, if mask used
		objMask = new boolean[nx][ny][nz];
		for (x=1;x<nx-1;x++)
			for (y=1;y<ny-1;y++)
				for (z=1;z<nz-1;z++) {
                    if (useWholeImage) {
                        if ( (cropBackground) && (image[x][y][z] <= val) )
                            objMask[x][y][z] = false;
                        else
                            objMask[x][y][z] = true;
                    } else if (mask.get((x-1)+(nx-2)*(y-1)+(nx-2)*(ny-2)*(z-1)) ) {
                        if ( (cropBackground) && (image[x][y][z] <= val) )
                            objMask[x][y][z] = false;
                        else
                            objMask[x][y][z] = true;
                    }
                }
		// remove the boundary from the computations
		for (x=0;x<nx;x++)
			for (y=0;y<ny;y++) {
				objMask[x][y][0] = false;
				objMask[x][y][nz-1] = false;
			}
		for (y=0;y<ny;y++)
			for (z=0;z<nz;z++) {
				objMask[0][y][z] = false;
				objMask[nx-1][y][z] = false;
			}
		for (z=0;z<nz;z++)
			for (x=0;x<nx;x++) {
				objMask[x][0][z] = false;
				objMask[x][ny-1][z] = false;
			}

		return objMask;
	} // createObjectMask
        
    /** sets the processing lower and upper boundaries */
    private void computeProcessingBoundaries(boolean[][][] objMask) {
		int 		x,y,z;
        
        x0 = nx;
        xN = 0;
        y0 = ny;
        yN = 0;
        z0 = nz;
        zN = 0;
        for (x=1;x<nx-1;x++)
			for (y=1;y<ny-1;y++)
				for (z=1;z<nz-1;z++) {
                    if (objMask[x][y][z]) {
                        if (x < x0) x0 = x;
                        if (x > xN) xN = x;
                        if (y < y0) y0 = y;
                        if (y > yN) yN = y;
                        if (z < z0) z0 = z;
                        if (z > zN) zN = z;
                    }
                }
        // debug
        System.out.print("boundaries: ["+x0+","+xN+"] ["+y0+","+yN+"] ["+z0+","+zN+"]\n");
        
        return;
    }

    	/** create smaller image (for saving memory) */
	private float[][][] reduceImageSize(float[][][] image) {
		float[][][] smaller = new float[mx][my][mz];

		for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
			smaller[x][y][z] = 0.0f;
		}
		for (int x=x0;x<=xN;x++) {
            for (int y=y0;y<=yN;y++) {
                for (int z=z0;z<=zN;z++) {
					smaller[x-x0+1][y-y0+1][z-z0+1] = image[x][y][z];
				}
			}
		}
		return smaller;
	}
	private boolean[][][] reduceImageSize(boolean[][][] image) {
		boolean[][][] smaller = new boolean[mx][my][mz];

		for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
			smaller[x][y][z] = false;
		}
		for (int x=x0;x<=xN;x++) {
            for (int y=y0;y<=yN;y++) {
                for (int z=z0;z<=zN;z++) {
					smaller[x-x0+1][y-y0+1][z-z0+1] = image[x][y][z];
				}
			}
		}
		return smaller;
	}
	/** retrieve original size from smaller image */
	private float[][][] extendImageSize(float[][][] image) {
		float[][][] larger = new float[nx][ny][nz];

		for (int x=0;x<nx;x++) {
			for (int y=0;y<ny;y++) {
				for (int z=0;z<nz;z++) {
					larger[x][y][z] = 0.0f;
				}
			}
		}		
 		for (int x=x0;x<=xN;x++) {
            for (int y=y0;y<=yN;y++) {
                for (int z=z0;z<=zN;z++) {
					larger[x][y][z] = image[x-x0+1][y-y0+1][z-z0+1];
				}
			}
		}
		return larger;
	}

	/** retrieve original size from smaller image */
	private double[][] extendTransform(double[][] mat) {
		double[][] larger = new double[4][4];

		for (int i=0;i<4;i++) for (int j=0;j<4;j++) 
            larger[i][j] = mat[i][j];
        // add the offset: (Id-R^T)P0
		larger[0][3] += ( (x0-1)*rx - mat[0][0]*(x0-1)*rx - mat[0][1]*(y0-1)*ry - mat[0][2]*(z0-1)*rz);
		larger[1][3] += ( (y0-1)*ry - mat[1][0]*(x0-1)*rx - mat[1][1]*(y0-1)*ry - mat[1][2]*(z0-1)*rz);
		larger[2][3] += ( (z0-1)*rz - mat[2][0]*(x0-1)*rx - mat[2][1]*(y0-1)*ry - mat[2][2]*(z0-1)*rz);
        
		return larger;
	}

	/** retrieve original size from smaller image */
	private float[] bufferFromImage(float[][][] image) {
		float[] larger = new float[(nx-2)*(ny-2)*(nz-2)];

		for (int n=0;n<(nx-2)*(ny-2)*(nz-2);n++)
			larger[n] = 0.0f;
		
		for (int x=x0;x<=xN;x++) {
            for (int y=y0;y<=yN;y++) {
                for (int z=z0;z<=zN;z++) {
					larger[(x-1)+(nx-2)*(y-1)+(nx-2)*(ny-2)*(z-1)] = image[x-x0+1][y-y0+1][z-z0+1];
				}
			}
		}
		return larger;
	}

	/** retrieve original size from smaller image */
	private byte[] orderedBufferFromImage(byte[][][] image, int[] id) {
		byte[] larger = new byte[(nx-2)*(ny-2)*(nz-2)];

		for (int n=0;n<(nx-2)*(ny-2)*(nz-2);n++)
			larger[n] = 0;
		
		for (int x=x0;x<=xN;x++) {
            for (int y=y0;y<=yN;y++) {
                for (int z=z0;z<=zN;z++) {
					larger[(x-1)+(nx-2)*(y-1)+(nx-2)*(ny-2)*(z-1)] = (byte)id[(int)image[x-x0+1][y-y0+1][z-z0+1]];
				}
			}
		}
		return larger;
	}

}