package edu.jhmi.rad.medic.algorithms;

import edu.jhmi.rad.medic.dialogs.*;
import edu.jhmi.rad.medic.methods.*;
import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.utilities.*;
import edu.jhu.ece.iacl.jist.io.ArrayIntegerTxtReaderWriter;
import edu.jhu.ece.iacl.utility.ArrayUtil;
import edu.jhu.ece.iacl.utility.VectorParser;

import gov.nih.mipav.model.algorithms.AlgorithmBase;
import gov.nih.mipav.model.structures.*;
import gov.nih.mipav.view.*;

import java.io.*;
import java.util.*;
import java.lang.*;
import java.text.*;

import org.apache.commons.lang.ArrayUtils;
 
/**
 *
 *   computes several measurements for quantifying volumetric images.
 *   <p>
 *	 It can compare hard segmentations and fuzzy memberships,
 *	 computing volume, overlap, topology, intensity measures.
 *
 *	@version    May 2007
 *	@author     Pierre-Louis Bazin
 *	@author 	John Bogovic
 *  @see 		JDialogVolumeStatistics
 *		
 *
*/
public class AlgorithmVolumeStatistics extends AlgorithmBase {

    private ModelImage  intImage;
    private ModelImage  refImage;
    private int[][] 	groups;		// optionally add statistics for groups of labels
    private int[][]		confMtx;
	
    private ViewUserInterface       userInterface;

    // image information
	private int 		nx,ny,nz,nt; 			// original image dimensions
	private float 		rx,ry,rz; 			// original image resolutions
	
	// algorithm parameters
	private ArrayList<String>	statistics;		// the type of algorithm to use
	private	boolean			useVOI;				// whether to use VOIs or not
	
	private	String			connectivity;			// connectivity	
	private	int				cObj, cBg;
	
	private ArrayList<String>	output;		// the type of algorithm to use
	private boolean		writeToFile;
	private	String		filename;
	private	final		String 		delim;
		    
	private 	boolean		verbose = true;
	private     boolean		debug = true;
	
	private	static final	float	INF = 1e30f;
	private	static final	float	ZERO = 1e-30f;
	
	/**
    *	Constructor for 3D images in which changes are placed in a predetermined destination image.
    *   @param targetImg      Image model where result image is to stored.
    *   @param srcImg       Source image model.
    */
	public AlgorithmVolumeStatistics(ModelImage srcImg, ModelImage intImg, ModelImage refImg, 
										ArrayList<String> stats_, 
										boolean voi_,
										String connect_,
										boolean tf_, String file_, String del_) {
			
		super(null, srcImg);
		userInterface = ViewUserInterface.getReference();
        
		intImage = intImg;
		refImage = refImg;
		
		statistics = stats_;
		useVOI = voi_;
		connectivity = connect_;
			 if (connectivity.equals("6/18")) { cObj = 6; cBg = 18; }
		else if (connectivity.equals("6/26")) { cObj = 6; cBg = 26; }
		else if (connectivity.equals("18/6")) { cObj = 18; cBg = 6; }
		else if (connectivity.equals("26/6")) { cObj = 26; cBg = 6; }

		
		writeToFile = tf_;
		filename = file_;
		delim = del_;
	}
	
	public void setGroups(int[][] groups){
		this.groups=groups;
	}
	
	public int[][] getGroups(){
		return groups;
	}

    /**
    *	Prepares this class for destruction.
    */
	public void finalize(){
        super.finalize();
	    srcImage = null;
        intImage = null;
		refImage = null;
        System.gc();
	}

    /**
    *	Constructs a string of the contruction parameters and outputs the string to the messsage frame if the logging
    *   procedure is turned on.
    *	@param destinationFlag	If true the log includes the name of the destination flag.
    */
	/*
    private void constructLog(boolean destinationFlag) {
        if ( destinationFlag == false) {
            historyString = new String( "VolumeStatistics(" + ")");
        }
        else  {
            historyString = new String( "VolumeStatistics(" + ")");
        }
        historyString += "\n";  // append a newline onto the string
        writeLog();
    }
	*/

    /**
    *   Starts the algorithm.
    */
	public void runAlgorithm() {

        if (srcImage  == null) {
            displayError("Source image is null");
            return;
        }
        
		// start the timer to compute the elapsed time
        setStartTime();

		// computations here
		//constructLog(true);
		
		// init dimensions and resolutions
		if (srcImage.getNDims() == 2) {
			nx = srcImage.getExtents()[0];
			ny = srcImage.getExtents()[1];
			nz = 1;
			nt = 1;
			
			rx = srcImage.getFileInfo()[0].getResolutions()[0];
			ry = srcImage.getFileInfo()[0].getResolutions()[1];
			rz = 1.0f;			
		} else if (srcImage.getNDims() == 3) {
			nx = srcImage.getExtents()[0];
			ny = srcImage.getExtents()[1];
			nz = srcImage.getExtents()[2];
			nt = 1;
			
			rx = srcImage.getFileInfo()[0].getResolutions()[0];
			ry = srcImage.getFileInfo()[0].getResolutions()[1];
			rz = srcImage.getFileInfo()[0].getResolutions()[2];
		} else if (srcImage.getNDims() == 4) {
			nx = srcImage.getExtents()[0];
			ny = srcImage.getExtents()[1];
			nz = srcImage.getExtents()[2];
			nt = srcImage.getExtents()[3];
			
			rx = srcImage.getFileInfo()[0].getResolutions()[0];
			ry = srcImage.getFileInfo()[0].getResolutions()[1];
			rz = srcImage.getFileInfo()[0].getResolutions()[2];
		}
		
		for (int t=0;t<nt;t++) {
		
			// 1. Main image
			int[] 	imgbuffer = null;
			float[]	intbuffer = null;
			int[]	refbuffer = null;
			try {
				if (srcImage.isColorImage()) {
					imgbuffer = new int[4*nx*ny*nz];	
					srcImage.exportData(t*4*nx*ny*nz,4*nx*ny*nz, imgbuffer); // locks and releases lock				
				} else {
					imgbuffer = new int[nx*ny*nz];	
					srcImage.exportData(t*nx*ny*nz,nx*ny*nz, imgbuffer); // locks and releases lock
				}
				if (intImage!=null) {
					intbuffer = new float[nx*ny*nz];
					intImage.exportData(t*nx*ny*nz,nx*ny*nz, intbuffer); // locks and releases lock
				}
				if (refImage!=null) {
					if (refImage.isColorImage()) {
						refbuffer = new int[4*nx*ny*nz];
						refImage.exportData(t*4*nx*ny*nz,4*nx*ny*nz, refbuffer); // locks and releases lock
					} else {
						refbuffer = new int[nx*ny*nz];
						refImage.exportData(t*nx*ny*nz,nx*ny*nz, refbuffer); // locks and releases lock
					}
				}
			} catch (IOException error) {
				imgbuffer = null;
				intbuffer = null;
				refbuffer = null;
				errorCleanUp("Algorithm: source images locked", true);
				return;
			} catch (OutOfMemoryError e){
				imgbuffer = null;
				intbuffer = null;
				refbuffer = null;
				errorCleanUp("Algorithm: Out of memory creating process buffer", true);
				return;
			}
			
			computeStatistics(imgbuffer, intbuffer, refbuffer);
		}
		// compute the elapsed time
        computeElapsedTime();

        setCompleted(true);
    } // end runAlgorithm()
	
	/**
    *	compute the size and overlap of the new image with regard to reference
	*	it is assumed that the labels may differ but follow the same ordering in intensity
    */
    private void computeStatistics(int[] img, float[] intens, int[] ref){
		int[][][]				image;
		float[][][]				intensity = null;
		int[][][]				reference = null;
		ImageCropper			crop = new ImageCropper(nx,ny,nz);
		
		int[][] imgMapping = null;
		int[][] refMapping = null;
		
		int mod;
		int progress;
		
		fireProgressStateChanged("initialization...");
        
		try {
            if (srcImage.isColorImage()) {
				imgMapping = crop.mapColorsToLabels(img);
				image = crop.convertColorToLabelImage(img,imgMapping);
			} else {
				image = crop.convertArray(img);
			}
            img = null;
            if (intens!=null) intensity = crop.convertArray(intens);
            intens = null;
            if (ref!=null) {
				if (refImage.isColorImage()) {
					refMapping = crop.mapColorsToLabels(ref);
					reference = crop.convertColorToLabelImage(ref,refMapping);
				} else {
					reference = crop.convertArray(ref);
				}
			}
            ref = null;
        } catch (OutOfMemoryError e) {
            img = null; image = null;
            intens = null; intensity = null;
            ref = null; reference = null;
            errorCleanUp("Algorithm: Out of memory creating process buffer", true);
            setCompleted(false);
            return;
        }

		//***************************************//
		//*          MAIN ALGORITHM             *//
		//***************************************//
		
		// record time
		long start_time = System.currentTimeMillis();
		
		String line;
		
		String imgtag,reftag,noreftag,inttag,nointtag;
		imgtag = delim+srcImage.getImageName();
		reftag = delim+refImage.getImageName();
		noreftag = delim+"no_reference";
		inttag = delim+intImage.getImageName();
		nointtag = delim+"no_intensity";
		
		// standard computations for the object
		output = new ArrayList();
		
		// get the set of labels
		int[] imgLabels = ObjectProcessing.listOrderedLabels(image,nx,ny,nz);
		
		TreeSet<Integer> labelset = new TreeSet<Integer>();
		int NimgLabels = imgLabels.length;
		
		for(int i=0; i<NimgLabels; i++){
			labelset.add(imgLabels[i]);
		}
		
		// create an array with indices that map from the 
		// set of seg labels to the set of total labels
		// (this is used for measures that depend on the seg only - not on the ref)
		int[] imgLabelsToSetLabelIdxs = null;
		if(reference!=null){
			System.out.println("\n\nadding reference labels\n\n");
			int[] refLabels = ObjectProcessing.listOrderedLabels(reference,nx,ny,nz);
			for(int i=0; i<refLabels.length; i++){
				labelset.add(refLabels[i]);
			}
			
		}
		System.out.println("labelset:\n"+labelset);
		
		int Nlabels = labelset.size();
		NimgLabels = imgLabels.length; // reset the value of this, because 'zero' may have been removed
		
		// build an array containing all labels
		Integer[] a = labelset.toArray(new Integer[Nlabels]);
		
		int[] allLabels = ArrayUtils.toPrimitive(a);
		Arrays.sort(allLabels);
		
		System.out.println("allLabels:\n"+ArrayUtil.printArray(allLabels));
		System.out.println("nlabels: "+allLabels.length);
		
		if(ref!=null){
			imgLabelsToSetLabelIdxs = new int[imgLabels.length];
			for(int i=0; i<imgLabels.length; i++){
				imgLabelsToSetLabelIdxs[i] = Arrays.binarySearch(allLabels, imgLabels[i]);
			}
		}else{
			imgLabelsToSetLabelIdxs = VectorParser.parseColonInt("0:"+Integer.toString(imgLabels.length-1));
		}
		
		System.out.println("imgLabelsToSetLabelIdxs:\n"+ArrayUtil.printArray(imgLabelsToSetLabelIdxs));
		
		// deal with groups
		int Ngroups = 0;
		if(groups!=null){
			Ngroups = groups.length;
			// sort each row
			for(int i=0; i<groups.length; i++){
				Arrays.sort(groups[i]);
			}
		}
		
		
//		int[] allLabels = new int[Nlabels];
//		Iterator<Integer> it = labelset.iterator();
//		int k=0;
//		while(it.hasNext()){
//			allLabels[k]=it.next().intValue();
//			k++;
//		}

//		// remove the zero label
//		if (imgLabels[0]==0) {
//			int[] tmp = new int[Nlabels-1];
//			for (int n=1;n<Nlabels;n++)
//				tmp[n-1] = imgLabels[n];
//			Nlabels--;
//			imgLabels = tmp;
//		}
		
		line = "Image_labels"+imgtag+noreftag+nointtag;
		for (int n=0;n<Nlabels;n++) line += (delim+allLabels[n]);
		// assign group labels
		for (int n=0;n<Ngroups;n++){
			line += delim;
			line += "[";
			for(int m=0; m<groups[n].length; m++){
				line += (groups[n][m]+",");
			}
			line += "]";
			
		}
		line +=("\n");
		output.add(line);
		
		// display the color mapping
		if (srcImage.isColorImage()) {
			line = "Image_color_1"+imgtag+noreftag+nointtag;
			for (int n=0;n<Nlabels;n++) line += (delim+imgMapping[n][0]);
			line +=("\n");
			output.add(line);
			line = "Image_color_2"+imgtag+noreftag+nointtag;
			for (int n=0;n<Nlabels;n++) line += (delim+imgMapping[n][1]);
			line +=("\n");
			output.add(line);
			line = "Image_color_3"+imgtag+noreftag+nointtag;
			for (int n=0;n<Nlabels;n++) line += (delim+imgMapping[n][2]);
			line +=("\n");
			output.add(line);
			line = "Image_color_4"+imgtag+noreftag+nointtag;
			for (int n=0;n<Nlabels;n++) line += (delim+imgMapping[n][3]);
			line +=("\n");
			output.add(line);
		}
		
		// build object list from VOIs
		if (useVOI) {
			// get list of VOI names
			int Nvoi;
			String[] voiName;
		
			// transfer each VOI into a mask
			boolean[][][][] voiMask;
			// TO DO
		}
		
		int Nreflb = 0;
		int[] refLabels = null;
		if (reference!=null) {
			Nreflb= ObjectProcessing.countLabels(reference,nx,ny,nz);
			refLabels = ObjectProcessing.listOrderedLabels(reference,nx,ny,nz);
			
			// remove the zero label
			if (refLabels[0]==0) {
				int[] trf = new int[Nreflb-1];
				for (int n=1;n<Nreflb;n++)
					trf[n-1] = refLabels[n];
				Nreflb--;
				refLabels = trf;
			}
			/* not needed ?
			line = "Reference_labels"+imgtag+reftag+nointtag;
			for (int n=0;n<Nreflb;n++) line = line+(delim+refLabels[n]);
			line +=("\n");
			output.add(line);
			*/
		}
		
		//TODO: might be a bit faster if the boolean[][][] object arrays were computed once
		// for each object instead of being recomputed for each statistic
		
		// compute the statistics
		for (int s=0; s<statistics.size(); s++) {
			System.out.print("Statistic: "+statistics.get(s)+"\n");
			if (statistics.get(s).equals("Voxels")) {
				// compute the volumes
				float[] volume = new float[Nlabels+Ngroups];
				for (int nn=0;nn<NimgLabels;nn++) {
					int n = imgLabelsToSetLabelIdxs[nn];
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					volume[n] = ObjectProcessing.volume(obj,nx,ny,nz);
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					volume[Nlabels+nn] = ObjectProcessing.volume(obj,nx,ny,nz);
				}
				line = "Voxels"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line += (delim+(int)volume[n]);
				line +="\n";
				output.add(line);
			}
			if (statistics.get(s).equals("Volume")) {
				// compute the volumes
				float[] volume = new float[Nlabels+Ngroups];
				for (int nn=0;nn<NimgLabels;nn++) {
					int n = imgLabelsToSetLabelIdxs[nn];
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					volume[nn] = ObjectProcessing.volume(obj,nx,ny,nz);
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					volume[Nlabels+nn] = ObjectProcessing.volume(obj,nx,ny,nz);
				}
				line = "Volume"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line += (delim+volume[n]*rx*ry*rz);
				line +="\n";
				output.add(line);
			}
			if (statistics.get(s).equals("Surface")) {
				// compute the volumes
				float[] surface = new float[Nlabels+Ngroups];
				for (int nn=0;nn<NimgLabels;nn++) {
					int n = imgLabelsToSetLabelIdxs[nn];
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					surface[nn] = ObjectProcessing.boundaryArea(obj,nx,ny,nz,rx,ry,rz);
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					surface[Nlabels+nn] = ObjectProcessing.boundaryArea(obj,nx,ny,nz,rx,ry,rz);
				}
				line = "Surface"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line += (delim+surface[n]*rx*ry*rz);
				line +="\n";
				output.add(line);
			}
			
			if (statistics.get(s).equals("Euler_characteristic")) {
				// compute the Euler characteristics, number of parts and holes
				float[] euler = new float[Nlabels+Ngroups];
				for (int nn=0;nn<NimgLabels;nn++) {
					int n = imgLabelsToSetLabelIdxs[nn];
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					euler[nn] = ObjectProcessing.eulerCharacteristic(obj,nx,ny,nz,cObj,cBg);
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					euler[Nlabels+nn] = ObjectProcessing.eulerCharacteristic(obj,nx,ny,nz,cObj,cBg);
				}
				line = "Euler_characteristic"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+(int)euler[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Object_parts")) {
				int[] parts = new int[Nlabels+Ngroups];
				for (int nn=0;nn<NimgLabels;nn++) {
					int n = imgLabelsToSetLabelIdxs[nn];
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					parts[nn] = ObjectProcessing.countParts(obj,nx,ny,nz,cObj,cBg);
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					parts[Nlabels+nn] = ObjectProcessing.countParts(obj,nx,ny,nz,cObj,cBg);
				}
				line = "Object_parts"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+parts[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Object_cavities")) {
				int[] holes = new int[Nlabels+Ngroups];
				for (int nn=0;nn<NimgLabels;nn++) {
					int n = imgLabelsToSetLabelIdxs[nn];
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					holes[nn] = ObjectProcessing.countHoles(obj,nx,ny,nz,cObj,cBg);
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					holes[Nlabels+nn] = ObjectProcessing.countHoles(obj,nx,ny,nz,cObj,cBg);
				}
				line = "Object_cavities"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+holes[n]);
				line+=("\n");
				output.add(line);
			}
			
			if (statistics.get(s).equals("Surface_ripples")) {
				// compute the volumes
				float[] surface = new float[Nlabels+Ngroups];
				float[] opened = new float[Nlabels+Ngroups];
				float[] closed = new float[Nlabels+Ngroups];
				boolean[][][] mask = new boolean[3][3][3];
				for (int i=-1;i<=1;i++) for (int j=-1;j<=1;j++) for (int l=-1;l<=1;l++) {
					if (i*i+j*j+l*l<=1) mask[1+i][1+j][1+l] = true;
					else mask[1+i][1+j][1+l] = false;
				}
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,imgLabels[n],imgLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					surface[n] = ObjectProcessing.boundaryArea(obj,nx,ny,nz,rx,ry,rz);
					boolean[][][] obj2 = Morphology.erodeObject(obj,nx,ny,nz,mask,1,1,1);
					obj2 = Morphology.dilateObject(obj2,nx,ny,nz,mask,1,1,1);
					opened[n] = ObjectProcessing.boundaryArea(obj2,nx,ny,nz,rx,ry,rz);
					obj2 = Morphology.dilateObject(obj,nx,ny,nz,mask,1,1,1);
					obj2 = Morphology.erodeObject(obj2,nx,ny,nz,mask,1,1,1);
					closed[n] = ObjectProcessing.boundaryArea(obj2,nx,ny,nz,rx,ry,rz);
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					surface[Nlabels+nn] = ObjectProcessing.boundaryArea(obj,nx,ny,nz,rx,ry,rz);
					boolean[][][] obj2 = Morphology.erodeObject(obj,nx,ny,nz,mask,1,1,1);
					obj2 = Morphology.dilateObject(obj2,nx,ny,nz,mask,1,1,1);
					opened[Nlabels+nn] = ObjectProcessing.boundaryArea(obj2,nx,ny,nz,rx,ry,rz);
					obj2 = Morphology.dilateObject(obj,nx,ny,nz,mask,1,1,1);
					obj2 = Morphology.erodeObject(obj2,nx,ny,nz,mask,1,1,1);
					closed[Nlabels+nn] = ObjectProcessing.boundaryArea(obj2,nx,ny,nz,rx,ry,rz);
				}
				line = "Surface_ripples"+imgtag+noreftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line += (delim+((2*surface[n]-opened[n]-closed[n])/(ZERO+surface[n])));
				line +="\n";
				output.add(line);
			}
			
			if (statistics.get(s).equals("Dice_overlap")) {
				// compare: requires same labels (use the reference as basis)
				float[] dice = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] xor = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.XOR,nx,ny,nz);
						float diff = ObjectProcessing.volume(xor,nx,ny,nz);
										
						dice[n] = ((vol1+vol2-diff)/(vol1+vol2));
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
					float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
					boolean[][][] xor = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.XOR,nx,ny,nz);
					float diff = ObjectProcessing.volume(xor,nx,ny,nz);
									
					dice[Nlabels+nn] = ((vol1+vol2-diff)/(vol1+vol2));
				}
				line = "Dice_overlap"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+dice[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Jaccard_overlap")) {
				// compare: requires same labels (use the reference as basis)
				float[] jaccard = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] xor = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.XOR,nx,ny,nz);
						float diff = ObjectProcessing.volume(xor,nx,ny,nz);
										
						jaccard[n] = ((vol1+vol2-diff)/(vol1+vol2+diff));
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
					float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
					boolean[][][] xor = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.XOR,nx,ny,nz);
					float diff = ObjectProcessing.volume(xor,nx,ny,nz);
					
					jaccard[Nlabels+nn] = ((vol1+vol2-diff)/(vol1+vol2+diff));
				}
				line = "Jaccard_overlap"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+jaccard[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Mountford_overlap")) {
				// compare: requires same labels (use the reference as basis)
				float[] mountford = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] xor = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.XOR,nx,ny,nz);
						float diff = ObjectProcessing.volume(xor,nx,ny,nz);
										
						mountford[n] = (vol1+vol2-diff)/(2.0f*vol1*vol2 - (vol1+vol2)*(vol1+vol2-diff)/2.0f);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
					float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
					boolean[][][] xor = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.XOR,nx,ny,nz);
					float diff = ObjectProcessing.volume(xor,nx,ny,nz);
					
					mountford[Nlabels+nn] = (vol1+vol2-diff)/(2.0f*vol1*vol2 - (vol1+vol2)*(vol1+vol2-diff)/2.0f);
				}
				line = "Mountford_overlap"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+mountford[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("True_positive_ratio")) {
				// compare: requires same labels (use the reference as basis)
				float[] truepos = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
//						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float inter = ObjectProcessing.volume(and,nx,ny,nz);
										
						truepos[n] = (inter/vol2);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
//					float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
					float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
					boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
					float inter = ObjectProcessing.volume(and,nx,ny,nz);
					
					truepos[Nlabels+nn] = (inter/vol2);
				}
				line = "True_positive_ratio"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+truepos[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("False_positive_ratio")) {
				// compare: requires same labels (use the reference as basis)
				float[] falsepos = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float inter = ObjectProcessing.volume(and,nx,ny,nz);
										
						falsepos[n] = ((vol1-inter)/vol2);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
					float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
					boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
					float inter = ObjectProcessing.volume(and,nx,ny,nz);
					
					falsepos[Nlabels+nn] = ((vol1-inter)/vol2);
				}
				line = "False_positive_ratio"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+falsepos[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("False_negative_ratio")) {
				// compare: requires same labels (use the reference as basis)
				float[] falseneg = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
//						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float inter = ObjectProcessing.volume(and,nx,ny,nz);
										
						falseneg[n] = ((vol2-inter)/vol2);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
//					float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
					float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
					boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
					float inter = ObjectProcessing.volume(and,nx,ny,nz);
					
					falseneg[Nlabels+nn] = ((vol2-inter)/vol2);
				}
				line = "False_negative_ratio"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+falseneg[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("True_segmented_ratio")) {
				// compare: requires same labels (use the reference as basis)
				float[] trueseg = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
//						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float inter = ObjectProcessing.volume(and,nx,ny,nz);
										
						trueseg[n] = (inter/vol1);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
//					float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
					boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
					float inter = ObjectProcessing.volume(and,nx,ny,nz);
					
					trueseg[Nlabels+nn] = (inter/vol1);
				}
				line = "True_segmented_ratio"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+trueseg[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("False_segmented_ratio")) {
				// compare: requires same labels (use the reference as basis)
				float[] falseseg = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
//						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float inter = ObjectProcessing.volume(and,nx,ny,nz);
										
						falseseg[n] = ((vol1-inter)/vol1);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
//					float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
					boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
					float inter = ObjectProcessing.volume(and,nx,ny,nz);
					
					falseseg[Nlabels+nn] = ((vol1-inter)/vol1);
				}
				line = "False_segmented_ratio"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+falseseg[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Volume_difference")) {
				// compare: requires same labels (use the reference as basis)
				float[] voldiff = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
										
						voldiff[n] = Numerics.abs(vol1/vol2-1.0f);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
					float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
					
					voldiff[Nlabels+nn] = Numerics.abs(vol1/vol2-1.0f);
				}
				line = "Volume_difference"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+voldiff[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Containment_index(seg_in_ref)")) {
				// compare: requires same labels (use the reference as basis)
				float[] seginref = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
//						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float over = ObjectProcessing.volume(and,nx,ny,nz);
										
						seginref[n] = (over/vol1);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
//					float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
					boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
					float over = ObjectProcessing.volume(and,nx,ny,nz);
					
					seginref[Nlabels+nn] = (over/vol1);
				}
				line = "Containment_index(seg_in_ref)"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+seginref[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Containment_index(ref_in_seg)")) {
				// compare: requires same labels (use the reference as basis)
				float[] refinseg = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
//						float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
						boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
						float over = ObjectProcessing.volume(and,nx,ny,nz);
										
						refinseg[n] = (over/vol2);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
//					float vol1 = ObjectProcessing.volume(obj1,nx,ny,nz);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
					float vol2 = ObjectProcessing.volume(obj2,nx,ny,nz);
					boolean[][][] and = ObjectProcessing.binaryOperation(obj1,obj2,ObjectProcessing.AND,nx,ny,nz);
					float over = ObjectProcessing.volume(and,nx,ny,nz);
					
					refinseg[Nlabels+nn] = (over/vol2);
				}
				line = "Containment_index(ref_in_seg)"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+refinseg[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Average_surface_distance")) {
				// compare: requires same labels (use the reference as basis)
				float[] dist = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						
						dist[n] = ObjectProcessing.averageSurfaceDistance(obj1,obj2,rx,ry,rz,nx,ny,nz);
						
						System.out.println("asd "+n+": "+dist[n]);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
					dist[Nlabels+nn] = ObjectProcessing.averageSurfaceDistance(obj1,obj2,rx,ry,rz,nx,ny,nz);
					
				}
				line = "Average_surface_distance"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+dist[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Average_surface_difference")) {
				// compare: requires same labels (use the reference as basis)
				float[] dist = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						
						dist[n] = ObjectProcessing.averageSurfaceDifference(obj1,obj2,rx,ry,rz,nx,ny,nz);
//						System.out.println("asd "+n+": "+dist[n]);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
					dist[Nlabels+nn] = ObjectProcessing.averageSurfaceDifference(obj1,obj2,rx,ry,rz,nx,ny,nz);
				}
				line = "Average_surface_difference"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+dist[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Average_squared_surface_distance")) {
				// compare: requires same labels (use the reference as basis)
				float[] dist = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						
						dist[n] = ObjectProcessing.averageSquaredSurfaceDistance(obj1,obj2,rx,ry,rz,nx,ny,nz);
						
						System.out.println("assd "+n+": "+dist[n]);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
					dist[Nlabels+nn] = ObjectProcessing.averageSquaredSurfaceDistance(obj1,obj2,rx,ry,rz,nx,ny,nz);
				}
				line = "Average_squared_surface_distance"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+dist[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Hausdorff_distance")) {
				// compare: requires same labels (use the reference as basis)
				float[] hausdorff = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
//					for (int m=0;m<Nlabels;m++) if (imgLabels[m]==refLabels[n]) {
						boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
						
						hausdorff[n] = ObjectProcessing.hausdorffDistance(obj1,obj2,rx,ry,rz,nx,ny,nz);
						
//						System.out.println("hd "+n+": "+hausdorff[n]);
//					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj1 = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					boolean[][][] obj2 = ObjectProcessing.objectFromLabelImage(reference, nx, ny, nz, groups[nn]);
					hausdorff[Nlabels+nn] = ObjectProcessing.hausdorffDistance(obj1,obj2,rx,ry,rz,nx,ny,nz);
				}
				line = "Hausdorff_distance"+imgtag+reftag+nointtag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+hausdorff[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Sum_intensity")) {
				float[] sum = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
					sum[n] = 0.0f;
				}
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					for (int n=0;n<Nlabels;n++) if ( image[x][y][z]==allLabels[n])
						sum[n] += intensity[x][y][z];
					
				}
				
				// DO GROUPS
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
						if ( obj[x][y][z])
							sum[Nlabels+nn] += intensity[x][y][z];
						
					}
				}
				line = "Sum_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+sum[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Median_intensity")) {
				float[] med = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
					med[n] = 0.0f;
				}
				ArrayList<Float > intensities = new ArrayList<Float>();
				for (int n=0;n<Nlabels;n++){
					intensities.clear();
					for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++){ 
						if ( image[x][y][z]==allLabels[n]) 
							intensities.add(intensity[x][y][z]);
					}
					Collections.sort(intensities);
					if (intensities.size()%2==0)
						med[n] = 0.5f*(intensities.get(intensities.size()/2) + intensities.get(intensities.size()/2 - 1));
					else
						med[n]= intensities.get(intensities.size()/2);
						
				}
				
				// DO GROUPS
				for(int nn=0; nn<Ngroups; nn++){
					intensities.clear();
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++){ 
						if (obj[x][y][z]) 
							intensities.add(intensity[x][y][z]);
					}
					Collections.sort(intensities);
					if (intensities.size()%2==0)
						med[Nlabels+nn] = 0.5f*(intensities.get(intensities.size()/2) + intensities.get(intensities.size()/2 - 1));
					else
						med[Nlabels+nn]= intensities.get(intensities.size()/2);
				}
				line = "Median_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+med[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Mean_intensity")) {
				float[] mean = new float[Nlabels+Ngroups];
				float[] den = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
					mean[n] = 0.0f;
					den[n] = 0.0f;
				}
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					for (int n=0;n<Nlabels;n++) if (image[x][y][z]==allLabels[n]) {
						mean[n] += intensity[x][y][z];
						den[n] += 1.0f;
					}
				}
				for (int n=0;n<Nlabels;n++) {
					if (den[n]>0) mean[n] = mean[n]/den[n];
				}
				
				// DO GROUPS
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
						if (obj[x][y][z]) {
							mean[Nlabels+nn] += intensity[x][y][z];
							den[Nlabels+nn] += 1.0f;
						}
					}
				}
				for(int nn=0; nn<Ngroups; nn++){
					if (den[Nlabels+nn]>0) mean[Nlabels+nn] = mean[Nlabels+nn]/den[Nlabels+nn];
				}
				
				line = "Mean_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+mean[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Min_intensity")) {
				float[] min = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
					min[n] = INF;
				}
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					for (int n=0;n<Nlabels;n++) if (image[x][y][z]==allLabels[n]) {
						if (intensity[x][y][z]<min[n]) min[n] = intensity[x][y][z];
					}
				}
				
				// DO GROUPS
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
						if (obj[x][y][z]) {
							if (intensity[x][y][z]<min[Nlabels+nn]) min[Nlabels+nn] = intensity[x][y][z];
						}
					}
				}
				line = "Min_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+min[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Max_intensity")) {
				float[] max = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
					max[n] = -INF;
				}
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					for (int n=0;n<Nlabels;n++) if (image[x][y][z]==allLabels[n]) {
						if (intensity[x][y][z]>max[n]) max[n] = intensity[x][y][z];
					}
				}
				
				// DO GROUPS
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
						if (obj[x][y][z]) {
							if (intensity[x][y][z]>max[Nlabels+nn]) max[Nlabels+nn] = intensity[x][y][z];
						}
					}
				}
				line = "Max_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+max[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("Stdev_intensity")) {
				float[] mean = new float[Nlabels+Ngroups];
				float[] std = new float[Nlabels+Ngroups];
				float[] den = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels+Ngroups;n++) {
					mean[n] = 0.0f;
					std[n] = 0.0f;
					den[n] = 0.0f;
				}
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					for (int n=0;n<Nlabels;n++) if (image[x][y][z]==allLabels[n]) {
						mean[n] += intensity[x][y][z];
						den[n] += 1.0f;
					}
				}
				for (int n=0;n<Nlabels;n++) {
					if (den[n]>0) mean[n] = mean[n]/den[n];
				}
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					for (int n=0;n<Nlabels;n++) if (image[x][y][z]==allLabels[n]) {
						std[n] += (intensity[x][y][z]-mean[n])*(intensity[x][y][z]-mean[n]);
					}
				}
				for (int n=0;n<Nlabels;n++) {
					if (den[n]>0) std[n] = (float)Math.sqrt(std[n]/den[n]);
				}
				
				
				// DO GROUPS
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					
					for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
						if (obj[x][y][z]) {
							mean[Nlabels+nn] += intensity[x][y][z];
							den[Nlabels+nn]  += 1.0f;
						}
					}
					if (den[Nlabels+nn]>0) mean[Nlabels+nn] = mean[Nlabels+nn]/den[Nlabels+nn];
					
					for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
						if (obj[x][y][z]) {
							std[Nlabels+nn] += (intensity[x][y][z]-mean[Nlabels+nn])*(intensity[x][y][z]-mean[Nlabels+nn]);
						}
					}
					
					if (den[Nlabels+nn]>0) std[Nlabels+nn] = (float)Math.sqrt(std[Nlabels+nn]/den[Nlabels+nn]);
					
					
				}
			
				line = "Stdev_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+std[n]);
				line+=("\n");
				output.add(line);
			}			
			if (statistics.get(s).equals("50_intensity")) {
				float[] per = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					per[n] = ObjectProcessing.objectCentile(obj, intensity, 50, 100, nx, ny, nz);
				}
				
				// DO GROUPS
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					per[Nlabels+nn] = ObjectProcessing.objectCentile(obj, intensity, 50, 100, nx, ny, nz);
				}
				
				line = "50_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+per[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("90_intensity")) {
				float[] per = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					per[n] = ObjectProcessing.objectCentile(obj, intensity, 90, 100, nx, ny, nz);
				}
				
				// DO GROUPS
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					per[Nlabels+nn] = ObjectProcessing.objectCentile(obj, intensity, 90, 100, nx, ny, nz);
				}
				
				line = "90_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+per[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("10_intensity")) {
				float[] per = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					per[n] = ObjectProcessing.objectCentile(obj, intensity, 10, 100, nx, ny, nz);
				}
				
				// DO GROUPS
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					per[Nlabels+nn] = ObjectProcessing.objectCentile(obj, intensity, 10, 100, nx, ny, nz);
				}
				
				line = "10_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+per[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("75_intensity")) {
				float[] per = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					per[n] = ObjectProcessing.objectCentile(obj, intensity, 75, 100, nx, ny, nz);
				}
				
				// DO GROUPS
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					per[Nlabels+nn] = ObjectProcessing.objectCentile(obj, intensity, 75, 100, nx, ny, nz);
				}
				
				line = "75_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+per[n]);
				line+=("\n");
				output.add(line);
			}
			if (statistics.get(s).equals("25_intensity")) {
				float[] per = new float[Nlabels+Ngroups];
				for (int n=0;n<Nlabels;n++) {
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image,nx,ny,nz,allLabels[n],allLabels[n],ObjectProcessing.SUPEQUAL,ObjectProcessing.INFEQUAL);
					per[n] = ObjectProcessing.objectCentile(obj, intensity, 25, 100, nx, ny, nz);
				}
				
				// DO GROUPS
				for(int nn=0; nn<Ngroups; nn++){
					boolean[][][] obj = ObjectProcessing.objectFromLabelImage(image, nx, ny, nz, groups[nn]);
					per[Nlabels+nn] = ObjectProcessing.objectCentile(obj, intensity, 25, 100, nx, ny, nz);
				}
				
				line = "25_intensity"+imgtag+noreftag+inttag;
				for (int n=0;n<Nlabels+Ngroups;n++) line+=(delim+per[n]);
				line+=("\n");
				output.add(line);
			}
		}
		
		if (statistics.contains("Confusion_Matrix")) {
			confMtx = ObjectProcessing.confusionMatrix(image, reference, nx, ny, nz);
		}
		// write the output to standard window or file
		if (!writeToFile) {
			for (int n=0; n<output.size(); n++) {
				MedicUtilPublic.displayMessage(output.get(n));
			}
		} else {
			addStatisticsToFile(filename);
		}
		setCompleted(true);
    } // calcSegmentation
	
	public final ArrayList<String> getOutput() { return output; }
	
	public int[][] getConfusionMatrix(){
		return confMtx;
	}
	
	
	private final void addStatisticsToFile(String name) {
		// check if the statistics file exists
		
		// open the file
		ArrayList<String> 	previous = loadStatisticsFile(filename);
		
		// merge the output
		appendStatistics(previous, output);
		
		// save the result
		writeStatisticsFile(previous, name);
	}
	
	private	final void appendStatistics(ArrayList<String> main, ArrayList<String> added) {
		for (int n=0;n<added.size();n++) {
			// extract statistics type
			String type = added.get(n).substring(0,added.get(n).indexOf(delim));
			System.out.println(added.get(n));
			System.out.println(type);
			
			// find the last line with this type
			int last=-1;
			for (int m=0;m<main.size();m++) {
				if (main.get(m).indexOf(delim)>-1)
					if (main.get(m).substring(0,main.get(m).indexOf(delim)).equals(type)) last = m;
			}
			if (last>-1) {
				main.add(last+1, added.get(n));
			} else {
				// add a space after each different statistic
				main.add(added.get(n));
				main.add(" \n");
			}
		}
	}
	
	private final ArrayList<String> loadStatisticsFile(String name) {
		ArrayList<String> list = new ArrayList();
		try {
            System.out.println("reading previous statistic file: "+name);
            File f = new File(filename);
            FileReader fr = new FileReader(f);
            BufferedReader br = new BufferedReader(fr);
            String line = br.readLine();
			
			// Exact corresponding template for first line ?
            if (!line.startsWith("MIPAV Volumetric Statistics File")) {
                System.out.println("not a proper MIPAV statistics file");
                br.close();
                fr.close();
                return null;
            }
			line = br.readLine();
			while (line!=null) {
				list.add(line+"\n");
				line = br.readLine();
				System.out.println(line);
			}
			br.close();
            fr.close();
        }
        catch (FileNotFoundException e) {
            System.out.println(e.getMessage());
        }
        catch (IOException e) {
            System.out.println(e.getMessage());
        } 
		catch (OutOfMemoryError e){
			System.out.println(e.getMessage());
		}
		catch (Exception e) {
			System.out.println(e.getMessage());
        }
		return list;	
	}
	
	

	private final void writeStatisticsFile(ArrayList<String> list, String name) {
		try {
            File f = new File(filename);
            FileWriter fw = new FileWriter(f);
            PrintWriter pw = new PrintWriter( fw );
			pw.write("MIPAV Volumetric Statistics File\n");
            for (int n=0;n<list.size();n++) {
				pw.write(list.get(n));
				System.out.print(list.get(n));
			}
			pw.close();
            fw.close();
       }
        catch (FileNotFoundException e) {
            System.out.println(e.getMessage());
        }
        catch (IOException e) {
            System.out.println(e.getMessage());
        } 
		catch (OutOfMemoryError e){
			System.out.println(e.getMessage());
		}
		catch (Exception e) {
			System.out.println(e.getMessage());
        }
	}
}		

