package edu.jhmi.rad.medic.methods;

import java.io.*;
import java.util.*;
import gov.nih.mipav.view.*;
import gov.nih.mipav.model.structures.jama.*;

import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.structures.*;
import edu.jhmi.rad.medic.utilities.*;
 
/**
 *
 *  This algorithm handles registration algorithms
 *	of images with a variant of the Demons algorithm
 *  using levelset segmentations
 *
 *	@version    august 2010
 *	@author     Pierre-Louis Bazin
 *		
 *
 */
 
public class DemonsLevelsetRegistration {
		
	// numerical quantities
	private static final	float   INF=1e30f;
	private static final	float   ZERO=1e-30f;
	
	// convenience tags
	private static final	int   X=0;
	private static final	int   Y=1;
	private static final	int   Z=2;
	private static final	int   T=3;
	
	// flag labels
	public static final	int   COMPOSITIVE 		= 101;
	public static final	int   MINDIV		 	= 102;
	public static final	int   MINDIVSEP		 	= 103;
	
	public static final	int   FIXED 			= 201;
	public static final	int   MOVING 			= 202;
	public static final	int   SYMMETRIC 		= 203;
	
	public static final	int   GAUSS_FLUID 		= 301;
	public static final	int   GAUSS_DIFFUSION 	= 302;
	public static final	int   GAUSS_MIXED	 	= 303;
	public static final	int   DIV_FLUID 		= 304;
	public static final	int   DIV_DIFFUSION 	= 305;
	
	public static final int	  RAW				= 401;
	public static final int	  NORMALIZED		= 402;

	private static final byte MATCH = 1;
	private static final byte IC = 2;
	private static final byte NONE = 0;
	
	// data buffers
	private 	float[][]		image;  		// source image
	private 	float[][]	    target;			// target image
	private		float[][]		w;				// weight from target to source space			
	private		float[][]		c;				// added transform from target space to original space (multiscale)
	private		float[][]		s;				// added transform from target space to original space (multiscale)
	private		float[][]		u;				// added transform update from target space to original space (multiscale)
	private		float[][]		iw;				// weight from source to target space
	private		float[][]		ic;				// added transform from target space to original space (multiscale)
	private		float[][]		is;				// added transform from target space to original space (multiscale)
	private		float[][]		iu;				// added transform update from target space to original space (multiscale)
	private static	int		nix,niy,niz;   	// image dimensions 
	private static	int		ntx,nty,ntz;   	// target dimensions
	private static	int		nsx,nsy,nsz;   	// forward deformation dimensions
	private static	int		nisx,nisy,nisz;   	// backward deformation dimensions
	private static	float 	scale;
	private static	float	rix,riy,riz;   	// image resolutions (no pyramid)
	private static	float	rtx,rty,rtz;   	// target resolutions (no pyramid)
	private static 	int		nc;
	private		float		imin, imax;
	private		float		tmin, tmax;
	private		float[]		lb;
	private 		float[][]	transform;		// prior transform matrix (for instance from prior alignment)		
	private 		float[][]	itransform;		// prior transform inverse matrix (for instance from prior alignment)		
	private		float		nband;
	
	// parameters
	private		float		smoothingKernel; 	// smoothing kernel size
	private		float		spatialScale; 		// scale of transformation
	private		boolean		useMask;
	private		float  		maskingThreshold;			// background threshold
	private		int		regType;
	private		int		forceType;
	private		int		fieldType;
	private		int		preType;
	private		int		Niter;
	
	// computation variables
	private		float			sigma2; 		// scale of transformation
	

	private		float[][]		gaussKernel;
	private		int				gx,gy,gz;
	private		float[][]		igaussKernel;
	private		int				igx,igy,igz;
    
	// computation flags
	private 	boolean 		isWorking;
	private 	boolean 		isCompleted;
	
	// for debug and display
	static final boolean		debug=true;
	static final boolean		verbose=true;
    
	/**
	 *  constructor
     *  note: the membership function mems_ must have sum = 0 on the masked areas
	 */
	public DemonsLevelsetRegistration(float[] image_, float[] target_,
									int nix_, int niy_, int niz_,
									float rix_, float riy_, float riz_,
									int ntx_, int nty_, int ntz_,
									float rtx_, float rty_, float rtz_,
									float smoothing_,
									float spScale_,
									boolean useMask_,
									float maskVal_,
									float scale_, int Ni_, int Nt_,
									int reg_, int force_, int field_,
									int pre_,
									float[][] trans_) {
	
		scale = scale_;
		
		smoothingKernel = smoothing_;
		spatialScale = spScale_;
		sigma2 = spatialScale*spatialScale;
		
		useMask = useMask_;
		maskingThreshold = maskVal_;
		Niter = Ni_;

		nband = 10.0f;
		
		transform = trans_;
		if (transform!=null) itransform = Numerics.invert3Dtransform(transform);
		else itransform = null;
		
		nix = nix_;
		niy = niy_;
		niz = niz_;
		
		rix = rix_;
		riy = riy_;
		riz = riz_;
		
		ntx = ntx_;
		nty = nty_;
		ntz = ntz_;
		
		rtx = rtx_;
		rty = rty_;
		rtz = rtz_;
			
		nsx = Numerics.ceil(ntx/scale);
		nsy = Numerics.ceil(nty/scale);
		nsz = Numerics.ceil(ntz/scale);
		
		nisx = Numerics.ceil(nix/scale);
		nisy = Numerics.ceil(niy/scale);
		nisz = Numerics.ceil(niz/scale);
		
		regType = reg_;
		forceType = force_;
		fieldType = field_;
		preType = pre_;
		
		//initialize the smoothing kernel
		gaussKernel = ImageFunctions.separableGaussianKernel(smoothingKernel/rtx,smoothingKernel/rty,smoothingKernel/rtz);
		gx = (gaussKernel[X].length-1)/2;
		gy = (gaussKernel[Y].length-1)/2;
		gz = (gaussKernel[Z].length-1)/2;
				
		igaussKernel = ImageFunctions.separableGaussianKernel(smoothingKernel/rix,smoothingKernel/riy,smoothingKernel/riz);
		igx = (igaussKernel[X].length-1)/2;
		igy = (igaussKernel[Y].length-1)/2;
		igz = (igaussKernel[Z].length-1)/2;
				
		try {			
			preprocessInputImages(image_, target_);
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
		isWorking = true;
		if (debug) MedicUtilPublic.displayMessage("Demons:initialisation\n");
	}

	final public void finalize() {
		image = null;
		target = null;
		s = null; u = null; c = null;
		is = null; iu = null; ic = null;
		System.gc();
	}
    
   public final float[][] getCurrentTransform() {
	   return s;
   }
   public final float[][] getCurrentUpdate() {
		return u;
	}
    
   public final float[][] getCurrentInverseTransform() {
	   return is;
   }
   public final float[][] getCurrentInverseUpdate() {
		return iu;
	}
    
	public final boolean isWorking() { return isWorking; }
	public final boolean isCompleted() { return isCompleted; }
    
	
	private final void preprocessInputImages(float[] img, float[] trg) {
		if (preType==RAW) {
			// find max number of objects and list them; assume the target has proper labeling
			// then create a level set function for each object
			lb = ObjectProcessing.listLabels(trg,ntx,nty,ntz);
			nc = lb.length-1;
			
			image = new float[nc][];
			target = new float[nc][];
			
			boolean[] obj;
			
			int c=0;
			for (int n=0;n<lb.length;n++) {
				if (lb[n]!=maskingThreshold) {
					obj = ObjectProcessing.objectFromImage(img,nix,niy,niz,lb[n],ObjectProcessing.EQUAL);
					image[c] = ObjectProcessing.signedDistanceFunction(obj,nix,niy,niz);
					obj = ObjectProcessing.objectFromImage(trg,ntx,nty,ntz,lb[n],ObjectProcessing.EQUAL);
					target[c] = ObjectProcessing.signedDistanceFunction(obj,ntx,nty,ntz);
					c++;
				}			
			}
			img = null;
			trg = null;
			obj = null;
		} else if (preType==NORMALIZED) {
			nc = Numerics.round(img.length/(nix*niy*niz));
			image = new float[nc][nix*niy*niz];
			target = new float[nc][ntx*nty*ntz];
			for (int n=0;n<nc;n++) for (int xyz=0;xyz<nix*niy*niz;xyz++) {
				image[n][xyz] = img[xyz + n*nix*niy*niz];
			}
			for (int n=0;n<nc;n++) for (int xyz=0;xyz<ntx*nty*ntz;xyz++) {
				target[n][xyz] = trg[xyz + n*ntx*nty*ntz];
			}
			img = null;
			trg = null;
		}
	}

    /** initialize the weights for images */
    public final void initializeWeights() {
				
		// initialization: start from prior transform or zero
		w = new float[nc][nsx*nsy*nsz];
		for (int n=0;n<nc;n++) for (int x=0;x<nsx;x++) for (int y=0;y<nsy;y++) for (int z=0;z<nsz;z++) {
			float xt = x*scale;
			float yt = y*scale;
			float zt = z*scale;
			
			float val = ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt,ntx,nty,ntz);
			w[n][x+nsx*y+nsx*nsy*z] = (float)Math.exp(-0.5*(val*val)/(nband*nband));
		}
			
		iw = new float[nc][nisx*nisy*nisz];
		for (int n=0;n<nc;n++) for (int x=0;x<nisx;x++) for (int y=0;y<nisy;y++) for (int z=0;z<nisz;z++) {
			float xi = x*scale;
			float yi = y*scale;
			float zi = z*scale;
			
			float val = ImageFunctions.linearClosestInterpolation(image[n],xi,yi,zi,nix,niy,niz);
			iw[n][x+nisx*y+nisx*nisy*z] = (float)Math.exp(-0.5*(val*val)/(nband*nband));
		}
	}
			    
	/** initialize the transform from previous estimate, if exists */
    public final void initializeTransform() {
				
		// initialization: start from prior transform or zero
		s = new float[3][nsx*nsy*nsz];
		if (transform==null) {
			for (int x=0;x<nsx;x++) for (int y=0;y<nsy;y++) for (int z=0;z<nsz;z++) {
				int xyz = x+nsx*y+nsx*nsy*z;
				s[X][xyz] = x*scale*rtx/rix;
				s[Y][xyz] = y*scale*rty/riy;
				s[Z][xyz] = z*scale*rtz/riz;
			}
		} else {
			for (int x=0;x<nsx;x++) for (int y=0;y<nsy;y++) for (int z=0;z<nsz;z++) {
				int xyz = x+nsx*y+nsx*nsy*z;
				s[X][xyz] = transform[X][X]*x*scale + transform[X][Y]*y*scale + transform[X][Z]*z*scale + transform[X][T];
				s[Y][xyz] = transform[Y][X]*x*scale + transform[Y][Y]*y*scale + transform[Y][Z]*z*scale + transform[Y][T];
				s[Z][xyz] = transform[Z][X]*x*scale + transform[Z][Y]*y*scale + transform[Z][Z]*z*scale + transform[Z][T];
			}
		}
				
		// update always zero
		u = new float[3][nsx*nsy*nsz];
		for (int xyz=0;xyz<nsx*nsy*nsz;xyz++) {
			u[X][xyz] = 0.0f;
			u[Y][xyz] = 0.0f;
			u[Z][xyz] = 0.0f;
		}
		
		// composite update always zero
		c = new float[3][nsx*nsy*nsz];
		for (int xyz=0;xyz<nsx*nsy*nsz;xyz++) {
			c[X][xyz] = 0.0f;
			c[Y][xyz] = 0.0f;
			c[Z][xyz] = 0.0f;
		}
		
		return;
    }
    
    /** initialize the transform from previous estimate, if exists */
    public final void initializeInverseTransform() {
				
		// initialization: start from prior transform or zero
		is = new float[3][nisx*nisy*nisz];
		if (transform==null) {
			for (int x=0;x<nisx;x++) for (int y=0;y<nisy;y++) for (int z=0;z<nisz;z++) {
				int xyz = x+nisx*y+nisx*nisy*z;
				is[X][xyz] = x*scale*rix/rtx;
				is[Y][xyz] = y*scale*riy/rty;
				is[Z][xyz] = z*scale*riz/rtz;
			}
		} else {
			for (int x=0;x<nisx;x++) for (int y=0;y<nisy;y++) for (int z=0;z<nisz;z++) {
				int xyz = x+nisx*y+nisx*nisy*z;
				s[X][xyz] = itransform[X][X]*x*scale + itransform[X][Y]*y*scale + itransform[X][Z]*z*scale + itransform[X][T];
				s[Y][xyz] = itransform[Y][X]*x*scale + itransform[Y][Y]*y*scale + itransform[Y][Z]*z*scale + itransform[Y][T];
				s[Z][xyz] = itransform[Z][X]*x*scale + itransform[Z][Y]*y*scale + itransform[Z][Z]*z*scale + itransform[Z][T];
			}
		}
				
		// update always zero
		iu = new float[3][nisx*nisy*nisz];
		for (int xyz=0;xyz<nisx*nisy*nisz;xyz++) {
			iu[X][xyz] = 0.0f;
			iu[Y][xyz] = 0.0f;
			iu[Z][xyz] = 0.0f;
		}
		
		// composite update always zero
		ic = new float[3][nisx*nisy*nisz];
		for (int xyz=0;xyz<nisx*nisy*nisz;xyz++) {
			ic[X][xyz] = 0.0f;
			ic[Y][xyz] = 0.0f;
			ic[Z][xyz] = 0.0f;
		}
		
		return;
    }
    
	
    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public void computeImageToTargetEnergy(float[] stats, byte flag) {
		float meanDiff = 0.0f;
		float maxDiff = 0.0f;
		float meanIC = 0.0f;
		float maxIC = 0.0f;
		float meanDiv = 0.0f;
		float maxDiv = 0.0f;
		for (int x=1;x<nsx-1;x++) for (int y=1;y<nsy-1;y++) for (int z=1;z<nsz-1;z++) {
			int xyz = x+nsx*y+nsx*nsy*z;
		
			// compute the update field
			float xs = s[X][xyz];
			float ys = s[Y][xyz];
			float zs = s[Z][xyz];
		
			float xt = x*scale;
			float yt = y*scale;
			float zt = z*scale;
			
			float xsi = xs/scale, ysi = ys/scale, zsi = zs/scale;
			
			for (int n=0;n<nc;n++) {
				// image differences
				float diff = (ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt,ntx,nty,ntz) 
							- ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nix,niy,niz));				
				
				// inverse consistent coupling: backprojected coordinates
				float iTx = xt - ImageFunctions.linearClosestInterpolation(is[X],xsi,ysi,zsi,nisx,nisy,nisz);
				float iTy = yt - ImageFunctions.linearClosestInterpolation(is[Y],xsi,ysi,zsi,nisx,nisy,nisz);
				float iTz = zt - ImageFunctions.linearClosestInterpolation(is[Z],xsi,ysi,zsi,nisx,nisy,nisz);
				
				meanDiff += w[n][xyz]*Numerics.abs(diff);
				if (w[n][xyz]*Numerics.abs(diff)>maxDiff) maxDiff = w[n][xyz]*Numerics.abs(diff);
				/*
				meanIC += w[n][xyz]*(float)Math.sqrt(iTx*iTx + iTy*iTy + iTz*iTz);
				if (w[n][xyz]*Math.sqrt(iTx*iTx + iTy*iTy + iTz*iTz)>maxIC) 
					maxIC =  w[n][xyz]*(float)Math.sqrt(iTx*iTx + iTy*iTy + iTz*iTz);
				*/
				meanIC += (float)Math.sqrt(iTx*iTx + iTy*iTy + iTz*iTz);
				if (Math.sqrt(iTx*iTx + iTy*iTy + iTz*iTz)>maxIC) 
					maxIC =  (float)Math.sqrt(iTx*iTx + iTy*iTy + iTz*iTz);
			}
		}
		meanDiff /= (nsx*nsy*nsz*nc);
		meanIC /= (nsx*nsy*nsz*nc);
		
		if (debug) System.out.print("mean intensity difference "+meanDiff+"\n");
		if (debug) System.out.print("max. intensity difference "+maxDiff+"\n");
		if (debug) System.out.print("mean consistency difference "+meanIC+"\n");
		if (debug) System.out.print("max. consistency difference "+maxIC+"\n");

		if (flag==MATCH) {
			stats[8] = stats[0];
			stats[9] = stats[1];
			stats[0] = meanDiff;
			stats[1] = maxDiff;
		} else if (flag==IC) {
			stats[10] = stats[2];
			stats[11] = stats[3];
			stats[2] = meanIC;
			stats[3] = maxIC;
		}
        return;
    } // 

    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public float registerImageToTargetUpdate() {
		float meanDiff = 0.0f;
		float maxDiff = 0.0f;
		for (int x=1;x<nsx-1;x++) for (int y=1;y<nsy-1;y++) for (int z=1;z<nsz-1;z++) {
			int xyz = x+nsx*y+nsx*nsy*z;
		
			// compute the update field
			float xs = s[X][xyz];
			float ys = s[Y][xyz];
			float zs = s[Z][xyz];
		
			float xsmx = s[X][xyz-1];
			float ysmx = s[Y][xyz-1];
			float zsmx = s[Z][xyz-1];
	
			float xspx = s[X][xyz+1];
			float yspx = s[Y][xyz+1];
			float zspx = s[Z][xyz+1];
	
			float xsmy = s[X][xyz-nsx];
			float ysmy = s[Y][xyz-nsx];
			float zsmy = s[Z][xyz-nsx];
	
			float xspy = s[X][xyz+nsx];
			float yspy = s[Y][xyz+nsx];
			float zspy = s[Z][xyz+nsx];
	
			float xsmz = s[X][xyz-nsx*nsy];
			float ysmz = s[Y][xyz-nsx*nsy];
			float zsmz = s[Z][xyz-nsx*nsy];
	
			float xspz = s[X][xyz+nsx*nsy];
			float yspz = s[Y][xyz+nsx*nsy];
			float zspz = s[Z][xyz+nsx*nsy];
			
			float xt = x*scale;
			float yt = y*scale;
			float zt = z*scale;
			
			float xsi = xs/scale, ysi = ys/scale, zsi = zs/scale;
			float xsmxi = xsmx/scale, ysmxi = ysmx/scale, zsmxi = zsmx/scale;
			float xspxi = xspx/scale, yspxi = yspx/scale, zspxi = zspx/scale;
			float xsmyi = xsmy/scale, ysmyi = ysmy/scale, zsmyi = zsmy/scale;
			float xspyi = xspy/scale, yspyi = yspy/scale, zspyi = zspy/scale;
			float xsmzi = xsmz/scale, ysmzi = ysmz/scale, zsmzi = zsmz/scale;
			float xspzi = xspz/scale, yspzi = yspz/scale, zspzi = zspz/scale;
			
			u[X][xyz] = 0.0f;
			u[Y][xyz] = 0.0f;
			u[Z][xyz] = 0.0f;
			float den = 0.0f;
			for (int n=0;n<nc;n++) {
				// image differences
				float diff = (ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt,ntx,nty,ntz) 
							- ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nix,niy,niz));				
				
				// image gradient
				float Jx, Jy, Jz;
				if (forceType==FIXED) {
					Jx = 0.5f/rtx*(ImageFunctions.linearClosestInterpolation(target[n],xt+scale,yt,zt,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt-scale,yt,zt,ntx,nty,ntz));
					Jy = 0.5f/rty*(ImageFunctions.linearClosestInterpolation(target[n],xt,yt+scale,zt,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt,yt-scale,zt,ntx,nty,ntz));
					Jz = 0.5f/rtz*(ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt+scale,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt-scale,ntx,nty,ntz));
				} else if (forceType==MOVING) {
					Jx = 0.5f/rix*(ImageFunctions.linearClosestInterpolation(image[n],xspx,yspx,zspx,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmx,ysmx,zsmx,nix,niy,niz));
					Jy = 0.5f/riy*(ImageFunctions.linearClosestInterpolation(image[n],xspy,yspy,zspy,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmy,ysmy,zsmy,nix,niy,niz));
					Jz = 0.5f/riz*(ImageFunctions.linearClosestInterpolation(image[n],xspz,yspz,zspz,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmz,ysmz,zsmz,nix,niy,niz));
				} else {
					Jx = 0.25f/rtx*(ImageFunctions.linearClosestInterpolation(target[n],xt+scale,yt,zt,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt-scale,yt,zt,ntx,nty,ntz));
					Jy = 0.25f/rty*(ImageFunctions.linearClosestInterpolation(target[n],xt,yt+scale,zt,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt,yt-scale,zt,ntx,nty,ntz));
					Jz = 0.25f/rtz*(ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt+scale,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt-scale,ntx,nty,ntz));

					Jx += 0.25f/rix*(ImageFunctions.linearClosestInterpolation(image[n],xspx,yspx,zspx,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmx,ysmx,zsmx,nix,niy,niz));
					Jy += 0.25f/riy*(ImageFunctions.linearClosestInterpolation(image[n],xspy,yspy,zspy,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmy,ysmy,zsmy,nix,niy,niz));
					Jz += 0.25f/riz*(ImageFunctions.linearClosestInterpolation(image[n],xspz,yspz,zspz,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmz,ysmz,zsmz,nix,niy,niz));
				}
				
				// inverse consistent coupling: backprojected coordinates
				float iTx = xt - ImageFunctions.linearClosestInterpolation(is[X],xsi,ysi,zsi,nisx,nisy,nisz);
				float iTy = yt - ImageFunctions.linearClosestInterpolation(is[Y],xsi,ysi,zsi,nisx,nisy,nisz);
				float iTz = zt - ImageFunctions.linearClosestInterpolation(is[Z],xsi,ysi,zsi,nisx,nisy,nisz);
				
				// erse consistent coupling: inverse transform jacobian
				float[][] diT = new float[3][3];
				diT[X][X] = 0.5f/rix*(ImageFunctions.linearClosestInterpolation(is[X],xspxi,yspxi,zspxi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[X],xsmxi,ysmxi,zsmxi,nisx,nisy,nisz));
				diT[X][Y] = 0.5f/riy*(ImageFunctions.linearClosestInterpolation(is[X],xspyi,yspyi,zspyi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[X],xsmyi,ysmyi,zsmyi,nisx,nisy,nisz));
				diT[X][Z] = 0.5f/riz*(ImageFunctions.linearClosestInterpolation(is[X],xspzi,yspzi,zspzi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[X],xsmzi,ysmzi,zsmzi,nisx,nisy,nisz));

				diT[Y][X] = 0.5f/rix*(ImageFunctions.linearClosestInterpolation(is[Y],xspxi,yspxi,zspxi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[Y],xsmxi,ysmxi,zsmxi,nisx,nisy,nisz));
				diT[Y][Y] = 0.5f/riy*(ImageFunctions.linearClosestInterpolation(is[Y],xspyi,yspyi,zspyi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[Y],xsmyi,ysmyi,zsmyi,nisx,nisy,nisz));
				diT[Y][Z] = 0.5f/riz*(ImageFunctions.linearClosestInterpolation(is[Y],xspzi,yspzi,zspzi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[Y],xsmzi,ysmzi,zsmzi,nisx,nisy,nisz));

				diT[Z][X] = 0.5f/rix*(ImageFunctions.linearClosestInterpolation(is[Z],xspxi,yspxi,zspxi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[Z],xsmxi,ysmxi,zsmxi,nisx,nisy,nisz));
				diT[Z][Y] = 0.5f/riy*(ImageFunctions.linearClosestInterpolation(is[Z],xspyi,yspyi,zspyi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[Z],xsmyi,ysmyi,zsmyi,nisx,nisy,nisz));
				diT[Z][Z] = 0.5f/riz*(ImageFunctions.linearClosestInterpolation(is[Z],xspzi,yspzi,zspzi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[Z],xsmzi,ysmzi,zsmzi,nisx,nisy,nisz));

				// putting it all together
				float J2 = Jx*Jx+Jy*Jy+Jz*Jz;
				
				float diT2 = Numerics.determinant3D(diT);
				diT2 *= diT2;				
				
				float sigmaI = diff*diff/sigma2;
				
				den += sigmaI*(1.0f+diT2) + w[n][xyz]*J2;
				
				u[X][xyz] += w[n][xyz]*diff*Jx;
				u[Y][xyz] += w[n][xyz]*diff*Jy;
				u[Z][xyz] += w[n][xyz]*diff*Jz;
				
				u[X][xyz] += sigmaI*( iTx*diT[X][X] + iTy*diT[X][Y] + iTz*diT[X][Z] );
				u[Y][xyz] += sigmaI*( iTx*diT[Y][X] + iTy*diT[Y][Y] + iTz*diT[Y][Z] );
				u[Z][xyz] += sigmaI*( iTx*diT[Z][X] + iTy*diT[Z][Y] + iTz*diT[Z][Z] );
				
				meanDiff += w[n][xyz]*Numerics.abs(diff);
				if (w[n][xyz]*Numerics.abs(diff)>maxDiff) maxDiff = w[n][xyz]*Numerics.abs(diff);
			}
			u[X][xyz] /= Numerics.max(ZERO,den);
			u[Y][xyz] /= Numerics.max(ZERO,den);
			u[Z][xyz] /= Numerics.max(ZERO,den);			
		}
		meanDiff /= (ntx*nty*ntz);
		
		//if (debug) MedicUtilPublic.displayMessage("mean intensity difference "+meanDiff+"\n");
		//if (debug) MedicUtilPublic.displayMessage("max. intensity difference "+maxDiff+"\n");

        return meanDiff;
    } // 

    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public float registerImageToTargetUpdateDecoupled() {
		float meanDiff = 0.0f;
		float maxDiff = 0.0f;
		for (int x=1;x<nsx-1;x++) for (int y=1;y<nsy-1;y++) for (int z=1;z<nsz-1;z++) {
			int xyz = x+nsx*y+nsx*nsy*z;
		
			// compute the update field
			float xs = s[X][xyz];
			float ys = s[Y][xyz];
			float zs = s[Z][xyz];
		
			float xsmx = s[X][xyz-1];
			float ysmx = s[Y][xyz-1];
			float zsmx = s[Z][xyz-1];
	
			float xspx = s[X][xyz+1];
			float yspx = s[Y][xyz+1];
			float zspx = s[Z][xyz+1];
	
			float xsmy = s[X][xyz-nsx];
			float ysmy = s[Y][xyz-nsx];
			float zsmy = s[Z][xyz-nsx];
	
			float xspy = s[X][xyz+nsx];
			float yspy = s[Y][xyz+nsx];
			float zspy = s[Z][xyz+nsx];
	
			float xsmz = s[X][xyz-nsx*nsy];
			float ysmz = s[Y][xyz-nsx*nsy];
			float zsmz = s[Z][xyz-nsx*nsy];
	
			float xspz = s[X][xyz+nsx*nsy];
			float yspz = s[Y][xyz+nsx*nsy];
			float zspz = s[Z][xyz+nsx*nsy];
			
			float xt = x*scale;
			float yt = y*scale;
			float zt = z*scale;
						
			u[X][xyz] = 0.0f;
			u[Y][xyz] = 0.0f;
			u[Z][xyz] = 0.0f;
			float den = 0.0f;
			for (int n=0;n<nc;n++) {
				// image differences
				float diff = (ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt,ntx,nty,ntz) 
							- ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nix,niy,niz));				
				
				// image gradient
				float Jx, Jy, Jz;
				if (forceType==FIXED) {
					Jx = 0.5f/rtx*(ImageFunctions.linearClosestInterpolation(target[n],xt+scale,yt,zt,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt-scale,yt,zt,ntx,nty,ntz));
					Jy = 0.5f/rty*(ImageFunctions.linearClosestInterpolation(target[n],xt,yt+scale,zt,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt,yt-scale,zt,ntx,nty,ntz));
					Jz = 0.5f/rtz*(ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt+scale,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt-scale,ntx,nty,ntz));
				} else if (forceType==MOVING) {
					Jx = 0.5f/rix*(ImageFunctions.linearClosestInterpolation(image[n],xspx,yspx,zspx,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmx,ysmx,zsmx,nix,niy,niz));
					Jy = 0.5f/riy*(ImageFunctions.linearClosestInterpolation(image[n],xspy,yspy,zspy,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmy,ysmy,zsmy,nix,niy,niz));
					Jz = 0.5f/riz*(ImageFunctions.linearClosestInterpolation(image[n],xspz,yspz,zspz,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmz,ysmz,zsmz,nix,niy,niz));
				} else {
					Jx = 0.25f/rtx*(ImageFunctions.linearClosestInterpolation(target[n],xt+scale,yt,zt,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt-scale,yt,zt,ntx,nty,ntz));
					Jy = 0.25f/rty*(ImageFunctions.linearClosestInterpolation(target[n],xt,yt+scale,zt,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt,yt-scale,zt,ntx,nty,ntz));
					Jz = 0.25f/rtz*(ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt+scale,ntx,nty,ntz) - ImageFunctions.linearClosestInterpolation(target[n],xt,yt,zt-scale,ntx,nty,ntz));

					Jx += 0.25f/rix*(ImageFunctions.linearClosestInterpolation(image[n],xspx,yspx,zspx,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmx,ysmx,zsmx,nix,niy,niz));
					Jy += 0.25f/riy*(ImageFunctions.linearClosestInterpolation(image[n],xspy,yspy,zspy,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmy,ysmy,zsmy,nix,niy,niz));
					Jz += 0.25f/riz*(ImageFunctions.linearClosestInterpolation(image[n],xspz,yspz,zspz,nix,niy,niz)-ImageFunctions.linearClosestInterpolation(image[n],xsmz,ysmz,zsmz,nix,niy,niz));
				}
				
				// putting it all together
				float J2 = Jx*Jx+Jy*Jy+Jz*Jz;
				
				den += diff*diff/sigma2 + w[n][xyz]*J2;
				
				u[X][xyz] += w[n][xyz]*diff*Jx;
				u[Y][xyz] += w[n][xyz]*diff*Jy;
				u[Z][xyz] += w[n][xyz]*diff*Jz;
				
				meanDiff += w[n][xyz]*Numerics.abs(diff);
				if (w[n][xyz]*Numerics.abs(diff)>maxDiff) maxDiff = w[n][xyz]*Numerics.abs(diff);
			}
			u[X][xyz] /= Numerics.max(ZERO,den);
			u[Y][xyz] /= Numerics.max(ZERO,den);
			u[Z][xyz] /= Numerics.max(ZERO,den);			
		}
		meanDiff /= (ntx*nty*ntz);
		
		//if (debug) MedicUtilPublic.displayMessage("mean intensity difference "+meanDiff+"\n");
		//if (debug) MedicUtilPublic.displayMessage("max. intensity difference "+maxDiff+"\n");

        return meanDiff;
    } // 

    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public float registerImageToTargetDivergence(boolean reinit) {
		float meanDiv = 0.0f;
		float maxDiv = 0.0f;
		for (int x=2;x<nsx-2;x++) for (int y=2;y<nsy-2;y++) for (int z=2;z<nsz-2;z++) {
			int xyz = x+nsx*y+nsx*nsy*z;
		
			// compute the update field
			float xs = s[X][xyz];
			float ys = s[Y][xyz];
			float zs = s[Z][xyz];
		
			float xsmx = s[X][xyz-1];
			float xspx = s[X][xyz+1];
			float ysmy = s[Y][xyz-nsx];
			float yspy = s[Y][xyz+nsx];
			float zsmz = s[Z][xyz-nsx*nsy];
			float zspz = s[Z][xyz+nsx*nsy];
					
			float xsmxmy = s[X][xyz-1-nsx];
			float xsmxpy = s[X][xyz-1+nsx];
			float xspxmy = s[X][xyz+1-nsx];
			float xspxpy = s[X][xyz+1+nsx];
			
			float xsmxmz = s[X][xyz-1-nsx*nsy];
			float xsmxpz = s[X][xyz-1+nsx*nsy];
			float xspxmz = s[X][xyz+1-nsx*nsy];
			float xspxpz = s[X][xyz+1+nsx*nsy];
			
			float ysmxmy = s[Y][xyz-1-nsx];
			float ysmxpy = s[Y][xyz-1+nsx];
			float yspxmy = s[Y][xyz+1-nsx];
			float yspxpy = s[Y][xyz+1+nsx];
			
			float ysmymz = s[Y][xyz-nsx-nsx*nsy];
			float ysmypz = s[Y][xyz-nsx+nsx*nsy];
			float yspymz = s[Y][xyz+nsx-nsx*nsy];
			float yspypz = s[Y][xyz+nsx+nsx*nsy];
			
			float zsmxmz = s[Z][xyz-1-nsx*nsy];
			float zsmxpz = s[Z][xyz-1+nsx*nsy];
			float zspxmz = s[Z][xyz+1-nsx*nsy];
			float zspxpz = s[Z][xyz+1+nsx*nsy];
			
			float zsmymz = s[Z][xyz-nsx-nsx*nsy];
			float zsmypz = s[Z][xyz-nsx+nsx*nsy];
			float zspymz = s[Z][xyz+nsx-nsx*nsy];
			float zspypz = s[Z][xyz+nsx+nsx*nsy];
				
			if (reinit) {
				u[X][xyz] = 0.0f;
				u[Y][xyz] = 0.0f;
				u[Z][xyz] = 0.0f;
			}
			
			// transform divergence
			float div = 3.0f*scale - 0.5f*(xspx-xsmx + yspy-ysmy + zspz-zsmz);
				
			// divergence gradient
			float Dx, Dy, Dz;
			/*
			Dx = 0.5f*(xspx-2.0f*xs+xsmx) + 0.25f*(yspxpy+ysmxmy-yspxmy-ysmxpy) + 0.25f*(zspxpz+zsmxmz-zspxmz-zsmxpz); 
			Dy = 0.25f*(xspxpy+xsmxmy-xspxmy-xsmxpy) + 0.5f*(yspy-2.0f*ys+ysmy) + 0.25f*(zspypz+zsmymz-zspymz-zsmypz); 
			Dz = 0.25f*(xspxpz+xsmxmz-xspxmz-xsmxpz) + 0.25f*(yspypz+ysmymz-yspymz-ysmypz) + 0.5f*(zspz-2.0f*zs+zsmz); 
			*/
			Dx = (xspx-2.0f*xs+xsmx) + 0.25f*(yspxpy+ysmxmy-yspxmy-ysmxpy) + 0.25f*(zspxpz+zsmxmz-zspxmz-zsmxpz);
			Dy = 0.25f*(xspxpy+xsmxmy-xspxmy-xsmxpy) + (yspy-2.0f*ys+ysmy) + 0.25f*(zspypz+zsmymz-zspymz-zsmypz);
			Dz = 0.25f*(xspxpz+xsmxmz-xspxmz-xsmxpz) + 0.25f*(yspypz+ysmymz-yspymz-ysmypz) + (zspz-2.0f*zs+zsmz);			
						
			// putting it all together
			float D2 = Dx*Dx+Dy*Dy+Dz*Dz;
			
			float den = Numerics.max(ZERO, div*div + D2);
			
			u[X][xyz] += -0.001f*div*Dx/den;
			u[Y][xyz] += -0.001f*div*Dy/den;
			u[Z][xyz] += -0.001f*div*Dz/den;
			
			meanDiv += Numerics.abs(div);
			if (Numerics.abs(div)>maxDiv) maxDiv = Numerics.abs(div);
		}
		meanDiv /= ((ntx-2)*(nty-2)*(ntz-2));
		
		//if (debug) MedicUtilPublic.displayMessage("mean divergence difference "+meanDiv+"\n");
		//if (debug) MedicUtilPublic.displayMessage("max. divergence difference "+maxDiv+"\n");

        return meanDiv;
    } // 

    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public float registerImageToTargetCompose() {		
		if ( (regType==GAUSS_FLUID || regType==GAUSS_MIXED) && (smoothingKernel>0) ) {
			//if (debug) MedicUtilPublic.displayMessage("GAUSS_FLUID regularization \n");
		
			// smooth the result with a gaussian kernel
			u[X] = ImageFunctions.separableConvolution(u[X],nsx,nsy,nsz,gaussKernel,gx,gy,gz);
			u[Y] = ImageFunctions.separableConvolution(u[Y],nsx,nsy,nsz,gaussKernel,gx,gy,gz);
			u[Z] = ImageFunctions.separableConvolution(u[Z],nsx,nsy,nsz,gaussKernel,gx,gy,gz);
		} else if (regType==DIV_FLUID && smoothingKernel>0) {
			//u = ImageFunctions.rotational3D(u,nsx,nsy,nsz);	
			u = ImageFunctions.reduceDivergence(u,nsx,nsy,nsz,smoothingKernel);
			//System.out.print(displayDivergence(u,nsx,nsy,nsz));
		}
		
		// always compose the transformations
		//if (fieldType==COMPOSITIVE) {
			//if (debug) MedicUtilPublic.displayMessage("compose with current transform \n");
							
			for (int x=0;x<nsx;x++) for (int y=0;y<nsy;y++) for (int z=0;z<nsz;z++) {
				int xyz = x+nsx*y+nsx*nsy*z;
				
				float xu = x+u[X][xyz];
				float yu = y+u[Y][xyz];
				float zu = z+u[Z][xyz];
				
				// note: if outside, extrapolate as X+u
				c[X][xyz] = ImageFunctions.linearClosestInterpolation(s[X],xu,yu,zu,nsx,nsy,nsz) - x*scale;
				c[Y][xyz] = ImageFunctions.linearClosestInterpolation(s[Y],xu,yu,zu,nsx,nsy,nsz) - y*scale;
				c[Z][xyz] = ImageFunctions.linearClosestInterpolation(s[Z],xu,yu,zu,nsx,nsy,nsz) - z*scale;
			}
		//}
		
		if ( (regType==GAUSS_DIFFUSION || regType==GAUSS_MIXED) && (smoothingKernel>0) ) {
			//if (debug) MedicUtilPublic.displayMessage("GAUSS_DIFFUSION regularization \n");
			
			// smooth the result with a gaussian kernel
			c[X] = ImageFunctions.separableConvolution(c[X],nsx,nsy,nsz,gaussKernel,gx,gy,gz);
			c[Y] = ImageFunctions.separableConvolution(c[Y],nsx,nsy,nsz,gaussKernel,gx,gy,gz);
			c[Z] = ImageFunctions.separableConvolution(c[Z],nsx,nsy,nsz,gaussKernel,gx,gy,gz);
		} else if (regType==DIV_DIFFUSION && smoothingKernel>0) {
			//c = ImageFunctions.rotational3D(c,nsx,nsy,nsz);	
			c = ImageFunctions.reduceDivergence(c,nsx,nsy,nsz,smoothingKernel);	
			//System.out.print(displayDivergence(c,nsx,nsy,nsz));
		}
					
		float meanC = 0.0f;
		for (int x=0;x<nsx;x++) for (int y=0;y<nsy;y++) for (int z=0;z<nsz;z++) {
			int xyz = x+nsx*y+nsx*nsy*z;
			
			s[X][xyz] = x*scale + c[X][xyz];
			s[Y][xyz] = y*scale + c[Y][xyz];
			s[Z][xyz] = z*scale + c[Z][xyz];
			
			meanC += c[X][xyz]*c[X][xyz]+c[Y][xyz]*c[Y][xyz]+c[Z][xyz]*c[Z][xyz];
		}
		meanC /= (nsx*nsy*nsz);
		
		//if (debug) MedicUtilPublic.displayMessage("mean update size "+meanC+"\n");

        return meanC;
    } // 
    
    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public void computeTargetToImageEnergy(float[] stats, byte flag) {
		float meanDiff = 0.0f;
		float maxDiff = 0.0f;
		float meanIC = 0.0f;
		float maxIC = 0.0f;
		for (int x=1;x<nisx-1;x++) for (int y=1;y<nisy-1;y++) for (int z=1;z<nisz-1;z++) {
			int xyz = x+nisx*y+nisx*nisy*z;
		
			// compute the update field
			float xs = is[X][xyz];
			float ys = is[Y][xyz];
			float zs = is[Z][xyz];
		
			float xsi = xs/scale, ysi = ys/scale, zsi = zs/scale;
			
			float xt = x*scale;
			float yt = y*scale;
			float zt = z*scale;
			
			for (int n=0;n<nc;n++) {
				// image differences
				float diff = (ImageFunctions.linearClosestInterpolation(image[n],xt,yt,zt,nix,niy,niz) 
							- ImageFunctions.linearClosestInterpolation(target[n],xs,ys,zs,ntx,nty,ntz));				
				
				// inverse consistent coupling: backprojected coordinates
				float iTx = xt - ImageFunctions.linearClosestInterpolation(s[X],xsi,ysi,zsi,nsx,nsy,nsz);
				float iTy = yt - ImageFunctions.linearClosestInterpolation(s[Y],xsi,ysi,zsi,nsx,nsy,nsz);
				float iTz = zt - ImageFunctions.linearClosestInterpolation(s[Z],xsi,ysi,zsi,nsx,nsy,nsz);
				
				meanDiff += iw[n][xyz]*Numerics.abs(diff);
				if (iw[n][xyz]*Numerics.abs(diff)>maxDiff) maxDiff = iw[n][xyz]*Numerics.abs(diff);
				/*
				meanIC += iw[n][xyz]*(float)Math.sqrt(iTx*iTx + iTy*iTy + iTz*iTz);
				if (iw[n][xyz]*Math.sqrt(iTx*iTx + iTy*iTy + iTz*iTz)>maxIC) 
					maxIC = iw[n][xyz]*(float)Math.sqrt(iTx*iTx + iTy*iTy + iTz*iTz);
				*/
				meanIC += (float)Math.sqrt(iTx*iTx + iTy*iTy + iTz*iTz);
				if (Math.sqrt(iTx*iTx + iTy*iTy + iTz*iTz)>maxIC) 
					maxIC = (float)Math.sqrt(iTx*iTx + iTy*iTy + iTz*iTz);
			}
		}
		meanDiff /= (nisx*nisy*nisz*nc);
		meanIC /= (nisx*nisy*nisz*nc);

		if (debug) System.out.print("mean intensity difference "+meanDiff+"\n");
		if (debug) System.out.print("max. intensity difference "+maxDiff+"\n");
		if (debug) System.out.print("mean consistency difference "+meanIC+"\n");
		if (debug) System.out.print("max. consistency difference "+maxIC+"\n");

		if (flag==MATCH) {
			stats[12] = stats[4];
			stats[13] = stats[5];
			stats[4] = meanDiff;
			stats[5] = maxDiff;
		} else if (flag==IC) {
			stats[14] = stats[6];
			stats[15] = stats[7];
			stats[6] = meanIC;
			stats[7] = maxIC;
		}
        return;
    } // 
		
    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public float registerTargetToImageUpdate() {
		float meanDiff = 0.0f;
		float maxDiff = 0.0f;
		for (int x=1;x<nisx-1;x++) for (int y=1;y<nisy-1;y++) for (int z=1;z<nisz-1;z++) {
			int xyz = x+nisx*y+nisx*nisy*z;
		
			// compute the update field
			float xs = is[X][xyz];
			float ys = is[Y][xyz];
			float zs = is[Z][xyz];
		
			float xsmx = is[X][xyz-1];
			float ysmx = is[Y][xyz-1];
			float zsmx = is[Z][xyz-1];
	
			float xspx = is[X][xyz+1];
			float yspx = is[Y][xyz+1];
			float zspx = is[Z][xyz+1];
	
			float xsmy = is[X][xyz-nisx];
			float ysmy = is[Y][xyz-nisx];
			float zsmy = is[Z][xyz-nisx];
	
			float xspy = is[X][xyz+nisx];
			float yspy = is[Y][xyz+nisx];
			float zspy = is[Z][xyz+nisx];
	
			float xsmz = is[X][xyz-nisx*nisy];
			float ysmz = is[Y][xyz-nisx*nisy];
			float zsmz = is[Z][xyz-nisx*nisy];
	
			float xspz = is[X][xyz+nisx*nisy];
			float yspz = is[Y][xyz+nisx*nisy];
			float zspz = is[Z][xyz+nisx*nisy];
			
			float xsi = xs/scale, ysi = ys/scale, zsi = zs/scale;
			float xsmxi = xsmx/scale, ysmxi = ysmx/scale, zsmxi = zsmx/scale;
			float xspxi = xspx/scale, yspxi = yspx/scale, zspxi = zspx/scale;
			float xsmyi = xsmy/scale, ysmyi = ysmy/scale, zsmyi = zsmy/scale;
			float xspyi = xspy/scale, yspyi = yspy/scale, zspyi = zspy/scale;
			float xsmzi = xsmz/scale, ysmzi = ysmz/scale, zsmzi = zsmz/scale;
			float xspzi = xspz/scale, yspzi = yspz/scale, zspzi = zspz/scale;
			
			float xt = x*scale;
			float yt = y*scale;
			float zt = z*scale;
			
			iu[X][xyz] = 0.0f;
			iu[Y][xyz] = 0.0f;
			iu[Z][xyz] = 0.0f;
			float den = 0.0f;
			for (int n=0;n<nc;n++) {
				// image differences
				float diff = (ImageFunctions.linearClosestInterpolation(image[n],xt,yt,zt,nix,niy,niz) 
							- ImageFunctions.linearClosestInterpolation(target[n],xs,ys,zs,ntx,nty,ntz));				
				
				// image gradient
				float Jx, Jy, Jz;
				if (forceType==FIXED) {
					Jx = 0.5f/rix*(ImageFunctions.linearClosestInterpolation(image[n],xt+scale,yt,zt,nix,niy,niz) - ImageFunctions.linearClosestInterpolation(image[n],xt-scale,yt,zt,nix,niy,niz));
					Jy = 0.5f/riy*(ImageFunctions.linearClosestInterpolation(image[n],xt,yt+scale,zt,nix,niy,niz) - ImageFunctions.linearClosestInterpolation(image[n],xt,yt-scale,zt,nix,niy,niz));
					Jz = 0.5f/riz*(ImageFunctions.linearClosestInterpolation(image[n],xt,yt,zt+scale,nix,niy,niz) - ImageFunctions.linearClosestInterpolation(image[n],xt,yt,zt-scale,nix,niy,niz));
				} else if (forceType==MOVING) {
					Jx = 0.5f/rtx*(ImageFunctions.linearClosestInterpolation(target[n],xspx,yspx,zspx,ntx,nty,ntz)-ImageFunctions.linearClosestInterpolation(target[n],xsmx,ysmx,zsmx,ntx,nty,ntz));
					Jy = 0.5f/rty*(ImageFunctions.linearClosestInterpolation(target[n],xspy,yspy,zspy,ntx,nty,ntz)-ImageFunctions.linearClosestInterpolation(target[n],xsmy,ysmy,zsmy,ntx,nty,ntz));
					Jz = 0.5f/rtz*(ImageFunctions.linearClosestInterpolation(target[n],xspz,yspz,zspz,ntx,nty,ntz)-ImageFunctions.linearClosestInterpolation(target[n],xsmz,ysmz,zsmz,ntx,nty,ntz));
				} else {
					Jx = 0.25f/rix*(ImageFunctions.linearClosestInterpolation(image[n],xt+scale,yt,zt,nix,niy,niz) - ImageFunctions.linearClosestInterpolation(image[n],xt-scale,yt,zt,nix,niy,niz));
					Jy = 0.25f/riy*(ImageFunctions.linearClosestInterpolation(image[n],xt,yt+scale,zt,nix,niy,niz) - ImageFunctions.linearClosestInterpolation(image[n],xt,yt-scale,zt,nix,niy,niz));
					Jz = 0.25f/riz*(ImageFunctions.linearClosestInterpolation(image[n],xt,yt,zt+scale,nix,niy,niz) - ImageFunctions.linearClosestInterpolation(image[n],xt,yt,zt-scale,nix,niy,niz));
					
					Jx += 0.25f/rtx*(ImageFunctions.linearClosestInterpolation(target[n],xspx,yspx,zspx,ntx,nty,ntz)-ImageFunctions.linearClosestInterpolation(target[n],xsmx,ysmx,zsmx,ntx,nty,ntz));
					Jy += 0.25f/rty*(ImageFunctions.linearClosestInterpolation(target[n],xspy,yspy,zspy,ntx,nty,ntz)-ImageFunctions.linearClosestInterpolation(target[n],xsmy,ysmy,zsmy,ntx,nty,ntz));
					Jz += 0.25f/rtz*(ImageFunctions.linearClosestInterpolation(target[n],xspz,yspz,zspz,ntx,nty,ntz)-ImageFunctions.linearClosestInterpolation(target[n],xsmz,ysmz,zsmz,ntx,nty,ntz));
				}
			
				// inverse consistent coupling: backprojected coordinates
				float iTx = xt - ImageFunctions.linearClosestInterpolation(s[X],xsi,ysi,zsi,nsx,nsy,nsz);
				float iTy = yt - ImageFunctions.linearClosestInterpolation(s[Y],xsi,ysi,zsi,nsx,nsy,nsz);
				float iTz = zt - ImageFunctions.linearClosestInterpolation(s[Z],xsi,ysi,zsi,nsx,nsy,nsz);
				
				// erse consistent coupling: inverse transform jacobian
				float[][] diT = new float[3][3];
				diT[X][X] = 0.5f/rtx*(ImageFunctions.linearClosestInterpolation(s[X],xspxi,yspxi,zspxi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[X],xsmxi,ysmxi,zsmxi,nsx,nsy,nsz));
				diT[X][Y] = 0.5f/rty*(ImageFunctions.linearClosestInterpolation(s[X],xspyi,yspyi,zspyi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[X],xsmyi,ysmyi,zsmyi,nsx,nsy,nsz));
				diT[X][Z] = 0.5f/rtz*(ImageFunctions.linearClosestInterpolation(s[X],xspzi,yspzi,zspzi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[X],xsmzi,ysmzi,zsmzi,nsx,nsy,nsz));

				diT[Y][X] = 0.5f/rtx*(ImageFunctions.linearClosestInterpolation(s[Y],xspxi,yspxi,zspxi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[Y],xsmxi,ysmxi,zsmxi,nsx,nsy,nsz));
				diT[Y][Y] = 0.5f/rty*(ImageFunctions.linearClosestInterpolation(s[Y],xspyi,yspyi,zspyi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[Y],xsmyi,ysmyi,zsmyi,nsx,nsy,nsz));
				diT[Y][Z] = 0.5f/rtz*(ImageFunctions.linearClosestInterpolation(s[Y],xspzi,yspzi,zspzi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[Y],xsmzi,ysmzi,zsmzi,nsx,nsy,nsz));

				diT[Z][X] = 0.5f/rtx*(ImageFunctions.linearClosestInterpolation(s[Z],xspxi,yspxi,zspxi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[Z],xsmxi,ysmxi,zsmxi,nsx,nsy,nsz));
				diT[Z][Y] = 0.5f/rty*(ImageFunctions.linearClosestInterpolation(s[Z],xspyi,yspyi,zspyi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[Z],xsmyi,ysmyi,zsmyi,nsx,nsy,nsz));
				diT[Z][Z] = 0.5f/rtz*(ImageFunctions.linearClosestInterpolation(s[Z],xspzi,yspzi,zspzi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[Z],xsmzi,ysmzi,zsmzi,nsx,nsy,nsz));
				
				// putting all together
				float J2 = Jx*Jx+Jy*Jy+Jz*Jz;
				
				float diT2 = Numerics.determinant3D(diT);
				diT2 *= diT2;				
				
				float sigmaI = diff*diff/sigma2;
				
				den += sigmaI*(1.0f+diT2) + iw[n][xyz]*J2;
				
				iu[X][xyz] += iw[n][xyz]*diff*Jx;
				iu[Y][xyz] += iw[n][xyz]*diff*Jy;
				iu[Z][xyz] += iw[n][xyz]*diff*Jz;
				
				iu[X][xyz] += sigmaI*( iTx*diT[X][X] + iTy*diT[X][Y] + iTz*diT[X][Z] );
				iu[Y][xyz] += sigmaI*( iTx*diT[Y][X] + iTy*diT[Y][Y] + iTz*diT[Y][Z] );
				iu[Z][xyz] += sigmaI*( iTx*diT[Z][X] + iTy*diT[Z][Y] + iTz*diT[Z][Z] );
				
				meanDiff += iw[n][xyz]*Numerics.abs(diff);
				if (iw[n][xyz]*Numerics.abs(diff)>maxDiff) maxDiff = iw[n][xyz]*Numerics.abs(diff);
			}
			iu[X][xyz] /= Numerics.max(ZERO,den);
			iu[Y][xyz] /= Numerics.max(ZERO,den);
			iu[Z][xyz] /= Numerics.max(ZERO,den);			
		}
		meanDiff /= (nix*niy*niz);

		//if (debug) MedicUtilPublic.displayMessage("mean intensity difference "+meanDiff+"\n");
		//if (debug) MedicUtilPublic.displayMessage("max. intensity difference "+maxDiff+"\n");

        return meanDiff;
    } // 
		
    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public float registerTargetToImageUpdateDecoupled() {
		
		float meanDiff = 0.0f;
		float maxDiff = 0.0f;
		for (int x=1;x<nisx-1;x++) for (int y=1;y<nisy-1;y++) for (int z=1;z<nisz-1;z++) {
			int xyz = x+nisx*y+nisx*nisy*z;
		
			// compute the update field
			float xs = is[X][xyz];
			float ys = is[Y][xyz];
			float zs = is[Z][xyz];
		
			float xsmx = is[X][xyz-1];
			float ysmx = is[Y][xyz-1];
			float zsmx = is[Z][xyz-1];
	
			float xspx = is[X][xyz+1];
			float yspx = is[Y][xyz+1];
			float zspx = is[Z][xyz+1];
	
			float xsmy = is[X][xyz-nisx];
			float ysmy = is[Y][xyz-nisx];
			float zsmy = is[Z][xyz-nisx];
	
			float xspy = is[X][xyz+nisx];
			float yspy = is[Y][xyz+nisx];
			float zspy = is[Z][xyz+nisx];
	
			float xsmz = is[X][xyz-nisx*nisy];
			float ysmz = is[Y][xyz-nisx*nisy];
			float zsmz = is[Z][xyz-nisx*nisy];
	
			float xspz = is[X][xyz+nisx*nisy];
			float yspz = is[Y][xyz+nisx*nisy];
			float zspz = is[Z][xyz+nisx*nisy];
			
			float xt = x*scale;
			float yt = y*scale;
			float zt = z*scale;
			
			iu[X][xyz] = 0.0f;
			iu[Y][xyz] = 0.0f;
			iu[Z][xyz] = 0.0f;
			float den = 0.0f;
			for (int n=0;n<nc;n++) {
				// image differences
				float diff = (ImageFunctions.linearClosestInterpolation(image[n],xt,yt,zt,nix,niy,niz) 
							- ImageFunctions.linearClosestInterpolation(target[n],xs,ys,zs,ntx,nty,ntz));				
				
				// image gradient
				float Jx, Jy, Jz;
				if (forceType==FIXED) {
					Jx = 0.5f/rix*(ImageFunctions.linearClosestInterpolation(image[n],xt+scale,yt,zt,nix,niy,niz) - ImageFunctions.linearClosestInterpolation(image[n],xt-scale,yt,zt,nix,niy,niz));
					Jy = 0.5f/riy*(ImageFunctions.linearClosestInterpolation(image[n],xt,yt+scale,zt,nix,niy,niz) - ImageFunctions.linearClosestInterpolation(image[n],xt,yt-scale,zt,nix,niy,niz));
					Jz = 0.5f/riz*(ImageFunctions.linearClosestInterpolation(image[n],xt,yt,zt+scale,nix,niy,niz) - ImageFunctions.linearClosestInterpolation(image[n],xt,yt,zt-scale,nix,niy,niz));
				} else if (forceType==MOVING) {
					Jx = 0.5f/rtx*(ImageFunctions.linearClosestInterpolation(target[n],xspx,yspx,zspx,ntx,nty,ntz)-ImageFunctions.linearClosestInterpolation(target[n],xsmx,ysmx,zsmx,ntx,nty,ntz));
					Jy = 0.5f/rty*(ImageFunctions.linearClosestInterpolation(target[n],xspy,yspy,zspy,ntx,nty,ntz)-ImageFunctions.linearClosestInterpolation(target[n],xsmy,ysmy,zsmy,ntx,nty,ntz));
					Jz = 0.5f/rtz*(ImageFunctions.linearClosestInterpolation(target[n],xspz,yspz,zspz,ntx,nty,ntz)-ImageFunctions.linearClosestInterpolation(target[n],xsmz,ysmz,zsmz,ntx,nty,ntz));
				} else {
					Jx = 0.25f/rix*(ImageFunctions.linearClosestInterpolation(image[n],xt+scale,yt,zt,nix,niy,niz) - ImageFunctions.linearClosestInterpolation(image[n],xt-scale,yt,zt,nix,niy,niz));
					Jy = 0.25f/riy*(ImageFunctions.linearClosestInterpolation(image[n],xt,yt+scale,zt,nix,niy,niz) - ImageFunctions.linearClosestInterpolation(image[n],xt,yt-scale,zt,nix,niy,niz));
					Jz = 0.25f/riz*(ImageFunctions.linearClosestInterpolation(image[n],xt,yt,zt+scale,nix,niy,niz) - ImageFunctions.linearClosestInterpolation(image[n],xt,yt,zt-scale,nix,niy,niz));
					
					Jx += 0.25f/rtx*(ImageFunctions.linearClosestInterpolation(target[n],xspx,yspx,zspx,ntx,nty,ntz)-ImageFunctions.linearClosestInterpolation(target[n],xsmx,ysmx,zsmx,ntx,nty,ntz));
					Jy += 0.25f/rty*(ImageFunctions.linearClosestInterpolation(target[n],xspy,yspy,zspy,ntx,nty,ntz)-ImageFunctions.linearClosestInterpolation(target[n],xsmy,ysmy,zsmy,ntx,nty,ntz));
					Jz += 0.25f/rtz*(ImageFunctions.linearClosestInterpolation(target[n],xspz,yspz,zspz,ntx,nty,ntz)-ImageFunctions.linearClosestInterpolation(target[n],xsmz,ysmz,zsmz,ntx,nty,ntz));
				}
			
				// putting all together
				float J2 = Jx*Jx+Jy*Jy+Jz*Jz;
				
				den += diff*diff/sigma2 + iw[n][xyz]*J2;
				
				iu[X][xyz] += iw[n][xyz]*diff*Jx;
				iu[Y][xyz] += iw[n][xyz]*diff*Jy;
				iu[Z][xyz] += iw[n][xyz]*diff*Jz;
				
				meanDiff += iw[n][xyz]*Numerics.abs(diff);
				if (iw[n][xyz]*Numerics.abs(diff)>maxDiff) maxDiff = iw[n][xyz]*Numerics.abs(diff);
			}
			iu[X][xyz] /= Numerics.max(ZERO,den);
			iu[Y][xyz] /= Numerics.max(ZERO,den);
			iu[Z][xyz] /= Numerics.max(ZERO,den);			
		}
		meanDiff /= (nix*niy*niz);

		//if (debug) MedicUtilPublic.displayMessage("mean intensity difference "+meanDiff+"\n");
		//if (debug) MedicUtilPublic.displayMessage("max. intensity difference "+maxDiff+"\n");

        return meanDiff;
    } // 
		
    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public float registerTargetToImageDivergence(boolean reinit) {		
		float meanDiv = 0.0f;
		float maxDiv = 0.0f;
		for (int x=2;x<nisx-2;x++) for (int y=2;y<nisy-2;y++) for (int z=2;z<nisz-2;z++) {
			int xyz = x+nisx*y+nisx*nisy*z;
		
			// compute the update field
			float xs = is[X][xyz];
			float ys = is[Y][xyz];
			float zs = is[Z][xyz];
		
			float xsmx = is[X][xyz-1];
			float xspx = is[X][xyz+1];
			float ysmy = is[Y][xyz-nisx];
			float yspy = is[Y][xyz+nisx];
			float zsmz = is[Z][xyz-nisx*nisy];
			float zspz = is[Z][xyz+nisx*nisy];
			
			float xsmxmy = is[X][xyz-1-nisx];
			float xsmxpy = is[X][xyz-1+nisx];
			float xspxmy = is[X][xyz+1-nisx];
			float xspxpy = is[X][xyz+1+nisx];
			
			float xsmxmz = is[X][xyz-1-nisx*nisy];
			float xsmxpz = is[X][xyz-1+nisx*nisy];
			float xspxmz = is[X][xyz+1-nisx*nisy];
			float xspxpz = is[X][xyz+1+nisx*nisy];
			
			float ysmxmy = is[Y][xyz-1-nisx];
			float ysmxpy = is[Y][xyz-1+nisx];
			float yspxmy = is[Y][xyz+1-nisx];
			float yspxpy = is[Y][xyz+1+nisx];
			
			float ysmymz = is[Y][xyz-nisx-nisx*nisy];
			float ysmypz = is[Y][xyz-nisx+nisx*nisy];
			float yspymz = is[Y][xyz+nisx-nisx*nisy];
			float yspypz = is[Y][xyz+nisx+nisx*nisy];
			
			float zsmxmz = is[Z][xyz-1-nisx*nisy];
			float zsmxpz = is[Z][xyz-1+nisx*nisy];
			float zspxmz = is[Z][xyz+1-nisx*nisy];
			float zspxpz = is[Z][xyz+1+nisx*nisy];
			
			float zsmymz = is[Z][xyz-nisx-nisx*nisy];
			float zsmypz = is[Z][xyz-nisx+nisx*nisy];
			float zspymz = is[Z][xyz+nisx-nisx*nisy];
			float zspypz = is[Z][xyz+nisx+nisx*nisy];

			if (reinit) {
				iu[X][xyz] = 0.0f;
				iu[Y][xyz] = 0.0f;
				iu[Z][xyz] = 0.0f;
			}
			// transform divergence
			float div = 3.0f*scale - 0.5f*(xspx-xsmx + yspy-ysmy + zspz-zsmz);
			
			// divergence gradient
			float Dx, Dy, Dz;
			/*
			Dx = 0.5f*(xspx-2.0f*xs+xsmx) + 0.25f*(yspxpy+ysmxmy-yspxmy-ysmxpy) + 0.25f*(zspxpz+zsmxmz-zspxmz-zsmxpz); 
			Dy = 0.25f*(xspxpy+xsmxmy-xspxmy-xsmxpy) + 0.5f*(yspy-2.0f*ys+ysmy) + 0.25f*(zspypz+zsmymz-zspymz-zsmypz); 
			Dz = 0.25f*(xspxpz+xsmxmz-xspxmz-xsmxpz) + 0.25f*(yspypz+ysmymz-yspymz-ysmypz) + 0.5f*(zspz-2.0f*zs+zsmz); 
			*/
			Dx = (xspx-2.0f*xs+xsmx) + 0.25f*(yspxpy+ysmxmy-yspxmy-ysmxpy) + 0.25f*(zspxpz+zsmxmz-zspxmz-zsmxpz);
			Dy = 0.25f*(xspxpy+xsmxmy-xspxmy-xsmxpy) + (yspy-2.0f*ys+ysmy) + 0.25f*(zspypz+zsmymz-zspymz-zsmypz);
			Dz = 0.25f*(xspxpz+xsmxmz-xspxmz-xsmxpz) + 0.25f*(yspypz+ysmymz-yspymz-ysmypz) + (zspz-2.0f*zs+zsmz);
			
			// putting all together
			float D2 = Dx*Dx+Dy*Dy+Dz*Dz;
			
			float den = Numerics.max(ZERO, div*div + D2);
			
			iu[X][xyz] += 0.1f*div*Dx/den;
			iu[Y][xyz] += 0.1f*div*Dy/den;
			iu[Z][xyz] += 0.1f*div*Dz/den;
			
			meanDiv += Numerics.abs(div);
			if (Numerics.abs(div)>maxDiv) maxDiv = Numerics.abs(div);
		}
		meanDiv /= ((ntx-2)*(nty-2)*(ntz-2));
		
		//if (debug) MedicUtilPublic.displayMessage("mean divergence difference "+meanDiv+"\n");
		//if (debug) MedicUtilPublic.displayMessage("max. divergence difference "+maxDiv+"\n");

        return meanDiv;
    } // 
    
    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public float registerTargetToImageCompose() {
		
		if ( (regType==GAUSS_FLUID || regType==GAUSS_MIXED) && (smoothingKernel>0) ) {
			//if (debug) MedicUtilPublic.displayMessage("GAUSS_FLUID regularization \n");
		
			// smooth the result with a gaussian kernel
			iu[X] = ImageFunctions.separableConvolution(iu[X],nisx,nisy,nisz,igaussKernel,igx,igy,igz);
			iu[Y] = ImageFunctions.separableConvolution(iu[Y],nisx,nisy,nisz,igaussKernel,igx,igy,igz);
			iu[Z] = ImageFunctions.separableConvolution(iu[Z],nisx,nisy,nisz,igaussKernel,igx,igy,igz);
		} else if (regType==DIV_FLUID && smoothingKernel>0) {
			//iu = ImageFunctions.rotational3D(iu,nisx,nisy,nisz);	
			iu = ImageFunctions.reduceDivergence(iu,nisx,nisy,nisz,smoothingKernel);	
			//System.out.print(displayDivergence(iu,nisx,nisy,nisz));
		}
		
		// compose the transformations
		//if (fieldType==COMPOSITIVE) {
			//if (debug) MedicUtilPublic.displayMessage("compose with current transform \n");
							
			for (int x=0;x<nisx;x++) for (int y=0;y<nisy;y++) for (int z=0;z<nisz;z++) {
				int xyz = x+nisx*y+nisx*nisy*z;
				
				float xu = x+iu[X][xyz];
				float yu = y+iu[Y][xyz];
				float zu = z+iu[Z][xyz];
				
				// note: if outside, extrapolate as X+u
				ic[X][xyz] = ImageFunctions.linearClosestInterpolation(is[X],xu,yu,zu,nisx,nisy,nisz) - x*scale;
				ic[Y][xyz] = ImageFunctions.linearClosestInterpolation(is[Y],xu,yu,zu,nisx,nisy,nisz) - y*scale;
				ic[Z][xyz] = ImageFunctions.linearClosestInterpolation(is[Z],xu,yu,zu,nisx,nisy,nisz) - z*scale;
			}
		//}
		
		if ( (regType==GAUSS_DIFFUSION || regType==GAUSS_MIXED) && (smoothingKernel>0) ) {
			//if (debug) MedicUtilPublic.displayMessage("GAUSS_DIFFUSION regularization \n");
			
			// smooth the result with a gaussian kernel
			ic[X] = ImageFunctions.separableConvolution(ic[X],nisx,nisy,nisz,igaussKernel,igx,igy,igz);
			ic[Y] = ImageFunctions.separableConvolution(ic[Y],nisx,nisy,nisz,igaussKernel,igx,igy,igz);
			ic[Z] = ImageFunctions.separableConvolution(ic[Z],nisx,nisy,nisz,igaussKernel,igx,igy,igz);
		} else if (regType==DIV_DIFFUSION && smoothingKernel>0) {
			//ic = ImageFunctions.rotational3D(ic,nisx,nisy,nisz);	
			ic = ImageFunctions.reduceDivergence(ic,nisx,nisy,nisz,smoothingKernel);	
			//System.out.print(displayDivergence(ic,nisx,nisy,nisz));
		}
					
		float meanC = 0.0f;
		for (int x=0;x<nisx;x++) for (int y=0;y<nisy;y++) for (int z=0;z<nisz;z++) {
			int xyz = x+nisx*y+nisx*nisy*z;
			
			is[X][xyz] = x*scale + ic[X][xyz];
			is[Y][xyz] = y*scale + ic[Y][xyz];
			is[Z][xyz] = z*scale + ic[Z][xyz];
			
			meanC += ic[X][xyz]*ic[X][xyz]+ic[Y][xyz]*ic[Y][xyz]+ic[Z][xyz]*ic[Z][xyz];
		}
		meanC /= (nisx*nisy*nisz);
		
		//if (debug) MedicUtilPublic.displayMessage("mean update size "+meanC+"\n");

        return meanC;
    } // 
    
   /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public float registerImageToTargetInverseConsistent() {
		float meanDiff = 0.0f;
		float maxDiff = 0.0f;
		for (int x=1;x<nsx-1;x++) for (int y=1;y<nsy-1;y++) for (int z=1;z<nsz-1;z++) {
			int xyz = x+nsx*y+nsx*nsy*z;
		
			// compute the update field
			float xs = s[X][xyz];
			float ys = s[Y][xyz];
			float zs = s[Z][xyz];
		
			float xsmx = s[X][xyz-1];
			float ysmx = s[Y][xyz-1];
			float zsmx = s[Z][xyz-1];
	
			float xspx = s[X][xyz+1];
			float yspx = s[Y][xyz+1];
			float zspx = s[Z][xyz+1];
	
			float xsmy = s[X][xyz-nsx];
			float ysmy = s[Y][xyz-nsx];
			float zsmy = s[Z][xyz-nsx];
	
			float xspy = s[X][xyz+nsx];
			float yspy = s[Y][xyz+nsx];
			float zspy = s[Z][xyz+nsx];
	
			float xsmz = s[X][xyz-nsx*nsy];
			float ysmz = s[Y][xyz-nsx*nsy];
			float zsmz = s[Z][xyz-nsx*nsy];
	
			float xspz = s[X][xyz+nsx*nsy];
			float yspz = s[Y][xyz+nsx*nsy];
			float zspz = s[Z][xyz+nsx*nsy];
			
			float xt = x*scale;
			float yt = y*scale;
			float zt = z*scale;
			
			float xsi = xs/scale, ysi = ys/scale, zsi = zs/scale;
			float xsmxi = xsmx/scale, ysmxi = ysmx/scale, zsmxi = zsmx/scale;
			float xspxi = xspx/scale, yspxi = yspx/scale, zspxi = zspx/scale;
			float xsmyi = xsmy/scale, ysmyi = ysmy/scale, zsmyi = zsmy/scale;
			float xspyi = xspy/scale, yspyi = yspy/scale, zspyi = zspy/scale;
			float xsmzi = xsmz/scale, ysmzi = ysmz/scale, zsmzi = zsmz/scale;
			float xspzi = xspz/scale, yspzi = yspz/scale, zspzi = zspz/scale;
			
			u[X][xyz] = 0.0f;
			u[Y][xyz] = 0.0f;
			u[Z][xyz] = 0.0f;
			float den = 0.0f;
			for (int n=0;n<nc;n++) {
				// inverse consistent coupling: backprojected coordinates
				float iTx = xt - ImageFunctions.linearClosestInterpolation(is[X],xsi,ysi,zsi,nisx,nisy,nisz);
				float iTy = yt - ImageFunctions.linearClosestInterpolation(is[Y],xsi,ysi,zsi,nisx,nisy,nisz);
				float iTz = zt - ImageFunctions.linearClosestInterpolation(is[Z],xsi,ysi,zsi,nisx,nisy,nisz);
				
				// inverse consistent coupling: inverse transform jacobian
				float[][] diT = new float[3][3];
				diT[X][X] = 0.5f/rix*(ImageFunctions.linearClosestInterpolation(is[X],xspxi,yspxi,zspxi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[X],xsmxi,ysmxi,zsmxi,nisx,nisy,nisz));
				diT[X][Y] = 0.5f/riy*(ImageFunctions.linearClosestInterpolation(is[X],xspyi,yspyi,zspyi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[X],xsmyi,ysmyi,zsmyi,nisx,nisy,nisz));
				diT[X][Z] = 0.5f/riz*(ImageFunctions.linearClosestInterpolation(is[X],xspzi,yspzi,zspzi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[X],xsmzi,ysmzi,zsmzi,nisx,nisy,nisz));

				diT[Y][X] = 0.5f/rix*(ImageFunctions.linearClosestInterpolation(is[Y],xspxi,yspxi,zspxi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[Y],xsmxi,ysmxi,zsmxi,nisx,nisy,nisz));
				diT[Y][Y] = 0.5f/riy*(ImageFunctions.linearClosestInterpolation(is[Y],xspyi,yspyi,zspyi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[Y],xsmyi,ysmyi,zsmyi,nisx,nisy,nisz));
				diT[Y][Z] = 0.5f/riz*(ImageFunctions.linearClosestInterpolation(is[Y],xspzi,yspzi,zspzi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[Y],xsmzi,ysmzi,zsmzi,nisx,nisy,nisz));

				diT[Z][X] = 0.5f/rix*(ImageFunctions.linearClosestInterpolation(is[Z],xspxi,yspxi,zspxi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[Z],xsmxi,ysmxi,zsmxi,nisx,nisy,nisz));
				diT[Z][Y] = 0.5f/riy*(ImageFunctions.linearClosestInterpolation(is[Z],xspyi,yspyi,zspyi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[Z],xsmyi,ysmyi,zsmyi,nisx,nisy,nisz));
				diT[Z][Z] = 0.5f/riz*(ImageFunctions.linearClosestInterpolation(is[Z],xspzi,yspzi,zspzi,nisx,nisy,nisz)-ImageFunctions.linearClosestInterpolation(is[Z],xsmzi,ysmzi,zsmzi,nisx,nisy,nisz));

				// putting it all together
				float diT2 = Numerics.determinant3D(diT);
				diT2 *= diT2;				
				
				float iT2 = iTx*iTx + iTy*iTy + iTz*iTz;
				
				den += iT2 + diT2;
				
				u[X][xyz] += 0.1f*scale*( iTx*diT[X][X] + iTy*diT[Y][X] + iTz*diT[Z][X] );
				u[Y][xyz] += 0.1f*scale*( iTx*diT[X][Y] + iTy*diT[Y][Y] + iTz*diT[Z][Y] );
				u[Z][xyz] += 0.1f*scale*( iTx*diT[X][Z] + iTy*diT[Y][Z] + iTz*diT[Z][Z] );
				
				meanDiff += (float)Math.sqrt(iT2);
				if (Math.sqrt(iT2)>maxDiff) maxDiff = (float)Math.sqrt(iT2);
			}
			u[X][xyz] /= Numerics.max(ZERO,den);
			u[Y][xyz] /= Numerics.max(ZERO,den);
			u[Z][xyz] /= Numerics.max(ZERO,den);			
		}
		meanDiff /= (ntx*nty*ntz);

		//if (debug) MedicUtilPublic.displayMessage("mean inverse consistency: "+meanDiff+"\n");
		//if (debug) MedicUtilPublic.displayMessage("max. inverse consistency: "+maxDiff+"\n");

        return meanDiff;
    } // 
    
    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public float registerTargetToImageInverseConsistent() {
		float meanDiff = 0.0f;
		float maxDiff = 0.0f;
		for (int x=1;x<nisx-1;x++) for (int y=1;y<nisy-1;y++) for (int z=1;z<nisz-1;z++) {
			int xyz = x+nisx*y+nisx*nisy*z;
		
			// compute the update field
			float xs = is[X][xyz];
			float ys = is[Y][xyz];
			float zs = is[Z][xyz];
		
			float xsmx = is[X][xyz-1];
			float ysmx = is[Y][xyz-1];
			float zsmx = is[Z][xyz-1];
	
			float xspx = is[X][xyz+1];
			float yspx = is[Y][xyz+1];
			float zspx = is[Z][xyz+1];
	
			float xsmy = is[X][xyz-nisx];
			float ysmy = is[Y][xyz-nisx];
			float zsmy = is[Z][xyz-nisx];
	
			float xspy = is[X][xyz+nisx];
			float yspy = is[Y][xyz+nisx];
			float zspy = is[Z][xyz+nisx];
	
			float xsmz = is[X][xyz-nisx*nisy];
			float ysmz = is[Y][xyz-nisx*nisy];
			float zsmz = is[Z][xyz-nisx*nisy];
	
			float xspz = is[X][xyz+nisx*nisy];
			float yspz = is[Y][xyz+nisx*nisy];
			float zspz = is[Z][xyz+nisx*nisy];
			
			float xsi = xs/scale, ysi = ys/scale, zsi = zs/scale;
			float xsmxi = xsmx/scale, ysmxi = ysmx/scale, zsmxi = zsmx/scale;
			float xspxi = xspx/scale, yspxi = yspx/scale, zspxi = zspx/scale;
			float xsmyi = xsmy/scale, ysmyi = ysmy/scale, zsmyi = zsmy/scale;
			float xspyi = xspy/scale, yspyi = yspy/scale, zspyi = zspy/scale;
			float xsmzi = xsmz/scale, ysmzi = ysmz/scale, zsmzi = zsmz/scale;
			float xspzi = xspz/scale, yspzi = yspz/scale, zspzi = zspz/scale;
			
			float xt = x*scale;
			float yt = y*scale;
			float zt = z*scale;
			
			iu[X][xyz] = 0.0f;
			iu[Y][xyz] = 0.0f;
			iu[Z][xyz] = 0.0f;
			float den = 0.0f;
			for (int n=0;n<nc;n++) {
				// inverse consistent coupling: backprojected coordinates
				float iTx = xt - ImageFunctions.linearClosestInterpolation(s[X],xsi,ysi,zsi,nsx,nsy,nsz);
				float iTy = yt - ImageFunctions.linearClosestInterpolation(s[Y],xsi,ysi,zsi,nsx,nsy,nsz);
				float iTz = zt - ImageFunctions.linearClosestInterpolation(s[Z],xsi,ysi,zsi,nsx,nsy,nsz);
				
				// erse consistent coupling: inverse transform jacobian
				float[][] diT = new float[3][3];
				diT[X][X] = 0.5f/rtx*(ImageFunctions.linearClosestInterpolation(s[X],xspxi,yspxi,zspxi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[X],xsmxi,ysmxi,zsmxi,nsx,nsy,nsz));
				diT[X][Y] = 0.5f/rty*(ImageFunctions.linearClosestInterpolation(s[X],xspyi,yspyi,zspyi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[X],xsmyi,ysmyi,zsmyi,nsx,nsy,nsz));
				diT[X][Z] = 0.5f/rtz*(ImageFunctions.linearClosestInterpolation(s[X],xspzi,yspzi,zspzi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[X],xsmzi,ysmzi,zsmzi,nsx,nsy,nsz));

				diT[Y][X] = 0.5f/rtx*(ImageFunctions.linearClosestInterpolation(s[Y],xspxi,yspxi,zspxi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[Y],xsmxi,ysmxi,zsmxi,nsx,nsy,nsz));
				diT[Y][Y] = 0.5f/rty*(ImageFunctions.linearClosestInterpolation(s[Y],xspyi,yspyi,zspyi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[Y],xsmyi,ysmyi,zsmyi,nsx,nsy,nsz));
				diT[Y][Z] = 0.5f/rtz*(ImageFunctions.linearClosestInterpolation(s[Y],xspzi,yspzi,zspzi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[Y],xsmzi,ysmzi,zsmzi,nsx,nsy,nsz));

				diT[Z][X] = 0.5f/rtx*(ImageFunctions.linearClosestInterpolation(s[Z],xspxi,yspxi,zspxi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[Z],xsmxi,ysmxi,zsmxi,nsx,nsy,nsz));
				diT[Z][Y] = 0.5f/rty*(ImageFunctions.linearClosestInterpolation(s[Z],xspyi,yspyi,zspyi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[Z],xsmyi,ysmyi,zsmyi,nsx,nsy,nsz));
				diT[Z][Z] = 0.5f/rtz*(ImageFunctions.linearClosestInterpolation(s[Z],xspzi,yspzi,zspzi,nsx,nsy,nsz)-ImageFunctions.linearClosestInterpolation(s[Z],xsmzi,ysmzi,zsmzi,nsx,nsy,nsz));
				
				// putting all together
				float diT2 = Numerics.determinant3D(diT);
				diT2 *= diT2;				
				
				float iT2 = iTx*iTx + iTy*iTy + iTz*iTz;
				
				den += iT2 + diT2;
				
				iu[X][xyz] += 0.1f*scale*( iTx*diT[X][X] + iTy*diT[Y][X] + iTz*diT[Z][X] );
				iu[Y][xyz] += 0.1f*scale*( iTx*diT[X][Y] + iTy*diT[Y][Y] + iTz*diT[Z][Y] );
				iu[Z][xyz] += 0.1f*scale*( iTx*diT[X][Z] + iTy*diT[Y][Z] + iTz*diT[Z][Z] );
				
				meanDiff += (float)Math.sqrt(iT2);
				if (Math.sqrt(iT2)>maxDiff) maxDiff = (float)Math.sqrt(iT2);
			}
			iu[X][xyz] /= Numerics.max(ZERO,den);
			iu[Y][xyz] /= Numerics.max(ZERO,den);
			iu[Z][xyz] /= Numerics.max(ZERO,den);			
		}
		meanDiff /= (nix*niy*niz);

		//if (debug) MedicUtilPublic.displayMessage("mean inverse consistency "+meanDiff+"\n");
		//if (debug) MedicUtilPublic.displayMessage("max. inverse consistency "+maxDiff+"\n");

        return meanDiff;
    } // 
    
    /** 
	 *	runs the algorithm
	 */
	public final void runRegistration() {        
        // top level
        if (debug) MedicUtilPublic.displayMessage("initialize all parameters \n");
        initializeTransform();
        initializeInverseTransform();
        initializeWeights();
		
		float[] stats = new float[16];
		for (int t=0;t<Niter;t++) {
			if (debug) MedicUtilPublic.displayMessage("iteration "+(t+1)+"\n");
			
			System.out.println("- Consistency matched -");
			computeImageToTargetEnergy(stats,MATCH);
			computeTargetToImageEnergy(stats,MATCH);	
			displayDivergence(s, 1.0f, nsx, nsy, nsz);
			displayDivergence(is, 1.0f, nisx, nisy, nisz);
			
			if (fieldType==COMPOSITIVE) {
				registerImageToTargetUpdateDecoupled();
				registerTargetToImageUpdateDecoupled();
			} else if (fieldType==MINDIV) {
				registerImageToTargetUpdateDecoupled();
				registerImageToTargetDivergence(false);
				
				registerTargetToImageUpdateDecoupled();
				registerImageToTargetDivergence(false);
			} else if (fieldType==MINDIVSEP) {
				registerImageToTargetUpdateDecoupled();
				registerTargetToImageUpdateDecoupled();
			}
			
			registerImageToTargetCompose();
			registerTargetToImageCompose();
			
			System.out.println("- Image matched -");
			computeImageToTargetEnergy(stats,IC);
			computeTargetToImageEnergy(stats,IC);	
			displayDivergence(s, 1.0f, nsx, nsy, nsz);
			displayDivergence(is, 1.0f, nisx, nisy, nisz);
			
			if (fieldType==MINDIVSEP) {
				registerImageToTargetDivergence(true);
				registerImageToTargetDivergence(true);
			
				registerImageToTargetCompose();
				registerTargetToImageCompose();
				
				System.out.println("- Div matched -");
				computeImageToTargetEnergy(stats,NONE);
				computeTargetToImageEnergy(stats,NONE);				
				displayDivergence(s, 1.0f, nsx, nsy, nsz);
				displayDivergence(is, 1.0f, nisx, nisy, nisz);
			}
			registerImageToTargetInverseConsistent();
			registerTargetToImageInverseConsistent();

			registerImageToTargetCompose();
			registerTargetToImageCompose();		
				
			if (t>0) {
				MedicUtilPublic.displayMessage("registration score: "+((stats[0]-stats[8])/stats[8])+", "+((stats[4]-stats[12])/stats[12])+"\n");
			}
		}
    }   
	
   /** 
	 *	runs the algorithm
	 */
	public final void runConsistentRegistration() {        
        // top level
        if (debug) MedicUtilPublic.displayMessage("initialize all parameters \n");
        initializeTransform();
        initializeInverseTransform();
        initializeWeights();
		
		float[] stats = new float[16];
		for (int t=0;t<Niter;t++) {
			if (debug) MedicUtilPublic.displayMessage("iteration "+(t+1)+"\n");
			
			System.out.println("- Consistency matched -");
			computeImageToTargetEnergy(stats,MATCH);
			computeTargetToImageEnergy(stats,MATCH);		
			
			if (fieldType==COMPOSITIVE) {
				registerImageToTargetUpdateDecoupled();
				registerTargetToImageUpdateDecoupled();
			} else if (fieldType==MINDIV) {
				registerImageToTargetUpdateDecoupled();
				registerImageToTargetDivergence(false);
				
				registerTargetToImageUpdateDecoupled();
				registerImageToTargetDivergence(false);
			} else if (fieldType==MINDIVSEP) {
				registerImageToTargetUpdateDecoupled();
				registerTargetToImageUpdateDecoupled();
			}
			
			registerImageToTargetCompose();
			registerTargetToImageCompose();
			
			System.out.println("- Image matched -");
			computeImageToTargetEnergy(stats,IC);
			computeTargetToImageEnergy(stats,IC);	
			
			if (fieldType==MINDIVSEP) {
				registerImageToTargetDivergence(true);
				registerImageToTargetDivergence(true);
			
				registerImageToTargetCompose();
				registerTargetToImageCompose();
				
				System.out.println("- Div matched -");
				computeImageToTargetEnergy(stats,NONE);
				computeTargetToImageEnergy(stats,NONE);				
			}

			int n=0;
			while ( (stats[3]>0.5 || stats[7]>0.5) && n<20) {
				registerImageToTargetInverseConsistent();
				registerTargetToImageInverseConsistent();

				registerImageToTargetCompose();
				registerTargetToImageCompose();		
				
				System.out.println("- More consistency needed ("+(n+1)+") -");
				computeImageToTargetEnergy(stats,IC);
				computeTargetToImageEnergy(stats,IC);		
			
				n++;
			}
			if (t>0) {
				MedicUtilPublic.displayMessage("registration score: "+((stats[0]-stats[8])/stats[8])+", "+((stats[4]-stats[12])/stats[12])+"\n");
			}
		}
    }   
    /** 
	 *	runs the algorithm
	 */
	public final void runDecoupledRegistration() {        
        // top level
        if (debug) MedicUtilPublic.displayMessage("initialize all parameters \n");
        initializeTransform();
        initializeInverseTransform();
        initializeWeights();
		
		float[] stats = new float[16];
		for (int t=0;t<Niter;t++) {
			if (debug) MedicUtilPublic.displayMessage("iteration "+(t+1)+"\n");
			
			System.out.println("- Energy -");
			computeImageToTargetEnergy(stats,MATCH);
			System.out.println(displayDivergence(s, 1.0f, nsx, nsy, nsz));
			computeTargetToImageEnergy(stats,MATCH);		
			System.out.println(displayDivergence(is, 1.0f, nisx, nisy, nisz));
			
			if (fieldType==COMPOSITIVE) {
				registerImageToTargetUpdateDecoupled();
				registerTargetToImageUpdateDecoupled();
			} else if (fieldType==MINDIV) {
				registerImageToTargetUpdateDecoupled();
				registerImageToTargetDivergence(false);
				
				registerTargetToImageUpdateDecoupled();
				registerImageToTargetDivergence(false);
			} else if (fieldType==MINDIVSEP) {
				registerImageToTargetUpdateDecoupled();
				registerTargetToImageUpdateDecoupled();
			}
			
			registerImageToTargetCompose();
			registerTargetToImageCompose();

			System.out.println("- Image matched -");
			computeImageToTargetEnergy(stats,IC);
			System.out.println(displayDivergence(s, 1.0f, nsx, nsy, nsz));
			computeTargetToImageEnergy(stats,IC);	
			System.out.println(displayDivergence(is, 1.0f, nisx, nisy, nisz));
			
			if (fieldType==MINDIVSEP) {
				registerImageToTargetDivergence(true);
				registerTargetToImageDivergence(true);
			
				registerImageToTargetCompose();
				registerTargetToImageCompose();

				System.out.println("- Div matched -");
				computeImageToTargetEnergy(stats,NONE);
				System.out.println(displayDivergence(s, 1.0f, nsx, nsy, nsz));
				computeTargetToImageEnergy(stats,NONE);	
				System.out.println(displayDivergence(is, 1.0f, nisx, nisy, nisz));
			}
			if (t>0) {
				MedicUtilPublic.displayMessage("registration score: "+((stats[0]-stats[8])/stats[8])+", "+((stats[4]-stats[12])/stats[12])+"\n");
			}
		}
    }   
    
    private final String displayDivergence(float[][] vect, float offset, int nx, int ny, int nz) {
    	float meanDiv = 0.0f;
    	float maxDiv = 0.0f;
		for (int x=1;x<nx-1;x++) for (int y=1;y<ny-1;y++) for (int z=1;z<nz-1;z++) {
			int xyz = x+nx*y+nx*ny*z;
			
			float div = 3.0f*scale*offset - ( 0.5f*(vect[X][xyz+1]-vect[X][xyz-1])
											+ 0.5f*(vect[Y][xyz+nx]-vect[Y][xyz-nx])
											+ 0.5f*(vect[Z][xyz+nx*ny]-vect[Z][xyz-nx*ny]) );
			div = Numerics.abs(div);			
						
			meanDiv += div;
			if (div>maxDiv) maxDiv = div;
		}
		meanDiv /= (nx-2)*(ny-2)*(nz-2);
		
		return ("Divergence: mean = "+meanDiv+", max = "+maxDiv+"\n");
    }
	
	/** 
	 *	returns the transformed coordinates
	 */
	public final float[] getCurrentMapping(int x, int y, int z) {
		float xs = x/scale;
		float ys = y/scale;
		float zs = z/scale;
		
		float[] Xs = new float[3];
		Xs[X] = ImageFunctions.linearClosestInterpolation(s[X],xs,ys,zs,nsx,nsy,nsz);
		Xs[Y] = ImageFunctions.linearClosestInterpolation(s[Y],xs,ys,zs,nsx,nsy,nsz);
		Xs[Z] = ImageFunctions.linearClosestInterpolation(s[Z],xs,ys,zs,nsx,nsy,nsz);
		return Xs;
	}// getCurrentMapping

	/** 
	 *	returns the transformed coordinates
	 */
	public final float[] getCurrentInverseMapping(int x, int y, int z) {
		float xs = x/scale;
		float ys = y/scale;
		float zs = z/scale;
		
		float[] Xs = new float[3];
		Xs[X] = ImageFunctions.linearClosestInterpolation(is[X],xs,ys,zs,nisx,nisy,nisz);
		Xs[Y] = ImageFunctions.linearClosestInterpolation(is[Y],xs,ys,zs,nisx,nisy,nisz);
		Xs[Z] = ImageFunctions.linearClosestInterpolation(is[Z],xs,ys,zs,nisx,nisy,nisz);
		return Xs;
	}// getCurrentMapping

    /** 
	 *	returns the transformed image
	 */
	public final float[][] exportTransformedImage() {
		float 	xs,ys,zs;
		float[][]	img;
		
		if (preType==NORMALIZED) img = new float[nc][ntx*nty*ntz];
		else img = new float[1][ntx*nty*ntz];
		
		for (int x=0;x<ntx;x++) for (int y=0;y<nty;y++) for (int z=0;z<ntz;z++) {
			int xyz = x+ntx*y+ntx*nty*z;
			xs = ImageFunctions.linearClosestInterpolation(s[X],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			ys = ImageFunctions.linearClosestInterpolation(s[Y],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			zs = ImageFunctions.linearClosestInterpolation(s[Z],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			
				// compute interpolated values
			if (preType==RAW) {
				float nbest = maskingThreshold;
				float best = 0.0f;
				int c = 0;
				for (int n=0;n<lb.length;n++) if (lb[n]!=maskingThreshold) {
					float score = ImageFunctions.linearClosestInterpolation(image[c],xs,ys,zs,nix,niy,niz);
					if (score>best) {
						best = score;
						nbest = lb[n];
					}
					c++;
				}
				img[0][xyz] = nbest;
			} else if (preType==NORMALIZED) {
				for (int n=0;n<nc;n++) {
					img[n][xyz] = ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nix,niy,niz);
				}
			}
		}
		return img;
	} // exportTransformedImage

     /** 
	 *	returns the transformed image
	 */
	public final float[][] exportInverseTransformedTarget() {
		float 	xs,ys,zs;
		float[][]	img;
		
		if (preType==NORMALIZED) img = new float[nc][nix*niy*niz];
		else img = new float[1][nix*niy*niz];
		
		 for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			int xyz = x+nix*y+nix*niy*z;
			xs = ImageFunctions.linearClosestInterpolation(is[X],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			ys = ImageFunctions.linearClosestInterpolation(is[Y],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			zs = ImageFunctions.linearClosestInterpolation(is[Z],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			
				// compute interpolated values
			if (preType==RAW) {
				float nbest = maskingThreshold;
				float best = 0.0f;
				int c = 0;
				for (int n=0;n<lb.length;n++) if ( (!useMask) || (lb[n]!=maskingThreshold) ) {
					float score = ImageFunctions.linearClosestInterpolation(target[c],xs,ys,zs,ntx,nty,ntz);
					if (score>best) {
						best = score;
						nbest = lb[n];
					}
					c++;
				}
				img[0][xyz] = nbest;
			} else if (preType==NORMALIZED) {
				for (int n=0;n<nc;n++) {
					img[n][xyz] = ImageFunctions.linearClosestInterpolation(target[n],xs,ys,zs,ntx,nty,ntz);
				}
			}
		}
		return img;
	} // exportTransformedImage

    /** 
	 *	returns the transform field v defined as X' = X+v (=s)
	 */
	public final float[][] exportTransformField() {
		float 	xs,ys,zs;
		float[][]	vec = new float[3][ntx*nty*ntz];
		
        for (int x=0;x<ntx;x++) for (int y=0;y<nty;y++) for (int z=0;z<ntz;z++) {
			int xyz = x+ntx*y+ntx*nty*z;
			
			xs = ImageFunctions.linearClosestInterpolation(s[X],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			ys = ImageFunctions.linearClosestInterpolation(s[Y],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			zs = ImageFunctions.linearClosestInterpolation(s[Z],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			
			vec[X][xyz] = xs-x;
			vec[Y][xyz] = ys-y;
			vec[Z][xyz] = zs-z;			
 		}
		return vec;
	} // exportTransformedImage

    /** 
	 *	returns the transform field v defined as X' = X+v (=s)
	 */
	public final float[][] exportInverseTransformField() {
		float 	xs,ys,zs;
		float[][]	vec = new float[3][nix*niy*niz];
		
        for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			int xyz = x+nix*y+nix*niy*z;
			
			xs = ImageFunctions.linearClosestInterpolation(is[X],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			ys = ImageFunctions.linearClosestInterpolation(is[Y],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			zs = ImageFunctions.linearClosestInterpolation(is[Z],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			
			vec[X][xyz] = xs-x;
			vec[Y][xyz] = ys-y;
			vec[Z][xyz] = zs-z;			
 		}
		return vec;
	} // exportTransformedImage

	/** 
	 *	returns the image residuals
	 */
	public final float[][] exportResiduals() {
		float 	xs,ys,zs;
		float[][]	img;
		
		if (preType==NORMALIZED) img = new float[nc][ntx*nty*ntz];
		else img = new float[1][ntx*nty*ntz];
		
		for (int x=0;x<ntx;x++) for (int y=0;y<nty;y++) for (int z=0;z<ntz;z++) {
			int xyz = x+ntx*y+ntx*nty*z;
    		xs = ImageFunctions.linearClosestInterpolation(s[X],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			ys = ImageFunctions.linearClosestInterpolation(s[Y],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			zs = ImageFunctions.linearClosestInterpolation(s[Z],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			
				// compute interpolated values
			if (preType==RAW) {
				img[0][xyz] = 0.0f;
				for (int n=0;n<nc;n++) {
					img[0][xyz] += Numerics.square(target[n][xyz]-ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nix,niy,niz));
				}
				img[0][xyz] = (float)Math.sqrt(img[0][xyz]);
			} else if (preType==NORMALIZED) {
				for (int n=0;n<nc;n++) {
					img[n][xyz] = Numerics.abs(target[n][xyz]-ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nix,niy,niz));
				}
			}
		}
		return img;
	} // exportResiduals

	/** 
	 *	returns the image residuals
	 */
	public final float[][] exportInverseResiduals() {
		float 	xs,ys,zs;
		float[][]	img;
		
		if (preType==NORMALIZED) img = new float[nc][nix*niy*niz];
		else img = new float[1][nix*niy*niz];
		
		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			int xyz = x+nix*y+nix*niy*z;
    		xs = ImageFunctions.linearClosestInterpolation(is[X],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			ys = ImageFunctions.linearClosestInterpolation(is[Y],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			zs = ImageFunctions.linearClosestInterpolation(is[Z],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			
				// compute interpolated values
			if (preType==RAW) {
				img[0][xyz] = 0.0f;
				for (int n=0;n<nc;n++) {
					img[0][xyz] += Numerics.square(image[n][xyz]-ImageFunctions.linearClosestInterpolation(target[n],xs,ys,zs,ntx,nty,ntz));
				}
				img[0][xyz] = (float)Math.sqrt(img[0][xyz]);
			} else if (preType==NORMALIZED) {
				for (int n=0;n<nc;n++) {
					img[n][xyz] = Numerics.abs(image[n][xyz]-ImageFunctions.linearClosestInterpolation(target[n],xs,ys,zs,ntx,nty,ntz));
				}
			}
		}
		return img;
	} // exportResiduals

	/** 
	 *	returns the transform residuals
	 */
	public final float[] exportForwardBackwardDistance() {
		float 	xs,ys,zs;
		float 	xsi,ysi,zsi;
		float[]	img;
		
		img = new float[ntx*nty*ntz];
		
		for (int x=0;x<ntx;x++) for (int y=0;y<nty;y++) for (int z=0;z<ntz;z++) {
			int xyz = x+ntx*y+ntx*nty*z;
    		xs = ImageFunctions.linearClosestInterpolation(s[X],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			ys = ImageFunctions.linearClosestInterpolation(s[Y],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			zs = ImageFunctions.linearClosestInterpolation(s[Z],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			
			xsi = ImageFunctions.linearClosestInterpolation(is[X],xs/scale,ys/scale,zs/scale,nisx,nisy,nisz);
			ysi = ImageFunctions.linearClosestInterpolation(is[Y],xs/scale,ys/scale,zs/scale,nisx,nisy,nisz);
			zsi = ImageFunctions.linearClosestInterpolation(is[Z],xs/scale,ys/scale,zs/scale,nisx,nisy,nisz);
			
			float wt = ImageFunctions.linearClosestInterpolation(w[0],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			
			img[xyz] = (float)Math.sqrt( Numerics.square(x-xsi) + Numerics.square(y-ysi) + Numerics.square(z-zsi) );
		}
		return img;
	} // exportResiduals

	
	/** 
	 *	returns the transform residuals
	 */
	public final float[] exportBackwardForwardDistance() {
		float 	xs,ys,zs;
		float 	xsi,ysi,zsi;
		float[]	img;
		
		img = new float[nix*niy*niz];
		
		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			int xyz = x+nix*y+nix*niy*z;
    		xs = ImageFunctions.linearClosestInterpolation(is[X],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			ys = ImageFunctions.linearClosestInterpolation(is[Y],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			zs = ImageFunctions.linearClosestInterpolation(is[Z],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			
			xsi = ImageFunctions.linearClosestInterpolation(s[X],xs/scale,ys/scale,zs/scale,nsx,nsy,nsz);
			ysi = ImageFunctions.linearClosestInterpolation(s[Y],xs/scale,ys/scale,zs/scale,nsx,nsy,nsz);
			zsi = ImageFunctions.linearClosestInterpolation(s[Z],xs/scale,ys/scale,zs/scale,nsx,nsy,nsz);
			
			float wi = ImageFunctions.linearClosestInterpolation(iw[0],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			
			img[xyz] = (float)Math.sqrt( Numerics.square(x-xsi) + Numerics.square(y-ysi) + Numerics.square(z-zsi) );
		}
		return img;
	} // exportResiduals

    /** 
	 *	returns the transform field v defined as X' = X+v (=s)
	 */
	public final byte[] exportTransformGrid(int ngrid) {
		float 	xs,ys,zs;
		byte[]	grid = new byte[ntx*nty*ntz];
		
        for (int x=0;x<ntx;x++) for (int y=0;y<nty;y++) for (int z=0;z<ntz;z++) {
        	int xyz = x+ntx*y+ntx*nty*z;
			
			xs = ImageFunctions.linearClosestInterpolation(s[X],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			ys = ImageFunctions.linearClosestInterpolation(s[Y],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			zs = ImageFunctions.linearClosestInterpolation(s[Z],x/scale,y/scale,z/scale,nsx,nsy,nsz);
			
			byte val = 0;
        	if ( (Numerics.abs(xs%ngrid)<1.0) || (Numerics.abs(ys%ngrid)<1.0) || (Numerics.abs(zs%ngrid)<1.0) ) val = 1;
			
        	grid[xyz] = val;
		}
		return grid;
	} // exportTransformedImage

    /** 
	 *	returns the transform field v defined as X' = X+v (=s)
	 */
	public final byte[] exportInverseTransformGrid(int ngrid) {
		float 	xs,ys,zs;
		byte[]	grid = new byte[nix*niy*niz];
		
        for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			int xyz = x+nix*y+nix*niy*z;
			
			xs = ImageFunctions.linearClosestInterpolation(is[X],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			ys = ImageFunctions.linearClosestInterpolation(is[Y],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			zs = ImageFunctions.linearClosestInterpolation(is[Z],x/scale,y/scale,z/scale,nisx,nisy,nisz);
			
			byte val = 0;
        	if ( (Numerics.abs(xs%ngrid)<1.0) || (Numerics.abs(ys%ngrid)<1.0) || (Numerics.abs(zs%ngrid)<1.0) ) val = 1;
			
        	grid[xyz] = val;
		}
		return grid;
	} // exportTransformedImage


	
}
