package edu.jhmi.rad.medic.methods;

import java.io.*;
import java.util.*;
import gov.nih.mipav.view.*;
import gov.nih.mipav.model.structures.jama.*;

import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.structures.*;
import edu.jhmi.rad.medic.utilities.*;
 
/**
 *
 *  This algorithm handles registration algorithms
 *	of images with the Demons algorithm
 *	and a multiscale / multigrid technique
 *	(original Demons method, not robust)
 *
 *
 *	@version    November 2004
 *	@author     Pierre-Louis Bazin
 *		
 *
 */
 
public class DemonsRegistration2 {
		
	// numerical quantities
	private static final	float   INF=1e30f;
	private static final	float   ZERO=1e-30f;
	
	// convenience tags
	private static final	int   X=0;
	private static final	int   Y=1;
	private static final	int   Z=2;
	private static final	int   T=3;
	
	// flag labels
	public static final	int   COMPOSITIVE 		= 101;
	public static final	int   DIFFEOMORPHIC 	= 102;
	public static final	int   FIXED 			= 201;
	public static final	int   MOVING 			= 202;
	public static final	int   SYMMETRIC 		= 203;
	
	public static final	int   GAUSS_FLUID 		= 301;
	public static final	int   GAUSS_DIFFUSION 	= 302;
	public static final	int   GAUSS_MIXED	 	= 303;
	public static final	int   DIV_FLUID 		= 304;
	public static final	int   DIV_DIFFUSION 	= 305;
	public static final	int   GAUSS_DIV_FLUID 		= 306;
	public static final	int   GAUSS_DIV_DIFFUSION 	= 307;
	
	public static final int	  RAW				= 401;
	public static final int	  NORMALIZED		= 402;
	public static final int	  SCALED			= 403;
	public static final int	  SEGMENTATION		= 404;
	public static final int	  FCM				= 405;
	public static final int	  MULTICHANNEL		= 406;
	public static final int	  LEVELSET			= 407;
	
	
	// data buffers
	private 	float[][]		image;  		// source image
	private 	float[][]	    target;			// target image
	private 	float[][]		fixed;  		// fixed image (multiscale)
	private 	float[][]	    moving;			// moving image (multiscale)
	private		float[][]		c;				// added transform from target space to original space (multiscale)
	private		float[][]		s;				// added transform from target space to original space (multiscale)
	private		float[][]		u;				// added transform update from target space to original space (multiscale)
	private static	int[]	nix,niy,niz;   	// image dimensions (pyramid)
	private static	int[]	ntx,nty,ntz;   	// target dimensions (pyramid)
	private static	float	rix,riy,riz;   	// image resolutions (no pyramid)
	private static	float	rtx,rty,rtz;   	// target resolutions (no pyramid)
	private static 	int		nc;
	private		float		imin, imax;
	private		float[]		lb;
	
	
	private static	int		nmx,nmy,nmz;   	// image dimensions (no pyramid)
	private static	int		nfx,nfy,nfz;   	// target dimensions (no pyramid)
	private static	float	rmx,rmy,rmz;   	// image resolutions (no pyramid)
	private static	float	rfx,rfy,rfz;   	// target resolutions (no pyramid)
	private 	float[][]	transform;		// prior transform matrix (for instance from prior alignment)		
	
	// pyramid handling
	private		int			levels;		// number of levels in the pyramid
	private		int			Niter;		// iterations for the first pass at a level
	private		int			Ntop;		// iterations for the pass at top level
	
	// parameters
	private		float		smoothingKernel; 	// smoothing kernel size
	private		float		spatialScale; 		// scale of transformation
	private		boolean		useMask;
	private		float  		maskingThreshold;			// background threshold
	private		int		regType;
	private		int		forceType;
	private		int		fieldType;
	private		int		preType;
	
	// computation variables
	private		float			sigma2; 		// scale of transformation
	

	private		float[][]		gaussKernel;
	private		int				gx,gy,gz;
    
	private		float[][]		divKernelX, divKernelY, divKernelZ;

	// computation flags
	private 	boolean 		isWorking;
	private 	boolean 		isCompleted;
	
	// for debug and display
	static final boolean		debug=true;
	static final boolean		verbose=true;
    
	/**
	 *  constructor
     *  note: the membership function mems_ must have sum = 0 on the masked areas
	 */
	public DemonsRegistration2(float[] image_, float[] target_,
									int nix_, int niy_, int niz_,
									float rix_, float riy_, float riz_,
									int ntx_, int nty_, int ntz_,
									float rtx_, float rty_, float rtz_,
									float smoothing_,
									float scale_,
									boolean useMask_,
									float maskVal_,
									int lev_, int Ni_, int Nt_,
									int reg_, int force_, int field_,
									int pre_,
									float[][] trans_) {
	
		smoothingKernel = smoothing_;
		spatialScale = scale_;
		sigma2 = spatialScale*spatialScale;
		
		useMask = useMask_;
		maskingThreshold = maskVal_;

		levels = lev_;
		Niter = Ni_;
		Ntop = Nt_;

		transform = trans_;
		
		rix = rix_;
		riy = riy_;
		riz = riz_;
			
		rtx = rtx_;
		rty = rty_;
		rtz = rtz_;
			
		regType = reg_;
		forceType = force_;
		fieldType = field_;
		preType = pre_;
		
		// init all the arrays : allocate the pyramids
		try {
			nix = new int[levels];
			niy = new int[levels];
			niz = new int[levels];
			ntx = new int[levels];
			nty = new int[levels];
			ntz = new int[levels];
            // compute the pyramid dimensions
			nix[0] = nix_;
			niy[0] = niy_;
			niz[0] = niz_;
			ntx[0] = ntx_;
			nty[0] = nty_;
			ntz[0] = ntz_;
			for (int l=1;l<levels;l++) {
				nix[l] = Numerics.floor(nix[l-1]/2.0);
				niy[l] = Numerics.floor(niy[l-1]/2.0);
				niz[l] = Numerics.floor(niz[l-1]/2.0);
				ntx[l] = Numerics.floor(ntx[l-1]/2.0);
				nty[l] = Numerics.floor(nty[l-1]/2.0);
				ntz[l] = Numerics.floor(ntz[l-1]/2.0);
			}
			
			preprocessInputImages(image_, target_);
		
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
		isWorking = true;
		if (debug) MedicUtilPublic.displayMessage("Demons:initialisation\n");
	}

	final public void finalize() {
		image = null;
		target = null;
		fixed = null; moving = null;
		s = null; u = null; c = null;
		System.gc();
	}
    
   public final float[][] getCurrentTransform() {
	   return s;
   }
   public final float[][] getCurrentUpdate() {
		return u;
	}
    
	public final boolean isWorking() { return isWorking; }
	public final boolean isCompleted() { return isCompleted; }
    
	private final void preprocessInputImages(float[] img, float[] trg) {
		if (preType==RAW) {
			nc = 1;
			image = new float[nc][];
			target = new float[nc][];
			image[0] = img;
			target[0] = trg;
		} else if (preType==NORMALIZED) {
			// find min, max
			imin = ImageFunctions.robustMinimum(img, 0.01f, 2, nix[0], niy[0], niz[0]);
			imax = ImageFunctions.robustMaximum(img, 0.01f, 2, nix[0], niy[0], niz[0]);
			
			float tmin = ImageFunctions.robustMinimum(trg, 0.01f, 2, ntx[0], nty[0], ntz[0]);
			float tmax = ImageFunctions.robustMaximum(trg, 0.01f, 2, ntx[0], nty[0], ntz[0]);

			if (debug) MedicUtilPublic.displayMessage("Image min / max: "+imin+" / "+imax+"\n");
			if (debug) MedicUtilPublic.displayMessage("Target min / max: "+tmin+" / "+tmax+"\n");
			
			// allocate
			nc = 1;
			image = new float[nc+1][];
			target = new float[nc][];
			image[0] = ImageFunctions.normalize(img, nix[0],niy[0],niz[0], imin, imax);
			target[0] = ImageFunctions.normalize(trg, ntx[0],nty[0],ntz[0], tmin, tmax);
			// keep original image for results
			image[1] = img;
		} else if (preType==SCALED) {
			// find min, max
			imin = ImageFunctions.robustMinimum(img, 0.01f, 2, nix[0], niy[0], niz[0]);
			imax = ImageFunctions.robustMaximum(img, 0.01f, 2, nix[0], niy[0], niz[0]);
			
			float tmin = ImageFunctions.robustMinimum(trg, 0.01f, 2, ntx[0], nty[0], ntz[0]);
			float tmax = ImageFunctions.robustMaximum(trg, 0.01f, 2, ntx[0], nty[0], ntz[0]);
			
			if (debug) MedicUtilPublic.displayMessage("Image min / max: "+imin+" / "+imax+"\n");
			if (debug) MedicUtilPublic.displayMessage("Target min / max: "+tmin+" / "+tmax+"\n");
			
			ImageFunctions.rescale(img, nix[0],niy[0],niz[0], imin, imax);
			ImageFunctions.rescale(trg, ntx[0],nty[0],ntz[0], tmin, tmax);
			
			// allocate
			nc = 1;
			image = new float[nc][];
			target = new float[nc][];
			image[0] = img;
			target[0] = trg;
			
		} else if (preType==SEGMENTATION) {
			// find max number of objects and list them; assume the target has proper labeling
			lb = ObjectProcessing.listLabels(trg,ntx[0],nty[0],ntz[0]);
			nc = lb.length;
			if (useMask) nc--;
			
			image = new float[nc][];
			target = new float[nc][];
			
			int c=0;
			for (int n=0;n<lb.length;n++) {
				if ( (!useMask) || (lb[n]!=maskingThreshold) ) {
					image[c] = ObjectProcessing.floatObjectFromImage(img,nix[0],niy[0],niz[0],lb[n],ObjectProcessing.EQUAL);
					target[c] = ObjectProcessing.floatObjectFromImage(trg,nix[0],niy[0],niz[0],lb[n],ObjectProcessing.EQUAL);
					c++;
				}			
			}
		} else if (preType==FCM) {
			// perform basic FCM segmentation
			nc = 3;
			//...
		} else if (preType==MULTICHANNEL) {
			nc = Numerics.round(img.length/(nix[0]*niy[0]*niz[0]));
			image = new float[nc][nix[0]*niy[0]*niz[0]];
			target = new float[nc][ntx[0]*nty[0]*ntz[0]];
			for (int n=0;n<nc;n++) for (int xyz=0;xyz<nix[0]*niy[0]*niz[0];xyz++) {
				image[n][xyz] = img[xyz + n*nix[0]*niy[0]*niz[0]];
			}
			for (int n=0;n<nc;n++) for (int xyz=0;xyz<ntx[0]*nty[0]*ntz[0];xyz++) {
				target[n][xyz] = trg[xyz + n*ntx[0]*nty[0]*ntz[0]];
			}
			img = null;
			trg = null;
		} else {
			nc = 1;
			image = new float[nc][];
			target = new float[nc][];
			image[0] = img;
			target[0] = trg;
		}	
		moving = new float[nc][];
		fixed = new float[nc][];
	}
	
	/** create an image pyramid from the original data, initialize Jacobians, etc */
	private final void initializeImages(int lvl) {

		int scale = 1;
		
		if (lvl>0) {
			for (int l=0;l<lvl;l++) scale *= 2;
			
			for (int n=0;n<nc;n++) {
				moving[n] = ImageFunctions.subsample(image[n], nix[0], niy[0], niz[0], scale);
				fixed[n] = ImageFunctions.subsample(target[n], ntx[0], nty[0], ntz[0], scale);
			}
		} else {
			for (int n=0;n<nc;n++) {
				moving[n] = image[n];
				fixed[n] = target[n];
			}
		}
		
		nmx = nix[lvl];
		nmy = niy[lvl];
		nmz = niz[lvl];
		
		nfx = ntx[lvl];
		nfy = nty[lvl];
		nfz = ntz[lvl];
		
		rmx = rix*scale;
		rmy = riy*scale;
		rmz = riz*scale;

		rfx = rtx*scale;
		rfy = rty*scale;
		rfz = rtz*scale;

		// use fixed scale smoothing ? best to smooth more at higher levels
		//gaussKernel = ImageFunctions.separableGaussianKernel(smoothingKernel/rfx,smoothingKernel/rfy,smoothingKernel/rfz);
		gaussKernel = ImageFunctions.separableGaussianKernel(smoothingKernel/rtx,smoothingKernel/rty,smoothingKernel/rtz);
		gx = (gaussKernel[X].length-1)/2;
		gy = (gaussKernel[Y].length-1)/2;
		gz = (gaussKernel[Z].length-1)/2;
		
		divKernelX = new float[3][];
		divKernelX[X] = new float[3];
		divKernelX[Y] = new float[1];
		divKernelX[Z] = new float[1];
		divKernelX[X][0] = 1.0f/4.0f;
		divKernelX[X][1] = 1.0f/2.0f;
		divKernelX[X][2] = 1.0f/4.0f;
		divKernelX[Y][0] = 1.0f;
		divKernelX[Z][0] = 1.0f;
				
		divKernelY = new float[3][];
		divKernelY[X] = new float[1];
		divKernelY[Y] = new float[3];
		divKernelY[Z] = new float[1];
		divKernelY[X][0] = 1.0f;
		divKernelY[Y][0] = 1.0f/4.0f;
		divKernelY[Y][1] = 1.0f/2.0f;
		divKernelY[Y][2] = 1.0f/4.0f;
		divKernelY[Z][0] = 1.0f;
				
		divKernelZ = new float[3][];
		divKernelZ[X] = new float[1];
		divKernelZ[Y] = new float[1];
		divKernelZ[Z] = new float[3];
		divKernelZ[X][0] = 1.0f;
		divKernelZ[Y][0] = 1.0f;
		divKernelZ[Z][0] = 1.0f/4.0f;
		divKernelZ[Z][1] = 1.0f/2.0f;
		divKernelZ[Z][2] = 1.0f/4.0f;
				
		if (debug) MedicUtilPublic.displayMessage("scale "+(lvl+1)+": \n img ("+nmx+"x"+nmy+"x"+nmz+", "+rmx+"|"+rmy+"|"+rmz+") \n trg ("+nfx+"x"+nfy+"x"+nfz+", "+rfx+"|"+rfy+"|"+rfz+") \n");
					
	}
	
    /** initialize the transform from previous estimate, if exists */
    private final void initializeTransform(int lvl) {
				
		// initialization: start from prior transform or zero
		float[][] sn = new float[3][nfx*nfy*nfz];
		if (transform==null) {
			for (int x=0;x<nfx;x++) for (int y=0;y<nfy;y++) for (int z=0;z<nfz;z++) {
				int xyz = x+nfx*y+nfx*nfy*z;
				sn[X][xyz] = x;
				sn[Y][xyz] = y;
				sn[Z][xyz] = z;
			}
		} else {
			for (int x=0;x<nfx;x++) for (int y=0;y<nfy;y++) for (int z=0;z<nfz;z++) {
				int xyz = x+nfx*y+nfx*nfy*z;
				sn[X][xyz] = transform[X][X]*x + transform[X][Y]*y + transform[X][Z]*z + transform[X][T];
				sn[Y][xyz] = transform[Y][X]*x + transform[Y][Y]*y + transform[Y][Z]*z + transform[Y][T];
				sn[Z][xyz] = transform[Z][X]*x + transform[Z][Y]*y + transform[Z][Z]*z + transform[Z][T];
			}
		}
		
		if (lvl==levels-1) {
			s = sn;
		} else {
			// sample from previous scale
			for (int x=0;x<ntx[lvl+1];x++) for (int y=0;y<nty[lvl+1];y++) for (int z=0;z<ntz[lvl+1];z++) {
				int xyz  = x+ntx[lvl+1]*y+ntx[lvl+1]*nty[lvl+1]*z;
				int xyz2  = 2*x+nfx*2*y+nfx*nfy*2*z;
				
				if (xyz2>=nfx*nfy*nfz) {
					MedicUtilPublic.displayMessage("too big: ("+x+","+y+","+z+") -> "+xyz2+" ["+nfx+"|"+nfy+"|"+nfz+"]\n");
					return;
				}
				if (xyz>=ntx[lvl+1]*nty[lvl+1]*ntz[lvl+1]) {
					MedicUtilPublic.displayMessage("too big: ("+x+","+y+","+z+") -> "+xyz+" ["+ntx[lvl+1]+"|"+nty[lvl+1]+"|"+ntz[lvl+1]+"]\n");
					return;
				}
				// no scaling of the coordinates: it is done in the resolution computations
				sn[X][xyz2] = 2.0f*s[X][xyz];
				sn[Y][xyz2] = 2.0f*s[Y][xyz];
				sn[Z][xyz2] = 2.0f*s[Z][xyz];
				
				sn[X][xyz2+1] = 2.0f*s[X][xyz]+1.0f;
				sn[Y][xyz2+1] = 2.0f*s[Y][xyz];
				sn[Z][xyz2+1] = 2.0f*s[Z][xyz];
				
            	sn[X][xyz2+nfx] = 2.0f*s[X][xyz];
				sn[Y][xyz2+nfx] = 2.0f*s[Y][xyz]+1.0f;
				sn[Z][xyz2+nfx] = 2.0f*s[Z][xyz];
				
            	sn[X][xyz2+nfx*nfy] = 2.0f*s[X][xyz];
				sn[Y][xyz2+nfx*nfy] = 2.0f*s[Y][xyz];
				sn[Z][xyz2+nfx*nfy] = 2.0f*s[Z][xyz]+1.0f;
				
				sn[X][xyz2+1+nfx] = 2.0f*s[X][xyz]+1.0f;
				sn[Y][xyz2+1+nfx] = 2.0f*s[Y][xyz]+1.0f;
				sn[Z][xyz2+1+nfx] = 2.0f*s[Z][xyz];
				
            	sn[X][xyz2+nfx+nfx*nfy] = 2.0f*s[X][xyz];
				sn[Y][xyz2+nfx+nfx*nfy] = 2.0f*s[Y][xyz]+1.0f;
				sn[Z][xyz2+nfx+nfx*nfy] = 2.0f*s[Z][xyz]+1.0f;
				
            	sn[X][xyz2+nfx*nfy+1] = 2.0f*s[X][xyz]+1.0f;
				sn[Y][xyz2+nfx*nfy+1] = 2.0f*s[Y][xyz];
				sn[Z][xyz2+nfx*nfy+1] = 2.0f*s[Z][xyz]+1.0f;
				
            	sn[X][xyz2+1+nfx+nfx*nfy] = 2.0f*s[X][xyz]+1.0f;
				sn[Y][xyz2+1+nfx+nfx*nfy] = 2.0f*s[Y][xyz]+1.0f;
				sn[Z][xyz2+1+nfx+nfx*nfy] = 2.0f*s[Z][xyz]+1.0f;
				
            }
			s = null;
			s = sn;
        }
		
		// update always zero
		u = new float[3][nfx*nfy*nfz];
		for (int xyz=0;xyz<nfx*nfy*nfz;xyz++) {
			u[X][xyz] = 0.0f;
			u[Y][xyz] = 0.0f;
			u[Z][xyz] = 0.0f;
		}
		
		// composite update always zero
		c = new float[3][nfx*nfy*nfz];
		for (int xyz=0;xyz<nfx*nfy*nfz;xyz++) {
			c[X][xyz] = 0.0f;
			c[Y][xyz] = 0.0f;
			c[Z][xyz] = 0.0f;
		}
		
		return;
    }
    
	
    /**
	 * compute the image position given the target
	 * for a given level l
     * performs only one iteration
	 */
    final public void registerImageToTarget() {
		
		if (debug) MedicUtilPublic.displayMessage("update: ");
		
		float meanDiff = 0.0f;
		for (int x=1;x<nfx-1;x++) for (int y=1;y<nfy-1;y++) for (int z=1;z<nfz-1;z++) {
			int xyz = x+nfx*y+nfx*nfy*z;
		
			// compute the update field
			float xs = s[X][xyz];
			float ys = s[Y][xyz];
			float zs = s[Z][xyz];
		
			float xsmx = s[X][xyz-1];
			float ysmx = s[Y][xyz-1];
			float zsmx = s[Z][xyz-1];
	
			float xspx = s[X][xyz+1];
			float yspx = s[Y][xyz+1];
			float zspx = s[Z][xyz+1];
	
			float xsmy = s[X][xyz-nfx];
			float ysmy = s[Y][xyz-nfx];
			float zsmy = s[Z][xyz-nfx];
	
			float xspy = s[X][xyz+nfx];
			float yspy = s[Y][xyz+nfx];
			float zspy = s[Z][xyz+nfx];
	
			float xsmz = s[X][xyz-nfx*nfy];
			float ysmz = s[Y][xyz-nfx*nfy];
			float zsmz = s[Z][xyz-nfx*nfy];
	
			float xspz = s[X][xyz+nfx*nfy];
			float yspz = s[Y][xyz+nfx*nfy];
			float zspz = s[Z][xyz+nfx*nfy];
			
			float den = 0.0f;
			u[X][xyz] = 0.0f;
			u[Y][xyz] = 0.0f;
			u[Z][xyz] = 0.0f;
			for (int n=0;n<nc;n++) {
			
				float mov = ImageFunctions.linearClosestInterpolation(moving[n],xs,ys,zs,nmx,nmy,nmz);
				
				float diff = (fixed[n][xyz] - mov);
				
				float Jx, Jy, Jz;
				if (forceType==FIXED) {
					Jx = 0.5f/rfx*(fixed[n][xyz+1] 		 - fixed[n][xyz-1]);
					Jy = 0.5f/rfy*(fixed[n][xyz+nfx] 	 - fixed[n][xyz-nfx]);
					Jz = 0.5f/rfz*(fixed[n][xyz+nfx*nfy] - fixed[n][xyz-nfx*nfy]);
				} else if (forceType==MOVING) {
					Jx = 0.5f/rmx*(ImageFunctions.linearClosestInterpolation(moving[n],xspx,yspx,zspx,nmx,nmy,nmz)-ImageFunctions.linearClosestInterpolation(moving[n],xsmx,ysmx,zsmx,nmx,nmy,nmz));
					Jy = 0.5f/rmy*(ImageFunctions.linearClosestInterpolation(moving[n],xspy,yspy,zspy,nmx,nmy,nmz)-ImageFunctions.linearClosestInterpolation(moving[n],xsmy,ysmy,zsmy,nmx,nmy,nmz));
					Jz = 0.5f/rmz*(ImageFunctions.linearClosestInterpolation(moving[n],xspz,yspz,zspz,nmx,nmy,nmz)-ImageFunctions.linearClosestInterpolation(moving[n],xsmz,ysmz,zsmz,nmx,nmy,nmz));
				} else {
					Jx = 0.25f/rfx*(fixed[n][xyz+1] 	  - fixed[n][xyz-1]);
					Jy = 0.25f/rfy*(fixed[n][xyz+nfx] 	  - fixed[n][xyz-nfx]);
					Jz = 0.25f/rfz*(fixed[n][xyz+nfx*nfy] - fixed[n][xyz-nfx*nfy]);
					
					Jx += 0.25f/rmx*(ImageFunctions.linearClosestInterpolation(moving[n],xspx,yspx,zspx,nmx,nmy,nmz)-ImageFunctions.linearClosestInterpolation(moving[n],xsmx,ysmx,zsmx,nmx,nmy,nmz));
					Jy += 0.25f/rmy*(ImageFunctions.linearClosestInterpolation(moving[n],xspy,yspy,zspy,nmx,nmy,nmz)-ImageFunctions.linearClosestInterpolation(moving[n],xsmy,ysmy,zsmy,nmx,nmy,nmz));
					Jz += 0.25f/rmz*(ImageFunctions.linearClosestInterpolation(moving[n],xspz,yspz,zspz,nmx,nmy,nmz)-ImageFunctions.linearClosestInterpolation(moving[n],xsmz,ysmz,zsmz,nmx,nmy,nmz));
				}
			
				float J2 = Jx*Jx+Jy*Jy+Jz*Jz;
				den += diff*diff/sigma2 + J2;
				
				u[X][xyz] += diff*Jx;
				u[Y][xyz] += diff*Jy;
				u[Z][xyz] += diff*Jz;
				
				meanDiff += Numerics.abs(diff);
			}
			u[X][xyz] /= Numerics.max(ZERO,den);
			u[Y][xyz] /= Numerics.max(ZERO,den);
			u[Z][xyz] /= Numerics.max(ZERO,den);
		}
		meanDiff /= (nfx*nfy*nfz);
		
		if (regType==GAUSS_FLUID || regType==GAUSS_MIXED || regType==GAUSS_DIV_FLUID) {
			if (debug) MedicUtilPublic.displayMessage("GAUSS_FLUID regularization \n");
		
			// smooth the result with a gaussian kernel
			u[X] = ImageFunctions.separableConvolution(u[X],nfx,nfy,nfz,gaussKernel,gx,gy,gz);
			u[Y] = ImageFunctions.separableConvolution(u[Y],nfx,nfy,nfz,gaussKernel,gx,gy,gz);
			u[Z] = ImageFunctions.separableConvolution(u[Z],nfx,nfy,nfz,gaussKernel,gx,gy,gz);
		} else if (regType==DIV_FLUID || regType==GAUSS_DIV_FLUID) {
		
			// smooth the result with a gaussian kernel
			u[X] = ImageFunctions.separableConvolution(u[X],nfx,nfy,nfz,divKernelX,1,0,0);
			u[Y] = ImageFunctions.separableConvolution(u[Y],nfx,nfy,nfz,divKernelY,0,1,0);
			u[Z] = ImageFunctions.separableConvolution(u[Z],nfx,nfy,nfz,divKernelZ,0,0,1);
		}
		
		// compose the transformations
		if (fieldType==COMPOSITIVE) {
			if (debug) MedicUtilPublic.displayMessage("compose with current transform \n");
							
			for (int x=0;x<nfx;x++) for (int y=0;y<nfy;y++) for (int z=0;z<nfz;z++) {
				int xyz = x+nfx*y+nfx*nfy*z;
				
				float xu = x+u[X][xyz];
				float yu = y+u[Y][xyz];
				float zu = z+u[Z][xyz];
				
				// note: if outside, extrapolate as X+u
				c[X][xyz] = ImageFunctions.linearClosestInterpolation(s[X],xu,yu,zu,nfx,nfy,nfz) - x;
				c[Y][xyz] = ImageFunctions.linearClosestInterpolation(s[Y],xu,yu,zu,nfx,nfy,nfz) - y;
				c[Z][xyz] = ImageFunctions.linearClosestInterpolation(s[Z],xu,yu,zu,nfx,nfy,nfz) - z;
			}
		}
		
		if (regType==GAUSS_DIFFUSION || regType==GAUSS_MIXED || regType==GAUSS_DIV_DIFFUSION) {
			if (debug) MedicUtilPublic.displayMessage("GAUSS_DIFFUSION regularization \n");
			
			// smooth the result with a gaussian kernel
			c[X] = ImageFunctions.separableConvolution(c[X],nfx,nfy,nfz,gaussKernel,gx,gy,gz);
			c[Y] = ImageFunctions.separableConvolution(c[Y],nfx,nfy,nfz,gaussKernel,gx,gy,gz);
			c[Z] = ImageFunctions.separableConvolution(c[Z],nfx,nfy,nfz,gaussKernel,gx,gy,gz);
		} else if (regType==DIV_DIFFUSION || regType==GAUSS_DIV_DIFFUSION) {
			if (debug) MedicUtilPublic.displayMessage("GAUSS_DIFFUSION regularization \n");
			
			// smooth the result with a gaussian kernel
			c[X] = ImageFunctions.separableConvolution(c[X],nfx,nfy,nfz,divKernelX,1,0,0);
			c[Y] = ImageFunctions.separableConvolution(c[Y],nfx,nfy,nfz,divKernelY,0,1,0);
			c[Z] = ImageFunctions.separableConvolution(c[Z],nfx,nfy,nfz,divKernelZ,0,0,1);
		}
					
		float meanC = 0.0f;
		for (int x=0;x<nfx;x++) for (int y=0;y<nfy;y++) for (int z=0;z<nfz;z++) {
			int xyz = x+nfx*y+nfx*nfy*z;
			
			s[X][xyz] = x + c[X][xyz];
			s[Y][xyz] = y + c[Y][xyz];
			s[Z][xyz] = z + c[Z][xyz];
			
			meanC += c[X][xyz]*c[X][xyz]+c[Y][xyz]*c[Y][xyz]+c[Z][xyz]*c[Z][xyz];
		}
		meanC /= (nfx*nfy*nfz);
		
		if (debug) MedicUtilPublic.displayMessage("convergence "+meanC+" -> "+meanDiff+"\n");

        return;
    } // 
    
    /** 
	 *	runs the gaussian pyramid algorithm
	 */
	public final void runGaussianPyramid() {        
        // top level
        if (debug) MedicUtilPublic.displayMessage("gaussian pyramid : top\n");
		
		if (debug) MedicUtilPublic.displayMessage("initialize all parameters \n");
        initializeImages(levels-1);
		initializeTransform(levels-1);
		
		for (int t=0;t<Ntop;t++) {
			if (debug) MedicUtilPublic.displayMessage("iteration "+(t+1)+"\n");
			registerImageToTarget();
		}
		
		// going down to level 0
		for (int l=levels-2;l>=0;l--) {
			if (debug) MedicUtilPublic.displayMessage("gaussian pyramid : level"+(l+1)+"\n");
		
			if (debug) MedicUtilPublic.displayMessage("initialize all parameters \n");
			initializeImages(l);
			initializeTransform(l);
		
			for (int t=0;t<Niter;t++) {
			
			if (debug) MedicUtilPublic.displayMessage("iteration "+(t+1)+"\n");
				registerImageToTarget();
			}
		}
    }//runGaussianPyramid
    
    /** 
	 *	returns the transformed coordinates
	 */
	public final float[] getCurrentMapping(int x, int y, int z) {
		int xyz = x+nfx*y+nfx*nfy*z;
		float[] Xs = new float[3];
		Xs[X] = s[X][xyz];
		Xs[Y] = s[Y][xyz];
		Xs[Z] = s[Z][xyz];
		return Xs;
	}// getCurrentMapping


    /** 
	 *	returns the transformed image
	 */
	public final float[][] exportTransformedImage() {
		float 	xs,ys,zs;
		float[][]	img;
		
		if (preType==MULTICHANNEL) img = new float[nc][nfx*nfy*nfz];
		else img = new float[1][nfx*nfy*nfz];
		
		if (preType==SCALED) ImageFunctions.unscale(image[0],nmx,nmy,nmz,imin,imax);
		
        for (int x=0;x<nfx;x++) for (int y=0;y<nfy;y++) for (int z=0;z<nfz;z++) {
			int xyz = x+nfx*y+nfx*nfy*z;
            xs = s[X][xyz];
			ys = s[Y][xyz];
			zs = s[Z][xyz];
			
				// compute interpolated values
			if (preType==RAW) {
				img[0][xyz] = ImageFunctions.linearClosestInterpolation(image[0],xs,ys,zs,nmx,nmy,nmz);
			} else if (preType==NORMALIZED) {
				img[0][xyz] = ImageFunctions.linearClosestInterpolation(image[1],xs,ys,zs,nmx,nmy,nmz);
			} else if (preType==SCALED) {
				img[0][xyz] = ImageFunctions.linearClosestInterpolation(image[0],xs,ys,zs,nmx,nmy,nmz);
			} else if (preType==SEGMENTATION) {
				float nbest = maskingThreshold;
				float best = 0.0f;
				int c = 0;
				for (int n=0;n<lb.length;n++) if ( (!useMask) || (lb[n]!=maskingThreshold) ) {
					float score = ImageFunctions.linearClosestInterpolation(image[c],xs,ys,zs,nmx,nmy,nmz);
					if (score>best) {
						best = score;
						nbest = lb[n];
					}
					c++;
				}
				img[0][xyz] = nbest;
			} else if (preType==MULTICHANNEL) {
				for (int n=0;n<nc;n++) {
					img[n][xyz] = ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nmx,nmy,nmz);
				}
			}
		}
		return img;
	} // exportTransformedImage

    /** 
	 *	returns the transform field v defined as X' = X+v (=s)
	 */
	public final float[][] exportTransformField() {
		float 	xs,ys,zs;
		float[][]	vec = new float[3][nfx*nfy*nfz];
		
        for (int x=0;x<nfx;x++) for (int y=0;y<nfy;y++) for (int z=0;z<nfz;z++) {
			int xyz = x+nfx*y+nfx*nfy*z;
			vec[X][xyz] = s[X][xyz]-x;
			vec[Y][xyz] = s[Y][xyz]-y;
			vec[Z][xyz] = s[Z][xyz]-z;			
 		}
		return vec;
	} // exportTransformedImage

	/** 
	 *	returns the image residuals
	 */
	public final float[][] exportResiduals() {
		float 	xs,ys,zs;
		float[][]	img;
		
		if (preType==MULTICHANNEL) img = new float[nc][nfx*nfy*nfz];
		else img = new float[1][nfx*nfy*nfz];
		
		for (int x=0;x<nfx;x++) for (int y=0;y<nfy;y++) for (int z=0;z<nfz;z++) {
			int xyz = x+nfx*y+nfx*nfy*z;
            xs = s[X][xyz];
			ys = s[Y][xyz];
			zs = s[Z][xyz];
			
				// compute interpolated values
			if (preType==RAW || preType==NORMALIZED || preType==SCALED) {
				img[0][xyz] = Numerics.square(target[0][xyz] - ImageFunctions.linearClosestInterpolation(image[0],xs,ys,zs,nmx,nmy,nmz));
			} else if (preType==SEGMENTATION) {
				img[0][xyz] = 0.0f;
				for (int n=0;n<nc;n++) {
					img[0][xyz] += Numerics.square(target[n][xyz]-ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nmx,nmy,nmz));
				}
			} else if (preType==MULTICHANNEL) {
				for (int n=0;n<nc;n++) {
					img[n][xyz] = Numerics.square(target[n][xyz]-ImageFunctions.linearClosestInterpolation(image[n],xs,ys,zs,nmx,nmy,nmz));
				}
			}
		}
		return img;
	} // exportResiduals

	
}
