package edu.jhmi.rad.medic.methods;
 import java.io.*;
import java.util.*;
import gov.nih.mipav.view.*;
import edu.jhmi.rad.medic.utilities.*;

/**
 *
 *  This algorithm handles the main Fuzzy C-Means operations
 *  for 3D data in FANTASM (membership, centroid computations)
 *	with different options: outliers, inhomogeneity correction, edge maps
 *	(the corresponding functions are to be found in different classes)
 *
 *	@version    December 2004
 *	@author     Pierre-Louis Bazin
 *		
 *
 */
 
public class SegmentationFCM {
		
	// numerical quantities
	private static final	float   INF=1e30f;
	private static final	float   ZERO=1e-30f;
	
	// data buffers
	private 	float[][][]			image;  			// original image
	private 	float[][][][]		mems;				// membership function
	private 	float[]				centroids;			// cluster centroids
	private 	float[][][]			field;  			// inhomogeneity field
	private 	float[][][][]		fields;  			// separate centroid fields
	private 	float[][][][]		edges;  			// edge field (3 directions; forward convention)
	private 	boolean[][][]		mask;   			// image mask: true for data points
	private		static	int			nx,ny,nz;   		// image dimensions
	
	// parameters
	private 	int 		clusters;   // number of clusters
	private 	int 		classes;    // number of classes in original membership: > clusters if outliers
	private 	float 		smoothing;	// MRF smoothing
    private		float		outlier;	// outlier distance
	private		float		fuzziness = 2.0f;	// fuzziness factor
	private		PowerTable	power, invpower;
	private		float		imgvar;		// image variance
			
	// computation variables
	private		float[]		prev;	// previous membership values
	
	// computation flags
	private 	boolean 		isWorking;
	private 	boolean 		isCompleted;
	private 	boolean 		useEdges;
	private 	int 			correctionType;	// selects the appropriate type of inhomogeneity correction
	
	private		int				distMode = 1;
	private	static final int	GAUSSIAN = 1;
	private	static final int	RAYLEIGH = 2;
	private	static final int	RAYLEIGH2 = 3;
	private	static final int	RICIAN = 4;
	
	
	// for debug and display
	ViewUserInterface			UI;
    ViewJProgressBar            progressBar;
	static final boolean		debug=true;
	static final boolean		verbose=true;
	
	/**
	 *  constructor
	 *	note: all images passed to the algorithm are just linked, not copied
	 */
	public SegmentationFCM(float[][][] image_, boolean [][][] mask_, 
					int nx_, int ny_, int nz_,
					int classes_, int clusters_, 
					float smoothing_, float fuzzy_, float outlierRatio_,
					ViewUserInterface UI_, ViewJProgressBar bar_) {
		this(image_, mask_, nx_, ny_, nz_, classes_, clusters_, 
			 smoothing_, fuzzy_, outlierRatio_, "Gaussian", UI_, bar_);
	}
	public SegmentationFCM(float[][][] image_, boolean [][][] mask_, 
					int nx_, int ny_, int nz_,
					int classes_, int clusters_, 
					float smoothing_, float fuzzy_, float outlierRatio_,
					String dist_,
					ViewUserInterface UI_, ViewJProgressBar bar_) {
		
		image = image_;
		mask = mask_;
		
		nx = nx_;
		ny = ny_;
		nz = nz_;
		
		classes = classes_;
		clusters = clusters_;
		fuzziness = fuzzy_;
		
		smoothing = smoothing_;
        outlier = outlierRatio_;
		
		useEdges = false;
		edges = null;
		
		correctionType = InhomogeneityCorrection.NONE;
		field = null;
		fields = null;
		
		if (dist_.equals("Gauss-Rayleigh")) distMode = RAYLEIGH;
		else if (dist_.equals("Gauss-Rayleigh2")) distMode = RAYLEIGH2;
		else if (dist_.equals("Rician")) distMode = RICIAN;
		else distMode = GAUSSIAN;
		
		UI = UI_;
        progressBar = bar_;
		
		// init all the new arrays
		try {
			//mems = new float[classes][nx][ny][nz];
			mems = new float[nx][ny][nz][classes];
			centroids = new float[clusters];
			prev = new float[classes];
			if (fuzziness!=2) {
				power = new PowerTable(0.0f , 1.0f , 0.000001f , fuzziness );
				float Imin = INF, Imax = -INF;
				for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
					if (image[x][y][z]<Imin) Imin = image[x][y][z];
					if (image[x][y][z]>Imax) Imax = image[x][y][z];
				}
				invpower = new PowerTable(0, (Imax-Imin)*(Imax-Imin), 0.000001f*(Imax-Imin)*(Imax-Imin), 1.0f/(1.0f-fuzziness) );
			} else {
				power = null;
				invpower = null;
			}
		} catch (OutOfMemoryError e){
			isWorking = false;
            finalize();
			System.out.println(e.getMessage());
			return;
		}
		isWorking = true;

		// init values
		for (int x=0;x<nx;x++) for (int y=0;y<ny;y++) for (int z=0;z<nz;z++) {
			for (int k=0;k<classes;k++) {
				//mems[k][x][y][z] = 0.0f;
				mems[x][y][z][k] = 0.0f;
			}
		}
		for (int k=0;k<clusters;k++) {
			centroids[k] = 0.0f;
		}

		if (debug) MedicUtilPublic.displayMessage("FCM:initialisation\n");
	}

	/** clean-up: destroy membership and centroid arrays */
	public final void finalize() {
		mems = null;
		centroids = null;
		prev = null;
		System.gc();
	}
	
    /** accessor for computed data */ 
    public final float[][][][] getMemberships() { return mems; }
    /** accessor for computed data */ 
    public final float[] getCentroids() { return centroids; }
    /** accessor for computed data */ 
	final public void setCentroids(float[] cent) { 
		centroids = cent; 
		if (debug) {
			MedicUtilPublic.displayMessage("centroids: ("+centroids[0]);
			for (int k=1;k<clusters;k++) MedicUtilPublic.displayMessage(", "+centroids[k]);
			MedicUtilPublic.displayMessage(")\n");
		}

	}
    /** accessor for computed data */ 
	final public void importCentroids(float[] cent) { 
		for (int k=0;k<clusters;k++) centroids[k] = cent[k]; 
		if (debug) {
			MedicUtilPublic.displayMessage("centroids: ("+centroids[0]);
			for (int k=1;k<clusters;k++) MedicUtilPublic.displayMessage(", "+centroids[k]);
			MedicUtilPublic.displayMessage(")\n");
		}

	}
    /** change parameters */
    public final void setMRF(float smooth) { smoothing = smooth; }
    public final float getMRF() { return smoothing; }
    /** add inhomogeneity correction */
    public final void addInhomogeneityCorrection(float[][][] field_, int type_) {
        field = field_;
        correctionType = type_;
    }
    public final void addSeparateInhomogeneityCorrection(float[][][][] fields_) {
		fields = fields_;
        correctionType = InhomogeneityCorrection.SEPARATE;
    }
	/** add edge parameter */
    public final void addEdgeMap(float[][][][] edges_) {
        edges = edges_;
        useEdges = true;
    }
	/** computation flags */
	public final boolean isWorking() { return isWorking; }
	/** computation flags */
	public final boolean isCompleted() { return isCompleted; }
	
    /** 
	 *  compute the FCM membership functions given the centroids
	 *	with the different options (outliers, field, edges, MRF)
	 */
    final public float computeMemberships() {
        float distance,dist;
        int x,y,z,k,m;
        int progress, mod;
        long inner_loop_time;
        float den,num;
        float neighbors, ngb;
        
		if (fuzziness!=2) return computeGeneralMemberships();
		
		if (distMode==RICIAN) computeImageVariance();
		
        distance = 0.0f;
		progress = 0;
        mod = nx*ny*nz/20; // mod is 1 percent of length

        inner_loop_time = System.currentTimeMillis();
        for (x=1;x<nx-1;x++) for (y=1;y<ny-1;y++) for (z=1;z<nz-1;z++) {
			progress++;
			if ( (verbose) && (progress%mod==0) )
                if (progressBar!=null) progressBar.updateValue(Math.round( (float)progress/(float)mod*5.0f),false);

			if ( mask[x][y][z] ) {
				den = 0;
				// remember the previous values
				for (k=0;k<classes;k++) 
					//prev[k] = mems[k][x][y][z];
					prev[k] = mems[x][y][z][k];

				for (k=0;k<classes;k++) {
					
					// data term
					if (distMode==RAYLEIGH && k==0) {
						if (correctionType==InhomogeneityCorrection.NONE) 
							num = rayleighDistance(image[x][y][z],centroids[k]);
						else if (correctionType==InhomogeneityCorrection.IMAGE) 
							num = rayleighDistance(field[x][y][z]*image[x][y][z],centroids[k]);
						else
							num = INF;
					} else if (distMode==RAYLEIGH2 && k==0) {
						if (correctionType==InhomogeneityCorrection.NONE) 
							num = rayleighDirectDistance(image[x][y][z],centroids[k]);
						else if (correctionType==InhomogeneityCorrection.IMAGE) 
							num = rayleighDirectDistance(field[x][y][z]*image[x][y][z],centroids[k]);
						else
							num = INF;
					} else if (distMode==RICIAN) {
						if (correctionType==InhomogeneityCorrection.NONE) 
							num = ricianApproxDistance(image[x][y][z],centroids[k]);
						else if (correctionType==InhomogeneityCorrection.IMAGE) 
							num = ricianApproxDistance(field[x][y][z]*image[x][y][z],centroids[k]);
						else
							num = INF;
					} else if (k<clusters) {
						if (correctionType==InhomogeneityCorrection.NONE) 
							num = (image[x][y][z]-centroids[k])*(image[x][y][z]-centroids[k]);
						else if (correctionType==InhomogeneityCorrection.IMAGE) 
							num = (field[x][y][z]*image[x][y][z]-centroids[k])*(field[x][y][z]*image[x][y][z]-centroids[k]);
						else if (correctionType==InhomogeneityCorrection.CENTROIDS) 
							num = (image[x][y][z]-field[x][y][z]*centroids[k])*(image[x][y][z]-field[x][y][z]*centroids[k]);
						else if (correctionType==InhomogeneityCorrection.SEPARATE)
							num = (image[x][y][z]-fields[x][y][z][k]*centroids[k])*(image[x][y][z]-fields[x][y][z][k]*centroids[k]);
						else num = INF;
					} else {
						num = outlier*outlier;
					}
					
					// spatial smoothing
					if (smoothing > 0.0f) { 
						ngb = 0.0f;  
						neighbors = 0.0f;
						// case by case	: X+
						if (mask[x+1][y][z]) for (m=0;m<classes;m++) if (m!=k) {
							//if (useEdges) ngb += edges[0][x][y][z]*mems[m][x+1][y][z]*mems[m][x+1][y][z];
							//else ngb += mems[m][x+1][y][z]*mems[m][x+1][y][z];
							if (useEdges) ngb += edges[0][x][y][z]*mems[x+1][y][z][m]*mems[x+1][y][z][m];
							else ngb += mems[x+1][y][z][m]*mems[x+1][y][z][m];
							neighbors ++;
						}
						// case by case	: X-
						if (mask[x-1][y][z]) for (m=0;m<classes;m++) if (m!=k) {
							//if (useEdges) ngb += edges[0][x-1][y][z]*mems[m][x-1][y][z]*mems[m][x-1][y][z];
							//else ngb += mems[m][x-1][y][z]*mems[m][x-1][y][z];
							if (useEdges) ngb += edges[0][x-1][y][z]*mems[x-1][y][z][m]*mems[x-1][y][z][m];
							else ngb += mems[x-1][y][z][m]*mems[x-1][y][z][m];
							neighbors ++;
						}
						// case by case	: Y+
						if (mask[x][y+1][z]) for (m=0;m<classes;m++) if (m!=k) {
							//if (useEdges) ngb += edges[1][x][y][z]*mems[m][x][y+1][z]*mems[m][x][y+1][z];
							//else ngb += mems[m][x][y+1][z]*mems[m][x][y+1][z];
							if (useEdges) ngb += edges[1][x][y][z]*mems[x][y+1][z][m]*mems[x][y+1][z][m];
							else ngb += mems[x][y+1][z][m]*mems[x][y+1][z][m];
							neighbors ++;
						}
						// case by case	: Y-
						if (mask[x][y-1][z]) for (m=0;m<classes;m++) if (m!=k) {
							//if (useEdges) ngb += edges[1][x][y-1][z]*mems[m][x][y-1][z]*mems[m][x][y-1][z];
							//else ngb += mems[m][x][y-1][z]*mems[m][x][y-1][z];
							if (useEdges) ngb += edges[1][x][y-1][z]*mems[x][y-1][z][m]*mems[x][y-1][z][m];
							else ngb += mems[x][y-1][z][m]*mems[x][y-1][z][m];
							neighbors ++;
						}
						// case by case	: Z+
						if (mask[x][y][z+1]) for (m=0;m<classes;m++) if (m!=k) {
							//if (useEdges) ngb += edges[2][x][y][z]*mems[m][x][y][z+1]*mems[m][x][y][z+1];
							//else ngb += mems[m][x][y][z+1]*mems[m][x][y][z+1];
							if (useEdges) ngb += edges[2][x][y][z]*mems[x][y][z+1][m]*mems[x][y][z+1][m];
							else ngb += mems[x][y][z+1][m]*mems[x][y][z+1][m];
							neighbors ++;
						}
						// case by case	: Z-
						if (mask[x][y][z-1]) for (m=0;m<classes;m++) if (m!=k) {
							//if (useEdges) ngb += edges[2][x][y][z-1]*mems[m][x][y][z-1]*mems[m][x][y][z-1];
							//else ngb += mems[m][x][y][z-1]*mems[m][x][y][z-1];
							if (useEdges) ngb += edges[2][x][y][z-1]*mems[x][y][z-1][m]*mems[x][y][z-1][m];
							else ngb += mems[x][y][z-1][m]*mems[x][y][z-1][m];
							neighbors ++;
						}
						if (neighbors>0.0) num = num + smoothing*ngb/neighbors;
					}
					// invert the result
					if (num>ZERO) num = 1.0f/num;
					else num = INF;

					//mems[k][x][y][z] = num;
					mems[x][y][z][k] = num;
					den += num;
				}

				// normalization
				for (k=0;k<classes;k++) {
					//mems[k][x][y][z] = mems[k][x][y][z]/den;
					mems[x][y][z][k] = mems[x][y][z][k]/den;

                    // compute the maximum distance
                    //dist = Math.abs(mems[k][x][y][z]-prev[k]);
                    dist = Math.abs(mems[x][y][z][k]-prev[k]);
                    if (dist > distance) distance = dist;
                }
			} else {
				for (k=0;k<classes;k++) 
					//mems[k][x][y][z] = 0.0f;
					mems[x][y][z][k] = 0.0f;
			}
		}
        if (debug) System.out.print("inner loop time: (milliseconds): " + (System.currentTimeMillis()-inner_loop_time) +"\n"); 

        return distance;
    } // computeMemberships
    
    /** 
	 *  compute the FCM membership functions given the centroids
	 *	with the different options (outliers, field, edges, MRF)
	 */
    final public float computeGeneralMemberships() {
        float distance,dist;
        int x,y,z,k,m;
        int progress, mod;
        long inner_loop_time;
        float den,num;
        float neighbors, ngb;
        
        distance = 0.0f;
		progress = 0;
        mod = nx*ny*nz/20; // mod is 1 percent of length

        inner_loop_time = System.currentTimeMillis();
        for (x=1;x<nx-1;x++) for (y=1;y<ny-1;y++) for (z=1;z<nz-1;z++) {
			progress++;
			if ( (verbose) && (progress%mod==0) )
                if (progressBar!=null) progressBar.updateValue(Math.round( (float)progress/(float)mod*5.0f),false);

			if ( mask[x][y][z] ) {
				den = 0;
				// remember the previous values
				for (k=0;k<classes;k++) 
					prev[k] = mems[x][y][z][k];

				for (k=0;k<classes;k++) {
					
					// data term
					if (k<clusters) {
						if (correctionType==InhomogeneityCorrection.NONE) 
							num = (image[x][y][z]-centroids[k])*(image[x][y][z]-centroids[k]);
						else if (correctionType==InhomogeneityCorrection.IMAGE) 
							num = (field[x][y][z]*image[x][y][z]-centroids[k])*(field[x][y][z]*image[x][y][z]-centroids[k]);
						else if (correctionType==InhomogeneityCorrection.CENTROIDS) 
							num = (image[x][y][z]-field[x][y][z]*centroids[k])*(image[x][y][z]-field[x][y][z]*centroids[k]);
						else if (correctionType==InhomogeneityCorrection.SEPARATE)
								num = (image[x][y][z]-fields[x][y][z][k]*centroids[k])*(image[x][y][z]-fields[x][y][z][k]*centroids[k]);
						else num = INF;
					} else {
						num = outlier*outlier;
					}
					
					// spatial smoothing
					if (smoothing > 0.0f) { 
						ngb = 0.0f;  
						neighbors = 0.0f;
						// case by case	: X+
						if (mask[x+1][y][z]) for (m=0;m<classes;m++) if (m!=k) {
							if (useEdges) ngb += edges[0][x][y][z]*power.lookup(mems[x+1][y][z][m],fuzziness);
							else ngb += power.lookup(mems[x+1][y][z][m],fuzziness);
							neighbors ++;
						}
						// case by case	: X-
						if (mask[x-1][y][z]) for (m=0;m<classes;m++) if (m!=k) {
							if (useEdges) ngb += edges[0][x-1][y][z]*power.lookup(mems[x-1][y][z][m],fuzziness);
							else ngb += power.lookup(mems[x-1][y][z][m],fuzziness);
							neighbors ++;
						}
						// case by case	: Y+
						if (mask[x][y+1][z]) for (m=0;m<classes;m++) if (m!=k) {
							if (useEdges) ngb += edges[1][x][y][z]*power.lookup(mems[x][y+1][z][m],fuzziness);
							else ngb += power.lookup(mems[x][y+1][z][m],fuzziness);
							neighbors ++;
						}
						// case by case	: Y-
						if (mask[x][y-1][z]) for (m=0;m<classes;m++) if (m!=k) {
							if (useEdges) ngb += edges[1][x][y-1][z]*power.lookup(mems[x][y-1][z][m],fuzziness);
							else ngb += power.lookup(mems[x][y-1][z][m],fuzziness);
							neighbors ++;
						}
						// case by case	: Z+
						if (mask[x][y][z+1]) for (m=0;m<classes;m++) if (m!=k) {
							if (useEdges) ngb += edges[2][x][y][z]*power.lookup(mems[x][y][z+1][m],fuzziness);
							else ngb += power.lookup(mems[x][y][z+1][m],fuzziness);
							neighbors ++;
						}
						// case by case	: Z-
						if (mask[x][y][z-1]) for (m=0;m<classes;m++) if (m!=k) {
							if (useEdges) ngb += edges[2][x][y][z-1]*power.lookup(mems[x][y][z-1][m],fuzziness);
							else ngb += power.lookup(mems[x][y][z-1][m],fuzziness);
							neighbors ++;
						}
						if (neighbors>0.0) num = num + smoothing*ngb/neighbors;
					}
					// invert the result
					if (num>ZERO) num = (float)invpower.lookup(num,1.0f/(1.0f-fuzziness) );
					else num = INF;

					mems[x][y][z][k] = num;
					den += num;
				}

				// normalization
				for (k=0;k<classes;k++) {
					mems[x][y][z][k] = mems[x][y][z][k]/den;

                    // compute the maximum distance
                   dist = Math.abs(mems[x][y][z][k]-prev[k]);
                    if (dist > distance) distance = dist;
                }
			} else {
				for (k=0;k<classes;k++) 
					mems[x][y][z][k] = 0.0f;
			}
		}
        if (debug) System.out.print("inner loop time: (milliseconds): " + (System.currentTimeMillis()-inner_loop_time) +"\n"); 

        return distance;
    } // computeGeneralMemberships
    
    /**
	 * compute the centroids given the membership functions
	 */
    final public void computeCentroids() {
        int x,y,z,k;
		float num,den;
		
		if (fuzziness!=2) {
			computeGeneralCentroids();
			return;
		}
        
		for (k=0;k<clusters;k++) {
			if (distMode==RAYLEIGH2 && k==0) {
				num = 0;
				den = 0;
				for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
					if (mask[x][y][z]) {
						if (correctionType==InhomogeneityCorrection.NONE) {
							num += mems[x][y][z][k]*mems[x][y][z][k]*image[x][y][z]*image[x][y][z];
							den += mems[x][y][z][k]*mems[x][y][z][k];
						} else if (correctionType==InhomogeneityCorrection.IMAGE) { 
							num += mems[x][y][z][k]*mems[x][y][z][k]*field[x][y][z]*image[x][y][z]*field[x][y][z]*image[x][y][z];
							den += mems[x][y][z][k]*mems[x][y][z][k];
						} else if (correctionType==InhomogeneityCorrection.CENTROIDS) { 
							num += mems[x][y][z][k]*mems[x][y][z][k]*field[x][y][z]*image[x][y][z]*field[x][y][z]*image[x][y][z];
							den += mems[x][y][z][k]*mems[x][y][z][k]*field[x][y][z]*field[x][y][z];
						} else if (correctionType==InhomogeneityCorrection.SEPARATE) { 
							num += mems[x][y][z][k]*mems[x][y][z][k]*fields[x][y][z][k]*image[x][y][z]*fields[x][y][z][k]*image[x][y][z];
							den += mems[x][y][z][k]*mems[x][y][z][k]*fields[x][y][z][k]*fields[x][y][z][k];
						}
					}
				}
				if (den>0.0f && num>0.0f) {
					centroids[k] = (float)Math.sqrt(num/(2.0f*den));
				} else {
					centroids[k] = 0.0f;
				}
			} else {
				num = 0;
				den = 0;
				for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
					if (mask[x][y][z]) {
						if (correctionType==InhomogeneityCorrection.NONE) {
							num += mems[x][y][z][k]*mems[x][y][z][k]*image[x][y][z];
							den += mems[x][y][z][k]*mems[x][y][z][k];
						} else if (correctionType==InhomogeneityCorrection.IMAGE) { 
							num += mems[x][y][z][k]*mems[x][y][z][k]*field[x][y][z]*image[x][y][z];
							den += mems[x][y][z][k]*mems[x][y][z][k];
						} else if (correctionType==InhomogeneityCorrection.CENTROIDS) { 
							num += mems[x][y][z][k]*mems[x][y][z][k]*field[x][y][z]*image[x][y][z];
							den += mems[x][y][z][k]*mems[x][y][z][k]*field[x][y][z]*field[x][y][z];
						} else if (correctionType==InhomogeneityCorrection.SEPARATE) { 
							num += mems[x][y][z][k]*mems[x][y][z][k]*fields[x][y][z][k]*image[x][y][z];
							den += mems[x][y][z][k]*mems[x][y][z][k]*fields[x][y][z][k]*fields[x][y][z][k];
						}
					}
				}
				if (den>0.0) {
					centroids[k] = num/den;
				} else {
					centroids[k] = 0.0f;
				}
			}				
        }
        if (verbose) {
			MedicUtilPublic.displayMessage("centroids: ("+centroids[0]);
			for (k=1;k<clusters;k++) MedicUtilPublic.displayMessage(", "+centroids[k]);
			MedicUtilPublic.displayMessage(")\n");
		}   
        return;
    } // computeCentroids
    
    /**
	 * compute the centroids given the membership functions
	 */
    final public void computeImageVariance() {
        int x,y,z,k;
		float num,den;
		
		num = 0.0f;
		den = 0.0f;
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
			for (k=0;k<clusters;k++) {
				if (mask[x][y][z]) {
					if (correctionType==InhomogeneityCorrection.NONE) {
						num += mems[x][y][z][k]*mems[x][y][z][k]*(image[x][y][z]-centroids[k])*(image[x][y][z]-centroids[k]);
						den += mems[x][y][z][k]*mems[x][y][z][k];
					} else if (correctionType==InhomogeneityCorrection.IMAGE) { 
						num += mems[x][y][z][k]*mems[x][y][z][k]*(field[x][y][z]*image[x][y][z]-centroids[k])*(field[x][y][z]*image[x][y][z]-centroids[k]);
						den += mems[x][y][z][k]*mems[x][y][z][k];
					} else if (correctionType==InhomogeneityCorrection.CENTROIDS) { 
						num += mems[x][y][z][k]*mems[x][y][z][k]*(field[x][y][z]*image[x][y][z]-centroids[k])*(field[x][y][z]*image[x][y][z]-centroids[k]);
						den += mems[x][y][z][k]*mems[x][y][z][k]*field[x][y][z]*field[x][y][z];
					} else if (correctionType==InhomogeneityCorrection.SEPARATE) { 
						num += mems[x][y][z][k]*mems[x][y][z][k]*(fields[x][y][z][k]*image[x][y][z]-centroids[k])*(fields[x][y][z][k]*image[x][y][z]-centroids[k]);
						den += mems[x][y][z][k]*mems[x][y][z][k]*fields[x][y][z][k]*fields[x][y][z][k];
					}
				}
			}
		}
		if (den>0.0) {
			imgvar = num/den;
		} else {
			imgvar = 1.0f;
		}
        if (verbose) {
			MedicUtilPublic.displayMessage("image variance: ("+imgvar+")\n");
		}   
        return;
    } // computeImageVariance
    
    /**
	 * compute the centroids given the membership functions
	 * for any value of the fuzziness
	 */
    final public void computeGeneralCentroids() {
        int x,y,z,k;
		float num,den;
		float val;
		
		for (k=0;k<clusters;k++) {
            num = 0;
            den = 0;
            for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				if (mask[x][y][z]) {
					val = (float)power.lookup(mems[x][y][z][k],fuzziness);
					if (correctionType==InhomogeneityCorrection.NONE) {
						num += val*image[x][y][z];
						den += val;
					} else if (correctionType==InhomogeneityCorrection.IMAGE) { 
						num += val*field[x][y][z]*image[x][y][z];
						den += val;
					} else if (correctionType==InhomogeneityCorrection.CENTROIDS) { 
						num += val*field[x][y][z]*image[x][y][z];
						den += val*field[x][y][z]*field[x][y][z];
					} else if (correctionType==InhomogeneityCorrection.SEPARATE) { 
						num += val*fields[x][y][z][k]*image[x][y][z];
						den += val*fields[x][y][z][k]*fields[x][y][z][k];
					}
				}
            }
            if (den>0.0) {
                centroids[k] = num/den;
            } else {
                centroids[k] = 0.0f;
            }
        }
        if (verbose) {
			MedicUtilPublic.displayMessage("centroids: ("+centroids[0]);
			for (k=1;k<clusters;k++) MedicUtilPublic.displayMessage(", "+centroids[k]);
			MedicUtilPublic.displayMessage(")\n");
		}   
        return;
    } // computeCentroids
    
    /** 
	 *  compute the FCM energy for the data and smoothness errors,
	 *	and produces a smoothing coefficient beta = factor * data / smoothness.
	 */
    final public float computeEnergyFactor(float factor) {
        float Edata,Esmooth;
        int x,y,z,k,m;
        float num;
        float neighbors, ngb;
        
        if (fuzziness!=2) {
			return computeGeneralEnergyFactor(factor);
		}
       
		Edata = 0.0f; Esmooth = 0.0f;
		for (x=1;x<nx-1;x++) for (y=1;y<ny-1;y++) for (z=1;z<nz-1;z++) {
			if ( mask[x][y][z] ) {
				for (k=0;k<classes;k++) {
					
					// data term
					if (k<clusters) {
						if (correctionType==InhomogeneityCorrection.NONE)
							num = (image[x][y][z]-centroids[k])*(image[x][y][z]-centroids[k]);
						else if (correctionType==InhomogeneityCorrection.IMAGE)
							num = (field[x][y][z]*image[x][y][z]-centroids[k])*(field[x][y][z]*image[x][y][z]-centroids[k]);
						else if (correctionType==InhomogeneityCorrection.CENTROIDS)
							num = (image[x][y][z]-field[x][y][z]*centroids[k])*(image[x][y][z]-field[x][y][z]*centroids[k]);
						else if (correctionType==InhomogeneityCorrection.SEPARATE)
							num = (image[x][y][z]-fields[x][y][z][k]*centroids[k])*(image[x][y][z]-fields[x][y][z][k]*centroids[k]);
						else
							num = INF;
					} else {
						num = outlier*outlier;
					}
					Edata += mems[x][y][z][k]*mems[x][y][z][k]*num;
					
					// spatial smoothing
					if (smoothing > 0.0f) { 
						ngb = 0.0f;  
						neighbors = 0.0f;
						// case by case	: X+
						if (mask[x+1][y][z]) for (m=0;m<classes;m++) if (m!=k) {
							//if (useEdges) ngb += edges[0][x][y][z]*mems[m][x+1][y][z]*mems[m][x+1][y][z];
							//else ngb += mems[m][x+1][y][z]*mems[m][x+1][y][z];
							if (useEdges) ngb += edges[0][x][y][z]*mems[x+1][y][z][m]*mems[x+1][y][z][m];
							else ngb += mems[x+1][y][z][m]*mems[x+1][y][z][m];
							neighbors ++;
						}
						// case by case	: X-
						if (mask[x-1][y][z]) for (m=0;m<classes;m++) if (m!=k) {
							//if (useEdges) ngb += edges[0][x-1][y][z]*mems[m][x-1][y][z]*mems[m][x-1][y][z];
							//else ngb += mems[m][x-1][y][z]*mems[m][x-1][y][z];
							if (useEdges) ngb += edges[0][x-1][y][z]*mems[x-1][y][z][m]*mems[x-1][y][z][m];
							else ngb += mems[x-1][y][z][m]*mems[x-1][y][z][m];
							neighbors ++;
						}
						// case by case	: Y+
						if (mask[x][y+1][z]) for (m=0;m<classes;m++) if (m!=k) {
							//if (useEdges) ngb += edges[1][x][y][z]*mems[m][x][y+1][z]*mems[m][x][y+1][z];
							//else ngb += mems[m][x][y+1][z]*mems[m][x][y+1][z];
							if (useEdges) ngb += edges[1][x][y][z]*mems[x][y+1][z][m]*mems[x][y+1][z][m];
							else ngb += mems[x][y+1][z][m]*mems[x][y+1][z][m];
							neighbors ++;
						}
						// case by case	: Y-
						if (mask[x][y-1][z]) for (m=0;m<classes;m++) if (m!=k) {
							//if (useEdges) ngb += edges[1][x][y-1][z]*mems[m][x][y-1][z]*mems[m][x][y-1][z];
							//else ngb += mems[m][x][y-1][z]*mems[m][x][y-1][z];
							if (useEdges) ngb += edges[1][x][y-1][z]*mems[x][y-1][z][m]*mems[x][y-1][z][m];
							else ngb += mems[x][y-1][z][m]*mems[x][y-1][z][m];
							neighbors ++;
						}
						// case by case	: Z+
						if (mask[x][y][z+1]) for (m=0;m<classes;m++) if (m!=k) {
							//if (useEdges) ngb += edges[2][x][y][z]*mems[m][x][y][z+1]*mems[m][x][y][z+1];
							//else ngb += mems[m][x][y][z+1]*mems[m][x][y][z+1];
							if (useEdges) ngb += edges[2][x][y][z]*mems[x][y][z+1][m]*mems[x][y][z+1][m];
							else ngb += mems[x][y][z+1][m]*mems[x][y][z+1][m];
							neighbors ++;
						}
						// case by case	: Z-
						if (mask[x][y][z-1]) for (m=0;m<classes;m++) if (m!=k) {
							//if (useEdges) ngb += edges[2][x][y][z-1]*mems[m][x][y][z-1]*mems[m][x][y][z-1];
							//else ngb += mems[m][x][y][z-1]*mems[m][x][y][z-1];
							if (useEdges) ngb += edges[2][x][y][z-1]*mems[x][y][z-1][m]*mems[x][y][z-1][m];
							else ngb += mems[x][y][z-1][m]*mems[x][y][z-1][m];
							neighbors ++;
						}
						if (neighbors>0.0) Esmooth += 0.5f*mems[x][y][z][k]*mems[x][y][z][k]*ngb/neighbors;
					}
				}
			}
		}
		//System.out.println("Ed:"+Edata+ "Es:"+Esmooth);
        return factor*Edata/Esmooth;
    } // computeEnergyFactor
    
    /** 
	 *  compute the FCM energy for the data and smoothness errors,
	 *	and produces a smoothing coefficient beta = factor * data / smoothness.
	 */
    final public float computeGeneralEnergyFactor(float factor) {
        float Edata,Esmooth;
        int x,y,z,k,m;
        float num;
        float neighbors, ngb;
        
        Edata = 0.0f; Esmooth = 0.0f;
		for (x=1;x<nx-1;x++) for (y=1;y<ny-1;y++) for (z=1;z<nz-1;z++) {
			if ( mask[x][y][z] ) {
				for (k=0;k<classes;k++) {
					
					// data term
					if (k<clusters) {
						if (correctionType==InhomogeneityCorrection.NONE)
							num = (image[x][y][z]-centroids[k])*(image[x][y][z]-centroids[k]);
						else if (correctionType==InhomogeneityCorrection.IMAGE)
							num = (field[x][y][z]*image[x][y][z]-centroids[k])*(field[x][y][z]*image[x][y][z]-centroids[k]);
						else if (correctionType==InhomogeneityCorrection.CENTROIDS)
							num = (image[x][y][z]-field[x][y][z]*centroids[k])*(image[x][y][z]-field[x][y][z]*centroids[k]);
						else if (correctionType==InhomogeneityCorrection.SEPARATE)
							num = (image[x][y][z]-fields[x][y][z][k]*centroids[k])*(image[x][y][z]-fields[x][y][z][k]*centroids[k]);
						else
							num = INF;
					} else {
						num = outlier*outlier;
					}
					Edata += power.lookup(mems[x][y][z][k],fuzziness)*num;
					
					// spatial smoothing
					if (smoothing > 0.0f) { 
						ngb = 0.0f;  
						neighbors = 0.0f;
						// case by case	: X+
						if (mask[x+1][y][z]) for (m=0;m<classes;m++) if (m!=k) {
							if (useEdges) ngb += edges[0][x][y][z]*power.lookup(mems[x+1][y][z][m],fuzziness);
							else ngb += power.lookup(mems[x+1][y][z][m],fuzziness);
							neighbors ++;
						}
						// case by case	: X-
						if (mask[x-1][y][z]) for (m=0;m<classes;m++) if (m!=k) {
							if (useEdges) ngb += edges[0][x-1][y][z]*power.lookup(mems[x-1][y][z][m],fuzziness);
							else ngb += power.lookup(mems[x-1][y][z][m],fuzziness);
							neighbors ++;
						}
						// case by case	: Y+
						if (mask[x][y+1][z]) for (m=0;m<classes;m++) if (m!=k) {
							if (useEdges) ngb += edges[1][x][y][z]*power.lookup(mems[x][y+1][z][m],fuzziness);
							else ngb += power.lookup(mems[x][y+1][z][m],fuzziness);
							neighbors ++;
						}
						// case by case	: Y-
						if (mask[x][y-1][z]) for (m=0;m<classes;m++) if (m!=k) {
							if (useEdges) ngb += edges[1][x][y-1][z]*power.lookup(mems[x][y-1][z][m],fuzziness);
							else ngb += power.lookup(mems[x][y-1][z][m],fuzziness);
							neighbors ++;
						}
						// case by case	: Z+
						if (mask[x][y][z+1]) for (m=0;m<classes;m++) if (m!=k) {
							if (useEdges) ngb += edges[2][x][y][z]*power.lookup(mems[x][y][z+1][m],fuzziness);
							else ngb += power.lookup(mems[x][y][z+1][m],fuzziness);
							neighbors ++;
						}
						// case by case	: Z-
						if (mask[x][y][z-1]) for (m=0;m<classes;m++) if (m!=k) {
							if (useEdges) ngb += edges[2][x][y][z-1]*power.lookup(mems[x][y][z-1][m],fuzziness);
							else ngb += power.lookup(mems[x][y][z-1][m],fuzziness);
							neighbors ++;
						}
						if (neighbors>0.0) Esmooth += 0.5f*power.lookup(mems[x][y][z][k],fuzziness)*ngb/neighbors;
					}
				}
			}
		}
		//System.out.println("Ed:"+Edata+ "Es:"+Esmooth);
        return factor*Edata/Esmooth;
    } // computeEnergyFactor
    
	/** 
	 *	returns the hard classification (max_{clusters}(Mems)) 
	 */
	public final byte[][][] exportHardClassification() {
		int 	x,y,z,t,k,best;
		byte[][][]	classification = new byte[nx][ny][nz];
        float bestmem;
        
		for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {                    
			if (mask[x][y][z]) {
				best = -1;bestmem = 0.0f;
				for (k=0;k<classes;k++) {
					//if (mems[k][x][y][z] > bestmem) {
					if (mems[x][y][z][k] > bestmem) {
						best = k;
						//bestmem = mems[k][x][y][z];
						bestmem = mems[x][y][z][k];
					}
				}   
				if (best==-1)
					classification[x][y][z] = 0;
				else
					classification[x][y][z] = (byte)(best+1);
			} else {
				classification[x][y][z] = 0;
			}
		}
		return classification;
	} // exportHardClassification

	/** 
	 *	export membership functions 
	 */
	public final float[][][][] exportMemberships() {
		int 	x,y,z,k,t;
		float[][][][]	Mems = new float[classes][nx][ny][nz];
		
        for (k=0;k<classes;k++) {
            for (x=0;x<nx;x++) for (y=0;y<ny;y++) for (z=0;z<nz;z++) {
				//Mems[k][x][y][z] = mems[k][x][y][z];
				Mems[k][x][y][z] = mems[x][y][z][k];
			}
		}
		return Mems;
	} // exportMemberships
	
	/** 
	 *	create ids for the centroids based on ordering
	 */
	public final int[] computeCentroidOrder() {
		int 	k,l;
		int[]	id = new int[classes+1];
		int		lowest;
		float[] cent = new float[clusters];
		
		// copy the centroids
        for (k=0;k<clusters;k++) 
			cent[k] = centroids[k];
		
		// add the zero
		id[0] = 0;

		// order them from smallest to largest
		for (k=0;k<clusters;k++) {
			lowest = 0;
			for (l=1;l<clusters;l++) {
				if (cent[l] < cent[lowest]) {
					lowest = l;
				}
			}
			id[k+1] = lowest+1;
			cent[lowest] = INF;
		}
		// keep order for other class types (outliers, etc)
		for (k=clusters;k<classes;k++)
			id[k+1] = k+1;
		
        if (debug) {
			MedicUtilPublic.displayMessage("ordering: ("+id[0]);
			for (k=0;k<classes;k++) MedicUtilPublic.displayMessage(", "+id[k+1]);
			MedicUtilPublic.displayMessage(")\n");
		}   
 		return id;
	} // computeCentroidOrder

	private final float rayleighDistance(float val, float mean) {
		// not properly normalized ?
		if (val<=0) return INF;
		else return (float)Math.PI/4.0f*(val*val/mean*mean-1) - (float)Math.log(val/mean);
	}

	private final float rayleighDirectDistance(float val, float sig) {
		// using the distribution parameter rather than mean and variance
		if (val<=0) return INF;
		else return val*val/sig*sig - 2.0f*(float)Math.log(val/sig) + (float)Math.log(Math.PI/2.0f) - (float)Math.PI/2.0f;
	}

	private final float ricianApproxDistance(float val, float mean) {
		// using the distribution parameter rather than mean and variance
		if (val<=0) return INF;
		else return (val-mean)*(val-mean) - imgvar*(float)Math.log(val/mean);
	}
}
