package edu.jhmi.rad.medic.visualization.models;

import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.GL;
import javax.media.opengl.GL2;

import java.awt.Color;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.util.Vector;

import edu.jhmi.rad.medic.visualization.framework.DrawParam;
import edu.jhmi.rad.medic.visualization.framework.MedicDrawable;
import edu.jhmi.rad.medic.visualization.framework.OpenGLHelper;
import edu.jhmi.rad.medic.visualization.primitives.DrawableBoundingBox;
import edu.jhmi.rad.medic.visualization.primitives.Line;
import edu.jhmi.rad.medic.visualization.structures.AdjacencyList;
import edu.jhmi.rad.medic.visualization.structures.LookUpTable;
import edu.jhmi.rad.medic.visualization.structures.ScalarFloatData;
import edu.jhmi.rad.medic.visualization.structures.Triangle;
import javax.swing.JPanel;
import javax.vecmath.Vector3f;
import javax.vecmath.Point3f;

import com.jogamp.common.nio.Buffers;


/**
 * @author Bhaskar Kishore (bhaskar@jhu.edu)
 */
public class TriangleMesh extends MedicDrawable {
	private IntBuffer VBOIDBuffer;
	private FloatBuffer vertexBuffer;
	private FloatBuffer normalBuffer;
	private IntBuffer triangleBuffer;
	private ScalarFloatData scalarBuffer;
	private FloatBuffer colorBuffer;
	private int polygonCount;
	private boolean drawEdges = false;
	
	private AdjacencyList adjacencyList = null;
	
	private boolean useColor = false;
	
	private int vertexInsertCounter;
	private int polygonInsertCounter;
	
	//Constants
	public static final int FILE_ASCII	= 1;			
	public static final int FILE_BINARY	= 2;
		
	public TriangleMesh() {
		vertexInsertCounter = 0;
		polygonInsertCounter = 0;
		scalarBuffer = null;
		m_BoundingBox = new DrawableBoundingBox();
		System.out.println("TMESH NIO");
	}
		
	protected void finalize() {		
		dispose();
	}
	
	public void allocateBuffers(int vertexCount, int triangleCount) {
		this.allocatePolygonBuffer(triangleCount);
		this.allocateVertexBuffer(vertexCount);
	}
	
	public void dispose() {
		System.out.println("Surface `" + getName() + "` freeing up memory ... ");		
		
		vertexBuffer.clear();
		normalBuffer.clear();
		triangleBuffer.clear();
		
		if(scalarBuffer != null)
			scalarBuffer.dispose();
		
		scalarBuffer = null;		
		vertexBuffer = null;
		normalBuffer = null;
		triangleBuffer = null;

		System.runFinalization();
		System.gc();
	}
			
	public void initVBO(GLAutoDrawable glAutoDrawable) {
		GL2 gl = glAutoDrawable.getGL().getGL2();
		VBOIDBuffer = Buffers.newDirectIntBuffer(4);
		gl.glGenBuffers(4, VBOIDBuffer);
		vertexBuffer.rewind();
		normalBuffer.rewind();
		triangleBuffer.rewind();
		VBOIDBuffer.rewind();
		gl.glBindBuffer(GL.GL_ARRAY_BUFFER, VBOIDBuffer.get(0));
		gl.glBufferData(GL.GL_ARRAY_BUFFER, this.getVertexCount() * 3 * Buffers.SIZEOF_FLOAT, vertexBuffer, GL.GL_STATIC_DRAW);
		gl.glBindBuffer(GL.GL_ARRAY_BUFFER, VBOIDBuffer.get(1));
		gl.glBufferData(GL.GL_ARRAY_BUFFER, this.getVertexCount() * 3 * Buffers.SIZEOF_FLOAT, normalBuffer, GL.GL_STATIC_DRAW);
		gl.glBindBuffer(GL.GL_ELEMENT_ARRAY_BUFFER, VBOIDBuffer.get(2));
		gl.glBufferData(GL.GL_ELEMENT_ARRAY_BUFFER, this.getPolygonCount() * 3 * Buffers.SIZEOF_INT, triangleBuffer, GL.GL_STATIC_DRAW);
		if(this.useColor) {
			colorBuffer.rewind();
			gl.glBindBuffer(GL.GL_ARRAY_BUFFER, VBOIDBuffer.get(3));
			gl.glBufferData(GL.GL_ARRAY_BUFFER, this.getVertexCount() * 3 * Buffers.SIZEOF_FLOAT, colorBuffer, GL.GL_STATIC_DRAW);
		}
		
		
		System.gc();
		this.m_IsInitialized = true;
	}
	
	public void drawGL(GLAutoDrawable glAutoDrawable) {
		GL2 gl = glAutoDrawable.getGL().getGL2();
		// Color Hack
		if(useColor) {
			float mat_ambient[] = { 0.0f, 0.0f, 0.0f, 0.15f };
			float mat_specular[] = { 0.0f, 0.0f, 0.0f, 0.4f };
		    float mat_shininess[] = { 2.0f };
		    gl.glMaterialfv(GL.GL_FRONT_AND_BACK, GL2.GL_AMBIENT, mat_ambient, 0);
		    gl.glMaterialfv(GL.GL_FRONT_AND_BACK, GL2.GL_SPECULAR, mat_specular, 0);
		    gl.glMaterialfv(GL.GL_FRONT_AND_BACK, GL2.GL_SHININESS, mat_shininess, 0);
		}	
		
		if(OpenGLHelper.isVBOSupported()) {	
			if(this.isInit() == false)
				initVBO(glAutoDrawable);
	
			gl.glEnableClientState(GL2.GL_NORMAL_ARRAY);
			gl.glEnableClientState(GL2.GL_VERTEX_ARRAY);
			if(useColor) {
				gl.glEnable ( GL2.GL_COLOR_MATERIAL ) ;
				gl.glEnableClientState(GL2.GL_COLOR_ARRAY);
			}
			
			gl.glBindBuffer(GL.GL_ARRAY_BUFFER, VBOIDBuffer.get(0));
			gl.glBindBuffer(GL.GL_ELEMENT_ARRAY_BUFFER, VBOIDBuffer.get(2));
	
			gl.glVertexPointer(3, GL.GL_FLOAT, 0, 0);
			
			gl.glBindBuffer(GL.GL_ARRAY_BUFFER, VBOIDBuffer.get(1));
			gl.glNormalPointer(GL.GL_FLOAT, 0, 0);
			
			if(useColor) {
				gl.glBindBuffer(GL.GL_ARRAY_BUFFER, VBOIDBuffer.get(3));
				gl.glColorPointer(3, GL.GL_FLOAT, 0, 0);
			}
			
			if(!drawEdges)
				gl.glPolygonMode(GL.GL_FRONT_AND_BACK, GL2.GL_FILL);
			else
				gl.glPolygonMode(GL.GL_FRONT_AND_BACK, GL2.GL_LINE);
			
			gl.glDrawElements(GL.GL_TRIANGLES, getPolygonCount() * 3, GL.GL_UNSIGNED_INT, 0);
			
			gl.glDisableClientState(GL2.GL_VERTEX_ARRAY);
			gl.glDisableClientState(GL2.GL_NORMAL_ARRAY);
			if(useColor) {
				gl.glDisableClientState(GL2.GL_COLOR_ARRAY);
				gl.glDisable(GL2.GL_COLOR_MATERIAL);
			}
			
			gl.glBindBuffer(GL2.GL_ARRAY_BUFFER, 0);
			gl.glBindBuffer(GL2.GL_ELEMENT_ARRAY_BUFFER, 0);
		}
		else {
			/*
			 * Vertex Array Method
			 */
			gl.glEnableClientState(GL2.GL_VERTEX_ARRAY);
			gl.glEnableClientState(GL2.GL_NORMAL_ARRAY);
			if(useColor) {
				gl.glEnableClientState(GL2.GL_COLOR_ARRAY);
			}
			vertexBuffer.rewind();
			normalBuffer.rewind();
			triangleBuffer.rewind();
			gl.glVertexPointer(3, GL.GL_FLOAT, 0, vertexBuffer);
			gl.glNormalPointer(GL.GL_FLOAT, 0, normalBuffer);
			if(useColor) {
				colorBuffer.rewind();
				gl.glEnable (GL2.GL_COLOR_MATERIAL);
				gl.glColorPointer(3, GL.GL_FLOAT, 0, colorBuffer);				
			}
			if(!drawEdges)
				gl.glPolygonMode(GL.GL_FRONT_AND_BACK, GL2.GL_FILL);
			else
				gl.glPolygonMode(GL.GL_FRONT_AND_BACK, GL2.GL_LINE);
			gl.glDrawElements(GL.GL_TRIANGLES, this.getPolygonCount() * 3, GL.GL_UNSIGNED_INT, triangleBuffer);
			
			gl.glDisableClientState(GL2.GL_VERTEX_ARRAY);
			gl.glDisableClientState(GL2.GL_NORMAL_ARRAY);
			if(useColor) {
				gl.glDisable(GL2.GL_COLOR_MATERIAL);
				gl.glDisableClientState(GL2.GL_COLOR_ARRAY);
			}
		}

	}
		
	/*
	 * Computes the bounding box for this surface.
	 */
	public void computeBoundingBox() {		
		int vnum = getVertexCount();	
		if(vnum < 3)
			return;
					
		Point3f m_BoxMax = new Point3f(Float.MIN_VALUE, Float.MIN_VALUE, Float.MIN_VALUE);
		Point3f m_BoxMin = new Point3f(Float.MAX_VALUE, Float.MAX_VALUE, Float.MAX_VALUE);
				
		for(int i = 0; i  < vnum; ++i) {			
			Point3f temp = new Point3f(getVertex(i));
			
			if(m_BoxMin.x > temp.x)
				m_BoxMin.x = temp.x;
			if(m_BoxMin.y > temp.y)
				m_BoxMin.y = temp.y;
			if(m_BoxMin.z > temp.z)
				m_BoxMin.z = temp.z;
				
			if(m_BoxMax.x < temp.x)
				m_BoxMax.x = temp.x;
			if(m_BoxMax.y < temp.y)
				m_BoxMax.y = temp.y;
			if(m_BoxMax.z < temp.z)
				m_BoxMax.z = temp.z;
			
			temp = null;
		}		
			
		m_BoundingBox.set(m_BoxMin, m_BoxMax);
		
		m_BoxMin = m_BoxMax = null;
	}
	
	/*
	 * Computes the per-vertex normals. Simple averaging of face normals is used.
	 */
	public void computeNormals() {		
		int vnum = getVertexCount();
		int pnum = getPolygonCount();
		Vector<Vector3f> normals = new Vector<Vector3f>(vnum);
		Vector3f avec, bvec, normal;
		Point3f p0, p1, p2;
		int[] polygon;		
		
		p0		= new Point3f();
		p1		= new Point3f();
		p2		= new Point3f();
		avec	= new Vector3f();
		bvec	= new Vector3f();
		normal	= new Vector3f();
		
		System.out.println("Computing Per-Vertex Normals ... ");
		
		for(int i = 0; i < vnum; ++i)
			normals.add(new Vector3f(0.0f, 0.0f, 0.0f));	
		
		for(int i = 0; i < pnum; ++i) {
			polygon = getPolygon(i);			
			p0.set(getVertex(polygon[0]));			
			p1.set(getVertex(polygon[1]));
			p2.set(getVertex(polygon[2]));
			
			avec.sub(p1, p0);
			bvec.sub(p2, p0);
			normal.cross(avec, bvec);
		
			normals.elementAt(polygon[0]).add(normal);
			normals.elementAt(polygon[1]).add(normal);
			normals.elementAt(polygon[2]).add(normal);
			
			polygon = null;
		}
		
		for(int i = 0; i < vnum; ++i) {
			normal = normals.get(i);
			normal.normalize();
			normalBuffer.put(normal.getX());
			normalBuffer.put(normal.getY());
			normalBuffer.put(normal.getZ());
		}
	
		p0 = p1 = p2 = null;
		avec = bvec = normal = null;		
	}
	
	/*
	 * Gets the number of polygons in the surface
	 *	@return		Number of polygons.
	 */
	public int getPolygonCount() {
		return polygonCount;
	}
	
	/*
	 * Gets the number of vertices in the surface
	 *	@return		Number of vertices
	 */
	public int getVertexCount() {
		return vertexBuffer.capacity() / 3;
	}
		
	/*
	 * Allocates buffers for vertex and normal data
	 * 	@param	vCount	Number of vertices to create buffer space for.
	 */
	public void allocateVertexBuffer(int vCount) {
		vertexBuffer = Buffers.newDirectFloatBuffer(vCount * 3);
		normalBuffer = Buffers.newDirectFloatBuffer(vCount * 3);
	}
	
	/*
	 * Adds a vertex to the surface
	 *	@param	v	Vertex position array
	 */
	public void addVertex(float[] v) {
		if(v == null)
			return;
		if(v.length != 3)
			return;
		if(vertexInsertCounter < getVertexCount()) {
			vertexBuffer.put(v[0]);
			vertexBuffer.put(v[1]);
			vertexBuffer.put(v[2]);
			vertexInsertCounter++;
		}
	}
	
	public void addVertex(float x, float y, float z) {
		if(vertexInsertCounter < getVertexCount()) {
			vertexBuffer.put(x);
			vertexBuffer.put(y);
			vertexBuffer.put(z);
			vertexInsertCounter++;
		}
	}
	
	/*
	 * Returns the vertex using the specified index.
	 * 	@param	index	index of vertex to return
	 * 	@return			If a valid index is given the float[3]
	 * 					containing x,y,z values for vertex else null;
	 */
	public float[] getVertex(int index) {
		if(index >= 0 && index < getVertexCount()){
			float [] vertex = new float[3];
			vertex[0] = vertexBuffer.get(index * 3);
			vertex[1] = vertexBuffer.get(index * 3 + 1);
			vertex[2] = vertexBuffer.get(index * 3 + 2);
			return vertex;
		}
		return null;
	}
		
	/*
	 * Allocates buffer space for storing polygon data
	 *	@param	pCount	Number of polygons to create buffer space for.
	 */
	public void allocatePolygonBuffer(int pCount) {
		triangleBuffer = Buffers.newDirectIntBuffer(pCount * 3);
		polygonCount = pCount;
	}
	
	/*
	 * Adds a polygon to the surface
	 *	@param	poly	polygon indices array
	 */
	public void addPolygon(int[] poly) {
		if(poly == null)
			return;
		if(poly.length != 3)
			return;
		if(polygonInsertCounter < getPolygonCount()) {
			triangleBuffer.put(poly[0]);
			triangleBuffer.put(poly[1]);
			triangleBuffer.put(poly[2]);
			polygonInsertCounter++;
		}
	}
	
	public void addPolygon(int v1, int v2, int v3) {		
		if(polygonInsertCounter < getPolygonCount()) {
			triangleBuffer.put(v1);
			triangleBuffer.put(v2);
			triangleBuffer.put(v3);
			polygonInsertCounter++;
		}
	}
	
	/*
	 * Returns the triangle at specified index.
	 * 	@param	index	Index of polygon to return
	 * 	@return			If a valid index is give then int[3]
	 * 					containing vertex indices else null
	 */
	public int[] getPolygon(int index) {
		if(index >= 0 && index < getPolygonCount()){
			int [] polygon = new int[3];
			polygon[0] = triangleBuffer.get(index * 3);
			polygon[1] = triangleBuffer.get(index * 3 + 1);
			polygon[2] = triangleBuffer.get(index * 3 + 2);
			return polygon;
		}
		return null;
	}
	
	public void appendMesh(FloatBuffer vbuf, IntBuffer pbuf, FloatBuffer nbuf) {
		int cVnum = vertexBuffer.capacity();
		System.out.println("V: " + (vertexBuffer.capacity() + vbuf.capacity()) * 4  + " P: " + (triangleBuffer.capacity() + pbuf.capacity()) * 4);
		FloatBuffer tempVertexBuffer = Buffers.newDirectFloatBuffer(vertexBuffer.capacity() + vbuf.capacity());
		FloatBuffer	tempNormalBuffer = Buffers.newDirectFloatBuffer(normalBuffer.capacity() + nbuf.capacity());
		IntBuffer	tempTriangleBuffer = Buffers.newDirectIntBuffer(triangleBuffer.capacity() + pbuf.capacity());
		
		for(int i = 0; i < vertexBuffer.capacity(); ++i) {
			tempVertexBuffer.put(vertexBuffer.get(i));
			tempNormalBuffer.put(normalBuffer.get(i));
		}
		
		for(int i = 0; i < vbuf.capacity(); ++i) {
			tempVertexBuffer.put(vbuf.get(i));
			tempNormalBuffer.put(nbuf.get(i));
		}
		
		for(int i = 0; i < triangleBuffer.capacity(); ++i) {
			tempTriangleBuffer.put(triangleBuffer.get(i));
		}
		
		for(int i = 0; i < pbuf.capacity(); ++i) {
			tempTriangleBuffer.put(pbuf.get(i) + cVnum /3);
		}
		
		triangleBuffer.clear();		
		vertexBuffer.clear();
		normalBuffer.clear();
		
		triangleBuffer=null;		
		vertexBuffer=null;
		normalBuffer=null;
		
		triangleBuffer = tempTriangleBuffer;
		vertexBuffer = tempVertexBuffer;
		normalBuffer = tempNormalBuffer;
		
		triangleBuffer.rewind();
		vertexBuffer.rewind();
		normalBuffer.rewind();
	}
	
	public void setScalarBuffer(ScalarFloatData dataBuffer) {
		scalarBuffer = dataBuffer;
		colorBuffer = Buffers.newDirectFloatBuffer(this.getVertexCount() * 3);
		float min, max;
		max = min = scalarBuffer.getDataPoint(0);
		for(int i = 1; i < scalarBuffer.capacity(); ++i) {
			if(min > scalarBuffer.getDataPoint(i))
				min = scalarBuffer.getDataPoint(i);
			if(max < scalarBuffer.getDataPoint(i))
				max = scalarBuffer.getDataPoint(i);
		}
		Color c;
		LookUpTable lut = new LookUpTable(min, max);
		for(int i = 0; i < getVertexCount(); ++i) {
			c = lut.getColor(scalarBuffer.getDataPoint(i));
			colorBuffer.put(c.getRed()/255.0f);
			colorBuffer.put(c.getGreen()/255.0f);
			colorBuffer.put(c.getBlue()/255.0f);
		}
		this.useColor = true;
		System.out.println("Scalar Array Size: " + scalarBuffer.capacity() + " " + min + " " + max);
	}
	
	public void printPoints() {
		for(int i = 0; i < getVertexCount(); ++i) {
			System.out.println("Vertex #" + i + " : " + vertexBuffer.get(i*3) + ", " + vertexBuffer.get(i*3 + 1)
					+ ", " + vertexBuffer.get(i*3 +2));
			
		}
	}
	
	public void computeColorBuffer() {
		
	}
		
	public JPanel getModelPanel() {
		return null;
	}
	
	public AdjacencyList getAdjacencyList() {
		if(adjacencyList == null)
			adjacencyList = new AdjacencyList(this);
		return adjacencyList;
	}
		
	/*
	 * Intersects the surface with a plane and returns the lines of intersection for each polygon.
	 *	@param	point	A point on the plane
	 *	@param 	normal	Normal of the plane
	 *	@return			A vector contain the lines formed by intersecting a plane with a polygon
	 */
	public Vector<Line> intersectPlane(Vector3f point, Vector3f normal) {
		Vector<Line> lines = new Vector<Line>();
		int polynum = getPolygonCount();
		Vector3f p1, i1, i2;
		p1 = new Vector3f(0,0,0);
		i1 = new Vector3f(0,0,0);		
		i2 = new Vector3f(0,0,0);		
		float d = point.dot(normal);		
		float u;
		int intersects = 0;
		Line l;
		int vIndex0, vIndex1, vIndex2;
		Vector3f v0, v1, v2;
		v0 = new Vector3f();
		v1 = new Vector3f();
		v2 = new Vector3f();
		
//		System.out.println("Slice :"  + point);		
		for(int i = 0; i < polynum; ++i) {
			vIndex0 = triangleBuffer.get((i * 3));
			vIndex1 = triangleBuffer.get((i * 3) + 1);
			vIndex2 = triangleBuffer.get((i * 3) + 2);
			
			v0.set(vertexBuffer.get(vIndex0 * 3), vertexBuffer.get((vIndex0 * 3) + 1), vertexBuffer.get((vIndex0 * 3) +2));
			v1.set(vertexBuffer.get(vIndex1 * 3), vertexBuffer.get((vIndex1 * 3) + 1), vertexBuffer.get((vIndex1 * 3) +2));
			v2.set(vertexBuffer.get(vIndex2 * 3), vertexBuffer.get((vIndex2 * 3) + 1), vertexBuffer.get((vIndex2 * 3) +2));
			
			intersects = 0;			
			i1.set(0,0,0);	
			i2.set(0,0,0);
			
			// Side A
			p1.set(0,0,0);				
			p1.sub(v1 , v0);
			u = computePlaneIntersection(normal, v0, p1, d);
			if(u >= 0 && u <= 1) {
				i1.set(v0);
				p1.scale(u);
				i1.add(p1);				
				intersects++;				
			}
			
			// Side B
			p1.set(0,0,0);			
			p1.sub(v2 , v0);
			u = computePlaneIntersection(normal, v0, p1, d);
			if(u >= 0 && u <= 1) {				
				p1.scale(u);
				if(intersects == 0) {				
					i1.set(v0);
					i1.add(p1);
				}
				else {			
					i2.set(v0);	
					i2.add(p1);
				}
				intersects++;					
			}
			
			// Side C
			if(intersects != 2) {
				p1.set(0,0,0);			
				p1.sub(v2, v1);
				u = computePlaneIntersection(normal, v0, p1, d);
				if(u >= 0 && u <= 1) {
					i2.set(v1);
					p1.scale(u);
					i2.add(p1);
					intersects++;										
				}
			}
			
			if(intersects == 2) {
				l = new Line();
				Vector3f p = new Vector3f(i1);
				p.sub(i2);
				if(p.length() > 0.00001) {
					if(i1.z != point.z || i2.z != point.z)
						continue;
					l.p1.set(i1.x, i1.y, i1.z);
					l.p2.set(i2.x, i2.y, i2.z);
					lines.addElement(l);
				//	System.out.println("Line : " + l);
				}
			}			
		}				
		return lines;
	}
	
	/*
	 * Computes a plane-ray intersection.
	 *	@param	normal	Normal of the plane
	 *	@param	start	Vector indicating starting position of the ray
	 *	@param	line	Vector indicating direction
	 *	@param	d		Distance from origin of the plane
	 *	@return			Scalar value. Point of intersection = start + line * scalar
	 */
	private float computePlaneIntersection(Vector3f normal, Vector3f start, Vector3f line, float d) {
		float denom = 0, u = -1;
		denom = line.dot(normal);
		if(denom == 0)
			return -1;
		u = d - start.dot(normal);
		return u / denom;
	}
	
	
	/**
	 * Compute distance to mesh by testing distance to all triangles.
	 * @param point
	 * @return
	 */
	public float computeDistance2(Point3f point) {
		float distance = Float.POSITIVE_INFINITY;
		float d1;
				
		// Force computation of adjacency list
		getAdjacencyList();
		
		Triangle [] triangles = adjacencyList.getAllPrimitives();
		//int closestTriangle = -1;
		
		for(int i = 0; i< triangles.length; ++i) {
			d1 = triangles[i].distance(point);
			if(d1 < distance ) {
				distance = d1;
			//	closestTriangle = i;
			}
		}
		
		System.out.println("All Triangle Computation. Point is " + point.toString() + ". Distance is " + distance + ".");
		
		return distance;
	}
	
	/**
	 * Compute distance to mesh by finding closest vertex and then searching triangles in a 1 ring neighborhood.
	 * @param point
	 * @return
	 */
	public float computeDistance(Point3f point) {
		float distance = Float.POSITIVE_INFINITY;
		float d1;
		Vector3f edge1 = new Vector3f();
		Point3f v = new Point3f();
		int vNum = getVertexCount();
		int closestVertex = -1;
				
		System.out.println("Point is :  "  + point.x + "," + point.y + "," + point.z);
		
		// Force computation of adjacency list
		getAdjacencyList();
		
		//Step 1: Compute the closes vertex to this object.
		for(int i = 0; i < vNum; ++i) {
			v.set(getVertex(i));
			edge1.sub(point, v);
			d1 = edge1.length();			
			if(distance > d1) {				
				distance = d1;
				closestVertex = i;
			}
		}
		
		if(closestVertex == -1) 
			return Float.NaN;
		
		// Step 2 : Get the list of triangles that are associated with closest vertex
		Triangle [] triangles = adjacencyList.getPrimitives(closestVertex);
				
		// Step 3 : Computes the distances to all these triangles and set the distance to mesh as the minimum of those.
		if(triangles != null) 
			for(int i = 0; i < triangles.length; ++i) {
				d1 = triangles[i].distance(point);
				if(distance > d1)
					distance = d1;
			}
		
		System.out.println("Distance is : " + distance);
		return distance;
	}
	
	public void setDrawParam(DrawParam drawParams) {
		if(drawParams.getName().contentEquals("Edges")) {
			drawEdges = Boolean.parseBoolean(drawParams.getValue());
		}
	}
}