package edu.jhmi.rad.medic.visualization.primitives;

import javax.vecmath.Point3f;

import edu.jhmi.rad.medic.visualization.framework.Disposable;

/**
 * @author Bhaskar Kishore (bhaskar@jhu.edu)
 */
public class BoundingBox implements Disposable {
	protected Point3f minCoords;		//Minimum coordinate of bounding box
	protected Point3f maxCoords;		//Maximum coordinate of bounding box
	
	public BoundingBox() {
		minCoords = new Point3f(0.0f,0.0f,0.0f);
		maxCoords = new Point3f(0.0f,0.0f,0.0f);
	}

	public BoundingBox(BoundingBox b) {
		if(b != null) {
			minCoords  = new Point3f(b.getMinCoords().x, b.getMinCoords().y, b.getMinCoords().z);
			maxCoords  = new Point3f(b.getMaxCoords().x, b.getMaxCoords().y, b.getMaxCoords().z);
		}
		else {
			minCoords = new Point3f(0.0f,0.0f,0.0f);
			maxCoords = new Point3f(0.0f,0.0f,0.0f);
		}
	}
		
	public BoundingBox(Point3f min, Point3f max) {
		minCoords = new Point3f(min.x, min.y, min.z);
		maxCoords = new Point3f(max.x, max.y, max.z);		
	}
	
	public void dispose() {
		minCoords = maxCoords = null;
	}	
	
	protected void finalize() {
		dispose();
	}
		
	public void set(BoundingBox b) {
		if(b != null) {
			minCoords.set(b.getMinCoords());
			maxCoords.set(b.getMaxCoords());
		}
	}
	
	public void setMinCoords(Point3f min) {
		minCoords.set(min.x, min.y, min.z);	
	}
	
	public Point3f getMinCoords() {
		return minCoords;
	}
	
	public void setMaxCoords(Point3f max) {
		maxCoords.set(max.x, max.y, max.z);
	}
	
	public Point3f getMaxCoords() {
		return maxCoords;
	}
	
	public void set(Point3f min, Point3f max) {
		minCoords.set(min.x, min.y, min.z);	
		maxCoords.set(max.x, max.y, max.z);
	}
		
	
	/** 
	 * Computes the union of two bounding boxes
	 * @param b	Bounding box to perform union operation with
	 */
	public void union(BoundingBox b) {
		if(minCoords.x > b.getMinCoords().getX())
			minCoords.x = b.getMinCoords().getX();
		if(minCoords.y > b.getMinCoords().getY())
			minCoords.y = b.getMinCoords().getY();
		if(minCoords.z > b.getMinCoords().getZ())
			minCoords.z = b.getMinCoords().getZ();
		if(maxCoords.x < b.getMaxCoords().getX())
			maxCoords.x = b.getMaxCoords().getX();
		if(maxCoords.y < b.getMaxCoords().getY())
			maxCoords.y = b.getMaxCoords().getY();
		if(maxCoords.z < b.getMaxCoords().getZ())
			maxCoords.z = b.getMaxCoords().getZ();
	}
	
	/**
	 * Checks if a point is within this box.
	 * @param p	Point to verify
	 * @return	True if point if within the box otherwise false.
	 */
	public boolean isInside(Point3f p) {
		return (p.x >= minCoords.x) && (p.x <= maxCoords.x)
				&& (p.y >= minCoords.y) && (p.y <= maxCoords.y)
				&& (p.z >= minCoords.z) && (p.z <= maxCoords.z);
	}
	
	/**
	 * Computes distance from specified point
	 * @param p	Point from which distance to box is computed
	 * @return	Distance
	 */
	public float distance(Point3f p) {		
		if(isInside(p))
			return Float.NaN;
		Point3f dP = new Point3f();		
		dP.x = Math.max(Math.min(p.x, maxCoords.x), minCoords.x);
		dP.y = Math.max(Math.min(p.y, maxCoords.y), minCoords.y);
		dP.z = Math.max(Math.min(p.z, maxCoords.z), minCoords.z);
		return dP.distance(p); 
	}
	
	public String toString() {
		String value = "Min [" + minCoords.x +
						"," + minCoords.y +
						"," + minCoords.z +
						"] Max [" + maxCoords.x +
						"," + maxCoords.y +
						"," + maxCoords.z +
						"]";
		return value;
	}
}