package edu.jhmi.rad.medic.visualization.structures;

import java.util.ArrayList;

import edu.jhmi.rad.medic.visualization.models.TriangleMesh;

/**
 * Adjacency List structure for meshes.</br>
 * 
 * @author Bhaskar Kishore
 *
 */
public class AdjacencyList {
	private TriangleMesh mesh;
	private ArrayList<ArrayList<Integer>> v2p;
	private ArrayList<ArrayList<Integer>> p2v;
	
	public AdjacencyList(TriangleMesh t) {
		mesh = t;
		v2p = new ArrayList<ArrayList<Integer>>();
		p2v = new ArrayList<ArrayList<Integer>>();
		compute();
	}
	
	public void clear() {
		v2p.clear();
		p2v.clear();		
	}
	
	public void compute() {
		int primitivenum = mesh.getPolygonCount();
		int vertexnum = mesh.getVertexCount();
		
		clear();
		
		p2v.ensureCapacity(primitivenum);
		v2p.ensureCapacity(vertexnum);
			
		for(int i = 0; i < vertexnum; ++i)
			v2p.add(new ArrayList<Integer>());
		
		int[] triangle;
		
		for(int i = 0; i < primitivenum; ++i) {
			p2v.add(new ArrayList<Integer>());
			triangle = mesh.getPolygon(i);
			if(triangle == null)
				continue;
			if(!p2v.get(i).contains(triangle[0]))
				p2v.get(i).add(triangle[0]);
			if(!p2v.get(i).contains(triangle[1]))
				p2v.get(i).add(triangle[1]);
			if(!p2v.get(i).contains(triangle[2]))
				p2v.get(i).add(triangle[2]);
			
			if(!v2p.get(triangle[0]).contains(i))
				v2p.get(triangle[0]).add(i);
			if(!v2p.get(triangle[1]).contains(i))
				v2p.get(triangle[1]).add(i);
			if(!v2p.get(triangle[2]).contains(i))
				v2p.get(triangle[2]).add(i);
		}
	}
	
	public void setMesh(TriangleMesh t) {
		mesh = t;
		compute();
	}
	
	public TriangleMesh getMesh() {
		return mesh;
	}
		
	/**
	 * Returns the list of primitives belonging to vertex specified by index parameter.
	 * @param index	Indice of the vertex
	 * @return	list containing indices of primitives which are associated with this vertex
	 */
	public ArrayList<Integer> getPrimitiveList(int index) {
		if(index < v2p.size()) 
			return v2p.get(index);
		return null;
	}
	
	/**
	 * Returns the list of vertices belonging to primitives specified by index parameter.
	 * @param index	Indice of the primitive
	 * @return	list containing indices of vertices which are associated with this vertex.
	 */
	public ArrayList<Integer> getVertexList(int index) {
		if(index < p2v.size()) 
			return p2v.get(index);
		return null;
	}

	/**
	 * Returns a set of triangle primitives associated with vertex specified by index paramater.	
	 * @param index	Indice of vertex
	 * @return	array of triangles, null if index exceeds list size.
	 */
	public Triangle [] getPrimitives(int index) {
		if(index < v2p.size()) {
			Triangle [] triangles = new Triangle[v2p.get(index).size()];
			for(int i = 0; i < v2p.get(index).size(); ++i) {
				int[] triIndex = mesh.getPolygon(v2p.get(index).get(i));
				triangles[i] = new Triangle(mesh.getVertex(triIndex[0]),
											mesh.getVertex(triIndex[1]),
											mesh.getVertex(triIndex[2]));
			}
			return triangles;
		}
		return null;
	}
	
	/**
	 * Returns all triangle primitives associated with this mesh.
	 * @return	array of all triangles
	 */
	public Triangle[] getAllPrimitives() {
		Triangle[] triangles = new Triangle[mesh.getPolygonCount()];		
		for(int i = 0; i < mesh.getPolygonCount(); ++i) {
			int [] triIndex = mesh.getPolygon(i);
			triangles[i] = new Triangle(mesh.getVertex(triIndex[0]),
					mesh.getVertex(triIndex[1]),
					mesh.getVertex(triIndex[2]));
		}
		return triangles;
	}
}
