package edu.jhu.ece.iacl.algorithms.SpineSeg;
 import java.io.*;
import java.util.*;
import gov.nih.mipav.view.*;
import gov.nih.mipav.model.structures.jama.*;
import gov.nih.mipav.model.file.FileInfoBase;

import edu.jhmi.rad.medic.libraries.*;
import edu.jhmi.rad.medic.structures.*;
import edu.jhmi.rad.medic.utilities.*;
import edu.jhu.ece.iacl.jist.structures.image.ImageData;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataByte;
import edu.jhu.ece.iacl.jist.structures.image.ImageDataFloat;
import edu.jhu.ece.iacl.jist.structures.image.ImageHeader;

/**
 *
 *  This algorithm performs a simple alignment of the image with a topology template
 *	<p>
 *	The algorithm handles all the little things needed for image cropping, conversion
 * 	to an image buffer, padding, etc.
 *
 *	@version    March 2007
 *	@author     Pierre-Louis Bazin
 *		
 *
 */
 
public class SpinePreprocess {
		
	// data buffers
	private 	ImageDataFloat	origImage;  			// original images
	private 	ImageDataByte		topology;  			// topology template
	//private		float[]			transform; 				// rigid transform from atlas space to image space
	private 	int				nix,niy,niz;   			// image dimensions
	private 	int				mix,miy,miz;   			// cropped image dimensions
	private 	float			rix,riy,riz;   			// image resolutions
	private 	int				x0i,y0i,z0i;   			// cropped image origin
	private 	int				xNi,yNi,zNi;   			// cropped image final point
	private 	float			xCi,yCi,zCi;   			// coordinates of the center
	private 	float			xCt,yCt,zCt;   			// coordinates of the center
	private 	int				ntx,nty,ntz;   			// template dimensions
	private 	float			rtx,rty,rtz;   			// template resolutions
	private   	float			orx,ory,orz;			// image axis orientation
	private		int				nc;						// number of image channels
	private		boolean			cropped;				// check whether the images are the originals or the cropped ones
	private		boolean			transformed;			// check whether the topology is the originals or the cropped ones
	private		boolean			normalized;				// check whether the images are normalized in [0,1]
	
	// structure parameters
	private 	int 		nobj;    	// number of structures
	private		byte		bgLabel;    // label of the background (usually 1)
	private		float		bgRatio;	// ratio of image intensity used as background
	private		int			bs;	// the amount of extra borders
	//private		float[]		Imin,Imax;	// image min,max
	private		float[]		Ilow,Ihigh;	// image robust normalised min,max
	
	// for debug and display
	private static final boolean		debug				=	true;
	private static final boolean		verbose				=	true;
    
	// constants
	private static final	float   ISQRT2 = (float)(1.0/Math.sqrt(2.0f));
	private static final	float   ZERO = 1E-20f;
	private static final	float   INF = 1E20f;
	
	/**
	 *  constructor
	 */
	public SpinePreprocess(ImageDataFloat img_, int nc_,
									ImageDataByte topo_, 
									int nobjs_, byte bgLb_, float bgRatio_,
									int bdSz_) {

		origImage = img_;

		nc = nc_;
		nix = origImage.getRows();
		niy = origImage.getCols();
		niz = origImage.getSlices();
		rix = origImage.getHeader().getDimResolutions()[0];
		riy = origImage.getHeader().getDimResolutions()[1];
		riz = origImage.getHeader().getDimResolutions()[2];
		xCi = nix/2.0f;
		yCi = niy/2.0f;
		zCi = niz/2.0f;
		cropped = false;
		normalized = false;
		
		
		topology = topo_;
		ntx = topology.getRows();
		nty = topology.getCols();
		ntz = topology.getSlices();
		rtx = topology.getHeader().getDimResolutions()[0];
		rty = topology.getHeader().getDimResolutions()[1];
		rtz = topology.getHeader().getDimResolutions()[2];
		xCt = ntx/2.0f;
		yCt = nty/2.0f;
		zCt = ntz/2.0f;
		transformed = false;
		
		nobj = nobjs_;
		bgLabel = bgLb_;
		bgRatio = bgRatio_;
		bs = bdSz_;
		
		// orientation: initialize a rotation
	//	transform = new float[6];
		// note : assumes a rotation around the image center
		if (origImage.getHeader().getImageOrientation() == ImageHeader.ImageOrientation.AXIAL) {
	//		transform[0] = 0.0f;
	//		transform[1] = 0.0f;
	//		transform[2] = 0.0f;
			
			if (origImage.getHeader().getAxisOrientation()[0]==ImageHeader.AxisOrientation.L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (origImage.getHeader().getAxisOrientation()[1]==ImageHeader.AxisOrientation.P2A_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if ((origImage.getHeader().getAxisOrientation()[2]==ImageHeader.AxisOrientation.S2I_TYPE)) orz = -1.0f;
			else orz = 1.0f;
		} else if (origImage.getHeader().getImageOrientation() == ImageHeader.ImageOrientation.CORONAL) {
	//		transform[0] = -ISQRT2;
	//		transform[1] = 0.0f;
	//		transform[2] = 0.0f;
			
			if (origImage.getHeader().getAxisOrientation()[0]==ImageHeader.AxisOrientation.L2R_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (origImage.getHeader().getAxisOrientation()[1]==ImageHeader.AxisOrientation.I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (origImage.getHeader().getAxisOrientation()[2]==ImageHeader.AxisOrientation.P2A_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else if (origImage.getHeader().getImageOrientation() == ImageHeader.ImageOrientation.SAGITTAL) {
	//		transform[0] = -0.5f;
	//		transform[1] = -0.5f;
	//		transform[2] = 0.5f;
			
			if (origImage.getHeader().getAxisOrientation()[0]==ImageHeader.AxisOrientation.P2A_TYPE) orx = -1.0f;
			else orx = 1.0f;
			if (origImage.getHeader().getAxisOrientation()[1]==ImageHeader.AxisOrientation.I2S_TYPE) ory = -1.0f;
			else ory = 1.0f;
			if (origImage.getHeader().getAxisOrientation()[2]==ImageHeader.AxisOrientation.R2L_TYPE) orz = -1.0f;
			else orz = 1.0f;
		} else {
			// default is axial
	//		transform[0] = 0.0f;
	//		transform[1] = 0.0f;
	//		transform[2] = 0.0f;
			
			orx = 1.0f;
			ory = 1.0f;
			orz = 1.0f;
		}
	//	transform[3] = 0.0f;
	//	transform[4] = 0.0f;
	//	transform[5] = 0.0f;
		
		// init the image cropping					
		x0i = 0;
		y0i = 0;
		z0i = 0;
		xNi = nix-1;
		yNi = niy-1;
		zNi = niz-1;
		
		// compute min,max
		//Imin = new float[nc];
		//Imax = new float[nc];
		Ilow = new float[nc];
		Ihigh = new float[nc];
		for (int c=0;c<nc;c++) {
			// compute the robust min, max right away
			Ilow[c] = ImageFunctions.robustMinimum(origImage.toArray3d(), 0.001f, 2, nix, niy, niz );
			Ihigh[c] = ImageFunctions.robustMaximum(origImage.toArray3d(), 0.001f, 2, nix, niy, niz );
			//if (debug) MedicUtilPublic.displayMessage("image "+(c+1)+" range: ["+Imin[c]+", "+Imax[c]+"]\n");
			//if (debug) 
				MedicUtilPublic.displayMessage("image "+(c+1)+" eff. range: ["+Ilow[c]+", "+Ihigh[c]+"]\n");
		}
		
		if (debug) MedicUtilPublic.displayMessage("initialisation: done\n");
	}

	final public void finalize() {
		origImage = null;
	//	transform = null;
		System.gc();
	}
	
	public final ImageDataFloat getImages() { return origImage; }
    /*
	public final void setTransform(float[] trans) {
        if (trans.length==6) transform = trans;
		else System.err.println("wrong transform type");
    }
    */
/*    public final float[] getTransform() {
        return transform;
    }
*/
    public final float[] getIntensityMax() {
        return Ihigh;
    }

	public float[] getIntensityScale() {
		float[] scale = new float[nc];
		
		for (int c=0;c<nc;c++) 
			scale[c] = (Ihigh[c]-Ilow[c]);
		
		return scale;
	}
/*
    public final float[] exportTransform() {
		float[] trans = new float[6];
		for (int i=0;i<6;i++)
			trans[i] = transform[i];
        return trans;
    }
	
	public final String displayTransform() {
		String info;
		
		info = "transform: ("+transform[0]+", "+transform[1]+", "+transform[2]+", "
						     +transform[3]+", "+transform[4]+", "+transform[5]+")\n";
		
		return info;
	}
	*/
	/** current dimensions */
	public final int[] getCurrentImageDimensions() {
		int[] dim = new int[3];
        if (cropped) {
			dim[0] = mix; dim[1] = miy; dim[2] = miz;
		} else {
			dim[0] = nix; dim[1] = niy; dim[2] = niz;
		}
		if (debug) System.out.print("current image dimensions: "+dim[0]+" x "+dim[1]+" x "+dim[2]+"\n");
        
		return dim;
    }
	
	public final int[] getOriginalImageDimensions() {
		int[] dim = new int[3];
        dim[0] = nix; dim[1] = niy; dim[2] = niz;
		
		return dim;
    }
	public final int[] getOriginalImageArrayDimensions(int size) {
		int[] dim = new int[4];
        dim[0] = nix; dim[1] = niy; dim[2] = niz; dim[3] = size;
		
		return dim;
    }
	public final int getOriginalImageSize() {
		return nix*niy*niz;
    }
	public final int[] getCroppedImageDimensions() {
		int[] dim = new int[3];
        dim[0] = mix; dim[1] = miy; dim[2] = miz;
		
		return dim;
    }
	
	/** current resolutions, with the sign corresponding to the axis orientation */
	public final float[] getSignedImageResolutions() {
		float[] res = new float[3];
        res[0] = rix/orx; res[1] = riy/ory; res[2] = riz/orz;
		
		if (debug) System.out.print("current image resolutions: "+res[0]+" x "+res[1]+" x "+res[2]+"\n");
        
		return res;
    }
	
	public final float[] getImageResolutions() {
		float[] res = new float[3];
        res[0] = rix; res[1] = riy; res[2] = riz;
		
		if (debug) System.out.print("current image resolutions: "+res[0]+" x "+res[1]+" x "+res[2]+"\n");
        
		return res;
    }
	
	public final float[] getImageOrientations() {
		float[] ori = new float[3];
        ori[0] = orx; ori[1] = ory; ori[2] = orz;
		
		if (debug) System.out.print("current image orientations: "+ori[0]+" x "+ori[1]+" x "+ori[2]+"\n");
        
		return ori;
    }
	
	/** current dimensions */
	public final void updateTransformedTemplate(ImageDataByte tpl) {	
		topology = tpl;
		transformed = true;
	}	
	
    /** sets the bounding box for cropping from a vector image */
	public void findCroppingBoundaries() {
		x0i = nix-1;
        xNi = 0;
        y0i = niy-1;
        yNi = 0;
        z0i = niz-1;
        zNi = 0;

		RotationMatrix rotation = new RotationMatrix();
	//	rotation.setParameters(transform[0],transform[1],transform[2]);
		
		// check all images 
		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			for (int c=0;c<nc;c++) {
				if ( ( (normalized) && (origImage.getFloat(x,y,z)> bgRatio) ) 
					|| (origImage.getFloat(x,y,z)> bgRatio*(Ihigh[c]-Ilow[c]) ) ) {
					if (x < x0i) x0i = x;
					if (x > xNi) xNi = x;
					if (y < y0i) y0i = y;
					if (y > yNi) yNi = y;
					if (z < z0i) z0i = z;
					if (z > zNi) zNi = z;
				}
			}
		}
		
		// check for the topology: from topology to image space
		/*
		for (int x=0;x<ntx;x++) for (int y=0;y<nty;y++) for (int z=0;z<ntz;z++) {
			if (topology[x][y][z]>bgLabel) {
				// get image coordinates
				float[] XI = computeImageCoordinates(x,y,z,transform,rotation.getMatrix());
				
				if (XI[0] < x0i) x0i = Numerics.floor(XI[0]);
				if (XI[0] > xNi) xNi = Numerics.ceil(XI[0]);
				if (XI[1] < y0i) y0i = Numerics.floor(XI[1]);
				if (XI[1] > yNi) yNi = Numerics.ceil(XI[1]);
				if (XI[2] < z0i) z0i = Numerics.floor(XI[2]);
				if (XI[2] > zNi) zNi = Numerics.ceil(XI[2]);
			}
		}*/
        // debug
        if (debug) System.out.print("boundaries: ["+x0i+","+xNi+"] ["+y0i+","+yNi+"] ["+z0i+","+zNi+"]\n");
        
        return;
    }
	
    /** sets the bounding box for cropping from a vector image */
	public void findSimpleTransformedCroppingBoundaries(float[][] trans) {
		x0i = nix-1;
        xNi = 0;
        y0i = niy-1;
        yNi = 0;
        z0i = niz-1;
        zNi = 0;

		// check all images 
		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			for (int c=0;c<nc;c++) {
				if ( ( (normalized) && (origImage.getFloat(x,y,z)> bgRatio) ) 
					|| (origImage.getFloat(x,y,z)> bgRatio*(Ihigh[c]-Ilow[c]) ) ) {
					if (x < x0i) x0i = x;
					if (x > xNi) xNi = x;
					if (y < y0i) y0i = y;
					if (y > yNi) yNi = y;
					if (z < z0i) z0i = z;
					if (z > zNi) zNi = z;
				}
			}
		}
		
		// check for the topology: from topology to image space
		for (int x=0;x<ntx;x++) for (int y=0;y<nty;y++) for (int z=0;z<ntz;z++) {
			if (topology.getByte(x,y,z)>bgLabel) {
				// get image coordinates
				float[] XI = new float[3];
				
				for (int i=0;i<3;i++) {
					XI[i] = trans[i][0]*x + trans[i][1]*y + trans[i][2]*z + trans[i][3];
				}
				
				if (XI[0] < x0i) x0i = Numerics.floor(XI[0]);
				if (XI[0] > xNi) xNi = Numerics.ceil(XI[0]);
				if (XI[1] < y0i) y0i = Numerics.floor(XI[1]);
				if (XI[1] > yNi) yNi = Numerics.ceil(XI[1]);
				if (XI[2] < z0i) z0i = Numerics.floor(XI[2]);
				if (XI[2] > zNi) zNi = Numerics.ceil(XI[2]);
			}
		}
        // debug
        if (debug) System.out.print("boundaries: ["+x0i+","+xNi+"] ["+y0i+","+yNi+"] ["+z0i+","+zNi+"]\n");
        
        return;
    }
	
	/** crop the images */
	public void cropImages() {
		if (cropped) return;
		
		mix = xNi-x0i+1+2*bs;
		miy = yNi-y0i+1+2*bs;
		miz = zNi-z0i+1+2*bs;
		
		// update the transform parameters
		xCi = mix/2.0f;
		yCi = miy/2.0f;
		zCi = miz/2.0f;
		
		/*RotationMatrix rotation = new RotationMatrix();
		rotation.setParameters(transform[0],transform[1],transform[2]);
		float[][] R = rotation.getMatrix();
		transform[3] += R[0][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[0][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[0][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		transform[4] += R[1][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[1][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[1][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		transform[5] += R[2][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[2][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[2][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		*/
		ImageDataFloat smaller = new ImageDataFloat(mix,miy,miz);
		smaller.setHeader(origImage.getHeader());
		smaller.setName(origImage.getName());

			for (int x=0;x<mix;x++) for (int y=0;y<miy;y++) for (int z=0;z<miz;z++) {
				smaller.set(x,y,z, 0.0f);
			}
			for (int x=x0i;x<=xNi;x++) {
				for (int y=y0i;y<=yNi;y++) {
					for (int z=z0i;z<=zNi;z++) {
						if ( (x<0) || (x>=nix) || (y<0) || (y>=niy) || (z<0) || (z>=niz) )
							smaller.set(x-x0i+bs,y-y0i+bs,z-z0i+bs,0.0f);
						else
							smaller.set(x-x0i+bs,y-y0i+bs,z-z0i+bs, origImage.getFloat(x,y,z));
					}
				}
			}
		// replace the original images
		origImage = smaller;
		cropped = true;
		
		return;
	}
	
	
	/** crop the images */
	public ImageData cropImage(ImageData imageIn, float bgVal) {
		//if (cropped) return;
		
		mix = xNi-x0i+1+2*bs;
		miy = yNi-y0i+1+2*bs;
		miz = zNi-z0i+1+2*bs;
		
		// update the transform parameters
		xCi = mix/2.0f;
		yCi = miy/2.0f;
		zCi = miz/2.0f;
		
		//RotationMatrix rotation = new RotationMatrix();
		/*
		rotation.setParameters(transform[0],transform[1],transform[2]);
		float[][] R = rotation.getMatrix();
		transform[3] += R[0][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[0][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[0][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		transform[4] += R[1][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[1][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[1][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		transform[5] += R[2][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[2][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[2][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		*/
		ImageDataFloat smaller = new ImageDataFloat(mix,miy,miz);
		smaller.setHeader(imageIn.getHeader());
		smaller.setName(imageIn.getName());

		for (int c=0;c<nc;c++) {
			for (int x=0;x<mix;x++) for (int y=0;y<miy;y++) for (int z=0;z<miz;z++) {
				smaller.set(x,y,z,bgVal);
			}
			for (int x=x0i;x<=xNi;x++) {
				for (int y=y0i;y<=yNi;y++) {
					for (int z=z0i;z<=zNi;z++) {
						if ( (x<0) || (x>=nix) || (y<0) || (y>=niy) || (z<0) || (z>=niz) )
							smaller.set(x-x0i+bs,y-y0i+bs,z-z0i+bs,bgVal);
						else
							smaller.set(x-x0i+bs,y-y0i+bs,z-z0i+bs,imageIn.get(x,y,z));
					}
				}
			}
		}
		
		return smaller;
	}
	
	
	/** uncrop the images */
	public void uncropImages() {
		if (!cropped) return;
		
		// update the transform parameters
		xCi = nix/2.0f;
		yCi = niy/2.0f;
		zCi = niz/2.0f;
		
		RotationMatrix rotation = new RotationMatrix();
		/*
		rotation.setParameters(transform[0],transform[1],transform[2]);
		float[][] R = rotation.getMatrix();
		transform[3] -= R[0][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[0][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[0][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		transform[4] -= R[1][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[1][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[1][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		transform[5] -= R[2][0]*(mix/2.0f+x0i-bs-nix/2.0f)*rix/orx+R[2][1]*(miy/2.0f+y0i-bs-niy/2.0f)*riy/ory+R[2][2]*(miz/2.0f+z0i-bs-niz/2.0f)*riz/orz;
		*/
		ImageDataFloat larger = new ImageDataFloat(nix,niy,niz);
		larger.setHeader(origImage.getHeader());
		larger.setName(origImage.getName());
		
		for (int c=0;c<nc;c++) {

			for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
				larger.set(x,y,z, 0.0f);
			}		
			for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
				for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
					for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
						larger.set(x,y,z, origImage.getFloat(x-x0i+bs,y-y0i+bs,z-z0i+bs));
					}
				}
			}
		}
		origImage = larger;
		cropped = false;
		return;
	}
	
	/** uncrop the images */
	public ImageData uncropImage(ImageData volIn, float bgVal) {
		
		ImageDataFloat larger = new ImageDataFloat(nix,niy,niz);
		larger.setHeader(volIn.getHeader());
		larger.setName(volIn.getName());
		
		for (int c=0;c<nc;c++) {

			for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
				larger.set(x,y,z, bgVal);
			}		
			for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
				for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
					for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
						larger.set(x,y,z, volIn.getFloat(x-x0i+bs,y-y0i+bs,z-z0i+bs));
					}
				}
			}
		}
		return larger;
	}

	/** uncrop the image, send it to a 1D buffer*/
	public float[] uncropAndBuffer(float[][][] src, float bgVal) {
		float[] larger = new float[nix*niy*niz];

		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			larger[x + nix*y + nix*niy*z] = bgVal;
		}		
		for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
			for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
				for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
					larger[x + nix*y + nix*niy*z] = src[x-x0i+bs][y-y0i+bs][z-z0i+bs];
				}
			}
		}
		
		return larger;
	}

	/** uncrop the image, send it to a 1D buffer*/
	public byte[] uncropAndBuffer(byte[][][] src, byte bgVal) {
		byte[] larger = new byte[nix*niy*niz];

		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			larger[x + nix*y + nix*niy*z] = bgVal;
		}		
		for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
			for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
				for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
					larger[x + nix*y + nix*niy*z] = src[x-x0i+bs][y-y0i+bs][z-z0i+bs];
				}
			}
		}
		
		return larger;
	}

	//send it to a 1d buffer
	public byte[] buffer(byte[][][] src, byte bgVal) {
		byte[] larger = new byte[nix*niy*niz];

		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			larger[x + nix*y + nix*niy*z] = src[x][y][z];
		}		
		/*
		for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
			for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
				for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
					larger[x + nix*y + nix*niy*z] = src[x-x0i+bs][y-y0i+bs][z-z0i+bs];
				}
			}
		}
		*/
		return larger;
	}

	
	/** uncrop the image, send it to a 1D buffer*/
	public float[] uncropAndBuffer(float[][][][] src, float bgVal, int nv) {
		float[] larger = new float[nv*nix*niy*niz];

		for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) for (int v=0;v<nv;v++) {
			larger[x + nix*y + nix*niy*z + nix*niy*niz*v] = bgVal;
		}		
		for (int x=Numerics.max(x0i-bs,0);x<=Numerics.min(xNi+bs,nix-1);x++) {
			for (int y=Numerics.max(y0i-bs,0);y<=Numerics.min(yNi+bs,niy-1);y++) {
				for (int z=Numerics.max(z0i-bs,0);z<=Numerics.min(zNi+bs,niz-1);z++) {
					for (int v=0;v<nv;v++) {
						larger[x + nix*y + nix*niy*z + nix*niy*niz*v] = src[v][x-x0i+bs][y-y0i+bs][z-z0i+bs];
					}
				}
			}
		}
		
		return larger;
	}

	
    /** 
	 *	returns the transformed topology template.
	 *	the template is cropped and a border is added.
	 *	note: this transformation should only be attempted once!!
	 */
/*	public final void transformTopology() {
		if (transformed) return;
		
		// new dimensions
		int mx = xNi-x0i+1+2*bs;
		int my = yNi-y0i+1+2*bs;
		int mz = zNi-z0i+1+2*bs;
		
		byte[][][] smaller = new byte[mx][my][mz];
		
		RotationMatrix rotation = new RotationMatrix();
		//rotation.setParameters(transform[0],transform[1],transform[2]);
		
		for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
			smaller[x][y][z] = bgLabel;
		}
		for (int x=x0i;x<=xNi;x++) for (int y=y0i;y<=yNi;y++) for (int z=z0i;z<=zNi;z++) {
			// coordinates in topology space
			float[] XT = computeTemplateCoordinates(x,y,z,transform,rotation.getMatrix());
			
			smaller[x-x0i+bs][y-y0i+bs][z-z0i+bs] = ImageFunctions.nearestNeighborInterpolation(topology,bgLabel,XT[0],XT[1],XT[2],ntx,nty,ntz);
		}
		topology = smaller;
		transformed = true;
		
		return;
	} // transformTopology
*/
    /** 
	 *	returns the transformed topology template.
	 *	the template is cropped and a border is added.
	 *	note: this transformation should only be attempted once!!
	 */
	/*public final byte[][][] generateTransformedTopology() {
		if (transformed) return topology;
		
		// new dimensions
		int mx = xNi-x0i+1+2*bs;
		int my = yNi-y0i+1+2*bs;
		int mz = zNi-z0i+1+2*bs;
		
		byte[][][] smaller = new byte[mx][my][mz];
		
		RotationMatrix rotation = new RotationMatrix();
		rotation.setParameters(transform[0],transform[1],transform[2]);
		
		for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
			smaller[x][y][z] = bgLabel;
		}
		for (int x=x0i;x<=xNi;x++) for (int y=y0i;y<=yNi;y++) for (int z=z0i;z<=zNi;z++) {
			// coordinates in topology space
			float[] XT = computeTemplateCoordinates(x,y,z,transform,rotation.getMatrix());
			
			smaller[x-x0i+bs][y-y0i+bs][z-z0i+bs] = ImageFunctions.nearestNeighborInterpolation(topology,bgLabel,XT[0],XT[1],XT[2],ntx,nty,ntz);
		}
		return smaller;
	} // transformTopology
*/
    /** 
	 *  transform an image aligned with the topology
	 *	the template is cropped and a border is added.
	 */
	/*public final float[][][] transformNewImage(float[][][] img, float bgVal) {
		
		// new dimensions
		int mx = xNi-x0i+1+2*bs;
		int my = yNi-y0i+1+2*bs;
		int mz = zNi-z0i+1+2*bs;
		
		float[][][] smaller = new float[mx][my][mz];
		
		RotationMatrix rotation = new RotationMatrix();
		rotation.setParameters(transform[0],transform[1],transform[2]);
		
		for (int x=0;x<mx;x++) for (int y=0;y<my;y++) for (int z=0;z<mz;z++) {
			smaller[x][y][z] = bgVal;
		}
		for (int x=x0i;x<=xNi;x++) for (int y=y0i;y<=yNi;y++) for (int z=z0i;z<=zNi;z++) {
			// coordinates in topology space
			float[] XT = computeTemplateCoordinates(x,y,z,transform,rotation.getMatrix());
			
			smaller[x-x0i+bs][y-y0i+bs][z-z0i+bs] = ImageFunctions.linearInterpolation(img,bgVal,XT[0],XT[1],XT[2],ntx,nty,ntz);
		}
		return smaller;
	} // transformTopology
*/
	/** 
	 *	computes the transformed coordinates from template to image space
	 */
	private final float[] computeImageCoordinates(int x,int y,int z,float[] trans, float[][] rot) {
		float[] XI = new float[3];
		XI[0] = (rot[0][0]*((x-xCt)*rtx-trans[3])+rot[1][0]*((y-yCt)*rty-trans[4])+rot[2][0]*((z-zCt)*rtz-trans[5]))*orx/rix + xCi;
		XI[1] = (rot[0][1]*((x-xCt)*rtx-trans[3])+rot[1][1]*((y-yCt)*rty-trans[4])+rot[2][1]*((z-zCt)*rtz-trans[5]))*ory/riy + yCi;
		XI[2] = (rot[0][2]*((x-xCt)*rtx-trans[3])+rot[1][2]*((y-yCt)*rty-trans[4])+rot[2][2]*((z-zCt)*rtz-trans[5]))*orz/riz + zCi;
		
		return XI;
	}
	
	/** 
	 *	computes the transformed coordinates from image to template space
	 */
	private final float[] computeTemplateCoordinates(int x,int y,int z,float[] trans, float[][] rot) {
		float[] XT = new float[3];
		
		XT[0] = (rot[0][0]*(x-xCi)*rix/orx+rot[0][1]*(y-yCi)*riy/ory+rot[0][2]*(z-zCi)*riz/orz+trans[3])/rtx + xCt;
		XT[1] = (rot[1][0]*(x-xCi)*rix/orx+rot[1][1]*(y-yCi)*riy/ory+rot[1][2]*(z-zCi)*riz/orz+trans[4])/rty + yCt;
		XT[2] = (rot[2][0]*(x-xCi)*rix/orx+rot[2][1]*(y-yCi)*riy/ory+rot[2][2]*(z-zCi)*riz/orz+trans[5])/rtz + zCt;
		
		return XT;
	}
	
	/** normalize the images */
	public void normalizeImages() {
		if (normalized) return;
		
		/* normalize over the robust min, max  but keeping 0 as the min value 
		   (equivalent span for all images, but different means if needed, and masking is preserved) */
		for (int c=0;c<nc;c++) for (int x=0;x<nix;x++) for (int y=0;y<niy;y++) for (int z=0;z<niz;z++) {
			origImage.set(x,y,z, origImage.getFloat(x,y,z)/(Ihigh[c]-Ilow[c]));
		}
		
		normalized = true;
		
		return;
	}
/*
	///align the topology and the intensity center of the images 
	public void alignImagesAndTopolgyCenters(byte[] label, float[][] centroid, int classes) {
		float pix,piy,piz;
		float ptx,pty,ptz;
		float den, w;
		pix = 0; piy = 0; piz = 0; den = 0;
		for (int c=0;c<nc;c++) for (int x=x0i;x<xNi;x++) for (int y=y0i;y<yNi;y++) for (int z=z0i;z<zNi;z++) {
			if (normalized) {
				w = images[c][x-x0i+bs][y-y0i+bs][z-z0i+bs];
			} else {
				w = (images[c][x-x0i+bs][y-y0i+bs][z-z0i+bs])/(Ihigh[c]-Ilow[c]);
			}
			pix += w*x;
			piy += w*y;
			piz += w*z;
			den += w;
		}
		pix /= den;
		piy /= den;
		piz /= den;
		
		ptx = 0; pty = 0; ptz = 0; den = 0;
		for (int x=0;x<ntx;x++) for (int y=0;y<nty;y++) for (int z=0;z<ntz;z++) {
			for (int k=0;k<classes;k++) if (topology[x][y][z]==label[k]) {
				for (int c=0;c<nc;c++) {
					ptx += centroid[c][k]*x;
					pty += centroid[c][k]*y;
					ptz += centroid[c][k]*z;
					den += centroid[c][k];
				}
			}
		}
		ptx /= den; 
		pty /= den;
		ptz /= den;
		
		// transform: translation
		transform[3] = (ptx-xCt)*rtx - (pix-xCi)*rix; 
		transform[4] = (pty-yCt)*rty - (piy-yCi)*riy; 
		transform[5] = (ptz-zCt)*rtz - (piz-zCi)*riz; 
		
		return;
	}
*/
	/** 
	 *  compute the FCM membership functions for initial tissues
	 */
  /*  final public void initialMemberships(ImageDataFloat[] mems, byte[][][][] best, int approx, float[][] centroid, int classes) {
		if ( (approx<classes) && (approx>1) ) initialApproxMemberships(mems, best, approx, centroid, classes);
		else initialExactMemberships(mems, centroid, classes);
	}
    final public void initialMemberships(ImageDataFloat[] mems, byte[][][][] best, SpineDeformableAtlas atlas, int approx, float[][] centroid, int classes) {
		if ( (approx<classes) && (approx>1) ) initialApproxMemberships(mems, best, atlas, approx, centroid, classes);
		else initialExactMemberships(mems, centroid, classes);
	}
	*/
		
    final public void initialExactMemberships(ImageDataFloat[] mems, float[][] centroid, int classes) {
        float dist, dist0;
        float den,num;
        float mean,count;
		boolean	used;
		float[] grouping = new float[classes];
		
        for (int x=x0i;x<xNi;x++) for (int y=y0i;y<yNi;y++) for (int z=z0i;z<zNi;z++) {
			den = 0;
			for (int k=0;k<classes;k++) {
				num = 0;
				for (int c=0;c<nc;c++) {
					num += (origImage.getFloat(x-x0i+bs,y-y0i+bs,z-z0i+bs)-centroid[c][k])
						  *(origImage.getFloat(x-x0i+bs,y-y0i+bs,z-z0i+bs)-centroid[c][k]);
				}
				// invert the result
				if (num>ZERO) num = 1.0f/num;
				else num = INF;

				mems[k].set(x-x0i+bs,y-y0i+bs,z-z0i+bs,num);
				den += num;
				
				// grouping: count the number of classes with same centroids
				grouping[k] = 0.0f;
				for (int m=0;m<classes;m++)
					if (centroid[0][m]==centroid[0][k])
						grouping[k]++;
			}
			
			// normalization
			if (den>0.0f) {
				for (int k=0;k<classes;k++) {
					mems[k].set(x-x0i+bs,y-y0i+bs,z-z0i+bs, 
						grouping[k]*mems[k].getFloat(x-x0i+bs,y-y0i+bs,z-z0i+bs)/den);
				}
			}
			//normTime += System.currentTimeMillis()-start;				
		}
        return;
    }// initialMemberships

	/** 
	 *  compute the FCM membership functions for initial tissues, with approximation
	 */
    /*
    final public void initialApproxMemberships(ImageDataFloat[] mems, byte[][][][] best, SpineDeformableAtlas atlas, int approx, float[][] centroid, int classes) {
        float dist, dist0;
        float den;
		float[] num = new float[classes];
        float mean,count;
		boolean	used;
		float[] grouping = new float[classes];
		
		for (int x=x0i;x<xNi;x++) for (int y=y0i;y<yNi;y++) for (int z=z0i;z<zNi;z++) {
			den = 0;
			float[] shape = atlas.getShapeArray(x-x0i+bs,y-y0i+bs,z-z0i+bs);
			for (int k=0;k<classes;k++) {
				num[k] = 0;
				for (int c=0;c<nc;c++) {
					num[k] += (origImage.getFloat(x-x0i+bs,y-y0i+bs,z-z0i+bs)-centroid[c][k])
					  *(origImage.getFloat(x-x0i+bs,y-y0i+bs,z-z0i+bs)-centroid[c][k]);
				}
				// add the atlas prior
				for (int m=0;m<classes;m++) if (m!=k) {
					num[k] += shape[m]*shape[m];
				}
				
				// invert the result
				if (num[k]>ZERO) num[k] = 1.0f/num[k];
				else num[k] = INF;

				den += num[k];
				
				// grouping: count the number of classes with same centroids
				//grouping[k] = 0.0f;
				//for (int m=0;m<classes;m++)
					//if (centroid[0][m]==centroid[0][k])
						//grouping[k]++;
						
			}
			// normalization
			if (den>0.0f) {
				for (int k=0;k<classes;k++) {
					//num[k] = grouping[k]*num[k]/den;
					num[k] = num[k]/den;
				}
			} else {
				System.err.print("\0");
			}
			Numerics.bestIndex(best[x-x0i+bs][y-y0i+bs][z-z0i+bs],mems[x-x0i+bs][y-y0i+bs][z-z0i+bs],num,approx,Numerics.bestIndex(shape));
			//normTime += System.currentTimeMillis()-start;				
		}
        return;
       
    }*/// initialMemberships

	/** 
	 *  compute the FCM membership functions for initial tissues, with approximation
	 */
    /*
    final public void initialApproxMemberships(ImageDataFloat mems, byte[][][][] best, int approx, float[][] centroid, int classes) {
        float dist, dist0;
        float den;
		float[] num = new float[classes];
        float mean,count;
		boolean	used;
		float[] grouping = new float[classes];
		
		for (int x=x0i;x<xNi;x++) for (int y=y0i;y<yNi;y++) for (int z=z0i;z<zNi;z++) {
			den = 0;
			for (int k=0;k<classes;k++) {
				num[k] = 0;
				for (int c=0;c<nc;c++) {
					num[k] += (origImage.getFloat(x-x0i+bs,y-y0i+bs,z-z0i+bs)-centroid[c][k])
					  *(origImage.getFloat(x-x0i+bs,y-y0i+bs,z-z0i+bs)-centroid[c][k]);
					
				}
				// invert the result
				if (num[k]>ZERO) num[k] = 1.0f/num[k];
				else num[k] = INF;

				den += num[k];
				
				// grouping: count the number of classes with same centroids
				grouping[k] = 0.0f;
				for (int m=0;m<classes;m++)
					if (centroid[0][m]==centroid[0][k])
						grouping[k]++;
			}
			// normalization
			if (den>0.0f) {
				for (int k=0;k<classes;k++) {
					num[k] = grouping[k]*num[k]/den;
				}
			} else {
				System.err.print("\0");
			}
			Numerics.bestIndex(best[x-x0i+bs][y-y0i+bs][z-z0i+bs],mems[x-x0i+bs][y-y0i+bs][z-z0i+bs],num,approx);
			//normTime += System.currentTimeMillis()-start;				
		}
        return;
    }// initialMemberships
*/
	/**
	 *	compute the cluster centroids given the template
	 *	using template and image intensity 
	 */
    public final float[][] initialCentroids(float[][] prior, int classes) {
        float num,den,imgT;
		float Nobj;
		float[][] centroid = new float[nc][classes];
		
		//RotationMatrix rotation = new RotationMatrix();
		//rotation.setParameters(transform[0],transform[1],transform[2]);
		
		// find image centroids : already done at initialization
		for (int c=0;c<nc;c++) {
			/*
			if (cropped) {
				Ilow[c] = ImageFunctions.robustMinimum(images[c], 0.01f, 2, mix, miy, miz );
				Ihigh[c] = ImageFunctions.robustMaximum(images[c], 0.01f, 2, mix, miy, miz );
			} else {
				Ilow[c] = ImageFunctions.robustMinimum(images[c], 0.01f, 2, nix, niy, niz );
				Ihigh[c] = ImageFunctions.robustMaximum(images[c], 0.01f, 2, nix, niy, niz );
			}
			if (debug) System.out.println("image centroid range: ["+(Imin[c]+Ilow[c]*(Imax[c]-Imin[c]) )+", "
																   +(Imin[c]+Ihigh[c]*(Imax[c]-Imin[c]) )+"]\n");
			*/
			
			// adjust the first centroids
			for (int k=0;k<classes;k++) {
				//centroid[c][k] = Ilow + ratio[k]*(Ihigh-Ilow);
				//centroid[c][k] = Ilow[c] + prior[c][k]*(Ihigh[c]-Ilow[c]);
				// already normalized!!!
				centroid[c][k] = Ilow[c]/(Ihigh[c]-Ilow[c]) + prior[c][k];
			}
			
		}
		if (debug) {
			String info = "";;
			for (int c=0;c<nc;c++) {
				info += "image "+c+" centroids: ("+(centroid[c][0]*(Ihigh[c]-Ilow[c]) );
				for (int k=1;k<classes;k++) 
					info += ", "+(centroid[c][k]*(Ihigh[c]-Ilow[c]) );
				info += ")\n";
			}
			System.out.println(info);
		}
		return centroid;
    } // initCentroids
    
	/**
	 *	compute the lesion centroid given the regular centroids
	 */
    public final float[] initialLesionCentroid(float[] prior) {
        float num,den,imgT;
		float Nobj;
		float[] lesion = new float[nc];
		
		RotationMatrix rotation = new RotationMatrix();
		//rotation.setParameters(transform[0],transform[1],transform[2]);
		
		// find image centroids
		for (int c=0;c<nc;c++) {
			lesion[c] = Ilow[c]/(Ihigh[c]-Ilow[c]) + prior[c];
		}
		if (debug) {
			String info = "";;
			for (int c=0;c<nc;c++) {
				info += "image "+c+" lesion: ("+(lesion[c]*(Ihigh[c]-Ilow[c]) );
				info += ")\n";
			}
			System.out.println(info);
		}
		return lesion;
    } // initLesionCentroids
    

}
