package edu.jhu.ece.iacl.algorithms.graphics.edit;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.Stack;

import javax.vecmath.Point3d;
import javax.vecmath.Point3f;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SelfIntersectionTriangle;
import edu.jhu.ece.iacl.algorithms.graphics.surf.HierarchicalSurface;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface;
import edu.jhu.ece.iacl.algorithms.graphics.surf.SelfIntersectingPreventionProgressiveSurface;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface.EdgeCollapse;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface.MeshTopologyOperation;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface.SurfaceDistanceVertexMetric;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.data.BinaryMinHeap;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;
import edu.jhu.ece.iacl.jist.structures.geom.VertexIndexed;

/**
 * Remesh two surfaces so that they are in 1-to-1 vertex correspondence. This
 * method better preserves the geometry of both surfaces and does not create
 * self-intersections of either surface.
 * 
 * @author Blake
 * 
 */
public class RemeshCoupledSurfaces extends AbstractCalculation {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.6 $");
	}

	protected EmbeddedSurface source;
	protected EmbeddedSurface target;
	protected EmbeddedSurface remeshedSource;
	protected EmbeddedSurface remeshedTarget;
	protected static int THICKNESS = 0;
	protected static int MATCHED_POINT = 1;
	protected static int FID = 4;
	protected boolean preventSelfIntersection = false;
	protected Stack<MeshTopologyOperation> vertStack1 = new Stack<MeshTopologyOperation>();
	protected Stack<MeshTopologyOperation> vertStack2 = new Stack<MeshTopologyOperation>();
	protected SelfIntersectingPreventionProgressiveSurface ps1, ps2;

	/**
	 * Constructor
	 * 
	 * @param parent
	 *            parent calculation
	 * @param source
	 *            source surface
	 * @param target
	 *            target surface
	 */
	public RemeshCoupledSurfaces(AbstractCalculation parent,
			EmbeddedSurface source, EmbeddedSurface target) {
		super(parent);
		this.source = source;
		this.target = target;
		setLabel("Remesh Surfaces");
	}

	/**
	 * Constructor
	 * 
	 * @param source
	 *            source surface
	 * @param target
	 *            target surface
	 */
	public RemeshCoupledSurfaces(EmbeddedSurface source, EmbeddedSurface target) {
		this.source = source;
		this.target = target;
		setLabel("Remesh Surfaces");
	}

	/**
	 * Remesh surface
	 * 
	 * @param decimationAmt
	 *            Amount to decimate surface after remeshing [0,1]
	 */
	public void remesh(float decimationAmt) {
		int vertSourceCount = source.getVertexCount();
		int vertTargetCount = target.getVertexCount();
		HierarchicalSurface hierarchialTarget = new HierarchicalSurface(target);
		setTotalUnits((vertSourceCount + vertTargetCount));
		Point3d sp, tp;
		int v1, v2, v3;
		double[] stats;
		for (int i = 0; i < vertSourceCount; i++) {
			stats = source.getVertexData(i);
			// get barycentric coordinates from vertex data
			tp = new Point3d(stats[MATCHED_POINT], stats[MATCHED_POINT + 1],
					stats[MATCHED_POINT + 2]);
			int fid = (int) stats[MATCHED_POINT + 3];
			v1 = target.getCoordinateIndex(fid * 3);
			v2 = target.getCoordinateIndex(fid * 3 + 1);
			v3 = target.getCoordinateIndex(fid * 3 + 2);
			// Insert vertex into surface by using barycentric interpolation
			hierarchialTarget.insert(GeometricUtilities
					.interpolatePointFromBary(tp, target.getVertex(v1), target
							.getVertex(v2), target.getVertex(v3)), source
					.getVertexData(i), fid);
			incrementCompletedUnits();
		}
		markCompleted();

		// Construct surface with inserted vertices
		remeshedTarget = hierarchialTarget.getSurface(true);
		remeshedTarget.setName(target.getName() + "_remesh");
		HierarchicalSurface hierarchialSource = new HierarchicalSurface(source);
		for (int i = 0; i < vertTargetCount; i++) {
			stats = target.getVertexData(i);
			// get barycentric coordinates from vertex data
			sp = new Point3d(stats[MATCHED_POINT], stats[MATCHED_POINT + 1],
					stats[MATCHED_POINT + 2]);
			int fid = (int) stats[MATCHED_POINT + 3];
			v1 = source.getCoordinateIndex(fid * 3);
			v2 = source.getCoordinateIndex(fid * 3 + 1);
			v3 = source.getCoordinateIndex(fid * 3 + 2);
			// Insert vertex into surface by using barycentric interpolation
			hierarchialSource.insert(GeometricUtilities
					.interpolatePointFromBary(sp, source.getVertex(v1), source
							.getVertex(v2), source.getVertex(v3)), target
					.getVertexData(i), fid);
			incrementCompletedUnits();
		}
		// Construct surface with inserted vertices
		remeshedSource = hierarchialSource.getSurface(false);
		remeshedSource.setName(source.getName() + "_remesh");
		System.out.println("SOURCE COUNT " + remeshedSource.getVertexCount()
				+ " TARGET COUNT " + remeshedTarget.getVertexCount());
		// Generate self-intersection prevention PS
		markCompleted();
		ps1 = new SelfIntersectingPreventionProgressiveSurface(this,
				remeshedSource);
		ps2 = new SelfIntersectingPreventionProgressiveSurface(this,
				remeshedTarget);
		ps1.setMetric(ProgressiveSurface.VertexMetric.SURFACE_DISTANCE);
		ps2.setMetric(ProgressiveSurface.VertexMetric.SURFACE_DISTANCE);
		// Decimate surface as long as it will not create self-intersections of
		// either surface
		if (decimationAmt > 0) {
			ps1.regularize(vertStack1);
			ps2.regularize(vertStack2);
			System.out.println("DECIMATE");
			decimate(-1, decimationAmt);
		}
		// The hierarchial surface will likely produce a surface of poor
		// quality. Regularizing the surface will drastically improve mesh
		// quality.
		ps1.regularize(vertStack1);
		ps2.regularize(vertStack2);
		// Create surface after decimation and regularization
		remeshedSource = ps1.createSurface();
		remeshedTarget = ps2.createSurface();

		System.out.println("SOURCE COUNT " + remeshedSource.getVertexCount()
				+ " TARGET COUNT " + remeshedTarget.getVertexCount());
		double[][] vertData = remeshedSource.getVertexData();
		double[][] newVertData = new double[vertData.length][1];
		// Copy vertex data from source surface to both target and source
		for (int i = 0; i < vertData.length; i++) {
			newVertData[i][0] = vertData[i][0];
		}
		remeshedTarget.setVertexData(newVertData);
		remeshedSource.setVertexData(newVertData);
	}

	/**
	 * Heap metric for decimation derived from the minimum of the heap metric
	 * from both progressive surfaces.
	 * 
	 * @param id
	 *            vertex id
	 * @return heap metric
	 */
	public double heapMetric(int id) {
		return Math.min(ps1.heapMetric(id), ps2.heapMetric(id));
	}
	/**
	 * Decimate surface to specified threshold heap metric amount
	 * @param threshold threshold for heap metric to indicate termination of decimation
	 * @param maxPercent max percent decimation [0,1]
	 * @return
	 */
	protected int decimate(double threshold, double maxPercent) {
		int vertCount = ps1.getSurface().getVertexCount();
		BinaryMinHeap heap = new BinaryMinHeap(vertCount, vertCount, 1, 1);
		int count = 0;
		for (int i = 0; i < vertCount; i++) {
			if (ps1.neighborVertexVertexTable[i].length > 0
					&& ps2.neighborVertexVertexTable[i].length > 0) {
				// Insert un-decimated points into heap
				VertexIndexed vox = new VertexIndexed(heapMetric(i));
				vox.setPosition(i);
				heap.add(vox);
			} else {
				// Count removed points
				count++;
			}
		}
		int v2;
		int step = 0;
		boolean removed = false;
		MeshTopologyOperation vs1 = null;
		MeshTopologyOperation vs2 = null;
		VertexIndexed vox;
		int[] nbrs;
		setLabel("Decimate");
		setTotalUnits((int) Math.ceil(maxPercent * vertCount));
		while (!heap.isEmpty() && (vertCount - count) > 4
				&& count < maxPercent * vertCount) {
			vox = (VertexIndexed) heap.remove();
			if (-vox.getValue() < threshold)
				break;
			v2 = vox.getRow();
			if (ps1.neighborVertexVertexTable[v2].length == 0
					|| ps2.neighborVertexVertexTable[v2].length == 0) {
				// V2 has already been removed
				continue;
			}
			vs1 = ps1.createEdgeCollapse();
			vs2 = ps2.createEdgeCollapse();

			int v1ps1 = ((SelfIntersectingPreventionProgressiveSurface.SelfIntersectingEdgeCollapse) vs1)
					.isCollapsable(v2);
			int v1ps2 = ((SelfIntersectingPreventionProgressiveSurface.SelfIntersectingEdgeCollapse) vs2)
					.isCollapsable(v2);
			removed = (v1ps1 != -1 && v1ps2 != -1);
			if (removed) {
				if (!vs1.apply(v1ps1, v2) || !vs2.apply(v1ps2, v2)) {
					System.out.println("COULD NOT COLLAPSE " + v2);
				}
				vertStack1.push(vs1);
				vertStack2.push(vs2);
				// Recompute curvatures
				nbrs = vs1.getChangedVerts();
				// Removing edge may change curvature of neighboring vertexes,
				// so update them as well
				for (int i = 0; i < nbrs.length; i++) {
					v2 = nbrs[i];
					vox = new VertexIndexed(heapMetric(v2));
					vox.setPosition(v2);
					heap.change(v2, 0, 0, vox);
				}
				nbrs = vs2.getChangedVerts();
				// Removing edge may change curvature of neighboring vertexes,
				// so update them as well
				for (int i = 0; i < nbrs.length; i++) {
					v2 = nbrs[i];
					vox = new VertexIndexed(heapMetric(v2));
					vox.setPosition(v2);
					heap.change(v2, 0, 0, vox);
				}
				step++;
				count++;
			} else {
				// Haven't hit this statement yet, but it maybe possible
				System.err
						.println("COULD NOT REMOVE VERTEX BECAUSE OF TOPOLOGY "
								+ v2 + " " + vox.getValue());
			}
			incrementCompletedUnits();
		}
		markCompleted();
		heap.makeEmpty();
		return vertCount - count;
	}
	/**
	 * Get remeshed source
	 * @return source
	 */
	public EmbeddedSurface getRemeshedSource() {
		return remeshedSource;
	}
	/**
	 * Get remeshed target
	 * @return target
	 */
	public EmbeddedSurface getRemeshedTarget() {
		return remeshedTarget;
	}
}
