package edu.jhu.ece.iacl.algorithms.graphics.locator.kdtree;

import java.util.Vector;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

/**
 * Bounding box data structure to store triangles or other bounding boxes.
 * Bounding box intersection tests are much faster than triangle intersection
 * tests.
 * 
 * @author Blake Lucas
 * 
 */
public class BBox{

	// Locations of the corners of the boudning box
	public KdPoint3 minPoint, maxPoint;

	public Vector<BBox> children;

	protected int depth;

	/**
	 * Constructor
	 */
	public BBox() {
		this(0);
	}
	/**
	 * Constructor
	 * @param depth tree depth for bounding box
	 */
	public BBox(int depth) {
		minPoint = new KdPoint3(0, 0, 0);
		maxPoint = new KdPoint3(0, 0, 0);
		this.depth = depth;
		children = new Vector<BBox>();
	}
	/**
	 * Get tree depth
	 * @return depth
	 */
	public int getDepth() {
		return depth;
	}
	/**
	 * Set tree depth
	 * @param d depth
	 */
	public void setDepth(int d) {
		depth = d;
	}
	/**
	 * Get minimum point for bounding box
	 * @return minimum point
	 */
	public Point3f getMin() {
		return minPoint;
	}
	/**
	 * Get maximum point for bounding box
	 * @return maximum point
	 */
	public Point3f getMax() {
		return maxPoint;
	}
	/**
	 * Get children bounding boxes
	 * @return
	 */
	public Vector<BBox> getChildren() {
		return children;
	}
	/**
	 * Get bounding box volume 
	 * @return volume
	 */
	public double volume() {
		return (maxPoint.z - minPoint.z) * (maxPoint.y - minPoint.y)
				* (maxPoint.x - minPoint.x);
	}
	/**
	 * Determine if bounding box intersects this bounding box
	 * @param test bounding box
	 * @return true if bounding box intersects this bounding box
	 */
	public boolean intersects(BBox test) {
		double dx = Math.min(test.maxPoint.x, maxPoint.x)
				- Math.max(test.minPoint.x, minPoint.x);
		double dy = Math.min(test.maxPoint.y, maxPoint.y)
				- Math.max(test.minPoint.y, minPoint.y);
		double dz = Math.min(test.maxPoint.z, maxPoint.z)
				- Math.max(test.minPoint.z, minPoint.z);
		return (dx >= 0) && (dy >= 0) && (dz >= 0);
	}

	public static final float EPS = 0;
	/**
	 * Determine if point is inside this bounding box
	 * @param test point to test if its inside bounding box
	 * @return true if point is inside bounding box
	 */
	public boolean inside(Point3f test) {
		return (test.x >= minPoint.x) && (test.x <= maxPoint.x)
				&& (test.y >= minPoint.y) && (test.y <= maxPoint.y)
				&& (test.z >= minPoint.z) && (test.z <= maxPoint.z);
	}

	public String toString() {
		return ("BBox " + minPoint + " " + maxPoint);
	}
	/**
	 * Update bounding box so that it contains all its children.
	 */
	public void update() {
		minPoint.x = minPoint.y = minPoint.z = 1E10f;
		maxPoint.x = maxPoint.y = maxPoint.z = -1E10f;
		for (BBox child : children) {
			child.update();
			Point3f p = child.getMin();
			minPoint.x = Math.min(minPoint.x, p.x);
			minPoint.y = Math.min(minPoint.y, p.y);
			minPoint.z = Math.min(minPoint.z, p.z);

			p = child.getMax();
			maxPoint.x = Math.max(maxPoint.x, p.x);
			maxPoint.y = Math.max(maxPoint.y, p.y);
			maxPoint.z = Math.max(maxPoint.z, p.z);
		}
		// Vector3f v = new Vector3f();
		// v.sub(maxPoint, minPoint);
		// v.scale(2);
		// SA = v.x + v.y + v.z;
	}
	/**
	 * Measure distance from point to bounding box. Return -1 if point is inside bounding box.
	 * @param p point
	 * @return distance to bounding box
	 */
	public double distanceToBox(Point3f p) {
		if (inside(p)) {
			return -1;
		}
		Point3f ref = new Point3f();
		ref.x = Math.max(Math.min(p.x, maxPoint.x), minPoint.x);
		ref.y = Math.max(Math.min(p.y, maxPoint.y), minPoint.y);
		ref.z = Math.max(Math.min(p.z, maxPoint.z), minPoint.z);
		return p.distance(ref);
	}

	private static final int LEFT = 0, RIGHT = 1, MIDDLE = 2;
	/**
	 * Determine if ray intersects bounding box
	 * @param org origin
	 * @param dr ray direction
	 * @return true if ray intersects bounding box
	 */
	public boolean intersects(Point3f org, Vector3f dr) {
		double minB[] = { minPoint.x, minPoint.y, minPoint.z };
		double maxB[] = { maxPoint.x, maxPoint.y, maxPoint.z };
		double origin[] = { org.x, org.y, org.z };
		double dir[] = { dr.x, dr.y, dr.z };

		double[] coord = new double[3];
		int[] quadrant = new int[3];
		boolean inside = true;
		int i;
		int whichPlane;
		double maxT[] = { 0, 0, 0 };
		double[] candidatePlane = new double[3];

		/*
		 * Find candidate planes; this loop can be avoided if rays cast all from
		 * the eye(assume perpsective view)
		 */
		for (i = 0; i < 3; i++) {
			if (origin[i] < minB[i]) {
				quadrant[i] = LEFT;
				candidatePlane[i] = minB[i];
				inside = false;
			} else if (origin[i] > maxB[i]) {
				quadrant[i] = RIGHT;
				candidatePlane[i] = maxB[i];
				inside = false;
			} else {
				quadrant[i] = MIDDLE;
			}
		}
		if (inside) {
			return true;
		}
		/* Calculate T distances to candidate planes */
		for (i = 0; i < 3; i++) {
			if (quadrant[i] != MIDDLE && dir[i] != 0.) {
				maxT[i] = (candidatePlane[i] - origin[i]) / dir[i];
			} else {
				maxT[i] = -1.;
			}
		}
		/* Get largest of the maxT's for final choice of intersection */
		whichPlane = 0;
		for (i = 1; i < 3; i++) {
			if (maxT[whichPlane] < maxT[i])
				whichPlane = i;
		}
		/* Check final candidate actually inside box */;
		if (maxT[whichPlane] < 0) {
			return false;
		}
		for (i = 0; i < 3; i++) {
			if (whichPlane != i) {
				coord[i] = origin[i] + maxT[whichPlane] * dir[i];
				if (coord[i] < minB[i] || coord[i] > maxB[i]) {
					return false;
				}
			} else {
				coord[i] = candidatePlane[i];
			}
		}
		return true;
	}
	/**
	 * Determine if line segment intersects bounding box
	 * @param org start point
	 * @param end end point
	 * @return true if line segment intersects bounding box
	 */
	public boolean intersects(Point3f org, Point3f end) {
		if (inside(org) || inside(end))
			return true;
		Vector3f dr = new Vector3f();
		dr.sub(end, org);
		float len = dr.length();
		dr.scale(1.0f / len);
		double minB[] = { minPoint.x, minPoint.y, minPoint.z };
		double maxB[] = { maxPoint.x, maxPoint.y, maxPoint.z };
		double origin[] = { org.x, org.y, org.z };
		double dir[] = { dr.x, dr.y, dr.z };

		float[] coord = new float[3];
		int[] quadrant = new int[3];
		boolean inside = true;
		int i;
		int whichPlane;
		double maxT[] = { 0, 0, 0 };
		double[] candidatePlane = new double[3];

		/*
		 * Find candidate planes; this loop can be avoided if rays cast all from
		 * the eye(assume perpsective view)
		 */
		for (i = 0; i < 3; i++) {
			if (origin[i] < minB[i]) {
				quadrant[i] = LEFT;
				candidatePlane[i] = minB[i];
				inside = false;
			} else if (origin[i] > maxB[i]) {
				quadrant[i] = RIGHT;
				candidatePlane[i] = maxB[i];
				inside = false;
			} else {
				quadrant[i] = MIDDLE;
			}
		}
		if (inside) {
			return true;
		}
		/* Calculate T distances to candidate planes */
		for (i = 0; i < 3; i++) {
			if (quadrant[i] != MIDDLE && dir[i] != 0.) {
				maxT[i] = (candidatePlane[i] - origin[i]) / dir[i];
			} else {
				maxT[i] = -1.;
			}
		}
		/* Get largest of the maxT's for final choice of intersection */
		whichPlane = 0;
		for (i = 1; i < 3; i++) {
			if (maxT[whichPlane] < maxT[i])
				whichPlane = i;
		}
		/* Check final candidate actually inside box */;
		if (maxT[whichPlane] < 0) {
			return false;
		}
		for (i = 0; i < 3; i++) {
			if (whichPlane != i) {
				coord[i] = (float) (origin[i] + maxT[whichPlane] * dir[i]);
				if (coord[i] < minB[i] || coord[i] > maxB[i]) {
					return false;
				}
			} else {
				coord[i] = (float) candidatePlane[i];
			}
		}
		if (org.distance(new Point3f(coord)) <= len) {
			return true;
		} else {
			return false;
		}
	}
}
