package edu.jhu.ece.iacl.algorithms.graphics.map;

import javax.vecmath.Point2d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import no.uib.cipr.matrix.DenseVector;
import no.uib.cipr.matrix.Matrix;
import no.uib.cipr.matrix.Vector;
import no.uib.cipr.matrix.sparse.BiCG;
import no.uib.cipr.matrix.sparse.DefaultIterationMonitor;
import no.uib.cipr.matrix.sparse.FlexCompRowMatrix;
import no.uib.cipr.matrix.sparse.IterationMonitor;
import no.uib.cipr.matrix.sparse.IterativeSolverNotConvergedException;
import edu.jhu.ece.iacl.jist.algorithms.graphics.GeomUtil;
import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.algorithms.graphics.edit.SurfaceSlicer;
import edu.jhu.ece.iacl.algorithms.graphics.surf.ProgressiveSurface;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Improved spherical mapping procedure that enforces the mid-saggital plane
 * stay aligned with a specified plane. There are additional features for
 * different types of mappings and production of guaranteed bijective mappings.
 * The partially inflated surface is used to initialize the spherical mapping,
 * but distortion minimization is relative to the reference surface. Using the
 * inflated surface as the reference surface will likely mitigate problems
 * caused my negative weights.
 * 
 * @author Blake Lucas
 * 
 */
public class HemisphericalMap extends ProgressiveSurface {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.5 $");
	}

	public HemisphericalMap() {
		setLabel("Hemispherical Map");
	}

	int iters = 2;
	int resolutions = 2;
	boolean useElevations = false;
	double maxDecimation = 0.5;
	double hormannWeight = 0;

	public enum WeightFunction {
		TUETTE, CONFORMAL, AUTHALIC, CONFORMAL_AUTHALIC_BLEND, MEAN_VALUE
	};

	protected double blendMu = 0.5, blendLambda = 0.5;
	int[] labels;
	double[] dists;
	double[] elevations;
	double[] residuals;
	double[][] gradients;
	double[][] metrics;
	EmbeddedSurface labeledRefSurface;

	public void setMaxDecimation(double dec) {
		this.maxDecimation = dec;
	}

	public void setHormannWeight(double hormann) {
		this.hormannWeight = hormann;
	}

	public void setUseElevations(boolean elev) {
		this.useElevations = elev;
	}

	public void setBlendLambda(double lambda) {
		this.blendLambda = lambda;
	}

	public void setBlendMu(double mu) {
		this.blendMu = mu;
	}

	/**
	 * Cut surface in half using specified normal and center of mass for the
	 * inflated surface.
	 * 
	 * @param inflatedSurf
	 *            inflated surface
	 * @param refSurface
	 *            reference surface
	 * @param center
	 *            center of mass
	 * @param normal
	 *            plane normal
	 * @return inflated surface that has been split in half
	 */
	protected EmbeddedSurface labelSurface(EmbeddedSurface inflatedSurf,
			EmbeddedSurface refSurface, Point3f center, Vector3f normal) {
		// Split inflated surface
		EmbeddedSurface labeledSurf = SurfaceSlicer.labelCut(inflatedSurf,null,
				center, normal);
		int origVertCount = refSurface.getVertexCount();
		int vertCount = labeledSurf.getVertexCount();
		elevations = new double[vertCount];
		labeledRefSurface = labeledSurf.clone();
		init(labeledSurf, false);
		// Move inflated surface positions to reference surface positions
		for (int i = 0; i < vertCount; i++) {
			if (i >= origVertCount) {
				int[] nbrs = neighborVertexVertexTable[i];
				int count = 0;
				Point3f avgPt = new Point3f();
				for (int nbr : nbrs) {
					if (nbr < origVertCount) {
						count++;
						avgPt.add(refSurface.getVertex(nbr));
					}
				}
				if (count == 0) {
					System.out.println("NO NEIGHBORS " + labels[i]);
				} else {
					avgPt.scale(1.0f / count);
				}
				// For vertices that are along split, use average of neighbors
				this.labeledRefSurface.setVertex(i, avgPt);
			} else {
				// Use reference surface positions
				labeledRefSurface.setVertex(i, refSurface.getVertex(i));
			}
			elevations[i] = 1;
		}
		return labeledSurf;
	}

	/**
	 * Solve for a spherical map of a surface.
	 * 
	 * @param normal
	 *            normal direction for plane to split the surface in half.
	 * @param initSurface
	 *            initial surface used to initialize spherical mapping by
	 *            projecting surface to sphere.
	 * @param refSurface
	 *            reference surface used to minimize geometric distortion.
	 * @param maxCurvature
	 *            maximum curvature threshold used as a termination criterion
	 *            for decimation.
	 * @param correctMap
	 *            correct spherical map to be bijective.
	 * @param findElevations
	 *            find elevations for spherical map. This is highly
	 *            experimental.
	 * @param weightFunc
	 *            weighting function used to minimize distortion.
	 * @return
	 */
	public EmbeddedSurface[] solve(Vector3f normal,
			EmbeddedSurface initSurface, EmbeddedSurface refSurface,
			double maxCurvature, boolean correctMap, boolean findElevations,
			WeightFunction weightFunc) {
		Point3f center = initSurface.getCenterOfMass();
		int origVertCount = refSurface.getVertexCount();
		EmbeddedSurface labeledInflatedSurf = labelSurface(initSurface,
				refSurface, center, normal);
		int vertCount = labeledInflatedSurf.getVertexCount();
		double[][] vertData = labeledInflatedSurf.getVertexData();
		// Label Surface
		labels = new int[vertCount];
		for (int i = 0; i < vertCount; i++) {
			labels[i] = (int) Math.signum(vertData[i][0]);
		}
		// Decimate partially inflated surface using curvature metric
		setCollapseMethod(CollapseMethod.EDGE);
		setMetric(VertexMetric.CURVATURE);
		setMetric(EdgeMetric.DISTANCE);
		if (correctMap) {
			decimate(maxCurvature, maxDecimation, false, labels);
		}
		// Project surface to sphere for initialization
		for (int i = 0; i < vertCount; i++) {
			Point3f pt = labeledInflatedSurf.getVertex(i);
			pt.sub(center);
			GeometricUtilities.normalize(pt);
			surf.setVertex(i, pt);
		}
		// Choose the appropriate weighting function
		WeightVectorFunc func = null;
		switch (weightFunc) {
		case TUETTE:
			func = new TuetteWeightFunc(this, labels);
			break;
		case CONFORMAL:
			func = new ConformalWeightFunc(this, labels, labeledRefSurface);
			break;
		case AUTHALIC:
			func = new AuthalicWeightFunc(this, labels, labeledRefSurface);
			break;
		case CONFORMAL_AUTHALIC_BLEND:
			func = new ConformalAuthalicBlendWeightFunc(this, labels,
					labeledRefSurface, blendLambda, blendMu);
			break;
		case MEAN_VALUE:
			func = new MeanValueWeightFunc(this, labels, labeledRefSurface);
			break;
		}
		// Use stereographic projection to map sphere to cylinder
		mapToCylinder(surf);
		// Solve for mapping procedure in plane
		solveConjugateGradient2D(func, 1);
		// Map cylinder back to sphere
		mapToSphere(surf);
		// Use gauss-seidel procedure to adjust vertices that lie on
		// mid-saggital plane
		solveGaussSeidelRadial(func, 0.8, 1000, 1E6, 1E-30);
		// Label vertices that are not included in the mesh at the current
		// resolution
		for (int i = 0; i < vertCount; i++) {
			if (neighborVertexVertexTable[i].length != 0) {
				labels[i] = 0;
			} else {
				labels[i] = 1;
			}
		}
		// Tessellate surface to restore original parameterization
		if (correctMap) {
			tessellate(1.0);
		}
		// Use mean-value weighting function to position vertices that were
		// decimated
		solveConjugateGradient3D(new MeanValueWeightFunc(this, labels,
				labeledRefSurface), 1);
		// Solve for the elevations of each vertex
		// This is highly experimental
		if (findElevations) {
			solveConjugateGradientRadial(new TuetteWeightFunc(this, labels), 1);
		}
		// Remove split vertices by referring to original parameterization of
		// surface.
		initSurface = refSurface.clone();
		EmbeddedSurface sphere = refSurface.clone();
		for (int i = 0; i < origVertCount; i++) {
			Point3f pt = surf.getVertex(i);
			sphere.setVertex(i, pt);
		}
		// Correct spherical map to be bijective using multi-resolution approach
		if (correctMap) {
			RobustSphericalMapCorrection rsmc = new RobustSphericalMapCorrection(
					this, 0.9, 0.01, 3);
			rsmc.setHormannWeight(hormannWeight);
			sphere = rsmc.solve(refSurface, sphere);
		}
		// Scale vertices by elevations if the elevations are not equal to 1
		for (int i = 0; i < vertCount; i++) {
			Point3f pt = surf.getVertex(i);
			pt.scale((float) elevations[i]);
			surf.setVertex(i, pt);
		}
		for (int i = 0; i < origVertCount; i++) {
			Point3f pt = sphere.getVertex(i);
			pt.scale((float) elevations[i]);
			initSurface.setVertex(i, pt);
		}
		// Generate spherical surface
		sphere.setName(refSurface.getName() + "_sphere");
		sphere.setVertexData(refSurface.getVertexData());

		// Generate elevation surface
		if(findElevations){
			initSurface.setName(refSurface.getName() + "_elev");
			initSurface.setVertexData(refSurface.getVertexData());
			return new EmbeddedSurface[] { sphere, initSurface };
		} else {
			return new EmbeddedSurface[] { sphere};

		}
	}

	/**
	 * Map cylinder to sphere using inverse stereographic projection.
	 * 
	 * @param surf
	 *            cylinder surface
	 */
	protected void mapToSphere(EmbeddedSurface surf) {
		int vertCount = surf.getVertexCount();
		int l;
		for (int i = 0; i < vertCount; i++) {
			Point3f q = surf.getVertex(i);
			l = labels[i];
			Point3f p = GeometricUtilities
					.stereoToSphere(new Point2d(q.x, q.y));
			q.x = (((l > 0) ? -p.z : p.z));
			q.y = p.x;
			q.z = p.y;

			surf.setVertex(i, q);
		}
	}

	/**
	 * Map sphere to cylinder using stereo graphic projection.
	 * 
	 * @param surf
	 *            spherical surface
	 */
	protected void mapToCylinder(EmbeddedSurface surf) {
		int vertCount = surf.getVertexCount();
		int l;
		for (int i = 0; i < vertCount; i++) {
			Point3f p = surf.getVertex(i);
			Point2d q = GeometricUtilities.sphereToStereo(new Point3f(p.y, p.z,
					-Math.abs(p.x)));
			l = labels[i];
			if (l > 0) {
				surf.setVertex(i, new Point3f((float) q.x, (float) q.y, 1));
			} else if (l < 0) {
				surf.setVertex(i, new Point3f((float) q.x, (float) q.y, -1));
			} else {
				surf.setVertex(i, new Point3f((float) q.x, (float) q.y, 0));
			}
		}
	}

	public void setIterations(int iters) {
		this.iters = iters;
	}

	public void setResolutions(int res) {
		this.resolutions = res;
	}

	/**
	 * Reinsert decimated vertices and normalize their locations so that they
	 * lie on a sphere.
	 */
	@Override
	public void tessellate(double maxTessellation) {
		int maxCount = (int) Math
				.floor(maxTessellation * surf.getVertexCount());
		int count = 0;
		System.out.println("TESSELLATE " + maxCount + " " + vertStack.size());
		setTotalUnits(Math.min(vertStack.size(), maxCount));
		while (!vertStack.isEmpty() && count < maxCount) {
			EdgeCollapse vs = (EdgeCollapse) vertStack.pop();
			vs.restore();
			Point3f pt1 = surf.getVertex(vs.v1);
			GeometricUtilities.normalize(pt1);
			surf.setVertex(vs.v1, pt1);
			Point3f pt2 = surf.getVertex(vs.v2);
			GeometricUtilities.normalize(pt2);
			surf.setVertex(vs.v2, pt2);
			incrementCompletedUnits();
		}
	}

	/**
	 * Re-insert deleted edge so that it lies on a sphere.
	 */
	@Override
	protected void insertSphericalEdge(SphericalEdgeCollapse vs) {
		int v1 = vs.v1;
		int v2 = vs.v2;
		int nbr1 = vs.nbr1;
		int nbr2 = vs.nbr2;
		int[] nbrs1 = neighborVertexVertexTable[v1];
		int nbrIndex1 = find(v1, nbr1);
		int nbrIndex2 = find(v1, nbr2);
		int[] newNbrs1 = new int[((nbrIndex2 > nbrIndex1) ? (nbrIndex2 - nbrIndex1)
				: (nbrIndex2 + nbrs1.length - nbrIndex1)) + 2];
		int[] newNbrs2 = new int[nbrs1.length - newNbrs1.length + 4];
		newNbrs1[newNbrs1.length - 1] = v2;
		for (int i = 0; i < newNbrs1.length - 1; i++) {
			newNbrs1[i] = nbrs1[(nbrIndex1 + i) % nbrs1.length];
		}
		newNbrs2[0] = v1;
		int v3;
		for (int i = 0; i < newNbrs2.length - 1; i++) {
			v3 = newNbrs2[i + 1] = nbrs1[(nbrIndex2 + i) % nbrs1.length];
			if (i == 0) {
				connect(v3, v2, find(v3, v1));
			} else if (i == newNbrs2.length - 2) {
				connect(v3, v2, (find(v3, v1) + 1)
						% neighborVertexVertexTable[v3].length);
			} else {
				int e = find(v3, v1);
				neighborVertexVertexTable[v3][e] = v2;
			}
		}
		neighborVertexVertexTable[v1] = newNbrs1;
		neighborVertexVertexTable[v2] = newNbrs2;
		surf.setVertex(vs.v2, surf.getVertex(vs.v1));
		if (isWoundCorrectly(vs.v1) && isWoundCorrectly(vs.v2)) {
			vertInsertOptMethod.insert(vs);
		}
	}

	/**
	 * Use iterative projected gauss-seidel procedure to optimize vertex
	 * locations.
	 * 
	 * @param func
	 *            weighting function
	 * @param lambda
	 *            step size
	 * @param maxIters
	 *            maximum iterations
	 * @param errThreshold
	 *            error threshold for termination
	 * @param deltaThreshold
	 *            change in error threshold for termination
	 */
	public void solveGaussSeidelSphere(WeightVectorFunc func, double lambda,
			int maxIters, double errThreshold, double deltaThreshold) {
		int vertCount = surf.getVertexCount();
		int[] ordering = new int[vertCount];
		for (int id = 0; id < vertCount; id++)
			ordering[id] = id++;
		// Create random traversal order
		for (int i = 0; i < vertCount; i++) {
			int swapIndex = (int) Math.max(0, Math.min(vertCount - 1,
					(vertCount - 1) * Math.random()));
			int tmp = ordering[i];
			ordering[i] = ordering[swapIndex];
			ordering[swapIndex] = tmp;
		}
		double res;
		double totalRes = 1E30;
		int iter = 0;
		int updateInterval = 10;
		double lastError = 0;
		do {
			lastError = totalRes;
			totalRes = 0;
			;
			for (int id : ordering) {
				Vector3f error = func.getResidualVector(id);
				Point3f pt = surf.getVertex(id);
				Vector3f rError = new Vector3f(pt);
				rError.scale(error.dot(rError));
				error.sub(rError);

				res = error.length();
				error.scale((float) lambda);
				pt.add(error);
				GeomUtil.normalize(pt);
				surf.setVertex(id, pt);
				totalRes += res;
			}
			totalRes /= vertCount;
			iter++;
			if (iter % updateInterval == 0)
				System.out.println("Iteration " + iter + " " + totalRes);
		} while (totalRes > errThreshold
				&& Math.abs(totalRes - lastError) > deltaThreshold
				&& iter < maxIters);
		System.out.println("Residual " + totalRes + " iterations " + iter);
	}

	/**
	 * Radial gauss-seidel to find optimal location for vertices. Unfortunately
	 * a trivial optimal location is that all vertices lie at the origin.
	 * 
	 * @param func
	 *            weighting function
	 * @param lambda
	 *            step size
	 * @param maxIters
	 *            maximum iterations
	 * @param errThreshold
	 *            error threshold for termination
	 * @param deltaThreshold
	 *            change in error threshold for termination
	 */
	public void solveGaussSeidelRadial(WeightVectorFunc func, double lambda,
			int maxIters, double errThreshold, double deltaThreshold) {
		int vertCount = surf.getVertexCount();
		int[] ordering = new int[vertCount];
		for (int id = 0; id < vertCount; id++)
			ordering[id] = id++;
		// Create random traversal order
		for (int i = 0; i < vertCount; i++) {
			int swapIndex = (int) Math.max(0, Math.min(vertCount - 1,
					(vertCount - 1) * Math.random()));
			int tmp = ordering[i];
			ordering[i] = ordering[swapIndex];
			ordering[swapIndex] = tmp;
		}
		double res;
		double totalRes = 1E30;
		int iter = 0;
		int updateInterval = 10;
		double lastError = 0;
		do {
			lastError = totalRes;
			totalRes = 0;
			;
			for (int id : ordering) {
				Vector3f error = func.getResidualVector(id);
				Point3f pt = surf.getVertex(id);
				Vector3f rError = new Vector3f(pt);
				rError.scale(error.dot(rError));

				res = rError.length();
				rError.scale((float) lambda);
				pt.add(rError);
				GeomUtil.normalize(pt);
				surf.setVertex(id, pt);
				totalRes += res;
			}
			totalRes /= vertCount;
			iter++;
			if (iter % updateInterval == 0)
				System.out.println("Iteration " + iter + " " + totalRes);
		} while (totalRes > errThreshold
				&& Math.abs(totalRes - lastError) > deltaThreshold
				&& iter < maxIters);
		System.out.println("Residual " + totalRes + " iterations " + iter);
	}

	/**
	 * Solve for optimal location of vertices in 2D plane
	 * 
	 * @param func
	 *            weighting function
	 * @param lambda
	 *            step size
	 */
	public void solveConjugateGradient2D(WeightVectorFunc func, double lambda) {
		int vertCount = surf.getVertexCount();
		int revLookupTable[] = new int[vertCount];
		int currentId = 0;
		for (int i = 0; i < vertCount; i++) {
			int len = neighborVertexVertexTable[i].length;
			if (len == 0) {
				revLookupTable[i] = -1;
			} else {
				revLookupTable[i] = currentId++;
			}
		}
		int[] lookupTable = new int[currentId];
		for (int i = 0; i < vertCount; i++) {
			int id = revLookupTable[i];
			if (id != -1) {
				lookupTable[id] = i;
			}
		}
		vertCount = currentId;
		Matrix Ax = new FlexCompRowMatrix(vertCount, vertCount);
		Matrix Ay = new FlexCompRowMatrix(vertCount, vertCount);
		DenseVector pX = new DenseVector(vertCount);
		DenseVector pY = new DenseVector(vertCount);
		DenseVector pnextX = new DenseVector(vertCount);
		DenseVector pnextY = new DenseVector(vertCount);
		// Initialize matrix
		for (int i = 0; i < vertCount; i++) {
			func.populate(Ax, Ay, lookupTable, revLookupTable, i);
			int origIndex = lookupTable[i];
			Point3f p = surf.getVertex(origIndex);
			if (labels[origIndex] == 0) {
				pX.set(i, p.x);
				pY.set(i, p.y);
			} else {
				pX.set(i, 0);
				pY.set(i, 0);
			}
			pnextX.set(i, p.x);
			pnextY.set(i, p.y);
		}
		BiCG solverX = new BiCG(pX);
		BiCG solverY = new BiCG(pY);
		IterationMonitor reporterX = new VerboseIterationMonitor(10000, 1E-6,
				1E-50, 1E40);
		IterationMonitor reporterY = new VerboseIterationMonitor(10000, 1E-6,
				1E-50, 1E40);
		solverX.setIterationMonitor(reporterX);
		solverY.setIterationMonitor(reporterY);
		try {
			solverX.solve(Ax, pX, pnextX);
		} catch (IterativeSolverNotConvergedException e) {
		}
		System.out.println("X: Residual " + reporterX.residual()
				+ " ITERATIONS " + reporterX.iterations());
		try {
			solverY.solve(Ay, pY, pnextY);
		} catch (IterativeSolverNotConvergedException e) {
		}
		System.out.println("Y: Residual " + reporterY.residual()
				+ " ITERATIONS " + reporterY.iterations());

		for (int id = 0; id < vertCount; id++) {
			Point3f p = surf.getVertex(lookupTable[id]);
			Point3f np = new Point3f((float) pnextX.get(id), (float) pnextY
					.get(id), p.z);

			surf.setVertex(lookupTable[id], np);
		}
	}

	/**
	 * Radial implicit solver to find optimal location for vertices.
	 * Unfortunately, a trivial solution is to place all vertices at the origin.
	 * Therefore, constraints must be added so that the solver avoids this and
	 * other trivial solutions.
	 * 
	 * @param func
	 *            weighting function
	 * @param lambda
	 *            step size
	 */
	public void solveConjugateGradientRadial(WeightVectorFunc func,
			double lambda) {
		int vertCount = surf.getVertexCount();
		int revLookupTable[] = new int[vertCount];
		int currentId = 0;
		for (int i = 0; i < vertCount; i++) {
			int len = neighborVertexVertexTable[i].length;
			if (len == 0) {
				revLookupTable[i] = -1;
			} else {
				revLookupTable[i] = currentId++;
			}
		}
		int[] lookupTable = new int[currentId];
		for (int i = 0; i < vertCount; i++) {
			int id = revLookupTable[i];
			if (id != -1) {
				lookupTable[id] = i;
			}
		}
		vertCount = currentId;
		Matrix Ar = new FlexCompRowMatrix(vertCount, vertCount);

		DenseVector pR = new DenseVector(vertCount);

		DenseVector pnextR = new DenseVector(vertCount);

		// Initialize matrix
		for (int i = 0; i < vertCount; i++) {
			func.populate(Ar, lookupTable, revLookupTable, i);
			int origIndex = lookupTable[i];
			if (labels[origIndex] == 0) {
				pR.set(i, elevations[origIndex]);
			} else {
				pR.set(i, 0);
			}
			pnextR.set(i, elevations[origIndex]);
		}
		BiCG solverR = new BiCG(pR);
		VerboseIterationMonitor reporterR = new VerboseIterationMonitor(1000,
				1E-5, 1E-50, 1E10);
		solverR.setIterationMonitor(reporterR);
		try {
			solverR.solve(Ar, pR, pnextR);
		} catch (IterativeSolverNotConvergedException e) {
		}
		System.out.println("R: Residual " + reporterR.residual()
				+ " ITERATIONS " + reporterR.iterations());
		for (int id = 0; id < vertCount; id++) {
			elevations[lookupTable[id]] = pnextR.get(id);
		}
	}

	/**
	 * Monitor sparse linear solver
	 * 
	 * @author Blake Lucas
	 * 
	 */
	protected class VerboseIterationMonitor extends DefaultIterationMonitor {
		double maxError = 0;
		double targetError = 0;

		public VerboseIterationMonitor(int maxIters, double maxErr,
				double maxDelta, double maxChange) {
			super(maxIters, maxErr, maxDelta, maxChange);
			targetError = maxErr;
			setTotalUnits(100);
		}

		@Override
		public boolean converged(double r)
				throws IterativeSolverNotConvergedException {
			if (iter % 10 == 0)
				System.out.println("Iteration " + iter + ") Error " + r);
			maxError = Math.max(r, maxError);
			setCompletedUnits((maxError - r) / (maxError - targetError));
			return super.converged(r);
		}

		@Override
		public boolean converged(double r, Vector v)
				throws IterativeSolverNotConvergedException {
			if (iter % 10 == 0)
				System.out.println("Iteration " + iter + ") Error " + r);
			maxError = Math.max(r, maxError);
			setCompletedUnits((maxError - r) / (maxError - targetError));
			return super.converged(r, v);
		}

		@Override
		public boolean converged(Vector v)
				throws IterativeSolverNotConvergedException {
			double r = residual();
			if (iter % 10 == 0)
				System.out.println("Iteration " + iter + ") Error " + r);
			maxError = Math.max(r, maxError);
			setCompletedUnits((maxError - r) / (maxError - targetError));
			return super.converged(v);
		}
	}

	/**
	 * Implicitly solve for location of vertices in 3d and then normalize those
	 * locations so that they lie on a sphere.
	 * 
	 * @param func
	 *            weighting function
	 * @param lambda
	 *            step size
	 */
	public void solveConjugateGradient3D(WeightVectorFunc func, double lambda) {
		int vertCount = surf.getVertexCount();
		int revLookupTable[] = new int[vertCount];
		int currentId = 0;
		for (int i = 0; i < vertCount; i++) {
			int len = neighborVertexVertexTable[i].length;
			if (len == 0) {
				revLookupTable[i] = -1;
			} else {
				revLookupTable[i] = currentId++;
			}
		}
		int[] lookupTable = new int[currentId];
		for (int i = 0; i < vertCount; i++) {
			int id = revLookupTable[i];
			if (id != -1) {
				lookupTable[id] = i;
			}
		}
		vertCount = currentId;
		Matrix Ax = new FlexCompRowMatrix(vertCount, vertCount);
		Matrix Ay = new FlexCompRowMatrix(vertCount, vertCount);
		Matrix Az = new FlexCompRowMatrix(vertCount, vertCount);

		DenseVector pX = new DenseVector(vertCount);
		DenseVector pY = new DenseVector(vertCount);
		DenseVector pZ = new DenseVector(vertCount);

		DenseVector pnextX = new DenseVector(vertCount);
		DenseVector pnextY = new DenseVector(vertCount);
		DenseVector pnextZ = new DenseVector(vertCount);

		// Initialize matrix
		for (int i = 0; i < vertCount; i++) {
			func.populate(Ax, Ay, Az, lookupTable, revLookupTable, i);
			int origIndex = lookupTable[i];
			Point3f p = surf.getVertex(origIndex);
			if (labels[origIndex] == 0) {
				pX.set(i, p.x);
				pY.set(i, p.y);
				pZ.set(i, p.z);
			} else {
				pX.set(i, 0);
				pY.set(i, 0);
				pZ.set(i, 0);
			}
			pnextX.set(i, p.x);
			pnextY.set(i, p.y);
			pnextZ.set(i, p.z);
		}
		BiCG solverX = new BiCG(pX);
		BiCG solverY = new BiCG(pY);
		BiCG solverZ = new BiCG(pZ);
		IterationMonitor reporterX = new VerboseIterationMonitor(10000, 1E-6,
				1E-50, 1E10);
		IterationMonitor reporterY = new VerboseIterationMonitor(10000, 1E-6,
				1E-50, 1E10);
		IterationMonitor reporterZ = new VerboseIterationMonitor(10000, 1E-6,
				1E-50, 1E10);

		solverX.setIterationMonitor(reporterX);
		solverY.setIterationMonitor(reporterY);
		solverZ.setIterationMonitor(reporterZ);
		try {
			solverX.solve(Ax, pX, pnextX);
		} catch (IterativeSolverNotConvergedException e) {
		}
		System.out.println("X: Residual " + reporterX.residual()
				+ " ITERATIONS " + reporterX.iterations());
		try {
			solverY.solve(Ay, pY, pnextY);
		} catch (IterativeSolverNotConvergedException e) {
		}
		System.out.println("Y: Residual " + reporterY.residual()
				+ " ITERATIONS " + reporterY.iterations());
		try {
			solverZ.solve(Az, pZ, pnextZ);
		} catch (IterativeSolverNotConvergedException e) {
		}
		System.out.println("Z: Residual " + reporterZ.residual()
				+ " ITERATIONS " + reporterZ.iterations());
		for (int id = 0; id < vertCount; id++) {
			Point3f p = surf.getVertex(lookupTable[id]);
			Point3f np = new Point3f((float) pnextX.get(id), (float) pnextY
					.get(id), (float) pnextZ.get(id));
			GeometricUtilities.normalize(np);
			surf.setVertex(lookupTable[id], np);
		}
	}

	/**
	 * Interface for weighted vector functions to populate sparse matricies and
	 * to calculate residual vector.
	 * 
	 * @author Blake Lucas
	 * 
	 */
	public interface WeightVectorFunc {
		/**
		 * Populate sparse matrix corresponding to a planar parameterization of a surface.
		 * @param Ax X coordinate matrix
		 * @param Ay Y coordinate matrix
		 * @param lookupTable lookup table into original surface vertices
		 * @param revLookupTable reverse lookup table into unconstrained surface vertices
		 * @param index vertex id
		 * @return vertex weighting factor
		 */
		public double populate(Matrix Ax, Matrix Ay, int[] lookupTable,
				int[] revLookupTable, int index);
		/**
		 * Populate sparse matrix corresponding to a spherical parameterization of a surface.
		 * @param Ax X coordinate matrix
		 * @param Ay Y coordinate matrix
		 * @param Az Z coordinate matrix
		 * @param lookupTable lookup table into original surface vertices
		 * @param revLookupTable reverse lookup table into unconstrained surface vertices
		 * @param index vertex id
		 * @return vertex weighting factor
		 */
		public double populate(Matrix Ax, Matrix Ay, Matrix Az,
				int[] lookupTable, int[] revLookupTable, int index);
		/**
		 * Populate sparse matrix corresponding to radial component
		 * @param Ar radii matrix
		 * @param lookupTable lookup table into original surface vertices
		 * @param revLookupTable reverse lookup table into unconstrained surface vertices
		 * @param index vertex id
		 * @return weighting factor
		 */
		public double populate(Matrix Ar, int[] lookupTable,
				int[] revLookupTable, int index);
		/**
		 * Get residual vector for mapping metric
		 * @param index vertex id
		 * @return residual vector
		 */
		public Vector3f getResidualVector(int index);
	}

}
