package edu.jhu.ece.iacl.algorithms.graphics.map;

import javax.vecmath.Point3f;

import no.uib.cipr.matrix.DenseVector;
import no.uib.cipr.matrix.Matrix;
import no.uib.cipr.matrix.Vector;
import no.uib.cipr.matrix.sparse.BiCG;
import no.uib.cipr.matrix.sparse.DefaultIterationMonitor;
import no.uib.cipr.matrix.sparse.FlexCompRowMatrix;
import no.uib.cipr.matrix.sparse.IterativeSolverNotConvergedException;
import edu.jhu.ece.iacl.algorithms.graphics.GeometricUtilities;
import edu.jhu.ece.iacl.jist.pipeline.AbstractCalculation;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Up-sample spherical data by solving a sparse linear system. Mean-value
 * weights are used because the system may not converge if conformal weights are
 * used. Therefore, this is not true Laplacian interpolation, but mean-value
 * weights are close enough that the results look similar in the convergent
 * case.
 * 
 * @author Blake Lucas
 * 
 */
public class MeanValueSphericalMapInterpolator extends AbstractCalculation {
	public MeanValueSphericalMapInterpolator() {
		super();
		setLabel("Laplacian Interpolator");
	}

	/**
	 * Up-sample spherical map
	 * 
	 * @param targetSurface
	 *            higher-resolution surface that indexes reference
	 * @param sourceIndexSurface
	 *            spherical map with indexes stored in vertex data
	 * @return up-sampled spherical map
	 */
	public EmbeddedSurface solve(EmbeddedSurface targetSurface,
			EmbeddedSurface sourceIndexSurface) {
		int vertCount = targetSurface.getVertexCount();
		int idxVertCount = sourceIndexSurface.getVertexCount();
		double[][] idxData = sourceIndexSurface.getVertexData();
		int[] labels = new int[vertCount];
		double[][] refVertData = new double[vertCount][3];
		for (int i = 0; i < vertCount; i++) {
			labels[i] = 1;
			Point3f pt = targetSurface.getVertex(i);
			refVertData[i][0] = pt.x;
			refVertData[i][1] = pt.y;
			refVertData[i][2] = pt.z;
		}
		for (int i = 0; i < idxVertCount; i++) {
			int j = (int) idxData[i][0];
			labels[j] = 0;
			Point3f pt = sourceIndexSurface.getVertex(i);
			refVertData[j][0] = pt.x;
			refVertData[j][1] = pt.y;
			refVertData[j][2] = pt.z;

		}
		int[][] neighborVertexVertexTable = EmbeddedSurface
				.buildNeighborVertexVertexTable(targetSurface,
						EmbeddedSurface.Direction.COUNTER_CLOCKWISE);
		Matrix Ax = new FlexCompRowMatrix(vertCount, vertCount);
		DenseVector pX = new DenseVector(vertCount);
		DenseVector pnextX = new DenseVector(vertCount);
		Matrix Ay = new FlexCompRowMatrix(vertCount, vertCount);
		DenseVector pY = new DenseVector(vertCount);
		DenseVector pnextY = new DenseVector(vertCount);
		Matrix Az = new FlexCompRowMatrix(vertCount, vertCount);
		DenseVector pZ = new DenseVector(vertCount);
		DenseVector pnextZ = new DenseVector(vertCount);
		// Initialize matrix
		for (int id = 0; id < vertCount; id++) {
			populate(Ax, neighborVertexVertexTable, labels, id);
			populate(Ay, neighborVertexVertexTable, labels, id);
			populate(Az, neighborVertexVertexTable, labels, id);
			if (labels[id] == 0) {
				pX.set(id, refVertData[id][0]);
				pY.set(id, refVertData[id][1]);
				pZ.set(id, refVertData[id][2]);
			} else {
				pX.set(id, 0);
				pY.set(id, 0);
				pZ.set(id, 0);
			}
			pnextX.set(id, refVertData[id][0]);
			pnextY.set(id, refVertData[id][1]);
			pnextZ.set(id, refVertData[id][2]);
		}
		BiCG solverX = new BiCG(pX);
		BiCG solverY = new BiCG(pY);
		BiCG solverZ = new BiCG(pZ);
		VerboseIterationMonitor reporterX = new VerboseIterationMonitor(10000,
				1E-6, 1E-50, 1E10);
		VerboseIterationMonitor reporterY = new VerboseIterationMonitor(10000,
				1E-6, 1E-50, 1E10);
		VerboseIterationMonitor reporterZ = new VerboseIterationMonitor(10000,
				1E-6, 1E-50, 1E10);
		solverX.setIterationMonitor(reporterX);
		solverY.setIterationMonitor(reporterY);
		solverZ.setIterationMonitor(reporterZ);
		try {
			solverX.solve(Ax, pX, pnextX);
		} catch (IterativeSolverNotConvergedException e) {
		}
		System.out.println("X: Residual " + reporterX.residual()
				+ " ITERATIONS " + reporterX.iterations());
		try {
			solverY.solve(Ay, pY, pnextY);
		} catch (IterativeSolverNotConvergedException e) {
		}
		System.out.println("Y: Residual " + reporterY.residual()
				+ " ITERATIONS " + reporterY.iterations());
		try {
			solverZ.solve(Az, pZ, pnextZ);
		} catch (IterativeSolverNotConvergedException e) {
		}
		System.out.println("Z: Residual " + reporterZ.residual()
				+ " ITERATIONS " + reporterZ.iterations());

		// Initialize matrix
		for (int id = 0; id < vertCount; id++) {
			populate(Ax, targetSurface, neighborVertexVertexTable, labels, id);
			populate(Ay, targetSurface, neighborVertexVertexTable, labels, id);
			populate(Az, targetSurface, neighborVertexVertexTable, labels, id);
			if (labels[id] == 0) {
				pX.set(id, refVertData[id][0]);
				pY.set(id, refVertData[id][1]);
				pZ.set(id, refVertData[id][2]);
			} else {
				pX.set(id, 0);
				pY.set(id, 0);
				pZ.set(id, 0);
			}
			pnextX.set(id, refVertData[id][0]);
			pnextY.set(id, refVertData[id][1]);
			pnextZ.set(id, refVertData[id][2]);
		}
		try {
			solverX.solve(Ax, pX, pnextX);
		} catch (IterativeSolverNotConvergedException e) {
		}
		System.out.println("X: Residual " + reporterX.residual()
				+ " ITERATIONS " + reporterX.iterations());
		try {
			solverY.solve(Ay, pY, pnextY);
		} catch (IterativeSolverNotConvergedException e) {
		}
		System.out.println("Y: Residual " + reporterY.residual()
				+ " ITERATIONS " + reporterY.iterations());
		try {
			solverZ.solve(Az, pZ, pnextZ);
		} catch (IterativeSolverNotConvergedException e) {
		}
		System.out.println("Z: Residual " + reporterZ.residual()
				+ " ITERATIONS " + reporterZ.iterations());

		EmbeddedSurface refSurfaceClone = targetSurface.clone();
		for (int id = 0; id < vertCount; id++) {
			Point3f pt = new Point3f((float) pnextX.get(id), (float) pnextY
					.get(id), (float) pnextZ.get(id));
			GeometricUtilities.normalize(pt);
			refSurfaceClone.setVertex(id, pt);
		}
		return refSurfaceClone;
	}

	/**
	 * Monitor linear system solver
	 * 
	 * @author Blake Lucas
	 * 
	 */
	protected class VerboseIterationMonitor extends DefaultIterationMonitor {
		double maxError = 0;
		double targetError = 0;

		public VerboseIterationMonitor(int maxIters, double maxErr,
				double maxDelta, double maxChange) {
			super(maxIters, maxErr, maxDelta, maxChange);
			targetError = maxErr;
			setTotalUnits(100);
		}

		@Override
		public boolean converged(double r)
				throws IterativeSolverNotConvergedException {
			if (iter % 10 == 0)
				System.out.println("Iteration " + iter + ") Error " + r);
			maxError = Math.max(r, maxError);
			setCompletedUnits((maxError - r) / (maxError - targetError));
			return super.converged(r);
		}

		@Override
		public boolean converged(double r, Vector v)
				throws IterativeSolverNotConvergedException {
			if (iter % 10 == 0)
				System.out.println("Iteration " + iter + ") Error " + r);
			maxError = Math.max(r, maxError);
			setCompletedUnits((maxError - r) / (maxError - targetError));
			return super.converged(r, v);
		}

		@Override
		public boolean converged(Vector v)
				throws IterativeSolverNotConvergedException {
			double r = residual();
			if (iter % 10 == 0)
				System.out.println("Iteration " + iter + ") Error " + r);
			maxError = Math.max(r, maxError);
			setCompletedUnits((maxError - r) / (maxError - targetError));
			return super.converged(v);
		}
	}

	/**
	 * Populate matrix with uniform weights
	 * 
	 * @param Ax
	 *            matrix
	 * @param neighborVertexVertexTable
	 *            vertex-vertex table
	 * @param labels
	 *            constraints
	 * @param id
	 *            vertex id
	 */
	public void populate(Matrix Ax, int[][] neighborVertexVertexTable,
			int[] labels, int id) {
		if (labels[id] != 0) {
			int len = neighborVertexVertexTable[id].length;
			double weight = 1.0 / len;
			int nbr;
			double total = 0;
			for (int i = 0; i < len; i++) {
				nbr = neighborVertexVertexTable[id][i];
				Ax.set(id, nbr, -weight);
			}
			if (total > 0) {
				total = 1 / total;
			}
		}
		Ax.set(id, id, 1);
	}

	/**
	 * Populate matrix with mean-value weights
	 * 
	 * @param Ax
	 *            matrix
	 * @param refSurface
	 *            reference surface
	 * @param neighborVertexVertexTable
	 *            vertex-vertex table
	 * @param labels
	 *            constraints
	 * @param id
	 *            vertex id
	 */
	public void populate(Matrix Ax, EmbeddedSurface refSurface,
			int[][] neighborVertexVertexTable, int[] labels, int id) {
		if (labels[id] != 0) {
			int len = neighborVertexVertexTable[id].length;
			Point3f pivot = refSurface.getVertex(id);
			double weight;
			double tana, tanb;
			Point3f pnext, pcurr, plast;
			int nbr;
			double total = 0;
			for (int i = 0; i < len; i++) {
				pnext = refSurface
						.getVertex(neighborVertexVertexTable[id][(i + 1) % len]);
				pcurr = refSurface
						.getVertex(nbr = neighborVertexVertexTable[id][i]);
				plast = refSurface.getVertex(neighborVertexVertexTable[id][(i
						+ len - 1)
						% len]);

				tana = GeometricUtilities.tanHalfAngle(pnext, pcurr, pivot);
				tanb = GeometricUtilities.tanHalfAngle(plast, pcurr, pivot);
				double l = pivot.distance(pcurr);
				if (l > 1E-6) {
					weight = 0.5 * (tana + tanb) / l;
				} else {
					weight = 0;
				}
				total += weight;
				Ax.set(id, nbr, -weight);
			}
			Ax.set(id, id, total);
		} else {
			Ax.set(id, id, 1);
		}
	}
}
