package edu.jhu.ece.iacl.algorithms.graphics.smooth;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Stack;

import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;

import no.uib.cipr.matrix.DenseVector;
import no.uib.cipr.matrix.Matrix;
import no.uib.cipr.matrix.sparse.BiCG;
import no.uib.cipr.matrix.sparse.FlexCompRowMatrix;
import no.uib.cipr.matrix.sparse.IterativeSolverNotConvergedException;

import edu.jhu.ece.iacl.algorithms.VersionUtil;
import edu.jhu.ece.iacl.algorithms.graphics.intersector.SurfaceSelfIntersector;
import edu.jhu.ece.iacl.algorithms.graphics.surf.SelfIntersectingPreventionProgressiveSurface;
import edu.jhu.ece.iacl.jist.structures.geom.EmbeddedSurface;

/**
 * Smooth surface using an implicit approach that is computationally stable.
 * Soft constraints can be added by weighing the error function that produces
 * the sparse linear system.
 * 
 * @author Blake Lucas
 * 
 */
public class MultiResolutionImplicitSmoothing extends
		SelfIntersectingPreventionProgressiveSurface {
	public static String getVersion() {
		return VersionUtil.parseRevisionNumber("$Revision: 1.6 $");
	}

	protected float lambda = 0.8f;
	protected int smoothStepIters = 5;
	protected int maxIters = 3;
	protected float maxCurvatureThreshold = 0.2f;
	protected float maxDecimation = 1.0f;
	protected double[][] labels;
	protected boolean decimate = false;
	protected boolean regularize = false;
	protected boolean preventSelfIntersection = false;

	protected Stack<MeshTopologyOperation> vertStack = new Stack<MeshTopologyOperation>();
	protected SurfaceSelfIntersector intersector;

	public boolean isPreventSelfIntersection() {
		return preventSelfIntersection;
	}

	public void setPreventSelfIntersection(boolean preventSelfIntersection) {
		this.preventSelfIntersection = preventSelfIntersection;
	}

	public boolean isRegularize() {
		return regularize;
	}

	public void setMaxDecimation(float maxDec) {
		this.maxDecimation = maxDec;
	}

	public void setRegularize(boolean regularize) {
		this.regularize = regularize;
	}

	protected WeightVectorFunc weightVectorFunc;

	public MultiResolutionImplicitSmoothing(float maxCurv, float lambda,
			int smoothIters, int multiresIters) {
		this.maxCurvatureThreshold = maxCurv;
		this.lambda = lambda;
		this.maxIters = multiresIters;
		this.smoothStepIters = smoothIters;
		weightVectorFunc = new LaplacianWeightFunc();
		this.setLabel("Multi-Resolution Smoothing");
	}

	public MultiResolutionImplicitSmoothing() {
		weightVectorFunc = new LaplacianWeightFunc();
		this.setLabel("Multi-Resolution Smoothing");
	}

	public void setDecimate(boolean decimate) {
		this.decimate = decimate;
	}
	/**
	 * Smooth surface 
	 * @param origSurf original surface
	 * @return smoothed surface
	 */
	public EmbeddedSurface solve(EmbeddedSurface origSurf) {
		init(origSurf, false);
		setTotalUnits(maxIters * smoothStepIters);
		labels = new double[surf.getVertexCount()][1];
		if (preventSelfIntersection) {
			intersector = new SurfaceSelfIntersector(surf, 120);
			intersector.init();
			intersector.labelIntersections();
		}
		for (int count = 0; count < maxIters; count++) {
			System.out.println("MULTI-RESOLUTION ITERATION " + (count + 1));
			if (regularize) {
				regularize();
				// Do not restore regularization operations
				vertStack.empty();
			}
			decimate(maxCurvatureThreshold, maxDecimation);
			tessellate(1);
			if (regularize) {
				regularize();
				// Do not restore regularization operations
				vertStack.empty();
			}
			incrementCompletedUnits();
			smooth(smoothStepIters);
		}

		surf.setVertexData(origSurf.getVertexData());
		if (decimate) {
			if (regularize)
				regularize();
			decimate(maxCurvatureThreshold, maxDecimation);
		}
		if (regularize) {
			regularize();
		}
		if (decimate || regularize) {
			surf = createSurface();
		}
		/*
		 * if(preventSelfIntersection&&!intersector.sanityCheck(surf)){
		 * System.err.println("Self-Intersection Sanity Check Failed!"); }
		 */
		markCompleted();
		return surf;
	}
	
	public void regularize() {
		super.regularize(vertStack);
	}

	protected static final float MedianAbsoluteDeviation = 1.4826f;

	protected double scaleL;

	protected double lorentizanNorm() {
		int vertCount = surf.getVertexCount();
		double curv = 0;
		double kcurv = 0;
		for (int id = 0; id < vertCount; id++) {
			Vector3f c = getMeanCurvature(id);
			kcurv = c.length();
			curv += Math.log(1 + 0.5 * kcurv * kcurv / (scaleL * scaleL));
		}
		curv = Math.sqrt(curv / vertCount);
		System.out.println("LORENTZIAN " + curv);
		return curv;
	}

	protected double updateLorentzianScale() {
		int vertCount = surf.getVertexCount();
		ArrayList<Float> curvs = new ArrayList<Float>();
		for (int id = 0; id < vertCount; id++) {
			if (neighborVertexVertexTable[id].length > 0) {
				Vector3f c = getMeanCurvature(id);
				curvs.add(c.length());
			}
		}
		Collections.sort(curvs);
		float median;
		if (curvs.size() % 2 == 0) {
			median = 0.5f * (curvs.get(curvs.size() / 2) + curvs.get(curvs
					.size() / 2 + 1));
		} else {
			median = curvs.get(curvs.size() / 2);
		}
		for (int i = 0; i < curvs.size(); i++) {
			curvs.set(i, Math.abs(curvs.get(i) - median));
		}
		Collections.sort(curvs);
		if (curvs.size() % 2 == 0) {
			median = 0.5f * (curvs.get(curvs.size() / 2) + curvs.get(curvs
					.size() / 2 + 1));
		} else {
			median = curvs.get(curvs.size() / 2);
		}
		return MedianAbsoluteDeviation * median;
	}

	protected static float DELTA = 0.25f;
	protected static int MAX_ITERS = 1;
	protected double[] weights;
	/**
	 * Smooth with Stabilized Biconjugate gradient solver
	 * @param func weighting function
	 * @return smoothed vertex coordinates
	 */
	public DenseVector[] iterateConjugateGradient(WeightVectorFunc func) {
		int vertCount = surf.getVertexCount();
		Matrix Ax = new FlexCompRowMatrix(vertCount, vertCount);
		Matrix Ay = new FlexCompRowMatrix(vertCount, vertCount);
		Matrix Az = new FlexCompRowMatrix(vertCount, vertCount);
		DenseVector pX = new DenseVector(vertCount);
		DenseVector pY = new DenseVector(vertCount);
		DenseVector pZ = new DenseVector(vertCount);
		DenseVector pnextX = new DenseVector(vertCount);
		DenseVector pnextY = new DenseVector(vertCount);
		DenseVector pnextZ = new DenseVector(vertCount);
		// Initialize matrix
		for (int i = 0; i < vertCount; i++) {
			func.populate(Ax, Ay, Az, i);
			Point3f p = surf.getVertex(i);
			pX.set(i, p.x);
			pY.set(i, p.y);
			pZ.set(i, p.z);
			pnextX.set(i, p.x);
			pnextY.set(i, p.y);
			pnextZ.set(i, p.z);
		}

		// Scale curvature matrix by update increment
		Ax.scale(-lambda);
		Ay.scale(-lambda);
		Az.scale(-lambda);
		// Add identity term
		for (int i = 0; i < vertCount; i++) {
			Ax.set(i, i, 1 + Ax.get(i, i));
			Ay.set(i, i, 1 + Ay.get(i, i));
			Az.set(i, i, 1 + Az.get(i, i));
		}

		BiCG solverX = new BiCG(pX);
		BiCG solverY = new BiCG(pY);
		BiCG solverZ = new BiCG(pZ);
		try {
			solverX.solve(Ax, pX, pnextX);
			solverY.solve(Ay, pY, pnextY);
			solverZ.solve(Az, pZ, pnextZ);
			return new DenseVector[] { pnextX, pnextY, pnextZ };
		} catch (IterativeSolverNotConvergedException e) {
			e.printStackTrace();
			return null;
		}

	}

	public interface WeightVectorFunc {
		public double populate(Matrix Ax, Matrix Ay, Matrix Az, int index);
	}
	/**
	 * Laplacian weighting function
	 * @author Blake Lucas
	 *
	 */
	protected class LaplacianWeightFunc implements WeightVectorFunc {
		public double populate(Matrix Ax, Matrix Ay, Matrix Az, int id) {
			int len = neighborVertexVertexTable[id].length;
			if (len == 0)
				return 0;
			int nbr;
			double curv = weights[id];
			double w = curv * 1.0 / len;
			for (int i = 0; i < len; i++) {
				nbr = neighborVertexVertexTable[id][i];
				Ax.add(id, nbr, w);
				Ay.add(id, nbr, w);
				Az.add(id, nbr, w);
			}
			Ax.add(id, id, -curv);
			Ay.add(id, id, -curv);
			Az.add(id, id, -curv);
			return curv;
		}
	}
	/**
	 * Smooth surface with implicit solver
	 * @param iters maximum iterations
	 * @return smoothed surface
	 */
	public EmbeddedSurface smooth(int iters) {
		System.out.println("SMOOTH " + iters + " iterations");
		double maxCurvature = 0, curv;
		// meanCurvature=0;
		Point3f p;
		DenseVector[] pts;
		int vertCount = surf.getVertexCount();
		weights = new double[vertCount];
		for (int id = 0; id < vertCount; id++) {
			Vector3f c = getMeanCurvature(id);
			weights[id] = curv = Math.log(1 + c.length());
			maxCurvature = Math.max(curv, maxCurvature);
		}
		for (int id = 0; id < vertCount; id++) {
			weights[id] /= maxCurvature;
		}
		double oldVol = surf.volume();
		Point3f oldCenter = surf.getCenterOfMass();
		for (int i = 0; i < iters; i++) {
			pts = iterateConjugateGradient(weightVectorFunc);
			// Update surface point locations
			for (int id = 0; id < vertCount; id++) {
				if (weights[id] > 0) {
					Point3f np = new Point3f((float) pts[0].get(id),
							(float) pts[1].get(id), (float) pts[2].get(id));
					p = surf.getVertex(id);
					if (preventSelfIntersection) {
						intersector.updateVertex(id, np);
					} else {
						surf.setVertex(id, np);
					}
					pts[0].set(id, p.x);
					pts[1].set(id, p.y);
					pts[2].set(id, p.z);
				}
			}
			maxCurvature = 0;
			// Preserve Volume and Center of Mass
			if (!preventSelfIntersection) {
				surf.scaleVertices((float) Math.pow(oldVol / surf.volume(),
						0.333333));
				Point3f center = surf.getCenterOfMass();
				center.sub(oldCenter);
				center.negate();
				surf.translate(center);
			}
			for (int id = 0; id < vertCount; id++) {
				Vector3f c = getMeanCurvature(id);
				weights[id] = curv = Math.log(1 + c.length());
				maxCurvature = Math.max(curv, maxCurvature);
			}
			// Rescale volume since Laplacian will decrease volume
			System.out.println("ITERATION " + (i + 1) + " Max Curvature "
					+ maxCurvature);
			for (int id = 0; id < vertCount; id++) {
				weights[id] /= maxCurvature;
			}
			incrementCompletedUnits();
		}
		System.gc();
		// Map points back to original surface
		return surf;
	}
}
